/* -*- Mode: C++; tab-width: 8; c-basic-offset: 2; indent-tabs-mode: nil; -*- */

#include "FdTable.h"

#include <unordered_set>

#include "rr/rr.h"

#include "Session.h"
#include "task.h"

using namespace std;

Switchable FdTable::will_write(Task* t, int fd) {
  auto it = fds.find(fd);
  if (it != fds.end()) {
    return it->second->will_write(t);
  }
  return ALLOW_SWITCH;
}

void FdTable::did_write(Task* t, int fd,
                        const std::vector<FileMonitor::Range>& ranges) {
  auto it = fds.find(fd);
  if (it != fds.end()) {
    it->second->did_write(t, ranges);
  }
}

void FdTable::did_dup(int from, int to) {
  if (fds.count(from)) {
    fds[to] = fds[from];
  } else {
    fds.erase(to);
  }
  update_syscallbuf_fds_disabled(to);
}

void FdTable::did_close(int fd) {
  fds.erase(fd);
  update_syscallbuf_fds_disabled(fd);
}

static bool is_fd_monitored_in_any_task(AddressSpace* vm, int fd) {
  for (Task* t : vm->task_set()) {
    if (t->fd_table()->is_monitoring(fd)) {
      return true;
    }
  }
  return false;
}

void FdTable::update_syscallbuf_fds_disabled(int fd) {
  assert(fd >= 0);

  unordered_set<AddressSpace*> vms_updated;
  // It's possible for tasks with different VMs to share this fd table.
  // But tasks with the same VM might have different fd tables...
  for (Task* t : task_set()) {
    AddressSpace* vm = t->vm().get();
    if (vms_updated.find(vm) != vms_updated.end()) {
      continue;
    }
    vms_updated.insert(vm);

    if (!t->syscallbuf_fds_disabled_child.is_null() &&
        fd < SYSCALLBUF_FDS_DISABLED_SIZE) {
      bool disable =
          is_fd_monitored_in_any_task(vm, fd) || RR_RESERVED_ROOT_DIR_FD == fd;
      t->write_mem(t->syscallbuf_fds_disabled_child + fd, (char)disable);
    }
  }
}

void FdTable::init_syscallbuf_fds_disabled(Task* t) {
  if (t->syscallbuf_fds_disabled_child.is_null()) {
    return;
  }

  char disabled[SYSCALLBUF_FDS_DISABLED_SIZE];
  memset(disabled, 0, sizeof(disabled));
  assert(RR_RESERVED_ROOT_DIR_FD < SYSCALLBUF_FDS_DISABLED_SIZE);
  disabled[RR_RESERVED_ROOT_DIR_FD] = 1;

  // It's possible that some tasks in this address space have a different
  // FdTable. We need to disable syscallbuf for an fd if any tasks for this
  // address space are monitoring the fd.
  for (Task* vm_t : t->vm()->task_set()) {
    for (auto& it : vm_t->fd_table()->fds) {
      int fd = it.first;
      assert(fd >= 0);
      if (fd < SYSCALLBUF_FDS_DISABLED_SIZE) {
        disabled[fd] = 1;
      }
    }
  }

  t->write_mem(t->syscallbuf_fds_disabled_child, disabled,
               SYSCALLBUF_FDS_DISABLED_SIZE);
}

static bool is_fd_open(Task* t, int fd) {
  char path[PATH_MAX];
  sprintf(path, "/proc/%d/fd/%d", t->tid, fd);
  struct stat st;
  return 0 == lstat(path, &st);
}

void FdTable::update_for_cloexec(Task* t, TraceTaskEvent& event) {
  vector<int> fds_to_close;

  if (t->session().is_recording()) {
    for (auto& it : fds) {
      if (!is_fd_open(t, it.first)) {
        fds_to_close.push_back(it.first);
      }
    }
    event.set_fds_to_close(fds_to_close);
  } else {
    fds_to_close = event.fds_to_close();
  }

  for (auto fd : fds_to_close) {
    did_close(fd);
  }
}
