/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  hlub@knoware.nl
*/



#include "rlwrap.h"


char slaves_working_directory[MAXPATHLEN+1];

static FILE *log_fp;

static void
utils_warn(const char *message, va_list ap)
{

  int saved_errno = errno;
  char buffer[BUFFSIZE];
  

  snprintf1(buffer, sizeof(buffer) - 1, "%s: ", program_name);
  vsnprintf(buffer + strlen(buffer), sizeof(buffer) - strlen(buffer) - 1,
	    message, ap);
  if (saved_errno)
    snprintf(buffer + strlen(buffer), sizeof(buffer) - strlen(buffer) - 1,
	     ": %s", strerror(saved_errno));
  mystrlcat(buffer, "\n", sizeof(buffer));

  fflush(stdout);
  fputs(buffer, stderr);
  fflush(stderr);
  DPRINTF1(DEBUG_ALL,
	   "Error/warning: %s\n(you can get rid of warnings by using the -n option)",
	   buffer);
  /* we want this because sometimes error messages (esp. from client) are dropped) */

}

void
mywarn(const char *message, ...)
{
  va_list ap;
  char *warning_message;

  if (nowarn)
    return;
  warning_message = add2strings("warning: ", message);
  va_start(ap, message);
  utils_warn(warning_message, ap);
  va_end(ap);
}

void
myerror(const char *message, ...)
{
  va_list ap;
  char *error_message = add2strings("error: ", message);

  va_start(ap, message);
  utils_warn(error_message, ap);
  va_end(ap);
  if (!i_am_child)
    cleanup_rlwrap_and_exit(EXIT_FAILURE);
  else /* child: die and let parent clean up */
    exit(-1);
}




char *
mybasename(char *filename)
{				/* determine basename of "filename" */
  char *p;

  /* find last '/' in name (if any) */
  for (p = filename + strlen(filename) - 1; p > filename; p--)
    if (*(p - 1) == '/')
      break;
  return p;
}

char *
mydirname(char *filename)
{				/* determine directory component of "name" */
  char *p;

  /* find last '/' in name (if any) */
  for (p = filename + strlen(filename) - 1; p > filename; p--)
    if (*(p - 1) == '/')
      break;
  return (p == filename ? "." : mystrndup(filename, p - filename));
}



void *
mymalloc(size_t size)
{				/* malloc with simplistic error handling: just bail out when out of memory */
  void *ptr;

  ptr = malloc(size);
  if (ptr == NULL) {
    /* don't call myerror(), as this calls mymalloc() again */
    fprintf(stderr, "Out of memory: tried in vain to allocate %d bytes\n", size);
    exit(EXIT_FAILURE);
  }	
  return ptr;
}


/* wrappers around strlcat and strlcpy, if available, otherwise emulations
   of them. Both versions *assure* 0-termination, but don't check for
   truncation: return type is void */

void
mystrlcpy(char *dst, const char *src, size_t size)
{
#ifdef HAVE_STRLCPY
  strlcpy(dst, src, size);
#else
  strncpy(dst, src, size - 1);
  dst[size - 1] = '\0';
#endif
}

void
mystrlcat(char *dst, const char *src, size_t size)
{
#ifdef HAVE_STRLCAT
  strlcat(dst, src, size);
  dst[size - 1] = '\0';		/* we don't check for truncation, just assure '\0'-termination. */
#else
  strncat(dst, src, size - strnlen(dst, size) - 1);
  dst[size - 1] = '\0';
#endif
}

char *mystrstr(const char *haystack, const char *needle) {
  return strstr(haystack, needle);
}




char *
mysavestring(const char *string)
{
  return mystrndup(string, strlen(string));
}

char *
mystrndup(const char *string, int len)
{
  /* allocate copy of string on the heap */
  char *buf;

  buf = (char *)mymalloc(len + 1);
  mystrlcpy(buf, string, len + 1);
  return buf;
}



char *
add3strings(const char *str1, const char *str2, const char *str3)
{
  int size = strlen(str1) + strlen(str2) + strlen(str3) + 1;	/* total length plus 0 byte */
  char *buf = (char *)mymalloc(size);

  mystrlcpy(buf, str1, size);
  mystrlcat(buf, str2, size);
  mystrlcat(buf, str3, size);
  return buf;
}


void
open_logfile(const char *filename)
{
  time_t now;

  log_fp = fopen(filename, "a");
  if (!log_fp)
    myerror("Cannot write to logfile %s", filename);
  now = time(NULL);
  fprintf(log_fp, "\n\n[rlwrap] %s\n", ctime(&now));
}

void
write_logfile(const char *str)
{
  if (log_fp)
    fputs(str, log_fp);
}


char *
search_and_replace(char *patt, char *repl, const char *string, int cursorpos,
		   int *line, int *col)
{
  int i, j, k;
  int pattlen = strlen(patt);
  int replen = strlen(repl);
  int stringlen = strlen(string);
  int past_cursorpos = 0;
  int current_line = 1;
  int current_column = 1;
  size_t scratchsize;
  char *scratchpad, *result;


  DPRINTF2(DEBUG_READLINE, "string=%s, cursorpos=%d",
	   mangle_string_for_debug_log(string, 40), cursorpos);
  scratchsize = max(stringlen, (stringlen * replen) / pattlen) + 1;	/* worst case : repleng > pattlen and string consists of only <patt> */
  DPRINTF1(DEBUG_READLINE, "Allocating %d bytes for scratchpad", scratchsize);
  scratchpad = mymalloc(scratchsize);


  for (i = j = 0; i < stringlen;) {
    if (strncmp(patt, string + i, pattlen) == 0) {
      i += pattlen;
      for (k = 0; k < replen; k++)
	scratchpad[j++] = repl[k];
      current_line++;
      current_column = 1;
    } else {
      scratchpad[j++] = string[i++];
      current_column++;
    }
    if (i >= cursorpos && !past_cursorpos) {
      past_cursorpos = 1;
      if (line)
	*line = current_line;
      if (col)
	*col = current_column;
    }
  }
  scratchpad[j] = '\0';
  result = mysavestring(scratchpad);
  free(scratchpad);
  return (result);
}

char *
first_of(char **strings)
{				/* returns the first non-NULL element of an array of char* */
  char **p;

  for (p = strings; *p == NULL; p++);
  return *p;
}


char *
as_string(int i)
{
  char *newstring = mymalloc(100), *return_value;

  sprintf(newstring, "%d", i);
  return_value = mysavestring(newstring);
  free(newstring);
  return (return_value);
}

size_t
filesize(const char *filename)
{
  struct stat buf;

  if (stat(filename, &buf))
    myerror("couldn't stat file %s", filename);
  return (size_t) buf.st_size;
}

void
close_logfile()
{
  if (log_fp)
    fclose(log_fp);
}

void
close_open_files_without_writing_buffers() /* called from child just before exec(command) */
{
  if(log_fp)
    close(fileno(log_fp));  /* don't flush buffers to avoid avoid double double output output */
  if (debug)
    close(fileno(debug_fp));
}

static char mangle_buf[BUFFSIZE];

char *
mangle_string_for_debug_log(const char *string, int len)
{
  char *tmp = mangle_buf;
  char oct[4];
  int i, j;

  assert(len + 3 < BUFFSIZE - 1);


  for (i = j = 0; j < len - 3 && string[i];) {
    if (isprint(string[i]) && string[i] != '\n')
      tmp[j++] = string[i++];
    else {
      tmp[j++] = '\\';
      snprintf1(oct, sizeof(oct), "%03o", string[i++]);
      strncpy(&tmp[j], oct, 3);
      j += 3;
    }
  }

  if (j == len - 3)
    for (; j < len; j++)
      tmp[j] = '.';

  tmp[j] = '\0';

  return tmp;
}





void
change_working_directory()
{
  
  snprintf0(slaves_working_directory, MAXPATHLEN, "?");

#ifdef HAVE_PROC_PID_CWD
  static char proc_pid_cwd[MAXPATHLEN+1];
  static int initialized = FALSE;
  int linklen = 0;
  
  if (!initialized && child_pid > 0) { 	/* first time we're called after birth of child */
    snprintf1(proc_pid_cwd, MAXPATHLEN , "/proc/%d/cwd", child_pid);
    initialized = TRUE;
  }	
  if (chdir(proc_pid_cwd) == 0) {
#ifdef HAVE_READLINK
    linklen = readlink(proc_pid_cwd, slaves_working_directory, MAXPATHLEN);
    if (linklen > 0)
      slaves_working_directory[linklen] = '\0';
    else
      snprintf1(slaves_working_directory, MAXPATHLEN, "? (%s)", strerror(errno));
#endif /* HAVE_READLINK */
  }		    
#else
  /* do nothing at all */
#endif
}


 /* bleach(buffer) eliminate ANSI/xterm colour codes (regexp: \033\[[\d;]*m ) from buffer,
    this may miss colour codes that straddle a buffer boundary! @@@ */

void bleach (char*buffer) {
  char *scratch = mymalloc(BUFFSIZE), *p, *q, *r;
  assert(strnlen(buffer, BUFFSIZE +1) < BUFFSIZE); /* buffer is properly NULL-terminated */
  for (p = buffer, q = scratch; *p; p++, q++) {

    if(*p == '\033' && *(p+1) == '[') { /* got a colour code here? */
      r = p;  /* remember where we were */
      p += 2; /* skip ESC[ */
      while(*p == ';' || isdigit(*p)) /* skip digits */
	p++;
      if (*p++ == 'm') {/* yes, this was a colour code */
	q = p;          /* skip past last 'm' of code  */
      } else {
	p = r;          /* reset p to remembered value */
      }
    }
    
    *q = *p; /* copy one byte from buffer -> scratch */
  }
  *q = '\0';
  DPRINTF2(DEBUG_TERMIO,"buffer unbleached: <%s>, bleached: <%s>",
	   mangle_string_for_debug_log(buffer, 60), mangle_string_for_debug_log(scratch,60));
  strncpy(buffer, scratch, BUFFSIZE);
  free(scratch);
}	


      
static char *extract_separator(char *format) { /* find first few non-space characters in format */
  static char *separator = NULL, *p;
  if(! separator) {
    separator = mysavestring(format);
    for (p = separator; *p ; p++) {
      if(*p == ' ') {
	*p = '\0';
	break;
      }
    }
    assert (*separator);   
  }	
  return separator; 
}


static char *trim_from_last_separator(char *line, char *separator) {
  char *savedline = mysavestring(line),
       *chopoff= savedline,
       *shorter, *retval;
  
  while ((strlen(separator) < strlen(chopoff)) &&
	 (shorter = mystrstr(chopoff + strlen(separator), separator)))
    chopoff = shorter;

  if(chopoff != savedline)
    *chopoff = '\0';
  retval = mysavestring(savedline); 

  free(savedline);
  return retval;
}


static void replace_in(char **scrap, char *item, char*replacement) {
  char *retval = search_and_replace(item, replacement, *scrap, 0, NULL, NULL);
  free(*scrap);
  *scrap = retval;
}

char *append_and_expand_history_format(char *line_with_possible_appendage) {
  char *scrap , *p, *strftime_format, *chopoff, *shorter, *line, *result;
  struct timeval now;
  struct tm *today;
  
  
  scrap = mysavestring(history_format);
  
  change_working_directory(); /* actualise slaves_working_directory */
  replace_in (&scrap, "%D", slaves_working_directory);
  replace_in (&scrap, getenv("HOME"), "~");
  replace_in (&scrap, "%P", saved_rl_state.prompt);
  replace_in (&scrap, "%C", command_name);
  
  gettimeofday(&now, NULL);
  today = localtime(& now.tv_sec);

  strftime_format = mysavestring(scrap);  
  replace_in(&scrap, "%", "Wed Nov  8 16:18:49 CET 2006      "); /* should be big enough for %+ */
  strftime(scrap, strlen(scrap), strftime_format, today);  
  free(strftime_format);

  line  = trim_from_last_separator(line_with_possible_appendage,
				   extract_separator(history_format));
  assert(strlen(line) > 0);  
  result = add3strings(line, (line[strlen(line) - 1] == ' ' ? "" : " "), scrap);
  free(scrap);
  
  return result;		
} 	
	
  

/* print info about option, considering whether we HAVE_GETOPT_LONG and whether GETOPT_GROKS_OPTIONAL_ARGS */
static void print_option(char shortopt, char *longopt, char*argument, int optional, char *comment) {
  int long_opts, optional_args;
  char *format;
  char *maybe_optional = "";
  char *longoptional = "";

  
#ifdef HAVE_GETOPT_LONG
  long_opts = TRUE;
#else
  long_opts = FALSE;
#endif

#ifdef GETOPT_GROKS_OPTIONAL_ARGS
  optional_args = TRUE;
#else
  optional_args = FALSE;
#endif

  if (argument) {
    maybe_optional = (optional_args && optional ? add3strings("[", argument,"]") :  add3strings(" <", argument,">"));
    longoptional = (optional ? add3strings("[=", argument,"]") : add3strings("=<", argument, ">"));
  } 	

  format = add2strings ("  -%c%-24.24s", (long_opts ? " --%s%s" : ""));
  fprintf(stderr, format, shortopt, maybe_optional, longopt, longoptional);
  if (comment)
    fprintf(stderr, " %s", comment);
  fprintf(stderr, "\n");
  /* don't free allocated strings: we'll exit() soon */
}	



void
usage(int status)
{
  fprintf(stderr, "Usage: %s [options] command ...\n"
	          "\n"
	          "Options:\n", program_name);

  print_option('a', "always-readline", "password:", TRUE, NULL);
  print_option('b', "break_chars", "chars", FALSE, NULL);
  print_option('c', "complete-filenames", NULL, FALSE, NULL);
  print_option('C', "command-name", "name|N", FALSE, NULL);
  print_option('D', "history-no-dupes", "0|1|2", FALSE, NULL);
  print_option('f', "file", "completion list", FALSE,NULL);
  print_option('F', "history-format", "format string", FALSE,NULL);
  print_option('h', "help", NULL, FALSE, NULL);
  print_option('H', "history-filename", "file", FALSE, NULL);
  print_option('i', "case-insensitive", NULL, FALSE, NULL);
  print_option('l', "logfile", "file", FALSE, NULL);
  print_option('n', "no-warnings", NULL, FALSE, NULL);
  print_option('P', "pre-given","input", FALSE, NULL);
  print_option('m', "multi-line", "newline substitute", TRUE, NULL);
  print_option('r', "remember", NULL, FALSE, NULL);
  print_option('v', "version", NULL, FALSE, NULL);
  print_option('s', "histsize", "N", FALSE,"(negative: readonly)");	
  
#ifdef DEBUG
  fprintf(stderr, "\n");
  print_option('t', "test-terminal", NULL, FALSE, NULL);
  print_option('d', "debug", "mask", TRUE, add3strings("(output sent to ", DEBUG_FILENAME,")"));
  fprintf(stderr,
	  "             \n"
	  "The debugging mask is a bit mask obtaining by adding:\n"
	  "    %3d    if you want to debug termio\n"
	  "    %3d    if you want to debug signals\n"
	  "    %3d    if you want to debug readline handling\n"
	  "    omitting the mask means: debug everything \n", DEBUG_TERMIO,
	  DEBUG_SIGNALS, DEBUG_READLINE);
#endif

  fprintf(stderr,
	  "\n"
	  "bug reports, suggestions, updates:\n"
	  "http://utopia.knoware.nl/~hlub/uck/rlwrap/\n");

  exit(status);
}	    
  

  
