//FileIO.cpp, Copyright (c) 2001, 2002, 2003, 2004, 2005 R.Lackner
//read/write graphic objects
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <ctype.h>
#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags

#ifdef _WINDOWS
	#include <io.h>					//for read/write
#else
	#define O_BINARY 0x0
	#include <unistd.h>
#endif

extern GraphObj *CurrGO;			//Selected Graphic Objects
extern Default defs;
extern int dlgtxtheight;
extern char TmpTxt[];
extern int cPlots;

static notary *Notary = 0L;
static ReadCache *Cache = 0L;

unsigned long cObsW;				//count objects written

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// graphic input/output is driven by tables based on the descIO template
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
typedef struct {
	char *label;
	unsigned short type;
	void *ptr;
	long *count;
	}descIO;

//the type member of descIO describes the following data types pointed to by ptr
enum {typNONE, typNZINT, typINT, typLFLOAT, typNZLFLOAT, 
	typDWORD, typULONG, typFRECT, typNZLFPOINT, typLFPOINT, typPOINT3D,
	typAXDEF, typPTRAXDEF, typLINEDEF, typFILLDEF, typGOBJ,	typOBJLST,
	typFPLST, typFPLST3D, typIPLST, typTEXT, typTXTDEF, typPTRTXTDEF, 
	typLAST = 0x100};

static char *ptr =0L;
static long ptr_step = 2048;
static int cbOut, sizeOut = 0, iFile = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// output graph to file, elementary functions
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
int OpenOutputFile(char *file)
{
	time_t ti;

	if(ptr) free(ptr);
	ptr = 0L;	cbOut = 0;
	if(file && BackupFile(file)) {
		if(-1 ==(iFile = open(file, O_RDWR | O_BINARY | O_CREAT | O_TRUNC,
			S_IWRITE | S_IREAD))) return -1;
		ptr = (char *)malloc(sizeOut = 2048);
		if(!ptr) goto openerr;
		ti = time(0L);
		cbOut = sprintf(ptr,";RLP 1.0\n;File \"%s\" created by RLPlot version "
			""SZ_VERSION" for %s \n;Date/Time: %s ",
			file,  
#ifdef _WINDOWS
			"Windows",
#else
			"Qt",
#endif
			ctime(&ti));
		}
	return iFile;
openerr:
	ptr = 0L;	cbOut = sizeOut = 0;
	close(iFile);
	iFile = 0;
	return -1;
}

void CloseOutputFile()
{
	if(iFile > 0){
		if(cbOut) write(iFile, ptr, cbOut);
		close(iFile);
		}
	if(ptr) free(ptr);
	cbOut = sizeOut = 0;
	ptr = 0L;
}

void AddStringToOutput(char *str, int length)
{
	int i;
	char *tmp_ptr;

	if((cbOut+length)< sizeOut) {			//fit in buffer
		memcpy(ptr+cbOut, str, length);
		cbOut += length;
		}
	else if(iFile > 0){						//buffer full: write to file
		i = sizeOut-cbOut;
		memcpy(ptr+cbOut, str, i);
		write(iFile, ptr, 2048);
		memcpy(ptr, str+i, length-i);
		cbOut = length-i;
		}
	else {									//buffer full: resize buffer
		if(tmp_ptr = (char*)realloc(ptr, sizeOut+ptr_step)){
			ptr = tmp_ptr;		sizeOut += ptr_step;	ptr_step += (ptr_step >> 1);
			AddStringToOutput(str, length);
			}
		}
}

void WriteTypObjLst(GraphObj **ptr, int count)
{
	int i, j, c, n;
	char tmp[512];
	
	if(!ptr) return;
	for(i = c = 0; i <= count/16; i++) {
		if(i) c =  sprintf(tmp, "   ");
		for(j = 0; (n=i*16+j) <count && j < 16; j++) {
			c += sprintf(tmp+c, "%s%ld", j ? " " : "", Notary->RegisterGO(ptr[n]));
			}
		if(n >= count) c += sprintf(tmp+c, "}");
		c += sprintf(tmp+c, "\n");
		AddStringToOutput(tmp, c);
		}
}

void WriteTypIpLst(POINT *ptr, long count)
{
	long i, j, c, n;
	char tmp[256];

	if(!ptr) return;
	for(i = 0; i <= count/8; i++) {
		for(j = c = 0; (n =i*8+j) <count && j < 8; j++) {
			c += sprintf(tmp+c, " %ld", ptr[n].x);
			c += sprintf(tmp+c, " %ld", ptr[n].y);
			}
		if(n >= count) c += sprintf(tmp+c, "}");
		c += sprintf(tmp+c, "\n");
		AddStringToOutput(tmp, c);
		}
}

void WriteTypFpLst(lfPOINT *ptr, long count)
{
	long i, j, c, n;
	char tmp[256];

	if(!ptr) return;
	for(i = 0; i <= count/8; i++) {
		for(j = c = 0; (n =i*8+j) <count && j < 8; j++) {
			c += WriteFloatToBuff(tmp+c, ptr[n].fx);
			c += WriteFloatToBuff(tmp+c, ptr[n].fy);
			}
		if(n >= count) c += sprintf(tmp+c, "}");
		c += sprintf(tmp+c, "\n");
		AddStringToOutput(tmp, c);
		}
}

void WriteTypFpLst3D(fPOINT3D *ptr, long count)
{
	long i, j, c, n;
	char tmp[256];

	if(!ptr) return;
	for(i = 0; i <= count/5; i++) {
		for(j = c = 0; (n =i*5+j) <count && j < 5; j++) {
			c += WriteFloatToBuff(tmp+c, ptr[n].fx);
			c += WriteFloatToBuff(tmp+c, ptr[n].fy);
			c += WriteFloatToBuff(tmp+c, ptr[n].fz);
			}
		if(n >= count && j) c += sprintf(tmp+c, "}\n");
		else if(n < count) c += sprintf(tmp+c, "\n");
		AddStringToOutput(tmp, c);
		}
}

static char * esc_str = 0L;
static int esc_str_size = 0;
int WriteEscString(char *txt, char *buff, int cb)
{
	int i, j, l, lim = 60;

	if(!txt || !txt[0]) return 0;
	l = strlen(txt);
	if((l+10) > esc_str_size) esc_str = (char*)realloc(esc_str, esc_str_size = (l+10));
	j = 0;	esc_str[j++] = '"';
	for(i = 0; txt[i]; i++) {
		switch(txt[i]) {
		case '\\':	j += sprintf(esc_str+j, "\\\\");	break;
		case '\n':	j += sprintf(esc_str+j, "\\n");		break;
		default:	esc_str[j++] = txt[i];
			}
		if(j > esc_str_size -2) esc_str = (char*)realloc(esc_str, (esc_str_size += 20));
		if(j > lim) {
			j += sprintf(esc_str+j, "\"\n   \"");
			lim += 60;
			}
		}
	j += sprintf(esc_str+j, "\"\n");
	return sprintf(buff+cb, "%s", esc_str);
}

bool ExecOutput(long id, char *Class, descIO *Desc)
{
	static char buff[800];
	int i, cb, last;
	fRECT *fr;
	AxisDEF *ax;
	LineDEF *ld;
	FillDEF *fd;
	TextDEF *tx;

	cb = sprintf(buff, "\n[%ld=%s]\n", id, Class);
	cObsW++;
	for(i = 0; Desc[i].label; i++) {
		last = cb;
		cb += sprintf(buff+cb,"%s=", Desc[i].label);
		switch(Desc[i].type & 0xff){
		case typNZINT:
			if(!(*(int*)Desc[i].ptr)) {
				cb = last;
				break;
				}
			//if not zero value continue as if int
		case typINT:
			cb += sprintf(buff+cb," %d\n", *(int*)Desc[i].ptr);
			break;
		case typNZLFLOAT:
			if(fabs(*((double*)Desc[i].ptr)) < 1.0e-7) {
				cb = last;
				break;
				}
			//if not zero or negative continue as if float
		case typLFLOAT:
			cb += WriteFloatToBuff(buff+cb, *(double*)Desc[i].ptr);
			cb += sprintf(buff+cb, "\n");
			break;
		case typDWORD:
			cb += sprintf(buff+cb," 0x%08x\n", *(DWORD *)Desc[i].ptr);
			break;
		case typULONG:
			cb += sprintf(buff+cb," %ld\n", *(unsigned long*)Desc[i].ptr);
			break;
		case typFRECT:
			fr = (fRECT*)Desc[i].ptr;
			cb += WriteFloatToBuff(buff+cb, fr->Xmin);
			cb += WriteFloatToBuff(buff+cb, fr->Ymax);
			cb += WriteFloatToBuff(buff+cb, fr->Xmax);
			cb += WriteFloatToBuff(buff+cb, fr->Ymin);
			cb += sprintf(buff+cb, "\n");
			break;
		case typNZLFPOINT:
			if(((lfPOINT *)Desc[i].ptr)->fx == ((lfPOINT *)Desc[i].ptr)->fy &&
				((lfPOINT *)Desc[i].ptr)->fx == 0.0f){
				cb = last;
				break;
				}
			//if not zero continue as if fPOINT
		case typLFPOINT:
			cb += WriteFloatToBuff(buff+cb, ((lfPOINT *)Desc[i].ptr)->fx);
			cb += WriteFloatToBuff(buff+cb, ((lfPOINT *)Desc[i].ptr)->fy);
			cb += sprintf(buff+cb, "\n");
			break;
		case typPOINT3D:
			cb += WriteFloatToBuff(buff+cb, ((fPOINT3D *)Desc[i].ptr)->fx);
			cb += WriteFloatToBuff(buff+cb, ((fPOINT3D *)Desc[i].ptr)->fy);
			cb += WriteFloatToBuff(buff+cb, ((fPOINT3D *)Desc[i].ptr)->fz);
			cb += sprintf(buff+cb, "\n");
			break;
		case typAXDEF:
		case typPTRAXDEF:
			ax = (Desc[i].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[i].ptr : *(AxisDEF **)Desc[i].ptr;
			//we do not set ownership: reconstruct after read
			cb += sprintf(buff+cb, "0x%08x",  ax->flags);
			cb += WriteFloatToBuff(buff+cb, ax->min);
			cb += WriteFloatToBuff(buff+cb, ax->max);
			cb += WriteFloatToBuff(buff+cb, ax->loc[0].fx);
			cb += WriteFloatToBuff(buff+cb, ax->loc[0].fy);
			cb += WriteFloatToBuff(buff+cb, ax->loc[0].fz);
			cb += WriteFloatToBuff(buff+cb, ax->loc[1].fx);
			cb += WriteFloatToBuff(buff+cb, ax->loc[1].fy);
			cb += WriteFloatToBuff(buff+cb, ax->loc[1].fz);
			cb += WriteFloatToBuff(buff+cb, ax->Start);
			cb += WriteFloatToBuff(buff+cb, ax->Step);
			cb += WriteFloatToBuff(buff+cb, ax->Center.fx);
			cb += WriteFloatToBuff(buff+cb, ax->Center.fy);
			cb += WriteFloatToBuff(buff+cb, ax->Radius);
			cb += sprintf(buff+cb," %d", ax->nBreaks);
			if(ax->nBreaks) {
				cb += sprintf(buff+cb, " {");
				AddStringToOutput(buff, cb);
				cb = 0;
				WriteTypFpLst(ax->breaks, ax->nBreaks);
				}
			cb += sprintf(buff+cb, "\n");
			break;
		case typLINEDEF:
			ld = (LineDEF *)Desc[i].ptr;
			cb += WriteFloatToBuff(buff+cb, ld->width);
			cb += WriteFloatToBuff(buff+cb, ld->patlength);
			cb += sprintf(buff+cb, ld->color ? " 0x%08x" : " 0x0", ld->color);
			cb += sprintf(buff+cb, ld->pattern ? " 0x%08x\n" : " 0x0\n", ld->pattern);
			break;
		case typFILLDEF:
			fd = (FillDEF *)Desc[i].ptr;
			//we set the 'hatch' member to zero: reconstruct after read
			cb += sprintf(buff+cb," %d 0x%08x", fd->type, fd->color);
			cb += WriteFloatToBuff(buff+cb, fd->scale);
			cb += sprintf(buff+cb," 0x0 0x%08x\n", fd->color2);
			break;
		case typGOBJ:
			if(*(GraphObj **)(Desc[i].ptr)) cb += 
				sprintf(buff+cb," %ld\n", Notary->RegisterGO(*(GraphObj **)(Desc[i].ptr)));
			else cb = last;
			break;
		case typOBJLST:
			if(!*(GraphObj ***)(Desc[i].ptr)){
				cb = last;
				break;
				}
			cb += sprintf(buff+cb, "(%ld){", Desc[i].count ? *Desc[i].count : 0L);
			AddStringToOutput(buff, cb);
			cb = 0;
			WriteTypObjLst(*(GraphObj ***)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0L);
			break;
		case typIPLST:
			if(!*(POINT**)(Desc[i].ptr)){
				cb = last;
				break;
				}
			cb += sprintf(buff+cb, "(%ld){", Desc[i].count ? *Desc[i].count : 0L);
			AddStringToOutput(buff, cb);
			cb = 0;
			WriteTypIpLst(*(POINT**)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0L);
			break;
		case typFPLST:
			if(!*(lfPOINT**)(Desc[i].ptr)){
				cb = last;
				break;
				}
			cb += sprintf(buff+cb, "(%ld){", Desc[i].count ? *Desc[i].count : 0L);
			AddStringToOutput(buff, cb);
			cb = 0;
			WriteTypFpLst(*(lfPOINT**)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0L);
			break;
		case typFPLST3D:
			if(!*(fPOINT3D**)(Desc[i].ptr)){
				cb = last;
				break;
				}
			cb += sprintf(buff+cb, "(%ld){", Desc[i].count ? *Desc[i].count : 0L);
			AddStringToOutput(buff, cb);
			cb = 0;
			WriteTypFpLst3D(*(fPOINT3D**)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0L);
			break;
		case typTEXT:
			if(!*(char**)(Desc[i].ptr)) cb = last;
			else cb += WriteEscString(*(char**)(Desc[i].ptr), buff, cb);
			break;
		case typTXTDEF:
		case typPTRTXTDEF:
			tx = (Desc[i].type &0xff) == typTXTDEF ? (TextDEF *)Desc[i].ptr : *(TextDEF **)Desc[i].ptr;
			if(!tx) {
				cb = last;
				break;
				}
			cb += sprintf(buff+cb, " 0x%08x 0x%08x %g %g %g %d %d %d %d", 
				tx->ColTxt, tx->ColBg, tx->fSize, tx->RotBL, tx->RotCHAR,
				tx->Align, tx->Mode, tx->Style, tx->Font);
			if(tx->text) cb += sprintf(buff+cb, " \"%s\"\n", tx->text);
			else cb += sprintf(buff+cb, "\n");
			break;
			}
		if(Desc[i].type & typLAST) break;
		}
	AddStringToOutput(buff, cb);
	return true;
}

void ReadTypIpLst(POINT *ptr, long count, unsigned char *first)
{
	long f[20];
	int j, k;
	long i;

	if(!ptr || !first) return;
	k = sscanf((char*)first, "%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld%ld", 
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19]);
	for(i = 0,  j = 0; j < k && i < count; i++, j += 2) {
		ptr[i].x = f[j];	ptr[i].y = f[j+1];
		}
	while (i < count) {
		if(!Cache->GetInt(&ptr[i].x) || !Cache->GetInt(&ptr[i].y)) return;
		i++;
		}
}

void ReadTypFpLst(lfPOINT *ptr, long count, unsigned char *first)
{
	double f[20];
	int j, k;
	long i;

	if(!ptr || !first) return;
	k = sscanf((char*)first, "%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf", 
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19]);
	for(i = 0,  j = 0; j < k && i < count; i++, j += 2) {
		ptr[i].fx = f[j];	ptr[i].fy = f[j+1];
		}
	while (i < count) {
		if(!Cache->GetFloat(&ptr[i].fx) || !Cache->GetFloat(&ptr[i].fy)) return;
		i++;
		}
}

void ReadTypFpLst3D(fPOINT3D *ptr, long count, unsigned char *first)
{
	double f[21];
	int j, k;
	long i;

	if(!ptr || !first) return;
	k = sscanf((char*)first, "%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf",
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19], &f[20]);
	for(i = 0,  j = 0; j < k && i < count; i++, j += 3) {
		ptr[i].fx = f[j];	ptr[i].fy = f[j+1];		ptr[i].fz = f[j+2];
		}
	while (i < count) {
		if(!Cache->GetFloat(&ptr[i].fx) || !Cache->GetFloat(&ptr[i].fy) ||
			!Cache->GetFloat(&ptr[i].fz)) return;
		i++;
		}
}

void TranslateEscChar(char *txt)
{
	int i, j;

	if(txt && txt[0]) {
		for(i = j = 0; txt[i]; i++) {
			if(txt[i] == '\\') {
				switch(txt[i+1]) {
				case 'n':
					txt[j++] = 0x0a;	i++;	break;
				case 'r':
					txt[j++] = 0x0d;	i++;	break;
				case '"':	case 0x27:			case '\\':
					txt[j++] = txt[++i];		break;
				default:
					txt[j++] = txt[i];	break;
					}
				}
			else txt[j++] = txt[i];
			}
		txt[j] = 0;
		}
}

void AddLines(char **txt)
{
	char tmp[1000], *ntxt;
	bool mlines;
	int i, j, cb = strlen(*txt);

	do {
		mlines = false;
		Cache->ReadLine(tmp, sizeof(tmp));
		for(i = strlen(tmp); i > 0 &&(tmp[i-1] < 33 || (tmp[i-1] == 
			'"' && tmp[i-2] != '\\') ||	(tmp[i-1] == '\\' && 
			(mlines = true))); tmp[--i] = 0);
		for(i = 0; tmp[i] && (tmp[i] < 33 || tmp[i] == '"'); i++);
		TranslateEscChar(tmp);
		if(tmp[0] && (j = strlen(tmp+i)) && (ntxt = (char*)realloc(*txt, cb + j + 1))) {
			strcpy(ntxt+cb, tmp+i);
			cb += j;	*(txt) = ntxt;
			}
		} while (mlines);
}

bool ExecInput(descIO *Desc)
{
	char c, tmp[1000], tmp2[20];
	int i, j, k, l;
	bool match, mlines;
	unsigned long il, jl;
	AxisDEF *ax;
	POINT *lp;
	lfPOINT *lfp;
	fPOINT3D *lfp3d;
	LineDEF *ld;
	FillDEF *fd;
	fRECT *fr;
	GraphObj **gobs;
	TextDEF *tx;

	if(!Desc || !Desc[0].label) return false;
	for(j = k = 0; ; ) {
		do{
			c = Cache->Getc();
			switch (c) {
			case '[':					//next object
			case 0:						//probably eof
				return true;
			case '}':					//a lists hang over
				c = Cache->Getc();
				break;
				}
		} while(c <33);
		for(i = 1, tmp[0] = c; i < sizeof(tmp) && '=' != (tmp[i] = Cache->Getc()); i++){
			if(tmp[i] < 32 && tmp[i]) i = -1;			//some error conditions
			else if(!tmp[i] && Cache->eof) return true;
			else if(tmp[i] == '[') return true;
			}
		tmp[i] = 0;
		match = mlines = false;
		do {
			if(0 == strcmp(tmp, Desc[j].label)) {
				Cache->ReadLine(tmp, sizeof(tmp));
				switch(Desc[j].type & 0xff){
				case typNZINT:
				case typINT:
					sscanf(tmp, "%d", (int*)Desc[j].ptr);
					break;
				case typNZLFLOAT:
				case typLFLOAT:
					sscanf(tmp, "%lf", (double*)Desc[j].ptr);
					break;
				case typDWORD:
					sscanf(tmp, "%x", (DWORD*)Desc[j].ptr);
					break;
				case typULONG:
					sscanf(tmp, "%ld", (unsigned long*)Desc[j].ptr);
					break;
				case typFRECT:
					fr = (fRECT*) Desc[j].ptr;
					sscanf(tmp, "%lf%lf%lf%lf", &fr->Xmin, &fr->Ymax, &fr->Xmax, &fr->Ymin);
					break;
				case typNZLFPOINT:
				case typLFPOINT:
					lfp = (lfPOINT*) Desc[j].ptr;
					sscanf(tmp, "%lf%lf", &lfp->fx, &lfp->fy);
					break;
				case typPOINT3D:
					lfp3d = (fPOINT3D*) Desc[j].ptr;
					sscanf(tmp, "%lf%lf%lf", &lfp3d->fx, &lfp3d->fy, &lfp3d->fz);
					break;
				case typPTRAXDEF:
				case typAXDEF:
					ax = (Desc[j].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[j].ptr : *(AxisDEF **)Desc[j].ptr;
					//pointer for typPTRAXDEF and memory allocated by the Axis module!
					if(!ax) break;
					sscanf(tmp, "%x%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%d", &ax->flags, &ax->min, &ax->max,
						&ax->loc[0].fx,	&ax->loc[0].fy, &ax->loc[0].fz, &ax->loc[1].fx,
						&ax->loc[1].fy, &ax->loc[1].fz, &ax->Start, &ax->Step, &ax->Center.fx, 
						&ax->Center.fy, &ax->Radius, &ax->nBreaks);
					if(ax->nBreaks) {
						ax->breaks = (lfPOINT*)calloc(ax->nBreaks, sizeof(lfPOINT));
						for(i = 0; tmp[i] && tmp[i-1] != '{'; i++);
						if(tmp[i]) {
							ReadTypFpLst(ax->breaks, ax->nBreaks, (unsigned char*)tmp+i);
							SortAxisBreaks(ax);
							}
						}
					break;
				case typLINEDEF:
					ld = (LineDEF*) Desc[j].ptr;
					sscanf(tmp,"%lf%lf%x%x", &ld->width, &ld->patlength, &ld->color, &ld->pattern);
					break;
				case typFILLDEF:
					fd = (FillDEF*) Desc[j].ptr;
					sscanf(tmp, "%d%x%lf%x%x", &fd->type, &fd->color, &fd->scale, &fd->hatch, &fd->color2);
					fd->hatch = 0L;
					break;
				case typGOBJ:
					*(GraphObj**)(Desc[j].ptr) = Notary->PopGO(atol(tmp));
					break;
				case typOBJLST:
					for(i = 0; i < 10 && tmp[i] != '(' && tmp[i]; i++);
					if(!tmp[i]) break;
					if(sscanf(tmp+i+1, "%ld", &il) && il) {
						*Desc[j].count = il;
						if(!*(GraphObj***)(Desc[j].ptr)){
							*(GraphObj***)(Desc[j].ptr) = (GraphObj**)calloc(il, sizeof(GraphObj*));
							}
						if((gobs = *(GraphObj***)(Desc[j].ptr))){
							i += 4;
							while(tmp[i-1] != '{') i++; while(tmp[i-1] < 33) i++;
							strcat(tmp, " ");
							for( ;il >0; il--) {
								for(l = 0; l < sizeof(tmp2); ){
									if(tmp[i]) c = tmp[i++];
									else if(!(c = Cache->Getc())) break;
									if(c >='0' && c <='9') tmp2[l++] = c;
									else {
										tmp2[l] = 0;
										if(l)break;
										}
									}
								sscanf(tmp2, "%ld", &jl);
								*gobs++ = Notary->PopGO(jl);
								if(c == '}') break;
								}
							}
						}	
					break;
				case typIPLST:
					for(i = 0; i < 10 && tmp[i] != '(' && tmp[i]; i++);
					if(!tmp[i]) break;
					if(sscanf(tmp+i+1, "%ld", &il) && il) {
						*Desc[j].count = il;
						if(!*(POINT**)(Desc[j].ptr)){
							*(POINT**)(Desc[j].ptr) = (POINT*)calloc(il, sizeof(POINT));
							}
						if(!(lp = *(POINT**)(Desc[j].ptr)))return false;
						while(tmp[i-1] != '{') i++; while(tmp[i-1] < 33) i++;
						ReadTypIpLst(lp, il, (unsigned char*)tmp+i);
						}
					break;
				case typFPLST:
					for(i = 0; i < 10 && tmp[i] != '(' && tmp[i]; i++);
					if(!tmp[i]) break;
					if(sscanf(tmp+i+1, "%ld", &il) && il) {
						*Desc[j].count = il;
						if(!*(lfPOINT**)(Desc[j].ptr)){
							*(lfPOINT**)(Desc[j].ptr) = (lfPOINT*)calloc(il, sizeof(lfPOINT));
							}
						if(!(lfp = *(lfPOINT**)(Desc[j].ptr)))return false;
						while(tmp[i-1] != '{') i++; while(tmp[i-1] < 33) i++;
						ReadTypFpLst(lfp, il, (unsigned char*)tmp+i);
						}
					break;
				case typFPLST3D:
					for(i = 0; i < 10 && tmp[i] != '(' && tmp[i]; i++);
					if(!tmp[i]) break;
					if(sscanf(tmp+i+1, "%ld", &il) && il) {
						*Desc[j].count = il;
						if(!*(fPOINT3D**)(Desc[j].ptr)){
							*(fPOINT3D**)(Desc[j].ptr) = (fPOINT3D*)calloc(il, sizeof(fPOINT3D));
							}
						if(!Desc[j].ptr)return false;
						while(tmp[i-1] != '{') i++; while(tmp[i-1] < 33) i++;
						ReadTypFpLst3D(*(fPOINT3D**)(Desc[j].ptr), il, (unsigned char*)tmp+i);
						}
					break;
				case typTEXT:
					for(i = strlen(tmp); i > 0 &&(tmp[i-1] < 32 || (tmp[i-1] == 
						'"' && tmp[i-2] != '\\') ||	(tmp[i-1] == '\\' && 
						(mlines = true))); tmp[--i] = 0);
					for(i = 0; tmp[i] && (tmp[i] < 33 || tmp[i] == '"'); i++);
					TranslateEscChar(tmp);
					if(tmp[0]){
						*(char**)(Desc[j].ptr) = strdup(tmp+i);
						if(mlines) AddLines((char**)(Desc[j].ptr));
						}
					break;
				case typPTRTXTDEF:
				case typTXTDEF:
					tx = (Desc[j].type & 0xff) == typTXTDEF ? (TextDEF *)Desc[j].ptr : *(TextDEF **)Desc[j].ptr;
					if(!tx) {
						if((Desc[j].type & 0xff) == typTXTDEF) break;	//prabably wrong usage of typTXTDEF instad of
																//    typPTRTXTDEF
						tx = *(TextDEF **)(Desc[j].ptr) = (TextDEF*)calloc(1, sizeof(TextDEF));
						if(!tx) return false;					//memory allocation error
						}
					sscanf(tmp, "%x%x%lf%lf%lf%d%d%d%d", &tx->ColTxt, &tx->ColBg, 
						&tx->fSize, &tx->RotBL, &tx->RotCHAR,
						&tx->Align, &tx->Mode, &tx->Style, &tx->Font);
					tx->iSize = 0;
					for(i = strlen(tmp); i >0 && tmp[i] != '"'; i--);
					if(i) {
						tmp[i] = 0;
						for(l = 0; l <i && tmp[l] != '"'; l++);
						if(i && tmp[l+1]) tx->text = strdup(tmp+l+1);
						}
					break;
					}
				match = true;			j++;	k++;
				if(!Desc[j].label || (Desc[j-1].type & typLAST)) 
					j = k = 0;	//rewind: items in file not sorted
				}
			else {
				j++;
				k++;
				if(!Desc[j].label || (Desc[j-1].type & typLAST)) {				//Error:
					if(k > j){						//  item not defined in Desc
						match = true;				//  read parameters,
						Cache->ReadLine(tmp, sizeof(tmp));	//   then continue
						}
					j= 0;
					}
				}
			}while(!match);
		}
}

bool SaveGraphAs(GraphObj *g)
{
	char *name = 0L;
	int i;
	bool bRet = true;

	if(Notary || !g) {
		ErrorBox("Output pending or\nno graph.");
		return false;
		}
	cObsW = 0;
	Notary = new notary();
	if(g->Id == GO_GRAPH || g->Id == GO_PAGE) {
		if(((Graph*)g)->filename) name = ((Graph*)g)->filename;
		}
	name = SaveGraphAsName(name);
	if (name && Notary) {
		iFile = OpenOutputFile(name);
		if(iFile >=0) {
			if(g && g->FileIO(FILE_WRITE)){
				if(g->Id == GO_GRAPH || g->Id == GO_PAGE) {
					g->Command(CMD_FILENAME, name, 0L);
					}
				for(i = strlen(name); i >=0 && name[i] != '/' && name[i] != '\\'; i--);
				if(name[i]) i++;
				g->Command(CMD_SETNAME, name+i, 0L);
				g->Command(CMD_UPDHISTORY, 0L, 0L);
				}
			else ErrorBox("Could not write\ndata to file.");
			}
		else ErrorBox("Open failed for\noutput file.");
		CloseOutputFile();
		}
	else bRet = false;
	if(Notary) delete Notary;	Notary = 0L;
	return bRet;
}

char *GraphToMem(GraphObj *g, long *size)
{
	static char *ret;

	if(Notary || !g) {
		ErrorBox("Output pending or\nno graph.");
		return false;
		}
	cObsW = 0;
	iFile = cbOut = sizeOut = 0;	ptr = 0L;	ptr_step = 2048;
	if (Notary = new notary()) {
		if(g && g->FileIO(FILE_WRITE)){
			//all done
			}
		delete Notary;					Notary = 0L;
		if(ptr) ptr[cbOut] = 0;
		ret = ptr;						if(size) *size = cbOut;
		iFile = cbOut = sizeOut = 0;	ptr = 0L;	ptr_step = 2048;
		return ret;
		}
	return 0L;
}

void UpdGOfromMem(GraphObj *go, unsigned char *buff)
{
	int i=0;

	if(!go || !buff) return;
	iFile = cbOut = sizeOut = 0;	ptr = 0L;
	for(i = 0; buff[i] && buff[i] != ']'; i++);
	if(!buff[i])return;
	for(; buff[i] && buff[i] <33; i++);
	if(!buff[i] || i < 4) return;
	if(!(Cache = new MemCache(buff+i-1))) return;
	if ((Notary = new notary()) && go->Id > GO_UNKNOWN && go->Id < GO_DEFRW) {
		//notary not needed but saver if tree exists
		go->Command(CMD_FLUSH, 0L, 0L);
		go->FileIO(INIT_VARS);			go->FileIO(FILE_READ);
		delete Notary;					Notary = 0L;
		}
	delete Cache;		Cache = 0L;
}

bool OpenGraph(GraphObj *root, char *name, unsigned char *mem)
{
	unsigned char c, tmp[80];
	char debug[80];
	unsigned long id, lid;
	int i;
	unsigned int hv;
	GraphObj *go;

	if(Notary || Cache) {
		ErrorBox("Output pending:\nRead Error.");
		return false;
		}
	if(!(Notary = new notary()))return false;
	if(mem) {
		if(!(Cache = new MemCache(mem))) return false;
		}
	else if(Cache = new ReadCache()){
		if(!Cache->Open(name)) {
			delete Notary;		delete Cache;
			Notary = 0L;		Cache = 0L;
			ErrorBox("Error open file");
			return false;
			}
		}
	else return false;
	//DEBUG: skipping header
	do {
		c = Cache->Getc();
		} while(c && c != '[');
	if(!c) goto ReadErr;
	do {
		for(i = 0; i < sizeof(tmp) && c != '=' && c; i++){
			tmp[i] = c = Cache->Getc();
			if(c == '[') i = -1;
			}
		if(!c) goto ReadErr;
		tmp[i] = tmp[i-1] = 0;			id=0;
		sscanf((char*)tmp, "%ld", &id);
		if(!id) goto ReadErr;
		//go to class name
		while((tmp[0] = Cache->Getc())<31 && tmp[0]);
		if(!tmp[0]) goto ReadErr;
		for(i = 1; i < sizeof(tmp) && c!= ']' && c; i++)
			tmp[i] = c = Cache->Getc();
		if(!c) goto ReadErr;
		tmp[i-1] = 0;
		go = 0L;
		hv = HashValue(tmp);
		switch(hv) {
		case 3895:		go = new Axis(FILE_READ);			break;
		case 7496002:	go = new Bar(FILE_READ);			break;
		case 81384:		go = new Symbol(FILE_READ);			break;
		case 62229:		go = new Bubble(FILE_READ);			break;
		case 7892802:	go = new Box(FILE_READ);			break;
		case 15411:		go = new Arrow(FILE_READ);			break;
		case 1052406:	go = new ErrorBar(FILE_READ);		break;
		case 324566:	go = new Whisker(FILE_READ);		break;
		case 1031437:	go = new DropLine(FILE_READ);		break;
		case 4839:		go = new Tick(FILE_READ);			break;
		case 16832:		go = new Label(FILE_READ);			break;
		case 1071373:	go = new GridLine(FILE_READ);		break;
		case 963085:	go = new DataLine(FILE_READ);		break;
		case 61662266:	go = new DataPolygon(FILE_READ);	break;
		case 435228:	go = new segment(FILE_READ);		break;
		case 1741325:	go = new polyline(FILE_READ);		break;
		case 435258:	go = new polygon(FILE_READ);		break;
		case 6888037:	go = new rectangle(FILE_READ);		break;
		case 1780087:	go = new roundrec(FILE_READ);		break;
		case 78813:		go = new Sphere(FILE_READ);			break;
		case 15463:		go = new Brick(FILE_READ);			break;
		case 69952:		go = new Line3D(FILE_READ);			break;
		case 386257:	go = new ellipse(FILE_READ);		break;
		case 95680:		go = new mLabel(FILE_READ);			break;
		case 4819316:	go = new PlotScatt(FILE_READ);		break;
		case 117848:	go = new xyStat(FILE_READ);			break;
		case 15935312:	go = new BubblePlot(FILE_READ);		break;
		case 247376:	go = new BoxPlot(FILE_READ);		break;
		case 317384:	go = new StackBar(FILE_READ);		break;
		case 1205932:	go = new PieChart(FILE_READ);		break;
		case 16664:		go = new Graph(FILE_READ);			break;
		case 25108:		go = new GoGroup(FILE_READ);		break;
		case 300976:	go = new Scatt3D(FILE_READ);		break;
		case 297280:	go = new Plane3D(FILE_READ);		break;
		case 19227098:	go = new Regression(FILE_READ);		break;
		case 297997:	go = new RegLine(FILE_READ);		break;
		case 4318417:	go = new SDellipse(FILE_READ);		break;
		case 4843600:	go = new PolarPlot(FILE_READ);		break;
		case 977452:	go = new DensDisp(FILE_READ);		break;
		case 4465:		go = new Page(FILE_READ);			break;
		case 75120:		go = new Plot3D(FILE_READ);			break;
		case 17142080:	go = new GridLine3D(FILE_READ);		break;
		case 246688:	go = new Arrow3D(FILE_READ);		break;
		case 75562:		go = new Ribbon(FILE_READ);			break;
		case 16503104:	go = new DropLine3D(FILE_READ);		break;
		case 28859579:	go = new svgOptions(FILE_READ);		break;
		case 70259:		go = new Limits(FILE_READ);			break;
		case 17145824:	go = new GridRadial(FILE_READ);		break;
		case 1074714:	go = new Function(FILE_READ);		break;
		case 256075:	go = new FitFunc(FILE_READ);		break;
		case 273377:	go = new LegItem(FILE_READ);		break;
		case 1053744:	go = new FreqDist(FILE_READ);		break;
		case 68748:		go = new Legend(FILE_READ);			break;
		case 66800:		go = new Grid3D(FILE_READ);			break;
		case 967843:	go = new DefsRW(FILE_READ);			break;
		case 66848:		go = new Func3D(FILE_READ);			break;
		default:
			sprintf(debug, "Object %ld in file\n(Class = \"%s\")\nhash #%d\nis unknown.",
				id, tmp, hv);
			InfoBox(debug);
			}
		if(go) {
			if(((int)id) < 0) DeleteGO(go);		//temporary objects have id < 0
			else if(!Notary->PushGO(lid = id, go)) DeleteGO(go);
			}
		if('[' != Cache->Lastc()) do {			//search next object
			c = Cache->Getc();
			} while(c && c != '[');
		tmp[0] = 0;
		}while (c);
	Cache->Close();
	if((go = Notary->PopGO(lid))) {
		go->Command(CMD_SET_DATAOBJ, 0L, 0L);
		delete Notary;		Notary = 0L;
		if(root->Id == GO_PAGE) {
			if(go->Id == GO_PAGE){
				if(!(root->parent->Command(CMD_DROP_GRAPH,(void *)go, 0L))) DeleteGO(go);
				}
			else if(!(root->Command(CMD_DROP_GRAPH, (void *)go, 0L))) DeleteGO(go);
			}
		else if(!(root->Command(CMD_DROP_GRAPH, (void *)go, 0L))){
			DeleteGO(go);	go = 0L;
			}
		if(go) go->Command(CMD_FILENAME, name, 0L);
		}
	if(Notary) delete Notary;		Notary = 0L;
	delete Cache;					Cache = 0L;
	return true;

ReadErr:
	Cache->Close();
	close(iFile);
	delete Notary;
	Notary = 0L;
	delete Cache;
	Cache = 0L;
	if(!name || !defs.IniFile || strcmp(name, defs.IniFile))
		ErrorBox("An error occured during read.");
	return false;
}

bool InitVarsGO(descIO *Desc)
{
	int i;
	AxisDEF *ax;
	TextDEF *tx;

	for(i = 0; Desc[i].label; i++) {
		switch(Desc[i].type & 0xff) {
		case typNZINT:
		case typINT:
			*(int*)Desc[i].ptr = 0;			
			break;
		case typNZLFLOAT:
		case typLFLOAT:
			*(double*)Desc[i].ptr = 0.0;
			break;
		case typDWORD:
			*(DWORD*)Desc[i].ptr = 0x0L;
			break;
		case typULONG:
			*(unsigned long*)Desc[i].ptr = 0L;
			break;
		case typFRECT:
			((fRECT*)Desc[i].ptr)->Xmin = ((fRECT*)Desc[i].ptr)->Xmax =
				((fRECT*)Desc[i].ptr)->Ymin = ((fRECT*)Desc[i].ptr)->Ymax = 0.0;
			break;
		case typNZLFPOINT:
		case typLFPOINT:
			((lfPOINT*)Desc[i].ptr)->fx = ((lfPOINT*)Desc[i].ptr)->fy = 0.0;
			break;
		case typPOINT3D:
			((fPOINT3D*)Desc[i].ptr)->fx = ((fPOINT3D*)Desc[i].ptr)->fy =
				((fPOINT3D*)Desc[i].ptr)->fz = 0.0;
			break;
		case typPTRAXDEF:
		case typAXDEF:
			ax = (Desc[i].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[i].ptr : *(AxisDEF **)Desc[i].ptr;
			if(!ax) break;
			ax->owner = 0L;		ax->flags = 0L;		ax->breaks = 0L;	ax->nBreaks = 0;
			ax->min = ax->max = ax->Start = ax->Step = 0.0f;
			ax->loc[0].fx = ax->loc[0].fy = ax->loc[0].fz = 0.0f;
			ax->loc[1].fx = ax->loc[1].fy = ax->loc[1].fz = 0.0f;
			ax->Center.fx = ax->Center.fy = ax->Radius = 0.0f;
			break;
		case typLINEDEF:
			((LineDEF*)Desc[i].ptr)->width = defs.GetSize(SIZE_HAIRLINE);
			((LineDEF*)Desc[i].ptr)->patlength = defs.GetSize(SIZE_PATLENGTH);
			((LineDEF*)Desc[i].ptr)->color = ((LineDEF*)Desc[i].ptr)->pattern = 0x0L;
			break;
		case typFILLDEF:
			((FillDEF*)Desc[i].ptr)->type = FILL_NONE;
			((FillDEF*)Desc[i].ptr)->color = 0x00ffffffL;
			((FillDEF*)Desc[i].ptr)->scale = 1.0f;
			((FillDEF*)Desc[i].ptr)->hatch = (LineDEF*)0L;
			((FillDEF*)Desc[i].ptr)->color2 = 0x00ffffffL;
			break;
		case typGOBJ:
			*(GraphObj **)Desc[i].ptr = (GraphObj*)0L;
			break;
		case typOBJLST:
			*Desc[i].count = 0L;
			*(GraphObj ***)Desc[i].ptr = (GraphObj**)0L;
			break;
		case typIPLST:
			*Desc[i].count = 0L;
			*(POINT **)Desc[i].ptr = (POINT*)0L;
			break;
		case typFPLST:
			*Desc[i].count = 0L;
			*(lfPOINT **)Desc[i].ptr = (lfPOINT*)0L;
			break;
		case typFPLST3D:
			*Desc[i].count = 0L;
			*(fPOINT3D **)Desc[i].ptr = (fPOINT3D*)0L;
			break;
		case typTEXT:
			*(char **)Desc[i].ptr = (char*)0L;
			break;
		case typTXTDEF:
			tx = (TextDEF *)Desc[i].ptr;
			tx->ColTxt = 0x0L,			tx->ColBg = 0x00ffffffL;
			tx->fSize = defs.GetSize(SIZE_TEXT);
			tx->RotBL = tx->RotCHAR = 0.0;
			tx->Align = tx->Mode = tx->Style = tx->Font = 0L;
			tx->text = 0L;
			break;
		case typPTRTXTDEF:
			*(TextDEF**)Desc[i].ptr = (TextDEF*)0L;
			break;
			}
		if(Desc[i].type & typLAST) break;
		}
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Save object data to memory block
static unsigned char *SavVarBuf = 0L;
static long SavVarSize = 0,	SavVarPos = 0;
void SavVarInit(long len)
{
	if(SavVarBuf) free(SavVarBuf);
	SavVarBuf = (unsigned char *)malloc(SavVarSize = len+4096);
	SavVarPos = 0;
}

bool SavVarAdd(void *ptr, int size)
{
	int len;

	if(SavVarBuf) {
		len = sizeof(unsigned char *) + sizeof(int) + size;
		while (SavVarSize <= SavVarPos+len) {
			if(!(SavVarBuf = (unsigned char *)realloc(SavVarBuf, SavVarSize + 4096))) return false;
			SavVarSize += 4096;
			}
		memcpy(SavVarBuf+SavVarPos, &ptr, sizeof(unsigned char *));
		SavVarPos += sizeof(unsigned char *);
		memcpy(SavVarBuf+SavVarPos, &size, sizeof(int));	SavVarPos += sizeof(int);
		memcpy(SavVarBuf+SavVarPos, ptr, size);				SavVarPos += size;
		return true;
		}
	return false;
}

void *SavVarFetch()
{
	void *tmp;

	SavVarAdd(0L, 0);				
	tmp = SavVarBuf = (unsigned char *)realloc(SavVarBuf, SavVarPos);
	SavVarSize = SavVarPos = 0;		SavVarBuf = 0L;
	return tmp;
}

bool SaveVarGO(descIO *Desc)
{
	int i;
	AxisDEF *ax;
	TextDEF *tx;

	for(i = 0; Desc[i].label; i++) {
		switch(Desc[i].type & 0xff){
		case typNZINT:
		case typINT:
			SavVarAdd(Desc[i].ptr, sizeof(int));
			break;
		case typNZLFLOAT:
		case typLFLOAT:
			SavVarAdd(Desc[i].ptr, sizeof(double));
			break;
		case typDWORD:
			SavVarAdd(Desc[i].ptr, sizeof(DWORD));
			break;
		case typULONG:
			SavVarAdd(Desc[i].ptr, sizeof(unsigned long));
			break;
		case typFRECT:
			SavVarAdd(Desc[i].ptr, sizeof(fRECT));
			break;
		case typNZLFPOINT:
		case typLFPOINT:
			SavVarAdd(Desc[i].ptr, sizeof(lfPOINT));
			break;
		case typPOINT3D:
			SavVarAdd(Desc[i].ptr, sizeof(fPOINT3D));
			break;
		case typAXDEF:
		case typPTRAXDEF:
			ax = (Desc[i].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[i].ptr : *(AxisDEF **)Desc[i].ptr;
			if(ax) {
				SavVarAdd(ax, sizeof(AxisDEF));
				if(ax->breaks && ax->nBreaks) {
					SavVarAdd(ax->breaks, ax->nBreaks * sizeof(lfPOINT));
					}
				}
			break;
		case typLINEDEF:
			SavVarAdd(Desc[i].ptr, sizeof(LineDEF));
			break;
		case typFILLDEF:
			SavVarAdd(Desc[i].ptr, sizeof(FillDEF));
			break;
		case typGOBJ:	case typOBJLST:		//not supported
			break;
		case typIPLST:						//probably in the future
			break;
		case typFPLST:
			break;
		case typFPLST3D:
			SavVarAdd(*((void **)Desc[i].ptr), sizeof(fPOINT3D) * (*Desc[i].count));
			break;
		case typTEXT:
			if(*(char**)(Desc[i].ptr)) SavVarAdd(Desc[i].ptr, strlen(*(char**)(Desc[i].ptr))+1);
			break;
		case typTXTDEF:
		case typPTRTXTDEF:
			tx = (Desc[i].type &0xff) == typTXTDEF ? (TextDEF *)Desc[i].ptr : *(TextDEF **)Desc[i].ptr;
			if(tx) SavVarAdd(tx, sizeof(TextDEF) - sizeof(char*));
			break;
			}
		if(Desc[i].type & typLAST) break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Graphic object member funtions for IO
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
bool
svgOptions::FileIO(int rw)
{
	descIO Desc[] = {
		{"tagAttr", typTEXT, &svgattr, 0L},
		{"Script", typLAST | typTEXT, &script, 0L}};

	switch(rw) {
	case INIT_VARS:
		return InitVarsGO(Desc);
	case FILE_READ:
		return ExecInput(Desc);
		}
	return false;
}

bool
Symbol::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"Idx", typINT, &idx, 0L},
		{"Pos", typLFPOINT, &fPos, 0L},
		{"Size", typLFLOAT, &size, 0L},
		{"Line", typLINEDEF, &SymLine, 0L},
		{"FillCol", typDWORD, &SymFill.color, 0L},
		{"Text", typPTRTXTDEF, &SymTxt, 0L},
		{"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		size = parent ? parent->GetSize(SIZE_SYMBOL): defs.GetSize(SIZE_SYMBOL);
		SymLine.color = parent ? parent->GetColor(COL_SYM_LINE) : defs.Color(COL_SYM_LINE);
		SymLine.width = parent ? parent->GetSize(SIZE_SYM_LINE) : defs.GetSize(SIZE_SYM_LINE);
		SymFill.type = FILL_NONE;
		SymFill.color = parent ? parent->GetColor(COL_SYM_FILL) : defs.Color(COL_SYM_FILL);
		SymFill.scale = 1.0f;
		SymFill.hatch = (LineDEF *) 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		SymFill.hatch = (LineDEF *) 0L;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Symbol", Desc);
		}
	return false;
}


bool
Bubble::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"Pos", typLFPOINT, &fPos, 0L},
		{"Size", typLFLOAT, &fs, 0L},
		{"Line", typLINEDEF, &BubbleLine, 0L},
		{"FillLine", typLINEDEF, &BubbleFillLine, 0L},
		{"Fill", typFILLDEF, &BubbleFill, 0L},
		{"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		BubbleLine.width = defs.GetSize(SIZE_BUBBLE_LINE);
		BubbleFillLine.width = defs.GetSize(SIZE_BUBBLE_HATCH_LINE);
		BubbleLine.color = BubbleFillLine.color = defs.Color(COL_BUBBLE_LINE);
		BubbleFill.color = defs.Color(COL_BUBBLE_FILL);
		ssRef = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		BubbleFill.hatch = &BubbleFillLine;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Bubble", Desc);
		}
	return false;
}

bool
Bar::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"Pos", typLFPOINT, &fPos, 0L},
		{"Size", typLFLOAT, &size, 0L},
		{"Org", typLFPOINT, &BarBase, 0L},
		{"Line", typLINEDEF, &BarLine, 0L},
		{"Fill", typFILLDEF, &BarFill, 0L},
		{"FillLine", typLINEDEF, &HatchLine, 0L},
		{"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		type = BAR_VERTB;
		memcpy(&BarFill, defs.GetFill(), sizeof(FillDEF));
		if(BarFill.hatch) memcpy(&HatchLine, BarFill.hatch, sizeof(LineDEF));
		BarFill.hatch = &HatchLine;
		memcpy(&BarLine, defs.GetOutLine(), sizeof(LineDEF));
		size = parent ? parent->GetSize(SIZE_BAR): defs.GetSize(SIZE_BAR);
		mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		BarFill.hatch = &HatchLine;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Bar", Desc);
		}
	return false;
}

void
DataLine::FileValues(char *name, int type, double start, double step)
{
	FILE *file;
	int i, c;
	double fx;
	lfPOINT *tfp;

	if(!(file = fopen(name, "r"))) {
		sprintf(TmpTxt, "DataLine: open failed for file \"%s\"", name);
		ErrorBox(TmpTxt);
		return;
		}
	if(Values) free(Values);
	if(!(Values = (lfPOINT*)calloc( nPnt = 1000, sizeof(lfPOINT)))){
		fclose(file);
		return;
		}
	switch(type) {
	case 1:				//x and y values
		i = 0;
		do {
			c = fscanf(file, "%lf%lf", &Values[i].fx, &Values[i].fy);
			i++;
			if(i >= nPnt &&(tfp = (lfPOINT*)realloc(Values, (nPnt+1000)*sizeof(lfPOINT)))){
				Values = tfp;			nPnt += 1000;
				}
			else if(i >= nPnt) break;
			}while(c == 2);
		i--;
		break;
	case 2:				//only y values
		i = 0;	fx = start;
		do {
			c = fscanf(file, "%lf", &Values[i].fy);
			Values[i].fx = fx;
			i++;	fx += step;
			if(i >= nPnt &&(tfp = (lfPOINT*)realloc(Values, (nPnt+1000)*sizeof(lfPOINT)))){
				Values = tfp;			nPnt += 1000;
				}
			}while(c == 1);
		i--;
		break;
		}
	nPntSet = i-1;
	fclose(file);
}

bool
DataLine::FileIO(int rw)
{
	long i;
	char *file1 = 0L;
	char *file2 = 0L;
	double Start = 0.0f, Step = 0.0f;
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssXref", typTEXT, &ssXref, 0L},
		{"ssYref", typTEXT, &ssYref, 0L},
		{"BgCol", typDWORD, &BgColor, 0L},
		{"Line", typLINEDEF, &LineDef, 0L},
		{"Data", typFPLST, &Values, &i},
		{"file_xy", typTEXT, &file2, 0L},
		{"start_x", typNZLFLOAT, &Start, 0L},
		{"step_x", typNZLFLOAT, &Step, 0L},
		{"file_y", typLAST | typTEXT, &file1, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		isPolygon = false;
		nPnt = nPntSet = cp = 0;
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		BgColor = defs.Color(COL_BG);
		pts = 0L;		dirty = true;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = max.fx = max.fy = 0.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		nPnt = i;
		nPntSet = i-1;
		if(file2)FileValues(file2, 1, 0.0, 1.0);
		else if(file1)FileValues(file1, 2, Start, fabs(Step) > defs.min4log ? Step : 1.0);
		if(file1) free(file1);
		if(file2) free(file2);
		if(i && Values)return true;
		break;
	case FILE_WRITE:
		i = nPntSet+1;
		return ExecOutput(Notary->RegisterGO(this), "DataLine", Desc);
		}
	return false;
}

bool
DataPolygon::FileIO(int rw)
{
	long i;
	char *file1 = 0L;
	char *file2 = 0L;
	double Start = 0.0f, Step = 0.0f;
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssXref", typTEXT, &ssXref, 0L},
		{"ssYref", typTEXT, &ssYref, 0L},
		{"BgCol", typDWORD, &BgColor, 0L},
		{"Line", typLINEDEF, &LineDef, 0L},
		{"FillLine", typLINEDEF, &pgFillLine, 0L},
		{"Fill", typFILLDEF, &pgFill, 0L},
		{"Data", typFPLST, &Values, &i},
		{"file_xy", typTEXT, &file2, 0L},
		{"start_x", typNZLFLOAT, &Start, 0L},
		{"step_x", typNZLFLOAT, &Step, 0L},
		{"file_y", typLAST | typTEXT, &file1, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		isPolygon = true;
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		LineDef.pattern = 0L;
		BgColor = defs.Color(COL_BG);
		memcpy(&pgFill, defs.GetFill(), sizeof(FillDEF));
		if(pgFill.hatch) memcpy(&pgFillLine, pgFill.hatch, sizeof(LineDEF));
		pgFill.hatch = &pgFillLine;		dirty = true;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = max.fx = max.fy = 0.0f;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		nPnt = i;
		nPntSet = i-1;
		if(file2)FileValues(file2, 1, 0.0f, 1.0f);
		else if(file1)FileValues(file1, 2, Start, fabs(Step) > defs.min4log ? Step : 1.0);
		if(file1) free(file1);
		if(file2) free(file2);
		pgFill.hatch = &pgFillLine;
		if(i && Values)return true;
		break;
	case FILE_WRITE:
		i = nPntSet+1;
		return ExecOutput(Notary->RegisterGO(this), "DataPolygon", Desc);
		}
	return false;
}

bool
RegLine::FileIO(int rw)
{
	descIO Desc[] = {
		{"type", typNZINT, &type, 0L},
		{"nPoints", typINT, &nPoints, 0L},
		{"BgCol", typDWORD, &BgColor, 0L},
		{"Line", typLINEDEF, &LineDef, 0L},
		{"Range", typFRECT, &lim, 0L},
		{"uClip", typFRECT, &uclip, 0L},
		{"mx", typNZLFLOAT, &mx, 0L},
		{"my", typNZLFLOAT, &my, 0L},
		{"li1", typLFPOINT, &l1, 0L},
		{"li2", typLFPOINT, &l2, 0L},
		{"li3", typLFPOINT, &l3, 0L},
		{"li4", typLFPOINT, &l4, 0L},
		{"li5", typLAST | typLFPOINT, &l5, 0L}};

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		cp = 0;
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		BgColor = defs.Color(COL_BG);
		pts = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "RegLine", Desc);
		}
	return false;
}

void
SDellipse::RegGO(void *n)
{
	if(n) {
		if(rl)rl->RegGO(n);
		((notary*)n)->AddRegGO(this);
	}
}

bool
SDellipse::FileIO(int rw)
{
	descIO Desc[] = {
		{"type", typNZINT, &type, 0L},
		{"Line", typLINEDEF, &LineDef, 0L},
		{"Range", typFRECT, &lim, 0L},
		{"Regr", typGOBJ, &rl, 0L},
		{"Data", typLAST | typFPLST, &val, &nPoints}};

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&LineDef, defs.GetOutLine(), sizeof(LineDEF));
		pts = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(rl) rl->parent = this;
		return true;
	case FILE_WRITE:
		if(rl) rl->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "SDellipse", Desc);
		}
	return false;
}

bool
ErrorBar::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"Pos", typLFPOINT, &fPos, 0L},
		{"Err", typLFLOAT, &ferr, 0L},
		{"Size", typLFLOAT, &SizeBar, 0L},
		{"Line", typLAST | typLINEDEF, &ErrLine, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		SizeBar = defs.GetSize(SIZE_ERRBAR);
		ErrLine.width = defs.GetSize(SIZE_ERRBAR_LINE);
		ErrLine.color = defs.Color(COL_SYM_LINE);
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "ErrorBar", Desc);
		}
	return false;
}

bool
Arrow::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"moveable", typNZINT, &moveable, 0L},
		{"p1", typLFPOINT, &pos1, 0L},
		{"p2", typLFPOINT, &pos2, 0L},
		{"CapW", typLFLOAT, &cw, 0L},
		{"CapL", typLFLOAT, &cl, 0L},
		{"Line", typLAST | typLINEDEF, &LineDef, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cw = defs.GetSize(SIZE_ARROW_CAPWIDTH);
		cl = defs.GetSize(SIZE_ARROW_CAPLENGTH);
		LineDef.color = parent ? parent->GetColor(COL_DATA_LINE) : defs.Color(COL_DATA_LINE);
		LineDef.width = parent ? parent->GetSize(SIZE_ARROW_LINE) : defs.GetSize(SIZE_ARROW_LINE);
		type = ARROW_LINE;
		dh1 = dh2 = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Arrow", Desc);
		}
	return false;
}

bool
Box::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"High", typLFPOINT, &pos1, 0L},
		{"Low", typLFPOINT, &pos2, 0L},
		{"Size", typLFLOAT, &size, 0L},
		{"Line", typLINEDEF, &Outline, 0L},
		{"FillLine", typLINEDEF, &Hatchline, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&Outline, defs.GetOutLine(), sizeof(LineDEF));
		memcpy(&Fill, defs.GetFill(), sizeof(FillDEF));
		if(Fill.hatch)memcpy(&Hatchline, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &Hatchline;
		size = defs.GetSize(SIZE_BAR);
		ssRef = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Fill.hatch = &Hatchline;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Box", Desc);
		}
	return false;
}

bool
Whisker::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"High", typLFPOINT, &pos1, 0L},
		{"Low", typLFPOINT, &pos2, 0L},
		{"Size", typLFLOAT, &size, 0L},
		{"Line", typLAST | typLINEDEF, &LineDef, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		size = defs.GetSize(SIZE_WHISKER);
		LineDef.width = defs.GetSize(SIZE_WHISKER_LINE);
		LineDef.color = defs.Color(COL_WHISKER);
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Whisker", Desc);
		}
	return false;
}

bool
DropLine::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typINT, &type, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"Pos", typLFPOINT, &fPos, 0L},
		{"Line", typLAST | typLINEDEF, &LineDef, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		LineDef.color = defs.Color(COL_SYM_LINE);
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "DropLine", Desc);
		}
	return false;
}

bool
Sphere::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"Pos", typPOINT3D, &fPos, 0L},
		{"Size", typLFLOAT, &size, 0L},
		{"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};
	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		size = defs.GetSize(SIZE_SYMBOL);
		Line.color = defs.Color(COL_SYM_LINE);
		Line.width = defs.GetSize(SIZE_SYM_LINE);
		Fill.color = defs.Color(COL_SYM_FILL);
		scl = 0L;		nscl = 0;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Sphere", Desc);
		}
	return false;
}

bool
Plane3D::FileIO(int rw)
{
	descIO Desc[] = {
		{"Line", typLINEDEF, &Line, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"values", typLAST | typFPLST3D, &dt, &ndt}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		ipl = 0L;	pts = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Plane3D", Desc);
		}
	return false;
}

bool
Brick::FileIO(int rw)
{
	descIO Desc[] = {
		{"Line", typLINEDEF, &Line, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"Pos", typPOINT3D, &fPos, 0L},
		{"depth", typLFLOAT, &depth, 0L},
		{"width", typLFLOAT, &width, 0L},
		{"height", typLFLOAT, &height, 0L},
		{"flags", typDWORD, &flags, 0L},
		{"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		Line.color = defs.Color(COL_SYM_LINE);
		Line.width = defs.GetSize(SIZE_SYM_LINE);
		Fill.color = defs.Color(COL_SYM_FILL);
		faces = (plane**)calloc(6, sizeof(plane*));
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		mo = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Brick", Desc);
		}
	return false;
}

bool
DropLine3D::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typINT, &type, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"Pos", typPOINT3D, &fPos, 0L},
		{"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		Line.color = defs.Color(COL_SYM_LINE);
		Line.width = defs.GetSize(SIZE_HAIRLINE);	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		ls[0] = ls[1] = ls[2] = ls[3] = ls[4] = ls[5] = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "DropLine3D", Desc);
		}
	return false;
}

bool
Arrow3D::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typINT, &type, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"Org", typPOINT3D, &fPos1, 0L},
		{"Pos", typPOINT3D, &fPos2, 0L},
		{"CapW", typLFLOAT, &cw, 0L},
		{"CapL", typLFLOAT, &cl, 0L},
		{"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cw = defs.GetSize(SIZE_ARROW_CAPWIDTH);
		cl = defs.GetSize(SIZE_ARROW_CAPLENGTH);
		Line.color = defs.Color(COL_ARROW);
		Line.width = defs.GetSize(SIZE_ARROW_LINE);
		ls[0] = ls[1] = ls[2] = 0L;
		cap = 0L;
		type = ARROW_LINE;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Arrow3D", Desc);
		}
	return false;
}

bool
Line3D::FileIO(int rw)
{
	descIO Desc[] = {
		{"Line", typLINEDEF, &Line, 0L},
		{"ssRefX", typTEXT, &x_range, 0L},
		{"ssRefY", typTEXT, &y_range, 0L},
		{"ssRefZ", typTEXT, &z_range, 0L},
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"values", typLAST | typFPLST3D, &values, &nPts}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		Line.color = defs.Color(COL_DATA_LINE);
		Line.width = defs.GetSize(SIZE_DATA_LINE);
		ls = 0L;	pts = 0L;	npts = 0L;	mo=0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = min.fz = max.fx = max.fy = max.fz = 0.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(nPts > 1) ls = (line_segment **)calloc(nPts-1, sizeof(line_segment*));
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Line3D", Desc);
		}
	return false;
}

bool
Label::FileIO(int rw)
{
	descIO Desc[] = {
		{"ssRef", typIPLST, &ssRef, &cssRef},
		{"moveable", typNZINT, &moveable, 0L},
		{"Pos", typNZLFPOINT, &fPos, 0L},
		{"Dist", typNZLFPOINT, &fDist, 0L},
		{"Flags", typDWORD, &flags, 0L},
		{"TxtDef", typLAST | typTXTDEF, &TextDef, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		TextDef.ColTxt = 0x0L;
		TextDef.ColBg = 0x00ffffffL;
		TextDef.fSize = defs.GetSize(SIZE_TEXT);
		TextDef.RotBL = TextDef.RotCHAR = 0.0;
		TextDef.iSize = 0;
		TextDef.Align = TXA_VTOP | TXA_HLEFT;
		TextDef.Mode = TXM_TRANSPARENT;
		TextDef.Style = TXS_NORMAL;
		TextDef.Font = FONT_HELVETICA;
		TextDef.text = 0L;	bgcolor = 0x00ffffffL;
		bgLine.width = 0.0;		bgLine.patlength = 6.0;
		bgLine.color = bgcolor;	bgLine.pattern = 0L;
		CursorPos = 0;	defDisp = 0L;	bBGvalid = bModified = false;
		curr_z = 0.0;		is3D = false;	fmt_txt = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Label", Desc);
		}
	return false;
}

void
mLabel::RegGO(void *n)
{
	int i;

	if(n) {
		if(Lines) for(i = 0; i < nLines; i++) if(Lines[i]) Lines[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
	}
}

bool
mLabel::FileIO(int rw)
{
	descIO Desc[] = {
		{"moveable", typNZINT, &moveable, 0L},
		{"Pos", typNZLFPOINT, &fPos, 0L},
		{"Dist", typNZLFPOINT, &fDist, 0L},
		{"lspc", typLFLOAT, &lspc, 0L},
		{"Flags", typDWORD, &flags, 0L},
		{"TxtDef", typTXTDEF, &TextDef, 0L},
		{"Lines", typLAST | typOBJLST, &Lines, &nLines}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		//The lines inherit settings from this object.
		//We need not save them in this context
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		TextDef.ColTxt = 0x0L;
		TextDef.ColBg = 0x00ffffffL;
		TextDef.fSize = defs.GetSize(SIZE_TEXT);
		TextDef.RotBL = TextDef.RotCHAR = 0.0;
		TextDef.iSize = 0;
		TextDef.Align = TXA_VTOP | TXA_HLEFT;
		TextDef.Mode = TXM_TRANSPARENT;
		TextDef.Style = TXS_NORMAL;
		TextDef.Font = FONT_HELVETICA;
		TextDef.text = 0L;
		undo_flags = 0L;	lspc = 1.0;
		curr_z = 0.0;		is3D = false;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(Lines) for ( i = 0; i < nLines; i++)
			if(Lines[i]) Lines[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(Lines) for ( i = 0; i < nLines; i++)
			if(Lines[i]) Lines[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "mLabel", Desc);
		}
	return false;
}

bool
segment::FileIO(int rw)
{
	descIO Desc[] = {
		{"moveable", typNZINT, &moveable, 0L},
		{"cent", typLFPOINT, &fCent, 0L},
		{"ri", typNZLFLOAT, &radius1, 0L},
		{"ra", typLFLOAT, &radius2, 0L},
		{"start", typLFLOAT, &angle1, 0L},
		{"end", typLFLOAT, &angle2, 0L},
		{"shout", typNZLFLOAT, &shift, 0L},
		{"Line", typLINEDEF, &segLine, 0L},
		{"FillLine", typLINEDEF, &segFillLine, 0L},
		{"Fill", typLAST | typFILLDEF, &segFill, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		segLine.width = defs.GetSize(SIZE_SEGLINE);
		pts = 0L;	nPts = 0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		segFill.hatch = &segFillLine;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "segment", Desc);
		}
	return false;
}

bool
polyline::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"moveable", typNZINT, &moveable, 0L},
		{"Data", typFPLST, &Values, &nPoints},
		{"Line", typLINEDEF, &pgLine, 0L},
		{"FillLine", typLINEDEF, &pgFillLine, 0L},
		{"Fill", typLAST | typFILLDEF, &pgFill, 0L}};
	
	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&pgLine, defs.plLineDEF(0L), sizeof(LineDEF));
		memcpy(&pgFill, defs.pgFillDEF(0L), sizeof(FillDEF));
		if(pgFill.hatch) memcpy(&pgFillLine, pgFill.hatch, sizeof(LineDEF));
		pgFill.hatch = &pgFillLine;
		pts = 0L;		nPts = 0;
		pHandles = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		pgFill.hatch = &pgFillLine;
		return true;
	case FILE_WRITE:
		if(type != 1) Desc[3].type |= typLAST;	//skip fill for polyline
		return ExecOutput(Notary->RegisterGO(this), 
			type == 1 ? (char*)"polygon" : (char*)"polyline", Desc);
		}
	return false;
}

bool
rectangle::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"moveable", typNZINT, &moveable, 0L},
		{"p1", typLFPOINT, &fp1, 0L},
		{"p2", typLFPOINT, &fp2, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"FillLine", typLINEDEF, &FillLine, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"Rad", typNZLFLOAT, &rad, 0L},
		{"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&Line, defs.pgLineDEF(0L), sizeof(LineDEF));
		memcpy(&Fill, defs.pgFillDEF(0L), sizeof(FillDEF));
		if(Fill.hatch) memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &FillLine;
		pts = 0L;	nPts = 0L;
		rad = defs.GetSize(SIZE_RRECT_RAD);
		drc = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Fill.hatch = &FillLine;
		return true;
	case FILE_WRITE:
		if(type != 2) rad = 0.0;
		ExecOutput(Notary->RegisterGO(this), 
			type == 1? (char*)"ellipse" : type == 2? (char*)"roundrec" :
			(char*)"rectangle", Desc);
		return true;
		}
	return false;
}

void
LegItem::RegGO(void *n)
{
	if(n) {
		if(Sym) Sym->RegGO(n);
		if(Desc) Desc->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
LegItem::FileIO(int rw)
{
	descIO Des[] = {
		{"D_Line", typLINEDEF, &DataLine, 0L},
		{"O_Line", typLINEDEF, &OutLine, 0L},
		{"H_Line", typLINEDEF, &HatchLine, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"Sym", typGOBJ, &Sym, 0L},
		{"Text", typGOBJ, &Desc, 0L},
		{"flags", typLAST | typDWORD, &flags, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Des);
	case INIT_VARS:
		InitVarsGO(Des);
		Fill.hatch = &HatchLine;
		return true;
	case FILE_READ:
		ExecInput(Des);
		Fill.hatch = &HatchLine;
		if(Sym) Sym->parent=this;
		if(Desc) Desc->parent=this;
		return true;
	case FILE_WRITE:
		if(Sym) Sym->FileIO(rw);
		if(Desc) Desc->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "LegItem", Des);
		}
	return false;
}

void
Legend::RegGO(void *n)
{
	int i;

	if(n) {
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Legend::FileIO(int rw)
{
	descIO Desc[] = {
		{"pos", typLFPOINT, &pos, 0L},
		{"rec1", typFRECT, &B_Rect, 0L},
		{"rec2", typFRECT, &D_Rect, 0L},
		{"rec3", typFRECT, &F_Rect, 0L},
		{"Items", typLAST | typOBJLST, &Items, &nItems}};
	int i;
	double d;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		B_Rect.Ymin = defs.GetSize(SIZE_DRECT_TOP);
		B_Rect.Xmin = defs.GetSize(SIZE_DRECT_LEFT);
		B_Rect.Xmax = B_Rect.Xmin + 1.5*(d = defs.GetSize(SIZE_BAR));
		B_Rect.Ymin += d*0.2;
		B_Rect.Ymax = B_Rect.Ymin + d/2.0;
		D_Rect.Ymin = 0.0;			D_Rect.Xmin = d*0.7;
		D_Rect.Xmax = d*1.3;		D_Rect.Ymax = d*0.4;
		F_Rect.Ymin = 0.0;			F_Rect.Xmin = d*0.2;
		F_Rect.Xmax = d*1.3;		F_Rect.Ymax = d*0.4;
		to = 0L;
		trc.left = trc.right = trc.top = trc.bottom = 0;
		if(!name) name=strdup("Legend");
		return true;
	case FILE_READ:
		nItems = 0L;
		ExecInput(Desc);
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->parent=this;
		return true;
	case FILE_WRITE:
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Legend", Desc);
		}
	return false;
}

void
PlotScatt::RegGO(void *n)
{
	int i;

	if(n) {
		if(TheLine) TheLine->RegGO(n);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		if(Errors) for(i = 0; i < nPoints; i++) if(Errors[i]) Errors[i]->RegGO(n);
		if(Arrows) for(i = 0; i < nPoints; i++) if(Arrows[i]) Arrows[i]->RegGO(n);
		if(DropLines) for(i = 0; i < nPoints; i++) if(DropLines[i]) DropLines[i]->RegGO(n);
		if(Labels) for(i = 0; i < nPoints; i++) if(Labels[i]) Labels[i]->RegGO(n);
		if(Bars) for(i = 0; i < nPoints; i++) if(Bars[i]) Bars[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
PlotScatt::FileIO(int rw)
{
	descIO Desc[] = {
		{"hide", typNZINT, &hidden, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"DefSym", typNZINT, &DefSym, 0L},
		{"baDist", typLFPOINT, &BarDist, 0L},
		{"xRange", typTEXT, &xRange, 0L},
		{"yRange", typTEXT, &yRange, 0L},
		{"eRange", typTEXT, &ErrRange, 0L},
		{"lRange", typTEXT, &LbRange, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"Bars", typOBJLST, &Bars, &nPoints},
		{"Symbols", typOBJLST, &Symbols, &nPoints},
		{"PL", typGOBJ, &TheLine, 0L},
		{"ErrBars", typOBJLST, &Errors, &nPoints},
		{"Arrows", typOBJLST, &Arrows, &nPoints},
		{"dLines", typOBJLST, &DropLines, &nPoints},
		{"Labels", typLAST | typOBJLST, &Labels, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		DefSym = SYM_CIRCLE;
		DefSel = 0x01;
		dirty = true;
		if(name) {
			sprintf(TmpTxt, "xy-plot (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		nPoints = 0L;
		ExecInput(Desc);
		ForEach(FE_PARENT, 0L, 0L);
		return true;
	case FILE_WRITE:
		if(TheLine) TheLine->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		if(Errors) for(i = 0; i < nPoints; i++) if(Errors[i]) Errors[i]->FileIO(rw);
		if(Arrows) for(i = 0; i < nPoints; i++) if(Arrows[i]) Arrows[i]->FileIO(rw);
		if(DropLines) for(i = 0; i < nPoints; i++) if(DropLines[i]) DropLines[i]->FileIO(rw);
		if(Labels) for(i = 0; i < nPoints; i++) if(Labels[i]) Labels[i]->FileIO(rw);
		if(Bars) for(i = 0; i < nPoints; i++) if(Bars[i]) Bars[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "PlotScatt", Desc);
		}
	return false;
}

bool
xyStat::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"hide", typNZINT, &hidden, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"DefSym", typNZINT, &DefSym, 0L},
		{"baDist", typLFPOINT, &BarDist, 0L},
		{"confi", typLFLOAT, &ci, 0L},
		{"xRange", typTEXT, &xRange, 0L},
		{"yRange", typTEXT, &yRange, 0L},
		{"prefix", typTEXT, &case_prefix, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"Bars", typOBJLST, &Bars, &nPoints},
		{"Symbols", typOBJLST, &Symbols, &nPoints},
		{"PL", typGOBJ, &TheLine, 0L},
		{"ErrBars", typOBJLST, &Errors, &nPoints},
		{"Labels", typLAST | typOBJLST, &Labels, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		//most initialistion is done by PlotScatt::FileIO
		curr_data = 0L;
		case_prefix = 0L;
		ci = 95.0;
		return true;
	case FILE_READ:
		nPoints = 0L;
		ExecInput(Desc);
		ForEach(FE_PARENT, 0L, 0L);
		return true;
	case FILE_WRITE:
		if(TheLine) TheLine->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		if(Errors) for(i = 0; i < nPoints; i++) if(Errors[i]) Errors[i]->FileIO(rw);
		if(Labels) for(i = 0; i < nPoints; i++) if(Labels[i]) Labels[i]->FileIO(rw);
		if(Bars) for(i = 0; i < nPoints; i++) if(Bars[i]) Bars[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "xyStat", Desc);
		}
	return false;
}

void
FreqDist::RegGO(void *n)
{
	int i;

	if(n) {
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
FreqDist::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"hide", typNZINT, &hidden, 0L},
		{"ssRef", typTEXT, &ssRef, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"cl_start", typNZLFLOAT, &start, 0L},
		{"cl_size", typLFLOAT, &step, 0L},
		{"BarLine", typLINEDEF, &BarLine, 0L},
		{"BarFill", typFILLDEF, &BarFill, 0L},
		{"BarFillLine", typLINEDEF, &HatchLine, 0L},
		{"plots", typLAST | typOBJLST, &plots, &nPlots}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&BarFill, defs.GetFill(), sizeof(FillDEF));
		BarFill.color = 0x00c0ffffL;
		if(BarFill.hatch) memcpy(&HatchLine, BarFill.hatch, sizeof(LineDEF));
		BarFill.hatch = &HatchLine;
		memcpy(&BarLine, defs.GetOutLine(), sizeof(LineDEF));
		curr_data=0L;
		dirty = true;
		if(name) {
			sprintf(TmpTxt, "freq. dist. (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->parent=this;
		return true;
	case FILE_WRITE:
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "FreqDist", Desc);
		}
	return false;
}

void
Regression::RegGO(void *n)
{
	int i;

	if(n) {
		if(rLine) rLine->RegGO(n);
		if(sde) sde->RegGO(n);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Regression::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"hide", typNZINT, &hidden, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"xRange", typTEXT, &xRange, 0L},
		{"yRange", typTEXT, &yRange, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"Line", typGOBJ, &rLine, 0L},
		{"Ellipse", typGOBJ, &sde, 0L},
		{"Symbols", typLAST | typOBJLST, &Symbols, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		dirty = true;
		if(name) {
			sprintf(TmpTxt, "regression (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		nPoints = 0L;
		return ExecInput(Desc);
	case FILE_WRITE:
		if(rLine) rLine->FileIO(rw);
		if(sde) sde->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Regression", Desc);
		}
	return false;
}

void
BubblePlot::RegGO(void *n)
{
	int i;

	if(n) {
		if(Bubbles) for(i = 0; i < nPoints; i++) if(Bubbles[i]) Bubbles[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
BubblePlot::FileIO(int rw)
{
	descIO Desc[] = {
		{"hide", typNZINT, &hidden, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"Line", typLINEDEF, &BubbleLine, 0L},
		{"FillLine", typLINEDEF, &BubbleFillLine, 0L},
		{"Fill", typFILLDEF, &BubbleFill, 0L},
		{"Bubbles", typLAST | typOBJLST, &Bubbles, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		BubbleFill.color = defs.Color(COL_BUBBLE_FILL);
		BubbleLine.color = defs.Color(COL_BUBBLE_LINE);
		BubbleLine.width = defs.GetSize(SIZE_BUBBLE_LINE);
		BubbleFillLine.color = defs.Color(COL_BUBBLE_FILLLINE);
		BubbleFillLine.width = defs.GetSize(SIZE_BUBBLE_HATCH_LINE);
		BubbleFill.hatch = &BubbleFillLine;
		dirty = true;
		if(name) {
			sprintf(TmpTxt, "bubbles (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Bubbles) for(i = 0; i < nPoints; i++) if(Bubbles[i]) Bubbles[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "BubblePlot", Desc);
		}
	return false;
}

void
PolarPlot::RegGO(void *n)
{
	int i;

	if(n) {
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->RegGO(n);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
PolarPlot::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"hide", typNZINT, &hidden, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"ang_offs", typLFLOAT, &offs, 0L},
		{"Plots", typOBJLST, &Plots, (long*)&nPlots},
		{"Axes", typOBJLST, &Axes, (long*)&nAxes},
		{"FillLine", typLINEDEF, &FillLine, 0L},
		{"Fill", typLAST | typFILLDEF, &Fill, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		CurrDisp = 0L;
		InitVarsGO(Desc);
		if(name) {
			sprintf(TmpTxt, "polar root (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "PolarPlot", Desc);
		}
	return false;
}

void
BoxPlot::RegGO(void *n)
{
	int i;

	if(n) {
		if(Boxes) for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->RegGO(n);
		if(Whiskers) for(i = 0; i < nPoints; i++) if(Whiskers[i]) Whiskers[i]->RegGO(n);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		if(Labels) for(i = 0; i < nPoints; i++) if(Labels[i]) Labels[i]->RegGO(n);
		if(TheLine) TheLine->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
BoxPlot::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"hide", typNZINT, &hidden, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"xRange", typTEXT, &xRange, 0L},
		{"yRange", typTEXT, &yRange, 0L},
		{"prefix", typTEXT, &case_prefix, 0L},
		{"boDist", typLFPOINT, &BoxDist, 0L},
		{"ci_box", typNZLFLOAT, &ci_box, 0L},
		{"ci_err", typNZLFLOAT, &ci_err, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"Boxes", typOBJLST, &Boxes, &nPoints},
		{"Whiskers", typOBJLST, &Whiskers, &nPoints},
		{"Symbols", typOBJLST, &Symbols, &nPoints},
		{"Labels", typOBJLST, &Labels, &nPoints},
		{"Line", typLAST | typGOBJ, &TheLine, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		dirty = true;
		InitVarsGO(Desc);
		if(name) {
			sprintf(TmpTxt, "boxes (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		curr_data = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Boxes) for(i = 0; i < nPoints; i++) 
			if(Boxes[i]) Boxes[i]->FileIO(rw);
		if(Whiskers) for(i = 0; i < nPoints; i++) 
			if(Whiskers[i]) Whiskers[i]->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->FileIO(rw);
		if(Labels) for(i = 0; i < nPoints; i++) 
			if(Labels[i]) Labels[i]->FileIO(rw);
		if(TheLine) TheLine->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "BoxPlot", Desc);
		}
	return false;
}

void
DensDisp::RegGO(void *n)
{
	int i;

	if(n) {
		if(Boxes) for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
DensDisp::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"xRange", typTEXT, &xRange, 0L},
		{"yRange", typTEXT, &yRange, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"Line", typLINEDEF, &DefLine, 0L},
		{"FillLine", typLINEDEF, &DefFillLine, 0L},
		{"Fill", typFILLDEF, &DefFill, 0L},
		{"Boxes", typLAST | typOBJLST, &Boxes, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		return InitVarsGO(Desc);
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Boxes) for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "DensDisp", Desc);
		}
	return false;
}

void
StackBar::RegGO(void *n)
{
	int i;

	if(n) {
		if(Boxes) for(i = 0; i < numPlots; i++) if(Boxes[i]) Boxes[i]->RegGO(n);
		if(xyPlots) for(i = 0; i < numXY; i++) if(xyPlots[i]) xyPlots[i]->RegGO(n);
		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i]) Polygons[i]->RegGO(n);
		if(Lines) for(i = 0; i < numPL; i++) if(Lines[i]) Lines[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
StackBar::FileIO(int rw)
{
	descIO Desc[] = {
		{"Bounds", typFRECT, &Bounds, 0L},
		{"x_axis", typNZINT, &use_xaxis, 0L},
		{"y_axis", typNZINT, &use_yaxis, 0L},
		{"cumData", typNZINT, &cum_data_mode, 0L},
		{"StartVal", typNZLFLOAT, &StartVal, 0L},
		{"Dspm", typNZLFPOINT, &dspm, 0L},
		{"ssXrange", typTEXT, &ssXrange, 0L},
		{"ssYrange", typTEXT, &ssYrange, 0L},
		{"BoxBars", typOBJLST, &Boxes, (long*)&numPlots},
		{"Plots", typOBJLST, &xyPlots, (long*)&numXY},
		{"Polygons", typOBJLST, &Polygons, (long*)&numPG},
		{"Lines", typLAST | typOBJLST, &Lines, (long*)&numPL}};
	int i;

	switch(rw) {
	case INIT_VARS:
		dirty = true;	CumData = 0L;
		InitVarsGO(Desc);
		if(name) {
			sprintf(TmpTxt, "stack (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		return true;
	case FILE_WRITE:
		if(Boxes) for(i = 0; i < numPlots; i++) if(Boxes[i]) Boxes[i]->FileIO(rw);
		if(xyPlots) for(i = 0; i < numXY; i++) if(xyPlots[i]) xyPlots[i]->FileIO(rw);
		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i]) Polygons[i]->FileIO(rw);
		if(Lines) for(i = 0; i < numPL; i++) if(Lines[i]) Lines[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Stacked", Desc);
		}
	return false;
}

void
PieChart::RegGO(void *n)
{
	int i;

	if(n) {
		if(Segments) for(i = 0; i < nPts; i++) if(Segments[i]) Segments[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
PieChart::FileIO(int rw)
{
	descIO Desc[] = {
		{"ssRefA", typTEXT, &ssRefA, 0L},
		{"ssRefR", typTEXT, &ssRefR, 0L},
		{"CtDef", typLFPOINT, &CtDef, 0L},
		{"FacRad", typLFLOAT, &FacRad, 0L},
		{"Segs", typLAST | typOBJLST, &Segments, (long*)&nPts}};
	int i;

	switch(rw) {
	case INIT_VARS:
		Bounds.Xmax = Bounds.Ymax = 100.0f;
		Bounds.Xmin = Bounds.Ymin = -100.0f;
		InitVarsGO(Desc);
		CtDef.fx = 90.0;	CtDef.fy = 360.0;
		FacRad = 1.0;
		if(name) {
			sprintf(TmpTxt, "pie chart (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Segments) for(i = 0; i < nPts; i++) if(Segments[i]) Segments[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "SegChart", Desc);
		}
	return false;
}

void
GoGroup::RegGO(void *n)
{
	int i;

	if(n) {
		if(Objects) for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
GoGroup::FileIO(int rw)
{
	descIO Desc[] = {
		{"Pos", typNZLFPOINT, &fPos, 0L},
		{"Items", typLAST | typOBJLST, &Objects, (long*)&nObs}};
	int i;

	switch(rw) {
	case INIT_VARS:
		Bounds.Xmax = Bounds.Ymax = 100.0f;
		Bounds.Xmin = Bounds.Ymin = -100.0f;
		return InitVarsGO(Desc);
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Objects) for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "group", Desc);
		}
	return false;
}

void
Scatt3D::RegGO(void *n)
{
	int i;

	if(n) {
		if(Line) Line->RegGO(n);
		if(rib) rib->RegGO(n);
		if(Balls) for(i = 0; i < nBalls; i++) if(Balls[i]) Balls[i]->RegGO(n);
		if(Columns) for(i = 0; i < nColumns; i++) if(Columns[i]) Columns[i]->RegGO(n);
		if(DropLines) for(i = 0; i < nDropLines; i++) if(DropLines[i]) DropLines[i]->RegGO(n);
		if(Arrows) for(i = 0; i < nArrows; i++) if(Arrows[i]) Arrows[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Scatt3D::FileIO(int rw)
{
	descIO Desc[] = {
		{"ssRefX", typTEXT, &ssRefX, 0L},
		{"ssRefY", typTEXT, &ssRefY, 0L},
		{"ssRefZ", typTEXT, &ssRefZ, 0L},
		{"Line", typGOBJ, &Line, 0L},
		{"Balls", typOBJLST, &Balls, &nBalls},
		{"Columns", typOBJLST, &Columns, &nColumns},
		{"DropLines", typOBJLST, &DropLines, &nDropLines},
		{"ParaV", typGOBJ, &rib, 0L},
		{"Arrows", typLAST | typOBJLST, &Arrows, &nArrows}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		c_flags = 0L;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		dirty = true;
		if(name) {
			sprintf(TmpTxt, "xyz-plot (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		//now set parent in all children
		if(Balls) for(i = 0; i < nBalls; i++) if(Balls[i]) Balls[i]->parent = this;
		if(Columns) for(i = 0; i < nColumns; i++) if(Columns[i]) Columns[i]->parent = this;
		if(DropLines) for(i = 0; i < nDropLines; i++) if(DropLines[i]) DropLines[i]->parent = this;
		if(Arrows) for(i = 0; i < nArrows; i++) if(Arrows[i]) Arrows[i]->parent = this;
		if(Line) Line->parent = this;
		if(rib) rib->parent = this;
		return true;
	case FILE_WRITE:
		if(Line) Line->FileIO(rw);
		if(rib) rib->FileIO(rw);
		if(Balls) for(i = 0; i < nBalls; i++) if(Balls[i]) Balls[i]->FileIO(rw);
		if(Columns) for(i = 0; i < nColumns; i++) if(Columns[i]) Columns[i]->FileIO(rw);
		if(DropLines) for(i = 0; i < nDropLines; i++) if(DropLines[i]) DropLines[i]->FileIO(rw);
		if(Arrows) for(i = 0; i < nArrows; i++) if(Arrows[i]) Arrows[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Scatt3D", Desc);
		}
	return false;
}

void
Ribbon::RegGO(void *n)
{
	int i;

	if(n) {
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Ribbon::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"z-pos", typNZLFLOAT, &z_value},
		{"z-width", typNZLFLOAT, &z_width},
		{"relwidth", typNZLFLOAT, &relwidth},
		{"ssRefX", typTEXT, &ssRefX, 0L},
		{"ssRefY", typTEXT, &ssRefY, 0L},
		{"ssRefZ", typTEXT, &ssRefZ, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"values", typFPLST3D, &values, &nVal},
		{"Planes", typLAST | typOBJLST, &planes, &nPlanes}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		relwidth = 0.6;				dirty = true;
		if(name) {
			sprintf(TmpTxt, "ribbon (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		//now set parent in all children
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Ribbon", Desc);
		}
	return false;
}

void
Grid3D::RegGO(void *n)
{
	int i;

	if(n) {
		if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->RegGO(n);
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Grid3D::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"Start", typPOINT3D, &start, 0L},
		{"Step", typPOINT3D, &step, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"lines", typOBJLST, &lines, &nLines},
		{"planes", typLAST | typOBJLST, &planes, &nPlanes}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		step.fx = step.fz = 1.0;
		if(name) {
			sprintf(TmpTxt, "grid (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		//now set parent in all children
		if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->parent = this;
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->FileIO(rw);
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Grid3D", Desc);
		}
	return false;
}

bool
Limits::FileIO(int rw)
{
	descIO Desc[] = {
		{"Bounds", typLAST | typFRECT, &Bounds, 0L}};

	switch(rw) {
	case INIT_VARS:
		if(name) {
			sprintf(TmpTxt, "limits (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), "Limits", Desc);
		}
	return false;
}

void
Function::RegGO(void *n)
{
	if(n) {
		if(dl)dl->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Function::FileIO(int rw)
{
	descIO Desc[] = {
		{"x1", typNZLFLOAT, &x1, 0L},
		{"x2", typNZLFLOAT, &x2, 0L},
		{"xstep", typNZLFLOAT, &xstep, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"f_xy", typTEXT, &cmdxy, 0L},
		{"param", typTEXT, &param, 0L},
		{"DataLine", typLAST | typGOBJ, &dl, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cmdxy = param = 0L;
		memcpy(&Line, defs.GetLine(), sizeof(LineDEF));
		if(name) {
			sprintf(TmpTxt, "function (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(dl) dl->parent = this;
		return true;
	case FILE_WRITE:
		if(dl) dl->FileIO(rw);
		ExecOutput(Notary->RegisterGO(this), "Function", Desc);
		}
	return false;
}

void
FitFunc::RegGO(void *n)
{
	int i;

	if(n) {
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
FitFunc::FileIO(int rw)
{
	descIO Desc[] = {
		{"ssXref", typTEXT, &ssXref, 0L},
		{"ssYref", typTEXT, &ssYref, 0L},
		{"x1", typNZLFLOAT, &x1, 0L},
		{"x2", typNZLFLOAT, &x2, 0L},
		{"xstep", typNZLFLOAT, &xstep, 0L},
		{"conv", typNZLFLOAT, &conv, 0L},
		{"chi2", typNZLFLOAT, &chi2, 0L},
		{"maxiter", typNZINT, &maxiter, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"f_xy", typTEXT, &cmdxy, 0L},
		{"p_xy", typTEXT, &parxy, 0L},
		{"Symbols", typLAST | typOBJLST, &Symbols, &nPoints}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cmdxy = parxy = 0L;
		memcpy(&Line, defs.GetLine(), sizeof(LineDEF));
		conv = 1.0e-15;	maxiter = 100;
		dl = 0L;
		if(name) {
			sprintf(TmpTxt, "fit function (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "FitFunc", Desc);
		}
	return false;
}

bool
GridLine::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"Line", typLINEDEF, &LineDef, 0L},
		{"flags", typLAST | typDWORD, &flags, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		ncpts = 0;		cpts = 0L;	gl1 = gl2 = gl3 = 0L;	ls = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;	mo = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), 
			Id == GO_GRIDLINE ?(char*)"GridLine" : 
			Id == GO_GRIDRADIAL? (char*)"GridRadial" : (char*)"GridLine3D", Desc);
		}
	return false;
}

void
Tick::RegGO(void *n)
{
	if(n) {
		if(Grid && (flags & AXIS_GRIDLINE)) Grid->RegGO(n);
		if(label) label->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Tick::FileIO(int rw)
{
	GraphObj *gl = Grid;
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"Val", typLFLOAT, &value, 0L},
		{"Flags", typDWORD, &flags, 0L},
		{"Rot", typNZLFLOAT, &angle, 0L},
		{"GridType", typINT, &gl_type, 0L},
		{"Grid", typGOBJ, &gl, 0L},
		{"Label", typGOBJ, &label, 0L},
		{"Size", typLAST | typLFLOAT, &size, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if(Grid && (flags & AXIS_GRIDLINE)) Grid->FileIO(rw);
		if(label) label->FileIO(rw);
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		fix = fiy = 0.0f;	ls = 0L;	Grid = 0L;	mo = 0L;
		size = defs.GetSize(SIZE_AXIS_TICKS);
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Grid = (GridLine*) gl;
		if(Grid)Grid->parent = this;
		if(label)label->parent = this;
		return true;
	case FILE_WRITE:
		if(Grid && (flags & AXIS_GRIDLINE)) Grid->FileIO(rw);
		else gl = 0L;
		if(label) label->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Tick", Desc);
		}
	return false;
}

void
Axis::TickFile(char *name)
{
	ReadCache *ca;
	int i, j, k, nt;
	char line[500], item[20];
	Tick **ttck;

	if(!name) return;
	if(!(ca = new ReadCache())) return;
	if(! ca->Open(name)) {
		delete ca;
		sprintf(TmpTxt, "Error open file \"%s\"\nfor axis ticks", name);
		ErrorBox(TmpTxt);
		return;
		}
	Command(CMD_FLUSH, 0L, 0L);
	if(!(Ticks = ((Tick**)calloc(nt = 100, sizeof(Tick*))))) return;
	for(i = 0; ; i++) {
		j = k = 0;
		ca->ReadLine(line, sizeof(line));
		if(!line[0]) break;
		while(line[j] && line[j] < 33) j++;
		do{ item[k] = line[j++]; }
			while(item[k] >32 && item[k++] != '=' && k <sizeof(item) && j <sizeof(line));
		item[k--] = 0;		if(!line[j-1])j--;
		while(k && !(isdigit(item[k])))item[k--]=0;
		while(line[j] && (line[j]<33 || line[j] == '"'))j++;
		k = strlen(line);
		while(k >=j && (line[k] < 33 || line[k] == '"')) line[k--] = 0;
		//realloc table if necessary
		if(NumTicks >= nt) {
			if((ttck= (Tick**)realloc(Ticks, (nt += 1000)*sizeof(Tick*))))Ticks= ttck;
			else NumTicks--;
			}
		//now add tick to table
		if(!(Ticks[NumTicks] = new Tick(this, data, atof(item),
			line[j] ? axis->flags : axis->flags | AXIS_MINORTICK)))break;
		Ticks[NumTicks]->Command(CMD_SETTEXT, line+j, 0L);
		NumTicks++;
		}
	ca->Close();
	if(!NumTicks && Ticks) {
		free(Ticks);
		NumTicks = 0;
		}
	delete ca;
}

void
Axis::RegGO(void *n)
{
	int i;

	if(n) {
		for(i = 0; Ticks && i< NumTicks; i++) if(Ticks[i]) Ticks[i]->RegGO(n);
		if(axisLabel) axisLabel->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Axis::FileIO(int rw)
{
	char *tickfile = 0L;
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"sAxLine", typLFLOAT, &sizAxLine, 0L},
		{"sAxTick", typLFLOAT, &sizAxTick, 0L},
		{"BrkGap", typLFLOAT, &brkgap, 0L},
		{"BrkSymSize", typLFLOAT, &brksymsize, 0L},
		{"BrkSym", typINT, &brksym, 0L},
		{"sTickLabel", typLFLOAT, &sizAxTickLabel, 0L},
		{"tick_type", typNZINT, &tick_type, 0L},
		{"tick_angle", typNZLFLOAT, &tick_angle, 0L},
		{"LbDist", typNZLFPOINT, &lbdist, 0L},
		{"TickLbDist", typNZLFPOINT, &tlbdist, 0L}, 
		{"Color", typDWORD, &colAxis, 0L},
		{"AxisDef", typPTRAXDEF, &axis},
		{"GridLine", typLINEDEF, &GridLine, 0L},
		{"GridType", typINT, &gl_type, 0L},
		{"Ticks", typOBJLST, &Ticks, (long*)&NumTicks},
		{"Label", typGOBJ, &axisLabel, 0L},
		{"TickFile", typTEXT, &tickfile, 0L},
		{"ssRefTV", typTEXT, &ssMATval, 0L},
		{"ssRefTL", typTEXT, &ssMATlbl, 0L},
		{"ssRefMT", typTEXT, &ssMITval, 0L},
		{"tlbDef", typLAST | typTXTDEF, &tlbdef, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		sizAxLine = defs.GetSize(SIZE_AXIS_LINE);
		sizAxTick = defs.GetSize(SIZE_AXIS_TICKS);
		sizAxTickLabel = defs.GetSize(SIZE_TICK_LABELS);
		colAxis = parent ? parent->GetColor(COL_AXIS) : defs.Color(COL_AXIS);
		GridLine.color = 0x00808080L;
		GridLine.pattern = 0xf8f8f8f8L;
		brksymsize = defs.GetSize(SIZE_TICK_LABELS);
		brkgap = defs.GetSize(SIZE_AXIS_TICKS);
		brksym = 2;
		tlbdef.ColTxt = parent ? parent->GetColor(COL_AXIS) : defs.Color(COL_AXIS);
		tlbdef.ColBg = parent ? parent->GetColor(COL_BG) : defs.Color(COL_AXIS);
		tlbdef.RotBL = tlbdef.RotCHAR = 0.0f;
		tlbdef.fSize = parent ? parent->GetSize(SIZE_TICK_LABELS) : defs.GetSize(SIZE_TICK_LABELS);
		tlbdef.Align = TXA_VCENTER | TXA_HCENTER;
		tlbdef.Style = TXS_NORMAL;
		tlbdef.Mode = TXM_TRANSPARENT;
		tlbdef.Font = FONT_HELVETICA;
		tlbdef.text = 0L;	l_segs = 0L;	nl_segs = 0;
		drawOut = scaleOut = 0L;			bModified = false;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		mo = 0L;
		return true;
	case FILE_READ:
		if(axisLabel)DeleteGO(axisLabel);
		if(tickfile) free(tickfile);		if(ssMATval) free(ssMATval);
		if(ssMATlbl) free(ssMATlbl);		if(ssMITval) free(ssMITval);
		tickfile = 0L;
		if(ExecInput(Desc) && tickfile && tickfile[0]){
			TickFile(tickfile);
			free(tickfile);					tickfile = 0L;
			}
		if(axis) axis->owner = this;
		if(axisLabel)axisLabel->parent = this;
		if(Ticks) for(i = 0; i < NumTicks; i++) if(Ticks[i]) Ticks[i]->parent = this;
		return true;
	case FILE_WRITE:
		//do all ticks
		for(i = 0; Ticks && i< NumTicks; i++) if(Ticks[i]) Ticks[i]->FileIO(rw);
		if(axisLabel) axisLabel->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Axis", Desc);
		}
	return false;
}

void
Plot3D::RegGO(void *n)
{
	int i;

	if(n) {
		if(plots) for(i = 0; plots && i< nPlots; i++) if(plots[i]) plots[i]->RegGO(n);
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Plot3D::FileIO(int rw)
{
	fPOINT3D rot_vec, rot_ang;
	//DEBUG: must initialize rot_vec, rot_ang in case they are not present in file
	descIO Desc[] = {
		{"xBounds", typLFPOINT, &xBounds, 0L},
		{"yBounds", typLFPOINT, &yBounds, 0L},
		{"zBounds", typLFPOINT, &zBounds, 0L},
		{"Corner1", typPOINT3D, &cub1, 0L},
		{"Corner2", typPOINT3D, &cub2, 0L},
		{"Center", typPOINT3D, &rotC, 0L},
		{"rot_vec", typPOINT3D, &rot_vec, 0L},
		{"rot_ang", typPOINT3D, &rot_ang, 0L},
		{"Axes", typOBJLST, &Axes, (long*)&nAxes},
		{"Plots", typLAST | typOBJLST, &plots, (long*)&nPlots}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		drag = 0L;		moveable = 1;
		//set up RotDef
		RotDef[0] = 0.919384;		RotDef[1] = 0.389104;		RotDef[2] = -0.057709;
		RotDef[3] = 0.327146;		RotDef[4] = 0.944974;		RotDef[5] = 1.0-RotDef[4];
		cub1.fx = defs.GetSize(SIZE_GRECT_LEFT) + defs.GetSize(SIZE_DRECT_LEFT);
		cub2.fx = defs.GetSize(SIZE_GRECT_LEFT) + defs.GetSize(SIZE_DRECT_RIGHT);
		cub1.fy = defs.GetSize(SIZE_GRECT_TOP) + defs.GetSize(SIZE_DRECT_BOTTOM);
		cub2.fy = defs.GetSize(SIZE_GRECT_TOP) + defs.GetSize(SIZE_DRECT_TOP);
		cub1.fy += defs.GetSize(SIZE_DRECT_TOP);	cub2.fy += defs.GetSize(SIZE_DRECT_TOP);
		cub1.fz = 0.0;
		cub2.fz = defs.GetSize(SIZE_DRECT_BOTTOM) - defs.GetSize(SIZE_DRECT_TOP);
		rotC.fx = (cub1.fx + cub2.fx)/2.0;		rotC.fy = (cub1.fy + cub2.fy)/2.0;
		rotC.fz = (cub1.fz + cub2.fz)/2.0;
		dispObs = 0L;		nmaxObs = 0;	crea_flags = 0L;		dirty = true;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		if(name) {
			sprintf(TmpTxt, "3D-root (%s)", name);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		RotDef[0] = rot_vec.fx;	RotDef[1] = rot_vec.fy;	RotDef[2] = rot_vec.fz;
		RotDef[3] = rot_ang.fx;	RotDef[4] = rot_ang.fy;	RotDef[5] = rot_ang.fz;
		return true;
	case FILE_WRITE:
		rot_vec.fx = RotDef[0];	rot_vec.fy = RotDef[1];	rot_vec.fz = RotDef[2];
		rot_ang.fx = RotDef[3];	rot_ang.fy = RotDef[4];	rot_ang.fz = RotDef[5];
		//do all plots
		for(i = 0; plots && i< nPlots; i++) if(plots[i]) plots[i]->FileIO(rw);
		//do all axes
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Plot3D", Desc);
		}
	return false;
}

void
Func3D::RegGO(void *n)
{
}

bool
Func3D::FileIO(int rw)
{
	fPOINT3D rot_vec, rot_ang;
	//DEBUG: must initialize rot_vec, rot_ang in case they are not present in file
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"xBounds", typLFPOINT, &xBounds, 0L},
		{"yBounds", typLFPOINT, &yBounds, 0L},
		{"zBounds", typLFPOINT, &zBounds, 0L},
		{"Corner1", typPOINT3D, &cub1, 0L},
		{"Corner2", typPOINT3D, &cub2, 0L},
		{"Center", typPOINT3D, &rotC, 0L},
		{"rot_vec", typPOINT3D, &rot_vec, 0L},
		{"rot_ang", typPOINT3D, &rot_ang, 0L},
		{"Axes", typOBJLST, &Axes, (long*)&nAxes},
		{"Plots", typOBJLST, &plots, (long*)&nPlots},
		{"x1", typNZLFLOAT, &x1, 0L},
		{"x2", typNZLFLOAT, &x2, 0L},
		{"xstep", typNZLFLOAT, &xstep, 0L},
		{"z1", typNZLFLOAT, &x1, 0L},
		{"z2", typNZLFLOAT, &x2, 0L},
		{"zstep", typNZLFLOAT, &xstep, 0L},
		{"g_idx", typNZINT, &g_idx, 0L},
		{"Line", typLINEDEF, &Line, 0L},
		{"Fill", typFILLDEF, &Fill, 0L},
		{"f_xz", typTEXT, &cmdxy, 0L},
		{"param", typLAST | typTEXT, &param, 0L}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		x1 = -20.0;		x2 = 20.0;	xstep = 2.0;
		z1 = -20.0;		z2 = 20.0;	zstep = 2.0;
		gda = 0L;		gob = 0L;
		g_idx = 0;		param = cmdxy = 0L;
		Line.width = defs.GetSize(SIZE_HAIRLINE);
		Line.patlength = defs.GetSize(SIZE_PATLENGTH);
		Line.color = Line.pattern = 0x0L;
		Fill.color = 0x00c0c0c0;
		Fill.color2 = 0x00ffffff;
		Fill.hatch = 0L;
		Fill.type = FILL_LIGHT3D;
		if(name) {
			sprintf(TmpTxt, "3D function (Plot %d)", cPlots);
			free(name);		name=strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		RotDef[0] = rot_vec.fx;	RotDef[1] = rot_vec.fy;	RotDef[2] = rot_vec.fz;
		RotDef[3] = rot_ang.fx;	RotDef[4] = rot_ang.fy;	RotDef[5] = rot_ang.fz;
		return true;
	case FILE_WRITE:
		rot_vec.fx = RotDef[0];	rot_vec.fy = RotDef[1];	rot_vec.fz = RotDef[2];
		rot_ang.fx = RotDef[3];	rot_ang.fy = RotDef[4];	rot_ang.fz = RotDef[5];
		//do all plots
		for(i = 0; plots && i< nPlots; i++) if(plots[i]) plots[i]->FileIO(rw);
		//do all axes
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		ExecOutput(Notary->RegisterGO(this), "Func3D", Desc);
		}
	return false;
}

void
Graph::RegGO(void *n)
{
	int i;

	if(n) {
		if(Plots) for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->RegGO(n);
		if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Graph::FileIO(int rw)
{
	int ixax, iyax;
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"Units", typNZINT, &units, 0L},
		{"GRect", typFRECT, &GRect, 0L},
		{"DRect", typFRECT, &DRect, 0L},
		{"Bounds", typFRECT, &Bounds, 0L},
		{"ColFrame", typDWORD, &ColGR, 0L},
		{"ColFrameL", typDWORD, &ColGRL, 0L},
		{"ColRec", typDWORD, &ColDR, 0L},
		{"ColAxis", typDWORD, &ColAX, 0L},
		{"Xaxis", typAXDEF, &x_axis, 0L},
		{"Yaxis", typAXDEF, &y_axis, 0L},
		{"DefXAxis", typINT, &ixax, 0L},
		{"DefYAxis", typINT, &iyax, 0L},
		{"Axes", typOBJLST, &Axes, (long*)&NumAxes},
		{"Plots", typLAST | typOBJLST, &Plots, (long*)&NumPlots}};
	int i;
	bool bConvert = false;

	ixax = iyax = -1;
	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		units = defs.cUnits = defs.dUnits;
		OwnDisp = false;
		dirty = true;
		GRect.Ymin = defs.GetSize(SIZE_GRECT_TOP);		GRect.Ymax = defs.GetSize(SIZE_GRECT_BOTTOM);
		GRect.Xmin = defs.GetSize(SIZE_GRECT_LEFT);		GRect.Xmax = defs.GetSize(SIZE_GRECT_RIGHT);
		DRect.Ymin = defs.GetSize(SIZE_DRECT_TOP);		DRect.Ymax = defs.GetSize(SIZE_DRECT_BOTTOM);
		DRect.Xmin = defs.GetSize(SIZE_DRECT_LEFT);		DRect.Xmax = defs.GetSize(SIZE_DRECT_RIGHT);
		ColGR = defs.Color(COL_GRECT);					ColGRL = defs.Color(COL_GRECTLINE);
		ColDR = defs.Color(COL_DRECT);					ColBG = defs.Color(COL_BG);
		ColAX = defs.Color(COL_AXIS);
		x_axis.max = y_axis.max = 1.0;
		x_axis.owner = y_axis.owner = (void *)this;
		rcDim.left = rcDim.right = rcDim.top = rcDim.bottom = 0;
		rcUpd.left = rcUpd.right = rcUpd.top = rcUpd.bottom = 0;
		CurrGO = 0L;		Disp = 0L;			Sc_Plots = 0L;
		AxisTempl = 0;		nscp = 0;			CurrDisp = 0L;
		ToolMode = TM_STANDARD;
		zoom_def = 0L;			tl_pts = 0L;			tl_nPts = 0;
		tickstyle = zoom_level = 0;
		frm_g = frm_d = 0L;		filename = 0L;
		rc_mrk.left = rc_mrk.right = rc_mrk.top = rc_mrk.bottom = -1;
		return true;
	case FILE_READ:
		units = 0;			//default to mm if statement mising in file
		if((bConvert =ExecInput(Desc)) && ixax>=0 && iyax >=0 && Axes && 
			NumAxes >= ixax+1 && NumAxes >= iyax) {
			if(Axes[ixax]) Axes[ixax]->Command(CMD_SET_AXDEF, &x_axis, 0L);
			if(Axes[iyax]) Axes[iyax]->Command(CMD_SET_AXDEF, &y_axis, 0L);
			return true;
			}
		return bConvert;
	case FILE_WRITE:
		bModified = false;
		//find default axes
		if(Axes) for(i = 0; Axes && i < NumAxes; i++) {
			if(Axes[i] && Axes[i]->GetAxis() == &x_axis) ixax = i;
			else if(Axes[i] && Axes[i]->GetAxis() == &y_axis) iyax = i;
			}
		if(Id == GO_GRAPH)RegGO(Notary);
		//do all plots
		if(Plots) for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		//do all axes
		if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Graph", Desc);
		}
	return false;
}

void
Page::RegGO(void *n)
{
	int i;

	if(n) {
		if(Plots) for(i = 0; i< NumPlots; i++) 
			if(Plots[i] && Plots[i]->Id != GO_GRAPH) Plots[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Page::FileIO(int rw)
{
	descIO Desc[] = {
		{"Type", typNZINT, &type, 0L},
		{"Units", typNZINT, &units, 0L},
		{"GRect", typFRECT, &GRect, 0L},
		{"Plots", typLAST | typOBJLST, &Plots, (long*)&NumPlots}};
	int i;

	switch(rw) {
	case INIT_VARS:
		//assume that Graph::FileIO(INIT_VARS) has been executed
		GRect.Xmin = GRect.Ymin = 0.0;
		GetPaper(&GRect.Xmax, &GRect.Ymax);
		ColBG = 0x00e8e8e8L;
		LineDef.width = 0.0;	LineDef.patlength = 1.0;
		LineDef.color = LineDef.pattern = 0x0L;
		FillDef.type = FILL_NONE;
		FillDef.color = 0x00ffffffL;	//use white paper
		FillDef.scale = 1.0;
		FillDef.hatch = 0L;		filename =0L;
		return true;
	case FILE_READ:
		Graph::FileIO(rw);
		return true;
	case FILE_WRITE:
		//do all plots
		bModified = false;
		if(Id == GO_PAGE)RegGO(Notary);
		if(Plots) for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), "Page", Desc);
		}
	return false;
}

bool
DefsRW::FileIO(int rw)
{
	descIO Desc[] = {
		{"dUnits", typINT, &defs.dUnits, 0L},
		{"cUnits", typINT, &defs.dUnits, 0L},
		{"dtHeight", typINT, &dlgtxtheight, 0L},
		{"ss_txt", typLFLOAT, &defs.ss_txt, 0L},
		{"File1", typTEXT, &defs.File1, 0L},
		{"File2", typTEXT, &defs.File2, 0L},
		{"File3", typTEXT, &defs.File3, 0L},
		{"File4", typTEXT, &defs.File4, 0L},
		{"File5", typTEXT, &defs.File5, 0L},
		{"File6", typLAST | typTEXT, &defs.File6, 0L}};

	switch(rw) {
	case FILE_READ:
		ExecInput(Desc);
		return true;
	case FILE_WRITE:
		Notary = new notary();
		unlink(defs.IniFile);
		iFile = OpenOutputFile(defs.IniFile);
		if(iFile >=0) {
			ExecOutput(-1, "Defaults", Desc);
			}
		CloseOutputFile();	if(Notary) delete Notary;	Notary = 0L;
		return true;
		}
	return false;
}
