%{
/*
 mfcalc.y, mfcalc.cpp, Copyright (c) 2004 R.Lackner
 parse string and simple math: based on the bison 'mfcalc' example

    This file is part of RLPlot.

    RLPlot is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RLPlot is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RLPlot; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include "rlplot.h"

struct symrec {
	char *name;
	int type, row, col;
	struct {
		double var;
		double (*fnctptr)(...);
		} value;
	int arg_type;
	struct symrec *next;
};

typedef struct{
	double  val;
	int type;
	symrec  *tptr;
	double *a_data;
	int a_count;

}YYSTYPE;

symrec *putsym (char *sym_name, int sym_type, int arg_type);
symrec *getsym (char *sym_name);
int push(YYSTYPE *res, YYSTYPE *val);
int get_range(YYSTYPE *res, char *first, char *last);
void yyerror(char *s);
int yylex(void);

static double line_result;
static DataObj *curr_data;
%}

%token <val>  NUM ARR PI E
%token <tptr> VAR FNCT
%type  <val>  exp

%right  '='
%left	','		/* list separator       */
%right	':'		/* range separator      */
%left	AND OR
%left   EQ NE GT GE LT LE
%left   '-' '+'
%left   '*' '/'
%left   NEG 		/* negation-unary minus */
%right  '^'	 	/* exponentiation       */

/* Grammar follows */
%%
input:    /* empty string */
	| input line
;

line:	 '\n' ';'
	| exp '\n'		{line_result = $1; return 0;}
	| exp ';'		{line_result = $1; return 0;}
	| error '\n'		{yyerrok;}
;

exp:	NUM				{$$ = $1;}
	|PI				{$$ = 3.14159265358979;}
	|E				{$$ = 2.71828182845905;}
	|VAR				{$$ = $1->value.var;}
	|VAR '=' exp		{$$ = $1->value.var = $3;
					if($1->row >=0 && $1->col >= 0 && curr_data) curr_data->SetValue($1->row, $1->col, $3);}
	|FNCT '(' exp ')'	{$$ = ($1->arg_type == ARR) ? ((*$1->value.fnctptr)(&yyvsp[-1])) : 
					(*($1->value.fnctptr))($3);}
	|exp AND exp		{$$ = (($1 != 0) && ($3 != 0))? 1 : 0;}
	|exp OR exp		{$$ = (($1 != 0) || ($3 != 0))? 1 : 0;}
	|exp EQ exp		{$$ = ($1 == $3) ? 1 : 0;}
	|exp NE exp		{$$ = ($1 != $3) ? 1 : 0;}
	|exp GT exp		{$$ = ($1 > $3) ? 1 : 0;}
	|exp GE exp		{$$ = ($1 >= $3) ? 1 : 0;}
	|exp LT exp		{$$ = ($1 < $3) ? 1 : 0;}
	|exp LE exp		{$$ = ($1 <= $3) ? 1 : 0;}
	|exp '+' exp		{$$ = $1 + $3;}
	|exp '-' exp		{$$ = $1 - $3;}
	|exp '*' exp		{$$ = $1 * $3;}
	|exp '/' exp		{if($3 != 0.0) $$ = $1 / $3;
					else $$ = (getsym("zdiv"))->value.var; }
	|exp ',' exp		{push(&yyval, &yyvsp[0]);}
	|VAR ':' VAR		{get_range(&yyval, $1->name, $3->name);}
	|'-' exp  %prec NEG	{$$ = -$2;}
	|exp '^' exp		{$$ = pow($1, $3);}
	|'(' exp ')'		{$$ = $2;}
;
%%

void yyerror(char *s)
{  
	//Called by yyparse on error
	if(curr_data) curr_data->Command(CMD_ERROR, s, 0L);
	else printf("%s\n", s);
}

/* more functions */
double sign(double v)
{
	if(v > 0.0) return 1.0;
	if(v < 0.0) return -1.0;
	return 0.0;
}

#undef min
double min(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) 
			if(sr->a_data[i] < sr->val) sr->val = sr->a_data[i];
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	return sr->val;
}

#undef max
double max(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) 
			if(sr->a_data[i] > sr->val) sr->val = sr->a_data[i];
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	return sr->val;
}

double count(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		sr->val = (double)sr->a_count;
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	else sr->val = 1.0;
	return sr->val;
}

double sum(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) sr->val += sr->a_data[i];
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	return sr->val;
}

double calc_mean(double *values, int n)
{
	int i;
	double sum;

	for(i = 1, sum = values[0]; i < n; i++) sum += values[i];
	return (sum/n);
}

double calc_variance(double *values, int n)
{
	int i;
	double ss, d, mean = calc_mean(values, n);

	for(i=0, ss=0.0; i < n; i++) ss += ((d=values[i]-mean)*d);
	return (ss/(n-1));
}

double mean(YYSTYPE *sr) 
{
	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		sr->val = calc_mean(sr->a_data, sr->a_count);
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	return sr->val;
}

double variance(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		sr->val = calc_variance(sr->a_data, sr->a_count);
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	return sr->val;
}

double stdev(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		sr->val = sqrt(calc_variance(sr->a_data, sr->a_count));
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	return sr->val;
}

double sterr(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->type == ARR && sr->a_data && sr->a_count){
		sr->val = sqrt(calc_variance(sr->a_data, sr->a_count))/sqrt(sr->a_count);
		free(sr->a_data);
		sr->a_data = 0L;
		sr->a_count = 0;
		sr->type = NUM;
		}
	return sr->val;
}

struct init
{
	char *fname;
	double (*fnct)(double);
	int arg_type;
};

struct init arith_fncts[] = {
	{"variance", (double(*)(double))&variance, ARR},
	{"stdev", (double(*)(double))&stdev, ARR},
	{"sterr", (double(*)(double))&sterr, ARR},
	{"min", (double(*)(double))&min, ARR},
	{"max", (double(*)(double))&max, ARR},
	{"count", (double(*)(double))&count, ARR},
	{"sum", (double(*)(double))&sum, ARR},
	{"mean", (double(*)(double))&mean, ARR},
	{"sign", sign, VAR},
	{"abs", fabs, VAR},
	{"asin", asin, VAR},
	{"acos", acos, VAR},
	{"atan", atan, VAR},
	{"sinh", sinh, VAR},
	{"cosh", cosh, VAR},
	{"tanh", tanh, VAR},
	{"sin",  sin, VAR},
	{"cos",  cos, VAR},
	{"atan", atan, VAR},
	{"log10", log10, VAR},
	{"ln",   log, VAR},
	{"exp",  exp, VAR},
	{"sqrt", sqrt, VAR},
	{0, 0, 0}};

//The symbol table: a chain of `struct symrec'
static symrec *sym_table = (symrec *) 0;

// Put arithmetic functions and predifened variables in table
void init_table (void)
{
	int i;
	symrec *ptr;

	for (i = 0; arith_fncts[i].fname != 0; i++) {
		ptr = putsym (arith_fncts[i].fname, FNCT, arith_fncts[i].arg_type);
		ptr->value.fnctptr = (double (*)(...))arith_fncts[i].fnct;
		}
	ptr = putsym("zdiv", VAR, 0);	ptr->value.var = 1.0;
}

void clear_table()
{
	symrec *ptr, *next;

	for (ptr = sym_table; ptr != (symrec *) 0;){
		if(ptr) {
			next = (symrec*)ptr->next;
			free(ptr);
			}
		ptr = next;
		}
	sym_table = (symrec *) 0;
}

symrec *
putsym (char *sym_name, int sym_type, int arg_type)
{
	symrec *ptr;

	ptr = (symrec *) malloc (sizeof (symrec));
	ptr->name = (char *) malloc (strlen (sym_name) + 1);
	strcpy (ptr->name,sym_name);
	ptr->type = sym_type;
	ptr->value.var = 0; /* set value to 0 even if fctn.  */
	ptr->arg_type = arg_type;
	ptr->col = ptr->row = -1;
	ptr->next = (struct symrec *)sym_table;
	sym_table = ptr;
	return ptr;
}

symrec *
getsym (char *sym_name)
{
	symrec *ptr;
	int row, col;
	double value;
	AccRange *ar;

	if(!sym_name || !sym_name[0]) return 0;
	for (ptr = sym_table; ptr != (symrec *) 0; ptr = (symrec *)ptr->next)
		if (!(strcmp(ptr->name,sym_name))) return ptr;
        if((curr_data) && (isalpha(sym_name[0]) || sym_name[0] == '$') && isdigit(sym_name[strlen(sym_name)-1])) {
		if((ar = new AccRange(sym_name)) && ar->GetFirst(&col, &row) && 
			(ptr = putsym(sym_name, VAR, 0))) {
			if(curr_data->GetValue(row, col, &value)) ptr->value.var = value;
			else ptr->value.var = 0.0;
			ptr->row = row;	ptr->col = col;
			delete(ar);
			return ptr;
			}
		}
	return 0;
}

int
push(YYSTYPE *res, YYSTYPE *val)
{
	if(val->type == ARR) {
		if(res->type == NUM) {
			if(!(val->a_data=(double*)realloc(val->a_data, (val->a_count+2)*sizeof(double))))return 0;
			val->a_data[val->a_count++] = res->val;
			res->a_data = val->a_data;		res->a_count = val->a_count;
			res->type = ARR;			val->a_data = 0L;
			val->a_count = 0;			val->type = NUM;
			val->val = res->val;			return 1;
			}
		if(res->type == ARR) {
			if(!(res->a_data=(double*)realloc(res->a_data, (val->a_count+res->a_count)*sizeof(double))))return 0;
			memcpy(&res->a_data[res->a_count], val->a_data, val->a_count*sizeof(double));
			res->a_count += val->a_count;		free(val->a_data);
			val->a_data = 0L;			val->a_count = 0;
			val->type = NUM;			return 1;
			}
		}
	if(res->type == NUM) {
		if(!(res->a_data =  (double*)malloc(2*sizeof(double))))return 0;
		res->a_data[0] = res->val;			res->a_data[1] = val->val;
		res->a_count = 2;				res->type = ARR;
		return 1;
		}
	if(res->type == ARR) {
		if(!(res->a_data = (double*)realloc(res->a_data, (res->a_count+1)*sizeof(double))))return 0; 
		res->a_data[res->a_count] = val->val;		res->a_count++;
		return 1;
		}
	return 0;
}

int
get_range(YYSTYPE *res, char *first, char *last)
{
	char r_txt[40];
	AccRange *r;
	int row, col;
	double value;
	YYSTYPE tmp;

	if(!res || !first || !last || !curr_data) return 0;
	sprintf(r_txt, "%s:%s", first, last);
	if(res->type != ARR && (r = new AccRange(r_txt)) && r->GetFirst(&col, &row)) {
		if(!(res->a_data =  (double*)malloc(r->CountItems() * sizeof(double)))) return 0;
		res->a_count = 0;
		res->type = ARR;
		for( ; r->GetNext(&col, &row); ) {
			if(curr_data->GetValue(row, col, &value)) res->a_data[res->a_count++] = value;
			}
		delete r;		return 1;
		}
	if((r = new AccRange(r_txt)) && r->GetFirst(&col, &row)) {
		tmp.type = NUM;
		for( ; r->GetNext(&col, &row); ) {
			if(curr_data->GetValue(row, col, &value)) {
				tmp.val = value;
				push(res, &tmp);
				}
			}
		delete r;		return 1;
		}
	return 0;
}

static char *buffer = 0L;
static int buff_pos = 0;

struct parse_info  {
	char *buffer;
	int buff_pos;
	double line_result;
	DataObj *curr_data;
	symrec *sym_table;
	YYSTYPE yylval;
	struct parse_info *next;
};
static parse_info *parse_stack = 0L;

void push_parser()
{
	parse_info *ptr;

	ptr = (parse_info *) malloc(sizeof(parse_info));
	ptr->buffer = buffer;			ptr->buff_pos = buff_pos;
	ptr->line_result = line_result;		ptr->curr_data = curr_data;
	ptr->sym_table = sym_table;		sym_table = 0L;
	memcpy(&ptr->yylval, &yylval, sizeof(YYSTYPE));
	ptr->next = parse_stack;
	parse_stack = ptr;
}

void pop_parser()
{
	parse_info *ptr;

	if(ptr = parse_stack) {
		parse_stack = ptr->next;
		buffer = ptr->buffer;			buff_pos = ptr->buff_pos;
		line_result = ptr->line_result;		curr_data = ptr->curr_data;
		sym_table = ptr->sym_table;
		memcpy(&yylval, &ptr->yylval, sizeof(YYSTYPE));
		free(ptr);
		}
}

static symrec *curr_sym;
int yylex (void)
{
	int i, c, tok;
	char tmp_txt[80];
	symrec *s;

	while((c = buffer[buff_pos++]) == ' ' || c == '\t');	//get first nonwhite char
	if(!c) return 0;
	//test for number
	if(c == '.' || isdigit(c)) {
		for(buff_pos--, i = 0; i < 79 && ((c = buffer[buff_pos]) == '.' || isdigit(c)); buff_pos++) {
			tmp_txt[i++] = (char)c;
			if(buffer[buff_pos+1] == 'e' && (buffer[buff_pos+2] == '-' || buffer[buff_pos+2] == '+')){
				tmp_txt[i++] = buffer[++buff_pos];
				tmp_txt[i++] = buffer[++buff_pos];
				}
			}
		tmp_txt[i] = 0;
		sscanf(tmp_txt, "%lf", &yylval.val);
		yylval.type = NUM;
		return NUM;
		}
	//test for name
	if(isalpha(c) || c=='$') {
 		for(buff_pos--, i = 0; i < 79 && ((c = buffer[buff_pos]) && (isalnum(c) || c == '$')); buff_pos++) {
			tmp_txt[i++] = (char)c; 
			}
		tmp_txt[i] = 0;
		if(!(strcmp(tmp_txt, "pi"))) return PI;
		if(!(strcmp(tmp_txt, "e"))) return E;
		if(!(s = getsym(tmp_txt))) s = putsym(tmp_txt, VAR, 0);
		curr_sym = yylval.tptr = s;	return s->type;
		}
	tok = 0;
	switch(c) {
	case '=':
		if(buffer[buff_pos] == '=') tok = EQ;
		break;
	case '!':
		if(buffer[buff_pos] == '=') tok = NE;
		break;
	case '>':
		if(buffer[buff_pos] == '=') tok = GE;
		else return GT;
		break;
	case '<':
		if(buffer[buff_pos] == '=') tok = LE;
		else if(buffer[buff_pos] == '>') tok = NE;
		else return LT;
		break;
	case '&':
		if(buffer[buff_pos] == '&') tok = AND;
		break;
	case '|':
		if(buffer[buff_pos] == '|') tok = OR;
		break;
		}
	if(tok) {
		buff_pos++;		return tok;
		}
	//Any other character is a token by itself
	return c;
}

bool do_xyfunc(DataObj *d, double x1, double x2, double step, char *expr, lfPOINT **pts, long *npts, char *param)
{
	double x, y;
	symrec *s;
	lfPOINT *new_points;
	long npoints = 0;
	int length;

	if(x1 < x2) step = fabs(step);
	else step = -fabs(step);
	if(!(new_points = (lfPOINT*)calloc((iround(fabs(x2-x1)/fabs(step))+2), sizeof(lfPOINT))))
		return false;
	if(d) curr_data = d;
	init_table();
	if(param) {
		length = strlen(param);
		if(!(buffer = (char*)malloc(length+2))){
			pop_parser();
			return false;
			}
		strcpy(buffer, param);	buffer[length++] = ';';
		buffer[length] = 0;	buff_pos = 0;
		do {
			yyparse();
			}while(buff_pos < length);
		free(buffer);		buffer = 0L;
		}		
	length = strlen(expr);
	buffer = expr;		s = putsym("x", VAR, 0);
	for(x = x1; step > 0.0 ? x <= x2 : x >= x2; x += step) {
		if(s = getsym("x")){
			s->value.var = x;	buff_pos = 0;
			do {
				yyparse();
				}while(buff_pos < length);
			if(s = getsym("y")) y = s->value.var;
			else y = line_result;
			new_points[npoints].fx = (getsym("x"))->value.var;
			new_points[npoints++].fy = y;
			}
		}
	*pts = new_points;	*npts = npoints;
	clear_table();
	if(curr_data) {
		curr_data->Command(CMD_CLEAR_ERROR, 0L, 0L);
		curr_data->Command(CMD_REDRAW, 0L, 0L);
		}
	return true;
}

bool do_formula(DataObj *d, char *expr, double *result)
{
	int length;

	if(d) curr_data = d;
	if(!expr || !expr[0]) return false;
	push_parser();		//make code reentrant
	init_table();		length = strlen(expr);
	if(!(buffer = (char*)malloc(length+2))){
		pop_parser();
		return false;
		}
	strcpy(buffer, expr);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = 0;
	do {
		yyparse();
		}while(buff_pos < length);
	*result = line_result;
	free(buffer);		buffer = 0L;
	clear_table();
	pop_parser();
	return true;
}

bool MoveFormula(DataObj *d, char *of, char *nf, int dx, int dy)
{
	int length, tok, pos, i;
	char *res, desc1[2], desc2[2];

	if(d) curr_data = d;
	if(!curr_data || !of || !nf) return false;
	push_parser();		//make code reentrant
	init_table();		length = strlen(of);
	if(!(buffer = (char*)malloc(length+2))){
		pop_parser();
		return false;
		}
	strcpy(buffer, of);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = pos = 0;
	res = (char *)calloc(length*2+10, sizeof(char));
	do {
		tok = yylex ();
		if(tok && tok < 256) {
			if(res[pos-1] == ' ') pos--;
			res[pos++] = (char)tok;
			}
		else switch(tok) {
			case NUM:
				pos += sprintf(res+pos, "%g ", yylval.val);
				break;
			case FNCT:
				pos += sprintf(res+pos, "%s", curr_sym->name);
				break;
			case VAR:
				if(curr_sym->col >= 0 && curr_sym->row >= 0) {
					desc1[0] = desc1[1] = desc2[0] = desc2[1] = 0;
					for(i=strlen(curr_sym->name)-1; i>0 && isdigit(curr_sym->name[i]); i--);
					if(curr_sym->name[0] == '$') desc1[0] = '$';
					if(curr_sym->name[i] == '$') desc2[0] = '$';
					pos += sprintf(res+pos, "%s%s%s%d", desc1, 
						Int2ColLabel(desc1[0] ? curr_sym->col : curr_sym->col+dx, false),
						desc2, desc2[0]? curr_sym->row+1 : curr_sym->row+1+dy);
					}
				else pos += sprintf(res+pos, "%s ", curr_sym->name);
				break;
			case PI:
				pos += sprintf(res+pos, "pi ");
				break;
			case E:
				pos += sprintf(res+pos, "e ");
				break;
			case AND:
				pos += sprintf(res+pos, " && ");
				break;
			case OR:
				pos += sprintf(res+pos, " || ");
				break;
			case EQ:
				pos += sprintf(res+pos, " == ");
				break;
			case NE:
				pos += sprintf(res+pos, " != ");
				break;
			case GT:
				pos += sprintf(res+pos, " > ");
				break;
			case GE:
				pos += sprintf(res+pos, " >= ");
				break;
			case LT:
				pos += sprintf(res+pos, " < ");
				break;
			case LE:
				pos += sprintf(res+pos, " <= ");
				break;
			}
		}while(buff_pos < length);
	while((res[pos-1] == ';' || res[pos-1] == ' ') && pos > 0) { res[pos-1] = 0; pos--;} 
	strcpy(nf, res);	free(res);
	free(buffer);		buffer = 0L;
	clear_table();
	pop_parser();
	return true;
}

static char *txt_formula;	//function to fit
static double **parval;		//pointers to parameter values
static void fcurve(double x, double **a, double *y, double dyda[], int ma)
{
	int i, length;
	double tmp, y1, y2;
	symrec *symx, *s=0L;

	if(!(symx = getsym("x"))) symx = putsym("x", VAR, 0);
	//swap parameters to requested set
	if(a != parval) for(i = 0; i < ma; i++) {
		tmp = *parval[i];	*parval[i]  = *a[i];	*a[i] = tmp;
		}
	//calc result
	symx->value.var = x;	buffer = txt_formula;
	buff_pos = 0;		length = strlen(txt_formula);
	do {	yyparse();	}while(buff_pos < length);
	if(s = getsym("y")) *y = s->value.var;
	else *y = line_result;
	if(*y == HUGE_VAL || *y == -HUGE_VAL) {
		for(i = 0, *y = 0.0; i < ma; dyda[i++] = 0.0);
		return;
		}
	//partial derivatives for each parameter by numerical differentiation
	for(i = 0; i < ma; i++) {
		if(*parval[i] != 0.0) {
			tmp = *parval[i];
			*parval[i] = tmp*.995;
			buff_pos = 0;
			do {	yyparse();	}while(buff_pos < length);
			y1 = s ? s->value.var : line_result;
			*parval[i] = tmp*1.005;
			buff_pos = 0;
			do {	yyparse();	}while(buff_pos < length);
			y2 = s ? s->value.var : line_result;
			*parval[i] = tmp;
			dyda[i] = (y2-y1)*100.0/tmp;
			}
		else dyda[i] = 0.0;
		}
	//swap parameters back to original
	if(a != parval) for(i = 0; i < ma; i++) {
		tmp = *parval[i];	*parval[i]  = *a[i];	*a[i] = tmp;
		}
}

int do_fitfunc(DataObj *d, char *rx, char *ry, char *rz, char **par, char *expr, double conv, int maxiter)
{
	int length, i, j, k, l, ndata, nparam, r1, r2, r3, c1, c2, c3, *lista, itst, itst1;
	symrec *tab1, *tab2, *csr, **parsym;
	AccRange *arx, *ary, *arz;
	double *x, *y, *z, currx, curry, currz, alamda, chisq, ochisq;
	double **covar, **alpha;
	char tmp_txt[500];

	if(d) curr_data = d;
	txt_formula = expr;
	if(!curr_data || !par || !expr || !rx || !ry) return 0;
	//process ranges and create arrays
	arx = ary = arz = 0L;	x = y = z = 0L;	parval = 0L;	parsym = 0L;
	if(!(arx = new AccRange(rx)))return 0;
	i = arx->CountItems()+1;
	if(!(ary = new AccRange(ry))){
		delete arx;	return 0;
		}
	if(rz && !(arz = new AccRange(rz))){
		delete ary;	delete arx;	return 0;
		}
	if(!(x = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		delete ary;	delete arx;	return 0;
		}
	if(!(y = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		free(x);	delete arx;	delete ary;	return 0;
		}
	if(rz && !(y = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		free(y);	free(x);	delete arx;	delete ary;	return 0;
		}
	arx->GetFirst(&c1, &r1);	ary->GetFirst(&c2, &r2);
	if(rz) arz->GetFirst(&c3, &r3);
	for(ndata = j = 0; j < i; j++) {
		if(rz) {
			if(arx->GetNext(&c1, &r1) && ary->GetNext(&c2, & r2) && arz->GetNext(&c3, &r3) &&
				curr_data->GetValue(r1, c1, &currx) && curr_data->GetValue(r2, c2, &curry) &&
				curr_data->GetValue(r3, c3, &currz)) {
				x[ndata] = currx;	y[ndata] = curry;	z[ndata] = currz;	ndata++;
				}
			}
		else {
			if(arx->GetNext(&c1, &r1) && ary->GetNext(&c2, & r2) &&
				curr_data->GetValue(r1, c1, &currx) && curr_data->GetValue(r2, c2, &curry)) {
				x[ndata] = currx;	y[ndata] = curry;	ndata++;
				}
			}
		}
	//common initialization for parser tasks
	push_parser();		//make code reentrant
	init_table();		length = strlen(*par);
	//process parameters
	if(!(buffer = (char*)malloc(length+2))){
		clear_table();	pop_parser();
		if(arz) delete arz;
		free(y);	free(x);	delete arx;	delete ary;
		return 0;
		}
	strcpy(buffer, *par);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = 0;
	tab1 = sym_table;
	do {
		yyparse();
		}while(buff_pos < length);
	tab2 = sym_table;	free(buffer);	buffer =0L;
	for(nparam = 0, csr=tab2; csr != tab1; nparam++, csr = csr->next);
	parsym = (symrec**)malloc((nparam+1)*sizeof(symrec*));
	parval = (double**)malloc((nparam+1)*sizeof(double*));
	for(i = 0, csr=tab2; csr != tab1 && i < nparam; i++, csr = csr->next){
		parsym[i] = csr;	parval[i] = &csr->value.var;
		}
	//do iteratations to optimize fit
	lista = (int*)malloc(sizeof(int)*nparam);
	for(i = 0; i< nparam; i++) lista[i] = i;
	covar = dmatrix(1, nparam, 1, nparam);
	alpha = dmatrix(1, nparam, 1, nparam);
	alamda = -1.0;		itst = 0;
	mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
	if(!Check_MRQerror()) {
		for(itst = itst1 = 0, ochisq = chisq; itst < maxiter && chisq > conv && ochisq >= chisq && itst1 < 9; itst++) {
			ochisq = chisq;
			mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
			if(ochisq == chisq) itst1++;
			else itst1 = 0;
			}
		alamda = 0.0;
		mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
		Check_MRQerror();
		}
	for(i = nparam-1, j = k = l = 0; i >= 0; l = 0, i--) {
		if(k > 20) {
			if(tmp_txt[j-1] == ' ') j--;
			if(tmp_txt[j-1] == ';') j--;
			l = sprintf(tmp_txt+j, "\n");
			j += l;		k = 0;
			}
		l += sprintf(tmp_txt+j, "%s%s=%g;", j && k ? " " : "", parsym[i]->name, parsym[i]->value.var);
		j += l;			k += l;
		}
	free(*par);	*par = strdup(tmp_txt);
	//write back spreadsheet data if necessary
	buffer = *par;	length = strlen(buffer);
	do {
		yyparse();
		}while(buff_pos < length);
	buffer = 0L;
	free_dmatrix(alpha, 1, nparam, 1, nparam);
	free_dmatrix(covar, 1, nparam, 1, nparam);
	if(arz) delete arz;		if(z) free(z);
	free(y);	free(x);	delete arx;	delete ary;
	if(parval) free(parval);	if(parsym) free(parsym);
	clear_table();
	pop_parser();
	if(curr_data){
		curr_data->Command(CMD_CLEAR_ERROR, 0L, 0L);
		curr_data->Command(CMD_REDRAW, 0L, 0L);
		}
	return itst < maxiter ? itst+1 : maxiter;
}














