#ifndef _RHEOLEF_INTERPOLATE_H
#define _RHEOLEF_INTERPOLATE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
// 
// interpolation with a class-function or a function
//
#include "rheolef/field.h"
namespace rheolef { 

// --------------------------------------------------------------------------
// implementation details of the interpolate() function
// --------------------------------------------------------------------------
namespace details {

template <class T, class M, class Function>
field_basic<T,M>
interpolate_tag (const space_basic<T,M>& V, Function f, const T&)
{
  check_macro (V.valued_tag() == space_constant::scalar,
	"interpolate: invalid "<<V.valued()<<"-valued space and scalar function");
  typedef typename space_basic<T,M>::size_type size_type;
  field_basic<T,M> u (V, std::numeric_limits<T>::max());
  for (size_type idof = 0, ndof = V.ndof(); idof < ndof; idof++) {
    u.dof (idof) = V.momentum (f, idof);
  }
  return u;
}
template <class T, class M, class Function>
field_basic<T,M>
interpolate_tag (const space_basic<T,M>& V, Function f, const point_basic<T>&)
{
  check_macro (V.valued_tag() == space_constant::vector,
	"interpolate: invalid "<<V.valued()<<"-valued space and vector function");
  typedef typename space_basic<T,M>::size_type size_type;
  field_basic<T,M> u (V, std::numeric_limits<T>::max());
  size_type dim = V.get_geo().dimension();
  point_basic<T> value;
  for (size_type vec_idof = 0, vec_ndof = V.ndof()/dim; vec_idof < vec_ndof; vec_idof++) {
    value = V.vector_momentum (f, vec_idof);
    for (size_type i_comp = 0; i_comp < dim; i_comp++) {
      size_type idof = vec_idof + i_comp*vec_ndof;
      u.dof (idof) = value [i_comp];
    }
  }
  return u;
}

} // namespace details

// --------------------------------------------------------------------------
// interface of the interpolate() function
// --------------------------------------------------------------------------

/*Class:interpolate
NAME: @code{interpolate} - Lagrange interpolation of a function
@findex  interpolate
@clindex space
@clindex field

DESCRIPTION:
 The function @code{interpolation} implements the
 Lagrange interpolation of a function or a class-function.
SYNOPSYS:
 template <class Function>
 field interpolate (const space& Vh, const Function& f);
EXAMPLE:
@noindent
 The following code compute the Lagrange interpolation
 @code{pi_h_u} of u(x).
@example
  Float u(const point& x);
  ...
  geo omega("square");
  space Xh (omega, "P1");
  field pi_h_u = interpolate (Xh, u);
@end example
ADVANCED EXAMPLE:
  It is possible the replace the function @code{u} 
  by a variable of the @code{field} type that represents
  a picewise polynomial function: this invocation allows
  the reinterpolation of a field on another mesh or with
  another approximation.
@example
  geo omega2 ("square2");
  space X2h (omega2, "P1");
  field uh2 = interpolate (X2h, pi_h_u);
@end example
End: */
//<interpolate:
template <class T, class M, class Function>
inline
field_basic<T,M>
interpolate (const space_basic<T,M>& V, Function f)
//>interpolate:
{
  typedef typename Function::result_type  result_type ;
  return details::interpolate_tag (V, f, result_type());
}
// specialization for scalar-valued functions:
template <class T, class M>
inline
field_basic<T,M>
interpolate (const space_basic<T,M>& V, T (*f)(const point_basic<T>&))
{
  return details::interpolate_tag (V, f, T());
}
// specialization for vector-valued functions:
template <class T, class M>
inline
field_basic<T,M>
interpolate (const space_basic<T,M>& V, point_basic<T> (*f)(const point_basic<T>&))
{
  return details::interpolate_tag (V, f, point_basic<T>());
}
// specialization for re-interpoltion of fields (change of mesh, of approx, ect):
template<class T, class M>
field_basic<T,M>
interpolate (const space_basic<T,M>& V2h, const field_basic<T,M>& u1h);

}// namespace rheolef
#endif // _RHEOLEF_INTERPOLATE_H
