///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// main assembly algorithm
//
// author: Pierre.Saramito@imag.fr
//
// date: 13 march 1999
//
# include "rheolef/form.h"
namespace rheolef { 

template <class T, class M>
void
form_basic<T,M>::assembly (
    const form_element<T,M>&   form_e,
    const geo_basic<T,M>&      X_geo,
    const geo_basic<T,M>&      Y_geo,
    bool                       X_geo_is_background)
{
    using namespace std;
    //
    // use temporary asr = dynamic matrix structures
    // 
    asr<T,M> auu (_Y.iu_ownership(), _X.iu_ownership());
    asr<T,M> aub (_Y.iu_ownership(), _X.ib_ownership());
    asr<T,M> abu (_Y.ib_ownership(), _X.iu_ownership());
    asr<T,M> abb (_Y.ib_ownership(), _X.ib_ownership());
    ublas::matrix<T> ak;
    std::vector<size_type> dis_idy, dis_jdx;

    for (size_type ie = 0, ne = X_geo.size(); ie < ne; ie++) {
      const geo_element& Kx = X_geo [ie];
      const geo_element& Ky = Y_geo [ie];
      // --------------------------------
      // compute elementary matrix ak
      // --------------------------------
      if (X_geo_is_background) {
          form_e (Kx, ak);
      } else {
          form_e (Ky, ak); 
      }
      // --------------------------------
      // compute idofs
      // --------------------------------
      if (! form_e.is_on_band()) {
        _X.dis_idof (Kx, dis_jdx);
        _Y.dis_idof (Ky, dis_idy);
      } else {
        const geo_basic<T,M>& lambda = form_e.get_band().band();
        size_type L_ie = form_e.get_band().sid_ie2bnd_ie (ie);
        const geo_element& Lx = lambda [L_ie];
        const geo_element& Ly = lambda [L_ie];
        _X.dis_idof (Lx, dis_jdx);
        _Y.dis_idof (Ly, dis_idy);
      }
      // ------------------------------
      // assembly ak in sparse matrix a
      // ------------------------------
      for (  size_type loc_idof = 0, ny = ak.size1(); loc_idof < ny; loc_idof++) {
        for (size_type loc_jdof = 0, nx = ak.size2(); loc_jdof < nx; loc_jdof++) {

	    const T& value = ak (loc_idof, loc_jdof);
	    // reason to perform:
            // - efficient : lumped mass, structured meshes => sparsity increases
	    // reason to avoid:
            // - conserve the sparsity pattern, even with some zero coefs
	    //   usefull when dealing with solver::update_values()
	    // - also solver_pastix: assume sparsity pattern symmetry
	    //   and failed when a_ij=0 (skipped) and a_ji=1e-15 (conserved) i.e. non-sym pattern
	    //   note: this actual pastix wrapper limitation could be suppressed
	    if (value == T(0)) continue;

	    size_type dis_idof = dis_idy [loc_idof];
	    size_type dis_jdof = dis_jdx [loc_jdof];

	    size_type dis_iub = _Y.dis_iub (dis_idof);
	    size_type dis_jub = _X.dis_iub (dis_jdof);

	    if   (_Y.dis_is_blocked(dis_idof))
	      if (_X.dis_is_blocked(dis_jdof)) abb.dis_entry (dis_iub, dis_jub) += value;
              else                             abu.dis_entry (dis_iub, dis_jub) += value;
	    else 
	      if (_X.dis_is_blocked(dis_jdof)) aub.dis_entry (dis_iub, dis_jub) += value;
              else                             auu.dis_entry (dis_iub, dis_jub) += value;
        }
      }
    }
    //
    // since all is local, axx.dis_entry_assembly() compute only axx.dis_nnz
    //
    auu.dis_entry_assembly();
    aub.dis_entry_assembly();
    abu.dis_entry_assembly();
    abb.dis_entry_assembly();
    //
    // convert dynamic matrix asr to fixed-size one csr
    //
    _uu = csr<T,M>(auu);
    _ub = csr<T,M>(aub);
    _bu = csr<T,M>(abu);
    _bb = csr<T,M>(abb);
    //
    // set pattern dimension to uu:
    // => used by solvers, for efficiency: direct(d<3) or iterative(d=3)
    //
    size_type map_dim = X_geo.map_dimension();
    _uu.set_pattern_dimension (map_dim);
    _ub.set_pattern_dimension (map_dim);
    _bu.set_pattern_dimension (map_dim);
    _bb.set_pattern_dimension (map_dim);
    //
    // when form_e is symmetric, uu & bb are also
    // => used by solvers, for efficiency: LDL^t or LU, CG or GMRES
    //
    bool is_symm = form_e.is_symmetric();
    _uu.set_symmetry (is_symm);
    _bb.set_symmetry (is_symm);
}

}// namespace rheolef
