///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// TODO: merge into the "field" command !

/*Prog:mfield
NAME: @code{mfield} -- handle multi-field files
@pindex mfield
@cindex RHEOPATH environment variable
@fiindex @file{.mfield} multi-field 
@fiindex @file{.field} field 

SYNOPSIS:
  @example
  	mfield @{-@var{field-name}@}+ [-I@var{dir}] @var{filename}
  @end example
EXAMPLE:
  This command performs a @emph{velocity} graphical
  output for vector-valued fields:
  @example
	mfield -u square.mfield -velocity
	mfield -u square.mfield -deformation
	mfield -s square.mfield -proj -tensor
  @end example
  It is also convenient for selecting some scalar fields, e.g.
  to pipe to another processor
  @example
	mfield -psi square.mfield | field -
	mfield -u0  square.mfield | field -
	mfield -u1  square.mfield | field -
  @end example

DESCRIPTION:      
@findex catchmark iostream manipulator
  Read and output multiple finite element fields from file,
  in field text file format.
  Fields are preceded by a label, in the following @file{.mfield} file format:
  @example
      mfield
      1 2
      #u0
        @var{field u0...}
      #u1
        @var{field u1...}
      #p
        @var{field p...}
      #psi
        @var{field psi...}
  @end example
  Such labeled file format could be easily generated
  using the @code{catchmark} stream manipulator
  (@pxref{iorheo ialgorithm}).
  @example
      cout << catchmark("u")   << uh
           << catchmark("p")   << ph
           << catchmark("psi") << psih;
  @end example
OPTIONS:
    @table @code

    @itemx -I@var{dir}
	add @var{dir} to the RHEOPATH search path.
    	See also @ref{geo class} for RHEOPATH mechanism. 

    @itemx @var{filename}
	specifies the name of the file containing
      	the input field.

    @itemx -all
	all fields are selected for stdout printing.

    @itemx -
	read field on standard input instead on a file.

    @itemx -ndigit @var{int}
        number of digits used to print floating point values
        when using the @samp{-geo} option.
        Default depends upon the machine precision associated to the
        @code{Float} type.

    @itemx -verbose
        print messages related to graphic files created and
       command system calls (this is the default).

    @itemx -noverbose
        does not print previous messages.

    @itemx -round [@var{float}]
        Round the output, in order for the non-regression tests to
        be portable across platforms. Floating point are machine-dependent.
	The round value is optional. default value is 1e-10.

    @end table
RENDER OPTIONS:
    @table @code

    @itemx -velocity
        Render vector-valued fields as arrows 
	using @code{plotmtv}, @code{mayavi} or @code{vtk}.
	The the numeric extension is assumed:
	use e.g. @samp{-u} option instead of @samp{-u0}, @samp{-u1} options.

    @itemx -deformation
        Render vector-valued fields as deformed mesh
	using @code{plotmtv}, @code{mayavi} or @code{vtk}.
	The the numeric extension is assumed:
	use e.g. @samp{-u} option instead of @samp{-u0}, @samp{-u1} options.

    @itemx -tensor
        Render tensor-valued fields as ellipses
	using @code{mayavi}.
	The the numeric extension is assumed:
	use e.g. @samp{-tau} option instead of @samp{-tau00}, @samp{-tau01}...
	Warning: in development stage.

    @itemx -vscale @var{float}
        scale vector values by specified factor. Default value is 1.
    
    @itemx -proj
        Convert all selected fields to P1-continuous approximation by using a L2 projection.
	Only valid yet together with the @code{-tensor} option.
    @end table

RENDER TOOL SELECTION:
@toindex @code{mayavi}
@toindex @code{vtk}
@toindex @code{plotmtv}

    @table @code
    @itemx -mayavi
        use @code{mayavi} tool.
        This is the default for bi- and tridimensional geometries. 
    	The environment variable RHEOLEF_RENDER may be set to one of the choices below to override this default.
    
    @itemx -plotmtv
        use @code{plotmtv} plotting command.

    @itemx -vtk
        use @code{vtk} tool.
        Old style for 3d problems: superseted by @code{-mayavi}.

    @end table
OTHERS RENDER OPTIONS:
    @table @code
    @itemx -color
    @itemx -gray
    @itemx -black-and-white
	Use (color/gray scale/black and white) rendering.
	Color rendering is the default.
    @cindex anaglyph 3D stereo rendering
    @itemx -stereo
    @itemx -nostereo
  	rendering mode: suitable for red-blue anaglyph 3D stereoscopic glasses.
    @itemx -fill
	vector norm isolines intervals are filled with color.
    @itemx -nofill
	vector norm isolines are drawed by using colored mesh edges.
	This is the default.

    @itemx -verbose
	print messages related to graphic files created and
       command system calls (this is the default).

    @itemx -noverbose
	does not print previous messages.

    @itemx -clean
	clear temporary graphic files (this is the default).
    
    @itemx -noclean
	does not clear temporary graphic files.

    @itemx -execute
	execute graphic command (this is the default).
    
    @itemx -noexecute
	does not execute graphic command. Generates only 
	graphic files. This is usefull in conjuction with the
	@code{-noclean} command.
    @end table
TO DO:
    @noindent
    Check if selected fields appear in file.
    Otherwise, send some error message.
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
SEE ALSO:
    class "field(3)"
DATE:
    2 february 2001
End:
*/
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

void usage()
{
      cerr << "mfield: usage: mfield"
	   << " [{-fieldname}+|-all]"
	   << " [-velocity|-deformation|-tensor]"
	   << " [-mayavi|-gnuplot|-vtk|-plotmtv]"
	   << " [-[no]stereo|-[no]fill]"
	   << " [-color|-gray|-black-and-white]"
	   << " [-vscale float]"
	   << " [-autovscale]"
	   << " [-proj]"
	   << " [-[no]verbose] [-[no]clean] [-[no]execute]"
	   << " [-dom[ains]]"
	   << " [-round [float]]"
	   << " [-ndigit int]"
	   << " {-Igeodir}*"
	   << " -|filename"
	   << endl;
      exit (1);
}
// round error, for non-regression tests,
Float round_value = 1e-10;
inline Float my_round (const Float& x) { return (fabs(x) < round_value) ? Float(0) : x; }

void
put_graphic_tensor (const string fname, const string& file_name, 
	istream& is, bool do_domains, bool do_proj)
{
    check_macro (scatch (is, "\n#"+fname+"00"), "tensor component `" << fname << "00' not found.");
    field tau00;
    is >> tau00;
    const space& V0 = tau00.get_space();
    const geo&   g  = tau00.get_geo();
    if (g.dimension() <= 1) {
       error_macro ("unexpected vector in " << g.dimension() << "D.");
    }
    space V (g, V0.get_approx(), "tensor");
    field tau (V);
    tau(0,0) = tau00;
    check_macro (scatch (is, "\n#"+fname+"01"), "tensor component `" << fname << "01' not found.");
    tau(0,1) = field(V0, is);
    check_macro (scatch (is, "\n#"+fname+"11"), "tensor component `" << fname << "11' not found.");
    tau(1,1) = field(V0, is);
    if (g.dimension() == 3) {
       check_macro (scatch (is, "\n#"+fname+"02"), "tensor component `" << fname << "02' not found.");
       tau(0,2) = field(V0, is);
       check_macro (scatch (is, "\n#"+fname+"12"), "tensor component `" << fname << "12' not found.");
       tau(1,2) = field(V0, is);
       check_macro (scatch (is, "\n#"+fname+"22"), "tensor component `" << fname << "22' not found.");
       tau(2,2) = field(V0, is);
    }
    if (do_proj) { // proj to P1-C0 approx
      space S (tau.get_geo(), "P1", "tensor");
      form m  (S, S, "mass");
      form p  (V, S, "mass");
      ssk<Float> fact_m = ldlt(m.uu);
      field tau_p1(S);
      tau_p1.u = fact_m.solve((p*tau).u);
      tau = tau_p1;
    }
    if (do_domains) cout << domains;
    cout << setbasename(file_name) 
         << setprecision(numeric_limits<Float>::digits10)
         << tau;
}
void
put_graphic_vector (const string fname, const string& file_name, 
	istream& is, bool do_velocity, bool do_domains)
{
    string target = "\n#" + fname + itos(0);
    if (!scatch (is, target)) {
       error_macro ("vector component `" << fname << "0' not found.");
    }
    field u0;
    is >> u0;
    const space& V0 = u0.get_space();
    const geo&   g  = u0.get_geo();
    space V;
    if (g.dimension() <= 1) {
       error_macro ("unexpected vector in " << g.dimension() << "D.");
    }
    target = "\n#" + fname + itos(1);
    if (!scatch (is, target)) {
       error_macro ("vector component `" << fname << "1' not found.");
    }
    field u1(V0, is);
    field u;
    if (g.dimension() == 2) {
       u = fld_cat2(u0,u1);
    } else {
        target = "\n#" + fname + itos(2);
        if (!scatch (is, target)) {
           error_macro ("vector component `" << fname << "2' not found.");
        }
        field u2(V0, is);
        u = fld_cat3(u0,u1,u2);
    }
    if (do_velocity) {
	cout << velocity;
    } else {
	cout << deformation;
    }
    if (do_domains) cout << domains;
    cout << setbasename(file_name) 
         << setprecision(numeric_limits<Float>::digits10)
         << u;
}
void
put_text (bool select_all, bool do_round, const set<string>& field_name,
	const string& file_name, istream& is, ostream& os, int digits10) 
{
    os << setbasename(file_name) 
       << setprecision(digits10);

    if (select_all || field_name.size() >= 2) {
       os << "#!mfield" << endl;
    }
    while (is) {
        if (!scatch (is, "\n#")) {
           continue;
        }
        string target;
        is >> target;
        if (target.size() == 0 || target[0] == '!') {
	    // skip a "#!field" or a "#!mfield"
	    continue;
        }
        if (select_all || field_name.find(target) != field_name.end()) {
            field u;
            is >> u;
            if (do_round) {
                u = transform(u,my_round);
    	    }
            os << catchmark(target) << u;
        }
    }
}
int main(int argc, char**argv)
{
    if (argc <= 1) usage();
    clog << verbose; bool bverbose = true;
    int digits10 = numeric_limits<Float>::digits10;
    bool on_stdin = false;
    bool select_all = false;
    bool do_round = false;
    bool do_velocity = false;
    bool do_domains = false;
    bool do_deformation = false;
    bool do_tensor = false;
    bool do_proj = false;
    typedef enum { no_render, vtk_render, plotmtv_render, mayavi_render, gnuplot_render } render_type;
    render_type render = no_render;
    cout << setvectorscale(1.0);
    cout << nofill;

    string file_name;
    set<string> field_name;
    string last_field_name;
    for (int i = 1; i < argc; i++) {

             if (strcmp (argv[i], "-ndigit") == 0)    { digits10 = atoi(argv[++i]); }
        else if (strcmp (argv[i], "-noverbose") == 0) { bverbose = false; clog << noverbose; }
        else if (strcmp (argv[i], "-verbose") == 0)   { bverbose = true; clog << verbose; }
        else if (strcmp (argv[i], "-all") == 0)       { select_all = true; }

        else if (strcmp (argv[i], "-velocity") == 0)    { do_velocity = true; }
        else if (strcmp (argv[i], "-deformation") == 0) { do_deformation = true; }
        else if (strcmp (argv[i], "-tensor") == 0)      { do_tensor = true; }

        else if (strcmp (argv[i], "-proj") == 0)      { do_proj = true; } 
	else if (strcmp ("-domains", argv [i]) == 0||
		 strcmp ("-dom",     argv [i]) == 0)  { do_domains = true; }

        else if (strcmp (argv[i], "-mayavi") == 0)    { render = mayavi_render; }
        else if (strcmp (argv[i], "-gnuplot") == 0)   { render = gnuplot_render; }
        else if (strcmp (argv[i], "-vtk") == 0)       { render = vtk_render; }
        else if (strcmp (argv[i], "-plotmtv") == 0)   { render = plotmtv_render; }

        else if (strcmp (argv[i], "-stereo") == 0)   { cout << stereo; }
        else if (strcmp (argv[i], "-nostereo") == 0) { cout << nostereo; }
        else if (strcmp (argv[i], "-fill") == 0)     { cout << fill; }
        else if (strcmp (argv[i], "-nofill") == 0)   { cout << nofill; }
        else if (strcmp (argv[i], "-autovscale") == 0)   { cout << autovscale; }
        else if (strcmp (argv[i], "-grid") == 0)   { cout << grid; }

        else if (strcmp (argv[i], "-color") == 0)           { cout << color; }
        else if (strcmp (argv[i], "-gray") == 0)            { cout << gray; }
        else if (strcmp (argv[i], "-black-and-white") == 0) { cout << black_and_white; }

        else if (strcmp (argv[i], "-verbose") == 0)   { clog << verbose; }
        else if (strcmp (argv[i], "-noverbose") == 0) { clog << noverbose; }
        else if (strcmp (argv[i], "-clean") == 0)     { clog << clean; }
        else if (strcmp (argv[i], "-noclean") == 0)   { clog << noclean; }
        else if (strcmp (argv[i], "-execute") == 0)   { clog << execute; }
        else if (strcmp (argv[i], "-noexecute") == 0) { clog << noexecute; }
        else if (strcmp (argv[i], "-round") == 0)     { 
	    do_round = true;
            if (i+1 < argc && is_float(argv[i+1])) {
                round_value = to_float (argv[++i]);
	    }
        } else if (strcmp (argv[i], "-vscale") == 0)   {

            if (i+1 == argc || !is_float(argv[i+1])) usage();
            Float vscale = to_float (argv[++i]);
            cout << setvectorscale(vscale);

	} else if (argv [i][0] == '-' && argv [i][1] == 'I') {

	    append_dir_to_rheo_path (argv[i]+2);
        
	} else if (strcmp (argv [i], "-") == 0) {
	    
	    on_stdin = true;
	    file_name = "output";

	} else if (argv [i][0] == '-') {

	    last_field_name = string(argv[i]+1); // skip '-'
	    field_name.insert(last_field_name);

	} else {
            // input geo on file
            file_name = delete_suffix (delete_suffix (argv[i], "gz"), "mfield");
            string dir_name = get_dirname (argv[i]);
	    prepend_dir_to_rheo_path (dir_name);
        }
    }
    if (!on_stdin && file_name == "") {
	cerr << "mfield: no input specified" << endl;
	usage();
    }
    if (!select_all && field_name.size() ==0) {
    	if (!on_stdin) {
		cerr << "mfield: no field name specified, using filename " << get_basename(delete_suffix (delete_suffix (file_name, "gz"), "mfield")) << endl;
		last_field_name = get_basename(delete_suffix (delete_suffix (file_name, "gz"), "mfield"));
		field_name.insert(last_field_name);
	} else {
		cerr << "mfield: no field name specified" << endl;
		usage();
	}
    }
    if (do_tensor) {
        cout << mayavi;
        if (on_stdin) {
            put_graphic_tensor (last_field_name, file_name, cin, do_domains, do_proj);
	} else {
            irheostream is(file_name, "mfield");
            if (!is) error_macro("\"" << file_name << "[.mfield[.gz]]\" not found.");
            put_graphic_tensor (last_field_name, file_name, is, do_domains, do_proj);
        }
	return 0;
    }
    if (do_velocity || do_deformation) {
	if (render == no_render) {
	    char* env_render = getenv("RHEOLEF_RENDER");
	    if (env_render!=NULL) 
	    {
		if (strcmp (env_render, "mayavi") == 0)    	{ render = mayavi_render; }
		else if (strcmp (env_render, "gnuplot") == 0)	{ render = gnuplot_render; }
		else if (strcmp (env_render, "vtk") == 0)	{ render = vtk_render; }
		else if (strcmp (env_render, "plotmtv") == 0)	{ render = plotmtv_render; }
		else render = mayavi_render;
	    }
	    else render = mayavi_render;
	}
	if (render == mayavi_render) {
            cout << mayavi;
	} else if (render == plotmtv_render) {
            cout << plotmtv;
        } else if (render == gnuplot_render) {
	    cout << gnuplot;
	} else {
            cout << vtk;
        }
        if (on_stdin) {
            put_graphic_vector (last_field_name, file_name, cin, do_velocity, do_domains);
	} else {
            irheostream is(file_name, "mfield");
            if (!is) error_macro("\"" << file_name << "[.mfield[.gz]]\" not found.");
            put_graphic_vector (last_field_name, file_name, is, do_velocity, do_domains);
        }
	return 0;
    }
    // do text output
    if (on_stdin) {
        // input field on standard input
        if (bverbose) clog << "! load field on stdin" << endl;
        put_text(select_all, do_round, field_name,file_name,cin,cout,digits10);
    } else {
        irheostream is(file_name, "mfield");
        if (!is) error_macro("\"" << file_name << "[.mfield[.gz]]\" not found.");
        put_text(select_all, do_round, field_name,file_name,is,cout,digits10);
    }
    return 0;
}
