///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/geo.h"
#include "rheolef/geo_domain.h"
#include "rheolef/geo_connectivity.h"
#include "rheolef/rheostream.h"

namespace rheolef {

// =========================================================================
// geo_base_rep members
// =========================================================================
template <class T, class M>
geo_base_rep<T,M>::~geo_base_rep()
{
  warning_macro ("dstor geo_base_rep: name="<<name()<<", dim="<<dimension()<<", map_dim="<<map_dimension()<<" size="<<size());
}
template <class T, class M>
geo_base_rep<T,M>::geo_base_rep (const geo_base_rep<T,M>& omega2)
{
  error_macro ("physical copy of geo_base_rep: dim="<<omega2.dimension()<<", dis_size="<<omega2.dis_size());
}
template <class T, class M>
const domain_indirect_basic<M>&
geo_base_rep<T,M>::get_domain_indirect (const std::string& name) const
{
	for (typename std::vector<domain_indirect_basic<M> >::const_iterator
			iter = _domains.begin(),
			last = _domains.end(); iter != last; iter++) {
		const domain_indirect_basic<M>& dom = *iter;
		if (name == dom.name()) return dom;
	}
	error_macro ("undefined domain \""<<name<<"\" in mesh \"" << _name << "\"");
	return *(_domains.begin()); // not reached
}
// =========================================================================
// geo_rep<seq> members
// =========================================================================
/// @brief io for geo
template <class T>
	idiststream&
geo_rep<T,sequential>::get (idiststream& ips)
{
warning_macro ("get...");
  using namespace std;
  istream& is = ips.is();
warning_macro ("get [0]");
  check_macro (is.good(), "bad input stream for geo.");

warning_macro ("get [0.b]");
  if (!scatch(is,"\nmesh"))
  	error_macro("input stream does not contains a geo.");
  //
  // 1) get header
  //
warning_macro ("get [1]");
  size_type n_vert;
  size_type n_elt;
  size_type n_edg = 0;
  size_type n_fac = 0;

  base::_name = "unnamed";

  is >> base::_version
  	>> base::_dimension
  	>> n_vert
  	>> n_elt;
  if (base::_version < 2) {
  	warning_macro ("mesh version < 2 is obsolete");
  } else {
  	if (base::_dimension >= 3) {
  		is >> n_fac;
  	}
  	if (base::_dimension >= 2) {
  		is >> n_edg;
  	}
  }
  //
  // 2) get coordinates
  //
warning_macro ("get [2]");
  base::_vertex.resize (n_vert);
  base::_vertex.get_values (ips, _point_get<T>(geo_base_rep<T,sequential>::_dimension));
  base::_geo_element[0].resize (base::_vertex.ownership());
  size_type first_iv = base::_vertex.ownership().first_index();
  for (size_type iv = 0; iv < n_vert; iv++) {
  	base::_geo_element[0][iv] = geo_element_p(first_iv + iv);
  	geo_element& P = base::_geo_element[0][iv];
  	P.set_ios_dis_ie (first_iv + iv);
  	P.set_dis_ie     (first_iv + iv);
  }
  check_macro (is.good(), "bad input stream for geo.");
  //
  // 3) get elements
  //
warning_macro ("get [3]");
  polymorphic_array<geo_element,sequential> elt (n_elt);
  elt.get_values (ips);
  base::_map_dimension = 0;
  for (size_type ie = 0; ie < n_elt; ie++) {
  	geo_element& K = elt[ie];
  	base::_map_dimension = std::max (K.dimension(), base::_map_dimension);
  	K.set_ios_dis_ie (ie);
  	K.set_dis_ie (ie);
  }
  if (base::_map_dimension == base::_dimension) {
  	base::_geo_element[base::_map_dimension] = elt;
  }
  //
  // 4) get faces & edges
  //
warning_macro ("get [4]");
  if   (base::_version  >= 2 && base::_dimension >= 3) {
  	base::_geo_element[2].resize (n_fac);
  	base::_geo_element[2].get_values (ips);
  	for (size_type ifac = 0; ifac < n_fac; ifac++) {
  		geo_element& F = base::_geo_element[2] [ifac];
  		F.set_ios_dis_ie (ifac);
  		F.set_dis_ie     (ifac);
  	}
  }
  if   (base::_version  >= 2 && base::_dimension >= 2) {
    	base::_geo_element[1].resize (n_edg);
    	base::_geo_element[1].get_values (ips);
    	for (size_type iedg = 0; iedg < n_edg; iedg++) {
      		geo_element& E = base::_geo_element[1] [iedg];
      		E.set_ios_dis_ie (iedg);
      		E.set_dis_ie     (iedg);
    	}
  }
  //
  // 5) get domain, until end-of-file
  //
warning_macro ("get [5]");
  vector<set<size_type> > ball [4];
  domain_indirect_basic<sequential> dom;
  while (dom.get (ips, *this, ball)) {
warning_macro ("get [5] domain...");
     base::_domains.push_back (dom);
warning_macro ("get [5] domain done: " << dom.name());
  }
  //
  // 6) size by
  //
warning_macro ("get [6]");
  base::reset_size_by();
warning_macro ("get done");
  return ips;
}
template <class T, class M>
void
geo_base_rep<T,M>::reset_size_by()
{
  std::fill (_size_by_dimension, _size_by_dimension+4, 0);
  std::fill (_size_by_variant,   _size_by_variant+reference_element::max_size, 0);
  for (size_type dim = 0; dim <= _map_dimension; dim++) {
    _size_by_dimension [dim] = _geo_element [dim].ownership().size();
    if (dim <= 1) {
      _size_by_variant [dim] = _size_by_dimension [dim];
      continue;
    }
    const polymorphic_array<geo_element,M>& ge = _geo_element[dim]; // otherwise bug with physical copy
    for (typename polymorphic_array<geo_element,M>::const_iterator iter = ge.begin(), last = ge.end();
		iter != last; iter++) {
  	const geo_element& K = (*iter);
        _size_by_variant [K.variant()]++;
    }
  }
}
template <class T>
odiststream&
geo_rep<T,sequential>::put (odiststream& ops)  const {
  using namespace std;
  size_type n_vert = base::_vertex.dis_size ();
  size_type n_elt  = base::_geo_element[base::_map_dimension].dis_size ();
  //
  // put header
  //
  ops << "#!geo" << endl
      << endl
      << "mesh" << endl
      << geo_base_rep<T,sequential>::_version << " " 
      << geo_base_rep<T,sequential>::_dimension << " " 
      << n_vert << " "
      << n_elt;
  if   (base::_version >= 2) {
    if (base::_dimension >= 3) {
      ops << " " << base::_geo_element[2].dis_size();
    }
    if (base::_dimension >= 2) {
      ops << " " << base::_geo_element[1].dis_size();
    }
  }
  ops << endl << endl;
  //
  // put vertices & elements
  //
  geo_base_rep<T,sequential>::_vertex.put_values (ops, _point_put<T>(geo_base_rep<T,sequential>::_dimension));
  ops << endl;
  base::_geo_element[base::_map_dimension].put_values (ops);
  ops << endl;
  //
  // put faces & edges
  //
  if   (base::_version >= 2 && base::_dimension >= 3) {
        base::_geo_element[2].put_values (ops); 
  }
  if   (base::_version >= 2 && base::_dimension >= 2) {
        base::_geo_element[1].put_values (ops); 
  }
  //
  // put domains
  //
  for (typename std::vector<domain_indirect_basic<sequential> >::const_iterator
        iter = base::_domains.begin(), last = base::_domains.end();
	iter != last; ++iter) {
    ops << endl;
    (*iter).put (ops);
  }
  return ops;
}
template <class T>
void
geo_rep<T,sequential>::dump (std::string name)  const {
  geo_base_rep<T,sequential>::_vertex.dump        (name + "-vert");
  base::_geo_element[base::_map_dimension].dump(name + "-elem");
}
// ----------------------------------------------------------------------------
// read from file
// ----------------------------------------------------------------------------
template <class T>
void
geo_rep<T,sequential>::load (std::string filename) 
{
  idiststream ips;
  ips.open (filename, "geo");
  check_macro(ips.good(), "\"" << filename << "[.geo[.gz]]\" not found.");
  get (ips);
  std::string root_name = delete_suffix (delete_suffix(filename, "gz"), "geo");
  std::string name = get_basename (root_name);
  base::_name = name;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class geo_base_rep<Float,sequential>;
#ifdef _RHEOLEF_HAVE_MPI
template class geo_base_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

template class geo_rep<Float,sequential>;

template geo_basic<Float,sequential> compact (const geo_basic<Float,sequential>&);

} // namespace rheolef
