#ifndef _RHEOLEF_DOMAIN_MASK_H
#define _RHEOLEF_DOMAIN_MASK_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/array.h"
#include "rheolef/geo_element.h"

namespace rheolef {

// =====================================================================
// 1) domain pair = pair (sign orientation ; size_t element_index)
// =====================================================================
// TODO: compact the sign bit and size_t in a long int
struct domain_pair_type {

// typedefs:

    typedef array<size_t>::size_type      size_type;
    typedef geo_element::orientation_type orientation_type;

// allocators:

    domain_pair_type () : _pos(true), _ige (std::numeric_limits<size_type>::max()) {}
    domain_pair_type (orientation_type orient, size_type ige) : _pos(orient > 0), _ige(ige) {}

// accessors:

    orientation_type orientation() const { return (_pos ?  1 : - 1); }
    size_type        index()       const { return _ige; }

// modifiers:

    void set_orientation (orientation_type orient) { _pos = (orient > 0); }
    void set_index       (size_type ige)           { _ige = ige; }
    void set (orientation_type orient, size_type ige) {
        set_orientation (orient);
        set_index       (ige);
    }

// i/o:

    friend std::istream&  operator>> (std::istream& is, domain_pair_type& x);
    friend std::ostream&  operator<< (std::ostream& os, const domain_pair_type& x);
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) { ar & _pos; ar & _ige; }

// data:
protected:
    bool      _pos;
    size_type _ige;
};

} // namespace rheolef

#ifdef _RHEOLEF_HAVE_MPI
// =====================================================================
// Some serializable types, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous 
// copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <> struct is_mpi_datatype<rheolef::domain_pair_type> : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

namespace rheolef {

// foward declarations:
template <class U, class M> class geo_rep;

// ==================================================================================
// 2) domain_indirect
// 2.1) representation: domain_indirect_base_rep<M> and domain_indirect_rep<M>
// ==================================================================================
template <class M>
class domain_indirect_base_rep : public array<domain_pair_type,M> {
public:

// typedefs:

    typedef array<domain_pair_type,M> base;
    typedef typename domain_pair_type::size_type        size_type;
    typedef typename domain_pair_type::orientation_type orientation_type;
    typedef typename base::iterator                     iterator_ioige;
    typedef typename base::const_iterator               const_iterator_ioige;

// allocators:

    domain_indirect_base_rep();
    void resize (size_type n) { base::resize (n); }

// accessors & modifiers:

    size_type             size()     const { return base::size(); }
    size_type             dis_size() const { return base::dis_size(); }

    const_iterator_ioige ioige_begin() const { return base::begin(); }
    const_iterator_ioige ioige_end()   const { return base::end(); }
          iterator_ioige ioige_begin()       { return base::begin(); }
          iterator_ioige ioige_end()         { return base::end(); }

    const domain_pair_type& oige (size_type ioige) const { return base::operator[] (ioige); }

    std::string name ()    const { return _name; }
    size_type map_dimension () const { return _map_dim; }
    void set_name (std::string name) { _name = name; }
    void set_map_dimension (size_type map_dim) { _map_dim = map_dim; }

protected:
// data:
    std::string           _name;
    size_type             _map_dim;
};
template <class M>
inline
domain_indirect_base_rep<M>::domain_indirect_base_rep()
 : array<domain_pair_type,M>(),
   _name(),
   _map_dim(0)
{
}
// ---------------------------------------------------------------------
template <class M> class domain_indirect_rep {};

template<>
class domain_indirect_rep<sequential> : public domain_indirect_base_rep<sequential> {
public:

// typedefs:

    typedef domain_indirect_base_rep<sequential>  base;
    typedef base::size_type              size_type;
    typedef base::iterator_ioige         iterator_ioige;
    typedef base::const_iterator_ioige   const_iterator_ioige;
    typedef base::orientation_type       orientation_type;

// allocators:

    domain_indirect_rep () : domain_indirect_base_rep<sequential>() {}
    void resize (size_type n) { base::resize (n); }

// accessors:

    size_type size()       const { return base::size(); }
    size_type dis_size()   const { return base::dis_size(); }

    const_iterator_ioige ioige_begin() const { return base::ioige_begin(); }
    const_iterator_ioige ioige_end()   const { return base::ioige_end(); }
          iterator_ioige ioige_begin()       { return base::ioige_begin(); }
          iterator_ioige ioige_end()         { return base::ioige_end(); }

    const domain_pair_type& oige (size_type ioige) const {
	return base::oige (ioige); }

    void set_name (std::string name)   { base::set_name(name); }
    void set_map_dimension (size_type map_dim) { base::set_map_dimension(map_dim); }
    std::string name ()    const { return base::name(); }
    size_type map_dimension () const { return base::map_dimension(); }

// i/o:

    template <class U>
    idiststream& get (idiststream& ips, const geo_rep<U,sequential>& omega,
    	std::vector<std::set<size_type> > ball[4]);

    odiststream& put (odiststream& ops) const;
};
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template<>
class domain_indirect_rep<distributed> : public domain_indirect_base_rep<distributed> {
public:

// typedefs:

    typedef domain_indirect_base_rep<distributed>          base;
    typedef base::size_type              size_type;
    typedef base::iterator_ioige         iterator_ioige;
    typedef base::const_iterator_ioige   const_iterator_ioige;
    typedef base::orientation_type       orientation_type;

// allocators:

    domain_indirect_rep();

// accessors & modifiers:

    size_type size()       const { return base::size(); }
    size_type dis_size()   const { return base::dis_size(); }

    const domain_pair_type& oige (size_type ioige) const {
	return base::oige (ioige); }

    void set_name (std::string name)   { base::set_name(name); }
    void set_map_dimension (size_type map_dim) { base::set_map_dimension(map_dim); }
    std::string name ()    const       { return base::name(); }
    size_type map_dimension () const       { return base::map_dimension(); }

// distributed specific acessors:

    const distributor& ini_ownership () const { return _ini_ioige2dis_ioige.ownership(); }
    size_type ioige2ini_dis_ioige (size_type ioige)     const { return _ioige2ini_dis_ioige [ioige]; }
    size_type ini_ioige2dis_ioige (size_type ini_ioige) const { return _ini_ioige2dis_ioige [ini_ioige]; }

// i/o:

    template <class U>
    idiststream& get (idiststream& ips, const geo_rep<U,distributed>& omega);
    template <class U>
    odiststream& put (odiststream& ops, const geo_rep<U,distributed>& omega) const;

protected:
    template <class U1,class M1> friend class geo_rep; // for geo_rep::build_from_domain
// data:
    array<size_type,distributed>        _ioige2ini_dis_ioige;
    array<size_type,distributed>        _ini_ioige2dis_ioige;
    size_type		                _ini_size_by_variant [reference_element::max_size];
};
inline
domain_indirect_rep<distributed>::domain_indirect_rep()
  : domain_indirect_base_rep<distributed>(),
    _ioige2ini_dis_ioige(),
    _ini_ioige2dis_ioige(),
    _ini_size_by_variant()
{
}
#endif // _RHEOLEF_HAVE_MPI
// ====================================================================
// 2.2) wrapper: domain_indirect_basic<M>
// ====================================================================
/*Class:domain_indirect
NAME: @code{domain_indirect} - a named part of a finite element mesh
@cindex  mesh boundary
@clindex domain_indirect
DESCRIPTION:
  @noindent
  The @code{domain_indirect} class defines a container for a part of a
  finite element mesh.
  This describes the connectivity of edges or faces.
  This class is usefull for boundary condition setting.
IMPLEMENTATION NOTE:
  The @code{domain} class is splitted into two parts.
  The first one is the @code{domain_indirect} class, that contains the main
  renumbering features: it acts as a indirect on a @code{geo} class(@pxref{geo class}).
  The second one is the @code{domain} class, that simply contains two
  smart_pointers: one on a @code{domain_indirect} and the second on the
  @code{geo} where renumbering is acting.
  Thus, the domain class develops a complete @code{geo}-like interface, 
  via the @code{geo_abstract_rep} pure virtual class derivation,
  and can be used by the @code{space} class (@pxref{space class}).
  The split between domain_indirect and domain is necessary,
  because the @code{geo} class contains a list of domain_indirect.
  It cannot contains a list of @code{domain} classes, that refers
  to the geo class itself: a loop in reference counting
  leads to a blocking situation in the automatic deallocation.

DATE: 12 may 1997
End:
*/
/// @brief the finite element boundary domain
template <class M = rheo_default_memory_model>
class domain_indirect_basic {
public:
};
typedef domain_indirect_basic<rheo_default_memory_model> domain_indirect;
// ---------------------------------------------------------------------
//<verbatim:
template <>
class domain_indirect_basic<sequential> : public smart_pointer<domain_indirect_rep<sequential> > {
public:

// typedefs:

    typedef domain_indirect_rep<sequential> rep;
    typedef smart_pointer<rep>        base;
    typedef rep::size_type            size_type;
    typedef rep::iterator_ioige       iterator_ioige;
    typedef rep::const_iterator_ioige const_iterator_ioige;

// allocators:

    domain_indirect_basic ();
    void resize (size_type n);

// accessors:

    size_type size()     const;
    size_type dis_size() const;
    const distributor& ownership() const;

    const_iterator_ioige ioige_begin() const;
    const_iterator_ioige ioige_end()   const;
          iterator_ioige ioige_begin();
          iterator_ioige ioige_end();

    const domain_pair_type& oige (size_type ioige) const;

    void set_name (std::string name);
    void set_map_dimension (size_type map_dim);
    std::string name ()    const;
    size_type map_dimension () const;

// i/o:

    odiststream& put (odiststream&) const;
    template <class T>
    idiststream& get (idiststream& ips, const geo_rep<T,sequential>& omega,
    	std::vector<std::set<size_type> > ball[4]);
};
//>verbatim:
inline
domain_indirect_basic<sequential>::domain_indirect_basic ()
 : smart_pointer<domain_indirect_rep<sequential> > (new_macro(domain_indirect_rep<sequential>))
{
}
inline
void
domain_indirect_basic<sequential>::resize (size_type n)
{
  return base::data().resize (n);
}
inline
domain_indirect_basic<sequential>::size_type
domain_indirect_basic<sequential>::size() const
{
  return base::data().size();
}
inline
domain_indirect_basic<sequential>::size_type
domain_indirect_basic<sequential>::dis_size() const
{
  return base::data().dis_size();
}
inline
const distributor&
domain_indirect_basic<sequential>::ownership() const
{
  return base::data().ownership();
}
inline
const domain_pair_type&
domain_indirect_basic<sequential>::oige (size_type ioige) const
{
  return base::data().oige (ioige);
}
inline
domain_indirect_basic<sequential>::const_iterator_ioige
domain_indirect_basic<sequential>::ioige_begin() const
{
  return base::data().ioige_begin();
}
inline
domain_indirect_basic<sequential>::const_iterator_ioige
domain_indirect_basic<sequential>::ioige_end() const
{
  return base::data().ioige_end();
}
inline
domain_indirect_basic<sequential>::iterator_ioige
domain_indirect_basic<sequential>::ioige_begin()
{
  return base::data().ioige_begin();
}
inline
domain_indirect_basic<sequential>::iterator_ioige
domain_indirect_basic<sequential>::ioige_end()
{
  return base::data().ioige_end();
}
inline
std::string
domain_indirect_basic<sequential>::name() const
{
  return base::data().name();
}
inline
domain_indirect_basic<sequential>::size_type
domain_indirect_basic<sequential>::map_dimension() const
{
  return base::data().map_dimension();
}
inline
void
domain_indirect_basic<sequential>::set_name(std::string name)
{
  return base::data().set_name (name);
}
inline
void
domain_indirect_basic<sequential>::set_map_dimension (size_type map_dim)
{
  return base::data().set_map_dimension (map_dim);
}
inline
odiststream&
domain_indirect_basic<sequential>::put (odiststream& ops) const
{
  return base::data().put (ops);
}
template<class T>
inline
idiststream&
domain_indirect_basic<sequential>::get (
    idiststream&                       ips,
    const geo_rep<T,sequential>&      omega,
    std::vector<std::set<size_type> > ball[4])
{
  return base::data().template get (ips, omega, ball);
}
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
//<verbatim:
template <>
class domain_indirect_basic<distributed> : public smart_pointer<domain_indirect_rep<distributed> > {
public:

// typedefs:

    typedef domain_indirect_rep<distributed>  rep;
    typedef smart_pointer<rep>            base;
    typedef rep::size_type                size_type;

// allocators:

    domain_indirect_basic ();

// accessors/modifiers:

    size_type size()     const;
    size_type dis_size() const;
    const distributor& ownership() const;

    const domain_pair_type& oige (size_type ioige) const;

    void set_name (std::string name);
    void set_map_dimension (size_type map_dim);
    std::string name () const;
    size_type map_dimension () const;

// distributed specific acessors:

    const distributor& ini_ownership() const;
    size_type ioige2ini_dis_ioige (size_type ioige) const;
    size_type ini_ioige2dis_ioige (size_type ini_ioige) const;

// i/o:

    template <class T>
    idiststream& get (idiststream& ips, const geo_rep<T,distributed>& omega);
    template <class T>
    odiststream& put (odiststream& ops, const geo_rep<T,distributed>& omega) const;
};
//>verbatim:

inline
domain_indirect_basic<distributed>::domain_indirect_basic ()
 : smart_pointer<rep> (new_macro(rep))
{
}
inline
domain_indirect_basic<distributed>::size_type
domain_indirect_basic<distributed>::size() const
{
  return base::data().size();
}
inline
domain_indirect_basic<distributed>::size_type
domain_indirect_basic<distributed>::dis_size() const
{
  return base::data().dis_size();
}
inline
const distributor&
domain_indirect_basic<distributed>::ownership() const
{
  return base::data().ownership();
}
template<class T>
inline
idiststream&
domain_indirect_basic<distributed>::get (
    idiststream&                     ips,
    const geo_rep<T,distributed>&   omega)
{
  return base::data().template get (ips, omega);
}
template<class T>
inline
odiststream&
domain_indirect_basic<distributed>::put (
    odiststream&                   ops,
    const geo_rep<T,distributed>& omega) const
{
  return base::data().template put (ops, omega);
}
inline
std::string
domain_indirect_basic<distributed>::name() const
{
  return base::data().name();
}
inline
domain_indirect_basic<distributed>::size_type
domain_indirect_basic<distributed>::map_dimension() const
{
  return base::data().map_dimension();
}
inline
void
domain_indirect_basic<distributed>::set_name(std::string name)
{
  return base::data().set_name (name);
}
inline
void
domain_indirect_basic<distributed>::set_map_dimension (size_type map_dim)
{
  return base::data().set_map_dimension (map_dim);
}
inline
const domain_pair_type&
domain_indirect_basic<distributed>::oige (size_type ioige) const
{
  return base::data().oige (ioige);
}
inline
const distributor&
domain_indirect_basic<distributed>::ini_ownership() const
{
  return base::data().ini_ownership();
}
inline
domain_indirect_basic<distributed>::size_type
domain_indirect_basic<distributed>::ioige2ini_dis_ioige (size_type ioige) const
{
    return base::data().ioige2ini_dis_ioige (ioige);
}
inline
domain_indirect_basic<distributed>::size_type
domain_indirect_basic<distributed>::ini_ioige2dis_ioige (size_type ini_ioige) const
{
    return base::data().ini_ioige2dis_ioige (ini_ioige);
}

#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
#endif // _RHEOLEF_DOMAIN_MASK_H
