///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:geo
NAME: @code{geo} - plot a finite element mesh 
@pindex geo
@cindex plotting
@cindex plotting mesh
@cindex mesh graphic representation
@clindex geo
@clindex rheostream
@fiindex @file{.ps} postscript
@fiindex @file{.tex} latex
@fiindex @file{.fig} Fig, xfig
@toindex @code{gnuplot}
@toindex @code{xfig}
@toindex @code{plotmtv}
@toindex @code{vtk}
@toindex @code{mayavi}
@toindex @code{x3d}
@toindex @code{PlotM}
SYNOPSIS:
  @example
	geo @var{options} @var{mesh}[.geo]
  @end example
DESCRIPTION:       
  @noindent
  Plot a 2d or 3d finite element mesh.
  
  @noindent
  The input file format is described with the @code{geo}
  class manual (@pxref{geo class}).

EXAMPLE: 
  Enter commands as:
  @example
    	geo -vtk bielle.geo
    	geo -vtk -fill bielle.geo
  @end example
  or, for 2D meshes:
  @example
    	geo -plotmtv square.geo
  @end example

INPUT FILE SPECIFICATION:
  @table @code
  @itemx @var{filename}
  	specifies the name of the file containing
      	the input mesh. The ".geo" extension is assumed.
	
@cindex RHEOPATH environment variable

  @itemx -I@var{geodir} 
      	add @var{geodir} to the mesh search path.
      	This mechanism initializes a search path given by the environment variable
      	@samp{RHEOPATH}. If the environment variable 
	@samp{RHEOPATH} is not set, the
      	default value is the current directory
	(@pxref{geo class}).

  @itemx -
  	read mesh on standard input instead on a file.
  @itemx -name
	when mesh comes from standard input, the mesh name
	is not known and is set to "output" by default.
	This option allows to change this default.
	Useful when dealing with output formats (graphic, format conversion)
	that creates auxilliary files, based on this name.
  @end table

INQUIRE OPTIONS:
  @table @code
  @itemx -size 
     print the number of elements in the mesh and exit.
  @itemx -n-node 
     print the number of nodes in the mesh and exit.
  @itemx -hmin
     print the smallest edge length in the mesh and exit.
  @itemx -hmax
     print the largest edge length in the mesh and exit.
  @itemx -xmin
     print the lower-left-bottom vertice in the mesh and exit.
  @itemx -xmax
     print the upper-right-top vertice in the mesh and exit.
  @end table

RENDER OPTIONS:
@toindex @code{plotmtv}
@toindex @code{gnuplot}
@toindex @code{vtk}
@toindex @code{mayavi}
@toindex @code{x3d}
@toindex @code{PlotM}
  The default render could also specified by using the RHEOLEF_RENDER environment variable.
  @table @code
  @itemx -mayavi
    	use Mayavi.
	This is the default.
  @itemx -plotmtv
  	use plotmtv tool.
  @itemx -gnuplot
  	use gnuplot tool.
  @itemx -vtk
    	use Visualization Toolkit.
  @itemx -x3d
    	use x3d visualization tool.
  @itemx -atom
  	use PlotM atom representation (from lassp, Cornell).
  @end table
GRAPHIC OPTIONS:
  @table @code
@cindex @code{stereo 3D rendering}
  @itemx -color
	use colors. This is the default.
  @itemx -black-and-white
	Option only available with @code{mayavi}.
  @itemx -stereo
  	Rendering mode suitable for red-blue anaglyph 3D stereoscopic glasses.
	Option only available with @code{mayavi}.
  @itemx -grid
    	rendering mode. This is the default.
  @itemx -fill
    	rendering mode. Fill mesh faces using light effects when available.
  @itemx -shrink
  	rendering mode. Shrink 3d elements (with vtk only).
  @itemx -tube
  	rendering mode. All edges appears as tubes (with vtk only).
  @itemx -ball
  	rendering mode. Vertices appears as balls (with vtk only).
  @itemx -full
  	rendering mode. All edges appears (with vtk only).
  @itemx -cut
    	rendering mode. cut by plane and clip (with vtk only).
        Works with @samp{-full}, @samp{-tube} or @samp{-ball} modes.
        Does not clip mesh with @samp{-fill} and @samp{-grid} modes.
        Does not work yet with @samp{-shrink} mode.
        This option is still in development.

  @itemx -origin @var{x} [@var{y} [@var{z}]]
	set the origin point of the cutting plane 
	when using with the @samp{-cut} option.
	Default is (0.5,0.5,0.5).

  @itemx -normal @var{nx} [@var{ny} [@var{nz}]]
  	set the normal vector of the cutting plane.
	when using with the @samp{-cut} option.
	Default is (1,0,0).
  @itemx -split
	split each edge by inserting a node.
	The resulting mesh is P2-iso-P1.
  @end table
OUTPUT FORMAT OPTIONS:
  @table @code
  @itemx -geo
    	output mesh on standard output stream in geo text file format,
	instead of plotting it.

@toindex @code{bamg}
  @itemx -bamg
    	output mesh on standard output stream in bamg text file format,
	instead of plotting it.

@toindex @code{gmsh}
  @itemx -gmsh
    	output mesh on standard output stream in gmsh ascii file format,
	instead of plotting it.

@toindex @code{tegen}
  @itemx -tegen
    	output mesh on stream in tegen text file format,
	instead of plotting it (TODO).

@toindex @code{mmg3d}
  @itemx -mmg3d
    	output mesh on standard output stream in mmg3d text file format,
	instead of plotting it.

@fiindex @file{.cemagref} cemagref topographic mesh
@cindex  @code{cemagref} topographic mesh

  @itemx -cemagref
    	output mesh on standard output stream in cemagref text file format,
	instead of plotting it. Notices that the mesh does not contains 
        the topographic elevation. If you want to include the elevation,
        in the output, use the @code{field} command with the @code{-cemagref} option.
  @itemx -upgrade
    	edges and faces are numbered. This numbering is required
	for the quadratic approximation.

  @itemx -ndigit @var{int}
    	number of digits used to print floating point values 
	when using the @samp{-geo} option.
	Default depends upon the machine precision
	of the @code{Float} type.
  @item -output-as-double
        Avoid variation of output float formats
        when using high precision @code{doubledouble} or 
	@code{bigfloat} classes.
	This option is used for non-regression test purpose.

  @end table
INPUT FORMAT OPTIONS:
@cindex mesh file format conversion

  @table @code
  @itemx -input-geo
	read the mesh in the @file{.geo} format.
	This is the default.

@fiindex @file{.bamg} bamg mesh
@toindex @code{bamg}
  @itemx -input-bamg
	read the mesh in the @file{.bamg} format.
        Since edges are not numbered in @file{.bamg} format,
	this file format is not suitable for using P2 elements.
	This format is supported only for format conversion purpose.
        See @code{-upgrade} for edge numbering while
	converting to @file{.geo}.

@fiindex @file{.msh} gmsh mesh
@toindex @code{gmsh}
  @itemx -input-gmsh
	read the mesh in the @file{.msh} format.
        Since edges in 2D and faces in 3D are not numbered in @file{.msh}
        format, this file format is not suitable for using P2 elements.
	This format is supported only for format conversion purpose.
        See @code{-upgrade} for edge numbering while
	converting to @file{.geo}.

@fiindex @file{.node} tetgen mesh nodes
@fiindex @file{.ele} tetgen mesh elements
@fiindex @file{.face} tetgen mesh boundary faces
@toindex @code{tetgen}
  @itemx -input-tetgen
	read the mesh as the concatenation of @file{.node}, @file{.ele} and @file{.face} tetgen format.
        Since in 3D are not all numbered in tetgen
        format, this file format is not suitable for using P2 elements.
	This format is supported only for format conversion purpose.
        See @code{-upgrade} for edge numbering while
	converting to @file{.geo}.

@fiindex @file{.mmg3d} mmg3d mesh
@toindex @code{mmg3d}
  @itemx -input-mmg3d
	read the mesh in the @file{.mmg3d} format.
        Since edges and faces are not numbered in @file{.mmg3d} format,
	this file format is not suitable for using P2 elements.
	This format is supported only for format conversion purpose.
        See @code{-upgrade} for edge numbering while
	converting to @file{.geo}.

@cindex regions in geometry
@toindex @code{tri}, grummp mesh generator
@toindex @code{tetra}, grummp mesh generator
@fiindex @file{.template} grummp mesh file format specification
@fiindex @file{.g} grummp mesh
  @itemx -input-grummp
    	read the mesh on standard input stream in grummp text file format.
	The @code{.g} grummp file format is defined
        grummp @file{.template} specification file
        @example
          newfile g
          "grummp"
          "2" nverts ncells nbfaces
          verts: coords
          cells: verts region
          bdryfaces: verts bc
        @end example
        Conversely, for tridimensionnal meshes, the 
        grummp @file{.template} specification file is:
        @example
          newfile g
          "grummp"
          "3" nverts ncells nbfaces
          verts: coords
          cells: verts region
          bdryfaces: verts bc
        @end example
        Such files can be generated with the @code{tri} and @code{tetra}
        grummp mesh file generators. The grummp mesh generators suport
        multi-regions geometries. These regions are translated to bi- or
        tri-dimensional domains, that are no boundary domains.
        A file format conversion writes:
        @example
           geo -input-grummp -geo -upgrade -noverbose - < hex-multi.g > hex-multi.geo
        @end example

@fiindex @file{.qmg} qmg mesh
@toindex @code{qmg}
  @itemx -input-qmg
    	output mesh on standard output stream in qmg text file format.
	instead of plotting it.
        Since edges are not numbered in @file{.qmg} format,
	this file format is not suitable for using P2 elements.
	This format is supported only for format conversion purpose.
        See @code{-upgrade} for edge numbering while
	converting to @file{.geo}.

@fiindex @file{.dmn} domain names
@cindex mesh file format conversion
  @itemx -dmn @var{domain-names-file}
	An auxilliary @file{.dmn} file defining the boundary domain names
	as used by @code{rheolef} could be used, since @code{bamg}, @code{mmg3d}, @code{tetgen}
	and @code{grummp} use numeric
	labels for domains. Example of a @file{multi-domain.dmn} 
	with two regions and four boundary domains file:
	@example
	  EdgeDomainNames
	    4
	    bottom
	    right
	    top
	    left
	  FaceDomainNames
            2
            hard
            soft
	@end example
        @code{geo} recognized this format as an extension at the end
        of the @file{.bamg}, @code{.mesh} (mmg3d) or the @file{.g} file.
	The complete file format conversion writes:
        @example
          cat multi-domain.g multi-domain.dmn | \\
            geo -input-grummp -upgrade -geo -  > multi-domain.geo
        @end example
        If this file is not provided, boundary domains are named
	@code{boundary1}, @code{boundary2}, ...
        and region domain are named
	@code{region1}, @code{region2}, ...
	The tridimensionnal domain name file contains
        the @code{FaceDomainNames} and @code{VolumeDomainNames}
	keywords, for boundary and region domains, respectively.

@fiindex @file{.cemagref} cemagref topographic mesh
@cindex  @code{cemagref} topographic mesh

  @itemx -input-cemagref
	read the mesh in the @file{.cemagref} format.
	Notices that the reader skip the topographic elevation informations.
	If you want to load the elevation, use the @code{field} 
	command with the @code{-input-cemagref} option.

  @end table

OTHERS OPTIONS:
  @table @code
    @itemx -verbose
    	print messages related to graphic files created and
       	command system calls (this is the default).

    @itemx -noverbose
    	does not print previous messages.

    @itemx -clean
    	clear temporary graphic files (this is the default).
    
    @itemx -noclean
    	does not clear temporary graphic files.

    @itemx -execute
    	execute graphic command (this is the default).
    
    @itemx -noexecute
   	does not execute graphic command. Generates only 
	graphic files. This is usefull in conjuction with the
	"-noclean" command.

  @itemx -check
  @itemx -dump
  	used for debug purpose.
  @end table

GRAPHIC OUTPUT FILE:
    You can generate a @code{postscript}, 
    @code{latex} or @code{Fig}
    output of your mesh, using 
    the @code{geo} command with
    @samp{-gnuplot -noclean} options, and then modifying the 
    @samp{.plot} generated file.

GEO FILE FORMAT:       
@noindent
This is the default mesh file format. It contains
two entitiess, namely a mesh and a list of domains. The mesh 
entity starts with the header. The header contains the @code{mesh}
keyword followed by a line containing a format version number, 
the space dimension (e.g. 1, 2 or 3), the number of vertices, the
number of elements. For file format version 2,
when dimension is three, the number of faces is specified,
and then,
when dimension is two or three, the number of edges is also specified.
Follows the vertex coordinates, the elements and the faces or edges.
One element starts with a letter indicating the element type
@table @samp
       	@itemx p
       		point 
       	@itemx e
       		edge 
	@itemx t
		triangle
	@itemx q
		quadrangle
	@itemx T
		tetrahedron
        @itemx P
		prism
	@itemx H
		hexaedron
@end table
@noindent
Then, we have the vertex indexes.
A vertex index is numbered in the C-style,
i.e. the first index is numbered 0.
An edge is a couple of index.

Geo version 1 file format (obsolete) does neither give any information about edges
for 2d meshes nor for faces in 3d. This information is requiered for
maintaining P2 approximation file in a consistent way, since edge and face
numbering algorithm should vary from one version to another, and thus may be stored
on file.

Nevertheless, geo version 1 file format is still supported, since it is
a convenient way to build and import meshes (@pxref{geo command}).

@noindent
A sample mesh is in version 1 writes
@example
    	mesh
        1 2 4 2
        0 0
        1 0
        1 1
        0 1
        t   0 1 3
        t   1 2 3
@end example

@noindent
the same in version 2
@example
    	mesh
        2 2 4 2 5
        0 0
        1 0
        1 1
        0 1
        t   0 1 3
        t   1 2 3
        0 1
        1 2
        2 3
        3 0
        1 3
@end example

@noindent
The second entity is a list of domains, that finishes with the
end of file. A domain starts with the 
@code{domain} keyword, followed by a domain name. 
Then, the format version number (i.e. 1 today),
and the number of sides. Follows the sides
@example
       	domain
       	bottom
       	1 1 1
       	e   0 1
   
       	domain
       	top
       	1 1 1
       	e   2 3
@end example
AUTHOR: Pierre.Saramito@imag.fr
SEE ALSO:
    "geo"(3), "rheostream"(3), "gnuplot"(1)
DATE:
    12 may 1997
End:
*/

#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

extern "C" {
    void exit(int);
}
void usage()
{
      cerr << "geo: usage:" << endl 
           << "geo [-size|-n-node|-hmin|-hmax|-xmin|-xmax]" << endl
           << "geo "
	   << "{-Igeodir}* "
	   << "[-name string] "
	   << "[-input-bamg|-input-gmsh|-input-tetgen|-input-mmg3d|-input-grummp|-input-qmg|-input-cemagref|-input-geo] "
	   << "[-geo|-bamg|-gmsh|-tetgen|-mmg3d|-cemagref|-gnuplot|-plotmtv|-atom|-x3d|-vtk|-mayavi|-dump|-check] "
	   << "[-color|-black-and-white|-stereo|-grid|-fill|-shrink|-tube|-full|-cut|-split] "
    	   << "-origin ox oy oz"
    	   << "-normal nx ny nz"
	   << "[-[no]clean] [-[no]execute] [-[no]verbose] "
	   << "[-output-as-double] "
	   << "{-|file[.geo[.gz]]|file[.msh[.gz]]}" 
	   << endl;
      exit (1);
}
bool
atofloat (const char* s, Float& x)
{
    double tmp;
    if (1 != sscanf(s, "%lf", & tmp)) return false;
    x = tmp;
    return true;
}
typedef enum { 
	no_inquire, 
	get_size, 
	get_n_node,
	get_hmin,
	get_hmax,
	get_xmin,
	get_xmax
} inquire_type;

typedef enum { 
	input_geo, 
	input_bamg, 
	input_tetgen, 
	input_mmg3d, 
	input_grummp, 
	input_gmsh, 
	input_qmg, 
	input_cemagref
} input_type;

typedef enum { 
	no_render, 
	gnuplot_render,
	mayavi_render,
	plotmtv_render,
	vtk_render,
	atom_render,
	x3d_render,
	file_render
} render_type;


int main(int argc, char **argv)
{
    if (argc <= 1) usage();
  
    bool has_geo = false;
    bool do_dump = false;
    bool do_check = false;
    string domain_names;
    string name = "output";
    string showdom = "";
    render_type render = no_render;

    point origin (0,0,0);
    point normal (1,0,0);
    //
    // set default options
    //
    bool upgrade = false ;
    int digits10 = numeric_limits<Float>::digits10;
    input_type input_format = input_geo;
    inquire_type inquire = no_inquire;
    clog << verbose; bool bverbose = true; 
    cout << grid;
    cout << nofill;
    cout << mayavi;

    geo g;
    for (int i = 1; i < argc; i++) {

      // input format

             if (strcmp (argv[i], "-input-geo") == 0)      input_format = input_geo;
        else if (strcmp (argv[i], "-input-bamg") == 0)     input_format = input_bamg;
        else if (strcmp (argv[i], "-input-tetgen") == 0)   input_format = input_tetgen;
        else if (strcmp (argv[i], "-input-mmg3d") == 0)    input_format = input_mmg3d;
        else if (strcmp (argv[i], "-input-grummp") == 0)   input_format = input_grummp; 
        else if (strcmp (argv[i], "-input-gmsh") == 0)     input_format = input_gmsh; 
        else if (strcmp (argv[i], "-input-qmg") == 0)      input_format = input_qmg; 
        else if (strcmp (argv[i], "-input-cemagref") == 0) input_format = input_cemagref; 

        else if (strcmp (argv[i], "-name") == 0) {
	    if (i == argc-1) {
	    	cerr << "geo: -name option: argument missing" << endl;
		usage();
	    }
	    name = argv[++i]; 
        }
      // inquire action

        else if (strcmp (argv[i], "-size") == 0)      inquire = get_size;
        else if (strcmp (argv[i], "-n-node") == 0)    inquire = get_n_node;
        else if (strcmp (argv[i], "-hmin") == 0)      inquire = get_hmin;
        else if (strcmp (argv[i], "-hmax") == 0)      inquire = get_hmax;
        else if (strcmp (argv[i], "-xmin") == 0)      inquire = get_xmin;
        else if (strcmp (argv[i], "-xmax") == 0)      inquire = get_xmax;

      // output format or render or action

        else if (strcmp (argv[i], "-geo") == 0)       { cout << rheo;     render = file_render; }
        else if (strcmp (argv[i], "-bamg") == 0)      { cout << bamg;     render = file_render; }
        else if (strcmp (argv[i], "-mmg3d") == 0)     { cout << mmg3d;    render = file_render; }
        else if (strcmp (argv[i], "-tetgen") == 0)    { cout << tetgen;   render = file_render; }
        else if (strcmp (argv[i], "-gmsh") == 0)      { cout << gmsh;     render = file_render; }
        else if (strcmp (argv[i], "-cemagref") == 0)  { cout << cemagref; render = file_render; }

        else if (strcmp (argv[i], "-gnuplot") == 0)   { cout << gnuplot; render = gnuplot_render; }
        else if (strcmp (argv[i], "-mayavi") == 0)    { cout << mayavi;  render = mayavi_render; }
        else if (strcmp (argv[i], "-plotmtv") == 0)   { cout << plotmtv; render = plotmtv_render; }
        else if (strcmp (argv[i], "-vtk") == 0)       { cout << vtk;     render = vtk_render; }
        else if (strcmp (argv[i], "-atom") == 0)      { cout << atom;    render = atom_render; }
        else if (strcmp (argv[i], "-x3d") == 0)       { cout << x3d;     render = x3d_render; }

        else if (strcmp (argv[i], "-check") == 0)     do_check = true;
        else if (strcmp (argv[i], "-dump") == 0)      do_dump = true;
        
      // graphic options

	else if (strcmp (argv[i], "-only") == 0)      showdom = argv[++i];
	else if (strcmp (argv[i], "-ndigit") == 0)    digits10 = atoi(argv[++i]);
	else if (strcmp (argv[i], "-color") == 0)    cout << color;
	else if (strcmp (argv[i], "-black-and-white") == 0) cout << black_and_white;
	else if (strcmp (argv[i], "-stereo") == 0)    cout << stereo;
	else if (strcmp (argv[i], "-fill") == 0)      cout << fill;
	else if (strcmp (argv[i], "-grid") == 0)      cout << grid;
	else if (strcmp (argv[i], "-shrink") == 0)    cout << shrink;
	else if (strcmp (argv[i], "-tube") == 0)      cout << tube;
	else if (strcmp (argv[i], "-ball") == 0)      cout << ball;
	else if (strcmp (argv[i], "-full") == 0)      cout << full;
	else if (strcmp (argv[i], "-cut") == 0)       cout << cut;
	else if (strcmp (argv[i], "-split") == 0)     cout << split;
	else if (strcmp (argv[i], "-origin") == 0)    {

	    if (i == argc-1 || ! atofloat(argv[++i], origin[0])) {
		cerr << "geo: -origin option: numeric argument(s) missing" << endl;
		usage();
	    }
	    (i != argc-1 && atofloat(argv[++i], origin[1]) &&
	     i != argc-1 && atofloat(argv[++i], origin[2]));
	    cout << setorigin (origin);
	}
	else if (strcmp (argv[i], "-normal") == 0)    {

	    if (i == argc-1 || ! atofloat(argv[++i], normal[0])) {
		cerr << "geo: -normal option: numeric argument(s) missing" << endl;
		usage();
	    }
	    (i != argc-1 && atofloat(argv[++i], normal[1]) &&
	     i != argc-1 && atofloat(argv[++i], normal[2]));
	    cout << setnormal (normal);
	}
      // other options

	else if (strcmp (argv[i], "-upgrade") == 0)   upgrade = true;
	else if (strcmp (argv[i], "-clean") == 0)     clog << clean;
	else if (strcmp (argv[i], "-noclean") == 0)   clog << noclean;
        else if (strcmp (argv[i], "-execute") == 0)   clog << execute;
        else if (strcmp (argv[i], "-noexecute") == 0) clog << noexecute;
        else if (strcmp (argv[i], "-verbose") == 0)   { bverbose = true; clog << verbose; }
        else if (strcmp (argv[i], "-noverbose") == 0) { bverbose = false; clog << noverbose; }
        else if (strcmp (argv[i], "-output-as-double") == 0) { numeric_flags<Float>::output_as_double(true); }


        else if (argv [i][0] == '-' && argv [i][1] == 'I') {

	    append_dir_to_rheo_path (argv[i]+2);
        }
	else if (strcmp (argv[i], "-") == 0) {

	    // input geo on standard input
	         if (input_format == input_bamg)     cin >> bamg;
	    else if (input_format == input_mmg3d)    cin >> mmg3d;
	    else if (input_format == input_tetgen)   cin >> tetgen;
	    else if (input_format == input_grummp)   cin >> grummp;
	    else if (input_format == input_gmsh)     cin >> gmsh;
	    else if (input_format == input_qmg)      cin >> qmg;
	    else if (input_format == input_cemagref) cin >> cemagref;
	    string thename;
	    if (name != "output") thename = name;
	    cin  >> setbasename(thename) >> g;
	    cout << setbasename(name);
	    has_geo = true;
        }
	else if (argv[i][0] != '-') {
	    // input geo on file
	    if (name=="output") name = get_basename(delete_suffix (delete_suffix (argv[i], "gz"), "geo"));
            g = geo (argv[i]);
	    cout << setbasename(name);
	    has_geo = true;
        }
	else {
	    usage();
	}
    }
    cout << setprecision(digits10);
    if (!has_geo) {
	cerr << "geo: no input specified" << endl;
	usage();
    }
    if (render == no_render) {
        if (g.dimension() == 1) {
            cout << gnuplot;
        } else {
	    char* env_render = getenv("RHEOLEF_RENDER");
	    if (env_render!=NULL) 
	    {
		if (strcmp (env_render, "mayavi") == 0)    	
			{ cout << mayavi; render = mayavi_render; }
		else if (strcmp (env_render, "gnuplot") == 0)	
			{ cout << gnuplot; render = gnuplot_render; }
		else if (strcmp (env_render, "vtk") == 0)	
			{ cout << vtk; render = vtk_render; }
		else if (strcmp (env_render, "plotmtv") == 0)	
			{ cout << plotmtv; render = plotmtv_render; }
		else { render = mayavi_render; cout << mayavi; }
	    }
	    else { render = mayavi_render; cout << mayavi; }
        }
    }
    if (upgrade)  g.upgrade() ;
    if (do_dump) {
	g.dump(cerr);
    }
    if (showdom!="") {
        // Skip delimiters at beginning.
	showdom=showdom+"+";
    	string::size_type lastPos = showdom.find_first_not_of("+", 0);
    	// Find first "non-delimiter".
    	string::size_type pos     = showdom.find_first_of("+", lastPos);
        domain d= g[showdom.substr(lastPos, pos - lastPos)];
	while (string::npos != pos && string::npos != lastPos)
	{
	    lastPos = showdom.find_first_not_of("+", pos);
	    // Find next "non-delimiter"
	    pos = showdom.find_first_of("+", lastPos);
	    if (string::npos == pos) break;
            d=d+g[showdom.substr(lastPos, pos - lastPos)];
	};
	g=geo(g, d);
	if (name=="output") cout << setbasename(showdom+"_from_"+name);
    }
    if (do_check) g.check();
    if (inquire == no_inquire) {
        cout << g;
        if (!cout) return 1;
        return 0;
      }
    // inquire
    switch (inquire) {
        case get_size   : cout << g.size() << endl; break;
        case get_n_node : cout << g.n_node() << endl; break;
        case get_hmin   : cout << g.hmin() << endl; break;
        case get_hmax   : cout << g.hmax() << endl; break;
        case get_xmin   : cout << g.xmin().put(cout,g.dimension()) << endl; break;
        case get_xmax   : cout << g.xmax().put(cout,g.dimension()) << endl; break;
	default: break;
    }
    return 0;
}
