#ifndef _RHEOLEF_VEC_EXPR_H
#define _RHEOLEF_VEC_EXPR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// expressions of vectors without temporaries, based on boost::proto
// and that uses an iterator for the assignment
//
// author: Pierre.Saramito@imag.fr
//
// date: 13 january 2011
//
#include "rheolef/vec.h"
#include "rheolef/pretty_name.h"

#include <boost/mpl/bool.hpp>
#include <boost/proto/core.hpp>
#include <boost/proto/debug.hpp>
#include <boost/proto/context.hpp>
#include <boost/proto/transform.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/typeof/std/vector.hpp>
#include <boost/typeof/std/complex.hpp>
#include <boost/type_traits/remove_reference.hpp>

namespace rheolef {
namespace mpl   = boost::mpl;
namespace proto = boost::proto;
using proto::_;

template <class Expr>
struct vec_expr;

// Here is an evaluation context that indexes into a vec
// expression and combines the result.
struct vec_subscript_context {
    typedef vec<Float>::size_type size_type;
    vec_subscript_context (size_type i) : _i(i) {}
    // Unless this is a vector terminal, use the default evaluation context
    template<class Expr, class EnableIf = void>
    struct eval : proto::default_eval<Expr, const vec_subscript_context> {};
    // Index vector terminals with our subscript.
    template<class Expr>
    struct eval<Expr, typename boost::enable_if<
              proto::matches<Expr, proto::terminal<vec<_, _> > > >::type> {
         typedef typename proto::result_of::value<Expr>::type::value_type result_type;
         result_type operator() (Expr &expr, const vec_subscript_context& ctx) const {
             return proto::value(expr)[ctx._i];
         }
    };
    size_type _i;
};
// get size
struct vec_get_size_context {
    typedef vec<Float>::size_type size_type;
    vec_get_size_context () {}

    template<class Expr, class EnableIf = void>
    struct eval : proto::null_eval<Expr, const vec_get_size_context> {};

    template<class Expr>
    struct eval<Expr, typename boost::enable_if<
            proto::matches<Expr, proto::terminal<vec<_,_> > > >::type> {
        typedef void result_type;
        result_type operator() (Expr &expr, const vec_get_size_context& ctx) const {
            ctx._size      = proto::value(expr).size();
            ctx._ownership = proto::value(expr).ownership();
        }
    };
    size_type          size()      const { return _size; };
    const distributor& ownership() const { return _ownership; };
    mutable size_type   _size;
    mutable distributor _ownership;
};
// check that sizes match
struct vec_check_size_context {
    typedef vec<Float>::size_type size_type;
    vec_check_size_context (size_type size) : _size(size) {}

    template<class Expr, class EnableIf = void>
    struct eval : proto::null_eval<Expr, const vec_check_size_context> {};

    template<class Expr>
    struct eval<Expr, typename boost::enable_if<
            proto::matches<Expr, proto::terminal<vec<_,_> > > >::type> {
        typedef void result_type;
        result_type operator() (Expr &expr, const vec_check_size_context& ctx) const {
            if (ctx._size != proto::value(expr).size()) {
                error_macro ("incompatible sizes "<<ctx._size<< " and " 
		  << proto::value(expr).size() << " in vec<T> expression "
                  <<typeid_name_macro(Expr));
            }
        }
    };
    size_type _size;
};
template<class Iterator>
struct vec_iterator_wrapper {
    typedef Iterator iterator;
    explicit vec_iterator_wrapper (Iterator iter1) : iter(iter1) {} 
    Iterator iter;
};
struct vec_begin : proto::callable {
    template<class Sig>
    struct result;

    template<class This, class Container>
    struct result<This(Container)> : proto::result_of::as_expr
       <
        vec_iterator_wrapper<
	  typename boost::remove_reference<Container>::type::const_iterator>
       > {};

    template<class Container>
    typename result<vec_begin(const Container&)>::type
    operator ()(const Container& cont) const {
        vec_iterator_wrapper<typename Container::const_iterator> iter (cont.begin());
        return proto::as_expr (iter);
    }
};
// Here is the grammar that replaces vector terminals with their begin iterators
struct vec_grammar_begin : proto::or_<
        proto::when <proto::terminal<vec<_, _> >, vec_begin(proto::_value)>
      , proto::when <proto::terminal<_> >
      , proto::when <proto::nary_expr<_, proto::vararg<vec_grammar_begin> > >
    >
{};

// Here is an evaluation context that dereferences iterator terminals.
struct vec_dereference_context {
    // Unless this is an iterator terminal, use the default evaluation context
    template<class Expr, class EnableIf = void>
    struct eval : proto::default_eval<Expr, const vec_dereference_context> {};

    // Dereference iterator terminals.
    template<class Expr>
    struct eval<Expr,
       typename boost::enable_if<
         proto::matches<Expr, proto::terminal<vec_iterator_wrapper<_> > >
        >::type
       > {
        typedef typename proto::result_of::value<Expr>::type IteratorWrapper;
        typedef typename IteratorWrapper::iterator iterator;
        typedef typename std::iterator_traits<iterator>::reference result_type;

        result_type operator() (Expr &expr, const vec_dereference_context&) const {
            return *proto::value(expr).iter;
        }
    };
};
// Here is an evaluation context that increments iterator
// terminals.
struct vec_increment_context {
    // Unless this is an iterator terminal, use the default evaluation context
    template<class Expr, class EnableIf = void>
    struct eval : proto::null_eval<Expr, const vec_increment_context> {};

    // advance iterator terminals.
    template<class Expr> struct eval<Expr,
	typename boost::enable_if<
            proto::matches<Expr, proto::terminal<vec_iterator_wrapper<_> > >
        >::type
      > {
        typedef void result_type;

        result_type operator() (Expr &expr, const vec_increment_context&) const {
            ++proto::value(expr).iter;
        }
    };
};
// A grammar which matches all the assignment operators,
// so we can easily disable them.
struct vec_assign_operators : proto::switch_<struct vec_assign_operators_cases> {};

// Here are the cases used by the switch_ above.
struct vec_assign_operators_cases {
    template<class Tag, int D = 0> struct case_  : proto::not_<_> {};

    template<int D> struct case_< proto::tag::plus_assign, D >         : _ {};
    template<int D> struct case_< proto::tag::minus_assign, D >        : _ {};
    template<int D> struct case_< proto::tag::multiplies_assign, D >   : _ {};
    template<int D> struct case_< proto::tag::divides_assign, D >      : _ {};
    template<int D> struct case_< proto::tag::modulus_assign, D >      : _ {};
    template<int D> struct case_< proto::tag::shift_left_assign, D >   : _ {};
    template<int D> struct case_< proto::tag::shift_right_assign, D >  : _ {};
    template<int D> struct case_< proto::tag::bitwise_and_assign, D >  : _ {};
    template<int D> struct case_< proto::tag::bitwise_or_assign, D >   : _ {};
    template<int D> struct case_< proto::tag::bitwise_xor_assign, D >  : _ {};
};
// An expression conforms to the vec_grammar if it is a terminal or some
// op that is not an assignment op. (Assignment will be handled specially.)
struct vec_grammar
  : proto::or_<
        proto::terminal<_>
      , proto::and_<
            proto::nary_expr<_, proto::vararg<vec_grammar> >
          , proto::not_<vec_assign_operators>
        >
    >
{};

// Expressions in the vec_domain will be wrapped in vec_expr<>
// and must conform to the vec_grammar
struct vec_domain : proto::domain<proto::generator<vec_expr>, vec_grammar> {};

// Here is vec_expr, a wrapper for expression types in the vec_domain.
template<class Expr>
struct vec_expr : proto::extends<Expr, vec_expr<Expr>, vec_domain> {
    explicit vec_expr (const Expr& expr)
      : proto::extends<Expr, vec_expr<Expr>, vec_domain>(expr)
    {}
#ifdef TO_CLEAN
    typename boost::result_of<vec_grammar_begin(const Expr&)>::type
    begin() const {
        return vec_grammar_begin() (*this);
    }
#endif // TO_CLEAN
private:
    // hide this:
    using proto::extends<Expr, vec_expr<Expr>, vec_domain>::operator[];
};
// Define a trait type for detecting vec terminals, to
// be used by the BOOST_PROTO_DEFINE_OPERATORS macro below.
template<class T>
struct is_vec : mpl::false_ {};

template<class T, class M>
struct is_vec<vec<T, M> > : mpl::true_ {};

// -------------------------------------------
// x+y, x-y, ect
// -------------------------------------------
// This defines all the overloads to make expressions involving
// vec to build expression templates.
#define RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, Left, Right) \
    typename boost::proto::detail::enable_binary< \
	DOMAIN, \
	TRAIT<Left >, \
	Left, \
	TRAIT<Right >, \
	Right,     \
        typename boost::proto::functional::make_expr<TAG, DOMAIN>::impl< \
		const Left&, const Right&>::result_type\
          >::type const \
    operator OP (const Left& left, const Right& right) { \
        return boost::proto::functional::make_expr<TAG, DOMAIN>::impl< \
            const Left&, const Right&>() (left, right);\
    }

#define DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN) \
    template<typename T>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, T, vec<T>) \
    template<typename T>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, vec<T>, T) \
    template<typename T>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, int, vec<T>) \
    template<typename T>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, vec<T>, int) \
    template<typename T1, typename T2>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, vec<T1>, vec<T2>) \
    template<typename E1, typename T2>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, vec_expr<E1>, vec<T2>) \
    template<typename T1, typename E2>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, vec<T1>, vec_expr<E2>) \
    template<typename E1, typename E2>  \
    RAW_DEFINE_BINARY_OPERATOR(OP, TAG, TRAIT, DOMAIN, vec_expr<E1>, vec_expr<E2>)

#define DEFINE_OPERATORS(TRAIT, DOMAIN)  \
    DEFINE_BINARY_OPERATOR(+, boost::proto::tag::plus,       TRAIT, DOMAIN) \
    DEFINE_BINARY_OPERATOR(*, boost::proto::tag::multiplies, TRAIT, DOMAIN) \

DEFINE_OPERATORS(is_vec, vec_domain)

#ifdef TO_CLEAN
namespace vec_op {
BOOST_PROTO_DEFINE_OPERATORS(is_vec, vec_domain)
} // namespace vec_op
#endif // TO_CLEAN

namespace vec_detail {
    struct assign_op {
        template<class T, class U>
        void operator() (T &t, U const &u) const { t = u; }
    };
    struct plus_assign_op {
        template<class T, class U>
        void operator() (T &t, U const &u) const { t += u; }
    };
    struct minus_assign_op {
        template<class T, class U>
        void operator() (T &t, U const &u) const { t -= u; }
    };

#ifdef TODO
    struct sin_ {
        template<class Sig>
        struct result;

        template<class This, class Arg>
        struct result<This(Arg)>
          : boost::remove_const<typename boost::remove_reference<Arg>::type>
        {};

        template<class Arg>
        Arg operator ()(Arg const &a) const { return std::sin(a); }
    };
    template<class A>
    typename proto::result_of::make_expr<proto::tag::function,
	vec_domain, const sin_, const A&>::type
    sin (const A& a) {
        return proto::make_expr<proto::tag::function, vec_domain> (sin_(), boost::ref(a));
    }
#endif // TODO
    template<class FwdIter, class Expr, class Op>
    void evaluate (FwdIter begin, FwdIter end, const Expr& expr, Op op) {
        const vec_increment_context   inc   = {};
        const vec_dereference_context deref = {};
        typename boost::result_of<vec_grammar_begin(const Expr&)>::type expr_begin
	  = vec_grammar_begin() (expr);
        for (; begin != end; ++begin) {
            op (*begin, proto::eval(expr_begin, deref));
            proto::eval (expr_begin, inc);
        }
    }
} // namespace vec_detail

// -------------------------------------------
// x = expr; x += expr; x -= expr;
// -------------------------------------------
template<class T, class M>
template<class Expr>
inline
vec<T,M>& 
vec<T,M>::operator= (const Expr& expr) {
    // get and check sizes:
    const vec_get_size_context get_size;
    proto::eval (proto::as_expr<vec_domain>(expr), get_size);
    size_type   expr_size      = get_size.size();
    if (array<T,M>::par_size() == 0) {
        distributor expr_ownership = get_size.ownership();
	resize (expr_ownership);
    }
    const vec_check_size_context check_size (array<T,M>::size());
    proto::eval (proto::as_expr<vec_domain>(expr), check_size); // error if the sizes don't match
    // perform all computations here:
    vec_detail::evaluate (array<T,M>::begin(), array<T,M>::end(), proto::as_expr<vec_domain>(expr), vec_detail::assign_op());
    return *this;
}
template<class T, class M>
template<class Expr>
inline
vec<T,M>::vec (const Expr& expr) {
    operator= (expr);
}

// Add-assign to a vector from some expression.
template<class T, class M>
template<class Expr>
inline
vec<T,M>&
vec<T,M>::operator+= (const Expr& expr) {
    vec_detail::evaluate (array<T,M>::begin(), array<T,M>::end(), proto::as_expr<vec_domain>(expr), vec_detail::plus_assign_op());
    return *this;
}
// Minus-assign to a vector from some expression.
template<class T, class M>
template<class Expr>
inline
vec<T, M>&
vec<T, M>::operator-= (const Expr& expr) {
    vec_detail::evaluate (array<T,M>::begin(), array<T,M>::end(), proto::as_expr<vec_domain>(expr), vec_detail::minus_assign_op());
    return *this;
}
// -------------------------------------------
// dot(x,y)
// -------------------------------------------
template <class T>
inline
T
dot (const vec<T>& x, const vec<T>& y) {
  return par_inner_product (x.begin(), x.end(), y.begin(), x.comm());
}
// TODO: avoid temporaries vec<T1>(y) by using an iterator y.begin()
template <class T1, class E2>
inline
T1
dot (const vec<T1>& x, const vec_expr<E2>& y) {
  return dot (x, vec<T1>(y));
}
template <class E1, class T2>
inline
T2
dot (const vec_expr<E1>& x, const vec<T2>& y) {
  return dot (vec<T2>(x),y);
}
#ifdef TODO
// I don't known how to compute the return type T !
// also, I don't known how to compute x.end() or y.end()...
// it is possible to use x[i] and y[i] together with x.size()
template <class E1, class E2>
inline
T
dot (const vec_expr<E1>& x, const vec_expr<E2>& y) {
  return dot (vec<T2>(x), vec<T>(y));
}
#endif // TODO

} // namespace rheolef
#endif // _RHEOLEF_VEC_EXPR_H
