#ifndef _RHEO_DISTRIBUTOR_H
#define _RHEO_DISTRIBUTOR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

# include "rheolef/parallel.h"
# include "rheolef/par_macros.h"
# include "rheolef/Vector.h"

namespace rheolef {

/*Class:distributor
NAME:  distributor - data distribution table (@PACKAGE@-@VERSION@)
SYNOPSYS:
   Used by "array"(1), "asr"(1) and "csr"(1).
   and such classes that distribute data as chunk.
SEE ALSO: "array"(1), "asr"(1), "csr"(1).
AUTHORS: Pierre.Saramito@imag.fr
DATE:   27 november 1998
End:
*/
//<distributor:
class distributor : public Vector<std::allocator<int>::size_type> {
public:

// typedefs:

	typedef std::allocator<int>::size_type size_type;
	typedef Vector<size_type>	       _base;
	typedef _base::iterator                iterator;
	typedef _base::const_iterator          const_iterator;
	typedef int                            tag_type;
	typedef communicator                   communicator_type;

// constants:

	static const size_type decide = size_type(-1);

// allocators/deallocators:

        distributor(
		size_type par_size = 0,
		const communicator_type& c = communicator_type(),
		size_type loc_size = decide);

        distributor(const distributor&);
	~distributor();

        void resize(
		size_type par_size = 0,
		const communicator_type& c = communicator_type(),
		size_type loc_size = decide);

// accessors:

	const communicator_type& comm() const;

	/// global and local sizes
        size_type par_size () const;

	/// current process id
        size_type process () const;
	
	/// number of processes
        size_type n_process () const;

	/// global index range and local size owned by current process
        size_type first_index () const;
        size_type last_index () const;
        size_type size () const;

	/// global index range and local size owned by k-th process
        size_type first_index (size_type k) const;
        size_type last_index (size_type k) const;
        size_type size (size_type k) const;

	/// return process id that store a global index
	size_type owner(size_type global_index) const;

	/// returns a new tag
	static tag_type get_new_tag();

 	/// find proc index associated to an index: CPU=log(nproc)
	size_type find_owner (size_type index) const;

// data:
protected:
	communicator_type _comm;
};
//>distributor:

// inline'd
inline
const distributor::communicator_type&
distributor::comm() const
{
  	return _comm;
}
inline
distributor::size_type
distributor::first_index(size_type j) const 
{
    	return at(j);
}
inline
distributor::size_type
distributor::last_index(size_type j) const 
{
    	return at(j+1);
}
inline
distributor::size_type
distributor::size(size_type j) const 
{
    	return last_index(j) - first_index(j);
}
inline
distributor::size_type
distributor::n_process() const 
{
#ifdef _RHEOLEF_HAVE_MPI
    	return _comm.size();
#else // _RHEOLEF_HAVE_MPI
    	return 1;
#endif // _RHEOLEF_HAVE_MPI
}
inline
distributor::size_type
distributor::process() const 
{
#ifdef _RHEOLEF_HAVE_MPI
    	return _comm.rank();
#else // _RHEOLEF_HAVE_MPI
    	return 0;
#endif // _RHEOLEF_HAVE_MPI
}
inline
distributor::size_type
distributor::first_index() const 
{
    	return first_index(process());
}
inline
distributor::size_type
distributor::last_index() const 
{
    	return last_index(process());
}
inline
distributor::size_type
distributor::size() const 
{
    	return size(process());
}
inline
distributor::size_type
distributor::par_size() const 
{
	return at(n_process());
}
} // namespace rheolef
#endif // _RHEO_DISTRIBUTOR_H
