///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// solve:
//  -\Delta u = 1  in \Omega
//          u = u0  on \partial\Omega
// where \Omega=]0,1[^2
// for a predefined boundary condition u0.
//
// usage:
//  form_solve P1 -I../data carre top bottom left right | field -I../data -

#include "rheolef/rheolef.h"
#include "rheolef/cg.h"
using namespace rheolef;
using namespace std;
//
// the exact solution
//
Float u_fct_1d (const point& x) {
    return x[0]*(1-x[0])/2;
}
Float u_fct_2d (const point& x) {
    return (x[0]*(1-x[0]) + x[1]*(1-x[1]))/4;
}
Float u_fct_3d (const point& x) {
    return (x[0]*(1-x[0]) + x[1]*(1-x[1]) + x[2]*(1-x[2]))/6;
}
void usage()
{
      cerr << "form_solve: usage: form_solve"
	   << " [P1|P2]"
           << " [-ndigit int]"
	   << " [-round]"
	   << " [-name string]"
	   << " [-direct|-cg]"
	   << " {-Igeodir}*"
	   << " -|mesh[.geo]"
	   << " {domain}+"
	   << endl;
      exit (1);
}
// round error, since P2 error is eps mach
Float eps1 = 1e-6;
template <class T>
inline T my_round (const T& x) { return eps1*T(int(x/eps1+0.5)); }

int main(int argc, char**argv)
{
    bool verbose = true;
    bool direct = true;
    bool do_round = false;
    if (argc <= 2) usage();
    string approx;
    string name;
    geo g;
    space V;
    int digits10 = numeric_limits<Float>::digits10;
    bool have_geo = false;

    for (int io = 1; io < argc; io++) {
        if (strcmp (argv[io], "-round") == 0) {
            do_round = true;
        } else if (strcmp (argv[io], "-ndigit") == 0) {
	    digits10 = atoi(argv[++io]);
    	    cout << setprecision(digits10);
        } else if (strcmp (argv[io], "-name") == 0) {
	    // usefull when mesh comes from cin
	    name = argv[++io];
            iorheo::setbasename(cin, name); // TODO: create manip: cin >> setbasename(name);
	    cout << setbasename(name);
        } else if (strcmp (argv[io], "-direct") == 0) {
	} else if (strcmp (argv[io], "-cg") == 0) {
            direct = false;
    	} else if (argv [io][0] == '-' && argv [io][1] == 'I') {
	    append_dir_to_rheo_path (argv[io]+2);
	} else if (strcmp (argv[io], "-") == 0) {
	    // input geo on standard input
            cerr << "! form_solve: geo on stdin\n";
	    cin >> g;
    	    if (approx == "") usage();
    	    V = space (g, approx);
	    have_geo = true;
        } else {
    	    if (approx == "") {
		approx = argv[io];
	    } else if (!have_geo) {
	        // input geo on file
  	        g = geo(argv[io]);
    		V = space (g, approx);
		have_geo = true;
            } else {
		// block a domain
		cerr << "! form_solve: block `" << argv[io] << "'\n";
		V.block (argv[io]);
	    }
        }
    }
    //
    // build space & forms
    //
    form a(V,V,"grad_grad");
    form m (V,V,"mass");
    if (V.n_blocked() == 0) {
        cerr << "no blocked degrees of freedom.\n";
        usage();
    }
    //
    // build exact solution and right-hand-side
    //
    field u_ex;
    if (g.dimension() == 1)
      u_ex = interpolate(V, u_fct_1d);
    else if (g.dimension() == 2)
      u_ex = interpolate(V, u_fct_2d);
    else
      u_ex = interpolate(V, u_fct_3d);

    //
    // build approximate solution
    //
    field u(V);
    u.u = 0;
    u.b = u_ex.b;
    
    Float eps = 1e-7;
    int iter_max = 100;

    field f(V, 1.0);
    int status = 0;
    if (direct) {
        ssk<Float> fa = ldlt(a.uu);
	u.u = fa.solve(m.uu*f.u + m.ub*f.b - a.ub*u.b);
    } else {
        status = cg (a.uu, u.u, m.uu*f.u + m.ub*f.b - a.ub*u.b, basic_diag<Float>(a.uu), iter_max, eps);
        cerr << "iter_max = " << iter_max << endl;
        cerr << "eps = " << eps << endl;
    } 
    Float err_l2 = sqrt(dot(m*(u-u_ex), u-u_ex));
    if (do_round) err_l2 = my_round(err_l2);

    cerr << "|u - u_ex| = " << double(err_l2) << endl;

    // force conversion to double for hight prec classes
    // for non-regression test purpose
    numeric_flags<Float>::output_as_double(true);
    cout << rheo << transform(u,my_round);
    return status;
}
