#ifndef _RHEOLEF_SPACE_H
#define _RHEOLEF_SPACE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/geo.h"
#include "rheolef/element.h"

namespace rheolef {

// =====================================================================
// space_act = a domain + an act (block, unblock)
// =====================================================================

template <class T, class M>
class space_act {
public:

// typedefs:

    enum act_type {
        block = 0,
        unblock = 1
    };

// allocators:

    space_act() : _dom(), _act() {}
    space_act(const domain_basic<T,M>& dom, act_type act)
	: _dom(dom), _act(act) {}
    space_act(const space_act<T,M>& sp)
	: _dom(sp._dom), _act(sp._act) {}

// accessors:

    const domain_basic<T,M>& get_domain() const { return _dom; }
    act_type                 get_act()    const { return _act; }
    
// data:
protected:
    domain_basic<T,M> _dom;
    act_type          _act;
};
// =====================================================================
// space_constitution = a table of acts
// =====================================================================

template <class T, class M>
class space_constitution_rep : public std::vector<space_act<T,M> > {
public:
    typedef std::vector<space_act<T,M> >   base;
    typedef typename  base::size_type      size_type;
    typedef typename  base::const_iterator const_iterator;

// allocators:

    space_constitution_rep()
      : base(), _omega(), _element() {}
    space_constitution_rep(const geo_basic<T,M>& omega, std::string approx)
      : base(), _omega(omega), _element(approx) {}
    space_constitution_rep (const space_constitution_rep<T,M>& scr)
      : base(scr), _omega(scr._omega), _element(scr._element)
    {
	error_macro ("physical copy of space_constitution_rep: size="<< base::size());
    }

// accessors:

    const geo_basic<T,M>& get_geo() const { return _omega; }
    const element<T>& get_element() const { return _element; }

    size_type      size()  const { return base::size(); }
    const_iterator begin() const { return base::begin(); }
    const_iterator end()   const { return base::end(); }

// modifiers

    void set_geo (const geo_basic<T,M>& omega)  { _omega = omega; }
    void set_element (const element<T>& elt)    { _element = elt; }

    void block  (const domain_basic<T,M>& dom) {
    	warning_macro ("block domain "<<dom.name());
	base::push_back (space_act<T,M> (dom, space_act<T,M>::block));
    }
    void unblock(const domain_basic<T,M>& dom) {
	base::push_back (space_act<T,M> (dom, space_act<T,M>::unblock));
    }

// data:
protected:
    geo_basic<T,M>                   _omega;
    element<T>                       _element;
};
template <class T, class M>
class space_constitution : public smart_pointer<space_constitution_rep<T,M> > {
public:

    typedef space_constitution_rep<T,M>   rep;
    typedef smart_pointer<rep>            base;
    typedef typename  rep::size_type      size_type;
    typedef typename  rep::const_iterator const_iterator;

// allocators:

    space_constitution()
     : smart_pointer<rep>(new_macro(rep)) {}

    space_constitution(const geo_basic<T,M>& omega, std::string approx)
     : smart_pointer<rep>(new_macro(rep (omega,approx))) {}

// accessors:

    const geo_basic<T,M>& get_geo() const { return base::data().get_geo(); }
    const element<T>& get_element() const { return base::data().get_element(); }

    size_type      size()  const { return base::data().size(); }
    const_iterator begin() const { return base::data().begin(); }
    const_iterator end()   const { return base::data().end(); }

// modifiers

    void set_geo (const geo_basic<T,M>& omega) { base::data().set_geo (omega); }
    void set_element (const element<T>& elt)   { base::data().set_element (elt); }

    void block  (const domain_basic<T,M>& dom) {
    	warning_macro ("block domain "<<dom.name());
	base::data().block(dom);
    }
    void unblock(const domain_basic<T,M>& dom) { base::data().unblock(dom); }
};
// =====================================================================
// a dof = a degree-of-freedom 
//       = space_pair
//       = pair (bool is_blocked ; size_t iub)
// =====================================================================
// TODO: compact the bool as an extra bit in size_type ?
struct space_pair_type {
    typedef array<size_t>::size_type size_type;
    space_pair_type () : _blk(false), _iub (std::numeric_limits<size_type>::max()) {}
    space_pair_type (bool blk, size_type iub) : _blk(blk), _iub(iub) {}
    bool is_blocked() const { return _blk; }
    size_type iub()   const { return _iub; }
    void set_iub (size_type iub) { _iub = iub; }
    void set_blocked (bool blk)  { _blk = blk; }
    friend std::ostream&  operator<< (std::ostream& os, const space_pair_type& x) {
	return os << "{" << x.is_blocked() << "," << x.iub() << "}"; }
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) { ar & _blk; ar & _iub; }
protected:
    bool      _blk;
    size_type _iub;
};

} // namespace rheolef

#ifdef _RHEOLEF_HAVE_MPI
// =====================================================================
// Some serializable types, like geo_element, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous 
// copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <> struct is_mpi_datatype<rheolef::space_pair_type> : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

namespace rheolef {

// =====================================================================
// 1) representation: space_rep, space_seq_rep and space_mpi_rep
// =====================================================================
template <class T, class M>
class space_rep {
public:

// typedefs:

    typedef typename space_pair_type::size_type size_type;

// allocators:

    space_rep ();
    space_rep (const geo_basic<T,M>& omega, std::string approx);

// accessors:

    void block  (const domain_basic<T,M>& dom) {
    	warning_macro ("block domain "<<dom.name());
	no_freeze_guard();
	_constit.block  (dom);
    }
    void unblock(const domain_basic<T,M>& dom) {no_freeze_guard(); _constit.unblock(dom);}

    const distributor&  ownership() const { return _idof2blk_iub.ownership(); }
    size_type                size() const { return ownership().size(); }
    size_type            par_size() const { return ownership().par_size(); }
    const communicator&      comm() const { return ownership().comm(); }

    const geo_basic<T,M>& get_geo() const { return _constit.get_geo(); }
    const element<T>& get_element() const { return _constit.get_element(); }

    bool                 is_blocked (size_type     idof) const { freeze_guard(); return _idof2blk_iub [idof].is_blocked(); }
    size_type                   iub (size_type     idof) const { freeze_guard(); return _idof2blk_iub [idof].iub(); }

    const distributor& iu_ownership() const { freeze_guard(); return _iu_ownership; }
    const distributor& ib_ownership() const { freeze_guard(); return _ib_ownership; }

    void set_par_dof (const geo_element& K, std::vector<size_type>& par_idof) const;

protected:
// internal:
    void freeze_guard() const {
    	if (_is_freezed) return;
    	_is_freezed = true;
    	freeze_body();
    }
    void no_freeze_guard() const {
    	check_macro (!_is_freezed, "freezed space cannot accept new (un)blocked domains");
    }
    void base_freeze_body() const;
    virtual void freeze_body() const { return base_freeze_body(); }
// data:
    space_constitution<T,M>          _constit;
    mutable bool                     _is_freezed;
    mutable array<space_pair_type,M> _idof2blk_iub;  // pair (is_blocked ; iu_or_ib); use ownership
    mutable distributor              _iu_ownership;  // unknown values distribution
    mutable distributor              _ib_ownership;  // blocked values distribution
};
// ---------------------------------------------------------------------
template <class T>
class space_seq_rep : public space_rep<T,sequential> {
public:

// typedefs:

    typedef space_rep<T,sequential> base;
    typedef typename base::size_type size_type;

// allocators:

    space_seq_rep (const geo_basic<T,sequential>& omega, std::string approx);

// compatibility with the distributed interface:

    bool             par_is_blocked (size_type par_idof) const { return is_blocked(par_idof); }
    size_type               par_iub (size_type par_idof) const { return iub(par_idof); }
    size_type     idof2par_old_idof (size_type     idof) const { return idof; }
};
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class space_mpi_rep : public space_rep<T,distributed> {
public:

// typedefs:

    typedef space_rep<T,distributed> base;
    typedef typename base::size_type size_type;

// allocators:

    space_mpi_rep (const geo_basic<T,distributed>& omega, std::string approx);

// accessors:

    const communicator&      comm() const { return base::comm(); }

    bool             par_is_blocked (size_type par_idof) const;
    size_type               par_iub (size_type par_idof) const;
    size_type     idof2par_old_idof (size_type     idof) const { base::freeze_guard(); return _idof2par_old_idof [idof]; }

protected:
    typedef std::map <size_type, size_type, std::less<size_type>,
            heap_allocator<std::pair<size_type,size_type> > >           map_type;
    typedef std::map <size_type, space_pair_type, std::less<size_type>,
            heap_allocator<std::pair<size_type,space_pair_type> > >     map_pair_type;
// internal:
    void freeze_body() const;
// data:
    array<size_type>               _idof2par_old_idof;   // permut to/from old dof numbering (before geo part), for i/o
// mutable data, affected by freeze_*()const:
    mutable map_pair_type          _ext_par_idof2blk_par_iub; // same for partition bdry dofs
};
#endif // _RHEOLEF_HAVE_MPI
// ====================================================================
// 2) wrapper class: seq & mpi specializations
// ====================================================================
/*Class:space
NAME: @code{space} -- piecewise polynomial finite element space
@clindex space
DESCRIPTION:
@noindent
The @code{space} class contains some numbering 
for unknowns and blocked degrees of freedoms
related to a given mesh and polynomial approximation.
SEE ALSO: "geo"(3)
AUTHORS: Pierre.Saramito@imag.fr
DATE:   14 december 2010
End:
*/
/// @brief the finite element space
template <class T, class M = rheo_default_memory_model>
class space_basic {
public:
};
typedef space_basic<Float> space;
// ---------------------------------------------------------------------
//<verbatim:
template <class T>
class space_basic<T,sequential> : public smart_pointer<space_seq_rep<T> > {
public:

// typedefs:

    typedef space_seq_rep<T>        rep;
    typedef smart_pointer<rep>      base;
    typedef typename rep::size_type size_type;

// allocators:

    space_basic (const geo_basic<T,sequential>& omega = geo_basic<T,sequential>(), std::string approx = "");

// accessors:

    void block  (std::string dom_name);
    void unblock(std::string dom_name);
    void block  (const domain_basic<T,sequential>& dom);
    void unblock(const domain_basic<T,sequential>& dom);

    const distributor&  ownership() const;
    const communicator& comm() const;
    size_type           size() const;
    size_type           par_size() const;

    const geo_basic<T,sequential>& get_geo() const;
    const element<T>&   get_element() const;

    void set_par_dof (const geo_element& K, std::vector<size_type>& par_idof) const;

    const distributor& iu_ownership() const;
    const distributor& ib_ownership() const;

    bool      is_blocked (size_type     idof) const;
    size_type        iub (size_type     idof) const;
    bool  par_is_blocked (size_type par_idof) const;
    size_type    par_iub (size_type par_idof) const;

    size_type idof2par_old_idof (size_type idof) const;
};
//>verbatim:
template<class T>
inline
space_basic<T,sequential>::space_basic (const geo_basic<T,sequential>& omega, std::string approx)
 : base (new_macro(rep(omega, approx)))
{
}
template<class T>
inline
const distributor&
space_basic<T,sequential>::ownership() const
{
    return base::data().ownership();
}
template<class T>
inline
const communicator&
space_basic<T,sequential>::comm() const
{
    return base::data().comm();
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::size() const
{
    return base::data().size();
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::par_size() const
{
    return base::data().par_size();
}
template<class T>
inline
const geo_basic<T,sequential>&
space_basic<T,sequential>::get_geo() const
{
    return base::data().get_geo();
}
template<class T>
inline
const element<T>&
space_basic<T,sequential>::get_element() const
{
    return base::data().get_element();
}
template<class T>
inline
void
space_basic<T,sequential>::set_par_dof (const geo_element& K, std::vector<size_type>& par_idof) const
{
    return base::data().set_par_dof (K, par_idof);
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::idof2par_old_idof (size_type idof) const
{
    return base::data().idof2par_old_idof (idof);
}
template<class T>
inline
const distributor&
space_basic<T,sequential>::iu_ownership() const
{
    return base::data().iu_ownership();
}
template<class T>
inline
const distributor&
space_basic<T,sequential>::ib_ownership() const
{
    return base::data().ib_ownership();
}
template<class T>
inline
bool
space_basic<T,sequential>::is_blocked (size_type idof) const
{
    return base::data().is_blocked (idof);
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::iub (size_type idof) const
{
    return base::data().iub (idof);
}
template<class T>
inline
bool
space_basic<T,sequential>::par_is_blocked (size_type par_idof) const
{
    return base::data().par_is_blocked (par_idof);
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::par_iub (size_type par_idof) const
{
    return base::data().par_iub (par_idof);
}
template<class T>
inline
void
space_basic<T,sequential>::block (std::string dom_name)
{
    return base::data().block (get_geo()[dom_name]);
}
template<class T>
inline
void
space_basic<T,sequential>::unblock (std::string dom_name)
{
    return base::data().unblock (get_geo()[dom_name]);
}
template<class T>
inline
void
space_basic<T,sequential>::block (const domain_basic<T,sequential>& dom)
{
    return base::data().block (dom);
}
template<class T>
inline
void
space_basic<T,sequential>::unblock (const domain_basic<T,sequential>& dom)
{
    return base::data().unblock (dom);
}
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
//<verbatim:
template <class T>
class space_basic<T,distributed> : public smart_pointer<space_mpi_rep<T> > {
public:

// typedefs:

    typedef space_mpi_rep<T>        rep;
    typedef smart_pointer<rep>      base;
    typedef typename rep::size_type size_type;

// allocators:

    space_basic (const geo_basic<T,distributed>& omega = geo_basic<T,distributed>(), std::string approx = "");

// accessors:

    void block  (std::string dom_name);
    void unblock(std::string dom_name);
    void block  (const domain_basic<T,distributed>& dom);
    void unblock(const domain_basic<T,distributed>& dom);

    const distributor&  ownership() const;
    const communicator& comm() const;
    size_type           size() const;
    size_type           par_size() const;

    const geo_basic<T,distributed>& get_geo() const;
    const element<T>&   get_element() const;

    void set_par_dof (const geo_element& K, std::vector<size_type>& par_idof) const;

    const distributor& iu_ownership() const;
    const distributor& ib_ownership() const;

    bool      is_blocked (size_type     idof) const;
    size_type        iub (size_type     idof) const;

    bool  par_is_blocked (size_type par_idof) const;
    size_type    par_iub (size_type par_idof) const;

    size_type idof2par_old_idof         (size_type idof) const;
};
//>verbatim:

template<class T>
inline
space_basic<T,distributed>::space_basic (const geo_basic<T,distributed>& omega, std::string approx)
 : base (new_macro(rep(omega, approx)))
{
}
template<class T>
inline
const distributor&
space_basic<T,distributed>::ownership() const
{
    return base::data().ownership();
}
template<class T>
inline
const communicator&
space_basic<T,distributed>::comm() const
{
    return base::data().comm();
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::size() const
{
    return base::data().size();
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::par_size() const
{
    return base::data().par_size();
}
template<class T>
inline
const geo_basic<T,distributed>&
space_basic<T,distributed>::get_geo() const
{
    return base::data().get_geo();
}
template<class T>
inline
const element<T>&
space_basic<T,distributed>::get_element() const
{
    return base::data().get_element();
}
template<class T>
inline
void
space_basic<T,distributed>::set_par_dof (const geo_element& K, std::vector<size_type>& par_idof) const
{
    return base::data().set_par_dof (K, par_idof);
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::idof2par_old_idof (size_type idof) const
{
    return base::data().idof2par_old_idof (idof);
}
template<class T>
inline
const distributor&
space_basic<T,distributed>::iu_ownership() const
{
    return base::data().iu_ownership();
}
template<class T>
inline
const distributor&
space_basic<T,distributed>::ib_ownership() const
{
    return base::data().ib_ownership();
}
template<class T>
inline
bool
space_basic<T,distributed>::is_blocked (size_type idof) const
{
    return base::data().is_blocked (idof);
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::iub (size_type idof) const
{
    return base::data().iub (idof);
}
template<class T>
inline
bool
space_basic<T,distributed>::par_is_blocked (size_type par_idof) const
{
    return base::data().par_is_blocked (par_idof);
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::par_iub (size_type par_idof) const
{
    return base::data().par_iub (par_idof);
}
template<class T>
inline
void
space_basic<T,distributed>::block (std::string dom_name)
{
    return base::data().block (get_geo()[dom_name]);
}
template<class T>
inline
void
space_basic<T,distributed>::unblock (std::string dom_name)
{
    return base::data().unblock (get_geo()[dom_name]);
}
template<class T>
inline
void
space_basic<T,distributed>::block (const domain_basic<T,distributed>& dom)
{
    warning_macro ("block domain "<<dom.name());
    base::data().block (dom);
}
template<class T>
inline
void
space_basic<T,distributed>::unblock (const domain_basic<T,distributed>& dom)
{
    base::data().unblock (dom);
}
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
#endif // _RHEOLEF_SPACE_H
