# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Simple dialog box -- Prompts for a comment and returns it
# Returns an empty string on cancel
#
# Written for Red Hat Inc. by Benjamin Mar Kuck <bmarkuck@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require deleteMessageRaiser
package provide getComment 1.0

# OK, cancel, and clear button handlers

proc _getComment_OK {} {
    global originalComment
    global getComment_text
    global objHandle

    set getComment_text [.getComment.commentFrame.entryBox get]

    if {[string length $getComment_text] > 0} {

        # Start the busy indicator

        status loadNotify 1

        if { [catch {set errorString [$objHandle commentOn $getComment_text]} errorText] } {
            tk_messageBox -icon error -title "Error" \
                -message "An error occurred while processing this directive:\n$errorText\nThe operation was aborted."
            return
        }

        # Lower the busy indicator

        status loadNotify 0

        if { $errorString != "" } {
            tk_messageBox -icon error -title "Error" \
                -message "The backend returned the error:\n $errorString"
            return
        }
    } else {

        # Start the busy indicator

        status loadNotify 1

        if { [catch {set errorString [$objHandle commentDrop]} errorText] } {
            tk_messageBox -icon error -title "Error" \
                -message "An error occurred while processing this directive:\n$errorText\nThe operation was aborted."
            return
        }

        # Lower the busy indicator

        status loadNotify 0

        if { $errorString != "" } {
            tk_messageBox -icon error -title "Error" \
                -message "The backend returned the error:\n $errorString"
            return
        }
    }

    if { $errorString == "" } {
        # Issue a refresh to reflect the possibly updated comment
        # Note that this hardcodes the hyperlink dispatcher / hyperlink
        # watcher names

        hyperlinker selectorMoveRequest [hyperlinkWatcher getCurrentLocation]

        # Now, clean up the dialog by calling the cancel procedure

        _getComment_Cancel
    }
}

proc _getComment_Cancel {} {

    # Just clean-up the dialog

    destroy .getComment
}

proc _getComment_Clear {} {

    # Clear the entry box if there is a comment in it
    
    set comment [.getComment.commentFrame.entryBox get]
    if {[string length $comment] > 0} {
        .getComment.commentFrame.entryBox delete 0 [string length $comment]
    }
}

# Get a name and return it

proc getComment { windowTitle objectType dialogPrompt handle } {
    global objectName
    global originalComment
    global getComment_text
    global objHandle

    set objectName [$handle getName]
    set originalComment [$handle getDescription]
    set getComment_text ""
    set objHandle $handle

    # If there is any possibility of error during commenting (e.g. not 
    # supported), put a check for it in here.

    if {[string compare $objectType "operator"] == 0} {
	if {![$handle isCommentingPossible]} {
	    tk_messageBox -icon error -title "Error" \
		-message "The function that implements this operator cannot be\
                          found.\nAny commenting functionality related to this \
                          operator is therefore disabled."
	    return
	}
    }

    # Create the dialog window

    toplevel .getComment

    # Disable the main program window

    wm transient .getComment .
    grab .getComment

    # Set the window title
   
    wm title .getComment $windowTitle

    # Create a frame to hold the entry widget
   
    frame .getComment.commentFrame -relief groove -borderwidth 1

    # Create an entry widget

    iwidgets::entryfield .getComment.commentFrame.entryBox -textbackground white -labeltext $dialogPrompt -width 40
    
    # Display the current comment associated with the object in the entry widget
    
    .getComment.commentFrame.entryBox insert 0 $originalComment

    # Pack the entry widget

    pack .getComment.commentFrame -side top -fill x -expand y -ipady 2 -ipadx 2
    pack .getComment.commentFrame.entryBox -side top -fill x -expand y

    # When the box appears to enter the name, immediately give focus to the
    # text box and allow the user to hit Enter instead of clicking OK to proceed.
    set comp [.getComment.commentFrame.entryBox component entry]
    focus $comp
    bind $comp <Return> {
        if {[string length [.getComment.commentFrame.entryBox get]] > 0} {
	    _getComment_OK
	}
    }

    # Add the ok, cancel, and clear buttons 

    # If user is hitting OK, make sure he knows he is deleting the commnent
    button .getComment.okButton -text "OK" -underline 0 \
	-command "checkConfirmationOnNullCommentAndExit"

    button .getComment.clearButton -text "Clear" -underline 4 \
	-command "_getComment_Clear"
    
    button .getComment.cancelButton -text "Cancel" -underline 0 \
	-command "_getComment_Cancel"
   
    bind .getComment <Alt-o> {checkConfirmationOnNullCommentAndExit; break}
    bind .getComment <Alt-r> {_getComment_Clear; break}
    bind .getComment <Alt-c> {_getComment_Cancel; break}
    bind .getComment.okButton <Return> {checkConfirmationOnNullCommentAndExit; break}
    bind .getComment.clearButton <Return> {_getComment_Clear; break}
    bind .getComment.cancelButton <Return> {_getComment_Cancel; break}
    bind .getComment <Escape> {_getComment_Cancel; break}

    pack .getComment.cancelButton .getComment.clearButton .getComment.okButton -side right 

    tkwait window .getComment

    return $getComment_text
}


proc checkConfirmationOnNullCommentAndExit {} {
    global objectName
    global originalComment
    global objHandle
    global originalComment

    set getComment_text [.getComment.commentFrame.entryBox get]

    if {[string length $getComment_text] == 0 && [string length $originalComment] != 0 } {

	eval "deleteMessageRaise -objectHandle $objHandle -objectType comment -destructiveAction commentDrop -deleteName {$objectName}"
	_getComment_Cancel
	
	# Issue a refresh
	hyperlinker selectorMoveRequest [hyperlinkWatcher getCurrentLocation]

    } else {
	_getComment_OK
    }
}

# End of file.
