saveDfRda <- function(reposDF, outFile="repdatadesc.rda") {
    ## DF can be of any type, but parameter "type" must define
    ## it properly for correct operations

    reposDF
    save(reposDF,file=outFile)
}

getPackDcfMatrix <- function(file, fields=c("Package","Version"),
                             mangSep=":v:") {
    ## Will retrieve the fields requested from the filename
    ## Will remove the first field and use that as the set of
    ## rownames, then return the result
    mtrx <- read.dcf(file,fields)

    if (ncol(mtrx) > 1) {
        mtrx <- mangleRowNames(mtrx,mangSep)
    }
    else if (ncol(mtrx) == 1) {
        rownames(mtrx) <- mtrx[,1]
    }
    else {
        mtrx <- NULL
    }

    return(mtrx)
}

getPackDcfList <- function(file, fields=c("Package","Version"),
                           mangSep=":v:") {
    ## Will retrieve the fields requested from the filename
    ## Will remove the first field and use that as the set of
    ## rownames, then return the result
    mtrx <- read.dcf(file,fields)

    if (ncol(mtrx) > 1) {
        mtrx <- mangleRowNames(mtrx,mangSep)
    }
    else if (ncol(mtrx) == 1) {
        rownames(mtrx) <- mtrx[,1]
    }

    dcfList <- split(mtrx, col(mtrx))
    dcfList[[length(dcfList)+1]] <- rownames(mtrx)
    names(dcfList) <- c(colnames(mtrx),"Mangled")

    return(dcfList)
}


reposDfFromList <- function(dcfList, fields=c("Package","Version")) {
    ## Creates a data frame where each column is constructed by
    ## an element of the passed in list.  Column names are from the
    ## list element names.  The last element of the list becomes the
    ## row.names of the data frame.

    ## Sanity check list
    if (length(dcfList) != (length(fields)+1)) {
        emsg <- "List does not contain the proper elements"
        stop(emsg)
    }

    reposDf <- data.frame(I(dcfList[[1]]))
    for (i in 2:(length(fields))) {
        reposDf <- cbind(reposDf,I(dcfList[[i]]))
    }

    colnames(reposDf) <- fields
    rownames(reposDf) <- dcfList[[length(dcfList)]]
    return(reposDf)
}

splitDFstrings <- function(df, fields, delim=",[[:space:]]*") {
    ## Will take a data frame and a set of field names.  For all those
    ## fields, will split the current string field into a vector of
    ## strings based on 'delim'.
    for (i in 1:length(fields)) {
        if (!all(is.na(df[,fields[i]]))) {
            tmp <- paste("df$",fields[i],
                         " <- I(strsplit(df$",
                         fields[i],", delim))",sep="")
            eval(parse(text=tmp))
        }
    }
    return(df)
}


buildPkgDf <- function(PACKAGEfile="PACKAGES.in", infoDir = ".",
                       prefixPath=NULL, mangSep=":v:") {
    ## Builds the repository Rda file from various sources

    cDir <- getwd()
    on.exit(setwd(cDir))
    setwd(infoDir)

    ## Note the fields to be used in the Rda file from
    ## PACKAGE.in

    ## FIXME: "Uses" is deprecated, only in here for backwards
    ##        compatability.  Remove for BioC 1.6.
    fields <- c("Package","Version", "Keywords", "Depends", "Title",
                "Suggests", "Imports", "Replaces", "Description", "URL",
                "Author", "Maintainer", "License", "Status",
                "Priority", "SystemRequirements", "ReleaseLevel",
                "Contains", "Uses")

    dcfList <- getPackDcfList(PACKAGEfile, fields, mangSep)
    ## Change the matrix into a data frame
    df <- reposDfFromList(dcfList,fields)
    ## Change any of the fields which are supposed to be vectors that
    ## are strings into vectors
    stringVex <- c("Keywords", "Depends", "Suggests",
                   "Imports","Replaces")
    df <- splitDFstrings(df, stringVex)
    df <- splitDFstrings(df, "Contains", delim="[[:space:]]+")
    ## Replace version strings w/ VersionNumber objects
    df <- replaceDFversions(df)
    ## Create an empty matrix containing any fields that are not yet
    ## to be filled in, then attach it to the data frame
    xFields <- c("OSspecific","KeySearchOrder")
    xMtrx <- matrix(nrow=nrow(df),ncol=length(xFields))
    colnames(xMtrx) <- xFields
    df <- cbind(df,xMtrx)

    df <- addOsSpecific(df, prefixPath)
    class(df) <- c(class(df),"Pkg")
    return(df)
}

replaceDFversions <- function(df) {
    ## Will replace all of the df's version strings w/ VersionNumber
    ## objects

    tmpVers <- list()
    for (i in seq(along=row.names(df))) {
        tmpVers[[i]] <- buildVersionNumber(as.character(df[i,]$Version))
    }
    df$Version <- I(tmpVers)
    return(df)
}

addOsSpecific <- function(df, prefixPath=NULL, ext=".status",mangSep=":v:") {
## Adds in specific package information
    oss <- dir(pattern=paste(".*",ext,sep=""))
    fields <- c("File","Status","Rvers", "Date")

    df$OSspecific <- list()
    for (i in seq(along=row.names(df))) {
        df$OSspecific[[i]] <- list(NA)
    }

    for (j in seq(along=oss)) {
        curOS <- gsub(ext,"",oss[j])
        osMtrx <- getReposSpecialFile(df, oss[j], mangSep)

        ## Add in any prefixPath
        if (!is.null(prefixPath)) {
            osMtrx[,"File"] <- file.path(prefixPath, osMtrx[,"File"])
        }
        for (i in 1:nrow(osMtrx)) {
            mangRow <- rownames(osMtrx)[i]
            whichDFRow <- match(mangRow,row.names(df))

            osFields <- osMtrx[i,fields]
            tmpList <- as.list(osFields)

            if (is.null(df$OSspecific[[whichDFRow]][[1]]$File)) {
                df$OSspecific[[whichDFRow]] <- list(tmpList)
                names(df$OSspecific[[whichDFRow]]) <- curOS
            }
            else {
                tmpNames <- names(df$OSspecific[[whichDFRow]])
                df$OSspecific[[whichDFRow]][[j]] <- tmpList
                tmpNames <- c(tmpNames,curOS)
                names(df$OSspecific[[whichDFRow]]) <- tmpNames
            }
        }
    }
    return(df)
}

getReposSpecialFile <- function(df, fileName, mangSep) {
    ## Read it in via Dcf, mangle and create rownames,
    ## Determine which rows are in the dataframe, and then
    ## return the resultant matrix
    mtrx <- read.dcf(fileName)
    mtrx <- fillBlankVersions(mtrx,df)
    mtrx <- mangleRowNames(mtrx,mangSep)
    ## Determine which mangled names are in the dataframe
    found <- rownames(mtrx) %in% row.names(df)
    mtrx <- mtrx[found,,drop=FALSE]
    return(mtrx)
}

fillBlankVersions <- function(mtrx, df) {
    ## Will take the repos file dcf matrix and find situations where
    ## there is no version #.  Will then match it to the highest
    ## version # available for that package.  If no such package is
    ## found, will drop from matrix
    blanks <- which(mtrx[,"Version"]=="NA")
    for (blank in blanks) {
        blankPkg <- mtrx[blank,"Package"]
        pkgMatches <- which(df$Package == blankPkg)
        pkgVersions <- df[pkgMatches,]$Version
        ## !! see comments in getMaxVersion
        blankVers <- getMaxVersion(pkgVersions)
        ## We still want to use characters at this point ...
        mtrx[blank,"Version"] <- as.character(blankVers)
    }
    return(mtrx)
}

mangleRowNames <- function(mtrx, mangSep) {
    ## Given a matrix, will combine the first two fields
    ## separated by 'mangSep', and will make these the row names
    if (ncol(mtrx) < 2) {
        return(mtrx)
    }
    rownames(mtrx) <- paste(mtrx[,1],mtrx[,2],sep=mangSep)
    return(mtrx)
}

buildVigDf <- function(VIGfile="Vignettes.in", infoDir = ".") {
    cDir <- getwd()
    on.exit(setwd(cDir))
    setwd(infoDir)
    vigFields <- c("VignetteIndexEntry","VignetteTitle", "VignettePackage",
                   "VignetteVersion", "VignetteDepends",
                   "VignetteKeywords", "PDFpath")
    vigMtrx <- read.dcf(VIGfile,vigFields)
    vigDcfList <- split(vigMtrx, col(vigMtrx))
    vDf <- data.frame(I(vigDcfList[[1]]))
    for (i in 2:length(vigFields)) {
        vDf <- cbind(vDf,I(vigDcfList[[i]]))
    }
    colnames(vDf) <- vigFields
    vDf <- splitDFstrings(vDf,c("VignetteDepends","VignetteKeywords"))
    class(vDf) <- c(class(vDf),"Vignette")
    return(vDf)
}

