reposToolsCurrentInstalls <- list()




syncLocalLibList <- function(lib=reposToolsLibPaths(), quiet=TRUE) {
    ## Need to see if there currently exists a 'liblisting.Rda' in lib
    ## if so, load it, otherwise set the object NULL
    if (!quiet)
        cat("\nSynching your local package management information ...\n")

    ## Create a locale for any temporary lib dirs, if needed
    tmpLibDir <- file.path(tempdir(),"tempLibs")
    if (! file.exists(tmpLibDir))
        dir.create(tmpLibDir)

    ## Now sync the local pkg info
    for (i in seq(along=lib)) {
        if (!file.exists(lib[i]))
            next

        ## Check to make sure we can write to this lib
        if (file.access(lib[i],mode=2) != 0) {
            if (!quiet) {
                cat("\n\t")
                rt_note(paste("reposTools can not access ",lib[i],".\n\t",
                           "This will not affect your R session unless ",
                           "you wish \n\tto install/update/remove packages ",
                           "from this directory\n",sep=""))
            }
            writeable <- FALSE
        }
        else
            writeable <- TRUE


        if (load.locLib(lib[i]) == FALSE) {
            locLibList <- NULL
        }

        if (length(dir(lib[i])) > 0) {
            ## Get listing of all packages installed in this library
            options(show.error.messages=FALSE)
            on.exit(options(show.error.messages=TRUE), add=TRUE)
            pkgs <- try(installed.packages(lib[i]))
            options(show.error.messages=TRUE)
            if (is(pkgs,"try-error")) {
                warning(paste(lib[i],"does not seem to be a valid",
                              "R package library, skipping ..."))
                next
            }

            if (length(pkgs) == 0)
                next

            ## Figure out which packages are already in the database,
            ## and which ones are new
            curLocPkgs <- unlist(lapply(locLibList, Package))
            oldPkgs <- pkgs[,"Package"] %in% curLocPkgs
            old <- pkgs[oldPkgs,,drop=FALSE]
            new <- pkgs[!oldPkgs,,drop=FALSE]
            gonePkgs <- which(!(curLocPkgs %in% pkgs[,"Package"]))

            if (nrow(old) > 0) {
                ## Need to find any updated packages ...
                remPkgs <- numeric()
                for (j in 1:nrow(old)) {
                    ## For each package, check to see if any of its
                    ## version has changed, if so, add it to the 'new' list
                    llPkg <- match(old[j,"Package"],curLocPkgs)
                    if (buildVersionNumber(old[j,"Version"])
                        != PkgVersion(locLibList[[llPkg]])) {
                        new <- rbind(new,old[j,])
                        remPkgs <- c(remPkgs, llPkg)
                    }
                }
                if (length(remPkgs) > 0)
                    locLibList <- locLibList[-remPkgs]
            }

            if (nrow(new) > 0) {
                locLibFields <- c("Package","Version","Keywords","Repos",
                                  "Depends", "Suggests", "Imports", "Bundle")
                ## Extract the information for any new packages
                locLibMtrx <- matrix(nrow=nrow(new),ncol=length(locLibFields))
                colnames(locLibMtrx) <- locLibFields
                pkgsFlds <- colnames(locLibMtrx)[colnames(locLibMtrx) %in% colnames(pkgs)]
                locLibMtrx[,pkgsFlds] <- new[,pkgsFlds]
                locLibMtrx[is.na(locLibMtrx)] <- "NA"
                ## Convert versions to VersionNumber
                vers <- locLibMtrx[,"Version"]
                versList <- list()
                for (k in 1:length(vers)) {
                    versList[[k]] <- buildVersionNumber(vers[k])
                }

                for (l in 1:nrow(locLibMtrx))
                    locLibList[[length(locLibList)+1]] <- new("localPkg",
                                                              Package=locLibMtrx[l,1],
                                                              PkgVersion=versList[[l]],
                                                              Keywords=locLibMtrx[l,3],
                                                              Repos=locLibMtrx[l,4],
                                                              Depends=locLibMtrx[l,5],
                                                              Suggests=locLibMtrx[l,6],
                                                              Imports=locLibMtrx[l,7],
                                                              Bundle=locLibMtrx[l,8])

                outNewPkgs <- paste(locLibMtrx[,"Package"], collapse="\n\t")
                ## If this is a non-writeable directory, packages
                ## there will show up as "new" but they shouldn't
                ## be reported.
                if ((writeable)&&(!quiet))
                    cat("Packages which have been added/updated:",
                        outNewPkgs,"\n",sep="\n\t")
            }


            if (length(gonePkgs) > 0) {
                ## Also need to detect removed packages
                outPkgs <-
                    paste(curLocPkgs[gonePkgs],collapse="\n\t")
                if (!quiet)
                    cat("Packages which have been removed:",outPkgs,"\n",
                        sep="\n\t")
                locLibList <- locLibList[-(gonePkgs)]
            }

            if (writeable)
                save.locLib(locLibList, lib[i])
            else {
                ## !! write the locliblist to a temporary directory
                curDir <- tempfile(tmpdir=tmpLibDir)
                dir.create(curDir)
                save.locLib(locLibList, curDir)
            }
        }
    }
}

load.locLib <- function(lib=reposToolsLibPaths()[1]) {
    if (length(lib) != 1) {
        warning("load.locLib() does not accept vectorized inputs")
        return(FALSE)
    }

    libFile <- file.path(lib,"liblisting.Rda")
    if (file.exists(libFile)) {
        load(libFile,envir=parent.frame())
        return(TRUE)
    }
    else {
        return(FALSE)
    }
}

closeLocLib <- function() {
    ## Removes the locLibList w/o saving
    ## Probably unecessary in most cases, but can be used to prevent
    ## confusion.
    if ("locLibList" %in% ls(envir=parent.frame()))
        rm("locLibList",envir=parent.frame())
}

save.locLib <- function(locLibList, lib) {
    locLibList
    libFile <- file.path(lib,"liblisting.Rda")

    if (file.access(lib,mode=2) == 0) {
        save(locLibList, file=libFile)
    }
    else {
       warning(paste("Incorrect permissions to edit package database,",
                   libFile))
    }
}

getLocalPkgs <- function(pkgs, libs=reposToolsLibPaths()) {
    pkgInfos <- list()

    for (lib in libs) {
        if (load.locLib(lib) == FALSE) {
            next
        }
c
        locPkgs <- lapply(locLibList, Package)
        locVers <- lapply(locLibList, PkgVersion)

        if (missing(pkgs))
            pkgs <- locPkgs

        for (i in seq(along=locPkgs)) {
            if (locPkgs[i] %in% pkgs)
                pkgInfos[[length(pkgInfos)+1]] <-
                    buildPkgInfo(as.character(locPkgs[[i]]),
                                 locVers[i][[1]])
        }
        closeLocLib()
    }

    return(pkgInfos)
}

findInstallDepends <- function(pkg, vers, libList, depColName, remove=FALSE) {
    out <- NULL
    if (length(libList) == 0)
        return(out)

    depCol <- switch(depColName,
                     "depends on"=lapply(libList,Depends),
                     "suggests"=lapply(libList,Suggests),
                     "imports"=lapply(libList,Imports),
                     stop("Invalid depColName"))

    for (i in 1:length(depCol)) {
         curDeps <- depCol[[i]]
        if ((length(curDeps) == 1)&&((is.na(curDeps))||
                   (curDeps == "NA")))
            next
        if (length(curDeps) != 0) {
            depMtrx <- getDependsMatrix(curDeps)
            if (is.null(depMtrx))
                next

            if (length(row <- grep(paste("^",pkg,"$",sep=""),
                                   depMtrx[,1])) > 0) {
                if (remove == TRUE) {
                    ## This is going to break a depency no matter what
                    out <- c(out,paste(Package(libList[[i]]),"version",
                                       stringRep(PkgVersion(libList[[i]])),
                                       depColName,
                                       depMtrx[row,1],depMtrx[row,2]))
                }
                else if (depMtrx[row,2] != "") {
                    ## Specific version requirement.  If they're not
                    ## removing the file, and there's no version req
                    ## then no dep problem
                    if (checkVers(vers,getReqOper(depMtrx[row,2]),
                                  getReqVers(depMtrx[row,2])) == FALSE) {
                        ## Broken dependency
                        out <- c(out,paste(Package(libList[[i]]),"version",
                                           stringRep(PkgVersion(libList[[i]])),
                                           depColName,
                                           depMtrx[row,1],depMtrx[row,2]))
                    }
                }
            }
        }
    }
    return(out)
}

is.installed <- function(pkg, vers=NULL,
                         verbose=TRUE, oper="==", libs) {
    ## Given a package name will check to see if it is installed
    ## if vers != NULL, will look for a specific version
    ## if oper != NULL and vers != NULL, won't look for a specific
    ## version, but rather will look to see if there's a version
    ## that satifies 'version oper vers'
    if ((!is.null(vers))&&(!is(vers, "VersionNumber")))
        vers <- buildVersionNumber(vers)

    if (missing(libs))
        libs <- reposToolsLibPaths()

    for (lib in libs) {
        if (load.locLib(lib)) {
            locPkgs <- unlist(lapply(locLibList, Package))
            whichPkg <- which(locPkgs == pkg)

            if (length(whichPkg) != 0) {
                ## Found package
                if (is.null(vers)) {
                    return(TRUE)
                }
                else {
                    pkgVers <- lapply(locLibList[whichPkg],PkgVersion)
                    if (any(unlist(lapply(pkgVers,checkVers, oper,vers))))
                        return(TRUE)
                }
            }
            closeLocLib()
        }
        else {
            if (verbose == TRUE)
                rt_note(paste("No locliblisting in R library",lib))
        }
    }
    return(FALSE)
}

getPkgVers <- function(pkg, libs=reposToolsLibPaths(), verbose=TRUE) {
    ## Will get the version of an installed package
    ## returns a list of versions if multiple installs
    ## returns an empty list if not intalled
    vers <- list()
    for (lib in libs) {
        if (load.locLib(lib)) {
            locPkgs <- unlist(lapply(locLibList, Package))
            whichPkg <- which(locPkgs == pkg)
            if (length(whichPkg) > 0) {
                vers[length(vers)+1] <- lapply(locLibList, PkgVersion)[whichPkg]
            }
            else {
                ## if we couldn't find it, it might be a bundle
                ## In this case we know that we'll have multiple
                ## packages all reporting back as part of the same
                ## bundle, so just grab the first one.
                whichPkg <- match(pkg, lapply(locLibList, Bundle))
                if (length(whichPkg) > 0)
                    vers[length(vers)+1] <- lapply(locLibList, PkgVersion)[whichPkg]
            }
            closeLocLib()
        }
        else {
            if (verbose == TRUE)
                rt_note(paste("No locliblisting in R library",lib))
        }
    }
    return(vers)
}

getLocLibURLs <- function(pkgs, lib) {
    ## match up any requested packages to
    ## the 'Repos' field of the locLib.

    if (load.locLib(lib) == FALSE) {
        return(NULL)
    }
    locPkgs <- unlist(lapply(locLibList, Package))
    urls <- unlist(lapply(locLibList,Repos))
    urls <- urls[match(pkgs,locPkgs)]
    if (all(is.na(urls)))
        return(NULL)

    closeLocLib()
    return(as.vector(urls))
}

replaceLocLibBundles <- function(pkgs, lib=.libPaths()[1]) {
    ## Find if any of the requested packages are actually part
    ## of a bundle, and replace it

    if (load.locLib(lib) == FALSE)
        return(pkgs)

    libPkgs <- unlist(lapply(locLibList,Package))
    whichPkgs <- match(pkgs, libPkgs)
    bndls <- unlist(lapply(locLibList, function(x) {
        z <- Bundle(x)
        if (length(z) == 0)
            "NA"
        else
            z
    }))[whichPkgs]
    bNames <- unique(bndls[bndls != "NA"])
    if (length(bNames) > 0) {
        pkgs <- pkgs[-match(bNames, bndls)]
        pkgs <- c(pkgs, bNames)
    }
    pkgs
}


updateLocLib <- function(lib, pkg, vers, repEntry) {
    if (load.locLib(lib) == FALSE) {
        syncLocalLibList(lib)
        load.locLib(lib)
    }

    if (!is(vers,"VersionNumber")) {
        vers <- try(buildVersionNumber(vers))
        if (is(vers, "try-error"))
            stop("Invalid vers input")
    }

    pI <- buildPkgInfo(pkg, vers)

    ## FIXME: This can be done in the usual fashion
    ## for the 1.5 release, providing some backwards compat here
    imports <- imports(repEntry, pI)
    if (is.null(imports))
        imports <- "NA"

    newPkg <- new("localPkg", Package=pkg, PkgVersion=vers,
                  Keywords=keywords(repEntry, pI),
                  Depends=depends(repEntry, pI),
                  Suggests=suggests(repEntry, pI),
                  Imports=imports,
                  Repos=repURL(repEntry))

    listPkg <- length(locLibList)
    curPkgs <- unlist(lapply(locLibList, Package))
    if (is.na(oP <- match(pkg, curPkgs)))
        listPkg <- listPkg + 1
    else
       listPkg <- oP

        locLibList[[listPkg]] <- newPkg

    save.locLib(locLibList, lib)
    return(TRUE)
}

resolve.depends <- function(pkg, repEntry, force=FALSE,
                            forward=TRUE, lib=reposToolsLibPaths()[1],
                            remove=FALSE, depends=TRUE,
                            suggests=TRUE, imports=TRUE,
                            searchOptions=TRUE, getAllDeps=FALSE,
                            versForce=TRUE, method="auto",
                            getNewest=FALSE, develOK=FALSE,
                            curRepList=NULL) {
    ## Passed a pkgInfo object.  Will make sure this package does not
    ## break any current dependencies - it will error if trying a
    ## level w/ a FALSE value, and warn if TRUE.
    errors <- c(depends,suggests,imports)
    if (length(which(errors)) > 0) {
        levels <- c("depends on","suggests","imports")
        halt <- FALSE
        out <- character()

        retEnv <- new.env()
        retPSlist <- new("pkgStatusList",statusList=list())

        if (forward == TRUE) {
            for (i in seq(along=levels)) {
                ## Now check for forward depends, if they need
                ## to obtain any other packages
                fdEnv <- solveForwardDepends(pkg, repEntry,
                                             lib, levels[i],
                                             searchOptions=searchOptions,
                                             getAllDeps=getAllDeps,
                                             method=method,
                                             getNewest=getNewest,
                                             develOK=develOK,
                                             curRepList=curRepList)

                curOut <- get("out", fdEnv)
                statusList(retPSlist) <- get("pkgList",fdEnv)
                out <- c(out, curOut)

                if ((length(curOut) > 0)&&(errors[i] == TRUE))
                    halt <- TRUE
            }
        }

        for (curLib in reposToolsLibPaths()) {
            if (load.locLib(curLib) == TRUE) {
                for (i in seq(along=levels)) {
                    ## Find any "reverse" depends - ie if this breaks anything
                    ## currently installed
                    curOut <- findInstallDepends(pkgName(pkg),pkgVersion(pkg),
                                                 locLibList, levels[i],
                                                 remove)
                    out <- c(out, curOut)
                    if ((length(curOut) > 0)&&(errors[i] == TRUE))
                        halt <- TRUE
                }
                closeLocLib()
            }
        }
    }

    if ((length(out) > 0) && (force==FALSE))
            warning(paste("\n",paste(out,collapse="\n"),"\n"))

    if ((halt == TRUE)&&(force==FALSE))
        assign("halt",TRUE,retEnv)
    else
        assign("halt",FALSE,retEnv)

    assign("pkgList",retPSlist,retEnv)
    return(retEnv)
}

solveForwardDepends <- function(pkgInfo, repEntry, lib, depLevel,
                                searchOptions=TRUE,getAllDeps=FALSE,
                                versForce=TRUE, method="auto",
                                getNewest=getNewest,
                                develOK=FALSE, curRepList=NULL) {
    if (develOK)
      warning("The develOK option is deprecated and will be ignored.\n",
              "If you would like to install development versions of\n",
              "Bioconductor packages, please install a development\n",
              "version of R.")

    pkg <- pkgName(pkgInfo)
    vers <- pkgVersion(pkgInfo)
    curPkgInfo <- new("pkgInfo", pkgName=pkg, pkgVersion=vers)

    ## Begin a hack to double check for circular references
    cIlen <- length(reposToolsCurrentInstalls)+1
    reposToolsCurrentInstalls[[cIlen]] <<- curPkgInfo

    out <- NULL

    retEnv <- new.env()
    retPSlist <- new("pkgStatusList",statusList=list())

    curMtrx <- switch(depLevel,
                      "depends on"=getDependsMatrix(depends(repEntry, pkgInfo)),
                      "suggests"=getDependsMatrix(suggests(repEntry, pkgInfo)),
                      "imports"=getDependsMatrix(imports(repEntry, pkgInfo))
                      )
    ## Now have a dependency matrix for this package on the
    ## current level of depends
    for (i in seq(along=curMtrx[,1])) {
        depPkg <- curMtrx[i,1]
        ## If this is a R depend and it has a version (we already
        ## know they have *a* version of R), check it
        if (depPkg == "R") {
            if (curMtrx[i,2]!="") {
                ## Handle R depends differently
                curRvers <- buildRversnum()
                depVers <- getReqVers(curMtrx[i,2])
                depOper <- getReqOper(curMtrx[i,2])
                if (checkVers(curRvers, depOper, depVers) == FALSE)
                {
                    out <- c(out, paste("Package",pkg,"version",
                                        vers, "requires R version",
                                        depVers,"but you only have",curRvers,
                                        "- please see www.r-project.org to",
                                        "upgrade your R install.\n"))
                }
            }
        }
        else {
            ## Package dependency
            if (curMtrx[i,2] != "") {
                depVers <- getReqVers(curMtrx[i,2])
                depOper <- getReqOper(curMtrx[i,2])
            }
            else
                depOper <- depVers <- NULL

            if (havePackage(depPkg, depVers, depOper) == FALSE) {
                ## Broken dependency
                brk <- TRUE
                errMsg <- paste("Package",pkg,"version",vers,
                                depLevel, depPkg)
                if (!is.null(depVers))
                    errMsg <- paste(errMsg,"version",depOper,depVers)
                if (getAllDeps == FALSE) {
                    ##outErrMsg <- paste(errMsg,", would you like to try ",
                    ##                   "to install this package?",
                    ##                   sep="")
                    ##ans <- userQuery(outErrMsg,c("yes","no"))
                    ans <- "no"
                }
                else
                    ans <- "yes"

                if (ans == "yes") {
                    inOut<- install.packages2(depPkg, curRepList,
                                              lib, getNewest=getNewest,
                                              searchOptions=searchOptions,
                                              getAllDeps=getAllDeps,
                                              method=method,
                                              develOK=FALSE)
                    updPkgs <- updatedPkgs(inOut)
                    ## Now check again to see if we have the package
                    if (length(updPkgs) > 0) {
                        if (havePackage(depPkg, depVers, depOper))
                            brk <- FALSE
                        statusList(retPSlist) <- inOut
                    }
                }
                if (brk == TRUE) {
                    out <- c(out,errMsg)
                    rt_note(paste("Package",depPkg,
                               "not found in any known repository."))
                }
            }
        }
    }

    assign("out",out,retEnv)
    assign("pkgList",retPSlist, retEnv)

    ## Remove this package from the circRef checker
    reposToolsCurrentInstalls <<- reposToolsCurrentInstalls[-cIlen]

    return(retEnv)
}

havePackage <- function(depPkg, depVers, depOper) {
    if (is.installed(depPkg,depVers, verbose=FALSE, depOper))
        return(TRUE)
    else {
        ## check the reposToolsCurrentInstalls list
        pkgNames <- unlist(lapply(reposToolsCurrentInstalls, pkgName))
        if (!is.na(match(depPkg, pkgNames))) {
            ## Package is in the reposToolsCurrentInstalls
            if (is.null(depVers))
                return(TRUE)
            else {
                pkgVers <- lapply(reposToolsCurrentInstalls, pkgVersion)
                if (any(unlist(lapply(pkgVers, checkVers, depOper,
                                      depVers))))
                    return(TRUE)
            }
        }
    }
    return(FALSE)
}

repositories <- function(recurse=TRUE, method="auto") {
    repList <- getOptReposList(recurse, method=method)
    title <- "Available Repositories: Select By Number, 0 For None"
    repIndex <- menu(repNames(repList),title=title)
    if (repIndex == 0)
        return(NULL)
    repEntry <- getRepEntry(repList, repIndex)
    return(repEntry)
}


remove.packages2 <- function(pkgs, lib, force=FALSE) {
    .Deprecated("remove.packages", package="utils")
    if (missing(lib) || is.null(lib)) {
        lib <- reposToolsLibPaths()[1]
        rt_note(paste("argument `lib' is missing: using", lib))
    }
    if (load.locLib(lib) == FALSE) {
        return(FALSE)
    }

    for (pkg in pkgs) {
        if (!is.installed(pkg)) {
            rt_note(paste("Package",pkg,"is not installed"))
            next
        }
        vers <- getPkgVers(pkg, lib)[[1]]
        ## Check to see if removing this package will break any
        ## dependencies.  Suggests level deps will be flagged
        ## here as well (which is not true for installation)
        ## to be extra careful.
        rd <- resolve.depends(buildPkgInfo(pkg,vers),NULL,force,
                              lib=lib, forward=FALSE, remove=TRUE,
                              suggests=TRUE)

        if (get("halt",rd) == TRUE)
            stop(paste("Could not remove package",pkg))

        ## Remove R from the local install
        print(paste("Removing package",pkg,"from system ...."))
        remove.packages(pkg, lib)

        options(show.error.messages=FALSE)
        z <- try(.find.package(pkg, lib))
        options(show.error.messages=TRUE)
        if (! is(z,"try-error")) {
            warning("Could not remove package",pkg)
            next
        }

        ## Remove this package from the localLibList
        locPkgs <- unlist(lapply(locLibList, Package))
        whichPkg <- which(locPkgs == pkg)
        if (length(whichPkg) != 0)
            locLibList <- locLibList[-whichPkg]
        print("Removal complete")

    }
    save.locLib(locLibList,lib)
}


install.packages2 <- function(pkgs=NULL, repEntry=NULL, lib, versions,
                              type, method="auto", theme=NULL,
                              recurse=TRUE, searchOptions=FALSE, getNewest=TRUE,
                              force=FALSE, versForce=TRUE, getAllDeps=FALSE,
                              fileSelect=baseFileSelect, develOK=FALSE) {
    .Deprecated("install.packages", package="utils")
    if (develOK)
      warning("The develOK option is deprecated and will be ignored.\n",
              "If you would like to install development versions of\n",
              "Bioconductor packages, please install a development\n",
              "version of R.")

    if (missing(lib))
        lib <- reposToolsLibPaths()[1]
    else
        if (! lib %in% .libPaths()) {
            paths <- .libPaths()
            paths <- c(lib, paths)
            .libPaths(paths)
        }

    ## We need to make sure that lib is in the R_LIBS
    ## environment variable, due to source installs.  Win32 installs
    ## don't need this, but it doesn't matter if it is here.
    curRLIBSEnv <- Sys.getenv("R_LIBS")
    on.exit(Sys.putenv(R_LIBS=curRLIBSEnv), add=TRUE)
    curRLIBS <- strsplit(curRLIBSEnv, ":")
    if (length(curRLIBS) > 0) {
        if (! lib %in% curRLIBS)
            Sys.putenv(R_LIBS=paste(c(curRLIBS, lib), collapse=":"))
    }

    ## We need one of pkgs, repEntry or theme to be defined
    if (is.null(repEntry) && is.null(pkgs) && is.null(theme)) {
        repEntry <- repositories()
        if (is.null(repEntry))
            stop(paste("No valid repository was chosen.",
                       "\nCan not continue."))
    }

    ## Now generate the repository list to use
    if (is(repEntry, "ReposList"))
        repList <- repEntry
    else
        repList <- buildRepList(repEntry, recurse, searchOptions, method=method)

    ## Double check to make sure there are entries
    if (numReps(repList) == 0)
        stop("No repositories to search")

    if (is.null(theme)) {
        if (missing(versions))
            versions <- NULL
        else {
            if (length(pkgs) == 0)
                return(new("pkgStatusList",statusList=list()))

            if (length(pkgs) != length(versions))
                stop("Must supply the same number of values in ",
                     "version as are specified in pkgs")
            ## Create a list of pkgInfo objects
            newPkgs <- vector(mode="list",length=length(pkgs))

            for (i in 1:length(pkgs))
                newPkgs[[i]] <- buildPkgInfo(pkgs[i], versions[i])
            pkgs <- newPkgs
        }
        ## If pkgs is NULL, we want all packages in the repository
        if (is.null(pkgs))
            pkgs <- repPkgs(repEntry)
    }
    else {
        if (is.null(repEntry))
            checkRep <- repList(repList)
        else
            checkRep <- list(repEntry)

        pkgs <- list()
        for (curTheme in seq(along=theme)) {
            for (curRepEntry in seq(along=checkRep)) {
                pkgs <- c(pkgs, getReposThemePkgs(theme[curTheme],
                                                  checkRep[[curRepEntry]]))
            }
            if (length(pkgs) == 0)
                rt_note(paste("Theme '", theme[curTheme],
                           "' was not found.", sep=""))
        }
        if (length(pkgs) == 0)
            return(invisible(NULL))
        versions <- sapply(lapply(pkgs, pkgVersion), as.character)
    }

    ## Verify the download type
    if (missing(type) || is.null(type)) {
        bioCOpt <- getOption("BioC")
        type <- bioCOpt$reposEntry$type
        if (is.null(type))
            stop("Can not determine default download type")
        else
            rt_note(paste("Using download type:", type))
    }

    ## Make sure that the requested lib directory is synch'ed
    syncLocalLibList(lib)

    pStatList <- new("pkgStatusList",statusList=list())

    if (length(pkgs) > 0) {

        ## Retrieve the pkgList object for this download
        pkgList <- buildPkgListing(repList, pkgs, type,
                                   develOK=FALSE,method=method)

        if (!is.null(versions)) {
            ## If versions were specified then we are downloading
            ## exactly what was found.  Otherwise need to select
            ## from what was available
            statusList(pStatList) <- installPkgListing(pkgList, lib, type, method,
                                                       versForce=versForce,
                                                       force=force,
                                                       searchOptions=searchOptions,
                                                       getAllDeps=getAllDeps,
                                                       getNewest=getNewest)
        }
        else {
            statusList(pStatList) <- fileSelect(pkgList, lib, type,
                                                getNewest=getNewest,
                                                versForce=versForce, force=force,
                                                searchOptions=searchOptions,
                                                getAllDeps=getAllDeps,
                                                method=method,
                                                develOK=FALSE,
                                                curRepList=repList)
        }
    }

    ## FIXME:
    pkgsDLd <- foundPkgs(pStatList)
    if (length(pkgsDLd) < length(pkgs)) {
        nfPkgs <- pkgs[! pkgs %in% pkgsDLd]
        availPkgs <- repPkgs(repList)

        for (i in seq(along = nfPkgs)) {
            if (length(nfPkgs[i]) > 0) {
                sim <- agrep(nfPkgs[i], availPkgs,
                             max.distance=list(all=0.05,
                             insertions=3, deletions=3, substitutions=3))
                if (length(sim) > 0) {
                    newList <- vector(mode="list", length=1)
                    names(newList) <- nfPkgs[i]
                    newList[[1]] <- availPkgs[sim]
                    matchesList(pStatList) <- newList
                }
            }
        }
    }

    return(pStatList)
}


download.packages2 <- function(pkgs, repEntry, destDir, versions,
                               type, method="auto") {
    .Deprecated("download.packages", package="utils")
    if (missing(repEntry))
        stop("No repository provided")
    else {
        if (is(repEntry,"ReposEntry")) {
            if (repType(repEntry) != "package")
                stop("download.packages2 requires a package repository")
        }
        else {
            stop("Supplied repEntry argument does not point to a valid repository.")
        }
    }

    if (missing(destDir)) {
        rt_note("destDir parameter missing, using current directory")
        destDir <- getwd()
    }
    else
        path.expand(destDir)

    if (missing(type) || is.null(type)) {
        bioCOpt <- getOption("BioC")
        type <- bioCOpt$reposEntry$type
        if (is.null(type))
            stop("Can not determine default download type")
        else
            rt_note(paste("Using download type:", type))
    }

    if (missing(pkgs)) {
        rt_note("pkgs parameter missing, downloading all packages in repository")
        pkgs <- repPkgs(repEntry)
    }

    if (missing(versions)) {
        ## Get the listing of pkgs, filenames, versions, etc from the repository
        repPkgDetails <- repObjects(repEntry, pkgs, type=type)
    }
    else {
        if (length(versions) != length(pkgs))
            stop("Must supply the same number of values in version ",
                 "as are specified in pkgs.")
    }

    filenames <- character()

    for (i in seq(along=pkgs)) {
        if (missing(versions)) {
            ## Get the highest version of the package in the repository
            whichPkgs <- which(repPkgDetails[,"pkg"] == pkgs[i])
            if (length(whichPkgs) == 0)
                stop("Package ", pkgs[i], " is not available at ",
                     repURL(repEntry))
            else
                pkgDetails <- repPkgDetails[whichPkgs,,drop=FALSE]

            ## Extract the set of version numbers, convert them
            ## to VersionNumber objects and use getMaxVersion()
            pkgVersions <- pkgDetails[,"vers"]
            pkgVersions <- lapply(pkgVersions, buildVersionNumber)
            pkgVersion <- getMaxVersion(pkgVersions)
        }
        else {
            ## Just use the supplied version
            pkgVersion <- versions[i]
        }

        fileName <- downloadFile(repEntry, pkgs[i], pkgVersion,
                                     type, method=method, dest=destDir)
        if (is.null(fileName))
            warning("Package ", pkgs[i], " version ", versions[i],
                    "is not available at ", repURL(repEntry))
        else
            filenames <- c(filenames, fileName)
    }
    return(filenames)
}


update.packages2 <- function(pkgs=NULL, repEntry=NULL, theme=NULL,
                             libs=reposToolsLibPaths(),
                             type, method="auto", prevRepos=TRUE, recurse=TRUE,
                             searchOptions=FALSE, force=FALSE, versForce=TRUE,
                             getAllDeps=FALSE, develOK=FALSE) {
    .Deprecated("update.packages", package="utils")

    if (develOK)
      warning("The develOK option is deprecated and will be ignored.\n",
              "If you would like to install development versions of\n",
              "Bioconductor packages, please install a development\n",
              "version of R.")

    ## First get the list of libraries that we are updating
    if (missing(libs) || is.null(libs)) {
        libs <- reposToolsLibPaths()
        rt_note(paste("argument `lib' is missing: using", libs))
    }

    ## Make sure that these libraries are synch'd
    syncLocalLibList(libs, quiet=TRUE)

    repList <- buildRepList(repEntry, recurse, searchOptions, method=method)

    ## Get the list of packages to update
    if (! is.null(theme)) {
        if (is.null(repEntry))
            checkRep <- repList(repList)
        else
            checkRep <- list(repEntry)

        pkgs <- list()
        for (curRepEntry in seq(along=checkRep)) {
            for (curTheme in seq(along=theme)) {
                pkgs <- c(pkgs, getReposThemePkgs(theme[curTheme],
                                                  checkRep[[curRepEntry]]))
            }
        }
        ## FIXME: Now subset against what is installed
        pkgs <- unlist(lapply(pkgs, pkgName))
        pkgs <- pkgs[unlist(lapply(pkgs, is.installed, libs=libs))]
    }
    else if (is.null(pkgs)) {
        for (i in 1:length(libs)) {
            if (load.locLib(libs[i]) == FALSE)
                stop("Invalid library directory: ",libs[i])
            pkgs <- c(pkgs, unlist(lapply(locLibList,Package)))
            closeLocLib()
        }
    }

    if (missing(type) || is.null(type)) {
        bioCOpt <- getOption("BioC")
        type <- bioCOpt$reposEntry$type
        if (is.null(type))
            stop("Can not determine default download type")
        else
            rt_note(paste("Using download type:", type))
    }

    out <- vector(mode="list", length=length(libs))
    names(out) <- libs

    for (lib in libs) {
        instURLs <- NULL

        ## go through each package requested and see if it is really
        ## part of a bundle
        pkgs <- replaceLocLibBundles(pkgs, lib)

        if (prevRepos) {
            ## First check all the prevRepos packages
            instList <- checkPrevRepos(pkgs, lib, type, force,
                                       versForce, recurse, getAllDeps,
                                       method)
            instURLs <- instList$toUpdate
            pkgs <- pkgs[! pkgs %in% instList$pkgs]
        }

        if (length(pkgs) != 0) {

            pkgList <- buildPkgListing(repList, pkgs, type,
                                       develOK=FALSE, method=method)
            Packages <- pkgList(pkgList)
            if (length(Packages) == 0)
                return(new("pkgStatusList",statusList=list()))
            for (i in 1:length(Packages)) {
                rep <- 1
                index <- 1
                for (repCheck in 1:length(Packages[[i]])) {
                    curMax <- buildVersionNumber("0")
                    idx <- getMaxElement(Packages[[i]][[rep]])
                    if (Packages[[i]][[rep]][[idx]] >= curMax) {
                        index <- idx
                        rep <- repCheck
                    }
                }
                pkgName <- names(Packages)[i]
                pkgVer <- Packages[[i]][[rep]][[index]]
                if (pkgVer > getMaxVersion(getPkgVers(pkgName, lib))) {
                    pkgInfo <- buildPkgInfo(pkgName, pkgVer)
                    repIndex <- as.numeric(names(Packages[[i]][rep]))
                    curURL <- getRepURL(pkgList, repIndex)
                    whichName <- which(names(instURLs) == curURL)
                    if (length(whichName) > 0) {
                        len <- length(instURLs[[curURL]])
                        instURLs[[curURL]][[len+1]] <- pkgInfo
                    }
                    else {
                        instURLs[[curURL]] <- list(pkgInfo)
                    }
                }
            }
        }

        pStatList <- new("pkgStatusList",statusList=list())

        if (!is.null(instURLs)) {
            for (i in 1:length(instURLs)) {
                pkgs <- unlist(lapply(instURLs[[i]], function(x) {
                    pkgName(x)}))
                statusList(pStatList) <-
                    install.packages2(pkgs,
                                      getReposEntry(names(instURLs)[i], method=method),
                                      lib,
                                      unlist(lapply(instURLs[[i]],
                                                    function(x) {
                                                        as.character(pkgVersion(x))})),
                                      type=type, method=method,
                                      recurse=recurse,
                                      searchOptions=searchOptions,
                                      force=force, versForce=versForce,
                                      getAllDeps=getAllDeps,
                                      develOK=FALSE)
            }
            out[[lib]] <- pStatList
        }
    }
    lapply(out, function(x) {if (is.null(x))
                                 paste("No updates found for this",
                                       "library directory")
                             else
                                 x})
}

checkPrevRepos <- function(pkgs, lib, type, force, versForce,
                           searchOptions, recurse, getAllDeps,
                           method) {
    ## This checks the any previous repositories - returns a list
    ## where the first element is the list of packages that have
    ## specified repositories and the second element is those
    ## packages with avail updates.

    instList <- NULL
    outPkgs <- character()

    urls <- getLocLibURLs(pkgs, lib)

    if (length(urls) == 0)
        return(list(pkgs=outPkgs, toUpdate=instList))

    uSplit <- split(1:length(urls), urls)
    uSplit <- uSplit[names(uSplit) != "NA"]

    if (length(uSplit) > 0) {

        for (i in 1:length(uSplit)) {
            repE <- getReposEntry(names(uSplit)[i], method=method)
            if (!is(repE, "reposEntry"))
                next
            curPkgs <- pkgs[ uSplit[[i]] ]
            outPkgs <- c(outPkgs, curPkgs)
            for (pkg in curPkgs) {
                curVers <- getPkgVers(pkg, lib)[[1]]
                maxRepInfo <- getMaxVersReposInfo(pkg, repE, type)
                if ((!is.null(maxRepInfo))&&(pkgVersion(maxRepInfo) >
                                             curVers)) {
                    whichName <- which(names(instList) ==
                                       names(uSplit)[i])
                    if (length(whichName) > 0) {
                        len <- length(instList[[whichName]])
                        instList[[whichName]][[len + 1]] <- maxRepInfo
                    }
                    else {
                        instList[[length(instList)+1]] <- maxRepInfo
                        names(instList)[length(instList)] <- names(uSplit)[i]
                    }
                }
            }
        }
    }

    list(pkgs=unique(outPkgs), toUpdate=instList)
}

baseFileSelect <- function(pkgList, lib, type, getNewest=TRUE,
                           versForce=TRUE, force=FALSE, searchOptions=FALSE,
                           getAllDeps=TRUE, method="auto",
                           develOK=FALSE, curRepList=NULL) {
    if (develOK)
      warning("The develOK option is deprecated and will be ignored.\n",
              "If you would like to install development versions of\n",
              "Bioconductor packages, please install a development\n",
              "version of R.")

    pkgStats <- new("pkgStatusList",statusList=list())
    Packages <- pkgList(pkgList)
    if (length(Packages) == 0)
        return(pkgStats)
    pkgs <- packages(pkgList)

    ## gotPkgs denotes packages that were acquired as a side effect
    ## of dependency checking.  Some of these might be ones that we
    ## were already looking for, so need to detect this
    gotPkgs <- NULL

    ## Packages is a list where each element is one of the
    ## packages we are looking for, with version & repository
    ## information on where it was found.
    for (i in 1:length(Packages)) {
        ## See if we already have this package
        if (pkgs[i] %in% gotPkgs)
            next

        ok <- TRUE

        ## Select which version of this package to retrieve
        if (getNewest == TRUE) {
            ## Simply find the newest version of each package and
            ## download
            rep <- 1
            index <- 1
            for (repCheck in 1:length(Packages[[i]])) {
                curMax <- buildVersionNumber("0")
                idx <- getMaxElement(Packages[[i]][[rep]])
                if (Packages[[i]][[rep]][[idx]] >= curMax) {
                    index <- idx
                    rep <- repCheck
                }
            }
        }
        else {
            ## Provide a menu selection of the available downloads and
            ## get the one selected in each case
            pvl <- pkgVersionList(pkgList, pkgs[i], "names")
            if (length(pvl) == 0) {
                next()
            }
            choices <- vector()
            repNames <- names(pvl)
            choiceReps <- vector()
            for (curRN in 1:length(pvl)) {
                curPvl <- pvl[[curRN]]
                for (curVer in 1:length(curPvl)) {
                    choiceReps <- c(choiceReps, curRN)
                    choices <- c(choices,
                                 paste(repNames[curRN],
                                       "has version",
                                       stringRep(curPvl[[curVer]])))
                }
            }

            if (length(choices) > 1) {
                title <- paste("\nPlease select (by number) a download ",
                               "site for package ", pkgs[i],":",sep="")
                index <- menu(choices,title=title)
                if (index == 0) {
                    rt_note(paste("Skippping package",pkgs[i]))
                    next()
                }
            }
            else {
                index <- 1
            }
            rep <- choiceReps[index]
        }
        pkg <- pkgs[i]
        pkgVer <- Packages[[i]][[rep]][[index]]

        pkgInfo <- buildPkgInfo(pkg, pkgVer)

        repIndex <- as.numeric(names(Packages[[i]][rep]))
        repEntry <- getRepEntry(pkgList, repIndex)

        ## Check to see if changing this file will break any
        ## other package's dependencies on it
        rd <- resolve.depends(pkgInfo, repEntry, force, lib=lib,
                              searchOptions=searchOptions,
                              getAllDeps=getAllDeps,
                              versForce=versForce,
                              getNewest=getNewest, develOK=FALSE,
                              curRepList=curRepList, method=method)

        rdPkgs <- get("pkgList",rd)
        if (length(packages(rdPkgs)) > 0) {
            gotPkgs <- c(gotPkgs,packages(rdPkgs))
            statusList(pkgStats) <- rdPkgs
        }

        if (get("halt",rd) == TRUE) {
            ok <- FALSE
        }
        if (ok == TRUE) {
            ret <- downAndInPkg(pkg, repEntry, stringRep(pkgVer), lib=lib,
                                type=type, method=method,
                                versForce=versForce)
        }
        else {
            ret <- FALSE
        }

    ## Add this package to the pkgStatusList
    statusList(pkgStats) <- new("pkgStatus",package=pkg,
                                found=TRUE, updated=ret,
                                url=repURL(repEntry),
                                pkgVersion=pkgVer)
    }
    return(pkgStats)
}

downAndInPkg <- function(pkg, repEntry, pkgVer, lib, type, method,
                         versForce) {
    ## Downloads and installs a package

    fileName <- download.packages2(pkg, repEntry,
                                   destDir=tempdir(),
                                   versions=pkgVer,
                                   type=type, method=method)
    ret <- handleDownloadedFile(fileName, pkg, pkgVer, type, lib,
                                versForce, repEntry)
    ret
}


handleDownloadedFile <- function(fileName, pkg, pkgVer, type, lib,
                                 versForce, repEntry) {
    if (is.null(fileName)) {
        warning(paste("No package ",pkg," version ",
                      stringRep(pkgVers)," of type ",type,
                      " exists at ",
                      repURL(repEntry),",skipping",sep=""))
        return(FALSE)
    }

    if (installPkg(fileName, pkg, pkgVer, type, lib, repEntry,
                   versForce)==TRUE) {
        updateLocLib(lib, pkg, pkgVer, repEntry)
    }

    unlink(fileName)
    return(TRUE)
}

installPkg <- function(fileName, pkg, vers, type, lib, repEntry,
                       versForce=TRUE) {
    if ((!is.null(fileName))&&(file.exists(fileName))) {
        OST <- .Platform$OS.type
        print(paste("Installing",pkg))
        if (type == "Win32") {
            ## Check for version compat)
            Rvers <- buildRversnum()
            builtVers <- pkgRVersion(repEntry, pkg, vers, type)
            out <- paste("Running R version ",Rvers," and package ",
                         pkg," was built for R version ",
                         builtVers,"\n",sep="")
            if (builtVers < Rvers) {
                if (versForce==FALSE) {
                    out <- paste(out, ", skipping.\nPlease ",
                              "look at the option versForce if you ",
                              "would like to continue.",sep="")
                    warning(out)
                    return(FALSE)
                }
                else {
                    out <- paste(out,", installing anyway.\n")
                    rt_note(out)
                }
            }
            if (OST != "windows") {
                warning(paste("Attempting to install Win32 version of",
                               pkg,"on a",OST,"system, skipping."))
                return(FALSE)
            }
            status <- zip.unpack(fileName, lib)
        }
        else if (type == "Source") {
            if (OST == "windows") {
                cmd <- paste(file.path(R.home(), "bin", "Rcmd"),
                             "install -l", lib, fileName)
            }
            else if (OST == "unix") {
                cmd <- paste(file.path(R.home(), "bin", "R"),
                             "CMD INSTALL -l", lib, fileName)
            }
            else {
                warning(paste("Attempting to install Unix version of",
                           pkg,"on a",OST,"system, skipping."))
                return(FALSE)
            }
                status <- system(cmd)
        }
        else {
            warning(paste("Do not know how to install packages of type",
                          " ",type,", skipping.",sep=""))
            return(FALSE)
        }

        if (status != 0) {
            warning(paste("Installation of package", pkg,
                          "had non-zero exit status"))
            return(FALSE)
        }
        else {
            print("Installation complete")
        }
    }
    syncLocalLibList(lib, quiet=TRUE)
    return(TRUE)
}

getOptReposList <- function(recurse=TRUE, method="auto") {
    ## Takes the option "repositories" and returns a list of
    ## reposEntry objects from those entries.
    reps <- as.character(getReposOption())
    if (length(reps) > 0) {
        repL <- lapply(reps,getReposEntry, method=method)
        ## remove NULL entries
        repL <- repL[!sapply(repL,is.null)]
        return(getReposList(repL, recurse, method=method))
    }
    else {
        return(NULL)
    }
}

reposToolsLibPaths <- function(baseLib=.libPaths(),
                               tmpLibPath=file.path(tempdir(),"tempLibs")) {
    ## Will add any temporarily maintained liblisting.Rda files to
    ## the search path
    newDirs <- character()

    if ((file.exists(tmpLibPath))&&(file.info(tmpLibPath)$isdir)) {
        newDirs <- dir(tmpLibPath, full.names=TRUE)
        if (length(newDirs) > 0) {

            ## Filter out any that aren't actually directories as these are
            ## obviously garbage
            areDirs <- unlist(lapply(lapply(newDirs,file.info),function(x){return(x$isdir)}))
            newDirs <- newDirs[areDirs]
        }
    }

    return(c(baseLib, newDirs))
}

getMaxVersReposInfo <- function(pkg, reposEntry, type) {
    ## Will return pkgInfo object corresponding to the
    ## max version element Will return NULL if it does
    ## not exist in the repository

    repInfos <- repPkgInfos(reposEntry, pkg, type)
    if (is.null(repInfos))
        return(NULL)
    versions <- lapply(repInfos, pkgVersion)
    maxEle <- getMaxElement(versions)
    repInfos[[maxEle]]
}



buildRepList <- function(repEntry=NULL, recurse=TRUE,
                         searchOptions=TRUE, method="auto") {
    repList <- new("ReposList", repList=list())

    if (!is.null(repEntry)) {
        if (is(repEntry,"ReposEntry")) {
            if (repType(repEntry) != "package")
                stop("update.packages2 requires a package repository")
        }
        else {
            stop("Supplied repEntry argument does not point to a valid repository.")
        }

        repList(repList) <- repEntry
        if (recurse == TRUE) {
            repList(repList) <- getSubRepList(repEntry, method=method)
        }
        if (searchOptions == TRUE) {
            repList(repList) <- getOptReposList(recurse, method=method)
        }
    }
    else
        repList(repList) <- getOptReposList(recurse, method=method)

    repList
}


browseRepos <- function(repEntry) {
    brURL <- paste(repURL(repEntry), "/html/index.html", sep="")

    curErr <- getOption("show.error.messages")
    options(show.error.messages = FALSE)
    on.exit(options(show.error.messages = curErr))
    ## Check to see if it is there
    testConn <- url(brURL)
    z <- try(readLines(testConn, n=1))
    close(testConn)
    if (is(z, "try-error") == TRUE) {
        warning("No browsable HTML for this repository")
        return(FALSE)
    }
    else
        browseURL(brURL)
}

checkRepos <- function(repEntry) {
    rdd <- repdataframe(repEntry)
    osS <- rdd$OSspecific

    repU <- repURL(repEntry)

    out <- character()

    oOpt <- getOption("show.error.messages")
    on.exit(options(show.error.messages=oOpt), add=TRUE)

    for (i in seq(along=osS)) {
        for (j in seq(along=osS[[i]])) {
            link <- url(paste(repU, osS[[i]][[j]]$File, sep="/"))
            options(show.error.messages=FALSE)
            lines <- try(readLines(link, n=1))
            options(show.error.messages=oOpt)
            close(link)
            if ((is(lines, "try-error"))||(lines == "<html>")) {
                out <-c(out, basename(osS[[i]][[j]]$File))
            }
        }
    }

    if (length(out) > 0) {
        cat("The following packages are not accessible at ",
            repName(z), ":\n\t", sep="")
        cat(paste(out, collapse="\n\t"))
        cat("\n")
    }
    else
        NULL
}
