/* Copyright (C) 2001 Ian Campbell <ijc@hellion.org.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdarg.h>

#include <rep.h>
#include <xmms/xmmsctrl.h>

#define SESSION 0

DEFUN("xmms-play", Fxmms_play, Sxmms_play, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-play::
xmms-play

Start XMMS playing.
::end:: */
{
	xmms_remote_play(SESSION);
	return Qt;
}

DEFUN("xmms-stop", Fxmms_stop, Sxmms_stop, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-stop::
xmms-stop

Stops XMMS playing.
::end:: */
{
	xmms_remote_stop(SESSION);	
	return Qt;
}

DEFUN("xmms-pause", Fxmms_pause, Sxmms_pause, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-pause::
xmms-pause

Pauses XMMS.
::end:: */
{
	xmms_remote_pause(SESSION);	
	return Qt;
}

DEFUN("xmms-get-version", Fxmms_get_version, Sxmms_get_version, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-get-version::
xmms-get-version

Returns the XMMS remote protocol version number.
::end:: */
{
	gint version;
	version = xmms_remote_get_version(SESSION);
	return rep_MAKE_INT(version);
}

DEFUN("xmms-playing-p", Fxmms_playing_p, Sxmms_playing_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-playing-p::
xmms-playing-p

Returns t if XMMS is currently playing.
::end:: */
{
	if ( xmms_remote_is_playing(SESSION) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-paused-p", Fxmms_paused_p, Sxmms_paused_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-paused-p::
xmms-paused-p

Returns t if XMMS is currently paused.
::end:: */
{
	if ( xmms_remote_is_paused(SESSION) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-playlist-pos-get", Fxmms_playlist_pos_get, Sxmms_playlist_pos_get, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-playlist-pos-get::
xmms-playlist-pos-get

Returns the current position in the playlist.
::end:: */
{
	return rep_MAKE_INT(xmms_remote_get_playlist_pos(SESSION));
}

DEFUN("xmms-playlist-pos-set", Fxmms_playlist_pos_set, Sxmms_playlist_pos_set, (repv pos), rep_Subr1) /*
::doc:xmms.remote#xmms-playlist-pos-set::
xmms-playlist-pos-set ARG

Sets the current playlist position to ARG.
::end:: */
{
	rep_DECLARE1 ( pos, rep_INTP );

	xmms_remote_set_playlist_pos(SESSION, rep_INT(pos));
	return Qt;
}

DEFUN("xmms-playlist-length", Fxmms_playlist_length, Sxmms_playlist_length, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-playlist-length::
xmms-playlist-length

Returns the length of the current playlist.
::end:: */
{
	return rep_MAKE_INT(xmms_remote_get_playlist_length(SESSION));
}

DEFUN("xmms-playlist-add", Fxmms_playlist_add, Sxmms_playlist_add, (repv list), rep_Subr1) /*
::doc:xmms.remote#xmms-playlist-add::
xmms-playlist-add LIST

Adds list to the playlist.
::doc:: */
{
	repv tmp;
	GList *play_list = NULL;

	rep_DECLARE1(list, rep_LISTP);

	for (tmp = list; tmp != Qnil ; tmp = rep_CDR(tmp) ) {
		gchar * str;
		if ( !rep_STRINGP(rep_CAR(tmp)) ) {
			g_list_free(play_list);
			rep_signal_arg_error(tmp,1);
			return rep_NULL;
		}
		str = rep_STRING(rep_CAR(tmp))->data;
		play_list = g_list_prepend(play_list, str);
	}
	play_list = g_list_reverse(play_list);
	xmms_remote_playlist_add(SESSION, play_list);
	g_list_free(play_list);
	return Qt;
}

DEFUN("xmms-playlist-delete", Fxmms_playlist_delete, Sxmms_playlist_delete, (repv pos), rep_Subr1) /*
::doc:xmms.remote#xmms-playlist-delete::
xmms-playlist-delete POS

Deletes track at POS.
::doc:: */
{
	/* rep_DELARE1(pos, rep_INTP); */

	xmms_remote_playlist_delete(SESSION, rep_INT(pos));
	return Qt;
}

DEFUN("xmms-playlist-clear", Fxmms_playlist_clear, Sxmms_playlist_clear, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-playlist-clear::
xmms-playlist-clear

Clears the playlist.
::end:: */
{
	xmms_remote_playlist_clear(SESSION);
	return Qt;
}

DEFUN("xmms-get-output-time", Fxmms_get_output_time, Sxmms_get_output_time, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-get-output-time::
xmms-get-output-time

Returns the current track time in seconds.
::end:: */
{
	return rep_MAKE_INT(xmms_remote_get_output_time(SESSION));
}

DEFUN("xmms-jump-to-time", Fxmms_jump_to_time, Sxmms_jump_to_time, (repv t), rep_Subr1) /*
::doc:xmms.remote#xmms-jump-to-time::
xmms-jump-to-time ARG

Jumps to the time ARG in the current track.
::end:: */
{
	rep_DECLARE(1, t, rep_INTP(t));
	
	xmms_remote_jump_to_time(SESSION, rep_INT(t));
	return Qt;
}

DEFUN("xmms-volume-get", Fxmms_volume_get, Sxmms_volume_get, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-volume-get::
xmms-volume-get

Returns a cons cell containing the left and right volume levels as a percentage.
::end:: */
{
	gint vl, vr;
	xmms_remote_get_volume(SESSION, &vl, &vr);
	return Fcons(rep_MAKE_INT(vl),rep_MAKE_INT(vr));
}

DEFUN("xmms-main-volume-get", Fxmms_main_volume_get, Sxmms_main_volume_get, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-main-volume-get::
xmms-main-volume-get

Returns the overall volume.
::end:: */
{
	return rep_MAKE_INT(xmms_remote_get_main_volume(SESSION));
}

DEFUN("xmms-balance-get", Fxmms_balance_get, Sxmms_balance_get, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-balance-get::
xmms-balance-get

Returns the current balance level.
::end:: */
{
	return rep_MAKE_INT(xmms_remote_get_balance(SESSION));
}

DEFUN("xmms-volume-set", Fxmms_volume_set, Sxmms_volume_set, (repv lr), rep_Subr1) /*
::doc:xmms.remote#xmms-volume-set::
xmms-volume-set ( LEFT . RIGHT )

Set the left channel volume to LEFT and the right channel volume to RIGHT.
::end:: */
{
	rep_DECLARE(1,lr, rep_CONSP(lr));
	rep_DECLARE(1,rep_CAR(lr), rep_INTP(rep_CAR(lr)));
	rep_DECLARE(1,rep_CDR(lr), rep_INTP(rep_CDR(lr)));
	
	xmms_remote_set_volume(SESSION,
			rep_INT(rep_CAR(lr)), 
			rep_INT(rep_CDR(lr)));
	return Qt;
}

DEFUN("xmms-main-volume-set", Fxmms_main_volume_set, Sxmms_main_volume_set, (repv vol), rep_Subr1) /*
::doc:xmms.remote#xmms-main-volume-set:: 
xmms-main-volume-set VOL

Set the volume to VOL.
::end:: */
{
	rep_DECLARE(1,vol,rep_INTP(vol));

	xmms_remote_set_main_volume(SESSION, rep_INT(vol));
	return Qt;
}

DEFUN("xmms-balance-set", Fxmms_balance_set, Sxmms_balance_set, (repv b), rep_Subr1) /*
::doc:xmms.remote#xmms-balance-set::
xmms-balance-set BAL

Set the balance to BAL.
::end:: */
{
	rep_DECLARE(1, b,rep_INTP(b));
	
	xmms_remote_set_balance(SESSION, rep_INT(b));
	return Qt;
}

DEFUN("xmms-playlist-file", Fxmms_playlist_file, Sxmms_playlist_file, (repv pos), rep_Subr1) /*
::doc:xmms.remote#xmms-playlist-file::
xmms-playlist-file POS

Returns the filename of the track at POS in the playlist.
::end:: */
{
	gchar *fname;
	rep_DECLARE1 (pos, rep_INTP);
	fname = xmms_remote_get_playlist_file(SESSION, rep_INT(pos));
	return rep_string_dup(fname);
}

DEFUN("xmms-playlist-title", Fxmms_playlist_title, Sxmms_playlist_title, (repv pos), rep_Subr1) /*
::doc:xmms.remote#xmms-playlist-title::
xmms-playlist-title POS

Returns the title of the track at POS in the playlist.
::end:: */
{
	gchar *fname;

	rep_DECLARE1 (pos, rep_INTP);
	fname = xmms_remote_get_playlist_title(SESSION, rep_INT(pos));
	return rep_string_dup(fname);
}

DEFUN("xmms-get-playlist-time", Fxmms_get_playlist_time, Sxmms_get_playlist_time, (repv pos), rep_Subr1) /*
::doc:xmms.remote#xmms-get-playlist-time::
xmms-get-playlist-time POS

Get the length of the track at position POS in the playlist.
::end:: */
{
	rep_DECLARE1(pos, rep_INTP);
	return rep_MAKE_INT(xmms_remote_get_playlist_time(SESSION, rep_INT(pos)));
}

DEFUN("xmms-get-info", Fxmms_get_info, Sxmms_get_info, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-get-info::
xmms-get-playlist-time

Returns a list containing the current bitrate, frequence and number of channels.
::end:: */
{
	gint rate, freq, nch;
	xmms_remote_get_info(SESSION, &rate, &freq, &nch);
	return rep_LIST_3(rep_MAKE_INT(rate), 
			rep_MAKE_INT(freq),
			rep_MAKE_INT(nch));
}

DEFUN("xmms-main-visible", Fxmms_main_visible, Sxmms_main_visible, (repv visible), rep_Subr1) /*
::doc:xmms.remote#xmms-main-visible::
xmms-main-visible SHOW

If SHOW is t then show the main window, else hide it.
::end:: */
{
	if ( rep_NILP(visible) ) {
		xmms_remote_main_win_toggle(SESSION, FALSE);
	} else {
		xmms_remote_main_win_toggle(SESSION, TRUE);
	}
	return Qt;
}

DEFUN("xmms-playlist-visible", Fxmms_playlist_visible, Sxmms_playlist_visible, (repv visible), rep_Subr1) /*
::doc:xmms.remote#xmms-playlist-visible::
xmms-playlist-visible SHOW

If SHOW is t then show the playlist window, else hide it.
::end:: */
{
	if ( rep_NILP(visible) ) {
		xmms_remote_pl_win_toggle(SESSION, FALSE);
	} else {
		xmms_remote_pl_win_toggle(SESSION, TRUE);
	}
	return Qt;
}

DEFUN("xmms-eq-visible", Fxmms_eq_visible, Sxmms_eq_visible, (repv visible), rep_Subr1) /*
::doc:xmms.remote#xmms-eq-visible::
xmms-eq-visible SHOW

If SHOW is t then show the equalizer window, else hide it.
::end:: */
{
	if ( rep_NILP(visible) ) {
		xmms_remote_eq_win_toggle(SESSION, FALSE);
	} else {
		xmms_remote_eq_win_toggle(SESSION, TRUE);
	}
	return Qt;
}

DEFUN("xmms-main-p", Fxmms_main_p, Sxmms_main_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-main-p::
xmms-main-p

Return t if the main window is visible.
::end:: */
{
	if ( xmms_remote_is_main_win(SESSION ) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-playlist-p", Fxmms_playlist_p, Sxmms_playlist_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-playlist-p::
xmms-playlist-p

Return t if the playlist window is visible.
::end:: */
{
	if ( xmms_remote_is_pl_win(SESSION ) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-eq-p", Fxmms_eq_p, Sxmms_eq_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-eq-p::
xmms-eq-p

Return t if the equalizer window is visible.
::end:: */
{
	if ( xmms_remote_is_eq_win(SESSION ) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-prefs-show", Fxmms_prefs_show, Sxmms_prefs_show, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-prefs-show::
xmms-prefs-show

Popup the XMMS preferences window.
::end:: */
{
	xmms_remote_show_prefs_box(SESSION);
	return Qt;
}

DEFUN("xmms-always-on-top-toggle", Fxmms_aot_toggle, Sxmms_aot_toggle, (repv toggle), rep_Subr1) /*
::doc:xmms.remote#xmms-always-on-top-toggle:
xmms-always-on-top-toggle TOGGLE

If TOGGLE is t then make XMMS be always-on-top, else don't.
::end:: */
{
	if ( rep_NILP ( toggle ) )
		xmms_remote_toggle_aot(SESSION, FALSE);
	else 
		xmms_remote_toggle_aot(SESSION, TRUE);
	return Qt;
}

DEFUN("xmms-eject", Fxmms_eject, Sxmms_eject, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-eject::
xmms-eject

Eject XMMS (popup add new tracks window). 
::end:: */
{
	xmms_remote_eject(SESSION);
	return Qt;
}

DEFUN("xmms-prev", Fxmms_prev, Sxmms_prev, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-prev::
xmms-prev

Jump to the previous track.
::end:: */
{
	xmms_remote_playlist_prev(SESSION);
	return Qt;
}

DEFUN("xmms-next", Fxmms_next, Sxmms_next, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-next::
xmms-next

Jump to the next track.
::end:: */
{
	xmms_remote_playlist_next(SESSION);
	return Qt;
}

DEFUN("xmms-playlist-add-url", Fxmms_playlist_add_url, Sxmms_playlist_add_url, (repv string), rep_Subr1) /*
::doc:xmms.remote#xmms-playlist-add-url::
xmms-playlist-add-url URL

Add URL to the playlist.
::end:: */
{
	rep_DECLARE1(string, rep_STRINGP);
	xmms_remote_playlist_add_url_string(SESSION, rep_STR(string));
	return Qt;
}

DEFUN("xmms-running-p", Fxmms_running_p, Sxmms_running_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-running-p::
xmms-running-p

Returns t if XMMS is running.
::end:: */
{
	if ( xmms_remote_is_running(SESSION) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-repeat-toggle", Fxmms_repeat_toggle, Sxmms_repeat_toggle, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-repeat-toggle::
xmms-repeat-toggle

Toggle repeat.
::end:: */
{
	xmms_remote_toggle_repeat(SESSION);
	return Qt;
}

DEFUN("xmms-shuffle-toggle", Fxmms_shuffle_toggle, Sxmms_shuffle_toggle, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-shuffle-toggle::
xmms-shuffle-toggle

Toggle shuffle.
::end:: */
{
	xmms_remote_toggle_shuffle(SESSION);
	return Qt;
}

DEFUN("xmms-repeat-p", Fxmms_repeat_p, Sxmms_repeat_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-repeat-p::
xmms-repeat-p

Returns t if XMMS is currently set to repeat.
::end:: */
{
	if ( xmms_remote_is_repeat(SESSION) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-shuffle-p", Fxmms_shuffle_p, Sxmms_shuffle_p, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-shuffle-p::
xmms-shuffle-p

Returns t if XMMS is currently set to shuffle.
::end:: */
{
	if ( xmms_remote_is_shuffle(SESSION) )
		return Qt;
	else
		return Qnil;
}

DEFUN("xmms-quit", Fxmms_quit, Sxmms_quit, (void), rep_Subr0) /*
::doc:xmms.remote#xmms-quit::
xmms-quit

Quits XMMS.
::end:: */
{
	xmms_remote_quit(SESSION);
	return Qt;
}

repv
rep_dl_init (void)
{
	repv tem = rep_push_structure ("xmms.remote");

  	Fset (Qdocumentation_file, rep_string_dup(REP_EXECDIR "/xmms/DOC"));

  	Fset(Qdocumentation_files, Fcons(Fsymbol_value(Qdocumentation_file, Qt),
  				         Fsymbol_value(Qdocumentation_files, Qt)));
	
	rep_ADD_SUBR(Sxmms_aot_toggle);
	rep_ADD_SUBR(Sxmms_balance_get);
	rep_ADD_SUBR(Sxmms_balance_set);
	rep_ADD_SUBR(Sxmms_eject);
	rep_ADD_SUBR(Sxmms_eq_p);
	rep_ADD_SUBR(Sxmms_eq_visible);
	rep_ADD_SUBR(Sxmms_get_info);
	rep_ADD_SUBR(Sxmms_get_output_time);
	rep_ADD_SUBR(Sxmms_get_playlist_time);
	rep_ADD_SUBR(Sxmms_get_version);
	rep_ADD_SUBR(Sxmms_jump_to_time);
	rep_ADD_SUBR(Sxmms_main_p);
	rep_ADD_SUBR(Sxmms_main_visible);
	rep_ADD_SUBR(Sxmms_main_volume_get);
	rep_ADD_SUBR(Sxmms_main_volume_set);
	rep_ADD_SUBR(Sxmms_next);
	rep_ADD_SUBR(Sxmms_pause);
	rep_ADD_SUBR(Sxmms_paused_p);
	rep_ADD_SUBR(Sxmms_play);
	rep_ADD_SUBR(Sxmms_playing_p);
	rep_ADD_SUBR(Sxmms_playlist_add);
	rep_ADD_SUBR(Sxmms_playlist_add_url);
	rep_ADD_SUBR(Sxmms_playlist_clear);
	rep_ADD_SUBR(Sxmms_playlist_delete);
	rep_ADD_SUBR(Sxmms_playlist_file);
	rep_ADD_SUBR(Sxmms_playlist_length);
	rep_ADD_SUBR(Sxmms_playlist_p);
	rep_ADD_SUBR(Sxmms_playlist_pos_get);
	rep_ADD_SUBR(Sxmms_playlist_pos_set);
	rep_ADD_SUBR(Sxmms_playlist_title);
	rep_ADD_SUBR(Sxmms_playlist_visible);
	rep_ADD_SUBR(Sxmms_prefs_show);
	rep_ADD_SUBR(Sxmms_prev);
	rep_ADD_SUBR(Sxmms_quit);
	rep_ADD_SUBR(Sxmms_repeat_p);
	rep_ADD_SUBR(Sxmms_repeat_toggle);
	rep_ADD_SUBR(Sxmms_running_p);
	rep_ADD_SUBR(Sxmms_shuffle_p);
	rep_ADD_SUBR(Sxmms_shuffle_toggle);
	rep_ADD_SUBR(Sxmms_stop);
	rep_ADD_SUBR(Sxmms_volume_get);
	rep_ADD_SUBR(Sxmms_volume_set);

	return rep_pop_structure (tem);
}

