/*
    Copyright (C) 2001-2002 by theKompany.com <www.thekompany.com>
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    See COPYING.GPL file.

    In addition, as a special exception, theKompany.com gives permission
    to link the code of this program with the tkwidgets library (or with
    modified versions of tkwidgets that use the same license as tkwidgets),
    and distribute linked combinations including the two.  You must obey
    the GNU General Public License in all respects for all of the code used
    other than tkwidgets.  If you modify this file, you may extend this
    exception to your version of the file, but you are not obligated to do so.
    If you do not wish to do so, delete this exception statement from your
    version.

    This license grants you the ability to use tkwidgets with Rekall only
    and may not be used outside of Rekall.
    See also http://www.rekall.a-i-s.co.uk/products/license.txt for details.
*/


#include	<qregexp.h>

#ifndef _WIN32
#include "tktexteditor.moc"
#else
#include "tktexteditor.h"
#endif

#include "tktextmanager.h"
#include "tktextview.h"
#include "tktextdoc.h"
#include "tktextline.h"
#include "tkhighlight.h"

#include <qapplication.h>
#include <qclipboard.h>
#include <qtextcodec.h>
#include <qfileinfo.h>
#include <qaction.h>
#include <qsimplerichtext.h>
#include <qpaintdevicemetrics.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qimage.h>
#include <qdir.h>
#include	<qregexp.h>

#ifdef QWS
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#endif

TKTextEditor::TKTextEditor(TKTextDocument *document, QWidget *parent, const char *name)
: QFrame(parent, name),
  document(document), manager(document->textManager()), view(0), textCodec(QTextCodec::codecForLocale())
{
#ifdef KOBOL
  fLinuNumberMargin = true;
#endif

  setFrameStyle(Panel | Sunken);

  path = QString::null;
  view = new TKTextView(this);
  view->installEventFilter(this);

  setFocusProxy(view);
}

TKTextEditor::~TKTextEditor()
{
  delete view;
  if (isLastView())
    delete document;
}

bool TKTextEditor::eventFilter(QObject *obj, QEvent *ev)
{
  if (obj == (QObject *)view && ev->type() == QEvent::FocusIn)
    document->m->editorActivated(this);

  return QFrame::eventFilter(obj, ev);
}

void TKTextEditor::setFilePath(const QString &name)
{
  path = QFileInfo(name).absFilePath();
  document->setHighlight(QFileInfo(path).fileName());
  emit filePathChange();
}

QString TKTextEditor::filePath()
{
  return path;
}

QString TKTextEditor::title()
{
  if (filePath().isEmpty())
    return tr("Untitled");

  return QFileInfo(filePath()).fileName();
}

QString TKTextEditor::html(int from, int to)
{
  if (to == -1)
    to = document->lastLine();
  else
    to = QMIN(to, document->lastLine());

  QRegExp rx("^(?:<[^>]*>)?(\\s*)"); // for replacing chars with &nbsp;
  QString result = "<pre>\n";
  for (int line = from; line <= to; line++) {
      QString l = document->html(line);

      // replace spaces at the start of line with &nbsp; for correct printing with qt html engine
      if ( rx.search(l) != -1 ) {
          int pos = rx.pos(1);
          int len = rx.cap(1).length();
          QString spaces;
          for ( int i=0; i<len; i++ )
            spaces += "&nbsp;";
          l = l.replace(pos,len,spaces);

      }

      result +=  l + "\n";//"<br>";
  }
  result += "</pre>";
  //const char *res = result.latin1();
  //qDebug( "%s", res  );
  return result;
}

bool TKTextEditor::isLastView()
{
  return document->viewsCount() <= 1;
}

bool TKTextEditor::open(const QString &path)
{
  setFilePath(path);

  QFile file(path);
  if (!file.open(IO_ReadOnly))
    return false;

  QTextStream ts(&file);
  ts.setCodec(textCodec);
  bool f = document->load(ts);
  file.close();
  return f;
}

bool TKTextEditor::saveAs(const QString &path)
{
  setFilePath(path);
  document->modified = true;
  return save();
}

bool TKTextEditor::save()
{
  if (!isModified())
    return true;

  if (filePath().isEmpty())
    return false;

#ifdef QWS
  struct stat status;
  bool haveAttr = stat((const char *)filePath(), &status) == 0;
#endif

  QFile file(filePath());
  if (!file.open(IO_WriteOnly))
    return false;

  QTextStream ts(&file);
  ts.setCodec(textCodec);
#ifndef KOBOL
  bool f = document->save(ts);
#else
	bool f;
	if (isLinuNumberMarginVisible())
		f = document->saveWith6Numbers(ts);
	else
		f = document->save(ts);
#endif
  file.close();

#ifdef QWS
  if (f && haveAttr)
    chmod((const char *)filePath(), status.st_mode);
#endif

  return f;
}

void TKTextEditor::fileNew()
{
  document->clear();
  setFilePath(QString::null);
}

bool TKTextEditor::findRegexp(const QString &pattern, bool caseSensitive, bool wildcard, bool fromCursor)
{
  int startLine = 0;
  int startPos = 0;
  if (fromCursor) {
    startLine = line();
    startPos = column();
  }

  bool rc = false ;
  view->clearFound();
  QRegExp searchRegExp = QRegExp(pattern, caseSensitive, wildcard);
  for (int line = startLine; line <= document->lastLine(); line++) {
    int pos = searchRegExp.search(*document->lineOf(line), startPos);
    startPos = 0;
    if (pos != -1) {
      view->setCursorPosition(line, pos + searchRegExp.matchedLength());
      view->setFound(line, pos, searchRegExp.matchedLength());
      rc = true ;
      break;
    }
  }
  view->updateView(false);
  return rc ;
}
//******** change type of returned value from 'void' to 'int' - return count of replacements
/*int*/ void TKTextEditor::replaceAll(const QString &text, const QString &replaceText, bool caseSensitive, bool wholeWord, bool inSelection)
{
  //* int count = 0;
  find(text, caseSensitive, wholeWord, false, inSelection, false);
  if (!hasFound())
    return /*0*/;

  view->lockSeletion(true);
  document->recordStart(line(), column());
  document->lockRecordStart(true);
  document->lockRecordEnd(true);
  while (hasFound()) {
    //* count++;
    view->lockSeletion(true);
    document->selectionCommand(view, TKTextDocument::SelectionRemove);
    document->insert(view, replaceText);
    view->lockSeletion(false);
    find(text, caseSensitive, wholeWord, true, inSelection, false);
  }
  document->lockRecordStart(false);
  document->lockRecordEnd(false);
  view->lockSeletion(true);
  document->recordEnd(view, line(), column());
  view->lockSeletion(false);
  //* return count;
}

void TKTextEditor::replaceFound(const QString &text)
{
  view->lockSeletion(true);
  document->lockRecordEnd(true);
  document->selectionCommand(view, TKTextDocument::SelectionRemove);
  document->lockRecordStart(true);
  document->lockRecordEnd(false);
  document->insert(view, text);
  document->lockRecordStart(false);
  view->lockSeletion(false);
}

bool TKTextEditor::find(const QString &text, bool caseSensitive, bool wholeWord, bool fromCursor, bool inSelection, bool cycle)
{
  view->lockSeletion(true);
  view->clearFound();
  QRegExp nonword("[^0-9a-zA-Z_]");
  int l = text.length();
  QRect selection = view->selection();

  int startLine = 0;
  int startPos = 0;
  if (fromCursor) {
    startLine = line();
    startPos = column();
  }
  if (inSelection) {
    startLine = QMAX(startLine, selection.top());
    startPos = QMAX(startPos, selection.left());
  }

  bool vs = manager->fVerticalSelection;
  bool found = false;
  for (int line = startLine; line <= (inSelection ? selection.bottom() : document->lastLine()); line++) {
    TKTextLine *s = document->lineOf(line);
    int pos = s->find(text, startPos, caseSensitive);
    startPos = inSelection && vs ? selection.left() : 0;

    if (pos == -1)
      continue;

    if (inSelection && vs && pos + l > selection.right())
      continue;

    if (inSelection && line == selection.bottom() && pos + l > selection.right())
      continue;

    if (!wholeWord) {
      view->setCursorPosition(line, pos + l, false);
      view->setFound(line, pos, l);
      found = true;
      break;
    }

    if (pos == 0 || nonword.search(QConstString(&s->ref(pos - 1), 1).string()) == 0) {
      if (pos + l == (int)s->length() || nonword.search(QConstString(&s->ref(pos + l), 1).string()) == 0) {
        view->setCursorPosition(line, pos + l, false);
        view->setFound(line, pos, l);
        found = true;
        break;
      }
    }
    startPos = pos + l;
    --line;
  }

  if (found)
    view->updateView(false);
  else if (cycle && fromCursor)
    find(text, caseSensitive, wholeWord, false, inSelection);

  view->lockSeletion(false);
  return found;
}

void TKTextEditor::print()
{
#if 0
#ifndef QT_NO_PRINTER
  if (!manager->setupPrinter())
    return;

  qApp->processEvents();
  QApplication::setOverrideCursor(Qt::waitCursor);

  bool preview = false;
  QSize previewSize(100, 100);

  QPaintDeviceMetrics metrics(document->m->printer);
  int dpix = metrics.logicalDpiX();
  int dpiy = metrics.logicalDpiY();

  QRect margins = document->m->margins;
  QRect body(margins.x() * dpix / 72,
             margins.y() * dpiy / 72,
             metrics.width() - (margins.x() + margins.width()) * dpix / 72,
             metrics.height() - (margins.y() + margins.height()) * dpiy / 72);
  QSimpleRichText richText(html(), document->m->font(), QString::null, 0, QMimeSourceFactory::defaultFactory(), body.height());
  QRect viewRect(body);
  int page = 1;

  QPainter p;
  QPixmap pix;
  if (preview) {
    pix.resize(metrics.width(), metrics.height());
    p.begin(&pix);
  } else
    p.begin(document->m->printer);

  richText.setWidth(&p, body.width());
  do {
    if (preview)
      pix.fill(Qt::white);
    p.setClipRect(body);
    richText.draw(&p, body.left(), body.top(), viewRect, colorGroup());
    viewRect.moveBy(0, body.height());
    p.translate(0 , -body.height());
    p.setFont(document->m->font());
    p.setClipping(false);
    p.drawText(viewRect.right() - p.fontMetrics().width(QString::number(page)),
               viewRect.bottom() + p.fontMetrics().ascent() + 5,
               QString::number(page));

    if (preview) {
      QImage image = pix.convertToImage();
      image = image.smoothScale(previewSize, QImage::ScaleMax);
      QPixmap gen;
      gen.convertFromImage(image);

      gen.save(QString("/page%1.bmp").arg(page), "BMP");
    } else
        if ( viewRect.top() < richText.height() ) {
            document->m->printer->newPage();
        }
      

    page++;
  } while (viewRect.top() < richText.height());

  QApplication::restoreOverrideCursor();
#endif
#endif
}

void TKTextEditor::notifyChange(TKTextEditor::ChangeId id)
{
  emit changed(id);
}

int TKTextEditor::line()
{
  return document->lineOf(view->cursor->line())->number();
}

int TKTextEditor::column()
{
  return view->cursor->column();
}

bool TKTextEditor::isModified()
{
  return document->isModified();
}

void TKTextEditor::setModified(bool f)
{
	document->modified = f;
	notifyChange(TKTextEditor::Modification);
}

void TKTextEditor::setWrap(bool wrap)
{
	document->setWrap( wrap );
}

bool TKTextEditor::getWrap()
{
  return document->getWrap();
}

QString TKTextEditor::highlight()
{
  return document->highlight()->name();
}

void TKTextEditor::setHighlight(const QString &name)
{
  document->setCustomHighlight(name);
}

void TKTextEditor::printPreview()
{
/*
  qApp->processEvents();

  if (!manager->setupPrinter())
    return;

  QApplication::setOverrideCursor(Qt::waitCursor);

  QDir().mkdir("/_pages");
  QSize previewSize(400, 400);

  QPaintDeviceMetrics metrics(document->m->printer);
  QPixmap pix(metrics.width(), metrics.height());

//  int dpix = metrics.logicalDpiX();
//  int dpiy = metrics.logicalDpiY();

  QRect margins = document->m->margins;
  QRect body(margins.x(),
             margins.y(),
             metrics.width() - (margins.x() + margins.width()),
             metrics.height() - (margins.y() + margins.height()));
  QSimpleRichText richText(html(), document->m->font(), QString::null, 0, QMimeSourceFactory::defaultFactory(), body.height());
  QRect viewRect(body);
  int page = 1;

  QPainter p(&pix);
  richText.setWidth(&p, body.width());
  do {
    pix.fill(Qt::white);
    p.setClipRect(body);
    richText.draw(&p, body.left(), body.top(), viewRect, colorGroup());
    viewRect.moveBy(0, body.height());
    p.translate(0 , -body.height());
    p.setFont(document->m->font());
    p.setClipping(false);
    p.setPen(black);
    p.drawText(viewRect.right() - p.fontMetrics().width(QString::number(page)),
               viewRect.bottom() + p.fontMetrics().ascent() + 5,
               QString::number(page));

    QImage image = pix.convertToImage();
    image = image.smoothScale(previewSize, QImage::ScaleMax);
    QPixmap gen;
    gen.convertFromImage(image);

    pix.save(QString("/_pages/page%1.bmp").arg(page), "BMP");
    page++;
  } while (viewRect.top() < richText.height());

  QApplication::restoreOverrideCursor();
*/
}
/*
void TKTextEditor::addMarginItem(TKTextMarginManager *m, int line, int itemId)
{
  int id = manager->textMarginClassId(m);
  if (id == -1 || line > document->lastLine())
    return;

  document->lineOf(line)->setMargin(id, itemId);
  document->tagLine(line);
}

void TKTextEditor::removeMarginItem(TKTextMarginManager *m, int line, int itemId)
{
  int id = manager->textMarginClassId(m);
  if (id == -1 || line > document->lastLine())
    return;

  document->lineOf(line)->delMargin(id, itemId);
  document->tagLine(line);
}
*/
void TKTextEditor::clear()
{
  document->clear();
}

QString TKTextEditor::getSelectedText()
{
  return document->selectionCommand(view, TKTextDocument::SelectionText);
}

QString TKTextEditor::text()
{
  return document->text();
}

void TKTextEditor::setText(const QString &text)
{
  document->clear();
  append(text);
}

void TKTextEditor::append(const QString &text)
{
  document->append(text);
  document->updateViews();
}

void TKTextEditor::setReadOnly(bool f)
{
  document->setReadOnly(f);
}

bool TKTextEditor::isReadOnly()
{
  return document->isReadOnly();
}

void TKTextEditor::insertLine(const QString &text, int line)
{
  document->insert(0, text, line, 0);
  document->updateViews();
}

QFont TKTextEditor::textFont() const
{
  return manager->font(0);
}

void TKTextEditor::setTextFont(const QFont &font)
{
  manager->setFont(font);
}

void TKTextEditor::makeFold(int fromLine, int toLine)
{
  document->makeFold(view, fromLine, toLine);
}

void TKTextEditor::cut()
{
  if (view->hasSelection())
    document->selectionCommand(view, TKTextDocument::SelectionCut);
}

void TKTextEditor::copy()
{
  if (view->hasSelection())
    document->selectionCommand(view, TKTextDocument::SelectionCopy);
}
//*****
void TKTextEditor::paste()
{
  document->selectionCommand(view, TKTextDocument::SelectionRemove);
  QString cb;
#ifndef QT_NO_CLIPBOARD
  cb = QApplication::clipboard()->text();
  #ifdef Q_WS_WIN
    cb = cb.replace(QRegExp("\r\n"), "\n");
    cb = cb.replace(QRegExp("\r"), "\n");
  #endif
  document->insert(view, cb);
#else
  cb = manager->clipboard;
  #ifdef Q_WS_WIN
    cb = cb.replace(QRegExp("\r\n"), "\n");
    cb = cb.replace(QRegExp("\r"), "\n");
  #endif
  document->insert(view, cb);
#endif
}

void TKTextEditor::undo()
{
  document->undo(view);
}

void TKTextEditor::redo()
{
  document->redo(view);
}

void TKTextEditor::del()
{
  if (view->hasSelection())
    document->selectionCommand(view, TKTextDocument::SelectionRemove);
  else
    document->del(view);
}

bool TKTextEditor::canUndo()
{
  return document->canUndo();
}

bool TKTextEditor::canRedo()
{
  return document->canRedo();
}

void TKTextEditor::completeWord()
{
  view->completeWord();
}

void TKTextEditor::indent()
{
  document->indent(view);
}

void TKTextEditor::unindent()
{
  document->unindent(view);
}

bool TKTextEditor::hasSelections()
{
  return hasSelection() | hasFound();
}

bool TKTextEditor::hasSelection()
{
  return view->hasSelection(true, false);
}

bool TKTextEditor::hasFound()
{
  return view->hasSelection(false, true);
}

void TKTextEditor::moveCursor(int line, int column)
{
  view->centerCursor(line, column);
}

void TKTextEditor::clearFound()
{
  view->clearFound();
}

void TKTextEditor::setTextCodec(QTextCodec * codec)
{
  textCodec = codec;
}

#ifdef KOBOL
bool TKTextEditor::isLinuNumberMarginVisible()
{
  return fLinuNumberMargin;
}
#endif

void TKTextEditor::setLinuNumberMarginVisible(bool f)
{
#ifdef KOBOL
  fLinuNumberMargin = f;
  textView()->updateMarginsState();
#endif
}
bool TKTextEditor::viewSetFocusOK ()
{
	return true ;
}
