/***************************************************************************
    file	         : tkc_tkemapper.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qfile.h>
#include	<qdom.h>

#include	<qwidget.h>
#include	<qtooltip.h>
#include	<qfont.h>
#include	<qsimplerichtext.h>
#include	<qcursor.h>
#include	<qapplication.h>
#include	<qpainter.h>
#include	<qtimer.h>
#include	<qguardedptr.h>
#include	<qlayout.h>
#include	<qregexp.h>

#include	"kb_locator.h"
#include	"kb_dictionary.h"
#include	"kb_dialog.h"

#ifndef 	_WIN32
#include	"tkc_tkemapper.moc"
#else
#include 	"tkc_tkemapper.h"
#endif

#include	"tktexteditor.h"
#include	"tktextview.h"
#include	"tktextdoc.h"
#include	"tktextline.h"


/*  getDictionary: Get the python helper dictionary			*/
/*  (returns)	 : KBDictionary * : The dictionary			*/


LIBKBASE_API KBDictionary *getPythonDict()

{
	static	KBDictionary	*dict	;
	if (dict == 0) dict = new KBDictionary ("python_sip") ;
	return	dict	;
}


/*  ------------------------------------------------------------------  */

/*  TKCTKEHelper								*/
/*  TKCTKEHelper: Constructor for syntax helper widget			*/
/*  editor	: TKTextEditor *  : Editor widget			*/
/*  helpText	: const QString & : Help text				*/
/*  (returns)	: QWidget *	  : Helper widget			*/

TKCTKEHelper::TKCTKEHelper
	(	TKTextEditor	*editor,
		const QString	&helpText
	)
	:
	QWidget
	(	0,
		"tkeHelper",
		Qt::WStyle_Customize|Qt::WStyle_StaysOnTop|Qt::WStyle_NoBorder
	),
	doc	(helpText, QFont())
{
	TKTextView	*view	= editor->textView ()	     ;
	int		lheight	= view->textDocument()->lineHeight() ;
	QPoint		relPos	= view->Cursor()->position() ;
	QPoint		absPos	= view->mapToGlobal (relPos) + QPoint (6, lheight) ;

	/* The code in this section is derived from the QWhatsThis	*/
	/* class ...							*/
	const int	shadowWidth	= 6  ;
	const int	vMargin		= 8  ;
	const int	hMargin		= 12 ;

	doc.adjustSize () ;
	doc.setWidth   (doc.widthUsed  () + 32) ;

	int	x	= absPos.x() ;
	int	y	= absPos.y() ;
	int 	w	= doc.width () + 2 * hMargin ;
	int 	h	= doc.height() + 2 * vMargin ;

	int	sx	= QApplication::desktop()->x     () ;
	int	sy	= QApplication::desktop()->y     () ;
	int	sw	= QApplication::desktop()->width () ;
	int	sh	= QApplication::desktop()->height() ;

	if (x + w > sw) x  = sw - w ;
	if (y + h > sh) y  = y - lheight - 2 * 6 - h ;

	if (x     < sx) x = sx ;
	if (y     < sy) y = sy ;

	setBackgroundMode (QWidget::NoBackground) 	;
	setPalette	  (QToolTip::palette(), true)	;
	setGeometry	  (x, y, w + shadowWidth, h + shadowWidth) ;
	show		  ()	;
}

void	TKCTKEHelper::paintEvent
	(	QPaintEvent		*
	)
{
	QPainter p  (this) ;
	int	 w  = width  () ;
	int	 h  = height () ;

	const int	vMargin		= 8  ;
	const int	hMargin		= 12 ;

	p.setPen    (colorGroup().foreground()) ;
	p.drawRect  (0, 0, w, h) ;
	p.setPen    (colorGroup().mid	  ()) ;
	p.setBrush  (colorGroup().background()) ;
	p.drawRect  (1, 1, w - 2, h - 2) ;
	p.setPen    (colorGroup().foreground()) ;

	doc.draw
	(	&p,
		hMargin, vMargin,
		QRect (0, 0, doc.width() + hMargin, doc.height() + vMargin),
		colorGroup(),
		0
	)	;

	p.setPen    (colorGroup().shadow()) ;
	p.drawPoint (w + 5, 6) ;
	p.drawLine  (w + 3, 6, w + 5,  8) ;
	p.drawLine  (w + 1, 6, w + 5, 10) ;

	int	i	;

	for (i = 7 ; i < h ; i += 2)
		p.drawLine (w, i, w + 5, i + 5 ) ;
	for (i = w - i + h ; i > 6 ; i -= 2 )
		p.drawLine (i, h, i + 5, h + 5 ) ;
	for ( ; i > 0 ; i -= 2 )
		p.drawLine (6, h + 6 - i, i + 5, h + 5) ;
}

/*  scanForMethod: Scan for method and return help text if any		*/
/*  editor	 : TKCTKTextEditor * : Editor				*/
/*  (returns)	 : QString	     : Method or null string if none	*/

static	QString	scanForMethod
	(	TKTextEditor	*editor
	)
{
	int	lno	= editor->line	 () ;
	int	col	= editor->column () ;

	QString	text	= editor->textDocument()->lineOf(lno)->left(col) ;

	static	QRegExp *scanner ;
	if (scanner == 0)
		scanner	= new QRegExp ("\\.[ \t]*([A-Za-z_][A-Za-z0-9_]*)[ \t]*$") ;

	if (scanner->match (text))
		return	scanner->capturedTexts()[1] ;

	return	QString::null ;
}

/*  getMethodHelp: Scan for method and return help text if any		*/
/*  editor	 : TKTextEditor *: Editor				*/
/*  (returns)	 : QString	 : Help text or QString::null if none	*/

static	QString	getMethodHelp
	(	TKTextEditor	*editor
	)
{
	QString	method	= scanForMethod (editor) ;
	if (method.isEmpty()) return QString::null  ;

//	fprintf	(stderr, "--->lookup[%s]\n", (const char *)method) ;
	QString	help 	= getPythonDict()->getAttrDescription ("__PythonMethod", method) ;
	if (help  .isEmpty()) return QString::null  ;

	return	help ;
}


/*  ------------------------------------------------------------------  */

/*  TKCTKEFindReplace							*/
/*  TKCTKEFindReplace							*/
/*		: Modal dialog wrapper round the find/replace dialog	*/
/*  (returns)	: TKCTKEFindReplace	:				*/

TKCTKEFindReplace::TKCTKEFindReplace ()
	:
	QDialog	((QWidget *)0, "", true),
	m_fr	(this, "tkeFindReplace")
{
	QVBoxLayout *lay = new QVBoxLayout (this) ;
	lay->addWidget (&m_fr) ;

	connect	(&m_fr, SIGNAL(closeDialog()), SLOT(slotClose	  ())) ;
	connect	(&m_fr, SIGNAL(find       ()), SLOT(slotFind      ())) ;
	connect	(&m_fr, SIGNAL(next   	  ()), SLOT(slotFindNext  ())) ;
	connect	(&m_fr, SIGNAL(replace    ()), SLOT(slotReplace   ())) ;
	connect	(&m_fr, SIGNAL(replaceAll ()), SLOT(slotReplaceAll())) ;
}

void	TKCTKEFindReplace::showEvent
	(	QShowEvent	*e
	)
{
	_KBDialog::setupLayout (&m_fr, -1, -1) ;

	m_fr.show	   ()  ;
	QDialog::showEvent (e) ;
	m_fr.exec 	   ()  ;
}

int	TKCTKEFindReplace::exec
	(	TKTextEditor	*editor,
		bool		replace
	)
{
	m_editor = editor ;

	if (replace)
		m_fr.prepareForReplace () ;
	else	m_fr.prepareForFind    () ;

	m_fr.enableInSelection (m_editor->hasSelection()) ;

	return	 QDialog::exec  () ;
}

void	TKCTKEFindReplace::slotFind ()
{
	if (m_editor)
		m_editor->find
		(	m_fr.findText	 (),
			m_fr.matchCase	 (),
			m_fr.wholeWord	 (),
			false,
			m_fr.inSelections()
		)	;
}

void	TKCTKEFindReplace::slotFindNext ()
{
	if (m_editor)
		m_editor->find
		(	m_fr.findText	 (),
			m_fr.matchCase   (),
			m_fr.wholeWord	 (),
			true,
			m_fr.inSelections()
		)	;
}

void	TKCTKEFindReplace::slotReplace ()
{
	if (m_editor)
		if (m_editor->hasFound())
		{
			m_editor->replaceFound (m_fr.replaceText()) ;
			slotFindNext () ;
		}
		else	slotFind () ;
}

void	TKCTKEFindReplace::slotReplaceAll ()
{
	if (m_editor)
		m_editor->replaceAll
		(	m_fr.findText	 (),
			m_fr.replaceText (),
			m_fr.matchCase	 (),
			m_fr.wholeWord	 (),
			m_fr.inSelections()
		)	;
}

void	TKCTKEFindReplace::slotClose ()
{
	m_editor = 0 ;
	done	(0)  ;
}

TKCTKEFindReplace *getFindReplace ()
{
	static	TKCTKEFindReplace *frDlg ;
	if (frDlg == 0) frDlg = new TKCTKEFindReplace ;
	return	frDlg	;
}

/*  ------------------------------------------------------------------  */

/*  TKCTKEMapper							*/
/*  TKCTKEMapper: Constructor for tkEditor-specific key mapper		*/
/*  editor	: TKTextEditor * : Editor				*/
/*  (returns)	: TKCTKEMapper	 :					*/

TKCTKEMapper::TKCTKEMapper
	(	TKTextEditor	*editor
	)
	:
	m_editor (editor)
{
	static	bool	first	= true	;

	m_helper  	= 0	;
	m_helperShown	= false ;
	m_editor->textView()->installEventFilter (this) ;

	if (first)
	{
		QString	kPath	;

		if (!(kPath = getenv ("REKALL_KEYMAPDIR")).isNull())
		{
			kPath   += "/" 	    ;
			kPath   += "keymap" ;
		}
		else
			kPath	= locateFile ("appdata", "keymap") ;


		fprintf	(stderr, "KeyMap: \"%s\"\n", (const char *)kPath) ;

		QString r = loadKeyMap (kPath) ;
		if (!r.isNull())
			fprintf (stderr, "KeyMap: %s\n", (const char *)r) ;
	}

}

/*  TKCTKEMapper							*/
/*  ~TKCTKEMapper: Destructor for tkEditor-specific key mapper		*/
/*  (returns)	 :		 :					*/

TKCTKEMapper::~TKCTKEMapper ()
{
	if (m_helper) delete m_helper  ;
}


/*  TKCTKEMapper							*/
/*  showHelper	: Show helper popup					*/
/*  helpText	: const QString & : Text				*/
/*  (returns)	: void		  :					*/

void	TKCTKEMapper::showHelper
	(	const QString	&helpText
	)
{
#ifndef	_WIN32
//	fprintf	(stderr, "------->showHelper\n") ;
	m_helperShown = false ;
	m_helper = new TKCTKEHelper (m_editor, helpText) ;

	/* Because we may be in a modal loop and the helper widget	*/
	/* cannot be parented, we cannot filter stuff like mouse clicks	*/
	/* on the helper, either directly or through the application,	*/
	/* so we have to use FocusOut on the editor to dismiss the	*/
	/* helper. But! we have to delay this since we get a focus out	*/
	/* under QT3 when we create the helper, before focus returns	*/
	/* to the editor. The following timer does this. Yek.		*/
	QTimer::singleShot (200, this, SLOT(helperShowing())) ;
#endif
}

/*  TKCTKEMapper							*/
/*  hideHelper	: Hide helper if present				*/
/*  (returns)	: void		:					*/

void	TKCTKEMapper::hideHelper ()
{
	if (m_helper)
	{	delete m_helper ;
		m_helperShown = false ;
//		fprintf	(stderr, "------->hideHelper\n") ;
	}
}

/*  TKCTKEMapper								*/
/*  useAtOuterLevel							*/
/*		: See if key should be processed at outer level		*/
/*  key		: int		: Key code from QKeyEvent		*/
/*  ctrl	: bool		: Control key pressed			*/
/*  (returns)	: bool		: Key consumed by mapper		*/

bool	TKCTKEMapper::useAtOuterLevel
	(	int		key,
		bool		ctrl
	)
{
	if (key == Qt::Key_ParenLeft)
	{
		hideHelper () ;

		QString	help	= getMethodHelp (m_editor) ;

		if (!help.isNull())
		{
			showHelper (help) ;
			m_helperLine = m_editor->line() ;

		}
	}
	return	ctrl	;
}

/*  TKCTKEMapper								*/
/*  codeToCode	: Convert text representation code code to action code	*/
/*  name	: const QString & : Text representation			*/
/*  (returns)	: int		  : Code				*/

int	TKCTKEMapper::codeToCode
	(	const QString	&name
	)
{
	struct	CodeToCode
	{	const char	*name	;
		Action		code	;
	}	;

	static	CodeToCode	mapCodeToCode[] =
	{
		{	"StartOfDoc",	StartOfDoc		},
		{	"EndOfDoc",	EndOfDoc		},
		{	"StartOfLine",	StartOfLine		},
		{	"EndOfLine",	EndOfLine		},
		{	"NextWord",	NextWord		},
		{	"PreviousWord",	PreviousWord		},
		{	"DelWordLeft",	DelWordLeft		},
		{	"DelWordRight",	DelWordRight		},
		{	"DelLine",	DelLine			},
		{	"Cut",		Cut			},
		{	"Copy",		Copy			},
		{	"Paste",	Paste			},
		{	"ScrollDown",	ScrollDown		},
		{	"ScrollUp",	ScrollUp		},
		{	"SearchForward",SearchForward		},
		{	"SearchBack",	SearchBack		},
		{	"Undo",		Undo			},
		{	"Redo",		Redo			},

		/* Add more cases here to match the Enum ...		*/
		{	0,		NoAction		}
	}	;

	for (CodeToCode *cp = &mapCodeToCode[0] ; cp->name != 0 ; cp += 1)
		if (cp->name == name)
			return	cp->code ;

	return	0 ;
}

/*  TKCTKEMapper								*/
/*  helper	: Execute an editor helper				*/
/*  code	: int		  : Helper code				*/
/*  text	: const QString & : Helper text				*/
/*  (returns)	: bool		  : True to clear callers mapping	*/

bool	TKCTKEMapper::helper
	(	int		,
		const QString	&text
	)
{
	/* ShowMethods							*/
	/* Show all possible matches for a method. We can for an	*/
	/* immediately preceding method and then get all descriptions	*/
	/* for which the method text is a prefix.			*/
	if (text == "ShowMethods")
	{
		hideHelper () ;

		QString	method	= scanForMethod (m_editor) ;
		if (method.isEmpty()) return QString::null ;

		QList<KBDictEntry> matched = getPythonDict()->getMatching ("__PythonMethod", method) ;
		QStringList	   helpers ;

		for (KBDictEntry *entry  = matched.first() ;
				  entry != 0 ;
				  entry  = matched. next())
			helpers.append (entry->m_descrip) ;

		showHelper   ("<qt>" + helpers.join ("<br/>") + "</qt>") ;
		m_helperLine = m_editor->line() ;
	}

	return	true	;
}

/*  TKCTKEMapper								*/
/*  function	: Execute an editor function				*/
/*  code	: int		  : Function code			*/
/*  text	: const QString & : Function text			*/
/*  (returns)	: bool		  : True to clear callers mapping	*/

bool	TKCTKEMapper::function
	(	int		code,
		const QString	&text
	)
{
	TKTextDocument	*doc	= m_editor->textDocument() ;
	TKTextView	*view	= m_editor->textView	() ;
	TKEditorCursor	*cursor	= view	  ->Cursor	() ;
	int		action	= -1	;

	/* First check for the cases which map directory onto cursor	*/
	/* actions ....							*/
	switch (code)
	{
		case StartOfDoc  : action = TKEditorCursor::MoveHome		; break ;
		case EndOfDoc	 : action = TKEditorCursor::MoveEnd		; break ;
		case StartOfLine : action = TKEditorCursor::MoveLineStart	; break ;
		case EndOfLine	 : action = TKEditorCursor::MoveLineEnd		; break	;
		case NextWord	 : action = TKEditorCursor::MoveWordForward	; break	;
		case PreviousWord: action = TKEditorCursor::MoveLineEnd		; break	;
		case ScrollUp	 : action = TKEditorCursor::MovePgUp		; break	;
		case ScrollDown	 : action = TKEditorCursor::MovePgDown		; break	;

		default		 : break  ;
	}

	if (action != -1)
	{
		cursor->move ((TKEditorCursor::CursorAction)action, false) ;
		return	true ;
	}

	if ((code == SearchForward) || (code == SearchBack))
	{
		getFindReplace()->exec(m_editor, false) ;
		return true ;
	}

	switch (code)
	{

#if	0
		case DelWordLeft :
			m_scintilla->delWordLeft   () ;
			break	 ;

		case DelWordRight:
			m_scintilla->delWordRight  () ;
			break	 ;
#endif
		case DelLine	 :
			doc ->removeLine (m_editor->line()) ;
			view->updateView (false) ;
			break	 ;

		case Cut	 :
			m_editor->copy	() ;
			m_editor->del   () ;
			break	 ;

		case Copy	 :
			m_editor->copy	() ;
			break	 ;

		case Paste	 :
			m_editor->paste	() ;
			break	 ;

#if	0
		case ToUpper	 :
			m_scintilla->upperCase	   () ;
			break	 ;

		case ToLower	 :
			m_scintilla->lowerCase     () ;
			break	 ;

		case ScrollUp	 :
			m_scintilla->lineScrollUp  () ;
			break	 ;

		case ScrollDown  :
			m_scintilla->lineScrollDown() ;
			break	 ;
#endif
		case Undo	 :
			m_editor->undo	() ;
			break	 ;

		case Redo	 :
			m_editor->redo	() ;
			break	 ;

		default	:
			fprintf	(stderr, "--->function[%d][%s]\n", code, (const char *)text) ;
			break	;
	}

	return	true	;
}

/*  TKCTKEMapper								*/
/*  checkChangeLine							*/
/*		: See if line has changed since helper was displayed	*/
/*  (returns)	: void		:					*/

void	TKCTKEMapper::checkChangeLine ()
{
	if (m_helper && (m_helperLine != m_editor->line()))
		hideHelper () ;
}

void	TKCTKEMapper::helperShowing ()
{
//	fprintf	(stderr, "------->helperShowing\n") ;
	m_helperShown = true ;
}

/*  TKCTKEMapper								*/
/*  filterEditor: Filter events to the tkEditor editor			*/
/*  e		: QEvent *	: The event				*/
/*  (returns)	: bool		: True if consumed			*/

bool	TKCTKEMapper::filterEditor
	(	QEvent		*e
	)
{
	switch (e->type())
	{
		case QEvent::KeyPress :
		{	/* Trap keystrokes and feed them through the	*/
			/* kep mapper. On return, start a timer if	*/
			/* there is a helper then start a timer. This	*/
			/* is needed so that we can remove it if the	*/
			/* user changes line.				*/
			QKeyEvent *k	= (QKeyEvent *)e ;

			if (m_helper) switch (k->key())
			{
				case Qt::Key_ParenRight  :
				case Qt::Key_Escape	 :
					delete	m_helper ;
					break	;

				default	:
					break	;
			}

			bool	rc	= applyKey (k->key(), (k->state() & Qt::ControlButton) != 0) ;
			if (m_helper) QTimer::singleShot(50, this, SLOT(checkChangeLine())) ;
			return	  rc	;
		}

		case QEvent::MouseButtonPress   :
		case QEvent::MouseButtonRelease :
		case QEvent::MouseButtonDblClick:
			/* All mouse clicks delete the helper		*/
			hideHelper ()	;
			return	false	;

		case QEvent::FocusOut		:
//			fprintf	(stderr, "------->focus out\n") ;
			if (m_helperShown) hideHelper () ;
			return	false	;

		case QEvent::FocusIn		:
//			fprintf	(stderr, "------->focus in\n") ;
			return	false	;

		default	:
			break	;
	}

	return	false	;
}


/*  TKCTKEMapper								*/
/*  eventFilter	: Preprocess events					*/
/*  o		: QObject *	: Object				*/
/*  e		: QEvent *	: Event					*/
/*  (returns)	: bool		: True if event consumed		*/

bool	TKCTKEMapper::eventFilter
	(	QObject		*,
		QEvent		*e
	)
{
	return	filterEditor (e) ;
}

