/***************************************************************************
    file	         : kb_wizardbits.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qregexp.h>
#include	<qapplication.h>


#ifndef		_WIN32
#include	"kb_wizard.moc"
#else
#include	"kb_wizard.h"
#endif

#include	"std.h"
#include	"eli.h"
#include	"interp.h"
#include	"syn.h"
#include	"code.h"



/*  KBWizardCtrl							*/
/*  KBWizardCtrl							*/
/*		: Constructor for wizard control base class		*/
/*  name	: const QString & : Control name			*/
/*  (returns)	: KBWizardCtrl	  :					*/

KBWizardCtrl::KBWizardCtrl
	(	KBWizardPage	*page,
		const QString	&name
	)
	:
	m_page		(page),
	m_name		(name),
	m_required	(true),
	m_wide		(false)
{
	m_changed = false ;
}

/*  KBWizardCtrl							*/
/*  ~KBWizardCtrl							*/
/*		: Destructor for wizard control base class		*/
/*  (returns)	:		:					*/

KBWizardCtrl::~KBWizardCtrl ()
{
}

/*  KBWizardCtrl							*/
/*  pageShown	: Embedding page is shown				*/
/*  next	: bool		: Moving to next page			*/
/*  (returns)	: void		:					*/

void	KBWizardCtrl::pageShown
	(	bool
	)
{
}

/*  KBWizardCtrl							*/
/*  ctrlChanged	: Notify change on control value			*/
/*  (returns)	: void		:					*/

void	KBWizardCtrl::ctrlChanged ()
{
//	fprintf
//	(	stderr,
//		"KBWizardCtrl::ctrlChanged: name=[%s]\n",
//		(cchar *)m_name
//	)	;
	m_changed = true ;
	m_page->ctrlChanged (this) ;
}

/*  KBWizardCtrl							*/
/*  ok		: Check if control value is acceptable			*/
/*  (returns)	: bool		: True if OK				*/

bool	KBWizardCtrl::ok ()
{
	if (m_required)
		if (value().isEmpty())
			return	false ;

	return	true ;
}

/*  KBWizardCtrl							*/
/*  save	: Save any required settings				*/
/*  rc		: int		: Dialog exit code			*/
/*  (returns)	: void		:					*/

void	KBWizardCtrl::save
	(	int
	)
{
}

/*  KBWizardCtrl							*/
/*  attribute	: Return control attribute value			*/
/*  name	: const QString & : Attribute name			*/
/*  (returns)	: QVariant	  : Value				*/

QVariant KBWizardCtrl::attribute
	 (	const QString	&
	 )
{
	/* Overridden by controls that have attributes ...		*/
	return	QVariant() ;
}

/*  ------------------------------------------------------------------  */

/*  KBWizardHidden							*/
/*  KBWizardHidden							*/
/*		: Constructor for line edit element in wizard page	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  defval	: const QString & : Default value			*/
/*  (returns)	: KBWizardHidden  :					*/

KBWizardHidden::KBWizardHidden
	(	KBWizardPage	*page,
		const QString	&name,
		const QString	&defval
	)
	:
	KBWizardCtrl	(page, name)
{
	m_value	= defval ;
}

/*  KBWizardHidden							*/
/*  ~KBWizardHidden							*/
/*		: Destructor for wizard line edit			*/
/*  (returns)	:		:					*/

KBWizardHidden::~KBWizardHidden ()
{
}

/*  KBWizardHidden							*/
/*  setValue	: Set value						*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBWizardHidden::setValue
	(	const QString	&value
	)
{
//	fprintf
//	(	stderr,
//		"KBWizardHidden::setValue [%s]<-[%s]\n",
//		(cchar *)name(),
//		(cchar *)value
//	)	;
	m_value	= value ;
}

/*  KBWizardHidden							*/
/*  value	: Get value						*/
/*  (returns)	: QString	: Value					*/

QString	KBWizardHidden::value ()
{
	return	m_value	;
}


/*  ------------------------------------------------------------------  */

/*  KBWizardLineEdit							*/
/*  KBWizardLineEdit							*/
/*		: Constructor for line edit element in wizard page	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  defval	: const QString & : Default value			*/
/*  pwd		: bool		  : Display as password			*/
/*  (returns)	: KBWizardLineEdit:					*/

KBWizardLineEdit::KBWizardLineEdit
	(	KBWizardPage	*page,
		const QString	&name,
		const QString	&defval,
		bool		pwd
	)
	:
	KBWizardCtrl	(page, name)
{
	setWidget (m_lineEdit = new TKLineEdit (page)) ;
	m_lineEdit->setText (defval) ;

	if (pwd)
		m_lineEdit->setEchoMode (TKLineEdit::Password) ;

	connect
	(	m_lineEdit,
		SIGNAL(textChanged(const QString &)),
		SLOT  (ctrlChanged())
	)	;

	setChanged (false) ;
}

/*  KBWizardLineEdit							*/
/*  ~KBWizardLineEdit							*/
/*		: Destructor for wizard line edit			*/
/*  (returns)	:		:					*/

KBWizardLineEdit::~KBWizardLineEdit ()
{
}

/*  KBWizardLineEdit							*/
/*  setValue	: Set value						*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBWizardLineEdit::setValue
	(	const QString	&value
	)
{
	m_lineEdit->setText (value) ;
	setChanged (false) ;
}

/*  KBWizardLineEdit							*/
/*  value	: Get value						*/
/*  (returns)	: QString	: Value					*/

QString	KBWizardLineEdit::value ()
{
	return	m_lineEdit->text () ;
}


/*  ------------------------------------------------------------------  */

/*  ------------------------------------------------------------------  */

/*  KBWizardCheckBox							*/
/*  KBWizardCheckBox							*/
/*		: Constructor for checkbox element in wizard page	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  defval	: const QString & : Default value			*/
/*  (returns)	: KBWizardCheckBox:					*/

KBWizardCheckBox::KBWizardCheckBox
	(	KBWizardPage	*page,
		const QString	&name,
		const QString	&defval
	)
	:
	KBWizardCtrl	(page, name)
{
	setWidget (m_checkBox = new QCheckBox (page)) ;
	m_checkBox->setChecked (defval.toUInt()) ;

	connect
	(	m_checkBox,
		SIGNAL(toggled    (bool)),
		SLOT  (ctrlChanged())
	)	;

	setChanged (false) ;
}

/*  KBWizardCheckBox							*/
/*  ~KBWizardCheckBox							*/
/*		: Destructor for wizard checkbox			*/
/*  (returns)	:		:					*/

KBWizardCheckBox::~KBWizardCheckBox ()
{
}

/*  KBWizardCheckBox							*/
/*  setValue	: Set value						*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBWizardCheckBox::setValue
	(	const QString	&value
	)
{
	m_checkBox->setChecked (value.toInt()) ;
	setChanged (false) ;
}

/*  KBWizardCheckBox							*/
/*  value	: Get value						*/
/*  (returns)	: QString	: Value					*/

QString	KBWizardCheckBox::value ()
{
	return	m_checkBox->isChecked() ? "1" : "0" ;
}


/*  ------------------------------------------------------------------  */

/*  KBWizardComboBox							*/
/*  KBWizardComboBox							*/
/*		: Constructor for line edit element in wizard page	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  values	: QStringList &	  : Values				*/
/*  defval	: const QString & : Default value			*/
/*  editable	: bool		  : Control is editable			*/
/*  (returns)	: KBWizardComboBox:					*/

KBWizardComboBox::KBWizardComboBox
	(	KBWizardPage		*page,
		const QString		&name,
		const QStringList	&values,
		const QString		&defval,
		bool			editable
	)
	:
	KBWizardCtrl	(page, name)
{
	setWidget (m_comboBox = new QComboBox (page)) ;
	m_infoView = 0 ;

	int	at	= -1 ;
	for (uint idx = 0 ; idx < values.count() ; idx += 1)
	{	if (values[idx] == defval) at = idx  ;
		m_comboBox->insertItem (values[idx]) ;
	}

	if (at >= 0) m_comboBox->setCurrentItem(at)  ;

	m_comboBox->setEditable (editable) ;

	connect
	(	m_comboBox,
		SIGNAL(activated  (int)),
		SLOT  (ctrlChanged())
	)	;

	setChanged (false) ;
}

/*  KBWizardComboBox							*/
/*  ~KBWizardComboBox							*/
/*		: Destructor for wizard combobox			*/
/*  (returns)	:		:					*/

KBWizardComboBox::~KBWizardComboBox ()
{
}

/*  KBWizardComboBox							*/
/*  setValue	: Set value						*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBWizardComboBox::setValue
	(	const QString	&value
	)
{
	for (int idx = 0 ; idx < m_comboBox->count() ; idx += 1)
		if (m_comboBox->text(idx) == value)
		{	m_comboBox->setCurrentItem (idx) ;
			break	;
		}
}

/*  KBWizardComboBox							*/
/*  value	: Get value						*/
/*  (returns)	: QString	: Value					*/

QString	KBWizardComboBox::value ()
{
	return	m_comboBox->currentText () ;
}


/*  KBWizardComboBox							*/
/*  attribute	: Return control attribute value			*/
/*  name	: const QString & : Attribute name			*/
/*  (returns)	: QVariant	  : Value				*/

QVariant KBWizardComboBox::attribute
	 (	const QString	&name
	 )
{
	if (name == "index")
		return	QVariant (m_comboBox->currentItem()) ;

	return	KBWizardCtrl::attribute (name) ;
}

/*  KBWizardComboBox							*/
/*  setInfoList	: Set extended information text				*/
/*  infoList	: QStringList &	: Information list			*/
/*  (returns)	: void		:					*/

void	KBWizardComboBox::setInfoList
	(	const QStringList	&infoList
	)
{
	/* If this is the first time then create the information viewer	*/
	/* and register it with the page.				*/
	if (m_infoView == 0)
	{
		m_infoView = new QTextView (page()) ;
		page()->setInfoCtrl (m_infoView) ;
	}

	/* Save the information list and display the entry for the	*/
	/* current item.						*/
	m_infoList = infoList ;
	m_infoView->setText (m_infoList[m_comboBox->currentItem()]) ;
}

/*  KBWizardCtrl							*/
/*  ctrlChanged	: Notify change on control value			*/
/*  (returns)	: void		:					*/

void	KBWizardComboBox::ctrlChanged ()
{
//	fprintf
//	(	stderr,
//		"KBWizardComboBox::ctrlChanged: name=[%s]\n",
//		(cchar *)name()
//	)	;

	if (m_infoView != 0)
		m_infoView->setText (m_infoList[m_comboBox->currentItem()]) ;

	KBWizardCtrl::ctrlChanged () ;
}


/*  ------------------------------------------------------------------  */

/*  KBWizardPage							*/
/*  KBWizardPage: Constructor for a wizard page				*/
/*  wizard	: KBWizard *	  : Parent wizard			*/
/*  name	: const QString & : Page name				*/
/*  parent	: QWidget *	  : Parent widget			*/
/*  (returns)	: KBWizardPage	  :					*/

KBWizardPage::KBWizardPage
	(	KBWizard	*wizard,
		QWidget		*parent,
		const QString	&name
	)
	:
	QWidget		(parent),
	m_wizard	(wizard),
	m_name		(name)
{
	m_elNext = 0	 ;
	m_elOK	 = 0	 ;
	m_finish = false ;
	m_info	 = false ;
	m_layout = new QGridLayout (this) ;
	_KBDialog::setupLayout (m_layout, -1, -1) ;
}

/*  KBWizardPage							*/
/*  ~KBWizardPage: Destructor for a wizard page				*/
/*  (returns)	 :		:					*/

KBWizardPage::~KBWizardPage ()
{
	if (m_elNext != 0) free (m_elNext) ;
	if (m_elOK   != 0) free (m_elOK  ) ;
}

/*  KBWizardPage							*/
/*  init	: Initialise page					*/
/*  page	: QDomElement &	: Definition element			*/
/*  (returns)	: void		:					*/

void	KBWizardPage::init
	(	const QDomElement	&page
	)
{
	m_elem	 = page ;
	m_finish = page.attribute("finish").toUInt() ;
	setTitle  (page.attribute("title")) ;

	for (QDomNode node = page.firstChild () ;
			    !node.isNull() ;
		      node = node.nextSibling() )
	{
		QDomElement elem = node.toElement () ;
		if (elem.isNull()) continue ;

		if (elem.nodeName() == "blurb")
		{
			setBlurb (elem.text()) ;
			continue   ;
		}

		addCtrl (elem) ;
	}

	addedAll () ;
}

/*  KBWizardPage							*/
/*  addHiddenCtrl							*/
/*		: Add a hidden control to the page			*/
/*  name	: cchar *	   : Control name			*/
/*  legend	: const QString &  : Not used				*/
/*  defval	: const QString &  : Default value			*/
/*  (returns)	: KBWizardHidden * :					*/

KBWizardHidden
	*KBWizardPage::addHiddenCtrl
	(	const QString	&name,
		const QString	&,
		const QString	&defval
	)
{
	KBWizardHidden	*ctrl	= new KBWizardHidden (this, name, defval) ;
	m_ctrls.append (ctrl)  ;
	return	ctrl	;
}

/*  KBWizardPage							*/
/*  addHiddenCtrl							*/
/*		: Add a hidden control to the page			*/
/*  ctrlElem	: QDomElement &	   : Specification element		*/
/*  (returns)	: KBWizardHidden * :					*/

KBWizardHidden
	*KBWizardPage::addHiddenCtrl
	(	const QDomElement	&ctrlElem
	)
{
	return	addHiddenCtrl
		(	ctrlElem.attribute("name"   ),
			ctrlElem.attribute("legend" ),
			ctrlElem.attribute("default")
		)	;
}

/*  KBWizardPage							*/
/*  addTextCtrl	: Add a line edit control to the page			*/
/*  name	: cchar *	     : Control name			*/
/*  legend	: const QString &    : Legend for display		*/
/*  defval	: const QString &    : Default value			*/
/*  pwd		: bool		     : Display as password		*/
/*  (returns)	: KBWizardLineEdit * :					*/

KBWizardLineEdit
	*KBWizardPage::addTextCtrl
	(	const QString	&name,
		const QString	&legend,
		const QString	&defval,
		bool		pwd
	)
{
	KBWizardLineEdit*ctrl	= new KBWizardLineEdit (this, name, defval, pwd) ;
	QLabel		*label	= new QLabel (this) ;

	m_layout->addWidget (label,	     m_ctrls.count(), 0) ;
	m_layout->addWidget (ctrl->widget(), m_ctrls.count(), 1) ;
	m_ctrls . append    (ctrl)  ;
	m_labels. append    (label)  ;

	label   ->setText   (legend) ;

	return	ctrl	;
}

/*  KBWizardPage							*/
/*  addTextCtrl	: Add a line edit control to the page			*/
/*  ctrlElem	: QDomElement &	     : Specification element		*/
/*  (returns)	: KBWizardLineEdit * :					*/

KBWizardLineEdit
	*KBWizardPage::addTextCtrl
	(	const QDomElement	&ctrlElem
	)
{
	return	addTextCtrl
		(	ctrlElem.attribute("name"    ),
			ctrlElem.attribute("legend"  ),
			ctrlElem.attribute("default" ),
			ctrlElem.attribute("password").toUInt()
		)	;
}

/*  KBWizardPage							*/
/*  addChoiceCtrl							*/
/*		: Add a combobox control to the page			*/
/*  name	: cchar *	     : Control name			*/
/*  legend	: const QString &    : Legend for display		*/
/*  values	: QStringList &	     : Values				*/
/*  defval	: const QString &    : Default value			*/
/*  editable	: bool		     : Control is editable		*/
/*  (returns)	: KBWizardComboBox * :					*/

KBWizardComboBox
	*KBWizardPage::addChoiceCtrl
	(	const QString		&name,
		const QString		&legend,
		const QStringList	&values,
		const QString		&defval,
		bool			editable
	)
{
	KBWizardComboBox
			*ctrl	= new KBWizardComboBox (this, name, values, defval, editable) ;
	QLabel		*label	= new QLabel (this) ;

	m_layout->addWidget (label,	     m_ctrls.count(), 0) ;
	m_layout->addWidget (ctrl->widget(), m_ctrls.count(), 1) ;
	m_ctrls  .append    (ctrl )  ;
	m_labels .append    (label)  ;

	label   ->setText   (legend) ;

	return	ctrl	;
}

/*  KBWizardPage							*/
/*  addChoiceCtrl: Add a choice control to the page			*/
/*  ctrlElem	  : QDomElement &      : Specification element		*/
/*  (returns)	  : KBWizardComboBox * :				*/

KBWizardComboBox
	*KBWizardPage::addChoiceCtrl
	(	const QDomElement	&ctrlElem
	)
{
	QStringList	values	;
	QStringList	infoList;
	QString		defval	;

	for (QDomNode node = ctrlElem.firstChild () ;
			     !node    .isNull     () ;
		      node = node     .nextSibling() )
	{
		QDomElement elem = node.toElement () ;
		if (elem.isNull()) continue ;

		if (elem.nodeName() == "value")
		{
			values  .append (elem.attribute("text")) ;
			infoList.append (elem.text()) ;

			if (!elem.attribute("default").isNull())
				defval = elem.attribute("text") ;
		}
	}

	KBWizardComboBox
			*ctrl	= addChoiceCtrl
				  (	ctrlElem.attribute("name"  ),
					ctrlElem.attribute("legend"),
					values,
					defval,
					ctrlElem.attribute("editable").toUInt()
				  )	;

	if (ctrlElem.attribute("info").toInt())
		ctrl->setInfoList (infoList) ;

	return	ctrl	;
}

/*  KBWizardPage							*/
/*  addCheckCtrl: Add a checkbox control to the page			*/
/*  name	: cchar *	     : Control name			*/
/*  legend	: const QString &    : Legend for display		*/
/*  defval	: const QString &    : Default value			*/
/*  (returns)	: KBWizardCheckBox * :					*/

KBWizardCheckBox
	*KBWizardPage::addCheckCtrl
	(	const QString	&name,
		const QString	&legend,
		const QString	&defval
	)
{
	KBWizardCheckBox *ctrl	= new KBWizardCheckBox (this, name, defval) ;

	ctrl->checkBox()->setText (legend) ;

	m_layout->addWidget (ctrl->widget(), m_ctrls.count(), 1) ;
	m_ctrls . append    (ctrl) ;

	return	ctrl	;
}

/*  KBWizardPage							*/
/*  addCheckCtrl: Add a checkbox control to the page			*/
/*  ctrlElem	: QDomElement &	     : Specification element		*/
/*  (returns)	: KBWizardCheckBox * :					*/

KBWizardCheckBox
	*KBWizardPage::addCheckCtrl
	(	const QDomElement	&ctrlElem
	)
{
	return	addCheckCtrl
		(	ctrlElem.attribute("name"   ),
			ctrlElem.attribute("legend" ),
			ctrlElem.attribute("default")
		)	;
}

/*  KBWizardPage							*/
/*  addCtrl	: Add control to page					*/
/*  elem	: QDomElement &	 : Specification element		*/
/*  (returns)	: KBWizardCtrl * : Control				*/

KBWizardCtrl
	*KBWizardPage::addCtrl
	(	const QDomElement	&elem
	)
{
	KBWizardCtrl	*ctrl	;

	if	(elem.nodeName() == "text")
	{
		ctrl	= addTextCtrl   (elem) ;
	}
	else if (elem.nodeName() == "choice")
	{
		ctrl	= addChoiceCtrl (elem) ;
	}
	else if (elem.nodeName() == "check")
	{
		ctrl	= addCheckCtrl (elem) ;
	}
	else
	{
		ctrl   = KBWizardCtrlReg::makeWizardCtrl
			     (		elem.nodeName(),
			     		this,
			     		elem
			     )	;

		if (ctrl == 0) return 0 ;

		if (ctrl->wide())
		{
			m_layout->addMultiCellWidget
			(	ctrl->widget (),
				m_ctrls.count(),
				m_ctrls.count(),
				0, 1
			)	;

			m_ctrls  .append    (ctrl )  ;
		}
		else
		{
			QLabel	     *label  = new QLabel (this) ;

			m_layout->addWidget (label,	     m_ctrls.count(), 0) ;
			m_layout->addWidget (ctrl->widget(), m_ctrls.count(), 1) ;
			m_ctrls  .append    (ctrl )  ;
			m_labels .append    (label)  ;

			label   ->setText   (elem.attribute ("legend")) ;
		}
	}

	if (ctrl != 0) ctrl->setRequired (elem.attribute("required", "1").toInt()) ;
	return	ctrl ;
}

/*  KBWizardPage							*/
/*  setInfoCtrl	: Set page information control				*/
/*  info	: QTextView *	: Information control			*/
/*  (returns)	: void		:					*/

void	KBWizardPage::setInfoCtrl
	(	QTextView	*info
	)
{
	m_info	= info ;
	m_info->setTextFormat (Qt::RichText) ;
}

/*  KBWizardPage							*/
/*  settings	: Get settings from controls on page			*/
/*  dict	: QDict<QString> & : Results dictionary			*/
/*  changed	: bool		   : Return only changed values		*/
/*  (returns)	: void		   :					*/

void	KBWizardPage::settings
	(	QDict<QString>	&dict,
		bool		changed
	)
{
	for (uint idx = 0 ; idx < m_ctrls.count() ; idx += 1)
	{
		KBWizardCtrl *ctrl = m_ctrls.at(idx) ;

		if (!changed || ctrl->changed())
			dict.insert
			(	ctrl->name (),
				new QString(ctrl->value())
			)	;
	}
}

/*  KBWizardPage							*/
/*  addedAll	: Called after all ctrls are added			*/
/*  (returns)	: void		   :					*/

void	KBWizardPage::addedAll ()
{
	/* If there is an information contrl then add this. It should	*/
	/* expand to fill the page.					*/
	if (m_info)
	{
		m_layout->addMultiCellWidget
		(	m_info,
			m_ctrls.count(),
			m_ctrls.count(),
			0,
			1
		)	;
	}

	/* Unless the "nofill" option is specified, set the stretch	*/
	/* factor on the next empty row so that the controls are	*/
	/* packed at the top of the page.				*/
	if (m_elem.attribute("nofill").toUInt() == 0)
		m_layout->setRowStretch (m_ctrls.count(), 1) ;
}

/*  KBWizardPage							*/
/*  pageShown	: Page is being shown					*/
/*  next	: bool		: Moving to next page			*/
/*  (returns)	: void		   :					*/

void	KBWizardPage::pageShown
	(	bool		next
	)
{
	for (uint idx = 0 ; idx < m_ctrls.count() ; idx += 1)
		m_ctrls.at(idx)->pageShown(next) ;
}

/*  KBWizardPage							*/
/*  clear	: Clear all controls from the page			*/
/*  (returns)	: void		   :					*/

void	KBWizardPage::clear ()
{
	m_ctrls .setAutoDelete (true ) ;
	m_labels.setAutoDelete (true ) ;

	m_ctrls .clear () ;
	m_labels.clear () ;

	m_ctrls .setAutoDelete (false) ;
	m_labels.setAutoDelete (false) ;

	delete	 m_layout ;
	m_layout = new QGridLayout (this) ;
	m_info	 = 0 	  ;
}


void	KBWizardPage::setCtrl
	(	const QString	&name,
		const QString	&value
	)
{
	for (uint idx = 0 ; idx < m_ctrls.count() ; idx += 1)
		if (m_ctrls.at(idx)->name() == name)
		{
			m_ctrls.at(idx)->setValue (value)  ;
			break	;
		}
}

void	KBWizardPage::setCtrl
	(	uint		index,
		const QString	&value
	)
{
	if (index < m_ctrls.count())
		m_ctrls.at(index)->setValue (value) ;
}

QString	KBWizardPage::ctrlValue
	(	uint		index
	)
{
	return	index < m_ctrls.count() ? m_ctrls.at(index)->value() : QString::null ;
}

KBWizardCtrl
	*KBWizardPage::findCtrl
	(	const QString	&name,
		const char *	klass
	)
{
	for (uint idx = 0 ; idx < m_ctrls.count() ; idx += 1)
		if (m_ctrls.at(idx)->name() == name)
		{
			KBWizardCtrl *ctrl =  m_ctrls.at(idx) ;
			if ((klass != 0) && !ctrl->inherits (klass))
				return	0  ;
			return	ctrl	;
		}

	return	0 ;
}

void	KBWizardPage::ctrlChanged
	(	KBWizardCtrl	*ctrl
	)
{
//	fprintf
//	(	stderr,
//		"KBWizardPage::ctrlChanged: name=[%s] ctrl=[%s] w=[%p]\n",
//		(cchar *)m_name,
//		(cchar *)ctrl->name(),
//		(void  *)m_wizard
//	)	;

	if (m_wizard != 0)
		m_wizard->ctrlChanged (this, ctrl) ;
}

/*  KBWizardPage							*/
/*  compile	: Compile code from named element			*/
/*  name	: const QString & : Element name			*/
/*  (returns)	: CBUFF		  : Code buffer or null if none/error	*/

CBUFF	*KBWizardPage::compile
	(	const QString	&name
	)
{
	static	bool	first	= true	;

	if (true)
	{
		extern	LIBEL32_API MC	el_lib1sp[] ;
		el_initlib (el_lib1sp)	;
		first	= false		;
	}

	DOMITER_BEGIN(m_elem,name,elem)
	{
		QString	expr = elem.text() ;

		fprintf
		(	stderr,
			"KBWizardPage::compile: [%s][%s]\n",
			(cchar *)name,
			(cchar *)expr
		)	;

		if (expr.isEmpty()) break ;

		QString script = "global print ; public f (page) { " + expr + "; }" ;
 
	        CBUFF   *cbuff  = el_compile (0, 0, 0, script, 0) ;
		if (cbuff == 0)
		{
			fprintf
			(	stderr,
				"KBWizardPage::nextPage: compile error\n"
			)	;

			return	0 ;
		}

		return	cbuff	;
	}
	DOMITER_END

	return	0 ;
}

/*  KBWizardPage							*/
/*  execute	: Execute code in buffer				*/
/*  cbuff	: CBUFF *	: Code buffer				*/
/*  (returns)	: QString	: Return value or null on error		*/

QString	KBWizardPage::execute
	(	CBUFF		*cbuff
	)
{
	extern	ELTAG	wiz_page_TAG	;

	if (!el_loadtext (cbuff))
	{
		fprintf
		(	stderr,
			"KBWizardPage::execute: load error\n"
		)	;

		return	0 ;
	}

	VALUE	argv	= VALUE    ((void *)this, &wiz_page_TAG) ;
	VALUE	res	= el_vexec ("", "f", 1, &argv) ;
	QString	rv	;

	switch	(res.tag->tag)
	{
		case V_NUM	:
			rv	= QString::number (res.val.num) ;
			break	;

		case V_DBL	:
			rv	= QString::number (res.val.num) ;
			break	;

		case V_STR	:
			rv	= res.val.str->text ;
			break	;

		default		:
			fprintf
			(	stderr,
				"KBWizardPage::execute: unexpected tag [%d]\n",
				res.tag->tag
			)	;
			rv	= "" ;
			break	;
	}

//	fprintf
//	(	stderr,
//		"KBWizardPage::execute -> [%s]\n",
//		(cchar *)rv
//	)	;
	return	rv	;
}

/*  KBWizardPage							*/
/*  ok		: Check if page values are acceptable			*/
/*  (returns)	: bool		: True if OK				*/

bool	KBWizardPage::ok ()
{
//	fprintf
//	(	stderr,
//		"KBWizardPage::ok: called\n"
//	)	;

	/* Start by seeing if there is any code associated with the	*/
	/* OK check, in which case execute that and pass its result	*/
	/* back.							*/
	if (m_elOK == 0)
		m_elOK = compile ("ok") ;

	if (m_elOK != 0)
		return	execute  (m_elOK).toUInt() ;

	/* No code so fall back on the default which is to check that	*/
	/* each of the controls is OK.					*/
	for (uint idx = 0 ; idx < m_ctrls.count() ; idx += 1)
		if (!m_ctrls.at(idx)->ok())
			return	false	;

	return	true ;
}

/*  KBWizardPage							*/
/*  save	: Save any required settings				*/
/*  rc		: int		: Dialog exit code			*/
/*  (returns)	: void		:					*/

void	KBWizardPage::save
	(	int		rc
	)
{
	for (uint idx = 0 ; idx < m_ctrls.count() ; idx += 1)
		m_ctrls.at(idx)->save(rc) ;
}

/*  KBWizardPage							*/
/*  nextPage	: Get name of next page					*/
/*  (returns)	: QString	: Page name or null for default		*/

QString	KBWizardPage::nextPage ()
{
//	fprintf
//	(	stderr,
//		"KBWizardPage::nextPage: called\n"
//	)	;

	/* See if there is any code associated with the next check, and	*/
	/* use that if so. If not then return null, in which case	*/
	/* control should move to the next sequential page.		*/
	if (m_elNext == 0)
		m_elNext = compile ("next") ;

	if (m_elNext != 0)
		return	execute  (m_elNext) ;

	return	QString::null	;
}

