/***************************************************************************
    file	         : kb_wizard.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<errno.h>

#include	<qfile.h>

#if		__KB_KDE

#include	<klocale.h>
#include	<kglobal.h>

#endif		// __KB_KDE

#include	"kb_wizard.h"
#include	"kb_locator.h"


#include	"std.h"
#include	"eli.h"
#include	"interp.h"
#include	"syn.h"
#include	"code.h"


extern	METHSET	wiz_ctrl_methset ;
extern	METHSET	wiz_page_methset ;
extern	METHSET	wiz_wiz_methset  ;

ELTAG	wiz_ctrl_TAG	=
{
	128,
	0,
	&wiz_ctrl_methset,
	"wizardCtrl"
}	;

ELTAG	wiz_page_TAG	=
{
	129,
	0,
	&wiz_page_methset,
	"wizardPage"
}	;

ELTAG	wiz_wiz_TAG	=
{
	130,
	0,
	&wiz_wiz_methset,
	"wizard"
}	;

static	VALUE	if_ctrlValue
	(	VALUE	*argv
	)
{
	KBWizardCtrl	*ctrl	= (KBWizardCtrl *)argv[0].val.user ;
	QString		value	= ctrl->value() ;

	if (value.isNull()) value = ""  ;
	return	new STRING((cchar *)value) ;
}

static	VALUE	if_ctrlSetValue
	(	VALUE	*argv
	)
{
	KBWizardCtrl	*ctrl	= (KBWizardCtrl *)argv[0].val.user ;

	ctrl->setValue	(argv[1].val.str->text) ;

	return	0 ;
}

static	VALUE	if_ctrlAttr
	(	VALUE	*argv
	)
{
	KBWizardCtrl	*ctrl	= (KBWizardCtrl *)argv[0].val.user ;
	cchar		*attr	= argv[1].val.str->text ;
	QVariant	value	= ctrl->attribute(attr) ;

	if (!value.isValid())
		return	0 ;

	switch (value.type())
	{
		case QVariant::Int  :
			return	value.asInt () ;

		case QVariant::UInt :
			return	value.asUInt() ;

		case QVariant::StringList :
		{
			QStringList strings = value.toStringList() ;
			VEC	    *rv	    = new VEC(strings.count()) ;

			for (uint idx = 0 ; idx < strings.count() ; idx += 1)
				rv->vals[idx] = new STRING((cchar *)strings[idx]) ;

			return	rv   ;
		}

		default	:
			break	;
	}

	QString	text = value.toString() ;
	if (text.isNull()) text = ""    ;
	return	new STRING((cchar *)text) ;
}

static	VALUE	if_ctrlOK
	(	VALUE	*argv
	)
{
	KBWizardCtrl	*ctrl	= (KBWizardCtrl *)argv[0].val.user ;
	return	ctrl->ok() ;
}

static	VALUE	if_pageCtrl
	(	VALUE	*argv
	)
{
	KBWizardPage	*page	= (KBWizardPage *)argv[0].val.user ;
	cchar		*name	= argv[1].val.str->text ;

	KBWizardCtrl	*ctrl	= page->findCtrl (name) ;
	if (ctrl == 0)
	{
		fprintf
		(	stderr,
			"if_pageCtrl: not found: [%s]\n",
			name
		)	;

		return	0 ;
	}

	return	VALUE ((void *)ctrl, &wiz_ctrl_TAG) ;
}

static	VALUE	if_pageWizard
	(	VALUE	*argv
	)
{
	KBWizardPage	*page	= (KBWizardPage *)argv[0].val.user ;
	KBWizard	*wizard	= page->wizard() ;

	if (wizard == 0)
	{
		fprintf
		(	stderr,
			"if_pageWizard: no wizard\n"
		)	;

		return	0 ;
	}

	return	VALUE ((void *)wizard, &wiz_wiz_TAG) ;
}

static	VALUE	if_wizardPage
	(	VALUE	*argv
	)
{
	KBWizard	*wizard	= (KBWizard *)argv[0].val.user ;
	cchar		*name	= argv[1].val.str->text ;

	KBWizardPage	*page	= wizard->findPage (name) ;
	if (page == 0)
	{
		fprintf
		(	stderr,
			"if_wizardPage: not found: [%s]\n",
			name
		)	;

		return	0 ;
	}

	return	VALUE ((void *)page, &wiz_page_TAG) ;
}

MC	wiz_ctrl_methods[] =
{
	{	"value",
		{
	 		0
		},
		if_ctrlValue
	},
	{	"attr",
		{
			&tagSTR,
			0
		},
		if_ctrlAttr
	},
	{	"setValue",
		{
			&tagSTR,
			0
		},
		if_ctrlSetValue
	},
	{	"ok",
		{
			0
		},
		if_ctrlOK
	},
	{
		0,
		{
			0
		},
		0
	}
}	;

MC	wiz_page_methods[] =
{
	{	"ctrl",
		{
			&tagSTR,
	 		0
		},
		if_pageCtrl
	},
	{	"wizard",
		{
	 		0
		},
		if_pageWizard
	},
	{
		0,
		{
			0
		},
		0
	}
}	;

MC	wiz_wiz_methods[] =
{
	{	"page",
		{
			&tagSTR,
	 		0
		},
		if_wizardPage
	},
	{
		0,
		{
			0
		},
		0
	}
}	;

ELTAG	*wiz_ctrl_allowed[] =
{
	&wiz_ctrl_TAG
}	;

ELTAG	*wiz_page_allowed[] =
{
	&wiz_page_TAG
}	;

ELTAG	*wiz_wiz_allowed[] =
{
	&wiz_wiz_TAG
}	;

METHSET	wiz_ctrl_methset = 
{
	0,
	wiz_ctrl_allowed,
	wiz_ctrl_methods
}	;

METHSET	wiz_page_methset = 
{
	0,
	wiz_page_allowed,
	wiz_page_methods
}	;

METHSET	wiz_wiz_methset = 
{
	0,
	wiz_wiz_allowed,
	wiz_wiz_methods
}	;


/*  ------------------------------------------------------------------	*/

class	KBWizardCtrlMaker
{
public	:

	cchar		*m_name	;
	MKWIZARDCTRL	m_fn	;

	KBWizardCtrlMaker (cchar *, MKWIZARDCTRL) ;
}	;

KBWizardCtrlMaker::KBWizardCtrlMaker
	(	cchar		*name,
		MKWIZARDCTRL	fn
	)
	:
	m_name	(name),
	m_fn	(fn)
{
}


static	QDict<KBWizardCtrlMaker>	wizardCtrlMaker	;

KBWizardCtrlReg::KBWizardCtrlReg
	(	cchar		*name,
		MKWIZARDCTRL	fn
	)
{
	fprintf
	(	stderr,
		"KBWizardCtrlReg::KBWizardCtrlReg: [%s]\n",
		name
	)	;

	wizardCtrlMaker.insert (name, new KBWizardCtrlMaker (name, fn)) ;
}


KBWizardCtrl
	*KBWizardCtrlReg::makeWizardCtrl
	(	const QString		&name,
		KBWizardPage		*page,
		const QDomElement	&elem
	)
{
	KBWizardCtrlMaker *maker = wizardCtrlMaker.find (name) ;

	fprintf
	(	stderr,
		"KBWizardCtrlReg::makeWizardCtrl: [%s]->[%p\n",
		(cchar *)name,
		(void  *)maker
	)	;


	return	maker == 0 ? 0 : (*maker->m_fn) (page, elem) ;
}

/*  ------------------------------------------------------------------  */

class	KBWizardMaker
{
public	:

	cchar		*m_name	;
	MKWIZARD	m_fn	;

	KBWizardMaker (cchar *, MKWIZARD) ;
}	;

KBWizardMaker::KBWizardMaker
	(	cchar		*name,
		MKWIZARD	fn
	)
	:
	m_name	(name),
	m_fn	(fn)
{
}
static	QDict<KBWizardMaker>		wizardMaker	;

KBWizardReg::KBWizardReg
	(	cchar		*name,
		MKWIZARD	fn
	)
{
	fprintf
	(	stderr,
		"KBWizardReg::KBWizardReg: [%s]\n",
		name
	)	;

	wizardMaker.insert (name, new KBWizardMaker (name, fn)) ;
}


KBWizard
	*KBWizardReg::makeWizard
	(	const QString		&name,
		KBLocation		&location,
		const QString		&server
	)
{
	KBWizardMaker *maker = wizardMaker.find (name) ;

	fprintf
	(	stderr,
		"KBWizardReg::makeWizard: [%s]->[%p\n",
		(cchar *)name,
		(void  *)maker
	)	;


	return	maker == 0 ? 0 : (*maker->m_fn) (location, server) ;
}

/*  ------------------------------------------------------------------  */

/*  KBWizard								*/
/*  KBWizard	: Constructor for rekall wizard class			*/
/*  dbInfo	: KBDBInfo *		: Database information		*/
/*  server	: const QString &	: Server name			*/
/*  (returns)	: KBWizard		:				*/

KBWizard::KBWizard
	(	KBDBInfo	*dbInfo,
		const QString	&server
	)
	:
	_KBDialog	("", true, "kbwizard"),
	m_dbInfo	(dbInfo),
	m_server	(server),
	m_title		(this),
	m_blurb		(this),
	m_stack		(this),
	m_bPrevious	(this),
	m_bNext		(this),
	m_bFinish	(this),
	m_bCancel	(this)
{
	QVBoxLayout	*layMain = new QVBoxLayout (this   ) ;
	QHBoxLayout	*layTitle= new QHBoxLayout (layMain) ;

	layTitle->addWidget (&m_title) ;

	QFrame		*hbar1	 = new QFrame (this) ;
	hbar1  ->setFrameStyle  (QFrame::Sunken|QFrame::HLine) ;
	hbar1  ->setFixedHeight (12)	;
	layMain->addWidget	(hbar1)	;

	QHBoxLayout	*layData = new QHBoxLayout (layMain) ;

	QFrame		*hbar2	 = new QFrame (this) ;
	hbar2  ->setFrameStyle  (QFrame::Sunken|QFrame::HLine) ;
	hbar2  ->setFixedHeight (12)	;
	layMain->addWidget	(hbar2)	;

	QHBoxLayout	*layButt = new QHBoxLayout (layMain) ;

	layData->addWidget (&m_blurb) ;
	layData->addWidget (&m_stack) ;

	layButt->addStretch() ;
	layButt->addWidget (&m_bPrevious) ;
	layButt->addWidget (&m_bNext	) ;
	layButt->addWidget (&m_bFinish	) ;
	layButt->addWidget (&m_bCancel	) ;

	m_title.setTextFormat (Qt::RichText) ;
	m_blurb.setTextFormat (Qt::RichText) ;

	m_bPrevious.setText   (TR("Previous")) ;
	m_bNext    .setText   (TR("Next"    )) ;
	m_bFinish  .setText   (TR("Finish"  )) ;
	m_bCancel  .setText   (TR("Cancel"  )) ;

	connect	(&m_bPrevious,	SIGNAL(clicked()), SLOT(clickPrevious())) ;
	connect	(&m_bNext,	SIGNAL(clicked()), SLOT(clickNext    ())) ;
	connect	(&m_bFinish,	SIGNAL(clicked()), SLOT(clickFinish  ())) ;
	connect	(&m_bCancel,	SIGNAL(clicked()), SLOT(clickCancel  ())) ;

	m_curPage = 0 ;

	m_blurb.setMinimumWidth  (180) ;
	m_blurb.setMinimumHeight (250) ;
	m_stack.setMinimumWidth  (320) ;

	static	bool	first	= true ;
	if (first)
	{
		el_init    (1024 * 64, 4096) ;
		first	= false ;
	}
}

/*  KBWizard								*/
/*  showPage	: Show specified page					*/
/*  idx		: uint		 : Page index				*/
/*  page	: KBWizardPage * : The page				*/
/*  push	: bool		 : Push page onto history		*/
/*  (returns)	: void		 :					*/

void	KBWizard::showPage
	(	uint		idx,
		KBWizardPage	*page,
		bool		push
	)
{
	/* NOTE: Arguments are redunant but we need both, and the 	*/
	/*	 public methods below derive one from the other.	*/
	if (push)
		m_history.push(m_pages.at(m_curPage)) ;

	m_stack	   .raiseWidget (page) ;
	m_bPrevious.setEnabled  (m_history.count() > 0) ;

	m_title    .setText	("<qt><b>" + page->title() + "</b></qt>") ;
	m_blurb    .setText	("<qt>" + page->blurb() + "</qt>") ;

	m_curPage  = idx 	;
	page->pageShown (push)	;

	ctrlChanged (page, 0)	;
}

/*  KBWizard								*/
/*  showPage	: Show specified page					*/
/*  page	: KBWizardPage * : The page				*/
/*  push	: bool		 : Push page onto history		*/
/*  (returns)	: void		 :					*/

void	KBWizard::showPage
	(	KBWizardPage	*page,
		bool		push
	)
{
	for (uint idx = 0 ; idx < m_pages.count() ; idx += 1)
		if (m_pages.at(idx) == page)
		{	showPage (idx, page, push) ;
			return	 ;
		}
}

/*  KBWizard								*/
/*  showPage	: Show specified page					*/
/*  idx		: uint		 : Page index				*/
/*  push	: bool		 : Push page onto history		*/
/*  (returns)	: void		 :					*/

void	KBWizard::showPage
	(	uint		idx,
		bool		push
	)
{
	if (idx < m_pages.count())
	{	showPage (idx, m_pages.at(idx), push) ;
		return	 ;
	}
}

/*  KBWizard								*/
/*  settings	: Get settings of all controls				*/
/*  dict	: QDict<QString> & : Results dictionary			*/
/*  changed	: bool		   : Return only changed values		*/
/*  (returns)	: void		   :					*/

void	KBWizard::settings
	(	QDict<QString>	&dict,
		bool		changed
	)
{
	for (uint idx = 0 ; idx < m_pages.count() ; idx += 1)
		m_pages.at(idx)->settings(dict, changed) ;
}

void	KBWizard::clickPrevious ()
{
	if (m_history.count() > 0)
		showPage (m_history.pop(), false) ;
}

void	KBWizard::clickNext ()
{
	QString	next = m_pages.at(m_curPage)->nextPage() ;
	for (uint idx = 0 ; idx < m_pages.count() ; idx += 1)
		if (m_pages.at(idx)->name() == next)
		{	showPage (idx, true) ;
			return	 ;
		}

	if (m_curPage < m_pages.count() - 1)
		showPage (m_curPage + 1, true) ;
}

void	KBWizard::clickFinish ()
{
	done	(1) ;
}

void	KBWizard::clickCancel ()
{
	done	(0) ;
}


KBWizardPage
	*KBWizard::addNewPage
	(	const QDomElement	&pageElem
	)
{
	int		pageNo	= m_pages.count	  ()	 ;
	KBWizardPage	*page	= new KBWizardPage(this, &m_stack, pageElem.attribute("name")) ;

	page->init (pageElem) ;

	m_stack.addWidget (page, pageNo) ;
	m_pages.append	  (page)	 ;

	return	page ;
}

/*  KBWizard								*/
/*  init	: Initialise wizard					*/
/*  doc		: QDomDocument & : Specification document		*/
/*  (returns)	: bool		 : Success				*/

bool	KBWizard::init
	(	const QDomDocument	&doc
	)
{
	QDomElement	docElem	= doc.documentElement() ;

	for (QDomNode node = docElem.firstChild () ;
			    !node.isNull() ;
		      node = node.nextSibling() )
	{
		QDomElement elem = node.toElement () ;
		if (elem.isNull()) continue ;

		/* Caption						*/
		/* Specifies the dialog caption text			*/
		if (elem.nodeName() == "caption")
		{
			setCaption (elem.text()) ;
			continue   ;
		}

		/* Page							*/
		/* Specifies a wizard page. The details are contined	*/
		/* within this element.					*/
		if (elem.nodeName() == "page")
		{
			addNewPage (elem) ;
			continue   ;
		}

		/* Text							*/
		/* This is the text that will be parsed or otherwise	*/
		/* inserted into the document, after substitution.	*/
		if (elem.nodeName() == "text")
		{
			m_text	= elem.text() ;
			continue  ;
		}
	}

	return	true	  ;
}

/*  KBWizard								*/
/*  init	: Initialise wizard					*/
/*  wName	: const QString & : Specification file name		*/
/*  (returns)	: bool		  : Success				*/

bool	KBWizard::init
	(	const QString	&wName
	)
{
	QFile	wFile	(wName) ;

	if (!wFile.open (IO_ReadOnly))
	{
		m_error	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot open \"%1\"")).arg(wName),
				strerror(errno),
				__ERRLOCN
			  )	;
		return	false	;
	}

	QDomDocument	doc	;
	if (!doc.setContent (&wFile))
	{
		m_error	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot parse \"%1\"")).arg(wName),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	return	init (doc)	;
}

/*  KBWizard								*/
/*  exec	: Execute wizard					*/
/*  start	: bool		: Start at first page			*/
/*  (returns)	: int		: Dialog exit code			*/

int	KBWizard::exec
	(	bool		start
	)
{
	showPage (start ? (uint)0 : m_curPage, false) ;

	int	rc  = _KBDialog::exec () ;
	save   (rc) ;
	return	rc  ;
}

int	KBWizard::execute ()
{
	return	exec	(true) ;
}


/*  KBWizard								*/
/*  findPage	: Find named page					*/
/*  name	: const QString & : Name of page			*/
/*  (returns)	: KBWizardPage *  : Page or null if not found		*/

KBWizardPage
	*KBWizard::findPage
	(	QString const	&name
	)
{
	for (uint idx = 0 ; idx < m_pages.count() ; idx += 1)
		if (m_pages.at(idx)->name() == name)
			return	m_pages.at(idx)	;

	return	0 ;
}

/*  KBWizard								*/
/*  findCtrl	: Find named control					*/
/*  page	: const QString & : Name of page			*/
/*  name	: const QString & : Name of control			*/
/*  klass	: const char *	  : Required class			*/
/*  (returns)	: KBWizardCtrl *  : Control or null if not found	*/

KBWizardCtrl
	*KBWizard::findCtrl
	(	QString const	&page,
		QString const	&name,
		const char	*klass
	)
{
	for (uint idx = 0 ; idx < m_pages.count() ; idx += 1)
		if ((page == "*") || (page == m_pages.at(idx)->name()))
		{
			KBWizardCtrl *ctrl = m_pages.at(idx)->findCtrl (name, klass) ;

			if (ctrl !=   0) return ctrl ;
			if (page != "*") return 0    ;
		}

	return	0 ;
}

/*  KBWizard								*/
/*  ctrlValue	: Get control value					*/
/*  page	: const QString & : Name of page			*/
/*  name	: const QString & : Name of control			*/
/*  (returns)	: QString	  : Value or null if no control		*/

QString	KBWizard::ctrlValue
	(	const QString 	&page,
		const QString	&name
	)
{
	KBWizardCtrl *ctrl = findCtrl (page, name) ;
	return	ctrl == 0 ? QString::null : ctrl->value() ;
}

/*  KBWizard								*/
/*  ctrlAttribute							*/
/*		: Get control attribute					*/
/*  page	: const QString & : Name of page			*/
/*  name	: const QString & : Name of control			*/
/*  attr	: const QString & : Name of attribute			*/
/*  (returns)	: QVariant	  : Value or null if no control/attr	*/

QVariant
	KBWizard::ctrlAttribute
	(	const QString 	&page,
		const QString	&name,
		const QString	&attr
	)
{
	KBWizardCtrl *ctrl = findCtrl (page, name) ;
	return	ctrl == 0 ? QVariant() : ctrl->attribute(attr) ;
}

/*  KBWizard								*/
/*  ctrlChanged	: Called when a control is changed			*/
/*  page	: KBWizardPage * : Page containing control		*/
/*  ctrl	: KBWizardCtrl * : Control in question			*/
/*  (returns)	: void		 :					*/

void	KBWizard::ctrlChanged
	(	KBWizardPage	*page,
		KBWizardCtrl	*
	)
{
//	fprintf
//	(	stderr,
//		"KBWizard::ctrlChanged: p=[%s] c=[%s] f=[%d]\n",
//		(cchar *)page->name(),
//		ctrl == 0 ? "" : (cchar *)ctrl->name(),
//		page->finish()
//	)	;

	/* If the page values are acceptable then enable the next and	*/
	/* finish buttons according to the finish value:		*/
	/*	0	- Next only					*/
	/*	1	- Finish only					*/
	/*	2	- Next and finish both allowed			*/
	if (page->ok())
	{
		uint	finish	= page->finish() ;

		m_bNext  .setEnabled ((finish == 0) || (finish == 2)) ;
		m_bFinish.setEnabled ((finish == 1) || (finish == 2)) ;
		return	;
	}

	/* Contents not acceptable, neither of the next or finish	*/
	/* buttons are enabled.						*/
	m_bNext  .setEnabled (false) ;
	m_bFinish.setEnabled (false) ;
}

/*  KBWizard								*/
/*  save	: Save any required settings				*/
/*  rc		: int		: Dialog exit code			*/
/*  (returns)	: void		:					*/

void	KBWizard::save
	(	int		rc
	)
{
	for (uint idx = 0 ; idx < m_pages.count() ; idx += 1)
		m_pages.at(idx)->save(rc) ;
}

/*  KBWizard								*/
/*  currentPageName							*/
/*		: Return name of current page				*/
/*  (returns)	: QString	: Page name				*/

QString	KBWizard::currentPageName ()
{
	return	m_pages.at(m_curPage)->name() ;
}

void	KBWizard::setCookie
	(	const QString	&key,
		void		*cookie
	)
{
	m_cookies.insert (key, cookie) ;
}

void	*KBWizard::cookie
	(	const QString	&key
	)
{
	return	m_cookies[key]	;
}

QString	KBWizard::wizForLocale
	(	const QString		&wiz
	)
{
	QString	wizXML	;

#if	__KB_KDE
	KLocale	*locale	= KGlobal::locale () ;
	wizXML	= locateFile ("appdata", "wizards/" + locale->language() + "/wiz" + wiz + ".wiz") ;
	if (!wizXML.isNull()) return wizXML ;
#endif
	wizXML	= locateFile ("appdata", "wizards/wiz" + wiz + ".wiz") ;
	return	wizXML	;
}
