/***************************************************************************
    file	         : kb_sizer.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qcursor.h>

#ifndef 	_WIN32
#include	"kb_sizer.moc"
#else
#include 	"kb_sizer.h"
#endif

#include	"kb_framer.h"
#include	"kb_ctrl.h"
#include	"kb_layout.h"
#include	"kb_display.h"
#include	"kb_options.h"
#include	"kb_report.h"
#include	"kb_writer.h"


static	const	QColor	sbColIdle	(63,	63,	63  ) ;
static	const	QColor	sbColLeader	(0,	0,	255 ) ;
static	const	QColor	sbColFollower	(0,	255,	0   ) ;
static	const	QColor	sbColActive	(255,   0,	0   ) ;

#if	QT_VERSION >= 300
static	QCursor		cMove		(Qt::SizeAllCursor  ) ;
static	QCursor		cSizeVer	(Qt::SizeVerCursor  ) ;
static	QCursor		cSizeHor	(Qt::SizeHorCursor  ) ;
static	QCursor		cSizeFDiag	(Qt::SizeFDiagCursor) ;
LIBKBASE_API QCursor 	cNoCursor	(Qt::BlankCursor    ) ;
#else
static	QCursor 	cMove	   	(    SizeAllCursor  ) ;
static	QCursor		cSizeVer   	(    SizeVerCursor  ) ;
static	QCursor 	cSizeHor   	(    SizeHorCursor  ) ;
static	QCursor 	cSizeFDiag 	(    SizeFDiagCursor) ;
QCursor			cNoCursor	(    BlankCursor    ) ;
#endif

static	int const	BlobSize 	= 6  ;

static	KBSizerInfoSet	defaultInfo	=
{
	SZF_LEFT|SZF_RIGHT|SZF_DBL,

	{	0,	SZF_X|SZF_Y|SZF_MOVE,	&cMove		},
	{	0,	SZF_X,			&cSizeHor	},
	{	0,	SZF_Y,			&cSizeVer	},
	{	0,	SZF_X|SZF_Y,		&cSizeFDiag	}
}	;


void	KBSizer::defaultInfoSet
	(	KBSizerInfoSet	&infoSet
	)
{
	infoSet	= defaultInfo	;
}

/*  KBSizerBlob	: Constructor for sizer tool marker			*/
/*  parent	: QWidget *	: Parent display widget			*/
/*  object	: KBObject *	: Object being manipulated		*/
/*  flags	: uint		: Sizing flags				*/
/*  cursor	: QCursor *	: Cursor to be used			*/
/*  (returns)	: KBSizerBlob	:					*/

KBSizerBlob::KBSizerBlob
	(	QWidget		*parent,
		KBObject	*object,
		uint		flags,
		QCursor		*cursor
	)
	:
	QWidget	(parent),
	object	(object),
	flags	(flags)
{
	setGeometry (0, 0, BlobSize, BlobSize) ;
	setPalette  (QPalette (sbColIdle)) ;

	if ((cursor != 0) && (cursor != &cNoCursor))
		setCursor (*cursor) ;

	show	    () ;
}

/*  KBSizerBlob								*/
/*  mousePressEvent							*/
/*		: Handle mouse press to start tracking			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBSizerBlob::mousePressEvent
	(	QMouseEvent *e
	)
{
	if ((e->button() == Qt::LeftButton) && (flags != 0))
		object->getSizer()->trackStart (e, this) ;
}

/*  KBSizerBlob								*/
/*  mouseMoveEvent							*/
/*		: Handle mouse movement during tracking			*/
/*  event	: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBSizerBlob::mouseMoveEvent
	(	QMouseEvent 	*event
	)
{
	object->getSizer()->trackMove (event) ;
}

/*  KBSizerBlob								*/
/*  mouseReleaseEvent							*/
/*		: Handle mouse release during tracking			*/
/*  event	: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBSizerBlob::mouseReleaseEvent
	(	QMouseEvent 	*
	)
{
	object->getSizer()->trackDone () ;
}

void	KBSizerBlob::move
	(	int		x,
		int		y
	)
{
	KBReport *report ;
	if  ((report = object->isReport()) != 0)
	{
		int	left	;
		int	top	;

		report->margins (left, top) ;

		x	+= (int)(left * pixelsPerMM()) ;
		y	+= (int)(top  * pixelsPerMM()) ;
	}

	QWidget::move (x, y)  ;
}

/*  KBSizer								*/
/*  KBSizer	: Constructor for resize/positioning control		*/
/*  object	: KBObject *	   : Object being worked upon		*/
/*  parent	: KBDisplay *	   : Parent display			*/
/*  ctrl	: QWidget *	   : Associated control			*/
/*  blobInfo	: KBSizerInfoSet * : Sizer blob options			*/
/*  (returns)	: KBSizer	   :					*/

KBSizer::KBSizer
	(	KBObject	*object,
		KBDisplay	*display,
		QWidget		*ctrl,
		KBSizerInfoSet	*blobInfo
	)
	:
	object	(object),
	display	(display),
	szCtrl	(ctrl),
	evCtrl	(ctrl),
	trap	(SZF_LEFT|SZF_RIGHT|SZF_DBL)
{
	init	(blobInfo) ;
}	

/*  KBSizer								*/
/*  KBSizer	: Constructor for resize/positioning control		*/
/*  object	: KBObject *	   : Object being worked upon		*/
/*  parent	: KBDisplay *	   : Parent display			*/
/*  ctrl	: KBControl *	   : Associated control			*/
/*  blobInfo	: KBSizerInfoSet * : Sizer blob options			*/
/*  (returns)	: KBSizer	   :					*/

KBSizer::KBSizer
	(	KBObject	*object,
		KBDisplay	*display,
		KBControl	*ctrl,
		KBSizerInfoSet	*blobInfo
	)
	:
	object	(object),
	display	(display),
	szCtrl	(ctrl-> topWidget()),
	evCtrl	(ctrl->mainWidget()),
	trap	(SZF_LEFT|SZF_RIGHT|SZF_DBL)
{
	init	(blobInfo) ;
}

/*  KBSizer								*/
/*  init	: Initialise routine					*/
/*  blobInfo	: KBSizerInfoSet * : Sizer blob options			*/
/*  (returns)	: KBSizer	   :					*/

void	KBSizer::init
	(	KBSizerInfoSet	*blobInfo
	)
{
	/* Unless the object is a block, a framer, or a component (in	*/
	/* which case they handle their own mouse events through the	*/
	/* KBDisplay class), install an event filter.			*/
	if ( (object->isFramer    () == 0) &&
	     (object->isBlock     () == 0) &&
	     (object->isComponent () == 0) )
				evCtrl->installEventFilter (this)  ;

	if (blobInfo == 0) blobInfo = &defaultInfo ;

	KBObject *otl = blobInfo->tl.proxy == 0 ? object : blobInfo->tl.proxy ;
	KBObject *otr = blobInfo->tr.proxy == 0 ? object : blobInfo->tr.proxy ;
	KBObject *obl = blobInfo->bl.proxy == 0 ? object : blobInfo->bl.proxy ;
	KBObject *obr = blobInfo->br.proxy == 0 ? object : blobInfo->br.proxy ;

	btl	= new KBSizerBlob (szCtrl, otl, blobInfo->tl.flags, blobInfo->tl.cursor) ;
	btr	= new KBSizerBlob (szCtrl, otr, blobInfo->tr.flags, blobInfo->tr.cursor) ;
	bbl	= new KBSizerBlob (szCtrl, obl, blobInfo->bl.flags, blobInfo->bl.cursor) ;
	bbr	= new KBSizerBlob (szCtrl, obr, blobInfo->br.flags, blobInfo->br.cursor) ;
	setBlobs () ;

	altered	= false ;
	tracking= false ;
	trap	= blobInfo->trap ;
}

/*  KBSizer								*/
/*  ~KBSizer	: Destructor for sizer object				*/
/*  (returns)	:		:					*/

KBSizer::~KBSizer ()
{
	if (object->getLayout() != 0)
		object->getLayout()->dropSizer (this) ;
	DELOBJ	(btl)	;
	DELOBJ	(btr)	;
	DELOBJ	(bbl)	;
	DELOBJ	(bbr)	;
}

void	KBSizer::hide ()
{
	btl->hide () ;
	btr->hide () ;
	bbl->hide () ;
	bbr->hide () ;
}

void	KBSizer::show ()
{
	btl->show () ;
	btr->show () ;
	bbl->show () ;
	bbr->show () ;
}

/*  KBSizer								*/
/*  trackStart	: Start tracking movement for resize			*/
/*  e		: QMouseEvent *	: Originating mouse event		*/
/*  blob	: KBSizerBlob *	: Blob originating request		*/
/*  (returns)	: void		:					*/

void	KBSizer::trackStart
	(	QMouseEvent	*e,
		KBSizerBlob	*blob
	)
{
	/* Ingore if already tracking, left button not pressed, or the	*/
	/* blob is marked out.						*/
	if (tracking)
		return	;

	if (e->button() != Qt::LeftButton)
		return	;

	if ((blob != 0) && (blob->getFlags() == 0))
		return	;

	if ((track = blob) == 0)
		track = btl ;

	tracking = true	  	  ;
	altered	 = false	  ;
	startMX	 = e->globalX ()  ;
	startMY	 = e->globalY ()  ;

	minSize	 = object->getMinSize() ;
	limit	 = object->getLayout ()->addSizer (this, (e->state() & Qt::ShiftButton) != 0) ;

//	fprintf
//	(	stderr,
//		"KBSizer::trackStart: limit (%d,%d,%d,%d)\n",
//		limit.left,
//		limit.right,
//		limit.up,
//		limit.down
//	)	;

	track->grabMouse () ;
}

/*  KBSizer								*/
/*  trackMove	: Movement tracker					*/
/*  e		: QMouseEvent *	: Associated mouse event		*/
/*  (returns)	: void		:					*/

void	KBSizer::trackMove
	(	QMouseEvent	*e
	)
{
	/* If tracking then move pass the movement to the active sizer	*/
	/* with the distance the most has moved since starting ...	*/
	if (tracking)
	{
		int	dx	= e->globalX() - startMX ;
		int	dy	= e->globalY() - startMY ;
		bool	move	= (track->getFlags() & SZF_MOVE) != 0 ;

		if (move)
		{
			if (dx < limit.left ) dx = limit.left  ;
			if (dx > limit.right) dx = limit.right ;
			if (dy < limit.up   ) dy = limit.up    ;
			if (dy > limit.down ) dy = limit.down  ;
		}

		if ((track->getFlags() & SZF_X) == 0) dx = 0 ;
		if ((track->getFlags() & SZF_Y) == 0) dy = 0 ;

		object->getLayout()->trackMove (dx, dy, move) ;
	}
}

/*  KBSizer								*/
/*  trackDone	: Signal end of tracking				*/
/*  (returns)	: void		:					*/

void	KBSizer::trackDone ()
{
	/* Pass this up to the document root node, which is		*/
	/* responsible for passing information to all selected		*/
	/* controls.							*/
	if (tracking)
	{
		tracking = false ;
		track ->releaseMouse () ;
		object->getLayout()->releaseSizer () ;
	}
}

/*  KBSizer								*/
/*  doDesignPopup: Show design popup unless disabled			*/
/*  e		 : QMouseEvent * : Invoking event			*/
/*  (returns)	 : bool		 : Popup shown				*/

bool	KBSizer::doDesignPopup
	(	QMouseEvent	*e
	)
{
	if ((trap & SZF_RIGHT) != 0)
	{
		object->designPopup (e, 0) ;
		return	true	;
	}

	return	false	;
}

/*  KBSizer								*/
/*  eventFilter	: Event filter for associated widget			*/
/*  obj		: QObject *	: Actually pointer at the widget	*/
/*  e		: QEvent *	: Event to process			*/
/*  (returns)	: bool		: False to continue processing		*/

bool	KBSizer::eventFilter
	(	QObject	*,
		QEvent	*e
	)
{
	if (e->type() == QEvent::MouseButtonPress)
	{
		/* A left-button mouse press starts either tracking to	*/
		/* move a control, or marking to sweep an aread in a	*/
		/* block. We don't don'need to trap the following	*/
		/* movements and left-button release, as the tracking	*/
		/* code grabs the mouse.				*/
		if (((QMouseEvent *)e)->button() == QMouseEvent::LeftButton)
		{
			if ((trap & SZF_LEFT) != 0)
			{
				if (object->isBlock())
					display->markStart ((QMouseEvent *)e) ;
				else	trackStart ((QMouseEvent *)e, 0) ;

				return	true	;
			}
		}

		/* A right-button brings up a (design) context menu	*/
		/* for the object associated with the control.		*/
		if (((QMouseEvent *)e)->button() == QMouseEvent::RightButton)
		{
			if (doDesignPopup ((QMouseEvent *)e)) return true ;
		}

		return	false	;
	}

	/* Double-click brings up the property dialog for the object	*/
	/* associated with the control.					*/
	if (e->type() == QEvent::MouseButtonDblClick)
	{
		if ((trap & SZF_DBL) != 0)
		{
			if (((QMouseEvent *)e)->button() == QMouseEvent::LeftButton)
			{	object->propertyDlg () ;
				return	true	;
			}
		}

		return	false	;
	}

#if	QT_VERSION >= 300
	/* QT3 provides a more general context-menu event ...		*/
	if (e->type() == QEvent::ContextMenu)
	{
		if ((trap & SZF_RIGHT) != 0)
		{
			object->designPopup ((QMouseEvent *)e, 0) ;
			return	true	;
		}

		return	false	;
	}
#endif

	/* Anything else is handled by the widget itself ...		*/
	return	false ;
}

/*  KBSizer								*/
/*  getPosition	: Get control position					*/
/*  (returns)	: QRect		: Position				*/

QRect	KBSizer::getPosition ()
{
	return	object->geometry () ;
}

/*  KBSizer								*/
/*  setBlobs	: Position sizer blobs					*/
/*  (returns)	: void		:					*/

void	KBSizer::setBlobs ()
{
	QRect	rect	= getPosition() ;
	int	x	;
	int	y	;
	int	w	;
	int	h	;

	/* If there is no parent display (ie., the control is not	*/
	/* itself embedded in a display) then this must be the top	*/
	/* level block ....						*/
	if (object->getParent() == 0)
	{
		display->cvtCtrlToView (rect) ;
		x = rect.left   () ;
		y = rect.top    () ;
		w = rect.width  () ;
		h = rect.height () ;
	}
	else
	{	x = 0 ;
		y = 0 ;
		w = rect.width  () ;
		h = rect.height () ;
	}

	/* This routine is typically called after the control		*/
	/* properties (which include position and size) have been	*/
	/* manually edited, and the control resized and/or repositioned	*/
	btl  ->move (x, y) ;
	btr  ->move (x + w - BlobSize, 0) ;
	bbl  ->move (x, y + h - BlobSize) ;
	bbr  ->move (x + w - BlobSize, y + h - BlobSize) ;
}

/*  KBSizer								*/
/*  setGeometry	: Set position and size of the control			*/
/*  (returns)	: void		:					*/

void	KBSizer::setGeometry
	(	QRect	pRect
	)
{
	szCtrl->resize (pRect.size()) ;
	if (display != 0) display->moveChild (szCtrl, pRect.x(), pRect.y()) ;
	setBlobs () ;
}
		
/*  KBSizer								*/
/*  doResize	: Handle control resize or move				*/
/*  dx		: int		: X-change				*/
/*  dy		: int		: Y-change				*/
/*  move	: bool		: Action is move rather than resize	*/
/*  (returns)	: void		:					*/

void	KBSizer::doResize
	(	int	dx,
		int	dy,
		bool	move
	)
{
	if ((dx != 0) || (dy != 0))
		altered = true ;

	if (move)
	{	object->move (startCX + dx, startCY + dy) ;
		return		 ;
	}

	KBFramer *framer ;
	if ((framer = object->isFooter()) != 0) dy = -dy ;
			

	/* If changing size then check that we are not going below the	*/
	/* minimum size.						*/
	int	nw = startW + dx ;
	int	nh = startH + dy ;

	if (nw < minSize.width ()) nw = minSize.width () ;
	if (nh < minSize.height()) nh = minSize.height() ;

	object->resize (nw, nh)  ;
//	setBlobs       ()	 ;
}

/*  KBSizer								*/
/*  setState	: Set sizer state					*/
/*  state	: KBSizerState	: New state				*/
/*  (returns)	: void		:					*/

void	KBSizer::setState
	(	KBSizerState	state
	)
{
	QColor	color	= state == sbActive   ? sbColActive   :
			  state == sbLeader   ? sbColLeader   :
			  state == sbFollower ? sbColFollower : sbColIdle ;

	btl->setPalette (color) ;
	btr->setPalette (color) ;
	bbl->setPalette (color) ;
	bbr->setPalette (color) ;

	/* Note the initial control position and size so that we can	*/
	/* calculate new positions and sizes if this is the start of a	*/
	/* move or resize.						*/
	QRect	area	= getPosition () ;
	startCX	= area.left  ()	;
	startCY	= area.top   ()	;
	startW  = area.width () ;
	startH	= area.height() ;
}

/*  KBSizer								*/
/*  accept	: Save values back to associated object			*/
/*  doSnap	: bool		: Forciply snap to grid			*/
/*  (returns)	: void		:					*/

void	KBSizer::accept
	(	bool	doSnap
	)
{
	QRect	  cRect	= getPosition () ;

	extern	void	snapRect (int &, int &, int &, int &) ;

	int	x	= cRect.x     () ;
	int	y	= cRect.y     () ;
	int	w	= cRect.width () ;
	int	h	= cRect.height() ;

	if (doSnap || (altered && KBOptions::snappingOn()))
		snapRect (x, y, w, h) ;


	tracking	= false	;
	object->setGeometry (QRect (x, y, w, h)) ;
	object->setChanged  ()  ;

	setBlobs () ;
	altered	 = false ;
}

/*  KBSizer								*/
/*  snapToGrid	: Snap position to grid					*/
/*  (returns)	: bool		:					*/

void	KBSizer::snapToGrid ()
{
	accept	(true)	;
}
