/***************************************************************************
    file	         : kb_reportblock.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifndef 	_WIN32
#include 	"kb_reportblock.moc"
#else
#include 	"kb_reportblock.h"
#endif

#include	"kb_framer.h"
#include	"kb_report.h"
#include	"kb_writer.h"
#include	"kb_layout.h"
#include	"kb_qrybase.h"
#include	"kb_display.h"
#include	"kb_nodereg.h"
#include	"kb_popupmenu.h"
#include	"kb_blockevents.h"
#include	"kb_attrdict.h"


/*  KBYObject								*/
/*  KBYObject	: Constructor for Y-order sorting object		*/
/*  object	: KBObject *	: Underlying object			*/
/*  (returns)	: KBYObject	:					*/

KBYObject::KBYObject
	(	KBObject *object
	)
	:
	object	(object)
{
	needs	= 0 ;
	y	= object->geometry().y() ;
}


/*  ------------------------------------------------------------------  */

/*  KBYSort								*/
/*  compareItems: Compare function for Y-order sorted			*/
/*  o1		: QCollection::Item : First KBYObject			*/
/*  o2		: QCollection::Item : Second KBYObject			*/
/*  (returns)	: int		    : Ordering value			*/

int	KBYSort::compareItems
	(	QCollection::Item	o1,
		QCollection::Item	o2
	)
{
	return	((KBYObject *)o1)->y - ((KBYObject *)o2)->y ;
}


/*  ------------------------------------------------------------------  */

/*  KBReportBlock							*/
/*  KBReportBlock: Constructor for extant mreport block element		*/
/*  parent	 : KBNode *	: Parent element			*/
/*  aList	 : const QDict<QString> &				*/
/*				: List of attributes			*/
/*  element	 : cchar *	: Element name				*/
/*		 : bool *	:					*/
/*  (returns)	 : KBReportBlock:					*/

KBReportBlock::KBReportBlock
	(	KBNode			*parent,
		const QDict<QString>	&aList,
		cchar			*element,
		bool			*
	)
	:
	KBBlock	(parent, aList,		element),
	pThrow	(this,	 "pthrow",	aList)
{
	ySort.setAutoDelete (true) ;
}

/*  KBReportBlock							*/
/*  KBReportBlock: Constructor for new report block element		*/
/*  parent	 : KBObject *	 : Parent element			*/
/*  rect	 : const QRect & : Size and position			*/
/*  blkType	 : BlkType	 : Initial block type			*/
/*  ok		 : bool &	 : Success				*/
/*  element	 : cchar *	 : Element name				*/
/*  (returns)	 : KBReportBlock :					*/

KBReportBlock::KBReportBlock
	(	KBObject	*parent,
		const QRect	&rect,
		BlkType		_blkType,
		bool		&ok,
		cchar		*element
	)
	:
	KBBlock	(parent, rect,     _blkType, ok, element),
	pThrow	(this,   "pthrow", false)
{
	if (ok) addFramers  () ;
	ySort.setAutoDelete (true) ;
}

/*  KBReportBlock							*/
/*  KBReportBlock: Constructor for extant report block element		*/
/*  _parent	: KBNode *	  : Parent element			*/
/*  _block	: KBReportBlock * : Extant block			*/
/*  (returns)	: KBReportBlock	  :					*/

KBReportBlock::KBReportBlock
	(	KBNode		*_parent,
		KBReportBlock	*_block
	)
	:
	KBBlock	(_parent, 	    _block),
	pThrow	(this,    "pthrow", _block)
{
	ySort.setAutoDelete (true) ;
}

/*  KBReportBlock							*/
/*  ~KBReportBlock: Destructor for form block element			*/
/*  (returns)	: void		:					*/

KBReportBlock::~KBReportBlock ()
{
}

/*  KBReportBlock							*/
/*  addFramers	: Add header and footer					*/
/*  (returns)	: void		:					*/

void	KBReportBlock::addFramers ()
{
	QRect	r = geometry () ;

	if (r.height() > 120)
	{
		KBAttrDict hDict ;
		KBAttrDict fDict ;

		hDict.addValue ("x",	0		) ;
		hDict.addValue ("y",	0		) ;
		hDict.addValue ("w",	r.width ()	) ;
		hDict.addValue ("h",	40		) ;
		hDict.addValue ("name", "header"	) ;

		fDict.addValue ("x",	0		) ;
		fDict.addValue ("y",	r.height() - 40	) ;
		fDict.addValue ("w",	r.width ()	) ;
		fDict.addValue ("h",	40		) ;
		fDict.addValue ("name", "footer"	) ;

		header	= new KBHeader (this, hDict, "KBBlockHeader") ;
		footer	= new KBFooter (this, fDict, "KBBlockFooter") ;
	}
}

/*  KBReportBlock							*/
/*  ySortObjects: Sort embedded objects into Y order			*/
/*  (returns)	: void		:					*/

void	KBReportBlock::ySortObjects ()
{
	ySort.clear () ;

	/* We need to group controls into sets, divided up by nested	*/
	/* blocks, so that we can output in the appropriate order. To	*/
	/* start, order all objects in increasing Y-order.		*/
	CITER
	(	Object,
		o,
		if ((o->isHidden() == 0) && (o->isFramer()) == 0)
			ySort.inSort (new KBYObject (o))
	)

	/* We now need to find out how much space (depth) is used by	*/
	/* each set, so that we can generate page throws. We find the	*/
	/* depth for each set, and note it against the first control	*/
	/* in that set.							*/
	KBYObject *unknown	= ySort.at(0) ;
	uint	  deltaP	= header == 0 ? 0 : header->height() ;

	for (uint idx = 0 ; idx < ySort.count() ; idx += 1)
	{
		KBYObject	*yObj	= ySort.at(idx) ;
		KBReportBlock	*repBlk	= yObj->object->isReportBlock() ;

		if (repBlk != 0)
		{
			unknown->needs = yObj->y - deltaP  ;
			deltaP	       = yObj->y + repBlk->geometry().height() ;
			unknown	       = idx + 1 >= ySort.count() ?
							 0 : ySort.at(idx + 1) ;
		}
	}

	if (unknown != 0)
	{
		int	h = geometry().height() ;

		if (footer == 0)
			unknown->needs = h - deltaP ;
		else	unknown->needs = h - deltaP - footer->height() ;
	}

//	LITER (KBYObject, ySort, o)
//		fprintf (stderr, "[%d][%d]\n", o->y, o->needs) ;
}

/*  KBReportBlock							*/
/*  showData	 : Display values from query in fields			*/
/*  (returns)	 : bool		: Success				*/

bool	KBReportBlock::showData ()
{
	uint	 qRows	 = m_query->getNumRows (qryLvl) ;
	KBWriter *writer = getReport ()->getWriter () ;


	/* Sort the block objects into Y-order, this is the order that	*/
	/* they will be output so that we move down the page and handle	*/
	/* nested containers correctly.					*/
	ySortObjects () ;


	/* Unlike a form, where all control are prepared once when the	*/
	/* form is executed, in a report we need to prepare all		*/
	/* controls at and below a block each time the block data is	*/
	/* shows. This will, for instance, reset all summary controls.	*/
	prepare	() ;


	/* The header is only output here if there are no data rows,	*/
	/* otherwise it gets output inside the loop (see comment there)	*/
	/* Its not obvious that we get called with no rows, but lets be	*/
	/* safe.							*/
	if (qRows == 0)
		if (header != 0)
		{
			header->writeData () ;
			writer->setOffset (false, QPoint (0, header->height())) ;
		}


	/* If there is a footer then reserver space for it. Page throws	*/
	/* will occur when the footer reservation would otherwise be	*/
	/* encroached upon.						*/
	if (footer != 0)
		writer->reserve (footer->height()) ;


	for (curQRow = 0, curDRow = 0 ; curQRow < qRows ; curQRow += 1)
	{
		int	extra	= 0 ;
		uint	delta	= header == 0 ? 0 : header->height() ;
		bool	evRc	;

		/* Load the current row from the query. This will also	*/
		/* load any header and/or footer fields.		*/
		m_query->setCurrentRow (qryLvl, curQRow) ;
		m_query->loadItems     (qryLvl, curQRow) ;

		KBValue	arg ((int)curQRow) ;
		if (!eventHook (events->onCurrent, 1, &arg, evRc)) return false ;


		/* If this is the first row and there is a header then	*/
		/* generate it now. It is deferred to this point so	*/
		/* that the first query row has been loaded, so that	*/
		/* data values can appear in the header (importantly 	*/
		/* including those with "=....." display expressions).	*/
		if ((curQRow == 0) && (header != 0))
		{
			header->writeData () ;
			writer->setOffset (false, QPoint (0, header->height())) ;
		}

		/* Now loop through all the objects to be displayed,	*/
		/* outputing their values ....				*/
		LITER
		(	KBYObject,
			ySort,
			yObj,

			KBObject	*obj	= yObj->object ;
			KBReportBlock	*repBlk	= obj ->isReportBlock() ;

			/* At each nested block, advance down by the	*/
			/* appropriate amount and recurse into the	*/
			/* block.					*/
			if (repBlk != 0)
			{
				writer->setOffset
				(	false,
					QPoint (0, yObj->y - delta)
				)	;

				if (!repBlk->requery ())
				{	setError (repBlk->lastError()) ;
					return   false ;
				}

				if (!repBlk->showData ())
				{	setError (repBlk->lastError()) ;
					return   false ;
				}

				delta	 = yObj->y + repBlk->geometry().height() ;
				continue ;
			}

			/* See if there is enough space for the next	*/
			/* set of controls. If not then finish this	*/
			/* page and page throw. 			*/
			if (!writer->spaceAvailable (yObj->needs))
			{

				/* Check to prevent looping if we run	*/
				/* out of space (ie., we get squeezed	*/
				/* out by headers and footers)		*/
				if (writer->pageIsEmpty())
				{
					setError
					(	KBError::Error,
						TR("Insufficient space on page"),
						QString(TR("Object %1: needs %2"))
							.arg(yObj->object->getName())
							.arg(yObj->needs),
						__ERRLOCN
					)	;
					return	false	;
				}

				finishPage	(true)  ;
				writer->newPage ()	;

				curDRow	= curQRow 	;

				/* Start a new page since we know that	*/
				/* there must be at least one control	*/
				/* to output.				*/
				startPage ()		;
			}

			/* OK, enough space, so output. As we do so,	*/
			/* keep track of the maximum extra extension	*/
			/* resulting from expanding controls (like	*/
			/* memos and rich text).			*/
			int e ;
			if (!obj->write (writer, QPoint(0, -delta), curDRow == curQRow, e))
				return	false	;

			e += obj->geometry().height() ;
			if (e > extra) extra = e ;
		)

		/* Here we finish with one row of data, so advance down	*/
		/* the page for the space occupied by the last set of	*/
		/* controls. If there has been extra space for expended	*/
		/* controls beyond the nominal advances then allow for	*/
		/* that.						*/
		int h = geometry().height() - delta ;
		if (footer != 0) h -= footer->height() ;

		if (extra > h)
		{
			fprintf
			(	stderr,
				"KBReportBlock::showData: extra %d -> %d\n",
				h,
				extra
			)	;

			h = extra ;
		}
			
		writer->setOffset (false, QPoint(0, h)) ;

		/* If set to page-per-row then finish this page. We	*/
		/* also start a new page if thete are more rows to data	*/
		/* to come.						*/
		if (pThrow.getIntValue() == 2)
		{
			finishPage 	(true) ;
			writer->newPage	() ;
			curDRow	= curQRow  ;

			if (curQRow < qRows - 1) startPage() ;
		}
	}

	/* If set to page-per-group then finish the page now and start	*/
	/* a new one ....						*/
	if (pThrow.getIntValue() == 1)
	{
		finishPage 	(true) ;
		writer->newPage	() ;
		return	true	;
	}

	/* ... otherwise if there is a footer then output that. Note	*/
	/* that if this is the topmost block then set the offset to	*/
	/* get the footer at the bottom on the page.			*/
	if (footer != 0)
	{
		if (getParent() == 0)
			writer->setOffset (true, QPoint(0, writer->reserve (0))) ;

		footer->writeData (false) ;
		writer->setOffset (false, QPoint(0,footer->height())) ;

		/* In case we are not the topmost block, "unreserve"	*/
		/* the footer space.					*/
		writer->reserve	  (- footer->height()) ;
	}

	return	true ;
}

/*  KBReportBlock							*/
/*  startPage	: Start a new output page				*/
/*  (returns)	: void		:					*/

void	KBReportBlock::startPage ()
{
	KBWriter *writer = getReport()->getWriter() ;

	if (getBlock() != 0)
		getBlock()->isReportBlock()->startPage () ;


	if (header != 0)
	{	header->writeData () ;
		writer->setOffset (false, QPoint (0, header->height())) ;
	}

	if (footer != 0) writer->reserve (footer->height()) ;
}

/*  KBReportBlock							*/
/*  finishPage	: Finish a page of output				*/
/*  goDown	: bool		: Move down to reservation point	*/
/*  (returns)	: void		:					*/

void	KBReportBlock::finishPage
	(	bool	goDown
	)
{
	KBWriter *writer = getReport()->getWriter() ;

	if (goDown)
		writer->setOffset (true, QPoint(0, writer->reserve (0))) ;

	if (footer != 0)
	{
		footer->writeData (true) ;
		writer->setOffset (false, QPoint(0, footer->height())) ;
	}

	if (getBlock() != 0)
		getBlock()->isReportBlock()->finishPage(false) ;
}

/*  KBReportBlock							*/
/*  getBlockVal	: Get block value pointer				*/
/*  (returns)	: KBValue *	: Value pointer or null at top level	*/

KBValue	*KBReportBlock::getBlockVal ()
{
	KBBlock	*parent	= getBlock () ;
	if ((parent == 0) || (parent->getBlkType() == BTNull))
		return	0 ;

	static	KBValue	mValue	;
//	mValue	= parent->getValue (parent->getCurQRow()) ;
	mValue	= parent->getRowValue (this, parent->getCurQRow()) ;
	return	&mValue	;
}

/*  KBReportBlock							*/
/*  getValue	: Get value						*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBReportBlock::getValue
	(	uint	qrow
	)
{
	return	m_query->getField (qryLvl, qrow, qryIdx.qryIdx()) ;
}

/*  KBBlock								*/
/*  setValue	: Set value display row					*/
/*  qrow	: uint		 : Query row				*/
/*  value	: const KBValue &: Value to set				*/
/*  (returns)	: bool		 : Success				*/

bool	KBReportBlock::setValue
	(	uint		,
		const KBValue	&
	)
{
//	if (mValues[0].value.getRawText() != value.getRawText())
//	{	mValues[0].value = value ;
//		mValues[0].delta = true	 ;
//	}

	return	true	;
}

/*  KBReportBlock							*/
/*  newNode	: Add a new node					*/
/*  id		: int		: Actually NodeSpec pointer for node	*/
/*  (returns)	: KBNode *	: The new node				*/

KBNode	*KBReportBlock::newNode
	(	int	id
	)
{
#if	__KB_RUNTIME

	return	0 ;

#else

	NodeSpec	*spec	= (NodeSpec *)id ;
	QRect		cRect	= newCtrlRect () ;
	KBAttrDict	aDict	(0) ;
	bool		ok	;
	KBNode		*node	;

	aDict.addValue ("x", 	    cRect.x     ()) ;
	aDict.addValue ("y", 	    cRect.y     ()) ;
	aDict.addValue ("w", 	    cRect.width ()) ;
	aDict.addValue ("h", 	    cRect.height()) ;
	aDict.addValue ("align",    Qt::AlignLeft ) ;

	node	= spec->nodeFunc (this, aDict, &ok) ;

	if (!ok) return 0 ;

	if (node->isObject())
	{
		node->isObject()->buildDisplay (blkDisp) ;
		buildCtrls (0, 0, 0) ;
	}
	node->showAs (KB::ShowAsDesign) ;
	getLayout()->setChanged() ;

	if (node->isObject())
		getLayout()->addSizer (node->isObject()->getSizer(), true) ;

	return	node	;
#endif
}

/*  KBReportBlock							*/
/*  replicate	: Replicate the block					*/
/*  _parent	: KBNode *	: Parent node				*/
/*  (returns)	: KBNode *	: Replicant				*/

KBNode	*KBReportBlock::replicate
	(	KBNode	*_parent
	)
{
	return	replicateBelow (new KBReportBlock (_parent, this)) ;
}

/*  KBReportBlock							*/
/*  docPropDlg	: Show report property dialog				*/
/*  (returns)	: void		:					*/

void	KBReportBlock::docPropDlg ()
{
	getReport()->propertyDlg () ;
}

/*  KBReportBlock							*/
/*  newTableBlock: Add a new table block				*/
/*  (returns)	 : void		:					*/

void	KBReportBlock::newTableBlock ()
{
#if	! __KB_RUNTIME
	QRect	rect	= newCtrlRect () ;

	rect.setX	(0) ;
	rect.setWidth	(geometry().width()) ;

	if (checkOverlap (rect.x(), rect.y(), rect.width(), rect.height()))
		return	;

	bool	ok 	;
	KBBlock	*b 	= new KBReportBlock (this, rect, KBBlock::BTTable, ok) ;

	if (!ok)
	{
		delete	b ;
		return	;
	}

	b->buildDisplay (blkDisp) ;
	b->showAs   (KB::ShowAsDesign) ;
	b->getContainer()->show() ;
	getLayout()->setChanged() ;
#endif
}

/*  KBReportBlock								*/
/*  newSQLBlock	: Add a new SQL block					*/
/*  (returns)	: void		:					*/

void	KBReportBlock::newSQLBlock ()
{
#if	! __KB_RUNTIME
	QRect	rect	= newCtrlRect () ;

	rect.setX	(0) ;
	rect.setWidth	(geometry().width()) ;

	if (checkOverlap (rect.x(), rect.y(), rect.width(), rect.height()))
		return	;

	bool	ok  ;
	KBBlock	*b = new KBReportBlock (this, newCtrlRect(), KBBlock::BTSQL, ok) ;

	if (!ok)
	{
		delete	b ;
		return	;
	}

	b->buildDisplay (blkDisp) ;
	b->showAs   (KB::ShowAsDesign) ;
	b->getContainer()->show() ;
	getLayout()->setChanged() ;
#endif
}

/*  KBReportBlock							*/
/*  newQueryBlock: Add a new query block				*/
/*  (returns)	 : void		:					*/

void	KBReportBlock::newQueryBlock ()
{
#if	! __KB_RUNTIME
	QRect	rect	= newCtrlRect () ;

	rect.setX	(0) ;
	rect.setWidth	(geometry().width()) ;

	if (checkOverlap (rect.x(), rect.y(), rect.width(), rect.height()))
		return	;

	bool	ok  ;
	KBBlock	*b = new KBReportBlock (this, rect, KBBlock::BTQuery, ok) ;

	if (!ok)
	{
		delete	b ;
		return	;
	}

	b->buildDisplay (blkDisp) ;
	b->showAs   (KB::ShowAsDesign) ;
	b->getContainer()->show() ;
	getLayout()->setChanged() ;
#endif
}

/*  KBReportBlock							*/
/*  addHeader	: Add a header						*/
/*  (returns)	: void		:					*/

void	KBReportBlock::addHeader ()
{
#if	! __KB_RUNTIME
	KBAttrDict ad	;
	ad.addValue ("h", 20) ;

	bool	   ok 	;
	KBHeader   *h 	= new KBHeader (this, ad, "KBBlockHeader", &ok) ;

	if (!ok)
	{
		delete	h ;
		return	  ;
	}

	h->buildDisplay (blkDisp) ;
	h->showAs   (KB::ShowAsDesign) ;
	h->getContainer()->show() ;
	getLayout()->setChanged() ;
#endif
}

/*  KBReportBlock							*/
/*  addFooter	: Add a footer						*/
/*  (returns)	: void		:					*/

void	KBReportBlock::addFooter ()
{
#if	! __KB_RUNTIME
	KBAttrDict ad	;
	ad.addValue ("h", 20) ;

	bool	   ok 	;
	KBFooter   *f 	= new KBFooter (this, ad, "KBBlockFooter", &ok) ;

	if (!ok)
	{
		delete	f ;
		return	  ;
	}

	f->buildDisplay (blkDisp) ;
	f->showAs   (KB::ShowAsDesign) ;
	f->getContainer()->show() ;
	getLayout()->setChanged() ;
#endif
}


/*  KBReportBlock							*/
/*  makeNewPopup: Make a new-thingy popup menu for a report		*/
/*  cancel	: bool		: Show cancel item			*/
/*  _rect	: QRect		: New object area			*/
/*  (returns)	: void		:					*/

KBPopupMenu
	*KBReportBlock::makeNewPopup
	(	bool	cancel,
		QRect	_rect
	)
{
#if	__KB_RUNTIME

	return	0 ;

#else

	extern void 	makeReportMenu (QPopupMenu *, QObject *, uint) ;

	KBPopupMenu	*newPopup = new KBPopupMenu (&bState) ;
	KBPopupMenu	*blkPopup = new KBPopupMenu (&bState) ;

	bool		header	  = false ;
	bool		footer	  = false ;

	CITER
	(	Header,
		h,
		header = true ; break ;
	)
	CITER
	(	Footer,
		f,
		footer = true ; break ;
	)

	blkPopup->insertEntry (false,	TR("Table Block"), this, SLOT(newTableBlock())) ;
	blkPopup->insertEntry (false,	TR("SQL Block"  ), this, SLOT(newSQLBlock  ())) ;
	blkPopup->insertEntry (false,	TR("Query Block"), this, SLOT(newQueryBlock())) ;

	if (cancel)
		newPopup->insertItem (TR("Cancel")) ;

	newPopup->insertItem      (TR("New B&lock"),   blkPopup) ;
	newPopup->insertSeparator () ;
	newPopup->insertEntry (header,  TR("Add Header" ), this, SLOT(addHeader    ())) ;
	newPopup->insertEntry (footer,  TR("Add Footer" ), this, SLOT(addFooter    ())) ;
	newPopup->insertSeparator () ;
	makeReportMenu (newPopup, this, KF_BLOCK|KF_STATIC|KF_DATA) ;

	newRect	= _rect		;
	return	newPopup	;
#endif
}

/*  KBReportBlock							*/
/*  move	: Move the block in its parent				*/
/*  x		: int		: New X-position			*/
/*  y		: int		: New Y-position			*/
/*  (returns)	: void		:					*/

void	KBReportBlock::move
	(	int	,
		int	y
	)
{
	KBBlock::move (0, y) ;
}

/*  KBReportBlock							*/
/*  resize	: Resize the block in its parent			*/
/*  w		: int		: New width				*/
/*  h		: int		: New height				*/
/*  (returns)	: bool		: Size changed				*/

bool	KBReportBlock::resize
	(	int	w,
		int	h
	)
{
	if (getParent() != 0)
		w = getParent()->isObject()->geometry().width() ;

	return	KBBlock::resize (w, h) ;
}

/*  KBReportBlock							*/
/*  rowsInBlock	: Get number of rows					*/
/*  (returns)	: uint		: Number of rows			*/

uint	KBReportBlock::rowsInBlock ()
{
	/* This is only used in design mode, when the value is aways	*/
	/* unity.							*/
	return	1	;
}

NEWNODENAMED	(ReportBlock, (cchar *)0, KF_REPORT, ReportBlock   )
NEWNODENAMED	(ReportBlock, (cchar *)0, KF_REPORT, ReportSubBlock)
