/***************************************************************************
    file	         : kb_propdlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qheader.h>
#include	<qcache.h>
#include	<qregexp.h>
#include	<qfile.h>
#include	<qtextstream.h>


#ifndef 	_WIN32
#include	"kb_propdlg.moc"
#else 
#include 	"kb_propdlg.h"
#endif

#include	"kb_object.h"
#include	"kb_qrybase.h"
#include	"kb_font.h"
#include	"kb_script.h"
#include	"kb_docroot.h"
#include	"kb_slot.h"
#include	"kb_slotdlg.h"
#include	"kb_options.h"

#include	"tk_colordialog.h"
#include	"tk_fontdialog.h"
#include	"tk_messagebox.h"
#include	"tk_config.h"

#include	"tk_helpproxy.h"



#if	__KB_EMBEDDED
#define	LWIDTH		(100)
#else
#define	LWIDTH		(160)
#endif



class	LIBKBASE_API	KBAttrListViewItem : public QListViewItem
{
	QString		name		;
	QString		order		;

public	:

	KBAttrListViewItem
		(	QListViewItem	*qlv,
			QListViewItem	*after,
			QString		legend,
			QString		value,
			QString		name,
			uint		_order
		)
		:
		QListViewItem	(qlv, after, legend, value),
		name		(name)
	{
		order.sprintf	("%08x", _order) ;
	}

	inline	QString	getName ()
	{ 	return	name   ;
	}

	virtual	QString	key
		(	int	,
			bool
		)
		const
	{
		return order  ;
	}
			
}	;


static	IntChoice	choiceFMode[] =
{
	{	KBObject::FMFixed,	"fixed"		},
	{	KBObject::FMFloat,	"float"		},
	{	KBObject::FMStretch,	"stretch"	},
	{	-1,			0		}
}	;

static	IntChoice	choiceScaling[] =
{
	{	0,		"No scaling"				},
	{	1,		"Exact fit"				},
	{	2,		"Fit preserving aspect ratio"		},
	{	3,		"Fill preserving aspect ratio"		},
	{	-1,		0					}
}	;


/*  ------------------------------------------------------------------  */

/*  Generally, help tags are constructed from the attributes owner's	*/
/*  class name and the attribute name. However, in a few cases there	*/
/*  is one help item for a number of attributes. The code below is a	*/
/*  bit of a hack to handle these cases ....				*/

static	HelpMap	helpMap	[] =
{
	{	"x",		"position"	},
	{	"y",		"position"	},
	{	"w",		"position"	},
	{	"h",		"position"	},
	{	"xmode",	"floatmode"	},
	{	"ymode",	"floatmode"	},
	{	0,		0		}

}	;


/*  ------------------------------------------------------------------  */

/*  KBAttrSlots								*/
/*  -----------								*/
/*  Derived attribute item used to handle slots, so the list of slot	*/
/*  names is displayed.							*/

class	LIBKBASE_API KBAttrSlots : public KBAttrItem
{
	const QList<KBSlot>	&m_slotList ;
public:

	KBAttrSlots (KBAttr *, const QList<KBSlot> &) ;
	virtual ~KBAttrSlots() ;

	virtual	QString	displayValue();
};

/*  KBAttrSlots								*/
/*  KBAttrSlots	: Constructor for slots attribute item			*/
/*  attr	: KBAttr *	: Actual attribute			*/
/*  slotList	: QList<KBSlot> & : Slot list				*/
/*  (returns)	: KBAttrSlots	  :					*/

KBAttrSlots::KBAttrSlots
	(	KBAttr			*attr,
		const QList<KBSlot>	&slotList
	)
	:
	KBAttrItem (attr),
	m_slotList (slotList)
{
}

/*  KBAttrSlots								*/
/*  ~KBAttrSlots: Destructor for slots attribute item			*/
/*  (returns)	:		:					*/

KBAttrSlots::~KBAttrSlots ()
{
}

/*  KBAttrSlots								*/
/*  displayValue: Generate dialog display value				*/
/*  (returns)	: QString	: Text to display			*/

QString	KBAttrSlots::displayValue()
{
	QString	text	;
	cchar	*sep	= "" ;

	LITER
	(	KBSlot,
		m_slotList,
		slot,

		text	+= sep	;
		text	+= slot->name () ;
		sep	 = ", "	;
	)

	return	text	;
}

/*  ------------------------------------------------------------------  */

/*  KBAttrItem								*/
/*  KBAttrItem	: Constructor for property dialog attribute item	*/
/*  attr	: KBAttr *	: Attribute				*/
/*  (returns)	: KBAttrItem	:					*/

KBAttrItem::KBAttrItem
	(	KBAttr	*attr
	)
	:
	m_attr	(attr),
	m_value	(attr->getValue())
{
}

/*  KBAttrItem								*/
/*  ~KBAttrItem	: Destructor for property dialog attribute item		*/
/*  (returns)	:		:					*/

KBAttrItem::~KBAttrItem ()
{
}

/*  KBAttrItem								*/
/*  displayValue: Get value for display in attribute list		*/
/*  (returns)	: QString	: Display text				*/

QString	KBAttrItem::displayValue ()
{
	return	m_attr->displayValue (m_value) ;
}

/*  KBAttrItem								*/
/*  save	: Save value back into attribute			*/
/*  (returns)	: void		:					*/

void	KBAttrItem::save ()
{
	m_attr->setValue (m_value) ;
}

/*  KBAttrItem								*/
/*  value	: Get value						*/
/*  (returns)	:		: Value					*/

QString &KBAttrItem::value ()
{
	return	m_value	;
}

/*  KBAttrItem								*/
/*  setValue	: Set value						*/
/*  value	: const QString & : Value to set			*/
/*  (returns)	: void		  :					*/

void	KBAttrItem::setValue
	(	const QString	&value
	)
{
	m_value	= value ;
}

/*  KBAttrItem								*/
/*  clear	: Clear attribute					*/
/*  (returns)	: void		  :					*/

void	KBAttrItem::clear ()
{
	m_value	= QString::null ;
}

/*  KBAttrItem								*/
/*  appendItem	: Append display listview item to set of items		*/
/*  lvi		: QListViewItem * : Item in question			*/
/*  (returns)	: void		  :					*/

void	KBAttrItem::appendItem
	(	QListViewItem	*lvi
	)
{
	m_lvitems.append (lvi) ;
}

/*  KBAttrItem								*/
/*  display	: Display current value					*/
/*  (returns)	: void		  :					*/

void	KBAttrItem::display ()
{
	LITER
	(	QListViewItem,
		m_lvitems,
		lvi,
		lvi->setText (1, displayValue()) ;
	)
}

/*  KBAttrItem								*/
/*  getAttrDlg	: Get attribute dialog for this item			*/
/*  parent	: QWidget *		: Parent widget			*/
/*  attrDict	: QDict<KBAttrItem> &	: Disctionary of all attributes	*/
/*  (returns)	: KBAttrDlg *		:				*/

KBAttrDlg
	*KBAttrItem::getAttrDlg
	(	QWidget			*parent,
		QDict<KBAttrItem>	&attrDict
	)
{
	/* Tricky code alert!						*/
	/* Normally, the attribute dialogs can be retured via a method	*/
	/* on the attribute, which is what this methiod defaults to.	*/
	/* But, by routing the request for the dialog through a virtual	*/
	/* method on the KBAttrItem class, we get the ability to pass	*/
	/* a specific derived item to the dialog.			*/
	return	m_attr->getAttrDlg (parent, this, attrDict) ;
}

/*  KBAttrItem								*/
/*  showChoices	: Show set of choices					*/
/*  choice	: IntChoice *	  : Set of choices			*/
/*  value	: const QString & : Current value			*/
/*  cbox	: QComboBox *	  : Combo box to use			*/
/*  (returns)	: void		  :					*/

void	KBAttrItem::showChoices
	(	IntChoice	*choices,
		const QString	&value,
		QComboBox	*cbox
	)
{
	/* This method is static ....					*/
	int	at = -1 ;
	cbox->show ()   ;

	while (choices->value >= 0)
	{
		cbox->insertItem (choices->descr) ;
		if (choices->value == value.toInt())
			at = cbox->count() - 1 ;
		choices += 1 ;
	}

	if (at >= 0) cbox->setCurrentItem (at) ;
}

/*  ------------------------------------------------------------------  */

/*  KBAttrIntChoice							*/
/*  KBAttrIntChoice							*/
/*		: Constructor for integer choice attribute item		*/
/*  attr	: KBAttr *	  : Actual attribute			*/
/*  choices	: IntChoice *	  : List of choices			*/
/*  defval	: int		  : Default value				*/
/*  (returns)	: KBAttrIntChoice :					*/

KBAttrIntChoice::KBAttrIntChoice
	(	KBAttr		*attr,
		IntChoice	*choices,
		int		defval
	)
	:
	KBAttrItem	(attr),
	choices		(choices),
	defval		(defval)
{
}

/*  KBAttrIntChoice							*/
/*  ~KBAttrIntChoice							*/
/*		: Destructor for integer choice attribute item		*/
/*  (returns)	:		:					*/

KBAttrIntChoice::~KBAttrIntChoice ()
{
}

/*  KBAttrIntChoice							*/
/*  displayValue: Get value for display in attribute list		*/
/*  (returns)	: QString	: Display text				*/

QString	KBAttrIntChoice::displayValue ()
{
	int	v = m_value.isEmpty() ? defval : m_value.toInt() ;

	for (IntChoice *c = choices ; c->value >= 0 ; c += 1)
		if (c->value == v)
			return	c->descr ;

	return	TR("#Error#") ;
}

/*  ------------------------------------------------------------------  */

struct GroupMap
{	cchar	*m_legend	;
	int	m_flags		;
	bool	m_open		;
}	;

static	GroupMap groupMap[] =
{
	{	"Display",	KAF_GRPFORMAT,	false	},
	{	"Data",		KAF_GRPDATA,	false	},
	{	"Events",	KAF_GRPEVENT,	false	},
	{	"Other",	KAF_GRPOTHER,	false	},
	{	"Notes",	KAF_GRPNOTES,	false	},
	{	"All",		0xffffffff,	false	},
	{	0,		0,		false	}
}	;


/*  KBPropDlg								*/
/*  KBPropDlg	: Constructor for control properties dialog		*/
/*  node	: KBNode *	: Parent node				*/
/*  _caption	: cchar *	: Dialog box caption			*/
/*  attribs	: QList<KBAttr>&: List of attributes			*/
/*  iniattr	: cchar *	: Initial attribute			*/
/*  (returns)	: KBPropDlg	:					*/

KBPropDlg::KBPropDlg
	(	KBNode		*node,
		cchar		*_caption,
		QList<KBAttr>	&attribs,
		cchar		*iniattr
	)
	:
	_KBDialog 	(_caption, true),
	KBPlayer	("propdlg", node->getElement(), this),
	node	  	(node),
	topLayout	(this),
	butLayout	(&topLayout),
	topWidget	(this),
	propAllVal  	(&topWidget),
	descrip	  	(&topWidget),
	bOK	  	(this),
	bCancel	  	(this),
	bAccept	  	(this),
	bIgnore	  	(this),
	bClear		(this),
	bEdit		(this),
	bVerify		(this),
	bHelp		(this),
	attribs	  	(attribs),
	iniattr		(iniattr),
	editBox	  	(&topWidget),
	tkEditor	(new TKTextDocument (getTextManager(KBOptions::getScriptFont())), &topWidget),
	mapper		(&tkEditor),
	comboBox  	(&topWidget),
	checkBox  	(&topWidget),
	spinBox  	(&topWidget)
{
	propAllVal.setRootIsDecorated(true) ;
	propAllVal.setSorting	     (-1)   ;

	QListViewItem *group = 0 ;
	for (GroupMap *gm = &groupMap[0] ; gm->m_legend != 0 ; gm += 1)
	{	group = new QListViewItem(&propAllVal, group, TR(gm->m_legend)) ;
		m_groups.insert (gm->m_flags, group) ;
	}

	m_slotListDlg	= 0 ;
	m_configDlg	= 0 ;

	userWidget	= 0 ;
	attrDlg		= 0 ;
	xmodeItem	= 0 ;
	ymodeItem	= 0 ;
	caption		= _caption ;

	KBObject	*obj	;
	if ((obj = node->isObject()) != 0)
	{
		gRect	= obj->geometry() ;
		pRect	= QRect
			  (	obj->getAttrVal("x").toInt(),
			  	obj->getAttrVal("y").toInt(),
			  	obj->getAttrVal("w").toInt(),
			  	obj->getAttrVal("h").toInt()
			  )	;
	}



	topLayout.setDirection (QBoxLayout::BottomToTop) ;

	setupListView (propAllVal,    true ) ;
	positionCtrls () ;

	spinBox   .resize      (100,  20) ;
	checkBox  .setText     ("") ;


	bOK	   .setText      (TR("OK"     )) ;
	bCancel    .setText      (TR("Cancel" )) ;
	bAccept    .setText      (TR("Accept" )) ;
	bEdit      .setText      (TR("Edit"   )) ;
	bIgnore    .setText      (TR("Ignore" )) ;
	bClear     .setText      (TR("Clear"  )) ;
	bVerify    .setText      (TR("Verify" )) ;
	bHelp      .setText      (TR("Help"   )) ;

	topLayout  .addWidget    (&topWidget ) ;

	butLayout  .addWidget	 (&bOK,      0, 0) ;
	butLayout  .addWidget	 (&bCancel,  1, 0) ;
	butLayout  .addWidget	 (&bEdit,    0, 1) ;
	butLayout  .addWidget	 (&bAccept,  0, 2) ;
	butLayout  .addWidget	 (&bIgnore,  1, 2) ;
	butLayout  .addWidget	 (&bClear,   0, 3) ;
	butLayout  .addWidget	 (&bVerify,  1, 3) ;
	butLayout  .addWidget	 (&bHelp,    0, 5) ;
	butLayout  .setColStretch(4, 0) ;

	topLayout  .setStretchFactor (&topWidget, 1) ;
	topLayout  .setStretchFactor (&butLayout, 0) ;

	descrip	   .setTextFormat(Qt::RichText) ;


	descrip    .hide () ;
	editBox    .hide () ;
	tkEditor   .hide () ;
	comboBox   .hide () ;
	checkBox   .hide () ;
	spinBox    .hide () ;

	bOK        .show () ;
	bCancel    .show () ;
	bEdit      .show () ;
	bAccept    .show () ;
	bIgnore	   .show () ;
	bClear	   .show () ;
	bVerify    .show () ;

	/* The accept and ignore buttons are only enabled when a	*/
	/* property is being edited, so they start disabled.		*/
	bAccept   .setEnabled(false) ;
	bIgnore   .setEnabled(false) ;

	bEdit     .setEnabled(false) ;
	bClear    .setEnabled(false) ;
	bVerify   .setEnabled(false) ;
	bHelp     .setEnabled(false) ;

#if	! __KB_KDE && ! __KB_EMBEDDED
	bHelp	  .hide() ;
#endif

	connect (&bOK,	     SIGNAL(clicked  ()),    SLOT(clickOK     ()   )) ;
	connect (&bCancel,   SIGNAL(clicked  ()),    SLOT(clickCancel ()   )) ;
	connect (&bAccept,   SIGNAL(clicked  ()),    SLOT(clickAccept ()   )) ;
	connect (&bEdit,     SIGNAL(clicked  ()),    SLOT(clickEdit   ()   )) ;
	connect (&bIgnore,   SIGNAL(clicked  ()),    SLOT(clickIgnore ()   )) ;
	connect (&bClear,    SIGNAL(clicked  ()),    SLOT(clickClear  ()   )) ;
	connect (&bVerify,   SIGNAL(clicked  ()),    SLOT(clickVerify ()   )) ;
	connect (&bHelp,     SIGNAL(clicked  ()),    SLOT(clickHelp   ()   )) ;

	connect	(&topWidget, SIGNAL(resized(KBResizeWidget *, QSize)),
		 	     SLOT  (resized(KBResizeWidget *, QSize))) ;

	curAttrItem 	= 0 ;
	curListItem	= 0 ;
	clrAttrItem	= 0 ;

	attrByName  .setAutoDelete (true ) ;
	attrByLegend.setAutoDelete (false) ;

	iniRSize	= true	;

	TKConfig *config = TKConfig::getConfig() ;
	config->setGroup ("Property Editor") ;
	width	= config->readNumEntry ("width",  600) ;
	height	= config->readNumEntry ("height", 260) ;




	/* Install event filters on the listview. This is used to stop	*/
	/* the user changing the selection while a property is being	*/
	/* edited.							*/
	propAllVal.viewport()->installEventFilter (this) ;

	tkEditor.setHighlight	    ("NoHighlight") ;

	KBObject *object = node->isObject() ;
	if (object != 0)
	{
		const QList<KBSlot> objSlots = object->getSlots() ;
		LITER
		(	KBSlot,
			objSlots,
			slot,
			m_slotList.append(new KBSlot(0, slot))
		)
	}

	m_configDlg  = new KBConfigDlg (&topWidget, node) ;
	m_configDlg->hide() ;

	resize	    (width, height) ;
}

/*  KBPropDlg								*/
/*  ~KBPropDlg	: Destructor for control properties dialog		*/
/*  (returns)	:		:					*/

KBPropDlg::~KBPropDlg ()
{
	TKConfig *config   = TKConfig::getConfig () ;
	config->setGroup   ("Property Editor") ;

	config->writeEntry ("width",  width  ) ;
	config->writeEntry ("height", height ) ;
	config->sync	   () ;

	for (GroupMap *gm = &groupMap[0] ; gm->m_legend != 0 ; gm += 1)
	{
		QListViewItem *item = m_groups.find (gm->m_flags) ;
		if (item != 0) gm->m_open = item->isOpen() ;
	}
}

/*  KBPropDlg								*/
/*  positionCtrls: Set basic control positions				*/
/*  (return)	 : void		:					*/

void	KBPropDlg::positionCtrls ()
{
	int	x	= editLeftSide() ;
	int	y	= getMargin   () + 80 + getSpacing() ;

	descrip   .move	       (x, getMargin()) ;
	editBox   .move	       (x, y) ;
	tkEditor  .move	       (x, y) ;
	comboBox  .move	       (x, y + 70) ;
	checkBox  .move	       (x, y) ;
	spinBox   .move	       (x, y) ;
}

/*  KBPropDlg								*/
/*  eventFilter	: Event filter routine					*/
/*  o		: QObject *	: Object receiving event		*/
/*  e		: QEvent *	: The event				*/
/*  (returns)	: bool		: True to consume event			*/

bool	KBPropDlg::eventFilter
	(	QObject		*o,
		QEvent 		*e
	)
{
	if (o == propAllVal.viewport())
	{
		if (curListItem != 0)
		{
			switch (e->type())
			{
				case QEvent::MouseButtonPress	:
				case QEvent::MouseButtonRelease	:
				case QEvent::KeyPress		:
				case QEvent::KeyRelease		:
					return	true	;

				default	:
					break	;
			}

			return	false	;
		}

		return	false	;
	}

	return	false	;
}


/*  KBPropDlg								*/
/*  setUserWidget: Set user supplied display widget			*/
/*  _userWidget  : QWidget *	: Widget in question			*/
/*  (returns)	 : void		:					*/

void	KBPropDlg::setUserWidget
	(	QWidget	*_userWidget
	)
{
	if ((userWidget = _userWidget) != 0)
	{
		userWidget->move (editLeftSide(),  90) ;
		userWidget->show () ;
	}
}

/*  KBPropDlg								*/
/*  editLeftSide: Get left hand side of edit area			*/
/*  (returns)	: uint		: Coordinate				*/

uint	KBPropDlg::editLeftSide ()
{

#if	__KB_EMBEDDED
	return	getMargin() ;
#else
	return	getMargin() + LWIDTH + getSpacing() ;
#endif
}


/*  KBPropDlg								*/
/*  setSizes	: Set sizes of variable-size widgets			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::setSizes ()
{
	QSize	s	= topWidget.size () ;

	int	flags	= curAttrItem == 0 ? 0 : curAttrItem->attr()->getFlags () ;
	int	propW	= curAttrItem == 0 ? s.width() - 2 * getMargin() : editLeftSide() - getMargin() ;
	int	attrW	= s.width () - getMargin() - editLeftSide () ;
	int	propH	= s.height() - propAllVal.y() ;

	propAllVal.resize (propW, propH) ;
	descrip   .resize (attrW, 80)    ;

	if (userWidget == 0)
	{
		comboBox  .resize  (attrW, 20) ;
		checkBox  .resize  (attrW, 20) ;

		editBox   .resize
		(	attrW,
			(flags & KAF_SINGLE) != 0 ? 25  :
			(flags & KAF_MULTI)  == 0 ? 60  : s.height() - editBox.y()
		)	;

		tkEditor   .resize (attrW, s.height() - tkEditor   .y()) ;
	}
	else	userWidget->resize (attrW, s.height() - userWidget->y()) ;
}

/*  KBPropDlg								*/
/*  resizeEvent	: Handle dialog resizing				*/
/*  e		: QResizeEvent * : Resize event				*/
/*  (returns)	: void		 :					*/

void	KBPropDlg::resizeEvent
	(	QResizeEvent	*
	)
{
	/* Catch this to (a) note the size for when we close and (b)	*/
	/* to subvert the dialog box sizing.				*/
	QSize	s = size   () ;
	width	= s.width  () ;
	height	= s.height () ;
#if	! __KB_EMBEDDED
	setMinimumSize (600, 350) ;
#endif
}


/*  KBPropDlg								*/
/*  resized	: Top widget has been resized				*/
/*  widget	: KBResizeWidget * : Widget in question			*/
/*  size	: QSize		   : New size				*/
/*  (returns)	: void		   :					*/

void	KBPropDlg::resized
	(	KBResizeWidget	*,
		QSize		
	)
{
	setSizes () ;
}


/*  KBPropDlg								*/
/*  setupListView: Set up a attributes list view			*/
/*  lView	 : QListView &	: List view				*/
/*  visible	 : bool		: True if list view is visible		*/
/*  (returns)	 : void		:					*/

void	KBPropDlg::setupListView
	(	QListView	&lView,
		bool		visible
	)
{
	lView.setHScrollBarMode (QListView::AlwaysOff)	;
	lView.header()->setResizeEnabled (false)	;

	lView.move		(getMargin(), getMargin());
	lView.addColumn		(TR("Attribute"), LWIDTH) ;
	lView.addColumn		(TR("Value"),       1500) ;

	connect	(&lView, SIGNAL(currentChanged(QListViewItem *)),
		 this,   SLOT  (setCurrent    (QListViewItem *))) ;
	connect	(&lView, SIGNAL(doubleClicked (QListViewItem *)),
		 this,   SLOT  (pickProperty  (QListViewItem *))) ;

	if (visible)
	{
		lView.show ()	  ;
	}
	else	lView.hide ()	  ;
}

/*  KBPropDlg								*/
/*  preExec	: Pre-execute hook					*/
/*  (returns)	: void		:					*/

void	KBPropDlg::preExec ()
{
}

/*  KBPropDlg								*/
/*  exec	: Execute dialog					*/
/*  (returns)	: bool		: Success				*/

bool	KBPropDlg::exec ()
{
	/* We need to trap this method in order to load the attributes.	*/
	/* This cannot be done in the constructor, since the		*/
	/* "hideProperty" mechanism will not work at that stage.	*/
	LITER
	(	KBAttr,
		attribs,
		attr,

		if (!hideProperty (attr)) addAttrib (attr) ;
	)

	setProperty ("configs", m_configDlg->getText()) ;
	preExec	    () ;

	if (!iniattr.isNull())
		for (QListViewItem *g  = propAllVal.firstChild () ;
				    g != 0 ;
				    g  = g->nextSibling())
			for (QListViewItem *i  = g->firstChild () ;
					    i != 0 ;
					    i  = i->nextSibling())
				if (((KBAttrListViewItem *)i)->getName() == iniattr)
				{
					pickProperty (i)    ;
					goto gotit ;
				}

	gotit :

	for (GroupMap *gm = &groupMap[0] ; gm->m_legend != 0 ; gm += 1)
	{
		QListViewItem *item = m_groups.find (gm->m_flags) ;
		if (item != 0) item->setOpen(gm->m_open) ;
	}

	/* Now we really can run it ....				*/
#if	__KB_EMBEDDED
	showMaximized () ;
#endif
	return	QDialog::exec () ;
}

/*  KBPropDlg								*/
/*  getAttrItem	: Get attribute item for attribute			*/
/*  attr	: KBAttr *	: Attribute				*/
/*  (returns)	: KBAttrItem *	: Associated attribute item		*/

KBAttrItem *KBPropDlg::getAttrItem
	(	KBAttr	*attr
	)
{
	if (attr->getName() == "xmode")
		return	xmodeItem = new KBAttrIntChoice (attr, choiceFMode, KBObject::FMFixed) ;

	if (attr->getName() == "ymode")
		return	ymodeItem = new KBAttrIntChoice (attr, choiceFMode, KBObject::FMFixed) ;

	if (attr->getName() == "slots" )
		return	new KBAttrSlots (attr, m_slotList) ;

	if (attr->getName() == "autosize")
		return	new KBAttrIntChoice (attr, choiceScaling) ;

	return	attr->getAttrItem () ;
}

/*  KBPropDlg								*/
/*  addAttrib	: Add an attribute to the property list			*/
/*  attr	: KBAttrItem *	: Attribute item			*/
/*  lView	: QListView &	: List view into which to insert	*/
/*  (returns)	: void		:					*/

void	KBPropDlg::addAttrib
	(	KBAttrItem	*item,
		QListView	&lView
	)
{
	int		gFlags	= item->attr()->getFlags() & KAF_GRPMASK ;
	QListViewItem	*group	;
	QListViewItem	*last	;

	if ((group = m_groups.find (gFlags)) == 0)
	{
		group = new QListViewItem (&lView, QString().sprintf("Unknown %08x", gFlags)) ;
		m_groups.insert(gFlags, group) ;
	}

	last	= group->firstChild() ;
	while ((last != 0) && (last->nextSibling() != 0))
		last = last->nextSibling() ;

	item->appendItem
		(	new KBAttrListViewItem
			(	group, //&lView,
				last,
				item->attr()->getLegend(),
				item->displayValue     (),
				item->attr()->getName  (),
				item->attr()->getOrder ()
			)
		) ;

	if ((group = m_groups.find (0xffffffff)) != 0)
	{
		last	= group->firstChild() ;
		while ((last != 0) && (last->nextSibling() != 0))
			last = last->nextSibling() ;

		item->appendItem
			(	new KBAttrListViewItem
				(	group, //&lView,
					last,
					item->attr()->getLegend(),
					item->displayValue     (),
					item->attr()->getName  (),
					item->attr()->getOrder ()
				)
			) ;
	}
}

/*  KBPropDlg								*/
/*  addAttrib	: Add an attribute to the property list			*/
/*  attr	: KBAttr *	: Attribute in question			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::addAttrib
	(	KBAttr	*attr
	)
{
fprintf	(stderr, "add  ---> [%s]\n", (cchar *)attr->getName()) ;
	KBAttrItem *item ;

	if ((item = attrByLegend.find (attr->getLegend())) == 0)
	{	attrByName  .insert (attr->getName  (), item = getAttrItem (attr)) ;
		attrByLegend.insert (attr->getLegend(), item ) ;
	}

	addAttrib (item, propAllVal) ;
}

/*  KBPropDlg								*/
/*  warning	: Display a warning message				*/
/*  msg		: cchar *	: Message				*/
/*  (returns)	: bool		: Always false				*/

bool	KBPropDlg::warning
	(	cchar	*msg
	)
{
	TKMessageBox::sorry (0, msg, TR("Unknown dialog error")) ;
	return	false  ;
}


/*  KBPropDlg								*/
/*  dropProperty: Drop currently displayed property			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::dropProperty ()
{
	if (userWidget != 0)
	{
		userWidget->hide () ;
		userWidget = 0 ;
	}

	descrip   .setText ("")	;
	descrip   .hide	   ()	;
	editBox   .hide	   ()	;
	editBox   .clear   ()	;
	tkEditor  .hide	   ()	;
	tkEditor  .clear   ()	;
	comboBox  .hide	   ()	;
	comboBox  .clear   ()	;
	checkBox  .hide	   () 	;
	spinBox   .hide	   () 	;

	positionCtrls 	   () 	;

	disconnect (&comboBox, SIGNAL (activated(const QString &)),
		    this,      SLOT   (pickCombo(const QString &))) ;


	bVerify   .setEnabled (false) ;
}

/*  KBPropDlg								*/
/*  hideProperty: Check whether property should be hidden		*/
/*  attr	: KBAttr *	: Attribute in question			*/
/*  (returns)	: bool		: True to hide				*/

bool	KBPropDlg::hideProperty
	(	KBAttr	*
	)
{
	/* This method allows selective hiding of properties. The	*/
	/* default here is that everything goes.			*/
	return	false	;
}

/*  KBPropDlg								*/
/*  showChoices	: Display choice selection				*/
/*  attr	: KBAttr *	 : Attribute in question		*/
/*  choices	: IntChoice *	 : Set of choices			*/
/*  value	: const QString &: Current value			*/
/*  cbox	: QComboBox *	 : Combo box to use or null		*/
/*  (returns)	: void		 :					*/

void	KBPropDlg::showChoices
	(	KBAttrItem	*,
		IntChoice	*choices,
		const QString	&value,
		QComboBox	*cbox
	)
{
	if (cbox == 0) cbox = &comboBox ;
	KBAttrItem::showChoices (choices, value, cbox) ;
}


/*  KBPropDlg								*/
/*  saveChoices	: Save choice selection					*/
/*  item	: KBAttrIte *	 : Associated item			*/
/*  choices	: IntChoice *	 : Set of choices			*/
/*  cbox	: QComboBox *	 : Combo box to use or null		*/
/*  (returns)	: void		 :					*/

void	KBPropDlg::saveChoices
	(	KBAttrItem	*item,
		IntChoice	*choices,
		QComboBox	*cbox
	)
{
	int	slot	= cbox == 0 ?
				comboBox.currentItem () :
				cbox   ->currentItem () ;

	if (slot >= 0)
		setProperty (item, QString("%1").arg(choices[slot].value)) ;
	else	setProperty (item, "") ;
}

/*  KBPropDlg								*/
/*  showProperty: Display property for update				*/
/*  item	: KBAttrItem *	: Associated item			*/
/*  (returns)	: bool		: Property shown OK			*/

bool	KBPropDlg::showProperty
	(	KBAttrItem	*item
	)
{
	QString		text	;
	KBAttr		*attr	= item->attr() ;
	const QString	&name	= attr->getName () ;

	if ((attrDlg = item->getAttrDlg (&topWidget, attrByName)) != 0)
	{
		if (attrDlg->init())
		{
			item->display ()  ;
			return false 	  ;
		}

		setupLayout 	(attrDlg) ;
		setUserWidget	(attrDlg) ;

		bVerify .setEnabled((attr->getFlags() & KAF_GRPEVENT) != 0) ;
		return	true	;
	}

	/* Foreground/Background colour					*/
	/* Handle the colour attributes specially, bringing up the	*/
	/* standard colour dialog box. Note that in this case we must	*/
	/* handle the Accept/Reject functionallity here.		*/
	if ((name == "fgcolor") || (name == "bgcolor"))
	{
		TKColorDialog cDialog (this, TR("Colour"), true) ;
		cDialog.setColor (QColor (strtol(item->value(), 0, 0))) ;
		if (cDialog.exec ())
		{	
			resultVal.sprintf ("0x%06x", cDialog.color().rgb() & 0x00ffffff) ;
			clickAccept () ;
		}
		return	false	;
	}

	/* Font								*/
	/* Similarly for a font. We have to parse a font string		*/
	/* ouselves, since we don't really want to use X font strings.	*/
	if (name == "font")
	{
		TKFontDialog  fDialog (this, TR("Font"), false, true) ;
		fDialog.setFont (KBFont::specToFont (item->value())) ;

		if (fDialog.exec())
		{
			resultVal = KBFont::fontToSpec (fDialog.font()) ;
			clickAccept () ;
		}

		return	false	;
	}

	/*  Xmode/YMode                                                 */
	/*  X- and Y- fixed/stretch/float modes ...                     */	
	if ((name == "xmode") || (name == "ymode"))
	{
		showChoices (item, choiceFMode, item->value()) ;
		return	true ;
	}

	/* Helper							*/
	/* Display the combo box with the list of all registered	*/
	/* helpers.							*/
	if (name == "helper")
	{
		extern	QStringList &getHelperSet() ;

		comboBox.clear		 () ;
		comboBox.insertItem	 ("") ;
		comboBox.insertStringList(getHelperSet()) ;

		for (int idx = 0 ; idx < comboBox.count() ; idx += 1)
			if (comboBox.text(idx) == item->value())
			{	comboBox.setCurrentItem (idx) ;
				break	;
			}

		comboBox.show		() ;
		return	true ;
	}

	/* Slots							*/
	/* This brings up a list of extant slots, plus actions like add	*/
	/* and edit. All the real work is in the separate slot dialog.	*/
	if (name == "slots")
	{
		if (m_slotListDlg == 0)
			m_slotListDlg = new KBSlotListDlg
					(	&topWidget,
						m_slotList,
						node->isObject()
					) ;

		setUserWidget (m_slotListDlg) ;
		return	true  ;
	}

	if (name == "configs")
	{
		setUserWidget (m_configDlg) ;
		return	true	 ;
	}

	if (name == "autosize")
	{
		showChoices (item, choiceScaling, item->value()) ;
		return	true ;
	}

	/* Next check for boolean attributes ....			*/
	if (attr->getType() == KBAttr::Bool)
	{
		checkBox.setChecked  (item->value() == "Yes") ;
		checkBox.setText     (item->attr()->getLegend ()) ;
		checkBox.show	     () ;
		checkBox.setFocus    () ;
		return	true ;
	}

	/* ... and integers. Note that the spinbox range limits are	*/
	/* integers, hence use INT_MAX in the KBAttr::UInt case.	*/
	if (attr->getType() == KBAttr::Int)
	{
		spinBox .setRange    (INT_MIN, INT_MAX) ;
		spinBox .setValue    (item->value().toInt()) ;
		spinBox .show	     () ;
		spinBox .setFocus    () ;
		return	true ;
	}
	if (attr->getType() == KBAttr::UInt)
	{
		spinBox .setRange    (0, INT_MAX) ;
		spinBox .setValue    (item->value().toInt()) ;
		spinBox .show	     () ;
		spinBox .setFocus    () ;
		return	true ;
	}


	if ((attr->getFlags() & KAF_EDITOR) != 0)
	{
		tkEditor.setText(item->value()) ;
		tkEditor.show   () ;
		setSizes ()	;
		return	 true	;
	}

	/* Default case, show the edit box which will overlay the	*/
	/* property values list box, and set the current property	*/
	/* value therein.						*/
	editBox.clear	    () ;
	editBox.show	    () ;
	editBox.unsetFont   () ;

	if ((attr->getFlags() & KAF_MULTI ) != 0)
		editBox.setFont	(QFont (TR("Courier"), 12)) ;


	editBox.insert	    (item->value()) ;
	editBox.setFocus    () ;
	editBox.setCursorPosition (0xffff, 0xffff) ;

	setSizes ()	;
	return	 true	;
}

/*  KBPropDlg								*/
/*  setProperty	: Set a property value					*/
/*  item	: KBAttrItem *	 : Attribute in question		*/
/*  value	: const QString &: Value to set				*/
/*  (returns)	: bool		 : Set OK				*/

bool	KBPropDlg::setProperty
	(	KBAttrItem	*item,
		const QString	&value
	)
{
	if (item == 0)
		return	false	;

	if (!item->attr()->valueOK (value))
		return	warning	(QString(TR("%1 has an invalid value"))
					.arg(item->attr()->getLegend ())) ;

	item->setValue (value)   ;
	item->display  ()	 ;


	const	QString	&name	= item->attr()->getName() ;
	bool	xywhDelta	= false ;

	if (name == "x")
	{
		pRect.moveTopLeft (QPoint(value.toInt(), pRect.y())) ;
		xywhDelta = true  ;
	}
	if (name == "y")
	{
		pRect.moveTopLeft (QPoint(pRect.x(), value.toInt())) ;
		xywhDelta = true  ;
	}
	if (name == "w")
	{
		pRect.setWidth    (value.toInt()) ;
		xywhDelta = true  ;
	}
	if (name == "h")
	{
		pRect.setHeight   (value.toInt()) ;
		xywhDelta = true  ;
	}

	if (xywhDelta)
	{
		gRect	= node->isObject()->convGeometry
			  (	pRect,
				xmodeItem == 0 ? KBObject::FMFixed : (KBObject::FloatMode)xmodeItem->value().toInt(),
				ymodeItem == 0 ? KBObject::FMFixed : (KBObject::FloatMode)ymodeItem->value().toInt()
			  )	;
		return	true	;
	}

	if ((item == xmodeItem) || (item == ymodeItem))
	{
		pRect	= node->isObject()->convGeometry
			  (	gRect,
				(KBObject::FloatMode)xmodeItem->value().toInt(),
				(KBObject::FloatMode)ymodeItem->value().toInt()
			  )	;
		setProperty ("x", QString("%1").arg(pRect.x	())) ;
		setProperty ("y", QString("%1").arg(pRect.y	())) ;
		setProperty ("w", QString("%1").arg(pRect.width ())) ;
		setProperty ("h", QString("%1").arg(pRect.height())) ;
		return	true ;
	}

	return	true	;
}

/*  KBPropDlg								*/
/*  setProperty	: Set a property value					*/
/*  attr	: cchar *	 : Attribute name			*/
/*  value	: const QString &: Value to set				*/
/*  (returns)	: bool		 : Set OK				*/

bool	KBPropDlg::setProperty
	(	cchar		*name,
		const QString	&value
	)
{
	return	setProperty (attrByName.find(name), value) ;
}

/*  KBPropDlg								*/
/*  getProperty	: Get current property value				*/
/*  name	: cchar *	 : Attribute name			*/
/*  (returns)	: const QString &: Property value			*/

const QString
	&KBPropDlg::getProperty
	(	cchar	*name
	)
{
	KBAttrItem *attr = attrByName.find(name) ;
	return	attr == 0 ? QString::null : attr->value() ;
}

/*  KBPropDlg								*/
/*  saveProperty: Check and save attribute				*/
/*  item	: KBAttrItem *	: Assoiated item			*/
/*  (returns)	: bool		: Attribute OK				*/

bool	KBPropDlg::saveProperty
	(	KBAttrItem	*item
	)
{
	const QString	&name	= item->attr()->getName () ;

	if (attrDlg != 0)
	{
		attrDlg->save   () ;
		item   ->display() ;

		setUserWidget   (0) ;
		DELOBJ	        (attrDlg) ;
		return true ;
	}

	/* Colour and font results are stored on the "resultVal" member	*/
	/* whence they are stored here.					*/
	if ((name == "fgcolor") || (name == "bgcolor") || (name == "font"))
	{
		setProperty  (item, resultVal) ;
		return	true ;
	}


	/*  Xmode/YMode                                                 */
	/*  X- and Y- fixed/stretch/float modes ...                     */	
	if ((name == "xmode") || (name == "ymode"))
	{
		saveChoices (item, choiceFMode) ;
		return	true ;
	}


	/* Helper							*/
	/* Value is in the combo box					*/
	if (name == "helper")
	{
		setProperty (item, comboBox.currentText()) ;
		return	true ;
	}


	if (name == "slots")
	{
		m_slotListDlg->save (m_slotList) ;
		setProperty ("slots", QString::null); 
		return	true	;
	}

	if (name == "configs")
	{
		setProperty  ("configs", m_configDlg->getText()) ;
		return	true ;
	}

	if (name == "autosize")
	{
		saveChoices (item, choiceScaling) ;
		return	true	;
	}

	/* Now some more general cases. First, if the property is a	*/
	/* boolean then we will be using the checkbox, and the property	*/
	/* is set to "Yes" or "No".					*/
	if (item->attr()->getType() == KBAttr::Bool)
	{
		setProperty (item, checkBox.isChecked() ? "Yes" : "No") ;
		return	true	;
	}

	/* Next the integer and unsigned integer properties ..		*/
	if (item->attr()->getType() == KBAttr::Int)
	{
		setProperty (item, spinBox.cleanText()) ;
		return	true	;
	}
	if (item->attr()->getType() == KBAttr::UInt)
	{
		setProperty (item, spinBox.cleanText()) ;
		return	true	;
	}

	/* Check if the property was edited using the text editor	*/
	/* widget, in which case retrieve everything from there.	*/
	if ((item->attr()->getFlags() & KAF_EDITOR) != 0)
	{
		setProperty (item, tkEditor.text()) ;
		return	true ;
	}


	/* Final catchall ....						*/
	setProperty (item, editBox.text()) ;
	return	true ;
}

/*  KBPropDlg								*/
/*  findHelpMapping							*/
/*		: Find possible help mapping for name			*/
/*  name	: const QString & : Name				*/
/*  (returns)	: cchar *	  : Mapping or null if none		*/

cchar	*KBPropDlg::findHelpMapping
	(	const QString	&name
	)
{
	for (HelpMap *ptr = &helpMap[0] ; ptr->name != 0 ; ptr += 1)
		if (ptr->name == name)
			return	ptr->help ;

	return	0 ;
}

/*  KBPropDlg								*/
/*  getHelpTag	: Get the help tag for an item				*/
/*  item	: KBAttrItem *	: Item					*/
/*  (returns)	: QString	: Tag or null if no help file		*/

QString	KBPropDlg::getHelpTag
	(	KBAttrItem	*item
	)
{
	QString	tag	;
	QString	name	= item->attr()->getName	  () ;
	cchar	*owner	= item->attr()->getOwnerName() ;
	QString	ctag	= QString("%1_%2").arg(owner).arg(name) ;

	/* We maintain a cache of help tags based on the attribute name	*/
	/* and the owner's class name, to cut down on lookup times.	*/
	static	QCache<QString>	helpCache ;
	QString	*cached	= helpCache.find (ctag) ;
	if (cached != 0)
	{
		fprintf	(stderr, "helpCache %s_%s -> %s\n",
				 owner,
				 (cchar *)name,
				 (cchar *)*cached
			) ;
		return	*cached	;
	}

	/* First see if this is one of the special cases, in which case	*/
	/* rather then using the attribute name, we use the mapped	*/
	/* help name.							*/
	cchar	*hptr	= findHelpMapping (name) ;
	if (hptr != 0) name = hptr ;

	/* Now check for help files based on the attribute name and (a)	*/
	/* the attribute's owner's class name or (b) a more general	*/
	/* case just using he attribute name.				*/
	tag	= QString ("%1_%2") .arg(owner).arg(name) ;
	if (TKHelpProxy::self()->helpPageExists (tag))
	{	helpCache.insert (ctag, new QString(tag)) ;
		return tag ;
	}

	tag	= QString ("attr_%2").arg(name) ;
	if (TKHelpProxy::self()->helpPageExists (tag))
	{	helpCache.insert (ctag, new QString(tag)) ;
		return tag ;
	}

	fprintf	(stderr, "No help found for %s_%s\n",
			 (cchar *)owner,
			 (cchar *)name
		)	;
	return	QString::null	;
}


/*  KBPropDlg								*/
/*  setHelpEnabled							*/
/*		: Enable or disable help button appropriately for item	*/
/*  attr	: KBAttrItem *	: Item					*/
/*  (returns)	: void		:					*/

void	KBPropDlg::setHelpEnabled
	(	KBAttrItem	*attr
	)
{
	bHelp.setEnabled (!getHelpTag(attr).isNull()) ;
}

/*  KBPropDlg								*/
/*  pickProperty: Pick property to be edited				*/
/*  pItem	: QListViewItem*: Property item				*/
/*  (returns)	: void		:					*/

void	KBPropDlg::pickProperty
	(	QListViewItem *pItem
	)
{
	if (pItem->depth() == 0)
		return ;

	/* If this is the currently edited property then ignore it,	*/
	/* else if some other property is already being edited then	*/
	/* take it that the user accepts the new value.			*/
	if (curListItem == pItem)
		return ;

	if (curListItem != 0)
		if (!clickAccept ())
			return	;

	/* Make sure the item is highlighted. This is needed because of	*/
	/* the listview event filters, which let through double-click	*/
	/* but not (always) the single clicks.				*/
	propAllVal.setCurrentItem (pItem) ;

	curAttrItem = attrByLegend.find (pItem->text(0)) ;
	curListItem = pItem ;
	setHelpEnabled (curAttrItem) ;

	KBAttr	*attr	= curAttrItem->attr() ;

	if (!showProperty (curAttrItem))
	{
		curAttrItem	= 0 ;
		curListItem	= 0 ;

		setCurrent  (pItem) ;
		return	    ;
	}

	bEdit   .setEnabled (false) ;
	bClear  .setEnabled (false) ;
	bCancel .setEnabled (false) ;
	clrAttrItem = 0 ;

	/* The property values list box is partly hidden, and the	*/
	/* accept and ignore buttons enabled.				*/
	descrip  .show       () ;
	descrip  .setText    (attr->getDescription()) ;
	bAccept  .setEnabled (true) ;
	bIgnore  .setEnabled ((attr->getFlags() & KAF_SYNTHETIC) == 0) ;

	setSizes   () ;
	setCaption (QString("%1: %2").arg(caption).arg(curListItem->text(0))) ;
}

/*  KBPropDlg								*/
/*  clickEdit	: Handle click of the edit button			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::clickEdit ()
{
	if (curListItem == 0)
	{
		QListViewItem *pItem = propAllVal.currentItem() ;
		if (pItem != 0) pickProperty (pItem) ;
	}
}

/*  KBPropDlg								*/
/*  clickOK	: Handle click of the OK button				*/
/*  (returns)	: void		:					*/

void	KBPropDlg::clickOK ()
{
	/* If a property is currently being edited then take it that	*/
	/* the user accepts the newly edited value.			*/
	if (!clickAccept ())
		return ;

	/* Check that the property values are OK. If not then we do not	*/
	/* allow the dialog to be OK'ed					*/
	{
		QDictIterator<KBAttrItem> aIter (attrByName) ;
		KBAttrItem		  *item ;

		while ((item = aIter.current()) != 0)
		{	if (!propertyOK (item))
				return	;
			aIter	+= 1	;
		}
	}

	/* Copy back the property values. Since all properties are, at	*/
	/* the bottom, represented as strings, we can just pass the	*/
	/* text back from the property values list box.			*/
	{
		QDictIterator<KBAttrItem> aIter (attrByName) ;
		KBAttrItem		  *attr ;

		while ((attr = aIter.current()) != 0)
		{
			//attr->attr->setValue (attr->value) ;
			attr->save () ;
			aIter	+= 1  ;
		}
	}

	/* Similarly pass back the slots, if any. We just clear the	*/
	/* original slots, then associate each of the current slots	*/
	/* with the parent object.					*/
	KBObject *object = node->isObject() ;
	if (object != 0)
	{
		object->clearSlots () ;

		LITER
		(	KBSlot,
			m_slotList,
			slot,
			slot->setParent (object)
		)
	}

	m_configDlg->fixUp() ;


	done	(true) ;
}

/*  KBPropDlg								*/
/*  clickCancel	: Handle click of the Cancel button			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::clickCancel ()
{
	QDictIterator<KBAttrItem> aIter (attrByName) ;
	KBAttrItem		  *item ;

	while ((item = aIter.current()) != 0)
	{
		QString	vOld	= item->attr ()->getValue() ;
		QString	vNew	= item->value() ;

		if (vOld.isNull()) vOld = "" ;
		if (vNew.isNull()) vNew = "" ;

		fprintf
		(	stderr,
			"KBPropDlg::clickCancel: [%s] [%s][%s]\n",
			(cchar *)item->attr()->getName (),
			(cchar *)vOld,
			(cchar *)vNew
		)	;

		if (vOld != vNew)
		{
			if (TKMessageBox::questionYesNo
			   	(	0,
					TR("Some properties have been changed: cancel anyway?"),
					TR("Properties changed")
				)
				== TKMessageBox::Yes) break ;

			return	;
		}

		aIter	+= 1	;
	}

	done	(false) ;
}

/*  KBPropDlg								*/
/*  clickAccept	: Handle click of the Accept button			*/
/*  (returns)	: bool		: Value is acceptable			*/

bool	KBPropDlg::clickAccept ()
{
	/* We should only get here if there is a property being edited,	*/
	/* but check anyway. Then copy back the edited text, after	*/
	/* which we can treat it as an ignore operation.		*/
	if (curListItem != 0)
	{
		if (!saveProperty (curAttrItem))
			return false ;

		clickIgnore () ;
	}

	return	true ;
}

/*  KBPropDlg								*/
/*  clickIgnore	: Handle click of the Ignore button			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::clickIgnore ()
{
	if (curListItem != 0)
	{
		QListViewItem	*i = curListItem ;

		dropProperty 	      ()      ;
		bAccept   .setEnabled (false) ;
		bIgnore   .setEnabled (false) ;
		bCancel   .setEnabled ( true) ;

		DELOBJ	(attrDlg) ;

		curListItem = 0 ;
		curAttrItem = 0 ;
		setSizes () ;

		setCurrent  (i) ;
		setCaption  (caption) ;
	}
}

/*  KBPropDlg								*/
/*  clickClear	: Handle click of the clear button			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::clickClear ()
{
	if ((curListItem == 0) && (clrAttrItem != 0))
	{	clrAttrItem->clear   () ;
		clrAttrItem->display () ;
	}
}

/*  KBPropDlg								*/
/*  clickHelp	: Handle click of the help button			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::clickHelp ()
{
#if	__KB_KDE
	if (propAllVal.currentItem() != 0)
	{
		TKHelpProxy	*help	= TKHelpProxy::self() ;
		QListViewItem	*item	= propAllVal.currentItem() ;

		if (item->depth() == 0) return ;

		KBAttrItem	*attr	= attrByLegend.find (item->text(0)) ;
		QString		legend	= item->text(0) ;

		help->showHelpPage (getHelpTag (attr)) ;
	}
#endif
#if	__KB_EMBEDDED
	if (propAllVal.currentItem() != 0)
	{
		QListViewItem	*item	= propAllVal.currentItem() ;

		if (item->depth() == 0) return ;

		KBAttrItem	*attr	= attrByLegend.find (item->text(0)) ;
		QString		tag	= getHelpTag (attr) ;

		extern	void	helpPopup (const QString &) ;
		helpPopup (tag) ;
	}
#endif
}

/*  KBPropDlg								*/
/*  clickVerify : Handle click of the verify button			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::clickVerify  ()
{
	if (attrDlg != 0) attrDlg->verify() ;
}

/*  KBPropDlg								*/
/*  setCurrent	: Handle change in current item				*/
/*  pItem	: QListViewItem*: Property item				*/
/*  (returns)	: void		:					*/

void	KBPropDlg::setCurrent
	(	QListViewItem *pItem
	)
{
	if (pItem->depth() == 0)
		return ;

	if ((curListItem == 0) && (pItem != 0))
	{
		KBAttrItem *item  = attrByLegend.find (pItem->text(0)) ;
		bool	   enable = (item->attr()->getFlags() & (KAF_CLEAR|KAF_EDITOR)) != 0 ;

		clrAttrItem = enable ? item : 0 ;
		bClear.setEnabled  (enable) ;
		setHelpEnabled 	   (item)   ;

		bEdit .setEnabled  (true)   ;

		return	;
	}

	bEdit .setEnabled (false) ;
	bClear.setEnabled (false) ;
	clrAttrItem	= 0 ;
}


/*  KBPropDlg								*/
/*  pickCombo	: Handle user picking combo box element			*/
/*  value	: const QString &: Value selected			*/
/*  (returns)	: void		 :					*/

void	KBPropDlg::pickCombo
	(	const QString &value
	)
{
	editBox.insert 	  (value) ;
	editBox.setFocus  () ;
}

/*  KBPropDlg								*/
/*  propertyOK	: Check if property is OK				*/
/*  item	: KBAttrItem *	: Associated item			*/
/*  (returns)	: bool		: Property is OK			*/

bool	KBPropDlg::propertyOK
	(	KBAttrItem	*item
	)
{
	KBAttr	*attr	= item->attr() ;

	if (attr->required () && item->value().isEmpty())
		return	warning (attr->getNullcheck ()) ;

	if (!attr->valueOK (item->value()))
		return	warning	(QString(TR("%1 has an invalid value"))
					.arg(attr->getLegend ())) ;
	return	true	;
}


/*  KBPropDlg								*/
/*  playerPerform							*/
/*		: Perform a stanza from a score				*/
/*  action	: const QString		: Action			*/
/*  args	: const QStringList &	: Arguemnts			*/
/*  pError	: KBError &		: Error return			*/
/*  (returns)	; bool			: Success			*/

bool	KBPropDlg::playerPerform
	(	const QString		&action,
		const QStringList	&args,
		KBError			&pError
	)
{
	/* First few actions have no arguments to we check them first.	*/
	/* These just correspond to button clicks.			*/
	if (action == "ok")
	{
		clickOK	()	;
		return	true	;
	}

	if (action == "cancel")
	{
		clickCancel ()	;
		return	true	;
	}

	if (action == "accept")
	{
		clickAccept ()	;
		return	true	;
	}

	if (action == "ignore")
	{
		clickIgnore ()	;
		return	true	;
	}

	/* All following actions require a property item, which is	*/
	/* identified by name, so scan for this now. Note that we use	*/
	/* the current group and match against the actual property name	*/
	/* and not the legend shown in the list view.			*/
//	QListViewItem	*item	= curGroup->firstChild() ;
//	while (item != 0)
//	{
//		if (((KBAttrListViewItem *)item)->getName() == args[0]) break ;
//		item	= item->nextSibling() ;
//	}

	QListViewItem *item = 0 ;
	if (item == 0)
	{
		KBError::EError
		(	"Cannot find property\n",
			args[0],
			__ERRLOCN
		)	;
		return	true	;
	}

	/* Set								*/
	/* Set the value of the currently selected item to the value	*/
	/* specified in argument zero. This is equivalent to the user	*/
	/* macking a change and then clicking the "Accept" button.	*/
	if (action == "set")
	{
		QString	arg0 = args[0]	;

		clickAccept	()	;
		setProperty	(attrByName.find (arg0), args[1]) ;

		if ((arg0 != "fgcolor") && (arg0 != "bgcolor") && (arg0 != "font"))
			pickProperty	(item)	;

		return		true	;
	}

	/* Pick								*/
	/* Pick and display a property as if the user had doubleclicked	*/
	/* in the list view.						*/
	if (action == "pick")
	{
		pickProperty	(item)	;
		return		true	;
	}

	return	KBPlayer::playerPerform	(action, args, pError) ;
}



/*  KBPropDlg								*/
/*  playerName	: Return the name					*/
/*  (returns)	: QString	: Name					*/

QString	KBPropDlg::playerName ()
{
	return	node->isObject() != 0 ? node->isObject()->getName() : (QString)"notNamed" ;
}

/*  KBPropDlg								*/
/*  mouseDoubleClickEvent						*/
/*		: Handle mouse double click				*/
/*  e		: QMouseEvent *	: Associated event			*/
/*  (returns)	: void		:					*/

void	KBPropDlg::mouseDoubleClickEvent
	(	QMouseEvent	*e
	)
{
#if	0 && !__KB_RUNTIME && ! __KB_EMBEDDED
	/* Documentation and debugging hook. Double-click with the	*/
	/* control and shift keys down prints the attribute set to a	*/
	/* file which is named for the node.				*/
	if ((e->state() & (ShiftButton|ControlButton)) != (ShiftButton|ControlButton))
		return	;

	/* If the alt key is down then we will only print attributes	*/
	/* which are actually in the derived node class, rather than	*/
	/* all attributes including those in base classes.		*/
	extern	void	exportSGML (KBNode *, bool) ;
	exportSGML (node, (e->state() & AltButton) != 0) ;
#endif
	_KBDialog::mouseDoubleClickEvent (e) ;
}

