/***************************************************************************
    file	         : kb_layout.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<limits.h>

#include	<qevent.h>
#include	<qwidget.h>
#include	<qlist.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"
#include	"kb_block.h"
#include	"kb_sizer.h"
#include	"kb_layout.h"
#include	"kb_options.h"
#include	"kb_gui.h"

#if		! __KB_RUNTIME

#include	"kb_formcopier.h"
#include	"kb_component.h"
#include	"kb_compaccessdlg.h"

/*  KBLayout								*/
/*  initSizer	: Initialise sizers on entry to design mode		*/
/*  (returns)	: void		:					*/

void	KBLayout::initSizer()
{
	while (sizerList.count() > 0)
	{
		sizerList.at(0)->getObject()->setMonitorSelect (false) ;
		sizerList.at(0)->setState(KBSizer::sbIdle);
		sizerList.remove((uint)0);
	}
}

/*  KBLayout								*/
/*  releaseSizer: Release sizer at end of update			*/
/*  (returns)	: void		:					*/

void	KBLayout::releaseSizer ()
{
	/* Tell all selected sizers to note the new size and position	*/
	/* settings, and return the leading sizer to the leading state	*/
	/* (as it was in the active state for the duration of the	*/
	/* update).							*/
	for (uint idx = 0 ; idx < sizerList.count() ; idx += 1)
		sizerList.at(idx)->accept (false) ;

	sizerList.at(0)->setState(KBSizer::sbLeader) ;
}

/*  KBLayout								*/
/*  dropSizer	: Drop sizer from list					*/
/*  sizer	: KBSizer *	: Sizer in question			*/
/*  (returns)	: void		:					*/

void	KBLayout::dropSizer
	(	KBSizer	*sizer
	)
{
	sizer->getObject()->setMonitorSelect (false) ;
	sizerList.remove (sizer) ;
	setGUIEnables	 ()	 ;
}

/*  KBLayout								*/
/*  add		: User has selected a sizer tool			*/
/*  sizer	: KBSizer *	: Selected sizer			*/
/*  multi	: bool		: Selected for multiple update		*/
/*  (returns)	: KBLimit	: Movement limits			*/

KBLimit	KBLayout::addSizer
	(	KBSizer	*sizer,
		bool	multi
	)
{
	KBLimit	limit	(SHRT_MIN, SHRT_MAX,SHRT_MIN, SHRT_MAX) ;

	/* If this is _not_ a multiple update then remove all existing	*/
	/* sizers from the list, reseting each to the idle state.	*/
	if (!multi) initSizer() ;

	if (sizer == 0) return limit ;

	/* Make sure the specified sizer is not in the list (in case	*/
	/* of multiple selection), then put it at the front of the list	*/
	/* whence it becomes the leading sizer.				*/
	sizerList.remove (   sizer) ;
	sizerList.insert (0, sizer) ;

	/* Inform all listed sizers other then the new one that they	*/
	/* are now followers, and inform this one that it is active.	*/
	for (uint idx1 = 1 ; idx1 < sizerList.count() ; idx1 += 1)
		sizerList.at(idx1)->setState (KBSizer::sbFollower) ;

	sizer->getObject()->setMonitorSelect (true) ;
	sizer->setState (KBSizer::sbActive) ;
	setGUIEnables   () ;

	/* Scan though all sizers to determin the limits on any		*/
	/* movement with their parent.					*/
	for (uint idx2 = 0 ; idx2 < sizerList.count() ; idx2 += 1)
	{
		KBLimit here = sizerList.at(idx2)->getObject()->getMoveLimit() ;

		if (here.left  > limit.left ) limit.left   = here.left  ;
		if (here.right < limit.right) limit.right  = here.right ;
		if (here.up    > limit.up   ) limit.up     = here.up    ;
		if (here.down  < limit.down ) limit.down   = here.down  ;
	}

	return	limit	;
}

/*  KBLayout								*/
/*  trackMove	: Pass movement tracking to active sizers		*/
/*  dx		: int		: X-movement				*/
/*  dy		: int		: Y-movement				*/
/*  move	: bool		: Operation is movement (not resize)	*/
/*  (returns)	: void		:					*/

void	KBLayout::trackMove
	(	int	dx,
		int	dy,
		bool	move
	)
{
	for (uint idx = 0 ; idx < sizerList.count() ; idx += 1)
		sizerList.at(idx)->doResize (dx, dy, move) ;
}

/*  KBLayout								*/
/*  newCtrlrect	: Get new control area					*/
/*  object	: KBObject *	 : object in which control will appear	*/
/*  bState	: Qt::ButtonState: Button state when option selected	*/
/*  defrect	: QRect		 : Default area				*/
/*  (returns)	: QRect		 : New control area			*/

QRect	KBLayout::newCtrlRect
	(	KBObject	*object,
		Qt::ButtonState	bState,
		QRect		defrect
	)
{
	/* If there are no sizers on the list then just return the	*/
	/* default rectangle; nothing else clever we can do.		*/
	if (sizerList.count() == 0)
		return	defrect	;

	/* If the new control is to go into a different object, then	*/
	/* just use the relative position of the existing leading	*/
	/* object.							*/
	if (sizerList.at(0)->getObject()->getParent() != object)
		return	sizerList.at(0)->getPosition () ;

	/* Case: more than one item on the sizer list. In this case	*/
	/* use the size of the most recent, and offset by the offset	*/
	/* between the most recent two.					*/
	if (sizerList.count() > 1)
	{
		QRect	rect0	= sizerList.at(0)->getPosition()    ;
		QRect	rect1	= sizerList.at(1)->getPosition()    ;
		QRect	rect	= rect0 ;

		rect.moveBy    (rect0.left() - rect1.left(),
				rect0.top () - rect1.top ()) ;
		return	rect	;
	}

	/* Case: one item on the sizer list. Use the same size and	*/
	/* offset right, unless the shift key was down in which case	*/
	/* offset down.							*/
	if (sizerList.count() == 1)
	{
		QRect	rect	= sizerList.at(0)->getPosition() ;
		int	space	= KBOptions::getCtrlSpace     () ;

		if ((bState & Qt::ShiftButton) == 0)
			rect.moveBy (rect.width() + space,  0) ;
		else	rect.moveBy (0, rect.height() + space) ;

		return	rect	;
	}

	/* Nothing on the sizer list so use the supplied default ...	*/
	return	defrect	;
}

/*  KBLayout								*/
/*  doCtrlAlign	: Handle control alignment				*/
/*  align	: KBCtrlAlign	: Requested alignment			*/
/*  (returns)	: void		:					*/

void	KBLayout::doCtrlAlign
	(	KB::CtrlAlign	align
	)
{
	if (sizerList.count() < 2) return ;

	QRect	pLeader	= sizerList.at(0)->getPosition() ;
	KBSizer	*sizer	= sizerList.first () ;

	while ((sizer = sizerList.next()) != 0)
	{
		QRect	pOther	= sizer->getPosition() ;

		switch (align)
		{
			case KB::AlignTop    :
				pOther.moveTopLeft
				(	QPoint
					(	pOther .left (),
						pLeader.top  ()
					)
				) ;
				break	;

			case KB::AlignLeft   :
				pOther.moveTopLeft
				(	QPoint
					(	pLeader.left (),
						pOther .top  ()
					)
				) ;
				break	;

			case KB::AlignBottom :
				pOther.moveBottomRight
				(	QPoint
					(	pOther .right (),
						pLeader.bottom()
					)
				) ;
				break	;

			case KB::AlignRight  :
				pOther.moveBottomRight
				(	QPoint
					(	pLeader.right (),
						pOther .bottom()
					)
				) ;
				break	;

			case KB::SameHeight  :
				pOther.setHeight (pLeader.height()) ;
				break	;

			case KB::SameWidth   :
				pOther.setWidth  (pLeader.width ()) ;
				break	;

			case KB::SameSize    :
				pOther.setHeight (pLeader.height()) ;
				pOther.setWidth  (pLeader.width ()) ;
				break	;

			default	:
				break	;
		}

		sizer->getObject()->setGeometry (pOther) ;
		sizer->accept (false)	;
	}
}

/*  KBLayout								*/
/*  setGUIEnables: Enable/disable GUI elements				*/
/*  (returns)	 : void		:					*/

void	KBLayout::setGUIEnables ()
{

	if (currGUI != 0)
	{
		currGUI->setEnabled (KB::GRCopy,  sizerList.count() > 0) ;
		currGUI->setEnabled (KB::GRAlign, sizerList.count() > 1) ;
		currGUI->setEnabled (KB::GRDelta, design ? dDesign : dData) ;
	}
}

/*  KBLayout								*/
/*  doCopy	: Copy selected items					*/
/*  (returns)	: void		:					*/

void	KBLayout::doCopy ()
{
	if (sizerList.count() == 0)
		return ;

	KBFormCopier::self()->clearCopy () ;

	LITER
	(	KBSizer,
		sizerList,
		sizer,

		KBObject *o = sizer->getObject() ;
		KBFormCopier::self()->addToCopy (o->replicate (0), o->objType())
	)
}

/*  KBLayout								*/
/*  doCut	: Cut selected items					*/
/*  (returns)	: void		:					*/

void	KBLayout::doCut ()
{
	if (sizerList.count() > 0)
	{
		setChanged ()	;
		doCopy	   ()	;

		while (sizerList.count() > 0)
			delete	sizerList.at(0)->getObject() ;
	}
}

/*  KBLayout								*/
/*  doPaste	: Paste set of nodes					*/
/*  (returns)	: void			:				*/

void	KBLayout::doPaste ()
{
	if (sizerList.count() == 0)
	{	parent->isObject()->pasteObjects () ;
		return	;
	}

	if (sizerList.count() == 1)
	{	sizerList.at(0)->getObject()->pasteObjects() ;
		return	;
	}

	KBError::EWarning
	(	TR("Cannot paste when several objects are selected"),
		QString::null,
		__ERRLOCN
	) ;
}

/*  KBLayout								*/
/*  doSaveComponent							*/
/*		: Save selected objects as a component			*/
/*  location	: KBLocation &	: Owning documents location		*/
/*  (returns)	: void		:					*/

void	KBLayout::doSaveComponent
	(	KBLocation	&location
	)
{
	QString		 svName	  = location.docLocn ;
	QString		 objName  ;
	QString		 comment  ;

	/* Run the component save-as dialog, which will return the	*/
	/* desired object name, server, and optionally a comment.	*/
	bool		   toFile ;
	KBComponentSaveDlg csDlg
			   (	objName,
				svName,
				comment,
				location.dbInfo,
				&toFile
			   )	;

	if (!csDlg.exec()) return ;


	QList<KBObject> copies	;
	QRect		range	(0, 0, 0, 0) ;

	copies.setAutoDelete	(true) ;

	LITER
	(	KBSizer,
		sizerList,
		sizer,

		KBObject *copy	= (KBObject *)sizer->getObject()->replicate(0) ;
		QRect	 r	= copy->geometry() ;
		range = range.unite  (r) ;

		copies.append (copy) ;
	)

	int	dx	= 20 - range.x() ;
	int	dy	= 20 - range.y() ;

	range.moveBy (dx, dy) ;

	LITER
	(	KBObject,
		copies,
		copy,

		QRect	 r	= copy->geometry() ;
		r.moveBy (dx, dy) ;
		copy->KBObject::move   (r.x(), r.y()) ;
		copy->KBObject::resize (r.width (), r.height()) ;
	)

	extern	QString	kbXMLEncoding  () ;

	QString	 text	= QString
			  (	"<?xml version=\"1.0\" encoding=\"%1\"?>\n"
				"<!DOCTYPE KBaseCompenent SYSTEM \"kbasecomponent.dtd\">\n"
			  	"<KBComponent w=\"%2\" h=\"%3\" type=\"%4\" notes=\"%5\">\n" 
			  )
			  .arg	(kbXMLEncoding())
			  .arg	(range.width () + 40)
			  .arg	(range.height() + 40)
			  .arg	((uint)parent->objType())
			  .arg	(comment) ;


	LITER
	(	KBObject,
		copies,
		copy,
		copy->printNode(text, 2)
	)

	fprintf	  (stderr, "Component:[[[\n%s\n]]]\n", (cchar *)text) ;

	/*  .... and save the component where requested. The special	*/
	/* case is if the used specified save-to-file.			*/
	if (toFile)
	{
		saveComponentToFile (objName, text) ;
		return	;
	}

	KBLocation	newlocn
			(	location.dbInfo,
				"component",
				svName,
				objName
			)	;
	KBError		error	;

	if (!newlocn.save (QString::null, QString::null, text, error))
		error.DISPLAY() ;

}

/*  KBLayout								*/
/*  snapToGrid	: Snap selected controls to the grid			*/
/*  (returns)	: void		:					*/

void	KBLayout::snapToGrid ()
{
	if (parent->showingDesign ())
	{
		LITER
		(	KBSizer,
			sizerList,
			sizer,
			sizer->snapToGrid ()
		)
		setChanged () ;
	}
}

/*  KBLayout	:							*/
/*  doMultiProp	: Invoke multiple-object properties dialog		*/
/*  (returns)	: void		:					*/

void	KBLayout::doMultiProp ()
{
	if (sizerList.count() > 1)
	{
		QList<KBNode> nodeList ;

		for (uint idx = 0 ; idx < sizerList.count() ; idx += 1)
			nodeList.append (sizerList.at(idx)->getObject()) ;

		if (sizerList.at(0)->getObject()->doMultiProp (nodeList))
			setChanged (true) ;
	}
}

/*  KBLayout	:							*/
/*  doSingleProp: Invoke object properties dialog			*/
/*  (returns)	: void		:					*/

void	KBLayout::doSingleProp ()
{
	/* HACK ALERT:							*/
	/* Changed to invode the design popup menu. This routine should	*/
	/* be renamed.							*/
	if (sizerList.count() > 0)
	{
		QMouseEvent me
			    (	QEvent::MouseButtonPress,
				QPoint(),
				QCursor::pos(),
				QMouseEvent::RightButton,
				QMouseEvent::RightButton
			    )	;

		sizerList.at(0)->doDesignPopup (&me) ;
	}
}

/*  KBLayout								*/
/*  clear	: Clear layout						*/
/*  _design	: bool		: Design mode				*/
/*  (returns)	: void		:					*/

void	KBLayout::clear
	(	bool	_design
	)
{
	sizerList.clear ()	;
	if (!design) setChanged (false) ;
	design	= _design	;
}

#endif

/*  KBLayout								*/
/*  setChanged	: Set document changed status				*/
/*  changed	: bool		: True if changed			*/
/*  (returns)	: void		:					*/

void	KBLayout::setChanged
	(	bool	changed
	)
{
	bool	 &rChanged = design ? dDesign : dData ;

	if (changed == rChanged) return ;

	rChanged = changed ;
	if (currGUI != 0) currGUI->setEnabled (KB::GRDelta, changed) ;
}

/*  KBLayout								*/
/*  getChanged	: Get object changed status				*/
/*  both	: bool		: Check both data and design changes	*/
/*  (returns)	: cchar *	: Change message or null if none	*/

cchar	*KBLayout::getChanged
	(	bool	both
	)
{
	if (both)
		if (dDesign && dData)
			return	"data and design" ;

	if (( design || both) && dDesign) return "design"  ;
	if ((!design || both) && dData  ) return "data"	   ;

	return	0 ;
}



/*  KBLayout								*/
/*  setUnMorphedItem							*/
/*		: Note which control is not morphed			*/
/*  item	: KBItem *	: Item or null				*/
/*  drow	: uint		: Display row number			*/
/*  (returns)	: void		:					*/

void	KBLayout::setUnMorphedItem
	(	KBItem		*item,
		uint		drow
	)
{
//	fprintf
//	(	stderr,
//		"KBLayout::setUnMorphedItem: [%s][%d]\n",
//		item == 0 ? "NONE" : (cchar *)item->getName(),
//		drow
//	)	;

	/* NOTE: This routine keeps track of the unmorphed control when	*/
	/* there are one or more morphing control in a form.		*/
	if (unMorphedItem != 0)
		if ((unMorphedItem != item) || (unMorphedDRow != drow))
			unMorphedItem->reMorphItem (unMorphedDRow) ;

	unMorphedItem = item ;
	unMorphedDRow = drow ;
}
