/***************************************************************************
    file	         : kb_compaccessdlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>

#include	<qapplication.h>
#include	<qdir.h>
#include	<qdict.h>

#include	"kb_dom.h"
#include	"kb_location.h"
#include	"kb_dbdociter.h"
#include	"kb_component.h"
#include	"kb_config.h"
#include	"kb_nodereg.h"
#include	"kb_parse.h"
#include	"kb_component.h"
#include	"kb_ascii.h"
#include	"kb_display.h"
#include	"kb_attrdlg.h"
#include	"kb_font.h"
#include	"kb_locator.h"

#include	"tk_filedialog.h"
#include	"tk_fontdialog.h"
#include	"tk_colordialog.h"
#include	"tk_messagebox.h"

#ifndef		_WIN32
#include	"kb_compaccessdlg.moc"
#else
#include	"kb_compaccessdlg.h"
#endif


static	QString	lastServer	;

static	QString	localDir ()
{
	return	QDir::homeDirPath() + "/.rekall/" ;
}


/*  ------------------------------------------------------------------  */

/*  KBWizardAttrDlg							*/
/*  KBWizardAttrDlg							*/
/*		: Constructor for wizard control with attr dialog	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  legend	: const QString & : Legend				*/
/*  defval	: const QString & : Default value			*/
/*  attrDlg	: KBAttrDlg *	  : Special attribute dialog		*/
/*  (returns)	: KBWizardLineEdit:					*/

KBWizardAttrDlg::KBWizardAttrDlg
	(	KBWizardPage	*page,
		const QString	&name,
		const QString	&legend,
		const QString	&defval,
		KBAttrDlg	*attrDlg
	)
	:
	KBWizardCtrl	(page, name)
{
	QWidget		*cntr	= new QWidget	 (page) ;
	QPushButton	*btn	= new QPushButton(cntr) ;

	m_attrDlg	= new KBAttrDlgDialog	 (attrDlg, legend) ;
	m_lineEdit	= new TKLineEdit 	 (cntr)	   ;
	m_value		= defval ;

	m_attrDlg ->init    	 (m_value) ;
	m_lineEdit->setText 	 (m_attrDlg->displayValue ()) ;
	m_lineEdit->setReadOnly  (true)    ;
	m_lineEdit->setEraseColor(qApp->palette().disabled().mid()) ;
	setWidget (cntr) ;

	QHBoxLayout	*lay	= new QHBoxLayout(cntr) ;
	btn->setFixedHeight (m_lineEdit->sizeHint().height()) ;
	btn->setText	    ("...") ;

	lay->addWidget (m_lineEdit) ;
	lay->addWidget (btn) ;

	connect	(btn, SIGNAL(clicked()), SLOT(slotClickDlg())) ;
}

/*  KBWizardAttrDlg							*/
/*  ~KBWizardAttrDlg							*/
/*		: Destructor for attribute dialog control		*/
/*  (returns)	:		:					*/

KBWizardAttrDlg::~KBWizardAttrDlg ()
{
	delete	m_attrDlg ;
}

/*  KBWizardAttrDlg							*/
/*  slotClickDlg: User clicks dialog button				*/
/*  (returns)	: void		:					*/

void	KBWizardAttrDlg::slotClickDlg ()
{
	m_attrDlg->init (m_value) ;

	if (m_attrDlg->exec())
	{	m_attrDlg ->verify  () ;
		m_lineEdit->setText (m_attrDlg->displayValue()) ;
		m_value = m_attrDlg->value() ;
		ctrlChanged() ;
	}
}

/*  KBWizardAttrDlg							*/
/*  setValue	: Set value						*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBWizardAttrDlg::setValue
	(	const QString	&value
	)
{
	m_value	= value	;
	m_attrDlg ->init    (m_value) ;
	m_lineEdit->setText (m_attrDlg->displayValue ()) ;
}

/*  KBWizardAttrDlg							*/
/*  value	: Get value						*/
/*  (returns)	: QString	: Value					*/

QString	KBWizardAttrDlg::value ()
{
	return	m_value	;
}



/*  ------------------------------------------------------------------  */

/*  KBWizardModalCtrl							*/
/*  KBWizardModalCtrl							*/
/*		: Constructor for wizard control using a modal dialog	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  legend	: const QString & : Legend				*/
/*  defval	: const QString & : Default value			*/
/*  (returns)	: KBWizardLineEdit:					*/

KBWizardModalCtrl::KBWizardModalCtrl
	(	KBWizardPage	*page,
		const QString	&name,
		const QString	&,
		const QString	&defval
	)
	:
	KBWizardCtrl	(page, name)
{
	QWidget		*cntr	= new QWidget	 (page) ;
	QPushButton	*btn	= new QPushButton(cntr) ;

	m_lineEdit	= new TKLineEdit 	 (cntr)	;
	m_value		= defval ;

	m_lineEdit->setText 	 (m_value) ;
	m_lineEdit->setReadOnly  (true)    ;
	m_lineEdit->setEraseColor(qApp->palette().disabled().mid()) ;
	setWidget (cntr) ;

	QHBoxLayout	*lay	= new QHBoxLayout(cntr) ;
	btn->setFixedHeight (m_lineEdit->sizeHint().height()) ;
	btn->setText	    ("...") ;

	lay->addWidget (m_lineEdit) ;
	lay->addWidget (btn) ;

	connect	(btn, SIGNAL(clicked()), SLOT(slotClickDlg())) ;
}

/*  KBWizardModalCtrl							*/
/*  ~KBWizardModalCtrl							*/
/*		: Destructor for modal dialog control			*/
/*  (returns)	:		:					*/

KBWizardModalCtrl::~KBWizardModalCtrl ()
{
}

/*  KBWizardModalCtrl							*/
/*  setValue	: Set value						*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBWizardModalCtrl::setValue
	(	const QString	&value
	)
{
	m_lineEdit->setText (m_value = value) ;
}

/*  KBWizardModalCtrl							*/
/*  value	: Get value						*/
/*  (returns)	: QString	: Value					*/

QString	KBWizardModalCtrl::value ()
{
	return	m_value	;
}

/*  ------------------------------------------------------------------  */

/*  KBWizardColorCtrl							*/
/*  KBWizardColorCtrl							*/
/*		: Constructor for wizard control with color dialog	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  legend	: const QString & : Legend				*/
/*  defval	: const QString & : Default value			*/
/*  (returns)	: KBWizardLineEdit:					*/

KBWizardColorCtrl::KBWizardColorCtrl
	(	KBWizardPage	*page,
		const QString	&name,
		const QString	&legend,
		const QString	&defval
	)
	:
	KBWizardModalCtrl	(page, name, legend, defval)
{
}

/*  KBWizardColorCtrl							*/
/*  ~KBWizardColorCtrl							*/
/*		: Destructor for wizard control with color dialog	*/
/*  (returns)	:		:					*/

KBWizardColorCtrl::~KBWizardColorCtrl ()
{
}

/*  KBWizardColorCtrl							*/
/*  slotClickDlg: User clicks dialog button				*/
/*  (returns)	: void		:					*/

void	KBWizardColorCtrl::slotClickDlg ()
{
	TKColorDialog cDialog (0, TR("Colour"), true) ;
	cDialog.setColor (QColor (strtol(m_value, 0, 0))) ;
	if (cDialog.exec ())
	{
		m_value.sprintf ("0x%06x", cDialog.color().rgb() & 0x00ffffff) ;
		m_lineEdit->setText (m_value) ;
		ctrlChanged() ;
	}
}


/*  ------------------------------------------------------------------  */

/*  KBWizardFontCtrl							*/
/*  KBWizardFontCtrl							*/
/*		: Constructor for wizard control with font dialog	*/
/*  page	: KBWizardPage *  : Parent page				*/
/*  name	: const QString & : Control name			*/
/*  legend	: const QString & : Legend				*/
/*  defval	: const QString & : Default value			*/
/*  (returns)	: KBWizardLineEdit:					*/

KBWizardFontCtrl::KBWizardFontCtrl
	(	KBWizardPage	*page,
		const QString	&name,
		const QString	&legend,
		const QString	&defval
	)
	:
	KBWizardModalCtrl	(page, name, legend, defval)
{
}

/*  KBWizardFontCtrl							*/
/*  ~KBWizardFontCtrl							*/
/*		: Destructor for attribute dialog control		*/
/*  (returns)	:		:					*/

KBWizardFontCtrl::~KBWizardFontCtrl ()
{
}

/*  KBWizardFontCtrl							*/
/*  slotClickDlg: User clicks dialog button				*/
/*  (returns)	: void		:					*/

void	KBWizardFontCtrl::slotClickDlg ()
{
	TKFontDialog  fDialog (0, TR("Font"), false, true) ;
	fDialog.setFont (KBFont::specToFont (m_value)) ;
	if (fDialog.exec())
	{
		m_value = KBFont::fontToSpec (fDialog.font()) ;
		m_lineEdit->setText (m_value) ;
		ctrlChanged() ;
	}
}

/*  ------------------------------------------------------------------  */

/*  KBComponentWizardPage						*/
/*  addAttrDlg	: Add an attribute dialog control			*/
/*  name	: cchar *	     : Control name			*/
/*  legend	: const QString &    : Legend for display		*/
/*  defval	: const QString &    : Default value			*/
/*  dlg		: KBAttrDlg *	     : Special purpose attribute dialog	*/
/*  (returns)	: KBWizardAttrDlg *  :					*/

KBWizardAttrDlg
	*KBComponentWizardPage::addAttrDlg
	(	const QString	&name,
		const QString	&legend,
		const QString	&defval,
		KBAttrDlg	*attrDlg
	)
{
	KBWizardAttrDlg	*ctrl	= new KBWizardAttrDlg
				  (	this,
					name,
					legend,
					defval,
					attrDlg
				  )	;
	QLabel		*label	= new QLabel (this) ;

	m_layout->addWidget (label,	     m_ctrls.count(), 0) ;
	m_layout->addWidget (ctrl->widget(), m_ctrls.count(), 1) ;
	m_ctrls . append    (ctrl)  ;
	m_labels. append    (label)  ;

	label   ->setText   (legend) ;

	return	ctrl	;
}

/*  KBComponentWizardPage						*/
/*  addColorDlg	: Add a color dialog control				*/
/*  name	: cchar *	     : Control name			*/
/*  legend	: const QString &    : Legend for display		*/
/*  defval	: const QString &    : Default value			*/
/*  (returns)	: KBWizardColorDlg * :					*/

KBWizardColorCtrl
	*KBComponentWizardPage::addColorCtrl
	(	const QString	&name,
		const QString	&legend,
		const QString	&defval
	)
{
	KBWizardColorCtrl *ctrl	= new KBWizardColorCtrl
				  (	this,
					name,
					legend,
					defval
				  )	;
	QLabel		*label	= new QLabel (this) ;

	m_layout->addWidget (label,	     m_ctrls.count(), 0) ;
	m_layout->addWidget (ctrl->widget(), m_ctrls.count(), 1) ;
	m_ctrls . append    (ctrl)  ;
	m_labels. append    (label)  ;

	label   ->setText   (legend) ;

	return	ctrl	;
}

/*  KBComponentWizardPage						*/
/*  addFontDlg	: Add a font dialog control				*/
/*  name	: cchar *	     : Control name			*/
/*  legend	: const QString &    : Legend for display		*/
/*  defval	: const QString &    : Default value			*/
/*  (returns)	: KBWizardColorDlg * :					*/

KBWizardFontCtrl
	*KBComponentWizardPage::addFontCtrl
	(	const QString	&name,
		const QString	&legend,
		const QString	&defval
	)
{
	KBWizardFontCtrl *ctrl	= new KBWizardFontCtrl
				  (	this,
					name,
					legend,
					defval
				  )	;
	QLabel		*label	= new QLabel (this) ;

	m_layout->addWidget (label,	     m_ctrls.count(), 0) ;
	m_layout->addWidget (ctrl->widget(), m_ctrls.count(), 1) ;
	m_ctrls . append    (ctrl)  ;
	m_labels. append    (label)  ;

	label   ->setText   (legend) ;

	return	ctrl	;
}

/*  ------------------------------------------------------------------  */

/*  KBComponentLoadDlg							*/
/*  KBComponentLoadDlg							*/
/*		: Constructor for component selection dialog		*/
/*  dbInfo	: KBDBInfo *	    : Database information		*/
/*  svName	: const QString &   : Server name			*/
/*  language	: const QString &   : Scripting language		*/
/*  size	: QSize		    : Requested size			*/
/*  paste	: bool		    : Component is being pasted		*/
/*  reqdType	: KBComponent::Type : Required type			*/
/*  (returns)	: KBDocChooserDlg   :					*/

KBComponentLoadDlg::KBComponentLoadDlg
	(	KBDBInfo		*dbInfo,
		const QString		&svName,
		const QString		&language,
		QSize			size,
		bool			paste,
		KB::ObjType		reqdType
	)
	:
	_KBDialog	(TR("Pick component"), true),
	m_dbInfo	(dbInfo),
	m_svName	(svName),
	m_language	(language),
	m_tab		(this),

#if	__KB_EMBEDDED
	m_pick		(&m_tab),
#else
	m_pick		(this),
#endif
	m_cbServer	(&m_pick),
	m_docStack	(&m_pick),
	m_lbDocument 	(&m_docStack),
	m_lvStock 	(&m_docStack),
	m_lvLocal 	(&m_docStack),
	m_bOK		(&m_pick),
	m_bCancel	(&m_pick),

	m_comment	(&m_tab),
	m_wizard	(&m_tab),
	m_blank		(&m_wizard),
	m_example	(&m_tab),
	m_size		(size),
	m_paste		(paste),
	m_reqdType	(reqdType)
{
	QHBoxLayout	*layMain  = new QHBoxLayout (this) ;

#if	__KB_EMBEDDED
	layMain ->addWidget (&m_tab	) ;
#else
	QVBoxLayout	*layLeft  = new QVBoxLayout (layMain) ;
	QVBoxLayout	*layRight = new QVBoxLayout (layMain) ;

	layLeft ->addWidget (&m_pick    ) ;
	layRight->addWidget (&m_tab	) ;

	layMain ->setStretchFactor (layLeft,  0) ;
	layMain ->setStretchFactor (layRight, 1) ;
#endif

	QVBoxLayout	*layPick  = new QVBoxLayout (&m_pick) ;

	layPick ->addWidget (&m_cbServer) ;
	layPick ->addWidget (&m_docStack) ;

	QHBoxLayout	*layButt  = new QHBoxLayout (layPick) ;
	layButt ->addStretch()		  ;
	layButt ->addWidget (&m_bCancel ) ;
	layButt ->addWidget (&m_bOK	) ;


	m_bCancel  .setText (TR("Cancel"  )) ;
	m_bOK      .setText (TR("OK"      )) ;


#if	__KB_EMBEDDED
	m_tab	   .addTab	  (&m_pick,    TR("Pick"       )) ;
#endif
	m_tab	   .addTab	  (&m_comment, TR("Description")) ;
	m_tab      .addTab	  (&m_example, TR("Preview"    )) ;
	m_tab      .addTab	  (&m_wizard,  TR("Configure"  )) ;
	m_tab	   .setTabEnabled (&m_wizard,  false) ;
	m_tab	   .setTabEnabled (&m_example, false) ;


	m_comment   .setTextFormat	(Qt::RichText) ;
	m_comment   .setMinimumSize	(300, 200) ;

	m_docStack  .addWidget		(&m_lbDocument, -1) ;
	m_docStack  .addWidget		(&m_lvStock,	-1) ;

	m_lvStock.addColumn		(TR("Component")) ;
	m_lvStock.setRootIsDecorated    (true) ;
	m_lvLocal.addColumn		(TR("Component")) ;
	m_lvLocal.setRootIsDecorated 	(true) ;

	m_blank     .setMinimumSize	(300, 1) ;

	/* Load the servers into the server combo box. If requested,	*/
	/* an entry for stock components is prefixed to the list.	*/
	QListIterator<KBServerInfo> *svIter = dbInfo->getServerIter () ;
	KBServerInfo		    *svInfo ;
	int			    at	    = -1 ;

	if (lastServer.isEmpty())
		lastServer = KBLocation::m_pStock ;

	if (m_paste)
	{
		m_cbServer.insertItem (TR(KBLocation::m_pStock)) ;
		m_cbServer.insertItem (TR("Local Components"  )) ;
		if (svName == lastServer) at = 0 ;
	}

	m_cbServer.insertItem (KBLocation::m_pFile) ;

	while ((svInfo = svIter->current()) != 0)
	{
		if (svInfo->serverName() == lastServer)
			at = m_cbServer.count() ;

		m_cbServer.insertItem (svInfo->serverName()) ;
		(*svIter) += 1 ;
	}

	delete	svIter	;
	if (at >= 0) m_cbServer.setCurrentItem (at) ;

	/* Locate the stock compenent directory. This will probably be	*/
	/* needed so we get it once and for all.			*/
	m_stockDir = locateDir
		     (	"appdata",
			"stock/component/" + m_language + "/dummy"
		     )
		     +	"stock/component/" + m_language ;

	fprintf
	(	stderr,
		"KBComponentLoadDlg: stockDir [%s]->[%s] size [%s]\n",
		(cchar *)m_language,
		(cchar *)m_stockDir,
		_TEXT(m_size)
	)	;

	m_currPage  = 0	  	 ;
	m_selected  = false	 ;
	m_component = 0		 ;
	m_rendered  = 0		 ;
	m_topWidget = 0		 ;
	m_bOK.setEnabled (false) ;

	serverSelected  (m_cbServer.currentText ()) ;

	connect	(&m_cbServer,   SIGNAL(activated       (const QString &)),
		 	        SLOT  (serverSelected  (const QString &))) ;
	connect	(&m_lbDocument, SIGNAL(highlighted     (const QString &)),
		 	        SLOT  (documentSelected(const QString &))) ;
	connect	(&m_lvStock,    SIGNAL(clicked         (QListViewItem *)),
		 	        SLOT  (stockSelected   (QListViewItem *))) ;
	connect	(&m_lvLocal,    SIGNAL(clicked         (QListViewItem *)),
		 	        SLOT  (localSelected   (QListViewItem *))) ;
	connect	(&m_tab,	SIGNAL(currentChanged  (QWidget	      *)),
				SLOT  (tabPageChanged  (QWidget	      *))) ;

	connect	(&m_bOK,	SIGNAL(clicked	       ()),
				SLOT  (clickOK	       ())) ;
	connect	(&m_bCancel,	SIGNAL(clicked	       ()),
				SLOT  (clickCancel     ())) ;

	m_example.installEventFilter (this) ;
	qApp    ->installEventFilter (this) ;

#if	__KB_EMBEDDED
	m_showMaxed = true ;
#endif
}

/*  KBComponentLoadDlg							*/
/*  ~KBComponentLoadDlg							*/
/*		: Destructor for component selection dialog		*/
/*  (returns)	:		:					*/

KBComponentLoadDlg::~KBComponentLoadDlg ()
{
	DELOBJ	(m_component) ;
	DELOBJ	(m_rendered ) ;
}

/*  KBComponentLoadDlg							*/
/*  eventFilter	: Local event filtering					*/
/*  o		: QObject *	: Target object				*/
/*  e		: QEvent  *	: Event					*/
/*  (returns)	: bool		: True to continue processing		*/

bool	KBComponentLoadDlg::eventFilter
	(	QObject		*o,
		QEvent		*e
	)
{
	/* Check for resize events going to the example widget, in	*/
	/* order to do a matching resize on the top-level widget.	*/
	if (o == &m_example)
	{
		if (e->type() == QEvent::Resize)
			if (m_topWidget != 0)
				m_topWidget->resize (m_example.size()) ;

		return	false	;
	}

	/* We must also prevent the displayed component actually doing	*/
	/* anything! Hence, filter offending events for any widget that	*/
	/* is a child of the example widget.				*/
	if (o->isWidgetType()) 
		for (QWidget *w = (QWidget *)o ; w != 0 ; w = w->parentWidget())
			if (w == &m_example)
				switch (e->type())
				{
					case QEvent::MouseButtonPress    :
					case QEvent::MouseButtonRelease  :
					case QEvent::MouseButtonDblClick :
					case QEvent::KeyPress		 :
					case QEvent::KeyRelease		 :
					case QEvent::Wheel		 :
					case QEvent::FocusIn		 :
					case QEvent::FocusOut		 :
						return	true	;

					default	:
						return	false	;
				}

	return	false	;
}

/*  KBComponentLoadDlg							*/
/*  getStockComponents							*/
/*		: Get stock components and fill component tree		*/
/*  path	: const QString & : Path to component directory		*/
/*  listView	: QListView &	  : Parent list view			*/
/*  parent	: QListViewItem * : Parent item or null at top level	*/
/*  (returns)	: void		  :					*/

void	KBComponentLoadDlg::getStockComponents
	(	const QString	&path,
		QListView	&listView,
		QListViewItem	*parent
	)
{
	QDir		    dir	   ;
	const QFileInfoList *eList ;

	fprintf
	(	stderr,
		"KBComponentLoadDlg::getStockComponents: [%s]\n",
		(cchar *)path
	)	;

	/* First scan through the directory locates subdirectories. For	*/
	/* each add a new entry and recurse down.			*/
	dir.setPath	  (path)	;
	dir.setNameFilter ("*" )	;
	dir.setFilter	  (QDir::Dirs)	;
	dir.setSorting	  (QDir::Name)	;

	if ((eList = dir.entryInfoList()) != 0)
	{
		QFileInfoListIterator eIter (*eList) ;
		QFileInfo *entry ;

		while ((entry = eIter.current()) != 0)
		{
			QString	name = entry->fileName() ;
			eIter	+= 1  ;

			if (name[0] == '.') continue ;

			QListViewItem *i ;
			if (parent == 0)
				i = new QListViewItem (&listView, name) ;
			else	i = new QListViewItem (parent,	  name) ;

			getStockComponents (path + "/" + name, listView, i) ;
		}
	}

	/* Second scan locates component files in the directory, and	*/
	/* adds an entry for each such.					*/
	dir.setPath	  (path)	;
	dir.setNameFilter ("*.cmp" )	;
	dir.setFilter	  (QDir::Files) ;
	dir.setSorting	  (QDir::Name ) ;

	if ((eList = dir.entryInfoList()) != 0)
	{
		QFileInfoListIterator eIter (*eList) ;
		QFileInfo *entry ;

		while ((entry = eIter.current()) != 0)
		{
			QString	name = entry->baseName() ;
			eIter	+= 1 ;

			if (name[0] == '.') continue ;

			if (parent == 0)
				new QListViewItem (&listView, name) ;
			else	new QListViewItem (parent,    name) ;
		}
	}
}

/*  KBComponentLoadDlg							*/
/*  serverSelected							*/
/*		: User selects a database server			*/
/*  svName	: const QString & : Server name				*/
/*  (returns)	: void		  :					*/

void	KBComponentLoadDlg::serverSelected
	(	const QString	&svName
	)
{
	m_lbDocument.clear   () ;
	m_lvStock   .clear   () ;
	m_comment   .setText (QString::null) ;

#if	! __KB_EMBEDDED
	m_tab.showPage      (&m_comment) ;
#endif
	m_tab.changeTab     (&m_comment, TR("Description")) ;
	m_tab.setTabEnabled (&m_example, false) ;
	m_tab.setTabEnabled (&m_wizard,  false) ;

	m_selected  = false   ;
	m_bOK.setEnabled (false) ;

	/* See if the user has selected the notional stock components	*/
	/* server, in which case raise the stock components list view	*/
	/* and popuplate it the first time round; and similarly for	*/
	/* local components.						*/
	if (m_paste)
	{

		if (m_cbServer.currentItem() == 0)
		{
			m_docStack.raiseWidget (&m_lvStock) ;

			if (m_lvStock.childCount() == 0)
				getStockComponents  (m_stockDir, m_lvStock, 0) ;

			return	;
		}

		if (m_cbServer.currentItem() == 1)
		{
			m_docStack.raiseWidget (&m_lvLocal) ;

			if (m_lvLocal.childCount() == 0)
				getStockComponents  (localDir(), m_lvLocal, 0) ;

			return	;
		}
	}

	/* Otherwise show the list view and populate that from the	*/
	/* selected server.						*/
	QString	    name    ;
	QString	    stamp   ;
	KBError	    error   ;

	KBDBDocIter docIter ;
	if (!docIter.init (m_dbInfo, svName, "component", "cmp", error))
	{	error.DISPLAY() ;
		return	;
	}

	while (docIter.getNextDoc (name, stamp))
		m_lbDocument.insertItem (name) ;

	documentSelected (m_lbDocument.text(0)) ;
	m_docStack.raiseWidget  (&m_lbDocument) ;
}

/*  KBComponentLoadDlg							*/
/*  getAllConfigs: Get all substituted configuration objects		*/
/*  object	 : KBObject *	     : Object from which to search	*/
/*  configSet	 : QList<KBConfig> & : Result list			*/
/*  fixup	 : bool		     : Fix width and height		*/
/*  changed	 : bool		     : Only process changed settings	*/
/*  (returns)	 : void		     :					*/

void	KBComponentLoadDlg::getAllConfigs
	(	KBObject	*object,
		QList<KBConfig>	&configSet,
		bool		fixup,
		bool		changed
	)
{
	/* Scan the object to find all configuration objects that are	*/
	/* embedded within it, and retrieve all the settings from the	*/
	/* wizard page.							*/
	QDict<QString> configVals ;

	object    ->findAllConfigs (configSet, QString::null) ;
	m_currPage->settings	   (configVals, changed) ;

	fprintf
	(	stderr,
		"KBComponentLoadDlg::getAllConfigs: fix=%d chg=%d\n",
		fixup,
		changed
	)	;
	LITER
	(	KBConfig,
		configSet,
		config,
		fprintf
		(	stderr,
			"..... KBConfig [%s][%s][%s]\n",
			(cchar *)config->path  (),
			(cchar *)config->attrib(),
			(cchar *)config->value ()
		)
	)
	QDictIterator<QString> iter (configVals) ;
	QString	*e ;
	while ((e = iter.current()) != 0)
	{	fprintf
		(	stderr,
			"..... value   [%s][%s]\n",
			(cchar *)iter.currentKey(),
			(cchar *)*e
		)	;
		iter += 1 ;
	}


	/* Load the settings. There are two special cases, width and	*/
	/* height, otherwise look up the value to be used in the	*/
	/* results from the wizard page. Note than in a particular	*/
	/* component, we assume that the identifier property uniquely	*/
	/* identifies a configuration object.				*/
	LITER
	(	KBConfig,
		configSet,
		config,

		if	(fixup && (config->attrib() == "w"))
		{
			if (m_size.width () > 0)
			{
				config->setValue   (QString("%1").arg(m_size.width ())) ;
				config->setChanged (true) ;
			}
		}
		else if (fixup && (config->attrib() == "h"))
		{
			if (m_size.height() > 0)
			{
				config->setValue   (QString("%1").arg(m_size.height())) ;
				config->setChanged (true) ;
			}
		}
		else
		{
			QString	*value	= configVals.find(config->ident()) ;
			if (value != 0)
				if (config->value() != *value)
				{	config->setValue   (*value) ;
					config->setChanged (true  ) ;
				}
		}
	)

	/* Done. Dump the wizard results and return. The argument list	*/
	/* contains the configuration object along with their values.	*/
	/* Also, the paths will have been set.				*/
	configVals.clear () ;
}

/*  KBComponentLoadDlg							*/
/*  substitute	: Render specified component with current values	*/
/*  component	: KBComponent *	: Component to render			*/
/*  (returns)	: void		  :					*/

void	KBComponentLoadDlg::substitute
	(	KBComponent	*component
	)
{
	/* Scan the component to find all configuration objects that	*/
	/* are embedded in it, and get the settings back from the	*/
	/* wizard page which can then be substituted into the component	*/
	QList<KBConfig>	configSet   ;
	getAllConfigs  (component, configSet, true, false)  ;

	LITER
	(	KBConfig,
		configSet,
		config,

		config->substitute (false) ;
		if (!config->user()) delete config ;
	)
}

/*  KBComponentLoadDlg							*/
/*  render	: Render currently selected component and values	*/
/*  component	: KBComponent *	: Component to render			*/
/*  (returns)	: void		  :					*/

void	KBComponentLoadDlg::render
	(	KBComponent	*component
	)
{
	DELOBJ	   (m_rendered) ;
	m_rendered = (KBComponent *)component->replicate(0)  ;
	substitute (m_rendered) ;

	/* We need to set the minimum size of the wizard stack to the	*/
	/* minimum size of all pages on the stack, so take the current	*/
	/* minimum, and expand it to the size hint for the new page.	*/
	QSize	s1	= m_wizard.visibleWidget()->minimumSize() ;
	QSize	s2	= m_currPage->sizeHint () ;
	QSize	size	;

	m_wizard.setMinimumSize (s1.expandedTo(s2)) ;
	m_wizard.raiseWidget    (m_currPage) ;
	m_rendered->showData	(&m_example,  size) ;

	m_topWidget = m_rendered->getDisplay()->getTopWidget() ;

	m_example.addWidget (m_topWidget, -1) ;
	m_topWidget->show   ()	 ;
}

/*  KBComponentLoadDlg							*/
/*  addSpecialConfig							*/
/*		: Check for special configuration cases			*/
/*  config	: KBConfig *	: Configuration				*/
/*  (returns)	: bool		: Special case handled			*/

bool	KBComponentLoadDlg::addSpecialConfig
	(	KBConfig	*config
	)
{
	static	QDict<KBAttrItem> dummy  ;

	/* Start by seeing if the attribute that the configuration	*/
	/* object modifies has an explicit dialog. If so we add a	*/
	/* control to the wizard page that handles modification via	*/
	/* the dialog.							*/
	QString		value	= config->value () ;
	QString		aname	= config->attrib() ;
	KBAttr	  	*attr	= config->getParent()->getAttr(aname) ;
	KBAttrDlg 	*dlg	= attr == 0 ? 0 : attr->getAttrDlg (0, 0, dummy) ;

	fprintf
	(	stderr,
		"KBComponentLoadDlg::addSpecialConfig: [%s][%s]->[%p][%p]\n",
		(cchar *)aname,
		(cchar *)value,
		(void  *)attr,
		(void  *)dlg
	)	;

	if (dlg != 0)
	{
		/* Hack! If not pasting and the attribute is an event	*/
		/* then the value is not used. This is since event code	*/
		/* inherits the component code; if the user leaves the	*/
		/* value empty then the event code will be used as-is.	*/
		if (!m_paste && (attr->isEvent() != 0)) value = QString::null ;

		m_currPage->addAttrDlg
		(	config->ident (),
			config->legend(),
			value,
			dlg
		)	;

		return	true	;
	}

	/* Next check for cases where the attribute is handled by one	*/
	/* of the modal dialog cases ...				*/
	if ((aname == "fgcolor") || (aname == "bgcolor"))
	{
		m_currPage->addColorCtrl
		(	config->ident (),
			config->legend(),
			value
		)	;

		return	true	;
	}

	if (aname == "font")
	{
		m_currPage->addFontCtrl
		(	config->ident (),
			config->legend(),
			value
		)	;

		return	true	;
	}

	/* Special cases are denoted by the value having the form	*/
	/* "tag:ident[:ident....]" (these are used in some of the	*/
	/* hand-crafted predefined stock components), so start by	*/
	/* splitting and rejecting anything that does not fit at all.	*/
	QStringList	bits	= QStringList::split (":", value) ;
	if (bits.count() < 2) return false ;


	/* First case will result in a combobox showing objects of a	*/
	/* specified type. The identifier denotes the type of object	*/
	/* that we are interested in.					*/
	if (bits[0] == "object")
	{
		static	cchar	*objectList[] =
		{
			"form",		"form",		"frm",
			"report",	"report",	"rep",
			"query",	"query",	"qry",
			0
		}	;

		/* Scan the object list for the specified type. If not	*/
		/* matched then silently return failure.		*/
		cchar	*type	= 0 ;
		cchar	*extn	= 0 ;

		for (cchar **ap = &objectList[0] ; *ap != 0 ; ap += 3)
			if (*ap == bits[1])
			{	type	= ap[1] ;
				extn	= ap[2] ;
				break	;
			}

		if (type == 0)
			return	false	;


		/* Run a document iterator for the specified type. If	*/
		/* this fails or yeilds no objects then just add a	*/
		/* default edit field.					*/
		KBDBDocIter	docIter	;
		KBError		error	;

		if (!docIter.init (m_dbInfo, m_svName, type, extn, error, false))
			goto	def	;

		QString		name	;
		QString		stamp	;
		QStringList	forms	;

		while (docIter.getNextDoc (stamp, name))
			forms.append (stamp) ;

		if (forms.count() == 0)
			goto	def	;

		/* Success, create a choice field with the derived list	*/
		/* of objects.						*/
		m_currPage->addChoiceCtrl
		(	config->ident (),
			config->legend(),
			forms,
			QString::null,
			true
		)	;

		return	true	;
	}

	/* Tag unrecognised, silently fail. The caller will simply show	*/
	/* an edit field.						*/
	return	false	;


	/* Default case, for instance when there are no objects to list	*/
	/* in which case fall back on an edit control, except that the	*/
	/* value is initially null.					*/
	def :
		m_currPage->addTextCtrl
		(	config->ident (),
			config->legend(),
			QString::null
		)	;
		return	true	;
}

/*  KBComponentLoadDlg							*/
/*  showDetails	: Show details and customisations for component		*/
/*  (returns)	: void		  :					*/

void	KBComponentLoadDlg::showDetails ()
{
	QDomDocument	dXML	;
	KBError		error	;
	QByteArray	data	;


	/* Clear any existing comment and raise the blank widget so	*/
	/* that existing customisation settings are hidden. Then get	*/
	/* the component definition text, and return if there is an	*/
	/* error.							*/
	DELOBJ (m_component)	;
	DELOBJ (m_rendered )	;
	m_topWidget = 0		;

	m_comment  .setText	(QString::null) ;
	m_wizard   .raiseWidget (&m_blank) ;

	if (!text (data, error))
	{	error.DISPLAY () ;
		m_currPage = 0	 ;
		return	;
	}

	/* Parse the component into an object tree. The top-level	*/
	/* component object should specify the type (form, report, ...)	*/
	/* with the notes being interpreted as a descriptive comment.	*/
	KBComponent *component = KBOpenComponentText (m_location, data, error) ;
	if (component == 0)
	{
		error.DISPLAY()	;
		return	;
	}

	switch (m_showType = component->objType())
	{
		case KB::ObjForm	:
#if	__KB_EMBEDDED
			m_tab.changeTab (&m_comment, "Form"  ) ;
#else
			m_tab.changeTab (&m_comment, "Form component"  ) ;
#endif
			break	;

		case KB::ObjReport:
#if	__KB_EMBEDDED
			m_tab.changeTab (&m_comment, "Report") ;
#else
			m_tab.changeTab (&m_comment, "Report component") ;
#endif
			break	;

		default	:
#if	__KB_EMBEDDED
			m_tab.changeTab (&m_comment, "Unknown") ;
#else
			m_tab.changeTab (&m_comment, "Unknown component type") ;
#endif
	}

	m_comment.setText (component->getAttrVal("notes")) ;


	/* See if there is an extant wizard page for the component. We	*/
	/* keep these in a dictionary keys by a value which will be	*/
	/* unique over all components, and which distinguises between	*/
	/* stock and user components. If there is such a page then	*/
	/* show it.							*/
	if ((m_currPage = m_pages.find (m_pageKey)) == 0)
	{
		m_currPage	= new KBComponentWizardPage (&m_wizard) ;
		m_pages.insert (m_pageKey, m_currPage) ;

		QList<KBConfig>	configSet ;
		component->findAllConfigs (configSet, QString::null) ;

		LITER
		(	KBConfig,
			configSet,
			config,

			if (!config->hidden())
			{
				config->fixupValue () ;

				if (!addSpecialConfig (config))
					m_currPage->addTextCtrl
					(	config->ident (),
						config->legend(),
						config->value ()
					)	;
			}
		)

		m_currPage->addedAll	() ;
	}
	else	m_wizard.raiseWidget (m_currPage) ;


	/* Note this as the currently dusplayed component and render	*/
	/* it into the example page.					*/
	m_component = component ;
	render	(m_component)	;
}

/*  KBComponentLoadDlg							*/
/*  documentSelected							*/
/*		: User selects an document				*/
/*  objName	: const QString & : Document name			*/
/*  (returns)	: void		  :					*/

void	KBComponentLoadDlg::documentSelected
	(	const QString	&
	)
{
	if (m_lbDocument.currentItem() >= 0)
	{
		m_document = m_lbDocument.currentText() ;
		m_pageKey  = QString("%1:%2")
				   .arg(m_cbServer.currentText())
				   .arg(m_document) ;
		showDetails ()	;

		m_selected = m_showType == m_reqdType ;
		m_bOK.setEnabled    (m_selected) ;
		m_tab.setTabEnabled (&m_wizard,  m_selected) ;
		m_tab.setTabEnabled (&m_example, m_component != 0) ;
	}
}

/*  KBComponentLoadDlg							*/
/*  stockSelected: User selects a stock component			*/
/*  lvItem	 : QListViewItem * : Item selected in list view		*/
/*  (returns)	 : void		   :					*/

void	KBComponentLoadDlg::stockSelected
	(	QListViewItem	*item
	)
{
	QString	path	;

	/* Check for the user clicking on a directory entry, in which	*/
	/* case ignore this action.					*/
	if ((item == 0) || (item->childCount() != 0)) return ;


	/* Backtrack up the tree from the selected item and build its	*/
	/* path name.							*/
	while (item != 0)
	{
		if (path.isNull())
			path	= item->text (0) ;
		else	path	= item->text (0) + "/" + path ;

		item	= item->parent() ;
	}

	m_document  = path	  ;
	m_stockPath = m_stockDir  + "/" + path + ".cmp" ;
	m_pageKey   = m_stockPath ;
	fprintf	(stderr, "KBComponentDlg: stockPath [%s]\n", (cchar *)m_stockPath) ;

	showDetails () ;

	m_selected = m_showType == m_reqdType ;
	m_bOK.setEnabled    (m_selected) ;
	m_tab.setTabEnabled (&m_wizard,  m_selected) ;
	m_tab.setTabEnabled (&m_example, m_component != 0) ;
}

/*  KBComponentLoadDlg							*/
/*  localSelected: User selects a local component			*/
/*  lvItem	 : QListViewItem * : Item selected in list view		*/
/*  (returns)	 : void		   :					*/

void	KBComponentLoadDlg::localSelected
	(	QListViewItem	*item
	)
{
	QString	path	;

	/* Check for the user clicking on a directory entry, in which	*/
	/* case ignore this action.					*/
	if ((item == 0) || (item->childCount() != 0)) return ;


	/* Backtrack up the tree from the selected item and build its	*/
	/* path name.							*/
	while (item != 0)
	{
		if (path.isNull())
			path	= item->text (0) ;
		else	path	= item->text (0) + "/" + path ;

		item	= item->parent() ;
	}

	m_document  = path ;
	m_stockPath = localDir() + path + ".cmp" ;
	m_pageKey   = m_stockPath ;
	fprintf	(stderr, "KBComponentDlg: stockPath [%s]\n", (cchar *)m_stockPath) ;

	showDetails () ;

	m_selected = m_showType == m_reqdType ;
	m_bOK.setEnabled    (m_selected) ;
	m_tab.setTabEnabled (&m_wizard,  m_selected) ;
	m_tab.setTabEnabled (&m_example, m_component != 0) ;
}

/*  KBComponentLoadDlg							*/
/*  tabPageChanged							*/
/*		: Tab page has changed					*/
/*  page	: QWidget *	: Now-current page			*/
/*  (returns)	: void		:					*/

void	KBComponentLoadDlg::tabPageChanged
	(	QWidget		*page
	)
{
	/* If this is the example page and there is a component then	*/
	/* display it, with substitutions.				*/
	if ((page == &m_example) && (m_component != 0))
		render (m_component) ;
}


/*  KBComponentLoadDlg							*/
/*  text	: Get raw text of currently selected document		*/
/*  data	: QByteArray &	: Data return				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBComponentLoadDlg::text
	(	QByteArray	&data,
		KBError		&pError
	)
{
	/* First case is for stock components, in which case load the	*/
	/* text from the associated file. The second case is similar,	*/
	/* for local components.					*/
	if (m_paste && (m_cbServer.currentItem() <= 1))
	{
		QFile	file (m_stockPath) ;

		if (!file.open (IO_ReadOnly))
		{
			pError	= KBError
				  (	KBError::Error,
					QString	("Failed to open \"%1\"").arg(m_stockPath),
					strerror(errno),
					__ERRLOCN
				  )	;
			return	false	;
		}

		data		= file.readAll() ;
		m_location	= KBLocation  () ;
		return	true	;
	}

	/* Otherwise, it is just the contents of the specified		*/
	/* document.							*/
	m_location = KBLocation
		     (		m_dbInfo,
				"component",
				m_cbServer  .currentText(),
				m_lbDocument.currentText(),
				"cmp"
		     )	;

	return	m_location.contents (data, pError) ;
}

/*  KBComponentLoadDlg							*/
/*  component	: Get component						*/
/*  (returns)	: KBComponent *	: Success				*/

KBComponent
	*KBComponentLoadDlg::component
	(	KBError		&pError
	)
{
	QByteArray	data	;

	if (m_currPage == 0)
		return	0	;

	if (!text (data, pError))
		return	0	;

	KBComponent *component = KBOpenComponentText (m_location, data, pError) ;
	if (component == 0)
		return	0	;

	substitute (component)	;

	return	component ;
}

/*  KBComponentLoadDlg							*/
/*  clickOK	: User clicks OK					*/
/*  (returns)	: void		:					*/

void	KBComponentLoadDlg::clickOK ()
{
	if (!m_selected)
		return	;

	if (m_currPage == 0)
		return	;

	QByteArray	data	;
	KBError		error	;


	if (!text (data, error))
	{	error.DISPLAY() ;
		return	;
	}

	KBComponent *component = KBOpenComponentText (m_location, data, error) ;
	if (component == 0)
	{	error.DISPLAY() ;
		return	;
	}

	/* Scan the component to find all configuration objects that	*/
	/* are embedded in it, and get the settings back from the	*/
	/* wizard page. We can then check that all required values are	*/
	/* set.								*/
	QList<KBConfig>	configSet  ;
	QDict<QString > configVals ;

	component ->findAllConfigs (configSet, QString::null) ;
	m_currPage->settings	   (configVals) ;

	LITER
	(	KBConfig,
		configSet,
		config,

		if ( config->hidden  ()) continue ;
		if (!config->required()) continue ;

		QString	*value	= configVals.find(config->ident()) ;
		if ((value != 0) && value->isEmpty())
		{
			KBError::EWarning
			(	QString	(TR("Please enter a value for '%1'"))
					.arg(config->legend()),
				QString::null,
				__ERRLOCN
			)	;
			return	;
		}
	)

	lastServer	= m_cbServer.currentText() ;
	done (true) ;
}

/*  KBComponentLoadDlg							*/
/*  clickCancel	: User clicks cancel					*/
/*  (returns)	: void		:					*/

void	KBComponentLoadDlg::clickCancel ()
{
	done	(false) ;
}


/*  ------------------------------------------------------------------  */

/*  KBComponentSaveDlg							*/
/*  KBComponentSaveDlg							*/
/*		: Save-as-component dialog				*/
/*  value	: QString &		: Returns component name	*/
/*  where	: QString &		: Returns save location		*/
/*  comment	: QString &		: Returns component comment	*/
/*  dbInfo	: KBDBInfo *		: Database information object	*/
/*  toFile	: bool *		: Save to file option		*/
/*  (returns)	: KBComponentSaveDlg	:				*/

KBComponentSaveDlg::KBComponentSaveDlg
	(	QString		&value,
		QString		&where,
		QString		&comment,
		KBDBInfo	*dbInfo,
		bool		*toFile
	)
	:
	KBPromptSaveDlg
	(	TR("Save component as ..."),
		TR("Enter component name"),
		value,
		where,
		dbInfo,
		true
	),
	m_textLabel	(this),
	m_textEdit	(this),
	m_comment	(comment),
	m_toFile	(toFile)
{
	m_lMain ->insertWidget (3, &m_textLabel) ;
	m_lMain ->insertWidget (4, &m_textEdit ) ;

	m_textLabel.setText (TR("Enter comment")) ;

	/* If there is a to-file return flag then insert a to-file	*/
	/* entry into the location list.				*/
	if (m_toFile != 0)
		m_cWhere.insertItem (TR("Save to file"), 1) ;
}

/*  KBComponentSaveDlg							*/
/*  accept	: User OK's dialog					*/
/*  (returns)	: void		:					*/

void	KBComponentSaveDlg::accept ()
{
	/* If the to-file option was specified then return the flag to	*/
	/* show whether the user selected this option or not.		*/
	if (m_toFile != 0)
		*m_toFile = m_cWhere.currentItem() == 1 ;

	m_comment = m_textEdit.text () ;
	KBPromptSaveDlg::accept	    () ;
}


/*  ------------------------------------------------------------------  */

void	saveComponentToFile
	(	const QString	&objName,
		const QString	&objText
	)
{
	KBFileDialog	fDlg
			(	".",
				"*.cmp|Components\n*.*|All file types",
				qApp->activeWindow(),
				"saveobject",
				true
			)	;

#if	! __KB_EMBEDDED
	fDlg.setSelection (objName) ;
#endif
	fDlg.setMode	  (KBFileDialog::AnyFile)	;
	fDlg.setCaption   (TR("Save to file ...."))	;

	if (!fDlg.exec ()) return ;

	QFile	file	;
	QString	name	= fDlg.selectedFile () ;
	file.setName	(name) ;

	if (QFileInfo(file).exists())
		if (TKMessageBox::questionYesNo
		   	(	0,
				QString (TR("%1 already exists: overwrite?")).arg(name),
				TR("Save to file ....")
			)
			!= TKMessageBox::Yes) return ;

	if (!file.open (IO_WriteOnly|IO_Truncate))
	{
		KBError::EError
		(	QString	(TR("Cannot open \"%1\"")).arg(name),
			strerror(errno),
			__ERRLOCN
		)	;
		return	;
	}


	QTextStream (&file) << objText ;
}



