/***************************************************************************
    file	         : tkc_keymapper.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qfile.h>
#include	<qintdict.h>


#include	"kb_classes.h"
#include	"kb_dom.h"

#ifndef 	_WIN32
#include	"tkc_keymapper.moc"
#else
#include 	"tkc_keymapper.h"
#endif

/*  TKCKeyMapperMap							*/
/*  ---------------							*/
/*  This class is used to represent a specific binding in a key map.	*/
/*  There will be one such element for each unique key prefix, which	*/
/*  contains information on what to do with that key prefix.		*/

class LIBKBASE_API TKCKeyMapperMap
{
	QIntDict<TKCKeyMapperMap>
				m_map		;

	enum	Binding
	{	BindNone,
		BindFunction,
		BindHelper
	}	;

	Binding			m_binding	;
	int			m_bindCode	;
	QString			m_bindText	;

public	:

	TKCKeyMapperMap	()	;

	void		bindKeyMapper	(int, TKCKeyMapperMap *) ;
	void		bindFunction	(int, const QString & = QString::null) ;
	void		bindHelper	(int, const QString & = QString::null) ;

	bool		activate	(TKCKeyMapper *) ;
	TKCKeyMapperMap	*applyKey	(int)	;
}	;


/*  TKCKeyMapperMap							*/
/*  TKCKeyMapperMap							*/
/*		: Constructor for key mapper binding object		*/
/*  (returns)	: TKCKeyMapperMap :					*/

TKCKeyMapperMap::TKCKeyMapperMap ()
{
	m_map.setAutoDelete (true) ;
	m_binding = BindNone	   ;
}

/*  TKCKeyMapperMap							*/
/*  ~TKCKeyMapperMap							*/
/*		: Destructor for key mapper binding object		*/
/*  (returns)	: void		:					*/

void	TKCKeyMapperMap::bindKeyMapper
	(	int		key,
		TKCKeyMapperMap	*mapper
	)
{
	m_map.insert (key, mapper) ;
}

/*  TKCKeyMapperMap							*/
/*  bindFunction: Bind a function to the binding object			*/
/*  bindCode	: int		  : Function code			*/
/*  bindText	: const QString & : Function text			*/
/*  (returns)	: void		  :					*/

void	TKCKeyMapperMap::bindFunction
	(	int		bindCode,
		const QString	&bindText
	)
{
	m_binding  = BindFunction ;
	m_bindCode = bindCode	  ;
	m_bindText = bindText	  ;
}

/*  TKCKeyMapperMap							*/
/*  bindHelper	: Bind a helper to the binding object			*/
/*  bindCode	: int		  : Helper code				*/
/*  bindText	: const QString & : Helper text				*/
/*  (returns)	: void		  :					*/

void	TKCKeyMapperMap::bindHelper
	(	int		bindCode,
		const QString	&bindText
	)
{
	m_binding  = BindHelper	;
	m_bindCode = bindCode	;
	m_bindText = bindText	;
}

/*  TKCKeyMapperMap							*/
/*  activate	: Activate binding object				*/
/*  mapper	: TKCKeyMapper * : Mapper to invoke			*/
/*  (returns)	: bool		 : True if function activated		*/

bool	TKCKeyMapperMap::activate
	(	TKCKeyMapper	*mapper
	)
{
	switch (m_binding)
	{
		case BindHelper   :
			/* If the binding is to a helper then invoke	*/
			/* the helper. The return code indicates if	*/
			/* the caller should clear the current mapping.	*/
			return	mapper->helper   (m_bindCode, m_bindText) ;

		case BindFunction :
			/* Invoke the function. The return code		*/
			/* indicates if the caller should clear the	*/
			/* current mapping.				*/
			return	mapper->function (m_bindCode, m_bindText) ;

		default	:
			/* Otherwise, we are presumably part-way	*/
			/* a key binding.				*/
			break	;
	}

	return	false	;
}


/*  TKCKeyMapperMap							*/
/*  applyKey	: Apply key to binding					*/
/*  key		: int		    : Key from QKeyEvent		*/
/*  (returns)	: TKCKeyMapperMap * : Next binding or null of none	*/

TKCKeyMapperMap
	*TKCKeyMapperMap::applyKey
	(	int		key
	)
{
	/* This is a simple lookup. The caller uses this method to step	*/
	/* through key binding sequences.				*/
	return	m_map.find (key) ;
}



/*  ------------------------------------------------------------------  */

QIntDict<TKCKeyMapperMap>
		TKCKeyMapper::m_map	;

/*  TKCKeyMapper							*/
/*  TKCKeyMapper: Constructor for key mapper base class			*/
/*  (returns)	: TKCKeyMapper	:					*/

TKCKeyMapper::TKCKeyMapper ()
{
	m_currentMap = 0	   ;
}

/*  TKCKeyMapper							*/
/*  ~TKCKeyMapper: Destructor for key mapper base class			*/
/*  (returns)	 :		:					*/

TKCKeyMapper::~TKCKeyMapper ()
{
}


/*  TKCKeyMapper							*/
/*  clearKeyMap	: Clear key maps					*/
/*  (returns)	: void		:					*/

void	TKCKeyMapper::clearKeyMap ()
{
	/* FIX ME! We should make sure that the m_currentMap pointers	*/
	/* of _all) extant TKCKeyMapper objects are reset!		*/
	m_map.clear  ()  ;
	m_currentMap = 0 ;
}

/*  TKCKeyMapper							*/
/*  findMapperMap: Find mapper element for specified key string		*/
/*  keyString	 : QValueList<int> & : Key string			*/
/*  (returns)	 : TKCKeyMapperMap * : Mapper element			*/

TKCKeyMapperMap
	*TKCKeyMapper::findMapperMap
	(	const QValueList<int>	&keyString
	)
{
	TKCKeyMapperMap	*map	;

	if ((map = m_map.find (keyString[0])) == 0)
	{
		map	= new TKCKeyMapperMap () ;
		m_map.insert (keyString[0], map) ;
	}

	for (uint idx = 1 ; idx < keyString.count() ; idx += 1)
	{
		TKCKeyMapperMap *next = map->applyKey (keyString[idx]) ;

		if (next == 0)
		{
			next = new TKCKeyMapperMap () ;
			map->bindKeyMapper (keyString[idx], next) ;
		}

		map	= next	;
	}

	return	map	;
}

void	TKCKeyMapper::error ()
{
	fprintf	(stderr, "--->error\n") ;
}

/*  TKCKeyMapper							*/
/*  bindFunction: Bind key sequence to function				*/
/*  keyString	: QValueList<int> : Key sequence			*/
/*  bindCode	: int		  : Function code			*/
/*  bindText	: const QString & : Function text			*/
/*  (returns)	: void		  :					*/

void	TKCKeyMapper::bindFunction
	(	const QValueList<int>	&keyString,
		int			bindCode,
		const QString		&bindText
	)
{
	findMapperMap(keyString)->bindFunction (bindCode, bindText) ;
}

/*  TKCKeyMapper							*/
/*  bindHelper	: Bind key sequence to helper				*/
/*  keyString	: QValueList<int> : Key sequence			*/
/*  bindCode	: int		  : Helper code				*/
/*  bindText	: const QString & : Helper text				*/
/*  (returns)	: void		  :					*/

void	TKCKeyMapper::bindHelper
	(	const QValueList<int>	&keyString,
		int			bindCode,
		const QString		&bindText
	)
{
	findMapperMap(keyString)->bindHelper   (bindCode, bindText) ;
}

/*  TKCKeyMapper							*/
/*  mapKeysToKeys: Map character kde sequence to code key sequence	*/
/*  keys	 : const QString & : Character sequence			*/
/*  (returns)	 : QValueList<int> : Code sequence			*/

QValueList<int>
	TKCKeyMapper::keysToKeys
	(	const QString	&keys
	)
{
	struct	KeysToCode
	{	char	key [4]	;
		int	code	;
	}	;

	static	KeysToCode	mapKeysToKeys[] =
	{
		{	"A",		Qt::Key_A		},
		{	"B",		Qt::Key_B		},
		{	"C",		Qt::Key_C		},
		{	"D",		Qt::Key_D		},
		{	"E",		Qt::Key_E		},
		{	"F",		Qt::Key_F		},
		{	"G",		Qt::Key_G		},
		{	"H",		Qt::Key_H		},
		{	"I",		Qt::Key_I		},
		{	"J",		Qt::Key_J		},
		{	"K",		Qt::Key_K		},
		{	"L",		Qt::Key_L		},
		{	"M",		Qt::Key_M		},
		{	"N",		Qt::Key_N		},
		{	"O",		Qt::Key_O		},
		{	"P",		Qt::Key_P		},
		{	"Q",		Qt::Key_Q		},
		{	"R",		Qt::Key_R		},
		{	"S",		Qt::Key_S		},
		{	"T",		Qt::Key_T		},
		{	"U",		Qt::Key_U		},
		{	"V",		Qt::Key_V		},
		{	"W",		Qt::Key_W		},
		{	"X",		Qt::Key_X		},
		{	"Y",		Qt::Key_Y		},
		{	"Z",		Qt::Key_Z		},

		/* Add more here as needed ...				*/
		{	"",		0			}
	}	;

	/* Scan the string key by key. converting to Qt key codes. In	*/
	/* the string, characters stand for themselves, except that	*/
	/* braces can be used for special cases, like "{F1}".		*/
	QValueList<int>	seq	;
	int		idx1	= 0;

	while (idx1 < (int)keys.length())
	{
		QString	key	;
		int	idx2	;

		if (keys.at(idx1) == "{")
		{
			if ((idx2 = keys.find ('}', idx1 + 1)) < 0)
				break	;

			key	= keys.mid (idx1 + 1, idx2 - idx1 - 1) ;
			idx1	= idx2 + 1 ;
		}
		else
		{	key	= keys.mid (idx1, 1) ;
			idx1    += 1 ;
		}

		for (KeysToCode *kp = &mapKeysToKeys[0] ; kp->code != 0 ; kp += 1)
			if (kp->key == key)
			{	seq.append (kp->code) ;
				break	;
			}
	}

	return	seq	;
}

/*  TKCKeyMapper							*/
/*  loadKeyMap	: Load a key map specification				*/
/*  kName	: const QString & : Specification file name		*/
/*  (returns)	: QString	  : Error or NULL if OK			*/

QString	TKCKeyMapper::loadKeyMap
	(	const QString	&kName
	)
{
	QFile	kFile	(kName) ;
	if (!kFile.open (IO_ReadOnly))
		return	QString(TR("%1: unable to open")).arg(kName) ;

	QDomDocument	kDoc	;
	if (!kDoc.setContent (&kFile))
		return	QString(TR("%1: unable to parse")).arg(kName) ;


	QDomNodeList	kFuncs	= kDoc.elementsByTagName ("function") ;
	QDomNodeList	kHelps	= kDoc.elementsByTagName ("helper"  ) ;

	for (uint idx1 = 0 ; idx1 < kFuncs.length() ; idx1 += 1)
	{
		QDomNode	kNode = kFuncs.item  (idx1) ;
		QDomElement	kElem = kNode.toElement () ;

		QString		kKeys = kElem.attribute ("keys") ;
		QString		kCode = kElem.attribute ("code") ;
		QString		kText = kElem.attribute ("text") ;

//		fprintf	(stderr, "Function[%s]->[%s][%s]\n",
//				 (const char *)kKeys,
//				 (const char *)kCode,
//				 (const char *)kText) ;
		bindFunction
		(	keysToKeys (kKeys),
			codeToCode (kCode),
			kText
		)	;
	}

	for (uint idx2 = 0 ; idx2 < kHelps.length() ; idx2 += 1)
	{
		QDomNode	kNode = kHelps.item  (idx2) ;
		QDomElement	kElem = kNode.toElement () ;

		QString		kKeys = kElem.attribute ("keys") ;
		QString		kCode = kElem.attribute ("code") ;
		QString		kText = kElem.attribute ("text") ;

//		fprintf	(stderr, "Helper  [%s]->[%s][%s]\n",
//				 (const char *)kKeys,
//				 (const char *)kCode,
//				 (const char *)kText) ;
		bindHelper
		(	keysToKeys (kKeys),
			codeToCode (kCode),
			kText
		)	;
	}

	return	QString::null	;
}

/*  TKCKeyMapper							*/
/*  useAtOuterLevel							*/
/*		: See if key should be processed at outer level		*/
/*  key		: int		: Key code from QKeyEvent		*/
/*  ctrl	: bool		: Control key pressed			*/
/*  (returns)	: bool		: Key consumed by mapper		*/

bool	TKCKeyMapper::useAtOuterLevel
	(	int		,
		bool		ctrl
	)
{
	/* At the outermost level, the default is we only handle	*/
	/* control keys. This means stuff like straight keystrokes,	*/
	/* cursor keys and the like are handled by the widget itself.	*/
	return	ctrl	;
}

/*  TKCKeyMapper							*/
/*  applyKey	: Apply key to the mapper				*/
/*  key		: int		: Key code from QKeyEvent		*/
/*  ctrl	: bool		: Control key pressed			*/
/*  (returns)	: bool		: Key consumed by mapper		*/

bool	TKCKeyMapper::applyKey
	(	int		key,
		bool		ctrl
	)
{
	if (m_currentMap == 0)
	{
		/* At the outermost level, check if we should use the	*/
		/* key ...						*/
		if (!useAtOuterLevel (key, ctrl))
			return	false	;

		/* OK, got a usable key. See if the key is in the map,	*/
		/* if not then report an error.				*/
		m_currentMap = m_map.find (key) ;

		if (m_currentMap == 0)
		{	error	()	;
			return	true	;
		}
	}
	else
	{
		/* At the inner levels, we handle all keys. If the key	*/
		/* leads nowhere then report an error.			*/
		m_currentMap = m_currentMap->applyKey (key) ;

		if (m_currentMap == 0)
		{	error	()	;
			return	true	;
		}
	}

	/* We are inside the mappings. See if this mapping consumes the	*/
	/* key, in which case we reset the current mapping pointer.	*/
	if (m_currentMap->activate (this))
	{
		m_currentMap = 0 ;
		return	true	 ;
	}

	/* Current mapping did not consume the key, so we consume the	*/
	/* key but leave the mapper in place.				*/
	return	true	;
}

/*  TKCKeyMapper							*/
/*  clearKey	: Clear any partial mapping				*/
/*  (returns)	: void		:					*/

void	TKCKeyMapper::clearKey ()
{
	m_currentMap = 0 ;
}

