/***************************************************************************
    file	         : kb_writer.cpp
    copyright            : (C)	1999,2000,2001,2002,2003,2004,2005
				 by Mike Richardson
			   (C)	2001,2002,2003,2004,2005
				by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qpainter.h>
#include	<qwidget.h>
#include	<qlist.h>
#include	<qpaintdevicemetrics.h>
#include	<qsimplerichtext.h>

#include	"kb_classes.h"
#include	"kb_ctrl.h"
#include	"kb_writer.h"
#include	"kb_ascii.h"
#include	"kb_labelskipdlg.h"

#include	"tk_messagebox.h"


/*  Printer settings like margins and virtual pages are give in		*/
/*  millimeters, but internal operation is in terms of pixels. Pending	*/
/*  better display information, PPM is defined to be a nominal pixels-	*/
/*  per-millimeter on the display.					*/

#define	PPM	(3.448)



struct	LIBKBASE_API	PageSizeMap
{	cchar	*name		;
	int	widthMM		;
	int	heightMM	;
}	;

static	PageSizeMap	pageSizeMap[] =
{
	{	"A0",		841,	1189		},
	{	"A1",		594,	841		},
	{	"A2",		420,	594		},
	{	"A3",		297,	420		},
	{	"A4",		210,	297		},
	{	"A5",		148,	210		},
	{	"A6",		105,	148		},
	{	"A7",		74,	105		},
	{	"A8",		52,	74		},
	{	"A9",		37,	52		},
	{	"B0",		1030,	1456		},
	{	"B1",		728,	1030		},
	{	"B10",		32,	45		},
	{	"B2",		515,	728		},
	{	"B3",		364,	515		},
	{	"B4",		257,	364		},
	{	"B5",		182,	257		},
	{	"B6",		128,	182		},
	{	"B7",		91,	128		},
	{	"B8",		64,	91		},
	{	"B9",		45,	64		},
	{	"C5E",		163,	229		},
	{	"Comm10E",	105,	241		},
	{	"DLE",		110,	220		},
	{	"Executive",	191,	254		},
	{	"Folio",	210,	330		},
	{	"Ledger",	432,	279		},
	{	"Legal",	216,	356		},
	{	"Letter",	216,	279		},
	{	"Tabloid",	279,	432		},
	{	0,		0,	0		}
}	;

LIBKBASE_API
	double	pixelsPerMM ()
{
	return	PPM	;
}

LIBKBASE_API
	void	getPixelPageSize
	(	cchar		*name,
		int		&widthMM,
		int		&heightMM
	)
{
	for (PageSizeMap *pmp = &pageSizeMap[0] ; pmp->name != 0 ; pmp += 1)
		if (qstricmp (name, pmp->name) == 0)
		{
			widthMM	 = (int)(pmp->widthMM  * pixelsPerMM()) ;
			heightMM = (int)(pmp->heightMM * pixelsPerMM()) ;
			return	 ;
		}

	widthMM	 = (int)(210 * pixelsPerMM())	;
	heightMM = (int)(297 * pixelsPerMM())	;
}

/*  ------------------------------------------------------------------  */

/*  KBWriterItem							*/
/*  KBWriterItem: Constructor for writer item base class		*/
/*  writer	: KBWriter *	: Parent writer				*/
/*  rect	: QRect		: Item position				*/
/*  (returns)	: KBWriterItem	:					*/

KBWriterItem::KBWriterItem
	(	KBWriter	*writer,
		QRect		rect
	)
	:
	m_writer(writer),
	m_rect	(rect)
{
	m_writer->add (this, m_rect) ;
}

/*  KBWriterItem							*/
/*  ~KBWriterItem: Destructor for writer item base class		*/
/*  (returns)	 :		:					*/

KBWriterItem::~KBWriterItem ()
{
}


/*  ------------------------------------------------------------------  */

/*  KBWriterText							*/
/*  KBWriterText: Constructor for write-text object			*/
/*  writer	: KBWriter *	   : Parent writer			*/
/*  rect	: QRect		   : Text area				*/
/*  pal		: const QPalette * : Palette				*/
/*  font	: const QFont *	   : Font				*/
/*  text	: const QString &  : Text				*/
/*  align	: int		   : Alignment				*/
/*  fSubs	: bool		   : Text substitution flag		*/
/*  (returns)	: void		   :					*/

KBWriterText::KBWriterText
	(	KBWriter	*writer,
		QRect		rect,
		const QPalette	*pal,
		const QFont	*font,
		const QString	&text,
		int		align,
		bool		fSubs
	)
	:
	KBWriterItem	(writer, rect),
	m_pal		(pal),
	m_font		(font),
	m_text		(text),
	m_align		(align),
	m_fSubs		(fSubs)
{
//	fprintf
//	(	stderr,
//		"KBWriterText: rect=%s t=%s\n",
//		_TEXT(rect),
//		_TEXT(text)
//	)	;

	m_extra	   = 0	   ;
	m_useExtra = false ;

	if (m_align == 0x1001)
	{
		QSimpleRichText rt (m_text, *m_font) ;
		rt.setWidth(m_rect.width()) ;
		m_extra	= rt.height() - m_rect.height() ;

		fprintf
		(	stderr,
			"KBWriterText::KBWriterText: m_extra=%d\n",
			m_extra
		)	;
		return	;
	}
}

/*  KBWriterText							*/
/*  ~KBWriterText: Destructor for write-text object			*/
/*  (returns)	 :		:					*/

KBWriterText::~KBWriterText ()
{
}

/*  KBWriterText							*/
/*  paintEvent	: Handle paint event for write-text object		*/
/*  e		: QPaintEvent *	: The originating paint event		*/
/*  p		: QPainter &	: Painter onto which to write		*/
/*  (returns)	: void		:					*/

void	KBWriterText::paintEvent
	(	QPaintEvent	*,
		QPainter	&p
	)
{
	QString	str	= m_fSubs ? m_writer->textSub (m_text) : m_text ;
	QRect	rect	= m_rect ;

	m_writer->adjust (rect) ;

//	fprintf	(stderr, "align     [%d]\n", align) ;
//	fprintf	(stderr, "writerText[%s]\n", (cchar *)str) ;

	p.save	()  ;

	if (m_align == 0x1001)
	{
		QSimpleRichText rt
			(	m_fSubs ? m_writer->textSub (m_text) : m_text,
				*m_font
			)	;

		if (m_useExtra && (m_extra > 0))
			rect.setHeight (rect.height() + m_extra) ;

		rt.setWidth(rect.width()) ;
		rt.draw	   (&p, rect.x(), rect.y(), rect, m_pal->active()) ;
	}
	else
	{
		p.setPen   (m_pal->active().foreground()) ;
		p.setFont  (*m_font) ;
		p.drawText (rect, m_align, str) ;
	}

	p.restore () ;
}


/*  ------------------------------------------------------------------  */

KBWriterPixmap::KBWriterPixmap
	(	KBWriter	*writer,
		QRect		rect,
		const QPixmap	&pixmap
	)
	:
	KBWriterItem	(writer, rect),
	m_pixmap	(pixmap)
{
	
}

KBWriterPixmap::~KBWriterPixmap ()
{
}

/*  KBWriterPixmap							*/
/*  paintEvent	: Handle paint event for image object			*/
/*  e		: QPaintEvent *	: The originating paint event		*/
/*  p		: QPainter &	: Painter onto which to write		*/
/*  (returns)	: void		:					*/

void	KBWriterPixmap::paintEvent
	(	QPaintEvent	*,
		QPainter	&p
	)
{
	QRect	rect = m_rect	;
	m_writer->adjust (rect) ;

	p.drawPixmap
	(	rect.x(),
		rect.y(),
		m_pixmap,
		0, 0,
		rect.width (),
		rect.height()
	) ;

}

/*  ------------------------------------------------------------------  */

/*  KBWriterBG								*/
/*  KBWriterBG	: Background object constructor				*/
/*  writer	: KBWriter *	  : Parent writer			*/
/*  rect	: QRect	   	  : Text area				*/
/*  color	: const QString & : Color specification			*/
/*  (returns)	: KBWriteBG	  :					*/


KBWriterBG::KBWriterBG
	(	KBWriter	*writer,
		QRect		rect,
		const QString	&color
	)
	:
	KBWriterItem	(writer, rect),
	m_color		(color)
{
}

/*  KBWriterBG								*/
/*  ~KBWriterBG	: Background object destructor				*/
/*  (returns)	:		:					*/

KBWriterBG::~KBWriterBG ()
{
}

/*  KBWriterBG								*/
/*  paintEvent	: Handle paint event for write-text object		*/
/*  e		: QPaintEvent *	: The originating paint event		*/
/*  p		: QPainter &	: Painter onto which to write		*/
/*  (returns)	: void		:					*/

void	KBWriterBG::paintEvent
	(	QPaintEvent	*,
		QPainter	&p
	)
{
	QRect	rect = m_rect	;
	m_writer->adjust (rect) ;

	if (!m_color.isEmpty())
		p.fillRect (rect, QColor (m_color.mid(2).toInt(0,16))) ;

//	fprintf
//	(	stderr,
//		"KBWriterBG::paintEvent [%s] %s\n",
//		(cchar *)color,
//		_TEXT(rect)
//	)	;
}


/*  ------------------------------------------------------------------  */

/*  KBWriterBox								*/
/*  KBWriterBox	: Constructor for box object				*/
/*  writer	: KBWriter *	: Parent writer				*/
/*  rect	: QRect	   	: Box area				*/
/*  (returns)	: KBWriterBox	:					*/

KBWriterBox::KBWriterBox
	(	KBWriter	*writer,
		QRect		rect
	)
	:
	KBWriterItem	(writer, rect)
{
}

/*  KBWriterBox								*/
/*  ~KBWriterBox: Destructor for box object				*/
/*  (returns)	:		:					*/

KBWriterBox::~KBWriterBox ()
{
}

/*  KBWriterBox								*/
/*  paintEvent	: Handle paint event for box object			*/
/*  e		: QPaintEvent *	: The originating paint event		*/
/*  p		: QPainter &	: Painter onto which to write		*/
/*  (returns)	: void		:					*/

void	KBWriterBox::paintEvent
	(	QPaintEvent	*,
		QPainter	&p
	)
{
	QRect	rect = m_rect	;
	m_writer->adjust (rect) ;

	p.setPen   (Qt::black)	;
	p.drawRect (rect)	;
}


/*  ------------------------------------------------------------------  */

/*  KBWriter								*/
/*  KBWriter: Constructor for widget writer class			*/
/*  parent	: QWidget *	: Parent widget				*/
/*  locn	: KBLocation &	: Location of associated document	*/
/*  (returns)	: KBWriter	:					*/

KBWriter::KBWriter
	(	QWidget		*parent,
		KBLocation	&location
	)
	:
	QWidget		(parent),
	m_dbInfo	(location.dbInfo ),
	m_docLocn	(location.docLocn)
{
	m_page		= 0	;
	m_report	= false	;

	m_vRows		= 1	;
	m_vCols		= 1	;
	m_vRowGap	= 0	;
	m_vColGap	= 0	;
	m_vRowHeight	= 0	;
	m_vColWidth	= 0	;
	m_vAtRow	= 0	;
	m_vAtCol	= 0	;
	m_vBorders	= false	;

#if	! __KB_EMBEDDED
	m_printer	= 0 ;
	m_painter	= 0 ;
#endif
}

/*  KBWriter								*/
/*  ~KBWriter: Constructor for widget writer class			*/
/*  (returns)	   :		:					*/

KBWriter::~KBWriter ()
{
#if	! __KB_EMBEDDED
	DELOBJ(m_painter) ;
	DELOBJ(m_printer) ;
#endif
}


#if	! __KB_EMBEDDED

/*  KBWriter								*/
/*  getPrinterSpec							*/
/*		: Get printer specification DOM element			*/
/*  name	: const QString & : Printer name			*/
/*  (returns)	: QDomElement	  : Will be null if printer not found	*/

QDomElement
	KBWriter::getPrinterSpec
	(	const QString	&name
	)
{
	KBLocation location
		   (	m_dbInfo,
			"print",
			m_docLocn,
			name.isNull() ? QString("Default") : name
		   )	;

	if (!location.exists())
	{
		if (!name.isEmpty())
			TKMessageBox::sorry
			(	0,
				QString(TR("Printer name: %1")).arg(name),
				TR("Printer not found")
			)	;
		return	QDomElement () ;
	}

	KBError		error	;
	QString		_doc	= location.contents (error) ;
	if (_doc.isNull())
	{	error.DISPLAY() ;
		return	QDomElement () ;
	}

	QDomDocument	pXML	;
	pXML.setContent	(_doc)	;
	QDomElement	root	= pXML.documentElement () ;
	if (root.isNull())
	{
		KBError::EError
		(	TR("Printer definition has no root element"),
			QString(TR("Printer: %1")).arg(name),
			__ERRLOCN
		)	;
		return	QDomElement () ;
	}

	return	root	;
}
#endif

/*  KBWriter								*/
/*  setup	: Set writer up prior to generating output		*/
/*  name	: const QString & : Printer name			*/
/*  showDlg	: bool		  : Show dialog				*/
/*  lMargin	: uint		  : Left margin				*/
/*  rMargin	: uint		  : Right margin			*/
/*  tMargin	: uint		  : Top margin				*/
/*  bMargin	: uint		  : Botton margin			*/
/*  isPrinting	: bool		  : Setting up for printing		*/
/*  (returns)	: bool		  : Success				*/

bool	KBWriter::setup
	(	const QString	&name,
		bool		showDlg,
		uint		lMargin,
		uint		rMargin,
		uint		tMargin,
		uint		bMargin,
		bool		isPrinting
	)
{
#if	! __KB_EMBEDDED
	QDomElement prSpec = getPrinterSpec (name) ;

	m_printer = new TKPrinter (TKPrinter::ResScreen) ;

	if (!prSpec.isNull())
		m_printer->loadSettings (prSpec) ;


	if (showDlg || prSpec.isNull())
		if (!m_printer->setup ())
		{	DELOBJ	(m_printer) ;
			return	false ;
		}

#if	__KB_KDE
	uint	    dpi	   = KBOptions::getDPI  () ;
	if (dpi != 0)
	{
		fprintf
		(	stderr,
			"KBWriter::setup: setResolution(%d)\n",
			dpi
		)	;
		m_printer->setResolution (dpi) ;
	}
#endif

	m_printer->setFullPage  (true) ;
	m_printer->saveSettings (m_settings) ;
#endif

	/* Note the margin values. These are given in millimeters and	*/
	/* are not in pixels.						*/
	m_lMargin	= lMargin ;
	m_rMargin	= rMargin ;
	m_tMargin	= tMargin ;
	m_bMargin	= bMargin ;


#if	__KB_EMBEDDED
	m_pWidth	= 240	  ;
	m_pHeight	= 280	  ;
#else

	/* HACK ALERT: We need the width and height even if we are	*/
	/* generating in "data" view. But, we need to associate a	*/
	/* painter with the printer before we can get the correct	*/
	/* metrics back. If we are not going to print then the print	*/
	/* is aborted, else we end up with an empty document.		*/
	m_painter = new QPainter() ;
	m_painter->begin  	(m_printer) ;
	QPaintDeviceMetrics dm	(m_printer) ;

//	fprintf	(stderr, "pdm width   : %d\n", dm.width      ()) ;
//	fprintf	(stderr, "pdm height  : %d\n", dm.height     ()) ;
//	fprintf	(stderr, "pdm widthMM : %d\n", dm.widthMM    ()) ;
//	fprintf	(stderr, "pdm heightMM: %d\n", dm.heightMM   ()) ;
//	fprintf	(stderr, "pdm logDpiX : %d\n", dm.logicalDpiX()) ;
//	fprintf	(stderr, "pdm logDpiY : %d\n", dm.logicalDpiY()) ;

	/* Calculate the printing width and height (the area less the	*/
	/* margins) in pixels.						*/
	m_pWidth	= (int)((double)(dm.widthMM () - m_lMargin - m_rMargin) * PPM) ;
	m_pHeight	= (int)((double)(dm.heightMM() - m_tMargin - m_bMargin) * PPM) ;

//	fprintf	(stderr, "pix width   : %d\n", m_pWidth ) ;
//	fprintf	(stderr, "pix height  : %d\n", m_pHeight) ;

	m_pageList	= m_printer->pageList() ;

//	fprintf	(stderr, "Page: List has %d entries\n", m_pageList.count()) ;
//	for (uint idx = 0 ; idx < m_pageList.count() ; idx += 1)
//		fprintf	(stderr, "Page: %2d -> %2d\n", idx, m_pageList[idx]) ;

	if (!isPrinting)
	{
		m_printer->abort() ;
		DELOBJ (m_painter) ;
		DELOBJ (m_printer) ;
	}
#endif

	m_vCols		= 1		;
	m_vRows		= 1		;
	m_vColGap	= 0		;
	m_vRowGap	= 0		;
	m_vColWidth	= m_pWidth	;
	m_vRowHeight	= m_pHeight	;
	m_vBorders	= false		;
	m_vAtCol	= 0		;
	m_vAtRow	= 0		;

	m_offset 	= QPoint(0,0)	;
	m_foot	 	= 0 ;
	m_cPNum	 	= 0 ;

	m_pages.setAutoDelete (true) ;
	clear	() 	;
	return	true	;
}

#if	! __KB_EMBEDDED
/*  KBWriter								*/
/*  setup	: Set writer up prior to generating output		*/
/*  showDlg	: bool		  : Show dialog				*/
/*  (returns)	: bool		  : Success				*/

bool	KBWriter::setup
	(	bool		showDlg
	)
{
	/* This call is used when a writer which has been used for data	*/
	/* view is to be reused for printing. In this case the printer	*/
	/* settings will have been saved in previously.			*/
	m_printer = new TKPrinter (TKPrinter::ResScreen, &m_settings) ;

	if (showDlg)
		if (!m_printer->setup ())
		{	DELOBJ	(m_printer) ;
			return false ;
		}

#if	__KB_KDE
	uint	    dpi	   = KBOptions::getDPI  () ;
	if (dpi != 0)
	{
		fprintf
		(	stderr,
			"KBWriter::setup: setResolution(%d)\n",
			dpi
		)	;
		m_printer->setResolution (dpi) ;
	}
#endif

	m_printer->setFullPage  (true) ;

	m_painter = new QPainter() ;
	m_painter->begin  	(m_printer) ;

	return	true	;
}
#endif

bool	KBWriter::setupVirtual
	(	uint		colWidth,
		uint		rowHeight,
		uint		colGap,
		uint		rowGap,
		bool		borders,
		bool		skip
	)
{
	fprintf
	(	stderr,
		"KBWriter::setupVirtual: page=(%d,%d) cell=(%d,%d) gap=(%d,%d)\n",
		m_pWidth,
		m_pHeight,
		rowHeight,
		colWidth,
		rowGap,
		colGap
	)	;

	/* Column width and row height of zero are mapped onto the full	*/
	/* width or height of the page.					*/
	if (colWidth  == 0) colWidth  = m_pWidth  ;
	if (rowHeight == 0) rowHeight = m_pHeight ;

	/* Calculate how may columns and rows can be accomodated on	*/
	/* the page ...							*/
	m_vCols		= (int)(((double)m_pWidth  / PPM  + (double)colGap) / (double)(colWidth  + colGap)) ;
	m_vRows		= (int)(((double)m_pHeight / PPM  + (double)rowGap) / (double)(rowHeight + rowGap)) ;

	/* ... and store the column widths, row heights and the gaps	*/
	/* in pixels.							*/
	m_vColWidth	= (int)((double)colWidth   * PPM) ;
	m_vRowHeight	= (int)((double)rowHeight  * PPM) ;
	m_vColGap	= (int)((double)colGap	   * PPM) ;
	m_vRowGap	= (int)((double)rowGap	   * PPM) ;

	m_vBorders	= borders ;
	m_vAtCol	= 0	  ;
	m_vAtRow	= 0	  ;

	fprintf
	(	stderr,
		"KBWriter::setupVirtual: (%d,%d) size=(%d,%d) gap=(%d,%d)\n",
		m_vRows,
		m_vCols,
		m_vRowHeight,
		m_vColWidth,
		m_vRowGap,
		m_vColGap
	)	;

	if (skip)
	{
		KBLabelSkipDlg	skipDlg (m_vBorders, m_vCols, m_vRows) ;
		if (!skipDlg.exec ()) return false ;

		m_vBorders = skipDlg.borders () ;
		m_vAtCol   = skipDlg.skipOver()	% m_vCols ;
		m_vAtRow   = skipDlg.skipOver()	/ m_vCols ;
	}

	return	true	;
}

/*  KBWriter								*/
/*  setOffset	: Set output offset					*/
/*  abs		: bool		: True->absolute, false->delta		*/
/*  o		: QPoint	: Offset				*/
/*  (returns)	: QPoint	: Original offset			*/

QPoint	KBWriter::setOffset
	(	bool	abs,
		QPoint	o
	)
{
	QPoint	r = m_offset  ;

//	fprintf
//	(	stderr,
//		"KBWriter::setOffset(%d,%s) %s->",
//		abs,
//		_TEXT(o),
//		_TEXT(m_offset)
//	)	;

	if (abs)
		m_offset  = o ;
	else	m_offset += o ;

//	fprintf
//	(	stderr,
//		"%s\n",
//		_TEXT(m_offset)
//	)	;

	return	r	;
}

/*  KBWriter								*/
/*  reserve	: Reserve footer space					*/
/*  f		: int		: Footer space required			*/
/*  (returns)	: int		: Offset of top of footer space		*/

int	KBWriter::reserve
	(	int	f
	)
{
//	fprintf	(stderr, "KBWriter::reserve(%d) %d->%d\n", f, m_foot, m_foot + f) ;

	m_foot	+= f ;
//	return	m_pHeight - m_foot ;
	return	m_vRowHeight - m_foot ;
}

/*  KBWriter								*/
/*  clear	: Clear writer preparatory to redisplay			*/
/*  (returns)	: void		:					*/

void	KBWriter::clear ()
{
	m_page	= 0 ;
	m_pages.clear () ;
	erase	      () ;
}

/*  KBWriter								*/
/*  spaceAvailable: See if space is available on page			*/
/*  y		  : int		: Space required			*/
/*  (returns)	  : bool	: True if available			*/

bool	KBWriter::spaceAvailable
	(	int	y
	)
{
//	return	m_offset.y() + y < m_pHeight - m_foot ;
	return	m_offset.y() + y < m_vRowHeight - m_foot ;
}

/*  KBWriter								*/
/*  pageIsEmpty	: See if page is empty					*/
/*  (returns)	: bool		: True if empty				*/

bool	KBWriter::pageIsEmpty ()
{
	return	(m_page == 0) || (m_page->count() == 0) ;
}

/*  KBWriter								*/
/*  newPage	: Start new output page					*/
/*  (returns)	: void		:					*/

void	KBWriter::newPage ()
{
	/* Note that this does not actually start a new page, rather	*/
	/* it just sets up for a new page. The page will actually be	*/
	/* created as and when the first output arrives for that page.	*/

	m_offset  = QPoint(0,0) ;
	m_foot	  = 0 ;

	fprintf
	(	stderr,
		"KBWriter::newPage: current (%d,%d) of (%d,%d)\n",
		m_vAtCol,
		m_vAtRow,
		m_vCols,
		m_vRows
	)	;

	m_vAtCol += 1 ;
	if (m_vAtCol < m_vCols) return ;

	m_vAtCol  = 0 ;
	m_vAtRow += 1 ;
	if (m_vAtRow < m_vRows) return ;

	m_vAtRow  = 0 ;

	fprintf
	(	stderr,
		"KBWriter::newPage: start new page\n"
	)	;

	m_page	  = 0 ;
}

/*  KBWriter								*/
/*  showPage	: Show specified page					*/
/*  pnum	: uint		: Logical page number			*/
/*  (returns)	: void		:					*/

void	KBWriter::showPage
	(	uint	pnum
	)
{
	fprintf
	(	stderr,
		"KBWriter::showPage : %d ",
		pnum
	)	;

	/* The page number is the physical number, ie., counting from	*/
	/* zero over the actual pages to print. We first map this to	*/
	/* the corresponding logical page number, ie., counting from	*/
	/* one for the generated pages ....				*/
	if (m_pageList.count() > 0)
	{
		if (pnum >= m_pageList.count())
			pnum  = m_pageList[m_pageList.count() - 1] ;
		else	pnum  = m_pageList[pnum] ;

		if (pnum >= 1)
			pnum -= 1 ;
	}

	/* ... and offset back, with paranoia checks to ensure that the	*/
	/* number is in range.						*/
	if (pnum >= m_pages.count())
		pnum = m_pages.count() - 1 ;


	fprintf	(stderr, "%d\n", pnum) ;

	/* Set the logical page number (so if say we print pages one	*/
	/* and three, and the pages show the page number, the output	*/
	/* will show one and three).					*/
	m_cPNum	= pnum ;
	m_page	= m_pages.at(pnum) ;
	erase	() ;
	update	() ;
}

/*  KBWriter								*/
/*  numPages	: Get number of pages					*/
/*  (returns)	: uint		: Number of pages			*/

uint	KBWriter::numPages ()
{
	return	m_pageList.count() > m_pages.count() ? m_pages   .count() :
		m_pageList.count() >		   0 ? m_pageList.count() :
						       m_pages   .count() ;
}

/*  KBWriter								*/
/*  getSize	: Get nominal page size					*/
/*  (returns)	: QSize		: Nominal page size			*/

QSize	KBWriter::getSize ()
{
	return	QSize
		(	m_pWidth  + (int)((double)(m_lMargin + m_rMargin) * PPM),
			m_pHeight + (int)((double)(m_tMargin + m_bMargin) * PPM)
		)	;
}

/*  KBWriter								*/
/*  paintEvent	: Handle paint event					*/
/*  e		: QPaintEvent *	: Pain event in question		*/
/*  (returns)	: void		:					*/

void	KBWriter::paintEvent
	(	QPaintEvent	*e
	)
{
	QPainter p (this) ;

	p.setBackgroundColor (QColor(255,255,255)) ;
	p.eraseRect	     (0, 0, width(), height()) ;

	if (m_page != 0)
		LITER
		(	KBWriterItem,
			*m_page,
			item,
			item->paintEvent (e, p)
		)

	/* If the borders setting is on then draw a border around each	*/
	/* virtual page. Useful for checking alignment without having	*/
	/* to fill each virtual page (or at least, mark the corners).	*/
	if (m_vBorders)
	{
		p.setPen   (Qt::black) ;

		for (uint row = 0 ; row < m_vRows ; row += 1)
			for (uint col = 0 ; col < m_vCols ; col += 1)
				p.drawRect				
				(	(int)(((double)m_lMargin * PPM) + col * (m_vColWidth  + m_vColGap)),
					(int)(((double)m_tMargin * PPM) + row * (m_vRowHeight + m_vRowGap)),
					m_vColWidth,
					m_vRowHeight
				)	;
	}
}

/*  KBWriter								*/
/*  startPage	: Start a page of output				*/
/*  (returns)	: void		:					*/

void	KBWriter::startPage ()
{
	m_page	= new KBWPage ;
	m_page->setAutoDelete (true) ;
	m_pages.append (m_page) ;
}

/*  KBWriter								*/
/*  add		: Add a writer item					*/
/*  item	: KBWriterItem * : The item				*/
/*  rect	: QRect &	 : Position				*/
/*  (returns)	: void		 :					*/

void	KBWriter::add
	(	KBWriterItem	*item,
		QRect		&rect
	)
{
	if (m_page == 0) startPage () ;
	m_page->append (item) ;

	fprintf
	(	stderr,
		"KBWriter::add: at (%d,%d) vby (%d,%d)\n",
		m_vAtCol,
		m_vAtRow,
		m_vAtCol * (m_vColWidth  + m_vColGap),
		m_vAtRow * (m_vRowHeight + m_vRowGap)
	)	;

	rect.moveBy
	(	m_vAtCol * (m_vColWidth  + m_vColGap) + m_offset.x(),
		m_vAtRow * (m_vRowHeight + m_vRowGap) + m_offset.y()
	)	;
}

/*  KBWriter								*/
/*  textSub	: Substitute in text					*/
/*  text	: const QString & : Text to be substituted		*/
/*  (returns)	: QString	  : Substituted version			*/

QString	KBWriter::textSub
	(	const QString	&text
	)
{
	QString	res	= "" 	;
	int	pos1	= 0	;
	int	pos2	;

	while ((pos2 = text.find ("%{", pos1)) >= 0)
	{
		res	+= text.mid (pos1, pos2 - pos1) ;
		pos1	 = pos2 + 2 ;

		if ((pos2 = text.find ("}", pos1)) < 0)
		{	res	+= "%{" ;
			break	;
		}

		QString	subs	= text.mid (pos1, pos2 - pos1) ;
		pos1	 = pos2 + 1 ;

		if (subs == "pageno")
		{
			res	+= QString().setNum(m_cPNum + 1) ;
			continue ;
		}
		if (subs == "pagecount")
		{
			res	+= QString().setNum(m_pages.count()) ;
			continue ;
		}

		res	+= "%{" + subs + "}" ;
	}

	res	+= text.mid (pos1) ;
	return	res ;
}

/*  KBWriter								*/
/*  printDoc	: Print the docucment					*/
/*  file	: const QString & : Not used				*/
/*  pageNo	: int		  : Page number of negative for all	*/
/*  (returns)	: void		  :					*/

void	KBWriter::printDoc
	(	const QString	&,
		int		pageNo
	)
{
#if	! __KB_EMBEDDED

	fprintf
	(	stderr,
		"KBWriter::printDoc: pageNo=%d\n",
		pageNo
	)	;

	if (m_printer == 0)
	{
		KBError::EError
		(	TR("Called KBWrite::printDoc but not printing"),
			QString::null,
			__ERRLOCN
		)	;
		return	;
	}


	QPaintDeviceMetrics pdm(m_printer) ;
	double	ppmmW = (double)pdm.width ()/(double)pdm.widthMM () ;
	double	ppmmH = (double)pdm.height()/(double)pdm.heightMM() ;


	m_painter->setViewport
	(	(int)((double)m_lMargin * ppmmW),
		(int)((double)m_tMargin * ppmmH),
		(int)((double)(pdm.widthMM () - m_lMargin - m_rMargin) * ppmmW), 
		(int)((double)(pdm.heightMM() - m_tMargin - m_bMargin) * ppmmH)
	)	;
	m_painter->setWindow
	(	0,
		0,
		(int)((double)(pdm.widthMM () - m_lMargin - m_rMargin) / 0.29),
		(int)((double)(pdm.heightMM() - m_tMargin - m_bMargin) / 0.29)
	)	; 

	m_painter->setClipRect
	(	(int)((double)m_lMargin * ppmmW),
		(int)((double)m_tMargin * ppmmH),
		(int)((double)(pdm.widthMM () - m_lMargin - m_rMargin) * ppmmW), 
		(int)((double)(pdm.heightMM() - m_tMargin - m_bMargin) * ppmmH)
	)	;
	m_painter->setClipping (true) ;


	if (pageNo < 0)
	{
		/* Ready to output all pages. Loop through the entries	*/
		/* in the page list, mapping to logical pages and	*/
		/* ensuring that we stay in range.			*/
		bool	 first	 = true ;

		if (m_pageList.count() == 0)
		{
			for (uint slot = 0 ; slot < m_pages.count() ; slot += 1)
			{
				m_cPNum = slot ;

				if (!first)
					m_printer->newPage() ;

				LITER
				(	KBWriterItem,
					*m_pages.at(m_cPNum),
					item,
					item->paintEvent (0, *m_painter)
				)

				first	= false ;
			}
		}
		else
		{
			for (uint slot = 0 ; slot < m_pageList.count() ; slot += 1)
			{
				m_cPNum = m_pageList[slot] ;
				if (m_cPNum >= 1)
					m_cPNum -= 1 ;

				if (m_cPNum >= m_pages.count())
					break ;

				if (!first)
					m_printer->newPage() ;

				LITER
				(	KBWriterItem,
					*m_pages.at(m_cPNum),
					item,
					item->paintEvent (0, *m_painter)
				)

				first	= false ;
			}
		}
	}
	else
	{
		/* Outputing a single specified page. Paranoia check	*/
		/* that the page exists.				*/
		m_cPNum = pageNo ;
		if (m_pages.at(m_cPNum) != 0)
			LITER
			(	KBWriterItem,
				*m_pages.at(m_cPNum),
				item,
				item->paintEvent (0, *m_painter)
			)
	}
	
	m_painter->end   () ;
	DELOBJ	(m_painter) ;
	DELOBJ	(m_printer) ;
#endif
}

/*  KBWriter								*/
/*  adjust	: Adjust control rectangle for margins			*/
/*  rect	: QRect &	: Recetangle to agjust			*/
/*  (returns)	: void		:					*/

void	KBWriter::adjust
	(	QRect		&rect
	)
{
#if	! __KB_EMBEDDED
	/* If we are *not* printing then we need to handle margin	*/
	/* offsets ourselves.						*/
	if (m_printer == 0)
		rect.moveBy
		(	(int)((double)m_lMargin * PPM),
			(int)((double)m_tMargin * PPM)
		)	;
#endif
}
