/***************************************************************************
    file	         : kb_docroot.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifndef		_WIN32
#include	"kb_docroot.moc"
#else
#include	"kb_docroot.h"
#endif

#include	"kb_callback.h"
#include	"kb_script.h"
#include	"kb_import.h"
#include	"kb_param.h"
#include	"kb_appptr.h"
#include	"kb_paramsetdlg.h"


/*  KBDocRoot								*/
/*  KBDoctRoot	: Common document root class constructor		*/
/*  parent	: KBNode *	  : Parent (actually, enclosing) node	*/
/*  children	: QList<KBNode> & : Children of said node		*/
/*  location	: KBLocation &	  : Document location			*/
/*  (returns)	: KBDocRoot	  :					*/

KBDocRoot::KBDocRoot
	(	KBNode			*parent,
		QList<KBNode>		&children,
		const KBLocation	&location
	)
	:
	m_parent	(parent),
	m_children	(children),
	m_location 	(location)
{
	reset () ;
}

/*  KBDocRoot								*/
/*  ~KBDoctRoot	: Common document root class destructor			*/

KBDocRoot::~KBDocRoot ()
{
	DELOBJ	(m_paramDict) ;
}

/*  KBDocRoot								*/
/*  setParamDict: Set parameter dictionary				*/
/*  _paramDict	: QDict<QString>&: Parameter dictionary			*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: KB::ShowRC	 : OK/Error/Cancel			*/

KB::ShowRC
	KBDocRoot::setParamDict
	(	const QDict<QString>	&_paramDict,
		KBError			&pError
	)
{
	/* Delete any existing parameter dictionary and set a new copy	*/
	/* of the new one. Then scan the document to locate all		*/
	/* defined paramaters; if there are none then we can return	*/
	/* without anything more to do.					*/
	DELOBJ	(m_paramDict) ;
	m_paramDict = new KBAttrDict (_paramDict) ;

	QDict<KBParamSet> paramSet    ;
	paramSet.setAutoDelete  (true) ;
	m_parent->findAllParams (paramSet) ;

	if (paramSet.count() == 0)
		return KB::ShowRCOK ;

	/* Scan the parameter dictionary and update any matching	*/
	/* parameters with the supplied values. Hence passed parameter	*/
	/* values override parameters defined in the document.		*/
	QDictIterator<QString> pIter (*m_paramDict) ;
	while (pIter.current() != 0)
	{
		KBParamSet *pSet = paramSet.find(pIter.currentKey()) ;
		if (pSet != 0)
		{
			pSet->m_value	= *pIter.current() ;
			pSet->m_set	= true	;
		}
		pIter += 1 ;
	}

	/* Run the user paramater set dialog, unless there we no	*/
	/* parameters left to set. If the user cancels this then	*/
	/* return in error.						*/
	bool	      ok    ;
	KBParamSetDlg pDlg  (TR("Set Parameters"), paramSet, this, pError, ok) ;
	if (!ok) return	 KB::ShowRCError ;

	if (pDlg.anyParams() && !pDlg.exec ())
	{
		pError	= KBError
			  (	KBError::Warning,
				TR("User cancelled parameter dialog"),
				QString::null,
				__ERRLOCN
			  )	;
		return	KB::ShowRCCancel ;
	}

	QDictIterator<KBParamSet> iter (paramSet) ;
	KBParamSet		  *pSet		  ;

	while ((pSet = iter.current()) != 0)
	{
		m_paramDict->replace (iter.currentKey(), new QString(pSet->m_value)) ;
		iter += 1 ;
	}

	return	KB::ShowRCOK ;
}

/*  KBDocRoot								*/
/*  reset	: Reset document root information			*/
/*  (returns)	: void		 :					*/

void	KBDocRoot::reset ()
{
	m_scrIface  		= 0     ;
	m_paramDict 		= 0	;
	m_gotExecError		= false	;
	m_docStatus.m_locking	= KBQryBase::NoLocking	;
}

/*  KBDocRoot								*/
/*  doExecError	: Execution error reported				*/
/*  (returns)	: void		:					*/

void	KBDocRoot::doExecError ()
{
	m_gotExecError	= true	;
	emit	execError ()	;
}

/*  KBDocRoot								*/
/*  doRequestClose							*/
/*		: Request to close the document				*/
/*  rc		: int		: Return code				*/
/*  (returns)	: void		:					*/

void	KBDocRoot::doRequestClose
	(	int	rc
	)
{
	emit	requestClose (rc) ;
}

void	KBDocRoot::doSetLocking
	(	KBQryBase::Locking	locking
	)
{
	m_docStatus.m_locking = locking  ;
	emit statusChange (&m_docStatus) ;
}

/*  KBDocRoot								*/
/*  getParamValue: Get value of named parameter				*/
/*  param	 : cchar *	: Parameter				*/
/*  (returns)	 : QString &	: Parameter value			*/

QString	KBDocRoot::getParamValue
	(	cchar	*param
	)
{
	/* The parameter may be either "name" or "name:defval", so we	*/
	/* split on the colon first ...					*/
	QString		res	= QString::null ;
	QStringList	bits	= QStringList::split (':', param) ;

	/* ... then, if there is a paramater dictionary, see if the	*/
	/* parameter has been defined.					*/
	if (m_paramDict != 0)
	{	QString	*val = m_paramDict->find (bits[0]) ;
		if (val != 0) res = *val ;
	}

	/* If we don't have a value, and the parameter was of the form	*/
	/* "name:defval", then use that default value.			*/
	if (res.isEmpty() && (bits.count() > 1)) res = bits[1] ;

	return	res	;
}

/*  KBDocRoot								*/
/*  setParamValue: Set value of named parameter				*/
/*  name	 : cchar *	   : Parameter name			*/
/*  value	 : const QString & : Parameter value			*/
/*  (returns)	 : void		   :					*/

void	KBDocRoot::setParamValue
	(	cchar		*name,
		const QString	&value
	)
{
	if (m_paramDict != 0)
		m_paramDict->replace (name, new QString(value)) ;
}

/*  KBDocRoot								*/
/*  getScriptIF	: Get script interface					*/
/*  language	: const QString & : Script language			*/
/*  ok		: bool &	  : Error flag				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: KBScriptIF *	  : Script interface or null on error	*/

KBScriptIF
	*KBDocRoot::getScriptIF
	(	QString		language,
		bool		&ok,
		KBError		&pError
	)
{
	KBScriptIF	*iface	= 0 ;

	if (language.isNull())
		language = m_parent->getAttrVal ("language") ;

	if (language.isEmpty())
	{
		pError	= KBError
			  (	KBError::Error,
				TR("No scripting language specified"),
				QString::null,
				__ERRLOCN
			  )	;
		ok	= false	;
		return	0	;
	}

	if ((iface = LinkKBScript (language, pError)) == 0)
	{	ok	= false ;
		return	0	;
	}

	ok	= true	;
	return	iface	;
}

/*  KBDocRoot								*/
/*  loadScripting: Load scripting and modules				*/
/*  ok		 : bool &	   : Error flag				*/
/*  pError	 : KBError &	   : Error return			*/
/*  (returns)	 : KBScriptIF *	   : Script interface or null on error	*/

KBScriptIF
	*KBDocRoot::loadScripting
	(	bool		&ok,
		KBError		&pError
	)
{
	/* See if a scripting interface is already present, in which	*/
	/* case return that.						*/
	if (m_scrIface != 0)
	{	ok	= true	   ;
		return	m_scrIface ;
	}

	/* See if a script language is specified, in which case we need	*/
	/* to load its interface. Set an error and return failure if	*/
	/* this fails.							*/
	QString	 language = m_parent->getAttrVal ("language") ;
	if (!language.isEmpty())
		if ((m_scrIface = LinkKBScript (language, pError)) == 0)
		{	ok	= false ;
			return	0	;
		}

	m_scripts.clear () ;
	m_imports.clear () ;

	LITER
	(	KBNode,
		m_children,
		child,

		if (child->isImport() != 0)
			m_imports.append (child->isImport()->getModuleName()) ;
		if (child->isScript() != 0)
			m_scripts.append (child->isScript()->getModuleName()) ;
	)

	/* Load all specified script modules. We check inside the loop	*/
	/* that we have a scripting interface, so that the user need	*/
	/* not set a language unless there is a script.			*/
	LITER
	(	KBNode,
		m_children,
		node,

		KBScript *script ;
		if ((script = node->isScript()) == 0) continue ;

		if (m_scrIface == 0)
		{
			pError	= KBError
				  (	KBError::Error,
					"No script language",
					QString("Modules %1,...").arg(script->getModuleName()),
					__ERRLOCN
				  )	;
			ok	= false	;
			return	0	;
		}

		KBLocation scrloc
			   (	m_location.dbInfo,
				"script",
				m_location.docLocn,
				script->getModuleName(),
				language
			   )	  ;
		QString	   eText  ;
		QString	   ePatt  ;

		if (!m_scrIface->load (scrloc, eText, ePatt, pError))
		{
			fprintf
			(	stderr,
				"KBDocRoot::loadScripting: scrIface->load failed [%s][%s][%s][%s]\n",
				(cchar *)scrloc.ident(),
				(cchar *)eText,
				(cchar *)pError.getMessage(),
				(cchar *)pError.getDetails()
			)	;

			KBAppPtr::getCallback()->editScript (scrloc, eText, ePatt, 0, pError) ;
			ok	= false	;
			return	0	;
		}
	)

	ok	= true	   ;
	return	m_scrIface ;
}

