/***************************************************************************
    file	         : kb_serverdlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qregexp.h>

#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_database.h"
#include	"kb_dbinfo.h"
#include	"kb_dbdociter.h"
#include	"kb_notifier.h"
#include	"kb_callback.h"
#include	"kb_appptr.h"
#include	"kb_attrdict.h"
#include	"kb_gui.h"
#include	"kb_options.h"
#include	"kb_tableinfo.h"
#include	"kb_rawsql.h"

#ifndef 	_WIN32
#include	"kb_serverdlg.moc"
#else
#include 	"kb_serverdlg.h"
#endif

#include	"kb_dbadvanceddlg.h"




class	KBServerDriver : public QListBoxText
{
	QString		m_tag	;

public	:

	inline	KBServerDriver
		(	QComboBox	*parent,
			const QString	&tag,
			const QString	&comment
		)
		:
		QListBoxText	(parent->listBox(), comment),
		m_tag		(tag)
	{
	}

	inline	const QString	&tag ()
	{
		return	m_tag	;
	}
}	;


/*  KBServerDlg								*/
/*  KBServerDlg	: Constructor for database server dialog		*/
/*  parent	: KBDBaseDlg *	  : Parent database dialog		*/
/*  dbInfo	: KBDBInfoData *  : Database information		*/
/*  gui		: KBaseGUI *	  : GUI for this viewer			*/
/*  (returns)	: KBServerDlg	  :					*/

KBServerDlg::KBServerDlg
	(	KBDBaseDlg	*parent,
		KBDBInfoData	*dbInfo,
		KBaseGUI	*
	)
	:
	QDialog	(parent),
	m_dbaseDlg	(parent),
	m_serverList	(this),
#if	__KB_EMBEDDED
	m_editWidget	(this),
#else
	m_editGroup 	(1, Qt::Horizontal, TR("Server Settings"), this),
	m_editWidget	(&m_editGroup ),
#endif
	m_serverName	(&m_editWidget),
	m_dbType	(&m_editWidget),
	m_hostName	(&m_editWidget),
	m_dbName	(&m_editWidget),
	m_userName	(&m_editWidget),
	m_password	(&m_editWidget),
	m_portNumber	(&m_editWidget),
	m_socketName	(&m_editWidget),
	m_flags		(&m_editWidget),
	m_showPwd	(TR("Show"	   ), &m_editWidget),
	m_disabled	(TR("Disabled"     ), &m_editWidget),
	m_autostart	(TR("AutoStart"    ), &m_editWidget),
	m_bEdit		(this),
	m_bAdvanced	(this),
	m_bSave		(this),
	m_bNew		(this),
	m_bCancel	(this),
	m_bDelete	(this),
	m_bDatabases	(&m_editWidget),
	m_dbInfo	(dbInfo)
{
	m_dbType.setListBox (new QListBox (&m_dbType)) ;

	QLabel	*lServerName	= new QLabel(TR("Server"	), &m_editWidget) ;
	QLabel	*lDbType	= new QLabel(TR("Type"		), &m_editWidget) ;
	QLabel	*lHostName	= new QLabel(TR("Host"		), &m_editWidget) ;
	QLabel	*lDbName	= new QLabel(TR("Database"	), &m_editWidget) ;
	QLabel	*lUserName	= new QLabel(TR("User"		), &m_editWidget) ;
	QLabel	*lPassword	= new QLabel(TR("Password"	), &m_editWidget) ;
	QLabel	*lPortSocket	= new QLabel(TR("Port/Socket"	), &m_editWidget) ;
	QLabel	*lFlags		= new QLabel(TR("Flags"		), &m_editWidget) ;

#if	__KB_EMBEDDED
	QVBoxLayout	*layMain    = new QVBoxLayout (this) ;
	layMain->addWidget (&m_serverList) ;
	layMain->addWidget (&m_editWidget) ;
	QHBoxLayout	*layButtons = new QHBoxLayout (layMain) ;

	connect	(&m_serverList, SIGNAL(activated(int)), SLOT(showServer())) ;

	m_editWidget .setMinimumWidth (200) ;
	m_serverList .setMinimumWidth (200) ;

	layButtons ->addStretch() ;
	layButtons ->addWidget (&m_bEdit,     0, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bAdvanced, 0, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bNew,      0, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bSave,     0, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bCancel,   0, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bDelete,   0, Qt::AlignRight) ;

#else
	QVBoxLayout	*layMain    = new QVBoxLayout (this) ;
	QHBoxLayout	*layEdit    = new QHBoxLayout (layMain) ;
	QGridLayout	*layButtons = new QGridLayout (layMain) ;

	layEdit   ->addWidget (&m_serverList) ;
	layEdit	  ->addWidget (&m_editGroup ) ;
	layEdit   ->setSpacing(KBOptions::getDlgSpacing()) ;
	layEdit   ->setMargin (KBOptions::getDlgMargin ()) ;

	connect	(&m_serverList, SIGNAL(selected   (int)), SLOT(clickEdit ())) ;
	connect	(&m_serverList, SIGNAL(highlighted(int)), SLOT(showServer())) ;

	layButtons ->setColStretch(0, 1) ;
	layButtons ->addWidget (&m_bEdit,     0, 0, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bAdvanced, 0, 1, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bNew,      0, 2, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bSave,     1, 0, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bCancel,   1, 1, Qt::AlignRight) ;
	layButtons ->addWidget (&m_bDelete,   1, 2, Qt::AlignRight) ;
#endif

	m_bEdit    .setText (TR("&Edit"    )) ;
	m_bSave    .setText (TR("&Save"    )) ;
	m_bNew	   .setText (TR("&New"     )) ;
#if	__KB_EMBEDDED
	m_bAdvanced.setText (TR("&Adv"     )) ;
	m_bCancel  .setText (TR("&Can"	   )) ;
	m_bDelete  .setText (TR("&Del"     )) ;
#else
	m_bAdvanced.setText (TR("&Advanced")) ;
	m_bCancel  .setText (TR("&Cancel"  )) ;
	m_bDelete  .setText (TR("&Delete"  )) ;
#endif

	QSize	s	= QSize(0,0) ;
	s	= s.expandedTo (m_bEdit    .sizeHint()) ;
	s	= s.expandedTo (m_bSave    .sizeHint()) ;
	s	= s.expandedTo (m_bNew     .sizeHint()) ;
	s	= s.expandedTo (m_bCancel  .sizeHint()) ;
	s	= s.expandedTo (m_bAdvanced.sizeHint()) ;
	s	= s.expandedTo (m_bDelete  .sizeHint()) ;

	m_bEdit	   .setMinimumSize (s) ;
	m_bSave	   .setMinimumSize (s) ;
	m_bNew	   .setMinimumSize (s) ;
	m_bCancel  .setMinimumSize (s) ;
	m_bAdvanced.setMinimumSize (s) ;
	m_bDelete  .setMinimumSize (s) ;

	m_bDatabases.setText     ("...") ;
	m_password  .setEchoMode (QLineEdit::Password) ;

	QGridLayout * laySettings = new QGridLayout (&m_editWidget) ;
	laySettings->addWidget		(lServerName,	0, 0) ;
	laySettings->addMultiCellWidget (&m_serverName,	0, 0,  1, 4) ;
	laySettings->addWidget		(lDbType,	1, 0) ;
	laySettings->addMultiCellWidget (&m_dbType,	1, 1,  1, 4) ;
	laySettings->addWidget		(lHostName,	2, 0) ;
	laySettings->addMultiCellWidget (&m_hostName,	2, 2,  1, 4) ;
	laySettings->addWidget		(lDbName,	3, 0) ;
	laySettings->addMultiCellWidget (&m_dbName,	3, 3,  1, 3) ;
	laySettings->addWidget		(&m_bDatabases, 3, 4) ;
	laySettings->addWidget		(lUserName,	4, 0) ;
	laySettings->addMultiCellWidget (&m_userName,	4, 4,  1, 3) ;
	laySettings->addWidget		(lPassword,	5, 0) ;
	laySettings->addMultiCellWidget (&m_password,	5, 5,  1, 3) ;
	laySettings->addMultiCellWidget	(&m_showPwd,	5, 5,  4, 4) ;
	laySettings->addWidget		(lPortSocket,	6, 0) ;
	laySettings->addMultiCellWidget	(&m_portNumber,	6, 6,  1, 2) ;
	laySettings->addMultiCellWidget	(&m_socketName,	6, 6,  3, 4) ;
	laySettings->addWidget		(lFlags,	7, 0) ;
	laySettings->addMultiCellWidget (&m_flags,	7, 7,  1, 4) ;
	laySettings->addMultiCellWidget (&m_disabled,	8, 8,  1, 1) ;
	laySettings->addMultiCellWidget (&m_autostart,	8, 8,  2, 4) ;

	laySettings->setSpacing(KBOptions::getDlgSpacing()) ;
	laySettings->setMargin (KBOptions::getDlgMargin ()) ;


	connect (&m_bEdit,     SIGNAL(clicked  ()), 	SLOT(clickEdit    ())) ;
	connect (&m_bAdvanced, SIGNAL(clicked  ()), 	SLOT(clickAdvanced())) ;
	connect (&m_bSave,     SIGNAL(clicked  ()), 	SLOT(clickSave    ())) ;
	connect (&m_bNew,      SIGNAL(clicked  ()), 	SLOT(clickNew     ())) ;
	connect (&m_bCancel,   SIGNAL(clicked  ()), 	SLOT(clickCancel  ())) ;
	connect (&m_bDelete,   SIGNAL(clicked  ()), 	SLOT(clickDelete  ())) ;
	connect	(&m_dbType,    SIGNAL(activated(int)),	SLOT(typeChange   ())) ;

	connect	(&m_bDatabases,SIGNAL(clicked  ()),	SLOT(showDatabases())) ;
	connect (&m_showPwd,   SIGNAL(toggled  (bool)) ,SLOT(showPassword (bool))) ;

	m_dbName.setEditable  (true) ;
	setEnabledAll (false, false) ;

	KBDriverDetailsList	drvList	;
	KBError			error	;

	if (KBServer::listDrivers (drvList, error))
	{
		new KBServerDriver (&m_dbType, QString::null, QString::null) ;

		for (uint idx = 0 ; idx < drvList.count() ; idx += 1)
			new KBServerDriver
			(	&m_dbType,
				drvList[idx].tag    (),
				drvList[idx].comment()
			)	;
	}
	else	error.DISPLAY () ;

	loadDBInfo    () ;

#if	! __KB_EMBEDDED
	int	w = m_serverList.sizeHint().width() ;
	if	(w < 100) w = 100 ;
	else if (w > 150) w = 150 ;
	m_serverList .setMinimumWidth (w) ;
#endif
}

/*  KBServerDlg								*/
/*  ~KBServerDlg: Destructor for database server dialog			*/
/*  (returns)	:		:					*/

KBServerDlg::~KBServerDlg ()
{
}

/*  KBServerDlg								*/
/*  loadDBInfo	: Load database information				*/
/*  (returns)	: void		:					*/

void	KBServerDlg::loadDBInfo ()
{
	QListIterator<KBServerInfo> *svIter = m_dbInfo->getServerIter() ;
	KBServerInfo  *svInfo ;

	m_serverList.clear () ;

	/* Insert the !Files entry first so that it appears in front of	*/
	/* all other server entries.					*/
	svInfo	= m_dbInfo->getFilesServer() ;
	m_serverList.insertItem (svInfo->m_serverName) ;

	while ((svInfo = svIter->current()) != 0)
	{
		(*svIter) += 1   ;
		if (svInfo == m_dbInfo->getFilesServer()) continue ;

		KBDBLink      dbLink  ;

		m_serverList.insertItem (svInfo->m_serverName) ;

		if (!svInfo->disabled())
		{
			if (!dbLink.connect (m_dbInfo, svInfo->m_serverName, true))
			{
				dbLink.lastError().DISPLAY() ;
				svInfo->m_isDisabled = true  ;
			}
		}
	}

	delete	svIter	;

	m_currServer  = 0 ;
	m_currIndex   = 0 ;
	m_editServer  = 0 ;

	setEnabledAll	(false, false) ;
	showServer	(0) ;
	m_serverList.setCurrentItem (0) ;
}

/*  KBServerDlg								*/
/*  clearServerInfo							*/
/*		: Clear all fields					*/
/*  (returns)	: void		:					*/

void	KBServerDlg::clearServerInfo ()
{
	m_serverName.setText ("") ;
	m_hostName  .setText ("") ;
	m_userName  .setText ("") ;
	m_password  .setText ("") ;
	m_portNumber.setText ("") ;
	m_socketName.setText ("") ;
	m_flags     .setText ("") ;
	m_disabled  .setChecked (false) ;
	m_autostart .setChecked (false) ;
	m_showPwd   .setChecked (false) ;

	m_dbName    .clear   ()	  ;
}

/*  KBServerDlg								*/
/*  showServerInfo							*/
/*		: Show server information				*/
/*  svInfo	: KBServerData * : Server				*/
/*  enable	: bool		 : True to enable update		*/
/*  (returns)	: void		 :					*/

void	KBServerDlg::showServerInfo
	(	KBServerData	*svInfo,
		bool		enable
	)
{
	m_serverName.setText    (svInfo->m_serverName) ;
	m_hostName  .setText    (svInfo->m_hostName  ) ;
	m_dbName    .setEditText(svInfo->m_dbName    ) ;
	m_userName  .setText    (svInfo->m_userName  ) ;
	m_password  .setText    (svInfo->m_password  ) ;
	m_portNumber.setText    (svInfo->m_portNumber) ;
	m_socketName.setText    (svInfo->m_socketName) ;
	m_flags     .setText    (svInfo->m_flags     ) ;
	m_disabled  .setChecked (svInfo->m_isDisabled) ;
	m_autostart .setChecked (svInfo->m_autoStart ) ;

	QString	 dbType	 = svInfo->m_dbType ;
	KBServer *server = dbType.isEmpty() ? 0 : getDriverServer(dbType) ;
	bool	 isFile	 = svInfo->m_serverName == KBLocation::m_pFile   ;

	setEnabledAll
	(	enable,
		enable && !isFile,
		server == 0 ? 0xffffffff : server->optionFlags()
	)	;

	DELOBJ	(server) ;
}

/*  KBServerDlg								*/
/*  setEnabledAll: Perform setEnabled on all edit controls		*/
/*  enable	 : bool		: Enable/disable			*/
/*  server	 : bool		: Enable/disable server name		*/
/*  optionFlags	 : uint		: Option enable flags			*/
/*  (returns)	 : void		:					*/

void	KBServerDlg::setEnabledAll
	(	bool	enable,
		bool	server,
		uint	opts
	)
{
	m_serverName.setEnabled (server) ;
	m_dbType    .setEnabled (enable) ;
	m_hostName  .setEnabled (enable && ((opts & AF_HOST        ) != 0)) ;
	m_dbName    .setEnabled (enable) ;
	m_userName  .setEnabled (enable && ((opts & AF_USERPASSWORD) != 0)) ;
	m_password  .setEnabled (enable && ((opts & AF_USERPASSWORD) != 0)) ;
	m_portNumber.setEnabled (enable && ((opts & AF_PORTNUMBER  ) != 0)) ;
	m_socketName.setEnabled (enable && ((opts & AF_SOCKETNAME  ) != 0)) ;
	m_flags     .setEnabled (enable && ((opts & AF_FLAGS       ) != 0)) ;
	m_disabled  .setEnabled (enable) ;
	m_autostart .setEnabled (enable) ;
}

/*  KBServerDlg								*/
/*  showServerType: Check if server type is recognised			*/
/*  svInfo	  : KBServerData *  : Server				*/
/*  (returns)	  : bool	    : Success				*/

bool	KBServerDlg::showServerType
	(	KBServerData	*svInfo
	)
{
	/* Scan the driver type combo for the driver type for this	*/
	/* server. If it is not found (either someone edited the	*/
	/* database file, or a driver is missing) then prompt if we	*/
	/* want to continue.						*/
	int	dbIdx  ;

	for (dbIdx = 1 ; dbIdx < m_dbType.count() ; dbIdx += 1)
		if (((KBServerDriver *)m_dbType.listBox()->item(dbIdx))->tag() == svInfo->m_dbType)
			break	;

	if (dbIdx >= m_dbType.count())
	{
		if (!svInfo->m_dbType.isEmpty() || (svInfo->m_serverName != KBLocation::m_pFile))
		{
			if (TKMessageBox::questionYesNo
				(	0,
					QString(TR("Server \"%1\" has an unrecognised type \"%2\", continue?"))
							.arg(svInfo->m_serverName)
							.arg(svInfo->m_dbType),
					TR("Edit server")
				) != TKMessageBox::Yes) return false ;
		}

		dbIdx	= 0 ;
	}

	m_dbType.setCurrentItem (dbIdx) ;
	return	true ;
}

/*  KBServerDlg								*/
/*  showServer	: Show server values					*/
/*  serverNo	: int		: Server index in list			*/
/*  (returns)	: void		:					*/

void	KBServerDlg::showServer
	(	int	serverNo
	)
{
	/* If some other property is already being edited then we	*/
	/* ignore this, also skip if we cannot find the server although	*/
	/* this should not happen.					*/
	if (m_editServer != 0) return ;

	KBServerData *svInfo = m_dbInfo->findServer (m_serverList.text (serverNo)) ;
	if (svInfo == 0) return ; 

	if (!showServerType (svInfo)) return ;
	showServerInfo (svInfo, false) ;

	m_bEdit	    .setEnabled (true ) ;
	m_bAdvanced .setEnabled (false) ;
	m_bSave	    .setEnabled (false) ;
	m_bNew	    .setEnabled (true ) ;
	m_bCancel   .setEnabled (false) ;
	m_bDelete   .setEnabled (serverNo != 0) ;
	m_bDatabases.setEnabled (false) ;
}

/*  KBServerDlg								*/
/*  showServer	: Show server values					*/
/*  (returns)	: void		:					*/

void	KBServerDlg::showServer ()
{
	showServer (m_serverList.currentItem()) ;
}

void	KBServerDlg::showPassword
	(	bool		show
	)
{
	m_password.setEchoMode (show ? QLineEdit::Normal : QLineEdit::Password) ;
}

/*  KBServerDlg								*/
/*  showDatabases							*/
/*		: Load database list into database combo		*/
/*  (returns)	: void		:					*/

void	KBServerDlg::showDatabases ()
{
	fprintf
	(	stderr,
		"KBServerDlg::clickShowDatabases: %p\n",
		(void *)m_editServer
	)	;

	if (m_editServer != 0)
	{
		KBServerDriver	*i	= (KBServerDriver *)m_dbType.listBox()->item(m_dbType.currentItem()) ;
		const QString	tag	= i->tag() ;
		KBServer	*server	= tag.isEmpty() ? 0 : getDriverServer (tag) ;

		if (server == 0) return ;

		KBServerInfo info
			     (		0,
					m_serverName.text(),
					tag,
					m_hostName  .text(),
					0,
					m_userName  .text(),
					m_password  .text()
			     )	;

		if (!server->connect (&info))
		{	server->lastError().DISPLAY() ;
			delete	server	;
			return	;
		}

		QStringList dbList ;
		if (!server->listDatabases (dbList))
		{	server->lastError().DISPLAY() ;
			delete	server	;
			return	;
		}

		QString	current	= m_dbName.currentText() ;

		m_dbName.clear () ;
		m_dbName.insertStringList (dbList) ;

		for (int idx = 0 ; idx < m_dbName.count() ; idx += 1)
			if (m_dbName.text(idx) == current)
			{	m_dbName.setCurrentItem (idx) ;
				break	;
			}

		delete	server	;
	}	
}

/*  KBServerDlg								*/
/*  clickEdit	: Handle edit operation					*/
/*  (returns)	: void		:					*/

void	KBServerDlg::clickEdit ()
{
	if (m_editServer != 0) return ;

	int		serverNo = m_serverList.currentItem() ;
	KBServerData	*svInfo	 = m_dbInfo->findServer (m_serverList.currentText()) ;

	if (svInfo == 0) return ; 

	if (svInfo->linkIsOpen ())
	{
		TKMessageBox::sorry
		(	0,
			QString(TR("Server %1 has an open connection")).arg(svInfo->m_serverName),
			TR("Cannot edit server")
		) ;
		return	   ;
	}

	if (!showServerType (svInfo)) return ;

	showServerInfo (svInfo, true) ;

	m_bEdit	    .setEnabled (false) ;
	m_bAdvanced .setEnabled (true ) ;
	m_bSave	    .setEnabled (true ) ;
	m_bNew	    .setEnabled (false) ;
	m_bCancel   .setEnabled (true ) ;
	m_bDelete   .setEnabled (false) ;
	m_serverList.setEnabled (false) ;
	m_bDatabases.setEnabled (true ) ;

	/* Note the index into the server list which is beign edited	*/
	/* and the associated server object. Then make a copy which	*/
	/* will be changed by editing; on completion we either replace	*/
	/* the original or, if the user cancels the changes, delete	*/
	/* the working copy.						*/
	m_currIndex 	= serverNo    ;
	m_currServer	= svInfo      ;
	m_editServer	= new KBServerData (m_dbInfo, svInfo) ;
}

/*  KBServerDlg								*/
/*  clickAdvanced: Handle advanced settings operation			*/
/*  (returns)	 : void		:					*/

void	KBServerDlg::clickAdvanced ()
{
	if (m_editServer != 0)
	{
		QString	 dbType	 = m_editServer->m_dbType ;
		KBServer *server = dbType.isEmpty() ? 0 : getDriverServer(dbType) ;

		KBDBAdvancedDlg advDlg
				(	m_editServer,
					server == 0 ? 0xffffffff : server->optionFlags()
				)	;
		advDlg.exec () ;
	}
}

/*  KBServerDlg								*/
/*  clickSave	: Handle save operation					*/
/*  (returns)	: void		:					*/

void	KBServerDlg::clickSave ()
{
	if (m_editServer == 0) return ;

	/* If inserting or editing any entry other than the files	*/
	/* entry, the server name must not be empty and the server	*/
	/* type must be set.						*/
	if ((m_currServer == 0) || (m_editServer->m_serverName != KBLocation::m_pFile))
	{
		/* Ignore the operation if the user has not set a	*/
		/* server name or a database server type.		*/
		if (m_serverName.text().isEmpty())
		{
			TKMessageBox::information
			(	0,
				TR("Please specify a server name"),
				TR("Cannot save server")
			)	;
			return	;
		}

		if (m_dbType.currentItem () == 0)
		{
			TKMessageBox::information
			(	0,
				TR("Please specify a server type"),
				TR("Cannot save server")
			)	;
			return	;
		}
	}


	/* If there is now an open link then we do not allow update.	*/
	/* This is unlikely, but the user might have done something	*/
	/* else like open a table, since this is not a modal dialog.	*/
	if (m_currServer != 0)
		if (m_currServer->linkIsOpen ())
		{
			TKMessageBox::sorry
			(	0,
				QString(TR("Server %1 now has an open connection")).arg(m_currServer->m_serverName),
				TR("Cannot save server")
			)	;
			return	;
		}


	/* Update the working copy with the contents of the dialog.	*/
	/* Any advanced changes will have been saved into the working	*/
	/* copy when the advanced dialog as OK'd			*/
	m_editServer->m_serverName = m_serverName.text () ;
	m_editServer->m_hostName   = m_hostName  .text () ;
	m_editServer->m_dbName     = m_dbName    .currentText () ;
	m_editServer->m_userName   = m_userName  .text () ;
	m_editServer->m_password   = m_password  .text () ;
	m_editServer->m_portNumber = m_portNumber.text () ;
	m_editServer->m_socketName = m_socketName.text () ;
	m_editServer->m_flags      = m_flags     .text () ;
	m_editServer->m_isDisabled = m_disabled  .isChecked () ;

	KBServerDriver    *i	   = (KBServerDriver *)m_dbType.listBox()->item(m_dbType.currentItem()) ;
	m_editServer->m_dbType     = i->tag() ;

	m_editServer->m_autoStart   = m_autostart .isChecked () ;
	m_editServer->m_useUserName = m_editServer->m_userName  ;
	m_editServer->m_usePassword = m_editServer->m_password  ;


	/* At this point, unless the server has been marked as		*/
	/* disabled or there is no server type (in which case it must	*/
	/* be the !Files entry), we connect to the server. This will	*/
	/* disable the server if no connection can be made.		*/
	if (!m_editServer->m_isDisabled && !m_editServer->m_dbType.isEmpty())
	{
		KBError	error	;
		if (m_editServer->getServer(error) == 0) error.DISPLAY() ;
	}


	/* If inserting a new entry then it will go at the end of the	*/
	/* server list. Otherwise, change the existing name in case it	*/
	/* has been renamed and detach the original entry.		*/
	QString	oldName	;

	if (m_currServer == 0)
	{
		m_currIndex = m_serverList.count ()	;
		m_serverList.insertItem (m_editServer->m_serverName) ;
	}
	else
	{	oldName	    = m_currServer->m_serverName ;
		m_serverList.changeItem (m_editServer->m_serverName, m_currIndex) ;
		m_dbInfo->remove (m_currServer, false) ;
	}


	/* We can now insert the server entry, which is either new or	*/
	/* a replacement, into the database information object.		*/
	m_dbInfo->insert (m_editServer) ;
	m_dbInfo->save   () ;


	DELOBJ	(m_currServer) ;

	/* Issue a notification that the server (may have) changed, so	*/
	/* that such things as the table lists are kept up to date.	*/
	KBLocation location
		   (
			m_dbInfo,
			"unknown",
			oldName,
			m_editServer->m_serverName
		   )	;
	KBNotifier::self()->nServerChanged (location) ;


	/* Note, if we were editing an existing server then the old	*/
	/* information object will have been deleted by the KBDBInfo	*/
	/* object inside the dbInfo->remove() call above.		*/
	m_currServer  = 0 ;
	m_editServer  = 0 ;

	clearServerInfo () ;
	setEnabledAll   (false, false) ;
	m_dbType    .setCurrentItem  (0) ;
	m_serverList.setEnabled   (true) ;

	if (m_currIndex >= 0)
	{
		m_serverList.setCurrentItem (m_currIndex) ;
		showServer  (m_currIndex) ;
	}

	m_currServer  = 0 ;
	m_currIndex   = 0 ;
}

/*  KBServerDlg								*/
/*  clickCancel	: Handle cancel operation				*/
/*  (returns)	: void		:					*/

void	KBServerDlg::clickCancel ()
{
	if (m_editServer == 0) return ;

	DELOBJ		(m_editServer) ;
	clearServerInfo () ;
	setEnabledAll   (false, false) ;
	m_dbType  .setCurrentItem  (0) ;

	m_serverList.setEnabled (true) ;

	int	show = m_currServer == 0 ? 0 : m_currIndex ;

	m_currServer  = 0	;
	m_currIndex   = 0	;

	m_serverList.setCurrentItem (show) ;
	showServer (show) ;
}

/*  KBServerDlg								*/
/*  clickNew	: Handle new operation					*/
/*  (returns)	: void		:					*/

void	KBServerDlg::clickNew ()
{
	if (m_editServer != 0) return ;

	clearServerInfo () ;
	setEnabledAll   (true, true) ;

	m_serverName.setText (TR("<unnamed>")) ;

	if (m_dbType.count() == 2)
		m_dbType.setCurrentItem (1) ;
	else	m_dbType.setCurrentItem (0) ;

	m_currServer   = 0    ;
	m_editServer   = new KBServerData (m_dbInfo) ;
	m_currIndex    = -1   ;

	m_serverList.setCurrentItem (m_currIndex)  ;

	m_bEdit	    .setEnabled (false) ;
	m_bAdvanced .setEnabled (true ) ;
	m_bSave	    .setEnabled (true ) ;
	m_bNew	    .setEnabled (false) ;
	m_bCancel   .setEnabled (true ) ;
	m_bDelete   .setEnabled (false) ;
	m_serverList.setEnabled (false) ;
	m_bDatabases.setEnabled (true ) ;

	m_serverName.setSelection (0, 9) ;
	m_serverName.setFocus	  ()	 ;
}

/*  KBServerDlg								*/
/*  clickDelete	: Handle delete operation				*/
/*  (returns)	: void		:					*/

void	KBServerDlg::clickDelete ()
{
	if (m_editServer != 0) return ;

	uint		svNo	= m_serverList.currentItem () ;
	KBServerData	*svInfo = m_dbInfo->findServer (m_serverList.text (svNo)) ;

	if ((svInfo == 0) || (svInfo->m_serverName == KBLocation::m_pFile))
		return	;

	QString		oldName	= svInfo->m_serverName ;

	m_dbInfo->remove (svInfo) ;
	m_dbInfo->save   () ;

	/* Issue a notification then the server (may have) changed, so	*/
	/* that such things as the table lists are kept up to date.	*/
	KBLocation location
		   (
			m_dbInfo,
			"unknown",
			oldName,
			""
		   )	;
	KBNotifier::self()->nServerChanged (location) ;

	m_serverList.removeItem     (svNo) ;
	m_serverList.setCurrentItem (0)	 ;
	showServer  (0) ;
}

void	KBServerDlg::typeChange ()
{
	if (m_editServer != 0)
	{
		DELOBJ	(m_editServer->m_advanced) ;

		KBServerDriver *i = (KBServerDriver *)m_dbType.listBox()->item(m_dbType.currentItem()) ;
		QString	 type	  = i->tag() ;

		if (!type.isEmpty())
			m_editServer->m_advanced = getDriverAdvanced (type)  ;

		KBServer *server = type.isEmpty() ? 0 : getDriverServer (type) ;
		bool	 isFile	 = m_editServer->m_serverName == KBLocation::m_pFile ;

		setEnabledAll
		(	true,
			!isFile,
			server == 0 ? 0xffffffff : server->optionFlags()
		)	;

		DELOBJ	(server) ;
	}
}

/*  KBServerDlg								*/
/*  autoStart	: Check for auto-start forms				*/
/*  svInfo	: KBServerInfo * : Server to check			*/
/*  autoLoc	: KBLocation &	 : Return started location if any	*/
/*  (returns)	: bool		 : Form opened				*/

bool	KBServerDlg::autoStartForm
	(	KBServerInfo	*svInfo,
		KBLocation	&autoLoc
	)
{
	QDict<QString> empty	;
	KBError	   error	;

#if	! __KB_RUNTIME
	/* The autostart option is ignored in the runtime version, that	*/
	/* is, we always act as if it is set.				*/
	if (!svInfo->m_autoStart)
		return	false	;
#endif

	autoLoc	= KBLocation
		  (	m_dbInfo,
			"form",
			svInfo->serverName(),
			svInfo->autoForm  ().isEmpty() ?
				QString("MainForm") :
				svInfo->autoForm ()
		  )	;

	QString	doc	= autoLoc.contents (error) ;
	if (doc.isEmpty())
		return	false	;

//	KBCallback *cb	= KBAppPtr::getCallback() ;
//	if (cb->openObject (autoLoc, KB::ShowAsData, empty, error, 0) == KB::ShowRCError)
//	{	error.DISPLAY ();
//		return	false	;
//	}

	fprintf
	(	stderr,
		"KBServerDlg::autoStartForm: got [%s]\n",
		(cchar *)autoLoc.ident()
	)	;

	return	true ;
}

/*  KBServerDlg								*/
/*  autoStart	: Check for auto-start forms				*/
/*  autoLoc	: KBLocation &	: Return started location if any	*/
/*  (returns)	: bool		: Form opened				*/

bool	KBServerDlg::autoStart
	(	KBLocation	&autoLoc
	)
{
	QListIterator<KBServerInfo> *svIter = m_dbInfo->getServerIter() ;
	KBServerInfo   *svInfo	;
	KBError	       error	;
	bool	       rc	= false	;

	while ((svInfo = svIter->current()) != 0)
	{
		if (autoStartForm (svInfo, autoLoc))
		{	rc = true ;
			break	  ;
		}

		(*svIter) += 1	;
	}

	delete	svIter	;

	if (!rc)
		if ((svInfo = m_dbInfo->getFilesServer()) != 0)
			if (autoStartForm (svInfo, autoLoc))
				rc	= true	;

	return	rc	;
}

void	KBServerDlg::showRawSQL
	(	KBaseApp	*kbaseApp
	)
{
	QString		server	= m_serverList.currentText()   ;
	KBServerData	*svInfo	= m_dbInfo->findServer(server) ;

	if (svInfo != 0)
		svInfo->showRawSQL (kbaseApp) ;
}

#if	! __KB_RUNTIME

void	KBServerDlg::dumpDatabase
	(	KBaseApp	*
	)
{
	extern	void	dumpDatabase
			(	KBDBInfo	*,
				const QString	&,
				const QString	&
			)	;

	KBError	error	;

	QString	target	= KBFileDialog::getExistingDirectory
			  (	QString::null,
				TR("Dump database to .....")
			  )	;
	if (target.isNull()) return ;

	dumpDatabase (m_dbInfo, m_serverList.currentText(), target) ;
}

void	KBServerDlg::loadDatabase
	(	KBaseApp	*
	)
{
	extern	void	loadDatabase
			(	KBDBInfo	*,
				const QString	&,
				const QString	&
			)	;

	KBError	error	;

	QString	target	= KBFileDialog::getExistingDirectory
			  (	QString::null,
				TR("Load database from .....")
			  )	;
	if (target.isNull()) return ;

	loadDatabase (m_dbInfo, m_serverList.currentText(), target) ;
}
#endif
