/***************************************************************************
    file	         : kb_odbc_mysql.cpp
    copyright            : (C) 1999,2000,2001 by Mike Richardson
			   (C) 2000,2001 by theKompany.com
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *     This program is licensed under the terms contained in the file      *
 *     LICENSE which is contained with the source code distribution.       *
 *                                                                         *
 ***************************************************************************/


#include	"kb_odbc.h"


namespace	ODBC_NS
{

/*  MySQLQryInsert							*/
/*  --------------							*/
/*  This is a wrapper class for KBODBCQryInsert, which can access the	*/
/*  MySQL last-inserted-key value.					*/

class	MySQLQryInsert : public KBODBCQryInsert
{
protected :

	SQLHSTMT	m_getHandle	;
	KBValue		m_newKey	;

public	:

	MySQLQryInsert (KBODBC *, bool, const QString &, const QString &) ;
virtual~MySQLQryInsert () ;

	virtual	bool	execute	   (uint, const KBValue *) ;
	virtual	bool	getNewKey  (const QString &, KBValue &, bool) ;
}	;

}

using	namespace	ODBC_NS ;

/*  ------------------------------------------------------------------  */

/*  MySQLQryInsert							*/
/*  MySQLQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBODBC *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: MySQLQryInsert  :					*/

ODBC_NS::MySQLQryInsert::MySQLQryInsert
	(	KBODBC		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBODBCQryInsert (server, data, query, tabName)
{
	m_getHandle = 0 ;
	if (m_stmHandle == 0) return ;

	if (!m_pServer->getStatement (m_getHandle)) return ;

	cchar	*g	= "select LAST_INSERT_ID()" ;
	long	odbcRC	;

	odbcRC	= SQLPrepare (m_getHandle, (uchar *)g, strlen(g)) ;
	if (!m_pServer->checkRCOK (m_getHandle, odbcRC, "Error preparing statement from ODBC"))
	{
		SQLFreeStmt (m_getHandle, SQL_DROP) ;
		m_getHandle = 0	    ;
		m_lError    = m_pServer->lastError() ;
		return	;
	}
}

/*  MySQLQryInsert							*/
/*  ~MySQLQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

ODBC_NS::MySQLQryInsert::~MySQLQryInsert ()
{
	if (m_getHandle != 0)
		SQLFreeStmt  (m_getHandle, SQL_DROP) ;
}


/*  MySQLQryInsert							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	ODBC_NS::MySQLQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	long	odbcRC	;

	if (!KBODBCQryInsert::execute (nvals, values)) return false ;

	SQLCloseCursor	(m_getHandle) ;

	odbcRC	= SQLExecute (m_getHandle) ;
	if (!m_pServer->checkRCOK(m_getHandle, odbcRC, "Error executing ODBC insert retrieve"))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	odbcRC = SQLFetch    (m_getHandle) ;
	if (!m_pServer->checkRCOK(m_getHandle, odbcRC, "Error fetching ODBC insert retrieve"))
	{
		m_lError = m_pServer->lastError () ;
		return	 false	;
	}

	char		buffer[32]	;
	SQLINTEGER	bufflen		;

	odbcRC = SQLGetData
		 (	m_getHandle,
			1,
			SQL_C_LONG,
			buffer,
			sizeof(buffer),
			&bufflen
		 )	;

	if (!m_pServer->checkRCOK(m_getHandle, odbcRC, "Error fetching ODBC retrieve"))
	{
		m_lError = m_pServer->lastError () ;
		return	 false	;
	}

	m_newKey = KBValue ((int)*(SQLINTEGER *)buffer) ;

	fprintf	(stderr, "MySQLQryInsert newKey [%s]\n", (cchar *)m_newKey.getRawText()) ;
	return	 true ;
}

/*  MySQLQryInsert							*/
/*  getNewKey	: Get new primary key value				*/
/*  primary	: const QString & : Key column name			*/
/*  _newKey	: KBValue &	  : New key value			*/
/*  prior	: bool		  : Pri-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	ODBC_NS::MySQLQryInsert::getNewKey
	(	const QString	&,
		KBValue		&newKey,
		bool		prior
	)
{
	if (prior)
	{
		newKey	 = KBValue () ;
		return	true ; 
	}

#if	0
	if (keycol != m_autocol)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Asking for insert key",
				QString ("%1, %2:%3").arg(m_tabName).arg(keycol).arg("m_autocol"),
				__ERRLOCN
			   )	;
		return	false	;
	}
#endif

	newKey	= m_newKey ;
	return	true	   ;
}





static	KBSQLSelect
	*ODBCMySQLQrySelect
	(	KBODBC		*odbc,
		bool		data,
		const QString	&select,
		bool		update
	)
{
	fprintf	(stderr, "ODBC: called ODBCMySQLQrySelect\n") ;
	return	new KBODBCQrySelect (odbc, data, select, update) ;
}

static	KBSQLUpdate
	*ODBCMySQLQryUpdate
	(	KBODBC		*odbc,
		bool		data,
		const QString	&update,
		const QString	&tabName
	)
{
	fprintf	(stderr, "ODBC: called ODBCMySQLQryUpdate\n") ;
	return new KBODBCQryUpdate (odbc, data, update, tabName) ;
}

static	KBSQLInsert
	*ODBCMySQLQryInsert
	(	KBODBC		*odbc,
		bool		data,
		const QString	&insert,
		const QString	&tabName
	)
{
	fprintf	(stderr, "ODBC: called ODBCMySQLQryInsert\n") ;
	return new MySQLQryInsert (odbc, data, insert, tabName) ;
}

static	KBSQLDelete
	*ODBCMySQLQryDelete
	(	KBODBC		*odbc,
		bool		data,
		const QString	&_delete,
		const QString	&tabName
	)
{
	fprintf	(stderr, "ODBC: called ODBCMySQLQryDelete\n") ;
	return new KBODBCQryDelete (odbc, data, _delete, tabName) ;
}


/*  KBODBC								*/
/*  ODBCMySQLDoListFields						*/
/*		: Field listing extension function			*/
/*  odbc	: KBODBC *	: Parent connection object		*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

static	bool	ODBCMySQLDoListFields
	(	KBODBC		*odbc,
		KBTableSpec	&tabSpec,
		KBError		&pError
	)
{
	SQLHSTMT stmHandle	;

	fprintf	(stderr, "ODBC: called ODBCMySQLDoListFields\n") ;

	if (!odbc->getStatement (stmHandle))
	{
		pError	= odbc->lastError() ;
		return false ;
	}

	QString	sAll	= QString("show columns from %1").arg(tabSpec.m_name) ;
	cchar	*s	= sAll ;
	long	odbcRC	;

	odbcRC	= SQLPrepare (stmHandle, (uchar *)s, strlen(s)) ;
	if (!odbc->checkRCOK (stmHandle, odbcRC, "Error preparing show columns"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		stmHandle = 0 ;
		pError	  = odbc->lastError() ;
		return	false ;
	}

	odbcRC	= SQLExecute (stmHandle) ;
	if (!odbc->checkRCOK(stmHandle, odbcRC, "Error executing show columns"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		stmHandle = 0  ;
		pError	  = odbc->lastError() ;
		return	 false ;
	}

	for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
	{
		odbcRC = SQLFetch (stmHandle) ;

		if (!odbc->checkRCOK(stmHandle, odbcRC, "Error fetching column info"))
		{
			SQLFreeStmt (stmHandle, SQL_DROP) ;
			stmHandle = 0  ;
			pError	  = odbc->lastError () ;
			return	  false	;
		}

		char		buffer[GETBUFSIZ+8]	;
		SQLINTEGER	bufflen			;

		odbcRC = SQLGetData
			 (	stmHandle,
				5 + 1,
				SQL_C_CHAR,
				buffer,
				GETBUFSIZ + 1,
				&bufflen
		 	)	;

		if (!odbc->checkRCOK(stmHandle, odbcRC, "Error fetching column info"))
		{
			SQLFreeStmt (stmHandle, SQL_DROP) ;
			stmHandle = 0  ;
			pError	  = odbc->lastError () ;
			return	  false	;
		}

		if (QString(buffer).find("auto_increment") >= 0)
			tabSpec.m_fldList.at(idx)->m_flags |= KBFieldSpec::Serial|KBFieldSpec::InsAvail ;
	}

	SQLFreeStmt (stmHandle, SQL_DROP) ;

	LITER
	(	KBFieldSpec,
		tabSpec.m_fldList,
		fSpec,

		/* If a column is a primary key and it is has fixed	*/
		/* data type, then view it as the Rekall default.	*/
		if ((fSpec->m_flags & KBFieldSpec::Primary) != 0)
			if (fSpec->m_typeIntl == KB::ITFixed)
				fSpec->m_typeName = "Primary Key"  ;
	)

	return	true	;
}

static	QString	ODBCMySQLLimitOffset
	(	int		limit,
		int		offset
	)
{
	return	QString(" limit %1,%2 ").arg(offset).arg(limit) ;

}

/*  ------------------------------------------------------------------  */

namespace	ODBC_NS
{
struct	ODBCDriverExtn	ODBCMySQLDriverExtn =
{
	"MySQL",

	ODBCMySQLQrySelect,
	ODBCMySQLQryUpdate,
	ODBCMySQLQryInsert,
	ODBCMySQLQryDelete,

	ODBCMySQLDoListFields,

	ODBCMySQLLimitOffset,

	ODBC_FLAGS_USED
}	;
}
