/***************************************************************************
    file	         : kb_mysql.cpp
    copyright            : (C) 1999,2000,2001,2002,2003,2004 by Mike Richardson
			   (C) 2000,2001,2002,2003,2004 by theKompany.com
			   (C) 2001,2002,2003,2004 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<time.h>
#include	<sys/types.h>

#ifdef 		_WIN32
#include 	<windows.h>
#endif

#include	<qarray.h>
#include	<qstring.h>
#include	<qcstring.h>
#include	<qdict.h>
#include	<qintdict.h>
#include	<qtextcodec.h>


#include	<mysql.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_databuffer.h"
#include	"kb_database.h"
#include	"kb_serverinfo.h"
#include	"kb_build.h"

#include	"kb_myadvanced.h"

#include	"kb_libloader.h"

struct	MySQLTypeMap
{	int		ident		;	/* MySQL identifier	*/
	KB::IType	itype		;	/* Internal type	*/
	char		mtype	[16]	;	/* MySQL type name	*/
	uint		flags		;	/* Various flags	*/
}	;



#define	FIELD_TYPE_UNSET	(-1)

static	MySQLTypeMap	typeMap[] =
{
/*	ident			itype		mtype		flags		*/


{	FIELD_TYPE_TINY,	KB::ITFixed,	"TinyInt",	0			},
{	FIELD_TYPE_SHORT,	KB::ITFixed,	"SmallInt",	0			},
{	FIELD_TYPE_INT24,	KB::ITFixed,	"MediumInt",	0			},
{	FIELD_TYPE_LONG,	KB::ITFixed,	"Integer",	0			},
{	FIELD_TYPE_LONGLONG, 	KB::ITFixed,	"BigInt",	0			},

{	FIELD_TYPE_FLOAT,	KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC	},
{	FIELD_TYPE_DOUBLE,	KB::ITFloat,	"Double",	FF_LENGTH|FF_PREC	},
{	FIELD_TYPE_DECIMAL,	KB::ITFloat,	"Decimal",	FF_LENGTH|FF_PREC	},

{	FIELD_TYPE_DATE,	KB::ITDate,	"Date",		0			},
{	FIELD_TYPE_TIME, 	KB::ITTime,	"Time",		0			},
{	FIELD_TYPE_DATETIME, 	KB::ITDateTime,	"DateTime",	0			},
{	FIELD_TYPE_TIMESTAMP, 	KB::ITDateTime,	"TimeStamp",	0			},

{	FIELD_TYPE_STRING,	KB::ITString,	"Char",		FF_LENGTH		},
{	FIELD_TYPE_VAR_STRING,	KB::ITString,	"VarChar",	FF_LENGTH		},

{	FIELD_TYPE_TINY_BLOB,	KB::ITBinary,	"Tinyblob",	0			},
{	FIELD_TYPE_BLOB,	KB::ITBinary,	"Blob",		0			},
{	FIELD_TYPE_MEDIUM_BLOB,	KB::ITBinary,	"Mediumblob",	0			},
{	FIELD_TYPE_LONG_BLOB,	KB::ITBinary,	"Longblob",	0			},

{	-1,			KB::ITString,	"Text",		0			}
}	;




/*  KBMySQLType								*/
/*  -----------								*/

class	KBMySQLType : public KBType
{
	MySQLTypeMap	*m_typeInfo	;

public	:

	KBMySQLType  (MySQLTypeMap *, uint, uint, bool) ;

	virtual	bool    isValid      (const QString &, KBError  &, const QString & = QString::null) ;
	virtual	void	getQueryText (KBDataArray   *, KBShared *, KBDataBuffer  &, QTextCodec * = 0) ;
}	;



/*  KBMySQL								*/
/*  -------								*/
/*  Implementation class for insertface to the MySQL database		*/

class	KBMySQL : public KBServer
{
	QString		m_socket	;
	QString		m_flags		;
	MYSQL		m_mysql		;
	QDict<QString>	m_variables	;

	bool		m_readOnly	;
	bool		m_ignoreCharset	;
	bool		m_foundRows	;

	void			loadVariables	() ;
	bool			tblCreateSQL	(QList<KBFieldSpec> &, const QString &, QString &, bool) ;
	bool			doListTables 	(KBTableDetailsList &, bool, uint) ;

	virtual	bool	 	doConnect    	(KBServerInfo  *) ;
	virtual	bool		doListTables 	(KBTableDetailsList &, uint = KB::IsAny) ;
	virtual	bool		doListFields 	(KBTableSpec   &) ;
	virtual	bool		doCreateTable	(KBTableSpec   &,  bool, bool = false) ;
	virtual	bool		doRenameTable	(cchar *, cchar *, bool) ;
	virtual	bool		doDropTable  	(cchar *,	   bool) ;

public	:

	KBMySQL	() ;
virtual~KBMySQL () ;


	virtual	KBSQLSelect	*qrySelect 	(bool, const QString &, bool) ;
	virtual	KBSQLUpdate	*qryUpdate 	(bool, const QString &, const QString &) ;
	virtual	KBSQLInsert	*qryInsert 	(bool, const QString &, const QString &) ;
	virtual	KBSQLDelete	*qryDelete 	(bool, const QString &, const QString &) ;

	virtual	bool		command		(bool, const QString &, uint, KBValue *, KBSQLSelect ** = 0) ;

	virtual bool		listDatabases	(QStringList &) ;
	virtual	QString		listTypes	() ;
	virtual	bool		tableExists	(const QString &, bool &) ;
	virtual	uint 		optionFlags	() ;

	virtual	bool		getSyntax	(QString &, KBServer::Syntax, ...) ;
	QString			getVariable	(const QString &) ;

	bool	execSQL
	(	const QString	&,
		QString		&,
		uint		, 
		const KBValue	*,
		QTextCodec	*,
		cchar		*,
		KBError		&
	)	;

	inline	int		getNumRows ()
	{
		return	mysql_affected_rows(&m_mysql) ;
	}
	inline	my_ulonglong	getInsertID()
	{
		return	mysql_insert_id    (&m_mysql) ;
	}
	inline	MYSQL_RES	*getResults()
	{
		return	mysql_store_result (&m_mysql) ;
	}
	inline	cchar		*mysqlErr  ()
	{
		return	mysql_error	   (&m_mysql) ;
	}
}	;


/*  KBMySQLQrySelect							*/
/*  ----------------							*/
/*  Implementation class for select queries on the MySQL database	*/

class	KBMySQLQrySelect : public KBSQLSelect
{
	KBMySQL		*m_server	;
	MYSQL_RES	*m_myres	;
	MYSQL_FIELD	*m_fields	;
	MYSQL_ROW	m_myrow		;
	uint		m_crow		;
	ulong		*m_lengths	;


public	:
	KBMySQLQrySelect (KBMySQL *, bool, const QString &) ;
virtual~KBMySQLQrySelect () ;

	KBMySQLQrySelect (KBMySQL *, bool, const QString &, MYSQL_RES *) ;

	virtual	bool	execute		(uint, const KBValue * ) ;
	virtual	KBValue	getField	(uint, uint, KBValue::VTrans) ;
	virtual	QString	getFieldName	(uint) ;
}	;


/*  KBMySQLQryUpdate							*/
/*  ----------------							*/
/*  Implementation class for update queries on the MySQL database	*/

class	KBMySQLQryUpdate : public KBSQLUpdate
{
	KBMySQL	*m_server	;

public	:
	KBMySQLQryUpdate (KBMySQL *, bool, const QString &, const QString &) ;
virtual~KBMySQLQryUpdate () ;

	virtual	bool	execute	 (uint, const KBValue * ) ;

}	;

/*  KBMySQLQryInsert							*/
/*  ----------------							*/
/*  Implementation class for insert queries on the MySQL database	*/

class	KBMySQLQryInsert : public KBSQLInsert
{
	KBMySQL		*m_server	;
	QString		m_autocol	;
	KBValue		m_newKey	;

public	:
	KBMySQLQryInsert (KBMySQL *, bool, const QString &, const QString &) ;
virtual~KBMySQLQryInsert () ;

	virtual	bool	execute	  (uint, const KBValue *) ;
	virtual	bool	getNewKey (const QString &, KBValue &, bool) ;
}	;

/*  KBMySQLQryDelete							*/
/*  ----------------							*/
/*  Implementation class for delete queries on the MySQL database	*/

class	KBMySQLQryDelete : public KBSQLDelete
{
	KBMySQL		*m_server	;

public	:
	KBMySQLQryDelete (KBMySQL *, bool, const QString &, const QString &) ;
virtual~KBMySQLQryDelete () ;

	virtual	bool	execute	 (uint, const KBValue * ) ;
}	;


static	QIntDict<MySQLTypeMap>	dIdentToType	;

/*  ------------------------------------------------------------------  */

/*  KBMySQLType								*/
/*  KBMySQLType	: Constructor for MySQL type object			*/
/*  typeInfo	: MySQLTypeMap * : Type information			*/
/*  length	: uint		 : Underlying database length		*/
/*  prec	: uint		 : Underlying database precision	*/
/*  nullOK	: bool		 : True if null is OK			*/
/*  (returns)	: KBMySQLType	 :					*/

KBMySQLType::KBMySQLType
	(	MySQLTypeMap	*typeInfo,
		uint	  	length,
		uint		prec,
		bool	  	nullOK
	)
	:
	KBType
	(	"MySQL",
		typeInfo == 0 ? KB::ITUnknown : typeInfo->itype,
		length,
		prec,
		nullOK
	),
	m_typeInfo	(typeInfo)
{
}

/*  KBMySQLType								*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Value to check			*/
/*  pError	: KBValue &	  : Error return			*/
/*  where	: const QString & : Caller				*/
/*  (returns)	: bool		  : Valid				*/

bool	KBMySQLType::isValid
	(	const QString	 &value,
		KBError		 &pError,
		const QString	&where
	)
{
	/* *** Need better checking here, eg., length			*/
	return	KBType::isValid (value, pError, where) ;
}

/*  KBMySQLType								*/
/*  getQueryText							*/
/*  value	: KBDataArray  * : Raw text of value to convert		*/
/*  d		: KBShared     * : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Results buffer			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: void		 :					*/

void	KBMySQLType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer,
		QTextCodec	*codec
	)
{
	if ((value != 0) && (m_iType == KB::ITBinary))
	{
		QCString res	= "" ;
		char	 *tmp	= (char *)malloc(2 * value->m_length + 1) ;

		mysql_escape_string (tmp, value->m_data, value->m_length) ;

		buffer.append ("'")	;
		buffer.append (tmp)	;
		buffer.append ("'")	;

		free	((void *)tmp) ;

		return	;
	}

	KBType::getQueryText (value, d, buffer, codec) ;
}





/*  ------------------------------------------------------------------  */

/*  KBMySQL								*/
/*  KBMySQL	: Constructor for MySQL database connection class	*/
/*  (returns)	: KBServer	:					*/

KBMySQL::KBMySQL ()
	:
	KBServer ()
{
	/* Initialise the MySQL connection structure. This does not	*/
	/* return a result.						*/
	mysql_init (&m_mysql) ;
	m__conn	   = 0 ;

}

/*  KBMySQL								*/
/*  ~KBMySQL	: Destructor for MySQL database connection class	*/
/*  (returns)	:		:					*/

KBMySQL::~KBMySQL ()
{
	/* Close the connection if there us one ....			*/
	if (m__conn) mysql_close (&m_mysql) ;
}

/*  KBMySQL								*/
/*  doConnect	: Open connection to database				*/
/*  svInfo	: KBServerInfo *: Server information			*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQL::doConnect
	(	KBServerInfo	*svInfo
	)
{
	m_readOnly	= svInfo->readOnly  () ;
	m_socket	= svInfo->socketName() ;
	m_flags		= svInfo->flags     () ;
	m_foundRows	= false	;

	if (svInfo->advanced() != 0)
		if (svInfo->advanced()->isType("mysql"))
		{
			KBMyAdvanced *a = (KBMyAdvanced *)svInfo->advanced() ;
			m_ignoreCharset	= a->m_ignoreCharset	;
			m_foundRows	= a->m_foundRows	;
		}
		else	/* If the type is wrong then show a warning.	*/
			/* This should never happen unless someone	*/
			/* hand edits the XML database file.		*/
			KBError::EError
			(	TR("Driver error"),
				TR("Invalid advanced options, ignoring"),
				__ERRLOCN
			)	;

	/* Check that we are not already connected ...			*/
	/* notConnected routine sets an error message.			*/
	if (m__conn)
	{	m_lError = KBError
			   (	KBError::Error,
				"Already connected to the MySQL server",
				QString::null,
				__ERRLOCN
			   )	;
		return	false ;
	}

	QString	host		= m_host.stripWhiteSpace() ;
	QString	port		= m_port.stripWhiteSpace() ;

#ifndef	_WIN32
	/* The driver supports SSH tunneling. If there is a tunnel	*/
	/* target then attempt to open the tunnel; if this succeeds	*/
	/* then the host becomes the local host and the port is that	*/
	/* returned by the tunneler.					*/
	fprintf
	(	stderr,
		"KBPgSQL::doConnect: sshTarget=[%s]\n",
		(cchar *)m_sshTarget
	)	;

	if (!m_sshTarget.isEmpty())
	{
		int local = openSSHTunnel(3306) ;
		if (local < 0) return false ;

		host	= "127.0.0.1" ;
		port	= QString("%1").arg(local) ;
	}
#endif

	/* Special case; if the database name is empty then establish	*/
	/* a connection to the default database with the "mysql" user.	*/
	/* In practice this is used to connect to get a list of		*/
	/* databases.							*/
	QString	database = m_database	 ;
	QString	user	 = m_user	 ;
	QString	password = m_password	 ;
	if (database.isEmpty())
	{
		if (user.isEmpty()) user = "mysql" ;
	}

	uint	flags	 = m_flags.toInt () ;
	if (m_foundRows) flags |= CLIENT_FOUND_ROWS ;

	if (mysql_real_connect
		(	&m_mysql,
			host,
			user,
			password,
			database,
			port    .toInt  (),
			m_socket.isEmpty() ? (cchar *)0 : (cchar *)m_socket,
			flags
		) == 0	)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Unable to connect to MySQL server",
				mysql_error (&m_mysql),
				__ERRLOCN
			   )	;
		return	false	;
	}

	m__conn	= true	;

	loadVariables() ;

	QString	charSet	= getVariable ("character_set") ;
	if (!charSet.isEmpty() && !m_ignoreCharset)
	{
		QTextCodec *codec = QTextCodec::codecForName(charSet) ;

		fprintf
		(	stderr,
			"KBMySQL::loadVariables: codec mapping [%s]->[%p]\n",
			(cchar *)charSet,
			(void  *)codec
		)	;

		if ((m_dataCodec == 0) && (codec != 0)) m_dataCodec = codec ;
		if ((m_objCodec  == 0) && (codec != 0)) m_objCodec  = codec ;
	}

	return	true	;
}

/*  KBMySQL								*/
/*  loadVariables: Load configuration variables				*/
/*  (returns)	 : void		:					*/

void	KBMySQL::loadVariables ()
{
	m_variables.setAutoDelete (true) ;

	QString	query	("show variables");
	if (!execSQL
		(	query,
			query,
			0,
			0,
			0,
			"Show variables query failed",
			m_lError
		))	return	;

	MYSQL_RES	*myres		;
	MYSQL_ROW	myrow		;
	ulong		*mylengths	;

	if ((myres = getResults()) == 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Show variables query failed",
				QString ("%1\n%2").arg(query).arg(mysql_error(&m_mysql)),
				__ERRLOCN
			   )	;
		return	 ;
	}

	int	nRows	= mysql_num_rows      (myres) ;

	for (int row = 0 ; row < nRows ; row += 1)
	{
		mysql_data_seek (myres, row) ;

		myrow	  = mysql_fetch_row     (myres) ;
		mylengths = mysql_fetch_lengths (myres) ;

		KBValue	name  (myrow[0], mylengths[0], &_kbString) ;
		KBValue	value (myrow[1], mylengths[1], &_kbString) ;

//		fprintf
//		(	stderr,
//			"KBMySQL::loadVariables [%s]->[%s]\n",
//			(cchar *)name .getRawText(),
//			(cchar *)value.getRawText()
//		)	;

		m_variables.insert
		(	name.getRawText(),
			new QString(value.getRawText())
		)	;
	}
}

/*  KBMySQL								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		  : Querying for data			*/
/*  select	: const QString & : Select query			*/
/*  update	: bool		  : Select for update query		*/
/*  (returns)	: KBSQLSelect   * : Select query class or NULL on error	*/

KBSQLSelect
	*KBMySQL::qrySelect
	(	bool		data,
		const QString	&select,
		bool
	)
{
	return	new KBMySQLQrySelect
		(	this,
			data,
			select
		)	;
}


/*  KBMySQL								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		  : Querying for data			*/
/*  update	: const QString & : Update query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryUpdate *   : Update query class or NULL on error	*/

KBSQLUpdate
	*KBMySQL::qryUpdate
	(	bool		data,
		const QString	&update,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting update query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBMySQLQryUpdate
		(	this,
			data,
			update,
			tabName
		)	;
}

/*  KBMySQL								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		  : Querying for data			*/
/*  insert	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryInsert *   : Insert query class or NULL on error	*/

KBSQLInsert
	*KBMySQL::qryInsert
	(	bool		data,
		const QString	&insert,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting insert query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBMySQLQryInsert
		(	this,
			data,
			insert,
			tabName
		)	;
}

/*  KBMySQL								*/
/*  qryDelete	: Open an delete query					*/
/*  data	: bool		  : Querying for data			*/
/*  _delete	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryDelete *   : Delete query class or NULL on error	*/

KBSQLDelete
	*KBMySQL::qryDelete
	(	bool		data,
		const QString	&_delete,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting delete query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBMySQLQryDelete
		(	this,
			data,
			_delete,
			tabName
		)	;
}

/*  KBMySQL								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Querying for data			*/
/*  rawqry	: const QString & : Query text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  select	: KBSQLSelect **  : Return for result queries		*/
/*  (returns)	: bool		  : Success				*/

bool	KBMySQL::command
	(	bool		data,
		const QString	&rawqry,
		uint		nvals,
		KBValue		*values,
		KBSQLSelect	**select
	)
{
	QString	  subqry ;
	MYSQL_RES *myres ;

	if (!execSQL
		(	rawqry,
			subqry,
			nvals,
			values,
			getCodec (data),
			"Query failed",
			m_lError
		))	return	false ;

	if ((myres = getResults()) == 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Query failed",
				QString ("%1\n%2").arg(subqry).arg(mysqlErr()),
				__ERRLOCN
			   )	;
		return	false	;
	}

	int	rows	= mysql_num_rows   (myres) ;
	int	fields	= mysql_num_fields (myres) ;

	fprintf
	(	stderr,
		"KBMySQL::command: rows=%d fields=%d\n",
		rows,
		fields
	)	;

	if (select == 0)
	{
		mysql_free_result (myres) ;
		return	true	;
	}

	if ((rows == 0) || (fields == 0))
	{
		*select	= 0	;
		mysql_free_result (myres) ;
		return	true	;
	}

	*select	= new KBMySQLQrySelect (this, data, rawqry, myres) ;
	return	true	;
}


/*  KBMySQL								*/
/*  execSQL	: Execute SQL statement					*/
/*  rawSql	: const QString &: Statement with placeholders		*/
/*  subSql	: QString &	 : Substituted statement for logging	*/
/*  nvals	: uint		 : Number of substitution values	*/
/*  values	: KBValue *	 : Substitution values			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  emsg	: cchar *	 : Error text on error			*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: bool		 : Success				*/

bool	KBMySQL::execSQL
	(	const QString	&rawSql,
		QString		&subSql,
		uint		nvals,
		const KBValue	*values,
		QTextCodec	*codec,
		cchar		*emsg,
		KBError		&pError
	)
{
	KBDataBuffer	exeSql	;
	bool	rc	= true	;

	if (!subPlaceList (rawSql, nvals, values, exeSql, codec, pError))
		return false	;

	subSql	= subPlaceList (rawSql, nvals, values, pError) ;
	if (subSql == QString::null)
		return false	;



	if (mysql_query (&m_mysql, exeSql.data()) != 0)
	{
		pError	= KBError
			  (	KBError::Error,
				emsg,
				QString ("%1\n%2").arg(subSql).arg(mysql_error (&m_mysql)),
				__ERRLOCN
			  )	;
		rc	= false	;
	}

	printQuery (rawSql, nvals, values, rc) ;
	return	rc ;
}

/*  KBMySQL								*/
/*  getSyntax	: Get text for syntax element				*/
/*  result	: QString &	: Return resule string			*/
/*  syntax	: Syntax	: Element				*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: QString	: Text					*/

bool	KBMySQL::getSyntax
	(	QString			&result,
		KBServer::Syntax	syntax,
		...
	)
{
	va_list	 ap ;
	va_start (ap, syntax) ;

	switch (syntax)
	{
		case Limit :
			{
				int	limit	= va_arg(ap, int) ;
				int	offset	= va_arg(ap, int) ;

				result	= QString(" limit %1,%2 ").arg(offset).arg(limit) ;
			}
			return	true	;

		default	:
			break	;
	}

	m_lError = KBError
		   (	KBError::Error,
			QString(TR("Driver does not support %1")).arg(syntaxToText(syntax)),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}

QString	KBMySQL::getVariable
	(	const QString	&variable
	)
{
	QString	*v = m_variables.find (variable) ;
	return	v == 0 ? QString::null : *v ;
}

/*  KBMySQL								*/
/*  listTypes	: Get list of types with information flags		*/
/*  (returns)	: QString	: List as bar-separated string		*/

QString	KBMySQL::listTypes ()
{
	static	QString	typeList ;

	if (typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0" ;

		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(MySQLTypeMap) ; idx += 1)
		{
			MySQLTypeMap *m = &typeMap[idx] ;

			if ((m->flags & FF_NOCREATE) == 0)
				typeList += QString("|%1,%2").arg(m->mtype).arg(m->flags) ;
		}
	}

	return	typeList ;
}

/*  KBMySQL								*/
/*  tableExists	: See if named table exists				*/
/*  table	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBMySQL::tableExists
	(	const QString	&table,
		bool		&exists
	)
{
	/* Workaround for a Windoze MySQL bug, whereby the server seems	*/
	/* to ignore case except on testing existance. So get a list of	*/
	/* all tables and do a case-insensitive match. Might break a	*/
	/* UNIX database, but does anyone really have to tables whose	*/
	/* names only differ in case??					*/
	KBTableDetailsList tabList ;
	if (!doListTables (tabList, true, KB::IsTable)) return false ;

	for (uint idx = 0 ; idx < tabList.count() ; idx += 1)
		if (tabList[idx].m_name.lower() == table.lower())
		{	exists	= true	;
			return	true	;
		}

	exists	= false	;
	return	true	;
}

/*  KBMySQL								*/
/*  doListTables: List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  alltables	: bool		: List all tables			*/
/*  type	: uint		: Type mask				*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQL::doListTables
	(	KBTableDetailsList	&tabList,
		bool			allTables,
		uint			type
	)
{
	MYSQL_RES *myres = mysql_list_tables (&m_mysql, 0) ;
	MYSQL_ROW myrow  ;

	if (myres == 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Error getting list of tables",
				mysql_error (&m_mysql),
				__ERRLOCN
			   )	;
		return	false	;
	}

	if ((type & KB::IsTable) == 0) return true ;

	for (uint idx = 0 ; idx < mysql_num_rows (myres) ; idx += 1)
	{
		myrow = mysql_fetch_row (myres) ;

		QString	tabName	= myrow[0] ;

		if (!allTables)
			if (tabName.left(8) == "__Rekall")
				continue ;

		tabList.append
		(	KBTableDetails
			(	tabName,
				KB::IsTable,
				QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
		)	)	;
	}

	mysql_free_result (myres) ;
	return	true ;
}

/*  KBMySQL								*/
/*  doListTable: List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  type	: uint		: Type flags				*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQL::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	return	doListTables (tabList, m_showAllTables, type) ;
}

bool	KBMySQL::listDatabases
	(	QStringList	&dbList
	)
{
	MYSQL_RES *myres = mysql_list_dbs (&m_mysql, 0) ;
	if (myres == 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"List databases request failed",
				QString ("%2").arg(mysqlErr()),
				__ERRLOCN
			   )	;
		return	false	;
	}

	int	nRows	= mysql_num_rows (myres) ;

	for (int row = 0 ; row < nRows ; row += 1)
	{
		mysql_data_seek (myres, row) ;
		dbList.append (mysql_fetch_row(myres)[0]) ;
	}

	mysql_free_result (myres) ;
	return	true	;
}

/*  KBMySQL								*/
/*  doListFields: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQL::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	MYSQL_RES 	*myres  ;
	MYSQL_FIELD	*fields	;
	uint		nFields	;
	QString		_query	;

	bool		anyAI	;
	uint		pkCount	;

#if	0
	QString	list	= QString("select * from %1 where 1 is null").arg(tabSpec.m_name) ;

	/* MySQL has a list-fields function, but we use a select so	*/
	/* that we can retrieve field information, specifically which	*/
	/* field, if any, is the primary key.				*/
	if (!execSQL (  list,
			_query,
			0,
			0,
			0,
			"Error getting list of fields in table",
			m_lError)) return false ;

	if ((myres = mysql_store_result (&m_mysql)) == 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Error getting list of fields in table",
				QString ("%1\n%2").arg(list).arg(mysql_error (&m_mysql)),
				__ERRLOCN
			   )	;
		return	false	;
	}
#endif
	if ((myres = mysql_list_fields (&m_mysql, tabSpec.m_name, 0)) == 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Error getting list of fields in table",
				QString ("%1").arg(mysql_error (&m_mysql)),
				__ERRLOCN
			   )	;
		return	false	;
	}

	nFields			= mysql_num_fields   (myres) ;
	fields			= mysql_fetch_fields (myres) ;
	tabSpec.m_prefKey	= -1	  ;
	tabSpec.m_keepsCase	= true  ;
	anyAI		  	= false ;
	pkCount		  	= 0	  ;

	for (uint idx = 0 ; idx < nFields ; idx += 1)
	{
		MySQLTypeMap *m = dIdentToType.find (fields[idx].type) ;
		bool	     pk = (fields[idx].flags & PRI_KEY_FLAG       ) != 0 ;
		bool	     nn = (fields[idx].flags & NOT_NULL_FLAG      ) != 0 ;
		bool	     ai = (fields[idx].flags & AUTO_INCREMENT_FLAG) != 0 ;
		bool	     uk = (fields[idx].flags & UNIQUE_KEY_FLAG    ) != 0 ;
		bool	     ix = (fields[idx].flags & MULTIPLE_KEY_FLAG  ) != 0 ;
		bool	     ts = (fields[idx].flags & TIMESTAMP_FLAG     ) != 0 ;
		bool	     bf = (fields[idx].flags & BINARY_FLAG        ) != 0 ;

#if	0
		fprintf
		(	stderr,
			"%3d: %8s(%3d) t=%3d f=%08x [%d]\n",
			idx,
			fields[idx].name,
			fields[idx].length,
			fields[idx].type,
			fields[idx].flags,
			m == 0 ? -1 : m->itype
		) ;
				 
#endif
		QString		mtype	;
		KB::IType	itype	;

		if (m != 0)
		{
			mtype	= m->mtype	;
			itype	= m->itype	;
		}
		else
		{
			mtype	= QString("<Unknown %1>").arg(fields[idx].type) ;
			itype	= KB::ITUnknown	;
		}

		/* HACK: If the field is a timestamp then clear not-	*/
		/*	 null. The column is marked not-null but we can	*/
		/*	 insert null (and MySQL sets the timestamp).	*/
		if (ts) nn = false ;

		switch (m->ident)
		{
			case FIELD_TYPE_BLOB	:
				/* Blob without a binary flag is really	*/
				/* a text column. Yuck.			*/
				if (!bf)
				{	mtype	= "Text"	;
					itype	= KB::ITString	;
				}
				break	;
		
			case FIELD_TYPE_DECIMAL	:
				fields[idx].length -= 2 ;
				break	;

			case FIELD_TYPE_LONG	:
				/* Long (ie., integers) with the auto-	*/
				/* increment flag is most treated as	*/
				/* the "Primary Key" pseudo-type.	*/
				if (ai)
					mtype	= "Primary Key" ;
				break	;
		}

		uint	flags	= 0 ;

		if (pk) flags |= KBFieldSpec::Primary | KBFieldSpec::Unique  ;
		if (nn) flags |= KBFieldSpec::NotNull ;
		if (uk) flags |= KBFieldSpec::Unique  ;
		if (ix) flags |= KBFieldSpec::Indexed ;
		if (ai) flags |= KBFieldSpec::Serial  | KBFieldSpec::ReadOnly;

		KBFieldSpec *fSpec = new KBFieldSpec
				     (		idx,
						fields[idx].name,
						mtype,
						itype,
						flags,
						fields[idx].length,
						fields[idx].decimals
				     ) ;

		fSpec->m_dbType	   = new KBMySQLType
				     (		m,
						fields[idx].length,
						fields[idx].decimals,
						!nn || ai
				     ) ;

		tabSpec.m_fldList.append (fSpec) ;

		if (pk && ai)
		{	tabSpec.m_prefKey = idx ;
			pkCount	         += 1   ;
		}
			
		if (ai)
			anyAI = true ;
	}

	mysql_free_result (myres) ;

	/* If there is any auto-increment column then we can retrieve	*/
	/* a unique key value from any unique column.			*/
	if (anyAI)
		LITER
		(	KBFieldSpec,
			tabSpec.m_fldList,
			fSpec,

			if ((fSpec->m_flags & KBFieldSpec::Unique) != 0)
				fSpec->m_flags |= KBFieldSpec::InsAvail ;
		)

	/* If there is no primary key column then the preferred key can	*/
	/* be a unique key column.					*/
	if (tabSpec.m_prefKey < 0)
		for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
		{	KBFieldSpec *fSpec = tabSpec.m_fldList.at(idx) ;
			if ((fSpec->m_flags & KBFieldSpec::Unique) != 0)
			{	tabSpec.m_prefKey = idx ;
				break	;
			}
		}

	/* If there is more that one column marked as primary key then	*/
	/* we don't handle this situation.				*/
	if (pkCount > 1)
		tabSpec.m_prefKey = -1 ;

	return	true ;
}

/*  KBMySQL								*/
/*  tblCreateSQL: Generate SQL for a table create statement		*/
/*  fldList	: QList<KBFieldSpec>&: Field specification		*/
/*  table	: const QString &    : Table name			*/
/*  create	: QString &	     : SQL text				*/
/*  best	: bool		     : Use best match			*/
/*  (returns)	: bool		     : Success				*/

bool	KBMySQL::tblCreateSQL
	(	QList<KBFieldSpec>	&fldList,
		const QString		&table,
		QString			&create,
		bool			best
	)
{
	MySQLTypeMap	*mapp	;
	cchar		*sep	= ""	;

	create	= QString ("create table %1\n(").arg(table) ;

	/* Main loop generates SQL create for the columns and column	*/
	/* types, plus options like auto-increment and primary key.	*/
	LITER
	(	KBFieldSpec,
		fldList,
		fSpec,

		mapp	= 0	;

		/* Special cases. If the field is "Primary Key" then we	*/
		/* create an "Int4" column marked as primary key, not	*/
		/* null and auto increment ...				*/
		QString		ftype	= fSpec->m_typeName ;
		KB::IType	itype	= fSpec->m_typeIntl ;

		if (ftype == "Primary Key")
		{
			create += QString ("%1\t%2 int not null primary key auto_increment")
					  .arg(sep	    )
					  .arg(fSpec->m_name) ;
			sep	= ",\n"	;
			continue   ;
		}
		/* ... while a foreign key is also "Int4" not null.	*/
		if (ftype == "Foreign Key")
		{
			create += QString ("%1\t%2 int not null")
					  .arg(sep	    )
					  .arg(fSpec->m_name) ;
			sep	= ",\n"	;
			continue   ;
		}

		/* Map the types used when creating the object and	*/
		/* design dictionary tables.				*/
		if	(ftype == "_Text"   ) ftype = "VarChar" ;
		else if (ftype == "_Integer") ftype = "Integer" ;
		else if (ftype == "_Binary" ) ftype = "Blob"	;


		/* Scan though the mapping table looking for the type	*/
		/* type.						*/
		for (uint typ = 0 ; typ < sizeof(typeMap)/sizeof(MySQLTypeMap) ; typ += 1)
			if (typeMap[typ].mtype == ftype)
				if ((typeMap[typ].flags & FF_NOCREATE) == 0)
				{
					mapp	= &typeMap[typ] ;
					break	;
				}

		/* If there is no mapping but the use-best-match flag	*/
		/* is set, then look for a mapping based on the		*/
		/* internal types.					*/
		if ((mapp == 0) && best)
		{
			if (itype == KB::ITBool)
				itype	= KB::ITFixed	;

			for (uint typ = 0 ; typ < sizeof(typeMap)/sizeof(MySQLTypeMap) ; typ += 1)
				if (typeMap[typ].itype == itype)
					if ((typeMap[typ].flags & FF_NOCREATE) == 0)
					{
						mapp	= &typeMap[typ] ;
						break	;
					}
		}


		if (mapp == 0)
		{	m_lError = KBError
				   (	KBError::Fault,
				  	"Error mapping column type",
				  	QString ("Type %1 for column %2 not known")
						.arg(ftype)
						.arg(fSpec->m_name),
				  	__ERRLOCN
				   )	;
			return	false	;
		}


		create += QString ("%1\t%2 %3").arg(sep          )
					       .arg(fSpec->m_name)
					       .arg(mapp ->mtype ) ;

		if ((mapp->flags & FF_LENGTH) != 0)
		{
			create	+= QString("(%1").arg(fSpec->m_length) ;
			if ((mapp->flags & FF_PREC) != 0)
				create	+= QString(",%1").arg(fSpec->m_prec) ;
			create	+= ")" ;
		}

		if ((fSpec->m_flags & KBFieldSpec::NotNull) != 0) create += " not null" ;
		if ((fSpec->m_flags & KBFieldSpec::Primary) != 0) create += " primary key auto_increment" ;

		sep	= ",\n"	;
	)

	/* Now add any unique columns. Note that we do not handle	*/
	/* multiple-column uniqueness.					*/
	LITER
	(	KBFieldSpec,
		fldList,
		fSpec,

		if ((fSpec->m_flags & KBFieldSpec::Unique ) != 0)
		{
			create	+= sep	;
			create	+= QString("unique (%1)").arg(fSpec->m_name) ;
			sep	= ",\n"	;
		}
		if ((fSpec->m_flags & KBFieldSpec::Indexed) != 0)
		{
			create	+= sep	;
			create	+= QString("index  (%1)").arg(fSpec->m_name) ;
			sep	= ",\n"	;
		}
	)
	
	create	+= "\n)";
	return	true	;
}

/*  KBMySQL								*/
/*  doCreateTable: Create a new table					*/
/*  tabSpec	 : KBTableSpec &: Table specification			*/
/*  assoc	 : bool		: Create associated objects		*/
/*  best	 : bool		: Use best match			*/
/*  (returns)	 : bool		: Success				*/

bool	KBMySQL::doCreateTable
	(	KBTableSpec	&tabSpec,
		bool		,
		bool		best
	)
{
	QString	create	;
	QString	_query	;

	if (!tblCreateSQL
		(	tabSpec.m_fldList,
			tabSpec.m_name,
			create,
			best
		))
		return false ;

	fprintf (stderr, "%s\n", (cchar *)create) ;

	return	execSQL (create, _query, 0, 0, 0, "Error creating table", m_lError) ;
}

/*  KBMySQL								*/
/*  doRenameTable: Rename a  table					*/
/*  oldName	 : cchar *	: Current table name			*/
/*  newName	 : cchar *	: New table name			*/
/*  assoc	 : bool		: Rename associated stuff - none here	*/
/*  (returns)	 : bool		: Success				*/

bool	KBMySQL::doRenameTable
	(	cchar	*oldName,
		cchar	*newName,
		bool
	)
{
	QString	rename	;

	rename	= "alter table " ;
	rename += oldName	 ;
	rename += " rename as "	 ;
	rename += newName	 ;

//	fprintf (stderr, "%s\n", (cchar *)rename) ;

	if (mysql_query (&m_mysql, rename) != 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Error renaming table",
				QString ("%1\n%2").arg(rename).arg(mysql_error (&m_mysql)),
				__ERRLOCN
			   )	;
		return	false	;
	}

	return	true	;
}

/*  KBMySQL								*/
/*  doDropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff - none here	*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQL::doDropTable
	(	cchar	*table,
		bool
	)
{
	QString	drop	;

	drop	= "drop table " ;
	drop   += table		;

//	fprintf (stderr, "%s\n", (cchar *)drop) ;

	if (mysql_query (&m_mysql, drop) != 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Error deleting table",
				QString ("%1\n%2").arg(drop).arg(mysql_error (&m_mysql)),
				__ERRLOCN
			   )	;
		return	false	;
	}

	return	true	;
}

/*  KBMySQL								*/
/*  optionFlags	: Get server option flags				*/
/*  (returns)	: uint		: Flags					*/

uint	KBMySQL::optionFlags ()
{
	/* Standard flags, plus we support SSH tunnelling.		*/
	return	KBServer::optionFlags() | AF_SSHTUNNEL ;
}


/*  ------------------------------------------------------------------- */

/*  KBMySQLQrySelect							*/
/*  KBMySQLQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBMySQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query			*/
/*  (returns)	: KBMySQLQrySelect:					*/

KBMySQLQrySelect::KBMySQLQrySelect
	(	KBMySQL		*server,
		bool		data,
		const QString	&query
	)	
	:
	KBSQLSelect (server, data, query),
	m_server    (server)
{
	m_nRows		= 0 ;
	m_nFields	= 0 ;
	m_fields	= 0 ;
	m_myres		= 0 ;
	m_myrow		= 0 ;
	m_crow	 	= 0 ;
	m_lengths	= 0 ;
}

KBMySQLQrySelect::KBMySQLQrySelect
	(	KBMySQL		*server,
		bool		data,
		const QString	&query,
		MYSQL_RES	*myres
	)
	:
	KBSQLSelect (server, data, query),
	m_server    (server)
{
	m_myres		= myres	;
	m_nRows		= mysql_num_rows      (m_myres) ;
	m_nFields	= mysql_num_fields    (m_myres) ;
	m_fields	= mysql_fetch_fields  (m_myres) ;
	m_myrow		= mysql_fetch_row     (m_myres) ;
	m_lengths	= mysql_fetch_lengths (m_myres) ;
	m_crow		= 0	;

	m_types		= new KBType *[m_nFields] ;

	for (uint idx = 0 ; idx < m_nFields ; idx += 1)
	{
		MySQLTypeMap *ptr = dIdentToType.find (m_fields[idx].type) ;
		bool	     nn	  = (m_fields[idx].flags & NOT_NULL_FLAG      ) != 0 ;
		bool	     ai	  = (m_fields[idx].flags & AUTO_INCREMENT_FLAG) != 0 ;

		m_types[idx] = new KBMySQLType
			       (	ptr,
					m_fields[idx].length,
					m_fields[idx].decimals,
					!nn || ai
			       )	;
	}
}


/*  KBMySQLQrySelect							*/
/*  ~KBMySQLQrySelect							*/
/*		: Destructor for select query object			*/
/*  (returns)	:		:					*/

KBMySQLQrySelect::~KBMySQLQrySelect ()
{
	if (m_myres != 0)
		mysql_free_result (m_myres) ;

//	if (m_types != 0)
//		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
//			m_types[idx]->deref() ;
}

/*  KBMySQLQrySelect							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQLQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (m_myres != 0)
	{	mysql_free_result (m_myres) ;
		m_myres = 0 ;
	}

	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Select query failed",
			m_lError
		))	return	false ;

	if ((m_myres = m_server->getResults()) == 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Select query failed",
				QString ("%1\n%2").arg(m_subQuery).arg(m_server->mysqlErr()),
				__ERRLOCN
			   )	;
		return	false	;
	}


	m_nRows		= mysql_num_rows      (m_myres) ;
	m_nFields	= mysql_num_fields    (m_myres) ;
	m_fields	= mysql_fetch_fields  (m_myres) ;
	m_myrow		= mysql_fetch_row     (m_myres) ;
	m_lengths	= mysql_fetch_lengths (m_myres) ;
	m_crow		= 0	;

	if (m_types == 0)
	{
	m_types		= new KBType *[m_nFields] ;

	for (uint idx = 0 ; idx < m_nFields ; idx += 1)
	{
		MySQLTypeMap *ptr = dIdentToType.find (m_fields[idx].type) ;
		bool	     nn	  = (m_fields[idx].flags & NOT_NULL_FLAG      ) != 0 ;
		bool	     ai	  = (m_fields[idx].flags & AUTO_INCREMENT_FLAG) != 0 ;

		m_types[idx] = new KBMySQLType
			       (	ptr,
					m_fields[idx].length,
					m_fields[idx].decimals,
					!nn || ai
			       )	;
	}
	}

	return	true	;
}

/*  KBMySQLQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBMySQLQrySelect::getField
	(	uint		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if ((int)qrow >= m_nRows  ) return KBValue() ;
	if (     qcol >= m_nFields) return KBValue() ;

	/* If we are already at the requested row then life is easy,	*/
	/* otherwise we have to get there first ...			*/
	if (qrow != m_crow)
	{
		if (qrow != m_crow + 1)
			mysql_data_seek (m_myres, qrow) ;

		m_myrow	  = mysql_fetch_row     (m_myres) ;
		m_lengths = mysql_fetch_lengths (m_myres) ;
		m_crow	  = qrow	;
	}

	/* Check that we have a set of results. This will not be the	*/
	/* case if the query returned no rows.				*/
	if (m_myrow == 0) KBValue() ;

	/* Next check for null values. In this case the field pointer	*/
	/* in the row will be empty.					*/
	if (m_myrow[qcol] == 0)
		return	KBValue (m_types[qcol]) ;

	/* Binary data gets special treatment as it is does not have a	*/
	/* zero terminator .....					*/
	if (m_types[qcol]->getIType() == KB::ITBinary)
	{
		int	length	= m_lengths[qcol] ;
		char	*data	= (char *)malloc(length) ;
		memcpy	(data, m_myrow[qcol], length) ;
		return	KBValue (QByteArray().assign(data, length), m_types[qcol]) ;
	}

	return	KBValue
		(	m_myrow  [qcol],
			m_lengths[qcol],
			m_types  [qcol],
			m_codec
		)	;
}

/*  KBMySQLQrySelect							*/
/*  getFieldname: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: Name					*/

QString	KBMySQLQrySelect::getFieldName
	(	uint	qcol
	)
{
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if (qcol >= m_nFields) return QString() ;

	return	m_fields[qcol].name ;
}


/*  KBMySQLQryUpdate							*/
/*  KBMySQLQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBMySQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Update query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KBMySQLQryUpdate:					*/

KBMySQLQryUpdate::KBMySQLQryUpdate
	(	KBMySQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBMySQLQryUpdate							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQLQryUpdate::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Update query failed",
			m_lError
		))	return	false ;


	m_nRows	= m_server->getNumRows () ;
	return	true ;
}

/*  KBMySQLQryUpdate							*/
/*  ~KBMySQLQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBMySQLQryUpdate::~KBMySQLQryUpdate ()
{
}


/*  KBMySQLQryInsert							*/
/*  KBMySQLQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBMySQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  tabName	: const QString & : Table being updated			*/
/*  query	: const QString & : Insert query			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBMySQLQruInsert:					*/

KBMySQLQryInsert::KBMySQLQryInsert
	(	KBMySQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBMySQLQryInsert							*/
/*  ~KBMySQLQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

KBMySQLQryInsert::~KBMySQLQryInsert ()
{
}

/*  KBMySQLQryInsert							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQLQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Insert query failed",
			m_lError
		))	return	false ;

	static	 KBMySQLType _kbMySQLKey (&typeMap[3], 0, 0, false) ;
	m_newKey = KBValue ((int)m_server->getInsertID(), &_kbMySQLKey) ;
	m_nRows	 = m_server->getNumRows () ;
	return	 true ;
}

/*  KBMySQLQryInsert							*/
/*  getNewKey	: Get new insert key					*/
/*  primary	: const QString & : Key column name			*/
/*  _newKey	: KBValue &	  : New key				*/
/*  prior	: bool		  : Pre-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	KBMySQLQryInsert::getNewKey
	(	const QString	&keycol,
		KBValue		&_newKey,
		bool		prior
	)
{
	/* The first time we execute this insert we need to get the	*/
	/* auto-increment column name, if any. We do this here rather	*/
	/* than on creation since it is more difficult to report errors	*/
	/* at query creation time.					*/
	if (m_autocol.isNull())
	{
		KBTableSpec tabSpec (m_tabName) ;
		if (!m_server->listFields (tabSpec))
		{	m_lError = m_server->lastError() ;
			return	 false ;
		}

		/* Set the column name to empty. This allows us to tell	*/
		/* later whether there really is no auto-increment	*/
		/* column, or whether there was an error above.		*/
		m_autocol = "" ;

		LITER
		(	KBFieldSpec,
			tabSpec.m_fldList,
			fSpec,

			if ((fSpec->m_flags & KBFieldSpec::Serial) != 0)
			{	m_autocol = fSpec->m_name ;
				break	;
			}
		)
	}

	if (prior)
	{
		_newKey = KBValue() ;
		return	true	;
	}

	if (keycol == m_autocol)
	{
		_newKey	= m_newKey ;
		return	true	   ;
	}

	m_lError = KBError
		   (	KBError::Error,
			"Asking for insert key",
			QString ("%1, %2:%3").arg(m_tabName).arg(keycol).arg(m_autocol),
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBMySQLQryDelete							*/
/*  KBMySQLQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBMySQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBMySQLQryDelete:					*/

KBMySQLQryDelete::KBMySQLQryDelete
	(	KBMySQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBMySQLQryDelete							*/
/*  ~KBMySQLQryDelete							*/
/*		: Destructor for delete query object			*/
/*  (returns)	:		:					*/

KBMySQLQryDelete::~KBMySQLQryDelete ()
{
}

/*  KBMySQLQryDelete							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBMySQLQryDelete::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Delete query failed",
			m_lError
		))	return	false ;

	m_nRows	= m_server->getNumRows () ;
	return	true	;
}


#ifndef _WIN32
KBFACTORY
(	KBMySQLFactory,
	"driver_mysql"
)

KBFACTORYIMPL
(	KBMySQLFactory,
	driver_mysql,
	"Rekall MySQL driver",
	"Plugin",
	"0",
	"7",
	"0"
)
#else
class	KBMySQLFactory : public KBFactory
{
public:
	inline	KBMySQLFactory() : KBFactory ()
	{
	}
	virtual	QObject	*create(QObject * = 0, const char *	= 0, const char * = 0, 
		const QStringList &	= QStringList());
	virtual const char* ident();
};

extern	"C"	__declspec(dllexport) void *init_libkbase_driver_mysql()			
{							
	return	new KBMySQLFactory;				
}							
#endif

QObject	*KBMySQLFactory::create
	(	QObject		  *parent,
		cchar		  *object,
		cchar		  *,
		const QStringList &
	)
{
	if (dIdentToType.count() == 0)
		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(MySQLTypeMap) ; idx += 1)
			if (typeMap[idx].ident != FIELD_TYPE_UNSET)
			{
				MySQLTypeMap *m = &typeMap[idx] ;
				dIdentToType.insert (m->ident, m) ;
			}

	if ((parent != 0) && !parent->inherits ("QWidget"))
	{
		fprintf	(stderr, "KBMySQLFactory: parent does not inherit QWidget\n") ;
		return	0  ;
	}

	if (strcmp (object, "driver"  ) == 0) return new KBMySQL () ;

	if (strcmp (object, "advanced") == 0) return new KBMyAdvanced () ;

	return	0 ;
}

cchar	*KBMySQLFactory::ident ()
{
	return	__KB_BUILD_IDENT	;
}
