% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/auxiliary.R
\name{auxiliary}
\alias{auxiliary}
\alias{lavaan.auxiliary}
\alias{cfa.auxiliary}
\alias{sem.auxiliary}
\alias{growth.auxiliary}
\title{Implement Saturated Correlates with FIML}
\usage{
auxiliary(model, data, aux, fun, ...)

lavaan.auxiliary(model, data, aux, ...)

cfa.auxiliary(model, data, aux, ...)

sem.auxiliary(model, data, aux, ...)

growth.auxiliary(model, data, aux, ...)
}
\arguments{
\item{model}{The analysis model can be specified with 1 of 2 objects:
\enumerate{
  \item  lavaan \code{\link[lavaan]{model.syntax}} specifying a hypothesized
         model \emph{without} mention of auxiliary variables in \code{aux}
  \item  a parameter table, as returned by \code{\link[lavaan]{parTable}},
         specifying the target model \emph{without} auxiliary variables.
         This option requires these columns (and silently ignores all others):
         \code{c("lhs","op","rhs","user","group","free","label","plabel","start")}
}}

\item{data}{\code{data.frame} that includes auxiliary variables as well as
any observed variables in the \code{model}}

\item{aux}{\code{character}. Names of auxiliary variables to add to \code{model}}

\item{fun}{\code{character}. Name of a specific lavaan function used to fit
\code{model} to \code{data} (i.e., \code{"lavaan"}, \code{"cfa"},
\code{"sem"}, or \code{"growth"}). Only required for \code{auxiliary}.}

\item{...}{additional arguments to pass to \code{\link[lavaan]{lavaan}}.}
}
\value{
a fitted \code{\linkS4class{lavaan}} object.  Additional
  information is stored as a \code{list} in the \code{@external} slot:
  \itemize{
    \item \code{baseline.model}. a fitted \code{\linkS4class{lavaan}}
          object. Results of fitting an appropriate independence model for
          the calculation of incremental fit indices (e.g., CFI, TLI) in
          which the auxiliary variables remain saturated, so only the target
          variables are constrained to be orthogonal. See Examples for how
          to send this baseline model to \code{\link[lavaan]{fitMeasures}}.
    \item \code{aux}. The character vector of auxiliary variable names.
    \item \code{baseline.syntax}. A character vector generated within the
          \code{auxiliary} function, specifying the \code{baseline.model}
          syntax.
  }
}
\description{
Automatically add auxiliary variables to a lavaan model when using full
information maximum likelihood (FIML) to handle missing data
}
\details{
These functions are wrappers around the corresponding lavaan functions.
You can use them the same way you use \code{\link[lavaan]{lavaan}}, but you
\emph{must} pass your full \code{data.frame} to the \code{data} argument.
Because the saturated-correlates approaches (Enders, 2008) treates exogenous
variables as random, \code{fixed.x} must be set to \code{FALSE}. Because FIML
requires continuous data (although nonnormality corrections can still be
requested), no variables in the model nor auxiliary variables specified in
\code{aux} can be declared as \code{ordered}.
}
\examples{
dat1 <- lavaan::HolzingerSwineford1939
set.seed(12345)
dat1$z <- rnorm(nrow(dat1))
dat1$x5 <- ifelse(dat1$z < quantile(dat1$z, .3), NA, dat1$x5)
dat1$x9 <- ifelse(dat1$z > quantile(dat1$z, .8), NA, dat1$x9)

targetModel <- "
  visual  =~ x1 + x2 + x3
  textual =~ x4 + x5 + x6
  speed   =~ x7 + x8 + x9
"

## works just like cfa(), but with an extra "aux" argument
fitaux1 <- cfa.auxiliary(targetModel, data = dat1, aux = "z",
                         missing = "fiml", estimator = "mlr")

## with multiple auxiliary variables and multiple groups
fitaux2 <- cfa.auxiliary(targetModel, data = dat1, aux = c("z","ageyr","grade"),
                         group = "school", group.equal = "loadings")

## calculate correct incremental fit indices (e.g., CFI, TLI)
fitMeasures(fitaux2, fit.measures = c("cfi","tli"))
## NOTE: lavaan will use the internally stored baseline model, which
##       is the independence model plus saturated auxiliary parameters
lavInspect(fitaux2@external$baseline.model, "free")

}
\references{
Enders, C. K. (2008). A note on the use of missing auxiliary
  variables in full information maximum likelihood-based structural equation
  models. \emph{Structural Equation Modeling, 15}(3), 434--448.
  \doi{10.1080/10705510802154307}
}
\author{
Terrence D. Jorgensen (University of Amsterdam; \email{TJorgensen314@gmail.com})
}
