% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ica.R
\name{step_ica}
\alias{step_ica}
\alias{tidy.step_ica}
\title{ICA Signal Extraction}
\usage{
step_ica(recipe, ..., role = "predictor", trained = FALSE,
  num_comp = 5, options = list(), res = NULL, num = NULL,
  prefix = "IC", skip = FALSE, id = rand_id("ica"))

\method{tidy}{step_ica}(x, ...)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose which
variables will be used to compute the components. See
\code{\link[=selections]{selections()}} for more details. For the \code{tidy}
method, these are not currently used.}

\item{role}{For model terms created by this step, what analysis
role should they be assigned?. By default, the function assumes
that the new independent component columns created by the
original variables will be used as predictors in a model.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{num_comp}{The number of ICA components to retain as new
predictors. If \code{num_comp} is greater than the number of columns
or the number of possible components, a smaller value will be
used.}

\item{options}{A list of options to
\code{\link[fastICA:fastICA]{fastICA::fastICA()}}. No defaults are set here.
\strong{Note} that the arguments \code{X} and \code{n.comp} should
not be passed here.}

\item{res}{The \code{\link[fastICA:fastICA]{fastICA::fastICA()}} object is stored
here once this preprocessing step has be trained by
\code{\link[=prep.recipe]{prep.recipe()}}.}

\item{num}{The number of components to retain (this will be
deprecated in factor of \code{num_comp} in version 0.1.5). \code{num_comp}
will override this option.}

\item{prefix}{A character string that will be the prefix to the
resulting new variables. See notes below.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[=bake.recipe]{bake.recipe()}}? While all operations are baked
when \code{\link[=prep.recipe]{prep.recipe()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_ica} object.}
}
\value{
An updated version of \code{recipe} with the new step
added to the sequence of existing steps (if any). For the
\code{tidy} method, a tibble with columns \code{terms} (the
selectors or variables selected), \code{value} (the loading),
and \code{component}.
}
\description{
\code{step_ica} creates a \emph{specification} of a recipe step
that will convert numeric data into one or more independent
components.
}
\details{
Independent component analysis (ICA) is a
transformation of a group of variables that produces a new set
of artificial features or components. ICA assumes that the
variables are mixtures of a set of distinct, non-Gaussian
signals and attempts to transform the data to isolate these
signals. Like PCA, the components are statistically independent
from one another. This means that they can be used to combat
large inter-variables correlations in a data set. Also like PCA,
it is advisable to center and scale the variables prior to
running ICA.

This package produces components using the "FastICA"
methodology (see reference below). This step requires the
\pkg{dimRed} and \pkg{fastICA} packages. If not installed, the
step will stop with a note about installing these packages.

The argument \code{num_comp} controls the number of components that
will be retained (the original variables that are used to derive
the components are removed from the data). The new components
will have names that begin with \code{prefix} and a sequence of
numbers. The variable names are padded with zeros. For example,
if \code{num_comp < 10}, their names will be \code{IC1} - \code{IC9}.
If \code{num_comp = 101}, the names would be \code{IC001} -
\code{IC101}.
}
\examples{
# from fastICA::fastICA
set.seed(131)
S <- matrix(runif(400), 200, 2)
A <- matrix(c(1, 1, -1, 3), 2, 2, byrow = TRUE)
X <- as.data.frame(S \%*\% A)

tr <- X[1:100, ]
te <- X[101:200, ]

rec <- recipe( ~ ., data = tr)

ica_trans <- step_center(rec,  V1, V2)
ica_trans <- step_scale(ica_trans, V1, V2)
ica_trans <- step_ica(ica_trans, V1, V2, num_comp = 2)

if (require(dimRed) & require(fastICA)) {
  ica_estimates <- prep(ica_trans, training = tr)
  ica_data <- bake(ica_estimates, te)

  plot(te$V1, te$V2)
  plot(ica_data$IC1, ica_data$IC2)

  tidy(ica_trans, number = 3)
  tidy(ica_estimates, number = 3)
}
}
\references{
Hyvarinen, A., and Oja, E. (2000). Independent
component analysis: algorithms and applications. \emph{Neural
Networks}, 13(4-5), 411-430.
}
\seealso{
\code{\link[=step_pca]{step_pca()}} \code{\link[=step_kpca]{step_kpca()}}
\code{\link[=step_isomap]{step_isomap()}} \code{\link[=recipe]{recipe()}} \code{\link[=prep.recipe]{prep.recipe()}}
\code{\link[=bake.recipe]{bake.recipe()}}
}
\concept{ica}
\concept{preprocessing}
\concept{projection_methods}
\keyword{datagen}
