% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rstar.R
\name{rstar}
\alias{rstar}
\title{Calculate R* convergence diagnostic}
\usage{
rstar(
  x,
  split = TRUE,
  uncertainty = FALSE,
  method = "rf",
  hyperparameters = NULL,
  training_proportion = 0.7,
  nsimulations = 1000,
  ...
)
}
\arguments{
\item{x}{(draws) A \code{\link{draws_df}} object or one coercible to a \code{draws_df} object.}

\item{split}{(logical) Should the estimate be computed on split chains? The
default is \code{TRUE}.}

\item{uncertainty}{(logical). Indicates whether to provide a vector of R*
values representing uncertainty in the calculated value (if \code{TRUE}) or a
single value (if \code{FALSE}). The default is \code{TRUE.}}

\item{method}{(string) The machine learning classifier to use (must be
available in the \pkg{caret} package). The default is \code{"rf"}, which calls
the random forest classifier.}

\item{hyperparameters}{(named list) Hyperparameter settings passed to the classifier.
The default for the random forest classifier (\code{method = "rf"}) is
\code{list(mtry = floor(sqt(nvariables(x))))}.
The default for the gradient-based model (\code{method = "gbm"}) is
\code{list(interaction.depth = 3, n.trees = 50, shrinkage = 0.1, n.minobsinnode = 10)}.}

\item{training_proportion}{(positive real) The proportion (in \verb{(0,1)}) of
iterations in used to train the classifier. The default is \code{0.7}.}

\item{nsimulations}{(positive integer) The number of R* values in the
returned vector if \code{uncertainty} is \code{TRUE}. The default is \code{1000.}}

\item{...}{Other arguments passed to \code{caret::train()}.}
}
\value{
A numeric vector of length 1 (by default) or length \code{nsimulations}
(if \code{uncertainty = TRUE}).
}
\description{
The \code{rstar()} function generates a measure of convergence for MCMC draws
based on whether it is possible to determine the Markov chain that generated
a draw with probability greater than chance. To do so, it fits a machine
learning classifier to a training set of MCMC draws and evaluates its
predictive accuracy on a testing set: giving the ratio of accuracy to
predicting a chain uniformly at random.
}
\details{
The \code{rstar()} function provides a measure of MCMC convergence based
on whether it is possible to determine the chain that generated a
particular draw with a probability greater than chance. To do so, it fits a
machine learning classifier to a subset of the original MCMC draws (the
training set) and evaluates its predictive accuracy on the remaining draws
(the testing set). If predictive accuracy exceeds chance (i.e. predicting
the chain that generated a draw uniformly at random), the diagnostic
measure R* will be above 1, indicating that convergence has yet to occur.
This statistic is recently developed, and it is currently unclear what is a
reasonable threshold for diagnosing convergence.

The statistic, R*, is stochastic, meaning that each time the test is run,
unless the random seed is fixed, it will generally produce a different
result. To minimize the implications of this stochasticity, it is
recommended to repeatedly run this function to calculate a distribution of
R*; alternatively, an approximation to this distribution can be obtained by
setting \code{uncertainty = TRUE}, although this approximation of uncertainty
will generally have a lower mean.

By default, a random forest classifier is used (\code{method = "rf"}), which tends
to perform best for target distributions of around 4 dimensions and above.
For lower dimensional targets, gradient boosted models (called via
\code{method = "gbm"}) tend to have a higher classification accuracy. On a given
MCMC sample, it is recommended to try both of these classifiers.
}
\examples{
\donttest{
if (require("caret", quietly = TRUE) && require("randomForest", quietly = TRUE)) {
  x <- example_draws("eight_schools")
  print(rstar(x))
  print(rstar(x, split = FALSE))
  print(rstar(x, method = "gbm"))
  # can pass additional arguments to methods
  print(rstar(x, method = "gbm", verbose = FALSE))

  # with uncertainty, returns a vector of R* values
  hist(rstar(x, uncertainty = TRUE))
  hist(rstar(x, uncertainty = TRUE, nsimulations = 100))

  # can use other classification methods in caret library
  print(rstar(x, method = "knn"))
}
}
}
\references{
Ben Lambert, Aki Vehtari (2020) R*: A robust MCMC convergence
diagnostic with uncertainty using gradient-boosted machines.
\emph{arXiv preprint} \code{arXiv:2003.07900}.
}
\seealso{
Other diagnostics: 
\code{\link{ess_basic}()},
\code{\link{ess_bulk}()},
\code{\link{ess_quantile}()},
\code{\link{ess_sd}()},
\code{\link{ess_tail}()},
\code{\link{mcse_mean}()},
\code{\link{mcse_quantile}()},
\code{\link{mcse_sd}()},
\code{\link{rhat_basic}()},
\code{\link{rhat}()}
}
\concept{diagnostics}
