% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/incidence.R, R/print.R
\name{incidence}
\alias{incidence}
\alias{incidence.default}
\alias{incidence.Date}
\alias{incidence.character}
\alias{incidence.integer}
\alias{incidence.numeric}
\alias{incidence.POSIXt}
\alias{print.incidence}
\title{Compute incidence of events from a vector of dates.}
\usage{
incidence(dates, interval = 1L, ...)

\method{incidence}{default}(dates, interval = 1L, ...)

\method{incidence}{Date}(
  dates,
  interval = 1L,
  standard = TRUE,
  groups = NULL,
  na_as_group = TRUE,
  first_date = NULL,
  last_date = NULL,
  ...
)

\method{incidence}{character}(
  dates,
  interval = 1L,
  standard = TRUE,
  groups = NULL,
  na_as_group = TRUE,
  first_date = NULL,
  last_date = NULL,
  ...
)

\method{incidence}{integer}(
  dates,
  interval = 1L,
  groups = NULL,
  na_as_group = TRUE,
  first_date = NULL,
  last_date = NULL,
  ...
)

\method{incidence}{numeric}(
  dates,
  interval = 1L,
  groups = NULL,
  na_as_group = TRUE,
  first_date = NULL,
  last_date = NULL,
  ...
)

\method{incidence}{POSIXt}(
  dates,
  interval = 1L,
  standard = TRUE,
  groups = NULL,
  na_as_group = TRUE,
  first_date = NULL,
  last_date = NULL,
  ...
)

\method{print}{incidence}(x, ...)
}
\arguments{
\item{dates}{A vector of dates, which can be provided as objects of the
class: integer, numeric, Date, POSIXct, POSIXlt, and character. (See Note
about \code{numeric} and \code{character} formats)}

\item{interval}{An integer or character indicating the (fixed) size of the
time interval used for computing the incidence; defaults to 1 day. This can
also be a text string that corresponds to a valid date interval: day, week,
month, quarter, or year. (See Note).}

\item{...}{Additional arguments passed to other methods (none are used).}

\item{standard}{(Only applicable to Date objects) When \code{TRUE} (default) and the
\code{interval} one of "week", "month", "quarter", or "year", then this will
cause the bins for the counts to start at the beginning of the interval
(See Note).}

\item{groups}{An optional factor defining groups of observations for which
incidence should be computed separately.}

\item{na_as_group}{A logical value indicating if missing group (NA) should be
treated as a separate group.}

\item{first_date, last_date}{optional first/last dates to be used in the
epicurve. When these are \code{NULL} (default), the dates from the first/last
dates are taken from the observations. If these dates are provided, the
observations will be trimmed to the range of [first_date, last_date].}

\item{x}{An 'incidence' object.}
}
\value{
An list with the class \code{incidence}, which contains the
following items:
\itemize{
\item \strong{dates}: The dates marking the left side of the bins used for counting
events. When \code{standard = TRUE} and the interval represents weeks, months,
quarters, or years, the first date will represent the first standard date
(See Interval specification, below).
\item \strong{counts}: A matrix of incidence counts, which one column per group (and
a single column if no groups were used).
\item \strong{timespan}: The length of the period for which incidence is computed, in
days.
\item \strong{interval}: The bin size. If it's an integer, it represents the number
of days between each bin. It can also be a character, e.g. "2 weeks" or
"6 months".
\item \strong{n}: The total number of cases.
\item \strong{weeks}: Dates in week format (YYYY-Www), where YYYY corresponds to the
year of the given week and ww represents the numeric week of the year.
This will be a produced from the function \code{\link[aweek:date2week]{aweek::date2week()}}. Note that
these will have a special \code{"week_start"} attribute indicating which day of
the ISO week the week starts on (see Weeks, below).
\item \strong{isoweeks}: ISO 8601 week format YYYY-Www, which is returned only when
ISO week-based weekly incidence is computed.
}
}
\description{
This function computes incidence based on dates of events provided in
various formats. A fixed interval, provided as numbers of days, is used to
define time intervals. Counts within an interval always include the first
date, after which they are labeled, and exclude the second. For instance,
intervals labeled as 0, 3, 6, ... mean that the first bin includes days 0, 1
and 2, the second interval includes 3, 4 and 5 etc.
}
\details{
For details about the \verb{incidence class}, see the dedicated
vignette:\cr \code{vignette("incidence_class", package = "incidence")}
}
\note{
\subsection{Input data (\code{dates})}{
\itemize{
\item \strong{Decimal (numeric) dates}: will be truncated with a warning
\item \strong{Character dates} should be in the unambiguous \code{yyyy-mm-dd} (ISO 8601)
format. Any other format will trigger an error.
}
}

\subsection{Interval specification (\code{interval})}{
If \code{interval} is a valid character (e.g. "week" or "1 month"), then
the bin will start at the beginning of the interval just before the first
observation by default. For example, if the first case was recorded on
Wednesday, 2018-05-09:
\itemize{
\item "week"    : first day of the week (i.e. Monday, 2018-05-07) (defaults to ISO weeks, see "Week intervals", below)
\item "month"   : first day of the month (i.e. 2018-05-01)
\item "quarter" : first day of the quarter (i.e. 2018-04-01)
\item "year"    : first day of the calendar year (i.e. 2018-01-01)
}

These default intervals can be overridden with \code{standard = FALSE}, which
sets the interval to begin at the first observed case.
}

\subsection{Week intervals}{

As of \emph{incidence} version 1.7.0, it is possible to construct standardized
incidence objects standardized to any day of the week thanks to the
\code{\link[aweek:date2week]{aweek::date2week()}} function from the \pkg{aweek} package. The default
state is to use ISO 8601 definition of weeks, which start on Monday. You can
specify the day of the week an incidence object should be standardised to by
using the pattern "{n} {W} weeks" where "{W}" represents the weekday in an
English or current locale and "{n}" represents the duration, but this can be
ommitted.  Below are examples of specifying weeks starting on different days
assuming we had data that started on 2016-09-05, which is ISO week 36 of
2016:
\itemize{
\item interval = "2 monday weeks" (Monday 2016-09-05)
\item interval = "1 tue week" (Tuesday 2016-08-30)
\item interval = "1 Wed week" (Wednesday 2016-08-31)
\item interval = "1 Thursday week" (Thursday 2016-09-01)
\item interval = "1 F week" (Friday 2016-09-02)
\item interval = "1 Saturday week" (Saturday 2016-09-03)
\item interval = "Sunday week" (Sunday 2016-09-04)
}

It's also possible to use something like "3 weeks: Saturday"; In addition,
there are keywords reserved for specific days of the week:
\itemize{
\item interval = "week", standard = TRUE (Default, Monday)
\item interval = "ISOweek"  (Monday)
\item interval = "EPIweek"  (Sunday)
\item interval = "MMWRweek" (Sunday)
}

The "EPIweek" specification is not strictly reserved for CDC epiweeks, but
can be prefixed (or posfixed) by a day of the week: "1 epiweek: Saturday".

}

\subsection{The \code{first_date} argument}{
Previous versions of \emph{incidence} had the \code{first_date} argument override
\code{standard = TRUE}. It has been changed as of \emph{incidence} version 1.6.0 to
be more consistent with the behavior when \code{first_date = NULL}. This, however
may be a change in behaviour, so a warning is now issued once and only once
if \code{first_date} is specified, but \code{standard} is not. To never see this
warning, use \code{options(incidence.warn.first_date = FALSE)}.
}

The intervals for "month", "quarter", and "year" will necessarily vary in the
number of days they encompass and warnings will be generated when the first
date falls outside of a calendar date that is easily represented across the
interval.
}
\examples{
## toy example
incidence(c(1, 5, 8, 3, 7, 2, 4, 6, 9, 2))
incidence(c(1, 5, 8, 3, 7, 2, 4, 6, 9, 2), 2)

## example using simulated dataset
if(require(outbreaks)) { withAutoprint({
  onset <- outbreaks::ebola_sim$linelist$date_of_onset

  ## daily incidence
  inc <- incidence(onset)
  inc
  plot(inc)

  ## weekly incidence
  inc.week <- incidence(onset, interval = 7, standard = FALSE)
  inc.week
  plot(inc.week)
  plot(inc.week, border = "white") # with visible border
  
  # Starting on Monday
  inc.isoweek <- incidence(onset, interval = "isoweek")
  inc.isoweek
  
  # Starting on Sunday
  inc.epiweek <- incidence(onset, interval = "epiweek")
  inc.epiweek

  # Starting on Saturday
  inc.epiweek <- incidence(onset, interval = "saturday epiweek")
  inc.epiweek

  ## use group information
  sex <- outbreaks::ebola_sim$linelist$gender
  inc.week.gender <- incidence(onset, interval = 7,
                               groups = sex, standard = FALSE)
  inc.week.gender
  head(inc.week.gender$counts)
  plot(inc.week.gender, border = "grey90")
  inc.satweek.gender <- incidence(onset, interval = "2 epiweeks: saturday",
                                  groups = sex)
  inc.satweek.gender
  plot(inc.satweek.gender, border = "grey90")

})}

# Use of first_date
d <- Sys.Date() + sample(-3:10, 10, replace = TRUE)

# `standard` specified, no warning
di <- incidence(d, interval = "week", first_date = Sys.Date() - 10, standard = TRUE)

# warning issued if `standard` not specified
di <- incidence(d, interval = "week", first_date = Sys.Date() - 10)

# second instance: no warning issued
di <- incidence(d, interval = "week", first_date = Sys.Date() - 10)


}
\seealso{
The main other functions of the package include:
\itemize{
\item \code{\link[incidence:plot.incidence]{incidence::plot.incidence()}}: Plot epicurves from an incidence object.
\item \code{\link[incidence:fit]{incidence::fit()}}: Fit log-linear model to computed incidence.
\item \code{\link[incidence:fit_optim_split]{incidence::fit_optim_split()}}: Find the optimal peak of the epidemic
and fits log-linear models on either side of the peak.
\item \code{\link[incidence:subset]{incidence::subset()}}: Handling of \code{incidence}
objects.
\item \code{\link[incidence:pool]{incidence::pool()}}: Sum incidence over groups.
\item \code{\link[incidence:as.data.frame.incidence]{incidence::as.data.frame.incidence()}}: Convert an \code{incidence} object to a
\code{data.frame}.
}

The following vignettes are also available:
\itemize{
\item \code{overview}: Provides an overview of the package's features.
\item \code{customize_plot}: Provides some tips on finer plot customization.
\item \code{incidence_class}: Details the content of the \code{incidence}
class.
}
}
\author{
Thibaut Jombart, Rich Fitzjohn, Zhian Kamvar
}
