% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/recode.R
\name{recode}
\alias{recode}
\alias{recode_factor}
\title{Recode values}
\usage{
recode(.x, ..., .default = NULL, .missing = NULL)

recode_factor(.x, ..., .default = NULL, .missing = NULL,
  .ordered = FALSE)
}
\arguments{
\item{.x}{A vector to modify}

\item{...}{Replacements. For character and factor \code{.x}, these should be named
and replacement is based only on their name. For numeric \code{.x}, these can be
named or not. If not named, the replacement is done based on position i.e.
\code{.x} represents positions to look for in replacements. See examples.

When named, the argument names should be the current values to be replaced, and the
argument values should be the new (replacement) values.

All replacements must be the same type, and must have either
length one or the same length as \code{.x}.

These dots support \link[rlang:tidy-dots]{tidy dots} features.}

\item{.default}{If supplied, all values not otherwise matched will
be given this value. If not supplied and if the replacements are
the same type as the original values in \code{.x}, unmatched
values are not changed. If not supplied and if the replacements
are not compatible, unmatched values are replaced with \code{NA}.

\code{.default} must be either length 1 or the same length as
\code{.x}.}

\item{.missing}{If supplied, any missing values in \code{.x} will be
replaced by this value. Must be either length 1 or the same length as
\code{.x}.}

\item{.ordered}{If \code{TRUE}, \code{recode_factor()} creates an
ordered factor.}
}
\value{
A vector the same length as \code{.x}, and the same type as
the first of \code{...}, \code{.default}, or \code{.missing}.
\code{recode_factor()} returns a factor whose levels are in the same order as
in \code{...}. The levels in \code{.default} and \code{.missing} come last.
}
\description{
This is a vectorised version of \code{\link[=switch]{switch()}}: you can replace
numeric values based on their position or their name, and character or factor
values only by their name. This is an S3 generic: dplyr provides methods for
numeric, character, and factors. For logical vectors, use \code{\link[=if_else]{if_else()}}. For
more complicated criteria, use \code{\link[=case_when]{case_when()}}.
}
\details{
You can use \code{recode()} directly with factors; it will preserve the existing
order of levels while changing the values. Alternatively, you can
use \code{recode_factor()}, which will change the order of levels to match
the order of replacements. See the \href{http://forcats.tidyverse.org/}{forcats}
package for more tools for working with factors and their levels.
}
\examples{
# For character values, recode values with named arguments only. Unmatched
# values are unchanged.
char_vec <- sample(c("a", "b", "c"), 10, replace = TRUE)
recode(char_vec, a = "Apple")
recode(char_vec, a = "Apple", b = "Banana")

# Use .default as replacement for unmatched values
recode(char_vec, a = "Apple", b = "Banana", .default = NA_character_)

# Use a named character vector for unquote splicing with !!!
level_key <- c(a = "apple", b = "banana", c = "carrot")
recode(char_vec, !!!level_key)

# For numeric values, named arguments can also be used
num_vec <- c(1:4, NA)
recode(num_vec, `2` = 20L, `4` = 40L)

# Or if you don't name the arguments, recode() matches by position.
# (Only works for numeric vector)
recode(num_vec, "a", "b", "c", "d")
# .x (position given) looks in (...), then grabs (... value at position)
# so if nothing at position (here 5), it uses .default or NA.
recode(c(1,5,3), "a", "b", "c", "d", .default = "nothing")

# Note that if the replacements are not compatible with .x,
# unmatched values are replaced by NA and a warning is issued.
recode(num_vec, `2` = "b", `4` = "d")
# use .default to change the replacement value
recode(num_vec, "a", "b", "c", .default = "other")
# use .missing to replace missing values in .x
recode(num_vec, "a", "b", "c", .default = "other", .missing = "missing")

# For factor values, use only named replacements
# and supply default with levels()
factor_vec <- factor(c("a", "b", "c"))
recode(factor_vec, a = "Apple", .default = levels(factor_vec))

# Use recode_factor() to create factors with levels ordered as they
# appear in the recode call. The levels in .default and .missing
# come last.
recode_factor(num_vec, `1` = "z", `2` = "y", `3` = "x")
recode_factor(num_vec, `1` = "z", `2` = "y", `3` = "x",
              .default = "D")
recode_factor(num_vec, `1` = "z", `2` = "y", `3` = "x",
              .default = "D", .missing = "M")

# When the input vector is a compatible vector (character vector or
# factor), it is reused as default.
recode_factor(letters[1:3], b = "z", c = "y")
recode_factor(factor(letters[1:3]), b = "z", c = "y")

# Use a named character vector to recode factors with unquote splicing.
level_key <- c(a = "apple", b = "banana", c = "carrot")
recode_factor(char_vec, !!!level_key)
}
\seealso{
\code{\link[=na_if]{na_if()}} to replace specified values with a \code{NA}.

\code{\link[=coalesce]{coalesce()}} to replace missing values with a specified value.

\code{\link[tidyr:replace_na]{tidyr::replace_na()}} to replace \code{NA} with a value.
}
