// printsxp.c: Print routines for debugging the jit compiler.
//
// This module provides routines for printing detailed information in
// SEXPs.  The routines will not force promises while printing, to
// minimize interference.
// A routine is also provided for tracing eval().
//
// RA_TODO this module could go away or be better integrated

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif
#include "Defn.h"
#include "Print.h"
#include "jit.h"
#include "printsxp.h"

#define STRLEN          80

int traceEvalFlag;    // set in R_ReadConsole to trace eval() after booting
int printSxpDepth;    // prevents eval of promises when in printSxp()

SEXP traceEval(CSEXP e, CSEXP env, const char msg[])
{
    if (printSxpDepth) {        // don't call eval when printing
        if (TYPEOF(e) == PROMSXP && PRVALUE(e) == R_UnboundValue) {
            // Rprintf("# Skipped promise eval while printing\n");
            // e = R_NilValue;     // this hack must be used with care
        }
    } else if (traceEvalFlag) {
        int i;
        Rprintf("\n");
        for (i = 0; i < R_EvalDepth; i++)
            Rprintf("=");
        Rprintf("%d %s ", R_EvalDepth, msg);
        printSxp(e, env, FALSE);
    }
    return e;
}

// Get the c function associated with the LANGSXP s if possible.
// If not possible, return NULL.

static CCODE getCfun(CSEXP s)
{
    CCODE cfun = NULL;
    if (TYPEOF(s) == LANGSXP) {
        CSEXP car = CAR(s);
        if (TYPEOF(car) == SYMSXP) {
                CSEXP symvalue = SYMVALUE(car);
                if (TYPEOF(symvalue) == SPECIALSXP)
                    cfun = PRIMFUN(symvalue);
        }
    }
    return cfun;
}

// Deparse p to a short string and append "..." if short string is too short.
//
// This partially expands expressions some expressions such as those
// beginning with "{" so you don't get just a single "{" back.
// It treats R_NilValue as a special case.
//
// Note that this is not re-entrant because it has a local static buffer.
// The strcats in the code are horribly inefficient but it doesn't matter.

char *deparseAsShortString(CSEXP s)
{
    const int SHORTLEN = STRLEN - 10;   // leave space for appending "..."
    CCODE cfun;
    static char str1[3 * STRLEN];	// 3* for strncats when cfun==do_begin
    const char *str2;
    if (s == R_NilValue)
        return "NULL";
    printSxpDepth++;
    assert(TYPEOF(s) == LANGSXP);
    str2 = CHAR(STRING_ELT(deparse1line(s, FALSE), 0));
    str1[0] = 0;    // use strncat because strncpy doesn't append 0
    strncat(str1, str2, SHORTLEN);
    cfun = getCfun(s);
    if (cfun == do_begin) {
        // expand and append body of "{"
        const char *str3 = CHAR(STRING_ELT(deparse1line(CADR(s), FALSE), 0));
        strncat(str1, " ", SHORTLEN);
        strncat(str1, str3, SHORTLEN);
        strcat(str1, " ... }");
    } else if (strlen(str2) >= SHORTLEN ||
               cfun == do_if     ||
               cfun == do_while  ||
               cfun == do_for    ||
               cfun == do_repeat) {
        strcat(str1, " ...");
    }
    printSxpDepth--;
    return str1;
}

static void printPROM(CSEXP s, CSEXP env, Rboolean details)
{
    CSEXP value = PRVALUE(s);
    Rprintf("{\n\tpromise-value      ");
    if (value == R_UnboundValue || TYPEOF(value) != CLOSXP)
        printSxp(value, env, details);
    else    // prevent recursion
        Rprintf("closure\n");
    Rprintf("\tpromise-expression ");
    printSxp(PRCODE(s), env, details);
    Rprintf("\t} # end-promise\n");
}

static void printLANG(CSEXP s, CSEXP env, Rboolean details)
{
    // The following call to deparse will invoke eval if there are
    // promises in s.  But traceEval checks printSxpDepth and
    // so will cause an immediate return from eval without
    // evaluating the promise.

    CSEXP attr = getAttrib(s, R_SourceSymbol);
    if (!isString(attr))
        Rprintf("%s\n", deparseAsShortString(s));
    else
        Rprintf("%s ...\n", CHAR(STRING_ELT(attr, 0)));
    if (details) {
        Rprintf("            language-car ");
        printSxp(CAR(s), env, details);
        Rprintf("            language-cdr ");
        printSxp(CDR(s), env, details);
    }
}

static void printSYM(CSEXP s, CSEXP env, Rboolean details)
{
    CSEXP cdr = CDR(s);
    Rprintf("\"%s\" ", CHAR(PRINTNAME(s)));
    if (!(TYPEOF(cdr) == SYMSXP &&   // prevent recursion
            0 == strcmp(CHAR(PRINTNAME(s)),
                        CHAR(PRINTNAME(cdr))))) {
        if (details) {
            Rprintf("symbol-value ");
            printSxp(cdr, env, details);
        }
        else
            Rprintf("\n");
        if (env == NILSXP)          // should never happen but it does
            Rprintf("\t\t\t\t\tenv is NILSXP!\n");
        else {
            CSEXP val = findVar(s, env);
            if (details && val != R_UnboundValue) {
                Rprintf("\t\t\t\t\t\tbound-value-of-symbol ");
                if (TYPEOF(val) == PROMSXP) // prevent recursion
                    Rprintf("%s promise\n",
                            (PRSEEN(val)? "evaluated": "unevaluated"));
                else
                    printSxp(val, env, FALSE);
            }
        }
    }
    if (TAG(s) != R_NilValue) {
        Rprintf("\t\t\t\tsymbol-internal ");
        printSxp(TAG(s), env, details);
    }
}

#if 0
static void printENV(CSEXP env)
{
    if (env == R_GlobalEnv)
        Rprintf("R_GlobalEnv ");
    else if (env == R_BaseEnv)
        Rprintf("base ");
    else if (env == R_EmptyEnv)
        Rprintf("R_EmptyEnv ");
    else if (R_IsPackageEnv(env))
        Rprintf("%s ",
                translateChar(STRING_ELT(R_PackageEnvName(env), 0)));
    else if (R_IsNamespaceEnv(env))
        Rprintf("namespace:%s ",
                translateChar(STRING_ELT(R_NamespaceEnvSpec(env), 0)));
    else
        Rprintf("%p ", env);
    Rprintf("\n\t\tenv-hashtab ");
    if (HASHTAB(env) == R_NilValue)
        Rprintf("none ");
    else
        Rprintf("%p ", env);
    Rprintf("\n\t\tenv-frame ");
    printSxp(FRAME(env),  R_NilValue, TRUE);
    Rprintf("\t\tenv-enclos ");
    printSxp(ENCLOS(env), R_NilValue, FALSE);
}
#endif

#if 0
static void printIfNonZero(const char *name, const unsigned i)
{
    if (i != 0)
        Rprintf("%s=%x ", name, i);
}
#endif

static void printSxpAux(CSEXP s, CSEXP env, Rboolean details)
{
    PROTECT(s); // necessary?
    switch (TYPEOF(s)) {
        case PROMSXP:
            printPROM(s, env, details);
            break;
        case LANGSXP:
            printLANG(s, env, details);
            break;
        case SYMSXP:
            printSYM(s, env, details);
            break;
#if 0
        case ENVSXP:
            printENV(s);
            break;
#endif
        case VECSXP:
        case CLOSXP:
            if (details)
                PrintValueRec(s, env);
            Rprintf("\n");
            break;
        case LISTSXP:
            Rprintf("\n\tpairlist-tag ");
            printSxp(TAG(s), env, details);
            Rprintf("\tpairlist-car ");
            printSxp(CAR(s), env, details);
            Rprintf("\tpairlist-cdr ");
            if (details)
                printSxp(CDR(s), env, details);
            else
                Rprintf("%s %s\n",
                        type2char(TYPEOF(CDR(s))),
                        (TYPEOF(CDR(s)) == NILSXP)? "": "[rest omitted]");
            break;
        case SPECIALSXP:
        case BUILTINSXP:
            Rprintf("\"%s\"\n", PRIMNAME(s));
            break;
        default:
            PrintValueRec(s, env);
            break;
    }
    UNPROTECT(1);
}

// Print s, truncate it if too long.
// This extends PrintValueRecord for jit compiler debugging.  If called via
// traceEval(), it will not evaluate promises to minimize heisenberging.
// It's a useful mess.  Change the #ifs below to suit your current purpose.

void printSxp(CSEXP s, CSEXP env, Rboolean details)
{
    printSxpDepth++;
    if (s == NULL)                  // should never happen but it does
        Rprintf("NULL\n");
    else if (s == R_UnboundValue)
        Rprintf("R_UnboundValue\n");
    else if (s == R_NilValue)
        Rprintf("R_NilValue\n");
    else {
        extern R_print_par_t R_print;
        int oldPrintMax = R_print.max;
        R_print.max = 5;
#if 0
        Rprintf("%p ", s);
#endif
        if (TYPEOF(s) != 0)
            Rprintf("%s ", type2char(TYPEOF(s)));
#if 0
        printIfNonZero("named", s->sxpinfo.named);
        if (s->sxpinfo.gp)
            Rprintf("gp=0x%4.4x ", s->sxpinfo.gp);
#endif
#if 0
        struct sxpinfo_struct info = s->sxpinfo;
        // printIfNonZero("type", info.type);
        printIfNonZero("obj", info.obj);
        // printIfNonZero("named", info.named);
        printIfNonZero("gp", info.gp);
        printIfNonZero("mark", info.mark);
        printIfNonZero("debug", info.debug);
        printIfNonZero("trace", info.trace);
        printIfNonZero("spare", info.spare);
        printIfNonZero("gcgen", info.gcgen);
        printIfNonZero("gccls", info.gccls);
#endif
#if 0   /* check for an uninited SEXP, see TORTURE_FILL_FREE in memory.c */
        if ((unsigned)CAR(s) == 0xeeeeeeee)
            Rprintf("[CAR(s) uninitialized] ");
        if ((unsigned)CDR(s) == 0xeeeeeeee)
            Rprintf("[CDR(s) uninitialized] ");
        if ((unsigned)TAG(s) == 0xeeeeeeee)
            Rprintf("[TAG(s) uninitialized] ");
#endif
        if (isVector(s)) {
            int len = LENGTH(s);
            Rprintf("length %d ", len);
            if (len < 0 || len > 1e6) {
                Rprintf("in hex 0x%x!\n", len);
                printSxpDepth--;
                return;
            }
        }
        printSxpAux(s, env, details);
        R_print.max = oldPrintMax;
        printSxpDepth--;
    }
}

// prints printf style message "format", followed by "s"

void printfSxp(CSEXP s, const char *format, ...)
{
    static const int BUFSIZE = 200;
    int nchars;
    char buf[BUFSIZE];
    va_list(ap);
    va_start(ap, format);
    nchars = vsnprintf(buf, BUFSIZE-1, format, ap);
    assert(nchars >= 0);
    buf[nchars] = 0;    // guarantee null termination
    va_end(ap);
    Rprintf("%s ", buf);
    printSxp(s, R_GlobalEnv, FALSE);
}
