/***************************************************************************
 *   Copyright (C) 2003 by Moris Ravasio                                   *
 *   moris_ravasio@hotmail.com                                             *
 *                                                                         *
 *   Copyright (C) 2007 by Matvey Kozhev                                   *
 *   sikon@lucidfox.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "qinkdlg.h"
#include "aboutdlg.h"
#include "device.h"

#include <QMenu>
#include <QSettings>

QInkDlg::QInkDlg(QWidget* parent, const char* name)
        : QDialog(parent, name)
{
    setupUi(this);
    
    QIcon icon(ICONFILE("64x64"));
    icon.addFile(ICONFILE("32x32"));
    icon.addFile(ICONFILE("16x16"));
    setWindowIcon(icon);
    
    grpError->hide();
    grpLevels->show();
    lblIcon->setPixmap(QPixmap(RESDIR "/app.png"));

    // make the window as compact as possible
    QSize suggestedSize = sizeHint();
    resize(QSize(width(), suggestedSize.height()));
    
    QRect desk;
    desk = QApplication::desktop()->geometry();
    move(desk.center() - rect().center());
    
    initBarMap();
    initTrayIcon(icon);
    loadState();
    autoDetect();
}

QInkDlg::~QInkDlg()
{
}

void QInkDlg::initTrayIcon(const QIcon& icon)
{
    trayIcon = new QSystemTrayIcon(this);
    trayIcon->setIcon(icon);
    connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),    SLOT(trayActivated(QSystemTrayIcon::ActivationReason)));
    
    QMenu *menu = new QMenu();
    QAction *action;
    
    action = menu->addAction(tr("&Show/hide"));
    connect(action, SIGNAL(triggered()), SLOT(toggleVisibility()));
    
    action = menu->addAction(tr("&About"));
    connect(action, SIGNAL(triggered()), SLOT(btnAbout_clicked()));
    
    menu->addSeparator();
    action = menu->addAction(tr("&Exit"));
    connect(action, SIGNAL(triggered()), SLOT(exit()));
    
    trayIcon->setContextMenu(menu);
    trayIcon->show();
}

void QInkDlg::initBarMap()
{
    barMap.insert(CARTRIDGE_BLACK, barBlack);
    barMap.insert(CARTRIDGE_COLOR, barColor);
    barMap.insert(CARTRIDGE_PHOTO, barPhoto);
    barMap.insert(CARTRIDGE_CYAN, barCyan);
    barMap.insert(CARTRIDGE_MAGENTA, barMagenta);
    barMap.insert(CARTRIDGE_YELLOW, barYellow);
//    barMap.insert(CARTRIDGE_PHOTOBLACK, barPhotoBlack); // TODO: Not implemented yet - add it?
    barMap.insert(CARTRIDGE_PHOTOCYAN, barPhotoCyan);
    barMap.insert(CARTRIDGE_PHOTOMAGENTA, barPhotoMagenta);
    barMap.insert(CARTRIDGE_PHOTOYELLOW, barPhotoYellow);
}

void QInkDlg::loadState()
{
    QSettings settings(QSettings::IniFormat, QSettings::UserScope, "LucidFox", "QInk");
    
    if(settings.contains("device"))
        lastDevice.deserialize(settings.value("device").toString());
    
    setVisible(settings.value("visible", true).toBool());
}

void QInkDlg::saveState()
{
    QSettings settings(QSettings::IniFormat, QSettings::UserScope, "LucidFox", "QInk");
    
    settings.setValue("device", lastDevice.serialize());
    settings.setValue("visible", isVisible());
    settings.sync();
}

void QInkDlg::trayActivated(QSystemTrayIcon::ActivationReason reason)
{
    if(reason == QSystemTrayIcon::DoubleClick)
        toggleVisibility();
}

void QInkDlg::toggleVisibility()
{
    if(isVisible())
    {
        hide();
    }
    else
    {
        updateView();
        show();
    }

    saveState();
}

void QInkDlg::close()
{
    hide();
    saveState();
}

void QInkDlg::cmbDevices_activated(const QString &)
{
    lastDevice = devList[cmbDevices->currentItem()];
    updateView();
}

void QInkDlg::btnAdd_clicked()
{}

void QInkDlg::btnDetect_clicked()
{
    // Auto detect printers
    autoDetect();
}

void QInkDlg::btnAbout_clicked()
{
    AboutDlg dlg(this);
    dlg.exec();
}

void QInkDlg::btnUpdate_clicked()
{
    updateView();
}

void QInkDlg::btnClose_clicked()
{
    close();
}

void QInkDlg::exit()
{
    delete trayIcon;
    saveState();
    QDialog::close();
}

void QInkDlg::autoDetect(void)
{
    Device::enumerate(devList);

    // Are there some printers?
    if (devList.empty())
    {
        // No printers detected: show error
        lastDevice = Device::invalid;
        cmbDevices->setEnabled(false);
        btnUpdate->setEnabled(false);

        showError(tr("<b>No printers detected on this system</b><br><br>It seems that there are no printer plugged to this computer. Please verify the connections of your devices; maybe a problem of wrong permissions on the device's port.<br><br>For parallel printers try <b>chmod 666 /dev/lpX</b> and <br><b>chmod 666 /dev/parportX</b> (as root), where X is your printer's parallel port number.<br><br>For USB printers, try <b>chmod 666 /dev/usblpX</b> (as root), where X is your printer's USB port number.<br><br>Click on <b>Detect</b> to try again."));
    }
    else
    {
        cmbDevices->setEnabled(true);
        btnUpdate->setEnabled(true);

        // Try to keep the focus on the previously selected device
        bool lastFound = false;
        
        for (int i = 0; i < devList.size(); i++)
        {
            cmbDevices->addItem(devList[i]);
        }
        
        for (int i = 0; i < devList.size(); i++)
        {
            if (lastDevice == devList[i])
            {
                lastFound = true;
                cmbDevices->setCurrentItem(i);
                break;
            }
        }

        if (!lastFound)
        {
            cmbDevices->setCurrentItem(0);
            lastDevice = devList[0];
        }

        grpError->hide();
        grpLevels->show();
        updateView();
    }
}

/*!
    \fn QInkDlg::showError(void)
 */
void QInkDlg::showError(QString errStr)
{
    lblErrorDescr->setText(errStr);
    grpLevels->hide();
    grpError->show();
}

void QInkDlg::updateView(void)
{
    // Disable all progress bars preventively
    QList<QProgressBar *> barList = barMap.values();
    
    for(QList<QProgressBar *>::iterator it = barList.begin(); it != barList.end(); ++it)
    {
        (*it)->setEnabled(false);
        (*it)->setValue(0);
    }
    
    grpError->hide();
    grpLevels->show();

    int callback = lastDevice.update();
    
    if(callback == INKLEVEL_OK)
    {
        QString toolTip = lastDevice + "\n\n";
        
        for(QMap<int, QProgressBar *>::iterator it = barMap.begin(); it != barMap.end(); ++it)
        {
            int level = lastDevice.getInkLevel(it.key());
            
            if(level != -1)
            {
                QProgressBar *bar = it.value();
                bar->setEnabled(true);
                bar->setValue(level);
                toolTip += tr("%1: %2%\n").arg(bar->toolTip()).arg(level);
            }
        }
        
        trayIcon->setToolTip(toolTip.trimmed());
    }
    else
    {
        switch(callback)
        {
        case DEV_PARPORT_INACCESSIBLE:
        case DEV_LP_INACCESSIBLE:
        case COULD_NOT_GET_DEVICE_ID:
        case DEV_USB_LP_INACCESSIBLE:
        case COULD_NOT_WRITE_TO_PRINTER:
        case COULD_NOT_READ_FROM_PRINTER:
        case ERROR:
        case UNKNOWN_PORT_SPECIFIED:
        case NO_DEVICE_CLASS_FOUND:
        case NO_CMD_TAG_FOUND:
        case NO_PRINTER_FOUND:
        case COULD_NOT_PARSE_RESPONSE_FROM_PRINTER:
            showError(tr("<b>Generic error</b><br><br>An error has occured communicating with the selected device. Click <b>Detect</b> and <b>Update</b> to refresh QInk status and try again. If the problem persists, e-mail me a description of the situation and all that can be helpful to me for reconstructing your execution environment."));
            break;

        case NO_INK_LEVEL_FOUND:
            showError(tr("<b>Cannot read ink level from device</b><br><br>Cannot retrieve ink levels from the selected device. Ensure your printer is switched on and click on <b>Update</b> to try again."));
            break;

        case PRINTER_NOT_SUPPORTED:
            showError(tr("<b>Unsupported printer!</b><br><br>Selected printer is not supported by this version of libinklevel. Take a look to <a href='http://home.arcor.de/markusheinz'>http://home.arcor.de/markusheinz</a> for a complete list of devices supported by your version of libinklevel."));
            break;
        }
    }
}
