#!/usr/bin/env python
# File created on 20 Oct 2010
from __future__ import division

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2011, The QIIME Project"
__credits__ = ["Greg Caporaso","Justin Kuczynski"]
__license__ = "GPL"
__version__ = "1.3.0"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"
__status__ = "Release"
 

from cogent.util.unit_test import TestCase, main
from qiime.parallel.beta_diversity import assemble_distance_matrix
from qiime.parse import parse_distmat_to_dict
from qiime.util import get_qiime_scripts_dir
import qiime.util

import tempfile
import string
import random
import os
import shutil
import subprocess

class ParallelBetaDiversityTests(TestCase):

    def setUp(self):
        """ """
        self.dm_comp1 = dm_comp1.split('\n')
        self.dm_comp2 = dm_comp2.split('\n')
        self.dm_comp3 = dm_comp3.split('\n')
        self.dm_comp4 = dm_comp4.split('\n')
        self.expected = expected.split('\n')

        self.dirs_to_remove = []

    def tearDown(self):
        """ """

        for d in self.dirs_to_remove:
            if os.path.exists(d):
                shutil.rmtree(d)
    def test_parallel_beta_diversity(self):
        """parallel should be the same dist mtx as serial"""
        qiime_config = qiime.util.load_qiime_config()
        tempdir = qiime_config['temp_dir'] or tempfile.gettempdir() 
        # tempfile may not work on cluster, if e.g. /tmp isn't mirrored via nfs
        maindir = os.path.join(tempdir,
         ''.join(random.choice(string.ascii_letters + string.digits) \
         for x in range(10)))


        os.makedirs(maindir)
        self.dirs_to_remove.append(maindir)
        otuf = os.path.join(maindir,'otuf')
        treef = os.path.join(maindir,'treef')

        otufh = open(otuf,'w')
        otufh.write(tutorial_otu_table)
        otufh.close()

        treefh = open(treef,'w')
        treefh.write(tutorial_tree)
        treefh.close()

        scripts_dir = get_qiime_scripts_dir()
        # parallel
        cmd = scripts_dir+'/parallel_beta_diversity.py -T -O 3 '+\
         '--retain_temp_files -i %s -o %s -m unifrac -t %s' %\
         (otuf, maindir+'/para1', treef)
        # -T so doesn't return yet
        proc = subprocess.Popen(cmd,shell=True, 
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        betaout, betaerr = proc.communicate()
        # first paralell version
        if betaout or betaerr:
            raise RuntimeError("parallel_beta_diversity.py should should "+\
              "not generate stdout or stderr. results:" + betaout + betaerr)

        # retain temp files doesn't matter, we just delete the folder

        # now with serial bdiv
        cmd=scripts_dir+'/beta_diversity.py -i %s -o %s -m unifrac -t %s' %\
         (otuf, maindir+'/serial1', treef)
        proc = subprocess.Popen(cmd,shell=True, 
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        betaout, betaerr = proc.communicate()
        if betaout or betaerr:
            raise RuntimeError("parallel_beta_diversity.py should should "+\
              "not generate stdout or stderr. results:" + betaout + betaerr)


        serialdist =\
            parse_distmat_to_dict(open(maindir+'/serial1/unifrac_otuf','U'))

        paradist =\
            parse_distmat_to_dict(open(maindir+'/para1/unifrac_otuf','U'))

        web_res = open(maindir+'/web_res','w')

        ## use unifrac_dmtx below, from fast unifrac website march 2011
        web_res.write(unifrac_dmtx)
        web_res.close()
        unifdist = parse_distmat_to_dict(open(maindir+'/web_res','U'))
        self.assertFloatEqual(serialdist, unifdist)
        self.assertFloatEqual(paradist, unifdist)

    def test_parallel_beta_diversity2(self):
        """ extra tips in tree should not affect the unifrac dmtx"""
        qiime_config = qiime.util.load_qiime_config()
        tempdir = qiime_config['temp_dir'] or tempfile.gettempdir() 
        # tempfile may not work on cluster, if e.g. /tmp isn't mirrored via nfs
        maindir = os.path.join(tempdir,
         ''.join(random.choice(string.ascii_letters + string.digits) \
         for x in range(10)))


        os.makedirs(maindir)
        self.dirs_to_remove.append(maindir)
        otuf = os.path.join(maindir,'otuf')
        treef = os.path.join(maindir,'treef')
        trim_treef = os.path.join(maindir,'trim_treef')

        otufh = open(otuf,'w')
        otufh.write(big_otu_table)
        otufh.close()

        treefh = open(treef,'w')
        treefh.write(big_tree)
        treefh.close()

        treefh = open(trim_treef,'w')
        treefh.write(big_tree_trimmed)
        treefh.close()

        scripts_dir = get_qiime_scripts_dir()


        # parallel
        cmd = scripts_dir+'/parallel_beta_diversity.py -T -O 3 '+\
         '--retain_temp_files -i %s -o %s -m unifrac -t %s' %\
         (otuf, maindir+'/para1', treef)
        # -T so doesn't return yet
        proc = subprocess.Popen(cmd,shell=True, 
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        betaout, betaerr = proc.communicate()
        if betaout or betaerr:
            raise RuntimeError("parallel_beta_diversity.py should should "+\
              "not generate stdout or stderr. results:" + betaout + betaerr)

        # parallel on trimmed_tree
        cmd = scripts_dir+'/parallel_beta_diversity.py -T -O 3 '+\
         '--retain_temp_files -i %s -o %s -m unifrac -t %s' %\
         (otuf, maindir+'/para_trim', trim_treef)
        proc = subprocess.Popen(cmd,shell=True, 
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        betaout, betaerr = proc.communicate()
        if betaout or betaerr:
            raise RuntimeError("parallel_beta_diversity.py should should "+\
              "not generate stdout or stderr. results:" + betaout + betaerr)


        # retain temp files doesn't matter, we just delete the folder

        # now with serial bdiv
        cmd=scripts_dir+'/beta_diversity.py -i %s -o %s -m unifrac -t %s' %\
         (otuf, maindir+'/serial1', treef)
        proc = subprocess.Popen(cmd,shell=True, 
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        betaout, betaerr = proc.communicate()
        if betaout or betaerr:
            raise RuntimeError(betaout + betaerr)


        serialdist =\
            parse_distmat_to_dict(open(maindir+'/serial1/unifrac_otuf','U'))

        paradist =\
            parse_distmat_to_dict(open(maindir+'/para1/unifrac_otuf','U'))

        paradist_trim =\
            parse_distmat_to_dict(open(maindir+'/para_trim/unifrac_otuf','U'))

        expected = open(maindir+'/expected','w')

        ## use unifrac_dmtx below, from fast unifrac website march 2011
        expected.write(big_dmtx)
        expected.close()
        unifdist = parse_distmat_to_dict(open(maindir+'/expected','U'))
        sam_keys = unifdist.keys()
        for i in range(len(sam_keys)):
            key_i = sam_keys[i]
            for j in range(i):
                key_j = sam_keys[j]
                self.assertFloatEqual(serialdist[key_i][key_j], unifdist[key_i][key_j])
                self.assertFloatEqual(paradist[key_i][key_j], unifdist[key_i][key_j])
                self.assertFloatEqual(paradist_trim[key_i][key_j], unifdist[key_i][key_j])

    def test_assemble_distance_matrix(self):
        """ assemble_distance_matrix functions as expected for a non-symmetric dm
        """
        actual = parse_distmat_to_dict(assemble_distance_matrix(
         [self.dm_comp1,self.dm_comp2,self.dm_comp3,self.dm_comp4]).split('\n'))
        exp = parse_distmat_to_dict(self.expected)
        self.assertEqual(actual,exp)
        
dm_comp1 = """	PC.354	PC.355	PC.356	PC.481	PC.593	PC.607	PC.634	PC.635	PC.636
PC.354	0.0	0.9999	0.637204555763	0.596124964451	0.59620370837	0.736037238732	0.790494137353	0.70551354446	0.758236466255
PC.355	0.623372333874	0.0	0.613486299912	0.631198829777	0.672502144191	0.766682588739	0.732473435165	0.738434734323	0.687363838411"""

dm_comp2 = """	PC.354	PC.355	PC.356	PC.481	PC.593	PC.607	PC.634	PC.635	PC.636
PC.356	0.637204555763	0.613486299912	0.0	0.691902878696	0.742922266508	0.740684448418	0.78590671873	0.723519724117	0.754301399992
PC.481	0.596124964451	0.631198829777	0.691902878696	0.0	0.668731241507	0.690956885033	0.652257003318	0.652311676944	0.661223123598"""

dm_comp3 = """	PC.354	PC.355	PC.356	PC.481	PC.593	PC.607	PC.634	PC.635	PC.636
PC.593	0.59620370837	0.672502144191	0.742922266508	0.668731241507	0.0	0.74048244192	0.760255173866	0.761741286862	0.751777287666
PC.607	0.736037238732	0.766682588739	0.740684448418	0.690956885033	0.74048244192	0.0	0.73093223871	0.688768250911	0.729047763792"""

dm_comp4 = """	PC.354	PC.355	PC.356	PC.481	PC.593	PC.607	PC.634	PC.635	PC.636
PC.634	0.790494137353	0.732473435165	0.78590671873	0.652257003318	0.760255173866	0.73093223871	0.0	0.583577161959	0.596033900207
PC.635	0.70551354446	0.738434734323	0.723519724117	0.652311676944	0.761741286862	0.688768250911	0.583577161959	0.0	0.6271222318
PC.636	0.758236466255	0.687363838411	0.754301399992	0.661223123598	0.751777287666	0.729047763792	0.596033900207	0.6271222318	0.0"""

expected = """	PC.354	PC.355	PC.356	PC.481	PC.593	PC.607	PC.634	PC.635	PC.636
PC.354	0.0	0.9999	0.637204555763	0.596124964451	0.59620370837	0.736037238732	0.790494137353	0.70551354446	0.758236466255
PC.355	0.623372333874	0.0	0.613486299912	0.631198829777	0.672502144191	0.766682588739	0.732473435165	0.738434734323	0.687363838411
PC.356	0.637204555763	0.613486299912	0.0	0.691902878696	0.742922266508	0.740684448418	0.78590671873	0.723519724117	0.754301399992
PC.481	0.596124964451	0.631198829777	0.691902878696	0.0	0.668731241507	0.690956885033	0.652257003318	0.652311676944	0.661223123598
PC.593	0.59620370837	0.672502144191	0.742922266508	0.668731241507	0.0	0.74048244192	0.760255173866	0.761741286862	0.751777287666
PC.607	0.736037238732	0.766682588739	0.740684448418	0.690956885033	0.74048244192	0.0	0.73093223871	0.688768250911	0.729047763792
PC.634	0.790494137353	0.732473435165	0.78590671873	0.652257003318	0.760255173866	0.73093223871	0.0	0.583577161959	0.596033900207
PC.635	0.70551354446	0.738434734323	0.723519724117	0.652311676944	0.761741286862	0.688768250911	0.583577161959	0.0	0.6271222318
PC.636	0.758236466255	0.687363838411	0.754301399992	0.661223123598	0.751777287666	0.729047763792	0.596033900207	0.6271222318	0.0"""


tutorial_tree = """(((((381:0.0213,(214:0.03728,253:0.00015)0.945:0.03224)0.763:0.00483,((269:0.02693,(231:0.00509,(105:0.01425,141:0.02641)0.846:0.01405)0.428:0.00519)0.622:0.00014,404:0.00524)0.795:0.00514)0.773:0.00508,(131:0.00518,(33:0.01631,((284:0.00828,(176:0.03098,388:0.01236)0.901:0.02175)0.885:0.01273,52:0.01046)0.743:0.00498)0.924:0.01603)0.779:0.00511)0.772:0.00014,153:0.00507)0.753:0.00602,(223:0.03237,(172:0.01733,81:0.03834)0.224:0.00414)0.845:0.01076,(136:0.00627,((((265:0.01557,200:0.00517)0.674:0.00014,((204:0.00015,(339:0.01613,(322:0.01633,268:0.01643)0.569:0.0107)0.885:0.00016)0.840:0.00527,((((((((280:0.02348,(395:0.00015,(48:0.03014,((30:0.02665,316:0.01921)0.813:0.01152,215:0.0242)0.850:0.01191)0.320:0.00016)0.912:0.02431)0.694:0.01482,(115:0.01526,364:0.08211)0.879:0.03637)0.677:0.03567,((((((162:0.06933,59:0.02113)0.991:0.08563,(308:0.02061,43:0.03488)0.894:0.04949)0.911:0.05006,(((344:0.00015,(146:0.00015,377:0.01634)0.924:0.0108)0.918:0.01069,((201:0.011,240:0.04792)1.000:0.00015,(61:0.00015,96:0.00523)0.781:0.00514)0.828:0.01056)0.809:0.00016,196:0.04505)0.213:0.00014)0.650:0.00529,(((161:0.01191,(390:0.04307,37:0.03893)0.933:0.03396)0.814:0.01401,68:0.04946)0.953:0.03303,((341:0.01127,393:0.02765)0.941:0.02238,(82:0.01112,(350:0.01141,(156:0.01636,356:0.00015)0.863:0.02214)0.946:0.02475)0.748:0.00565)0.761:0.00968)0.748:0.00836)0.927:0.0224,271:0.05902)0.753:0.00511,(((((((217:0.03796,379:0.00016)0.973:0.05805,(299:0.08963,(382:0.06426,((317:0.00016,((205:0.00532,264:0.03867)0.939:0.01605,(194:0.03374,(32:0.01052,(348:0.02212,157:0.02743)1.000:0.00014)0.868:0.02793)0.745:0.00531)0.336:0.01061)0.789:0.00604,334:0.02104)0.598:0.01527)0.687:0.00354)0.836:0.01564)0.811:0.01617,(292:0.06237,84:0.02159)0.934:0.04776)0.864:0.02103,301:0.06716)0.698:0.0046,272:0.00539)0.809:0.0115,88:0.05965)0.860:0.01208,(276:0.01065,279:0.03443)0.891:0.01124)0.090:0.00014)0.924:0.03938)0.953:0.05227,281:0.02828)0.691:0.00622,25:0.01213)0.727:0.00397,((261:0.01613,((147:0.01555,20:0.00016)0.967:0.02125,(107:0.01089,349:0.03426)0.757:0.00478)0.750:0.00518)0.799:0.0052,(259:0.01616,63:0.01053)0.764:0.00523)0.792:0.00511)1.000:0.00016,(72:0.05949,(1:0.01425,67:0.0377)0.751:0.00762)0.867:0.01609)0.807:0.00507,((49:0.01645,116:0.01633)0.736:0.00514,(398:0.00515,(((180:0.04458,99:0.0328)0.913:0.02521,(((410:0.05589,(((150:0.04425,(170:0.03163,((250:0.00693,331:0.00435)1.000:0.10845,357:0.01319)0.850:0.0225)0.879:0.02887)0.749:0.00795,(((((23:0.00919,248:0.08024)0.405:0.03691,(358:0.05635,369:0.07223)0.978:0.09469)0.888:0.05975,(234:0.07249,8:0.00016)0.712:0.01829)0.976:0.07916,(((275:0.094,(((114:0.0269,302:0.02202)0.985:0.06964,(213:0.06889,42:0.03436)0.415:0.01928)0.795:0.02064,((110:0.05188,342:0.01457)0.967:0.08524,((123:0.02756,343:0.0481)0.800:0.01738,((298:0.03283,(124:0.02507,6:0.03351)0.781:0.01076)0.939:0.03194,309:0.04124)0.820:0.01321)0.985:0.0961)0.928:0.06559)0.902:0.03886)0.684:0.03217,373:0.06838)0.909:0.03592,((290:0.02673,380:0.00015)1.000:0.16099,(((90:0.09952,192:0.10171)0.679:0.01316,(326:0.03972,45:0.09053)0.965:0.05309)0.115:0.00014,(375:0.00015,(221:0.00071,278:0.05255)1.000:0.08313)1.000:0.10921)0.623:0.0222)0.892:0.03509)0.465:0.00015)0.980:0.05443,(((306:0.08813,385:0.14214)0.269:0.00862,((256:0.01776,(273:0.07543,69:0.01333)0.591:0.02343)0.883:0.02549,((132:0.02365,219:0.01597)0.897:0.02388,(100:0.01243,50:0.0237)0.226:0.01766)0.961:0.04348)0.848:0.01577)0.998:0.08323,(241:0.23207,(130:0.24778,(53:0.12887,(129:0.07692,318:0.01288)0.900:0.04845)0.817:0.02143)0.888:0.05464)0.657:0.01537)0.822:0.01876)0.828:0.01549)0.773:0.01019,((98:0.12681,((148:0.0294,391:0.00571)0.989:0.07803,(389:0.10107,(252:0.00014,362:0.01104)0.964:0.06682)0.834:0.03217)0.762:0.0152)0.524:0.0181,(0:0.0483,(135:0.01151,(300:0.0175,(274:0.04561,((((166:0.02935,355:0.00015)0.833:0.00565,41:0.00014)0.807:0.00586,(226:0.01038,92:0.0044)0.792:0.00425)0.961:0.03236,((360:0.01752,7:0.0182)0.748:0.00495,(368:0.02316,288:0.01783)0.759:0.00622)0.707:0.00573)0.841:0.00015)0.949:0.02275)0.745:0.00559)0.855:0.02344)0.876:0.03532)0.885:0.02567)0.752:0.00645)0.782:0.00969,(((((((178:0.01576,(230:0.02704,64:0.02146)0.869:0.0108)0.809:0.01014,((122:0.00448,354:0.0225)0.855:0.01127,(333:0.01086,406:0.01648)0.748:0.00433)0.789:0.00624)0.171:0.00516,((416:0.04298,(400:0.01045,74:0.01051)0.923:0.00014)0.862:0.02166,(307:0.04097,(260:0.03574,335:0.0434)0.747:0.00875)0.916:0.02837)0.843:0.00987)0.804:0.00016,((237:0.09447,((370:0.01631,(319:0.04803,(60:0.01986,405:0.01742)0.560:0.01574)0.898:0.01971)0.918:0.01584,(384:0.02116,(245:0.01047,(177:0.0051,(183:0.03226,413:0.00014)0.826:0.00518)0.777:0.00501)0.923:0.0158)0.622:0.00016)0.685:0.00099)0.224:0.02406,((22:0.03142,5:0.06696)0.870:0.03448,47:0.0347)0.763:0.01052)0.847:0.01209)0.743:0.00534,((((62:0.00137,(121:0.00016,78:0.04376)1.000:0.10609)0.942:0.0378,(311:0.05626,407:0.06902)0.944:0.04614)0.703:0.00608,(((188:0.01993,202:0.02611)0.914:0.02118,(328:0.0273,337:0.00015)0.815:0.01019)0.852:0.01169,(330:0.03441,((386:0.13035,(392:0.00544,(321:0.02191,4:0.01061)0.763:0.0052)0.932:0.00014)0.671:0.01096,145:0.01556)0.829:0.01073)0.735:0.00529)0.840:0.01052)0.849:0.01531,(262:0.0683,((310:0.05551,((83:0.01296,(127:0.01909,212:0.01393)0.090:0.00499)0.876:0.01352,(104:0.00014,171:0.01061)0.895:0.01717)0.877:0.02683)0.940:0.03929,(119:0.0152,179:0.00197)0.889:0.02843)0.066:0.01551)0.839:0.01374)0.820:0.01069)0.869:0.01061,(((293:0.01741,168:0.04514)0.046:0.01491,345:0.03334)0.248:0.01629,(31:0.04727,97:0.04999)0.915:0.03556)0.811:0.01631)0.010:0.00016,(((94:0.0671,(108:0.00014,229:0.06991)0.630:0.01827)0.982:0.06031,(143:0.02201,((((((198:0.02745,(140:0.14724,75:0.02831)0.817:0.0209)0.851:0.01902,(((282:0.06783,54:0.00015)0.952:0.03641,((313:0.03746,80:0.00524)0.872:0.0215,2:0.07468)0.862:0.02589)0.916:0.03914,((367:0.0099,(((128:0.0425,((111:0.06727,11:0.00495)0.974:0.02953,283:0.02606)0.504:0.00357)0.862:0.02044,(289:0.04546,(399:0.00319,((((152:0.00014,19:0.06307)0.992:0.03752,154:0.00016)0.786:0.00014,134:0.06945)0.997:0.06109,51:0.00014)0.994:0.04556)0.353:0.00583)0.482:0.00828)0.933:0.03536,112:0.07957)0.734:0.00733)0.962:0.08492,403:0.10375)0.869:0.0525)0.894:0.03949)0.645:0.00925,((((287:0.00534,15:0.05518)0.920:0.03189,(((304:0.00508,409:0.00015)0.991:0.00014,(120:0.00015,(57:0.04309,56:0.0156)0.759:0.00015)0.902:0.01019)0.339:0.01644,173:0.094)0.787:0.01131)1.000:0.07731,(236:0.00625,((26:0.04569,(((351:0.005,(27:0.03624,(137:0.01569,(314:0.00015,408:0.03277)0.991:0.03257)0.806:0.00498)0.851:0.00588)0.928:0.01791,((133:0.04374,(227:0.00527,(412:0.00014,(175:0.00507,((95:0.01566,210:0.00014)0.438:0.01045,191:0.00016)0.781:0.00518)0.815:0.00508)0.859:0.01021)0.745:0.00667)0.735:0.01956,((((12:0.01588,415:0.01701)0.121:0.03139,(73:0.04886,(17:0.00016,(46:0.02083,378:0.01021)0.886:0.01027)0.785:0.019)0.719:0.02118)0.774:0.01959,329:0.01522)0.777:0.01121,(((286:0.00722,(394:0.01596,(372:0.00015,225:0.0446)0.884:0.0109)0.929:0.02558)0.584:0.00985,218:0.02283)0.888:0.01478,159:0.02121)0.739:0.00866)0.851:0.01129)0.728:0.00602)0.866:0.01998,93:0.04869)0.604:0.00297)0.648:0.01633,199:0.06704)0.788:0.01956)0.371:0.01052)0.827:0.01491,((244:0.0262,(126:0.00015,163:0.03192)0.984:0.04377)0.817:0.01306,((216:0.00014,(86:0.02257,(21:0.01127,34:0.01066)0.859:0.01088)0.622:0.017)0.998:0.19434,(233:0.00244,(182:0.01898,(239:0.02877,267:0.00015)0.839:0.01438)0.999:0.09419)0.975:0.15234)0.877:0.07457)0.893:0.0244)0.821:0.02013)0.998:0.10422,(195:0.10508,((249:0.0368,(336:0.04596,((263:0.02407,(277:0.01295,190:0.03788)0.823:0.01671)0.698:0.0068,197:0.01756)0.309:0.01631)0.860:0.01866)0.926:0.02656,(303:0.04293,(113:0.04423,347:0.04295)0.930:0.03972)0.885:0.02484)0.701:0.00015)0.902:0.03629)0.841:0.02905,(246:0.00014,(125:0.03009,184:0.0229)0.998:0.07478)0.999:0.10301)0.936:0.04978,((247:0.04204,((((((238:0.01393,(109:0.01081,39:0.02762)0.769:0.00519)0.758:0.00702,(257:0.01539,85:0.07408)0.746:0.00558)0.755:0.01039,(363:0.04294,155:0.00015)0.943:0.02426)0.894:0.01745,266:0.00586)0.948:0.03346,55:0.02705)0.739:0.00453,203:0.00015)0.855:0.02077)0.995:0.07638,327:0.00745)0.921:0.03692)0.553:0.01549)0.970:0.05544)0.858:0.02855,338:0.08163)0.892:0.03304)0.759:0.00673)0.945:0.02495,((((((((102:0.04317,36:0.02415)0.964:0.03758,65:0.00505)0.822:0.01078,366:0.00016)0.811:0.01537,(315:0.01071,((151:0.0,160:0.0):0.00016,340:0.00014)0.842:0.01037)0.951:0.02712)0.724:0.00057,(185:0.04527,(207:0.01304,76:0.00341)0.949:0.03474)0.845:0.0196)0.871:0.0106,(371:0.02805,(164:0.0104,242:0.02179)0.758:0.0052)0.771:0.00538)0.841:0.01097,174:0.13953)0.831:0.01033,(144:0.01866,(3:0.01578,312:0.00015)0.785:0.00532)0.780:0.00615)0.752:0.00572)0.667:0.00244)0.268:0.00339,((101:0.04199,77:0.00334)0.965:0.0345,((14:0.01106,294:0.00502)0.891:0.01811,(285:0.01062,397:0.01076)0.758:0.00896)0.163:0.01034)0.850:0.01331)0.563:0.00537)0.800:0.00519)0.930:0.00016)0.759:0.01023)1.000:0.00014)0.850:0.00015,(243:0.03373,220:0.01032)0.888:0.011)0.540:0.00014,(189:0.02629,(((139:0.0155,186:0.01757)0.420:0.01444,(((((((165:0.0059,58:0.03297)0.779:0.02132,((222:0.01678,(323:0.02243,44:0.04081)0.819:0.01102)0.063:0.00015,(106:0.03989,149:0.02047)0.775:0.01298)0.706:0.0074)0.957:0.03281,((((258:0.04247,87:0.0123)0.500:0.01067,235:0.00735)0.645:0.00296,208:0.00505)1.000:0.00015,((18:0.00454,(((10:0.04233,(414:0.00016,(142:0.01127,66:0.03479)0.756:0.00498)0.726:0.00685)0.486:0.01639,181:0.00014)0.784:0.00501,(167:0.01463,(320:0.00885,402:0.00881)0.791:0.00014)0.839:0.01499)0.773:0.00524)0.893:0.01079,(169:0.00517,(295:0.01586,297:0.03792)0.262:0.00016)0.778:0.00521)0.818:0.00528)0.764:0.01062)0.767:0.00486,70:0.00512)0.766:0.00495,(((332:0.00016,((325:0.01591,(383:0.00014,(361:0.01642,(138:0.04133,(158:0.0036,224:0.00657)0.840:0.01972)0.769:0.00881)0.777:0.00496)0.882:0.01036)0.752:0.00492,(24:0.03974,((((254:0.00541,(251:0.00015,(324:0.02187,((117:0.0052,(374:0.03165,270:0.02362)0.731:0.00708)0.791:0.00525,13:0.01621)0.757:0.00511)0.607:0.01283)0.889:0.0192)0.852:0.01583,305:0.01647)0.948:0.00015,211:0.00015)0.419:0.00016,(103:0.01686,209:0.05269)0.861:0.01595)0.937:0.01635)0.756:0.00523)0.878:0.01048)0.776:0.00238,(365:0.03251,((38:0.04434,79:0.00014)0.758:0.00016,(296:0.043,9:0.00518)0.693:0.0162)0.508:0.00805)0.766:0.00767)0.764:0.00313,(((359:0.02181,(16:0.04469,(232:0.01621,(118:0.03421,(29:0.01612,353:0.01494)0.293:0.01034)0.864:0.01326)0.747:0.01394)0.724:0.0072)0.911:0.01681,387:0.02755)0.761:0.00523,(346:0.01957,(376:0.04072,71:0.0547)0.829:0.0181)0.750:0.00673)0.823:0.01037)0.774:0.0054)0.789:0.005,(((228:0.00529,((401:0.02214,((187:0.00532,411:0.00526)0.801:0.00583,((89:0.027,193:0.00014)0.787:0.00524,91:0.01618)0.743:0.0045)0.548:0.00548)0.825:0.016,40:0.02807)0.778:0.00992)0.824:0.01011,255:0.05012)0.966:0.00014,(352:0.01585,396:0.00014)0.784:0.02134)0.880:0.0107)0.901:0.0194,(35:0.0209,(206:0.00836,291:0.06414)0.439:0.00793)0.753:0.00846)0.763:0.00968)0.942:0.02851,28:0.0208)0.742:0.01057)0.781:0.00811)0.802:0.02029)0.750:0.01578);"""

tutorial_otu_table = """#Full OTU Counts
#OTU ID	PC.354	PC.355	PC.356	PC.481	PC.593	PC.607	PC.634	PC.635	PC.636	Consensus Lineage
0	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
1	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
2	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Porphyromonadaceae;Parabacteroides
3	2	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
4	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
5	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
6	0	0	0	0	0	0	0	1	0	Root;Bacteria;Actinobacteria;Actinobacteria
7	0	0	2	0	0	0	0	0	2	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
8	1	1	0	2	4	0	0	0	0	Root;Bacteria;Firmicutes;"Bacilli";"Lactobacillales";Lactobacillaceae;Lactobacillus
9	0	0	2	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
10	0	1	0	0	0	0	0	0	0	Root;Bacteria
11	0	0	0	0	0	0	1	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
12	0	0	0	0	0	0	1	0	0	Root;Bacteria;Bacteroidetes
13	1	0	0	1	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
14	0	0	1	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
15	0	0	0	0	1	0	0	0	0	Root;Bacteria
16	1	0	2	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
17	0	0	0	1	0	0	4	10	37	Root;Bacteria;Bacteroidetes
18	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
19	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes
20	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
21	0	0	0	0	0	0	2	3	2	Root;Bacteria;Bacteroidetes
22	0	0	0	0	2	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
23	14	1	14	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Bacilli";"Lactobacillales";Lactobacillaceae;Lactobacillus
24	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
25	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
26	0	0	0	0	0	0	0	1	1	Root;Bacteria;Bacteroidetes
27	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes
28	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
29	6	0	4	0	2	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
30	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes
31	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
32	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
33	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
34	0	0	0	0	0	0	8	10	2	Root;Bacteria
35	1	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
36	1	0	1	0	0	0	0	1	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
37	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
38	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
39	0	0	0	0	0	0	0	1	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
40	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
41	0	0	1	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
42	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes
43	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
44	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
45	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Erysipelotrichi";"Erysipelotrichales";Erysipelotrichaceae;Coprobacillus
46	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes
47	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
48	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
49	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
50	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
51	0	1	0	0	0	0	0	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
52	0	2	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
53	0	0	0	0	0	0	2	0	1	Root;Bacteria;Proteobacteria;Deltaproteobacteria
54	0	0	0	0	0	0	5	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Porphyromonadaceae;Parabacteroides
55	0	0	0	0	0	0	1	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
56	0	0	0	0	0	1	0	0	0	Root;Bacteria;Bacteroidetes
57	0	0	0	0	0	0	0	1	0	Root;Bacteria
58	1	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
59	0	0	0	0	0	0	0	0	1	Root;Bacteria;Deferribacteres;Deferribacteres;Deferribacterales;Deferribacteraceae;Mucispirillum
60	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
61	0	0	1	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
62	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
63	1	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
64	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
65	0	0	0	6	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
66	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
67	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
68	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
69	0	0	1	0	0	0	0	0	0	Root;Bacteria
70	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
71	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
72	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
73	0	0	0	0	0	5	0	0	0	Root;Bacteria;Bacteroidetes
74	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
75	1	0	1	0	0	0	0	0	0	Root;Bacteria;Bacteroidetes
76	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
77	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
78	1	0	1	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
79	2	3	8	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
80	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Porphyromonadaceae;Parabacteroides
81	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
82	0	0	0	0	0	2	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
83	0	0	0	1	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
84	1	0	0	0	0	0	0	2	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae";Ruminococcus
85	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
86	0	0	0	0	0	0	0	1	0	Root;Bacteria
87	0	0	1	0	0	2	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
88	0	0	0	0	0	0	0	1	0	Root;Bacteria
89	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
90	0	0	0	9	0	0	3	0	0	Root;Bacteria;Firmicutes;"Erysipelotrichi";"Erysipelotrichales";Erysipelotrichaceae;Turicibacter
91	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";Butyrivibrio
92	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
93	0	0	0	0	0	0	2	1	0	Root;Bacteria;Bacteroidetes
94	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
95	0	0	0	2	0	0	0	0	0	Root;Bacteria;Bacteroidetes
96	0	0	0	1	0	1	0	1	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
97	0	0	0	0	0	1	0	0	0	Root;Bacteria
98	0	0	0	0	0	0	0	1	0	Root;Bacteria
99	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
100	0	0	0	1	0	0	0	0	0	Root;Bacteria
101	0	0	0	3	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
102	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
103	0	1	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
104	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
105	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
106	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
107	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
108	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;Incertae Sedis XIII;Anaerovorax
109	0	0	0	1	0	0	1	5	2	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
110	0	0	0	0	0	2	0	0	0	Root;Bacteria;Actinobacteria;Actinobacteria;Coriobacteridae;Coriobacteriales;Coriobacterineae;Coriobacteriaceae;Olsenella
111	0	0	0	0	0	0	1	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
112	0	0	0	0	0	0	1	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
113	0	0	0	0	0	1	0	0	0	Root;Bacteria
114	0	0	0	0	0	1	0	0	0	Root;Bacteria
115	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes
116	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
117	1	0	2	0	0	6	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
118	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
119	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
120	1	3	1	2	1	9	2	4	5	Root;Bacteria;Bacteroidetes
121	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
122	0	0	0	1	0	2	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
123	0	0	0	0	0	0	1	0	0	Root;Bacteria;Actinobacteria;Actinobacteria;Coriobacteridae;Coriobacteriales;Coriobacterineae;Coriobacteriaceae
124	0	0	0	0	0	0	1	0	0	Root;Bacteria;Actinobacteria;Actinobacteria;Coriobacteridae;Coriobacteriales;Coriobacterineae;Coriobacteriaceae
125	0	0	0	0	0	0	1	0	0	Root;Bacteria;Bacteroidetes
126	0	0	2	0	0	0	0	1	0	Root;Bacteria
127	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
128	0	0	0	0	0	0	1	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
129	0	0	0	1	0	0	0	0	0	Root;Bacteria
130	0	0	0	0	5	2	0	0	0	Root;Bacteria;Proteobacteria;Epsilonproteobacteria;Campylobacterales;Helicobacteraceae;Helicobacter
131	0	0	1	3	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
132	0	0	0	0	1	0	0	0	0	Root;Bacteria
133	0	0	1	0	0	0	0	0	0	Root;Bacteria
134	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
135	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
136	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
137	0	0	0	0	0	0	0	1	0	Root;Bacteria
138	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
139	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
140	0	0	0	0	0	0	1	3	0	Root;Bacteria
141	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
142	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
143	0	0	1	0	0	0	0	0	0	Root;Bacteria
144	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
145	0	0	2	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
146	1	0	0	0	2	0	2	0	3	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
147	0	1	0	1	1	0	0	0	3	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
148	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes
149	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
150	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
151	0	0	0	1	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
152	0	0	0	1	0	0	1	2	19	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
153	0	2	1	2	0	0	1	1	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
154	2	18	0	1	0	0	21	4	4	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
155	0	0	0	0	0	5	9	5	3	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
156	0	0	1	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
157	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
158	1	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
159	0	0	0	0	0	0	0	1	1	Root;Bacteria;Bacteroidetes
160	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
161	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
162	0	0	0	0	0	3	5	2	6	Root;Bacteria;Deferribacteres;Deferribacteres;Deferribacterales;Deferribacteraceae;Mucispirillum
163	0	0	0	0	0	0	0	0	1	Root;Bacteria
164	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
165	2	1	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
166	0	0	0	0	0	0	0	1	0	Root;Bacteria
167	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
168	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
169	0	2	0	7	0	0	0	2	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
170	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
171	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
172	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
173	0	0	0	0	0	1	0	0	0	Root;Bacteria
174	1	0	0	0	10	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Peptostreptococcaceae";"Peptostreptococcaceae Incertae Sedis"
175	0	0	0	0	1	0	0	0	0	Root;Bacteria;Bacteroidetes
176	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
177	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia"
178	0	0	0	2	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
179	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
180	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
181	1	4	2	6	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
182	0	0	0	0	0	1	0	0	0	Root;Bacteria
183	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia"
184	0	0	0	1	0	0	3	1	0	Root;Bacteria;Bacteroidetes
185	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
186	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
187	0	1	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
188	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
189	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
190	0	0	0	0	0	0	0	1	0	Root;Bacteria
191	2	1	10	2	24	0	0	1	1	Root;Bacteria
192	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Bacilli";"Lactobacillales";Streptococcaceae;Streptococcus
193	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";Butyrivibrio
194	0	0	2	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae";Acetanaerobacterium
195	0	0	0	0	0	1	0	0	0	Root;Bacteria
196	0	0	0	0	0	1	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
197	0	1	0	0	0	0	0	0	0	Root;Bacteria
198	0	2	0	0	0	1	0	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales
199	0	0	0	0	0	1	1	0	0	Root;Bacteria
200	0	0	0	2	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
201	0	0	0	1	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
202	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
203	0	2	2	4	0	5	1	5	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
204	1	4	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
205	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
206	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
207	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
208	0	2	0	2	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
209	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
210	0	0	0	0	0	0	0	0	1	Root;Bacteria
211	1	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
212	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
213	0	0	0	0	0	0	0	2	0	Root;Bacteria;Firmicutes
214	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
215	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
216	0	0	0	0	0	0	0	1	0	Root;Bacteria;Bacteroidetes
217	0	0	0	0	0	2	0	1	0	Root;Bacteria
218	0	0	0	0	9	1	0	0	0	Root;Bacteria;Bacteroidetes
219	0	0	0	0	1	0	0	0	0	Root;Bacteria
220	1	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
221	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes
222	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
223	0	0	0	0	0	0	0	2	2	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
224	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
225	0	2	1	0	0	0	0	0	0	Root;Bacteria;Bacteroidetes
226	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
227	0	1	2	0	9	1	1	1	3	Root;Bacteria;Bacteroidetes
228	16	0	0	0	12	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
229	0	0	0	0	0	1	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;Incertae Sedis XIII
230	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
231	0	19	2	0	2	0	3	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
232	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
233	0	0	0	0	1	0	0	0	0	Root;Bacteria;Bacteroidetes
234	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Bacilli";"Lactobacillales";Lactobacillaceae;Lactobacillus
235	0	1	1	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
236	0	0	0	0	0	2	0	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales
237	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
238	0	0	0	0	0	0	0	1	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
239	0	0	0	0	0	1	0	0	0	Root;Bacteria
240	0	0	0	0	0	1	0	0	0	Root;Bacteria
241	0	0	0	0	0	0	2	0	0	Root;Bacteria;TM7;TM7_genera_incertae_sedis
242	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
243	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
244	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes
245	0	0	0	1	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
246	0	0	0	0	0	0	0	1	0	Root;Bacteria
247	0	0	1	0	0	0	0	0	0	Root;Bacteria;Bacteroidetes
248	1	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Bacilli";"Lactobacillales";Lactobacillaceae;Lactobacillus
249	1	0	0	0	0	0	0	0	0	Root;Bacteria
250	1	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
251	0	0	0	1	4	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
252	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
253	0	0	0	0	2	0	0	5	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
254	11	13	6	13	2	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
255	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
256	0	0	0	0	0	0	1	0	0	Root;Bacteria
257	0	0	0	0	0	0	5	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
258	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
259	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
260	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
261	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
262	0	1	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";Bryantella
263	0	0	0	0	1	0	0	0	0	Root;Bacteria
264	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
265	0	0	0	0	0	2	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
266	0	0	0	2	0	0	0	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae;Alistipes
267	1	0	0	5	17	20	0	0	0	Root;Bacteria
268	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
269	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
270	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
271	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
272	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
273	0	0	0	0	0	0	1	0	0	Root;Bacteria
274	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
275	0	0	0	0	0	0	1	0	0	Root;Bacteria;Verrucomicrobia;Verrucomicrobiae;Verrucomicrobiales;Verrucomicrobiaceae;Akkermansia
276	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
277	1	0	0	0	0	0	0	0	0	Root;Bacteria
278	0	0	0	0	0	1	0	0	0	Root;Bacteria
279	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
280	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
281	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
282	0	0	0	0	0	0	2	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Porphyromonadaceae;Parabacteroides
283	0	0	0	0	0	0	2	1	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
284	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
285	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
286	0	2	3	1	4	0	5	0	4	Root;Bacteria;Bacteroidetes
287	0	0	0	0	0	0	1	1	1	Root;Bacteria;Bacteroidetes
288	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
289	0	0	0	0	3	0	0	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
290	0	0	0	0	0	0	0	0	2	Root;Bacteria;Firmicutes;"Bacilli";Bacillales;"Staphylococcaceae";Staphylococcus
291	0	0	0	0	1	0	0	0	0	Root;Bacteria
292	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
293	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
294	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
295	29	1	10	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
296	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
297	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
298	0	0	0	0	0	0	1	0	0	Root;Bacteria;Actinobacteria;Actinobacteria
299	0	0	0	0	0	0	1	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
300	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia"
301	0	0	0	0	0	0	2	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
302	0	0	0	0	0	1	0	0	0	Root;Bacteria
303	0	0	0	0	0	0	0	0	1	Root;Bacteria
304	0	0	0	0	0	0	0	1	0	Root;Bacteria;Bacteroidetes
305	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
306	0	0	0	0	0	0	0	0	1	Root;Bacteria
307	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
308	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae";"Ruminococcaceae Incertae Sedis"
309	0	0	0	1	0	0	0	0	0	Root;Bacteria;Actinobacteria;Actinobacteria;Coriobacteridae;Coriobacteriales;Coriobacterineae;Coriobacteriaceae;Denitrobacterium
310	0	0	1	0	0	0	0	0	0	Root;Bacteria
311	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
312	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
313	0	1	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Porphyromonadaceae;Parabacteroides
314	0	0	1	0	0	0	0	0	0	Root;Bacteria;Bacteroidetes
315	1	3	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
316	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
317	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
318	0	0	0	0	0	1	0	0	0	Root;Bacteria;Proteobacteria
319	0	2	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
320	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
321	0	0	0	0	0	0	0	0	1	Root;Bacteria
322	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
323	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
324	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
325	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
326	0	0	0	0	4	0	0	0	2	Root;Bacteria;Firmicutes;"Erysipelotrichi";"Erysipelotrichales";Erysipelotrichaceae;Erysipelotrichaceae Incertae Sedis
327	0	0	0	0	0	0	0	1	0	Root;Bacteria;Bacteroidetes
328	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
329	2	2	0	1	0	0	0	0	0	Root;Bacteria;Bacteroidetes
330	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes
331	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes
332	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
333	0	0	0	0	0	6	0	3	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
334	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
335	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
336	0	0	1	0	0	0	0	0	0	Root;Bacteria
337	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
338	0	0	0	0	0	0	0	1	0	Root;Bacteria
339	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
340	0	0	2	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
341	0	0	1	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
342	0	0	0	0	0	1	0	0	0	Root;Bacteria
343	0	0	0	0	0	0	0	0	1	Root;Bacteria;Actinobacteria;Actinobacteria;Coriobacteridae;Coriobacteriales;Coriobacterineae;Coriobacteriaceae
344	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
345	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
346	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
347	0	0	0	1	0	0	0	0	0	Root;Bacteria
348	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
349	0	0	0	0	0	0	1	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
350	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
351	0	0	0	0	2	2	1	4	1	Root;Bacteria;Bacteroidetes
352	3	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
353	0	4	4	0	1	2	0	2	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
354	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
355	0	0	0	0	0	0	0	1	0	Root;Bacteria
356	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
357	0	0	0	4	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
358	0	0	1	0	0	0	0	0	0	Root;Bacteria
359	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
360	0	0	1	0	0	0	0	1	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
361	2	0	2	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
362	1	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
363	0	0	0	0	0	1	0	1	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Rikenellaceae
364	1	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
365	0	0	0	0	0	2	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
366	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";Roseburia
367	0	0	0	0	1	0	0	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
368	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
369	0	0	0	0	0	1	0	0	0	Root;Bacteria
370	2	1	0	5	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
371	1	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
372	0	1	0	0	0	0	0	0	0	Root;Bacteria
373	0	1	0	0	0	0	3	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;Clostridiaceae;"Clostridiaceae 1";Clostridium
374	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
375	0	0	0	0	0	0	4	0	0	Root;Bacteria;Firmicutes;"Erysipelotrichi";"Erysipelotrichales";Erysipelotrichaceae;Erysipelotrichaceae Incertae Sedis
376	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
377	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
378	0	0	0	0	0	0	0	0	1	Root;Bacteria;Bacteroidetes
379	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Ruminococcaceae"
380	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Bacilli";Bacillales;"Staphylococcaceae";Staphylococcus
381	0	0	2	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
382	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
383	4	9	0	2	0	0	0	2	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
384	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
385	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Bacilli";"Lactobacillales";"Carnobacteriaceae";"Carnobacteriaceae 1"
386	0	0	1	0	0	0	0	0	0	Root;Bacteria
387	0	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
388	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
389	0	1	0	0	0	0	0	0	0	Root;Bacteria
390	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
391	0	0	0	0	0	0	0	0	1	Root;Bacteria;Firmicutes
392	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
393	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
394	0	0	1	0	0	0	0	0	0	Root;Bacteria
395	1	1	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
396	2	0	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
397	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
398	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
399	0	0	0	0	0	0	13	0	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Bacteroidaceae;Bacteroides
400	0	0	0	0	0	0	1	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
401	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
402	0	1	0	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
403	0	0	0	0	0	0	0	1	0	Root;Bacteria;Bacteroidetes;Bacteroidetes;Bacteroidales;Prevotellaceae
404	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae";"Lachnospiraceae Incertae Sedis"
405	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
406	0	0	0	0	0	1	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
407	1	0	0	0	0	4	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
408	1	5	3	2	0	0	0	0	1	Root;Bacteria;Bacteroidetes
409	0	0	0	0	0	0	0	1	1	Root;Bacteria;Bacteroidetes
410	0	0	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
411	0	0	0	1	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
412	0	0	0	0	2	0	0	0	0	Root;Bacteria;Bacteroidetes
413	0	0	0	0	0	0	0	1	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales
414	1	0	1	0	0	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales;"Lachnospiraceae"
415	0	0	0	0	0	7	0	2	2	Root;Bacteria;Bacteroidetes
416	0	1	0	0	1	0	0	0	0	Root;Bacteria;Firmicutes;"Clostridia";Clostridiales"""

unifrac_dmtx = """	PC.354	PC.355	PC.356	PC.481	PC.593	PC.607	PC.634	PC.635	PC.636
PC.354	0	0.623372333874	0.637204555763	0.596124964451	0.59620370837	0.736037238732	0.790494137353	0.70551354446	0.758236466255
PC.355	0.623372333874	0	0.613486299912	0.631198829777	0.672502144191	0.766682588739	0.732473435165	0.738434734323	0.687363838411
PC.356	0.637204555763	0.613486299912	0	0.691902878696	0.742922266508	0.740684448418	0.78590671873	0.723519724117	0.754301399992
PC.481	0.596124964451	0.631198829777	0.691902878696	0	0.668731241507	0.690956885033	0.652257003318	0.652311676944	0.661223123598
PC.593	0.59620370837	0.672502144191	0.742922266508	0.668731241507	0	0.74048244192	0.760255173866	0.761741286862	0.751777287666
PC.607	0.736037238732	0.766682588739	0.740684448418	0.690956885033	0.74048244192	0	0.73093223871	0.688768250911	0.729047763792
PC.634	0.790494137353	0.732473435165	0.78590671873	0.652257003318	0.760255173866	0.73093223871	0	0.583577161959	0.596033900207
PC.635	0.70551354446	0.738434734323	0.723519724117	0.652311676944	0.761741286862	0.688768250911	0.583577161959	0	0.6271222318
PC.636	0.758236466255	0.687363838411	0.754301399992	0.661223123598	0.751777287666	0.729047763792	0.596033900207	0.6271222318	0"""

big_tree = """((((((((836:0.06877,((549322:0.00892,522457:0.01408)1.000:0.09706,314761:0.09977)0.161:0.01566)0.882:0.00924,(((311539:0.04844,(((174835:0.01627,(34207:0.00082,45996:0.00334)0.863:0.00433)1.000.2:0.07429,((166057:0.00014,18084:0.00412)0.999:0.02776,51122:0.01515)1.000.3:0.09792)1.000.4:0.04652,(((((945:0.08077,(178877:0.01342,(29928:0.00726,35548:0.00187)0.748:0.01216)1.000.5:0.05924)0.975:0.01729,(137114:0.02299,(953:0.02284,(954:0.0119,171811:0.01395)0.291:0.01815)0.728:0.00684)0.999.2:0.02126)0.466:0.00503,(((568672:0.00896,(113094:0.01292,(276812:0.07699,259487:0.00588)0.961:0.00776)1.000.6:0.03018)0.959:0.01239,(19329:0.03945,((978:0.02293,165837:0.00202)0.645:0.00616,(172850:0.01382,111758:0.01417)0.963:0.00682)1.000.7:0.02943)0.825:0.0135)0.999.3:0.03212,(522630:0.01587,(167501:0.02286,(522485:0.01786,564980:0.033)0.895:0.00374)0.728.2:0.00798)1.000.8:0.0236)0.642:0.00704)1.000.9:0.04555,(((((913:0.00704,(911:0.01827,141263:0.01656)0.647:0.00124)0.883:0.01116,((72647:0.01021,(157150:0.00891,200231:0.03101)0.704:0.00183)0.958:0.00701,201472:0.0153)0.987:0.01376)0.953:0.01071,((177017:0.01029,78319:0.00771)1.000.10:0.06246,((910:0.00851,179370:0.01235)0.975.2:0.0106,195366:0.01217)0.965:0.01308)0.987.2:0.01741)0.983:0.01163,((253612:0.02025,921:0.02319)1.000.11:0.02765,(878:0.01141,186470:0.00936)0.989:0.00936)0.317:0.00383)0.206:0.003,(((151069:0.01311,((318531:0.009,144552:0.01694)0.999.4:0.01608,(279228:0.01056,143586:0.00345)0.409:0.00278)0.921:0.0069)0.993:0.01325,((144945:0.00466,141098:0.00155)0.467:0.01215,(143597:0.0242,141374:0.00512)0.857:0.00624)0.989.2:0.01462)0.909:0.00535,(936:0.02603,((923:0.01698,930:0.01356)0.961.2:0.00818,(533888:0.02476,916:0.01687)0.701:0.00462)0.197:0.00345)0.998:0.0124)0.865:0.00286)0.390:0.0043)0.771:0.00458,(223945:0.01911,111652:0.02053)1.000.12:0.05292)1.000.13:0.02401)0.975.3:0.01453)0.385:0.00175,(((212332:0.00401,73046:0.02081)1.000.14:0.02867,(584409:0.00652,16774:0.01061)1.000.15:0.02971)0.855:0.01497,(877:0.05319,(((((((148674:0.01916,(193936:0.00279,188346:0.03099)0.788:0.02078)0.999.5:0.02084,197113:0.04741)0.814:0.00014,18315:0.0076)0.866:0.00281,143713:0.00014)0.750:0.002,174652:0.01312)0.331:0.00661,(872:0.01639,178974:0.02045)1.000.16:0.03717)0.962:0.01027,203067:0.03469)0.314:0.00574)1.000.17:0.04322)0.964:0.01247)0.859:0.0039,179805:0.04611)0.958.2:0.01034)0.995:0.01869,(((919:0.07582,((144807:0.00015,838:0.00586)1.000.18:0.06395,((321466:0.04613,(591374:0.01813,137080:0.05603)0.961.3:0.01204)0.418:0.00629,((245328:0.02527,(278899:0.02811,143638:0.01723)0.316:0.00942)0.395:0.00702,(278219:0.02423,183128:0.02224)0.908:0.01079)0.997:0.02739)0.996:0.03456)1.000.19:0.05099)0.987.3:0.02192,((141677:0.053,((214838:0.02316,(201971:0.0119,(206806:0.01086,202234:0.02124)0.075:0.00371)0.957:0.01327)1.000.20:0.05304,(141995:0.01773,127101:0.00924)1.000.21:0.06994)0.403:0.01742)0.995.2:0.02401,(((165292:0.02435,199556:0.02102)0.940:0.0256,((173183:0.025,((163550:0.00287,(195473:0.00062,43314:0.0121)1.000.22:0.02353)1.000.23:0.05336,53714:0.00574)0.748.2:0.03251)0.773:0.01735,173953:0.00669)1.000.24:0.05171)0.998.2:0.0351,((127052:0.02842,(185056:0.0323,(244423:0.08976,((((581072:0.01145,552713:0.01131)0.908.2:0.00462,(115168:0.01781,586076:0.02113)0.525:0.00271)0.999.6:0.02484,(((253596:0.03165,241157:0.01498)1.000.25:0.03465,((143239:0.01921,244960:0.01184)1.000.26:0.05088,(255340:0.03307,(144887:0.01909,141403:0.02695)0.984:0.02489)1.000.27:0.03958)0.988:0.02553)0.698:0.00495,(319869:0.03238,(341277:0.01383,108942:0.01734)0.182:0.01595)1.000.28:0.06809)0.374:0.00693)0.611:0.00841,1006:0.08302)0.431:0.00941)0.766:0.01097)0.019:0.00915)0.956:0.01457,((133478:0.01171,264893:0.0273)0.987.4:0.01417,((580782:0.02543,(554809:0.01675,581017:0.01203)0.286:0.01139)0.943:0.02194,(((((242513:0.01463,(216453:0.00905,215305:0.02723)0.979:0.01631)0.941:0.00873,(46479:0.04779,30678:0.02913)0.972:0.01211)1.000.29:0.03528,(138353:0.00654,(137954:0.00343,(((100621:0.01623,31165:0.01567)0.949:0.00703,(439385:0.00721,30405:0.02667)0.715:0.0046)1.000.30:0.02781,(196754:0.01628,(((1126:0.02392,46043:0.01693)0.999.7:0.02444,((((((554594:0.01252,547138:0.01931)0.879:0.00748,574960:0.01568)0.703:0.00108,(570918:0.02151,511162:0.01497)0.924:0.00583)0.985:0.00611,((210741:0.01692,550229:0.01345)0.962.2:0.00665,140284:0.01307)0.925:0.00407)0.223:0.00806,(534777:0.0293,(1051:0.01663,573575:0.01305)0.704.2:0.00957)0.745:0.01307)0.856:0.00452,193496:0.01163)0.985.2:0.00953)0.381:0.00663,((((509765:0.02684,(540305:0.01675,(565796:0.02234,195741:0.00835)0.961.4:0.0048)0.577:0.00564)0.997.2:0.00848,511243:0.00558)0.635:0.00663,(349600:0.01855,545027:0.03114)0.586:0.02447)0.845:0.00859,((568829:0.03616,324008:0.0179)0.981:0.0138,(((185612:0.02018,(195292:0.00378,321018:0.02949)0.393:0.01062)0.898:0.00922,(549041:0.04274,((36985:0.01173,153762:0.01427)1.000.31:0.02855,(155789:0.01214,(155684:0.00963,(155495:0.01501,1029:0.01702)0.870:0.00613)0.971:0.0057)0.409.2:0.00161)0.755:0.00089)0.435:0.00523)0.898.2:0.00457,((((((323689:0.02448,235839:0.02747)0.683:0.01179,247303:0.02925)0.950:0.00242,((152483:0.01886,155526:0.02428)0.715.2:0.00425,(197473:0.0107,513153:0.05726)0.815:0.00899)0.386:0.00168)0.957.2:0.00451,(((1098:0.01269,(348549:0.02299,559394:0.01903)0.950.2:0.0058)0.355:0.00266,((1105:0.03473,(347857:0.03029,((312409:0.01194,108964:0.02814)0.937:0.01798,147873:0.00845)0.745.2:0.00253)0.248:0.00104)0.880:0.00965,(108923:0.02414,(109253:0.01813,(347913:0.02147,108961:0.0152)0.080:0.00317)0.938:0.00542)0.886:0.00552)1.000.32:0.02212)1.000.33:0.01836,((510828:0.01878,573586:0.01688)0.886.2:0.00508,(568724:0.01535,540244:0.01706)0.945:0.00583)0.790:0.00288)0.963.2:0.00611)0.980:0.00679,195377:0.02052)0.203:0.00435,513941:0.01861)0.995.3:0.01325)0.695:0.00429)0.302:0.0029)0.959.2:0.00866)0.984.2:0.01161)0.982:0.01723)0.996.2:0.01509)0.999.8:0.03008)0.984.3:0.03607)1.000.34:0.05007,((((546313:0.02661,580255:0.01797)0.756:0.00086,564501:0.01183)0.913:0.00332,(54107:0.02232,245272:0.01095)0.744:0.01019)0.994:0.00978,((((71074:0.01581,(252853:0.01956,(100679:0.01026,540349:0.01713)0.969:0.01024)0.795:0.00469)0.966:0.00913,(1140:0.01498,(101071:0.02534,(1143:0.00935,142373:0.00944)0.964.2:0.00844)0.999.9:0.01696)0.809:0.00698)0.869:0.00272,((1141:0.01651,248490:0.01829)0.972.2:0.00753,(1144:0.00998,512943:0.01434)0.952:0.00719)0.254:0.00178)0.703.2:0.00266,(236566:0.0152,(553459:0.0008,141417:0.00453)0.947:0.01522)0.890:0.00524)0.117:0.00014)0.999.10:0.0359)0.985.3:0.02917,(583985:0.0172,204290:0.02082)0.988.2:0.02322)0.872:0.02416)0.982.2:0.02353)1.000.35:0.03014)0.971.2:0.01558)0.989.3:0.01948)0.924.2:0.00818)0.714:0.00851,((269121:0.0978,(395774:0.01713,(530020:0.00873,142115:0.00756)0.058:0.00619)1.000.36:0.12141)0.959.3:0.02251,(((73608:0.00641,(177633:0.01346,142964:0.00909)0.592:0.00363)1.000.37:0.04759,((126212:0.00992,144071:0.01872)0.987.5:0.01622,((((((122375:0.0043,178858:0.0191)0.964.3:0.01125,(140252:0.01356,140627:0.01086)0.974:0.011)0.999.11:0.02051,((253014:0.01544,201133:0.00944)0.781:0.00551,(126348:0.00931,(542849:0.00906,((322321:0.01513,((255080:0.00467,(462844:0.01358,256174:0.02035)0.877:0.01628)0.966.2:0.005,444679:0.02013)0.933:0.00466)0.878:0.00316,((545878:0.01946,447740:0.00588)0.904:0.00279,540290:0.02478)0.774:0.00376)0.943.2:0.00578)1.000.38:0.02458)0.996.3:0.01359)0.971.3:0.01178)0.621:0.00749,(((535216:0.02964,(566583:0.0112,138956:0.03748)0.802:0.00938)1.000.39:0.02998,(542607:0.00803,119614:0.02638)0.984.4:0.0136)0.479:0.00974,(535188:0.00742,((562837:0.01138,588710:0.00589)0.885:0.00449,(523571:0.02185,(545095:0.0038,579781:0.02495)0.320:0.00479)0.498:0.00295)0.910:0.00523)1.000.40:0.08074)0.969.2:0.01163)0.512:0.00185,(512931:0.01274,308277:0.01411)1.000.41:0.02386)0.904.2:0.00743,((105679:0.03377,(107293:0.02073,105066:0.02998)1.000.42:0.04794)0.938.2:0.01519,((105531:0.02812,((346497:0.0032,144063:0.00537)0.982.3:0.00898,336090:0.02605)0.741:0.00368)0.996.4:0.01774,(((((515962:0.00182,253051:0.00092)0.990:0.01241,525569:0.02534)0.978:0.01051,((526250:0.00367,238793:0.00175)0.991:0.00886,(538819:0.02819,(519027:0.00367,235267:0.00452)0.753:0.01179)0.522:0.00358)0.979.2:0.00999)0.972.3:0.01099,((144537:0.01582,529521:0.01031)0.895.2:0.00643,((550657:0.01671,(594409:0.00471,202343:0.00459)0.999.12:0.01874)0.980.2:0.01345,(455926:0.01362,((246371:0.00815,43667:0.0089)0.998.3:0.01703,((282132:0.01399,248699:0.00761)0.957.3:0.00519,(245697:0.00742,((283549:0.01358,((425657:0.01691,(463052:0.01532,(435275:0.005,463639:0.03563)1.000.43:0.00424)0.817:0.00518)0.996.5:0.01674,284901:0.01862)0.308:0.00444)0.139:0.00391,(462098:0.03211,575865:0.01033)0.911:0.00582)0.962.3:0.00985)0.794:0.00336)0.958.3:0.00793)0.957.4:0.0081)0.936:0.00509)0.924.3:0.00587)1.000.44:0.01715)0.999.13:0.01641,(109236:0.01534,(107535:0.03048,539714:0.01193)0.992:0.01546)0.986:0.01201)0.993.2:0.01584)0.960:0.01309)0.634:0.01229)0.798:0.01219)0.752:0.01194)0.423:0.00826,(330020:0.02525,837:0.01274)0.999.14:0.02538)0.448:0.00433)0.980.3:0.01421)0.943.3:0.0107)0.764:0.01661,((533567:0.00699,(141876:0.01128,883:0.01564)0.500:0.00617)1.000.45:0.13671,((588213:0.02256,(159805:0.00029,144656:0.00204)0.936.2:0.01792)1.000.46:0.0343,(266718:0.02617,((258104:0.00267,142154:0.00275)0.439:0.01139,(141171:0.01714,142245:0.00304)0.912:0.01947)0.520:0.01487)0.945.2:0.02107)1.000.47:0.07508)0.997.3:0.03543)0.957.5:0.013,((((333274:0.03808,((((383277:0.01316,(104332:0.00855,203260:0.01485)0.965.2:0.00923)0.897:0.00294,(237:0.00746,((137840:0.01637,200103:0.01404)0.740:0.00103,200619:0.02167)0.884:0.00349)0.950.3:0.00593)0.654:0.00013,202265:0.02348)0.850:0.00291,((347380:0.01245,568129:0.01636)0.242:0.00776,539535:0.00964)0.731:0.00152)0.803:0.00759)1.000.48:0.069,((((282184:0.03032,16284:0.03681)0.989.4:0.01632,150361:0.02339)0.073:0.01012,348720:0.00637)1.000.49:0.077,(528115:0.0417,411190:0.04773)0.222:0.00687)0.559:0.0157)1.000.50:0.09966,(560816:0.19618,574071:0.31586)0.959.4:0.04422)0.618:0.02521,772:0.07833)1.000.51:0.04801)1.000.52:0.03846,((((((((256597:0.01878,(541455:0.01772,(140276:0.00693,(114830:0.00092,114763:0.00232)0.986.2:0.02571)0.917:0.00584)0.988.3:0.01118)0.980.4:0.02174,(242281:0.04131,((((((651:0.0237,649:0.01379)1.000.53:0.04617,(534667:0.02328,((533680:0.00014,149425:0.00273)0.932:0.00344,147807:0.03099)0.961.5:0.01984)0.999.15:0.02497)0.860:0.00674,((254043:0.01272,27020:0.01722)0.943.4:0.01157,(685:0.02403,(17995:0.03522,686:0.01763)0.960.2:0.01387)1.000.54:0.03602)0.915:0.00753)0.869.2:0.00644,((((241352:0.02836,156901:0.02106)0.947.2:0.01489,((28138:0.0346,(546023:0.00373,161282:0.0051)0.868:0.02851)0.868.2:0.01486,((((74440:0.01098,115205:0.01179)0.914:0.00577,(52324:0.00276,(173903:0.01415,674:0.00822)0.993.3:0.0132)1.000.55:0.02357)0.945.3:0.00653,34918:0.02638)0.991.2:0.01113,((((216813:0.00764,((146856:0.00526,115326:0.02277)0.969.3:0.00608,((155724:0.00749,106803:0.02256)0.993.4:0.01189,(47016:0.021,559983:0.00929)0.967:0.00988)0.718:0.00463)0.881:0.01018)0.964.4:0.00811,152976:0.00505)0.977:0.00649,(148382:0.0144,(81102:0.01213,166971:0.01326)0.766.2:0.00119)0.302.2:0.00013)0.844:0.00162,((((657:0.01108,544733:0.01477)0.778:0.00351,(236030:0.00674,(560969:0.01441,246508:0.0096)0.874:0.00985)0.890.2:0.00744)0.977.2:0.0091,(((103164:0.01237,103455:0.01104)0.952.2:0.01467,(253039:0.01512,37730:0.01833)0.825.2:0.01121)0.718.2:0.00606,(51196:0.01421,21979:0.01966)0.972.4:0.00583)0.292:0.00399)0.238:0.00015,665:0.01173)0.776:0.00307)0.950.4:0.00727)0.845.2:0.00554)0.717:0.00767)0.995.4:0.02266,(36299:0.03639,27978:0.00989)1.000.56:0.02733)0.263:0.00987,((((337881:0.02457,329769:0.00817)0.762:0.0589,(464512:0.02867,((575042:0.01127,((139004:0.03169,576268:0.01641)0.928:0.0037,((202883:0.02157,201749:0.00964)0.915.2:0.01026,563153:0.01295)0.936.3:0.00507)0.881.2:0.00015)1.000.57:0.02763,(570298:0.02956,136097:0.04147)0.946:0.01635)0.683.2:0.00879)0.262:0.0046)0.494:0.005,(((47968:0.01563,(((350806:0.0386,134188:0.03266)0.991.3:0.02468,169692:0.0003)0.481:0.01457,546382:0.05607)1.000.58:0.02948)0.945.4:0.01056,(591975:0.01783,(545708:0.0086,133970:0.0034)0.684:0.04082)0.461:0.00503)0.929:0.00629,(319392:0.0642,(587671:0.00841,270529:0.0225)0.999.16:0.02046)0.880.2:0.00666)0.973:0.00934)0.993.5:0.01956,((312709:0.01277,591811:0.01403)1.000.59:0.0408,((156718:0.01356,574256:0.00798)1.000.60:0.03022,((((576020:0.01593,344263:0.01895)0.416:0.00685,(327649:0.00411,(340992:0.00876,(339432:0.03926,(((536159:0.00715,548300:0.02589)1.000.61:0.02366,537698:0.00726)1.000.62:0.02386,(((562217:0.01492,156610:0.01836)1.000.63:0.0309,((99537:0.00573,218253:0.03534)0.894:0.01019,(574253:0.00441,349206:0.03151)0.765:0.003)0.886.3:0.01299)0.830:0.01514,550787:0.01001)0.942:0.00807)0.998.4:0.01468)0.810:0.00651)1.000.64:0.01945)1.000.65:0.02317)0.996.6:0.01715,((((199963:0.05732,35867:0.01476)0.882.2:0.00158,(565550:0.00094,151829:0.04353)0.837:0.00236)0.999.17:0.00482,((((138107:0.02726,(320196:0.00655,(341290:0.02079,200303:0.03112)0.368:0.00015)0.989.5:0.00966)0.901:0.00014,220293:0.01569)0.822:0.00079,614:0.00159)0.752.2:0.00539,355910:0.01818)0.981.2:0.01924)0.868.3:0.00412,((190922:0.01539,566259:0.02291)0.890.3:0.00871,((318497:0.02292,192430:0.01571)0.505:0.00148,(112125:0.02443,45762:0.00798)0.967.2:0.00618)0.874.2:0.0028)0.942.2:0.00491)0.481.2:0.00425)0.996.7:0.02263,39652:0.03439)1.000.66:0.02775)0.981.3:0.01559)0.947.3:0.02043)0.950.5:0.01241)0.962.4:0.01281)0.842:0.00812,(38000:0.00737,(23901:0.00603,136425:0.01826)0.998.5:0.01687)1.000.67:0.04647)0.448.2:0.0056,((545129:0.03049,((((316294:0.02047,(325843:0.01004,221900:0.0317)0.418.2:0.00765)0.561:0.00661,((312714:0.02823,(540673:0.01753,(((244042:0.00745,156685:0.01581)0.644:0.00943,(335260:0.01192,304972:0.03869)0.973.2:0.01328)0.988.4:0.01328,245569:0.01398)0.860.2:0.00446)0.993.6:0.01439)0.880.3:0.00622,(536949:0.00632,340382:0.00952)1.000.68:0.02745)0.404:0.00553)0.948:0.01088,(531181:0.02723,247483:0.01777)0.463:0.00601)0.898.3:0.00645,(239547:0.01266,((245441:0.00015,125425:0.03061)1.000.69:0.04681,(311512:0.02219,((((132088:0.02029,702:0.00963)0.909.2:0.00195,698:0.02425)0.855.2:0.00553,313962:0.04193)0.994.2:0.0172,(248065:0.02463,61001:0.00531)0.951:0.0078)0.951.2:0.01215)1.000.70:0.03083)0.997.4:0.02749)0.984.5:0.0194)0.930:0.00953)0.681:0.00697,(85668:0.04933,86270:0.01725)0.999.18:0.02363)0.928.2:0.00916)0.960.3:0.00921)0.440:0.01075)1.000.71:0.12196,((30999:0.03185,((296987:0.01648,(523320:0.00549,582796:0.02054)0.294:0.00251)1.000.72:0.01997,((336982:0.00954,70647:0.01059)0.965.3:0.00973,(572748:0.02738,159880:0.01194)0.991.4:0.01245)0.971.4:0.01117)0.742:0.00664)1.000.73:0.09086,((((137592:0.02133,137909:0.01771)1.000.74:0.02639,516270:0.08432)0.662:0.01241,(((530780:0.01921,(((108821:0.00229,555508:0.00641)0.854:0.00203,187194:0.01948)0.928.3:0.00015,103247:0.01991)0.999.19:0.01639)0.829:0.00776,(187016:0.03769,(102505:0.00804,((86454:0.00949,(272691:0.01365,22948:0.01651)0.933.2:0.00318)0.877.2:0.00155,146195:0.04529)0.775:0.0017)1.000.75:0.02794)0.988.5:0.01789)0.997.5:0.02288,((101882:0.01118,35354:0.02614)1.000.76:0.05185,((((36837:0.0456,(559229:0.01618,516:0.00432)0.251:0.01606)1.000.77:0.0515,((403:0.01209,(125712:0.00423,475:0.00296)0.995.5:0.01782)0.978.2:0.01152,((107074:0.00988,52710:0.01694)0.739:0.00638,((592689:0.04237,367815:0.01842)0.854.2:0.00167,(146358:0.01343,(451189:0.0294,467541:0.02515)0.083:0.01503)0.911.2:0.00209)0.926:0.01161)0.958.4:0.00759)0.977.3:0.01366)0.242.2:0.01005,((((((511207:0.03528,143256:0.01141)0.930.2:0.0074,(509926:0.00554,88818:0.00626)0.881.3:0.04574)0.582:0.00444,(((361:0.00607,160909:0.00879)0.745.3:0.01993,566371:0.01133)0.960.4:0.00741,((559288:0.01198,(262453:0.02841,157003:0.01024)0.696:0.0033)0.951.3:0.00757,343638:0.02905)0.987.6:0.01014)0.979.3:0.01065)0.663:0.00563,333115:0.02961)0.947.4:0.00685,((320939:0.01476,199952:0.01146)0.871:0.00275,(539582:0.00629,(265165:0.01766,211246:0.00529)0.929.2:0.00436)0.975.4:0.00495)0.999.20:0.01083)0.097:0.00271,(189149:0.0034,581916:0.02117)0.921.2:0.01793)0.227:0.00504)1.000.78:0.03331,((193607:0.01152,(154061:0.01144,(143846:0.01052,530720:0.03989)0.872.2:0.00539)0.895.3:0.00393)0.997.6:0.01755,((533814:0.00898,(453468:0.01168,242876:0.00855)0.893:0.0056)1.000.79:0.04132,((563839:0.03457,532287:0.02774)0.992.2:0.01765,(252377:0.01855,68323:0.02475)0.966.3:0.01883)0.994.3:0.02281)0.999.21:0.02713)0.856.2:0.0094)0.792:0.01007)0.973.3:0.01459)0.840:0.00936)0.996.8:0.03091,(((((235941:0.01459,(248477:0.00335,249243:0.02765)0.913.2:0.00478)0.920:0.00442,(254640:0.0131,243000:0.01439)0.702:0.00361)0.976:0.00911,568996:0.00149)0.480:0.02849,(233066:0.02313,(245210:0.02646,251210:0.00333)0.942.3:0.01434)0.962.5:0.01536)1.000.80:0.09094,((261976:0.02159,((263589:0.0193,149591:0.01623)0.856.3:0.00403,((216145:0.00767,108784:0.01846)0.812:0.00396,140425:0.02475)0.739.2:0.00104)0.840.2:0.00201)0.997.7:0.01702,((((99463:0.01097,22804:0.01544)0.947.5:0.00605,(552299:0.01322,(239243:0.001,234172:0.00096)0.838:0.01161)0.939:0.0064)1.000.81:0.02365,(((575365:0.02159,((554894:0.01451,(266149:0.00656,50770:0.02675)0.997.8:0.01317)0.960.5:0.01102,557845:0.01773)0.652:0.00755)0.968:0.01215,513268:0.01334)0.290:0.00453,536937:0.01259)1.000.82:0.02696)0.042:0.00254,(((557121:0.03054,524988:0.00818)0.940.2:0.00947,((164497:0.01251,((259704:0.02571,(264985:0.00353,361178:0.02552)0.816:0.00378)0.941.2:0.008,313193:0.01266)0.486:0.00449)0.859.2:0.00644,((((37119:0.00772,(349670:0.01913,310778:0.01449)0.878.2:0.0127)0.860.3:0.0058,(172057:0.01737,243787:0.01814)0.946.2:0.00946)0.961.6:0.00571,240565:0.01049)1.000.83:0.01663,(236057:0.01376,236619:0.01544)0.563:0.00379)0.816.2:0.00287)0.943.5:0.01003)0.704.3:0.00914,((((554587:0.02157,334489:0.00835)0.845.3:0.00244,(545392:0.00974,544385:0.02322)0.779:0.00499)0.993.7:0.01549,((559987:0.00483,106238:0.00211)0.957.6:0.03768,156913:0.00765)0.951.4:0.01714)1.000.84:0.04386,(535788:0.03127,(551081:0.02205,(565033:0.02752,212362:0.00253)0.362:0.0095)1.000.85:0.05873)1.000.86:0.03973)0.931:0.01976)0.661:0.00758)0.456:0.00355)1.000.87:0.05572)0.747:0.01787)0.996.9:0.02949)0.999.22:0.04577)0.926.2:0.02463,(((527781:0.00838,(139379:0.01283,14:0.01095)0.767:0.00963)1.000.88:0.0322,(((170268:0.00604,(99780:0.00238,104320:0.0075)0.456.2:0.00135)0.759:0.0008,(112414:0.00281,32461:0.01319)0.074:0.01774)1.000.89:0.05477,(47977:0.03065,((111898:0.01897,(111333:0.0082,133209:0.03017)1.000.90:0.04213)1.000.91:0.03165,((213250:0.01978,((58153:0.0262,58154:0.0031)0.806:0.00654,(368917:0.02771,142157:0.02394)0.385.2:0.00537)0.868.4:0.00834)0.792.2:0.00603,(369925:0.01956,209344:0.05121)0.112:0.00529)0.988.6:0.01624)0.925.2:0.01073)1.000.92:0.06698)0.945.5:0.02514)0.996.10:0.03891,((((249133:0.01486,124193:0.01352)0.999.23:0.03599,(342565:0.0354,((148715:0.01201,146124:0.01706)0.979.4:0.00861,((110525:0.01058,109822:0.01234)0.966.4:0.00915,307361:0.01291)0.754:0.00173)0.999.24:0.02323)1.000.93:0.05197)1.000.94:0.04026,(((457105:0.05324,(542257:0.0633,(286992:0.02648,(202209:0.00397,99169:0.00452)0.950.6:0.01156)1.000.95:0.04156)0.460:0.01143)0.968.2:0.01606,((548984:0.01018,(261044:0.02975,(201596:0.00728,158026:0.01697)0.885.2:0.00966)0.931.2:0.01051)1.000.96:0.06655,(((201067:0.00861,286035:0.02646)1.000.97:0.04878,(((62881:0.01864,329648:0.02134)0.998.6:0.02163,(570725:0.01019,(107533:0.00761,(547371:0.01261,140729:0.00557)0.994.4:0.01238)0.845.4:0.00442)0.999.25:0.02813)0.788.2:0.00807,(99429:0.02039,(((153944:0.01422,(269725:0.01169,330865:0.04322)0.999.26:0.02459)1.000.98:0.02817,((76146:0.0223,(136107:0.01212,569573:0.01154)0.996.11:0.01636)0.944:0.0075,((((567801:0.01568,(155920:0.01416,148487:0.01111)0.950.7:0.00565)0.637:0.00286,154238:0.01281)0.754.2:0.00148,(152962:0.02031,((152801:0.00384,(195401:0.01782,155055:0.01916)0.723:0.00902)0.905:0.01378,178088:0.03294)0.977.4:0.0094)0.958.5:0.00692)0.952.3:0.0076,170941:0.01131)0.965.4:0.00849)0.867:0.00603)0.805:0.00359,((155703:0.02539,(284456:0.01818,282634:0.02869)0.767.2:0.00315)0.994.5:0.01159,(153679:0.01994,52339:0.00891)0.894.2:0.00252)0.651:0.01374)1.000.99:0.02678)0.998.7:0.0246)1.000.100:0.03859)0.947.6:0.01531,((((63282:0.00697,63283:0.01907)0.837.2:0.00284,(216435:0.01222,207482:0.02407)0.963.3:0.00665)0.129:0.00267,213630:0.00811)1.000.101:0.10326,130029:0.10236)0.116:0.0118)0.992.3:0.02226)0.316.2:0.0075)0.074.2:0.01162,((((((((546805:0.00613,299208:0.00617)0.999.27:0.01107,(411589:0.00816,202764:0.01059)0.926.3:0.00403)1.000.102:0.01348,(531588:0.01141,(155086:0.02716,(448625:0.01576,(160749:0.00162,299490:0.00198)0.804:0.01551)0.860.4:0.0066)0.555:0.00324)0.750.2:0.00139)0.892:0.00285,(((589211:0.00469,558673:0.00538)1.000.103:0.03716,101828:0.03114)0.976.2:0.01361,((319002:0.02096,219191:0.03147)0.972.5:0.01563,(((410763:0.01244,274250:0.01789)0.977.5:0.01044,412056:0.02451)0.898.4:0.00534,(550661:0.01647,(550655:0.01752,(158701:0.02219,((567877:0.01988,515698:0.00606)0.926.4:0.01061,201690:0.01595)0.842.2:0.0057)0.943.6:0.00421)0.999.28:0.0117)0.639:0.00243)0.931.3:0.00608)0.435.2:0.00723)0.959.5:0.00686)0.684.2:0.00352,((((((558062:0.00362,379145:0.00455)0.868.5:0.00489,464997:0.03832)0.878.3:0.00782,73685:0.02347)0.949.2:0.00862,593427:0.01144)0.736:0.00142,((552392:0.01056,202976:0.01694)0.972.6:0.00621,(265193:0.01801,202462:0.02316)0.979.5:0.00869)0.940.3:0.00422)0.915.3:0.00403,(389053:0.021,384856:0.01015)0.720:0.00274)0.217:0.00101)0.969.4:0.01374,(407348:0.02101,((258365:0.00294,556117:0.02599)0.952.4:0.011,(467265:0.01595,(200343:0.00214,105390:0.0038)0.057:0.00945)0.925.3:0.0092)0.620:0.00893)0.935:0.01093)0.552:0.01059,(267307:0.00998,130088:0.02391)1.000.104:0.02866)1.000.105:0.06022,(421839:0.06549,438290:0.10958)0.941.3:0.03453)0.979.6:0.02254)0.873:0.02167)0.916:0.01532,((250392:0.06018,(148407:0.03997,(((146447:0.00056,311706:0.00683)1.000.106:0.03892,344532:0.03893)1.000.107:0.02886,((104364:0.00343,(250570:0.02181,108962:0.00519)0.316.3:0.00305)0.964.5:0.00811,((308755:0.01127,315098:0.00997)0.982.4:0.00957,((255494:0.0297,(351294:0.00637,(146866:0.01659,155847:0.01748)0.923:0.00884)0.416.2:0.00513)0.291.2:0.00264,715:0.02533)0.674:0.00049)0.977.6:0.00988)1.000.108:0.03441)0.163:0.00972)0.995.6:0.0323)1.000.109:0.08435,(190786:0.01973,(513408:0.01875,559633:0.03534)0.662.2:0.00835)0.995.7:0.02758)0.990.2:0.02933)0.999.29:0.05158)1.000.110:0.05616)0.974.2:0.02194,((105401:0.01195,(570008:0.02165,138592:0.01356)0.942.4:0.01054)1.000.111:0.06228,((267686:0.01052,(32110:0.01287,151209:0.0318)0.998.8:0.01184)0.707:0.07996,(519852:0.02245,138474:0.01843)1.000.112:0.06785)0.661.2:0.02301)1.000.113:0.05777)0.649:0.01274,((((136054:0.05371,(544697:0.02608,131861:0.02272)0.986.3:0.019)1.000.114:0.08848,(140100:0.07564,(140297:0.03524,(566419:0.00671,138385:0.0196)1.000.115:0.04312)0.950.8:0.02132)0.998.9:0.04046)1.000.116:0.04181,((589892:0.01316,253993:0.01175)1.000.117:0.09113,((124675:0.06585,((265152:0.04799,((245:0.01815,124738:0.00105)0.973.4:0.06389,(119938:0.02515,((245107:0.02392,245708:0.00907)0.532:0.00474,((124491:0.01608,(543071:0.02266,249:0.01514)0.834:0.00466)0.940.4:0.00634,252384:0.01096)0.941.4:0.02013)0.955:0.01031)1.000.118:0.04769)0.999.30:0.02764)0.521:0.01497,(101987:0.01771,(241739:0.0181,242069:0.02081)0.395.2:0.00558)1.000.119:0.04952)0.514:0.01782)0.989.6:0.02846,188272:0.03876)0.996.12:0.03167)0.961.7:0.02434)0.934:0.01013,(46:0.01913,(67147:0.01731,(((((((((105600:0.00982,(40123:0.00967,79:0.02156)0.875:0.00324)0.983.2:0.00764,(218697:0.00733,18599:0.01862)0.968.3:0.00762)0.573:0.00351,(84255:0.00365,77878:0.00262)0.918:0.0121)0.755.2:0.00341,(101553:0.00287,565906:0.00177)0.984.6:0.00846)0.979.7:0.00789,(249915:0.00495,(204:0.01559,198:0.01618)0.999.31:0.01116)0.952.5:0.01453)0.931.4:0.01534,99797:0.00088)0.919:0.00547,170726:0.0041)0.994.6:0.01351,(139922:0.02661,(((230854:0.00597,(((((323003:0.03386,(234409:0.00979,(353258:0.00147,(((150477:0.01228,155634:0.00912)0.895.4:0.00272,((186546:0.01352,589886:0.01308)0.332:0.00257,310429:0.01352)0.734:0.00105)0.997.9:0.01223,312333:0.03411)0.986.4:0.02128)0.962.6:0.01178)0.983.3:0.01263)0.992.4:0.01652,((160868:0.01464,(103975:0.01144,221937:0.01632)0.583:0.00721)0.998.10:0.01775,(((138:0.02089,(150:0.02083,139:0.02056)0.876:0.00461)0.418.3:0.00218,(24414:0.00014,547463:0.00255)0.762.2:0.01096)0.982.5:0.00837,(105197:0.03074,(167356:0.00903,22694:0.00978)0.331.2:0.00337)0.985.4:0.0125)0.375:0.00497)0.738:0.00616)0.760:0.00196,((104130:0.00583,319390:0.02084)0.193:0.00141,(333327:0.02243,303948:0.0153)0.981.4:0.00587)0.986.5:0.00955)0.740.2:0.00141,(((470690:0.00556,(((307876:0.01706,104804:0.02752)0.000:0.00015,152450:0.01188)0.794.2:0.00194,124:0.00664)0.989.7:0.01005)0.959.6:0.0045,(128:0.01404,86:0.00925)0.948.2:0.00668)0.980.5:0.00897,(228569:0.02347,107657:0.00766)0.967.3:0.00636)0.987.7:0.01351)0.892.2:0.00014,(100449:0.00667,262662:0.02273)0.969.5:0.01375)0.687:0.02595)0.983.4:0.02078,246686:0.0048)0.988.7:0.01221,166443:0.01329)0.899:0.00564)0.946.3:0.00739)1.000.120:0.01797,227:0.01179)0.994.7:0.01586)1.000.121:0.0543)1.000.122:0.05034)0.674.2:0.00989)0.978.3:0.01572,((584600:0.02073,(142640:0.01323,(20:0.00787,(589434:0.01838,157276:0.00745)0.210:0.00328)0.903:0.00675)0.817.2:0.00792)0.993.8:0.01606,(552640:0.00787,(176683:0.0153,((255315:0.00981,(246445:0.01693,240377:0.02883)0.096:0.00618)1.000.123:0.02428,(253591:0.02355,538763:0.01131)0.969.6:0.01327)0.671:0.00801)1.000.124:0.04181)1.000.125:0.02927)1.000.126:0.0479)0.734.2:0.00862,((((514441:0.0178,((109726:0.00899,(202400:0.02364,103714:0.00122)0.009:0.00064)0.942.5:0.00246,((807:0.01342,100398:0.01056)0.999.32:0.02424,(811:0.01944,90070:0.00372)0.843:0.00108)0.881.4:0.00311)0.936.4:0.00295)0.803.2:0.00485,137113:0.01548)1.000.127:0.0334,(326628:0.03827,(((50705:0.01447,((310630:0.0076,(73240:0.00014,104760:0.00386)0.970:0.00524)0.987.8:0.00994,(((((144391:0.00276,752:0.00107)0.907:0.00369,((177710:0.01742,108955:0.00935)0.994.8:0.0094,(158889:0.00437,71750:0.01198)0.990.3:0.0093)0.956.2:0.00727)0.992.5:0.01016,557548:0.01593)0.957.7:0.00789,589586:0.00318)0.978.4:0.00762,71588:0.00749)0.942.6:0.00633)0.710:0.00675)0.958.6:0.00811,518252:0.00193)0.463.2:0.00328,71597:0.01167)0.800:0.00794)1.000.128:0.07105)0.696.2:0.01053,(526964:0.08552,135883:0.15309)0.986.6:0.02811)0.636:0.01668)0.896:0.01069)0.858:0.01124,((51:0.22017,(247740:0.00818,773:0.05082)1.000.129:0.09853)0.871.2:0.03696,((200845:0.22054,(536258:0.20873,(269089:0.17103,(108925:0.07377,(569329:0.03102,509222:0.03913)1.000.130:0.05517)1.000.131:0.06622)0.264:0.02961)0.836:0.02882)0.001:0.01585,211317:0.16102)0.740.3:0.02856)0.608:0.01403)1.000.132:0.12261,((((((((((((((((((((((((((((((((((((((((((((((532519:0.00978,(380647:0.01961,458596:0.0315)0.910.2:0.00597)0.996.13:0.01015,107891:0.0079)0.784:0.00812,((93935:0.02485,(99274:0.01153,115376:0.02288)0.766.3:0.00166)0.907.2:0.0048,(561032:0.02235,93749:0.00679)0.683.3:0.00323)0.949.3:0.00853)0.988.8:0.01198,((23815:0.00015,73295:0.00949)0.972.7:0.00988,(171651:0.03331,12727:0.00818)0.955.2:0.00622)1.000.133:0.02051)0.943.7:0.00813,((546210:0.00525,325253:0.00864)0.849:0.00929,526237:0.01581)0.848:0.02439)0.912.2:0.00536,((19216:0.0148,18155:0.01193)0.943.8:0.00791,365921:0.00938)0.774.2:0.00459)1.000.134:0.01371,(170832:0.00633,(545511:0.03646,326457:0.01269)0.874.3:0.00409)0.986.7:0.01001)0.254.2:0.00165,(((312491:0.00592,(526486:0.01895,520365:0.00657)0.883.2:0.00299)0.967.4:0.0047,(378053:0.00579,((12733:0.01268,((164751:0.00857,521725:0.00026)0.968.4:0.00571,174732:0.01677)0.980.6:0.00656)0.493:0.00182,168713:0.00491)0.995.8:0.00772)0.519:0.00016)0.862:0.00351,(543731:0.00239,517605:0.03173)0.977.7:0.00864)0.955.3:0.00691)0.931.5:0.00571,(47148:0.00238,(550057:0.00559,(12736:0.00248,518959:0.00461)1.000.135:0.01185)0.995.9:0.01046)0.991.5:0.00991)0.864:0.00229,(172706:0.02751,279978:0.02672)0.946.4:0.00791)0.929.3:0.0049,(((201506:0.0058,135406:0.01324)0.998.11:0.01565,(200646:0.00944,135424:0.02043)0.955.4:0.00762)0.878.4:0.0051,(592605:0.01852,510617:0.01749)0.989.8:0.0083)0.715.3:0.00053)0.982.6:0.00702,(((167566:0.00426,135409:0.01247)0.884.2:0.00471,(109193:0.01563,(((395168:0.01077,(417333:0.02023,515746:0.0191)0.988.9:0.00976)0.921.3:0.00539,(329785:0.01514,347911:0.01579)0.768:0.00445)0.984.7:0.00834,(154789:0.00955,(511542:0.00619,210018:0.02364)0.992.6:0.01294)0.916.2:0.00456)0.820:0.0022)1.000.136:0.01759)0.749:0.00455,(549375:0.02058,((343580:0.01972,321654:0.00502)0.761:0.00252,(338708:0.00964,(113533:0.00775,12753:0.00015)0.869.3:0.00152)0.908.3:0.00251)0.838.2:0.00313)0.988.10:0.00941)0.982.7:0.00727)0.855.3:0.00222,(((560734:0.02195,(231733:0.00015,311573:0.00252)0.992.7:0.00941)0.760.2:0.00155,(108722:0.01582,108724:0.08556)0.942.7:0.01045)0.743:0.00193,(140338:0.01415,565105:0.02078)0.997.10:0.01038)0.991.6:0.0089)0.938.3:0.0056,(((((249430:0.01285,249330:0.00812)0.868.6:0.00301,(538835:0.01502,(232795:0.00897,145101:0.00513)0.894.3:0.00269)0.875.2:0.00014)0.979.8:0.00479,367851:0.00475)0.713:0.00014,((568735:0.0242,(275357:0.00447,154610:0.01888)0.666:0.00325)0.689:0.00974,315847:0.00884)0.136:0.00227)0.822.2:0.00315,(236025:0.02401,((428663:0.00553,((333655:0.00079,542913:0.00504)0.473:0.00615,(250746:0.01167,165030:0.00893)0.996.14:0.00979)0.299:0.0018)0.942.8:0.00082,12743:0.00898)0.767.3:0.00087)0.301:0.00873)0.950.9:0.00594)0.186:0.00526,165629:0.02585)1.000.137:0.02884,((522198:0.01192,(157893:0.02659,(172332:0.01971,((179479:0.01695,(205147:0.02404,537204:0.04195)0.951.5:0.0032)0.864.2:0.00157,469602:0.01206)0.934.2:0.00959)0.722:0.0054)0.785:0.00733)0.976.3:0.0076,(263308:0.01351,((341901:0.02719,(((237339:0.01397,216616:0.0209)0.346:0.00456,(549341:0.00605,132361:0.01222)0.821:0.00332)0.592.2:0.0045,377237:0.00587)0.995.10:0.0131)0.980.7:0.00975,(((((101707:0.01863,((552729:0.0186,34725:0.01548)0.860.5:0.00221,(((257538:0.0153,203408:0.02097)0.162:0.0078,(205152:0.00258,586825:0.00468)0.879.2:0.00155)0.747.2:0.0008,((((((((255802:0.00691,((98983:0.00095,101669:0.00258)0.891:0.01757,204284:0.01067)0.925.4:0.00555)0.943.9:0.00566,501795:0.00386)0.993.9:0.00881,(((((207298:0.00978,(99987:0.01048,516593:0.00094)0.960.6:0.0051)0.939.2:0.00573,433883:0.01782)0.896.2:0.00415,154684:0.01056)0.926.5:0.00529,(((243943:0.00761,(101852:0.01304,(325277:0.02885,167263:0.01941)0.965.5:0.00973)0.976.4:0.00804)0.857.2:0.00222,104329:0.0093)0.851:0.00155,563475:0.01482)0.910.3:0.00338)0.961.8:0.0051,214797:0.0078)0.941.5:0.00471)0.871.3:0.00266,96649:0.0105)0.914.2:0.00289,(360170:0.01106,((151383:0.00333,544884:0.0254)0.983.5:0.00693,501824:0.01109)0.656:0.00014)0.891.2:0.00219)0.849.2:0.00208,(16538:0.01761,151442:0.01102)0.776.2:0.00141)0.774.3:0.002,((((12763:0.00081,12762:0.00014)0.997.11:0.01171,(559914:0.00384,(222071:0.01372,12783:0.00828)0.916.3:0.00746)0.927:0.00735)0.795.2:0.00183,566859:0.01135)0.912.3:0.00289,(((533951:0.01357,286030:0.01568)0.921.4:0.00693,(138624:0.01577,(156466:0.00674,517399:0.02191)0.511:0.00332)0.834.2:0.00308)0.764.2:0.00202,(250656:0.01483,207728:0.00893)0.992.8:0.01254)0.921.5:0.00424)0.939.3:0.00434)0.926.6:0.00329,((335428:0.00738,(257075:0.00851,271202:0.01699)0.215:0.00093)0.971.5:0.00478,(60763:0.00155,(249247:0.00014,251908:0.00423)1.000.138:0.01077)1.000.139:0.00011)0.917.2:0.00426)0.865.2:0.00644)0.877.3:0.00181)0.232:0.00204)0.883.3:0.00319,208432:0.01427)0.943.10:0.00544,((104813:0.0083,45520:0.0174)0.999.33:0.01431,((((134599:0.01387,202304:0.00348)0.911.3:0.00029,102697:0.00619)0.805.2:0.00015,140932:0.011)0.999.34:0.01489,(((313252:0.01308,69420:0.02266)0.988.11:0.00863,(554391:0.01114,574022:0.00555)0.979.9:0.00828)0.871.4:0.00155,248733:0.00594)0.780:0.00421)0.785.2:0.00869)0.693:0.00364)0.051:0.00249,((166467:0.01898,((205833:0.00645,((217144:0.00521,493260:0.00379)0.967.5:0.00681,414150:0.00605)0.526:0.00315)1.000.140:0.01363,531515:0.00537)0.943.11:0.00331)0.836.2:0.0022,(197085:0.01349,(309743:0.02723,295675:0.02777)0.810.2:0.00398)0.982.8:0.01029)0.633:0.00337)0.978.5:0.00666,((549557:0.01126,12787:0.00944)0.966.5:0.004,(((114316:0.00853,246385:0.00717)0.972.8:0.00518,(137878:0.01063,12785:0.0025)0.956.3:0.0056)0.883.4:0.00156,(((243180:0.00549,455240:0.01426)0.868.7:0.00276,218299:0.01819)0.979.10:0.00706,((553334:0.008,106684:0.01139)0.782:0.00737,32247:0.011)0.691:0.0065)0.874.4:0.00459)0.874.5:0.00159)0.747.3:0.00076)0.945.6:0.00455)0.804.2:0.00312)0.571:0.00114)0.961.9:0.00683)0.969.7:0.00916,(305897:0.014,12801:0.00406)0.907.3:0.01394)0.990.4:0.01292,(411799:0.0222,(525694:0.01249,(340457:0.00931,538164:0.01275)0.979.11:0.00818)0.969.8:0.01064)0.983.6:0.01133)0.983.7:0.01662,(((591315:0.0251,248170:0.01816)1.000.141:0.04299,((((222313:0.02755,(((572745:0.0116,(201495:0.0089,((542225:0.01008,99241:0.01177)0.700:0.00183,(566820:0.01172,((200104:0.0072,((275402:0.0187,321069:0.01593)0.930.3:0.00771,(((272992:0.0112,260834:0.00269)0.999.35:0.0125,571157:0.00402)0.962.7:0.00487,(134809:0.02849,101549:0.01436)0.972.9:0.00815)0.947.7:0.00522)0.504:0.00454)0.956.4:0.00661,549945:0.01874)0.827:0.00513)0.924.4:0.00368)0.333:0.00198)0.949.4:0.00649)0.872.3:0.00425,517376:0.00344)1.000.142:0.02089,534601:0.01673)0.270:0.00317)0.988.12:0.01471,((153946:0.04063,(((587114:0.01242,((203186:0.00587,369490:0.00435)0.739.3:0.00613,138174:0.00677)0.947.8:0.00555)1.000.143:0.03745,(((550320:0.008,(276687:0.00862,272814:0.01021)0.794.3:0.00479)0.867.2:0.00478,((203278:0.00952,(536016:0.00655,546245:0.03253)0.738.2:0.00102)1.000.144:0.0117,(552440:0.00697,(591611:0.01309,((309705:0.00831,(350544:0.00966,(94651:0.00803,(273072:0.00248,273810:0.01847)0.964.6:0.00543)1.000.145:0.00955)0.861:0.00233)0.936.5:0.00834,((((200197:0.00782,((559221:0.00268,557965:0.00356)0.924.5:0.01083,354047:0.00731)0.727:0.0048)0.992.9:0.00887,((356778:0.00923,571148:0.01959)0.204:0.00219,((292798:0.00962,79332:0.00971)0.994.9:0.01006,354609:0.01133)0.899.2:0.01053)0.955.5:0.00651)0.903.2:0.00406,(355144:0.01501,514700:0.01959)0.707.2:0.00175)0.911.4:0.00321,(210359:0.00519,132652:0.01386)0.169:0.00159)1.000.146:0.03155)0.977.8:0.01018)0.968.5:0.00716)0.958.7:0.00587)0.913.3:0.00536)0.697:0.00508,(204467:0.01571,210533:0.03752)0.809.2:0.00564)0.995.11:0.01883)0.976.5:0.00922,((250138:0.0298,(169682:0.00949,172612:0.01105)0.307:0.0018)0.996.15:0.01009,((575324:0.00346,271885:0.02316)1.000.147:0.02286,(154599:0.01369,(525820:0.02066,517129:0.00685)0.977.9:0.01046)0.552.2:0.00493)0.351:0.00099)0.983.8:0.00951)0.905.2:0.00489)0.977.10:0.00935,((((511122:0.01886,(247638:0.01154,255203:0.00794)0.969.9:0.0072)0.979.12:0.00939,315608:0.01921)0.990.5:0.01414,(((247055:0.02943,12812:0.00248)0.813:0.03112,104847:0.00564)0.696.3:0.0031,(332548:0.00701,53290:0.01003)0.795.3:0.00564)0.777:0.1181)0.981.5:0.01398,(555988:0.02955,583854:0.02129)0.950.10:0.00934)0.871.5:0.00514)0.901.2:0.00776)1.000.148:0.03718,((((247286:0.01045,102750:0.03487)0.997.12:0.01862,((564553:0.01067,510433:0.00933)0.998.12:0.01559,569556:0.05298)0.477:0.00692)0.982.9:0.01288,(((94227:0.02597,((251247:0.0208,161544:0.02095)0.999.36:0.04171,(255515:0.0536,(562332:0.01424,94128:0.02689)0.991.7:0.02572)0.184:0.00876)0.919.2:0.01351)0.105:0.00277,(((538728:0.01424,225849:0.01667)0.981.6:0.01092,(562817:0.02615,585514:0.02355)0.993.10:0.01218)0.576:0.00331,((((((((328654:0.00854,512499:0.01667)0.806.2:0.00303,((160075:0.03091,(457819:0.01562,(210507:0.0137,(219366:0.01397,(511430:0.01215,543676:0.00784)0.983.9:0.00963)0.717.2:0.00309)0.917.3:0.00355)0.610:0.00322)0.950.11:0.01025,((139390:0.00983,12524:0.02195)1.000.149:0.03438,(591444:0.02665,((215846:0.04245,((151097:0.01461,329012:0.01833)0.563.2:0.00501,564648:0.00314)0.192:0.01407)0.814.2:0.00235,540668:0.02592)0.081:0.00155)0.977.11:0.01197)0.955.6:0.01743)0.755.3:0.01051)0.981.7:0.01073,((((543795:0.00204,580761:0.00684)0.739.4:0.01247,((565461:0.01473,588313:0.04775)0.932.2:0.00595,(119667:0.00077,236439:0.0025)0.850.2:0.01913)0.834.3:0.00243)0.894.4:0.00398,155882:0.01868)0.991.8:0.01212,(((260851:0.02847,((((573369:0.01454,547368:0.01832)0.763:0.0053,161055:0.00585)0.767.4:0.00493,145979:0.01378)0.994.10:0.01894,(229799:0.02076,((155352:0.00976,265139:0.0101)0.872.4:0.00506,348047:0.00602)0.972.10:0.00826)0.426:0.00948)0.606:0.00845)0.906:0.00571,((((11418:0.02225,(245061:0.02201,11402:0.03205)0.385.3:0.00465)0.498.2:0.0032,(((((142032:0.027,(193862:0.00415,203248:0.01289)0.926.7:0.00479)0.978.6:0.00764,(270964:0.01049,270309:0.01199)0.618.2:0.00577)0.999.37:0.01709,((389275:0.01249,545443:0.00548)0.876.2:0.0063,(576054:0.02198,574727:0.02785)0.884.3:0.00767)0.999.38:0.02097)0.999.39:0.02614,(546478:0.0351,(((550680:0.00492,172264:0.00816)0.950.12:0.00494,273824:0.0096)0.853:0.00363,554148:0.00957)0.999.40:0.01882)0.753.2:0.00773)0.323:0.00464,(272708:0.01801,(156409:0.02079,(344834:0.01938,(330721:0.01521,228452:0.02061)0.931.6:0.0063)0.871.6:0.00344)0.914.3:0.0054)1.000.150:0.01831)0.895.5:0.01141)0.786:0.00867,265482:0.02587)0.844.2:0.00307,(265696:0.00778,(215181:0.01246,(187806:0.01334,289344:0.00987)0.936.6:0.00917)0.953.2:0.007)0.999.41:0.01585)0.950.13:0.00669)0.868.8:0.00358,(410072:0.02343,532624:0.01606)1.000.151:0.01555)0.301.2:0.00211)0.864.3:0.0047)0.900:0.00501,((553502:0.00536,138117:0.02229)1.000.152:0.02807,(((((254130:0.0129,243453:0.04782)0.326:0.00347,545316:0.01781)0.848.2:0.00261,(((525442:0.0056,240178:0.00625)0.447:0.00259,558919:0.01774)0.939.4:0.00628,542020:0.03994)0.399:0.00171)0.926.8:0.00475,534888:0.02351)0.772:0.00608,535040:0.03034)0.910.4:0.01078)0.862.2:0.01663)0.471:0.00016,250067:0.05241)0.894.5:0.00687,(((582749:0.009,578659:0.00777)0.942.9:0.00706,(250987:0.02015,336584:0.00881)0.997.13:0.01242)1.000.153:0.01867,(56545:0.03927,((24260:0.02388,(518645:0.02194,(317641:0.01809,(174445:0.00679,574943:0.01238)0.696.4:0.00347)0.813.2:0.00301)0.899.3:0.00281)0.592.3:0.00249,((225319:0.01476,222364:0.01461)1.000.154:0.01494,(223441:0.00929,(207124:0.03124,(215003:0.01874,256166:0.01969)0.641:0.00405)0.984.8:0.01125)0.583.2:0.00228)0.294.2:0.00263)0.773.2:0.00152)0.867.3:0.0033)0.841:0.00432)0.932.3:0.00751,(((333608:0.01291,318902:0.01063)1.000.155:0.02714,((512041:0.00284,(216128:0.01542,217034:0.02102)0.457:0.00215)1.000.156:0.0129,(((113586:0.00708,112160:0.00698)1.000.157:0.0189,((65445:0.03251,586184:0.00299)0.762.3:0.0015,(517156:0.00821,342630:0.01871)0.740.4:0.0042)0.996.16:0.01634)0.951.6:0.00786,((265034:0.00865,257150:0.0223)1.000.158:0.01394,((533436:0.01889,593462:0.00407)0.757:0.00528,(261988:0.00833,343298:0.00714)0.433:0.00454)0.998.13:0.01376)0.974.3:0.00793)0.937.2:0.00461)0.710.2:0.00141)0.875.3:0.0054,(509223:0.03181,(245241:0.0172,(63126:0.00674,253229:0.02294)0.899.4:0.00512)0.808:0.01008)0.969.10:0.00919)0.143:0.00384)0.931.7:0.00751,(244549:0.02828,((125157:0.02926,((((254262:0.01129,(575662:0.00563,(348982:0.00869,212039:0.00546)0.799:0.01073)0.992.10:0.01022)0.991.9:0.00789,563192:0.00388)0.293:0.01343,(570504:0.02,(((324650:0.00957,(338423:0.005,271012:0.00492)0.910.5:0.0159)0.864.4:0.00578,338983:0.00686)0.999.42:0.01756,((160171:0.00767,582410:0.01708)1.000.159:0.02121,(((349633:0.01173,572076:0.01343)0.997.14:0.0126,((342593:0.02398,(310703:0.01729,270810:0.01532)1.000.160:0.03141)0.882.3:0.00994,((581177:0.00386,197548:0.04547)0.948.3:0.00727,(549352:0.0034,150770:0.03168)0.877.4:0.00878)0.433.2:0.00622)1.000.161:0.01863)0.643:0.00283,(152199:0.01196,((161639:0.01703,((550158:0.01385,550926:0.01512)0.973.5:0.01109,589088:0.00818)0.325:0.01583)0.955.7:0.00695,310346:0.02876)0.920.2:0.00536)0.980.8:0.00956)0.492:0.00357)0.916.4:0.00405)0.745.4:0.00341)0.914.4:0.00573)0.896.3:0.00768,((((592515:0.01205,263082:0.00232)0.970.2:0.00393,325555:0.00643)0.974.4:0.00459,(((((563246:0.01087,509624:0.00864)0.920.3:0.00532,403565:0.00719)0.973.6:0.00375,(260281:0.00718,583303:0.00736)0.681.2:0.00155)0.882.4:0.01124,(((298250:0.01322,(276603:0.0229,554927:0.02044)0.870.2:0.00611)0.759.2:0.00082,((581081:0.00698,459268:0.00865)0.870.3:0.00163,(140384:0.00777,270611:0.00162)0.872.5:0.01443)0.762.4:0.00081)0.960.7:0.00492,590548:0.01219)0.886.4:0.00403)0.902:0.00312,(63831:0.01172,(113200:0.01049,354357:0.01476)0.486.2:0.00303)0.926.9:0.00418)0.745.5:0.00073)0.884.4:0.00295,(526161:0.00512,((253077:0.00966,257743:0.02688)0.944.2:0.0037,(270308:0.01446,(236395:0.01449,137604:0.00706)0.030:0.00694)0.915.4:0.00272)0.951.7:0.00366)0.938.4:0.00433)0.995.12:0.01172)0.995.13:0.01264)0.988.13:0.01103,(282746:0.0237,113190:0.02303)0.930.4:0.00669)0.702.2:0.0052)0.997.15:0.01448)0.917.4:0.00707)0.965.6:0.01101)0.981.8:0.00914,(((((221218:0.0194,161665:0.01468)0.791:0.00182,(222011:0.01202,((11397:0.02298,(63307:0.01993,557394:0.01778)0.596:0.00268)0.857.3:0.00275,245969:0.0153)0.894.6:0.00252)0.936.7:0.00432)0.994.11:0.00929,((63304:0.00805,(48649:0.00844,249582:0.017)0.987.9:0.01306)1.000.162:0.02143,(140736:0.00653,160500:0.01891)0.740.5:0.00622)0.997.16:0.01282)0.885.3:0.00254,(548077:0.02883,(232306:0.01351,(((((11398:0.01739,157129:0.01217)0.876.3:0.02547,((578668:0.03482,((160517:0.0116,(241866:0.02475,364140:0.00347)0.942.10:0.00794)0.999.43:0.02076,((58322:0.02054,(225385:0.00442,63308:0.01815)0.984.9:0.01064)0.950.14:0.00796,(570851:0.00953,(532349:0.02741,((108882:0.00014,582989:0.00014)0.792.3:0.00161,269564:0.00979)0.183:0.00485)0.973.7:0.00991)0.958.8:0.00813)0.983.10:0.01405)0.266:0.00846)0.987.10:0.01481,43490:0.01687)0.938.5:0.00678)0.400:0.00415,((156849:0.01382,((((63301:0.01032,(11631:0.019,11536:0.0215)0.816.3:0.00201)0.984.10:0.00584,(34259:0.02093,(39131:0.01256,128358:0.01657)0.949.5:0.00568)0.702.3:0.00566)0.809.3:0.00273,(11632:0.01108,582560:0.01562)0.560:0.00389)0.763.2:0.00015,160047:0.00686)1.000.163:0.05376)0.902.2:0.00842,(156584:0.00557,159923:0.01627)0.921.6:0.01086)0.984.11:0.01245)0.764.3:0.00305,((11399:0.02378,113475:0.02163)0.323.2:0.00308,11396:0.02505)0.906.2:0.0041)0.730:0.00056,(211564:0.01205,235014:0.01804)0.872.6:0.0059)0.865.3:0.00404)0.932.4:0.00528)0.766.4:0.00015)0.508:0.00162,((263760:0.00566,236029:0.06319)0.932.5:0.00554,253790:0.02046)0.904.3:0.00318)0.970.3:0.00602)0.923.2:0.00593)0.812.2:0.00285,(556910:0.02254,(((256498:0.03914,529793:0.01098)0.994.12:0.01384,((((584272:0.03285,84258:0.01145)0.972.11:0.00954,(562659:0.03162,(236284:0.01518,279158:0.00831)0.964.7:0.00963)0.996.17:0.01754)0.950.15:0.00901,((594448:0.00818,155271:0.03063)0.985.5:0.00951,(243610:0.0195,257176:0.0146)0.964.8:0.00787)0.832:0.00319)0.898.5:0.00662,(584385:0.00922,(101604:0.00757,255449:0.02599)0.972.12:0.00836)1.000.164:0.015)0.977.12:0.00907)0.369:0.00225,((((((((209118:0.01206,348524:0.01018)0.321:0.00287,207912:0.01587)0.869.4:0.00374,580487:0.02589)0.802.2:0.00549,(534805:0.0008,559205:0.02858)0.957.8:0.00512)0.400.2:0.00095,350536:0.03362)0.954:0.0061,570023:0.01796)0.894.7:0.00747,156323:0.01373)0.935.2:0.00632,281116:0.0182)0.967.6:0.00828)1.000.165:0.02085)0.886.5:0.00838)0.998.14:0.01699)0.985.6:0.01549,((((323004:0.00191,113150:0.00459)0.981.9:0.01411,(((80775:0.02492,156755:0.01412)0.535:0.00363,((41242:0.00965,355212:0.0064)0.999.44:0.01468,206831:0.01285)0.720.2:0.005)0.985.7:0.01036,(346162:0.00375,(328890:0.00761,(312275:0.00325,257493:0.0038)0.969.11:0.00559)1.000.166:0.02085)0.872.7:0.00853)0.327:0.00354)1.000.167:0.03621,((260879:0.01333,464059:0.02111)0.999.45:0.02283,(585361:0.01194,(48363:0.02616,354168:0.02592)0.950.16:0.00762)0.933.3:0.009)0.996.18:0.02567)0.999.46:0.02616,((240287:0.02458,(588201:0.01583,511808:0.00989)0.999.47:0.02447)0.997.17:0.03157,((248375:0.03237,577683:0.01724)1.000.168:0.13447,(((((55589:0.0164,(42261:0.03181,248619:0.01744)0.878.5:0.00608)0.999.48:0.02042,((251393:0.0213,(215928:0.00847,237496:0.01841)0.976.6:0.01187)0.995.14:0.01951,224718:0.03025)0.759.3:0.00192)0.411:0.00524,574576:0.01665)0.941.6:0.00795,547749:0.03934)0.983.11:0.0107,(((128299:0.02642,(170956:0.008,((555635:0.01469,137747:0.0208)0.196:0.00626,587309:0.00537)0.795.4:0.00337)0.996.19:0.01394)0.350:0.0029,((121375:0.01586,(236133:0.02638,(210792:0.00464,217323:0.00166)0.605:0.00646)0.306:0.00538)0.991.10:0.01331,((591093:0.01545,361471:0.00858)0.987.11:0.01082,(253573:0.01209,581739:0.03194)0.575:0.0045)0.884.5:0.00629)0.963.4:0.0084)0.978.7:0.0101,580062:0.0129)0.968.6:0.01242)0.482:0.00895)0.933.4:0.01466)0.970.4:0.01728)0.869.5:0.01222)0.841.2:0.00985)0.848.3:0.01495,((((141216:0.00828,591604:0.01042)0.256:0.00532,(251707:0.01838,(588071:0.0087,(111386:0.00901,72811:0.0067)0.997.18:0.01321)0.970.5:0.01067)0.870.4:0.00483)1.000.169:0.0214,((((((108280:0.00563,(((((((150663:0.00931,(561317:0.00812,554990:0.00919)0.766.5:0.00239)0.770:0.00156,(509764:0.00809,549754:0.01342)0.781.2:0.00084)0.920.4:0.00536,(316839:0.0029,12043:0.01105)0.897.2:0.00928)0.801:0.00076,((168232:0.00894,100606:0.00471)0.887:0.00468,(593238:0.01345,51116:0.01846)0.462:0.00258)0.803.3:0.00077)0.994.13:0.00484,(((40889:0.00395,105867:0.00488)0.570:0.00248,432211:0.01036)0.982.10:0.00547,(141124:0.00639,104810:0.00391)0.839:0.00093)0.921.7:0.00073)1.000.170:0.00479,((38367:0.00405,(30118:0.00116,112062:0.01767)0.980.9:0.00707)0.963.5:0.00585,18830:0.00907)0.879.3:0.00426)0.839.2:0.00444,104661:0.02433)0.879.4:0.00593)0.865.4:0.00327,(579412:0.00404,(532962:0.00994,194841:0.00653)0.279:0.00157)0.956.5:0.00531)0.970.6:0.00587,(((300943:0.00166,((((277241:0.00357,261552:0.01405)0.853.2:0.00866,(574980:0.02071,191592:0.01355)0.980.10:0.00768)0.718.3:0.00267,(100801:0.01019,564329:0.01167)0.875.4:0.00246)0.231:0.00219,(63709:0.00638,(546990:0.01666,((249273:0.00697,(24444:0.00407,(189603:0.04029,155746:0.02877)0.816.4:0.00179)0.422:0.00077)0.567:0.00121,257173:0.01073)0.824:0.00308)0.987.12:0.00753)0.594:0.00223)0.992.11:0.00826)0.929.4:0.005,(591087:0.00668,(471204:0.01187,211377:0.01505)0.928.4:0.00602)0.725:0.00225)1.000.171:0.02969,(100915:0.0078,(243869:0.00014,589503:0.00079)0.847:0.0109)0.966.6:0.00549)0.703.3:0.00415)0.908.4:0.00362,((260552:0.01459,34539:0.00756)1.000.172:0.0179,(((174843:0.00682,12112:0.00581)0.640:0.00357,((546102:0.00549,((211188:0.00179,276680:0.03063)0.938.6:0.0054,19369:0.01912)0.883.5:0.00212)0.934.3:0.00514,552196:0.00168)0.954.2:0.00558)0.767.5:0.00199,((49520:0.00933,((575948:0.00628,(521949:0.0179,571069:0.00321)1.000.173:0.00325)0.858.2:0.00016,(102741:0.00388,253838:0.00156)0.878.6:0.00464)0.849.3:0.0017)0.946.5:0.00615,(((((12098:0.01461,550415:0.02002)0.936.8:0.00016,(12101:0.01384,(243267:0.01612,12065:0.00809)0.832.2:0.00147)0.741.2:0.00076)0.964.9:0.00466,17513:0.01457)0.968.7:0.00529,12106:0.0059)0.981.10:0.01014,(181053:0.00477,(575689:0.0087,108282:0.00374)0.880.4:0.00287)0.907.4:0.00491)0.439.2:0.00145)0.864.5:0.00368)0.938.7:0.00553)0.986.8:0.01112)0.648:0.00474,12131:0.02077)0.993.11:0.01274,((203722:0.03172,334140:0.01277)0.970.7:0.00734,((136417:0.01348,525863:0.00669)0.524:0.00564,(250813:0.01075,246169:0.01722)0.765.2:0.0055)0.534:0.00656)0.997.19:0.01485)0.218:0.00173)0.970.8:0.00898,(12128:0.02544,(11990:0.01849,11985:0.0391)0.669:0.01534)0.996.20:0.01725)0.977.13:0.01482)0.582.2:0.00592)0.192.2:0.00617,((((269352:0.01103,526038:0.01973)0.993.12:0.01274,((99316:0.01335,535545:0.00992)0.833:0.00537,((((349434:0.01784,(((((160968:0.00392,12941:0.02839)0.631:0.00076,297261:0.0049)0.847.2:0.00151,(216399:0.01399,((214049:0.00593,(181203:0.00704,328174:0.00392)0.694:0.00851)0.952.6:0.00381,(262254:0.00238,12938:0.00943)0.983.12:0.0056)0.734.3:0.00079)0.957.9:0.00339)0.156:0.00016,154705:0.01447)0.834.4:0.006,257520:0.02294)0.870.5:0.0048)0.998.15:0.01198,((244488:0.01844,((265971:0.0116,((12703:0.02428,(12701:0.00111,12702:0.00553)0.976.7:0.00941)0.998.16:0.01651,(((244000:0.00623,139156:0.0092)1.000.174:0.01633,556664:0.0131)0.890.4:0.00535,(((538144:0.0079,(((((((104427:0.00593,(257287:0.03034,544457:0.00147)0.850.3:0.00093)0.934.4:0.00313,((91736:0.00351,243785:0.01557)0.879.5:0.00467,141774:0.00262)0.900.2:0.00155)0.570.2:0.00014,310095:0.01648)0.967.7:0.0047,144336:0.00945)0.874.6:0.0019,((579406:0.00237,534213:0.00334)0.780.2:0.00122,82793:0.02294)0.735:0.00157)0.898.6:0.00378,273880:0.01064)0.724:0.00117,54677:0.01033)0.894.8:0.00338)0.634.2:0.00393,37332:0.00647)0.971.6:0.00707,((13016:0.00836,(554357:0.00555,(525091:0.00535,549001:0.00907)0.645.2:0.00366)0.871.7:0.00355)0.860.6:0.00604,((81456:0.01259,13019:0.01215)0.847.3:0.00591,558894:0.00785)0.776.3:0.00672)0.947.9:0.00686)0.863.2:0.0035)0.754.3:0.00386)0.988.14:0.01087)0.896.4:0.00345,(572368:0.01137,113711:0.0138)0.870.6:0.00287)0.976.8:0.00726)0.877.5:0.00444,555033:0.01964)0.851.2:0.00268)0.963.6:0.00913,((158633:0.00158,(536855:0.00091,550861:0.00311)0.907.5:0.00326)0.874.7:0.01837,(((12919:0.01505,147492:0.01108)0.954.3:0.00015,((43413:0.00543,279920:0.00812)0.860.7:0.00481,(150637:0.00487,(111157:0.01299,(((252366:0.00101,590285:0.00014)1.000.175:0.01593,223819:0.02865)0.825.3:0.00109,550950:0.00555)0.913.4:0.00065)0.766.6:0.00076)0.793:0.00327)0.330:0.00327)0.943.12:0.00432,((85863:0.00858,175954:0.01616)0.394:0.00159,(506998:0.02164,(((88338:0.01776,138207:0.00789)0.033:0.00291,(577382:0.0121,((400950:0.01239,314065:0.00606)0.957.10:0.00621,(35860:0.00952,(518306:0.0114,42025:0.00909)0.897.3:0.00308)0.835:0.00196)0.987.13:0.00638)0.726:0.00206)0.971.7:0.00525,383433:0.0089)0.907.6:0.00293)0.895.6:0.0035)0.965.7:0.00504)0.993.13:0.00935)0.957.11:0.00737)0.959.7:0.0061,((133017:0.00561,12988:0.01645)0.804.3:0.0019,(((12989:0.00661,537159:0.00269)0.895.7:0.00454,(((12986:0.01559,12978:0.01028)0.805.3:0.00015,(((152567:0.03073,(63345:0.00471,(272368:0.02224,216326:0.02167)0.844.3:0.00613)0.891.3:0.00171)0.797:0.00154,12980:0.02802)0.891.4:0.00233,(12963:0.00719,306457:0.00511)0.865.5:0.00706)0.792.4:0.00077)0.996.21:0.00809,12984:0.00548)0.976.9:0.00932)0.820.2:0.00555,(248764:0.00897,((155487:0.01557,(557167:0.00155,136536:0.01087)0.885.4:0.00466)0.888:0.00375,((275689:0.00664,(33829:0.01186,(12995:0.01677,12993:0.00582)0.789:0.00014)0.894.9:0.0061)0.959.8:0.00819,552367:0.00493)0.263.2:0.00289)0.608.2:0.00733)0.753.3:0.00165)0.921.8:0.0043)0.927.2:0.00525)0.754.4:0.00201)0.964.10:0.01219)0.929.5:0.0076,((550414:0.00586,(550105:0.01693,525473:0.01294)0.727.2:0.00355)0.738.3:0.00351,(379515:0.01092,313723:0.00981)0.957.12:0.00783)0.991.11:0.01482)0.227.2:0.00502,((((((238963:0.00534,(240258:0.03054,564093:0.01005)0.975.5:0.00671)0.986.9:0.0058,515421:0.00452)0.718.4:0.00064,535010:0.00448)0.984.12:0.00739,((138062:0.0491,(((((((211265:0.0121,(200014:0.00563,113935:0.01278)0.941.7:0.00544)0.998.17:0.01166,576440:0.01056)0.220:0.00019,(244332:0.00696,242607:0.02788)0.890.5:0.00742)0.785.3:0.00117,581811:0.02439)0.846:0.00253,564025:0.01938)0.065:0.00014,(560566:0.01901,139950:0.00449)0.466.2:0.00342)0.965.8:0.00544,(((((580850:0.00635,(357423:0.00591,542118:0.01762)0.907.7:0.00462)0.969.12:0.00314,557564:0.00709)1.000.176:0.00016,(((160066:0.01107,(((22701:0.009,(528979:0.00724,(150244:0.00936,589246:0.01533)0.530:0.00617)0.926.10:0.01154)0.974.5:0.00527,(154710:0.01753,(267960:0.00353,575601:0.02605)0.974.6:0.00323)0.559.2:0.00015)0.771.2:0.00418,568948:0.00602)0.921.9:0.00312)0.789.2:0.00391,(((78124:0.0043,(548516:0.00156,64356:0.00951)0.097.2:0.00245)0.893.2:0.00287,(((236904:0.00474,250572:0.07012)0.000.2:0.00015,156316:0.01733)0.782.2:0.00073,(39135:0.00919,24564:0.00931)0.794.4:0.00082)0.882.5:0.00955)0.959.9:0.00453,409932:0.00459)0.881.5:0.00014)0.113:0.00171,(304224:0.00043,248468:0.02777)0.859.3:0.00173)0.853.3:0.00372)0.934.5:0.00161,((((135946:0.0129,242784:0.01499)0.945.7:0.01465,554324:0.00818)0.000.3:0.00212,(12840:0.00727,252354:0.00664)0.940.5:0.00232)0.838.3:0.00233,277508:0.0088)0.855.4:0.00258)0.284:0.00131,(112340:0.01803,(12905:0.00587,519572:0.00562)0.878.7:0.00498)0.896.5:0.00703)0.911.5:0.00222)0.845.5:0.0016)0.970.9:0.00013,(359367:0.01045,(((552236:0.00734,250989:0.01334)0.985.8:0.00931,((42584:0.01042,237912:0.0266)0.937.3:0.00491,(35366:0.00654,(574020:0.00646,(370045:0.00507,112936:0.00521)0.982.11:0.00692)0.999.49:0.01548)0.956.6:0.00492)0.960.8:0.00686)0.909.3:0.00519,219743:0.00869)0.785.4:0.00271)0.290.2:0.00155)0.929.6:0.00461)0.837.3:0.00575,((555017:0.01322,(181645:0.01455,((535227:0.02225,((99171:0.00319,236358:0.00559)0.960.9:0.00417,(275938:0.00779,33446:0.01345)0.951.8:0.00454)0.869.6:0.00571)1.000.177:0.06019,(104900:0.00487,279281:0.01382)0.916.5:0.00557)0.448.3:0.00375)1.000.178:0.0225)0.999.50:0.02044,(((41449:0.00426,203127:0.01318)0.254.3:0.0024,(181134:0.00513,12950:0.00857)0.453:0.0028)1.000.179:0.06958,(343126:0.00892,574018:0.02205)0.921.10:0.01061)0.825.4:0.00979)0.894.10:0.00594)0.934.6:0.00906,166311:0.03905)0.952.7:0.00959)0.946.6:0.00954)0.892.3:0.00521,((111674:0.01961,570720:0.01175)0.998.18:0.01549,(112301:0.02598,128099:0.01003)1.000.180:0.02725)0.977.14:0.01202)0.956.7:0.00682,((139597:0.01656,(582429:0.01133,((((((((11825:0.00126,264063:0.00726)0.605.2:0.00115,81060:0.02043)0.940.6:0.00077,317386:0.01156)0.598:0.00085,132602:0.00787)0.995.15:0.00783,((591184:0.01673,213834:0.00872)0.885.5:0.00172,((((((198384:0.00726,(((102921:0.01799,((513415:0.01448,238534:0.00115)0.973.8:0.00418,(157671:0.01499,276073:0.00045)0.420:0.01189)0.775.2:0.00064)0.773.3:0.00091,((((((590298:0.018,11920:0.03111)0.000.4:0.0003,239270:0.00271)0.436:0.00078,((567355:0.01256,563236:0.02462)0.979.13:0.0003,(559795:0.02903,((305384:0.10898,180775:0.0875)0.687.2:0.00253,568012:0.16011)0.790.2:0.00096)0.827.2:0.00159)0.940.7:0.00074)1.000.181:0.00013,299931:0.01428)0.741.3:0.00027,((((272393:0.01029,(((16961:0.00737,11871:0.01908)0.738.4:0.00221,278310:0.01093)0.947.10:0.00248,(((350595:0.052,258807:0.00264)0.517:0.01592,555369:0.03681)1.000.182:0.00016,(245533:0.00859,531574:0.01702)0.000.5:0.00102)0.130:0.00077)0.916.6:0.00074)0.561.2:0.00123,(((((584092:0.02994,((133451:0.03121,(186079:0.00822,512303:0.03895)0.891.5:0.0026)0.697.2:0.00014,((247242:0.01775,245293:0.01054)0.955.8:0.02984,581380:0.02914)0.778.2:0.00275)0.321.2:0.00282)0.750.3:0.00141,587328:0.00177)0.892.4:0.00042,((242090:0.01269,(140767:0.01628,143008:0.02975)0.909.4:0.00823)0.945.8:0.0227,(16836:0.00011,246819:0.01394)0.797.2:0.00199)0.884.6:0.00014)0.000.6:0.00324,(307348:0.00769,(572519:0.02511,199958:0.01458)0.000.7:0.00165)0.364:0.00245)0.972.13:0.00016,11875:0.01646)0.717.3:0.00071)0.412:0.00077,332727:0.04042)0.919.3:0.00066,130132:0.01414)1.000.183:0.00014)0.999.51:0.00015,(137956:0.01985,156444:0.08063)0.794.5:0.00916)0.721:0.0031)0.898.7:0.00225,(136362:0.02747,23782:0.04039)0.974.7:0.00155)0.814.3:0.00016)0.982.12:0.00403,(((((157140:0.00014,317514:0.00028)0.913.5:0.01234,(((555939:0.00077,313124:0.04894)0.800.2:0.00408,276797:0.00284)0.723.2:0.00089,(151684:0.01362,264455:0.01154)0.836.3:0.01487)0.870.7:0.00276)0.957.13:0.00448,((199105:0.00014,101311:0.008)1.000.184:0.0111,535617:0.00263)0.738.5:0.00074)0.922:0.00293,(105114:0.00095,249849:0.00134)1.000.185:0.00653)0.945.9:0.00823,(589769:0.00571,567040:0.00853)0.393.2:0.00078)0.779.2:0.0008)0.565:0.00389,(((247947:0.02717,244680:0.00106)0.851.3:0.00092,((157248:0.02191,105617:0.00443)0.862.3:0.00081,((263669:0.00257,(273898:0.00211,576695:0.00497)0.830.2:0.00149)0.805.4:0.0086,234418:0.01374)0.776.4:0.00089)0.744.2:0.00157)0.929.7:0.00229,(151293:0.00735,571692:0.06179)0.001.2:0.00016)0.785.5:0.00079)0.799.2:0.00245,((250546:0.00241,(((538020:0.0234,262538:0.01339)0.908.5:0.00339,(49805:0.00668,((53467:0.0116,((251051:0.00218,134109:0.00727)0.851.4:0.00186,573794:0.01089)0.930.5:0.00233)0.936.9:0.00311,353728:0.00782)0.869.7:0.00311)0.671.2:0.00261)1.000.186:0.00092,352835:0.03756)1.000.187:0.00107)0.886.6:0.00155,94790:0.00638)0.892.5:0.00248)0.382:0.00179,((549935:0.00335,561221:0.00598)0.847.4:0.00151,(343365:0.00472,136568:0.01474)0.784.2:0.00358)0.411.2:0.00089)0.880.5:0.0049,(((((355361:0.02785,511101:0.00887)0.883.6:0.00235,553328:0.06464)0.805.5:0.00016,(((((238492:0.01964,177045:0.00417)0.869.8:0.00411,540453:0.0475)0.896.6:0.00159,(327769:0.00302,246683:0.01263)0.914.5:0.00165)0.805.6:0.00156,(569341:0.0078,((263935:0.00831,((((160135:0.00534,185820:0.00667)0.950.17:0.00397,(576909:0.00705,241092:0.01442)0.753.4:0.00173)0.168:0.00358,46451:0.00631)0.853.4:0.00171,((246667:0.02916,((((562973:0.00613,252723:0.01245)0.093:0.00232,(((221815:0.00276,313468:0.03215)0.832.3:0.00387,336298:0.00795)0.796:0.00143,(569156:0.00913,(275765:0.03254,(321528:0.0112,260903:0.05345)0.274:0.00434)0.872.8:0.00016)0.684.3:0.00906)0.637.2:0.00029)0.966.7:0.00378,((203618:0.01308,559760:0.00412)0.475:0.00016,(91947:0.01442,151419:0.00715)0.965.9:0.0039)0.791.2:0.00113)0.892.6:0.00211,(((242867:0.04106,256117:0.01202)0.733:0.01509,188942:0.01547)1.000.188:0.001,306796:0.0097)0.936.10:0.00015)0.829.2:0.00632)0.650:0.00155,(102862:0.01652,(202032:0.03617,33773:0.01298)0.925.5:0.00155)0.729:0.00073)1.000.189:0.00097)0.857.4:0.00144)0.900.3:0.00234,(46425:0.00168,177854:0.00314)0.999.52:0.00781)0.758:0.00087)0.786.2:0.00016)0.778.3:0.00085,((340050:0.01656,104459:0.00772)0.000.8:0.0003,((130081:0.01766,11761:0.00197)0.000.9:0.00028,264465:0.03873)1.000.190:0.0009)0.945.10:0.00555)0.828:0.00233)0.231.2:0.00374,(((237060:0.02316,(278711:0.00554,308934:0.00432)0.931.8:0.0097)0.811:0.00258,11774:0.00672)0.819:0.00169,132566:0.00692)1.000.191:0.00097)0.340:0.00077,199229:0.03417)0.939.5:0.00279)0.758.2:0.00104)0.932.6:0.00375)0.861.2:0.0016,132419:0.01202)0.997.20:0.00837,(153785:0.00273,139908:0.01923)0.471.2:0.00395)0.872.9:0.00236,((30701:0.00651,(151510:0.01674,11863:0.00639)0.770.2:0.00195)0.566:0.00164,(67228:0.00105,(184721:0.00249,11861:0.00801)0.987.14:0.0047)0.164:0.00914)0.863.3:0.00178)0.997.21:0.00933)0.913.6:0.00338)1.000.192:0.01485,575785:0.0172)0.074.3:0.00169)0.819.2:0.00364,((((305551:0.03671,467626:0.02611)0.982.13:0.01146,((104294:0.02325,451456:0.01637)0.733.2:0.00586,((234186:0.00636,586558:0.01717)0.995.16:0.01173,(11431:0.00314,((152323:0.01682,270517:0.00029)0.940.8:0.00462,585063:0.00573)0.872.10:0.01324)0.758.3:0.00246)0.991.12:0.01272)0.920.5:0.00636)0.947.11:0.00461,(((193229:0.00428,568643:0.00676)0.945.11:0.02507,(525816:0.01634,65420:0.01222)0.968.8:0.00885)0.191:0.00683,((267662:0.00583,557602:0.00375)1.000.193:0.02276,(((567387:0.00488,(586357:0.00593,(312041:0.01875,304085:0.02037)0.908.6:0.01195)0.903.3:0.01129)1.000.194:0.04857,(160626:0.02155,318920:0.00679)0.988.15:0.01385)0.896.7:0.01377,(((578307:0.01785,((((458334:0.00434,((235395:0.01768,((105395:0.01894,(313857:0.00248,(307023:0.01844,555795:0.00014)1.000.195:0.0521)0.990.6:0.01399)0.850.4:0.02006,((315209:0.01092,240198:0.00571)0.863.4:0.00199,331093:0.01591)0.959.10:0.0091)0.834.5:0.0054)0.947.12:0.00401,545460:0.01095)0.937.4:0.00472)0.950.18:0.00687,((152835:0.03378,((228795:0.01158,127821:0.00254)0.833.2:0.02331,240234:0.0067)0.985.9:0.00632)0.948.4:0.00574,(329744:0.01763,248827:0.01137)0.999.53:0.01402)0.958.9:0.00931)0.863.5:0.00758,(104500:0.00334,(349848:0.01249,(273604:0.00406,305194:0.00028)0.918.2:0.01781)0.966.8:0.00688)1.000.196:0.02098)0.951.9:0.00647,581541:0.00493)0.734.4:0.00781)0.999.54:0.0177,233361:0.01035)0.836.4:0.00597,323949:0.02191)0.995.17:0.01503)0.960.10:0.0095)0.967.8:0.00983)0.597:0.00096)0.979.14:0.00722,((((((((((160337:0.01155,104310:0.00708)0.408:0.00092,((406112:0.01581,(217413:0.01024,11622:0.01536)0.997.22:0.00015)0.929.8:0.00475,((73544:0.0056,(11621:0.01226,252713:0.00997)0.904.4:0.00344)0.926.11:0.00353,(102993:0.0049,240553:0.01518)0.896.8:0.00014)0.787:0.00169)0.956.8:0.00352)0.898.8:0.00333,259094:0.00752)0.993.14:0.00915,((254635:0.02061,(112323:0.01779,552883:0.02098)0.109:0.00257)0.314.2:0.00312,(252888:0.01848,((11620:0.0047,11492:0.00955)0.941.8:0.00322,(111476:0.03818,(((136379:0.0115,(11544:0.00961,41130:0.00155)0.906.3:0.0055)0.936.11:0.00249,258045:0.00028)0.793.2:0.00803,11491:0.01337)0.159:0.00016)0.884.7:0.00226)0.992.12:0.00951)0.890.6:0.005)0.942.11:0.00563)0.933.5:0.00354,(104265:0.00567,(250241:0.01206,(144241:0.00014,142506:0.01267)0.994.14:0.00913)0.960.11:0.00586)0.915.5:0.00355)0.645.3:0.00186,(379604:0.0139,(83120:0.01528,(254610:0.00463,(((257618:0.00387,(543740:0.01331,223578:0.00176)0.782.3:0.00472)0.732:0.00077,(151679:0.00683,83514:0.00465)0.979.15:0.00513)0.821.2:0.00387,((554377:0.00518,574265:0.01274)0.789.3:0.00203,(246986:0.00844,256178:0.01026)0.994.15:0.00857)0.659:0.002)0.936.12:0.00502)0.904.5:0.00442)0.988.16:0.00879)0.755.4:0.00429)0.092:0.00227,((237887:0.01938,225724:0.02747)0.935.3:0.00527,((255610:0.02257,(214812:0.01403,137991:0.0125)0.952.8:0.00922)0.991.13:0.01384,(((11429:0.00701,((139937:0.00233,353538:0.01717)0.982.14:0.00547,(((225825:0.00987,218165:0.01306)0.738.6:0.00149,248801:0.01613)0.972.14:0.00414,(11441:0.00561,11435:0.01188)0.400.3:0.00015)0.784.3:0.00769)0.550:0.00015)0.760.3:0.0009,266111:0.00896)0.950.19:0.00468,((90721:0.00439,((223418:0.00565,220439:0.02264)0.916.7:0.00237,366012:0.01501)0.883.7:0.00155)0.870.8:0.00367,(157616:0.01061,547588:0.00636)0.990.7:0.00787)0.816.5:0.00208)0.999.55:0.01539)0.877.6:0.00515)0.634.3:0.00255)0.593:0.00189,((11448:0.0072,((277895:0.01966,247315:0.00577)0.573.2:0.0106,76142:0.00626)0.987.15:0.00762)0.999.56:0.01245,(565489:0.02669,(248323:0.01868,563670:0.00496)0.957.14:0.00586)0.914.6:0.00614)0.907.8:0.00607)0.866.2:0.00323,(559986:0.02256,(83082:0.01311,(237583:0.00734,146122:0.00805)0.757.2:0.00286)0.877.7:0.00331)0.811.2:0.00282)0.957.15:0.0043,550173:0.02957)0.739.5:0.0044)0.684.4:0.00122)0.973.9:0.0077,(((((422976:0.0167,(99702:0.00583,(366196:0.00909,543710:0.01098)0.969.13:0.00681)0.971.8:0.0075)0.964.11:0.00644,((153548:0.01289,213535:0.00972)0.946.7:0.006,((((12697:0.01113,24568:0.00303)0.998.19:0.01701,((((543684:0.00813,(304148:0.00237,12663:0.03903)0.982.15:0.00546)0.128:0.00236,351340:0.00822)1.000.197:0.01695,269218:0.0065)0.363:0.00284,(((158679:0.01786,(140652:0.01144,12690:0.00922)0.946.8:0.0062)0.648.2:0.00067,(190663:0.00906,266368:0.02264)0.870.9:0.00816)0.844.4:0.00423,(576231:0.00639,165578:0.02175)0.934.7:0.00439)0.927.3:0.00461)0.869.9:0.0022)0.883.8:0.00363,86480:0.01287)0.377:0.00079,246070:0.00915)0.903.4:0.00314)0.861.3:0.00261)0.860.8:0.00264,(86523:0.01357,255647:0.00735)0.999.57:0.01701)0.497:0.00308,((386689:0.01027,571410:0.00394)0.936.13:0.01989,((((139055:0.02062,242450:0.01047)0.989.9:0.01222,578832:0.01943)0.846.2:0.00403,80600:0.04964)0.394.2:0.00171,(((556666:0.0092,(546690:0.00579,(379518:0.00443,310485:0.01528)0.887.2:0.0091)0.927.4:0.00405)0.835.2:0.00418,((148208:0.00865,(417515:0.00994,531674:0.01571)0.372:0.00183)0.960.12:0.00614,(((279715:0.00827,(((((249009:0.00125,291382:0.01616)0.927.5:0.00434,(538570:0.02038,(527048:0.00268,371761:0.00184)0.998.20:0.01137)0.282:0.0007)0.983.13:0.00587,136858:0.00995)0.480.2:0.00143,(207157:0.00417,277710:0.01036)0.896.9:0.00467)0.941.9:0.00587,510745:0.01128)0.738.7:0.00107)0.336:0.00153,91806:0.0168)0.813.3:0.00169,592423:0.01044)0.999.58:0.01416)0.946.9:0.00656)0.986.10:0.00878,(((573834:0.01475,(((259900:0.00453,(230342:0.01603,593992:0.025)0.953.3:0.00396)0.758.4:0.00014,(307298:0.01212,(((269755:0.01757,311254:0.00883)0.652.2:0.00384,542975:0.01446)0.873.2:0.00235,((240200:0.00704,240435:0.01509)0.949.6:0.00431,161643:0.0249)0.941.10:0.00463)0.772.2:0.00165)0.971.9:0.0101)0.903.5:0.00707,((((80013:0.00484,(316184:0.01299,(263950:0.00797,324987:0.00149)0.988.17:0.0082)0.568:0.00015)0.990.8:0.0073,(319282:0.00577,407095:0.01213)0.746:0.00405)0.879.6:0.00404,342044:0.0139)0.882.6:0.00599,59851:0.02793)0.981.11:0.00804)0.838.4:0.00424)0.847.5:0.00371,143961:0.01719)0.845.6:0.00204,((((200933:0.00245,440080:0.0048)0.906.4:0.00634,(((237108:0.01317,(((((((538790:0.01667,(152865:0.00293,204293:0.00723)0.814.4:0.00396)0.186.2:0.00015,(529444:0.00698,499844:0.00734)0.994.16:0.01169)0.989.10:0.0074,37441:0.00545)0.979.16:0.00459,(240821:0.00692,(205559:0.01014,270034:0.00655)0.887.3:0.00614)0.666.2:0.00178)0.927.6:0.00329,(223883:0.00482,132604:0.00729)0.817.3:0.00554)0.509:0.00016,161253:0.01457)0.786.3:0.00225,(217525:0.01024,(320871:0.00901,140315:0.00787)0.943.13:0.00327)0.961.10:0.00554)0.919.4:0.00015)0.938.8:0.00239,((((114159:0.01945,((((103201:0.04506,(((99439:0.02293,147737:0.00825)0.915.6:0.00042,(525029:0.01142,142011:0.01622)0.000.10:0.00028)0.000.11:0.00028,((231493:0.00765,363337:0.01213)0.861.4:0.00774,368639:0.02752)0.908.7:0.00238)0.993.15:0.00016)1.000.198:0.00061,((((((94370:0.01308,250101:0.01415)0.918.3:0.00444,569187:0.00487)0.971.10:0.00652,(110418:0.00542,(305962:0.03717,(264698:0.01149,499905:0.0153)0.926.12:0.00295)0.941.11:0.00348)0.941.12:0.00314)0.762.5:0.00114,12221:0.01372)0.935.4:0.00334,(348790:0.01624,(144540:0.00088,((97664:0.00417,300958:0.00351)0.977.15:0.00742,178912:0.00932)0.778.4:0.00142)1.000.199:0.00956)0.922.2:0.00015)0.739.6:0.0008,111124:0.01034)0.925.6:0.00263)0.916.8:0.00184,556676:0.03997)0.916.9:0.00051,((205717:0.01629,(573439:0.00979,(241903:0.01143,(((210760:0.01694,231226:0.01252)0.859.4:0.0015,78299:0.0011)0.905.3:0.00385,267000:0.0079)0.867.4:0.00154)0.948.5:0.00014)1.000.200:0.00015)1.000.201:0.00063,(((12253:0.01028,(557633:0.02698,356171:0.0071)0.781.3:0.00229)0.769:0.00077,((263590:0.01717,12257:0.01408)0.887.4:0.00166,((12214:0.00639,12220:0.00871)0.777.2:0.00075,(((553830:0.08041,12216:0.00014)0.850.5:0.00542,(70836:0.01886,540933:0.01781)0.132:0.00359)0.828.2:0.00014,(44928:0.02407,71669:0.00266)0.867.5:0.00093)1.000.202:0.00121)0.797.3:0.00386)0.930.6:0.00155)0.910.6:0.00016,587696:0.01899)0.901.3:0.00014)0.916.10:0.00052)0.916.11:0.00051)0.937.5:0.00413,(146721:0.00952,(543467:0.02031,(302479:0.0201,155177:0.03425)0.981.12:0.01206)0.821.3:0.00851)0.834.6:0.00483)0.998.21:0.01176,(288552:0.00903,((133578:0.00552,(283386:0.0077,(68745:0.00029,586126:0.00652)0.980.11:0.00507)0.724.2:0.00233)0.761.2:0.00093,(12284:0.00921,(350174:0.00769,546764:0.00555)0.995.18:0.01586)0.473.2:0.00267)0.889:0.00203)0.733.3:0.0007)0.966.9:0.00482,(258922:0.01853,(((((220783:0.00014,200217:0.01714)0.970.10:0.00401,139462:0.00847)0.872.11:0.00208,(365014:0.00619,12168:0.02249)0.816.6:0.00353)0.488:0.00984,268296:0.01524)0.809.4:0.00165,423397:0.01921)0.681.3:0.00302)0.975.6:0.00692)0.949.7:0.00389)0.646:0.00077,((310647:0.0073,577125:0.00734)0.951.10:0.00548,(((((89130:0.01507,94524:0.00488)0.908.8:0.00285,79881:0.0089)0.863.6:0.0016,(79897:0.00325,66587:0.0089)0.989.11:0.00632)0.735.2:0.00078,(149009:0.00898,((100011:0.04475,108713:0.00269)0.909.5:0.01304,82683:0.00556)0.745.6:0.0039)0.967.9:0.00474)0.964.12:0.00311,(((174021:0.01151,(587989:0.00108,83306:0.00012)0.970.11:0.00015)0.941.13:0.00895,(249999:0.02379,(197711:0.00865,226363:0.00267)0.809.5:0.00261)0.958.10:0.00499)0.873.3:0.00325,(((47210:0.00077,566849:0.00595)0.978.8:0.00391,(((((279373:0.01206,354551:0.00078)0.909.6:0.00321,132963:0.0076)0.421:0.0023,136675:0.00716)0.923.3:0.00238,(206162:0.00258,225374:0.00938)0.861.5:0.00156)0.836.5:0.00226,(105783:0.00349,113019:0.00943)0.666.3:0.00344)0.921.11:0.00015)0.777.3:0.00077,(((25558:0.01198,156083:0.00948)0.885.6:0.00157,218035:0.00789)0.769.2:0.00233,((((12264:0.00958,150282:0.00634)0.957.16:0.00015,(236702:0.01129,130238:0.00805)0.404.2:0.00232)0.950.20:0.00456,(223546:0.0039,233856:0.00254)0.943.14:0.00234)0.877.8:0.00171,191805:0.00785)0.932.7:0.00234)0.987.16:0.00248)0.901.4:0.00264)0.981.13:0.00515)0.401:0.00011)0.919.5:0.00315)0.963.7:0.00398)0.952.9:0.00492,((223593:0.0032,(230900:0.00939,295166:0.0008)1.000.203:0.01743)0.991.14:0.00731,(106397:0.04032,329994:0.00592)1.000.204:0.00971)0.822.3:0.00155)0.951.11:0.00434,(556645:0.01519,(556066:0.0121,(((585547:0.00919,(100282:0.00127,140229:0.0087)0.784.4:0.00598)0.991.15:0.01007,((30568:0.00082,12181:0.00783)0.264.2:0.00723,(95950:0.00312,(277142:0.02266,(12185:0.00523,12186:0.02167)0.372.2:0.00534)0.946.10:0.00636)0.980.12:0.00406)0.992.13:0.00567)0.791.3:0.0014,((71352:0.00588,241770:0.00901)0.949.8:0.00014,526926:0.03372)0.864.6:0.00014)0.979.17:0.00387)0.897.4:0.00701)0.994.17:0.00865)0.907.9:0.00368)0.954.4:0.00499)0.930.7:0.00542)0.992.14:0.01268)0.949.9:0.00579)0.773.4:0.00218,(((((((593699:0.0003,(574432:0.01021,321316:0.00811)0.867.6:0.00156)0.780.3:0.00075,343748:0.01059)0.769.3:0.00109,(556235:0.00606,((((559476:0.01805,65431:0.01515)0.936.14:0.0041,(149050:0.01246,(239951:0.01443,(((12369:0.02025,219876:0.01213)0.789.4:0.00342,585725:0.08048)0.836.6:0.00015,344945:0.00783)0.892.7:0.00476)0.502:0.00235)0.905.4:0.00315)0.877.9:0.00336,40964:0.01103)0.897.5:0.00016,((248784:0.00781,((588981:0.01259,211794:0.00014)0.792.5:0.00155,(106365:0.006,12364:0.00274)0.766.7:0.00374)0.902.3:0.00402)0.726.2:0.00155,((329045:0.01191,(541380:0.0047,559935:0.00327)0.854.3:0.00155)0.939.6:0.00627,((206642:0.00084,230798:0.00247)0.996.22:0.00057,(236808:0.02688,446480:0.02713)0.831:0.00196)0.953.4:0.00567)0.515:0.00014)0.980.13:0.00408)0.796.2:0.00237)0.379:0.00103)0.551:0.00237,133467:0.01724)0.188:0.00156,(153776:0.00787,20014:0.01587)0.617:0.00605)0.876.4:0.00621,((((40713:0.01694,((((299284:0.00896,(171867:0.00893,(161687:0.01318,12530:0.01928)1.000.205:0.00586)0.004:0.00911)0.877.10:0.00016,526633:0.00435)0.602:0.00472,113648:0.02183)0.932.8:0.0032,(431175:0.01868,(245679:0.00558,274278:0.00957)0.975.7:0.00556)0.733.4:0.00072)1.000.206:0.02006)0.903.6:0.00477,(575780:0.01032,(33876:0.00865,62967:0.0187)0.895.8:0.00373)0.587:0.00401)0.887.5:0.00686,((138116:0.02567,(187652:0.00917,(86198:0.01179,12541:0.01157)0.993.16:0.01118)0.900.4:0.00441)0.579:0.00383,((((12538:0.02069,(384657:0.02044,369064:0.01386)0.501:0.00572)0.749.2:0.0013,(149935:0.00855,566454:0.00865)0.997.23:0.01142)0.923.4:0.0056,240850:0.0121)0.939.7:0.00625,282092:0.00711)0.983.14:0.00814)0.865.6:0.0049)0.918.4:0.00364,(109814:0.01988,((237179:0.01565,((((531989:0.01522,(138659:0.0066,138602:0.01651)0.780.4:0.00137)0.820.3:0.00283,(499956:0.01205,(554402:0.00521,157133:0.00856)0.998.22:0.0102)0.931.9:0.00384)0.995.19:0.01497,500170:0.01349)0.987.17:0.00759,513203:0.00908)0.358:0.00342)0.343:0.00572,285415:0.01791)0.838.5:0.00622)0.669.2:0.00029)0.938.9:0.00472)0.988.18:0.00605,((((((429838:0.01595,(318628:0.01093,254649:0.01005)0.852:0.00282)0.838.6:0.00207,((((12315:0.00531,47463:0.01358)0.750.4:0.00941,556569:0.01021)0.963.8:0.00669,((((((272994:0.00392,557925:0.00731)0.968.9:0.00382,(101632:0.00493,(255914:0.0094,525081:0.01013)0.945.12:0.00395)0.960.13:0.00465)0.895.9:0.00304,(338568:0.01323,550104:0.00824)0.922.3:0.00306)0.955.9:0.00563,(235695:0.01366,101233:0.0073)0.677:0.00078)0.669.3:0.00063,(((161011:0.00341,532097:0.00195)0.977.16:0.02783,(252063:0.02623,((((289962:0.00481,12357:0.00333)0.937.6:0.00576,551705:0.00974)0.984.13:0.00615,(188500:0.01063,215512:0.01204)0.729.2:0.00297)0.950.21:0.00544,571690:0.01367)0.831.2:0.00317)0.993.17:0.00887)0.292.2:0.00337,((12343:0.01023,12346:0.00571)0.934.8:0.00406,(159316:0.00244,103732:0.01587)0.844.5:0.00148)0.873.4:0.00406)0.646.2:0.0098)0.840.3:0.00275,(204777:0.00121,39890:0.013)0.903.7:0.00315)0.877.11:0.00638)0.763.3:0.00133,((411183:0.00014,304693:0.0012)0.910.7:0.00578,280631:0.01437)0.886.7:0.00457)0.958.11:0.00429)0.982.16:0.00741,(103561:0.0028,39735:0.01369)0.971.11:0.00489)0.681.4:0.00883,((542896:0.0089,(315761:0.00608,(568991:0.01619,244196:0.00743)0.000.12:0.00147)0.983.15:0.00784)0.995.20:0.01044,(12542:0.01894,(((232214:0.01429,(509013:0.00606,511138:0.01126)0.546:0.00148)0.958.12:0.00451,(139651:0.01098,241541:0.00666)0.960.14:0.00417)0.000.13:0.00015,(147441:0.02017,400183:0.00822)0.999.59:0.01315)0.368.2:0.00572)0.945.13:0.00451)0.840.4:0.00245)0.707.3:0.00069,(((573103:0.01477,73335:0.00643)1.000.207:0.01688,((543089:0.01919,((105360:0.00459,587623:0.00389)0.766.8:0.00248,((152486:0.01082,153995:0.00808)0.928.5:0.00016,285868:0.00583)0.995.21:0.0032)0.729.3:0.00347)0.930.8:0.00546,(546946:0.00905,(142602:0.0359,114038:0.01567)1.000.208:0.03379)0.767.6:0.00436)0.976.10:0.00775)0.786.4:0.00267,(((78074:0.01324,545986:0.005)0.988.19:0.01092,294021:0.03803)0.894.11:0.00744,(((((582466:0.01444,((12592:0.01654,71888:0.00311)0.996.23:0.01336,81251:0.01485)0.857.5:0.00835)0.986.11:0.01468,((((((33696:0.00969,192142:0.02521)0.329:0.00274,(12590:0.00729,534592:0.00967)0.928.6:0.00429)0.568.2:0.00202,31123:0.00922)0.988.20:0.00646,(547904:0.00475,238830:0.0075)0.902.4:0.00232)0.096.2:0.00015,531040:0.01078)0.805.7:0.00306,(245707:0.01058,(112522:0.00728,112481:0.00862)0.998.23:0.0129)0.984.14:0.01296)0.988.21:0.01815)0.833.3:0.00549,((527314:0.0034,522040:0.00082)0.976.11:0.00709,((92262:0.01231,585167:0.014)1.000.209:0.02723,83011:0.01217)0.813.4:0.00942)0.454:0.00469)0.899.5:0.0066,(215248:0.00258,((514203:0.00893,384164:0.0073)0.991.16:0.0072,528198:0.00346)0.847.6:0.00374)0.999.60:0.01472)0.987.18:0.0123,((303554:0.01608,(181971:0.00204,109194:0.01161)0.964.13:0.00923)1.000.210:0.02114,((((79234:0.03788,((469488:0.00174,(((551774:0.00941,28461:0.0131)0.775.3:0.01039,((58541:0.00848,(278919:0.00644,((585106:0.00206,26061:0.00539)0.973.10:0.00659,(569067:0.00014,583523:0.00329)0.952.10:0.00648)0.568.3:0.00249)0.989.12:0.0114)0.377.2:0.00466,((584347:0.02003,92289:0.00764)0.982.17:0.01214,((492389:0.01376,561935:0.00345)0.997.24:0.01421,((274643:0.0573,130655:0.00078)0.756.2:0.02167,(551431:0.01054,(362473:0.00709,(566233:0.00479,103151:0.00873)0.725.2:0.00314)0.997.25:0.01224)0.854.4:0.00772)0.585:0.00818)0.660:0.00704)0.625:0.00618)0.933.6:0.00673)0.956.9:0.00014,((106166:0.00014,436034:0.01259)0.930.9:0.00218,(541516:0.00539,(((23149:0.00811,(188589:0.00088,547861:0.00447)0.882.7:0.00414)0.998.24:0.01124,571103:0.00725)0.922.4:0.00404,(24150:0.00312,49536:0.0069)0.750.5:0.00077)0.874.8:0.00187)0.715.4:0.00155)0.966.10:0.00358)0.903.8:0.00326)0.953.5:0.00657,553115:0.00787)0.865.7:0.00627)0.971.12:0.01298,(565742:0.02571,(33488:0.00089,591960:0.00014)0.997.26:0.01815)0.999.61:0.03139)0.797.4:0.01132,((19270:0.03154,541752:0.01946)0.859.5:0.00935,(((546280:0.00076,139504:0.00011)0.924.6:0.00727,588670:0.0355)1.000.211:0.03158,((470693:0.04668,((139948:0.01203,131712:0.00083)1.000.212:0.04608,(139969:0.00016,131783:0.00107)0.929.9:0.03875)0.878.8:0.01605)0.972.15:0.0116,((364752:0.01417,(202640:0.01487,554616:0.01482)0.959.11:0.01065)1.000.213:0.0271,(12568:0.03105,((569912:0.01383,((152869:0.00386,((153783:0.00529,90040:0.01684)0.998.25:0.01178,168533:0.00818)0.975.8:0.00634)0.952.11:0.00453,((561441:0.00706,29001:0.00542)1.000.214:0.01355,((23235:0.00131,((394801:0.00779,370432:0.01508)0.898.9:0.00311,(60136:0.00765,82476:0.01214)0.868.9:0.00168)0.862.4:0.00222)0.961.11:0.0046,((50842:0.00079,44425:0.00171)1.000.215:0.01084,437735:0.0104)0.863.7:0.00189)0.421.2:0.00078)0.900.5:0.0033)0.245:0.00016)0.855.5:0.00399,(12569:0.01124,560661:0.02414)0.937.7:0.00558)0.857.6:0.00473)0.979.18:0.01186)0.963.9:0.00966)0.889.2:0.00614)0.897.6:0.00815)0.591:0.00666)0.865.8:0.0064,(533159:0.01376,533806:0.02197)0.607:0.00555)0.073.2:0.00298)0.503:0.00867)0.984.15:0.01524)0.926.13:0.00507)0.981.14:0.00882)0.848.4:0.00723,(((464060:0.01667,(247720:0.00014,554922:0.00016)0.391:0.01351)0.949.10:0.00621,((417900:0.01341,((100122:0.0065,(114941:0.00317,((99934:0.00565,566396:0.00106)1.000.216:0.00203,580377:0.03308)0.871.8:0.00334)0.612:0.00414)0.784.5:0.00151,(22581:0.00414,(140760:0.03932,149651:0.00565)0.912.4:0.0051)0.948.6:0.00479)0.999.62:0.01247)0.885.7:0.00307,(((531300:0.0036,(174602:0.01177,529325:0.01502)0.982.18:0.0057)0.911.6:0.00337,261705:0.00947)0.919.6:0.00312,172245:0.00766)0.867.7:0.00222)0.520.2:0.00227)1.000.217:0.0272,(388134:0.01376,((530345:0.02155,591049:0.00948)1.000.218:0.02033,(((262645:0.01129,((457360:0.0059,((230376:0.0059,183694:0.00518)0.908.9:0.00327,((112715:0.01273,323263:0.01303)0.979.19:0.00915,((507146:0.00184,(103120:0.01199,136948:0.00967)0.977.17:0.00752)0.045:0.00619,273126:0.00309)0.379.2:0.00196)0.952.12:0.00643)0.413:0.00435)0.965.10:0.00451,12485:0.00898)0.571.2:0.00361)0.884.8:0.00359,182895:0.01081)0.999.63:0.01521,(((((232202:0.00342,(582705:0.02088,78681:0.00043)0.867.8:0.00015)0.947.13:0.00433,(((226642:0.00391,158234:0.00242)0.865.9:0.00157,((104744:0.01744,589054:0.00665)0.879.7:0.00274,((181936:0.00056,561067:0.02675)0.905.5:0.00098,110374:0.00496)0.889.3:0.01328)0.778.5:0.00104)0.957.17:0.00313,((((508890:0.03447,496942:0.03279)0.755.5:0.00101,567669:0.03138)0.263.3:0.00338,146560:0.00014)0.126:0.00619,((393934:0.01046,((538247:0.00015,543491:0.00572)0.240:0.00741,(400420:0.01934,471141:0.01155)0.994.18:0.0089)0.142:0.00111)0.987.19:0.00682,(588712:0.0084,224448:0.00924)0.960.15:0.00473)0.927.7:0.00446)0.904.6:0.0027)0.748.3:0.00077)0.790.3:0.00155,318338:0.01159)0.939.8:0.00394,(((151713:0.0045,(72222:0.00042,(302819:0.04518,(144051:0.01832,(((((335422:0.02523,(302973:0.03579,341415:0.03674)0.764.4:0.00086)0.629:0.003,289799:0.02862)0.989.13:0.00015,(290111:0.00928,289923:0.00543)0.353:0.00427)0.826:0.00078,299752:0.00396)0.860.9:0.00789,311160:0.02451)0.772.3:0.0036)0.550.2:0.00199)0.767.7:0.00161)0.730.2:0.00416)0.880.6:0.0018,(102954:0.01746,235825:0.01245)0.690:0.00225)0.733.5:0.00048,237889:0.04012)0.839.3:0.00188)0.883.9:0.00343,(68681:0.00699,((((44949:0.0107,(210986:0.00828,(((144771:0.10222,564646:0.1338)0.954.5:0.01649,(536480:0.00238,(252119:0.0265,(130481:0.05106,568913:0.02312)1.000.219:0.01956)0.987.20:0.0121)0.795.5:0.00171)0.573.3:0.00064,(200183:0.05956,568886:0.02284)0.109.2:0.0021)0.880.7:0.00168)0.771.3:0.00312)0.832.4:0.00144,287197:0.00788)0.866.3:0.00156,(569667:0.01255,((71728:0.00502,((((292918:0.01948,290069:0.01585)0.922.5:0.00046,107123:0.01126)0.921.12:0.00041,12457:0.01324)0.780.5:0.00077,(581042:0.02901,239892:0.00155)0.946.11:0.00233)0.991.17:0.00478)0.839.4:0.00176,(((152113:0.00465,383501:0.00638)0.812.3:0.00305,369524:0.00787)0.909.7:0.00235,584662:0.00647)0.835.3:0.00014)0.913.7:0.00321)0.953.6:0.00489)0.756.3:0.00089,(((12415:0.01765,(((554201:0.01115,543841:0.00354)0.872.12:0.00153,(244416:0.00353,(107764:0.01135,252828:0.02227)0.904.7:0.00532)0.975.9:0.0052)0.925.7:0.00235,240314:0.01096)0.850.6:0.00153)0.847.7:0.00077,(((191973:0.00251,((104706:0.0141,189233:0.00042)1.000.220:0.00633,248026:0.00701)0.999.64:0.0017)0.924.7:0.00785,412672:0.17099)0.999.65:0.00014,((((12408:0.00934,((136467:0.01456,((39179:0.00382,188551:0.03802)0.398:0.00822,200356:0.00858)0.874.9:0.00014)1.000.221:0.00016,(273134:0.01691,261709:0.01283)0.861.6:0.00519)0.792.6:0.0022)0.906.5:0.00016,((20720:0.02679,(197304:0.02355,12398:0.02225)0.903.9:0.00493)0.975.10:0.00714,279258:0.00797)0.000.14:0.00039)0.870.10:0.00016,((33821:0.01006,541256:0.01886)0.856.4:0.00487,(200110:0.02026,537426:0.02118)0.901.5:0.00732)0.667:0.00015)1.000.222:0.00079,(540565:0.01561,((137766:0.00823,274622:0.02821)0.875.5:0.00014,16350:0.0069)1.000.223:0.00015)0.786.5:0.00169)0.933.7:0.0017)0.612.2:0.00059)0.957.18:0.00362,(48874:0.00907,(251817:0.00337,(12434:0.00988,(101943:0.01164,(161042:0.01639,256966:0.00427)0.987.21:0.00467)0.925.8:0.00277)0.915.7:0.00281)0.990.9:0.00662)0.718.5:0.00155)0.872.13:0.0043)0.943.15:0.00301)0.907.10:0.00014)0.916.12:0.00414)0.779.3:0.00448)0.849.4:0.00376)0.993.18:0.01222)0.847.8:0.00647)0.976.12:0.00842)0.913.8:0.00512)0.964.14:0.00898)0.961.12:0.01359,(((268395:0.01019,((528131:0.01473,(276870:0.00478,34366:0.01432)0.810.3:0.007)0.950.22:0.00639,((106801:0.01623,160141:0.01411)0.183.2:0.00014,556233:0.01939)0.923.5:0.008)0.981.15:0.00832)0.930.10:0.00563,(402059:0.03519,((((((((320455:0.01259,111980:0.02231)0.303:0.00343,(513129:0.00357,359533:0.00409)0.371:0.0249)0.628:0.00311,435665:0.02349)0.891.6:0.00334,(544480:0.00964,(587045:0.0203,13401:0.0132)0.989.14:0.00965)0.902.5:0.00479)0.833.4:0.00411,((144223:0.00106,524627:0.01499)0.813.5:0.01179,(86061:0.01034,145619:0.0049)0.782.4:0.00166)0.983.16:0.00696)0.873.5:0.00202,((((13460:0.0063,(13459:0.00632,202911:0.06237)0.957.19:0.00313)0.743.2:0.02475,((13463:0.00803,((553111:0.00014,550470:0.00169)0.056:0.00947,(558402:0.00431,13467:0.0008)0.998.26:0.01233)0.951.12:0.00604)0.500.2:0.00169,(((((470178:0.01053,470221:0.00107)0.341:0.00725,((50265:0.00014,552026:0.00028)0.976.13:0.00573,(115099:0.00719,551636:0.01462)0.771.4:0.00157)0.933.8:0.00447)0.958.13:0.00449,(13438:0.00805,(503291:0.00183,((103750:0.00573,(239006:0.01287,527199:0.01811)0.881.6:0.00238)0.912.5:0.00344,(358019:0.01032,250779:0.01933)0.552.3:0.00347)0.999.66:0.0139)0.903.10:0.01295)0.501.2:0.00081)0.736.2:0.0008,(((492853:0.0161,(((544910:0.0184,(((374036:0.00607,13424:0.00914)0.924.8:0.00014,357432:0.00446)0.894.12:0.00091,408733:0.0276)1.000.224:0.00258)0.886.8:0.00467,(13419:0.00743,(25821:0.01699,(470207:0.00423,(272078:0.00218,13452:0.00658)0.903.11:0.00432)0.350.2:0.00152)0.919.7:0.00472)0.924.9:0.00587)0.869.10:0.00313,395362:0.03013)0.895.10:0.00653)0.255:0.0035,((534507:0.00094,463296:0.01437)0.969.14:0.00552,((462778:0.02166,427604:0.00255)0.872.14:0.0016,545862:0.00307)0.973.11:0.00547)0.929.10:0.00385)0.929.11:0.0041,((470190:0.00403,73590:0.01055)0.984.16:0.00608,376148:0.00901)0.823:0.0017)0.996.24:0.00774)0.938.10:0.00393,((397782:0.02229,((471130:0.00085,410908:0.01992)0.872.15:0.00609,((465447:0.00193,435639:0.02682)1.000.225:0.01341,271035:0.00822)0.295:0.00149)0.867.9:0.00451)0.090:0.00201,((450611:0.02624,(13428:0.01118,242625:0.0255)0.931.10:0.00746)0.893.3:0.00465,(((106751:0.02378,65332:0.03959)0.988.22:0.01481,((393598:0.03552,((27896:0.00016,583448:0.00014)0.998.27:0.01175,90728:0.01814)0.976.14:0.00892)0.965.11:0.00711,255771:0.01268)0.883.10:0.00515)0.949.11:0.00634,(396732:0.03111,((557799:0.00014,132506:0.0012)0.899.6:0.00666,470200:0.00658)1.000.226:0.0294)0.522.2:0.00213)0.905.6:0.00415)0.925.9:0.00484)0.906.6:0.00478)0.911.7:0.00296)0.300:0.00185)0.704.4:0.00173,(107869:0.00649,((13361:0.02338,218364:0.00232)0.829.3:0.00414,13365:0.00584)0.958.14:0.0042)0.998.28:0.00932)0.865.10:0.00244,(521762:0.01088,(235898:0.00425,((25532:0.00388,(((((211156:0.00689,495361:0.00028)0.994.19:0.00015,69384:0.01402)0.185:0.00595,(361871:0.00399,548125:0.02391)0.891.7:0.007)0.765.3:0.0019,(13411:0.00561,(471202:0.00106,13393:0.02083)0.922.6:0.0049)0.905.7:0.00471)0.082:0.00014,37997:0.0099)0.975.11:0.00557)0.971.13:0.0053,(104380:0.01943,18932:0.02611)0.860.10:0.00277)0.871.9:0.00334)0.866.4:0.00163)0.888.2:0.00267)0.307.2:0.00322)0.744.3:0.00209,499172:0.01846)0.908.10:0.005,(((103471:0.0078,(413048:0.03233,13367:0.00733)0.977.18:0.00551)0.903.12:0.00289,(((81096:0.01358,405759:0.00366)0.976.15:0.00599,337789:0.00559)0.723.3:0.00737,137456:0.00784)0.378:0.00157)0.941.14:0.00357,(26444:0.00923,(522984:0.01729,(457782:0.00932,380412:0.02026)1.000.227:0.01931)0.994.20:0.00937)0.839.5:0.00259)0.860.11:0.00412)0.997.27:0.01609)0.266.2:0.00358)0.958.15:0.00831,((((((((165440:0.0176,575299:0.0009)1.000.228:0.00547,(47805:0.01618,(((37109:0.0184,256702:0.01374)0.911.8:0.00071,(((573976:0.0157,((257679:0.00981,(241543:0.0585,240211:0.03211)0.813.6:0.00386)0.705:0.00059,202486:0.00551)0.887.6:0.00014)0.804.4:0.00584,(13581:0.00028,152649:0.02912)0.910.8:0.01239)0.446:0.00159,(546404:0.00186,343242:0.00029)0.997.28:0.01154)0.460.2:0.00169)1.000.229:0.00015,536612:0.00427)0.378.2:0.02029)0.970.12:0.00561)0.879.8:0.00319,(((155985:0.00028,13518:0.01847)0.000.15:0.01324,297451:0.0116)0.900.6:0.00029,13509:0.00135)0.850.7:0.00147)0.863.8:0.00015,(252284:0.00092,81710:0.00042)0.994.21:0.00782)0.878.9:0.00157,(((409497:0.00993,141953:0.00263)0.999.67:0.01951,132913:0.01742)0.673:0.00506,((551777:0.00945,237519:0.00078)0.735.3:0.0003,(84268:0.0064,(185640:0.01673,(79112:0.0079,(((549729:0.00515,(63690:0.02495,557103:0.00418)0.769.4:0.00076)0.978.9:0.00936,(106832:0.00061,276278:0.00983)0.876.5:0.00029)0.872.16:0.00161,((((188084:0.00722,82059:0.00727)0.760.4:0.00916,(238797:0.00973,(82373:0.01588,156449:0.00973)0.719:0.00078)0.772.4:0.00809)0.926.14:0.00347,(81999:0.00956,412821:0.01791)0.978.10:0.00637)0.837.4:0.00437,(13616:0.01072,(((43786:0.01022,148911:0.00327)0.741.4:0.00416,137712:0.02043)0.786.6:0.00627,(13634:0.00073,532451:0.01184)0.791.4:0.00397)0.609:0.00118)0.975.12:0.00739)0.865.11:0.00159)0.954.6:0.00485)0.985.10:0.00462)0.946.12:0.00342)0.965.12:0.00785)0.967.10:0.00501)0.913.9:0.00281)0.899.7:0.00294,(((78149:0.01828,571864:0.02126)1.000.230:0.03049,((((((241605:0.00311,526776:0.0004)0.982.19:0.00511,594167:0.0053)0.976.16:0.00687,(139959:0.00217,142287:0.00028)0.755.6:0.00606)0.899.8:0.00334,236778:0.01705)0.981.16:0.0079,(((((568873:0.01082,571393:0.01448)0.820.4:0.00223,127640:0.07029)0.998.29:0.00015,158459:0.00323)0.896.10:0.00029,(((322848:0.01268,(104596:0.00982,((154902:0.01172,239824:0.00399)0.782.5:0.00695,13302:0.01274)0.851.5:0.0025)0.809.6:0.00149)0.729.4:0.00082,(399970:0.01516,558953:0.01877)0.860.12:0.00308)0.965.13:0.00392,(579296:0.00687,192556:0.01465)0.746.2:0.00081)0.921.13:0.00966)0.890.7:0.0022,(579502:0.00816,459006:0.00076)1.000.231:0.00197)0.765.4:0.001)0.994.22:0.00932,99032:0.00644)0.829.4:0.00233)0.964.15:0.00568,(((214610:0.0259,590555:0.00657)0.902.6:0.00721,(146223:0.02376,(103057:0.01778,103349:0.0032)0.988.23:0.01008)0.352:0.00302)0.975.13:0.00869,13522:0.00669)0.570.3:0.00312)0.738.8:0.00015)0.699:0.00355,100442:0.03286)0.934.9:0.00476,(514259:0.02065,((414945:0.01477,(571264:0.00458,566868:0.01893)0.988.24:0.01076)0.937.8:0.00631,(((13069:0.00822,(112844:0.0097,255860:0.00098)0.976.17:0.00732)0.918.5:0.00438,((((590657:0.00328,(276601:0.00856,365607:0.01133)1.000.232:0.00547)0.152:0.0034,((28966:0.01249,((((197530:0.01013,(281424:0.0197,(((13115:0.00158,215776:0.01424)0.993.19:0.00867,(535768:0.00546,563608:0.01003)0.826.2:0.00673)0.899.9:0.00107,13081:0.00551)0.956.10:0.00567)0.859.6:0.00078)0.998.30:0.00014,(((574504:0.02508,594108:0.00733)0.733.6:0.00063,262897:0.02228)0.932.9:0.00381,99103:0.0156)0.940.9:0.00477)0.938.11:0.00224,114858:0.0133)0.912.6:0.00386,13250:0.00203)0.909.8:0.00085)0.713.2:0.00233,((13035:0.01337,13034:0.01319)0.679:0.0017,(((154552:0.00921,214595:0.01761)0.905.8:0.00706,104109:0.00842)0.122:0.00236,(207437:0.00222,174903:0.00384)0.992.15:0.01222)0.883.11:0.00278)0.873.6:0.00464)1.000.233:0.00016)1.000.234:0.00015,243988:0.0223)0.808.2:0.00236,((((300187:0.00872,(((556866:0.00914,13239:0.00053)0.942.12:0.00015,(166876:0.00543,(99500:0.02429,538038:0.00533)0.743.3:0.00076)0.786.7:0.00155)0.876.6:0.00155,202331:0.00574)0.861.7:0.00154)0.881.7:0.00187,((241158:0.00735,298629:0.01109)0.880.8:0.00014,(92235:0.00618,(13236:0.00399,13162:0.00249)0.531:0.00791)0.966.11:0.00467)0.910.9:0.00217)0.363.2:0.00014,(563112:0.00016,(143735:0.00653,523221:0.00921)0.877.12:0.00176)0.927.8:0.0054)0.058.2:0.00569,(93004:0.01587,(254659:0.00743,71298:0.01325)0.760.5:0.01087)0.916.13:0.00402)0.711:0.00097)0.863.9:0.00202)0.974.8:0.00715,225840:0.01666)0.997.29:0.01105)0.763.4:0.00496)0.958.16:0.00619)0.591.2:0.00336)0.999.68:0.0262)0.917.5:0.01698,(347376:0.08327,((((591851:0.01938,(86784:0.02488,12619:0.0065)0.920.6:0.00875)0.984.17:0.01438,(72858:0.00436,143171:0.01118)1.000.235:0.02392)0.592.4:0.01504,(203691:0.02573,(87249:0.0298,78555:0.02168)1.000.236:0.02467)0.734.5:0.00663)0.752.3:0.00708,(((300734:0.00743,243420:0.05308)0.481.3:0.00991,257642:0.0269)0.334:0.00194,((((568235:0.01129,514327:0.00646)0.760.6:0.00167,((((567846:0.01213,540945:0.00143)0.463.3:0.00015,367330:0.0047)0.906.7:0.00029,(((193082:0.00972,584375:0.00597)0.000.16:0.00429,326977:0.00232)0.995.22:0.01404,49417:0.02748)0.473.3:0.00236)0.954.7:0.00015,(102049:0.01173,471180:0.00015)0.868.10:0.00703)0.952.13:0.00558)0.941.15:0.00585,(570451:0.04473,((311636:0.01376,321598:0.01614)0.942.13:0.00744,339532:0.00333)0.899.10:0.00462)0.181:0.0024)0.969.15:0.00583,((561644:0.02272,531390:0.02037)0.920.7:0.00421,(((((309178:0.00681,542747:0.00014)0.745.7:0.00578,(533785:0.00284,471185:0.01045)0.961.13:0.00587)0.941.16:0.00548,(532979:0.02313,(295384:0.00914,581027:0.01477)0.906.8:0.00336)0.528:0.00221)0.998.31:0.01117,(12650:0.00689,((484303:0.00695,105328:0.00879)0.889.4:0.006,292521:0.01046)0.973.12:0.00465)0.985.11:0.00622)0.850.8:0.00241,((469852:0.01191,24773:0.01188)0.768.2:0.00227,55846:0.0069)0.963.10:0.0044)0.715.5:0.00326)0.856.5:0.00177)0.865.12:0.00265)0.883.12:0.01115)1.000.237:0.05255)0.999.69:0.0531)0.862.5:0.01956,201720:0.12599)0.961.14:0.02259,((223296:0.02952,525417:0.13548)1.000.238:0.07676,(201147:0.08249,276582:0.02715)0.345:0.01738)0.980.14:0.02344)0.691.2:0.01083,243646:0.18743)0.973.13:0.02388,(((3793:0.18032,((240123:0.00602,21997:0.02017)1.000.239:0.13332,(((51888:0.0164,(38013:0.02127,65483:0.0264)0.966.12:0.0077)0.197.2:0.00237,(233469:0.0051,560932:0.00014)0.924.10:0.00932)0.999.70:0.04236,((108313:0.00956,541272:0.01272)1.000.240:0.03666,(540449:0.01774,(((561904:0.0087,(336104:0.00723,111131:0.01876)0.845.7:0.00554)0.996.25:0.01337,(((335847:0.01366,(108479:0.01272,66916:0.01047)0.961.15:0.00733)0.971.14:0.00801,566297:0.01423)0.096.3:0.00729,((203148:0.01424,(576883:0.00547,(203480:0.00264,203045:0.02189)0.932.10:0.01313)0.964.16:0.01018)0.957.20:0.00845,((99923:0.00505,540398:0.00058)0.978.11:0.00788,(535650:0.01171,229130:0.0037)0.921.14:0.00721)0.998.32:0.01695)0.342:0.00147)0.987.22:0.01245)0.439.3:0.00488,(235018:0.0257,230004:0.00219)0.992.16:0.01381)0.732.2:0.00144)0.665:0.00709)0.996.26:0.02753)1.000.241:0.07136)1.000.242:0.07429)0.884.9:0.02776,((((((358725:0.01085,(516411:0.00154,92338:0.02801)0.426.2:0.02045)1.000.243:0.13521,(37498:0.00782,(584646:0.02107,587682:0.01861)0.763.5:0.01081)1.000.244:0.09804)0.994.23:0.0389,(138974:0.11406,(250068:0.06193,((100106:0.00554,199848:0.03179)0.776.5:0.00966,(238964:0.00872,105069:0.03192)0.982.20:0.01183)0.974.9:0.01769)1.000.245:0.08123)0.736.3:0.02128)0.379.3:0.01043,((273317:0.03219,540658:0.02788)0.999.71:0.04065,(141715:0.01018,221327:0.02362)1.000.246:0.13394)0.503.2:0.01649)0.210.2:0.0185,(((((((((((((3957:0.01654,((121373:0.02347,((3921:0.02573,3918:0.0082)0.833.5:0.00224,124523:0.01966)0.530.2:0.00418)0.132.2:0.0038,3922:0.01238)0.946.13:0.00879)1.000.247:0.02902,3906:0.01217)0.808.3:0.01152,((299619:0.01692,(239628:0.01524,36733:0.00785)0.913.10:0.00497)0.907.11:0.00968,(3909:0.01421,143148:0.00832)0.642.2:0.00177)0.937.9:0.00638)0.986.12:0.01651,(302412:0.02866,((289341:0.01422,579730:0.0117)0.993.20:0.01376,(47380:0.01891,73875:0.01922)0.500.3:0.00572)0.808.4:0.00585)1.000.248:0.04626)0.479.2:0.00915,((88036:0.00108,3949:0.00653)0.986.13:0.01311,293340:0.05657)0.817.4:0.00976)0.957.21:0.01346,((3952:0.01737,((301091:0.01279,((236512:0.00977,((3931:0.01021,3947:0.0077)0.771.5:0.00148,(484437:0.00549,574315:0.01683)0.792.7:0.0009)0.905.9:0.00314)0.973.14:0.00503,240104:0.01717)0.887.7:0.00291)0.841.3:0.00248,(3943:0.01127,72610:0.01919)0.834.7:0.00521)0.817.5:0.00502)0.452:0.00215,(3956:0.00674,(3944:0.0078,3945:0.00904)0.913.11:0.00543)0.966.13:0.00725)0.998.33:0.01687)0.998.34:0.01843,(46681:0.01207,(251512:0.01307,527931:0.01585)0.933.9:0.00659)0.998.35:0.01831)0.992.17:0.02022,(((((((42877:0.02873,(301895:0.00121,83155:0.00097)1.000.249:0.02818)0.999.72:0.02484,(((563323:0.00437,24210:0.00217)0.933.10:0.07349,(((552289:0.04459,(134428:0.02464,3826:0.08144)0.686:0.01155)0.992.18:0.02622,((275514:0.00581,339907:0.00357)1.000.250:0.04218,(71422:0.01132,333752:0.00341)0.915.8:0.04737)0.977.19:0.02056)0.862.6:0.01077,(287530:0.00038,136296:0.02924)1.000.251:0.03587)0.999.73:0.02715)0.338:0.01736,((29328:0.00744,3860:0.00997)1.000.252:0.04012,((3852:0.00358,453529:0.00926)0.963.11:0.00736,(573932:0.03827,3859:0.01926)0.929.12:0.00664)0.990.10:0.01491)0.998.36:0.02867)0.744.4:0.00816)0.901.6:0.01159,17854:0.04297)0.658:0.01096,(104346:0.02913,((345571:0.01562,103814:0.01249)0.980.15:0.01224,(((289851:0.01069,297835:0.01003)0.998.37:0.02217,(((((107163:0.00907,(170551:0.01022,267013:0.00523)0.724.3:0.00157)0.777.4:0.00872,(103223:0.00555,535291:0.01337)0.738.9:0.00229)0.930.11:0.00417,(578240:0.00842,576403:0.00275)0.237:0.00172)1.000.253:0.02008,104881:0.04763)0.994.24:0.01881,((3836:0.02608,565720:0.01204)1.000.254:0.02443,(299900:0.00886,(105320:0.01236,291957:0.01656)0.939.9:0.005)0.990.11:0.01048)0.932.11:0.00437)0.954.8:0.00621)0.990.12:0.0142,(68837:0.02766,332359:0.01677)0.628.2:0.00451)0.723.4:0.01123)0.876.7:0.00708)0.092.2:0.01389)1.000.255:0.02317,(324511:0.06207,(572161:0.02285,((526483:0.00735,328771:0.02164)0.996.27:0.02037,(95168:0.01576,(566353:0.01557,3861:0.00755)0.877.13:0.00811)0.999.74:0.02556)0.967.11:0.01511)0.187:0.00683)0.594.2:0.0107)0.784.6:0.00266,((78066:0.01051,3984:0.02858)1.000.256:0.03912,(141897:0.06359,((36987:0.00779,((143866:0.00853,146570:0.0196)0.915.9:0.00728,(114660:0.00015,77899:0.01882)0.874.10:0.00607)0.419:0.00385)0.984.18:0.01764,((((30690:0.00658,(143039:0.02068,(22641:0.0056,46814:0.00578)0.777.5:0.00115)0.776.6:0.00134)0.871.10:0.00445,(3977:0.01239,3969:0.03852)0.989.15:0.01523)0.956.11:0.00941,3879:0.00866)0.996.28:0.01201,4127:0.00588)0.995.23:0.01936)0.973.15:0.01853)0.870.11:0.00734)0.771.6:0.00635)0.928.7:0.01176,342601:0.08213)0.975.14:0.01881)0.992.19:0.0207,(((569396:0.01529,(585733:0.02011,513030:0.00749)0.986.14:0.013)0.956.12:0.00809,(262448:0.005,106960:0.03501)0.777.6:0.00329)0.951.13:0.01118,((((((((77632:0.00436,4116:0.02065)1.000.257:0.01999,(114537:0.0078,78252:0.01284)0.875.6:0.00438)0.656.2:0.00411,78017:0.0131)0.975.15:0.01114,(536096:0.02099,240875:0.02259)1.000.258:0.03026)0.501.3:0.00782,((4024:0.01125,4015:0.01106)1.000.259:0.02835,(47692:0.0027,534577:0.01035)1.000.260:0.03064)0.922.7:0.01152)0.907.12:0.00862,(((40147:0.00901,22449:0.02298)0.998.38:0.0147,(69324:0.01022,(72963:0.01633,241126:0.02918)0.625.2:0.00523)0.871.11:0.00875)0.982.21:0.01495,(143523:0.02785,(114784:0.01576,142334:0.01292)0.963.12:0.01097)0.862.7:0.007)0.914.7:0.00718)0.830.3:0.00543,(135878:0.01779,77621:0.01313)0.973.16:0.00965)0.996.29:0.01608,(((((((78032:0.02246,69426:0.02805)0.728.3:0.0076,(142741:0.02331,78117:0.01753)0.993.21:0.01057)0.878.10:0.00349,146387:0.01446)0.866.5:0.00441,((4009:0.00316,78075:0.01852)0.929.13:0.00451,(77678:0.01672,((4101:0.01259,4099:0.01741)0.997.30:0.01461,((32372:0.00472,(4086:0.01101,135874:0.00404)0.903.13:0.00624)0.983.17:0.00905,114570:0.0069)0.933.11:0.00565)0.769.5:0.00323)0.890.8:0.00369)0.996.30:0.01127)0.871.12:0.00271,(143013:0.03489,((141999:0.00015,(4004:0.03444,42498:0.00731)0.914.8:0.00529)1.000.261:0.02249,(21434:0.0219,555083:0.01928)0.083.2:0.00555)0.259:0.00495)0.995.24:0.01358)0.860.13:0.00281,((78179:0.01791,(69779:0.02194,69543:0.01796)0.997.31:0.02184)0.990.13:0.01557,((74426:0.01779,539398:0.03702)0.848.5:0.01005,((((4049:0.01768,536459:0.00631)1.000.262:0.01793,(72131:0.00964,(145031:0.02404,71891:0.01024)0.890.9:0.00411)0.990.14:0.01302)0.977.20:0.01089,(69563:0.02507,148743:0.02733)0.922.8:0.00709)0.808.5:0.00406,70909:0.01993)0.960.16:0.00946)0.908.11:0.00862)0.883.13:0.007)0.977.21:0.00774,((536207:0.03064,(((((276247:0.02931,(147671:0.01795,150200:0.0189)0.901.7:0.01386)0.202:0.00421,143152:0.03711)0.094:0.00348,(78119:0.01801,78155:0.00852)1.000.263:0.04537)0.915.10:0.00755,((((78092:0.01949,(149915:0.01909,(77877:0.0142,((141505:0.0145,77749:0.00568)0.996.31:0.01323,(77582:0.00015,570521:0.01627)0.527:0.00661)0.951.14:0.00605)0.883.14:0.00301)0.674.3:0.00346)0.952.14:0.00692,(77953:0.02004,77948:0.01535)0.157:0.00424)0.954.9:0.00739,(4094:0.01618,((141771:0.00352,(145132:0.01249,(78296:0.00732,((77905:0.00175,(78395:0.00922,77600:0.0049)0.970.13:0.01282)0.849.5:0.00893,(44741:0.01251,4087:0.02864)0.602.2:0.00611)0.969.16:0.00785)0.939.10:0.00493)0.918.6:0.00606)0.959.12:0.00719,(141059:0.01208,((143023:0.00589,(114698:0.01012,77677:0.02925)0.939.11:0.01439)0.243:0.00727,((143612:0.01085,77811:0.00749)0.345.2:0.00301,(149948:0.00891,114589:0.01475)0.949.12:0.00678)0.924.11:0.00015)0.999.75:0.00831)0.946.14:0.00504)0.980.16:0.00911)0.953.7:0.00702)0.987.23:0.00915,(86557:0.02747,(((144082:0.00559,143879:0.01693)0.989.16:0.00675,144546:0.0027)0.998.39:0.01722,(143974:0.01555,143026:0.0168)0.988.25:0.01335)0.772.5:0.00413)0.953.8:0.00752)0.460.3:0.00336)0.695.2:0.00199,((((146784:0.0141,146440:0.01171)0.990.15:0.01694,(4098:0.01253,(135859:0.03453,141454:0.0023)0.995.25:0.01816)1.000.264:0.04261)0.812.4:0.00859,((69074:0.01211,(70817:0.0093,44787:0.02784)0.102:0.00774)0.999.76:0.02579,((((143938:0.01938,74331:0.023)0.814.5:0.00369,(145168:0.02085,((78213:0.01011,((112977:0.03172,203550:0.01924)0.893.4:0.00506,((141191:0.00712,222726:0.02115)0.753.5:0.0022,141692:0.00817)0.707.4:0.00105)0.917.6:0.00444)0.887.8:0.0037,(73924:0.03024,(252938:0.02003,147373:0.01005)0.913.12:0.00459)0.979.20:0.00903)0.347:0.0024)0.925.10:0.00586)0.796.3:0.00559,((589527:0.0185,(48891:0.00015,574881:0.00516)0.951.15:0.00532)0.999.77:0.02419,(40667:0.01641,143978:0.0245)0.899.11:0.01223)0.731.2:0.00807)0.975.16:0.01068,((((((135860:0.02538,(((38586:0.00186,554108:0.00084)1.000.265:0.02098,((591066:0.01459,(260367:0.02015,(582361:0.00384,35642:0.02694)0.631.2:0.00318)0.992.20:0.00985)0.848.6:0.00408,((36150:0.0035,(25068:0.01534,547924:0.01958)0.974.10:0.00657)0.909.9:0.00326,((583190:0.02039,590884:0.01524)0.959.13:0.0098,(17555:0.01033,16871:0.0161)0.923.6:0.00595)0.466.3:0.00319)0.983.18:0.00816)0.987.24:0.00944)0.898.10:0.00592,(69510:0.03176,573443:0.02104)0.399.2:0.00753)0.900.7:0.00776)0.221:0.00604,114513:0.02018)0.408.2:0.00409,((584558:0.03103,143625:0.03134)0.956.13:0.00986,(46398:0.01651,543504:0.03292)0.941.17:0.00732)0.868.11:0.00487)0.906.9:0.00691,((((144495:0.01344,143469:0.00313)0.944.3:0.00588,(((142787:0.00565,141004:0.00619)0.584:0.00226,(((142666:0.01472,(((143550:0.00822,((142615:0.00981,(144101:0.02213,(142283:0.00137,(143279:0.00276,141015:0.00014)0.741.5:0.00297)0.963.13:0.00654)0.331.3:0.00328)0.694.2:0.00587,144810:0.00476)0.374.2:0.00192)0.952.15:0.00603,(142062:0.0012,(142553:0.01485,142437:0.01513)0.988.26:0.00978)0.618.3:0.00208)0.958.17:0.00627,141873:0.00447)0.935.5:0.00592)0.845.8:0.0026,(143072:0.01339,(((142336:0.01048,142038:0.016)0.754.5:0.00247,(143607:0.00652,144398:0.01241)0.822.4:0.00267)0.744.5:0.00203,143993:0.00014)0.946.15:0.00538)0.983.19:0.00734)0.949.13:0.00582,(141234:0.00788,(144064:0.00129,143943:0.02355)0.998.40:0.01209)0.581:0.00284)0.769.6:0.00166)0.753.6:0.00159,143477:0.01003)0.892.8:0.00476)0.996.32:0.00922,((((163772:0.00372,223178:0.00756)0.759.4:0.00105,(217949:0.01138,225887:0.00279)0.964.17:0.00687)0.984.19:0.00784,226218:0.02028)0.906.10:0.00296,((148565:0.00729,144644:0.00872)1.000.266:0.02285,(219267:0.0052,(172484:0.01412,220876:0.00147)0.908.12:0.02803)0.776.7:0.00677)0.812.5:0.00359)0.691.3:0.00084)0.302.3:0.00203,(((((144525:0.01396,143395:0.01095)0.858.3:0.00265,(148414:0.01753,(144659:0.0118,142965:0.00511)0.962.8:0.00809)0.260:0.00545)0.965.14:0.00869,(144334:0.00838,(142825:0.01161,(143044:0.01234,141888:0.00264)0.982.22:0.00962)0.986.15:0.01384)0.999.78:0.01769)0.982.23:0.01301,((142453:0.01796,((146849:0.00267,148033:0.00758)1.000.267:0.02088,219620:0.00305)0.997.32:0.01339)0.941.18:0.00892,((((((220612:0.0106,(4136:0.00729,162987:0.0131)0.372.3:0.00304)0.875.7:0.00539,203489:0.00467)0.755.7:0.01422,166677:0.00838)0.611.2:0.01571,(((144749:0.01331,(((144853:0.0041,(149097:0.00983,141261:0.00573)0.157.2:0.00217)0.998.41:0.01572,(142173:0.00989,(143400:0.00833,(218857:0.00739,218319:0.02748)0.994.25:0.01276)0.852.2:0.00352)0.991.18:0.0103)0.950.23:0.0055,(((((142828:0.0052,(141838:0.00618,148967:0.01535)0.798.2:0.00274)0.769.7:0.00183,143419:0.00316)0.843.2:0.00807,144329:0.02872)0.951.16:0.00612,144414:0.00583)0.771.7:0.00194,((((147944:0.00616,142232:0.00762)0.982.24:0.01351,((((143359:0.00654,143914:0.01237)0.930.12:0.00408,144020:0.01565)0.739.7:0.00097,142457:0.01617)0.950.24:0.00684,142914:0.00788)0.989.17:0.00758)0.932.12:0.00487,142205:0.00797)0.790.4:0.00529,142166:0.00955)0.953.9:0.00596)0.952.16:0.00585)0.983.20:0.00968)0.828.3:0.01047,148281:0.0179)0.470:0.00193,(149613:0.02439,(((144524:0.01812,(143516:0.00962,(144331:0.00623,(142867:0.00759,141054:0.01126)0.979.21:0.00694)0.760.7:0.00506)0.129.2:0.0055)0.554:0.00815,144438:0.00992)0.765.5:0.00317,142337:0.00507)0.155:0.00863)0.994.26:0.01809)0.292.3:0.00377)0.992.21:0.01776,(31994:0.02698,565920:0.02372)1.000.268:0.04137)0.265:0.00677,545719:0.05104)0.914.9:0.00905)0.932.13:0.01414)0.885.8:0.004,((224378:0.00643,(172977:0.01815,(220312:0.00371,219101:0.01192)0.984.20:0.01173)0.142.2:0.00335)0.422.2:0.00569,171504:0.0216)0.434:0.00359)0.943.16:0.00566)0.991.19:0.01159)0.995.26:0.01733,(((144465:0.01177,143904:0.01491)0.623:0.00477,(((142519:0.00385,141493:0.01802)0.863.10:0.00283,143683:0.0158)0.783:0.0036,(((142473:0.0093,(141295:0.01155,150371:0.01083)0.951.17:0.00659)0.912.7:0.00329,((((142561:0.01236,142763:0.01724)0.887.9:0.00703,(141867:0.01482,143125:0.00879)0.756.4:0.00305)0.178:0.00232,142118:0.01294)0.863.11:0.0025,(144192:0.01605,((144197:0.00608,143117:0.00233)0.919.8:0.00724,144238:0.01913)0.355.2:0.00124)0.978.12:0.00646)0.869.11:0.00261)0.328:0.00178,140996:0.01634)0.890.10:0.0047)0.392:0.0042)0.990.16:0.013,((223986:0.00742,221034:0.01676)0.874.11:0.00252,(((221134:0.01164,224431:0.01132)0.762.6:0.00083,(164828:0.02703,221761:0.0051)0.984.21:0.00569)0.196.2:0.00794,227117:0.00711)0.900.8:0.01456)0.994.27:0.01051)0.954.10:0.01117)0.966.14:0.00811,(143947:0.01547,(223738:0.01309,(148779:0.0139,(218236:0.00722,(226408:0.00579,226457:0.0035)0.499:0.00447)0.915.11:0.00498)0.917.7:0.00358)0.970.14:0.00679)0.938.12:0.00654)0.845.9:0.00323)0.516:0.00787)0.782.6:0.00562)0.855.6:0.00542,(((77591:0.02696,25824:0.01463)0.997.33:0.01975,(27323:0.00358,593083:0.00101)1.000.269:0.03401)0.695.3:0.01022,(((83413:0.01548,4080:0.01109)0.999.79:0.01953,(83445:0.01071,4038:0.02686)1.000.270:0.03366)0.977.22:0.01548,(4043:0.00863,(83652:0.01796,4076:0.01677)0.921.15:0.0114)0.994.28:0.02074)0.955.10:0.00873)0.834.8:0.00674)0.862.8:0.00338)0.937.10:0.00646)0.382.2:0.00302,70117:0.0325)0.837.5:0.00412)0.915.12:0.00636)0.975.17:0.01205)0.939.12:0.01407)0.970.15:0.01902,((561288:0.01256,(535661:0.0274,(238440:0.0104,593215:0.02992)0.522.3:0.00515)0.967.12:0.00914)0.999.80:0.01989,(518643:0.03414,((((535295:0.00967,(((4131:0.00535,(268088:0.01373,547268:0.00307)1.000.271:0.02187)0.771.8:0.00542,219079:0.01871)0.999.81:0.01304,136736:0.01383)0.810.4:0.00372)0.973.17:0.00663,109553:0.00633)0.992.22:0.01134,((3979:0.01703,(100628:0.00998,(318992:0.01457,(81100:0.00318,325466:0.01899)0.753.7:0.00469)0.954.11:0.00553)0.080.2:0.00272)0.921.16:0.00671,582112:0.01456)0.739.8:0.0079)0.420.2:0.00488,519715:0.01514)0.981.17:0.01386)0.483:0.00593)0.996.33:0.02134)1.000.272:0.0505,((141219:0.01058,(90465:0.00843,(101217:0.01773,(141189:0.00448,89329:0.01506)0.088:0.00822)0.987.25:0.01129)0.953.10:0.00961)1.000.273:0.05933,((266509:0.03948,((106877:0.01759,586280:0.01452)1.000.274:0.03475,((99408:0.02693,((((4205:0.01929,4204:0.01411)1.000.275:0.00247,4274:0.00664)0.943.17:0.00423,542593:0.00991)0.977.23:0.00871,4156:0.02831)0.894.13:0.00793)1.000.276:0.10749,(37803:0.11283,103241:0.03831)0.511.2:0.01959)0.208:0.01771)0.398.2:0.01514)0.974.11:0.02136,((79121:0.00386,((3806:0.01846,(108101:0.01233,341850:0.01822)0.848.7:0.00764)1.000.277:0.02076,(157633:0.00636,326880:0.02235)0.765.6:0.00687)0.890.11:0.0088)0.977.24:0.01627,(79373:0.00085,79522:0.07507)0.964.18:0.01808)1.000.278:0.05213)0.603:0.00949)0.995.27:0.02628)0.373:0.00582,((((583561:0.00519,560620:0.00271)0.999.82:0.01982,((77624:0.00664,38610:0.01537)0.542:0.00277,(77774:0.00853,142321:0.02119)0.798.3:0.00471)0.932.14:0.01299)1.000.279:0.03461,3812:0.08715)0.833.6:0.00791,(((((63216:0.03632,3801:0.00536)0.829.5:0.02684,(221514:0.02626,212910:0.02215)0.802.3:0.0096)1.000.280:0.04413,(((560424:0.03832,(107722:0.01531,257722:0.00837)1.000.281:0.01972)0.706:0.01346,((235371:0.01224,347105:0.0178)0.990.17:0.0154,(100167:0.02473,(159087:0.01515,103910:0.0093)0.943.18:0.00767)0.871.13:0.01419)0.969.17:0.01331)0.918.7:0.01058,((569758:0.0236,538030:0.02965)1.000.282:0.03387,(582122:0.02084,(581348:0.01487,(534608:0.00169,(262150:0.00993,540607:0.00936)0.887.10:0.00236)0.772.6:0.00232)1.000.283:0.02125)0.913.13:0.00657)0.968.10:0.01251)0.411.3:0.00546)0.998.42:0.01891,(251213:0.03132,((241975:0.00791,106187:0.01092)1.000.284:0.02277,((247846:0.02333,240820:0.01788)0.999.83:0.02304,((320359:0.00901,(101310:0.00828,249670:0.02726)0.718.6:0.00322)0.973.18:0.01106,(((256656:0.02796,255595:0.02436)0.984.22:0.01572,(594444:0.00511,(238250:0.00172,159471:0.01367)0.999.84:0.01543)1.000.285:0.02445)0.936.15:0.00958,(((62701:0.0223,538290:0.0048)0.796.4:0.0135,(3799:0.01445,592115:0.02243)0.839.6:0.00933)1.000.286:0.03103,(3802:0.01157,3807:0.01421)1.000.287:0.04048)0.742.2:0.01006)0.770.3:0.00787)0.975.18:0.00944)0.098:0.00358)0.925.11:0.01521)0.500.4:0.00517)0.391.2:0.00181,(((((342035:0.00613,(251576:0.0193,128374:0.01984)0.860.14:0.00535)1.000.288:0.0282,(242567:0.02141,306448:0.01515)0.703.4:0.01154)0.999.85:0.02271,(99658:0.03294,(104169:0.02085,(159150:0.02104,560417:0.01438)0.918.8:0.00755)1.000.289:0.02703)0.895.11:0.00827)0.928.8:0.00844,((((112885:0.01127,83737:0.01567)1.000.290:0.02631,(77729:0.02025,(225227:0.01853,570134:0.01909)0.342.2:0.00586)1.000.291:0.02505)1.000.292:0.03477,((168305:0.02502,548153:0.02347)0.991.20:0.01996,79117:0.07337)0.537:0.01041)0.943.19:0.01223,(139224:0.03916,(240434:0.01191,(182638:0.01684,257936:0.00875)0.989.18:0.01417)0.999.86:0.02752)0.998.43:0.02273)0.813.7:0.00358)0.986.16:0.01298,((237596:0.02261,(102429:0.02429,(103618:0.03266,(106409:0.01617,256467:0.00762)0.982.25:0.01121)0.691.4:0.0072)0.679.2:0.0077)1.000.293:0.0233,((105791:0.01124,244538:0.00514)0.997.34:0.01818,(244361:0.03296,(237886:0.00979,139238:0.00813)0.999.87:0.01919)0.225:0.00654)0.979.22:0.01245)0.786.8:0.00941)0.892.9:0.00668)0.905.10:0.0047)0.132.3:0.00393)0.995.28:0.02342,(288748:0.07632,(105838:0.04079,(277877:0.05492,(352774:0.03125,((172254:0.0063,(243633:0.01394,546956:0.01016)0.875.8:0.00389)0.978.13:0.0065,(((352549:0.00024,250680:0.00795)0.995.29:0.00775,28706:0.00994)0.888.3:0.0039,(165202:0.00566,161814:0.01789)0.911.9:0.00421)0.829.6:0.00559)0.964.19:0.009)0.900.9:0.01478)0.984.23:0.02582)0.997.35:0.02732)0.998.44:0.03081)0.999.88:0.03934)0.972.16:0.01412,(((77837:0.01104,(227066:0.00989,143791:0.01025)0.942.14:0.00793)0.998.45:0.02754,(((550451:0.0096,334061:0.02985)0.991.21:0.01526,317015:0.0173)0.737:0.01259,143845:0.00432)1.000.294:0.0334)0.998.46:0.03642,(97468:0.1329,(4386:0.08614,((148552:0.01508,4349:0.02855)0.942.15:0.01359,(562022:0.06572,(304981:0.05054,(584418:0.01532,211824:0.02404)0.895.12:0.01342)0.927.9:0.01669)0.905.11:0.01821)1.000.295:0.04795)0.987.26:0.03306)0.684.5:0.01589)0.997.36:0.03675)0.965.15:0.02584)0.985.12:0.02205,(591769:0.03228,((96577:0.03935,(454145:0.02567,559657:0.01752)0.979.23:0.0181)1.000.296:0.0331,(569847:0.03989,(335853:0.06043,((223314:0.0153,581213:0.00718)0.987.27:0.01484,(515032:0.01176,(559426:0.02049,515451:0.00768)0.993.22:0.016)0.447.2:0.00462)0.218.2:0.00482)0.658.2:0.00389)0.932.15:0.01158)1.000.297:0.04379)1.000.298:0.06725)0.629.2:0.0198)0.959.14:0.0143,(((((((10400:0.04767,(241136:0.03264,(136882:0.01013,138593:0.02178)0.876.8:0.00625)1.000.299:0.02749)0.725.3:0.00876,((557238:0.00796,574939:0.01615)1.000.300:0.02167,(241507:0.00564,249529:0.00966)0.940.10:0.01053)1.000.301:0.02953)0.626:0.00894,((299192:0.02896,(572163:0.0213,329685:0.01615)0.271:0.0081)0.998.47:0.024,(167257:0.00223,10402:0.01444)1.000.302:0.04553)0.983.21:0.01503)0.911.10:0.01229,((((((566260:0.02272,10408:0.01302)0.782.7:0.00324,533546:0.00645)0.446.2:0.00426,562895:0.00221)0.881.8:0.01017,(129416:0.01809,103231:0.00444)0.973.19:0.01341)0.999.89:0.02437,((((144444:0.0114,(335081:0.01558,(327333:0.00859,((341908:0.0154,337642:0.00728)0.456.3:0.00295,331480:0.0091)0.818:0.00227)0.813.8:0.0042)0.959.15:0.01011)1.000.303:0.03254,((255272:0.00326,249844:0.03377)1.000.304:0.0397,(94910:0.01334,(179230:0.00013,569058:0.2183)0.940.11:0.00827)0.521.2:0.00896)0.999.90:0.02334)0.998.48:0.02083,(((146574:0.02795,199746:0.01475)0.986.17:0.01218,((39982:0.00926,(114294:0.02374,143120:0.0168)0.996.34:0.01342)0.975.19:0.01404,(((159005:0.0416,((192132:0.00539,10517:0.00206)0.435.3:0.0135,(198560:0.00515,229139:0.00928)0.992.23:0.01349)0.781.4:0.00952)0.970.16:0.01326,((148915:0.0045,(220622:0.01033,143786:0.03143)0.985.13:0.00995)0.983.22:0.01292,(407754:0.02365,((((303251:0.01716,195233:0.0154)0.346.2:0.00495,(265607:0.00235,170836:0.00081)0.895.13:0.01424)0.857.7:0.01052,588730:0.00699)1.000.305:0.02524,(293365:0.02997,(265414:0.02508,572886:0.0392)0.933.12:0.0098)0.997.37:0.02051)0.747.4:0.00507)1.000.306:0.02966)0.797.5:0.0088)0.876.9:0.00579,((147874:0.01331,(224816:0.02572,(23507:0.00268,582695:0.00014)0.998.49:0.01828)0.817.6:0.00826)0.960.17:0.01407,(((((322341:0.02335,348053:0.0146)0.657:0.00767,304312:0.0226)0.948.7:0.00791,321186:0.02093)0.685:0.00705,(113978:0.00517,10518:0.00534)0.974.12:0.01141)0.898.11:0.01493,((145835:0.02731,(234242:0.01728,574243:0.0084)0.992.24:0.01828)1.000.307:0.03133,((326109:0.00246,170502:0.00284)0.707.5:0.03262,(((585650:0.00911,537892:0.00966)0.987.28:0.00945,(319780:0.01669,558603:0.01639)0.967.13:0.00924)0.873.7:0.0028,(((25583:0.00238,308451:0.03573)0.951.18:0.00346,364203:0.01282)0.876.10:0.00207,(((10526:0.00573,33199:0.0129)0.836.7:0.00162,585949:0.01404)0.982.26:0.00531,(30569:0.00527,(51535:0.0098,104185:0.00993)0.862.9:0.00417)0.994.29:0.01037)0.279.2:0.00079)0.826.3:0.0019)0.967.14:0.00916)0.752.4:0.00422)0.761.3:0.01074)0.945.14:0.00865)0.995.30:0.01366)0.925.12:0.01196)0.561.3:0.00666)0.972.17:0.00861,((148238:0.02126,144902:0.01703)1.000.308:0.03271,(10485:0.04021,(10486:0.03381,(((150205:0.00991,((142895:0.02645,147645:0.0076)0.513:0.00294,137367:0.01812)0.415:0.00216)0.952.17:0.00479,(149936:0.02131,361169:0.0126)0.876.11:0.00568)0.721.2:0.00035,(((253105:0.01579,78006:0.01279)0.996.35:0.01882,(201363:0.01693,78137:0.02138)0.966.15:0.01263)1.000.309:0.03247,368560:0.01333)0.752.5:0.00397)0.980.17:0.01175)0.957.22:0.00886)0.113.2:0.00577)0.997.38:0.02188)0.439.4:0.00777)0.982.27:0.01191,((26629:0.02409,((107875:0.00604,177966:0.01311)1.000.310:0.0407,(68874:0.00742,257276:0.02371)0.997.39:0.02224)0.567.2:0.00933)0.981.18:0.01689,((((235762:0.01637,567650:0.01303)0.917.8:0.00938,(219355:0.01273,(592620:0.00524,338150:0.0166)0.966.16:0.00691)0.690.2:0.00515)1.000.311:0.02729,(((((((586229:0.00967,10470:0.01477)0.612.3:0.00313,226732:0.0142)0.762.7:0.00156,559451:0.01922)0.879.9:0.00234,(243561:0.00471,351705:0.01663)0.955.11:0.00391)0.743.4:0.00374,392116:0.01539)0.996.36:0.01726,((266554:0.00358,561586:0.00454)0.994.30:0.0095,102523:0.00586)0.187.2:0.02113)0.997.40:0.01317,(10463:0.02934,564993:0.0328)0.815.2:0.00407)0.870.12:0.00602)0.998.50:0.02187,((102334:0.03131,(((226936:0.00074,161406:0.00164)1.000.312:0.03424,10449:0.0195)0.996.37:0.02383,(105315:0.02273,195717:0.02335)0.942.16:0.0124)0.981.19:0.01694)0.724.4:0.00281,(((570101:0.02845,((533801:0.02508,(((203149:0.0184,164422:0.02162)0.625.3:0.00233,10444:0.00348)0.982.28:0.01087,((10442:0.00467,547898:0.01583)0.981.20:0.01055,516164:0.01117)0.936.16:0.00887)0.991.22:0.01672)0.995.31:0.01489,((522596:0.00896,((565050:0.00014,80009:0.00242)0.974.13:0.00764,10437:0.01676)0.943.20:0.00676)0.989.19:0.01408,(264685:0.02515,(((257510:0.01928,((104557:0.01248,((10430:0.01043,87949:0.00534)0.947.14:0.00473,43170:0.01745)0.003:0.00217)0.754.6:0.0013,54085:0.00549)0.468:0.0032)0.987.29:0.00789,(72826:0.01526,(111852:0.01911,347187:0.01151)0.998.51:0.01029)0.893.5:0.00336)0.962.9:0.00481,570509:0.01055)0.994.31:0.01579)0.631.3:0.01125)0.950.25:0.01048)0.975.20:0.01075)0.753.8:0.00722,((81681:0.00806,140059:0.01619)1.000.313:0.03536,((34507:0.01652,10454:0.01399)1.000.314:0.02333,(170412:0.00898,((96653:0.0056,(536531:0.01293,(309022:0.0113,10455:0.01171)0.795.6:0.00359)0.310:0.01259)0.983.23:0.0061,584131:0.00332)0.978.14:0.00846)0.998.52:0.02449)0.583.3:0.00832)0.535.2:0.00815)0.981.21:0.01203,((102653:0.01197,(245878:0.00179,199794:0.0306)0.990.18:0.01339)0.997.41:0.01999,283392:0.05013)0.160:0.00394)0.932.16:0.00628)0.852.3:0.00476)0.954.12:0.0139)0.847.9:0.00717)0.979.24:0.01334)0.298:0.01083,(100488:0.02911,(73344:0.02118,(137297:0.02684,519284:0.00014)0.360:0.00763)0.959.16:0.01479)1.000.315:0.03958)0.561.4:0.007)0.999.91:0.03097,((236160:0.03878,231910:0.011)1.000.316:0.09484,((171291:0.04756,(200431:0.01252,305529:0.00208)0.978.15:0.02882)0.991.23:0.01788,((222885:0.05672,(510135:0.02502,571356:0.02628)0.998.53:0.02959)0.992.25:0.01729,(263284:0.03628,(((203274:0.00158,202279:0.00656)0.974.14:0.01819,151247:0.0068)0.698.2:0.01743,(176114:0.02325,22125:0.02402)0.981.22:0.01101)1.000.317:0.02202)0.850.9:0.00634)0.932.17:0.0085)0.764.5:0.0089)1.000.318:0.06935)0.983.24:0.02113,((554620:0.02696,(539552:0.03256,211216:0.01627)0.911.11:0.02)1.000.319:0.09566,(((510484:0.02152,350097:0.01174)1.000.320:0.04548,((568730:0.01217,(57551:0.01796,521936:0.00724)0.978.16:0.01442)1.000.321:0.05668,(74231:0.02086,272977:0.02819)0.812.6:0.01642)0.221.2:0.01342)0.791.5:0.01287,((25453:0.00634,(235381:0.00155,(179130:0.00701,96944:0.01363)0.925.13:0.00275)0.782.8:0.00144)1.000.322:0.06715,(43329:0.0653,(25709:0.00693,557175:0.01932)1.000.323:0.03844)0.992.26:0.02598)0.166:0.00683)1.000.324:0.06368)0.941.19:0.02642)0.937.11:0.01898,(((((139839:0.04507,(((29566:0.02299,11298:0.00209)1.000.325:0.04554,(11292:0.03346,447942:0.01362)0.830.4:0.01711)0.996.38:0.02044,((228694:0.01685,518404:0.00733)1.000.326:0.04672,(31235:0.04326,((73887:0.01093,(((236623:0.01062,70628:0.00832)0.740.6:0.00139,74391:0.01488)0.990.19:0.01668,((93158:0.01631,70038:0.02217)1.000.327:0.02528,((((59574:0.0234,92131:0.00878)0.729.5:0.00099,(529859:0.00492,(38061:0.02414,572818:0.00024)0.837.6:0.02919)0.934.10:0.00666)0.935.6:0.00456,237034:0.00633)0.667.2:0.00487,((92134:0.01356,70804:0.01503)0.981.23:0.00843,(74059:0.01006,(11297:0.00675,156697:0.02117)0.711.2:0.00243)0.990.20:0.00924)0.292.4:0.00303)0.993.23:0.01847)0.999.92:0.02539)0.999.93:0.02584)0.873.8:0.01448,566658:0.00831)0.941.20:0.01645)0.926.15:0.01396)0.045.2:0.00734)1.000.328:0.05158)0.996.39:0.02429,(((105657:0.00936,112138:0.00172)0.959.17:0.03405,(560396:0.00179,(42042:0.00222,(110394:0.01189,(241391:0.02106,240997:0.00508)0.999.94:0.0161)0.884.10:0.00281)1.000.329:0.02028)1.000.330:0.02018)0.951.19:0.01154,((591820:0.01228,532752:0.01925)0.996.40:0.01738,((((511044:0.02275,470440:0.00168)0.671.3:0.00353,((470429:0.02768,(((550936:0.01415,368299:0.00781)0.957.23:0.00563,(137187:0.01021,518865:0.02079)0.760.8:0.01562)0.387:0.00014,436432:0.00868)0.972.18:0.00692)0.875.9:0.00262,(470451:0.00784,288565:0.03557)0.934.11:0.00468)0.963.14:0.00669)0.997.42:0.01503,((581962:0.00244,276080:0.00628)0.899.12:0.01109,332164:0.01112)1.000.331:0.02423)0.947.15:0.00916,(11302:0.02345,99826:0.03365)0.950.26:0.01018)0.879.10:0.00611)0.989.20:0.01494)0.981.24:0.01434)1.000.332:0.11021,(200153:0.12727,(((((113012:0.02103,(586839:0.01023,(289791:0.01521,(15698:0.01442,(200351:0.00978,290290:0.00974)0.917.9:0.00595)0.254.4:0.00275)0.313:0.00501)1.000.333:0.03713)0.997.43:0.02713,((((322615:0.02994,294314:0.02389)1.000.334:0.02959,((((((559974:0.01365,(531345:0.0067,527512:0.002)1.000.335:0.01972)1.000.336:0.02891,148935:0.01606)0.963.15:0.0095,266198:0.02167)0.990.21:0.01411,(561959:0.04763,(((567437:0.01449,((185117:0.00757,515855:0.01497)0.989.21:0.01007,529404:0.01714)0.769.8:0.00575)0.982.29:0.009,((576850:0.01622,587510:0.01438)1.000.337:0.01986,514061:0.01132)0.163.2:0.00207)0.837.7:0.00296,516461:0.01384)0.457.2:0.00216)0.858.4:0.00604)0.925.14:0.01201,(301045:0.02548,305758:0.02441)0.989.22:0.01704)0.928.9:0.00825,(42395:0.01651,(145737:0.01454,((347587:0.01242,327180:0.00712)0.736.4:0.00098,297155:0.01742)0.975.21:0.00683)0.917.10:0.00518)0.995.32:0.01959)0.967.15:0.01191)0.770.4:0.00618,((291419:0.02564,(292635:0.00678,(148285:0.0236,265925:0.02106)0.100:0.00209)0.970.17:0.01188)1.000.338:0.02712,((((351463:0.00959,527076:0.01564)1.000.339:0.02503,((234951:0.01106,315402:0.00395)0.790.5:0.04933,(544009:0.01933,((((514671:0.01804,(207532:0.01477,((357591:0.00565,357766:0.02496)0.724.5:0.01366,((21652:0.00677,353853:0.01385)0.987.30:0.00932,(525156:0.01208,(269619:0.01134,165445:0.01599)0.797.6:0.00294)0.958.18:0.00632)0.948.8:0.00736)0.924.12:0.0061)0.828.4:0.00447)0.832.5:0.00242,521033:0.0114)0.897.7:0.00253,(207023:0.02578,((329032:0.00083,517044:0.00726)0.897.8:0.00272,247556:0.02323)0.861.8:0.0074)0.998.54:0.00014)0.949.14:0.00238,(((((325489:0.01808,(530751:0.01355,349892:0.01538)0.864.7:0.00224)0.757.3:0.00108,289399:0.01089)0.928.10:0.00292,((277165:0.01117,((268757:0.01931,274035:0.02207)0.845.10:0.00246,330098:0.01623)0.984.24:0.0076)0.749.3:0.00036,260536:0.01617)0.955.12:0.00357)0.897.9:0.00016,(323693:0.01667,346769:0.00204)0.405:0.00191)0.926.16:0.00383,518557:0.01301)0.766.9:0.00065)0.732.3:0.0058)0.697.3:0.00884)0.776.8:0.00698)0.995.33:0.01364,319287:0.03215)0.905.12:0.00626,((342452:0.02333,304971:0.01607)1.000.340:0.02306,(336245:0.03651,(310545:0.00475,(583845:0.01393,(210102:0.00405,363469:0.02078)0.677.2:0.00448)0.979.25:0.00946)0.972.19:0.00858)1.000.341:0.02753)0.941.21:0.01214)0.898.12:0.01174)1.000.342:0.02744)0.997.44:0.01742,(((((437656:0.00839,((301378:0.00982,513605:0.02665)0.895.14:0.00882,567506:0.00876)0.434.2:0.0028)0.920.8:0.00457,321762:0.00826)0.960.18:0.01028,((320640:0.02271,313733:0.03467)0.978.17:0.02129,(279179:0.00702,(309413:0.00514,301272:0.02031)0.990.22:0.01107)0.996.41:0.01817)1.000.343:0.02689)0.988.27:0.02042,((((512484:0.01063,259677:0.00828)0.747.5:0.00054,(202520:0.0041,(114288:0.01271,(299795:0.02292,531589:0.00794)0.962.10:0.01058)0.968.11:0.00949)0.833.7:0.00227)0.891.8:0.00468,(348767:0.06823,((352784:0.01865,580530:0.00675)0.812.7:0.00159,(234959:0.00327,266800:0.01254)1.000.344:0.02594)0.828.5:0.00364)0.990.23:0.01841)0.382.3:0.00712,(((195833:0.04479,356152:0.00601)0.983.25:0.00724,562962:0.00268)0.955.13:0.00637,((362288:0.00758,233848:0.00015)0.783.2:0.00923,555400:0.01136)0.739.9:0.00407)0.999.95:0.02188)0.966.17:0.01411)0.987.31:0.01829,((((328666:0.01179,460759:0.04156)0.826.4:0.00368,((232434:0.0069,325403:0.01446)0.019.2:0.02062,230132:0.03821)0.974.15:0.01023)0.651.2:0.00246,(305882:0.01196,(135943:0.03294,(524648:0.02192,525853:0.01463)0.770.5:0.00348)0.604:0.00511)0.995.34:0.01496)0.448.4:0.00294,((509454:0.01123,(277047:0.02053,(209031:0.02496,(292124:0.00921,((265197:0.0099,272602:0.00527)0.987.32:0.01022,270447:0.01572)0.996.42:0.01544)0.954.13:0.00651)0.841.4:0.00535)0.992.27:0.01139)1.000.345:0.02647,(((213966:0.01535,559991:0.02817)1.000.346:0.03272,((229533:0.01047,323989:0.02044)0.917.11:0.00629,(580081:0.0229,(519405:0.01584,50779:0.00991)0.971.15:0.01278)0.641.2:0.00909)0.957.24:0.01114)0.531.2:0.00648,((545091:0.00762,322460:0.01923)0.979.26:0.01171,(211308:0.04203,(304255:0.01573,271356:0.02014)1.000.347:0.02192)1.000.348:0.0244)0.859.7:0.00442)0.986.18:0.01425)0.470.2:0.00625)1.000.349:0.03403)0.336.2:0.00217)0.961.16:0.01531)1.000.350:0.06781,((((166158:0.02081,567222:0.01401)0.961.17:0.01204,(429439:0.01129,(53927:0.01074,546258:0.01615)0.556:0.00319)0.958.19:0.00958)1.000.351:0.04004,((251430:0.01218,((558391:0.01628,263447:0.00903)0.974.16:0.00723,142221:0.02218)0.294.3:0.00166)1.000.352:0.03286,(((71794:0.02725,568610:0.02542)0.989.23:0.01194,((129089:0.01592,206850:0.02565)0.964.20:0.00579,((78432:0.0182,85011:0.0048)0.742.3:0.00146,(586521:0.00922,((77171:0.00995,(591314:0.00191,(590934:0.02545,582830:0.02645)0.318:0.00434)0.996.43:0.0112)0.995.35:0.01197,576360:0.01427)0.296:0.00281)0.744.6:0.00276)0.739.10:0.0031)0.962.11:0.00855)0.995.36:0.01749,(104368:0.04295,(((99271:0.01832,534716:0.02983)0.116.2:0.00389,(139420:0.01269,557248:0.02812)0.950.27:0.00796)0.592.5:0.00338,(103907:0.01101,(574787:0.01712,15796:0.01704)0.874.12:0.00258)0.758.5:0.0108)0.438:0.00268)0.444:0.00347)1.000.353:0.03325)0.922.9:0.01651)0.996.44:0.02398,((15899:0.06725,((79153:0.00424,588056:0.00016)1.000.354:0.05625,(((((543508:0.00719,39647:0.01964)1.000.355:0.03196,(108671:0.01735,(146814:0.01424,147129:0.02289)0.937.12:0.00805)1.000.356:0.03219)0.940.12:0.01388,(((310116:0.00078,15878:0.00639)0.753.9:0.00183,((579601:0.02588,(15864:0.00661,123108:0.01507)0.527.2:0.00488)1.000.357:0.02479,(15865:0.02088,462568:0.01076)0.855.7:0.00989)1.000.358:0.03304)0.994.32:0.01711,(328570:0.03072,(537925:0.01409,((56134:0.01547,(15873:0.00082,591777:0.00014)0.863.12:0.00269)1.000.359:0.02907,(349001:0.00502,70042:0.01178)0.640.2:0.00465)0.806.3:0.00487)0.550.3:0.00639)0.979.27:0.01431)0.990.24:0.01502)0.813.9:0.00842,((58394:0.04051,(543369:0.03007,15906:0.0199)0.999.96:0.02579)0.998.55:0.02376,((555715:0.02429,36302:0.01879)1.000.360:0.04735,(((15931:0.02755,(107873:0.00078,346929:0.00014)0.999.97:0.0186)0.988.28:0.01251,((517560:0.01462,((578261:0.02039,(15947:0.01368,(70343:0.00849,(303415:0.00767,15948:0.01014)0.635.2:0.00152)0.955.14:0.00436)0.762.8:0.00257)0.965.16:0.00755,(71629:0.01412,(60702:0.01673,((74382:0.00784,(15953:0.01727,15954:0.0064)0.636.2:0.003)0.872.17:0.00215,(315295:0.00716,173388:0.00964)0.903.14:0.00247)0.728.4:0.00093)0.993.24:0.0123)0.354:0.00605)0.712:0.00302)0.922.10:0.00487,((19295:0.01406,(15936:0.01867,347915:0.01534)0.879.11:0.00542)0.931.11:0.00466,((183380:0.00439,(588281:0.01117,190324:0.01051)0.856.6:0.00302)0.405.2:0.00101,15943:0.01293)0.981.25:0.00557)0.928.11:0.00473)1.000.361:0.01659)0.889.5:0.00706,((200523:0.03321,((((((538840:0.00618,551259:0.00618)0.961.18:0.00456,((333099:0.01299,15916:0.02799)0.854.5:0.00409,329837:0.01389)0.933.13:0.00367)0.487:0.00239,(15912:0.0082,71155:0.01032)0.926.17:0.01591)0.989.24:0.00988,(186205:0.009,15918:0.01204)0.926.18:0.00568)0.762.9:0.001,(69777:0.01728,15919:0.00525)0.962.12:0.00929)0.949.15:0.0039,((15920:0.02567,(15921:0.01485,(342895:0.00937,179109:0.01604)0.913.14:0.0047)0.887.11:0.00522)1.000.362:0.01698,((142565:0.01928,338814:0.01299)0.988.29:0.00982,15922:0.02015)0.150:0.00404)0.871.14:0.00232)0.811.3:0.00284)0.517.2:0.00516,((536180:0.03092,(15924:0.01673,15925:0.00928)0.247:0.00346)0.788.3:0.00525,260430:0.01003)0.908.13:0.00381)0.998.56:0.02053)0.999.98:0.0203)0.825.5:0.0122)0.289:0.00819)0.738.10:0.01113,((567316:0.0224,(199102:0.02048,(67615:0.00014,233952:0.0003)0.837.8:0.01093)0.992.28:0.01388)0.975.22:0.01092,(85218:0.04364,(((15897:0.00752,(((573950:0.01449,((225520:0.00562,306647:0.00346)0.982.30:0.00657,(594001:0.00709,108816:0.00471)0.884.11:0.0034)0.592.6:0.00348)0.831.3:0.00699,15887:0.00701)0.764.6:0.00124,(317108:0.012,344619:0.01094)0.309:0.00225)0.962.13:0.00506)0.894.14:0.00258,(15889:0.00965,59733:0.00765)0.795.7:0.00211)0.759.5:0.00014,(565607:0.00344,537306:0.01027)0.942.17:0.00553)0.875.10:0.00603)0.999.99:0.01897)0.985.14:0.01561)0.878.11:0.00604)0.827.3:0.02141)1.000.363:0.08086,(((((((551749:0.01248,542429:0.02212)0.960.19:0.00846,((15827:0.00777,580923:0.02322)0.998.57:0.01752,269293:0.00872)0.689.2:0.00651)1.000.364:0.06392,(566189:0.0154,(107845:0.04264,(15834:0.00517,15839:0.01676)0.997.45:0.01427)0.128.2:0.00761)0.997.46:0.01994)0.960.20:0.01174,((((114530:0.01543,590759:0.02026)1.000.365:0.02479,(77956:0.07323,(114509:0.0252,78062:0.01904)1.000.366:0.04887)0.997.47:0.02439)0.129.3:0.00604,((77895:0.0084,77983:0.03481)0.998.58:0.0173,(243142:0.02406,548855:0.01797)0.965.17:0.01054)0.796.5:0.0079)1.000.367:0.05487,(149052:0.0604,((15850:0.00827,570232:0.00428)0.961.19:0.01203,(540483:0.01748,(559508:0.00988,111523:0.02143)0.971.16:0.01033)0.846.3:0.00457)0.704.5:0.00974)1.000.368:0.03765)0.335:0.008)0.995.37:0.02534,(((160148:0.00526,308020:0.00016)0.987.33:0.01182,(573062:0.0198,15837:0.00727)0.451:0.00847)0.998.59:0.03806,((((49164:0.02045,114673:0.01212)0.998.60:0.0221,(51934:0.05119,(136766:0.01659,(573881:0.00669,(282773:0.01295,316748:0.04262)0.919.9:0.00479)0.995.38:0.01393)1.000.369:0.03383)0.935.7:0.01168)0.664:0.00136,((97115:0.01664,(89356:0.01632,247626:0.01446)1.000.370:0.01947)1.000.371:0.02202,(136331:0.02087,542085:0.03621)0.730.3:0.00761)0.704.6:0.00466)1.000.372:0.11151,((23047:0.03364,209237:0.01378)0.997.48:0.03555,(87318:0.01852,(15860:0.06882,133123:0.03626)0.987.34:0.02378)0.982.31:0.02113)0.990.25:0.04368)1.000.373:0.10209)0.999.100:0.03757)1.000.374:0.0818,((287209:0.01116,(284191:0.02368,280502:0.0191)0.201:0.00536)1.000.375:0.05884,(280566:0.0274,568172:0.00975)0.992.29:0.0241)1.000.376:0.06291)0.941.22:0.0154,(219640:0.0212,255073:0.03712)1.000.377:0.19777)0.240.2:0.00972)0.976.18:0.01979)0.995.39:0.02532)0.347.2:0.01128,((((305348:0.00237,528244:0.01886)1.000.378:0.03286,(((215065:0.02709,((311535:0.0258,529828:0.02398)0.483.2:0.00774,(315971:0.0362,529539:0.01202)1.000.379:0.02187)0.836.8:0.00528)0.858.5:0.00615,(15703:0.03165,(15707:0.00319,510498:0.00607)1.000.380:0.01756)0.957.25:0.00796)1.000.381:0.02741,(262497:0.04286,555555:0.03565)0.923.7:0.01879)0.984.25:0.01628)0.964.21:0.01286,(((((103525:0.01934,((351154:0.01102,103806:0.0223)0.943.21:0.00653,((590004:0.00014,(15689:0.0079,15690:0.01939)0.430:0.00087)0.974.17:0.00827,243947:0.02341)0.966.18:0.00891)0.949.16:0.00753)1.000.382:0.02959,((((107774:0.02651,15688:0.01495)0.930.13:0.0065,((545715:0.01195,(274442:0.00462,271584:0.01978)0.708:0.00648)0.999.101:0.01971,148755:0.00765)1.000.383:0.02355)0.860.15:0.0055,(30323:0.01813,(15697:0.02869,103388:0.04291)0.889.6:0.01495)1.000.384:0.02796)0.441:0.00457,(278258:0.01616,542107:0.02665)1.000.385:0.03879)0.818.2:0.00661)0.364.2:0.00366,((575096:0.01846,(((55663:0.01941,((((32876:0.01804,270156:0.00982)0.483.3:0.00431,15674:0.01033)0.748.4:0.00089,(107488:0.01693,(186762:0.01641,(101068:0.01586,(105508:0.02197,202612:0.00426)0.766.10:0.0036)0.917.12:0.00073)0.788.4:0.00261)0.939.13:0.00476)0.912.8:0.00293,316248:0.00524)0.933.14:0.00495)0.603.2:0.00304,(15680:0.00858,250969:0.02804)0.786.9:0.00592)0.281:0.0029,60694:0.01583)0.708.2:0.00479)0.997.49:0.01853,(((93038:0.01263,345429:0.01439)0.897.10:0.00532,(369928:0.00885,15650:0.02376)0.882.8:0.00801)0.996.45:0.0188,((((((15646:0.01983,102155:0.02433)0.310.2:0.0021,15637:0.01879)1.000.386:0.01435,((63017:0.0244,(15648:0.01993,(15647:0.01099,131688:0.01924)0.881.9:0.00329)0.917.13:0.00418)0.977.25:0.00716,((260301:0.00921,69009:0.01671)0.957.26:0.00675,((143071:0.0093,209087:0.03765)0.921.17:0.00311,254955:0.01998)0.812.8:0.00331)0.995.40:0.00981)0.704.7:0.00079)0.888.4:0.00317,((186018:0.02,15590:0.01764)0.957.27:0.00541,(58412:0.03062,53611:0.01724)0.984.26:0.00953)0.832.6:0.00307)0.982.32:0.00895,(55569:0.00246,367461:0.17044)0.278:0.01289)0.886.9:0.0067,(64303:0.0344,(((156272:0.01242,(202793:0.02616,199895:0.00612)0.988.30:0.01166)1.000.387:0.00015,164955:0.00763)0.742.4:0.00092,54260:0.02034)0.138:0.0294)0.857.8:0.01168)0.999.102:0.02712)0.731.3:0.00719)1.000.388:0.03764)0.882.9:0.00485,15696:0.09152)0.823.2:0.00614,((252219:0.01388,232478:0.00465)1.000.389:0.06615,(((103956:0.0039,540644:0.00455)0.867.10:0.00435,592559:0.01898)0.940.13:0.0072,103238:0.01195)0.988.31:0.01661)0.899.13:0.01464)0.996.46:0.02173)1.000.390:0.04066,((267455:0.08074,((274141:0.04688,(((287510:0.00652,(469918:0.00538,21138:0.0112)0.014:0.00015)1.000.391:0.00278,(309852:0.01475,295705:0.01771)0.920.9:0.01199)0.999.103:0.02245,((129426:0.04488,(272141:0.00028,516319:0.00313)0.961.20:0.01671)0.999.104:0.02246,(592917:0.01885,(542780:0.01341,328719:0.02499)0.965.18:0.0095)0.916.14:0.01049)0.850.10:0.00809)0.981.26:0.01656)0.971.17:0.01763,(((((522353:0.01604,516223:0.01191)0.966.19:0.00635,(74862:0.01922,413456:0.01088)0.885.9:0.00265)0.927.10:0.0043,295081:0.02131)0.887.12:0.00791,((233953:0.0121,((188354:0.00314,151870:0.00112)0.975.23:0.00489,191700:0.01054)0.693.2:0.00454)1.000.392:0.02286,(231985:0.00238,576232:0.00455)0.977.26:0.00952)0.374.3:0.00792)0.991.24:0.01503,(((554904:0.0114,542934:0.00545)0.950.28:0.00797,(528074:0.01208,190872:0.01214)0.960.21:0.00645)0.999.105:0.01522,(136526:0.02352,(((550248:0.00649,(274239:0.00014,558752:0.00254)1.000.393:0.03245)0.356:0.00073,(((((200664:0.01722,251385:0.01906)0.827.4:0.00857,100232:0.00618)0.819.3:0.00588,104610:0.00753)1.000.394:0.0211,102627:0.00783)0.919.10:0.00521,(360238:0.00797,142563:0.00455)0.805.8:0.00177)0.989.25:0.00738)0.853.5:0.00257,15716:0.00903)0.994.33:0.01172)0.958.20:0.00941)0.956.14:0.01007)0.265.2:0.01193)0.926.19:0.01726)1.000.395:0.05606,((((109807:0.05324,((565289:0.02585,(362470:0.01323,(181074:0.01162,539382:0.00258)0.996.47:0.01165)0.999.106:0.01738)0.941.23:0.00778,(((((196708:0.009,222343:0.00236)1.000.396:0.0167,((301672:0.02743,49837:0.02001)0.914.10:0.00517,(288677:0.02563,(289655:0.03474,(148303:0.00721,(578197:0.00106,233573:0.00042)0.979.28:0.00674)0.700.2:0.00622)0.973.20:0.01307)0.528.2:0.00549)0.842.3:0.00587)0.929.14:0.00745,((((175699:0.01373,(274084:0.0267,(264699:0.03648,273953:0.03334)0.877.14:0.01067)0.995.41:0.02301)0.998.61:0.02701,115098:0.01154)1.000.397:0.05081,(433140:0.04334,((98789:0.00014,381629:0.00124)1.000.398:0.02244,(135952:0.02189,410665:0.02079)0.606.2:0.00371)0.997.50:0.01993)0.989.26:0.02107)0.326.2:0.01549,(449743:0.01427,227728:0.01314)0.999.107:0.02332)1.000.399:0.04809)0.417:0.00234,(49369:0.02388,(470369:0.00156,(184625:0.01556,(178347:0.00978,295024:0.00634)0.993.25:0.01349)1.000.400:0.02527)0.791.6:0.03427)0.630:0.00606)0.993.26:0.01497,((((((229069:0.01561,(15734:0.00249,(185935:0.02677,198054:0.01703)0.991.25:0.00577)0.913.15:0.01211)0.992.30:0.01223,(529292:0.02931,556643:0.01842)0.989.27:0.01503)0.709:0.0035,15729:0.01234)0.851.6:0.00474,493391:0.0236)0.912.9:0.00571,470381:0.01353)0.123:0.00538,((262375:0.01935,276677:0.02529)1.000.401:0.04168,195355:0.01983)0.857.9:0.00748)0.957.28:0.00965)0.607.2:0.00834)1.000.402:0.03085)0.874.13:0.01144,(188919:0.02162,(300511:0.00206,296332:0.01027)0.998.62:0.02765)0.997.51:0.02365)0.996.48:0.025,((((313063:0.03183,292053:0.01169)0.999.108:0.03068,(304473:0.02042,(531331:0.01108,(340537:0.03154,(296385:0.00892,351528:0.01373)0.984.27:0.00947)0.526.2:0.00177)0.874.14:0.00495)0.982.33:0.01869)0.999.109:0.03806,(264499:0.00166,266677:0.01879)1.000.403:0.08574)0.982.34:0.02561,(((((((230263:0.00251,349487:0.00206)0.987.35:0.00821,149744:0.00999)0.998.63:0.02289,(292372:0.04975,(149856:0.01299,((290864:0.01694,293994:0.00961)0.972.20:0.00961,(((340545:0.01533,533537:0.02015)0.128.3:0.00638,276258:0.02112)0.708.3:0.00348,276862:0.00963)0.955.15:0.00835)0.154:0.00305)0.997.52:0.02145)0.659.2:0.01236)1.000.404:0.03322,((537420:0.04288,((355885:0.02452,(349481:0.0003,330367:0.00277)0.960.22:0.00759)0.999.110:0.01811,((258244:0.02241,(146880:0.02431,((323045:0.026,164006:0.00755)0.934.12:0.00552,(297719:0.01008,553228:0.01444)0.908.14:0.00613)0.262.2:0.00501)0.852.4:0.00466)0.993.27:0.01301,(((274257:0.00184,540924:0.00268)0.817.7:0.00035,337579:0.00926)0.991.26:0.0121,(((260783:0.02745,(310012:0.00933,303004:0.01301)0.930.14:0.00744)0.972.21:0.00758,305211:0.02668)0.679.3:0.00196,(532556:0.02522,577460:0.02105)0.996.49:0.01676)0.175:0.0027)0.548:0.00702)0.992.31:0.01915)0.662.3:0.00726)0.532.2:0.0043,355889:0.02463)0.998.64:0.02644)0.443:0.01096,((560959:0.00014,592884:0.00014)0.905.13:0.00331,(198937:0.00266,176495:0.00806)0.761.4:0.0013)0.724.6:0.03262)0.641.3:0.0043,525159:0.03449)0.983.26:0.011,((470324:0.01224,(77904:0.00105,15731:0.01152)0.287:0.00145)1.000.405:0.02665,(((142208:0.01165,(140669:0.0035,352458:0.00645)0.995.42:0.0136)0.986.19:0.01254,(343415:0.02489,(588461:0.00868,324768:0.01673)0.995.43:0.01461)0.967.16:0.01089)0.563.3:0.00629,((142165:0.00014,114755:0.02659)0.976.19:0.02493,((559642:0.00551,330905:0.00655)0.722.2:0.00821,(500948:0.01068,513667:0.019)0.828.6:0.00584)1.000.406:0.03531)0.966.20:0.01205)0.988.32:0.01641)0.889.7:0.0074)0.574:0.00234)0.996.50:0.01915)0.998.65:0.02567,((353691:0.00436,(568077:0.01184,516873:0.01584)0.869.12:0.00455)1.000.407:0.09113,((((528858:0.00552,317703:0.001)0.826.5:0.00345,(530177:0.01372,512892:0.00281)0.872.18:0.00741)1.000.408:0.06276,247091:0.06316)0.404.3:0.01777,((158774:0.02423,529472:0.02711)1.000.409:0.03294,((259697:0.02653,585892:0.03419)1.000.410:0.0311,(293385:0.05756,(((235952:0.07304,((211927:0.00136,297865:0.01766)0.998.66:0.02122,((270947:0.02354,(15714:0.01051,15713:0.01626)0.981.27:0.01476)0.998.67:0.02754,(248473:0.02239,219788:0.03403)0.991.27:0.01968)0.149:0.00854)1.000.411:0.04115)0.944.4:0.01265,((304059:0.00587,346727:0.01831)0.066:0.01052,(547649:0.01853,((346271:0.01418,301292:0.01962)0.999.111:0.02193,565711:0.00495)0.277:0.00302)0.997.53:0.02127)0.998.68:0.0257)0.532.3:0.00614,(294333:0.05113,(15710:0.0333,(289242:0.00956,((308656:0.01283,293999:0.01453)1.000.412:0.01361,324304:0.00825)0.794.6:0.00508)0.989.28:0.01284)0.282.2:0.0061)0.976.20:0.01317)0.994.34:0.0222)0.777.7:0.00575)0.919.11:0.01458)0.991.28:0.0224)0.950.29:0.01694)0.174:0.01246)0.576.2:0.00685)0.685.2:0.01033)0.769.9:0.01552)0.926.20:0.01115,(((313232:0.06499,360268:0.03264)1.000.413:0.04894,((242868:0.0389,300708:0.03316)0.989.29:0.01809,((247217:0.02002,(110120:0.01026,559693:0.02658)1.000.414:0.02253)0.973.21:0.01661,(551818:0.00459,200584:0.00387)0.767.8:0.10356)0.841.5:0.01186)0.985.15:0.01964)0.984.28:0.01853,((581038:0.01312,263658:0.02912)1.000.415:0.06501,((160460:0.01818,63832:0.01007)1.000.416:0.03103,((243175:0.01519,511931:0.02012)0.993.28:0.01598,(236166:0.02333,545311:0.01275)0.957.29:0.01043)0.793.3:0.00954)1.000.417:0.04458)0.987.36:0.0202)0.992.32:0.0233)0.964.22:0.01518)0.941.24:0.0174)1.000.418:0.03792,(((225715:0.0089,(((78071:0.01434,81819:0.0054)0.934.13:0.00707,(143841:0.0089,107427:0.00634)0.889.8:0.00381)0.846.4:0.00363,(220693:0.00829,225145:0.02215)0.983.27:0.00906)0.272:0.00216)1.000.419:0.09773,((161631:0.0107,572828:0.0195)1.000.420:0.05235,(146168:0.03507,594324:0.02507)1.000.421:0.04507)0.995.44:0.04345)1.000.422:0.05442,(((167052:0.01488,442222:0.01291)1.000.423:0.11362,(110474:0.0194,(((224477:0.00955,279874:0.02647)0.972.22:0.01474,(112746:0.03183,360032:0.02839)0.346.3:0.01556)0.999.112:0.03143,(592973:0.0288,(233433:0.01372,94666:0.02056)0.990.26:0.01613)0.967.17:0.01345)0.785.6:0.01548)0.986.20:0.02541)0.896.11:0.0166,((((((142111:0.02101,3391:0.02201)0.055:0.00591,(3392:0.02242,3393:0.05201)0.687.3:0.00739)0.994.35:0.02025,((((385423:0.02845,303568:0.04417)0.923.8:0.01318,(((509525:0.14088,(((583906:0.02835,369386:0.01122)1.000.424:0.03887,(((202729:0.0191,138561:0.02917)1.000.425:0.12437,47887:0.02348)0.984.29:0.02808,42846:0.04477)1.000.426:0.04205)0.965.19:0.02135,155973:0.10236)0.998.69:0.03507)0.116.3:0.00954,(68830:0.03027,(25212:0.03015,(354744:0.01092,(547777:0.01708,(534428:0.01231,562434:0.01414)0.977.27:0.00015)0.983.28:0.00981)0.993.29:0.01864)0.938.13:0.01423)1.000.427:0.1296)0.972.23:0.0167,((((3450:0.01441,170175:0.01213)1.000.428:0.09626,((305139:0.02097,(175064:0.01264,314329:0.00621)0.564:0.00395)0.840.5:0.07345,((111470:0.01867,(97540:0.00532,558128:0.01685)1.000.429:0.03876)0.999.113:0.033,((566475:0.05557,(538813:0.02969,(330819:0.04246,(536414:0.0152,582822:0.02038)0.998.70:0.0245)0.466.4:0.00938)1.000.430:0.05778)1.000.431:0.03903,(333278:0.0739,((306176:0.01469,(563938:0.01658,(539917:0.01332,(542968:0.00835,593245:0.01217)0.945.15:0.0102)0.645.4:0.00476)0.400.4:0.0047)1.000.432:0.03283,(355064:0.04046,(326051:0.03716,553198:0.01977)0.992.33:0.02128)0.200:0.00476)0.973.22:0.01362)0.975.24:0.01783)0.763.6:0.01256)0.995.45:0.01968)0.947.16:0.01383)0.554.2:0.00561,(587811:0.08992,509956:0.09928)0.996.51:0.0243)0.930.15:0.01703,(((530315:0.02816,523310:0.00918)1.000.433:0.14386,(161126:0.06661,522238:0.10779)0.943.22:0.01924)0.765.7:0.01014,((((128298:0.02876,529902:0.02718)0.992.34:0.02011,((133071:0.03996,(532403:0.05224,526853:0.03117)0.445:0.00742)0.910.10:0.00806,((((((518211:0.01007,((521728:0.01131,321142:0.02413)0.683.4:0.00204,517401:0.02526)0.361:0.0045)1.000.434:0.0458,(373273:0.04096,(354548:0.01618,340965:0.02197)0.998.71:0.0299)1.000.435:0.04662)0.977.28:0.01536,517906:0.05544)0.205:0.00652,(516587:0.0618,527237:0.04367)0.964.23:0.0132)0.765.8:0.00643,(338024:0.0551,(134448:0.01735,528946:0.00562)0.340.2:0.06819)0.978.18:0.02262)0.368.3:0.00595,((344325:0.05427,(201849:0.19763,524130:0.1117)0.998.72:0.0531)0.294.4:0.01472,(334822:0.02206,((532345:0.006,520550:0.01547)0.993.30:0.01636,(111684:0.01321,(31682:0.02528,318179:0.01771)0.871.15:0.00352)0.996.52:0.01418)0.593.2:0.00896)0.999.114:0.02907)0.999.115:0.03169)0.960.23:0.0087)0.994.36:0.02063)0.875.11:0.00741,((128911:0.10412,(128835:0.04045,(516356:0.02133,(524229:0.01771,(102889:0.02949,(105269:0.02399,102748:0.02579)0.338.2:0.00676)0.872.19:0.00996)0.922.11:0.01066)0.988.33:0.01576)0.927.11:0.01301)0.990.27:0.02084,((((307798:0.01011,521730:0.00635)1.000.436:0.0306,524613:0.085)0.338.3:0.01099,((((309770:0.02469,532468:0.0529)0.997.54:0.0171,(((((((131823:0.02715,522441:0.03282)0.936.17:0.00989,(24874:0.02187,525501:0.01492)1.000.437:0.03309)0.437:0.00433,(344909:0.01802,546773:0.01348)0.999.116:0.0215)0.973.23:0.00956,(412359:0.05194,319562:0.04072)1.000.438:0.04693)0.476:0.00564,(540877:0.03638,(306940:0.0569,(515908:0.02138,(77335:0.01389,514721:0.02088)1.000.439:0.03927)1.000.440:0.03081)0.934.14:0.01565)0.199:0.0049)0.124:0.0078,33162:0.07021)0.902.7:0.00727,((552010:0.00802,530167:0.01257)1.000.441:0.02771,(327889:0.01619,514322:0.05149)0.304:0.00852)0.834.9:0.00729)0.993.31:0.01303)0.904.8:0.00583,((526522:0.00837,129536:0.01085)0.972.24:0.01116,(313631:0.07049,(407237:0.03137,561038:0.03785)1.000.442:0.02814)0.881.10:0.00958)0.995.46:0.01476)0.534.2:0.00757,(((527089:0.02344,((378739:0.01656,538872:0.01699)1.000.443:0.01937,((99983:0.01242,522715:0.01041)0.895.15:0.00378,(515118:0.01142,317715:0.01118)0.326.3:0.0021)1.000.444:0.02496)0.900.10:0.00616)0.887.13:0.00382,(515836:0.02975,((337021:0.00853,518900:0.01388)1.000.445:0.03776,520311:0.07058)0.870.13:0.0078)0.488.2:0.00414)0.967.18:0.01067,(354184:0.03239,(443968:0.01882,313799:0.02083)0.997.55:0.01539)0.986.21:0.01191)0.855.8:0.00564)0.843.3:0.01346)0.085:0.00936,(((337382:0.12142,138007:0.0978)0.991.29:0.0304,(332418:0.11276,((355652:0.03032,531223:0.05581)1.000.446:0.03724,(519272:0.04836,(340082:0.07786,321537:0.03882)0.645.5:0.01386)0.937.13:0.01678)1.000.447:0.03869)0.960.24:0.01755)0.083.3:0.01071,((314485:0.03099,312286:0.02994)1.000.448:0.06651,((324257:0.05461,306494:0.0336)1.000.449:0.06428,(520163:0.04709,(26708:0.01061,313122:0.02237)0.998.73:0.02552)1.000.450:0.03853)0.916.15:0.022)0.887.14:0.01449)0.996.53:0.01956)0.767.9:0.00535)0.985.16:0.01769)0.990.28:0.02535,(140201:0.2306,(203513:0.0753,(203220:0.01987,200301:0.01318)1.000.451:0.04113)1.000.452:0.12605)0.871.16:0.03751)0.687.4:0.01309)0.973.24:0.01857)0.898.13:0.01443)0.969.18:0.01419)0.964.24:0.01187,(((3538:0.09654,(((3536:0.00906,512561:0.02671)1.000.453:0.02389,((3527:0.0072,((260699:0.02612,42014:0.01342)0.927.12:0.01016,((((((3534:0.03147,559773:0.01417)1.000.454:0.02219,3549:0.01225)0.501.4:0.0022,129242:0.01691)0.872.20:0.00312,((((220978:0.01626,(387654:0.01114,(218011:0.01002,221754:0.02411)0.846.5:0.00385)0.927.13:0.00472)1.000.455:0.02616,(553722:0.03871,533803:0.02463)0.273:0.00564)0.999.117:0.01171,589524:0.00809)0.938.14:0.00369,(((245980:0.0371,(216254:0.03919,(35074:0.05833,((45967:0.08111,72055:0.13627)0.972.25:0.02304,29047:0.06216)0.954.14:0.01651)0.946.16:0.01062)0.700.3:0.0012)0.184.2:0.00115,((((129153:0.01269,3544:0.01432)0.993.32:0.00899,(159942:0.00056,469179:0.04226)0.868.12:0.00077)1.000.456:0.00016,(192539:0.03157,(198230:0.02205,243468:0.02935)0.929.15:0.00686)0.755.8:0.00322)0.928.12:0.00151,173746:0.0414)0.401.2:0.00095)0.869.13:0.00241,560209:0.02581)0.997.56:0.01035)0.173:0.00743)0.867.11:0.00234,(3550:0.01858,(3512:0.01047,128731:0.01804)0.946.17:0.00962)0.814.6:0.01102)0.892.10:0.00152,226053:0.02419)0.143.2:0.00133)0.833.8:0.00186)0.985.17:0.01239,(214073:0.02553,104786:0.02243)0.921.18:0.00977)0.695.4:0.00736)0.959.18:0.00935,(3556:0.03959,(3540:0.00079,532628:0.00014)1.000.457:0.03494)0.529:0.00577)0.836.9:0.00899)0.946.18:0.01115,((277315:0.02925,(130918:0.06249,3497:0.06215)0.988.34:0.02026)0.998.74:0.02363,((545380:0.01705,(200033:0.01274,(3490:0.04581,((3495:0.00532,48130:0.02236)1.000.458:0.05409,(49481:0.00711,569892:0.03505)0.742.5:0.00857)0.960.25:0.00862)0.494.2:0.00409)1.000.459:0.03089)0.730.4:0.00651,((203466:0.01801,225743:0.01045)1.000.460:0.03183,(201626:0.00747,200109:0.00544)0.910.11:0.02062)0.993.33:0.01724)0.494.3:0.00709)0.944.5:0.01038)0.852.5:0.0097,(141161:0.04781,512475:0.02805)0.050:0.00519)0.986.22:0.01706)0.944.6:0.01462,(((69432:0.00959,((141924:0.0141,588592:0.05172)0.996.54:0.01721,((219270:0.00229,(550329:0.00479,85239:0.01589)0.945.16:0.0094)0.658.3:0.00232,229641:0.01018)0.938.15:0.00713)0.999.118:0.019)0.869.14:0.00405,(((563073:0.00014,(155728:0.00966,238123:0.00188)0.979.29:0.00085)0.837.9:0.00157,510753:0.00876)0.710.3:0.01461,(25107:0.01386,(142022:0.01904,248338:0.0089)0.875.12:0.00469)0.391.3:0.00613)0.477.2:0.00627)1.000.461:0.02686,(((((3476:0.07239,(144177:0.00079,29775:0.00014)0.946.19:0.03701)0.838.7:0.00944,(((112309:0.02233,19387:0.03384)0.997.57:0.02142,((((554316:0.01905,(303559:0.0109,544751:0.01485)0.974.18:0.00708)0.737.2:0.00236,109937:0.02894)0.084:0.00086,((252116:0.01326,591956:0.00846)0.917.14:0.00763,344186:0.02112)0.973.25:0.00746)0.995.47:0.01564,(((111321:0.02542,(109523:0.01993,188204:0.0198)0.787.2:0.00612)0.848.8:0.00843,(110571:0.0214,3474:0.03589)0.560.2:0.00792)0.780.6:0.00845,((((((149877:0.01165,(149475:0.01406,146905:0.01274)0.946.20:0.01197)0.920.10:0.00375,(145942:0.01098,(147234:0.01888,(147909:0.01689,(146462:0.02007,(125237:0.01732,147450:0.01325)0.761.5:0.00254)0.313.2:0.00246)0.783.3:0.00161)0.835.4:0.00475)0.963.16:0.00566)0.728.5:0.00152,((148247:0.01982,146233:0.03018)0.625.4:0.00821,(148404:0.02253,147303:0.02183)0.932.18:0.0094)0.861.9:0.00583)0.722.3:0.00704,((146293:0.01637,540043:0.03363)0.810.5:0.00627,((215148:0.02411,110154:0.01736)0.955.16:0.00877,318881:0.03291)0.570.4:0.00344)0.973.26:0.01094)0.729.6:0.00589,((210947:0.01178,(187735:0.01244,3473:0.01034)0.832.7:0.0024)0.450:0.00202,(149804:0.01317,(305395:0.0114,(126325:0.01717,324259:0.01226)0.924.13:0.00424)0.954.15:0.00664)0.989.30:0.01202)0.995.48:0.01256)0.941.25:0.01009,(111173:0.02704,(109889:0.01218,((110805:0.00952,(110833:0.02659,(110187:0.02028,(110867:0.01666,110444:0.01097)0.949.17:0.0118)0.861.10:0.00582)0.940.14:0.00535)0.906.11:0.00523,111094:0.02241)0.842.4:0.00315)0.982.35:0.01477)0.990.29:0.01847)0.921.19:0.00888)0.786.10:0.00774)0.744.7:0.01021)0.930.16:0.00805,(129267:0.03308,(111005:0.0209,3475:0.02252)0.966.21:0.01392)0.862.10:0.01217)0.969.19:0.01102)0.889.9:0.00716,(144620:0.01531,27542:0.01546)1.000.462:0.02833)0.946.21:0.01023,(104053:0.03474,(100373:0.02297,366238:0.01502)0.990.30:0.01726)0.999.119:0.02332)0.482.2:0.00347,(((30150:0.08396,(45082:0.07874,(((137756:0.00187,508915:0.00584)0.775.4:0.06428,((555067:0.05886,((44394:0.01722,((210139:0.00998,83959:0.0206)0.267:0.00588,141745:0.01354)0.868.13:0.0033)0.272.2:0.00198,(137837:0.0405,(46777:0.015,354695:0.0246)0.939.14:0.00746)0.937.14:0.00888)0.995.49:0.02156)1.000.463:0.04759,(((362781:0.0202,(575555:0.01333,592858:0.01339)0.926.21:0.00871)1.000.464:0.03139,(319371:0.02586,(149793:0.00694,3448:0.01496)0.748.5:0.00817)1.000.465:0.02758)0.981.28:0.01304,((296128:0.04537,(227965:0.06058,(563800:0.02998,558119:0.03393)0.999.120:0.02735)0.999.121:0.02943)0.197.3:0.01053,(574214:0.03892,(((((329899:0.01789,(((((328963:0.01511,101999:0.00565)0.839.7:0.00364,(((305369:0.00774,((((30625:0.01387,370181:0.01174)0.773.5:0.00536,(317794:0.01273,(325174:0.01493,556625:0.00747)0.819.4:0.00465)0.962.14:0.00808)0.846.6:0.00154,588707:0.0227)0.879.12:0.00214,(582772:0.01528,568603:0.01671)0.766.11:0.00205)0.787.3:0.00659)0.903.15:0.00258,((583992:0.01895,101837:0.01608)0.753.10:0.00384,348903:0.00749)0.986.23:0.01044)0.349:0.0008,(((361147:0.02473,(344558:0.00913,(549748:0.02153,332808:0.00489)0.821.4:0.0065)0.916.16:0.00157)0.722.4:0.00016,583758:0.02957)0.755.9:0.00096,(102592:0.00625,347931:0.01283)0.943.23:0.00299)0.868.14:0.01032)0.891.9:0.00254)0.937.15:0.00396,((305627:0.01072,(347310:0.01223,(532823:0.00992,532647:0.01423)0.706.2:0.00435)0.889.10:0.00403)0.949.18:0.0062,(199329:0.00926,549656:0.03415)0.936.18:0.00587)0.227.3:0.00708)0.732.4:0.00319,346339:0.02266)0.895.16:0.00731,((306869:0.00882,(326137:0.01507,(106691:0.00166,340216:0.00668)0.968.12:0.00542)0.872.21:0.00315)0.709.2:0.00084,106374:0.01151)0.714.2:0.00209)0.798.4:0.00201)0.651.3:0.00645,(338106:0.00397,552055:0.01657)1.000.466:0.01836)0.753.11:0.0022,591206:0.02394)0.200.2:0.00331,573523:0.02043)0.855.9:0.00455,107146:0.02274)1.000.467:0.02044)0.974.19:0.01465)0.906.12:0.00853)0.836.10:0.00518)0.023:0.00439)0.998.75:0.02767,3479:0.05401)0.379.4:0.01058)0.242.3:0.00793)0.764.7:0.01476,(201125:0.13597,((3466:0.02215,(3468:0.00799,(140203:0.01317,3467:0.00744)0.969.20:0.00695)0.988.35:0.01273)0.934.15:0.01062,((((351106:0.01603,((234919:0.01021,233758:0.01041)0.933.15:0.00349,138411:0.01693)0.741.6:0.00121)0.627:0.00183,(((3462:0.00873,(230058:0.01411,(155941:0.01169,569927:0.02172)0.721.3:0.00088)0.898.14:0.00172)0.979.30:0.00387,229165:0.01436)0.824.2:0.00164,587965:0.02644)0.691.5:0.00263)0.813.10:0.0024,((247611:0.01036,(48463:0.00779,234436:0.02781)0.488.3:0.00161)0.843.4:0.00579,534046:0.00603)0.692:0.00245)0.896.12:0.00588,231091:0.01657)0.824.3:0.00737)0.226:0.00334)0.999.122:0.03561)0.304.2:0.00728,(28521:0.01809,47632:0.0303)1.000.468:0.04987)0.907.13:0.0065)0.591.3:0.00997)0.997.58:0.0157)0.943.24:0.01391)0.998.76:0.02285,(((((185678:0.05516,(253958:0.03974,((570149:0.01619,(242305:0.00731,249359:0.02452)0.901.8:0.00822)0.964.25:0.01017,(554203:0.02362,519568:0.01701)0.943.25:0.00848)0.624:0.00552)0.956.15:0.00934)0.974.20:0.00776,(((((136196:0.01735,567570:0.01705)0.992.35:0.01572,(99196:0.06272,(554238:0.01922,539884:0.01544)1.000.469:0.02698)0.913.16:0.00835)0.989.31:0.01143,((592401:0.04274,((188868:0.0017,536917:0.00252)1.000.470:0.01827,(548460:0.00802,265411:0.00962)0.982.36:0.01167)0.999.123:0.02194)0.972.26:0.01268,((240402:0.05853,(((((251680:0.00014,(348181:0.01865,553975:0.04501)0.978.19:0.00867)1.000.471:0.02918,((249216:0.01532,188968:0.01336)0.951.20:0.00631,(((194171:0.01775,(325973:0.01641,556036:0.01282)0.938.16:0.00766)0.138.2:0.00536,(195665:0.02378,554341:0.00707)0.986.24:0.01066)0.832.8:0.00259,184548:0.01727)0.981.29:0.00828)0.988.36:0.01303)0.212:0.007,(225630:0.02939,(353695:0.01122,537606:0.00949)1.000.472:0.03088)0.977.29:0.01072)0.170:0.00279,((169750:0.01453,196547:0.01646)0.988.37:0.01113,(94733:0.01733,(172516:0.02339,171931:0.00625)0.971.18:0.00784)0.998.77:0.02106)0.611.3:0.00386)0.890.12:0.00535,(159246:0.03111,(568166:0.01362,(577524:0.00102,159427:0.01836)0.017:0.00143)0.889.11:0.00603)1.000.473:0.01912)0.614:0.0069)0.368.4:0.00727,188528:0.03521)0.575.2:0.0088)0.333.2:0.00428)0.895.17:0.00293,(((183285:0.02447,(190474:0.01425,535035:0.02195)0.955.17:0.01008)0.958.21:0.01394,(239741:0.02929,579823:0.02848)0.963.17:0.01576)0.708.4:0.0078,((558034:0.01834,(177850:0.02502,3280:0.008)1.000.474:0.03231)1.000.475:0.03077,(((277811:0.01135,184822:0.02039)1.000.476:0.02861,(((189129:0.00269,3275:0.01398)0.866.6:0.00683,(170083:0.00786,169844:0.02658)0.973.27:0.00984)0.926.22:0.012,(559534:0.02936,(194024:0.01521,189054:0.003)0.806.4:0.01031)0.972.27:0.01541)0.946.22:0.00858)0.943.26:0.01012,(278028:0.03094,169748:0.02329)0.987.37:0.01383)0.891.10:0.00673)0.251.2:0.00138)0.984.30:0.00891)0.964.26:0.0063,((((585449:0.05586,(571008:0.04402,(3206:0.02802,(59101:0.01593,(59109:0.00593,373820:0.0034)0.891.11:0.00291)0.999.124:0.0194)0.488.4:0.00415)0.766.12:0.00752)0.994.37:0.01884,(((107506:0.01903,(590001:0.01012,513156:0.01375)0.997.59:0.01756)0.864.8:0.00768,(278670:0.01711,258105:0.02421)0.997.60:0.01754)0.980.18:0.01464,((((307381:0.01514,574890:0.0205)1.000.477:0.02648,((247622:0.01079,(47726:0.01174,252166:0.00988)0.458:0.00398)0.871.17:0.02793,(279660:0.02751,(163111:0.01304,(565556:0.00558,279122:0.01774)0.829.7:0.00363)0.973.28:0.01143)0.859.8:0.01348)0.440.2:0.00567)0.972.28:0.0086,(58439:0.05018,82269:0.03003)0.959.19:0.00893)0.654.2:0.00267,(194616:0.03716,(((347009:0.00056,271081:0.06134)0.932.19:0.02734,(548398:0.04336,192780:0.01558)0.751:0.00442)0.926.23:0.01022,((514114:0.02021,(560624:0.0078,585775:0.00902)0.505.2:0.00238)0.476.2:0.00204,(112426:0.00936,143213:0.01551)0.681.5:0.00349)1.000.478:0.04703)0.867.12:0.01105)0.899.14:0.0075)0.854.6:0.00497)0.871.18:0.01021)0.898.15:0.00713,(((((((((565214:0.01752,237105:0.01721)1.000.479:0.03176,(248532:0.01432,210875:0.02627)0.986.25:0.01739)0.460.4:0.00847,(238675:0.01859,((((((46080:0.00987,3207:0.00473)0.464:0.00923,(45178:0.01076,194304:0.01372)0.778.6:0.00738)0.937.16:0.00521,42381:0.00464)0.882.10:0.00429,(310390:0.01169,(45775:0.01164,(24838:0.0177,(80524:0.00546,(24335:0.01739,141154:0.01066)0.937.17:0.00433)0.728.6:0.00281)0.809.7:0.00285)0.707.6:0.00613)0.645.6:0.00346)1.000.480:0.02207,24455:0.0295)0.725.4:0.00618,(167278:0.01671,3208:0.0025)0.890.13:0.01803)0.885.10:0.00688)0.873.9:0.00565)0.446.3:0.0056,(190488:0.03096,(587573:0.01735,17093:0.02887)0.995.50:0.01857)0.926.24:0.00896)0.723.5:0.00464,331217:0.03354)0.943.27:0.006,(143320:0.02848,((((50782:0.00677,565917:0.01346)1.000.481:0.02796,71748:0.03966)0.780.7:0.00808,(3232:0.01979,56680:0.02438)0.939.15:0.01023)0.384:0.00721,39534:0.04518)0.889.12:0.01085)0.998.78:0.01332)0.479.3:0.0023,((((29085:0.01436,(62923:0.0068,(226595:0.00419,3231:0.00085)0.819.5:0.00221)1.000.482:0.03333)0.884.12:0.01021,(551634:0.00014,210194:0.0003)0.990.31:0.01101)0.893.6:0.00659,(576348:0.00683,(357897:0.00791,(221509:0.01092,563811:0.02353)0.884.13:0.00458)0.935.8:0.00474)0.996.55:0.01235)1.000.483:0.03957,((201540:0.03114,203219:0.02971)0.998.79:0.01908,(62924:0.00908,(268561:0.00014,100950:0.03734)0.998.80:0.02614)0.999.125:0.01737)0.222.2:0.0046)0.927.14:0.00614)0.878.12:0.00423,((237210:0.04237,((((236194:0.03073,((471226:0.00242,3112:0.03927)1.000.484:0.00136,3241:0.03325)0.000.17:0.03071)0.992.36:0.01773,356043:0.01272)0.799.3:0.00499,(531796:0.02132,49254:0.04001)0.894.15:0.00924)0.934.16:0.00723,3240:0.01956)0.498.3:0.00376)0.969.21:0.01161,((137041:0.01922,(566848:0.02608,(149021:0.02622,(149911:0.03609,(147523:0.00087,545082:0.0007)0.980.19:0.00969)0.990.32:0.01304)0.016:0.00444)0.903.16:0.00465)1.000.485:0.01796,((308876:0.03138,271091:0.02104)0.914.11:0.00785,(93641:0.01577,(67714:0.02028,(106754:0.02509,(165105:0.01038,35028:0.01481)0.395.3:0.00596)0.413.2:0.00435)0.938.17:0.0048)0.947.17:0.00641)0.987.38:0.01133)0.823.3:0.00598)0.963.18:0.00877)0.870.14:0.00652,((141959:0.02383,(3248:0.00942,277925:0.01892)0.962.15:0.00897)1.000.486:0.03824,(3247:0.03194,(175235:0.01898,200320:0.03463)0.997.61:0.02101)0.954.16:0.01078)0.993.34:0.01719)0.927.15:0.01315)0.959.20:0.01164,(((((((172573:0.02743,((182352:0.02414,241102:0.02344)0.760.9:0.00539,((549654:0.0103,167334:0.01083)1.000.487:0.01993,((((((536201:0.00551,509907:0.00492)0.932.20:0.02029,129771:0.02538)0.244:0.00535,171723:0.01524)0.955.18:0.00993,511951:0.01868)0.976.21:0.00708,(224486:0.03211,166256:0.01106)0.995.51:0.01277)0.229:0.00157,561309:0.00575)0.956.16:0.00768)0.604.2:0.00364)0.923.9:0.00545)0.953.11:0.00571,((573191:0.02815,505911:0.00389)1.000.488:0.03221,((71326:0.02771,((138616:0.0066,(386885:0.01943,512034:0.02513)0.704.8:0.00658)0.974.21:0.00926,405112:0.02268)0.720.3:0.00637)0.963.19:0.0094,(554144:0.02725,507150:0.02721)0.835.5:0.00466)0.978.20:0.01451)0.645.7:0.00436)0.969.22:0.00769,((193533:0.01721,509617:0.02087)0.999.126:0.02027,(355265:0.01643,(571269:0.01107,558201:0.01854)0.999.127:0.02739)0.983.29:0.01354)0.299.2:0.00458)0.834.10:0.00339,206639:0.05223)0.460.5:0.00465,((351877:0.0212,(278544:0.01238,277671:0.01574)1.000.489:0.02071)0.563.4:0.00655,((3094:0.03038,99755:0.02764)0.999.128:0.02067,(277783:0.0404,((342090:0.00619,443892:0.00293)0.999.129:0.02102,(137925:0.01092,279188:0.03884)0.222.3:0.00956)0.957.30:0.01278)0.280:0.0063)0.872.22:0.00764)0.941.26:0.00771)0.846.7:0.00335,(((250650:0.06118,(34564:0.01503,72915:0.01268)1.000.490:0.02326)0.961.21:0.01459,(427017:0.01047,159142:0.00645)0.999.130:0.02168)0.898.16:0.00746,(((((278296:0.01445,551186:0.01373)0.983.30:0.0157,(225760:0.01613,278136:0.02011)0.874.15:0.00815)0.771.9:0.00251,(278185:0.02003,278371:0.01707)0.873.10:0.00697)0.847.10:0.0036,161302:0.02437)0.960.26:0.00603,(((278573:0.01107,594327:0.02067)1.000.491:0.02048,((263124:0.03072,46480:0.0193)1.000.492:0.02438,((45181:0.02466,21210:0.00474)0.998.81:0.01412,(563194:0.02112,(239727:0.00924,36962:0.01289)0.622:0.00449)0.926.25:0.00781)0.768.3:0.00773)0.223.2:0.00879)0.825.6:0.00793,(103589:0.00244,94786:0.00484)1.000.493:0.05321)0.852.6:0.00354)0.990.33:0.01201)0.922.12:0.00618)0.981.30:0.01211,(((136145:0.00557,(3262:0.00838,549188:0.01268)0.844.6:0.00799)0.989.32:0.03184,((181870:0.01574,(186537:0.00575,(180477:0.02082,554079:0.00789)0.872.23:0.00627)0.985.18:0.01327)0.979.31:0.00916,((((((101565:0.02834,(189042:0.00691,257778:0.00761)0.927.16:0.00703)0.841.6:0.01539,(70960:0.02471,(45088:0.0047,(349281:0.00714,219193:0.00183)0.871.19:0.00248)0.950.30:0.00782)0.941.27:0.0097)0.672:0.01058,((3089:0.00014,175060:0.00028)1.000.494:0.03815,136186:0.02144)0.972.29:0.01333)0.462.2:0.00364,(105809:0.02543,(((185766:0.00755,(218884:0.01759,44224:0.01918)1.000.495:0.02334)0.853.6:0.0044,106108:0.02115)0.825.7:0.00261,158785:0.01161)0.996.56:0.01416)0.976.22:0.00874)0.677.3:0.0041,(50096:0.03505,((((74605:0.00864,266459:0.0155)0.414:0.00221,(((((111394:0.01686,111492:0.00971)0.279.3:0.01081,(199024:0.01769,258214:0.00258)0.999.131:0.01308)0.924.14:0.00547,(282462:0.00624,108006:0.0176)0.994.38:0.01185)0.884.14:0.00338,(139853:0.00367,583809:0.02158)0.964.27:0.00014)0.988.38:0.00992,(((105704:0.01155,(527300:0.01746,(278305:0.03302,((60833:0.00134,568411:0.00249)0.757.4:0.01485,(157669:0.02152,(551873:0.00156,323687:0.00057)0.981.31:0.00724)0.819.6:0.00523)0.291.3:0.00324)0.992.37:0.01086)0.307.3:0.00305)0.171:0.00312,73220:0.00665)0.989.33:0.01035,(108008:0.02194,108357:0.03164)0.937.18:0.00703)0.951.21:0.00873)0.921.20:0.00534)0.929.16:0.00381,((105653:0.02159,(178988:0.01763,((147260:0.01647,(209017:0.02364,542307:0.0105)0.892.11:0.01008)0.888.5:0.00255,554414:0.02699)0.998.82:0.0111)0.290.3:0.00498)0.944.7:0.00684,((142304:0.01843,101785:0.00859)0.941.28:0.00626,(((((148662:0.00438,((((238658:0.00914,(193916:0.01099,45521:0.02202)0.948.9:0.009)0.988.39:0.00939,((253820:0.00639,(((((348190:0.00644,23731:0.00906)0.748.6:0.0017,350292:0.07708)0.984.31:0.00631,279599:0.00903)0.980.20:0.00696,((278176:0.00049,235567:0.00257)0.000.18:0.00523,43551:0.0155)0.824.4:0.00732)0.997.62:0.01154,48640:0.01781)0.108:0.00014)0.758.6:0.00096,((3098:0.00758,245604:0.01152)0.436.2:0.00772,(101942:0.00609,(275504:0.00242,(265473:0.00475,(592912:0.00232,514364:0.01349)0.838.8:0.00154)0.994.39:0.01223)0.998.83:0.01348)0.998.84:0.01226)0.957.31:0.00612)0.830.5:0.00225)0.847.11:0.00173,178178:0.01302)0.877.15:0.00479,(237315:0.01358,(548891:0.02065,182078:0.00633)0.767.10:0.00558)0.979.32:0.0077)0.732.5:0.00077)0.972.30:0.00556,199561:0.00828)0.959.21:0.00483,(268782:0.00306,245919:0.02181)0.815.3:0.00261)0.872.24:0.00153,(198952:0.02121,333578:0.00536)0.987.39:0.00641)0.866.7:0.00158,267299:0.01188)0.952.18:0.00406)0.613:0.00301)0.778.7:0.00158)0.229.2:0.00195,170580:0.01456)0.972.31:0.00737)0.989.34:0.01067)0.787.4:0.00785,(((((261638:0.0077,((312572:0.00807,(169488:0.00569,33939:0.02014)0.891.12:0.00395)0.756.5:0.00124,((200652:0.00738,264448:0.00246)0.758.7:0.00077,3097:0.01921)0.827.5:0.0067)0.995.52:0.01434)0.999.132:0.02571,(166201:0.03499,((186291:0.01079,192607:0.005)0.837.10:0.00193,102789:0.01432)1.000.496:0.02929)0.313.3:0.00844)0.898.17:0.00938,180770:0.02556)0.519.2:0.00992,184879:0.00735)0.931.12:0.00851,(194216:0.00604,569165:0.02247)0.985.19:0.01338)0.925.15:0.01002)0.981.32:0.01079)0.997.63:0.01479)0.948.10:0.01048,((248773:0.02136,244883:0.00221)1.000.497:0.0326,40082:0.03977)0.768.4:0.00764)0.985.20:0.01134)0.564.2:0.00639)0.957.32:0.00793)0.939.16:0.00454)0.602.3:0.00332,((((3289:0.02369,((((101109:0.01611,3293:0.00732)0.974.22:0.0082,563165:0.03163)0.760.10:0.00575,186253:0.01003)0.890.14:0.00272,(3259:0.01173,589027:0.01308)0.960.27:0.00509)0.833.9:0.00733)0.997.64:0.0206,((328295:0.00935,166453:0.02701)1.000.498:0.04617,278727:0.03868)0.742.6:0.01301)0.993.35:0.01775,(((533880:0.0333,(56488:0.01751,344914:0.0284)0.553:0.00965)0.606.3:0.00491,(62919:0.02378,99912:0.03249)0.969.23:0.0125)0.787.5:0.00582,((193060:0.00119,(549739:0.00214,((144426:0.01027,546043:0.03203)0.103:0.00785,(38561:0.0165,(575713:0.01433,152148:0.01236)0.995.53:0.01112)0.600:0.0076)0.992.38:0.00888)0.834.11:0.00092)0.993.36:0.04329,(319713:0.02849,(161174:0.02479,((((278517:0.02769,369012:0.02645)0.984.32:0.01342,546448:0.02091)0.800.3:0.00185,((243796:0.03117,(582814:0.01282,240428:0.01359)0.904.9:0.01)0.917.15:0.0082,(69650:0.01587,((246287:0.00454,533744:0.01533)1.000.499:0.01524,(242781:0.00889,278809:0.01356)0.431.2:0.00225)0.984.33:0.00918)0.960.28:0.0098)0.904.10:0.00879)0.961.22:0.00968,257398:0.01247)0.767.11:0.0078)0.891.13:0.01331)0.155.2:0.00713)0.328.2:0.00305)0.828.7:0.00501)0.937.19:0.0069,(((361402:0.03195,583080:0.01684)0.999.133:0.02483,((((((550168:0.00955,54464:0.00109)0.965.20:0.00989,415283:0.02084)0.999.134:0.01188,512223:0.00238)0.431.3:0.0097,(((239262:0.01582,(((335985:0.04375,339509:0.03521)0.935.9:0.0026,236491:0.00214)0.891.14:0.01862,(((155313:0.00202,3344:0.0248)0.538:0.00493,(577243:0.01439,((((142703:0.00967,543597:0.00632)0.923.10:0.00164,202572:0.02304)0.800.4:0.00032,3333:0.01593)0.819.7:0.00501,558480:0.03169)0.875.13:0.00202)0.844.7:0.00525)0.939.17:0.00459,330378:0.00812)0.959.22:0.00494)0.925.16:0.00299)0.530.3:0.00271,144130:0.01352)0.946.23:0.00448,((533688:0.02108,404788:0.02896)0.749.4:0.00619,581223:0.0041)0.982.37:0.01821)0.951.22:0.00591)0.976.23:0.0118,60158:0.02857)1.000.500:0.02707,(244826:0.04428,((3308:0.02295,3307:0.06685)1.000.501:0.04205,142996:0.00776)0.825.8:0.01589)0.969.24:0.01581)0.986.26:0.01588)0.956.17:0.00998,(62921:0.06687,(587725:0.04233,(151457:0.01234,219119:0.01914)0.942.18:0.01237)0.973.29:0.0167)0.805.9:0.00813)0.837.11:0.00499)0.983.31:0.01122)0.567.3:0.00639,(62920:0.02733,107856:0.02394)0.999.135:0.02436)0.989.35:0.01397,((159447:0.04919,(216560:0.01588,3159:0.02019)0.998.85:0.02757)1.000.502:0.0395,(108670:0.05092,((212812:0.03001,237187:0.00392)0.981.33:0.01669,(152906:0.02502,154574:0.01518)0.929.17:0.01072)1.000.503:0.04179)0.339:0.0153)0.861.11:0.00788)0.943.28:0.00876)0.961.23:0.01303,(((241636:0.00015,218516:0.00894)1.000.504:0.04964,(16700:0.01883,(30337:0.01088,62918:0.04599)0.986.27:0.01718)0.903.17:0.01179)0.803.4:0.01376,(((190926:0.00814,140593:0.01622)0.287.2:0.01201,(180764:0.00365,112046:0.04521)0.630.2:0.00665)1.000.505:0.03453,(25314:0.05569,274997:0.03919)0.992.39:0.02145)0.529.2:0.00657)0.992.40:0.02162)1.000.506:0.0477,(((451807:0.12647,(((225363:0.01555,208107:0.02259)0.975.25:0.01142,((((328994:0.017,226346:0.01019)0.956.18:0.00933,(83349:0.01286,(221107:0.00732,278744:0.00746)0.998.86:0.0165)0.381.2:0.00241)0.511.3:0.00618,236597:0.02852)0.981.34:0.01018,74727:0.00732)0.444.2:0.00505)1.000.507:0.04866,(237637:0.03281,(343561:0.02891,516884:0.01171)0.443.2:0.00564)1.000.508:0.03419)0.997.65:0.02893)0.982.38:0.02518,(143658:0.06881,((589410:0.01145,((391054:0.00641,(581388:0.00968,(550894:0.00612,(316111:0.01324,542830:0.00806)1.000.509:0.01595)0.902.8:0.00337)0.863.13:0.00229)0.910.12:0.00312,577667:0.00723)0.920.11:0.00611)0.991.30:0.01702,((58155:0.02094,(342685:0.02328,(230921:0.01581,((351091:0.01559,310781:0.01024)0.953.12:0.00785,(278915:0.00173,559149:0.01749)0.871.20:0.00315)1.000.510:0.01835)0.998.87:0.01518)0.972.32:0.00941)0.209:0.00619,((111712:0.00643,(279048:0.00865,(((207400:0.01306,301924:0.02554)0.807:0.00864,(320115:0.01453,(569807:0.00433,((290683:0.02106,531787:0.00699)0.991.31:0.01055,(299551:0.01091,381666:0.01067)0.988.40:0.00912)0.550.4:0.00196)0.894.16:0.00524)0.973.30:0.01017)0.971.19:0.01182,(((((170218:0.00886,((303552:0.00163,301614:0.00447)0.991.32:0.00816,555951:0.00719)0.848.9:0.00398)0.982.39:0.00818,(269386:0.00016,404404:0.00983)0.947.18:0.01164)0.584.2:0.0024,((528576:0.02388,526124:0.00734)0.132.4:0.01206,((341111:0.01002,340514:0.01584)0.985.21:0.00941,(206256:0.01296,(294659:0.01008,(278650:0.00382,297585:0.00905)0.382.4:0.00014)0.981.35:0.0073)0.968.13:0.00636)0.870.15:0.00246)0.129.4:0.003)0.988.41:0.0111,(556936:0.01368,(592551:0.00221,151120:0.03252)1.000.511:0.02469)0.768.5:0.00443)0.761.6:0.00388,(378828:0.0205,180684:0.01688)0.957.33:0.00955)0.288:0.00464)0.947.19:0.01015)0.927.17:0.0084)0.943.29:0.00627,((373357:0.00983,(301639:0.02264,528691:0.00371)0.350.3:0.00199)0.996.57:0.01175,(571689:0.02085,((578358:0.01272,537033:0.0169)0.995.54:0.01008,(549658:0.00617,588446:0.02173)1.000.512:0.01914)0.746.3:0.0014)0.823.4:0.00392)0.816.7:0.00373)0.943.30:0.007)0.971.20:0.01556)1.000.513:0.03452)0.987.40:0.02519)0.926.26:0.01626,((538983:0.0135,89151:0.03438)0.995.55:0.02401,((562510:0.00814,((115117:0.00766,542202:0.00883)0.861.12:0.00166,556990:0.01605)0.829.8:0.00342)0.997.66:0.01571,(((115007:0.01484,147415:0.02431)0.999.136:0.02123,(582668:0.01915,227019:0.01815)0.951.23:0.0094)0.872.25:0.00559,89473:0.02054)0.778.8:0.00628)0.962.16:0.01633)1.000.514:0.04643)0.981.36:0.02182)0.997.67:0.03209)0.988.42:0.02678)0.911.12:0.01557)0.993.37:0.0302,((((((302768:0.01294,((528799:0.02845,(538695:0.02499,(156305:0.01958,152050:0.00374)0.952.19:0.0093)0.943.31:0.0125)0.953.13:0.00942,((11152:0.00481,47237:0.0314)0.829.9:0.00329,(579615:0.00871,(251647:0.00441,(571654:0.01094,(100267:0.00771,40818:0.01237)0.997.68:0.01163)0.976.24:0.00816)0.855.10:0.00523)0.768.6:0.00586)0.978.21:0.01219)1.000.515:0.05594)0.931.13:0.01061,(247748:0.00063,106788:0.03402)0.604.3:0.00428)0.992.41:0.02477,(106632:0.08896,(((93761:0.00014,(71173:0.01894,27536:0.01456)0.239:0.00829)0.933.16:0.01398,(67795:0.0046,(184732:0.01786,(210920:0.00264,141137:0.0054)0.928.13:0.00543)0.991.33:0.01066)0.865.13:0.00919)0.999.137:0.02228,((((114816:0.00301,581448:0.03527)0.956.19:0.0107,(78064:0.02414,135519:0.01359)0.703.5:0.01083)1.000.516:0.06146,((34981:0.00397,557968:0.01049)0.972.33:0.0098,579980:0.00959)1.000.517:0.07432)0.998.88:0.03288,(((252330:0.07008,(137208:0.02726,(((10980:0.0598,(((111336:0.00423,103154:0.01846)0.752.6:0.01853,(64325:0.0147,37024:0.02643)0.561.5:0.00519)1.000.518:0.02909,(73138:0.00043,(516347:0.00438,(11080:0.01899,((11085:0.02067,132728:0.02614)0.999.138:0.0208,11019:0.02068)0.850.11:0.01767)0.763.7:0.00335)0.753.12:0.01353)0.249:0.01848)0.976.25:0.01227)0.947.20:0.00958,((114024:0.01675,(114186:0.01771,(113235:0.0156,548699:0.03363)0.116.4:0.00759)0.715.6:0.01599)0.949.19:0.0056,(((545809:0.01489,(266613:0.01767,11039:0.00695)0.658.4:0.00671)0.959.23:0.00576,(11065:0.0043,11081:0.01881)0.375.2:0.0134)0.704.9:0.0009,(((113412:0.01579,110529:0.03252)0.336.3:0.00066,((16301:0.01475,((27055:0.03939,26677:0.01833)0.757.5:0.00109,(171157:0.00187,10960:0.00865)0.972.34:0.00595)0.953.14:0.00841)0.973.31:0.00918,10993:0.02044)0.974.23:0.01209)0.969.25:0.00402,((105695:0.0063,(68999:0.0136,470494:0.01467)0.856.7:0.00156)1.000.519:0.00949,(52155:0.00634,81166:0.01173)0.937.20:0.00313)0.956.20:0.0043)0.949.20:0.004)0.703.6:0.00074)0.703.7:0.00142)0.847.12:0.00209,(((36371:0.01399,205778:0.0198)0.988.43:0.0109,(470510:0.01034,((101810:0.00613,52884:0.01745)1.000.520:0.01357,470487:0.0123)0.583.4:0.00165)0.877.16:0.00174)0.387.2:0.0009,49735:0.01754)0.960.29:0.00567)0.953.15:0.01432)1.000.521:0.04019)0.817.8:0.01249,((204168:0.1097,((110074:0.02867,((93762:0.02626,((49390:0.00156,(568548:0.00854,538468:0.01557)0.919.12:0.01381)0.985.22:0.01038,104251:0.00778)0.988.44:0.01016)0.988.45:0.01047,((162164:0.01144,326035:0.0227)0.887.15:0.00671,(171429:0.00516,329225:0.02759)0.987.41:0.00359)0.366:0.00629)0.991.34:0.01236)0.997.69:0.02109,(((113750:0.01346,((217716:0.01449,206907:0.01702)0.985.23:0.0128,(((260773:0.0261,271688:0.01039)0.997.70:0.01665,(563303:0.01122,((((340421:0.00491,82461:0.03405)0.571.3:0.00813,(224044:0.01022,57484:0.0412)0.760.11:0.01815)0.937.21:0.00688,(112883:0.00693,101721:0.00844)0.795.8:0.00344)0.582.3:0.00158,((209624:0.00825,(((((((103892:0.01195,100516:0.01323)0.754.7:0.00857,(38197:0.01481,73836:0.01175)0.896.13:0.0183)0.419.2:0.00015,546926:0.00368)1.000.522:0.00886,(81472:0.00014,(11101:0.00428,((43869:0.0242,(70730:0.00664,70203:0.00961)0.862.11:0.00374)0.957.34:0.00616,572747:0.00571)0.936.19:0.00014)0.834.12:0.00081)0.929.18:0.00319)0.748.7:0.00101,98692:0.01488)0.940.15:0.00594,(142365:0.01308,48010:0.02442)0.799.4:0.00145)0.876.12:0.0028,(108823:0.01689,((((237892:0.03125,112344:0.01917)0.934.17:0.00781,(11109:0.01481,(((590100:0.00691,(222158:0.02238,144680:0.0068)0.940.16:0.00446)1.000.523:0.01308,(169835:0.00585,105327:0.01359)0.687.5:0.00229)0.740.7:0.00084,(144864:0.01605,568631:0.00359)0.544:0.00313)0.961.24:0.00941)0.959.24:0.01001)0.649.2:0.00294,(11114:0.01646,533317:0.02087)0.957.35:0.00718)0.947.21:0.0075,((510665:0.00803,96203:0.01242)0.779.4:0.00943,(565266:0.00504,141130:0.01549)0.760.12:0.00076)0.796.6:0.00657)0.831.4:0.00014)0.815.4:0.01498)0.547:0.0018)0.960.30:0.00702,(((142794:0.00791,(57482:0.01895,108911:0.00951)0.938.18:0.00615)0.943.32:0.00503,(278960:0.02406,(306303:0.00793,(98903:0.00633,(535076:0.00092,214786:0.01664)0.961.25:0.01464)0.972.35:0.00821)0.964.28:0.00765)0.938.19:0.0065)0.936.20:0.00489,(((((((424252:0.01598,11103:0.01247)0.867.13:0.00289,(((573260:0.01022,541356:0.01012)0.928.14:0.00256,549271:0.02382)0.736.5:0.00411,((593648:0.024,574279:0.01497)0.889.13:0.00716,529673:0.00884)0.963.20:0.00435)0.838.9:0.00051)0.890.15:0.0018,(571284:0.01697,306610:0.00239)0.971.21:0.01157)0.876.13:0.00247,(581740:0.01216,(134605:0.0041,142733:0.02779)0.980.21:0.0067)0.709.3:0.00015)0.811.4:0.00318,(113264:0.00533,(113830:0.00992,11113:0.0227)0.995.56:0.01394)0.982.40:0.00802)0.983.32:0.00538,283715:0.01594)0.921.21:0.00572,527383:0.00632)0.966.22:0.00565)0.968.14:0.00501)0.950.31:0.00545)0.935.10:0.0072)0.998.89:0.01661)0.993.38:0.01104,82215:0.01682)0.938.20:0.00669)0.955.19:0.007)0.901.9:0.00413,581232:0.02457)0.949.21:0.01098,((566005:0.0086,279233:0.01737)1.000.524:0.0418,(560545:0.02495,((((550768:0.00423,(343001:0.0173,536031:0.01977)0.997.71:0.01077)0.962.17:0.00637,510699:0.01763)0.569:0.00471,(205139:0.00987,516091:0.01677)0.974.24:0.00775)0.976.26:0.0059,(273995:0.00205,146253:0.00186)0.985.24:0.00016)0.623.2:0.00466)0.987.42:0.01495)0.692.2:0.01307)0.980.22:0.01843)0.917.16:0.01315)0.991.35:0.01743,(((11150:0.01912,(290493:0.02277,11124:0.01324)0.980.23:0.01642)1.000.525:0.04096,(((101619:0.01893,(105748:0.02349,67789:0.00174)0.836.11:0.00209)0.848.10:0.01728,(79600:0.03113,57475:0.02084)0.506:0.00704)0.968.15:0.0144,(((103492:0.00639,556468:0.02707)0.777.8:0.0088,((107393:0.01692,((11136:0.0008,157898:0.01211)0.932.21:0.01811,108123:0.02988)0.942.19:0.00677)0.105.2:0.00162,100136:0.02825)0.855.11:0.00014)0.993.39:0.01339,(((296261:0.00492,79345:0.0217)0.229.3:0.00165,79406:0.01397)0.956.21:0.00513,(158924:0.02494,300292:0.03209)0.919.13:0.0057)0.987.43:0.00917)0.966.23:0.01073)0.807.2:0.01433)0.961.26:0.01373,(((((((((103545:0.02624,(103408:0.01208,61874:0.00959)0.556.2:0.00405)0.797.7:0.00661,292405:0.01213)0.989.36:0.01549,67794:0.01139)0.868.15:0.00787,(67780:0.0167,214381:0.01863)0.507:0.00335)0.884.15:0.00658,(79474:0.02032,(528299:0.0163,67790:0.00744)0.370:0.00081)0.990.34:0.01067)0.955.20:0.00606,(((79418:0.01943,((((108125:0.0227,20059:0.00536)0.973.32:0.00931,(138321:0.01305,103225:0.01732)0.933.17:0.00472)0.923.11:0.00381,79140:0.019)0.756.6:0.00099,(67791:0.00638,268986:0.01683)0.934.18:0.00459)0.651.4:0.00228)0.988.46:0.00752,((11128:0.02175,(190937:0.00476,159736:0.02366)0.605.3:0.00473)0.976.27:0.00824,(175519:0.02242,((155145:0.01501,537179:0.00935)0.958.22:0.00417,((194025:0.00409,((538726:0.01286,154096:0.0152)0.941.29:0.01435,((((((57471:0.01724,57470:0.01072)0.908.15:0.00674,((139854:0.02758,250576:0.00078)0.938.21:0.00891,107839:0.03317)0.915.13:0.00324)0.957.36:0.00464,(((((82535:0.00251,(558481:0.00791,((98179:0.00423,((96015:0.0291,((101326:0.00402,(61878:0.00272,544300:0.00769)0.929.19:0.00452)0.976.28:0.00633,((536599:0.01296,(104446:0.00718,100222:0.01459)0.886.10:0.00266)0.966.24:0.00628,99597:0.02388)0.933.18:0.01461)0.981.37:0.00053)0.978.22:0.00554,(115002:0.00801,90529:0.00682)0.000.19:0.00286)0.544.2:0.00015)0.967.19:0.00014,(142040:0.0226,(11148:0.00488,(284524:0.00838,554142:0.01591)0.438.2:0.00247)0.967.20:0.01196)0.951.24:0.00151)0.962.18:0.00636)0.948.11:0.00319)0.891.15:0.0017,(563849:0.00924,544443:0.0169)0.219:0.00371)0.897.11:0.00308,(113844:0.00659,67787:0.01045)0.969.26:0.00637)0.750.6:0.00079,(549911:0.01317,511548:0.02055)0.951.25:0.00582)0.998.90:0.01033,11149:0.00804)0.856.8:0.00015)0.960.31:0.00459,((578417:0.00863,61855:0.01935)0.718.7:0.0017,((104597:0.00655,(108919:0.00407,96850:0.00727)0.975.26:0.00665)0.847.13:0.00224,((11139:0.01485,259007:0.00504)0.964.29:0.00708,518681:0.02695)0.839.8:0.00334)0.977.30:0.00564)0.928.15:0.00374)0.941.30:0.00352,(85073:0.01757,11138:0.00313)0.961.27:0.00531)0.806.5:0.00499,(108912:0.00597,94327:0.00635)0.980.24:0.00481)0.750.7:0.00074)0.879.13:0.00156)0.899.15:0.00691,((270785:0.00802,((579539:0.00504,343339:0.01354)0.986.28:0.00662,160940:0.01854)0.912.10:0.00296)0.747.6:0.00176,574165:0.0046)0.749.5:0.00123)1.000.526:0.01382)0.805.10:0.00145)0.986.29:0.0092)0.957.37:0.00404)0.907.14:0.00338,(((565854:0.01634,248911:0.01572)0.943.33:0.00406,((532048:0.02506,572319:0.01816)0.906.13:0.00946,(167013:0.01974,538168:0.03313)0.818.3:0.00278)0.988.47:0.00894)0.959.25:0.00564,159762:0.03373)0.944.8:0.0065)0.839.9:0.0016)0.996.58:0.01066,584327:0.04388)0.796.7:0.00466,((((292483:0.01232,103076:0.01475)0.998.91:0.01859,(295292:0.01593,((94941:0.00091,94413:0.02885)0.915.14:0.00348,11120:0.01029)0.996.59:0.01161)0.996.60:0.01875)0.989.37:0.01641,((520798:0.02173,((105257:0.03318,104287:0.0054)1.000.527:0.02436,(552622:0.00331,536849:0.02806)1.000.528:0.02945)0.926.27:0.00898)0.976.29:0.00958,(102927:0.01065,((((177809:0.0119,371282:0.00737)0.922.13:0.00399,((((565121:0.00708,((98432:0.00761,510296:0.00802)0.846.8:0.0032,(149235:0.0191,554288:0.01114)0.963.21:0.00015)0.842.5:0.00318)0.762.10:0.00094,202868:0.01382)0.925.17:0.00306,349595:0.00391)0.966.25:0.00077,587379:0.00638)0.945.17:0.00092)0.879.14:0.00233,((107659:0.00554,(100070:0.01124,99906:0.00471)0.397:0.00173)0.917.17:0.00159,((104899:0.01057,(107097:0.01381,106942:0.01529)0.997.72:0.00852)0.887.16:0.00398,103824:0.00938)0.652.3:0.00014)0.938.22:0.00238)0.740.8:0.00076,(((11127:0.02259,((100364:0.00844,57474:0.0117)0.654.3:0.00258,243991:0.01969)0.942.20:0.00502)0.819.8:0.00268,(569838:0.00699,258528:0.01356)0.904.11:0.00389)0.841.7:0.00234,262084:0.00614)0.985.25:0.00591)0.883.15:0.0021)0.698.3:0.00168)0.594.3:0.01005)0.893.7:0.01248,(99252:0.00898,79239:0.01039)0.998.92:0.01693)0.947.22:0.00763)0.981.38:0.00994,(106612:0.03485,((98839:0.01677,(107804:0.01514,(101920:0.0227,180340:0.03044)0.812.9:0.00689)0.886.11:0.00743)0.997.73:0.02205,61877:0.01222)0.927.18:0.01045)0.911.13:0.00597)0.956.22:0.00847)1.000.529:0.0222)0.948.12:0.00829)0.980.25:0.01216,(((((103202:0.02155,(104672:0.01383,249367:0.01622)0.567.4:0.00528)1.000.530:0.0252,(295702:0.02259,(((283100:0.00364,44032:0.00508)0.775.5:0.00423,(561090:0.01975,((11191:0.00502,142721:0.01198)0.339.2:0.0003,(137181:0.01654,578429:0.01518)0.775.6:0.00217)0.978.23:0.00777)0.913.17:0.00827)1.000.531:0.02918,(19883:0.01093,((141569:0.00014,111793:0.00081)0.963.22:0.01864,282130:0.00678)0.994.40:0.03128)0.966.26:0.016)0.723.6:0.01493)0.671.4:0.00796)0.776.9:0.00605,(((512603:0.02424,88092:0.02197)1.000.532:0.02345,((159487:0.019,99157:0.01372)0.676:0.0042,((106099:0.02785,104820:0.01641)0.994.41:0.01461,(275020:0.01885,(106449:0.02784,244236:0.01148)0.980.26:0.01306)0.107:0.00223)0.790.6:0.00643)0.807.3:0.00716)1.000.533:0.02924,(214502:0.02719,100828:0.01989)0.993.40:0.01925)0.141:0.00014)0.551.2:0.00733,(((142926:0.01001,59210:0.00764)0.969.27:0.02624,(102626:0.0207,(146284:0.01302,(143275:0.01536,147107:0.00239)0.225.2:0.00349)0.990.35:0.01116)0.998.93:0.01655)0.964.30:0.01063,(((113756:0.01899,((576665:0.02126,56461:0.00742)0.925.18:0.00729,((240175:0.06083,((549039:0.02183,(284123:0.01635,((113002:0.02307,278725:0.01026)0.942.21:0.00821,((545844:0.00852,533613:0.01323)0.998.94:0.01699,(586732:0.00083,(11247:0.00717,11230:0.002)0.908.16:0.00173)0.884.16:0.00211)0.851.7:0.00282)0.992.42:0.00915)0.802.4:0.00361)0.982.41:0.00831,124228:0.01383)0.277.2:0.0068)0.992.43:0.01212,(302158:0.01327,219965:0.01553)0.873.11:0.00571)0.258:0.00014)0.828.8:0.00442)0.957.38:0.00707,(267263:0.00014,542539:0.00758)1.000.534:0.02468)0.966.27:0.00664,(((201590:0.00677,242988:0.01084)0.902.9:0.00425,((((430191:0.0152,201896:0.00733)0.811.5:0.00365,470642:0.01417)0.709.4:0.00779,(484430:0.01961,(257763:0.03068,(253584:0.01935,386273:0.02364)0.983.33:0.01545)0.649.3:0.0098)0.826.6:0.01027)0.367:0.00308,(69947:0.02517,(11227:0.00595,200293:0.00119)0.869.15:0.01982)0.989.38:0.01698)0.968.16:0.00818)0.909.10:0.00504,(57347:0.01034,163231:0.00655)0.939.18:0.00484)0.887.17:0.00509)1.000.535:0.04126)0.989.39:0.01965)0.880.9:0.00554,(541385:0.04093,(455329:0.01936,(((((((11180:0.02338,(((279090:0.0025,546186:0.03617)0.447.3:0.00116,159999:0.00999)0.995.57:0.01512,(150295:0.00971,237719:0.01433)0.926.28:0.00819)0.921.22:0.0074)0.986.30:0.00993,(227146:0.02039,567703:0.00874)0.379.5:0.00157)0.909.11:0.00445,((349369:0.01203,571505:0.01939)0.971.22:0.00989,(51975:0.00831,((11163:0.00537,589078:0.01223)0.985.26:0.01278,20475:0.00226)0.990.36:0.0113)0.746.4:0.00623)0.858.6:0.00352)0.848.11:0.00356,(365874:0.01573,(538501:0.02165,(250174:0.0016,248438:0.01031)0.996.61:0.01409)0.931.14:0.00892)0.740.9:0.01156)0.978.24:0.01366,(((140349:0.00959,144024:0.02351)0.943.34:0.00583,(((((105384:0.01132,11164:0.00764)0.968.17:0.00549,561979:0.01378)0.914.12:0.00357,245826:0.00517)0.573.4:0.01555,140216:0.00549)0.970.18:0.00789,408325:0.04009)0.996.62:0.01935)0.998.95:0.01151,(277566:0.01905,(160519:0.0011,251292:0.02791)0.988.48:0.00798)0.150.2:0.00308)0.452.2:0.00108)0.969.28:0.0082,((113246:0.00811,(((104189:0.01217,(114358:0.01252,534177:0.00295)0.932.22:0.00454)0.952.20:0.01787,(211326:0.04267,(96366:0.01197,266102:0.02576)0.710.4:0.01345)0.996.63:0.01525)0.990.37:0.01697,(48263:0.00782,113754:0.01826)0.805.11:0.00402)0.997.74:0.01897)0.964.31:0.00848,(((284348:0.02725,((567689:0.0009,588023:0.0526)0.908.17:0.00843,((105639:0.00169,71048:0.00989)0.993.41:0.01177,11183:0.02994)0.951.26:0.00771)0.796.8:0.00998)1.000.536:0.0211,(240061:0.0218,239833:0.02118)0.983.34:0.01114)0.215.2:0.00171,70950:0.02547)0.947.23:0.00682)0.997.75:0.0146)0.970.19:0.00897,(60384:0.01853,(278352:0.01261,11162:0.00762)0.983.35:0.01178)0.836.12:0.00742)0.681.6:0.00448)0.998.96:0.02383)1.000.537:0.03553)0.977.31:0.01202)0.938.23:0.01084)0.166.2:0.01635)0.944.9:0.01286)0.562:0.02473)1.000.538:0.06533,(51780:0.0541,566496:0.06022)0.998.97:0.03443)1.000.539:0.04366,(((((569299:0.01144,550922:0.01101)0.958.23:0.01288,570826:0.00966)0.996.64:0.02192,(((((558597:0.00988,18643:0.01397)0.606.4:0.002,536608:0.0076)0.860.16:0.00014,227523:0.01409)0.628.3:0.00477,572046:0.00852)0.909.12:0.00525,(11286:0.03023,(544778:0.00761,594030:0.0152)1.000.540:0.0176)0.928.16:0.00883)0.996.65:0.02525)1.000.541:0.06362,((((218189:0.01618,((8039:0.03662,131064:0.02195)1.000.542:0.03189,293272:0.01797)0.309.2:0.0071)0.850.12:0.00487,93549:0.01643)1.000.543:0.0397,((((238646:0.02035,591677:0.0168)1.000.544:0.05855,((160103:0.02994,(34315:0.02365,(40122:0.01371,(200017:0.01799,560023:0.01125)0.000.20:0.0012)0.174.2:0.01113)0.966.28:0.01177)0.998.98:0.02827,(((63994:0.01403,349530:0.00873)0.862.12:0.00249,((63963:0.01067,63964:0.01551)0.964.32:0.01076,55005:0.03417)0.939.19:0.00684)1.000.545:0.108,((((((((((150145:0.01804,110392:0.02404)0.997.76:0.0151,((558915:0.0032,546434:0.00661)0.791.7:0.02159,((97708:0.00163,96705:0.00015)1.000.546:0.01877,(253342:0.02177,(151384:0.02021,141532:0.01471)0.911.14:0.00689)0.151:0.00208)0.689.3:0.0033)0.843.5:0.00489)0.964.33:0.00756,(113843:0.01788,(563047:0.01011,469129:0.0051)0.900.11:0.00757)0.979.33:0.00817)0.954.17:0.00703,((236325:0.02903,(((310487:0.02448,112901:0.00617)0.960.32:0.00467,113342:0.02229)0.880.10:0.00271,64874:0.00464)0.958.24:0.00402)0.987.44:0.00984,499198:0.0182)0.944.10:0.00755)0.711.3:0.00478,(253392:0.02142,((((225714:0.02032,570888:0.01426)0.997.77:0.01669,(170130:0.01946,591556:0.03905)0.896.14:0.00901)0.987.45:0.01047,((((((((75333:0.01559,114568:0.01242)0.906.14:0.01065,(132318:0.00587,7229:0.00579)0.873.12:0.0031)0.872.26:0.00157,((276631:0.01768,((592344:0.00996,(((24899:0.02266,(((49998:0.02072,17921:0.00471)1.000.547:0.00016,(211450:0.01859,129858:0.00373)0.872.27:0.01365)0.897.12:0.00044,((119349:0.00157,(219926:0.00028,125947:0.0034)0.938.24:0.00233)0.331.4:0.00274,169778:0.00028)0.984.34:0.00469)0.989.40:0.00015)0.922.14:0.00155,316759:0.0106)0.930.17:0.00016,(40058:0.01321,7222:0.00341)0.277.3:0.00187)0.425:0.00103)0.880.11:0.00323,((156580:0.01031,257726:0.00093)0.861.13:0.00552,(107719:0.06493,((471398:0.00077,((108849:0.00809,(((362426:0.02048,547752:0.01497)0.823.5:0.00233,107758:0.01349)0.962.19:0.0055,(((354052:0.00503,(47552:0.01018,198421:0.01283)0.875.14:0.00263)0.381.3:0.00388,((558833:0.03584,173767:0.00042)0.896.15:0.01296,142599:0.00423)0.860.17:0.00345)0.942.22:0.00731,(99125:0.01434,77828:0.01221)0.073.3:0.00285)0.867.14:0.00192)0.771.10:0.00313)0.737.3:0.00429,(258484:0.02658,7214:0.00602)0.915.15:0.00383)0.992.44:0.00901)0.355.3:0.00089,211076:0.01213)1.000.548:0.00076)0.957.39:0.00319)0.894.17:0.00227)0.742.7:0.00073)0.960.33:0.00386,(158930:0.01048,140518:0.01061)0.900.12:0.00768)0.954.18:0.00403)0.778.9:0.00078,144381:0.00952)0.913.18:0.00301,((236156:0.0237,253304:0.0381)0.000.21:0.00043,(339877:0.00525,94022:0.02649)0.965.21:0.00014)0.835.6:0.00965)0.955.21:0.00523,(367189:0.00734,172940:0.01261)0.922.15:0.00496)0.756.7:0.00139,(139065:0.01009,(512517:0.01164,(28964:0.01818,313830:0.00914)0.967.21:0.01179)1.000.549:0.02477)0.859.9:0.00298)0.962.20:0.00498,(287984:0.00275,(134384:0.00133,255307:0.03442)1.000.550:0.01029)0.980.27:0.00854)0.987.46:0.00872)0.852.7:0.00197,(583804:0.0307,((319986:0.00938,(331266:0.00092,259583:0.00623)0.871.21:0.01569)0.994.42:0.01651,((324381:0.0321,270171:0.00238)0.585.2:0.00266,(558389:0.01436,567560:0.00867)0.601:0.0084)0.778.10:0.00528)0.904.12:0.00648)0.994.43:0.01228)0.866.8:0.00576)0.892.12:0.0042)0.853.7:0.00557,(((((((((250119:0.02133,298947:0.0074)1.000.551:0.01473,(237924:0.00682,562108:0.01006)0.890.16:0.00553)0.842.6:0.00269,(156814:0.01916,(509038:0.01184,22530:0.01957)0.587.2:0.00202)0.991.36:0.00958)0.824.5:0.00225,60579:0.01056)0.720.4:0.00341,(253486:0.01224,(511916:0.02899,((282571:0.01325,387343:0.01441)0.948.13:0.00015,(582302:0.00421,581579:0.00314)0.866.9:0.00387)0.490:0.00222)0.845.11:0.00244)0.982.42:0.00892)0.727.3:0.00264,545945:0.00399)0.988.49:0.01065,(((204917:0.02393,(398839:0.01322,(361480:0.00235,165322:0.02611)0.844.8:0.01365)0.515.2:0.0101)1.000.552:0.04836,(206937:0.0435,(((547221:0.01094,(158112:0.00842,256434:0.00634)0.915.16:0.00514)0.999.139:0.01733,(216504:0.00912,(562244:0.00413,41229:0.01388)1.000.553:0.01586)0.991.37:0.01337)0.901.10:0.00615,((((272661:0.00832,252773:0.01345)0.629.3:0.00379,((173726:0.0008,111135:0.00185)0.992.45:0.00879,(6821:0.00554,576373:0.00824)0.900.13:0.00589)1.000.554:0.01366)0.972.36:0.00615,(((197476:0.02779,104793:0.00655)0.940.17:0.00397,(589923:0.00969,455791:0.01034)0.959.26:0.00434)0.925.19:0.00338,182833:0.00089)0.913.19:0.00728)0.801.2:0.0027,207344:0.01777)0.978.25:0.01015)0.941.31:0.00841)0.705.2:0.00848)0.976.30:0.01376,164280:0.01123)0.541:0.00641)0.613.2:0.00645,43158:0.01591)0.996.66:0.01513,(330902:0.0291,(220795:0.00605,226599:0.02254)0.842.7:0.00735)0.479.4:0.01009)0.642.3:0.00447)0.548.2:0.00308,(95883:0.02207,(309818:0.00661,((158339:0.0133,((65391:0.00557,(594220:0.01072,((593808:0.02031,181631:0.00902)0.869.16:0.00171,108905:0.00687)0.805.12:0.00247)0.990.38:0.01015)0.936.21:0.00462,((((31379:0.00277,(171796:0.01141,346019:0.02045)0.751.2:0.00234)0.926.29:0.00233,(((((382348:0.02091,((68624:0.01225,68623:0.01296)0.953.16:0.00481,329764:0.02297)0.305:0.00056)0.320.2:0.00156,89748:0.02499)0.507.2:0.00014,(56470:0.02428,(((249800:0.04336,582533:0.01277)0.896.16:0.00527,(228195:0.00976,(249846:0.02713,(206862:0.04819,228065:0.01844)0.139.2:0.00286)0.757.6:0.00142)0.466.5:0.00024)0.292.5:0.00029,(332634:0.01673,241343:0.00931)0.464.2:0.00071)0.807.4:0.0017)0.853.8:0.00077)0.874.16:0.00042,311931:0.01441)0.461.2:0.00014,226210:0.02403)0.503.3:0.00335)0.870.16:0.00156,565246:0.01529)0.851.8:0.00207,((((554733:0.01789,(584565:0.01067,511551:0.00655)0.726.3:0.00326)0.982.43:0.007,(((((540496:0.0138,528864:0.00357)0.890.17:0.00496,(55861:0.02747,526614:0.00476)0.387.3:0.00493)0.787.6:0.00477,(591899:0.00654,(556028:0.01229,(561894:0.01142,511905:0.01344)0.879.15:0.00513)1.000.555:0.02158)0.742.8:0.00541)0.853.9:0.00232,562181:0.01588)0.806.6:0.00015,(((6825:0.011,(541465:0.01927,161317:0.01133)0.549:0.00389)0.920.12:0.00408,((6892:0.00555,(110790:0.00584,303810:0.00907)0.968.18:0.00475)0.979.34:0.00654,((((35834:0.0076,70363:0.00875)1.000.556:0.02831,22985:0.02306)0.940.18:0.00659,512238:0.01664)0.790.7:0.00322,(77605:0.0049,102566:0.00723)0.884.17:0.00134)0.879.16:0.00254)0.644.2:0.00227)0.988.50:0.00845,((((100294:0.00015,279278:0.02265)0.020:0.02251,(26755:0.01576,((547998:0.01346,204226:0.01395)0.899.16:0.00161,(112764:0.0074,6857:0.00861)0.981.39:0.00477)0.873.13:0.00157)0.744.8:0.00091)0.740.10:0.00069,((((((154347:0.00736,(575630:0.00732,(149152:0.00317,178202:0.00199)0.716:0.00679)0.850.13:0.00791)0.710.5:0.00982,558108:0.03127)0.957.40:0.00785,6864:0.01845)0.939.20:0.00847,153898:0.01572)0.092.3:0.00981,6849:0.01772)0.141.2:0.00101,(110287:0.00327,279351:0.01032)0.919.14:0.00248)0.960.34:0.00385)0.978.26:0.0064,((128201:0.04734,558297:0.00556)0.220.2:0.00078,266661:0.03391)0.111:0.00014)0.739.11:0.00074)0.895.18:0.00158)0.983.36:0.00634)0.954.19:0.00467,(49531:0.01511,339734:0.01376)0.884.18:0.0023)0.761.7:0.00153,(537405:0.00735,568289:0.01042)0.891.16:0.00257)0.755.10:0.00344)0.854.7:0.00504)0.825.9:0.00196)0.866.10:0.00254,6876:0.02322)0.952.21:0.00623)0.822.5:0.0073)1.000.557:0.02907)0.305.2:0.00161,((((((38016:0.01261,574522:0.01898)0.994.44:0.02611,296722:0.09572)0.983.37:0.02074,(((238343:0.00405,561226:0.00182)0.862.13:0.02631,583154:0.00247)1.000.558:0.03193,(207982:0.02261,(228077:0.0362,536916:0.00997)0.997.78:0.01581)0.322:0.0053)0.975.27:0.0107)0.662.4:0.00115,356250:0.12958)0.402:0.0043,(((((551154:0.03677,(337734:0.02794,((246297:0.01394,(94716:0.01542,((151309:0.00879,237195:0.00895)0.839.10:0.01782,311155:0.01421)0.648.3:0.00413)0.603.3:0.00394)0.506.2:0.00422,155812:0.00759)0.756.8:0.00452)0.939.21:0.01371)1.000.559:0.0478,(((((((250114:0.00014,389746:0.00377)0.987.47:0.01043,(589010:0.0028,144697:0.01605)0.976.31:0.00827)0.329.2:0.00753,6370:0.00197)0.995.58:0.0202,(550256:0.02488,519635:0.01258)0.992.46:0.01794)0.419.3:0.00694,(64884:0.03211,146009:0.00544)0.990.39:0.01449)0.864.9:0.0078,(((538707:0.0138,207155:0.00184)0.994.45:0.00937,(140478:0.00993,(172109:0.00894,(589312:0.003,(6372:0.00875,280825:0.01533)0.972.37:0.00791)0.891.17:0.0051)0.726.4:0.00274)0.894.18:0.0043)0.737.4:0.0027,(164660:0.01249,(((171151:0.00239,262569:0.00107)0.878.13:0.00379,148906:0.0184)0.994.46:0.01563,195318:0.00833)0.991.38:0.01063)0.968.19:0.00726)0.990.40:0.01277)0.964.34:0.01173,((363957:0.0208,356874:0.02948)0.999.140:0.02302,(((((540352:0.04017,((((129475:0.01461,450185:0.00278)0.313.4:0.00805,(6378:0.00414,((134552:0.00107,42684:0.00185)0.940.19:0.00659,(75888:0.01352,(((238347:0.00834,(470243:0.01281,6387:0.00465)0.405.3:0.00983)0.852.8:0.00188,47990:0.00465)0.759.6:0.00154,(53569:0.00515,((63985:0.00647,((325612:0.00119,327035:0.02863)1.000.560:0.0047,(222032:0.01584,((470724:0.02095,(143795:0.0009,218785:0.03173)0.774.4:0.00477)0.865.14:0.01231,(517381:0.01146,132799:0.00698)0.885.11:0.00199)0.977.32:0.00311)0.435.4:0.00014)1.000.561:0.00549)0.706.3:0.00658,(360229:0.00651,6402:0.00208)0.909.13:0.00307)0.919.15:0.00768)0.971.23:0.00594)0.261:0.00303)0.469:0.00518)0.987.48:0.00716)0.329.3:0.00068)0.887.18:0.00232,104888:0.01216)0.734.6:0.00069,(((105593:0.00783,(56599:0.01228,(21796:0.00478,72999:0.00694)0.941.32:0.00632)0.983.38:0.00827)0.994.47:0.01042,((103564:0.00657,((536311:0.03704,92601:0.00551)0.979.35:0.0048,(470523:0.0051,125235:0.04149)0.879.17:0.00286)0.894.19:0.00236)0.765.9:0.00112,((126626:0.0201,(92167:0.00398,394191:0.00853)0.993.42:0.0099)0.910.13:0.0056,23991:0.0146)0.855.12:0.00377)0.973.33:0.00696)0.263.4:0.00141,104026:0.01173)0.943.35:0.00342)0.919.16:0.00294)0.568.4:0.00233,589124:0.00666)0.416.3:0.00265,55486:0.01247)0.963.23:0.00691,(104316:0.01273,(144630:0.0153,215563:0.01634)0.996.67:0.01078)0.918.9:0.00514)0.833.10:0.00341,(264994:0.02645,(6376:0.00588,544854:0.00322)0.702.4:0.00695)0.977.33:0.01487)0.732.6:0.00967)1.000.562:0.0231)0.934.19:0.00657)0.953.17:0.01347,((51448:0.01342,45071:0.01732)1.000.563:0.02909,(206561:0.01208,144094:0.01326)1.000.564:0.02556)0.906.15:0.01129)0.223.3:0.00913,((144420:0.01115,558688:0.01976)0.435.5:0.00832,(534652:0.04059,((((359570:0.00568,302050:0.01656)1.000.565:0.01864,(6368:0.00612,556357:0.02203)1.000.566:0.02327)0.379.6:0.00507,((148882:0.00416,290842:0.01709)0.980.28:0.0133,(558848:0.00542,(574480:0.0206,151797:0.01005)0.738.11:0.00151)0.945.18:0.01116)0.793.4:0.00902)0.510:0.00628,((342261:0.02,108299:0.01659)0.915.17:0.00423,((168478:0.01105,244223:0.01215)0.959.27:0.01134,560198:0.02824)0.915.18:0.00671)0.871.22:0.00854)0.581.2:0.01321)1.000.567:0.02444)0.980.29:0.01415)0.797.8:0.00975,(((((204179:0.01703,(205044:0.00868,240452:0.01593)0.944.11:0.01005)0.794.7:0.00562,(200807:0.01287,((41200:0.01079,(593511:0.01047,224900:0.02127)0.034:0.01483)0.949.22:0.00676,(194191:0.01278,100908:0.0168)0.891.18:0.00484)0.929.20:0.00606)0.994.48:0.01244)0.981.40:0.00958,(((234053:0.01297,578293:0.01387)0.878.14:0.00434,(576056:0.01944,164635:0.00357)0.503.4:0.00992)0.188.2:0.00169,((220551:0.01572,(114209:0.00801,((((154550:0.00781,509397:0.01708)0.439.5:0.00082,225296:0.01299)0.818.4:0.00411,((250010:0.00203,583255:0.00917)0.789.5:0.00115,(277681:0.00395,551424:0.01932)0.738.12:0.00315)0.896.17:0.00028)0.928.17:0.00654,(220011:0.04875,((218017:0.00467,215229:0.01254)0.949.23:0.00454,(222729:0.00844,(216418:0.0052,160762:0.01674)0.810.6:0.00336)0.861.14:0.00276)0.525.2:0.00242)0.788.5:0.00155)0.987.49:0.0091)0.891.19:0.0038)0.871.23:0.00246,285572:0.0098)0.949.24:0.00547)0.995.59:0.0167)0.996.68:0.01684,(225073:0.01652,((((221903:0.01386,((226979:0.01976,222745:0.00243)0.893.8:0.00171,144298:0.01735)0.537.2:0.00014)0.378.3:0.00491,307844:0.00468)1.000.568:0.03208,(267229:0.03508,((210313:0.00828,(((247235:0.00534,252485:0.00664)0.992.47:0.01158,(541147:0.01021,(214047:0.0212,((101437:0.00098,208823:0.00479)0.918.10:0.01233,(355149:0.0093,311995:0.01123)0.963.24:0.00599)0.876.14:0.00341)0.971.24:0.0071)0.931.15:0.00463)0.968.20:0.00521,567307:0.016)0.741.7:0.0007)0.940.20:0.00553,160908:0.00612)0.849.6:0.00444)0.963.25:0.00996)0.965.22:0.00876,((((((593905:0.01348,226091:0.01336)0.631.4:0.00252,((281246:0.02043,156479:0.00354)0.748.8:0.00106,545121:0.02461)0.982.44:0.0067)0.996.69:0.01208,(514085:0.02472,(565973:0.00524,((168042:0.00829,136641:0.00839)0.783.4:0.00101,(205301:0.00862,328181:0.00722)0.991.39:0.01853)0.862.14:0.00295)0.933.19:0.00676)0.974.25:0.00774)0.659.3:0.00208,(((((((165384:0.00814,(238426:0.00867,(235519:0.01747,(212958:0.0064,579008:0.02508)0.946.24:0.00811)0.871.24:0.00293)0.905.14:0.00291)0.886.12:0.00412,(584504:0.00831,(589570:0.0082,513965:0.01188)0.901.11:0.00392)0.880.12:0.00365)0.945.19:0.00459,560895:0.0128)0.927.19:0.00335,(224590:0.00486,(113247:0.00662,151439:0.00948)0.817.9:0.00346)0.999.141:0.0113)0.860.18:0.00181,223219:0.01452)0.883.16:0.00241,(555896:0.01645,((239352:0.00972,227269:0.03895)0.085.2:0.00963,(((36067:0.0047,(208009:0.00554,104916:0.0041)0.940.21:0.00471)0.952.22:0.00602,(205576:0.00786,(359069:0.00977,(536863:0.00314,(546694:0.01823,113273:0.02733)0.871.25:0.00789)0.756.9:0.00439)0.993.43:0.00811)0.886.13:0.00291)0.964.35:0.00585,((561129:0.01387,387042:0.01098)0.995.60:0.01357,256223:0.01107)0.655:0.00451)0.834.13:0.00278)0.909.14:0.00434)0.989.41:0.00694)0.789.6:0.00015,(((114229:0.03631,352281:0.04186)0.971.25:0.01492,224647:0.00686)0.254.5:0.00447,((206488:0.01683,236450:0.00844)0.727.4:0.00103,221665:0.00731)0.792.8:0.00276)0.981.41:0.00884)0.995.61:0.01042)0.912.11:0.00383,(236367:0.01046,(245246:0.01054,253948:0.02006)0.825.10:0.00259)0.932.23:0.00576)0.938.25:0.01147,(142009:0.00904,555772:0.00997)0.994.49:0.01657)0.643.2:0.00984)0.977.34:0.01196)0.996.70:0.01651)0.992.48:0.0113,((((110294:0.04494,(158161:0.00595,(157274:0.00748,175927:0.00418)0.990.41:0.01032)0.998.99:0.0215)0.957.41:0.01217,(((((177340:0.00312,264040:0.00105)0.836.13:0.00175,580735:0.02119)0.991.40:0.01558,(((((105022:0.01992,125742:0.00727)0.921.23:0.00554,274432:0.0062)0.958.25:0.0056,222196:0.00103)0.603.4:0.003,((((101445:0.01264,312772:0.00929)0.927.20:0.005,((105146:0.01524,103643:0.01281)0.996.71:0.01104,(556229:0.00248,(((520725:0.00796,((104555:0.01133,251118:0.01899)0.927.21:0.00014,343674:0.00105)0.914.13:0.00229)0.791.8:0.00331,((537948:0.012,(105727:0.00941,((107115:0.01667,250544:0.0035)0.956.23:0.00187,202563:0.01356)0.868.16:0.00392)0.861.15:0.00511)0.791.9:0.00353,105132:0.00471)0.851.9:0.0017)0.878.15:0.00158,306546:0.01441)0.545:0.00015)0.930.18:0.00574)0.986.31:0.00769)0.450.2:0.00375,(529407:0.01481,(554745:0.02409,156676:0.00107)0.621.2:0.01037)0.963.26:0.00722)0.740.11:0.00108,(93630:0.00434,135981:0.00219)0.999.142:0.01706)0.332.2:0.00348)0.408.3:0.00327,(((108436:0.01579,(219027:0.02589,(101169:0.01253,(289508:0.01104,103273:0.00873)0.845.12:0.0048)0.624.2:0.00354)0.830.6:0.00627)0.943.36:0.00646,(242585:0.01567,(((567795:0.02588,(101544:0.03397,354554:0.00643)0.945.20:0.00697)1.000.569:0.00249,582100:0.01748)0.878.16:0.00015,318930:0.03293)0.972.38:0.02033)0.977.35:0.00891)0.987.50:0.01085,(306270:0.03421,329876:0.01982)0.955.22:0.00527)0.947.24:0.00814)0.947.25:0.00766)0.225.3:0.00087,((96941:0.00226,264560:0.00989)0.985.27:0.00689,(137250:0.0166,240233:0.00167)0.985.28:0.00909)0.983.39:0.00985)1.000.570:0.01973,(((74988:0.03235,((168285:0.00014,535635:0.02906)0.843.6:0.01659,(553029:0.02702,257017:0.00807)0.999.143:0.01171)0.052:0.0049)0.801.3:0.00468,(551992:0.01942,(243280:0.01104,(184153:0.01112,150107:0.01609)0.497.2:0.00435)0.946.25:0.01164)0.869.17:0.00472)0.698.4:0.00071,((64665:0.01429,265614:0.00874)0.947.26:0.00883,(138498:0.00938,138290:0.01916)1.000.571:0.02416)0.955.23:0.00702)0.722.5:0.00072)0.976.32:0.00775)0.850.14:0.00235,(((511972:0.00015,296353:0.02487)0.976.33:0.04395,(((((49524:0.00727,(((69916:0.00777,6540:0.00504)0.955.24:0.00394,(((((74365:0.00247,239941:0.0062)0.979.36:0.00627,19636:0.00434)0.978.27:0.00786,(559849:0.0032,311145:0.02487)0.794.8:0.00323)0.742.9:0.00691,511499:0.00838)0.895.19:0.00215,(6539:0.00398,49602:0.00469)0.997.79:0.00968)0.914.14:0.00242)0.828.9:0.00273,((17356:0.00736,112204:0.01497)0.199.2:0.00343,19771:0.01549)0.800.5:0.00494)0.631.5:0.00413)0.890.18:0.00391,69852:0.01087)0.000.22:0.0013,((261884:0.00485,(46257:0.02826,73796:0.01233)0.660.2:0.00466)1.000.572:0.02256,(29792:0.02041,((251028:0.00492,(27293:0.02083,40156:0.02732)0.906.16:0.01005)0.970.20:0.00622,71515:0.01956)0.908.18:0.00684)0.954.20:0.00742)0.947.27:0.00674)0.756.10:0.00215,562879:0.00715)0.981.42:0.00856,(175017:0.01526,((6518:0.01989,40900:0.02298)0.615:0.0042,(145637:0.01028,105991:0.01462)0.932.24:0.01469)0.799.5:0.00201)0.422.3:0.00233)0.998.100:0.01448)0.809.8:0.01132,(((512437:0.00582,(((137480:0.00308,540799:0.01736)0.946.26:0.00316,(159843:0.00372,27004:0.01561)0.765.10:0.02326)0.949.25:0.00316,((((592888:0.00238,143954:0.01187)0.880.13:0.00156,(576649:0.02529,564986:0.00558)0.816.8:0.00489)0.746.5:0.00081,(((327025:0.01053,(361662:0.00777,534083:0.00456)0.987.51:0.00853)0.950.32:0.00525,((322993:0.00971,(198458:0.00109,((540849:0.00014,528137:0.00636)1.000.573:0.01417,356589:0.02227)0.944.12:0.00154)0.942.23:0.00878)0.795.9:0.00248,310147:0.01148)0.750.8:0.00088)0.939.22:0.00632,(307219:0.00505,108663:0.01281)0.391.4:0.00146)0.990.42:0.00556)0.781.5:0.00156,143501:0.01148)0.880.14:0.00162)0.955.25:0.00452)0.809.9:0.00347,(((((535045:0.02647,((336794:0.01143,(228790:0.01115,234810:0.02413)0.862.15:0.01621)0.696.5:0.00184,534649:0.0114)0.955.26:0.00515)0.858.7:0.00367,141618:0.00103)0.962.21:0.00633,336177:0.00876)0.943.37:0.00618,(((553899:0.02565,551555:0.04208)1.000.574:0.02253,(((16839:0.01613,178053:0.03071)0.666.4:0.00788,98919:0.03086)0.734.7:0.00151,(352419:0.01513,325021:0.01274)0.847.14:0.00159)0.949.26:0.00325)0.835.7:0.00151,(196465:0.00628,(245670:0.00381,234236:0.01471)0.844.9:0.00015)0.824.6:0.00313)0.987.52:0.00647)0.926.30:0.00525,(341843:0.01466,(325090:0.00152,(342918:0.00411,(255370:0.01598,571272:0.01253)0.979.37:0.00683)0.997.80:0.01144)0.918.11:0.0049)0.989.42:0.013)0.218.3:0.00544)0.992.49:0.01165,104211:0.02524)0.986.32:0.01186)0.839.11:0.00391)0.993.44:0.01112,((553957:0.02961,(220761:0.01786,207373:0.0158)0.987.53:0.01572)0.934.20:0.00944,((((((147236:0.01202,((551776:0.00613,171974:0.01877)0.881.11:0.00324,((245525:0.01908,533665:0.01771)0.280.2:0.00135,(574483:0.01223,222171:0.01706)0.910.14:0.00423)0.768.7:0.0185)0.894.20:0.00287)0.998.101:0.01378,(((((265840:0.01274,(106737:0.00624,530591:0.01457)0.944.13:0.00438)0.921.24:0.00391,155215:0.01876)0.925.20:0.00521,239439:0.02499)0.682:0.00348,111577:0.01198)0.950.33:0.00601,((((((555469:0.01931,106337:0.00817)0.747.7:0.00071,(((((((511775:0.02065,253963:0.00823)0.512.2:0.00165,107184:0.01853)0.927.22:0.00366,(590112:0.001,548316:0.01853)0.991.41:0.01279)0.838.10:0.00193,536858:0.00674)0.321.3:0.00159,200507:0.01321)0.758.8:0.00159,553799:0.01955)0.771.11:0.00292,(108155:0.01951,108329:0.01035)1.000.575:0.01263)0.975.28:0.00493)0.997.81:0.00832,(365866:0.00668,((236435:0.01044,369022:0.01803)0.974.26:0.00816,(((43046:0.00867,((235493:0.00892,551095:0.0054)0.984.35:0.00699,107192:0.01001)1.000.576:0.01297)0.976.34:0.00606,(351540:0.00506,224877:0.0288)0.725.5:0.00056)0.918.12:0.00693,551631:0.01734)0.974.27:0.00884)0.987.54:0.01579)0.962.22:0.00536)0.863.14:0.00227,((89219:0.00856,(106259:0.00163,163080:0.00182)0.594.4:0.00014)0.948.14:0.01887,(546813:0.00759,(150027:0.02284,6749:0.00787)0.955.27:0.00767)0.950.34:0.0072)0.994.50:0.01079)0.536:0.0033,(163176:0.01264,((224674:0.01486,567753:0.01083)0.990.43:0.01173,169725:0.00665)0.746.6:0.0079)0.954.21:0.00643)0.755.11:0.00164,190625:0.01121)0.931.16:0.0041)0.919.17:0.00542)0.864.10:0.00666,(568555:0.03639,333442:0.03203)0.980.30:0.01024)0.894.21:0.00422,(((277656:0.01469,202891:0.00964)0.924.15:0.00516,((77638:0.01986,(143225:0.02761,135858:0.02366)0.985.29:0.01522)0.934.21:0.00946,((201491:0.00482,147910:0.01104)0.994.51:0.01459,135855:0.04278)0.853.10:0.00502)0.983.40:0.01064)0.993.45:0.00799,(((((356823:0.00984,21532:0.01136)0.683.5:0.00359,(170831:0.00812,168182:0.01922)0.344:0.00196)0.947.28:0.00739,240971:0.0334)0.099:0.00984,(((((93931:0.01122,115286:0.01574)0.937.22:0.00574,539313:0.00645)0.627.2:0.00347,((74235:0.00934,(254556:0.01989,(149145:0.01602,(((143561:0.02075,255424:0.01107)0.741.8:0.00901,6792:0.01247)0.935.11:0.00701,167461:0.0055)0.639.2:0.00341)0.908.19:0.00486)0.976.35:0.00537)0.924.16:0.00014,((35148:0.01218,129289:0.00679)0.868.17:0.00395,((357120:0.01502,(((564159:0.0214,((278115:0.00566,(6783:0.00927,267500:0.01836)0.771.12:0.00106)0.929.21:0.00401,(233927:0.00086,179013:0.00107)0.812.10:0.00698)0.977.36:0.00614)0.326.4:0.00184,((521021:0.00198,(111043:0.01888,6748:0.00591)0.935.12:0.01619)0.758.9:0.00692,234901:0.02659)0.932.25:0.00659)0.726.5:0.00367,6750:0.01197)0.973.34:0.00957)0.986.33:0.00872,249511:0.00936)0.876.15:0.00236)0.917.18:0.00642)0.939.23:0.00499)0.972.39:0.0088,49117:0.0117)0.628.4:0.00577,560768:0.02561)0.962.23:0.00894)0.912.12:0.00742,((((147307:0.01244,(144724:0.02318,245301:0.00014)0.831.5:0.00214)0.940.22:0.00545,(((529251:0.01759,((77627:0.00816,367951:0.01863)0.950.35:0.0063,165690:0.00015)0.933.20:0.00471)0.681.7:0.00962,(136814:0.0207,80896:0.01437)0.714.3:0.00786)0.680:0.00435,((142103:0.0203,(107623:0.01324,(120677:0.01465,(((236679:0.00669,576785:0.02851)0.971.26:0.00823,(6692:0.00263,(140458:0.01136,(335623:0.01366,(436070:0.01802,(((6733:0.00809,587854:0.01111)0.989.43:0.00652,197955:0.00809)0.963.27:0.00482,149051:0.00821)0.936.22:0.00015)0.914.15:0.0027)0.831.6:0.00271)0.996.72:0.01688)0.997.82:0.01913)0.980.31:0.00938,((((306298:0.03605,485128:0.00571)0.948.15:0.004,((106422:0.02126,(6697:0.00432,((69212:0.00853,((26694:0.00162,99384:0.01185)0.988.51:0.0049,131456:0.01934)0.937.23:0.00503)0.960.35:0.00541,(((544084:0.01442,576674:0.01749)0.891.20:0.00475,(552654:0.02615,442735:0.00821)0.744.9:0.00083)0.404.4:0.00323,168595:0.01672)0.846.9:0.00289)0.973.35:0.00527)0.903.18:0.00322)0.946.27:0.00994,96448:0.02165)0.948.16:0.00391)0.954.22:0.00235,((161279:0.00626,(150832:0.00871,(334429:0.01515,427337:0.01009)0.900.14:0.00321)0.921.25:0.00274)0.739.12:0.00796,(((144181:0.0151,(551430:0.00449,153255:0.01138)0.491:0.00217)0.807.5:0.0031,140097:0.0072)0.792.9:0.00015,570372:0.00779)1.000.577:0.00015)0.853.11:0.00229)0.730.5:0.00085,((584742:0.01667,513084:0.00237)0.976.36:0.00551,((539491:0.00828,511268:0.00764)0.988.52:0.00715,525767:0.00877)0.938.26:0.00329)0.870.17:0.00318)0.901.12:0.00315)0.981.43:0.00743)0.862.16:0.00308)0.944.14:0.00598)0.934.22:0.00835,((((242252:0.01729,88575:0.00883)0.869.18:0.00283,((36121:0.00643,6725:0.00094)1.000.578:0.01661,(42580:0.00507,(557434:0.01281,((134476:0.03009,((54073:0.00014,592785:0.00421)0.999.144:0.02001,78240:0.00528)0.970.21:0.01183)0.911.15:0.008,545495:0.00646)0.703.8:0.00461)0.880.15:0.00594)0.913.20:0.0042)0.875.15:0.00337)0.951.27:0.00753,(6712:0.00625,(6711:0.00874,6713:0.00613)0.827.6:0.00496)0.677.4:0.00306)0.892.13:0.00402,26639:0.01823)0.862.17:0.00359)0.937.24:0.0052)0.954.23:0.00988)0.942.24:0.00582,(151035:0.00689,278568:0.0212)0.999.145:0.02106)0.983.41:0.00992,((208069:0.02252,(186145:0.01254,((463253:0.01666,149154:0.01159)0.178.2:0.01666,239340:0.0008)1.000.579:0.03313)1.000.580:0.01827)0.415.2:0.00214,(6673:0.00043,139914:0.00014)0.694.3:0.00775)0.947.29:0.00677)0.661.3:0.00495)0.381.4:0.00261)0.880.16:0.00262)0.951.28:0.00646,(((((278532:0.00441,(90189:0.00431,143394:0.00352)0.759.7:0.00466)1.000.581:0.02379,((261745:0.01709,(252984:0.02607,(106348:0.01695,(102864:0.00885,17553:0.0055)0.736.6:0.00132)0.925.21:0.00572)0.930.19:0.00848)0.918.13:0.00742,(276290:0.01161,(142298:0.01314,(404251:0.00904,(187854:0.00399,243583:0.01215)0.616:0.00277)0.952.23:0.00847)0.883.17:0.00599)0.923.12:0.00858)0.992.50:0.01409)0.925.22:0.01093,(510574:0.02446,((171073:0.01494,((536176:0.02176,540962:0.01678)0.969.29:0.00894,548352:0.00405)0.153:0.00098)0.838.11:0.0028,(212652:0.03973,78574:0.02375)0.992.51:0.01333)0.992.52:0.01682)0.978.28:0.01096)0.692.3:0.00354,(((((((567333:0.01481,((205571:0.0182,(583489:0.01655,(168070:0.00854,206149:0.01819)0.886.14:0.00189)0.963.28:0.00403)0.912.13:0.00385,(204943:0.03071,222917:0.00971)0.998.102:0.01552)0.993.46:0.01295)0.911.16:0.00679,252902:0.00973)0.999.146:0.01203,(257753:0.01557,((356045:0.00903,((240448:0.01341,331781:0.01273)0.842.8:0.00153,((218239:0.02355,537138:0.01396)0.997.83:0.00014,583039:0.00399)0.884.19:0.00161)0.837.12:0.00155)0.851.10:0.00203,(188905:0.01994,((240278:0.01455,150172:0.00496)0.910.15:0.00587,(155433:0.00447,(156101:0.02543,(554516:0.00315,(230327:0.00724,310913:0.0157)0.989.44:0.0096)0.968.21:0.00833)0.863.15:0.00224)0.987.55:0.00771)0.205.2:0.00404)0.970.22:0.00698)0.930.20:0.00533)0.616.2:0.00011)0.291.4:0.00404,(576127:0.01881,(508906:0.01661,114213:0.01453)0.754.8:0.00486)0.992.53:0.0103)0.998.103:0.01774,217902:0.02667)0.982.45:0.01149,(342642:0.01272,(254959:0.01193,(306861:0.04561,(355039:0.00043,117907:0.00077)0.665.2:0.01644)0.804.5:0.00692)0.997.84:0.01437)0.964.36:0.0085)0.678:0.00477,((552579:0.00212,256389:0.04768)0.756.11:0.0282,(579423:0.01633,(214210:0.02076,150728:0.00529)0.931.17:0.0092)0.992.54:0.01875)0.939.24:0.00954)0.916.17:0.00634)0.954.24:0.00862,((573240:0.00351,6794:0.00813)1.000.582:0.02433,((325942:0.01973,561445:0.02849)0.965.23:0.01017,(((71612:0.00399,406799:0.01566)1.000.583:0.02001,((325024:0.01787,(512321:0.01697,(585600:0.01407,(253630:0.01274,((102883:0.02191,567553:0.01679)0.888.6:0.00411,((541068:0.00669,159617:0.02283)0.973.36:0.00591,(((572311:0.01457,((253509:0.0499,249736:0.03574)0.978.29:0.02103,((551139:0.01868,543154:0.02226)0.882.11:0.00388,((575467:0.0196,(566943:0.02086,(534686:0.02158,549340:0.00862)0.624.3:0.00352)0.673.2:0.0031)0.775.7:0.00122,(361100:0.00322,((533848:0.01038,214749:0.01244)0.931.18:0.00501,202422:0.02072)0.738.13:0.00081)0.859.10:0.00496)0.853.12:0.00171)0.861.16:0.00168)0.879.18:0.00167)0.956.24:0.00891,560776:0.01409)0.517.3:0.00278,557496:0.01108)0.876.16:0.00193)0.939.25:0.00442)0.965.24:0.00565)0.934.23:0.0042)0.900.15:0.00312)0.960.36:0.00721)0.967.22:0.0085,((394416:0.00622,169002:0.00903)0.984.36:0.00731,538089:0.02396)0.789.7:0.00531)1.000.584:0.02055)0.924.17:0.00945,((352105:0.03936,(568985:0.01849,(241569:0.02155,342018:0.01545)0.182.2:0.00321)0.751.3:0.00633)0.910.16:0.01188,((557255:0.00924,((156973:0.00938,(210077:0.00101,86386:0.01334)0.976.37:0.00767)1.000.585:0.02413,(419579:0.01493,(248208:0.01714,(136009:0.00103,242598:0.00797)0.974.28:0.01261)0.387.4:0.00896)0.449:0.00741)1.000.586:0.01573)0.249.2:0.00211,(226082:0.00409,131220:0.01481)0.863.16:0.00265)0.985.30:0.01009)0.905.15:0.00525)0.954.25:0.0092)0.648.4:0.00609)0.318.2:0.00576)0.878.17:0.0058)0.939.26:0.00794,((((550530:0.0226,((101367:0.01533,(594282:0.01681,158397:0.03513)0.981.44:0.00864)0.904.13:0.00278,(180796:0.01318,146595:0.00874)0.893.9:0.00365)0.754.9:0.01071)0.992.55:0.01395,(6772:0.01473,107438:0.01801)0.955.28:0.00819)0.875.16:0.00523,(245116:0.01748,(297924:0.02917,((571956:0.00399,(186982:0.01612,145681:0.00325)0.988.53:0.00646)0.994.52:0.00411,583038:0.00624)0.871.26:0.00594)0.726.6:0.00718)0.969.30:0.00954)0.955.29:0.00777,((((569792:0.01018,(539260:0.012,543991:0.00624)1.000.587:0.01238)0.797.9:0.00528,((109307:0.00031,248472:0.03309)0.998.104:0.01345,110737:0.02542)0.426.3:0.0036)0.835.8:0.0056,150235:0.00843)0.993.47:0.01216,(563937:0.01502,((((114743:0.0112,63266:0.02328)0.998.105:0.0139,(563027:0.00985,583720:0.01032)0.559.3:0.00239)0.542.2:0.00408,(((107729:0.02778,6734:0.01575)0.875.17:0.0068,214916:0.0031)0.980.32:0.0109,(200484:0.00306,553673:0.01945)0.022:0.00606)0.994.53:0.01209)0.864.11:0.00826,(573892:0.03687,(559929:0.01943,516574:0.0107)0.994.54:0.01366)0.967.23:0.00925)0.982.46:0.0105)0.342.3:0.00234)0.987.56:0.01161)0.986.34:0.01241)0.745.8:0.00168)0.286.2:0.0037)0.814.7:0.00367)0.740.12:0.00143)0.963.29:0.00751)0.452.3:0.00326,(((99071:0.02865,204877:0.02341)0.749.6:0.00817,(530501:0.04313,334185:0.0202)0.994.55:0.01853)0.851.11:0.00313,(((((340957:0.0218,582930:0.00311)0.968.22:0.00605,(((224948:0.03236,331690:0.02004)0.956.25:0.00877,((340557:0.01395,533198:0.02222)0.924.18:0.00498,(591499:0.01692,((145512:0.01208,(202850:0.01617,363874:0.00328)0.599:0.00456)0.993.48:0.01138,(470759:0.00443,(263236:0.04566,217779:0.01142)0.973.37:0.01196)0.773.6:0.00214)0.944.15:0.00459)0.874.17:0.00257)0.774.5:0.00185)0.946.28:0.0062,(((7303:0.00015,94668:0.00935)0.853.13:0.00704,213506:0.00571)0.674.4:0.00389,((((((61192:0.00588,278562:0.00693)0.951.29:0.00836,256121:0.00492)0.947.30:0.00374,94238:0.01419)0.852.9:0.00981,((((143183:0.00172,(215915:0.00546,252382:0.00921)0.982.47:0.00393)0.674.5:0.00952,((109763:0.00547,111083:0.00646)0.761.8:0.00696,(189345:0.01823,110502:0.00323)0.957.42:0.00016)0.991.42:0.0061)0.843.7:0.00237,531165:0.00637)0.891.21:0.00245,(21324:0.0094,287560:0.02283)0.419.4:0.00106)0.809.10:0.00222)0.900.16:0.00243,((592988:0.00493,7344:0.00866)0.986.35:0.00698,(567104:0.00389,544493:0.02151)0.881.12:0.00394)0.827.7:0.00159)0.953.18:0.00585,(((144767:0.01689,(((251263:0.01285,533088:0.00453)0.549.2:0.00016,142425:0.01837)0.575.3:0.00345,(((587586:0.02956,((567842:0.00906,562567:0.01733)0.798.5:0.00266,586230:0.01734)0.875.18:0.01319)0.852.10:0.00363,(303815:0.01335,532409:0.00653)0.679.4:0.0118)0.920.13:0.00359,(((580561:0.01471,115250:0.00815)0.434.3:0.00091,((222289:0.01577,329302:0.0084)0.906.17:0.00381,(139019:0.00728,32294:0.0097)0.468.2:0.00308)0.977.37:0.00555)0.782.9:0.00083,(171310:0.01966,(352209:0.00707,(202769:0.00949,585352:0.00373)0.899.17:0.00472)0.504.2:0.00156)0.998.106:0.00811)0.753.13:0.00085)0.794.9:0.00134)0.714.4:0.00457)0.159.2:0.00311,(171844:0.01857,573270:0.00642)0.898.18:0.00235)0.763.8:0.00083,(76765:0.003,7289:0.00669)0.332.3:0.00783)0.908.20:0.00014)0.957.43:0.00415)0.918.14:0.00015)0.484:0.00403)0.956.26:0.00396,(105930:0.00169,((514468:0.00015,(546403:0.00014,(563357:0.00014,572792:0.00763)0.640.3:0.00015)1.000.588:0.02253)0.992.56:0.00763,526573:0.00487)0.886.15:0.00219)0.758.10:0.00467)0.876.17:0.00285,(((247545:0.01041,(537594:0.0077,(570154:0.05684,((((((((241809:0.00566,259460:0.00405)0.964.37:0.00783,((((253124:0.00364,((247129:0.00255,510182:0.00948)0.781.6:0.01002,162184:0.006)0.907.15:0.00967)0.950.36:0.00385,((365805:0.00638,(7316:0.00682,7347:0.0191)0.989.45:0.00858)0.378.4:0.00153,88301:0.04048)0.454.2:0.00166)0.848.12:0.00028,(410684:0.00507,102382:0.01417)0.756.12:0.00467)0.994.56:0.0047,((((590309:0.0059,551576:0.00339)0.906.18:0.01176,(466815:0.00475,494857:0.01119)0.822.6:0.00156)0.957.44:0.00154,106998:0.00607)0.096.4:0.00169,(((335466:0.00831,327622:0.03054)1.000.589:0.00079,342306:0.01683)0.907.16:0.0007,322045:0.02083)0.453.2:0.0011)0.850.15:0.0016)0.570.5:0.00014)0.599.2:0.00031,569272:0.0167)0.984.37:0.00409,(164523:0.01204,93999:0.01301)0.960.37:0.00365)0.985.31:0.00463,(((((286535:0.01026,(((335562:0.01731,141833:0.01314)0.940.23:0.00842,401837:0.00649)0.911.17:0.00311,((568152:0.00395,359091:0.01401)0.873.14:0.00826,7349:0.00042)0.939.27:0.00465)0.794.10:0.00224)1.000.590:0.00458,284177:0.05262)0.399.3:0.00362,(239263:0.01027,(329777:0.00698,278496:0.00791)0.860.19:0.00235)0.884.20:0.0017)0.744.10:0.00015,225140:0.00485)0.243.2:0.00016,81997:0.01732)0.434.4:0.00145)0.734.8:0.00247,140398:0.00897)0.201.2:0.00114,(225949:0.01778,583060:0.02735)0.789.8:0.00539)0.910.17:0.00741,122054:0.02919)0.995.62:0.01578)0.959.28:0.01128)0.941.33:0.00329)0.994.57:0.00791,((325947:0.00352,((133568:0.00477,97447:0.00728)0.956.27:0.00498,(536196:0.02356,67839:0.03598)0.554.3:0.00201)0.998.107:0.0117)0.531.3:0.00751,(533155:0.00464,170509:0.01046)0.884.21:0.00016)0.969.31:0.00544)0.936.23:0.00016,352640:0.00704)0.992.57:0.01705)0.996.73:0.0135,((((((537790:0.00092,570301:0.00014)0.000.23:0.01414,264116:0.00951)0.946.29:0.00989,((509575:0.00869,315886:0.02449)0.927.23:0.00629,546705:0.01842)0.910.18:0.00471)0.858.8:0.00453,(324539:0.0028,348675:0.00863)0.763.9:0.00784)1.000.591:0.02676,354510:0.00668)0.884.22:0.00477,(((339951:0.01428,((7396:0.01604,293753:0.01039)0.427:0.00188,((100095:0.0364,(44156:0.07971,243613:0.0255)0.000.24:0.00355)0.897.13:0.00155,111660:0.01814)0.499.2:0.00413)0.999.147:0.01282)0.998.108:0.01154,((346072:0.01393,(348496:0.00863,287646:0.02783)0.000.25:0.00028)0.949.27:0.0017,(563550:0.05151,38789:0.03152)0.984.38:0.00624)0.626.2:0.00278)0.574.2:0.01049,257119:0.03185)0.727.5:0.00559)0.952.24:0.00861)0.997.85:0.01008)0.992.58:0.00934)0.843.8:0.00171)0.979.38:0.00982,((((558170:0.02577,(320681:0.05375,303777:0.06904)0.974.29:0.01673)1.000.592:0.04345,((((545465:0.02799,28592:0.09625)0.179:0.00375,(320002:0.02084,(576928:0.02313,306159:0.01735)0.149.2:0.00363)0.975.29:0.00883)0.948.17:0.00751,((((170339:0.02367,561367:0.00798)0.939.28:0.00474,((((((((513449:0.03096,((310289:0.00115,335496:0.02798)0.996.74:0.02088,558043:0.02821)0.905.16:0.01146)0.647.2:0.00638,(256947:0.02505,574956:0.02442)0.978.30:0.01401)0.988.54:0.01215,(106627:0.07042,107152:0.02395)0.735.4:0.00163)0.869.19:0.00505,(((334615:0.0211,(((((((568325:0.01112,576292:0.01207)0.862.18:0.00015,(310843:0.00566,97627:0.01341)0.341.2:0.0018)0.924.19:0.00297,(560444:0.00721,(324457:0.00961,558523:0.00749)0.995.63:0.00952)0.894.22:0.00269)0.991.43:0.00555,(106252:0.00646,248136:0.00822)0.944.16:0.00318)0.849.7:0.00159,307150:0.01609)0.901.13:0.00244,(300229:0.02001,68619:0.0325)0.898.19:0.00497)0.606.5:0.00246,((155949:0.00749,7037:0.00256)0.719.2:0.0014,((244916:0.00028,374294:0.01186)0.971.27:0.00542,(((593598:0.02229,253142:0.01627)0.992.59:0.00471,65461:0.01236)0.931.19:0.00164,241338:0.00059)0.923.13:0.00234)0.937.25:0.00282)0.793.5:0.00014)0.990.44:0.00686)0.960.38:0.00016,(78243:0.02479,((((((7098:0.00311,7088:0.01733)0.764.8:0.00081,((((250332:0.00859,119723:0.01349)0.874.18:0.00016,126382:0.01906)0.952.25:0.00386,201510:0.01025)0.494.4:0.00358,99062:0.00093)0.854.8:0.00476)0.836.14:0.00139,((580865:0.00789,142675:0.00028)0.998.109:0.01188,39467:0.00718)0.924.20:0.00251)0.973.38:0.00471,((((251020:0.01591,(141709:0.0105,74685:0.01092)0.284.2:0.00403)0.888.7:0.00468,((565542:0.00512,(430242:0.01265,575562:0.03685)0.795.10:0.00079)0.793.6:0.00155,141042:0.00812)0.975.30:0.00311)0.607.3:0.00092,((83899:0.0076,342494:0.01116)0.922.16:0.00343,((315987:0.01423,191590:0.0129)0.667.3:0.0048,82958:0.00475)0.865.15:0.00384)0.987.57:0.00692)0.941.34:0.00237,(416615:0.00321,(89333:0.01326,(58864:0.01421,(95932:0.01167,499955:0.01197)0.926.31:0.0001)0.930.21:0.0024)0.857.10:0.0052)0.922.17:0.00845)0.889.14:0.00039)0.797.10:0.00274,((442577:0.00413,(((546335:0.00643,150777:0.00285)0.628.5:0.00058,107032:0.00887)0.981.45:0.00466,((578828:0.0066,552067:0.00014)1.000.593:0.02296,557874:0.00234)0.908.21:0.00261)0.995.64:0.01135)0.858.9:0.00387,147885:0.01056)0.902.10:0.0024)0.983.42:0.00548,7063:0.0094)0.960.39:0.00597)0.624.4:0.00275)0.502.2:0.00164,(((547824:0.00794,((200358:0.00548,(6940:0.0158,352511:0.00528)0.923.14:0.00878)1.000.594:0.01344,((((58604:0.00812,(((571114:0.01524,(6925:0.01016,187316:0.0003)0.941.35:0.00404)0.922.18:0.00317,((30714:0.05,576010:0.03671)0.245.2:0.00029,(181878:0.00562,540808:0.00565)0.992.60:0.00392)0.939.29:0.00299)0.928.18:0.00246,((522675:0.00014,(((501642:0.00381,167380:0.00673)0.989.46:0.01107,303409:0.00736)0.528.3:0.00014,(66687:0.00327,258052:0.01119)0.862.19:0.00485)0.991.44:0.0064)0.755.12:0.00094,108740:0.00925)0.927.24:0.00326)0.774.6:0.00082)0.353.2:0.00765,(263851:0.01064,108777:0.00745)0.991.45:0.0075)0.807.6:0.00492,(111232:0.01518,(105081:0.01346,(137154:0.00894,571375:0.00471)0.995.65:0.01395)0.979.39:0.01042)0.378.5:0.00119)0.938.27:0.0064,(((263146:0.03641,166388:0.00014)0.988.55:0.0061,((((584835:0.02477,((358807:0.02061,(306436:0.02505,239064:0.02893)0.843.9:0.00932)0.799.6:0.00502,((385196:0.00183,460813:0.00259)0.998.110:0.01006,21007:0.01304)0.998.111:0.01394)0.960.40:0.01005)0.967.24:0.00627,204756:0.00695)0.733.7:0.00074,447775:0.00921)0.995.66:0.00795,265983:0.00631)0.772.7:0.00097)0.407:0.00264,499118:0.01882)0.975.31:0.00913)0.873.15:0.0032)0.748.9:0.00104)0.957.45:0.00544,(112155:0.0252,250579:0.00801)0.940.24:0.00421)0.619:0.00449,327197:0.01011)0.960.41:0.00704)0.955.30:0.00532)0.743.5:0.00109,((152484:0.01056,((326023:0.00847,142262:0.03459)0.882.12:0.00625,141444:0.00709)0.985.32:0.01389)0.987.58:0.01383,278251:0.03031)0.954.26:0.00745)0.811.6:0.00217,(((580578:0.01589,533886:0.00879)0.976.38:0.00941,(((332121:0.00763,194785:0.01335)0.962.24:0.00441,((302458:0.00857,(107879:0.00472,164959:0.01084)0.782.10:0.00074)0.891.22:0.0024,(315287:0.00544,((113128:0.02261,(154792:0.02367,(269558:0.00174,224825:0.01924)0.777.9:0.00281)0.395.4:0.00032)0.900.17:0.00196,(377330:0.00275,76621:0.01071)1.000.595:0.01987)0.820.5:0.01177)0.860.20:0.00168)0.835.9:0.00014)0.881.13:0.00228,(112662:0.01968,210891:0.01196)0.923.15:0.00238)0.784.7:0.00168)0.288.2:0.00136,(((((124653:0.00313,((((243487:0.01522,((318371:0.00814,102136:0.0096)0.778.11:0.00784,(332569:0.017,270828:0.0064)0.775.8:0.00075)0.101:0.00015)0.946.30:0.00338,72406:0.00797)0.513.2:0.00079,227875:0.00502)0.770.6:0.00234,328912:0.01586)0.881.14:0.00184)0.922.19:0.00316,((342959:0.04124,((589483:0.02064,148264:0.0056)0.427.2:0.00388,211171:0.0244)0.856.9:0.0049)0.122.2:0.00155,229790:0.02371)0.812.11:0.00015)0.937.26:0.00237,345255:0.01045)0.892.14:0.00233,((312672:0.01734,(89424:0.01042,((((((59902:0.00866,62982:0.00728)0.870.18:0.00325,(50139:0.00791,227889:0.01198)0.972.40:0.00389)0.423.2:0.00509,(((222656:0.00186,6999:0.01328)0.854.9:0.00696,(104501:0.00753,(((590960:0.01552,230788:0.02157)0.938.28:0.00108,((162964:0.02065,353754:0.00964)0.872.28:0.00235,(228981:0.02784,(232802:0.00353,(216339:0.00745,538618:0.01488)0.479.5:0.00282)0.758.11:0.00016)1.000.596:0.00014)0.927.25:0.00227)0.479.6:0.00031,(551794:0.01476,263522:0.01309)0.892.15:0.00477)0.266.3:0.00087)0.459:0.00555)0.782.11:0.00225,(230607:0.01083,(236168:0.02147,546951:0.01632)0.628.6:0.00162)0.887.19:0.00268)0.971.28:0.00478)0.745.9:0.00107,((81130:0.00565,279450:0.01045)0.752.7:0.00403,141254:0.02448)0.861.17:0.00244)0.757.7:0.0008,(579946:0.01774,133557:0.00975)0.924.21:0.00327)0.840.6:0.00154,72607:0.02451)0.856.10:0.00244)0.969.32:0.00366)0.708.5:0.00014,((586108:0.00663,((291214:0.00705,(((323062:0.00876,323641:0.00294)0.826.7:0.00077,142773:0.02024)0.949.28:0.00312,((194205:0.00213,279595:0.02228)0.962.25:0.00015,53922:0.01511)0.945.21:0.00233)0.883.18:0.00329)0.838.12:0.0025,344929:0.01439)0.641.4:0.00636)0.901.14:0.00657,((106999:0.02079,544356:0.01191)0.887.20:0.00447,((315609:0.02207,(337173:0.029,(354682:0.02681,320222:0.0143)0.109.3:0.00572)0.232.2:0.00642)0.495:0.00631,(333920:0.02793,141075:0.01103)0.930.22:0.00698)0.970.23:0.00871)0.951.30:0.00618)0.910.19:0.00533)0.760.13:0.00141)0.933.21:0.00251,6997:0.00339)0.435.6:0.00164)0.945.22:0.0041)0.904.14:0.00358,((351718:0.00892,115132:0.01943)0.505.3:0.00193,325206:0.0025)0.967.25:0.00631)0.851.12:0.00252,((136160:0.01701,(((6955:0.0037,(156795:0.00078,236786:0.0012)0.760.14:0.01941)0.987.59:0.00628,(((((248059:0.00348,572939:0.02343)0.797.11:0.01031,(((6972:0.00787,564336:0.00618)0.724.7:0.0112,(((584580:0.0176,90602:0.00863)0.822.7:0.00312,((552264:0.00375,105640:0.00156)0.987.60:0.00625,(200302:0.00252,82776:0.01437)0.988.56:0.00016)0.842.9:0.00155)0.796.9:0.00227,(593538:0.00816,78049:0.00727)0.918.15:0.00015)0.952.26:0.00394)0.917.19:0.00283,557877:0.01329)0.570.6:0.00042)0.933.22:0.00071,6942:0.0028)0.431.4:0.00302,((519816:0.00644,521233:0.02203)0.994.58:0.004,511920:0.00537)0.941.36:0.00467)0.894.23:0.00311,547575:0.01488)0.942.25:0.00323)0.483.4:0.00327,144759:0.01207)0.902.11:0.00751)0.474:0.00507,98380:0.01038)0.970.24:0.00601)0.989.47:0.00601)0.860.21:0.00232,(((546567:0.0062,(42541:0.01519,339015:0.00107)0.998.112:0.00835)0.999.148:0.01049,(235991:0.00543,255518:0.02509)0.922.20:0.00339)0.973.39:0.00625,168682:0.00553)1.000.597:0.01579)0.666.5:0.00369,(347702:0.02081,(203349:0.03176,220686:0.01236)0.937.27:0.01051)0.978.31:0.01182)0.875.19:0.00529)0.589:0.00428,(((((((114194:0.0045,101552:0.0587)0.782.12:0.00082,((88142:0.0118,81341:0.00371)0.895.20:0.00015,(321156:0.01711,(590917:0.0105,(281172:0.00662,7151:0.00267)0.843.10:0.00332)0.975.32:0.00407)0.968.23:0.00527)0.994.59:0.0087)0.354.2:0.00714,404097:0.02065)0.898.20:0.00287,162332:0.00173)0.969.33:0.00552,(((((109619:0.0086,557460:0.01562)1.000.598:0.00031,((283210:0.00296,263929:0.0099)0.700.4:0.00014,7162:0.02407)0.637.3:0.01084)0.390.2:0.00161,(562835:0.00618,592171:0.02623)0.845.13:0.01165)0.936.24:0.00328,340633:0.0122)0.877.17:0.00613,261931:0.02262)0.927.26:0.00517)0.898.21:0.00266,((555018:0.0069,((143699:0.02044,233054:0.0059)0.640.4:0.005,(135315:0.00854,277679:0.00985)0.896.18:0.00473)0.987.61:0.0087)0.848.13:0.00235,141400:0.02083)0.930.23:0.00243)0.931.20:0.00396,((((101184:0.01219,(58127:0.01109,(556657:0.00353,160195:0.0009)0.986.36:0.00175)0.897.14:0.01022)0.793.7:0.00169,(103204:0.00672,142529:0.0131)0.928.19:0.00854)0.512.3:0.00316,(((((545997:0.02808,169856:0.02118)0.844.10:0.00511,(279440:0.00402,(333262:0.01456,(250274:0.00982,258496:0.01286)0.918.16:0.00156)0.743.6:0.00699)0.836.15:0.00221)0.480.3:0.00014,((348938:0.01366,117682:0.00329)0.254.6:0.00384,((((140904:0.00764,260565:0.00795)0.969.34:0.00564,140825:0.0003)0.855.13:0.00492,(136596:0.01004,((107880:0.0089,557756:0.01756)0.818.5:0.00495,216700:0.01117)0.740.13:0.00067)0.851.13:0.00346)0.745.10:0.00282,((313285:0.01894,585468:0.00237)0.960.42:0.00015,((322858:0.01235,553551:0.01254)0.151.2:0.00187,114783:0.00487)0.947.31:0.00225)0.792.10:0.00077)0.927.27:0.00316)0.864.12:0.00314)0.975.33:0.00392,((((221331:0.01447,43917:0.01412)0.987.62:0.00547,(217417:0.00611,(484628:0.01279,328467:0.00312)0.893.10:0.00273)0.905.17:0.00249)0.911.18:0.00247,(566695:0.00999,(254624:0.00672,((((364101:0.00525,((226906:0.00561,114102:0.0054)0.860.22:0.00555,246240:0.03174)0.931.21:0.01541)0.829.10:0.00014,(214611:0.04042,111853:0.01749)0.986.37:0.0113)0.960.43:0.00015,81483:0.01795)0.923.16:0.00073,((234716:0.01989,248403:0.02499)0.871.27:0.00015,(217402:0.01962,576053:0.00327)1.000.599:0.00575)0.914.16:0.00156)0.917.20:0.00015)0.909.15:0.00235)0.860.23:0.00157)0.896.19:0.00233,((153016:0.0091,305378:0.00474)0.397.2:0.00446,(98927:0.01119,239229:0.01408)0.882.13:0.00231)0.812.12:0.00078)0.985.33:0.00016)0.051.2:0.00155,(((105380:0.01539,282506:0.0076)0.966.29:0.00393,(((355367:0.00878,(323551:0.01209,172018:0.00858)0.964.38:0.00557)0.711.4:0.00163,(113507:0.0037,((203542:0.00721,340339:0.01912)0.763.10:0.00078,23589:0.01269)0.543:0.00469)0.879.19:0.00307)0.786.11:0.00554,(17679:0.00956,(218435:0.00662,38002:0.00943)0.892.16:0.00158)0.778.12:0.00083)0.862.20:0.00308)0.743.7:0.00077,((153307:0.01176,581639:0.01554)0.937.28:0.00402,143057:0.0087)0.838.13:0.0015)0.951.31:0.00234)0.990.45:0.00016)0.884.23:0.00015,(233730:0.00152,(159681:0.01198,(100284:0.00473,208255:0.02223)0.980.33:0.00675)0.941.37:0.00403)0.997.86:0.00716)0.936.25:0.00488)0.974.30:0.00742)0.823.6:0.00569)0.854.10:0.00593,103556:0.06046)0.043:0.00362,(7260:0.06981,((138425:0.0253,107473:0.01666)0.487.2:0.01665,124840:0.02657)0.951.32:0.01358)0.077:0.01135)0.967.26:0.01142)0.133:0.00934,257876:0.09643)0.627.3:0.01205)0.972.41:0.01377)0.994.60:0.02274)0.997.87:0.02442,(7495:0.03194,((32422:0.06689,279464:0.00096)0.922.21:0.0081,((320750:0.00945,(391196:0.01434,111241:0.01589)0.532.4:0.00016)0.952.27:0.0087,(549520:0.02275,((517407:0.00571,210453:0.02454)0.389:0.00155,146915:0.01862)0.852.11:0.00934)0.990.46:0.01085)0.465:0.00454)1.000.600:0.03206)1.000.601:0.04409)0.909.16:0.0096,((((100967:0.07954,(366048:0.03414,(229585:0.01213,370121:0.01122)0.989.48:0.01592)1.000.602:0.04399)0.785.7:0.02023,(((149350:0.00959,(7769:0.03664,7767:0.01052)0.877.18:0.01116)0.999.149:0.0226,(469903:0.001,(7765:0.00028,101179:0.00028)0.783.5:0.01112)0.978.32:0.01449)1.000.603:0.05694,((((515969:0.00682,154107:0.01195)0.313.5:0.00399,(249801:0.01059,(241923:0.00422,571953:0.01673)0.974.31:0.00654)0.998.113:0.01276)0.754.10:0.00353,(7760:0.01785,256896:0.01132)0.342.4:0.00309)1.000.604:0.03687,(591874:0.03104,((19288:0.02552,(242036:0.02675,(256790:0.03009,553257:0.0166)0.965.25:0.01063)0.976.39:0.0153)0.050.2:0.00762,(239328:0.00611,(100561:0.00946,(592571:0.01162,572949:0.02328)0.539:0.00207)0.998.114:0.01307)0.975.34:0.01027)0.998.115:0.02189)0.977.38:0.01439)0.950.37:0.0169)0.619.2:0.00778)0.408.4:0.00766,(((((((((((((((((38578:0.00553,163714:0.00476)0.515.3:0.00155,46026:0.01047)0.990.47:0.01334,((((((139424:0.0305,(140048:0.00382,170354:0.00683)0.878.18:0.00723)0.209.2:0.00235,251677:0.00349)0.245.3:0.00311,500246:0.00869)0.896.20:0.00379,576918:0.00965)0.520.3:0.00327,(((350331:0.00361,254070:0.01116)0.917.21:0.00234,8823:0.00391)0.988.57:0.00557,143326:0.01539)0.743.8:0.00102)0.909.17:0.00524,(248544:0.0054,((191114:0.00554,309588:0.00548)0.938.29:0.01246,155988:0.00718)0.850.16:0.0028)0.957.46:0.01)0.996.75:0.01756)1.000.605:0.03327,20333:0.02311)0.935.13:0.01874,(540876:0.01892,((332103:0.01214,(594066:0.00719,347006:0.02093)0.824.7:0.00851)1.000.606:0.03408,((256595:0.01814,(45496:0.0154,38060:0.01062)0.863.17:0.00575)0.900.18:0.00758,((268213:0.03116,(137101:0.02257,(((334913:0.01797,((8895:0.01126,522986:0.01584)0.878.19:0.00423,277954:0.02744)0.962.26:0.00478)0.985.34:0.0067,(316071:0.0235,((84007:0.01562,83231:0.00481)0.993.49:0.00804,((((281772:0.01436,(8886:0.01447,(((233965:0.01472,(184708:0.00306,198608:0.02125)0.340.3:0.0018)0.371.2:0.0032,314823:0.0025)0.467.2:0.00014,8878:0.02695)0.484.2:0.00142)0.507.3:0.00031)0.867.15:0.00088,(272031:0.02555,(107697:0.00397,363584:0.02956)0.911.19:0.00312)0.937.29:0.00156)0.649.4:0.00014,(201152:0.03043,591045:0.06336)0.933.23:0.00443)0.797.12:0.00311,(140788:0.08466,81621:0.00978)0.955.31:0.00468)0.889.15:0.01035)0.848.14:0.00176)0.877.19:0.00324)0.490.2:0.00314,(((284078:0.04053,((71399:0.01337,(311934:0.01533,(8872:0.00803,81059:0.01267)0.732.7:0.00927)0.957.47:0.00664)0.995.67:0.01136,((202397:0.0298,((541460:0.00406,(73940:0.02878,100854:0.0017)0.344.2:0.00638)0.997.88:0.02478,200178:0.03105)0.856.11:0.00992)0.717.4:0.00031,170276:0.00029)0.996.76:0.01336)0.768.8:0.00659)0.808.6:0.0039,((((439090:0.01272,160592:0.00671)0.763.11:0.00231,346907:0.01021)0.987.63:0.00626,140832:0.00173)0.690.3:0.00157,247917:0.01124)0.997.89:0.01332)0.845.14:0.00644,419632:0.01652)0.951.33:0.00739)0.825.11:0.00653)0.637.4:0.00585)0.923.17:0.00543,278583:0.03391)0.908.22:0.00554)1.000.607:0.03865)0.996.77:0.0208)0.729.7:0.00979)0.960.44:0.01183,(((73001:0.0155,342076:0.00043)0.960.45:0.01279,359108:0.01867)1.000.608:0.02707,((((((259542:0.01037,274227:0.01526)0.954.27:0.007,(71138:0.02085,(140774:0.02906,568943:0.00295)0.970.25:0.00607)0.198:0.00016)0.934.24:0.00313,(((279305:0.00231,156459:0.01075)0.996.78:0.01078,(275521:0.01556,586746:0.00951)0.979.40:0.00682)0.987.64:0.0078,(84701:0.01422,(((230079:0.02678,(((140838:0.01051,(140921:0.01806,140915:0.0372)0.983.43:0.01021)0.947.32:0.0031,(511449:0.00947,(71107:0.02348,126847:0.00028)0.509.2:0.00254)0.855.14:0.0003)0.933.24:0.00556,79398:0.00279)0.468.3:0.00942)0.927.28:0.0023,((((161179:0.03256,235663:0.01158)0.918.17:0.00341,101245:0.00736)0.174.3:0.00028,(332320:0.02019,112879:0.03654)0.225.4:0.00258)0.768.9:0.00028,(140854:0.02612,140837:0.03209)0.000.26:0.00126)0.999.150:0.0027)0.938.30:0.00378,((((140818:0.00698,140787:0.01437)0.781.7:0.00103,(140870:0.00769,140912:0.02187)0.933.25:0.00473)0.646.3:0.00955,84384:0.04841)0.593.3:0.00844,140829:0.01615)0.732.8:0.00125)0.873.16:0.00325)0.893.11:0.00416)0.862.21:0.00169)0.948.18:0.00512,((158267:0.01742,8957:0.0118)0.971.29:0.00991,140967:0.01639)0.916.18:0.00815)0.036:0.00578,(144856:0.00483,140805:0.01118)0.393.3:0.01301)1.000.609:0.02479,(246147:0.00946,545268:0.00951)0.999.151:0.02486)0.769.10:0.01058)0.965.26:0.01131)0.879.20:0.0118,(((192719:0.014,161359:0.00981)0.913.21:0.00253,((((((9118:0.00014,(586329:0.00301,23426:0.02626)0.858.10:0.00029)0.943.38:0.00403,(98089:0.00804,157483:0.01522)0.975.35:0.0108)0.605.4:0.00757,264272:0.00963)0.875.20:0.00312,71948:0.00463)0.909.18:0.00729,561093:0.01857)0.774.7:0.0009,81340:0.00964)0.728.7:0.00056)1.000.610:0.03658,((276843:0.03228,(319701:0.01934,((133396:0.00255,268766:0.00028)0.909.19:0.00898,554261:0.01732)0.991.46:0.01693)0.979.41:0.01484)0.996.79:0.01833,((((225795:0.00829,316234:0.01149)0.956.28:0.00729,(223511:0.01379,(32525:0.02284,(377895:0.00339,387053:0.00723)0.786.12:0.00181)0.528.4:0.00248)0.975.36:0.009)0.837.13:0.00508,((517013:0.01607,(((351345:0.01043,((268666:0.00029,144284:0.02511)0.997.90:0.0073,330658:0.00388)0.970.26:0.00544)0.590:0.004,(((188233:0.00157,(147816:0.00665,(569759:0.00464,574998:0.00785)0.488.5:0.0023)0.984.39:0.00392)0.804.6:0.00135,(541977:0.01112,351068:0.00134)0.473.4:0.00159)0.966.30:0.00413,(333961:0.02571,(222466:0.00487,266698:0.00845)0.401.3:0.00155)0.753.14:0.00089)0.897.15:0.00249)0.898.22:0.00327,(214514:0.00886,131585:0.00378)0.960.46:0.00682)0.607.4:0.00439)0.740.14:0.00581,((286422:0.00282,268355:0.03301)0.984.40:0.01803,(107632:0.00651,((73576:0.00328,144487:0.00497)0.763.12:0.0008,538492:0.01402)0.923.18:0.00478)0.888.8:0.00427)0.717.5:0.00831)0.989.49:0.01501)0.999.152:0.02216,((262781:0.01807,382239:0.03312)0.709.5:0.00861,(((((131602:0.00891,140089:0.01546)0.824.8:0.00353,(230971:0.01334,(574889:0.01982,(((365740:0.00083,281062:0.00095)0.994.61:0.00808,322290:0.01327)0.930.24:0.00444,368438:0.01229)0.629.4:0.0012)0.940.25:0.00719)0.750.9:0.00233)0.857.11:0.00427,(594414:0.01477,322845:0.01495)0.726.7:0.0062)0.864.13:0.00737,((269454:0.01291,(8864:0.00826,584370:0.00825)0.952.28:0.00412)0.885.12:0.00302,(268501:0.01667,(278848:0.00876,553622:0.01584)0.926.32:0.00573)0.991.47:0.00935)0.931.22:0.00493)0.884.24:0.00536,((110997:0.01436,163227:0.0049)0.863.18:0.01041,((((81437:0.03433,103969:0.03985)0.872.29:0.00302,(549634:0.00517,126233:0.03293)0.904.15:0.00213)0.958.26:0.00485,((531807:0.02406,146764:0.00233)0.988.58:0.0095,(((158539:0.01539,((257619:0.01991,(179604:0.0063,(554012:0.00483,558502:0.00646)1.000.611:0.0017)0.955.32:0.00312)0.989.50:0.00015,(248182:0.01779,(250915:0.00327,(79265:0.00804,275116:0.0161)0.791.10:0.00015)0.997.91:0.00015)0.980.34:0.00466)0.910.20:0.00657)0.949.29:0.01045,(((131520:0.01325,((235556:0.00484,141317:0.00968)0.977.39:0.01249,81695:0.01743)0.872.30:0.00015)0.967.27:0.00673,(((535384:0.0118,560509:0.02295)0.831.7:0.01033,552598:0.02927)0.274.2:0.00409,8854:0.01436)0.810.7:0.00238)0.872.31:0.00213,((541952:0.00283,581903:0.01936)0.753.15:0.00096,201388:0.01126)0.922.22:0.00238)0.748.10:0.00118)0.933.26:0.00539,(259552:0.0108,560989:0.00826)0.817.10:0.00875)0.390.3:0.00231)0.868.18:0.00383)0.953.19:0.00524,(26868:0.00339,272511:0.00312)0.916.19:0.00384)0.821.5:0.00203)0.550.5:0.00523)0.999.153:0.0228)0.984.41:0.01329)0.883.19:0.01064)0.996.80:0.01786)0.800.6:0.00451)0.999.154:0.0203,((((176420:0.01801,519520:0.04799)0.975.37:0.01384,((161003:0.01812,(564113:0.01172,246186:0.01252)0.983.44:0.00886)0.995.68:0.01122,(((((319651:0.01427,308043:0.02379)0.817.11:0.00246,309342:0.03206)0.865.16:0.01524,((((122102:0.00075,128085:0.03739)0.992.61:0.01523,(535488:0.00221,70325:0.00531)0.075.2:0.00663)0.452.4:0.00159,108410:0.00181)0.936.26:0.00308,(((9344:0.00543,197576:0.02372)0.968.24:0.01287,9288:0.00421)0.131:0.00208,(343589:0.0088,((((((((325564:0.01271,(161613:0.00707,(234491:0.03044,(269860:0.02532,((233383:0.01949,(233454:0.02717,228084:0.03119)0.212.2:0.0018)0.919.18:0.00603,228157:0.01262)0.736.7:0.00941)0.009.2:0.00451)0.259.2:0.0074)0.764.9:0.0072)0.743.9:0.00155,540433:0.043)0.775.9:0.00058,(586740:0.04552,(260031:0.0214,9314:0.02057)0.744.11:0.00164)0.765.11:0.00119)0.160.2:0.00016,247861:0.00286)0.872.32:0.00371,((((((552887:0.00241,533226:0.00029)0.923.19:0.00246,((128050:0.00014,525907:0.0111)0.908.23:0.0033,245610:0.00931)0.901.15:0.00251)0.927.29:0.00258,((263343:0.01018,137285:0.00517)0.990.48:0.00746,83112:0.00579)0.345.3:0.00095)0.908.24:0.00331,557957:0.00681)0.883.20:0.00315,554703:0.00633)0.852.12:0.00185,(82765:0.00675,162648:0.00716)0.975.38:0.00498)0.882.14:0.00231)0.919.19:0.00232,277815:0.01017)0.926.33:0.00016,((9303:0.00108,9217:0.01176)0.968.25:0.00686,106027:0.01315)0.913.22:0.00154)0.084.2:0.0055,(((((523223:0.01018,(536526:0.0164,((9286:0.00222,146030:0.00109)0.834.14:0.00145,109758:0.00517)0.862.22:0.00205)0.806.7:0.00621)0.971.30:0.00488,(60374:0.01857,570503:0.00634)0.743.10:0.00166)0.717.6:0.00044,(237322:0.01684,(257562:0.00159,((511250:0.011,((234088:0.023,353304:0.01382)0.355.4:0.00084,66026:0.00331)0.886.16:0.00262)0.000.27:0.00086,((((239927:0.01058,536960:0.00495)0.991.48:0.00865,(457104:0.01084,(198609:0.02175,231428:0.03486)0.830.7:0.00102)0.803.5:0.00174)0.750.10:0.00095,((9171:0.00601,(((277751:0.00082,559028:0.00256)0.958.27:0.00415,(82011:0.00086,548949:0.0117)0.930.25:0.00307)0.758.12:0.00083,202555:0.01733)0.765.12:0.0044)0.882.15:0.00248,342299:0.034)0.991.49:0.00016)0.815.5:0.00157,143887:0.01513)0.204.2:0.00076)0.889.16:0.00243)0.337:0.00014)0.924.22:0.00391)0.897.16:0.00253,234481:0.01907)0.763.13:0.00081,(231968:0.01502,(232634:0.02357,230146:0.01525)0.822.8:0.00815)0.745.11:0.0068)0.907.17:0.00233)0.376:0.00171)0.865.17:0.00157)0.860.24:0.00157)0.929.22:0.00564)0.870.19:0.00368,((345203:0.00552,(276875:0.00926,552220:0.00983)0.000.28:0.00149)0.864.14:0.00665,74170:0.02005)0.721.4:0.00039)0.341.3:0.00237,(((224355:0.02054,526734:0.02446)0.912.14:0.00816,(((339560:0.02616,579117:0.01285)0.953.20:0.00618,((9249:0.02931,(9246:0.03347,(9250:0.00178,9245:0.00014)0.998.116:0.01752)0.495.2:0.00962)0.213:0.008,559495:0.00896)0.928.20:0.0063)0.932.26:0.00659,(320634:0.01619,(353585:0.01131,((308893:0.01095,574764:0.00095)0.837.14:0.00492,(196433:0.00327,(553472:0.00516,534555:0.00313)0.994.62:0.01498)0.986.38:0.00776)0.733.8:0.00117)0.665.3:0.00673)0.957.48:0.00728)0.822.9:0.00245)0.963.30:0.00675,((573179:0.01044,369051:0.00937)0.948.19:0.00475,245931:0.11356)0.746.7:0.00527)1.000.612:0.01454)0.640.5:0.00336)0.893.12:0.00938)0.983.45:0.01715,((160210:0.01752,579828:0.01429)0.682.2:0.00171,(171395:0.0042,((((150188:0.00381,(167728:0.00494,(((178835:0.01605,(160207:0.02602,159955:0.02098)0.711.5:0.00114)0.834.15:0.0097,(556190:0.01547,(129244:0.00014,((279374:0.02309,245715:0.02442)0.942.26:0.0056,351250:0.00333)0.508.2:0.00155)0.558:0.00266)0.991.50:0.0063)0.754.11:0.00012,535616:0.01499)0.306.2:0.00077)0.950.38:0.00782)0.900.19:0.00452,(551390:0.0191,(160844:0.00015,564623:0.01285)0.983.46:0.00449)0.882.16:0.00248)0.948.20:0.00441,567653:0.01209)0.903.19:0.0026,((184311:0.01425,(114914:0.00687,102788:0.00396)0.869.20:0.00312)0.980.35:0.00738,((((160550:0.00659,183456:0.01531)0.624.5:0.00455,566122:0.00016)0.923.20:0.00379,348164:0.01106)0.963.31:0.00608,(41265:0.01294,((132680:0.0219,570650:0.00764)0.900.20:0.00664,((9142:0.00645,(((272556:0.02886,274273:0.02209)0.822.10:0.01199,160832:0.00483)1.000.613:0.00584,9141:0.00429)0.851.14:0.00265)1.000.614:0.02137,(549933:0.01305,(((111916:0.00891,511531:0.01043)0.357:0.00145,(((45389:0.00475,70596:0.0042)0.868.19:0.00251,539473:0.0016)0.604.4:0.00014,(9131:0.03057,9129:0.01999)0.744.12:0.0055)0.889.17:0.00289)0.971.31:0.00014,(147578:0.00986,(9128:0.01227,535720:0.00239)0.978.33:0.00868)0.972.42:0.00893)0.632:0.00402)0.987.65:0.00857)0.917.22:0.00549)0.904.16:0.00694)0.773.7:0.00664)0.408.5:0.0018)0.919.20:0.00409)0.905.18:0.00465)0.930.26:0.00599)0.841.8:0.00322)1.000.615:0.02834,((8994:0.00461,(8991:0.01031,((112983:0.02124,149694:0.02369)0.743.11:0.0136,(256478:0.00042,20204:0.01791)0.892.17:0.00245)1.000.616:0.0017)0.852.13:0.01478)0.999.155:0.01994,((9017:0.02186,((88337:0.02948,(101788:0.01035,110157:0.01343)0.697.4:0.00151)0.751.4:0.00153,(((((253801:0.0026,9005:0.01116)0.717.7:0.00399,349841:0.00015)1.000.617:0.00876,(18498:0.00409,(268568:0.00888,83672:0.00928)0.864.15:0.0019)0.915.19:0.00235)0.947.33:0.00389,((232605:0.01382,196033:0.00921)0.796.10:0.00539,213517:0.00661)0.844.11:0.00154)0.920.14:0.00379,(148004:0.01029,((((138313:0.00372,(131370:0.0614,((459403:0.03265,((((146524:0.00497,((108360:0.00188,(229009:0.02665,(151850:0.00671,137236:0.00397)0.770.7:0.00093)1.000.618:0.01193)0.998.117:0.00873,((575055:0.01927,(179131:0.00996,(133479:0.00028,84951:0.02657)0.953.21:0.00896)0.912.15:0.0027)0.568.5:0.00278,575480:0.00781)0.945.23:0.00485)0.853.14:0.00157)0.858.11:0.00157,((((101407:0.00932,22553:0.01014)0.948.21:0.00583,187803:0.00863)0.006:0.00492,(38945:0.02188,75631:0.00508)0.957.49:0.00633)0.759.8:0.0025,113783:0.01496)0.730.6:0.00078)0.958.28:0.00549,((318639:0.01596,242880:0.01301)0.539.2:0.01281,(((9061:0.02784,263552:0.02134)0.676.2:0.00016,(266148:0.0369,119545:0.01595)0.932.27:0.0034)0.016.2:0.00071,(61817:0.0142,438318:0.00164)0.865.18:0.00056)0.763.14:0.00029)0.943.39:0.00234)0.760.15:0.00078,((564653:0.05705,247972:0.00471)0.547.2:0.00307,19933:0.01474)0.047:0.00016)0.976.40:0.02334)0.859.11:0.0026,(278789:0.01306,197265:0.01176)0.841.9:0.00813)0.785.8:0.00174)0.913.23:0.00412)0.860.25:0.00283,(586719:0.00674,(139040:0.01865,140647:0.00764)0.971.32:0.0098)1.000.619:0.01676)0.942.27:0.00449,256999:0.00911)0.992.62:0.00954,147472:0.00376)0.987.66:0.01118)0.979.42:0.0069)0.937.30:0.00707)0.963.32:0.00818)1.000.620:0.01492,((155961:0.01628,152770:0.01164)0.839.12:0.01152,(253162:0.02618,((168803:0.00504,(341133:0.00929,128057:0.01452)0.850.17:0.0041)0.978.34:0.00957,((166811:0.00929,444831:0.02401)0.663.2:0.01243,9123:0.00394)0.985.35:0.00812)0.998.118:0.01511)0.376.2:0.00776)1.000.621:0.0198)0.807.7:0.00732)0.924.23:0.00756)0.597.2:0.00447)0.939.30:0.0103,((237124:0.00557,561170:0.01103)1.000.622:0.04051,(571322:0.01879,((252198:0.01208,274900:0.01476)0.888.9:0.00498,(543247:0.00811,((544980:0.04871,104600:0.00014)0.789.9:0.00551,9402:0.00162)0.985.36:0.00653)0.761.9:0.00134)0.862.23:0.00461)0.988.59:0.01389)0.170.2:0.00814)0.988.60:0.0182,(((((444872:0.01673,554761:0.01477)0.967.28:0.00408,161901:0.02833)0.116.5:0.00065,((162823:0.04308,172125:0.02054)0.440.3:0.00126,(546546:0.0155,25552:0.03491)0.816.9:0.00719)0.819.9:0.00081)0.881.15:0.00219,(215466:0.04645,423025:0.0198)0.702.5:0.00422)0.804.7:0.01018,(((372558:0.01875,(((((((93878:0.01191,152734:0.01003)1.000.623:0.06739,(((91315:0.01541,573612:0.03105)0.958.29:0.00926,(581129:0.01454,((91016:0.00321,9458:0.00107)0.044:0.01475,((583215:0.00131,578335:0.00143)0.981.46:0.00976,(9459:0.00725,590073:0.01973)0.844.12:0.00537)0.841.10:0.00338)0.908.25:0.00615)0.888.10:0.00459)0.939.31:0.01427,((547396:0.00722,9511:0.00143)1.000.624:0.02068,((((((((((((109008:0.01499,(579141:0.01822,((70512:0.0123,9514:0.00015)0.958.30:0.00376,29369:0.00725)0.965.27:0.00678)0.101.2:0.00186)0.772.8:0.0039,(536840:0.01262,(9466:0.01491,((136385:0.00066,286210:0.03118)0.911.20:0.00733,99002:0.00635)0.882.17:0.00512)0.989.51:0.00806)0.868.20:0.00289)0.757.8:0.00095,(35735:0.01534,(106518:0.01182,(23898:0.00843,74465:0.00981)0.933.27:0.00447)0.940.26:0.00432)0.939.32:0.00428)0.801.4:0.00178,((561902:0.00384,(453130:0.0029,(134081:0.00917,(289990:0.00868,129558:0.00028)0.907.18:0.00162)0.983.47:0.00324)0.955.33:0.00451)0.962.27:0.00528,((163461:0.00093,(9541:0.00166,(73432:0.00016,(9539:0.00014,164684:0.00328)0.914.17:0.00165)0.998.119:0.00552)0.882.18:0.00154)0.861.18:0.00227,(186374:0.00549,9551:0.0039)0.941.38:0.00844)0.868.21:0.00239)0.914.18:0.00285)0.936.27:0.00361,(((82728:0.00623,(9504:0.01091,((163465:0.00371,(106648:0.00757,((9617:0.00014,((72830:0.00778,102824:0.00895)0.767.12:0.00201,38515:0.00292)0.969.35:0.00385)0.422.4:0.00077,560709:0.00124)0.999.156:0.0092)0.876.18:0.00274)0.972.43:0.00538,71067:0.0089)0.863.19:0.00504)0.715.7:0.00277)0.955.34:0.00526,(9549:0.00695,579924:0.01602)0.071:0.00178)0.776.10:0.00172,((145814:0.00925,((74211:0.0172,(73428:0.0014,74197:0.00111)0.923.21:0.00209)0.585.3:0.00634,((101255:0.00663,(72378:0.00644,(9577:0.00042,(9565:0.00044,45954:0.00744)0.852.14:0.0008)0.910.21:0.00184)0.713.3:0.0003)0.995.69:0.00469,140514:0.00241)0.769.11:0.0003)0.973.40:0.00658)0.975.39:0.00606,(9550:0.01095,45916:0.00281)0.985.37:0.00733)0.865.19:0.00232)0.861.19:0.00249)0.804.8:0.0016,(32352:0.01282,(101090:0.00501,(271851:0.00148,517257:0.00421)0.952.29:0.00094)0.990.49:0.00708)0.963.33:0.00549)0.864.16:0.0016,(9548:0.01263,9546:0.00242)0.097.3:0.00919)0.746.8:0.00082,(((((36769:0.00701,91583:0.00813)0.960.47:0.0069,(588278:0.00594,83013:0.00939)0.839.13:0.00782)0.966.31:0.00888,((9467:0.00569,74034:0.00504)0.999.157:0.01828,(((558496:0.00014,557614:0.00734)0.977.40:0.00808,91910:0.00116)0.976.41:0.01009,(568276:0.00851,(547345:0.00518,(558276:0.00014,571594:0.00445)0.989.52:0.00635)0.924.24:0.00397)0.861.20:0.00572)0.783.6:0.00416)0.992.63:0.01324)0.897.17:0.00336,(91545:0.0069,(99711:0.00166,(9558:0.00335,(105388:0.00237,9557:0.00128)0.771.13:0.00082)0.931.23:0.00249)0.951.34:0.00945)0.993.50:0.01012)0.956.29:0.00455,((106675:0.01275,(9555:0.00015,293231:0.02455)0.993.51:0.00959)0.932.28:0.00518,582144:0.02677)0.012:0.00167)0.926.34:0.00383)1.000.625:0.00697,(((74655:0.00412,498497:0.02647)0.971.33:0.00473,(((164515:0.00456,((140393:0.00752,139012:0.01056)0.756.13:0.00415,(9498:0.0092,(9502:0.0003,9503:0.00317)0.962.28:0.00651)0.987.67:0.01061)0.909.20:0.00509)0.866.11:0.00353,(562676:0.00548,250243:0.00508)0.534.3:0.01695)0.915.20:0.00396,(17538:0.00655,219846:0.00927)0.741.9:0.00232)0.990.50:0.0065)0.923.22:0.00376,((9477:0.00611,(53170:0.00804,563824:0.00105)0.957.50:0.00395)0.958.31:0.00454,370964:0.00778)0.963.34:0.00621)0.999.158:0.00908)0.548.3:0.00016,((46278:0.00167,(236390:0.03069,(531822:0.02336,526872:0.00736)1.000.626:0.00382)0.878.20:0.00162)0.915.21:0.00255,(70974:0.00177,(251058:0.01355,(149650:0.00512,343424:0.0031)0.915.22:0.00312)0.802.5:0.0012)0.823.7:0.00503)0.989.53:0.00559)0.952.30:0.00525,81142:0.02639)0.853.15:0.00327,(9510:0.00015,(560950:0.00014,571320:0.01268)1.000.627:0.01597)0.355.5:0.02112)0.828.10:0.0061)0.987.68:0.01644)0.819.10:0.01384)0.999.159:0.01904,(((349391:0.02776,(((82772:0.01261,557577:0.01432)0.918.18:0.00377,(285542:0.00587,(550960:0.02959,(249580:0.00361,592303:0.0868)0.100.2:0.00557)0.417.2:0.00485)0.943.40:0.00634)0.950.39:0.00497,(((528990:0.00967,(531978:0.01556,522294:0.01716)0.969.36:0.00998)0.902.12:0.00431,((((10180:0.01026,(((338822:0.03183,(455227:0.03082,108266:0.02715)0.781.8:0.00818)0.997.92:0.01429,(45848:0.01123,538304:0.00486)0.978.35:0.00623)0.558.2:0.00014,10196:0.01716)0.876.19:0.00162)0.801.5:0.00212,(((((254397:0.0199,148901:0.01947)0.904.17:0.00464,9977:0.0096)0.777.10:0.00093,146591:0.00339)0.371.3:0.0015,(((542044:0.0239,52942:0.01562)0.904.18:0.00461,((33643:0.01821,(50763:0.06409,(((161060:0.0545,(((((339873:0.01114,(196767:0.00812,541777:0.01543)0.789.10:0.00396)0.843.11:0.00224,(582303:0.01168,554710:0.01732)0.648.5:0.00336)1.000.628:0.01303,((24924:0.00722,510297:0.00749)0.993.52:0.00961,(((50183:0.0156,(560404:0.01594,577120:0.01391)0.961.28:0.00716)0.408.6:0.00449,583227:0.01002)0.987.69:0.01051,(308102:0.00012,67659:0.00727)0.813.11:0.02928)0.291.5:0.00301)0.712.2:0.00417)0.863.20:0.00246,(36645:0.0104,(580880:0.02071,560641:0.01076)0.941.39:0.0049)0.783.7:0.00276)0.769.12:0.00347,((34972:0.01187,23853:0.0253)1.000.629:0.02939,((((((148394:0.00433,111650:0.02411)0.906.19:0.01115,(53693:0.01745,9647:0.02394)0.905.19:0.00492)0.448.5:0.00296,(171684:0.00922,17473:0.01445)0.999.160:0.01614)0.882.19:0.00285,(33113:0.0257,(9648:0.01445,40406:0.01974)0.997.93:0.0115)0.958.32:0.00708)0.500.5:0.00077,24034:0.03024)0.866.12:0.00855,(146724:0.01361,(9653:0.01484,9651:0.01408)0.733.9:0.00253)0.997.94:0.01339)0.758.13:0.00116)0.868.22:0.00503)1.000.630:0.0323)0.935.14:0.01217,((54656:0.01352,(9649:0.01949,52069:0.0215)0.818.6:0.0051)1.000.631:0.04544,(200690:0.01945,55799:0.02294)0.988.61:0.01685)0.967.29:0.01366)0.952.31:0.011,(141977:0.04201,(((63942:0.01396,32418:0.0197)1.000.632:0.02136,(((135999:0.01371,((201763:0.00487,200350:0.01087)0.999.161:0.01324,208236:0.01524)0.260.2:0.00212)0.940.27:0.00685,((48707:0.00955,17213:0.00889)0.993.53:0.01161,191713:0.01284)0.848.15:0.00279)0.900.21:0.00453,((136008:0.02825,101889:0.01952)0.954.28:0.0079,(138503:0.01207,(42640:0.00962,(74610:0.01302,(101732:0.01644,101382:0.00715)0.124.2:0.00557)0.959.29:0.00769)0.431.5:0.00191)0.309.3:0.0036)0.213.2:0.00407)1.000.633:0.0291)0.286.3:0.00588,((136328:0.02348,(136397:0.02144,136210:0.02019)0.989.54:0.01458)0.999.162:0.03163,((135411:0.01696,(((135960:0.03472,136270:0.0178)0.856.12:0.01222,(158110:0.00504,(136380:0.02899,137237:0.00649)0.911.21:0.00667)1.000.634:0.02029)0.798.6:0.00299,136019:0.03162)0.957.51:0.00769)0.864.17:0.0047,(136211:0.02534,(79106:0.0043,79588:0.02772)0.971.34:0.01038)0.834.16:0.00648)0.892.18:0.00833)0.999.163:0.01934)0.869.21:0.00441)0.827.8:0.00249)0.617.2:0.00863)0.819.11:0.00619)0.928.21:0.00749,(336436:0.0107,63945:0.00544)0.841.11:0.02918)0.952.32:0.00654)0.385.4:0.0028,((592467:0.03133,54828:0.02159)0.000.29:0.00387,27576:0.02397)0.962.29:0.00546)0.842.10:0.00203)0.996.81:0.00928,(560314:0.01111,(163631:0.0063,287604:0.00912)0.511.4:0.00113)1.000.635:0.02994)0.225.5:0.00328)0.852.15:0.00274,(((555870:0.00474,((88086:0.00213,565554:0.00456)0.000.30:0.00029,(9793:0.00238,(10305:0.0142,(((46931:0.00792,51229:0.00652)0.943.41:0.00471,(102194:0.00467,(61198:0.01387,61196:0.01169)0.742.10:0.0099)0.957.52:0.00444)0.363.3:0.00428,45637:0.00089)0.913.24:0.00634)0.984.42:0.00536)0.351.2:0.00982)0.946.31:0.00394)0.870.20:0.00314,(230213:0.00502,(549232:0.01651,10115:0.00798)0.036.2:0.00225)0.991.51:0.00794)0.906.20:0.00015,(((226176:0.00641,541724:0.01548)0.870.21:0.00156,153079:0.01189)0.805.13:0.00326,(((329096:0.01433,(540793:0.01134,575117:0.01578)0.966.32:0.00484)0.995.70:0.00709,535915:0.01834)0.862.24:0.00158,((((213522:0.01338,216947:0.0128)0.987.70:0.03249,(208648:0.05635,65383:0.05055)0.954.29:0.0176)0.771.14:0.00014,((583608:0.0216,((((355736:0.01041,332935:0.01354)0.981.47:0.00757,576533:0.01381)0.987.71:0.00782,586383:0.02554)0.379.7:0.00126,(352486:0.02478,99056:0.01107)0.949.30:0.00582)0.767.13:0.0036)0.903.20:0.00414,293839:0.0058)0.909.21:0.00155)0.646.4:0.0005,((10390:0.01483,(10342:0.00028,283014:0.0034)0.000.31:0.0003)0.884.25:0.00469,(((282207:0.00657,(248470:0.04025,274365:0.01475)0.812.13:0.00195)0.897.18:0.00156,((365162:0.0191,423291:0.00225)0.651.5:0.00222,307582:0.01084)0.786.13:0.00078)0.788.6:0.00078,(10169:0.00255,303883:0.01031)0.874.19:0.0044)0.783.8:0.00079)0.782.13:0.00376)0.857.12:0.00208)0.878.21:0.00156)0.893.13:0.00152)0.911.22:0.00313)0.862.25:0.00222,((((571767:0.02696,(((240810:0.01136,(181487:0.00251,144399:0.00772)0.466.6:0.00346)0.202.2:0.00321,((353216:0.03321,246330:0.02817)0.990.51:0.01302,200105:0.10421)0.755.13:0.00615)0.838.14:0.00123,((((368274:0.01585,217008:0.01136)0.994.63:0.00859,(543441:0.00852,540402:0.02067)0.890.19:0.00594)0.743.12:0.00077,(((552376:0.01086,(((166908:0.02208,((173654:0.02787,305760:0.00992)0.030.2:0.00758,320770:0.00732)0.781.9:0.00401)0.363.4:0.00491,569024:0.0205)0.430.2:0.00042,162322:0.00376)0.918.19:0.0073)0.876.20:0.00044,((148470:0.00402,(((((((((86519:0.02088,(211593:0.03261,567944:0.04145)0.546.2:0.00398)0.009.3:0.00155,(((139117:0.00902,(164430:0.01655,335722:0.01008)0.866.13:0.00156)0.720.5:0.00116,((226404:0.02121,(((236821:0.01664,305982:0.00642)0.860.26:0.00638,(259306:0.01382,581741:0.01584)0.148:0.00382)0.882.20:0.00678,(425949:0.02309,152115:0.01961)0.000.32:0.0006)0.000.33:0.00105)0.891.23:0.00028,23694:0.01002)0.739.13:0.00014)0.000.34:0.00014,(267416:0.01475,270355:0.03173)0.953.22:0.00545)0.736.8:0.00028)0.662.5:0.00155,200697:0.05237)0.629.5:0.00015,(210455:0.05895,(210269:0.02669,202356:0.01364)0.858.12:0.00641)0.919.21:0.00643)0.885.13:0.00147,(((((589308:0.00353,587124:0.01527)0.644.3:0.00015,((561327:0.01853,514015:0.02003)0.977.41:0.00914,(564431:0.02017,(460975:0.02191,201790:0.10393)0.017.2:0.00404)0.755.14:0.00319)0.417.3:0.00014)0.442:0.00154,541119:0.02685)0.597.3:0.00042,(537871:0.02028,(330933:0.00014,257349:0.00213)0.932.29:0.01139)0.793.8:0.00171)0.997.95:0.00014,(341740:0.01324,466445:0.01878)0.764.10:0.00156)0.923.23:0.00092)0.983.48:0.00312,(((((((436723:0.04326,544420:0.03198)0.000.35:0.00029,(291478:0.02891,270543:0.0394)0.853.16:0.00079)0.421.3:0.00127,512044:0.00813)0.085.3:0.00138,44635:0.0105)0.755.15:0.00162,(564752:0.02161,319632:0.04223)0.998.120:0.03675)0.958.33:0.01092,534128:0.02742)0.870.22:0.00261,((280706:0.01428,((9846:0.02595,(168759:0.02244,(258785:0.04262,241415:0.0265)0.999.164:0.02071)0.988.62:0.01089)0.759.9:0.0013,268585:0.07051)0.057.2:0.00878)0.884.26:0.00028,(257388:0.01468,((364163:0.0127,(356864:0.01125,166235:0.00738)0.937.31:0.00236)0.854.11:0.00205,(218220:0.00268,162215:0.04123)0.000.36:0.00671)0.963.35:0.00233)0.881.16:0.0067)0.999.165:0.00411)0.863.21:0.00014)0.948.22:0.00155,10116:0.01453)0.878.22:0.00466,111616:0.00868)1.000.636:0.00563,253061:0.00717)0.936.28:0.0025)0.965.28:0.00686,236285:0.00356)0.902.13:0.00312)0.924.25:0.00059,573696:0.01262)1.000.637:0.00015)0.986.39:0.0039,((287789:0.01653,323418:0.00845)1.000.638:0.04032,316916:0.02049)0.813.12:0.00077)0.862.26:0.00448)0.782.14:0.00015)0.948.23:0.00313,((578714:0.02212,((152527:0.01782,9870:0.00204)1.000.639:0.01351,(259659:0.00156,112784:0.01727)0.000.37:0.00029)0.756.14:0.00075)0.931.24:0.00238,((((((167120:0.00695,(9939:0.00463,((9838:0.00247,9657:0.01532)0.943.42:0.00377,(164789:0.00389,577440:0.00911)0.988.63:0.00787)0.931.25:0.00706)0.169.2:0.00216)0.745.12:0.00088,563359:0.01116)0.168.2:0.00157,((((27032:0.03108,357900:0.04138)0.000.38:0.00044,(168010:0.0007,534898:0.02588)0.890.20:0.0034)0.500.6:0.00014,(349130:0.00787,322033:0.06934)0.974.32:0.00985)0.763.15:0.00014,218720:0.05366)0.893.14:0.01147)0.861.21:0.0016,578058:0.0117)0.767.14:0.00104,((563485:0.01122,(459596:0.00974,101634:0.01268)0.871.28:0.00185)0.638:0.00016,((575074:0.02889,584827:0.0051)0.944.17:0.00412,(96864:0.01319,(484823:0.02399,499901:0.00774)0.926.35:0.01038)0.803.6:0.00155)0.840.7:0.0038)0.881.17:0.00171)0.929.23:0.0026,9807:0.00692)0.284.3:0.00404)0.865.20:0.00154)0.985.38:0.00774,(562424:0.00637,(9727:0.01594,444415:0.00752)0.250:0.00155)0.968.26:0.00805)0.969.37:0.00541,(540533:0.01037,119010:0.01712)0.981.48:0.00746)0.900.22:0.00327)0.905.20:0.00293)0.973.41:0.00871,((112070:0.01774,(204864:0.02502,(589251:0.00341,66558:0.00931)0.865.21:0.00159)0.758.14:0.00719)0.743.13:0.00392,(((((144049:0.00482,551931:0.00695)0.922.23:0.00537,163340:0.01247)0.739.14:0.00105,(10298:0.0058,((88447:0.00701,220541:0.01693)0.925.23:0.00044,(((129220:0.00337,356553:0.01991)0.940.28:0.00293,10243:0.00346)0.821.6:0.00562,((87232:0.00014,241221:0.00028)0.953.23:0.00372,91820:0.00824)0.951.35:0.0042)0.850.18:0.00188)0.987.72:0.0055)0.933.28:0.00311)0.842.11:0.00248,(((146743:0.00411,589696:0.02618)0.989.55:0.00611,171836:0.00488)0.766.13:0.00093,160641:0.00487)0.913.25:0.00155)0.799.7:0.00078,524492:0.00264)0.769.13:0.00568)0.930.27:0.0077)0.839.14:0.00356)0.808.7:0.00177)0.981.49:0.01065,(((21132:0.01715,(567856:0.00907,550234:0.02313)0.962.30:0.00575)0.935.15:0.0069,(52003:0.0216,(86631:0.0134,(139401:0.00155,89402:0.00353)1.000.640:0.00134)0.763.16:0.00075)0.770.8:0.00153)0.950.40:0.01173,210181:0.08248)0.943.43:0.00955)0.900.23:0.00811,142906:0.05142)0.926.36:0.0129)0.988.64:0.01531,(231765:0.01974,((511469:0.02327,(312352:0.00016,173383:0.00028)0.992.64:0.01167)0.945.24:0.00833,(585679:0.03062,99172:0.00814)0.257:0.00615)0.922.24:0.00642)1.000.641:0.02522)0.993.54:0.02396,((539467:0.0874,(355248:0.02001,101021:0.01019)1.000.642:0.07727)0.991.52:0.02998,(((29865:0.01862,((((518316:0.0115,322790:0.00028)0.835.10:0.00972,205408:0.00939)0.823.8:0.0036,9357:0.00927)0.816.10:0.02578,(60771:0.01138,(560865:0.00313,587570:0.00486)0.961.29:0.00667)1.000.643:0.01842)0.788.7:0.00699)0.911.23:0.00723,(158543:0.03065,((562801:0.00174,261254:0.00914)0.213.3:0.05074,(548716:0.00088,310914:0.00557)0.633.2:0.03228)0.895.21:0.01432)0.978.36:0.01507)0.999.166:0.02894,(9351:0.04626,((335220:0.0001,335954:0.00471)0.033.2:0.02713,(333930:0.00486,9349:0.00247)0.327.2:0.02174)0.989.56:0.01831)0.919.22:0.01939)0.779.5:0.01232)0.963.36:0.02415)0.960.48:0.01203,326744:0.00472)0.407.2:0.00159,526811:0.01286)0.844.13:0.00249)0.915.23:0.00515,255907:0.01163)1.000.644:0.01828,((208034:0.01232,136353:0.00597)0.778.13:0.00388,(254988:0.02222,539301:0.00877)0.840.8:0.00256)0.859.12:0.00376)0.973.42:0.01236)0.916.20:0.00833)1.000.645:0.02577,388951:0.02063)1.000.646:0.03714,((((((((8253:0.01129,8297:0.02022)0.912.16:0.00688,(251317:0.02361,160677:0.03204)0.918.20:0.00353)0.868.23:0.00468,(81385:0.00483,(27938:0.0094,24089:0.00506)1.000.647:0.00156)0.768.10:0.00767)0.064:0.0017,(((265950:0.01193,(79540:0.0256,(562316:0.00675,52888:0.00726)0.890.21:0.00788)0.898.23:0.00014)0.875.21:0.00106,271352:0.07191)0.842.12:0.00078,((((((140856:0.00284,152296:0.0547)1.000.648:0.0331,(262508:0.03942,(269774:0.03189,268361:0.02628)0.861.22:0.00227)0.542.3:0.01084)0.246:0.00014,140965:0.01832)0.509.3:0.00206,257083:0.01133)0.336.4:0.00264,140956:0.04082)1.000.649:0.00157,140806:0.0632)0.776.11:0.00076)0.833.11:0.00014)0.976.42:0.00555,327026:0.01444)0.994.64:0.01474,(((467388:0.01775,((8249:0.00665,((8261:0.00197,(223485:0.00243,582913:0.00015)0.999.167:0.01093)0.929.24:0.008,(360893:0.00953,246528:0.00559)0.988.65:0.00128)0.976.43:0.00997)0.865.22:0.00628,8265:0.00263)0.941.40:0.01011)0.680.2:0.0064,248916:0.02561)0.869.22:0.01018,(19444:0.03126,((220416:0.01271,112175:0.01437)0.994.65:0.02009,((379925:0.01543,8250:0.00487)1.000.650:0.03385,(146907:0.0114,(379908:0.02256,438391:0.00766)0.126.2:0.00249)0.044.2:0.00594)0.986.40:0.01279)0.957.53:0.0087)0.867.16:0.00723)0.214:0.01038)1.000.651:0.03396,(((((455948:0.00839,(84904:0.01261,171048:0.00973)0.600.2:0.00395)0.819.12:0.00375,(363410:0.01943,(147715:0.00242,552246:0.00341)0.739.15:0.00087)0.934.25:0.00634)0.552.4:0.00405,(((238484:0.02228,(548118:0.01374,153465:0.00175)0.982.48:0.0096)0.895.22:0.00737,(307143:0.01214,(324983:0.02153,548754:0.01008)0.768.11:0.00652)0.868.24:0.003)0.208.2:0.00016,(155859:0.02669,(228207:0.0174,158223:0.00807)0.939.33:0.02615)0.793.9:0.00268)0.999.168:0.01225)0.963.37:0.00722,((((((250517:0.01277,101152:0.01278)0.877.20:0.00268,(63614:0.00268,54545:0.00636)0.911.24:0.00569)0.909.22:0.00318,(((((248172:0.01332,(310018:0.0161,((257195:0.00962,(240740:0.031,213991:0.01065)0.399.4:0.00015)0.380:0.00226,(250734:0.00966,550870:0.04531)0.845.15:0.00147)0.436.3:0.00028)0.082.2:0.00042)0.384.2:0.00097,(283320:0.02365,360440:0.00471)0.803.7:0.00313)0.917.23:0.01314,((8400:0.00264,(522240:0.01665,(513114:0.02012,524433:0.01807)0.660.3:0.00193)0.292.6:0.00264)0.893.15:0.00014,(238344:0.04652,248562:0.00652)0.928.22:0.00155)0.773.8:0.00207)0.952.33:0.0031,(((30013:0.04142,314572:0.01635)0.958.34:0.03196,293172:0.00445)0.951.36:0.00859,415516:0.00567)0.911.25:0.00108)0.920.15:0.00239,(182826:0.00584,(186836:0.01262,(512270:0.00126,((((576851:0.00014,511035:0.02566)0.879.21:0.00982,209791:0.00605)0.851.15:0.00326,351275:0.00481)1.000.652:0.00382,229145:0.01623)0.961.30:0.00556)0.929.25:0.00313)0.953.24:0.00317)0.895.23:0.00231)0.940.29:0.00337)0.918.21:0.00296,(245702:0.20369,236829:0.08811)0.935.16:0.01978)0.387.5:0.00016,((((((170774:0.01855,((((318598:0.00956,(351148:0.01524,343089:0.01438)0.890.22:0.0014)0.000.39:0.00282,308433:0.02921)0.918.22:0.00042,347685:0.019)0.913.26:0.00016,165523:0.0193)0.923.24:0.0007)1.000.653:0.00016,(((216324:0.02638,302312:0.00997)0.742.11:0.00641,100228:0.01349)0.534.4:0.00024,((((242603:0.01767,(254162:0.02193,249191:0.03054)0.871.29:0.00713)0.838.15:0.01564,322094:0.03267)0.521.3:0.0021,578231:0.01467)0.862.27:0.00014,(328168:0.01083,(356287:0.04043,104391:0.03633)0.408.7:0.00014)0.046:0.00082)0.216:0.00071)1.000.654:0.00014)0.916.21:0.00258,(336987:0.00845,((((514506:0.00719,321738:0.00088)1.000.655:0.00992,290186:0.01271)0.000.40:0.00335,525865:0.00852)0.849.8:0.00488,(288912:0.01916,(291175:0.01874,8320:0.00698)0.711.6:0.00156)0.878.23:0.00094)0.784.8:0.00093)0.950.41:0.00234)0.773.9:0.00301,(((223907:0.00178,342421:0.01179)0.859.13:0.00014,(397091:0.0069,(60267:0.02008,577164:0.02838)0.911.26:0.00074)0.775.10:0.0054)0.826.8:0.00561,(277462:0.00029,(554705:0.01754,426536:0.01353)0.354.3:0.00094)0.920.16:0.01098)1.000.656:0.00011)0.645.8:0.00155,(((518077:0.00667,((222180:0.01309,294782:0.01622)0.963.38:0.01152,206329:0.01362)0.980.36:0.00252)0.899.18:0.00312,((296029:0.02196,73559:0.00834)0.770.9:0.00114,(563369:0.02272,562084:0.00329)0.994.66:0.00813)0.906.21:0.00215)0.458.2:0.00463,332182:0.07997)0.955.35:0.00632)0.767.15:0.0009,(159039:0.00573,((351236:0.00868,(513402:0.00384,(256549:0.01551,60258:0.03137)0.336.5:0.00365)0.477.3:0.00099)0.863.22:0.00124,131304:0.01458)0.750.11:0.00075)0.948.24:0.00295)0.756.15:0.00016)0.808.8:0.00157,208075:0.08857)0.975.40:0.0082)0.261.2:0.0059,(143131:0.03279,(136755:0.01898,124309:0.01479)1.000.657:0.02875)0.326.5:0.00451)0.977.42:0.0144)0.997.96:0.01962,(158243:0.05872,(105696:0.01353,(299539:0.02677,((351581:0.01752,99626:0.01716)0.819.13:0.00795,(158925:0.02249,524618:0.01536)0.261.3:0.00269)0.988.66:0.01326)0.524.2:0.00865)0.992.65:0.01777)0.662.6:0.00811)0.956.30:0.01262)0.898.24:0.00978,((57488:0.00462,(539880:0.02272,(328710:0.01193,585507:0.01383)0.996.82:0.01461)0.392.2:0.00644)1.000.658:0.03118,(568293:0.01852,(556062:0.0165,146995:0.01208)0.553.2:0.00347)1.000.659:0.03446)0.958.35:0.01134)0.922.25:0.00682,(((((156429:0.00436,8115:0.01348)0.102.2:0.00446,(55625:0.01465,((580770:0.01782,((101926:0.02329,(554249:0.00484,(517764:0.04366,369841:0.00432)0.693.3:0.00071)0.952.34:0.00683)0.970.27:0.00926,((((395814:0.0204,((((((524724:0.00857,570843:0.02586)0.783.9:0.00657,(367541:0.00705,(((137560:0.00139,137652:0.02591)0.789.11:0.00502,352742:0.01967)0.333.3:0.00032,((351851:0.00316,445432:0.04791)0.785.9:0.00177,583245:0.01266)0.852.16:0.00316)1.000.660:0.00015)0.822.11:0.00236)0.851.16:0.00159,(337872:0.01041,101344:0.00791)0.926.37:0.0033)0.416.4:0.0008,((557875:0.02432,(369627:0.02108,(25638:0.0013,531203:0.03881)0.958.36:0.01822)0.975.41:0.00477)0.753.16:0.00073,(((550587:0.03086,340723:0.00342)0.716.2:0.01104,(261457:0.00547,548677:0.01276)0.788.8:0.0017)0.903.21:0.00299,(((206143:0.00148,203183:0.06036)0.879.22:0.0126,(200804:0.00347,585456:0.01313)0.797.13:0.0088)0.996.83:0.00948,48370:0.02232)0.781.10:0.00106)0.874.20:0.00156)0.962.31:0.00389)0.873.17:0.00182,(((571136:0.01486,589958:0.02436)0.965.29:0.00603,369463:0.00727)0.756.16:0.00087,(((53048:0.01125,554728:0.00803)0.365:0.00779,113719:0.01495)0.767.16:0.00249,334873:0.0099)0.950.42:0.00313)0.904.19:0.00147)0.885.14:0.00247,((591123:0.02644,(((347570:0.01828,555230:0.01207)0.983.49:0.01324,38116:0.00722)0.930.28:0.00814,(242748:0.00635,((142849:0.01202,((24119:0.02553,363975:0.0132)0.852.17:0.00156,(100919:0.01158,(8146:0.0391,33823:0.00248)0.822.12:0.00156)1.000.661:0.00092)0.905.21:0.00335)0.331.5:0.00082,((140755:0.04514,357869:0.00178)0.752.8:0.00479,(((40862:0.00169,159766:0.0233)0.782.15:0.00313,317854:0.02813)0.961.31:0.00309,(51042:0.01987,((((570752:0.0068,143946:0.01486)0.000.41:0.00028,140938:0.03547)0.869.23:0.00013,(142776:0.03403,(144159:0.02187,143944:0.03264)0.993.55:0.01758)0.999.169:0.04151)0.846.10:0.00076,140764:0.05054)0.012.2:0.0017)0.811.7:0.00636)0.834.17:0.00823)0.880.17:0.00204)0.930.29:0.00335)0.891.24:0.00347)0.869.24:0.0055)0.873.18:0.00306,((269110:0.00818,358381:0.02237)0.831.8:0.00162,(579641:0.02663,((268384:0.00816,223015:0.01175)0.961.32:0.00394,(142310:0.02303,161092:0.00122)0.854.12:0.00193)0.863.23:0.00191)0.963.39:0.00689)0.869.25:0.00183)0.895.24:0.00316)0.940.30:0.00399)0.955.36:0.00415,(((((113253:0.00754,(558075:0.01465,582314:0.01067)0.958.37:0.00672)0.773.10:0.00161,566527:0.00791)0.817.12:0.00432,143508:0.02286)0.532.5:0.00142,((239013:0.018,33515:0.00442)0.484.3:0.0028,354374:0.00094)0.895.25:0.00218)0.940.31:0.00326,362476:0.00582)0.931.26:0.00408)0.814.8:0.00343,((571195:0.01479,156939:0.00587)0.980.37:0.00866,((((144634:0.01737,144625:0.0648)0.343.2:0.00122,69979:0.00237)0.490.3:0.00154,(141553:0.01371,(42325:0.00073,261928:0.00881)0.957.54:0.00325)0.764.11:0.00085)0.912.17:0.00548,324906:0.01168)0.963.40:0.00623)0.885.15:0.00367)0.899.19:0.00342,215031:0.01909)0.823.9:0.00385)0.990.52:0.01269)0.096.5:0.00517,((277216:0.01628,(268814:0.00318,367194:0.00959)0.903.22:0.00412)0.701.2:0.0074,(162545:0.0569,578822:0.00335)0.996.84:0.02074)0.978.37:0.01108)1.000.662:0.02016)0.910.22:0.00816)0.999.170:0.02384,(8111:0.03178,((213472:0.00198,110307:0.03751)0.996.85:0.01563,(140325:0.01056,33068:0.0072)0.562.2:0.00131)0.067:0.00466)1.000.663:0.02609)1.000.664:0.02722,((223694:0.0604,(319880:0.02697,294096:0.03959)1.000.665:0.04502)0.935.17:0.01444,(((((((((553872:0.02002,(540961:0.02721,((572903:0.03574,(233692:0.012,430759:0.01038)0.987.73:0.00595)0.176:0.00434,((586175:0.01417,((((67478:0.01943,(104276:0.01436,79550:0.01198)0.847.15:0.00666)0.890.23:0.00269,571917:0.00755)0.608.3:0.00158,((108910:0.0187,108916:0.026)0.979.43:0.0127,(101558:0.00014,114761:0.00093)0.156.2:0.00485)0.819.14:0.0111)0.980.38:0.00656,542157:0.00772)0.902.14:0.00355)0.958.38:0.00015,(550092:0.01645,547720:0.03101)0.944.18:0.0076)0.748.11:0.00297)0.856.13:0.00706)0.980.39:0.01374)0.963.41:0.01248,(369008:0.02567,560000:0.0179)0.301.3:0.00689)1.000.666:0.03655,(((202620:0.02065,200580:0.02516)1.000.667:0.20523,(102973:0.03902,((46892:0.01875,88180:0.02546)1.000.668:0.0232,(103126:0.01959,((103272:0.01097,(99043:0.01738,(103124:0.01325,103696:0.02019)0.291.6:0.003)0.428:0.00222)0.952.35:0.00011,99356:0.01265)0.875.22:0.00239)0.301.4:0.0042)0.855.15:0.01046)0.999.171:0.04485)0.985.39:0.03026,((((((109464:0.00634,(349392:0.01286,338445:0.01546)0.257.2:0.00011)0.904.20:0.00612,232506:0.00773)0.883.21:0.00324,(545848:0.01642,(228699:0.01906,(229084:0.00042,99933:0.00028)1.000.669:0.00416)0.971.35:0.00698)0.998.121:0.01215)0.989.57:0.01128,((573036:0.01001,(350230:0.00468,593801:0.02747)0.007:0.00779)0.854.13:0.02272,((252370:0.00993,323845:0.0322)0.726.8:0.00691,547357:0.01885)0.983.50:0.00965)0.951.37:0.00859)0.942.28:0.00987,(230774:0.0003,((233968:0.01535,103807:0.00732)0.212.3:0.00119,((124702:0.00705,229598:0.00379)0.975.42:0.00553,204166:0.02291)0.594.5:0.00333)0.853.17:0.0614)0.998.122:0.02751)1.000.670:0.03603,(((((119608:0.0017,331005:0.02227)0.993.56:0.01268,315563:0.01519)0.948.25:0.0102,108109:0.02972)0.904.21:0.00952,106179:0.01517)1.000.671:0.04345,((154863:0.0206,306650:0.01288)0.999.172:0.02511,(101609:0.02617,((173220:0.01855,538133:0.01184)0.995.71:0.01685,(541479:0.00598,581343:0.0169)0.988.67:0.01218)0.998.123:0.0201)0.930.30:0.00968)0.905.22:0.01671)0.999.173:0.02536)0.993.57:0.02216)0.588:0.01292)0.471.3:0.00975,235822:0.07472)0.971.36:0.01242,(((((((314588:0.0137,580917:0.00805)0.993.58:0.01155,(342186:0.01143,(143093:0.00916,141502:0.01313)0.948.26:0.00749)0.999.174:0.02569)0.992.66:0.01478,((((320352:0.01336,(275378:0.03332,144283:0.01702)0.942.29:0.01388)0.306.3:0.00514,(((97143:0.01319,279174:0.00311)0.918.23:0.00171,202861:0.01156)1.000.672:0.00519,319405:0.01735)0.351.3:0.01105)0.908.26:0.00452,(353635:0.0392,(351504:0.02674,(278807:0.00441,499941:0.01965)0.943.44:0.00737)0.575.4:0.00337)0.994.67:0.01041)0.488.6:0.00402,((589864:0.03415,(143401:0.03274,283378:0.00205)0.699.2:0.00119)0.993.59:0.01078,(329506:0.01205,(286184:0.00527,(255548:0.00351,587714:0.03858)0.000.42:0.00955)0.979.44:0.00662)0.645.9:0.00485)0.986.41:0.0116)1.000.673:0.02572)0.789.12:0.00988,(278394:0.01747,241890:0.03183)0.998.124:0.02068)0.918.24:0.0087,(336397:0.02703,(103157:0.01301,307129:0.02128)0.999.175:0.02157)0.917.24:0.01039)0.931.27:0.00813,((((344621:0.01069,(((243171:0.02287,(591238:0.01406,311244:0.01747)0.611.4:0.00269)0.965.30:0.00967,(150054:0.01039,157299:0.01614)0.450.3:0.00272)0.858.13:0.00277,99230:0.01137)0.379.8:0.00336)1.000.674:0.03222,(245893:0.01717,546332:0.01916)0.598.2:0.01117)0.918.25:0.01048,(((((101880:0.02682,(154198:0.01023,((269459:0.00594,267771:0.0076)0.991.53:0.00987,151941:0.00757)0.830.8:0.0031)1.000.675:0.01421)0.932.30:0.00749,((149537:0.0168,((((335870:0.01046,563606:0.01397)0.995.72:0.01049,(272039:0.0041,(159939:0.00924,542803:0.02428)0.983.51:0.01269)0.864.18:0.0049)0.991.54:0.00971,142486:0.00477)0.959.30:0.00589,(253176:0.01578,((((((338117:0.01548,246295:0.02736)0.992.67:0.01267,((578438:0.00891,268323:0.01656)0.951.38:0.00971,(149557:0.01051,529429:0.01247)0.947.34:0.01017)0.580:0.00152)0.990.53:0.00671,((316714:0.01535,544402:0.00459)0.784.9:0.00162,555216:0.01277)0.736.9:0.00063)0.966.33:0.00512,(153700:0.00884,97734:0.00971)0.903.23:0.00888)0.497.3:0.00159,(((((279425:0.006,592957:0.00976)0.867.17:0.00211,(139864:0.00907,138778:0.00781)0.929.26:0.00321)0.843.12:0.00177,(354827:0.00562,(545003:0.01929,((((542707:0.00991,336455:0.00739)0.792.11:0.00015,278961:0.00478)0.916.22:0.00185,(159069:0.01458,(236929:0.01511,217507:0.00415)0.993.60:0.00735)0.922.26:0.0031)0.747.8:0.00051,235343:0.01778)0.855.16:0.00161)0.958.39:0.00468)0.994.68:0.00842)0.977.43:0.00685,((278702:0.01786,245864:0.03787)0.988.68:0.01488,(((258367:0.00798,577998:0.01934)0.977.44:0.00802,((((45957:0.00704,591921:0.02514)0.860.27:0.00966,541798:0.00797)0.426.4:0.00156,236604:0.0082)0.921.26:0.00486,(241462:0.00653,(579964:0.01031,(206237:0.00896,126508:0.01327)0.949.31:0.00685)0.997.97:0.01395)0.057.3:0.00546)0.517.4:0.00513)0.997.98:0.01555,144746:0.0207)0.968.27:0.00836)0.508.3:0.00234)0.852.18:0.00156,((593222:0.01261,(533129:0.00245,220496:0.0174)0.811.8:0.00155)0.877.21:0.00163,(((((218228:0.01456,279539:0.0143)0.973.43:0.00695,359627:0.00376)0.998.125:0.01413,(((149014:0.00232,560245:0.0142)0.776.12:0.0013,(251148:0.01502,219188:0.01829)0.852.19:0.00245)0.992.68:0.0062,(576242:0.00571,308726:0.01547)0.886.17:0.00281)0.761.10:0.0013)0.869.26:0.00457,(((103055:0.00638,357193:0.00963)0.991.55:0.00871,((150972:0.00554,140026:0.00955)0.846.11:0.01147,150492:0.00639)0.908.27:0.00015)0.313.6:0.00302,(335051:0.01467,(510210:0.01381,(278398:0.0144,362119:0.01129)0.944.19:0.00542)0.320.3:0.00368)0.964.39:0.00825)0.936.29:0.00449)0.975.43:0.00555,549550:0.01584)0.960.49:0.00334)0.935.18:0.00328)0.908.28:0.00254)0.992.69:0.00613,351157:0.01146)0.931.28:0.00315)0.928.23:0.00515)0.998.126:0.01148)0.889.18:0.00362,((248442:0.00663,(575798:0.00541,570446:0.01074)0.962.32:0.00624)0.770.10:0.00592,(572372:0.01711,544776:0.00402)0.984.43:0.0098)0.998.127:0.014)0.974.33:0.0103)0.848.16:0.00783,((302021:0.00813,536934:0.02707)0.990.54:0.01355,(((593435:0.00619,(189314:0.00014,95349:0.00192)1.000.676:0.01514)0.971.37:0.00682,((261733:0.00968,278975:0.01136)0.889.19:0.0023,(320618:0.02061,(251646:0.00963,8181:0.018)0.679.5:0.0016)0.834.18:0.00167)0.942.30:0.00421)0.084.3:0.00276,293405:0.02427)0.997.99:0.01689)0.937.32:0.00949)0.943.45:0.00761,(((((110380:0.01327,298139:0.01627)1.000.677:0.03127,((255685:0.00571,67771:0.01147)0.949.32:0.0076,(330481:0.01066,(321910:0.0094,(257504:0.01889,309458:0.01266)1.000.678:0.01531)0.945.25:0.00544)0.948.27:0.00684)0.940.32:0.00846)0.599.3:0.00598,(((269778:0.01969,(310378:0.02262,541691:0.01962)0.239.2:0.00307)0.968.28:0.00943,(530532:0.02922,308842:0.02108)0.993.61:0.01344)0.966.34:0.01297,(((105318:0.00525,8204:0.01472)0.983.52:0.01051,(535454:0.00443,(534841:0.01452,236857:0.00875)0.999.176:0.01692)0.849.9:0.00624)0.994.69:0.01662,((239131:0.01391,246885:0.00982)0.995.73:0.01561,(540649:0.00459,(308505:0.04627,(572166:0.00561,536976:0.01368)0.849.10:0.00241)0.364.3:0.00164)1.000.679:0.02057)0.997.100:0.02365)0.753.17:0.00613)0.878.24:0.00838)0.658.5:0.00279,((((312030:0.00956,580484:0.00574)0.743.14:0.00314,592078:0.0036)0.738.14:0.00482,(575570:0.00539,330181:0.01912)0.861.23:0.00504)0.964.40:0.02759,((278648:0.01104,(223209:0.00634,348578:0.01874)0.785.10:0.00106)0.929.27:0.00475,(247764:0.02839,247833:0.00109)0.983.53:0.00543)0.994.70:0.01252)0.974.34:0.0132)0.970.28:0.00851,(((((((110129:0.00577,(278620:0.0095,(316034:0.01447,321327:0.01052)0.572:0.00291)0.999.177:0.01489)1.000.680:0.01618,((222753:0.0255,(283599:0.02682,((375603:0.00557,252978:0.00446)0.949.33:0.01681,(350896:0.01166,291807:0.01439)0.943.46:0.00666)0.993.62:0.01113)0.710.6:0.00183)0.991.56:0.00897,576714:0.00737)0.997.101:0.01099)0.797.14:0.00417,(111195:0.01263,38752:0.01378)0.993.63:0.01222)0.313.7:0.00293,550044:0.03387)0.935.19:0.00609,154853:0.02289)0.851.17:0.005,((((((8407:0.02118,275246:0.01931)0.988.69:0.01179,(582472:0.0092,(141875:0.006,((110075:0.01843,8405:0.00604)0.996.86:0.01069,172811:0.02149)0.471.4:0.00189)0.995.74:0.01232)0.296.2:0.00015)0.952.36:0.00758,(318714:0.00766,316869:0.0172)0.959.31:0.00649)0.978.38:0.00828,237439:0.02458)0.452.5:0.00108,(234841:0.01607,569907:0.00775)1.000.681:0.03216)0.532.6:0.00463,(396137:0.02107,(143166:0.0306,(198202:0.02189,194319:0.02288)0.615.2:0.00266)0.934.26:0.00759)0.955.37:0.01228)0.875.23:0.00571)0.916.23:0.01045,((165894:0.01061,563459:0.01075)1.000.682:0.03111,((222510:0.00859,95512:0.01376)0.761.11:0.01427,((558634:0.00401,(83738:0.00704,(257184:0.0169,573761:0.00143)0.996.87:0.00901)0.981.50:0.00774)0.934.27:0.00418,(563542:0.00338,((339495:0.01279,83122:0.00829)0.854.14:0.005,190729:0.00631)0.771.15:0.00919)0.952.37:0.01129)0.902.15:0.0058)1.000.683:0.02514)0.904.22:0.00951)0.464.3:0.00533)0.991.57:0.01396)0.644.4:0.00908,((559242:0.00771,546239:0.01038)0.937.33:0.01304,((237863:0.01121,(104331:0.00754,139001:0.0147)0.948.28:0.00553)0.494.5:0.00268,142467:0.00703)0.964.41:0.01316)1.000.684:0.02801)0.906.22:0.00826)0.724.8:0.00637,(581250:0.00553,147823:0.01165)1.000.685:0.02471)0.928.24:0.00849)0.832.9:0.00635,((216695:0.02638,((((161219:0.00111,201234:0.01666)0.986.42:0.01464,239612:0.08811)0.996.88:0.01881,((210420:0.01842,((585439:0.01998,(336560:0.01303,(567013:0.00925,306969:0.00439)0.793.10:0.00795)0.878.25:0.00783)0.966.35:0.01333,(98593:0.0058,(101892:0.00465,228537:0.03089)0.983.54:0.0071)0.937.34:0.00783)0.986.43:0.01286)0.810.8:0.00701,(167947:0.01232,(243868:0.01019,588434:0.01118)0.963.42:0.00881)0.856.14:0.00709)0.955.38:0.01137)0.879.23:0.00945,(181469:0.02151,329740:0.01752)0.990.55:0.01603)1.000.686:0.03236)0.994.71:0.01783,(((255358:0.02465,(334763:0.01378,255625:0.00659)0.200.3:0.00398)0.800.7:0.00715,(544395:0.00737,((323482:0.0128,313341:0.01377)0.783.10:0.01838,(((140175:0.00611,591094:0.02061)0.996.89:0.01198,(533531:0.014,((80175:0.01716,588515:0.00911)0.995.75:0.01408,(98298:0.01648,(361395:0.01664,139522:0.01752)0.482.3:0.00713)0.857.13:0.00313)0.753.18:0.0014)0.946.32:0.00558)0.869.27:0.0053,((224680:0.03504,227917:0.00014)0.996.90:0.01273,(92209:0.01091,337520:0.01662)0.555.2:0.00371)0.881.18:0.00454)0.767.17:0.00349)0.098.2:0.00729)0.987.74:0.01491)1.000.687:0.02933,(((136796:0.00498,582535:0.0015)0.894.24:0.0044,(104318:0.01715,(243013:0.00719,187488:0.00966)0.696.6:0.01653)0.734.9:0.00797)1.000.688:0.02848,((538642:0.03853,314090:0.01825)0.968.29:0.01023,(((105547:0.01774,(316013:0.00354,239588:0.01089)0.927.30:0.00668)0.880.18:0.00611,((139752:0.01292,235311:0.00253)0.173.2:0.00285,(152853:0.00338,322470:0.01288)0.995.76:0.01096)0.629.6:0.00176)0.997.102:0.01395,(((588928:0.00357,536666:0.03094)0.761.12:0.01091,160637:0.01344)0.222.4:0.00446,((532913:0.01633,(311960:0.03263,95477:0.00771)0.767.18:0.01119)0.799.8:0.00417,((230216:0.01258,((101015:0.01113,145918:0.01749)0.995.77:0.0097,(244390:0.01746,(((270852:0.00423,146687:0.01777)0.915.24:0.00849,262549:0.00832)0.994.72:0.01045,263897:0.00709)0.829.11:0.01389)0.439.6:0.0026)0.896.21:0.00034)0.925.24:0.00393,(583231:0.01067,(91272:0.02338,315848:0.01517)0.969.38:0.01004)0.996.91:0.0109)0.977.45:0.00785)0.891.25:0.00516)0.994.73:0.01301)0.980.40:0.01198)0.982.49:0.01189)0.668:0.00615)0.867.18:0.01222)0.781.11:0.00292)0.916.24:0.0047)0.369.2:0.00737,(((217600:0.03002,(107320:0.01959,24515:0.02039)0.753.19:0.0017)1.000.689:0.0298,329714:0.03266)0.978.39:0.01446,(154336:0.08828,((((565326:0.05511,181920:0.04415)0.988.70:0.01862,((((516971:0.04789,(((321939:0.00959,(326525:0.00632,47332:0.01129)0.535.3:0.00695)1.000.690:0.02577,(155199:0.02141,(559562:0.00811,(311032:0.00303,573705:0.00679)0.970.29:0.00812)0.929.28:0.00974)0.995.78:0.01869)0.983.55:0.01263,(268470:0.02369,(((8201:0.01116,461486:0.01514)0.995.79:0.01165,160907:0.00762)0.848.17:0.00269,592739:0.00689)0.992.70:0.00919)1.000.691:0.01678)0.403.2:0.00514)0.926.38:0.00754,(555032:0.00321,(161298:0.00895,546017:0.00366)0.697.5:0.02871)0.937.35:0.02949)0.970.30:0.01106,(536033:0.05232,(124798:0.0605,((537335:0.01118,(156385:0.01759,114569:0.00523)0.897.19:0.0026)0.864.19:0.0018,((((590825:0.00575,((552904:0.00414,(46306:0.00095,103719:0.00405)0.948.29:0.00467)0.974.35:0.00015,246377:0.00601)0.698.5:0.00318)0.856.15:0.00194,471467:0.00284)0.470.3:0.00434,((124036:0.00348,330338:0.00252)0.284.4:0.00155,566270:0.01614)0.947.35:0.00355)0.996.92:0.01319,554750:0.01271)0.698.6:0.00176)0.430.3:0.00124)0.965.31:0.03913)0.757.9:0.0035)0.990.56:0.01219,((((275661:0.01622,((((570214:0.00014,570916:0.0119)0.932.31:0.00015,102362:0.00014)0.930.31:0.00584,(247405:0.02078,(27457:0.03314,8195:0.00638)0.974.36:0.00924)0.790.8:0.00234)1.000.692:0.02073,114192:0.00997)0.996.93:0.01886)0.995.80:0.01463,(((8206:0.02271,326600:0.01729)0.969.39:0.00882,(((512228:0.01135,8193:0.01363)0.965.32:0.00605,(100683:0.12491,546823:0.00232)0.945.26:0.00812)0.152.2:0.00094,8194:0.06723)1.000.693:0.01663)0.997.103:0.01407,(((221209:0.01312,206232:0.02123)0.999.178:0.01589,(188182:0.01196,548989:0.01479)0.973.44:0.01024)0.976.44:0.01144,(534970:0.03195,((154145:0.00502,81975:0.01445)0.993.64:0.01237,376032:0.006)0.995.81:0.01165)0.985.40:0.00901)0.697.6:0.00339)0.378.6:0.00471)0.135:0.00506,(((95209:0.01357,(586625:0.00803,((309637:0.00819,252187:0.00794)0.900.24:0.0021,243296:0.0148)0.892.19:0.00533)0.982.50:0.0103)0.952.38:0.00616,(((544882:0.00622,8190:0.03804)0.994.74:0.01244,(358464:0.01674,(101822:0.00029,(101658:0.01032,79509:0.01455)1.000.694:0.01135)0.999.179:0.01249)0.964.42:0.00863)0.734.10:0.00163,(8176:0.01374,(8183:0.01028,8186:0.01194)0.959.32:0.00779)1.000.695:0.02318)0.967.30:0.00745)0.997.104:0.015,541001:0.05191)0.356.2:0.00394)0.781.12:0.00473,(196860:0.01925,245588:0.04109)0.968.30:0.01335)0.954.30:0.00907)0.928.25:0.0097)0.806.8:0.01217,((166304:0.02032,56490:0.01606)1.000.696:0.03204,((561936:0.03686,(104383:0.0049,(175318:0.0042,545665:0.00775)0.933.29:0.00569)0.850.19:0.0027)0.955.39:0.01068,(87543:0.01337,(155391:0.02134,(165647:0.01029,570256:0.03044)0.791.11:0.00545)0.059:0.00253)0.998.128:0.0191)0.892.20:0.00778)0.971.38:0.01093)0.946.33:0.00946,((532798:0.03779,(((564590:0.00914,109431:0.00707)0.999.180:0.0098,113799:0.00411)0.440.4:0.00076,((218897:0.02379,590022:0.01342)0.901.16:0.00412,252743:0.00873)0.966.36:0.00338)0.999.181:0.0129)1.000.697:0.01764,((241778:0.02382,568866:0.01932)0.348:0.00989,(420931:0.03612,(48284:0.02298,35099:0.01624)0.983.56:0.01292)1.000.698:0.029)0.980.41:0.01292)0.994.75:0.01281)0.389.2:0.00405)0.467.3:0.00585)0.542.4:0.00807)0.948.30:0.00713,267568:0.03028)0.256.2:0.00179,(554141:0.03639,(537713:0.02548,((79309:0.00531,316774:0.01907)0.916.25:0.00389,((139543:0.00453,578191:0.02022)0.893.16:0.00377,333373:0.00671)0.985.41:0.00895)0.998.129:0.01524)0.388:0.00661)0.958.40:0.00784)0.941.41:0.00755,(((8811:0.00028,(134569:0.01636,340031:0.01388)0.645.10:0.00014)0.980.42:0.00539,592685:0.02098)0.763.17:0.04531,((((353089:0.01426,(163334:0.01908,(211053:0.01071,573563:0.00708)0.914.19:0.00416)0.743.15:0.00233)0.955.40:0.00402,((((163520:0.02228,536009:0.006)0.911.27:0.00929,66955:0.00888)0.319:0.0009,(220569:0.00313,581833:0.00535)0.963.43:0.0066)0.928.26:0.0028,(((140400:0.0123,20696:0.01018)0.753.20:0.00079,565377:0.00816)0.845.16:0.00078,((59259:0.02009,549775:0.00359)0.826.9:0.01771,(136420:0.00639,(251584:0.01398,326599:0.05514)0.997.105:0.01467)0.991.58:0.0025)0.888.11:0.00157)0.376.3:0.0003)0.941.42:0.00333)0.929.29:0.0043,245157:0.00504)0.766.14:0.003,((547256:0.00698,(162875:0.00429,192998:0.01456)0.857.14:0.00259)0.960.50:0.00489,(((563189:0.01523,454067:0.02283)0.859.14:0.00016,(78569:0.00646,572202:0.01924)0.999.182:0.00941)0.939.34:0.0028,141623:0.06893)0.812.14:0.00753)0.880.19:0.00309)0.997.106:0.01847)0.992.71:0.02067)0.977.46:0.01111)0.357.2:0.00748)0.789.13:0.00781,((167879:0.02404,535138:0.00408)1.000.699:0.02489,((288207:0.03019,(255697:0.12517,(((((516256:0.0044,(38217:0.00955,140831:0.00701)0.773.11:0.00075)0.976.45:0.00684,(((64506:0.01113,((8488:0.01727,163170:0.02855)0.803.8:0.00312,583913:0.01336)0.985.42:0.00629)0.370.2:0.00314,540331:0.01054)0.977.47:0.00646,(568727:0.01999,168428:0.02517)0.743.16:0.00083)0.874.21:0.00165)0.997.107:0.00908,247240:0.00845)0.986.44:0.00557,((((((8493:0.01001,243015:0.03571)0.996.94:0.00135,((590820:0.03662,581887:0.00467)0.745.13:0.00313,((448893:0.02578,(279553:0.00315,(140946:0.02968,140772:0.02566)0.913.27:0.00476)0.167:0.00154)0.781.13:0.00077,((526385:0.00768,38888:0.0057)0.982.51:0.01178,(((154066:0.05592,140930:0.00015)0.994.76:0.01208,(151046:0.02325,151390:0.02151)1.000.700:0.00093)0.706.4:0.00045,152595:0.04369)1.000.701:0.00237)0.714.5:0.00016)0.956.31:0.00234)0.886.18:0.002)0.964.43:0.00541,72449:0.00085)0.994.77:0.01421,((515065:0.0101,((218565:0.02029,(114808:0.00069,583306:0.00759)0.794.11:0.00732)0.996.95:0.01241,(240439:0.01005,(220548:0.00235,585700:0.00693)0.914.20:0.01508)0.978.40:0.00875)0.999.183:0.0235)0.878.26:0.00159,166436:0.00559)0.935.20:0.00321)0.982.52:0.00703,((39176:0.00156,555622:0.00311)0.784.10:0.00193,(198175:0.00808,((70116:0.00803,((((144519:0.01478,174219:0.03018)0.032:0.00015,185735:0.00925)0.068:0.01218,(194461:0.02985,191096:0.01935)0.739.16:0.0033)0.780.8:0.00026,247615:0.02361)0.961.33:0.00391)0.768.12:0.00738,141280:0.01441)0.753.21:0.00093)0.859.15:0.00314)0.751.5:0.00274)0.750.12:0.00074,(((55384:0.03448,(((163790:0.013,275935:0.05932)0.552.5:0.00338,328340:0.0219)0.000.43:0.00056,(((271906:0.02128,276168:0.03363)0.842.13:0.00698,8525:0.01465)0.849.11:0.00256,263193:0.01238)0.965.33:0.00312)0.790.9:0.00016)0.363.5:0.00014,(170892:0.00895,161169:0.022)0.312:0.0016)0.882.21:0.00156,((53970:0.02252,84705:0.00731)0.974.37:0.00776,((((8433:0.01055,(46183:0.0257,(((590136:0.04909,((143958:0.02954,144098:0.0465)0.974.38:0.01298,161820:0.05303)0.917.25:0.0099)0.000.44:0.00082,285160:0.03892)0.346.4:0.0003,167148:0.02514)0.884.27:0.0017)0.881.19:0.00312)0.965.34:0.00391,(236231:0.00979,520731:0.03343)0.999.184:0.00878)0.789.14:0.00077,(535159:0.02091,(157402:0.00727,263013:0.01895)0.910.23:0.0089)0.762.11:0.0017)0.918.26:0.00234,(543389:0.01461,511565:0.01515)0.469.2:0.00867)0.956.32:0.00313)0.882.22:0.00156)0.954.31:0.00316)0.863.24:0.00161)0.723.7:0.00072,(((((108340:0.01258,(296076:0.03031,344935:0.00043)0.210.3:0.01397)0.887.21:0.00368,((((((202340:0.01265,312742:0.01232)0.852.20:0.00153,53457:0.01411)0.871.30:0.00198,(101162:0.01264,(260507:0.00735,((567049:0.00851,(531082:0.03098,8552:0.00131)0.872.33:0.00934)0.823.10:0.00157,517305:0.00441)0.945.27:0.00384)0.960.51:0.00604)0.375.3:0.00204)0.994.78:0.0079,(261544:0.0141,142026:0.01473)0.916.26:0.00356)0.896.22:0.00339,257108:0.01011)1.000.702:0.0143,71972:0.02437)0.996.96:0.01011)0.985.43:0.00677,((253171:0.00869,415304:0.00991)0.934.28:0.00429,186382:0.01667)0.933.30:0.0035)0.743.17:0.00085,(((263567:0.01054,((((292270:0.02769,130585:0.00309)0.864.20:0.00153,177261:0.03022)0.938.31:0.00236,(((((138914:0.03922,(((535517:0.02851,152423:0.01687)1.000.703:0.00076,(367789:0.02273,((253448:0.00421,(542974:0.02028,226426:0.03271)0.944.20:0.00321)0.861.24:0.00092,106541:0.01773)1.000.704:0.00312)0.772.9:0.00555)0.866.14:0.00155,((((260809:0.02197,((273674:0.13912,265867:0.0789)0.988.71:0.02275,265177:0.13615)0.848.18:0.00472)0.841.12:0.0051,((275708:0.01904,(262708:0.01258,269930:0.02129)0.931.29:0.00414)0.950.43:0.00071,398017:0.01946)0.651.6:0.00015)0.788.9:0.00212,(272707:0.09193,270710:0.08328)0.975.44:0.01525)0.867.19:0.00229,266458:0.03679)0.969.40:0.00309)0.916.27:0.00105)0.776.13:0.00077,(141206:0.02813,(217410:0.01912,279948:0.02838)0.642.4:0.00289)0.905.23:0.0041)0.805.14:0.00103,(351681:0.02053,((242875:0.03874,(549837:0.02041,(318906:0.02321,548694:0.00805)1.000.705:0.02876)0.912.18:0.00852)0.762.12:0.01023,(((279910:0.00496,539941:0.03538)1.000.706:0.01639,((259419:0.01917,(110621:0.01317,(555883:0.02688,(((141127:0.00893,195370:0.04744)0.621.3:0.00398,(237173:0.03943,(25337:0.01135,140966:0.0152)0.876.21:0.00718)0.000.45:0.00605)0.095:0.00016,66816:0.05064)0.788.10:0.00014)0.857.15:0.00077)0.859.16:0.00077)0.978.41:0.00016,(140916:0.03371,(103728:0.01684,((((223417:0.03638,208451:0.03804)0.851.18:0.00619,140933:0.02804)0.311:0.00645,293243:0.03107)0.944.21:0.01696,251989:0.06226)0.378.7:0.0049)0.463.4:0.00171)0.787.7:0.00154)0.865.23:0.00071)0.998.130:0.00031,350105:0.02947)0.691.6:0.00237)0.903.24:0.00015)0.805.15:0.00078)0.900.25:0.00155,((((107493:0.03678,114379:0.01292)0.927.31:0.00078,(246575:0.01254,279231:0.01986)0.924.26:0.00261)1.000.707:0.00045,(((250894:0.01537,(363719:0.02972,(161891:0.00971,201266:0.0073)0.444.3:0.00084)0.000.46:0.00042)0.798.7:0.00078,((284298:0.0162,(((140863:0.01329,((153457:0.0104,(89393:0.01297,571314:0.00802)0.937.36:0.00754)0.760.16:0.00076,131189:0.01603)0.878.27:0.00311)0.773.12:0.00208,258724:0.01555)0.869.28:0.00277,((272189:0.01926,410048:0.01048)0.870.23:0.00345,(277610:0.02278,416614:0.01998)0.028:0.01078)0.856.16:0.00439)0.725.6:0.00385)0.683.6:0.00156,(587601:0.03007,157102:0.01624)0.000.47:0.00722)0.944.22:0.00169)0.817.13:0.00155,(534544:0.01449,(103025:0.02534,(132521:0.06577,((147561:0.02267,251821:0.02216)0.867.20:0.00805,(((541223:0.0396,341197:0.03868)0.634.4:0.00015,167590:0.03305)0.738.15:0.00167,(241731:0.07749,238696:0.02229)0.909.23:0.0076)0.918.27:0.00227)0.679.6:0.00097)0.742.12:0.00091)0.822.13:0.00904)0.955.41:0.00438)0.775.11:0.00016)0.873.19:0.00015,(((60329:0.01419,((513078:0.02404,184602:0.03616)0.882.23:0.01716,216433:0.01281)0.934.29:0.02257)0.928.27:0.00446,227761:0.01209)0.452.6:0.00014,229657:0.02183)0.000.48:0.00148)0.986.45:0.00389)0.269:0.00015,(315411:0.01919,193843:0.02258)0.761.13:0.00075)0.812.15:0.00324)0.982.53:0.00394,(580877:0.00806,105848:0.00653)0.917.26:0.00234)0.864.21:0.00184)0.868.25:0.00208,(173619:0.01341,173614:0.0171)0.992.72:0.00704)0.753.22:0.0007,304190:0.01528)0.982.54:0.00482)0.869.29:0.00163,244248:0.03388)0.748.12:0.00157)0.274.3:0.00016)0.588.2:0.00187)0.918.28:0.00702,(532645:0.28504,(553810:0.08688,(201216:0.30001,167658:0.08049)0.996.97:0.04484)0.318.3:0.00894)0.215.3:0.0036)0.968.31:0.0144)1.000.708:0.02953)0.937.37:0.00924)0.891.26:0.00792,((501947:0.0161,99377:0.00663)1.000.709:0.05007,(((333882:0.01877,61960:0.00074)0.892.21:0.01049,((158847:0.00815,7716:0.00091)0.920.17:0.00394,(278985:0.01438,221119:0.01147)0.397.3:0.00199)0.981.51:0.01013)0.991.59:0.01844,(350293:0.03414,((353468:0.00858,(239634:0.03251,(241977:0.00463,((109271:0.01702,219817:0.00772)0.999.185:0.01114,((((113736:0.03518,287139:0.0061)0.986.46:0.01503,97463:0.01352)0.897.20:0.00724,257905:0.00792)1.000.710:0.02043,((340144:0.00094,119582:0.01499)0.865.24:0.01178,142117:0.01883)0.637.5:0.0078)0.800.8:0.00247)0.976.46:0.00893)0.639.3:0.00404)0.976.47:0.00759)0.174.4:0.00269,242662:0.00558)0.988.72:0.00547)0.986.47:0.0233)1.000.711:0.03871)0.958.41:0.01315)0.988.73:0.0117,(((129040:0.00455,(8040:0.00411,8035:0.01834)0.997.108:0.03155)0.315:0.00756,(71299:0.02373,((110041:0.00624,(380204:0.01911,208702:0.01844)0.614.2:0.00305)0.999.186:0.01347,358429:0.01084)0.805.16:0.00374)0.999.187:0.01855)0.984.44:0.01357,(254304:0.00951,((534315:0.01416,353064:0.00711)0.188.3:0.00149,(155867:0.04489,560778:0.00396)0.934.30:0.00888)0.951.39:0.00558)1.000.712:0.03492)0.923.25:0.00632)0.505.4:0.00353,((((((145621:0.01386,(555130:0.02552,(563923:0.00295,((70607:0.0089,158067:0.01359)0.914.21:0.00645,(113578:0.00326,(514782:0.00712,((572391:0.00728,24022:0.01414)0.854.15:0.00235,85522:0.01277)0.915.25:0.00362)0.862.28:0.00015)0.927.32:0.00542)0.918.29:0.0025)0.960.52:0.0052)0.857.16:0.00448)1.000.713:0.04074,((161529:0.01024,146843:0.02343)1.000.714:0.05553,((7704:0.0139,(7705:0.01103,200611:0.02053)0.852.21:0.00262)1.000.715:0.03323,(((544122:0.01213,511184:0.0203)0.945.28:0.00456,((7698:0.00276,((((106505:0.01106,106173:0.00541)0.720.6:0.0108,252812:0.00757)0.893.17:0.00294,(187293:0.00014,44251:0.0088)0.999.188:0.01448)0.996.98:0.00816,564045:0.00431)0.702.6:0.00161)0.974.39:0.00733,543364:0.01514)0.985.44:0.00957)0.901.17:0.00951,((((584322:0.0082,254539:0.01359)0.915.26:0.00496,(139639:0.02267,152006:0.02427)0.911.28:0.00566)0.587.3:0.00278,((142595:0.00171,112990:0.01725)0.756.17:0.00077,286578:0.01741)0.946.34:0.00413)0.962.33:0.00767,(((555777:0.00065,143984:0.04123)0.999.189:0.024,(550017:0.0221,161678:0.00361)0.767.19:0.00752)0.962.34:0.00787,142087:0.0165)0.971.39:0.00995)0.973.45:0.01133)0.995.82:0.02148)0.980.43:0.02349)0.975.45:0.024)0.931.30:0.01861,((((((((223386:0.02248,516890:0.00551)0.572.2:0.00172,((((583529:0.02727,(525302:0.00681,263054:0.01979)0.957.55:0.00741)0.952.39:0.00539,((113202:0.01013,(((8059:0.01111,8081:0.02398)0.964.44:0.00615,(563741:0.01805,(8051:0.01889,(235630:0.02922,40125:0.0084)0.936.30:0.00362)0.933.31:0.00458)0.555.3:0.00216)0.989.58:0.00738,(40043:0.00173,((140993:0.01664,(205851:0.0369,141872:0.01125)0.579.2:0.00792)0.928.28:0.00476,((110683:0.02035,8098:0.00865)0.903.25:0.00403,(141423:0.00804,((((135917:0.01106,150338:0.01104)0.971.40:0.00835,(509929:0.01656,48494:0.0047)0.932.32:0.00451)0.850.20:0.0034,(((548160:0.02001,154661:0.01348)0.609.2:0.00958,(212454:0.00885,145704:0.00656)0.879.24:0.00746)0.426.5:0.00402,94840:0.00825)0.942.31:0.00598)0.706.5:0.00014,(131019:0.01866,534688:0.01033)0.823.11:0.00494)0.966.37:0.00502)0.982.55:0.00941)0.914.22:0.00945)0.884.28:0.00425)0.782.16:0.00075)0.541.2:0.00157)0.908.29:0.00274,(8105:0.01932,(8107:0.01981,8106:0.02098)0.652.4:0.00733)0.982.56:0.01275)0.616.3:0.00526)0.583.5:0.00181,((224998:0.01436,335355:0.0054)0.955.42:0.00457,(160617:0.01006,(141697:0.00607,142869:0.00588)1.000.716:0.01357)0.284.5:0.00147)0.823.12:0.00196)0.904.23:0.00345,((284577:0.02866,(146120:0.01486,((((8099:0.01633,282693:0.01352)0.794.12:0.00102,109242:0.01055)0.895.26:0.0078,(8100:0.00762,16413:0.0034)0.799.9:0.00078)0.930.32:0.0024,112881:0.01035)0.762.13:0.0009)0.564.3:0.00495)0.987.75:0.00897,(((142950:0.00813,143230:0.03217)0.771.16:0.00163,328695:0.00771)0.854.16:0.00677,113920:0.03765)0.975.46:0.00809)0.979.45:0.01011)0.764.12:0.00016)0.991.60:0.00915,(218525:0.02032,294790:0.01807)0.735.5:0.00096)0.827.9:0.00753,(((265527:0.00329,((245267:0.01353,(8054:0.00773,8057:0.00158)0.815.6:0.00142)0.934.31:0.00407,(588559:0.00443,544207:0.02366)0.991.61:0.00872)0.921.27:0.00566)0.446.4:0.00382,8067:0.01341)0.508.4:0.00332,(142784:0.00828,45263:0.01843)0.885.16:0.00339)0.980.44:0.00992)0.948.31:0.00981,286126:0.03036)0.698.7:0.00907,(79330:0.00568,(144477:0.01638,549573:0.0084)0.172:0.00397)1.000.717:0.05417)0.593.4:0.0031,574649:0.04627)0.911.29:0.01057,537326:0.02855)0.708.6:0.01078)0.679.7:0.01152,(((279383:0.0538,((548050:0.00747,(237458:0.01117,(274135:0.00082,550427:0.01654)0.783.11:0.01521)0.553.3:0.00662)1.000.718:0.02431,(532153:0.00407,592523:0.02449)1.000.719:0.03467)0.854.17:0.00596)0.452.7:0.00501,((79413:0.0092,247229:0.01356)1.000.720:0.02594,(114206:0.03709,551031:0.02423)0.999.190:0.02633)0.968.32:0.01221)0.990.57:0.01932,(((237939:0.03354,((239869:0.0058,(139679:0.01511,305646:0.00973)0.979.46:0.00725)0.964.45:0.00795,(248874:0.02405,89641:0.03521)0.954.32:0.00903)0.993.65:0.01643)0.841.13:0.00645,(594672:0.00955,(536681:0.00733,529452:0.00786)0.747.9:0.00329)0.994.79:0.01464)0.987.76:0.01469,(((252006:0.02158,((339012:0.01717,((458132:0.01044,(267469:0.00237,107157:0.00434)0.983.57:0.00082)0.733.10:0.00248,522744:0.01271)0.967.31:0.0068)0.760.17:0.00898,(539709:0.01059,(148717:0.0162,260040:0.01398)0.629.7:0.0057)0.998.131:0.02161)0.974.40:0.01297)0.941.43:0.01052,(((138619:0.01036,102284:0.01083)1.000.721:0.0262,(30988:0.01853,46905:0.02113)0.996.99:0.02221)0.116.6:0.0061,((32089:0.01139,43018:0.01717)1.000.722:0.02254,((586481:0.01189,((544166:0.04318,565016:0.02561)0.826.10:0.00472,(551797:0.00873,18746:0.00072)0.934.32:0.00498)0.996.100:0.01176)0.852.22:0.0045,((99902:0.00757,7655:0.01228)0.827.10:0.00275,((86244:0.00575,135216:0.0061)0.786.14:0.01615,(100064:0.02102,513774:0.0043)0.838.16:0.00597)0.990.58:0.01308)0.986.48:0.0103)1.000.723:0.02243)0.926.39:0.0084)1.000.724:0.03491)0.896.23:0.01542,(((((((7642:0.02279,((277633:0.01786,((278109:0.01189,514911:0.0187)0.864.22:0.01135,329725:0.00955)0.907.19:0.0033)0.948.32:0.00642,260232:0.01177)0.965.35:0.00689)0.902.16:0.00554,(((334454:0.01757,(141144:0.01982,558574:0.02044)0.984.45:0.00905)0.075.3:0.00186,(548778:0.01479,(267556:0.00492,114239:0.01469)0.870.24:0.00808)0.943.47:0.01228)0.929.30:0.0038,589725:0.01645)0.875.24:0.0044)0.956.33:0.00952,((280984:0.02088,278683:0.01915)0.968.33:0.01092,(536365:0.03784,528349:0.00921)0.929.31:0.00946)0.810.9:0.00667)0.978.42:0.0117,311740:0.02311)0.952.40:0.01451,555129:0.02404)0.684.6:0.00955,537600:0.04393)0.770.11:0.01001,(581185:0.02771,((559867:0.01368,(421635:0.01314,(380009:0.01607,(98216:0.00229,349801:0.0113)0.134:0.00176)0.916.28:0.01098)0.876.22:0.0044)0.236:0.00234,((((149662:0.0124,324870:0.01621)0.846.12:0.011,527446:0.02967)0.857.17:0.00836,170060:0.01893)0.747.10:0.00097,(280842:0.00569,533041:0.01531)0.961.34:0.00522)1.000.725:0.01055)1.000.726:0.02806)0.877.22:0.01523)0.550.6:0.00713)0.295.2:0.00891)0.923.26:0.01245)0.599.4:0.01219)0.969.41:0.01176,((((((((((539237:0.01231,530692:0.00402)1.000.727:0.05214,(203475:0.00422,344595:0.00108)0.998.132:0.02671)0.999.191:0.02686,(334207:0.01644,279663:0.0072)1.000.728:0.04183)0.915.27:0.00858,((578973:0.02437,55626:0.02942)1.000.729:0.06591,(136759:0.02556,22253:0.03325)0.968.34:0.01638)0.826.11:0.01021)0.969.42:0.00758,(((((524595:0.0107,234468:0.00655)0.895.27:0.00222,16391:0.01165)0.727.6:0.00086,(349901:0.0122,((73506:0.01395,242763:0.02392)0.949.34:0.00686,576671:0.02267)0.968.35:0.00635)0.855.17:0.00443)0.917.27:0.00508,254316:0.00878)1.000.730:0.0597,(((240701:0.01527,99390:0.01344)0.947.36:0.00759,(282458:0.02079,244768:0.08634)0.186.3:0.00431)0.990.59:0.01167,((((251664:0.01476,(555059:0.00685,536873:0.00855)0.954.33:0.00779)0.984.46:0.01036,((((((238980:0.02137,(534043:0.01601,321458:0.02048)0.091:0.00531)1.000.731:0.02081,(((278735:0.03131,(511763:0.02321,325840:0.02944)0.190:0.00498)0.716.3:0.00841,587687:0.02788)0.638.2:0.00499,((317285:0.03095,(((511371:0.02759,359663:0.02897)0.923.27:0.00688,574974:0.01976)0.701.3:0.00419,357896:0.02597)0.430.4:0.00209)0.322.2:0.0039,((((((419180:0.01004,((245537:0.02065,(368366:0.01177,256519:0.01098)0.576.3:0.0061)0.952.41:0.00677,(333779:0.01209,(543739:0.00016,102805:0.00014)0.749.7:0.0104)0.926.40:0.00639)0.268:0.00129)0.600.3:0.00355,(362533:0.01227,320125:0.02095)0.901.18:0.00616)0.924.27:0.00633,(135830:0.02621,(227017:0.01621,77169:0.01692)0.922.27:0.00469)0.995.83:0.01)0.752.9:0.00094,((132329:0.01778,(526380:0.02003,552844:0.02358)0.941.44:0.00909)0.715.8:0.00132,(529786:0.01766,529296:0.01452)0.558.3:0.0029)0.958.42:0.00734)0.844.14:0.00013,(36918:0.0287,(212714:0.01517,455438:0.01257)0.994.80:0.01236)0.789.15:0.00584)0.862.29:0.00415,542155:0.02326)0.943.48:0.00552)0.881.20:0.00418)0.963.44:0.00859)1.000.732:0.03568,(218114:0.0177,353731:0.01026)1.000.733:0.03897)0.946.35:0.00957,((75050:0.03883,(257122:0.03219,137191:0.01215)0.991.62:0.01315)0.832.10:0.00661,(248221:0.00929,510274:0.00124)0.991.63:0.01649)0.987.77:0.01195)0.955.43:0.00899,((255441:0.03091,515598:0.02201)0.918.30:0.00767,(71246:0.02662,(216743:0.01923,239870:0.03059)0.825.12:0.00692)0.941.45:0.00846)0.985.45:0.01352)0.904.24:0.00552,((251564:0.00344,323858:0.01725)1.000.734:0.03899,(561011:0.01738,((225991:0.01717,63324:0.02734)0.988.74:0.01475,(347934:0.01944,((207732:0.00619,575883:0.00521)1.000.735:0.03163,(47013:0.00789,(36221:0.00733,(235968:0.00902,514449:0.00534)0.997.109:0.01056)0.140:0.00162)0.537.3:0.00195)0.921.28:0.00543)0.806.9:0.00571)0.869.30:0.00313)0.308.2:0.00199)0.852.23:0.0069)0.939.35:0.00619)0.889.20:0.00565,((112161:0.0187,237253:0.01224)0.764.13:0.00328,(279470:0.00642,160953:0.01733)0.843.13:0.00361)0.972.44:0.01021)0.360.2:0.00573,137507:0.03471)0.929.32:0.0062)0.984.47:0.01122)0.891.27:0.00784)0.957.56:0.01108,((276006:0.02089,550690:0.02679)1.000.736:0.0267,((268607:0.0161,(267408:0.01681,(264331:0.01241,263538:0.01598)0.796.11:0.00337)0.898.25:0.01024)1.000.737:0.03832,(((305235:0.01597,(270320:0.01103,546723:0.03488)0.959.33:0.01214)0.966.38:0.0107,559606:0.01321)1.000.738:0.02265,(((105678:0.01257,(100883:0.00671,((105627:0.01403,312981:0.01248)0.146:0.01015,80375:0.01785)0.417.4:0.00246)0.804.9:0.00679)1.000.739:0.02814,(230249:0.01324,574241:0.01558)1.000.740:0.03058)0.431.6:0.00533,((52016:0.01846,292187:0.0287)0.917.28:0.00688,((((549798:0.00271,(231714:0.02219,233007:0.01734)0.970.31:0.00539)1.000.741:0.01068,233794:0.0077)0.311.2:0.00113,231687:0.01832)0.995.84:0.01223,(((((554366:0.01581,(354600:0.01748,158090:0.01062)1.000.742:0.05463)0.943.49:0.00955,(101534:0.00276,542985:0.00014)1.000.743:0.02094)0.961.35:0.00846,7977:0.03273)0.801.6:0.00654,((8012:0.02202,191570:0.01109)0.767.20:0.00352,((((236913:0.00252,590656:0.03193)0.906.23:0.0033,((7972:0.01525,((99452:0.01424,115022:0.01251)0.285:0.00287,((((315994:0.00589,592165:0.02114)0.999.192:0.01351,(7999:0.02231,7998:0.00558)0.955.44:0.00593)0.885.17:0.00341,(500166:0.00925,572561:0.00653)0.566.2:0.00197)0.899.20:0.00373,(((311942:0.01007,(123281:0.02666,157215:0.0148)0.840.9:0.00316)0.980.45:0.00782,((463932:0.02215,((((((21751:0.01012,7949:0.00446)0.168.3:0.00159,(163406:0.01252,(7938:0.00658,145455:0.00802)0.835.11:0.00499)0.877.23:0.00629)0.566.3:0.00268,518474:0.00525)0.943.50:0.0051,((244967:0.01109,(154202:0.01429,(7933:0.00962,((7914:0.00369,7920:0.0143)0.789.16:0.00085,125639:0.00202)0.829.12:0.00651)0.757.10:0.00388)0.786.15:0.00165)0.924.28:0.00393,(7908:0.01579,516584:0.00912)0.694.4:0.00433)0.137:0.00185)0.823.13:0.00551,522337:0.02785)0.810.10:0.00187,(203056:0.01275,(((((557327:0.00242,229854:0.03583)0.941.46:0.00338,579691:0.0053)0.420.3:0.00079,(((7958:0.01403,259537:0.01126)0.736.10:0.00089,7954:0.00969)0.923.28:0.00299,(7955:0.00739,168390:0.01132)0.037:0.00078)0.923.29:0.00661)0.847.16:0.00175,298170:0.01972)0.827.11:0.00179,173398:0.01483)0.788.11:0.00173)0.748.13:0.00134)0.408.8:0.00201)0.168.4:0.00217,(153948:0.00963,257199:0.03139)0.991.64:0.01095)0.774.8:0.00114)0.761.14:0.00146,(171418:0.01299,((226108:0.00627,(331322:0.00737,(306074:0.00652,243485:0.0141)0.917.29:0.00546)0.994.81:0.00978)0.949.35:0.00443,318862:0.01854)0.964.46:0.00804)0.801.7:0.0041)0.925.25:0.00377)0.845.17:0.00334)0.894.25:0.00403)0.449.2:0.00173,(7971:0.01359,(166988:0.0104,7996:0.01976)0.596.2:0.00525)0.114:0.00374)0.907.20:0.00424)0.893.18:0.0038,((543450:0.02312,563948:0.00531)0.496:0.00245,7973:0.04349)0.916.29:0.00407)0.084.4:0.00159,(((7982:0.01471,(7981:0.01184,310241:0.00836)0.986.49:0.01)0.904.25:0.00566,(48785:0.01452,((500261:0.01597,551369:0.01631)0.964.47:0.00659,(238279:0.02452,(556395:0.00326,8008:0.0068)0.741.10:0.00494)0.769.14:0.00277)0.903.26:0.00447)0.100.3:0.00163)0.961.36:0.00481,(7976:0.02084,((328548:0.03192,(140123:0.01281,(549676:0.012,555343:0.01891)0.948.33:0.00805)0.835.12:0.00326)0.820.6:0.00451,(64661:0.02472,590087:0.0231)0.972.45:0.00744)0.900.26:0.00364)0.892.22:0.00345)0.736.11:0.00082)0.887.22:0.00313)0.899.21:0.0061)0.613.3:0.00587,(221947:0.01985,7978:0.0129)0.850.21:0.00526)0.920.18:0.00798)0.479.7:0.00432)0.935.21:0.0063)1.000.744:0.02302)0.997.110:0.01364)0.886.19:0.01029)0.921.29:0.01079)0.948.34:0.01039,((((((8011:0.02315,327533:0.01698)1.000.745:0.04942,(45765:0.02391,((100054:0.01132,(193694:0.02568,(575450:0.0057,148490:0.01243)0.884.29:0.00334)0.829.13:0.0059)0.970.32:0.01141,(256878:0.01694,225862:0.01602)0.935.22:0.00767)0.117.2:0.00506)0.994.82:0.02359)0.939.36:0.01223,(332601:0.02377,(336069:0.01342,252003:0.01545)0.981.52:0.01424)1.000.746:0.03201)0.894.26:0.00688,(((261085:0.01491,511554:0.02064)0.220.3:0.00299,(212023:0.02218,257693:0.01607)0.652.5:0.00904)1.000.747:0.05858,314835:0.08979)0.147:0.00371)0.747.11:0.00609,((105206:0.09102,(581984:0.00766,561050:0.02634)1.000.748:0.0362)0.054:0.00964,((535071:0.01039,((304869:0.0291,250583:0.00104)0.796.12:0.00133,362382:0.02207)0.589.2:0.02304)1.000.749:0.04435,(((64552:0.03117,(173408:0.02255,317805:0.00538)1.000.750:0.01829)0.816.11:0.00493,98992:0.03074)0.899.22:0.00873,365486:0.02928)0.986.50:0.02243)1.000.751:0.04906)0.992.73:0.02156)0.748.14:0.00518,(241611:0.02048,(149607:0.01314,(269523:0.0202,(575131:0.02859,(568157:0.00238,558342:0.0162)0.978.43:0.0086)0.917.30:0.006)0.821.7:0.00505)0.880.20:0.00562)1.000.752:0.04332)0.676.3:0.01076)0.958.43:0.01049,((((((326446:0.02639,(246142:0.0407,559916:0.01664)0.923.30:0.00978)0.912.19:0.00784,(((156557:0.00511,319778:0.01024)0.999.193:0.01872,(272458:0.02346,(19290:0.02002,(314162:0.02945,(251151:0.0155,(46260:0.01488,(8016:0.00014,103547:0.00267)1.000.753:0.0242)0.230:0.00336)0.661.4:0.00311)0.684.7:0.00543)0.885.18:0.0067)0.240.3:0.00585)0.901.19:0.00606,(154082:0.01126,547615:0.01288)0.999.194:0.02182)0.916.30:0.00648)0.819.15:0.00812,((((229650:0.01708,354403:0.01776)1.000.754:0.03581,(74504:0.02388,313983:0.03792)0.993.66:0.0247)0.940.33:0.01066,(337388:0.02361,(326155:0.02519,587874:0.01766)1.000.755:0.03549)0.917.31:0.01074)0.915.28:0.00926,(8026:0.01798,535383:0.02106)0.610.2:0.00558)0.873.20:0.00603)0.968.36:0.00983,((541438:0.01886,251636:0.02841)0.997.111:0.01905,(215917:0.0339,76053:0.03709)0.508.5:0.00823)0.879.25:0.00633)0.712.3:0.00372,(265658:0.03167,555799:0.02152)0.934.33:0.00665)0.688:0.00417,(303457:0.02964,523629:0.02688)1.000.756:0.04499)0.949.36:0.0147)0.935.23:0.01211,(((562971:0.05862,81107:0.04911)0.428.2:0.01625,81307:0.03049)0.962.35:0.01628,((582514:0.01682,((7509:0.00793,307237:0.00498)1.000.757:0.02141,((137953:0.02852,((7511:0.0132,7510:0.01406)0.810.11:0.00356,(73639:0.01791,573955:0.0113)1.000.758:0.04045)0.764.14:0.00576)0.189:0.00252,(593175:0.01143,(60453:0.01985,60454:0.00928)0.984.48:0.00837)0.970.33:0.0075)0.959.34:0.00892)0.984.49:0.01348)0.989.59:0.02011,(((81842:0.04245,(7505:0.02707,81342:0.03916)0.113.3:0.00993)0.152.3:0.00562,(219763:0.02662,110311:0.01802)1.000.759:0.05119)0.954.34:0.0116,(241048:0.0521,255153:0.04019)0.854.18:0.01291)0.684.8:0.00598)0.541.3:0.01283)0.984.50:0.017)0.738.16:0.00816,(591247:0.01564,583897:0.01092)1.000.760:0.05032)0.940.34:0.0101)0.276:0.00412,(((((((225004:0.01571,((333792:0.01921,311786:0.00881)0.948.35:0.00631,(342189:0.01322,369412:0.0164)0.968.37:0.00684)0.998.133:0.0107)0.961.37:0.00631,((539642:0.01713,(279517:0.00872,(((551733:0.01183,(((255117:0.02028,7561:0.01197)0.969.43:0.00593,515895:0.01107)0.879.26:0.01134,360759:0.01108)0.923.31:0.005)0.558.4:0.00248,(171373:0.01633,562993:0.00814)0.832.11:0.00526)0.955.45:0.00524,140611:0.00903)0.455:0.00363)0.956.34:0.00683)0.996.101:0.01236,577330:0.01451)0.682.3:0.00203)0.897.21:0.00466,((164052:0.01013,((557546:0.00876,(356042:0.01236,((152002:0.00954,(7542:0.00912,(151600:0.00839,535870:0.01176)0.897.22:0.00403)0.803.9:0.00319)0.839.15:0.00546,((552276:0.01631,(((552073:0.00824,333009:0.03709)0.800.9:0.00846,81479:0.00826)0.875.25:0.00289,584388:0.00836)0.810.12:0.00225)0.785.11:0.00105,278017:0.01029)0.896.24:0.00248)0.938.32:0.00271)0.947.37:0.0035)0.864.23:0.00185,334435:0.01456)0.966.39:0.00504)0.009.4:0.00323,(271040:0.01651,250968:0.01347)0.998.134:0.01332)0.989.60:0.01272)0.919.23:0.01445,((199419:0.01938,253267:0.01868)1.000.761:0.02826,((159131:0.02629,251374:0.01427)0.977.48:0.01224,((342353:0.00324,(159377:0.01114,(63851:0.01196,325389:0.01474)0.907.21:0.00385)0.970.34:0.00736)0.987.78:0.00615,(312169:0.01096,334288:0.01261)0.871.31:0.00181)0.910.24:0.01419)0.688.2:0.01195)0.915.29:0.01527)0.339.3:0.00741,(((240274:0.01003,((113554:0.0123,(144237:0.00923,579424:0.00806)0.211:0.00077)0.778.14:0.00331,(148618:0.00906,363270:0.01324)0.895.28:0.00545)0.959.35:0.00657)0.988.75:0.01641,(534024:0.02205,(277460:0.02176,(544029:0.00444,544895:0.0226)0.675:0.00383)0.885.19:0.00683)0.999.195:0.02796)1.000.762:0.04756,((329297:0.02262,(356150:0.02528,274153:0.02491)0.999.196:0.02826)0.678.2:0.0108,327786:0.01753)0.958.44:0.01788)0.984.51:0.01943)0.999.197:0.01603,((((246166:0.01111,(126228:0.00946,229693:0.00747)0.746.9:0.00977)1.000.763:0.03198,((559694:0.03799,(((539933:0.01386,297961:0.00959)0.907.22:0.01351,534467:0.00388)0.994.83:0.0128,(272502:0.00226,(241556:0.0173,278243:0.02154)0.955.46:0.0111)0.716.4:0.00754)0.995.85:0.01903)0.951.40:0.01579,(162865:0.02408,(240090:0.01072,146270:0.02399)0.753.23:0.00537)0.998.135:0.02694)0.373.2:0.00789)0.783.12:0.00668,(92419:0.05236,(592807:0.0111,556529:0.01078)1.000.764:0.04363)0.983.58:0.01532)0.839.16:0.0032,(((591879:0.00172,119196:0.00171)0.211.2:0.01371,(17992:0.02694,(593092:0.01127,99301:0.01375)0.894.27:0.00477)0.982.57:0.01237)1.000.765:0.06354,(103264:0.01386,(342374:0.01425,(517810:0.00976,105115:0.01781)0.904.26:0.00588)0.966.40:0.00799)1.000.766:0.04156)0.984.52:0.01799)0.916.31:0.00722)0.912.20:0.00651,(((((((592416:0.04245,7587:0.02603)0.944.23:0.0129,(524284:0.02371,332804:0.02084)0.999.198:0.01814)0.833.12:0.00471,((((533132:0.00702,((580575:0.01357,((7643:0.01458,263960:0.0239)0.577.2:0.00015,555574:0.01443)0.888.12:0.00973)0.967.32:0.006,590955:0.0325)0.936.31:0.00553)0.906.24:0.01107,(255109:0.01779,511313:0.00616)0.838.17:0.00896)0.414.2:0.00733,(269700:0.02437,(274139:0.01904,278306:0.01275)0.897.23:0.00769)0.915.30:0.00733)0.721.5:0.00674,((72638:0.01516,173031:0.01648)0.684.9:0.00138,(72559:0.02085,(164335:0.01346,307033:0.01209)0.973.46:0.00687)0.964.48:0.00669)0.516.2:0.00393)0.953.25:0.0091)0.676.4:0.00465,(((((570082:0.03133,(341837:0.0126,275872:0.00797)0.995.86:0.01544)0.999.199:0.01888,(533025:0.01016,(511798:0.0183,((566137:0.01655,516644:0.00724)0.963.45:0.00596,(516020:0.02316,((((24638:0.02543,237937:0.00249)0.807.8:0.00632,(((261529:0.00328,271777:0.00623)0.893.19:0.00336,(276375:0.01696,519729:0.00872)0.972.46:0.0071)0.815.7:0.00466,(268263:0.02087,149398:0.00644)0.899.23:0.00265)0.853.18:0.00723)0.787.8:0.0008,(584938:0.00875,96329:0.01329)0.882.24:0.00158)0.848.19:0.00487,(152376:0.01836,264798:0.01188)0.719.3:0.00452)0.870.25:0.00193)0.837.15:0.00186)0.948.36:0.00564)0.872.34:0.00606)0.998.136:0.01564)0.923.32:0.00562,((((217304:0.01358,333006:0.00906)0.515.4:0.00315,(538780:0.02143,(533809:0.01084,(98768:0.01593,261470:0.02365)0.899.24:0.00506)0.152.4:0.00297)0.792.12:0.0018)0.954.35:0.00538,(261375:0.00879,213821:0.01317)0.988.76:0.0076)0.967.33:0.00554,(254195:0.03721,(538297:0.03597,(((172425:0.02775,110147:0.03024)0.997.112:0.02019,((143226:0.02653,((571289:0.00999,(165797:0.01161,296708:0.02033)0.964.49:0.00913)0.117.3:0.0029,588431:0.01575)0.168.5:0.00467)0.901.20:0.00433,(75565:0.01652,(238961:0.01786,539105:0.00439)0.942.32:0.00631)0.880.21:0.00362)0.971.41:0.00793)0.926.41:0.00618,((429948:0.00653,100583:0.01267)1.000.767:0.03298,(288621:0.00374,(79132:0.02179,288087:0.00832)0.681.8:0.0105)1.000.768:0.03443)0.913.28:0.01964)0.605.5:0.00683)0.117.4:0.00878)0.857.18:0.00607)0.770.12:0.0018)0.768.13:0.00209,(546751:0.02256,(277870:0.02135,(250082:0.02938,(551973:0.0135,558137:0.01709)0.950.44:0.00817)0.858.14:0.0083)0.769.15:0.00591)0.994.84:0.01461)0.915.31:0.0058,(574766:0.01867,((190822:0.03973,175965:0.01052)0.999.200:0.02977,(((108755:0.01198,182899:0.01949)0.916.32:0.00454,(556389:0.00339,(79212:0.03491,254811:0.01739)0.994.85:0.01504)0.408.9:0.00516)0.907.23:0.00548,(241908:0.01532,180233:0.00947)0.990.60:0.01275)0.751.6:0.00628)0.995.87:0.01601)0.999.201:0.0153)0.980.46:0.00797)0.938.33:0.00671,(((313357:0.03422,(305967:0.00763,256660:0.01115)0.998.137:0.01813)0.966.41:0.01145,((557161:0.0052,149753:0.00782)1.000.769:0.02351,((105207:0.01265,(237453:0.01493,80943:0.0059)0.950.45:0.00516)0.988.77:0.00761,272596:0.00901)0.742.13:0.0239)0.823.14:0.0086)0.480.4:0.00402,((88634:0.00408,345688:0.0016)0.759.10:0.03084,((261461:0.02108,337743:0.03226)0.841.14:0.01083,((554477:0.02403,533202:0.00771)1.000.770:0.02351,(587974:0.01174,(177523:0.01093,(255361:0.01893,256886:0.02487)0.465.2:0.00632)0.974.41:0.01189)0.967.34:0.01174)0.923.33:0.0059)0.794.13:0.00458)0.972.47:0.00773)0.810.13:0.00636)0.541.4:0.00398,((((96854:0.04335,((528941:0.02269,(333875:0.03291,196530:0.01736)0.812.16:0.00993)0.869.31:0.00466,(159605:0.01837,(103037:0.01819,437967:0.01529)0.778.15:0.0033)1.000.771:0.01973)0.639.4:0.00661)0.348.2:0.00343,((((((((244283:0.01827,((255719:0.00255,(251900:0.0124,337092:0.00179)0.980.47:0.00646)0.964.50:0.00715,(186744:0.00791,186404:0.01138)0.954.36:0.00963)0.921.30:0.00877)0.978.44:0.01538,(((202055:0.01586,((185856:0.00314,327923:0.01282)0.998.138:0.01246,(201799:0.02669,200531:0.00813)0.962.36:0.01046)0.962.37:0.01135)0.974.42:0.01206,(((526781:0.0034,310153:0.02539)0.996.102:0.01021,(539152:0.01024,(159877:0.01188,62495:0.01216)0.400.5:0.00163)0.772.10:0.00147)0.997.113:0.01189,(370367:0.02756,(7460:0.00758,142319:0.00661)0.999.202:0.01435)0.890.24:0.00747)0.956.35:0.00895)0.858.15:0.00818,(172489:0.00774,((198205:0.00095,7457:0.01529)0.926.42:0.00452,(62638:0.02284,7452:0.02706)0.326.6:0.00313)1.000.772:0.02963)0.963.46:0.00722)0.983.59:0.01414)0.144:0.00593,(40387:0.02033,280943:0.01727)1.000.773:0.01812)0.834.19:0.00416,(153768:0.02931,(574068:0.00765,93272:0.01876)0.736.12:0.02445)1.000.774:0.02037)0.832.12:0.00439,((((344620:0.02018,(339792:0.00412,69845:0.04322)0.592.7:0.01391)0.962.38:0.01553,((552048:0.00106,568686:0.02862)0.942.33:0.00885,(70993:0.00683,58992:0.01252)0.764.15:0.004)0.997.114:0.01528)0.985.46:0.01339,525227:0.0108)0.262.3:0.00338,((583819:0.01196,341286:0.00318)0.993.67:0.01178,(256033:0.00374,(89646:0.01894,((7473:0.0081,571311:0.00364)0.993.68:0.01086,32582:0.00995)0.969.44:0.00702)0.848.20:0.00574)0.949.37:0.01076)0.978.45:0.00861)0.982.58:0.01283)0.186.4:0.00371,(349780:0.01358,201695:0.0196)1.000.775:0.032)0.954.37:0.00897,((((243441:0.02683,(207038:0.01661,158980:0.01309)1.000.776:0.03563)1.000.777:0.02769,530809:0.03882)0.549.3:0.00644,((543963:0.03742,242186:0.05658)0.655.2:0.00693,(140562:0.00505,51207:0.01767)1.000.778:0.0338)0.909.24:0.00647)0.936.32:0.00712,((((158638:0.00413,((324503:0.01418,234358:0.00542)0.048:0.01204,78477:0.00727)0.956.36:0.00687)0.999.203:0.01918,(336892:0.0083,559243:0.02203)1.000.779:0.02469)0.992.74:0.01479,(355416:0.024,239144:0.04256)0.967.35:0.01266)0.971.42:0.01215,((353756:0.02906,(((156986:0.01623,155536:0.00917)0.995.88:0.01538,(((587540:0.00636,254374:0.02224)0.852.24:0.00162,549530:0.02243)0.917.32:0.01825,((((((584489:0.03116,521786:0.0061)0.811.9:0.0039,248771:0.01568)0.023.2:0.00015,(365415:0.00028,313985:0.0467)0.800.10:0.00747)0.927.33:0.00291,(107402:0.00863,(463539:0.01305,551472:0.02806)0.863.25:0.00404)0.741.11:0.00076)0.998.139:0.01014,556730:0.01793)0.822.14:0.00192,(75633:0.03914,222645:0.01552)0.926.43:0.00775)0.880.22:0.00345)0.987.79:0.01169)0.971.43:0.00883,((277765:0.03518,255261:0.01608)1.000.780:0.024,((((((458397:0.01286,(589636:0.00792,332476:0.01434)0.155.3:0.00206)0.461.3:0.00523,537797:0.01044)0.569.2:0.00422,(232138:0.02247,154435:0.01096)0.200.4:0.00465)0.999.204:0.02037,((332574:0.02102,((593006:0.02586,226353:0.01272)0.991.65:0.01133,533140:0.00875)0.781.14:0.00705)0.815.8:0.00715,(532025:0.02118,(((((228030:0.05288,225685:0.00709)0.055.2:0.00107,357680:0.00611)1.000.781:0.01498,(544749:0.01983,156500:0.0155)0.301.5:0.00305)0.553.4:0.00186,((509622:0.01979,20938:0.01589)0.870.26:0.00269,(((366851:0.00161,235634:0.01539)0.944.24:0.0095,27669:0.01082)0.861.25:0.00151,100307:0.00765)0.964.51:0.0044)0.856.17:0.00171)0.948.37:0.00431,((((((208768:0.03476,(153003:0.00535,204786:0.00459)0.247.2:0.00479)0.999.205:0.02023,563462:0.01627)0.780.9:0.00145,150982:0.02475)0.875.26:0.00277,572643:0.00392)0.994.86:0.00973,364617:0.01893)0.812.17:0.00214,((560106:0.00504,310791:0.02437)0.501.5:0.00306,247980:0.02181)0.874.22:0.00338)0.885.20:0.00253)0.985.47:0.01069)0.348.3:0.00564)0.799.10:0.00723)0.999.206:0.02025,((564301:0.00652,(47916:0.00365,(277971:0.03941,274454:0.01125)0.754.12:0.00366)0.869.32:0.00637)0.999.207:0.02319,522839:0.01423)0.920.19:0.00792)0.930.33:0.00772,254180:0.02044)0.986.51:0.01333)0.940.35:0.00745)0.999.208:0.02059)0.896.25:0.00632,((93707:0.0142,168075:0.00881)1.000.782:0.01629,((((258787:0.01451,264970:0.01008)0.981.53:0.01136,(((239219:0.02236,(194499:0.02125,(278010:0.00988,(593502:0.0108,(331736:0.00773,(576461:0.00875,((154568:0.02941,526439:0.00554)0.260.3:0.00319,(269377:0.0159,(((90324:0.01713,((((((589587:0.00879,576093:0.00468)0.882.25:0.01814,7632:0.01026)0.884.30:0.00159,(98135:0.01893,569357:0.0012)0.786.16:0.01117)0.965.36:0.0044,(261285:0.0066,(((341821:0.0058,166442:0.01173)0.906.25:0.00244,((((561739:0.00988,252341:0.0153)0.838.18:0.00337,(154352:0.00934,(515916:0.0163,((581953:0.02609,553770:0.00105)0.401.4:0.01659,218656:0.00909)0.138.3:0.00122)0.898.26:0.00403)0.988.78:0.0087)0.927.34:0.00317,(202816:0.00952,(257849:0.01361,553394:0.01277)0.998.140:0.01437)0.899.25:0.00451)0.729.8:0.00081,511326:0.01193)0.945.29:0.00326)0.993.69:0.00692,((142048:0.01125,(155768:0.01622,(347251:0.0114,138690:0.00935)1.000.783:0.01858)0.994.87:0.01214)0.846.13:0.0027,201369:0.01288)0.702.7:0.00412)0.418.4:0.00078)0.862.30:0.00163)0.642.5:0.01048,((7634:0.00981,174482:0.01502)0.186.5:0.00092,269745:0.01645)0.723.8:0.00644)0.988.79:0.00719,252965:0.00782)0.190.2:0.00078)0.759.11:0.00069,(151987:0.01557,585866:0.01381)0.881.21:0.00236)0.865.25:0.00153,509976:0.01004)0.888.13:0.00331)0.922.28:0.005)0.977.49:0.00554)0.855.18:0.0017)0.282.3:0.01266)0.932.33:0.00723)0.511.5:0.00733)0.835.13:0.00521)0.950.46:0.00954,311399:0.0126)0.982.59:0.01408,(((336977:0.01843,(160646:0.01575,356249:0.01853)0.811.10:0.00607)0.463.5:0.00481,188380:0.01736)1.000.784:0.01527,(((361994:0.00756,338034:0.01432)0.984.53:0.00857,(566615:0.02117,(276698:0.00733,(((558595:0.02562,148963:0.00964)0.878.28:0.00229,(327055:0.01049,311073:0.00642)0.970.35:0.01154)0.736.13:0.00101,360752:0.01259)0.717.8:0.00087)0.994.88:0.01171)0.484.4:0.00433)0.372.4:0.003,(311213:0.03548,(((221067:0.00658,276755:0.00483)0.384.3:0.00345,(146081:0.00813,312465:0.03298)0.992.75:0.01266)0.791.12:0.00634,(169426:0.00802,554137:0.01464)0.832.13:0.00529)0.948.38:0.00891)0.263.5:0.00483)0.962.39:0.00784)0.327.3:0.00226)0.859.17:0.00746)0.765.13:0.00513,(301947:0.01554,264901:0.02716)0.942.34:0.00627)0.899.26:0.00463,((((181525:0.01952,563256:0.01449)0.825.13:0.00505,(317142:0.00984,(239488:0.01153,(352767:0.00565,540865:0.00399)0.922.29:0.02081)0.847.17:0.02352)0.975.47:0.0101)0.901.21:0.0046,((565462:0.00397,263799:0.0096)0.740.15:0.00285,(538992:0.01273,577112:0.01666)0.969.45:0.0065)0.983.60:0.01205)0.571.4:0.01102,195783:0.03124)0.782.17:0.00482)0.504.3:0.00401)0.925.26:0.0057)0.958.45:0.00985)0.851.19:0.00398)0.914.23:0.00665)0.806.10:0.00257,(257850:0.0487,(575230:0.03201,(89553:0.0252,539039:0.01439)0.992.76:0.01735)0.932.34:0.01449)0.993.70:0.01582)0.961.38:0.00686)0.816.12:0.00205,((265444:0.00391,292288:0.03559)1.000.785:0.02561,527507:0.01591)0.531.4:0.00554)0.879.27:0.00328,(((202468:0.03313,(251981:0.00579,278708:0.012)0.999.209:0.01614)0.871.32:0.00784,559269:0.0219)0.767.21:0.00439,(((7628:0.03511,(221888:0.00653,520399:0.00085)1.000.786:0.03553)0.987.80:0.01561,((157596:0.01635,168486:0.01635)0.999.210:0.02012,(245592:0.02807,260942:0.02634)0.999.211:0.02744)0.261.4:0.00498)0.952.42:0.00908,(186089:0.02514,(((564511:0.02472,171955:0.02255)0.885.21:0.00425,(((255342:0.01545,431838:0.02273)0.980.48:0.0151,(166871:0.04581,278349:0.0135)0.999.212:0.03436)0.209.3:0.00656,(550474:0.04172,(574531:0.02441,589171:0.00679)0.311.3:0.00623)0.450.4:0.00582)0.883.22:0.00748)0.739.17:0.00108,(((589714:0.02361,590516:0.01971)0.836.16:0.01024,571233:0.01808)0.927.35:0.00482,(95183:0.02992,272582:0.02155)0.834.20:0.00439)0.958.46:0.00628)0.980.49:0.00774)0.556.3:0.004)0.693.4:0.00141)0.930.34:0.00457)0.980.50:0.00822)0.840.10:0.00222,((362891:0.02347,(((241066:0.01052,(555330:0.00797,152647:0.01919)0.779.6:0.00464)0.999.213:0.02075,((((7577:0.01523,(148469:0.01476,(152604:0.03107,(113155:0.01589,(160599:0.01275,331064:0.02095)0.997.115:0.01246)0.592.8:0.00369)0.538.2:0.00481)0.997.116:0.01257)0.881.22:0.0031,(7575:0.01259,(241188:0.0118,(160275:0.01785,7586:0.01005)0.992.77:0.01266)0.944.25:0.00915)0.953.26:0.00771)0.966.42:0.00653,((201383:0.0313,((352911:0.00406,530747:0.0047)0.693.5:0.00015,32280:0.00561)0.974.43:0.00676)0.724.9:0.00498,7606:0.02017)0.944.26:0.00458)0.914.24:0.00297,(((7598:0.01529,(545897:0.02193,(7582:0.01766,169677:0.00662)0.853.19:0.00356)0.907.24:0.00454)0.992.78:0.00985,(235979:0.03634,(7619:0.02412,7624:0.00914)0.953.27:0.00687)0.810.14:0.00212)0.722.6:0.00182,(7622:0.00305,7641:0.00563)0.788.12:0.01723)0.925.27:0.00448)0.507.4:0.00385)0.994.89:0.01215,((199655:0.00104,131035:0.00275)0.995.89:0.01157,(565944:0.0022,104039:0.01458)0.916.33:0.00734)0.988.80:0.01101)0.877.24:0.0065)0.913.29:0.00906,((((519028:0.01711,(203160:0.02666,348014:0.00628)0.762.14:0.00381)0.999.214:0.01851,(((258311:0.03025,96909:0.03075)0.822.15:0.00765,((565421:0.00991,(202305:0.00647,266350:0.00158)0.767.22:0.01614)0.982.60:0.0092,((141619:0.01637,66569:0.00648)0.939.37:0.00889,(244634:0.01627,(106089:0.0198,(148733:0.00984,(159598:0.02642,((((243086:0.0061,511219:0.01444)0.917.33:0.00562,(541400:0.02032,367915:0.00666)0.953.28:0.01241)0.995.90:0.01901,(76061:0.02684,242102:0.00908)0.667.4:0.0049)0.823.15:0.00886,(220661:0.01837,553048:0.02513)0.862.31:0.00365)0.985.48:0.0139)0.836.17:0.00257)0.943.51:0.00515)0.997.117:0.01575)0.919.24:0.00974)0.997.118:0.01907)0.752.10:0.00716)0.994.90:0.01391,((589528:0.02978,((((114044:0.00879,(153466:0.02343,346479:0.0154)0.053:0.00412)1.000.787:0.03474,((332876:0.00664,253074:0.03797)0.626.3:0.00609,(585574:0.02633,510626:0.02263)0.857.19:0.00773)0.999.215:0.02096)0.451.2:0.00927,(592099:0.0295,(365755:0.00852,553141:0.0101)0.995.91:0.01859)0.938.34:0.01337)0.463.6:0.00997,340951:0.0146)0.752.11:0.00369)0.956.37:0.0088,((((304458:0.01263,585154:0.00913)0.769.16:0.00362,(258434:0.01401,(303531:0.00916,592478:0.01579)0.688.3:0.00399)0.218.4:0.00375)1.000.788:0.03399,(589101:0.02365,(313480:0.02395,261838:0.01339)0.951.41:0.00825)0.952.43:0.00839)0.120:0.00516,((587111:0.01576,(247661:0.00739,(118071:0.00393,573262:0.00057)0.907.25:0.01791)0.842.14:0.01409)1.000.789:0.01884,586541:0.01182)0.998.141:0.01698)0.958.47:0.01133)0.945.30:0.00886)0.962.40:0.00814)0.893.20:0.00515,(7650:0.04208,(509542:0.0325,((354483:0.00339,7649:0.00327)0.992.79:0.01551,(572591:0.01169,518194:0.00966)1.000.790:0.02869)0.522.4:0.00788)0.907.26:0.00794)0.954.38:0.00995)0.841.15:0.00543,((((316871:0.02932,(159272:0.01739,314376:0.01242)0.683.7:0.0043)0.999.216:0.01885,(((((315780:0.01965,(568930:0.00238,593544:0.00157)0.955.47:0.01333)0.937.38:0.00578,(((7553:0.0164,((211956:0.01294,241921:0.00402)0.935.24:0.00566,108086:0.02126)0.844.15:0.00328)0.795.11:0.00533,(546246:0.01281,(43810:0.00406,142800:0.01351)0.878.29:0.00562)0.827.12:0.00159)0.983.61:0.00772,107140:0.00511)0.939.38:0.00663)0.862.32:0.00398,(193004:0.00659,347412:0.02043)0.832.14:0.00403)0.906.26:0.00656,(((70247:0.02088,(7539:0.00901,7570:0.00287)0.757.11:0.00597)0.943.52:0.00651,((7470:0.01839,71426:0.00091)1.000.791:0.02238,71248:0.00561)0.888.14:0.01256)0.974.44:0.00747,(((528855:0.00758,292111:0.01901)0.405.4:0.00325,(((513221:0.0089,(512465:0.01317,303623:0.01444)0.740.16:0.00198)0.852.25:0.00163,(((((7524:0.03914,(((555855:0.01439,(559119:0.00117,7521:0.00502)0.951.42:0.01362)0.927.36:0.00609,(((7518:0.02107,7517:0.00496)0.327.4:0.01312,(219789:0.01607,589849:0.00943)0.578:0.00305)0.737.5:0.00211,(7519:0.00564,139719:0.00044)0.945.31:0.00484)0.952.44:0.00742)0.902.17:0.00563,((136053:0.00524,132687:0.00421)0.795.12:0.00862,(20680:0.03405,((25151:0.01262,7555:0.00837)0.887.23:0.00145,(513982:0.01617,237043:0.00832)0.745.14:0.00087)0.763.18:0.0029)0.895.29:0.0086)1.000.792:0.02081)0.945.32:0.00838)0.898.27:0.00517,((197820:0.01121,(562591:0.01012,(353702:0.00886,245986:0.01179)0.953.29:0.00451)0.950.47:0.00589)0.957.57:0.00679,329527:0.01044)0.683.8:0.00571)0.953.30:0.00495,(67654:0.02564,((160903:0.00639,(114439:0.00866,(575596:0.0108,310885:0.00602)0.992.80:0.01105)0.996.103:0.01307)0.853.20:0.0026,7532:0.01871)0.992.81:0.00921)0.904.27:0.0042)0.824.9:0.00435,150019:0.03764)0.618.4:0.00313,((140020:0.00029,7563:0.03556)0.875.27:0.01468,(7559:0.0166,(245668:0.00014,565851:0.00079)1.000.793:0.00872)0.878.30:0.0069)0.097.4:0.00425)0.947.38:0.00014)0.915.32:0.00329,(329501:0.01988,((((78036:0.0003,159208:0.02203)0.872.35:0.00616,349616:0.01244)0.267.2:0.00273,274061:0.01029)0.948.39:0.00521,7556:0.00932)0.948.40:0.00485)0.908.30:0.00454)0.869.33:0.0026)0.483.5:0.00396,312010:0.0221)0.927.37:0.00493)0.733.11:0.00148)0.930.35:0.00739,(168507:0.03348,568003:0.02733)0.989.61:0.01386)0.285.2:0.00566)0.916.34:0.00601,(91521:0.01652,((((7596:0.03145,7595:0.00891)0.969.46:0.00829,(7590:0.01947,(7594:0.01577,137055:0.01419)0.986.52:0.01186)0.933.32:0.00879)0.997.119:0.01506,(133344:0.02017,(7607:0.01616,7610:0.01245)0.259.3:0.00442)0.750.13:0.00309)0.161.2:0.00327,7613:0.01327)1.000.794:0.02192)0.974.45:0.00925)0.992.82:0.01088,((540861:0.02974,(155515:0.01545,(321477:0.00632,553528:0.00325)1.000.795:0.03714)0.905.24:0.00543)0.899.27:0.00729,(568472:0.02419,313486:0.02085)0.531.5:0.0099)0.999.217:0.0193)0.456.4:0.00014)0.196.3:0.00241)0.911.30:0.00491)0.896.26:0.00469)0.933.33:0.00495)0.988.81:0.01198)0.929.33:0.00965)0.811.11:0.01019,(((140186:0.03802,((244546:0.01886,527669:0.00627)0.992.83:0.0151,((528872:0.00203,(238940:0.02561,412992:0.01079)0.993.71:0.01281)0.938.35:0.00487,(64239:0.01186,(235478:0.00195,267648:0.00543)0.965.37:0.00951)0.935.25:0.00551)0.993.72:0.01238)0.869.34:0.00817)1.000.796:0.03848,((204859:0.00015,543665:0.00792)1.000.797:0.04846,(((590947:0.00741,(554795:0.00947,578673:0.01073)0.376.4:0.00356)0.997.120:0.01578,(322942:0.01488,(560016:0.02576,343417:0.01355)0.070:0.00449)0.880.23:0.00603)0.998.142:0.02248,(((299708:0.00918,(56552:0.01375,589703:0.01517)0.964.52:0.00757)0.446.5:0.00462,((213189:0.03537,(573373:0.01431,(58382:0.01672,233882:0.01053)0.968.38:0.01092)0.909.25:0.00461)0.703.9:0.00238,350596:0.01707)0.899.28:0.00675)0.686.2:0.00054,((155504:0.02101,539199:0.0268)0.896.27:0.01008,(363787:0.00231,57791:0.02976)0.791.13:0.02551)0.733.12:0.00139)0.724.10:0.00442)1.000.798:0.03461)0.924.29:0.01203)0.871.33:0.005,(((220181:0.01724,324704:0.00762)1.000.799:0.02703,((253642:0.00977,565631:0.03021)1.000.800:0.04187,((((240468:0.0349,(141825:0.01871,575267:0.01927)0.258.2:0.00834)0.996.104:0.01941,(278173:0.04165,((269231:0.01564,(344721:0.01416,552912:0.01621)0.849.12:0.00614)0.741.12:0.00469,((((192482:0.00066,175477:0.01366)0.898.28:0.00412,245026:0.01733)0.980.51:0.00975,(500169:0.00967,78370:0.02938)0.392.3:0.00784)0.998.143:0.01849,(37497:0.03317,499859:0.00907)0.336.6:0.0034)0.840.11:0.00359)0.983.62:0.01395)0.917.34:0.01273)0.420.4:0.00735,(158622:0.02002,((516554:0.00896,(182827:0.00532,7772:0.01981)0.792.13:0.00296)1.000.801:0.02014,(256598:0.01012,94102:0.0061)0.972.48:0.0111)0.808.9:0.00531)0.911.31:0.01194)0.801.8:0.00987,252728:0.03139)0.523:0.00827)0.596.3:0.01395)0.999.218:0.0256,(((50142:0.01577,238503:0.0092)1.000.802:0.05682,(320911:0.00488,(204506:0.00254,351052:0.01396)0.748.15:0.00081)0.756.18:0.01258)0.995.92:0.02425,((550784:0.04134,237966:0.01679)1.000.803:0.02942,((562488:0.02824,548785:0.0095)1.000.804:0.02719,((7792:0.02435,((101731:0.00425,566599:0.02827)0.986.53:0.01073,((554502:0.01883,264158:0.00974)0.985.49:0.00917,(534474:0.01174,((279341:0.02346,571774:0.01313)0.004.2:0.00168,(137895:0.00195,(300037:0.01393,(223206:0.02258,81357:0.00389)0.963.47:0.00545)0.978.46:0.00643)0.839.17:0.0039)0.982.61:0.00697)0.994.91:0.01159)0.966.43:0.01117)0.456.5:0.00254)0.890.25:0.00326,(((105121:0.01673,((562146:0.01779,(325248:0.00478,509760:0.01359)0.752.12:0.00721)0.988.82:0.0127,578119:0.02299)0.875.28:0.00645)0.917.35:0.00731,((149965:0.02323,(199667:0.01784,(353183:0.00531,75877:0.00838)0.868.26:0.00518)0.992.84:0.01195)0.854.19:0.00622,(((((((562931:0.00714,((((76525:0.00391,((77152:0.0009,79476:0.01447)0.995.93:0.02139,572324:0.0131)0.883.23:0.00157)0.910.25:0.00342,74091:0.00733)0.890.26:0.00509,122536:0.00728)0.905.25:0.00261,(245102:0.00153,554825:0.01596)0.919.25:0.00394)0.941.47:0.00353)0.184.3:0.00076,207813:0.01986)0.944.27:0.00346,358999:0.00737)0.926.44:0.00282,(((592861:0.01614,(321865:0.01285,88350:0.00924)0.865.26:0.00154)0.909.26:0.00236,(((241713:0.01454,(((457990:0.01992,253894:0.00291)0.957.58:0.00864,(((158677:0.00594,((361923:0.01026,(100144:0.00362,(114501:0.00699,587723:0.00393)0.718.8:0.00829)0.988.83:0.00815)0.889.21:0.0017,97259:0.00847)0.283:0.00234)0.899.29:0.0029,(((140061:0.00094,(383798:0.03994,((138668:0.00241,159749:0.00167)0.999.219:0.01178,(278947:0.00977,(140927:0.00837,585064:0.00601)0.953.31:0.00407)0.910.26:0.0036)0.594.6:0.00211)0.496.2:0.00387)1.000.805:0.01331,351629:0.00398)0.909.27:0.00266,(326693:0.00379,((((535329:0.00393,351746:0.02123)0.942.35:0.0063,(254119:0.00921,101503:0.01063)1.000.806:0.02263)0.864.24:0.00217,(333628:0.02493,363149:0.00317)0.929.34:0.01279)0.853.21:0.00247,506057:0.0036)0.953.32:0.00469)0.854.20:0.00229)0.947.39:0.00334)0.914.25:0.00391,101000:0.02703)0.983.63:0.00885)0.690.4:0.00545,7884:0.01471)0.349.2:0.00105)0.978.47:0.00602,((((208471:0.02796,(531786:0.00392,(149430:0.01351,48518:0.02297)0.765.14:0.00621)0.890.27:0.00307)0.733.13:0.00108,(((109333:0.00666,93199:0.03587)0.995.94:0.00893,(261369:0.05449,41720:0.00027)1.000.807:0.00931)0.963.48:0.00528,((246060:0.00959,((230970:0.00265,191835:0.00028)0.735.6:0.01132,(138884:0.00747,((180940:0.00267,181365:0.03025)0.922.30:0.00296,(99791:0.01112,276619:0.00835)0.914.26:0.00362)0.917.36:0.00305)0.746.10:0.00278)0.795.13:0.0015)0.918.31:0.00413,562124:0.02127)0.708.7:0.00087)0.972.49:0.00425)0.882.26:0.00325,(153182:0.01279,338929:0.01108)0.972.50:0.00951)0.839.18:0.00317,(((516809:0.03399,((((538384:0.05526,342503:0.13514)0.859.18:0.00707,((202238:0.00437,(200997:0.03806,200538:0.05316)1.000.808:0.05131)0.996.105:0.04429,194964:0.04243)0.757.12:0.00195)0.952.45:0.00304,219968:0.00625)0.888.15:0.0056,(129907:0.0319,521222:0.00306)0.143.3:0.00173)0.813.13:0.00235)0.889.22:0.00234,((((342928:0.0169,(221596:0.0061,170684:0.01565)0.891.28:0.0021)0.762.15:0.00343,(7813:0.00306,((261768:0.05317,259059:0.06953)0.928.29:0.01394,327532:0.01117)0.864.25:0.00354)0.964.53:0.00328)0.846.14:0.00676,569939:0.00686)0.999.220:0.00955,(365521:0.02216,138166:0.00708)0.959.36:0.0049)0.945.33:0.00513)0.904.28:0.01089,((391286:0.04148,160115:0.00257)0.462.3:0.00292,(43798:0.04475,438487:0.03047)0.499.3:0.00072)0.436.4:0.00268)0.965.38:0.00716)0.973.47:0.00616)0.956.38:0.00575,((((510060:0.01661,(102209:0.01167,(146003:0.05601,(150537:0.03183,107507:0.00014)0.833.13:0.00077)0.900.27:0.01365)0.188.4:0.00262)0.956.39:0.00823,((404492:0.0046,516410:0.0146)0.982.62:0.00609,(170471:0.00168,363793:0.00297)0.956.40:0.00295)0.999.221:0.00015)0.219.2:0.00552,(((155104:0.00342,275388:0.02265)0.469.3:0.00156,168681:0.00643)0.942.36:0.00015,254609:0.00793)0.953.33:0.00471)0.895.30:0.00272,(266094:0.00687,157598:0.00736)0.767.23:0.00656)0.330.2:0.00647)0.978.48:0.0076)0.860.28:0.00213,(142971:0.01882,((252239:0.00467,(548799:0.00406,(331342:0.02309,367733:0.00159)0.901.22:0.00307)0.854.21:0.0072)0.976.48:0.00723,537205:0.04476)0.817.14:0.0031)0.860.29:0.0028)0.840.12:0.003)0.998.144:0.01127,297307:0.01747)0.896.28:0.00332,257386:0.01334)0.731.4:0.00125,(((((((272797:0.00084,593908:0.01523)0.993.73:0.00771,236708:0.00798)0.950.48:0.0056,(113915:0.00541,202402:0.00809)0.895.31:0.00156)0.406:0.00015,(579953:0.00476,(143429:0.00081,156229:0.00877)0.948.41:0.00086)0.950.49:0.00871)0.950.50:0.0046,254845:0.01152)0.757.13:0.00663,((((101374:0.01303,587543:0.01459)0.896.29:0.00495,315730:0.00285)0.989.62:0.00679,370343:0.00591)0.950.51:0.00397,(((139220:0.01573,(547724:0.00799,107275:0.01001)0.811.12:0.0047)0.921.31:0.00356,((172808:0.00789,271896:0.00445)0.976.49:0.00476,366690:0.00856)0.755.16:0.00078)0.646.5:0.00146,(340120:0.01933,(539242:0.02315,559226:0.0078)0.051.3:0.003)0.879.28:0.00249)0.871.34:0.00348)0.987.81:0.00807)0.857.20:0.00574,((74443:0.00561,(218479:0.00143,136003:0.01402)0.948.42:0.00518)0.710.7:0.00346,(203659:0.00972,65469:0.02782)0.173.3:0.00396)0.999.222:0.01568)0.566.4:0.0063)0.997.121:0.01438)0.668.2:0.00453)0.902.18:0.00516,(((249305:0.01057,279266:0.02127)1.000.809:0.0227,(((362681:0.00828,102773:0.00707)0.997.122:0.01222,(121642:0.02558,(579635:0.01571,(126675:0.00213,369488:0.01151)0.937.39:0.012)0.259.4:0.00453)0.968.39:0.00822)0.997.123:0.01231,((245307:0.01009,((309216:0.01288,(537510:0.01025,323292:0.00753)0.992.85:0.01216)0.410:0.00418,585780:0.00818)0.975.48:0.00891)0.999.223:0.01626,(226164:0.032,(163937:0.01276,((((364648:0.00905,(278325:0.00851,(253405:0.00595,309065:0.01414)0.548.4:0.00511)0.942.37:0.00789)0.607.5:0.00157,(((((354823:0.01017,(67341:0.00157,((68655:0.02018,155699:0.01894)0.605.6:0.02234,236298:0.00397)0.961.39:0.00314)0.859.19:0.00015)0.784.11:0.0016,100907:0.03459)0.651.7:0.00142,106058:0.01837)0.915.33:0.00176,350831:0.01441)0.915.34:0.00306,((82238:0.00805,(133317:0.02275,81196:0.00645)0.969.47:0.00645)0.964.54:0.00482,(81676:0.00767,79105:0.00705)0.922.31:0.01473)0.950.52:0.00412)0.990.61:0.00977)0.937.40:0.00588,(177737:0.01874,370199:0.01322)1.000.810:0.02171)0.952.46:0.00735,(156994:0.04336,(71415:0.00823,(533999:0.01619,((329236:0.01343,(120471:0.01038,114384:0.00465)0.988.84:0.01547)0.878.31:0.00155,136764:0.01086)0.957.59:0.00536)0.891.29:0.00351)0.512.4:0.00259)0.881.23:0.00289)0.945.34:0.00575)0.999.224:0.01326)0.980.52:0.0097)0.944.28:0.00744)0.766.15:0.00335)0.801.9:0.0067,(204511:0.02074,(((160366:0.0121,209670:0.01368)0.637.6:0.0041,(244451:0.01728,((150574:0.00014,150541:0.00232)0.955.48:0.00673,(203227:0.00725,82032:0.00164)0.994.92:0.00932)0.386.2:0.00391)0.397.4:0.00496)0.910.27:0.0062,(110983:0.00906,137213:0.0217)0.983.64:0.00825)0.365.2:0.00182)0.998.145:0.01328)0.979.47:0.00872)0.296.3:0.00207)0.937.41:0.00719)0.941.48:0.01034)0.889.23:0.01049)1.000.811:0.0249)0.903.27:0.00759)0.856.18:0.00828)0.984.54:0.01577)0.846.15:0.0138)0.969.48:0.0196,(256882:0.01644,274344:0.0177)1.000.812:0.06133)0.940.36:0.01978)0.830.9:0.01458,((((((((305493:0.01605,(173557:0.0236,233724:0.00478)0.307.4:0.00819)0.040:0.035,(544240:0.04591,(((21579:0.02816,39058:0.0352)0.866.15:0.00635,154693:0.03222)0.423.3:0.00497,313251:0.01862)0.977.50:0.01317)0.962.41:0.01389)0.979.48:0.01673,(((144338:0.01925,254738:0.01971)0.996.106:0.03128,(141713:0.0108,((554270:0.00392,106778:0.01516)0.869.35:0.06437,(77711:0.01288,(141434:0.00332,(114511:0.01116,(141552:0.00192,77697:0.00288)0.996.107:0.01227)0.941.49:0.00482)1.000.813:0.01531)0.889.24:0.00562)0.659.4:0.01485)1.000.814:0.04896)1.000.815:0.04696,((553501:0.02217,90210:0.01408)0.432:0.07529,((550142:0.02677,((566774:0.02776,(45099:0.01429,(4551:0.0162,188414:0.0042)0.305.3:0.00916)1.000.816:0.02237)0.918.32:0.0171,180865:0.02882)0.211.3:0.00726)0.857.21:0.01093,(((253292:0.0141,579026:0.01476)0.998.146:0.02678,((143437:0.02638,78233:0.02172)1.000.817:0.06839,(148878:0.01884,143746:0.00486)1.000.818:0.06505)0.510.2:0.01086)0.776.14:0.01307,208265:0.1141)0.168.6:0.00847)0.983.65:0.02359)0.990.62:0.01933)0.547.3:0.00659)0.527.3:0.01725,(110755:0.04668,(252167:0.00865,352689:0.00849)0.523.2:0.00807)1.000.819:0.06943)0.972.51:0.02099,(((231677:0.02941,349842:0.00971)1.000.820:0.06834,(((325911:0.02378,(119998:0.00304,228764:0.01051)0.106:0.01705)1.000.821:0.06686,((125326:0.00558,547409:0.00601)0.934.34:0.00635,(227705:0.00891,116334:0.00908)0.978.49:0.01076)1.000.822:0.05205)0.968.40:0.02743,(160817:0.10783,((((560655:0.00177,320839:0.0452)0.569.3:0.03854,((231635:0.01821,((50807:0.0054,324916:0.03449)0.999.225:0.02421,((327871:0.00851,(322833:0.00057,260826:0.00078)0.847.18:0.0122)0.815.9:0.00224,333113:0.02571)0.855.19:0.01673)1.000.823:0.02547)0.660.4:0.00582,(230262:0.00472,276028:0.00718)0.451.3:0.0273)0.986.54:0.01605)0.998.147:0.02426,((327341:0.00805,(((((((315009:0.01292,(22763:0.00638,311134:0.016)0.350.4:0.00269)0.622.2:0.00302,(((316367:0.01795,230523:0.01361)0.785.12:0.00578,(307841:0.01791,(((5902:0.02905,(108804:0.0136,5900:0.01462)0.815.10:0.00131)0.857.22:0.00165,(68975:0.01599,229440:0.00883)0.968.41:0.01394)0.931.31:0.00357,107419:0.00897)0.385.5:0.00955)0.992.86:0.00921)0.196.4:0.00549,((326008:0.01685,((307973:0.01353,(321861:0.0217,(((337774:0.00776,(348493:0.01335,343807:0.02166)0.989.63:0.00833)0.749.8:0.00314,((558072:0.02594,550273:0.01959)0.974.46:0.00842,583834:0.00592)0.994.93:0.01)0.844.16:0.01471,336060:0.00782)0.731.5:0.00045)0.987.82:0.00862)0.618.5:0.0191,351210:0.00607)0.995.95:0.01777)0.860.30:0.00748,(234504:0.01643,548593:0.01411)0.817.15:0.01085)0.246.2:0.00233)0.978.50:0.00862)0.993.74:0.01356,((5904:0.01695,(583381:0.02857,540857:0.00806)0.094.2:0.00285)0.984.55:0.00939,(((((544967:0.01373,108860:0.01049)0.557:0.00081,((537980:0.00742,155191:0.02046)0.953.34:0.00629,154874:0.00886)0.760.18:0.00541)0.856.19:0.00301,(556156:0.0089,328361:0.01384)0.868.27:0.01225)0.887.24:0.00293,(((543492:0.01928,151130:0.02024)0.761.15:0.0007,102389:0.00548)0.921.32:0.00281,(125744:0.01656,5901:0.01859)0.835.14:0.00276)0.301.6:0.00137)0.735.7:0.00101,(153164:0.02717,580810:0.01376)0.668.3:0.00157)0.842.15:0.00286)0.974.47:0.00801)0.953.35:0.01173,349307:0.02661)0.908.31:0.00863,((((((559623:0.01925,336767:0.03331)0.917.37:0.01702,(((((61205:0.00944,538209:0.02883)0.766.16:0.01106,317606:0.01999)0.655.3:0.00118,564936:0.02233)0.987.83:0.01124,(307148:0.03141,(242994:0.00406,344462:0.0286)0.000.49:0.01219)0.830.10:0.0018)0.811.13:0.0061,(119251:0.00618,229684:0.00539)0.410.2:0.00443)0.847.19:0.00219)0.368.5:0.00354,306741:0.01892)0.890.28:0.004,(559061:0.01837,((543396:0.0106,108885:0.01621)0.220.4:0.00386,102110:0.01739)0.612.4:0.01691)0.996.108:0.0116)0.688.4:0.01032,((355076:0.0204,(((202120:0.01761,((353280:0.03582,(331734:0.01904,(342165:0.0288,356107:0.0128)0.777.11:0.00583)0.927.38:0.00093)0.176.2:0.0003,321636:0.01932)0.000.50:0.00122)0.876.23:0.00078,232405:0.05096)0.963.49:0.01239,336634:0.01248)0.866.16:0.00567)0.706.6:0.00096,((((343006:0.01486,354371:0.03496)0.974.48:0.0205,236718:0.02852)0.993.75:0.02198,562690:0.0027)0.997.124:0.01031,(314722:0.02021,332963:0.03205)0.997.125:0.01845)0.776.15:0.00181)0.998.148:0.01501)0.946.36:0.01499,588959:0.01247)0.911.32:0.00567)0.907.27:0.00483,((330709:0.01605,323110:0.00835)0.932.35:0.01979,((319166:0.01051,331614:0.00556)0.958.48:0.00561,332667:0.01677)0.958.49:0.00651)0.988.85:0.00843)0.871.35:0.00361,(353485:0.01317,339096:0.01958)0.651.8:0.00132)0.500.7:0.00461)0.970.36:0.01257,316482:0.06855)1.000.824:0.03363)1.000.825:0.07096,(347255:0.016,(123660:0.00329,330866:0.0095)0.974.49:0.02604)1.000.826:0.04377)0.954.39:0.02196)0.584.3:0.01239)0.858.16:0.01493)0.492.2:0.01541,(355833:0.06982,((((111655:0.04169,232609:0.014)1.000.827:0.04066,((112236:0.01313,(112194:0.02298,5950:0.00202)0.977.51:0.01319)0.994.94:0.01719,(141954:0.01098,((352699:0.01918,254127:0.00865)0.887.25:0.00391,(325735:0.01577,201547:0.01086)0.975.49:0.00878)0.920.20:0.00693)0.995.96:0.01588)0.959.37:0.0169)1.000.828:0.02989,((((66159:0.0403,(250018:0.01383,590270:0.01514)1.000.829:0.07086)0.639.5:0.0076,(560977:0.02199,136326:0.01894)1.000.830:0.02531)0.286.4:0.01104,5951:0.03697)1.000.831:0.04458,((236767:0.02419,536819:0.01611)1.000.832:0.03345,(513411:0.02317,(264506:0.01804,562891:0.00928)0.999.226:0.02133)0.986.55:0.01682)1.000.833:0.0678)0.968.42:0.01835)0.497.4:0.01806,(((434746:0.01915,(((6123:0.01002,((6112:0.01017,6104:0.00545)0.430.5:0.02184,518659:0.00402)0.978.51:0.00548)0.346.5:0.0054,105908:0.02872)0.686.3:0.00212,(106063:0.01542,(112673:0.01175,(43030:0.00059,83890:0.01428)0.821.8:0.00603)0.742.14:0.00669)0.984.56:0.00855)0.994.95:0.01431)1.000.834:0.04208,(((156880:0.00014,74038:0.02339)0.578.2:0.00737,528486:0.01171)0.997.126:0.02095,(59553:0.03439,254847:0.03268)0.616.4:0.00829)1.000.835:0.03041)0.982.63:0.01849,(6010:0.13505,(153869:0.01129,(6013:0.01877,17901:0.01028)0.502.3:0.00996)1.000.836:0.08694)0.470.4:0.01452)1.000.837:0.03457)0.706.7:0.01115)0.994.96:0.02313)0.539.3:0.01397)0.742.15:0.01222,((((((346935:0.00451,136508:0.00254)0.986.56:0.01512,(315089:0.02058,(573505:0.01411,202752:0.00788)0.544.3:0.00661)0.841.16:0.0054)1.000.838:0.04495,(((((((((((((323929:0.01338,(246412:0.01876,102846:0.01634)0.757.14:0.00077)0.953.36:0.00317,(565122:0.00787,(527154:0.02541,145164:0.02218)0.832.15:0.00291)0.970.37:0.00518)0.959.38:0.00773,((((362433:0.00177,((513441:0.02225,(47902:0.00633,414219:0.01664)0.857.23:0.00249)0.944.29:0.00464,(545515:0.01233,(38141:0.0073,544184:0.01293)0.977.52:0.00745)0.890.29:0.00314)0.888.16:0.02045)0.767.24:0.00561,((575053:0.00448,556935:0.01847)0.996.109:0.01476,136815:0.01156)0.628.7:0.0009)0.933.34:0.00379,((539380:0.0083,511099:0.0161)0.941.50:0.00636,283089:0.01716)0.065.2:0.00326)0.868.28:0.00014,(107317:0.02084,(202523:0.02185,(206948:0.01402,158376:0.01797)0.998.149:0.01257)0.984.57:0.00917)0.866.17:0.00155)0.768.14:0.00189)0.768.15:0.00231,(((46411:0.02996,535750:0.02165)0.924.30:0.00411,((((155596:0.03062,(548464:0.02255,155236:0.00615)0.137.2:0.00672)0.968.43:0.00783,((573545:0.0374,(201867:0.02182,189444:0.00275)0.000.51:0.00114)0.956.41:0.0063,((((204451:0.01975,(150328:0.02425,146818:0.01213)0.950.53:0.00531)0.859.20:0.00361,((559076:0.00188,512395:0.03736)0.316.4:0.00845,155467:0.01638)0.781.15:0.0064)0.866.18:0.00607,(((205713:0.0098,(((453324:0.0019,105525:0.00804)0.950.54:0.00546,(((192524:0.00582,321970:0.00971)0.963.50:0.00574,158965:0.00646)0.794.14:0.00339,243829:0.02165)0.168.7:0.00763)0.985.50:0.01238,539390:0.02091)0.905.26:0.00416)0.961.40:0.00473,149885:0.00724)0.962.42:0.00418,(5263:0.0221,(227405:0.00659,158953:0.0025)0.669.4:0.00726)0.799.11:0.00164)0.720.7:0.00134)0.686.4:0.00246,(222572:0.00314,(218154:0.01697,((((589676:0.01044,243842:0.00646)0.310.3:0.00016,((177007:0.0111,554002:0.012)0.938.36:0.0084,(567667:0.01856,459244:0.00658)0.989.64:0.00852)0.957.60:0.00707)1.000.839:0.01452,((216594:0.01005,(170189:0.02275,200464:0.01135)0.997.127:0.00795)0.931.32:0.00319,(((362402:0.00876,(590813:0.00136,350229:0.00135)0.879.29:0.00014)0.728.8:0.00408,573223:0.03225)1.000.840:0.00397,(217449:0.00295,283372:0.01993)0.423.4:0.00458)0.811.14:0.01417)0.847.20:0.00162)0.485:0.00146,((199327:0.01897,47066:0.00794)0.946.37:0.00602,((107637:0.0624,(203040:0.05709,107789:0.03401)0.934.35:0.00907)0.987.84:0.01156,(99135:0.00409,173386:0.00359)0.822.16:0.00661)0.872.36:0.00014)0.976.50:0.00615)0.885.22:0.0028)0.858.17:0.00157)0.930.36:0.00344)0.941.51:0.00485)0.885.23:0.00275)0.927.39:0.00318,((((147871:0.00233,166218:0.02159)0.849.13:0.00349,(103710:0.01412,568574:0.02148)0.753.24:0.00609)0.791.14:0.00421,((((423722:0.03881,466433:0.01026)0.897.24:0.01543,(146948:0.00355,534015:0.01343)0.988.86:0.01311)0.997.128:0.00961,((244132:0.02485,271876:0.07008)0.955.49:0.01367,(569436:0.02015,(591634:0.03665,((188340:0.01943,224706:0.01652)0.232.3:0.00731,144278:0.005)0.885.24:0.00155)0.461.4:0.00014)0.772.11:0.01613)0.937.42:0.00394)0.953.37:0.00407,((76582:0.00937,253214:0.00209)0.994.97:0.01775,144183:0.00645)0.864.26:0.0018)0.963.51:0.00569)0.929.35:0.00375,((((226596:0.02203,195715:0.00827)0.776.16:0.00312,((((232448:0.00015,(256527:0.01538,578761:0.0119)0.835.15:0.00655)0.913.30:0.01084,(((257030:0.00564,240333:0.01557)0.962.43:0.00469,(51455:0.00921,(74262:0.01279,49093:0.01423)0.863.26:0.00453)0.761.16:0.00077)0.920.21:0.00235,(405655:0.00938,(((((201562:0.01136,166577:0.02526)0.919.26:0.00514,513708:0.02745)0.970.38:0.00765,179906:0.01143)0.889.25:0.00758,120422:0.00965)0.892.23:0.00299,((312402:0.00854,(((244917:0.01019,22665:0.00881)0.950.55:0.00491,(152035:0.00326,564161:0.0194)0.983.66:0.00723)0.910.28:0.00678,((195857:0.00709,(235404:0.00177,539770:0.00689)0.929.36:0.00732)0.974.50:0.00527,(277780:0.00755,((((((135586:0.00075,5424:0.01669)0.932.36:0.00014,((((514145:0.06274,((144720:0.03984,26211:0.02498)0.992.87:0.01474,27851:0.00532)0.802.6:0.00165)0.921.33:0.00084,((143494:0.02318,242798:0.0137)0.978.52:0.04015,250524:0.00446)0.905.27:0.00014)0.985.51:0.00523,((329286:0.00877,244116:0.02146)0.548.5:0.00238,5441:0.01822)0.929.37:0.00553)0.865.27:0.00257,(232354:0.03024,200651:0.04038)0.057.4:0.00429)0.779.7:0.00015)0.969.49:0.01456,((142056:0.01177,(5266:0.02008,128112:0.00805)0.767.25:0.01172)0.278.2:0.00194,(567572:0.01759,250206:0.00549)0.958.50:0.00665)0.223.4:0.00351)0.635.3:0.00538,33043:0.01294)0.921.34:0.00461,66930:0.00925)0.995.97:0.01564,(215523:0.02608,((241968:0.01548,(368575:0.01504,(500111:0.01621,221997:0.01195)0.784.12:0.0068)0.700.5:0.00483)0.845.18:0.00333,(585427:0.00804,((421082:0.00622,164020:0.0145)0.979.49:0.01177,(((363192:0.00012,527777:0.00014)0.992.88:0.00765,((156180:0.01316,(5123:0.02842,77907:0.03379)0.929.38:0.01004)0.964.55:0.00743,273440:0.01352)0.833.14:0.00318)0.921.35:0.00333,262792:0.00872)0.904.29:0.00503)0.840.13:0.00276)0.576.4:0.00358)1.000.841:0.01673)0.978.53:0.01075)0.175.2:0.00339)0.701.4:0.00537)0.617.3:0.00015)0.007.2:0.00439)0.916.35:0.00739,(340714:0.02056,543669:0.01366)0.910.29:0.00548)0.730.7:0.00107)0.992.89:0.00713)0.904.30:0.00237)0.967.36:0.00504)0.942.38:0.00458,(38159:0.00944,(342329:0.00255,5439:0.00431)0.997.129:0.02454)0.762.16:0.00389)0.286.5:0.00206,323791:0.00703)0.863.27:0.00175)0.961.41:0.00393,223514:0.01064)0.959.39:0.01123,(221752:0.00014,548253:0.00014)1.000.842:0.015)0.959.40:0.0108)0.803.10:0.00012)0.642.6:0.00526,165611:0.02406)0.757.15:0.00096)0.960.53:0.00387,(542591:0.03715,(591010:0.00361,579419:0.02684)0.291.7:0.00103)1.000.843:0.02425)0.909.28:0.00014)0.980.53:0.00749,((((537767:0.01402,267825:0.01515)0.357.3:0.00313,(130944:0.00748,249985:0.02782)0.856.20:0.00539)0.195:0.00463,340224:0.01865)0.967.37:0.00736,(273074:0.02851,247484:0.01275)0.996.110:0.01132)0.785.13:0.0039)0.819.16:0.0019,(253753:0.01279,(267019:0.01802,(420642:0.0054,167601:0.0115)0.482.4:0.00043)0.901.23:0.00947)0.729.9:0.01945)0.919.27:0.00283,((354070:0.01679,(135940:0.00075,159335:0.02778)0.986.57:0.00839)0.746.11:0.00418,(((240545:0.01129,(((567089:0.00427,143303:0.02369)0.326.7:0.00275,(167459:0.01223,155982:0.01498)0.610.3:0.0031)0.966.44:0.00717,(217617:0.01088,(369965:0.00498,536784:0.01615)0.986.58:0.00654)0.836.18:0.01678)0.550.7:0.003)0.969.50:0.00891,((243410:0.02185,38898:0.03167)0.503.5:0.00574,(365493:0.01698,(5136:0.0137,252231:0.01959)0.802.7:0.00657)0.929.39:0.00975)0.939.39:0.00712)0.996.111:0.01073,557392:0.01517)0.974.51:0.00719)0.965.39:0.00443)0.860.31:0.00212,((((((24111:0.00878,(73511:0.017,(22289:0.00015,554537:0.0112)0.996.112:0.00961)0.904.31:0.00558)0.910.30:0.00303,((66931:0.00578,(5138:0.01761,151688:0.00868)0.912.21:0.00546)1.000.844:0.00014,200910:0.00363)0.950.56:0.00235)0.045.3:0.00015,552671:0.0044)0.933.35:0.00738,(518661:0.01618,(310532:0.01693,243131:0.0172)0.948.43:0.0058)0.930.37:0.00499)0.590.2:0.00196,32754:0.01497)0.885.25:0.00235,((345333:0.00726,(219107:0.02455,541971:0.00356)0.606.6:0.00185)0.752.13:0.00179,527879:0.01298)0.928.30:0.01054)0.935.26:0.00347)0.922.32:0.00315,(((226964:0.01572,224880:0.01228)0.961.42:0.00583,((((100042:0.01485,99884:0.00555)0.804.10:0.00117,(83392:0.00226,243907:0.03372)0.986.59:0.00606)0.978.54:0.00713,269321:0.00694)0.981.54:0.01415,257109:0.00095)0.881.24:0.00637)0.785.14:0.01147,552109:0.00439)0.987.85:0.00781)1.000.845:0.01627,(((((((304121:0.00559,510769:0.0052)0.727.7:0.02068,(535451:0.01754,(308393:0.01184,547134:0.01257)0.876.24:0.01076)0.952.47:0.01069)0.999.227:0.01979,(((353557:0.03627,((529803:0.00491,558229:0.03145)1.000.846:0.04884,556347:0.03672)0.830.11:0.01307)0.951.43:0.01291,(((((((557512:0.00942,(156189:0.00747,318118:0.01714)0.752.14:0.02832)0.999.228:0.02063,(156247:0.03189,((343405:0.01425,237511:0.01207)1.000.847:0.03059,(((215375:0.02514,(84103:0.01702,240591:0.00557)0.729.10:0.00049)0.986.60:0.0098,((315035:0.01166,548592:0.02518)0.920.22:0.01299,((308120:0.0186,584614:0.00876)0.680.3:0.00505,274918:0.02038)0.562.3:0.00261)0.968.44:0.01183)0.959.41:0.01161,112779:0.00643)0.999.229:0.02356)0.886.20:0.00813)0.661.5:0.0058)0.934.36:0.00633,322591:0.03003)0.436.5:0.00231,((569516:0.01262,260972:0.02887)0.961.43:0.00674,581316:0.00832)0.931.33:0.00517)0.742.16:0.00419,((270912:0.00461,514865:0.00521)1.000.848:0.04023,((269457:0.01504,571033:0.01942)0.586.2:0.00491,((330924:0.00352,79338:0.00919)0.980.54:0.01283,248919:0.03401)0.982.64:0.01704)0.690.5:0.00976)0.977.53:0.01221)0.271.2:0.0048,(276298:0.02926,(94969:0.05643,((203098:0.01156,265893:0.00093)0.802.8:0.00093,147965:0.01038)0.805.17:0.00855)0.523.3:0.00532)0.983.67:0.01041)0.992.90:0.0104,((259098:0.00014,538386:0.01338)1.000.849:0.03365,(((536382:0.01986,270010:0.01063)0.990.63:0.01352,((232375:0.01793,(320821:0.02295,(234182:0.00913,124830:0.0003)0.927.40:0.00437)0.974.52:0.01351)1.000.850:0.03988,(592876:0.02128,(289355:0.02698,231996:0.00428)0.786.17:0.00484)0.918.33:0.04272)0.997.130:0.02243)0.901.24:0.00703,((275744:0.03806,155758:0.01808)0.981.55:0.01254,((313217:0.01427,((563896:0.01493,(139871:0.01693,(583617:0.01147,101714:0.002)0.950.57:0.00538)0.921.36:0.00525)0.420.5:0.00226,554487:0.0118)0.897.25:0.02073)0.992.91:0.02253,(150582:0.0322,559688:0.01781)0.960.54:0.01558)0.802.9:0.01698)0.889.26:0.00961)0.731.6:0.00639)0.994.98:0.01285)0.775.12:0.00315)0.746.12:0.00718,(106379:0.05947,(210118:0.01121,(576887:0.00578,133782:0.03379)1.000.851:0.02476)1.000.852:0.02581)0.142.3:0.00209)0.910.31:0.00606)0.946.38:0.007,593947:0.03064)0.148.2:0.0021,(136018:0.03619,((239275:0.02165,(254012:0.03859,(303233:0.02327,150659:0.01145)0.975.50:0.00972)0.664.2:0.00763)0.991.66:0.01951,((((((((165558:0.00634,(82396:0.01259,502999:0.01539)0.870.27:0.00158)0.829.14:0.01309,(((350951:0.00231,107183:0.02268)0.941.52:0.00584,93573:0.01381)0.880.24:0.01129,((575419:0.01275,209427:0.01446)0.874.23:0.00254,113140:0.00622)0.000.52:0.00103)0.968.45:0.0074)0.717.9:0.00086,(99122:0.01176,(355188:0.01793,174315:0.01578)0.860.32:0.00191)0.976.51:0.01186)0.851.20:0.00372,((332501:0.00888,273185:0.02638)0.743.18:0.00086,(((((525087:0.00622,175383:0.00669)0.978.55:0.00617,563292:0.0116)0.801.10:0.00139,(550842:0.01837,(573755:0.00803,565555:0.01113)0.865.28:0.00614)0.992.92:0.00837)0.842.16:0.00198,(573013:0.02551,(243709:0.0179,((212196:0.01359,(215447:0.03452,544772:0.02302)0.918.34:0.01428)0.968.46:0.01473,(573246:0.01587,((((559356:0.01728,516056:0.01342)0.379.9:0.00344,453654:0.02488)0.866.19:0.00015,(540672:0.02479,255206:0.01932)0.960.55:0.0077)0.596.4:0.0046,130155:0.01116)0.279.4:0.00279)0.915.35:0.0133)0.302.4:0.00237)0.975.51:0.00759)0.901.25:0.00487)0.917.38:0.00431,(317411:0.00587,365288:0.02096)0.980.55:0.01025)0.886.21:0.00404)0.988.87:0.00792)0.375.4:0.0018,(224376:0.02293,((223711:0.00086,541036:0.03138)0.960.56:0.02604,(555082:0.00856,25284:0.01895)0.720.8:0.00359)0.700.6:0.00817)0.999.230:0.01115)0.855.20:0.00245,224543:0.00922)0.953.38:0.00741,(245814:0.02269,232353:0.00358)0.963.52:0.00825)0.713.4:0.00789,226508:0.01511)0.953.39:0.01104)0.991.67:0.01476)0.976.52:0.01076)0.912.22:0.00443,(((((((107461:0.0183,((467965:0.00668,536995:0.02117)0.995.98:0.01739,(173071:0.00891,87497:0.01792)0.998.150:0.01858)0.893.21:0.01309)0.107.2:0.0075,(548842:0.01911,582735:0.01138)0.985.52:0.01743)0.995.99:0.01927,(370667:0.02328,105473:0.01325)1.000.853:0.03198)0.991.68:0.0138,(((90420:0.02791,(222685:0.0226,171887:0.00919)0.898.29:0.00782)0.950.58:0.00741,(526198:0.0293,((344447:0.0194,(567929:0.02514,(113345:0.03542,(586430:0.01653,((304694:0.00173,574405:0.0046)0.982.65:0.00749,(209662:0.02804,(287211:0.00381,((326268:0.00699,338715:0.02208)0.886.22:0.00257,132848:0.01213)0.951.44:0.00495)0.700.7:0.00559)0.740.17:0.00469)0.934.37:0.00752)0.982.66:0.01178)0.817.16:0.00334)0.245.4:0.00169)0.637.7:0.00355,((351644:0.01651,(583178:0.02079,547229:0.00478)0.806.11:0.01509)0.919.28:0.00595,(113477:0.01403,558394:0.0213)0.989.65:0.01368)0.917.39:0.00862)0.782.18:0.00662)0.899.30:0.00611)0.979.50:0.00963,((50238:0.01848,((180502:0.0048,(273758:0.00105,174460:0.00016)0.927.41:0.00942)0.859.21:0.00169,(196623:0.00066,551756:0.0237)0.357.4:0.02426)1.000.854:0.02134)0.949.38:0.00743,(((((((246824:0.00802,523501:0.01922)0.913.31:0.00259,(100899:0.02028,4708:0.00677)0.862.33:0.01529)0.879.30:0.02792,(((174214:0.00455,((4650:0.00245,560403:0.03796)0.863.28:0.00536,(236974:0.00642,105545:0.01091)0.851.21:0.00231)0.953.40:0.00467)0.746.13:0.01462,350760:0.019)0.968.47:0.01045,221589:0.00925)0.979.51:0.01096)0.816.13:0.00759,(239819:0.01625,(((224138:0.02554,209096:0.02183)0.984.58:0.01194,207369:0.00114)0.921.37:0.00608,((146531:0.00978,20480:0.03181)0.024:0.00014,105233:0.00744)0.964.56:0.03225)0.900.28:0.00706)0.996.113:0.01726)0.980.56:0.01307,((((5497:0.0291,(379348:0.00346,212879:0.01669)0.220.5:0.00531)0.856.21:0.00198,558957:0.01028)0.233:0.0016,544008:0.01131)0.990.64:0.01554,(89155:0.02469,593232:0.02899)0.987.86:0.01722)0.597.4:0.01151)0.808.10:0.00623,(349473:0.0041,559544:0.01327)0.979.52:0.01872)0.929.40:0.0084,((203162:0.03832,(((203753:0.02081,(29077:0.04233,(587672:0.01221,(557772:0.00988,315470:0.00322)0.644.5:0.00319)0.826.12:0.01701)0.627.4:0.01003)0.974.53:0.01312,(((588231:0.03262,(4715:0.004,545895:0.01289)0.591.4:0.01449)0.984.59:0.01177,((560101:0.03559,540721:0.02095)0.993.76:0.01852,(143411:0.0278,(330207:0.00878,198655:0.01577)0.999.231:0.02112)0.944.30:0.01425)0.962.44:0.01158)0.846.16:0.01047,(279185:0.1011,(((575272:0.01205,5026:0.00475)0.755.17:0.01649,(517509:0.01484,(((153825:0.01158,(((((((237433:0.04352,211106:0.02606)0.983.68:0.01663,240912:0.03588)0.883.24:0.00731,346311:0.03322)0.867.21:0.00236,(277612:0.01818,271504:0.04641)0.876.25:0.01111)0.875.29:0.00091,(146397:0.02068,(160432:0.00941,159763:0.01319)0.899.31:0.00864)0.787.9:0.00046)0.864.27:0.00322,573764:0.02345)0.836.19:0.00092,(((((252635:0.01836,((138381:0.03052,(68786:0.00474,224099:0.03035)0.945.35:0.00318)0.870.28:0.00221,((111553:0.03415,166553:0.00639)0.952.48:0.00317,241046:0.00554)0.989.66:0.0081)0.927.42:0.00856)0.768.16:0.0127,284778:0.01793)0.902.19:0.00056,(29702:0.0265,107036:0.00982)0.976.53:0.006)0.767.26:0.00566,(4829:0.01501,4978:0.01833)0.748.16:0.00481)0.742.17:0.00614,(((65460:0.01508,(231834:0.01632,559211:0.0101)0.942.39:0.00525)0.843.14:0.00704,4972:0.00156)0.856.22:0.00076,197199:0.02739)1.000.855:0.00126)0.965.40:0.00311)0.883.25:0.00221)0.942.40:0.00883,(372774:0.01054,313427:0.02846)0.881.25:0.01792)0.428.3:0.00269,(543472:0.01846,280723:0.02012)0.985.53:0.01061)0.831.9:0.00239)0.966.45:0.00716)0.896.30:0.00378,572152:0.0194)0.461.5:0.00666)0.804.11:0.01025)0.989.67:0.01133)0.761.17:0.0032,((136238:0.02182,210722:0.03305)0.960.57:0.01044,(((212776:0.02565,253944:0.03236)0.987.87:0.01788,317632:0.02644)0.864.28:0.00593,((((((((179877:0.00959,(534993:0.0254,(39556:0.03123,(575164:0.00566,316335:0.01616)0.175.3:0.00468)0.100.4:0.00568)0.975.52:0.01115)0.998.151:0.01342,((365619:0.01093,(((159481:0.00088,135980:0.00245)0.997.131:0.01019,153648:0.01919)0.292.7:0.00711,((222350:0.00919,523319:0.01268)0.801.11:0.00408,(234081:0.02513,114090:0.02848)0.699.3:0.00107)0.959.42:0.00726)0.603.5:0.00277)0.868.29:0.00502,(358785:0.00787,(325065:0.00833,237595:0.01379)0.444.4:0.00221)0.904.32:0.00247)0.976.54:0.0069)0.074.4:0.00183,213691:0.01452)0.968.48:0.00735,(592746:0.00901,((101471:0.01606,149501:0.00653)0.967.38:0.00989,(256335:0.01259,534454:0.01105)0.940.37:0.00834)0.912.23:0.00767)0.233.2:0.00509)0.939.40:0.0054,(((248505:0.01515,585175:0.02862)0.822.17:0.00742,(60583:0.02857,((114139:0.01778,(112955:0.00766,119955:0.01172)0.515.5:0.00899)0.917.40:0.00503,306133:0.01306)0.918.35:0.00684)0.831.10:0.00643)0.973.48:0.00873,((257244:0.0214,(205156:0.00471,555816:0.00247)1.000.856:0.0208)0.946.39:0.00498,(((((318586:0.01175,553855:0.0081)0.991.69:0.0108,(114396:0.01366,256361:0.01351)0.909.29:0.00596)0.987.88:0.00977,361574:0.0121)0.539.4:0.00427,114057:0.01766)0.901.26:0.00329,((163571:0.01429,252670:0.01607)0.985.54:0.01568,((215423:0.00564,227957:0.02285)0.965.41:0.00916,(584598:0.00537,(511344:0.01,211157:0.00795)0.905.28:0.00491)0.909.30:0.00509)0.999.232:0.01718)0.970.39:0.00838)0.872.37:0.00293)0.765.15:0.00135)0.902.20:0.00638)0.843.15:0.00169,((((((237095:0.02388,(144888:0.01234,112834:0.01951)0.823.16:0.00206)0.675.2:0.00308,238490:0.01627)0.838.19:0.0108,20976:0.00627)0.901.27:0.00238,(((((107444:0.02209,76270:0.01586)0.864.29:0.00967,(327961:0.01234,343296:0.00963)0.955.50:0.00439)0.947.40:0.00458,308892:0.01176)0.508.6:0.00156,353828:0.00357)0.958.51:0.011,(((((97561:0.01514,546305:0.01083)0.847.21:0.0023,((156816:0.03063,113504:0.00801)0.859.22:0.01274,4721:0.01419)0.995.100:0.0075)0.902.21:0.00308,52053:0.00638)0.904.33:0.00014,(4681:0.0271,(566886:0.00475,572724:0.01298)0.873.21:0.00172)0.789.17:0.00234)0.903.28:0.00317,544615:0.01859)0.746.14:0.00072)0.756.19:0.00083)0.864.30:0.00232,((207374:0.00748,111439:0.01793)0.953.41:0.00687,((248385:0.01745,53533:0.02338)0.512.5:0.01112,583709:0.02351)0.742.18:0.00119)0.959.43:0.00608)0.679.8:0.00074,(265043:0.01068,152124:0.01303)0.825.14:0.00216)0.781.16:0.00218)0.778.16:0.00162,(((247905:0.0143,(154028:0.00959,238973:0.01822)0.783.13:0.00198)0.996.114:0.01205,(204919:0.01454,98457:0.02549)0.940.38:0.00608)0.979.53:0.00815,(315986:0.02177,218539:0.03795)0.356.3:0.00239)0.873.22:0.00369)0.940.39:0.00798,(254308:0.0305,((212484:0.0253,254348:0.02038)1.000.857:0.05722,(210974:0.02124,208963:0.02214)0.432.2:0.0044)0.931.34:0.00995)0.999.233:0.02786)0.743.19:0.00728)0.800.11:0.00318)0.961.44:0.00805)0.638.3:0.00271)0.221.3:0.00719,((((254744:0.03156,((((243360:0.00388,(248337:0.00872,226516:0.0162)0.768.17:0.00862)0.829.15:0.00457,(319033:0.0148,162188:0.01072)0.808.11:0.0021)0.988.88:0.00927,(((((434250:0.01063,(253274:0.01036,247879:0.01427)0.703.10:0.00714)0.798.8:0.00252,(138424:0.00955,169423:0.01875)0.999.234:0.01084)0.991.70:0.0105,((66234:0.00945,((277676:0.06584,(169404:0.01356,(358718:0.00337,538266:0.00172)0.788.13:0.00526)0.951.45:0.01032)0.498.4:0.00016,171280:0.01578)0.875.30:0.00798)0.743.20:0.02614,511192:0.02376)0.280.3:0.00183)0.885.26:0.00462,(((584480:0.00577,4803:0.01079)0.921.38:0.00805,(109048:0.01878,(318318:0.00081,156941:0.00158)0.243.3:0.00835)0.941.53:0.00998)0.878.32:0.01378,(362293:0.00326,237106:0.01887)0.984.60:0.01232)0.960.58:0.01303)0.219.3:0.00333,(((((421277:0.01565,(591699:0.011,493250:0.00555)0.999.235:0.01069)1.000.858:0.02026,(((367554:0.01347,((579710:0.01333,(546221:0.01244,(((528281:0.00523,548603:0.01591)0.967.39:0.01133,(261034:0.00106,(305729:0.03025,247045:0.03271)0.754.13:0.00505)0.686.5:0.00245)0.899.32:0.00336,((347247:0.00679,161482:0.02779)0.920.23:0.00092,107137:0.04022)0.852.26:0.00355)0.969.51:0.00626)0.238.2:0.00343)0.978.56:0.00709,(4797:0.01002,4805:0.00932)0.992.93:0.00813)0.662.7:0.00203)0.897.26:0.00335,(166094:0.00515,208196:0.03377)0.946.40:0.0052)0.995.101:0.00901,102764:0.01986)0.480.5:0.00087)0.889.27:0.00247,253825:0.0128)0.744.13:0.0077,((85734:0.01475,(527526:0.00312,4772:0.03879)0.754.14:0.00153)0.962.45:0.01026,(261309:0.02393,((160852:0.00929,500699:0.00919)0.970.40:0.00681,(342927:0.00654,(153508:0.06784,109372:0.00092)0.942.41:0.00233)0.897.27:0.00382)0.967.40:0.00696)0.984.61:0.01054)0.978.57:0.00765)0.973.49:0.00649,(341095:0.01183,(256447:0.00734,164604:0.01042)0.253:0.00243)0.969.52:0.00786)0.916.36:0.00554)0.888.17:0.0076)0.940.40:0.00594,(274150:0.01058,(410855:0.01295,552903:0.01365)0.971.44:0.00777)0.810.15:0.0016)0.971.45:0.00774)0.956.42:0.00742,(332834:0.01707,(452998:0.01641,310925:0.02369)0.966.46:0.00912)0.842.17:0.00291)0.861.26:0.00208,((528636:0.01101,(((567336:0.04023,223155:0.00628)0.913.32:0.00859,583653:0.00156)1.000.859:0.0241,4819:0.01032)0.984.62:0.00906)0.893.22:0.00365,(548469:0.01184,553778:0.02868)0.962.46:0.00724)0.969.53:0.00643)0.999.236:0.014,((((((((225511:0.01441,(518637:0.01159,333473:0.01291)0.702.8:0.00441)0.295.3:0.00305,(106192:0.01365,471401:0.00985)0.921.39:0.00448)0.720.9:0.00281,(4762:0.02029,306170:0.01064)0.971.46:0.00582)0.781.17:0.00178,((270150:0.01429,(51161:0.00963,(156677:0.03164,530044:0.01246)0.995.102:0.01293)0.895.32:0.0079)0.777.12:0.00184,19989:0.01044)0.910.32:0.00334)0.889.28:0.00178,((((43530:0.01244,(124699:0.01065,(144675:0.01099,364780:0.01013)0.957.61:0.00502)0.969.54:0.01151)0.976.55:0.00624,(252325:0.01486,(201019:0.01209,545434:0.01304)0.960.59:0.00672)0.073.4:0.00436)0.902.22:0.0048,4760:0.02465)0.134.2:0.00305,((570609:0.01555,552788:0.01242)0.989.68:0.00773,136235:0.02622)0.591.5:0.00385)0.736.14:0.00119)0.787.10:0.002,538998:0.02741)0.956.43:0.00617,501498:0.01737)0.907.28:0.00351,(544410:0.01207,(((586876:0.21786,(232812:0.01102,(((162925:0.00959,537864:0.01162)0.912.24:0.00256,156044:0.0204)0.949.39:0.00386,(4741:0.0032,(4752:0.01477,(4737:0.01893,4735:0.02045)0.773.13:0.00541)0.950.59:0.00529)0.910.33:0.00249)0.284.6:0.00015)0.735.8:0.00191)0.590.3:0.0005,(113362:0.00823,(152696:0.03492,4736:0.01436)0.783.14:0.00116)0.732.9:0.00887)0.780.10:0.0016,((570890:0.01562,((225488:0.01014,142726:0.01712)0.979.54:0.01189,106732:0.01867)0.328.3:0.00242)0.791.15:0.00494,((((137669:0.00742,255169:0.0176)0.922.33:0.00607,357993:0.00824)1.000.860:0.02584,(((556571:0.0136,(365947:0.01513,338761:0.00262)0.996.115:0.01309)0.961.45:0.00714,(158970:0.0024,154041:0.01937)0.957.62:0.00589)0.368.6:0.00361,242089:0.00864)0.999.237:0.01851)0.999.238:0.02157,(544990:0.02045,(((332410:0.02905,(548876:0.01846,152079:0.0132)0.867.22:0.00619)0.665.4:0.00156,155413:0.01448)0.791.16:0.00138,(112859:0.03718,((515986:0.00868,509544:0.00983)0.783.15:0.0029,58103:0.02152)0.595:0.00449)0.720.10:0.00119)0.896.31:0.0041)0.086:0.00376)0.994.99:0.01261)0.875.31:0.00509)0.811.15:0.00181)0.915.36:0.00399)0.991.71:0.01144)0.881.26:0.00943)0.962.47:0.01231)0.880.25:0.00513)0.777.13:0.00205)0.976.56:0.00849)0.294.5:0.00107,(225699:0.01978,513048:0.02296)0.990.65:0.01251)0.268.2:0.00318,(((253265:0.01932,238442:0.01725)0.933.36:0.01082,(156227:0.02604,152480:0.01691)0.978.58:0.01289)0.978.59:0.01089,((((202674:0.00352,(552944:0.00731,560143:0.01735)1.000.861:0.01884)0.200.5:0.02393,(306677:0.0033,(262760:0.01885,(4638:0.00828,178313:0.01527)0.894.28:0.00387)0.758.15:0.00856)0.946.41:0.00719)0.939.41:0.00717,((574086:0.02586,(583185:0.00622,(549098:0.02554,519258:0.01551)0.504.4:0.00399)0.957.63:0.00895)0.819.17:0.00937,(589466:0.02107,(((((591347:0.03163,4640:0.01497)0.990.66:0.01551,(((59721:0.00014,90031:0.00092)0.274.4:0.01602,26693:0.00643)0.973.50:0.00916,210886:0.0234)0.898.30:0.00398)0.878.33:0.00266,((((586152:0.01438,((4617:0.01935,208626:0.02302)0.881.27:0.00293,(((305643:0.02014,309465:0.00376)0.998.152:0.01097,246068:0.03685)0.794.15:0.0018,(573426:0.00569,(207866:0.02124,(529888:0.01556,533498:0.01583)0.325.2:0.00422)0.950.60:0.00537)0.926.45:0.00601)0.985.55:0.00819)0.996.116:0.01178)0.411.4:0.00252,138039:0.01369)0.619.3:0.00102,(561725:0.01973,318155:0.01894)0.960.60:0.00453)0.908.32:0.00257,((((114978:0.01339,((167855:0.00328,553511:0.008)0.983.69:0.00575,((47454:0.01898,131364:0.00028)0.839.19:0.0025,4619:0.00785)0.787.11:0.00527)0.826.13:0.00317)0.722.7:0.00146,172883:0.01746)0.873.23:0.00247,172128:0.01456)0.987.89:0.00904,144964:0.00818)1.000.862:0.02188)0.735.9:0.00319)0.937.43:0.0035,(((564600:0.02484,233793:0.03492)0.143.4:0.00316,151855:0.00575)0.938.37:0.00562,106838:0.00311)0.924.31:0.00383)0.982.67:0.00559,((579761:0.03045,(((4633:0.00939,34344:0.01456)1.000.863:0.00432,115024:0.02719)0.290.4:0.00016,(((559937:0.01729,573659:0.01565)0.962.48:0.00463,(333432:0.01867,88254:0.0165)0.881.28:0.00296)0.989.69:0.00664,4635:0.0084)0.880.26:0.00016)0.331.6:0.00774)0.530.4:0.00608,(52859:0.00953,198749:0.02323)0.996.117:0.01445)0.894.29:0.00406)0.921.40:0.00351)0.531.6:0.0017)0.891.30:0.00636)0.919.29:0.01003,((540718:0.02856,566212:0.03269)0.933.37:0.01505,((583294:0.00919,((66822:0.0139,304412:0.03328)0.979.55:0.01364,(240590:0.0038,211951:0.03206)0.855.21:0.01949)0.858.18:0.00479)0.653:0.00469,((591539:0.03002,(((182489:0.02118,254805:0.00907)0.999.239:0.01421,136944:0.01709)0.994.100:0.01071,573041:0.00374)0.890.30:0.00736)0.994.101:0.01357,((546964:0.02373,161555:0.01835)0.875.32:0.00348,525648:0.01185)0.634.5:0.0048)0.903.29:0.0043)0.999.240:0.02728)0.998.153:0.02369)0.876.26:0.00682)0.979.56:0.00961)0.872.38:0.003,((551423:0.02924,141437:0.0153)0.918.36:0.00889,(161408:0.03355,(150568:0.02055,(150577:0.01902,277481:0.02896)0.719.4:0.01001)0.999.241:0.02636)0.905.29:0.00964)0.948.44:0.00691)0.802.10:0.00332)0.969.55:0.00766,((509779:0.03279,(((512225:0.01576,569481:0.01964)0.853.22:0.00848,587323:0.01123)0.755.18:0.03253,(558255:0.03181,(259822:0.01281,306234:0.01293)1.000.864:0.03127)0.159.3:0.01086)0.443.3:0.00771)0.650.2:0.00959,((325677:0.02847,(122223:0.02233,593115:0.02583)0.940.41:0.01355)0.315.2:0.00984,((582995:0.0072,(161779:0.00322,(561163:0.00643,347417:0.01412)0.237.2:0.02559)0.978.60:0.01872)0.864.31:0.01345,(((114432:0.00925,(((581542:0.0023,342195:0.02401)0.853.23:0.00315,47270:0.01668)0.927.43:0.01925,((264175:0.01129,114859:0.01681)0.912.25:0.01423,94391:0.00094)0.999.242:0.00913)0.947.41:0.00696)1.000.865:0.0124,(194356:0.01932,111501:0.01781)0.212.4:0.00225)0.743.21:0.00669,92008:0.00359)0.898.31:0.008)0.995.103:0.0167)0.958.52:0.00635)0.127:0.00446)0.933.38:0.00466)0.116.7:0.00115,((((345716:0.01915,(123474:0.02956,562114:0.01659)0.395.5:0.01099)1.000.866:0.03526,(((545503:0.02266,(111182:0.0214,((548219:0.01736,(((567681:0.0161,587645:0.00908)0.323.3:0.00469,575915:0.02328)0.564.4:0.00321,21491:0.00782)0.977.54:0.00981)1.000.867:0.03078,(135997:0.00759,203076:0.0167)0.999.243:0.02257)0.444.5:0.0089)0.956.44:0.01012)0.972.52:0.0104,((165140:0.01661,296248:0.00784)0.988.89:0.00947,(278733:0.00816,(265891:0.01583,548305:0.00799)0.894.30:0.00404)0.979.57:0.00912)0.974.54:0.00866)0.955.51:0.00578,(((340342:0.01781,((170768:0.01138,(550862:0.01105,341787:0.01016)0.941.54:0.0092)0.645.11:0.00581,(114588:0.00937,594080:0.01192)0.998.154:0.02728)0.905.30:0.00452)0.927.44:0.00585,(559757:0.01592,(310049:0.02358,553131:0.01944)0.905.31:0.00464)0.960.61:0.00752)0.949.40:0.01179,((135247:0.01736,580769:0.01568)0.998.155:0.0161,133177:0.03269)0.818.7:0.00787)0.625.5:0.00257)0.829.16:0.00418)0.535.4:0.00954,(((((244920:0.01469,357791:0.01752)0.910.34:0.00466,((((593441:0.01447,(365967:0.00902,((360432:0.02672,((218626:0.00607,167156:0.01767)0.881.29:0.00775,((242423:0.01652,220653:0.0164)0.706.8:0.00709,(549083:0.00516,(172438:0.01467,165827:0.00777)0.549.4:0.00537)0.848.21:0.00485)0.891.31:0.00476)0.688.5:0.00338)0.182.3:0.00313,160252:0.03374)0.981.56:0.01031)0.878.34:0.00782)0.991.72:0.0112,(347422:0.02016,566238:0.01139)0.699.4:0.00422)0.998.156:0.01644,((110731:0.02371,319087:0.0096)0.973.51:0.01021,(306876:0.01635,(590969:0.00541,(320764:0.01077,166922:0.0078)0.995.104:0.01141)0.955.52:0.00774)0.930.38:0.00583)0.964.57:0.0119)0.610.4:0.00549,((106469:0.02443,((((348420:0.01557,(549500:0.00418,211048:0.02455)0.757.16:0.00133)0.868.30:0.01088,(((222965:0.00283,203414:0.01539)0.945.36:0.00659,((290742:0.02147,324499:0.00916)0.571.5:0.00316,(104513:0.01232,532882:0.00029)0.975.53:0.0081)0.999.244:0.0149)0.944.31:0.01137,(73128:0.00712,166987:0.02792)0.983.70:0.0096)0.940.42:0.0048)0.979.58:0.00755,((((423096:0.01818,(((577508:0.00479,(223080:0.00522,217471:0.00158)0.986.61:0.00769)1.000.868:0.01905,540427:0.00962)0.920.24:0.0087,(170315:0.0114,(137549:0.01327,(279616:0.01151,422581:0.01498)0.397.5:0.00393)0.942.42:0.00623)0.735.10:0.00105)0.749.9:0.0035)0.764.16:0.00188,(((103674:0.01563,587602:0.01696)0.996.118:0.01328,(141577:0.01768,240707:0.00249)0.999.245:0.02135)0.778.17:0.00714,(549968:0.0089,(27774:0.00832,170549:0.02766)0.911.33:0.00513)0.816.14:0.0063)0.982.68:0.00983)0.908.33:0.0044,(((((((246665:0.01415,106075:0.01543)0.823.17:0.00667,(510302:0.01474,61250:0.03191)0.995.105:0.01536)0.309.4:0.00481,(563888:0.02075,209744:0.01436)0.938.38:0.00642)0.898.32:0.00365,(328838:0.00745,306409:0.0354)0.995.106:0.01114)0.734.11:0.0008,(((((258074:0.0181,(262333:0.0178,(((580223:0.01024,(368703:0.0026,548120:0.0245)0.905.32:0.00342)0.993.77:0.00822,((287385:0.012,((113817:0.00461,61109:0.0057)0.536.2:0.00753,(297968:0.01286,((((315930:0.01628,(((572670:0.01197,592471:0.03737)0.889.29:0.00621,(579705:0.01293,(544093:0.0087,590614:0.0183)0.973.52:0.00915)0.412.2:0.00464)0.643.3:0.00586,71580:0.01265)0.719.5:0.00721)0.458.3:0.00302,(5681:0.01939,((592909:0.01965,5642:0.01832)0.884.31:0.00391,75228:0.0207)0.772.12:0.00154)0.937.44:0.00414)0.949.41:0.00016,((((560186:0.01171,353846:0.00526)0.966.47:0.0121,(75623:0.02378,(113877:0.00014,106101:0.00163)0.931.35:0.01028)0.669.5:0.00346)0.933.39:0.00947,(115231:0.01522,202723:0.01068)0.865.29:0.00867)0.841.17:0.00413,(149884:0.00643,(593530:0.01624,135905:0.01454)0.586.3:0.01695)0.965.42:0.00579)0.783.16:0.00081)0.989.70:0.0059,((((571405:0.02451,532672:0.01627)0.836.20:0.0025,154751:0.01103)0.998.157:0.01062,548236:0.02372)0.887.26:0.00422,((549110:0.0261,152925:0.00921)0.759.12:0.0015,82225:0.00933)0.920.25:0.00683)0.905.33:0.00301)0.985.56:0.00909)0.947.42:0.00512)0.838.20:0.00378)0.964.58:0.00851,151410:0.00471)0.407.3:0.00163)0.953.42:0.00335,(113364:0.01628,71804:0.00729)0.852.27:0.01685)0.883.26:0.00162)0.964.59:0.00675)0.798.9:0.00149,(342325:0.01507,(364573:0.00696,293005:0.01599)0.971.47:0.00704)0.860.33:0.00171)0.599.5:0.00178,(((277670:0.00723,((534739:0.01058,546437:0.02126)0.960.62:0.0079,(80289:0.01244,79260:0.02454)0.681.9:0.00367)0.869.36:0.00203)0.964.60:0.00478,(135479:0.02018,138759:0.01297)0.852.28:0.00325)0.871.36:0.00212,((103182:0.02273,251325:0.0389)0.954.40:0.00631,104479:0.0116)0.934.38:0.00429)0.964.61:0.00486)0.024.2:0.00165,((((254355:0.00234,364995:0.03071)0.353.3:0.00177,((226155:0.00855,145901:0.01066)0.961.46:0.01179,((352275:0.0114,(313326:0.01088,110751:0.02319)0.749.10:0.00185)0.963.53:0.00701,(320903:0.01372,264674:0.02174)0.726.9:0.00371)0.871.37:0.00293)0.986.62:0.01408)0.819.18:0.00476,511895:0.04183)0.823.18:0.00155,(114827:0.01396,(135261:0.02156,215354:0.00014)0.989.71:0.00908)0.913.33:0.00251)0.874.24:0.00176)0.907.29:0.00387,(38427:0.01341,155375:0.00574)0.999.246:0.01755)0.544.4:0.00397)0.869.37:0.00382,(306783:0.01004,(140242:0.01332,((100870:0.01998,(100404:0.01752,((220404:0.01576,((186153:0.00834,((276652:0.00242,243335:0.00452)0.808.12:0.02125,328066:0.01045)0.835.16:0.00728)0.956.45:0.00799,541462:0.00719)0.357.5:0.00194)0.999.247:0.01177,(277446:0.0071,322963:0.00767)0.884.32:0.00655)0.849.14:0.02049)0.265.3:0.00484)0.237.3:0.00368,145649:0.02057)0.767.27:0.00135)0.945.37:0.00653)0.938.39:0.01758)0.680.4:0.00228,((((222231:0.00952,(((240900:0.01765,46143:0.00361)0.929.41:0.00326,(341986:0.02424,((((555105:0.01448,265749:0.00559)0.155.4:0.00226,5629:0.01306)0.993.78:0.01074,569576:0.00389)0.985.57:0.01051,(364960:0.01687,271908:0.01109)0.974.55:0.007)0.900.29:0.00328)0.561.6:0.00014)0.987.90:0.00863,(247890:0.01595,(239375:0.01302,359190:0.03619)0.978.61:0.01266)0.997.132:0.01601)0.422.5:0.00124)0.937.45:0.00463,((((91134:0.01559,((101280:0.001,160406:0.02849)0.936.33:0.01065,150135:0.01802)0.976.57:0.00703)0.829.17:0.00324,138930:0.02331)0.848.22:0.00295,331090:0.01187)0.933.40:0.0037,(((((((5703:0.01254,(50418:0.01975,(83098:0.00421,304753:0.00109)0.980.57:0.01287)0.479.8:0.00334)0.971.48:0.00689,((((171023:0.0092,219732:0.02357)0.986.63:0.01189,(162482:0.01146,362911:0.00842)0.864.32:0.00745)0.106.2:0.00015,((((((347121:0.01194,537320:0.00578)0.978.62:0.01959,(564346:0.01951,554517:0.02098)0.943.53:0.00575)0.910.35:0.00341,109582:0.01593)0.926.46:0.00651,189161:0.02271)0.979.59:0.00578,(132018:0.01786,(356855:0.0267,(305389:0.00684,247324:0.02394)0.607.6:0.00564)0.657.2:0.00389)0.976.58:0.0076)0.958.53:0.00446,((142136:0.009,17447:0.00014)0.260.4:0.00668,569713:0.01273)0.982.69:0.00927)0.929.42:0.00378)0.967.41:0.00837,((315323:0.0149,(297563:0.01996,(68976:0.00082,134828:0.00015)0.877.25:0.0139)0.916.37:0.00361)0.884.33:0.00298,(548736:0.01148,536793:0.02155)0.852.29:0.01044)0.387.6:0.00252)0.839.20:0.00228)0.932.37:0.00405,((((45295:0.01302,565176:0.00754)0.862.34:0.0045,(153431:0.03557,567643:0.01802)0.872.39:0.00296)0.984.63:0.00839,325747:0.04539)0.814.9:0.00495,112728:0.00978)0.894.31:0.00016)0.935.27:0.00314,((104078:0.00345,159906:0.00014)0.817.17:0.00604,(278099:0.00127,539408:0.0073)0.974.56:0.00732)0.859.23:0.01678)0.844.17:0.00421,(((79185:0.02444,542437:0.01603)0.776.17:0.00362,(272287:0.01736,((581582:0.01441,154523:0.01206)0.963.54:0.0093,(337345:0.01897,253055:0.00839)0.972.53:0.01032)0.953.43:0.00828)0.918.37:0.00518)0.812.18:0.00342,(((334722:0.01784,((((575673:0.02205,336543:0.01935)0.274.5:0.00125,(338768:0.00823,((569378:0.0365,5706:0.00861)1.000.869:0.02876,(((591725:0.00747,278412:0.01747)0.964.62:0.00648,558277:0.01787)0.966.48:0.00773,417573:0.02532)0.619.4:0.00244)0.406.2:0.00093)0.992.94:0.0072)0.891.32:0.00396,(297421:0.00348,552641:0.03588)0.795.14:0.0078)0.958.54:0.00919,5672:0.03359)0.981.57:0.00929)0.883.27:0.00342,(((291829:0.0152,557153:0.00749)0.883.28:0.00499,548114:0.01996)0.981.58:0.00987,((245675:0.01945,590937:0.01413)0.855.22:0.00336,5645:0.004)0.115:0.0033)0.978.63:0.00899)0.896.32:0.00277,(((194519:0.01099,155908:0.01829)0.942.43:0.00658,((81559:0.00849,154786:0.01916)0.842.18:0.00572,346324:0.03338)0.928.31:0.00476)0.206.2:0.00014,(109216:0.01289,335148:0.01174)0.918.38:0.00393)0.872.40:0.00172)0.867.23:0.00475)0.958.55:0.00546)0.785.15:0.00243,((276906:0.02301,(26995:0.02014,135688:0.00671)0.917.41:0.0188)0.583.6:0.00332,((106356:0.00825,(163471:0.00189,546740:0.00259)0.999.248:0.01613)0.921.41:0.00526,(5632:0.01573,(99671:0.01372,146931:0.01821)0.874.25:0.01187)0.774.9:0.01277)0.363.6:0.00136)0.859.24:0.00485)0.285.3:0.00327,((168588:0.01362,(515404:0.02336,74722:0.01061)0.953.44:0.01103)0.343.3:0.00608,(((303567:0.00016,(147134:0.02188,368902:0.00424)0.965.43:0.02452)0.817.18:0.00503,((572325:0.01264,(327863:0.01213,276820:0.01098)0.810.16:0.00203)0.917.42:0.00316,(201085:0.04599,(159254:0.01396,257864:0.00843)0.981.59:0.01195)0.969.56:0.00869)0.971.49:0.00759)0.987.91:0.00901,(577629:0.01439,(535793:0.02407,(520995:0.00171,((((155937:0.02005,568416:0.01363)0.978.64:0.0099,(((((235464:0.01841,(170685:0.03428,516831:0.01827)0.370.3:0.00992)0.900.30:0.00535,((113266:0.01559,525855:0.02095)0.954.41:0.0138,((252343:0.00892,(257910:0.00723,(200932:0.0142,(278279:0.01445,75353:0.02241)0.706.9:0.00329)0.907.30:0.0154)0.973.53:0.00988)0.997.133:0.01627,5649:0.02043)0.984.64:0.01104)0.931.36:0.00775)0.280.4:0.00276,333407:0.02577)0.970.41:0.00986,(132289:0.02562,((5822:0.02039,(148930:0.00967,(397618:0.02014,5826:0.01003)0.949.42:0.00907)0.873.24:0.00832)0.314.3:0.00517,77962:0.01557)0.998.158:0.01826)0.543.2:0.00688)0.820.7:0.00221,((257450:0.00706,(140574:0.0116,135282:0.0147)0.673.3:0.0055)0.956.46:0.0079,(((((69921:0.01744,(242651:0.01962,5781:0.02394)0.913.34:0.00914)0.970.42:0.01263,140619:0.01682)0.976.59:0.01248,169231:0.02066)0.946.42:0.00814,(111058:0.0154,((239119:0.01775,((135896:0.012,(347949:0.01625,537679:0.00998)0.963.55:0.00838)0.990.67:0.01271,((336186:0.0197,259482:0.01202)0.977.55:0.01088,114720:0.01317)0.795.15:0.00215)0.861.27:0.00416)0.882.27:0.00359,((316923:0.02142,159684:0.01957)0.994.102:0.01555,(343990:0.02904,(167723:0.01108,(254341:0.00176,250733:0.02739)0.921.42:0.02463)0.935.28:0.00757)0.821.9:0.00537)0.649.5:0.00532)0.786.18:0.00308)0.796.13:0.00223)0.993.79:0.01082,((44392:0.00275,(239882:0.01871,30868:0.00351)0.535.5:0.00574)0.965.44:0.01102,237272:0.01025)0.817.19:0.00564)0.627.5:0.00606)0.964.63:0.00678)0.986.64:0.00877)0.816.15:0.00231,(140881:0.02393,451932:0.00296)0.964.64:0.00773)0.889.30:0.00379,((215772:0.02635,(254618:0.01937,534626:0.00985)1.000.870:0.00755)0.990.68:0.01195,556328:0.01484)0.979.60:0.00824)0.993.80:0.01007)0.904.34:0.00332)0.962.49:0.00569)0.955.53:0.00599)0.790.10:0.00384)0.987.92:0.01031)0.867.24:0.00431)0.503.6:0.00278)0.986.65:0.00818,(((139434:0.03058,((114675:0.01803,272692:0.02078)0.975.54:0.01142,(271240:0.01982,306945:0.0156)0.385.6:0.00206)0.936.34:0.00529)0.903.30:0.00324,((234741:0.00258,154128:0.00085)0.958.56:0.02881,((((562774:0.02326,((242666:0.01396,251424:0.03138)0.526.3:0.0031,((150257:0.00951,589505:0.01149)0.977.56:0.00883,(50766:0.00081,182573:0.00347)0.000.53:0.01773)0.846.17:0.00392)0.886.23:0.00401)0.965.45:0.0074,(154776:0.01219,313948:0.01049)0.986.66:0.01184)0.971.50:0.006,(251365:0.01415,((328931:0.06466,150457:0.00187)0.370.4:0.01472,((534136:0.00327,129026:0.02286)1.000.871:0.00016,(155367:0.01713,536235:0.01724)0.900.31:0.00828)0.909.31:0.00497)0.749.11:0.01384)0.996.119:0.00895)0.852.30:0.00169,365829:0.02769)0.654.4:0.00316)0.974.57:0.00569)0.876.27:0.00184,((((435644:0.01134,311767:0.01927)0.981.60:0.01204,((((172275:0.01654,((353196:0.02811,314415:0.01356)0.943.54:0.01053,(238470:0.00604,334773:0.0257)0.330.3:0.00301)0.961.47:0.01103)0.830.12:0.00465,165349:0.01454)0.990.69:0.01047,(248590:0.02242,(308804:0.00898,(172912:0.03134,365662:0.00099)0.945.38:0.00353)0.980.58:0.00746)0.918.39:0.00339)0.290.5:0.00014,(305694:0.01172,(275025:0.01567,554782:0.00311)0.966.49:0.01189)0.999.249:0.01167)0.987.93:0.00966)0.898.33:0.00412,235342:0.01931)0.938.40:0.00813,333817:0.01975)0.768.18:0.00129)0.983.71:0.00527)0.117.5:0.00014,(546187:0.01985,(348315:0.00856,19540:0.01089)0.905.34:0.00804)0.997.134:0.01288)0.986.67:0.00593)0.914.27:0.00429)0.938.41:0.00381,((((251266:0.01236,((32029:0.00109,5675:0.00242)0.780.11:0.00469,545124:0.02758)0.903.31:0.00442)0.850.22:0.00257,(103399:0.01437,(((313991:0.02261,168140:0.00386)0.918.40:0.0065,254991:0.00376)0.404.5:0.02213,236240:0.00404)0.966.50:0.007)0.762.17:0.00129)0.990.70:0.01192,((219093:0.00739,131552:0.00207)0.932.38:0.00946,(133336:0.02197,(135996:0.0335,133856:0.01454)0.731.7:0.00717)0.942.44:0.01329)0.928.32:0.00823)0.911.34:0.00703,(160768:0.02903,574171:0.00254)0.999.250:0.0194)0.948.45:0.00504)0.967.42:0.00623)0.990.71:0.01285,(308867:0.02169,(((220518:0.00835,543161:0.02811)0.227.4:0.00857,102398:0.021)0.960.63:0.00813,592117:0.00638)0.888.18:0.00692)0.761.18:0.00521)0.326.8:0.00614)0.240.4:0.00274,236326:0.00976)0.238.3:0.00429)0.997.135:0.03681)1.000.872:0.01591,350516:0.02362)0.916.38:0.0065,((((((103986:0.0121,237879:0.01257)0.967.43:0.00545,(5770:0.00659,(140910:0.05875,160854:0.00494)0.790.11:0.01731)0.902.23:0.00278)0.922.34:0.00334,((527854:0.02426,(164471:0.00489,(202277:0.00181,80137:0.00975)0.970.43:0.02824)0.761.19:0.00154)0.860.34:0.00368,(106521:0.00426,200338:0.0145)0.302.5:0.00547)0.811.16:0.00178)0.827.13:0.00183,((((47316:0.01147,(((510091:0.00885,(155546:0.0078,235419:0.0141)0.970.44:0.00461)0.725.7:0.00087,(163919:0.00532,245648:0.00969)0.852.31:0.01174)0.906.27:0.00999,(172486:0.01582,((519944:0.00865,509296:0.01334)0.982.70:0.00668,((((258406:0.00953,((5740:0.00453,172587:0.01142)0.970.45:0.00733,(323776:0.0118,(560743:0.01107,(44012:0.01001,244407:0.01184)0.979.61:0.00898)0.402.2:0.00429)0.879.31:0.00241)0.931.37:0.00373)0.912.26:0.00401,(78216:0.0045,((196204:0.00153,223075:0.01644)0.834.21:0.00083,(159464:0.015,71366:0.0261)0.903.32:0.01708)0.998.159:0.01116)0.259.5:0.00254)0.805.18:0.00384,(130794:0.01232,225243:0.01581)0.361.2:0.00499)0.864.33:0.00283,(532259:0.0041,394641:0.00734)0.663.3:0.00079)0.868.31:0.00167)0.859.25:0.00453)0.819.19:0.00359)0.812.19:0.002)0.845.19:0.00172,(494490:0.02175,(529216:0.00316,5779:0.01028)0.575.5:0.00234)0.912.27:0.0128)0.923.34:0.00345,(161602:0.10929,165945:0.00967)0.871.38:0.00252)0.976.60:0.00832,(533606:0.01225,5737:0.02223)0.996.120:0.01945)0.787.12:0.006)0.998.160:0.01416,((((((((560393:0.01534,((353031:0.01394,(201888:0.02153,5793:0.00863)0.314.4:0.00875)1.000.873:0.00016,325153:0.01175)0.775.13:0.00559)1.000.874:0.01784,214016:0.02369)0.995.107:0.0083,((532163:0.01886,526241:0.00028)0.918.41:0.0001,580990:0.03716)0.973.54:0.00488)0.621.4:0.00015,(((((((83643:0.01819,516703:0.00833)0.991.73:0.00991,(521994:0.01211,559361:0.00986)0.773.14:0.00246)0.861.28:0.00774,((((201726:0.01069,558958:0.01925)0.948.46:0.00842,(344279:0.01682,(((5802:0.00918,(225039:0.01857,562530:0.00155)0.622.3:0.00173)0.975.55:0.0042,321409:0.01931)0.973.55:0.00502,((67721:0.01057,(49391:0.00665,552274:0.01109)0.934.39:0.00436)0.887.27:0.0025,512277:0.02771)0.962.50:0.00457)0.087:0.00164)0.777.14:0.00162)0.971.51:0.00562,((201411:0.00884,267759:0.01108)0.757.17:0.00131,(199683:0.019,27188:0.01668)0.933.41:0.00327)0.892.24:0.00243)0.987.94:0.00671,240664:0.01011)0.198.2:0.00299)0.126.3:0.00897,(((182749:0.02428,548853:0.01138)1.000.875:0.01762,591092:0.01636)0.898.34:0.00563,571663:0.0108)0.125:0.00014)0.218.5:0.00274,277653:0.0174)0.996.121:0.00879,(569276:0.00616,(239689:0.00374,246019:0.0019)1.000.876:0.0198)0.850.23:0.00649)0.964.65:0.01983,((591654:0.01317,(568966:0.01389,(108320:0.00905,((535970:0.01087,238148:0.01445)0.891.33:0.0064,136291:0.02188)0.947.43:0.00437)0.804.12:0.00199)0.979.62:0.0069)0.376.5:0.00414,577267:0.01735)0.938.42:0.0046)0.990.72:0.00853)0.991.74:0.01516,115461:0.02452)0.496.3:0.00457,((207949:0.0117,(308558:0.01014,347153:0.00593)0.895.33:0.00434)0.849.15:0.00823,(172090:0.0046,((((323018:0.00508,(159870:0.00679,239526:0.01813)0.986.68:0.00971)0.987.95:0.0089,((140189:0.01707,533002:0.05282)0.958.57:0.00868,(551287:0.02136,(92016:0.01558,(210036:0.01961,355768:0.01564)0.841.18:0.01033)0.606.7:0.00556)0.967.44:0.00713)0.950.61:0.00579)0.726.10:0.00246,256939:0.02394)0.519.3:0.00483,294040:0.01245)0.966.51:0.00778)0.604.5:0.00534)0.875.33:0.00487)0.945.39:0.00545,(351622:0.01806,(((163121:0.013,(((((167222:0.02987,565474:0.00948)0.642.7:0.00015,(560005:0.02449,167766:0.0177)0.731.8:0.0169)0.989.72:0.00732,(233567:0.02551,593578:0.0134)0.999.251:0.01776)0.870.29:0.00174,(((((114095:0.00081,243032:0.02385)0.832.16:0.00905,560116:0.02167)0.927.45:0.00514,321297:0.01209)0.393.4:0.00311,(218406:0.01019,((593437:0.0111,(250408:0.00983,(251897:0.02108,99632:0.00962)0.978.65:0.00756)0.767.28:0.00163)0.940.43:0.00364,(254380:0.0055,314122:0.0119)0.879.32:0.00263)0.991.75:0.00698)0.830.13:0.00256)0.988.90:0.0101,568872:0.01332)0.956.47:0.01596)0.999.252:0.00952,((344892:0.0133,220130:0.01609)0.797.15:0.0033,(5780:0.01548,522283:0.01398)0.957.64:0.00551)0.960.64:0.0047)0.566.5:0.00014)0.758.16:0.00463,(263307:0.0352,100625:0.00571)0.907.31:0.00973)0.873.25:0.00428,(194746:0.00941,528484:0.02836)0.470.5:0.00696)0.169.3:0.00112)0.948.47:0.0054)0.892.25:0.00428,((((73883:0.00564,(62508:0.00183,148330:0.01428)0.981.61:0.00722)0.477.4:0.00276,(5753:0.01523,352619:0.00364)0.982.71:0.00843)0.998.161:0.01441,((156883:0.01327,156929:0.01833)0.994.103:0.01304,(332691:0.02194,(156734:0.00149,532993:0.01289)0.992.95:0.01748)0.802.11:0.00648)0.895.34:0.00489)0.978.66:0.00816,(307203:0.01639,(156780:0.02766,156862:0.01066)0.893.23:0.0032)0.974.58:0.00681)0.904.35:0.004)0.416.5:0.00563)0.942.45:0.00713,(555568:0.01222,(221018:0.01965,(5828:0.00218,((225594:0.00043,(519755:0.00739,514342:0.01731)1.000.877:0.01319)0.999.253:0.01659,291146:0.01679)0.994.104:0.01264)0.980.59:0.01363)0.830.14:0.011)0.735.11:0.00402)0.987.96:0.01324)0.993.81:0.01322,(((252028:0.02097,141539:0.04376)0.988.91:0.01879,316185:0.00518)0.579.3:0.03144,((((298881:0.02562,(263491:0.01479,296183:0.03818)0.336.7:0.00434)0.904.36:0.00618,(561259:0.02408,((581650:0.03262,((254434:0.01407,(560353:0.02209,464901:0.01003)0.971.52:0.008)0.977.57:0.01099,(316190:0.02352,(167302:0.02756,154801:0.01926)0.927.46:0.00945)0.873.26:0.00705)0.961.48:0.00812)0.888.19:0.00372,((5804:0.01638,((570345:0.01258,548610:0.01307)0.916.39:0.00459,(((535801:0.02541,309713:0.0065)0.970.46:0.00875,(501917:0.01045,(((524932:0.00666,530994:0.01654)0.961.49:0.00946,(151324:0.01881,138291:0.00673)0.520.4:0.00804)0.960.65:0.00767,518302:0.00565)0.968.49:0.0074)0.849.16:0.00283)0.884.34:0.00553,(580499:0.01336,(240459:0.00587,339119:0.01492)0.977.58:0.00937)0.944.32:0.008)0.895.35:0.00547)0.861.29:0.00386)0.595.2:0.00315,519237:0.01644)0.948.48:0.01002)0.872.41:0.00385)0.862.35:0.00271)0.951.46:0.00721,82177:0.03652)0.329.4:0.00725,((306401:0.01189,(80906:0.03311,(193660:0.01343,(((5726:0.02388,((349062:0.01297,(335432:0.01629,230067:0.01576)0.763.19:0.00214)0.969.57:0.00705,512526:0.00495)0.890.31:0.01209)0.166.3:0.01217,((586757:0.03494,519840:0.00088)1.000.878:0.02204,(584890:0.02557,278420:0.01614)0.988.92:0.01235)0.802.12:0.01342)0.710.8:0.01194,541086:0.03227)0.972.54:0.00994)0.956.48:0.00902)0.940.44:0.01017)0.866.20:0.00477,(200964:0.03129,163280:0.01831)0.992.96:0.01432)0.982.72:0.01243)0.858.19:0.00805)0.950.62:0.0109)0.737.6:0.00505)1.000.879:0.03501,((185425:0.06481,530853:0.06138)0.990.73:0.02422,200355:0.07183)0.976.61:0.01793)0.945.40:0.00913)0.920.26:0.00482,(((((273081:0.01198,((245498:0.01145,310212:0.0122)0.960.66:0.01014,259452:0.01585)0.744.14:0.00818)0.962.51:0.00832,312798:0.01078)0.836.21:0.00314,269733:0.01218)0.131.2:0.00368,(514057:0.03261,565524:0.01534)0.687.6:0.00269)0.997.136:0.01704,(521541:0.05252,268065:0.00417)0.971.53:0.03488)0.992.97:0.01298)0.933.42:0.00827,((((((532443:0.03842,((565583:0.01086,241958:0.01822)0.995.108:0.02181,(((332051:0.015,559803:0.01062)0.998.162:0.01652,(307689:0.00635,5863:0.01432)0.998.163:0.01502)0.142.4:0.00917,(((5875:0.01419,169804:0.00097)0.873.27:0.00431,154213:0.01293)0.995.109:0.01793,((151886:0.00639,140456:0.01663)0.816.16:0.00403,((101881:0.00142,5856:0.01169)0.919.30:0.00934,(100847:0.01397,165618:0.00514)0.996.122:0.01639)0.794.16:0.00287)0.962.52:0.01316)0.115.2:0.00597)0.992.98:0.02186)0.968.50:0.01794)0.992.99:0.02089,((111797:0.03038,(509163:0.02569,((18609:0.01024,((347767:0.01787,(546491:0.01748,(303607:0.02485,590926:0.03589)0.991.76:0.01446)0.558.5:0.00481)0.985.58:0.0092,(191430:0.02401,243897:0.0215)0.796.14:0.00708)0.919.31:0.00015)0.671.5:0.00404,211469:0.01008)0.997.137:0.01939)0.938.43:0.01677)0.999.254:0.02753,(519257:0.01121,530619:0.02145)0.999.255:0.03017)0.208.3:0.00981)0.411.5:0.00891,((((105062:0.01996,337659:0.01535)0.997.138:0.01792,((((286163:0.01646,27292:0.00403)0.942.46:0.00632,(113078:0.00109,5850:0.00016)0.959.44:0.01744)0.655.4:0.00376,(279285:0.00564,135262:0.00095)0.997.139:0.01527)0.312.2:0.00648,355130:0.01262)0.990.74:0.01439)0.838.21:0.00557,(102145:0.0359,((108547:0.01792,149505:0.03062)0.971.54:0.01697,(593605:0.00885,(222013:0.01924,5847:0.0169)0.869.38:0.00986)0.906.28:0.0103)0.993.82:0.01611)0.975.56:0.01501)0.976.62:0.01174,(556189:0.04405,(222908:0.02222,(((344513:0.00675,((185015:0.01936,250911:0.01681)0.525.3:0.00728,148699:0.03884)0.947.44:0.00834)0.976.63:0.01146,(330270:0.00937,326364:0.01857)0.942.47:0.00853)0.755.19:0.00126,(118311:0.00175,514038:0.00174)0.975.57:0.01829)0.989.73:0.01425)1.000.880:0.02624)0.993.83:0.01932)0.906.29:0.01447)0.980.60:0.01794,(((561524:0.00937,(90551:0.02008,(101219:0.00105,544221:0.01663)1.000.881:0.02296)0.821.10:0.00842)0.967.45:0.01213,226147:0.0142)0.997.140:0.02383,(318593:0.03334,(111917:0.02208,(80157:0.0168,261739:0.0141)0.992.100:0.01598)0.996.123:0.02751)0.671.6:0.0133)1.000.882:0.03774)0.905.35:0.01427,(586778:0.06402,(((70005:0.01488,(5563:0.0147,(93202:0.00192,104117:0.00359)0.897.28:0.0229)0.963.56:0.00981)0.975.58:0.01275,((272478:0.01088,573162:0.01286)0.482.5:0.00408,(171288:0.01902,554330:0.01414)0.997.141:0.02098)0.918.42:0.01224)0.818.8:0.00933,((((((336655:0.01082,355315:0.01496)0.912.28:0.00586,(154562:0.0145,(305547:0.01421,(257725:0.00465,346735:0.11254)0.920.27:0.01526)0.904.37:0.00697)0.986.69:0.00715)0.904.38:0.00238,((241110:0.01463,((5621:0.00163,5600:0.00236)0.992.101:0.00578,(90161:0.01581,89707:0.0039)0.725.8:0.01126)0.986.70:0.01041)0.848.23:0.00947,87601:0.01946)0.834.22:0.00152)0.884.35:0.01127,((110830:0.00756,139814:0.03472)0.860.35:0.00829,((509047:0.02625,260289:0.03933)0.343.4:0.00424,(588010:0.03884,250076:0.07587)0.859.26:0.02114)0.882.28:0.00183)0.948.49:0.0094)0.863.29:0.00841,5598:0.01469)0.988.93:0.01038,(((565413:0.00883,551851:0.01607)0.967.46:0.00628,(346819:0.02817,(((324088:0.01041,(587634:0.02037,568385:0.01553)0.999.256:0.02095)0.963.57:0.00552,((210590:0.02199,226283:0.01965)0.433.3:0.00635,231965:0.01249)0.829.18:0.00398)0.890.32:0.00194,(((((108986:0.00408,219048:0.01493)0.955.54:0.02312,((5508:0.01596,(46818:0.02442,(32988:0.00408,112129:0.01991)0.930.39:0.00242)0.847.22:0.00973)0.752.15:0.0016,285596:0.02403)0.628.8:0.0032)0.932.39:0.00336,(190723:0.00397,182027:0.00627)0.987.97:0.0104)0.866.21:0.00245,5552:0.02711)0.918.43:0.00015,(199553:0.01284,111099:0.01878)0.866.22:0.0049)0.999.257:0.01279)0.200.6:0.00222)0.746.15:0.00072)0.938.44:0.00397,(((240075:0.0315,(227434:0.01528,(345344:0.01791,(168212:0.00313,30983:0.04921)0.885.27:0.01732)0.558.6:0.001)0.614.3:0.00433)0.983.72:0.00964,(207510:0.01799,54353:0.01433)0.948.50:0.00626)0.918.44:0.00494,((538121:0.01918,(5554:0.01267,(165771:0.00384,568154:0.01579)0.839.21:0.00428)0.949.43:0.00427)0.832.17:0.00231,(316655:0.0242,((((106146:0.0094,(334299:0.00349,345034:0.00424)0.723.9:0.00785)0.978.67:0.00565,280403:0.01443)0.776.18:0.00156,164042:0.0152)0.944.33:0.00924,270365:0.00197)0.981.62:0.0123)0.730.8:0.00252)0.131.3:0.0022)0.878.35:0.00261)0.989.74:0.00942)0.838.22:0.00835)1.000.883:0.03449)0.888.20:0.01616)0.753.25:0.01061,(((((590507:0.00353,568425:0.00852)0.890.33:0.04056,(560382:0.03182,((362188:0.00161,562969:0.00174)1.000.884:0.00428,((511230:0.00733,(106821:0.02937,532912:0.01088)0.900.32:0.00579)0.447.4:0.00103,29952:0.01542)0.711.7:0.01314)0.895.36:0.01464)1.000.885:0.04736)0.994.105:0.02631,((((544302:0.01432,543839:0.01583)0.676.5:0.00419,((510799:0.04972,576858:0.02063)0.885.28:0.00594,((214914:0.00971,156154:0.00973)0.959.45:0.00765,534416:0.01887)0.760.19:0.00543)0.803.11:0.00458)0.988.94:0.01533,((153188:0.01139,((211555:0.00813,177583:0.00258)0.868.32:0.00649,561842:0.0329)0.937.46:0.00971)0.922.35:0.00975,217516:0.01686)0.969.58:0.01625)0.493.2:0.00555,(((555926:0.01648,542729:0.0186)0.989.75:0.01275,(((((234209:0.00852,(555322:0.024,158741:0.0186)0.919.32:0.00392)0.999.258:0.02666,(257891:0.01927,(350929:0.01399,(114227:0.00463,589812:0.0064)0.991.77:0.01369)0.383:0.00721)0.870.30:0.004)0.887.28:0.00335,(563484:0.01414,215343:0.02518)0.927.47:0.00635)0.996.124:0.01112,346206:0.01533)0.958.58:0.00891,(315241:0.02623,(141079:0.00805,(104521:0.01033,261055:0.01809)0.976.64:0.01044)0.900.33:0.00904)0.993.84:0.02032)0.983.73:0.01478)0.998.164:0.0177,204148:0.02719)0.977.59:0.01266)1.000.886:0.03019)0.982.73:0.01956,(((((252811:0.01416,(555718:0.01406,(254247:0.0171,93834:0.00646)0.893.24:0.00363)0.436.6:0.00382)1.000.887:0.02704,((163816:0.0044,206331:0.00514)1.000.888:0.05323,(((182626:0.01504,244312:0.01994)1.000.889:0.02451,((580777:0.01587,244972:0.01439)1.000.890:0.03165,((240281:0.01551,(578073:0.03368,103846:0.02613)0.966.52:0.01083)0.982.74:0.01244,((102414:0.01101,256677:0.01651)1.000.891:0.03778,((367617:0.01809,(171372:0.02074,154063:0.01084)0.460.6:0.00481)0.932.40:0.00516,340065:0.0213)0.998.165:0.01658)0.325.3:0.00539)0.994.106:0.01513)0.746.16:0.00494)0.488.7:0.00729,(370391:0.0334,((533126:0.00647,92371:0.03773)0.968.51:0.01254,546204:0.00935)0.992.102:0.02391)0.986.71:0.02)0.722.8:0.00503)1.000.892:0.02451)0.990.75:0.01616,((225468:0.00014,172062:0.00178)1.000.893:0.04893,(((573535:0.02697,(542969:0.0104,199150:0.00351)0.476.3:0.0191)0.999.259:0.02316,(174833:0.05363,(50973:0.03288,83629:0.02118)0.766.17:0.01515)0.952.49:0.01068)0.838.23:0.01408,(138013:0.01668,138266:0.01327)1.000.894:0.02988)0.923.35:0.00988)0.758.17:0.00642)0.533:0.00386,((156484:0.0463,(567827:0.01489,(574275:0.02081,250481:0.01719)0.911.35:0.01122)0.998.166:0.02892)0.990.76:0.01942,((((239904:0.0182,511848:0.02449)0.999.260:0.02678,(((((((200287:0.02867,545938:0.00162)0.325.4:0.02912,((63607:0.00014,181641:0.00014)0.916.40:0.03838,(561066:0.03509,(((589988:0.00275,256953:0.00563)0.908.34:0.00507,(148276:0.0264,25832:0.0062)1.000.895:0.04328)1.000.896:0.03104,(582926:0.00803,329326:0.00772)0.997.142:0.01865)0.070.2:0.00461)0.919.33:0.00783)0.930.40:0.01314)1.000.897:0.02052,(581688:0.02151,(572937:0.0222,313413:0.03754)0.997.143:0.01812)0.995.110:0.01756)0.369.3:0.00673,((282361:0.03459,(((106889:0.00876,4574:0.0115)0.752.16:0.02448,((279352:0.01448,278959:0.00259)0.925.28:0.00428,(91561:0.01246,310439:0.02466)0.076:0.0017)0.987.98:0.01229)0.931.38:0.00979,(((((36320:0.0231,(265895:0.00859,317373:0.01763)0.852.32:0.00441)0.993.85:0.01746,(((108446:0.01936,4583:0.00342)0.772.13:0.00667,349161:0.01864)0.990.77:0.01317,((115653:0.00877,228168:0.01152)0.968.52:0.00986,(4585:0.01318,(342146:0.02953,545263:0.00029)1.000.898:0.03307)0.467.4:0.00508)0.994.107:0.01737)0.663.4:0.00846)0.883.29:0.01389,(103709:0.03557,(338883:0.02725,(155692:0.00375,335943:0.02117)0.848.24:0.01918)1.000.899:0.03369)0.922.36:0.01795)0.994.108:0.02475,336443:0.03171)0.951.47:0.01121,(135990:0.02955,(577466:0.00614,(544803:0.02602,(200772:0.00016,279093:0.00104)0.999.261:0.01657)0.922.37:0.00667)0.951.48:0.00816)0.997.144:0.01355)0.733.14:0.00142)0.994.109:0.01781)0.954.42:0.01439,(584420:0.00354,60164:0.00903)1.000.900:0.04987)0.310.4:0.00435)0.801.12:0.00397,(((133137:0.03297,(322844:0.02703,(136616:0.00941,(356548:0.00675,156957:0.00791)0.999.262:0.0159)0.892.26:0.00605)0.562.4:0.00932)1.000.901:0.03012,(((304307:0.03238,(((138716:0.03847,5104:0.0547)0.988.95:0.01704,(545476:0.01946,(336148:0.03789,165418:0.01584)0.642.8:0.00972)1.000.902:0.02702)0.808.13:0.00838,(((336318:0.03704,342034:0.02907)0.596.5:0.00897,(156350:0.00844,516834:0.0205)1.000.903:0.0246)0.181.2:0.00814,304859:0.02794)0.273.2:0.00998)0.914.28:0.00884)0.925.29:0.01015,(((238374:0.01623,190304:0.01484)1.000.904:0.0287,((323748:0.01914,(325774:0.00735,(211365:0.00453,567782:0.00837)0.367.2:0.00265)0.954.43:0.00768)0.991.78:0.0117,(176117:0.01079,(574536:0.01396,552039:0.01065)0.974.59:0.00781)0.984.65:0.01001)0.764.17:0.00422)0.901.28:0.00551,((226756:0.0213,294380:0.01876)1.000.905:0.02808,((279075:0.00924,(217511:0.01779,340730:0.00833)0.989.76:0.01414)0.991.79:0.01314,(543882:0.0424,(277380:0.01673,565409:0.01815)0.975.59:0.0099)0.723.10:0.00499)0.793.11:0.00258)0.993.86:0.01197)0.939.42:0.0077)0.910.36:0.00566,(334647:0.03223,(271388:0.02448,(((247400:0.01643,(((263755:0.01073,269183:0.01735)0.455.2:0.00083,222549:0.0214)0.881.30:0.00261,267913:0.01986)0.903.33:0.00322)0.866.23:0.00195,266612:0.02134)0.869.39:0.00277,271998:0.02277)0.056.2:0.00383)0.981.63:0.00936)0.741.13:0.00545)0.899.33:0.00618)0.939.43:0.01152,((((542689:0.03354,572538:0.00805)0.999.263:0.01895,(267833:0.01245,((330657:0.0142,(60649:0.01616,272622:0.00894)0.925.30:0.00721)0.990.78:0.01285,140105:0.01979)0.927.48:0.00626)0.992.103:0.01471)0.988.96:0.01681,((593864:0.00422,218252:0.02957)0.642.9:0.04045,(233820:0.02928,(230514:0.00879,559711:0.0069)1.000.906:0.03744)0.942.48:0.01029)0.775.14:0.00857)0.966.53:0.01287,((278603:0.01692,(240749:0.02405,(319668:0.00718,250376:0.02451)0.929.43:0.00576)0.201.3:0.00312)0.992.104:0.01571,((531406:0.02614,(155886:0.0105,237235:0.0197)0.943.55:0.01755)0.992.105:0.02222,(538178:0.04665,(356610:0.01397,(307417:0.0087,(113643:0.00875,(354033:0.02388,326326:0.00773)0.787.13:0.00345)0.885.29:0.00816)0.904.39:0.00592)1.000.907:0.03168)0.865.30:0.00832)0.963.58:0.01188)0.521.4:0.00627)0.962.53:0.01066)0.974.60:0.01064)0.988.97:0.00871,((242336:0.03697,(541154:0.00837,(253590:0.00559,(182490:0.01279,222179:0.01751)0.426.6:0.00076)0.999.264:0.01156)1.000.908:0.03018)0.904.40:0.00919,((((((586348:0.00269,554299:0.00014)1.000.909:0.05276,(252376:0.02334,220224:0.01659)0.904.41:0.00977)0.957.65:0.01652,(((163171:0.02679,(220178:0.02455,91010:0.03539)0.986.72:0.01461)0.972.55:0.01121,(((271304:0.01733,(235586:0.00924,345110:0.00848)1.000.910:0.04247)0.991.80:0.01327,(((58862:0.03068,4546:0.00534)0.999.265:0.01455,326714:0.01015)1.000.911:0.03566,(260920:0.00732,((4572:0.01746,((520511:0.0053,((4552:0.02079,163174:0.00028)1.000.912:0.00411,63343:0.00868)0.929.44:0.00338)0.933.43:0.01297,(4586:0.00933,((75835:0.01031,(151417:0.0074,((151705:0.00185,4563:0.01333)0.948.51:0.00406,(152173:0.00732,131955:0.00552)0.893.25:0.00296)0.514.2:0.00236)0.747.12:0.00136)0.949.44:0.00503,(344876:0.01405,192269:0.01194)0.434.5:0.00249)0.897.29:0.00327)0.946.43:0.00453)0.944.34:0.00015)0.928.33:0.00505,(107382:0.01593,115023:0.02166)0.994.110:0.01281)0.225.6:0.00048)1.000.913:0.02421)0.944.35:0.00881)0.702.9:0.00534,(217798:0.03753,(252146:0.00353,(165020:0.01437,306356:0.00741)0.898.35:0.00412)0.235:0.00517)0.992.106:0.01568)0.956.49:0.00945)0.619.5:0.00591,32486:0.06627)0.772.14:0.00956)0.986.73:0.01168,(((94055:0.0219,(252132:0.02651,564942:0.0085)0.998.167:0.01736)0.989.77:0.01068,((235349:0.02741,552221:0.03138)0.939.44:0.00577,((211514:0.01443,((204935:0.02231,111976:0.00402)0.981.64:0.00482,((225101:0.00579,591569:0.00814)0.192.3:0.00156,331818:0.00224)0.747.13:0.00259)0.964.66:0.00659)0.997.145:0.01496,((197564:0.01092,284226:0.01709)0.967.47:0.00952,177607:0.02855)0.232.4:0.00198)0.967.48:0.00777)0.387.7:0.00259)0.939.45:0.00588,(345742:0.014,(((248751:0.01147,361107:0.00354)0.095.2:0.00095,566368:0.01776)0.991.81:0.01393,((154249:0.03689,(58135:0.00862,(513189:0.00234,262768:0.02447)0.972.56:0.01042)0.996.125:0.01652)0.804.13:0.00519,130838:0.01167)0.983.74:0.01463)0.999.266:0.0218)1.000.914:0.02137)0.423.5:0.00393)0.064.2:0.00206,((60574:0.01241,244241:0.01328)1.000.915:0.02851,(565880:0.03896,(322226:0.05525,(((257403:0.01597,256883:0.01193)0.992.107:0.01542,(493677:0.01228,357711:0.00928)0.988.98:0.01559)0.983.75:0.01428,((((113141:0.01129,113576:0.01747)0.975.60:0.01194,((154486:0.02407,33613:0.00876)0.964.67:0.01139,(4588:0.02343,573699:0.01207)0.991.82:0.01381)0.910.37:0.00902)0.990.79:0.01092,(255508:0.00953,534431:0.01081)0.299.3:0.00321)0.963.59:0.00712,((((567063:0.02995,(((246579:0.01227,581496:0.00712)1.000.916:0.01421,((113710:0.01495,590727:0.00907)0.942.49:0.00652,((226012:0.00642,365796:0.01147)1.000.917:0.01736,369734:0.01476)0.521.5:0.00343)0.980.61:0.00808)0.888.21:0.0031,((241292:0.0152,101876:0.0178)0.824.10:0.00466,78724:0.01709)0.847.23:0.00225)0.867.25:0.00501)0.978.68:0.00935,(76700:0.02188,250075:0.01123)0.998.168:0.01375)0.575.6:0.00218,((((244994:0.00855,(227319:0.0127,(225294:0.01644,250043:0.00948)0.983.76:0.01431)0.752.17:0.00504)0.943.56:0.00819,((292508:0.02608,((114006:0.01382,(114037:0.00761,(114085:0.02154,113512:0.01483)0.849.17:0.00639)0.806.12:0.00483)0.899.34:0.00467,112912:0.01526)0.855.23:0.00323)0.841.19:0.0036,(559267:0.00854,156709:0.00358)1.000.918:0.01702)0.338.4:0.0028)0.541.5:0.00426,((((225546:0.01708,210561:0.01037)0.951.49:0.00822,(93503:0.00839,(362845:0.00177,93985:0.02508)0.913.35:0.00423)0.947.45:0.0093)0.877.26:0.00876,542363:0.01147)0.930.41:0.00887,585228:0.01987)0.924.32:0.00822)0.964.68:0.00941,45882:0.047)0.962.54:0.0128)0.886.24:0.00516,((160882:0.01506,113965:0.01279)0.937.47:0.00816,(553482:0.01471,(514040:0.0084,(160071:0.00913,((210884:0.01216,152533:0.01304)0.071.2:0.00287,218380:0.00681)0.976.65:0.00711)0.970.47:0.00826)0.392.4:0.00356)0.989.78:0.01047)0.304.3:0.00429)0.864.34:0.006)0.971.55:0.00804)0.918.45:0.00809)0.721.6:0.00687)0.781.18:0.01031)0.331.7:0.00493)0.554.4:0.00796,(((232270:0.06038,((138667:0.01646,(568244:0.0199,(222372:0.02294,(560931:0.01096,550141:0.01174)0.995.111:0.01532)0.998.169:0.01914)0.731.9:0.01082)0.844.18:0.00867,(147088:0.01703,94511:0.03501)0.955.55:0.01102)0.928.34:0.01485)1.000.919:0.03886,(211913:0.06133,(509529:0.03056,((557664:0.01886,236397:0.01395)1.000.920:0.02762,(242184:0.02999,(256291:0.02129,576693:0.013)0.999.267:0.02182)0.914.29:0.00953)0.471.5:0.00371)0.981.65:0.01721)0.904.42:0.01094)0.587.4:0.0025,(25641:0.02909,((217201:0.02329,(331647:0.02419,((336803:0.02614,161246:0.01148)0.745.15:0.00401,((548127:0.02994,588604:0.01168)0.857.24:0.00912,357011:0.00557)0.784.13:0.00471)0.924.33:0.00598)0.743.22:0.00876)1.000.921:0.05075,(575614:0.02606,((26566:0.01181,(511480:0.01204,565179:0.00739)0.864.35:0.009)0.997.146:0.0229,537674:0.03371)0.652.6:0.0083)0.839.22:0.00715)0.968.53:0.01911)0.967.49:0.01183)0.965.46:0.00832)0.953.45:0.00821)0.935.29:0.00622)0.907.32:0.00398,(((329389:0.00472,(304619:0.02905,341445:0.0076)0.935.30:0.01074)1.000.922:0.03232,(((259136:0.00786,584854:0.00972)0.975.61:0.00912,340767:0.02873)0.316.5:0.00564,((558770:0.01848,512990:0.01103)0.851.22:0.00663,338740:0.02765)0.666.6:0.00371)0.999.268:0.01881)0.967.50:0.01027,(((261444:0.01455,571146:0.02089)0.408.10:0.00176,(151653:0.01058,(60160:0.00446,575921:0.00759)0.967.51:0.00632)0.990.80:0.00804)0.878.36:0.00392,289382:0.02329)0.950.63:0.00767)0.987.99:0.01303)0.400.6:0.00482)0.967.52:0.00842,((581302:0.02523,(((203164:0.00733,226468:0.01682)0.925.31:0.06184,((590575:0.01503,152931:0.00444)1.000.923:0.04537,(230033:0.01329,234763:0.0155)1.000.924:0.04506)0.993.87:0.02347)0.980.62:0.01911,(172572:0.05588,551819:0.02831)0.509.4:0.00554)0.930.42:0.00672)0.981.66:0.01017,((199869:0.02031,564010:0.00555)1.000.925:0.02539,((583127:0.01908,268555:0.00863)1.000.926:0.02882,(154449:0.01633,561208:0.00638)0.612.5:0.00345)0.987.100:0.00992)0.993.88:0.01087)0.407.4:0.00429)0.465.3:0.00363,(((((319588:0.03113,548884:0.02155)0.998.170:0.02831,(105948:0.03677,(50080:0.03887,580691:0.00107)0.028.2:0.00999)1.000.927:0.02969)0.454.3:0.0088,(((547681:0.01162,256038:0.0173)1.000.928:0.03252,((104436:0.00014,588899:0.02277)1.000.929:0.04113,(6144:0.00271,570120:0.00267)0.278.3:0.0397)0.544.5:0.01044)0.798.10:0.02006,((249530:0.00883,142475:0.01529)0.242.4:0.00216,((279302:0.0095,(139738:0.0044,328739:0.01205)0.977.60:0.00621)0.932.41:0.00416,245677:0.01981)0.980.63:0.00838)1.000.930:0.04651)0.497.5:0.00927)0.990.81:0.01572,((341281:0.01927,274973:0.02624)0.907.33:0.0065,(275770:0.01425,(240989:0.02487,96012:0.01767)0.998.171:0.01777)0.945.41:0.00622)0.983.77:0.01133)0.872.42:0.004,((273866:0.0238,(60184:0.00642,548705:0.00411)0.970.48:0.0072)0.801.13:0.00452,(138336:0.02408,594295:0.01389)0.992.108:0.01361)0.992.109:0.01427)0.672.2:0.00411)0.973.56:0.00994)0.940.45:0.00976)0.929.45:0.00925,((((217093:0.0392,267119:0.01009)0.996.126:0.02848,((((226512:0.01032,(540097:0.00993,(4404:0.01465,(144291:0.01294,554284:0.00481)0.747.14:0.00383)0.999.269:0.01899)0.990.82:0.01423)0.678.3:0.00428,(79493:0.01467,576328:0.0164)0.937.48:0.0061)0.874.26:0.00363,(103476:0.02999,(((222002:0.02049,(305362:0.00236,248092:0.01701)0.764.18:0.01165)0.797.16:0.00605,167845:0.03301)0.935.31:0.00712,((55243:0.01069,(((((163322:0.00445,359891:0.01191)0.909.32:0.0043,((281197:0.00122,((517152:0.01181,217968:0.00823)0.820.8:0.00267,345664:0.021)0.735.12:0.00849)0.999.270:0.01774,319537:0.01165)0.986.74:0.01162)0.978.69:0.00718,(114073:0.0064,((137794:0.00777,((164098:0.00492,(151603:0.01192,138031:0.01071)0.970.49:0.00798)0.914.30:0.00692,(197174:0.02037,246217:0.01985)0.978.70:0.01556)0.995.112:0.01691)0.358.2:0.005,(163698:0.01294,154116:0.01109)0.609.3:0.00391)0.999.271:0.01592)0.699.5:0.0051)0.741.14:0.0029,(((113068:0.00723,(4413:0.03601,((102348:0.00507,(547078:0.01174,265137:0.01301)0.966.54:0.0063)0.920.28:0.00375,((75850:0.015,(64695:0.03709,563370:0.00978)0.974.61:0.00513)0.832.18:0.00166,42326:0.01114)0.932.42:0.00497)0.999.272:0.01728)0.949.45:0.00813)0.678.4:0.00219,152603:0.0055)0.998.172:0.01476,(573663:0.01324,91688:0.02881)0.251.3:0.00408)0.919.34:0.00573)0.899.35:0.00352,(((269416:0.00921,204768:0.00676)0.799.12:0.01151,(((230389:0.02319,194558:0.01184)0.809.11:0.00703,(348570:0.00181,110099:0.0054)0.909.33:0.01784)0.725.9:0.00118,((177571:0.0129,(88754:0.02744,369436:0.00783)0.352.2:0.00106)0.977.61:0.00944,4416:0.01673)0.785.16:0.00656)0.957.66:0.00555)0.785.17:0.00479,(358386:0.0236,(163716:0.01783,590311:0.00225)0.959.46:0.00585)0.934.40:0.00563)0.974.62:0.00863)0.950.64:0.0045)0.964.69:0.00533,((249778:0.00687,548020:0.00015)0.992.110:0.01146,(314477:0.02127,(211664:0.00643,(394806:0.00848,280239:0.01528)0.961.50:0.00634)0.779.8:0.00387)0.900.34:0.00329)0.644.6:0.00336)0.778.18:0.00341)0.943.57:0.008)0.982.75:0.00944)0.991.83:0.02023,(((222512:0.01303,274248:0.01262)0.629.8:0.01441,(151650:0.00758,((152605:0.01856,((253895:0.01838,250255:0.01353)1.000.931:0.01977,151048:0.01917)0.455.3:0.0066)0.676.6:0.0041,223286:0.01614)0.644.7:0.00583)0.885.30:0.00526)0.247.3:0.00452,((((((((172972:0.01151,138221:0.00992)0.998.173:0.02922,(171696:0.00437,138090:0.00014)1.000.932:0.04643)0.965.47:0.01584,((112235:0.00637,241355:0.01062)0.980.64:0.0113,(177929:0.00562,243689:0.01487)0.987.101:0.02299)0.970.50:0.0113)0.938.45:0.009,((364038:0.00515,451765:0.02139)0.669.6:0.00157,(((360826:0.01164,(591379:0.00469,81029:0.02934)0.499.4:0.00383)0.422.6:0.00429,(304500:0.02048,((217521:0.00334,(532895:0.01288,361359:0.01107)0.693.6:0.00239)0.998.174:0.01437,(513956:0.00898,224956:0.00821)0.957.67:0.00692)0.502.4:0.00409)0.306.4:0.00415)0.807.9:0.00589,(512676:0.01601,(306635:0.03493,((32348:0.00819,(360500:0.01064,328033:0.00536)0.800.12:0.00229)0.948.52:0.00437,(227060:0.00822,152090:0.01442)0.433.4:0.00174)0.314.5:0.00118)0.747.15:0.00267)0.991.84:0.00966)0.998.175:0.01359)0.967.53:0.00635)0.909.34:0.00566,(21795:0.01464,(509496:0.01392,(572984:0.01222,222665:0.02074)0.685.3:0.00469)0.935.32:0.00504)0.249.3:0.00414)0.793.12:0.00197,((112752:0.01159,(136225:0.01624,568243:0.01089)0.665.5:0.00433)0.935.33:0.00564,(136201:0.01367,(573907:0.01179,564690:0.01241)0.918.46:0.00786)0.940.46:0.01107)0.943.58:0.00479)0.572.3:0.00141,((583138:0.00838,203885:0.00924)0.825.15:0.00377,((((((((((229359:0.01732,369229:0.01977)0.962.55:0.0088,128717:0.00521)0.964.70:0.01327,(((156232:0.02226,(240612:0.0218,4421:0.00105)0.000.54:0.00335)0.799.13:0.00309,((548999:0.02091,((241962:0.0055,138760:0.02083)0.879.33:0.00016,136553:0.00727)0.899.36:0.00545)0.757.18:0.00713,(278839:0.0003,(92777:0.00016,249632:0.07212)0.977.62:0.005)0.878.37:0.00159)0.994.111:0.0073)0.906.30:0.00532,91691:0.04405)0.868.33:0.00582)0.979.63:0.00757,((246133:0.01419,255006:0.02622)0.814.10:0.00529,509511:0.02332)0.928.35:0.00661)0.845.20:0.00394,4442:0.01643)0.545.2:0.00456,(247298:0.02176,4470:0.02433)0.815.11:0.00276)0.944.36:0.00627,((8030:0.07495,(462865:0.03686,122816:0.09836)0.998.176:0.04807)0.997.147:0.03618,(4449:0.02905,342971:0.00197)0.887.29:0.00293)0.336.8:0.00244)0.618.6:0.00259,(104162:0.01324,276553:0.01953)1.000.933:0.03235)0.740.18:0.00204,(44239:0.00892,356753:0.01451)0.364.4:0.00438)0.626.4:0.00909,143200:0.01385)1.000.934:0.0191)0.817.20:0.00185)0.582.4:0.00238,(((((263857:0.01726,(220042:0.0099,(141085:0.01781,407742:0.01183)0.991.85:0.01529)0.871.39:0.00643)0.263.6:0.00302,537239:0.0159)0.952.50:0.00907,(223657:0.02997,368986:0.00827)0.961.51:0.00666)0.927.49:0.0066,(360049:0.01446,158130:0.01097)0.871.40:0.00469)0.340.4:0.00237,(((53994:0.00581,((22545:0.0107,((39575:0.00493,266208:0.00349)0.969.59:0.00554,203235:0.00816)0.971.56:0.01287)0.971.57:0.00843,240252:0.00989)0.385.7:0.01713)1.000.935:0.01941,(((266659:0.0072,40217:0.00352)0.999.273:0.0142,((158537:0.00016,4490:0.03361)0.967.54:0.00673,239497:0.01437)0.576.5:0.00453)0.907.34:0.00508,227409:0.01181)0.947.46:0.01112)0.993.89:0.01355,(((43037:0.00632,170219:0.01619)0.929.46:0.00751,(112277:0.0152,((319672:0.00903,(((259611:0.01695,109654:0.01033)0.754.15:0.0007,154314:0.02385)0.883.30:0.00249,230396:0.02319)0.790.12:0.00393)0.905.36:0.00467,(509891:0.01928,((593096:0.00619,154354:0.02025)0.807.10:0.00927,223618:0.01324)0.837.16:0.00363)0.954.44:0.00673)0.784.14:0.00456)0.898.36:0.00015)0.950.65:0.00635,(315855:0.01947,4411:0.02601)0.882.29:0.0045)0.712.4:0.00521)0.920.29:0.00507)0.834.23:0.00371)0.765.16:0.0018)0.983.78:0.01274)1.000.936:0.02776)0.997.148:0.03177,4575:0.05985)0.822.18:0.01331,((((((149141:0.02697,((201801:0.01761,225998:0.00078)0.921.43:0.01445,77884:0.00787)0.982.76:0.01211)0.974.63:0.02472,(581447:0.03548,(145022:0.00918,(68790:0.00153,4515:0.00107)0.989.79:0.02098)0.923.36:0.00822)0.985.59:0.01847)0.957.68:0.01696,(((241639:0.0073,226882:0.00658)0.859.27:0.00463,583557:0.00799)1.000.937:0.03951,(45811:0.03547,((56046:0.00014,245577:0.00103)0.429:0.04106,(87741:0.04429,(240686:0.04059,(282743:0.01901,143736:0.00561)0.964.71:0.01244)0.961.52:0.01124)0.709.6:0.00692)0.936.35:0.01085)0.920.30:0.00995)0.757.19:0.00674)0.808.14:0.01031,((542625:0.0148,(548310:0.01481,318113:0.02129)0.350.5:0.00611)1.000.938:0.02645,(224678:0.04785,(258178:0.03267,(227239:0.00929,317951:0.01654)0.589.3:0.01034)0.735.13:0.00642)1.000.939:0.03095)0.994.112:0.01526)0.978.71:0.01285,(339039:0.01898,(((108907:0.03914,(252909:0.01901,(274426:0.01823,(271221:0.02666,546565:0.01058)0.956.50:0.00961)0.974.64:0.00957)0.952.51:0.01)0.979.64:0.00971,(((341359:0.03364,266165:0.03556)0.879.34:0.00874,((262719:0.02469,(568338:0.02704,(339472:0.01475,324421:0.01181)0.026:0.00388)0.841.20:0.00769)1.000.940:0.02841,((300711:0.0487,(545318:0.00984,585746:0.03138)0.982.77:0.01645)0.964.72:0.01214,4539:0.01867)0.834.24:0.00842)0.798.11:0.00286)0.989.80:0.01439,(((120521:0.02115,4597:0.02681)1.000.941:0.03611,((556015:0.01612,593795:0.02078)1.000.942:0.02791,(220363:0.01857,574659:0.03542)1.000.943:0.03238)0.988.99:0.02299)0.932.43:0.01125,344763:0.0436)0.855.24:0.0104)0.870.31:0.00409)0.986.75:0.01166,((573354:0.01434,535156:0.00606)1.000.944:0.02715,(160662:0.01688,(565674:0.04452,((70396:0.02522,((114201:0.01633,(570812:0.00285,62963:0.00291)0.936.36:0.00472)0.590.4:0.02044,(32841:0.02025,105950:0.00932)1.000.945:0.02155)0.968.54:0.00837)0.418.5:0.00378,(564264:0.00199,547550:0.01818)1.000.946:0.01858)0.960.67:0.0134)0.999.274:0.02556)0.267.3:0.0063)0.457.3:0.0066)0.975.62:0.01108)0.955.56:0.00929)0.970.51:0.01223,(162115:0.04813,((74063:0.02666,((156377:0.00582,(510201:0.01816,((560804:0.00456,130859:0.01045)0.995.113:0.01195,(170172:0.01244,93710:0.01336)0.957.69:0.00581)0.173.4:0.00333)0.647.3:0.00441)0.994.113:0.01123,(523098:0.01614,551249:0.01076)0.867.26:0.00339)0.824.11:0.00689)0.434.6:0.00831,((200462:0.01933,(((543918:0.00473,(259049:0.00954,(66183:0.01,4529:0.00636)0.939.46:0.007)0.988.100:0.01219)0.938.46:0.00518,((306797:0.04062,580150:0.01244)0.737.7:0.00132,(((68789:0.01991,282783:0.0009)0.840.14:0.00016,542958:0.00326)0.492.3:0.00341,(128131:0.00198,4522:0.00029)0.893.26:0.02886)0.805.19:0.02642)0.938.47:0.00412)0.466.7:0.0033,78316:0.0167)0.995.114:0.01282)0.607.7:0.00877,(217481:0.01464,37406:0.00925)0.988.101:0.01456)0.861.30:0.01374)0.999.275:0.02192)0.292.8:0.00676)0.985.60:0.01534)0.957.70:0.01191)0.977.63:0.01439)0.847.24:0.00741,((149766:0.02101,535965:0.01923)0.968.55:0.01036,(114903:0.03476,((277643:0.01225,356598:0.01714)0.800.13:0.00483,(((((132488:0.02157,((((342749:0.01446,70071:0.01519)0.902.24:0.0037,161477:0.03492)0.768.19:0.00271,((((570817:0.0409,(225933:0.02211,243743:0.01459)0.952.52:0.01116)0.423.6:0.00659,(248044:0.00952,(189882:0.01444,(312820:0.00768,50868:0.02393)0.881.31:0.01013)0.444.6:0.00031)0.783.17:0.00176)0.224:0.00193,(((240173:0.02674,(345961:0.00976,158370:0.00606)0.765.17:0.00324)0.972.57:0.00823,(((89681:0.00336,(((446302:0.00651,329366:0.0025)0.166.4:0.00221,307315:0.01492)0.000.55:0.00028,328359:0.00604)0.738.17:0.00016)0.989.81:0.00876,(152818:0.01007,((((487962:0.01745,(583102:0.01909,(((441555:0.00702,591751:0.01781)0.641.5:0.00247,254841:0.01382)0.821.11:0.00287,319666:0.02521)0.449.3:0.00193)0.483.6:0.00247)0.911.36:0.0031,(350117:0.0117,(162721:0.01086,173459:0.01235)0.859.28:0.00726)0.966.55:0.00555)0.777.15:0.0033,((219689:0.00922,344395:0.03024)0.879.35:0.00454,(102265:0.00643,367659:0.01116)0.749.12:0.00055)0.437.2:0.00144)0.981.67:0.00643,(47031:0.02376,(108001:0.01281,580805:0.02423)0.381.5:0.00336)0.868.34:0.00014)0.974.65:0.00507)0.928.36:0.00236)0.924.34:0.00245,(((74176:0.00946,(226138:0.01683,252977:0.00213)0.946.44:0.00858)0.537.4:0.00174,585288:0.03164)0.788.14:0.00234,220683:0.0157)0.935.34:0.00237)0.777.16:0.00138)0.878.38:0.0026,((((269203:0.00877,(329649:0.01306,(222326:0.00682,219544:0.00468)0.292.9:0.00515)0.902.25:0.00412)0.571.6:0.00472,(((((((((494426:0.01954,((6233:0.01539,99326:0.01515)0.984.66:0.00856,256074:0.01784)0.965.48:0.00391)0.968.56:0.00315,217266:0.01289)0.876.28:0.00156,509815:0.00801)0.866.24:0.00158,((312453:0.01966,70529:0.02188)1.000.947:0.00959,(217152:0.02506,60251:0.00414)0.876.29:0.00848)0.998.177:0.00951)0.793.13:0.00014,(209084:0.03185,316463:0.01275)0.928.37:0.00312)0.894.32:0.00174,(239208:0.00822,210631:0.00957)0.767.29:0.00277)0.727.8:0.0037,580992:0.01728)0.941.55:0.00015,(318717:0.00107,303645:0.00761)0.985.61:0.00484)0.936.37:0.0048,321764:0.0191)0.964.73:0.00688)0.991.86:0.00014,(584563:0.02359,235483:0.02953)0.683.9:0.00172)0.893.27:0.00156,(104492:0.00782,(535709:0.02438,243129:0.01722)0.868.35:0.00163)0.892.27:0.00233)0.931.39:0.00293)0.772.15:0.00872)0.875.34:0.00477,((((251580:0.01661,(358745:0.02307,533603:0.0251)0.943.59:0.00736)0.929.47:0.00606,((347342:0.01888,(((((321772:0.01065,511760:0.019)0.935.35:0.0003,(292322:0.02451,197687:0.01368)0.877.27:0.00263)0.948.53:0.00717,(549471:0.00789,514081:0.01015)0.091.2:0.00559)0.965.49:0.00467,((338207:0.00218,104644:0.01703)0.624.6:0.01327,104787:0.01563)0.869.40:0.0063)0.893.28:0.00266,577679:0.01715)0.449.4:0.00082)0.739.18:0.01344,((164787:0.01463,((161637:0.01325,130803:0.01616)0.861.31:0.00269,(162144:0.01296,(400524:0.00467,547624:0.01812)0.940.47:0.00375)0.729.11:0.0014)0.708.8:0.00384)0.946.45:0.00405,(((((((((564413:0.02025,(100015:0.02129,((155841:0.01544,589380:0.0059)1.000.948:0.02642,(((((108318:0.01087,(384499:0.02993,570614:0.01815)0.929.48:0.00312)0.994.114:0.0073,256514:0.00015)0.913.36:0.0104,(103369:0.00435,(361013:0.00815,209036:0.001)1.000.949:0.01297)0.759.13:0.00619)0.901.29:0.00845,(277792:0.00832,(203087:0.02095,190640:0.01172)0.487.3:0.00215)0.973.57:0.00625)0.989.82:0.00889,238220:0.01325)0.437.3:0.00318)0.954.45:0.00726)0.943.60:0.00736)0.926.47:0.0049,(261978:0.02162,(426622:0.01684,447014:0.03542)0.308.3:0.00578)0.969.60:0.0094)0.936.38:0.00643,(171409:0.03094,6263:0.01293)0.996.127:0.01506)0.752.18:0.00111,(6255:0.01854,143097:0.01375)0.520.5:0.00231)0.959.47:0.00575,577484:0.02809)0.856.23:0.0019,((217733:0.00718,133044:0.01459)0.990.83:0.0084,509987:0.02292)0.842.19:0.00223)0.906.31:0.00244,((545582:0.01156,(360253:0.01202,(446081:0.01108,(111304:0.01711,160417:0.00492)0.876.30:0.00826)0.768.20:0.00091)0.878.39:0.00167)0.859.29:0.00157,((576436:0.01219,((566147:0.011,(165838:0.00383,289006:0.00472)0.078:0.02086)0.501.6:0.00747,(570279:0.01277,((6300:0.00149,581178:0.00618)0.706.10:0.01401,(153588:0.00449,92490:0.00941)0.975.63:0.00678)0.545.3:0.00098)0.964.74:0.00331)0.846.18:0.00153)0.903.34:0.00312,((((202104:0.00253,347113:0.00359)0.959.48:0.02728,((89332:0.02169,90554:0.01616)0.966.56:0.00392,90483:0.00342)0.974.66:0.00394)0.889.31:0.00168,(313934:0.02792,281742:0.01158)0.950.66:0.01677)0.912.29:0.00231,189621:0.01977)0.313.8:0.00016)0.882.30:0.00803)0.911.37:0.00233)0.896.33:0.00239,((563756:0.03188,551494:0.01497)0.969.61:0.00878,(102512:0.0282,((202044:0.02072,6276:0.00637)0.886.25:0.00156,(436276:0.02403,405598:0.02474)0.884.36:0.00156)0.877.28:0.00503)0.641.6:0.00373)0.935.36:0.00532)0.945.42:0.00434,(((265240:0.02956,(329101:0.00678,6294:0.01514)0.728.9:0.00569)0.953.46:0.0133,164830:0.01373)0.808.15:0.00473,((((((154250:0.0122,(269458:0.00318,(157567:0.00817,555283:0.01366)0.916.41:0.00948)0.884.37:0.01337)0.972.58:0.00654,(((193988:0.01479,(53381:0.01182,((((162725:0.00392,219175:0.02219)0.929.49:0.00624,235560:0.02518)0.903.35:0.00595,588310:0.01334)0.939.47:0.00783,421125:0.00507)0.999.276:0.01335)0.492.4:0.00597)0.915.37:0.00561,(((572649:0.01966,31853:0.01643)0.730.9:0.00014,((551674:0.00784,(354006:0.02473,(538847:0.04564,128052:0.01247)0.925.32:0.02322)0.936.39:0.00375)0.968.57:0.0064,(175039:0.00716,160563:0.03353)1.000.950:0.00014)0.873.28:0.00145)0.891.34:0.00323,218419:0.02643)0.804.14:0.00449)0.995.115:0.0109,((445733:0.01623,(147665:0.01114,(256823:0.01211,(240257:0.00868,(352352:0.01657,258282:0.00697)0.964.75:0.00584)0.978.72:0.00736)0.791.17:0.00598)0.846.19:0.00476)0.800.14:0.00257,(((134748:0.00724,205122:0.01944)0.840.15:0.00228,263034:0.00407)0.700.8:0.00183,249507:0.01352)0.264.3:0.00087)0.862.36:0.006)0.871.41:0.00235)0.743.23:0.00015,531645:0.01907)0.870.32:0.00241,(349727:0.01405,(257791:0.01094,(114238:0.01954,(((518998:0.01212,((493339:0.01282,362799:0.01045)0.745.16:0.0018,153621:0.00528)0.977.64:0.00511)0.851.23:0.00397,584900:0.0062)0.993.90:0.00899,431743:0.00747)0.782.19:0.00638)0.696.7:0.00251)0.750.14:0.00116)0.977.65:0.00635)0.966.57:0.00625,(157745:0.0017,(18607:0.00015,(105077:0.01745,(((460604:0.01116,(264666:0.01118,79611:0.02318)0.980.65:0.01023)0.903.36:0.00308,106313:0.01149)0.118:0.00234,(166645:0.00158,164190:0.01921)0.942.50:0.01622)0.958.59:0.00696)0.997.149:0.01289)0.609.4:0.01018)0.353.4:0.00895)0.773.15:0.00701,191991:0.03107)0.991.87:0.00877)0.976.66:0.00743)0.915.38:0.00319)0.894.33:0.00435)0.828.11:0.00643)0.867.27:0.00293,(489291:0.02781,((305684:0.00566,(160939:0.01373,200533:0.05299)0.949.46:0.00753)0.967.55:0.00624,(278687:0.02121,(147648:0.00518,123759:0.02368)0.996.128:0.01312)0.715.9:0.00188)0.989.83:0.00915)0.972.59:0.00805)0.952.53:0.00923,((((275764:0.01424,467779:0.01711)0.639.6:0.00404,(32571:0.01053,(51138:0.01145,156959:0.00866)0.949.47:0.00592)0.221.4:0.00892)0.914.31:0.00331,((356631:0.01117,314378:0.03775)0.998.178:0.01427,((90311:0.01752,(456830:0.01422,((212603:0.02212,462836:0.00796)0.969.62:0.01113,(261294:0.00605,414888:0.00885)0.979.65:0.01435)0.790.13:0.00365)0.236.2:0.00278)0.898.37:0.00539,(((((352641:0.00792,158615:0.00014)0.878.40:0.00645,90697:0.02268)0.975.64:0.00015,(138401:0.0271,(438614:0.01783,((6157:0.00168,295655:0.01515)0.843.16:0.01446,((138735:0.00819,24311:0.01634)0.737.8:0.00128,539565:0.02201)0.752.19:0.00493)0.975.65:0.0058)0.764.19:0.00089)0.947.47:0.0055)0.444.7:0.00405,(((6195:0.00501,251683:0.0172)0.950.67:0.01348,258050:0.00962)0.833.15:0.00015,354264:0.00792)0.756.20:0.00701)0.914.32:0.0053,88144:0.01161)0.987.102:0.01046)0.612.6:0.00278)0.597.5:0.00334)0.886.26:0.00352,((154633:0.00571,6203:0.01184)0.863.30:0.0081,(386762:0.0205,578126:0.0215)0.897.30:0.00708)0.877.29:0.00591)0.997.150:0.01773)0.235.2:0.00719)0.968.58:0.00762)0.963.60:0.00654,(((360298:0.01459,(361840:0.01844,(152371:0.01366,238429:0.01123)0.904.43:0.00016)0.971.58:0.0041)0.856.24:0.00176,((511844:0.00983,582134:0.02062)0.995.116:0.01699,(339246:0.00241,582012:0.01477)0.748.17:0.0109)0.994.115:0.00895)0.563.5:0.00383,(((165325:0.01216,(543139:0.01736,(((317811:0.00492,(552687:0.02385,(((99195:0.01415,278777:0.0288)0.908.35:0.00888,(158069:0.00929,(((149938:0.01274,(251218:0.01703,226200:0.00655)0.800.15:0.00512)0.760.20:0.00123,222699:0.00707)1.000.951:0.02541,(238730:0.02294,141824:0.01561)0.740.19:0.00423)0.929.50:0.00427)0.852.33:0.00223)0.844.19:0.00284,(41806:0.02338,108043:0.014)0.871.42:0.00258)0.839.23:0.00247)0.917.43:0.0048)0.598.3:0.00015,(331282:0.01917,494339:0.01047)0.875.35:0.00486)0.830.15:0.00078,(((590580:0.01455,18013:0.01028)0.655.5:0.0021,(188464:0.01764,440356:0.01038)0.953.47:0.00659)0.816.17:0.00226,(354042:0.00132,(238508:0.01518,31402:0.0111)0.666.7:0.00514)1.000.952:0.02515)0.910.38:0.00396)0.760.21:0.00229)0.990.84:0.00759)0.427.3:0.00162,((((366814:0.02672,((344045:0.02485,336308:0.01269)0.746.17:0.01155,361963:0.00562)0.494.6:0.00599)0.813.14:0.02319,(555602:0.00592,364127:0.00631)0.583.7:0.00883)0.516.3:0.00182,(((((340165:0.02245,568062:0.0016)0.767.30:0.00554,370114:0.00337)0.988.102:0.00867,262714:0.00781)0.922.38:0.00443,(((511757:0.01996,(568463:0.0152,(170677:0.01108,136384:0.01045)0.772.16:0.00409)0.971.59:0.00703)0.000.56:0.00156,143386:0.02002)0.585.4:0.00236,((108379:0.01326,(203846:0.00485,307166:0.00831)0.985.62:0.01688)0.839.24:0.00276,145518:0.00791)0.921.44:0.00357)0.882.31:0.00389)0.970.52:0.00608,((514041:0.00819,242675:0.02493)0.978.73:0.01635,((114292:0.0119,131892:0.02441)0.976.67:0.01012,357660:0.00901)0.885.31:0.00759)0.331.8:0.00015)0.996.129:0.00798)0.997.151:0.01021,(220477:0.0035,44980:0.01737)0.973.58:0.00851)0.834.25:0.00189)0.998.179:0.01007,252410:0.02733)0.210.4:0.00159)0.681.10:0.00271)0.965.50:0.00622)1.000.953:0.01597,(256780:0.00651,(90664:0.02203,266620:0.00909)0.137.3:0.00334)0.887.30:0.01103)0.657.3:0.00051,445570:0.02396)0.972.60:0.00732,(142959:0.0222,128733:0.05063)0.900.35:0.00506)0.823.19:0.00739,(138076:0.03546,(56491:0.01414,(6365:0.00396,247680:0.01665)0.739.19:0.008)0.432.3:0.00491)0.999.277:0.02331)0.913.37:0.00705)0.592.9:0.01003)0.901.30:0.01618)1.000.954:0.04677)0.852.34:0.00852)0.722.9:0.00472)0.992.111:0.01606)0.983.79:0.01826,(((138986:0.0087,566744:0.01012)1.000.955:0.04604,(((113991:0.02042,(4594:0.0177,540846:0.00161)0.766.18:0.02455)1.000.956:0.04335,(25696:0.03151,(35557:0.01913,(185783:0.00028,304425:0.06663)0.936.40:0.01686)0.941.56:0.0082)1.000.957:0.03958)0.760.22:0.02409,(((114828:0.01587,443407:0.00553)0.843.17:0.02123,(562576:0.03508,((270023:0.0158,146977:0.02601)0.949.48:0.01052,(258350:0.01211,135801:0.01789)0.835.17:0.00844)1.000.958:0.02804)0.998.180:0.02588)0.000.57:0.00806,((565670:0.02068,310428:0.00354)0.926.48:0.0425,(584181:0.03127,342224:0.01888)1.000.959:0.07156)0.564.5:0.01183)0.824.12:0.0138)0.589.4:0.01611)0.982.78:0.02151,(((541547:0.00284,253678:0.00755)0.939.48:0.04403,468480:0.03762)0.999.278:0.02657,326754:0.05789)0.977.66:0.01552)0.886.27:0.00997)0.403.3:0.00529,(((((225544:0.00967,439353:0.02067)0.992.112:0.01971,(309146:0.03208,149199:0.01668)0.974.67:0.01806)1.000.960:0.06113,(((249410:0.01979,50794:0.02537)0.998.181:0.02562,(325441:0.03751,(113913:0.02809,342450:0.01078)0.984.67:0.01285)0.995.117:0.02364)0.998.182:0.02936,(146004:0.06061,((325439:0.00148,570138:0.0277)1.000.961:0.0407,((170743:0.01071,(278145:0.02502,153717:0.02731)0.700.9:0.00483)1.000.962:0.04945,(155559:0.02083,(345859:0.0329,265205:0.02862)0.982.79:0.01846)1.000.963:0.03326)0.588.3:0.01994)0.824.13:0.01673)0.515.6:0.01293)0.993.91:0.02317)0.999.279:0.03102,((114901:0.06677,((223237:0.01145,136114:0.02815)1.000.964:0.03941,(198250:0.02077,78776:0.01456)0.992.113:0.02263)0.999.280:0.03067)0.023.3:0.01342,(233001:0.0318,250742:0.01922)0.999.281:0.02992)0.999.282:0.03027)0.977.67:0.02101,((((105599:0.02006,(5397:0.01226,294146:0.01483)0.792.14:0.00186)0.995.118:0.02552,(201415:0.02684,((201637:0.02132,201190:0.02032)0.630.3:0.0057,((78389:0.01478,142611:0.0386)0.945.43:0.01589,((201136:0.02066,(78324:0.00634,145963:0.02078)0.961.53:0.00995)0.648.6:0.00352,77616:0.00712)0.439.7:0.00687)0.778.19:0.00605)0.906.32:0.00772)0.389.3:0.01394)1.000.965:0.12314,((300601:0.03713,528107:0.02386)0.992.114:0.0234,((165410:0.02262,((292910:0.02515,581652:0.01897)1.000.966:0.02632,((325850:0.02352,(135284:0.02372,113865:0.01742)0.913.38:0.00886)0.479.9:0.00651,(333201:0.04301,(211720:0.01475,149833:0.02405)0.812.20:0.00523)0.999.283:0.02433)0.086.2:0.00572)0.948.54:0.0123)0.920.31:0.00784,(40230:0.01797,518552:0.03307)0.989.84:0.01709)0.947.48:0.01548)1.000.967:0.05902)0.794.17:0.01738,((343481:0.00626,297362:0.00542)1.000.968:0.03368,265449:0.02072)1.000.969:0.07925)0.995.119:0.03048)0.976.68:0.01797)0.930.43:0.00835,((326279:0.02493,(((537810:0.0045,540951:0.01362)0.974.68:0.00612,583714:0.0033)0.829.19:0.0067,246883:0.01089)0.999.284:0.02136)1.000.970:0.04407,(241602:0.06064,(254391:0.05496,(554110:0.01272,(100730:0.01446,((571306:0.00682,252140:0.02833)0.892.28:0.00739,(226414:0.01758,221237:0.02053)0.886.28:0.00557)0.923.37:0.00747)0.491.2:0.00376)0.956.51:0.01462)1.000.971:0.02757)0.971.60:0.01213)0.944.37:0.01128)0.916.42:0.01346)0.999.285:0.0283,(197360:0.04667,((11282:0.01167,214607:0.01274)1.000.972:0.03604,((209788:0.00617,(11280:0.01085,11266:0.0192)0.991.88:0.00908)1.000.973:0.02445,((111324:0.01718,11267:0.01967)0.964.76:0.01336,(((11270:0.05918,593515:0.03142)0.791.18:0.00619,(11271:0.04887,214910:0.03303)0.538.3:0.00893)0.727.9:0.00239,(((11274:0.01021,11262:0.0095)0.848.25:0.01054,110832:0.01127)0.883.31:0.03029,(11265:0.02038,(11269:0.00925,(532856:0.03069,211943:0.00043)0.991.89:0.01549)0.906.33:0.01111)0.954.46:0.00955)0.815.12:0.00745)0.958.60:0.00949)0.680.5:0.00836)0.727.10:0.00838)0.896.34:0.01145)1.000.974:0.04131)0.842.20:0.01166,(222711:0.02094,111106:0.01198)1.000.975:0.1259)0.984.68:0.025)0.898.38:0.01696)0.902.26:0.00858,(((274794:0.10235,(189151:0.05206,((511221:0.08934,365052:0.05502)1.000.976:0.03226,(224255:0.02187,562287:0.02772)1.000.977:0.04803)0.621.5:0.02285)0.996.130:0.03064)0.999.286:0.05517,(330963:0.00343,263962:0.00302)1.000.978:0.12653)0.779.9:0.00678,((((237789:0.02647,170942:0.02174)1.000.979:0.07904,(((337066:0.01539,(343349:0.02995,(156878:0.00812,88339:0.02845)0.983.80:0.01158)0.864.36:0.00759)1.000.980:0.0607,(513958:0.03286,(346201:0.0177,261065:0.02095)0.858.20:0.01091)1.000.981:0.03127)0.959.49:0.01494,((143327:0.02436,10731:0.01651)1.000.982:0.06001,((592584:0.01759,(10724:0.01965,248927:0.01807)0.536.3:0.00863)0.989.85:0.02002,((277150:0.03457,253746:0.01353)0.713.5:0.01185,(217352:0.03432,(150949:0.0064,10677:0.01922)0.939.49:0.01385)0.000.58:0.00715)0.227.5:0.00779)0.991.90:0.0239)1.000.983:0.0436)0.404.6:0.0081)0.957.71:0.01152,(((((137474:0.00827,165809:0.02038)0.658.6:0.00423,(566857:0.01573,255615:0.01433)0.794.18:0.00475)0.439.8:0.00014,(594198:0.01478,((200786:0.01217,285313:0.01286)0.072:0.00396,(268413:0.01698,129752:0.00427)0.952.54:0.00461)0.653.2:0.00363)0.972.61:0.00823)1.000.984:0.04555,((327296:0.00967,98535:0.00844)1.000.985:0.04118,((256515:0.01902,559850:0.00913)1.000.986:0.05445,(590108:0.01629,587522:0.02118)1.000.987:0.07114)0.255.2:0.01366)0.783.18:0.01029)0.922.39:0.00789,(((((246839:0.01593,(156343:0.02523,(142379:0.01481,151807:0.01844)0.966.58:0.00961)0.995.120:0.01339)0.126.4:0.00398,79461:0.01869)1.000.988:0.0418,((574097:0.02874,509157:0.03537)0.998.183:0.03009,((((540283:0.01169,(170103:0.01289,10678:0.01443)0.452.8:0.00312)0.998.184:0.01725,((268308:0.019,(562534:0.00485,537106:0.02972)0.945.44:0.00536)0.899.37:0.00406,(560324:0.01452,(165464:0.01119,556385:0.01845)0.959.50:0.00816)0.930.44:0.00712)1.000.989:0.03458)0.933.44:0.00619,(547121:0.04015,((441506:0.00963,(572347:0.01992,106810:0.02761)0.354.4:0.00262)0.881.32:0.00372,(590338:0.01218,(512497:0.01926,(((554440:0.01946,(583342:0.02206,219858:0.01808)0.019.3:0.0032)0.742.19:0.00079,(553256:0.01123,534747:0.01596)0.740.20:0.01028)0.852.35:0.00157,((154329:0.02578,149503:0.00861)0.327.5:0.00162,265934:0.00973)0.999.287:0.00988)0.303.2:0.00231)0.882.32:0.00419)0.645.12:0.00176)0.886.29:0.00543)0.999.288:0.02262)0.215.4:0.00713,(536335:0.02147,(255844:0.01647,268698:0.01179)0.958.61:0.00913)0.999.289:0.02912)0.970.53:0.01564)0.725.10:0.00838)0.562.5:0.00161,((350620:0.01712,((563626:0.00984,580346:0.01293)0.893.29:0.00316,584265:0.00977)0.885.32:0.00604)1.000.990:0.03983,((((237813:0.08359,((((95801:0.03111,74264:0.00933)0.767.31:0.00225,104681:0.02266)0.753.26:0.00424,78341:0.00551)0.817.21:0.00765,252529:0.01239)0.956.52:0.02072)0.998.185:0.02984,((554210:0.02915,(10776:0.02042,545656:0.01126)0.500.8:0.02444)0.987.103:0.01544,(((((558853:0.01702,(542328:0.01493,169367:0.012)0.145:0.00724)0.343.5:0.00248,(102735:0.01168,566268:0.02641)0.899.38:0.0073)0.947.49:0.00578,(((508757:0.00502,(155465:0.00822,273242:0.01648)0.835.18:0.00324)0.943.61:0.01824,(373867:0.01136,155987:0.01565)0.859.30:0.00517)0.933.45:0.00661,(146307:0.00965,(533193:0.01503,268751:0.01371)0.997.152:0.01897)0.997.153:0.01934)0.943.62:0.00678)0.616.5:0.00546,537190:0.03411)0.704.10:0.0073,(166688:0.0325,567788:0.02643)0.809.12:0.00788)0.983.81:0.01823)0.995.121:0.0241)0.193.2:0.00569,(((276230:0.07333,((533366:0.01386,(560302:0.02032,(165767:0.01082,537262:0.01331)0.516.4:0.00541)0.816.18:0.00816)0.997.154:0.02041,((219894:0.01535,510138:0.01028)0.991.91:0.01207,((151758:0.01914,(28552:0.01825,(542416:0.01124,123413:0.02386)0.737.9:0.00556)0.994.116:0.01133)0.997.155:0.01518,(((591519:0.01756,242560:0.02383)0.997.156:0.01866,(66924:0.02216,19062:0.02076)0.286.6:0.00492)0.999.290:0.01993,(((160457:0.005,(578505:0.00837,262663:0.00931)0.951.50:0.00482)0.809.13:0.00154,273544:0.00561)1.000.991:0.01576,(587929:0.02348,(161119:0.01869,((((109915:0.00483,10758:0.00122)0.994.117:0.00985,(571810:0.0186,138280:0.00279)0.989.86:0.0084)0.402.3:0.00387,((107050:0.00588,316207:0.02575)0.806.13:0.00436,(571785:0.02389,((161965:0.01624,(77292:0.02465,(149053:0.00748,535606:0.01113)0.906.34:0.00274)0.784.15:0.00359)0.979.66:0.01047,(((581036:0.02705,(581775:0.01472,((10750:0.00715,163774:0.00716)0.994.118:0.01034,(155072:0.0043,(583853:0.00971,557936:0.00571)0.163.3:0.0023)0.940.48:0.00445)0.945.45:0.00634)0.975.66:0.01206)0.950.68:0.00916,(543826:0.01873,225430:0.017)0.839.25:0.00536)0.991.92:0.00931,138526:0.01563)0.309.5:0.0031)0.924.35:0.00793)0.888.22:0.0041)0.752.20:0.00166)0.967.56:0.00794,((294756:0.01536,10754:0.01308)0.963.61:0.00606,(10759:0.00684,257733:0.01477)0.996.131:0.01183)0.791.19:0.00285)0.926.49:0.00657)0.497.6:0.00795)0.585.5:0.00501)0.898.39:0.00491)0.586.4:0.00516)0.921.45:0.00694)0.999.291:0.02282)1.000.992:0.02038)0.926.50:0.00523,(((538541:0.02785,((552003:0.01414,31004:0.02819)0.813.15:0.00467,((569088:0.01542,(164530:0.01103,143971:0.02455)0.936.41:0.00667)0.894.34:0.00464,((221935:0.01203,(558943:0.01429,(310060:0.00181,104941:0.00785)0.967.57:0.0082)0.386.3:0.00429)0.995.122:0.01238,(140135:0.01617,580869:0.02028)0.975.67:0.01506)0.967.58:0.00889)0.223.5:0.00268)0.941.57:0.00825)0.655.6:0.00784,30060:0.0151)1.000.993:0.04105,(552824:0.0205,(173034:0.03081,535455:0.01786)0.852.36:0.01018)1.000.994:0.0415)0.981.68:0.01525)0.925.33:0.0054,(((245974:0.01835,571089:0.03065)1.000.995:0.04262,((26785:0.02425,572807:0.02903)0.940.49:0.00878,((91433:0.01271,590151:0.03584)0.999.292:0.02145,(71864:0.00775,10775:0.02142)1.000.996:0.02894)0.906.35:0.01172)1.000.997:0.0446)0.921.46:0.0105,((249482:0.04049,(218256:0.01593,(305714:0.02467,(202389:0.00753,148496:0.01104)0.988.103:0.01319)0.973.59:0.01448)1.000.998:0.06464)1.000.999:0.03401,(((264473:0.02416,546852:0.0247)0.995.123:0.01831,((188174:0.00886,561085:0.01729)0.936.42:0.00732,(10779:0.00867,542030:0.01018)0.997.157:0.01764)0.997.158:0.01821)0.973.60:0.01543,(((289758:0.04205,(((257496:0.037,(108098:0.01765,315992:0.0123)0.999.293:0.01956)0.330.4:0.00485,(((546128:0.01879,(160902:0.00531,(154843:0.00777,(542015:0.00885,(546119:0.00369,(109284:0.00239,544699:0.0144)0.750.15:0.00107)0.815.13:0.00266)0.144.2:0.00177)0.987.104:0.00892)0.936.43:0.00741)0.999.294:0.01993,((241850:0.02456,(132520:0.01214,159502:0.01555)0.935.37:0.00527)0.999.295:0.01635,(238062:0.0106,(590543:0.01953,(582637:0.00606,342766:0.00993)0.972.62:0.01091)0.555.4:0.00412)0.906.36:0.00669)0.992.115:0.01445)0.950.69:0.01039,(((248972:0.01926,138923:0.00662)0.999.296:0.01833,((24077:0.00437,544630:0.01452)0.997.159:0.01732,(10897:0.03847,(10865:0.02597,((158084:0.01616,114332:0.00993)0.895.37:0.00386,(159387:0.02288,159263:0.00258)0.868.36:0.00248)1.000.1000:0.03017)0.928.38:0.00956)0.725.11:0.00652)0.761.20:0.00592)0.896.35:0.00652,(((148326:0.02532,((594041:0.01915,272166:0.01823)0.986.76:0.01137,(10893:0.02825,(((173340:0.01016,159856:0.01011)0.701.5:0.0026,175430:0.00937)0.949.49:0.00471,173385:0.01243)0.986.77:0.00993)0.948.55:0.00773)0.684.10:0.00393)0.973.61:0.0074,(((568628:0.02407,(((164141:0.02337,(249902:0.00831,217198:0.00936)0.993.92:0.01564)0.988.104:0.01531,320845:0.02229)0.975.68:0.00917,(162529:0.01707,(556574:0.01974,537927:0.02016)0.908.36:0.00522)0.731.10:0.00289)0.598.4:0.00185)0.515.7:0.004,566855:0.02279)0.874.27:0.00412,((544425:0.00565,103639:0.01978)1.000.1001:0.01668,(((560623:0.02059,594000:0.01739)0.998.186:0.02109,((314793:0.01077,580063:0.00705)0.999.297:0.01596,((241805:0.00602,270636:0.00245)0.948.56:0.00483,(592871:0.01449,(10901:0.0032,258076:0.01188)1.000.1002:0.012)0.805.20:0.00183)1.000.1003:0.01844)0.986.78:0.01375)0.941.58:0.00603,((((10934:0.01831,(10867:0.02691,237295:0.00722)0.120.2:0.00348)1.000.1004:0.0682,534671:0.0383)0.911.38:0.01505,551572:0.043)0.919.35:0.00836,((169802:0.00622,(157733:0.00796,576852:0.00684)0.972.63:0.00728)1.000.1005:0.02418,((((10941:0.01397,47877:0.00078)0.874.28:0.01596,(((523104:0.01185,221104:0.01934)0.932.44:0.00717,10938:0.01672)0.773.16:0.0019,10939:0.00928)0.987.105:0.01101)0.931.40:0.00749,(86333:0.02458,86388:0.01546)0.923.38:0.0131)0.998.187:0.02212,((78421:0.00755,(307138:0.00771,(285579:0.01206,287932:0.00252)0.523.4:0.00276)0.975.69:0.00698)0.999.298:0.01547,((((86082:0.0117,((243934:0.00799,(78718:0.00986,143680:0.01836)0.908.37:0.00396)0.913.39:0.00355,((165612:0.00562,(112751:0.00976,(78986:0.0007,256085:0.00766)0.957.72:0.00606)1.000.1006:0.01118)0.930.45:0.00399,(587101:0.01621,(10933:0.00751,239765:0.01175)0.683.10:0.00178)0.518:0.0025)0.745.17:0.00103)0.996.132:0.01136)0.480.6:0.00074,(114027:0.00806,561928:0.00735)0.879.36:0.00292)0.712.5:0.00429,((51719:0.00539,(330092:0.00988,248725:0.03481)0.965.51:0.01314)1.000.1007:0.01665,((577257:0.02233,(544106:0.00824,(10927:0.01586,((132831:0.0018,(((10923:0.01126,160301:0.00637)0.802.13:0.00636,10925:0.00997)0.965.52:0.00475,10926:0.01529)0.524.3:0.00803)0.876.31:0.00251,10928:0.00745)0.818.9:0.01116)0.994.119:0.01394)0.974.69:0.01127)0.687.7:0.00865,(113207:0.01613,378495:0.01741)0.566.6:0.00785)0.567.5:0.00703)0.678.5:0.00398)0.975.70:0.00697,(545744:0.01621,106553:0.03158)0.918.47:0.00514)0.969.63:0.00947)0.991.93:0.01397)1.000.1008:0.03385)0.898.40:0.00821)0.908.38:0.00601)0.291.8:0.00014)0.981.69:0.00906)0.957.73:0.00629)0.881.33:0.00401,(((((151178:0.00728,514785:0.02659)0.999.299:0.01861,(306625:0.00738,((((10905:0.01019,153430:0.00878)0.259.6:0.00231,313754:0.02945)0.861.32:0.01704,304376:0.00072)0.985.63:0.01502,((570484:0.00239,574919:0.01369)0.929.51:0.00474,(289889:0.0085,(78733:0.00291,(112154:0.01435,583955:0.02346)0.762.18:0.00238)0.761.21:0.00244)0.939.50:0.00044)0.903.37:0.00657)0.919.36:0.00541)0.984.69:0.00888)0.813.16:0.00402,(((160182:0.02255,((557184:0.0154,(253758:0.0109,545988:0.01904)1.000.1009:0.02031)0.904.44:0.00582,(((((337177:0.01615,147720:0.017)0.994.120:0.01024,((((584754:0.01291,541666:0.01403)0.212.5:0.00323,(146208:0.0185,((((111869:0.01736,(542240:0.00991,202733:0.01167)0.865.31:0.00234)0.963.62:0.00457,19549:0.01084)0.969.64:0.00592,150803:0.00994)0.928.39:0.00432,(148508:0.00683,342977:0.01434)0.971.61:0.00713)0.971.62:0.00907)0.950.70:0.00658)0.783.19:0.00199,(183230:0.01413,278443:0.00514)0.922.40:0.00376)0.376.6:0.00284,((539378:0.01048,(348411:0.00909,545365:0.01285)0.954.47:0.00525)0.428.4:0.00166,((350206:0.007,(316203:0.01975,566138:0.01399)0.905.37:0.01569)0.834.26:0.00333,562630:0.01035)0.918.48:0.00277)0.811.17:0.00267)0.794.19:0.00409)0.849.18:0.00468,(((171180:0.00829,(237712:0.00394,535659:0.02119)0.839.26:0.0068)0.831.11:0.00256,((544326:0.00722,568791:0.0088)0.996.133:0.00918,556123:0.01424)0.866.25:0.00243)0.997.160:0.01014,(237632:0.0176,(155401:0.00404,558344:0.02366)0.998.188:0.01579)0.987.106:0.0121)0.885.33:0.00449)0.794.20:0.00304,554116:0.02096)0.889.32:0.00378,(((275331:0.02037,274595:0.02489)0.448.6:0.00433,(571102:0.02055,(556083:0.03164,513493:0.03006)0.880.27:0.00857)0.126.5:0.00015)0.345.4:0.00321,(((134929:0.02656,574156:0.00639)0.992.116:0.00776,(((113838:0.02044,(70417:0.00898,528626:0.028)0.828.12:0.00161)0.762.19:0.00079,(518308:0.01191,281311:0.01141)0.742.20:0.00077)0.856.25:0.00814,(583903:0.01134,562283:0.01145)0.961.54:0.0056)0.178.3:0.0015)0.886.30:0.00333,((589811:0.00078,105176:0.02447)0.933.46:0.01142,(((((341077:0.01394,533326:0.01756)0.885.34:0.00398,((192150:0.00883,184819:0.01739)0.802.14:0.00328,99562:0.02761)0.851.24:0.00242)0.758.18:0.00104,150305:0.02275)0.850.24:0.00351,((558428:0.01928,(244502:0.01454,252360:0.00738)0.873.29:0.00584)0.924.36:0.00452,(593687:0.0155,578329:0.00972)1.000.1010:0.01709)0.703.11:0.0009)0.996.134:0.01032,(584979:0.01205,(272732:0.00961,274713:0.01351)0.987.107:0.00852)0.947.50:0.00491)0.561.7:0.00243)0.927.50:0.00393)0.403.4:0.00101)0.830.16:0.00291)0.945.46:0.00621)0.718.9:0.00294)0.790.14:0.00401,(((160538:0.00014,161002:0.00084)0.870.33:0.01339,((425208:0.00758,225865:0.02079)0.825.16:0.00403,160788:0.01643)0.051.4:0.00642)1.000.1011:0.02386,156126:0.01785)0.920.32:0.00639)0.962.56:0.00702,((554551:0.00587,563858:0.00837)0.821.12:0.00365,(17424:0.01757,(154607:0.01287,10909:0.01536)0.989.87:0.00863)0.354.5:0.0021)0.922.41:0.00361)0.957.74:0.00607)0.846.20:0.00318,((326082:0.03064,(581992:0.01431,549171:0.02602)0.746.18:0.0016)0.597.6:0.00357,(246145:0.01771,354146:0.02853)0.039:0.00389)0.982.80:0.00869)0.830.17:0.00667,(((147459:0.01885,(10869:0.02025,556875:0.02467)0.915.39:0.00875)0.885.35:0.00413,(249930:0.01956,555385:0.0144)0.788.15:0.00555)0.831.12:0.00262,(353759:0.01896,((204801:0.03327,(165239:0.0121,((10876:0.00574,(139771:0.00804,100001:0.00669)0.982.81:0.0091)0.954.48:0.00796,166757:0.01457)0.853.24:0.00312)0.978.74:0.00962)0.149.3:0.00366,(267799:0.01074,(513028:0.00524,249559:0.02166)0.968.59:0.01246)1.000.1012:0.02399)0.987.108:0.009)0.772.17:0.00654)0.796.15:0.00464)0.450.5:0.00408)0.809.14:0.00593)0.959.51:0.00907)0.890.34:0.00666)0.999.300:0.01725,(574128:0.0165,((((130033:0.01853,260326:0.01989)0.979.67:0.01015,(555020:0.00614,(510495:0.01275,261390:0.01195)0.726.11:0.00235)0.879.37:0.00313)0.965.53:0.00558,(277069:0.01926,552223:0.0185)0.224.2:0.0004)0.908.39:0.00329,(173313:0.02745,252372:0.02763)0.973.62:0.00889)0.417.5:0.00313)0.998.189:0.02375)0.366.2:0.00706)0.946.46:0.01229,((((343091:0.01126,(249656:0.01202,238545:0.01615)0.946.47:0.0072)1.000.1013:0.01992,(((205279:0.01049,(395222:0.03582,10771:0.01678)0.909.35:0.01013)1.000.1014:0.04787,566534:0.04272)0.985.64:0.01702,((((572343:0.03295,((576990:0.0066,((566553:0.00488,10906:0.01527)0.877.30:0.00302,565202:0.02273)0.758.19:0.0017)0.956.53:0.005,(543630:0.02235,(343572:0.00431,(40264:0.01159,(513824:0.004,(299873:0.01146,564870:0.00964)0.811.18:0.00328)0.767.32:0.00489)0.910.39:0.00353)0.276.2:0.01371)0.963.63:0.00745)0.880.28:0.005)0.181.3:0.00547,(113166:0.02955,159166:0.02173)0.691.7:0.00946)0.559.4:0.00014,(591936:0.01967,(557390:0.01456,(514094:0.02046,((156136:0.01123,(260728:0.02317,10904:0.0327)0.404.7:0.00898)0.364.5:0.00316,592826:0.02227)0.793.14:0.00563)0.431.7:0.00211)0.859.31:0.00274)0.222.5:0.00228)0.903.38:0.00325,(298100:0.01723,(235297:0.0039,142790:0.02618)0.991.94:0.01121)0.899.39:0.00595)0.895.38:0.00302)0.223.6:0.00297)0.721.7:0.00459,((((566769:0.01903,(540242:0.01672,(169603:0.01761,328502:0.01416)0.714.6:0.00401)0.876.32:0.00259)0.935.38:0.00428,((339335:0.01671,(145576:0.02342,584649:0.01909)0.360.3:0.006)0.818.10:0.00309,(164090:0.03095,(166759:0.00416,241392:0.03057)0.990.85:0.01074)0.151.3:0.00125)0.353.5:0.00198)0.461.6:0.00175,(113116:0.01401,164969:0.02095)0.778.20:0.00443)0.625.6:0.00259,(536561:0.02307,(534317:0.01228,559617:0.01342)0.995.124:0.01369)0.984.70:0.01038)0.840.16:0.00262)0.994.121:0.01497,((252205:0.01024,548731:0.02186)0.689.4:0.00516,((592376:0.03444,267426:0.01393)0.938.48:0.00706,((235938:0.00306,159904:0.02096)0.992.117:0.01237,(136481:0.01673,((262547:0.00841,274235:0.01565)0.529.3:0.00269,129790:0.00705)0.991.95:0.00939)0.194:0.003)0.597.7:0.00254)0.861.33:0.00335)1.000.1015:0.03733)0.993.93:0.02189)0.909.36:0.00694,((537744:0.00969,(150612:0.0025,143077:0.02629)0.999.301:0.01824)0.800.16:0.00752,552734:0.00976)1.000.1016:0.02246)0.941.59:0.01142)0.245.5:0.01096)0.918.49:0.00822)0.985.65:0.01054)0.757.20:0.00175)0.982.82:0.01176,((((((94853:0.02281,(((128221:0.01706,538685:0.01202)0.999.302:0.01802,(274550:0.02109,(159708:0.02114,(260870:0.00717,136232:0.01524)0.986.79:0.00975)0.989.88:0.01272)0.446.6:0.00432)0.998.190:0.01708,((34985:0.03459,(546409:0.01128,295289:0.01157)0.984.71:0.01038)0.944.38:0.00697,(((((((284321:0.00771,187961:0.01843)0.891.35:0.00419,(143046:0.01974,(546120:0.00108,548614:0.0033)0.910.40:0.00671)0.949.50:0.00427)0.888.23:0.00346,(10845:0.00918,151989:0.0207)0.776.19:0.00762)0.883.32:0.00309,((((576808:0.01088,276971:0.01541)0.725.12:0.00754,10858:0.00775)0.728.10:0.00979,10859:0.00858)0.989.89:0.01064,(110803:0.00433,275861:0.01749)1.000.1017:0.0149)0.784.16:0.00439)0.842.21:0.00262,10857:0.02312)0.998.191:0.01328,((271079:0.01733,66927:0.02808)0.987.109:0.01426,(10849:0.00391,37165:0.03367)0.874.29:0.01837)0.554.5:0.00474)0.887.31:0.00321,((269237:0.02891,((587689:0.01676,(133542:0.00983,66928:0.01975)0.950.71:0.00591)0.230.2:0.001,((452423:0.01194,(586738:0.01749,(150930:0.00657,(21347:0.01523,460067:0.00251)0.909.37:0.00312)0.978.75:0.00728)0.268.3:0.00323)0.853.25:0.0024,(203169:0.02277,(534010:0.01393,(((162776:0.00487,63258:0.01314)0.879.38:0.00165,(256676:0.00263,579680:0.01254)0.704.11:0.00449)0.756.21:0.00095,158424:0.02856)0.802.15:0.00148)0.512.6:0.00359)0.874.30:0.00307)0.926.51:0.00248)0.887.32:0.00357)0.908.40:0.00662,102521:0.02275)0.741.15:0.00181)0.970.54:0.00564)0.838.24:0.00196)0.940.50:0.00745)0.791.20:0.0071,((579169:0.02935,269542:0.01785)0.943.63:0.0102,(280179:0.02266,110185:0.00922)0.999.303:0.01735)0.185.2:0.00264)0.987.110:0.01299,77467:0.02327)0.547.4:0.00411,(56266:0.0055,552783:0.01117)1.000.1018:0.026)1.000.1019:0.02908,((241617:0.1203,((((((311410:0.01657,((547552:0.01589,154397:0.01757)0.918.50:0.00324,(279049:0.01307,((337873:0.00789,(163348:0.00802,(112173:0.00919,192062:0.00754)0.947.51:0.0043)0.791.21:0.00701)0.420.6:0.00765,329633:0.01279)0.786.19:0.00514)0.938.49:0.00015)0.801.14:0.00822)0.178.4:0.0042,278071:0.01542)1.000.1020:0.02302,((((106795:0.01649,76328:0.01537)0.321.4:0.00517,(203745:0.01426,279354:0.0239)0.607.8:0.00431)1.000.1021:0.02411,((102062:0.01094,551463:0.03386)0.800.17:0.01306,(100161:0.01911,19356:0.01256)0.882.33:0.00751)0.998.192:0.01971)0.500.9:0.00691,((517763:0.0086,(106430:0.00644,103065:0.00014)0.985.66:0.01299)1.000.1022:0.03439,334143:0.05279)0.645.13:0.01102)0.912.30:0.01261)0.867.28:0.00553,((99697:0.02132,540615:0.01829)0.973.63:0.0106,(((((247260:0.01743,10829:0.01254)0.816.19:0.00242,((542644:0.01007,(47698:0.01199,(341591:0.01013,581441:0.01676)0.269.2:0.00509)0.913.40:0.00549)0.994.122:0.01009,(249672:0.01698,(105497:0.00442,((165751:0.00608,142768:0.01402)0.969.65:0.00954,(590669:0.00646,96561:0.02127)0.756.22:0.00191)0.929.52:0.00342)0.937.49:0.00529)0.528.5:0.00296)0.907.35:0.00644)0.994.123:0.01072,((10830:0.00178,86795:0.0131)0.993.94:0.01191,(563651:0.01334,(130219:0.00947,281431:0.01914)0.410.3:0.01152)0.968.60:0.00828)0.972.64:0.00971)0.621.6:0.00662,((((101171:0.0091,((113376:0.01074,550476:0.01043)0.762.20:0.00093,589607:0.0102)0.689.5:0.00542)0.897.31:0.00716,79376:0.02443)0.974.70:0.00861,((161369:0.0094,541497:0.02145)1.000.1023:0.02416,(569243:0.01177,77760:0.03366)0.378.8:0.00272)0.987.111:0.0096)0.987.112:0.01111,(221162:0.01868,(61882:0.00472,90296:0.01856)0.259.7:0.00307)0.950.72:0.00626)0.733.15:0.0046)0.951.51:0.01057,((((347713:0.00454,591865:0.01923)0.851.25:0.00443,((278958:0.01491,((260510:0.00576,259454:0.01172)0.988.105:0.01133,254088:0.02676)0.967.59:0.00737)0.951.52:0.00629,(((10837:0.00956,284851:0.015)0.903.39:0.0042,(109275:0.00391,(590606:0.01281,160918:0.01963)0.687.8:0.00712)0.977.68:0.00933)0.025:0.00239,170756:0.00911)0.953.48:0.00537)0.903.40:0.00334)0.806.14:0.00598,((163987:0.00967,166116:0.02192)0.842.22:0.00589,278259:0.02556)0.585.6:0.00311)0.704.12:0.00554,(312498:0.01009,526696:0.01075)0.011:0.00614)0.242.5:0.00659)0.893.30:0.0057)0.999.304:0.0215)0.888.24:0.00925,((162310:0.00885,141610:0.02191)0.999.305:0.01979,(296890:0.02791,316530:0.05433)0.789.18:0.00914)0.935.39:0.00982)0.012.3:0.00549,(534767:0.06272,((243494:0.02603,((333184:0.00472,(((583279:0.02464,(572479:0.0133,(254181:0.00029,250699:0.01617)0.827.14:0.00694)0.773.17:0.00314)0.879.39:0.005,564051:0.00162)0.765.18:0.00322,69671:0.00092)0.977.69:0.01658)0.989.90:0.00915,(111965:0.01367,(588015:0.00501,(((264490:0.00323,520134:0.01177)0.766.19:0.00202,(159745:0.0091,(98407:0.01459,341855:0.00319)0.829.20:0.00016)0.991.96:0.00688)0.827.15:0.00328,((545413:0.00733,(141426:0.0025,(231069:0.00171,105894:0.01634)0.913.41:0.005)0.823.20:0.00537)0.908.41:0.00626,(593167:0.01257,531935:0.01163)0.758.20:0.00246)0.884.38:0.00521)0.947.52:0.0057)0.956.54:0.00785)0.978.76:0.01038)0.812.21:0.00592)0.938.50:0.01048,(((((164944:0.0121,((319044:0.01202,541339:0.00296)0.698.8:0.00283,(25548:0.01375,87194:0.01263)0.674.6:0.0036)0.877.31:0.00265)0.893.31:0.00359,(134181:0.01893,((109273:0.00724,109272:0.0122)0.950.73:0.00432,(((320593:0.00183,307864:0.01729)0.952.55:0.00664,(540643:0.01384,540454:0.01213)0.971.63:0.00975)0.909.38:0.00406,(245906:0.00396,(20814:0.0045,(100033:0.00173,242966:0.00217)0.917.44:0.00314)0.972.65:0.00558)0.626.5:0.00325)0.951.53:0.00644)0.890.35:0.00509)0.410.4:0.00247)0.857.25:0.00572,((((10840:0.0254,510143:0.00701)0.922.42:0.00496,(512929:0.01117,((594293:0.00588,527049:0.01809)0.822.19:0.00639,509769:0.00758)0.632.2:0.00212)0.696.8:0.00231)0.935.40:0.00607,(138936:0.02369,(((((236813:0.00092,226272:0.00456)0.968.61:0.00447,((271328:0.00028,(111554:0.01695,544211:0.00107)0.658.7:0.00016)0.951.54:0.00441,(108099:0.01495,(223812:0.01642,591305:0.00727)0.856.26:0.00229)0.981.70:0.00803)0.731.11:0.00051)0.388.2:0.00085,184429:0.00616)0.449.5:0.00079,570415:0.00421)0.796.16:0.00229,(226332:0.01732,567464:0.01232)0.713.6:0.00418)0.929.53:0.00579)0.256.3:0.00411)0.786.20:0.00255,(309672:0.01294,(571713:0.01599,128136:0.00804)0.989.91:0.01085)0.580.2:0.00479)0.998.193:0.01564)0.801.15:0.00258,(((591626:0.01607,((10814:0.00015,10809:0.00717)0.842.23:0.03071,((78851:0.00313,(104747:0.01405,281137:0.01324)0.397.6:0.00444)0.522.5:0.00487,257411:0.00893)0.932.45:0.00531)0.913.42:0.00468)0.845.21:0.00199,(112333:0.00245,((((((549531:0.01409,(587903:0.02198,(590689:0.00233,535882:0.01412)0.704.13:0.00643)0.829.21:0.00247)0.984.72:0.00593,581436:0.00558)0.911.39:0.00721,308423:0.01212)0.746.19:0.00142,(82114:0.00742,(511303:0.01966,545830:0.00089)0.864.37:0.00507)0.388.3:0.00117)0.922.43:0.00237,(78728:0.01041,131172:0.00282)0.963.64:0.00399)0.939.51:0.00309,169282:0.00321)0.370.5:0.00076)0.965.54:0.00464)0.783.20:0.0016,((249075:0.01318,(561836:0.01154,(244699:0.01031,141040:0.01184)0.842.24:0.00255)0.979.68:0.00779)0.949.51:0.00622,(347426:0.01187,(548857:0.00185,((366378:0.00139,166413:0.02051)0.970.55:0.0058,266557:0.01284)0.958.62:0.00478)0.986.80:0.00866)0.985.67:0.00908)0.930.46:0.00431)0.771.17:0.0019)0.954.49:0.00965,((568793:0.0112,129152:0.00945)0.803.12:0.00164,(534504:0.00419,110727:0.01099)0.728.11:0.00161)0.998.194:0.01531)0.931.41:0.0116)0.995.125:0.02164)0.928.40:0.01126)0.997.161:0.02787)0.938.51:0.01431,(((532903:0.05493,(546717:0.02577,(317469:0.01715,184957:0.02802)0.207:0.00514)0.900.36:0.00893)1.000.1024:0.02762,(563571:0.04503,((551732:0.01571,(593701:0.00745,114153:0.02114)0.973.64:0.0078)1.000.1025:0.02274,(562989:0.04087,512958:0.0248)0.591.6:0.00862)0.496.4:0.00578)0.978.77:0.02131)0.997.162:0.02623,((81732:0.03506,(139228:0.00708,159337:0.03764)0.999.306:0.02615)1.000.1026:0.02824,((((146992:0.02988,56683:0.03195)1.000.1027:0.07526,299189:0.05314)0.804.15:0.00932,((66925:0.0298,(303539:0.01143,(191519:0.00789,151310:0.01285)0.926.52:0.00462)0.900.37:0.00397)0.908.42:0.00478,((66635:0.01574,581689:0.0123)0.972.66:0.00645,(274924:0.01091,(509370:0.03997,131091:0.01869)0.922.44:0.00801)0.903.41:0.00466)0.687.9:0.00459)1.000.1028:0.0458)0.876.33:0.00964,(268129:0.01231,133425:0.00875)1.000.1029:0.05397)0.774.10:0.0053)0.987.113:0.01418)0.317.2:0.0091)0.804.16:0.00327)0.551.3:0.00051,((139171:0.02494,558030:0.03042)1.000.1030:0.03533,(517432:0.01652,(67797:0.00487,57453:0.02071)0.995.126:0.01687)1.000.1031:0.04073)0.993.95:0.02011)0.996.135:0.01517)0.167.2:0.00121)0.965.55:0.0082)0.009.5:0.00547,(((19024:0.01001,518730:0.02939)1.000.1032:0.11683,((((143834:0.01736,543482:0.01907)0.987.114:0.01114,(10718:0.01012,262088:0.01188)0.872.43:0.0041)0.975.71:0.01109,554930:0.00668)0.942.51:0.01465,(569150:0.02378,(((63254:0.04591,(29387:0.01089,34143:0.02839)0.365.3:0.00266)0.866.26:0.00384,((563406:0.01081,590259:0.00702)0.859.32:0.00809,(273691:0.02154,263319:0.00941)0.795.16:0.00189)0.983.82:0.01021)0.397.7:0.00659,569641:0.01727)0.885.36:0.00747)0.982.83:0.01806)0.996.136:0.03923)0.956.55:0.02534,(588676:0.02493,(545469:0.02665,(327954:0.02577,155472:0.00427)0.187.3:0.01819)0.612.7:0.01155)1.000.1033:0.05139)0.993.96:0.02338)0.625.7:0.00925)0.810.17:0.00829)0.898.41:0.00639,((((((254193:0.02821,257948:0.01298)0.919.37:0.00606,((247777:0.02797,(227450:0.01211,209570:0.01811)0.270.2:0.00482)0.232.5:0.00413,256205:0.02662)0.931.42:0.00713)1.000.1034:0.08114,(113368:0.02269,343998:0.01937)1.000.1035:0.11195)0.391.5:0.02112,((185399:0.02866,542199:0.01916)1.000.1036:0.03707,((338371:0.0609,(74869:0.05212,((((49295:0.02408,(171640:0.02107,137427:0.01277)0.420.7:0.00309)0.783.21:0.00426,(38566:0.0211,247981:0.00778)0.812.22:0.00722)0.945.47:0.00837,(105561:0.01843,137132:0.02168)0.565.2:0.00537)0.976.69:0.01228,((105711:0.00179,307774:0.00732)0.762.21:0.01749,(213370:0.01454,243041:0.01836)0.994.124:0.01714)0.999.307:0.02615)0.862.37:0.00972)0.676.7:0.01238)0.999.308:0.0234,(((448975:0.01287,((353499:0.01315,(191550:0.01614,337324:0.01415)0.994.125:0.01209)1.000.1037:0.02717,(511558:0.02614,172283:0.03139)0.626.6:0.00792)0.008:0.00769)1.000.1038:0.0398,(320761:0.01894,232532:0.02572)0.987.115:0.01625)0.953.49:0.01513,(((236208:0.02678,(416757:0.0182,(227331:0.01602,265055:0.01402)0.421.4:0.00577)0.923.39:0.0064)0.901.31:0.00723,525171:0.01615)0.957.75:0.00805,246762:0.00918)1.000.1039:0.0569)0.730.10:0.00724)0.971.64:0.01893)1.000.1040:0.04887)0.765.19:0.01479,(((322881:0.01747,306090:0.00956)1.000.1041:0.07701,((230171:0.07164,(570846:0.02374,568507:0.02679)0.999.309:0.04974)1.000.1042:0.04135,(((253026:0.02768,(136786:0.0136,509119:0.01701)0.521.6:0.0065)0.998.195:0.01984,(233140:0.04894,(155720:0.04224,(149432:0.03148,553766:0.01415)0.999.310:0.02008)0.127.2:0.00464)0.741.16:0.00645)1.000.1043:0.02352,(((577706:0.00014,4709:0.00091)1.000.1044:0.05313,234197:0.0387)0.998.196:0.02345,((198258:0.0029,75871:0.01469)1.000.1045:0.07267,((79548:0.01578,548348:0.01621)0.999.311:0.02614,(355193:0.01373,590339:0.02792)1.000.1046:0.05738)0.953.50:0.01279)0.851.26:0.00432)0.659.5:0.00943)0.975.72:0.02089)0.992.118:0.02421)0.998.197:0.02523,((((269593:0.01542,367670:0.02027)1.000.1047:0.04594,((234175:0.00602,259981:0.00122)1.000.1048:0.04674,((106171:0.03801,(6258:0.0238,(560221:0.03625,((589130:0.03651,151578:0.00269)0.757.21:0.00015,559552:0.02101)0.848.26:0.00487)0.988.106:0.01354)1.000.1049:0.04838)0.895.39:0.01375,160719:0.09794)0.982.84:0.01681)0.872.44:0.00807)0.992.119:0.02279,((((236563:0.01772,(10690:0.02301,275503:0.0205)0.993.97:0.01655)0.852.37:0.00434,(273697:0.01637,10691:0.01512)0.746.20:0.00433)0.803.13:0.00515,260749:0.01672)1.000.1050:0.04683,(276800:0.0372,(10661:0.02476,(275844:0.00392,306845:0.03877)1.000.1051:0.02056)0.073.5:0.00997)0.999.312:0.03623)0.570.7:0.01333)1.000.1052:0.055,((((((281539:0.01939,254683:0.03264)0.991.97:0.01481,(((106006:0.02813,(141636:0.01088,(218005:0.01458,(141116:0.00653,(581069:0.02646,(38634:0.01522,75942:0.0104)0.116.8:0.00509)0.978.78:0.01068)0.798.12:0.00541)0.961.55:0.00624)0.947.53:0.00905)1.000.1053:0.02688,590861:0.07364)0.897.32:0.00768,((230010:0.01816,(239851:0.01099,229914:0.02443)0.896.36:0.00712)1.000.1054:0.02068,(((((277896:0.00943,240221:0.01637)1.000.1055:0.01898,277698:0.01987)0.954.50:0.01436,(252029:0.03976,(((274526:0.04918,((((22841:0.00014,(265128:0.02035,259755:0.02611)0.928.41:0.02853)0.977.70:0.0182,176161:0.02404)0.868.37:0.00371,(246590:0.03874,((254136:0.01118,256371:0.05875)0.700.10:0.01101,(190692:0.00247,273492:0.05954)0.040.2:0.0038)0.993.98:0.01469)0.782.20:0.00768)1.000.1056:0.01964,245729:0.03596)0.816.20:0.00518)0.788.16:0.00473,((262357:0.1043,(259725:0.01572,271096:0.09075)0.839.27:0.03291)0.489:0.00175,194185:0.00097)1.000.1057:0.0221)0.928.42:0.00636,(106487:0.0079,257600:0.01093)1.000.1058:0.03188)0.803.14:0.00919)0.379.10:0.00167)0.942.52:0.00789,((591460:0.02111,241853:0.03859)0.612.8:0.00653,(272012:0.04925,(185806:0.04236,249394:0.03085)0.172.2:0.00606)0.984.73:0.01317)0.982.85:0.01042)0.998.198:0.0107,(249153:0.0144,324529:0.00875)0.978.79:0.0091)0.994.126:0.015)0.830.18:0.00486)0.970.56:0.01266)0.505.5:0.00537,(518456:0.01687,515896:0.00946)1.000.1059:0.03988)0.983.83:0.02051,(((136377:0.01686,182905:0.00714)0.932.46:0.00539,((346586:0.00417,554390:0.005)0.799.14:0.00345,239066:0.03281)0.671.7:0.00283)1.000.1060:0.04633,580005:0.07324)0.690.6:0.01483)1.000.1061:0.06609,((388058:0.01224,109057:0.00792)1.000.1062:0.10129,((349447:0.02322,((241214:0.00811,559177:0.00823)1.000.1063:0.03115,246382:0.01002)0.982.86:0.0125)1.000.1064:0.03154,((218704:0.01838,(224375:0.01099,(223474:0.02762,520863:0.00912)0.414.3:0.01237)0.995.127:0.01577)0.988.107:0.01544,(233009:0.02733,(534885:0.02396,(((119236:0.01335,(339137:0.02083,(225970:0.00821,141274:0.01773)0.961.56:0.01156)0.934.41:0.00635)0.830.19:0.00594,(235795:0.01695,(251806:0.02129,226410:0.01165)0.948.57:0.00951)0.978.80:0.01299)0.970.57:0.00913,((531899:0.01541,509440:0.0175)0.916.43:0.00466,((159853:0.01801,239463:0.02171)0.770.13:0.00325,(49070:0.0455,(119981:0.0211,246321:0.02385)0.945.48:0.00965)0.901.32:0.00683)0.792.15:0.00216)0.479.10:0.00176)0.971.65:0.00988)0.381.6:0.00897)0.997.163:0.01636)0.722.10:0.01348)1.000.1065:0.05597)0.993.99:0.03688)0.721.8:0.01447,524002:0.16621)0.580.3:0.0126)0.947.54:0.01477)0.545.4:0.00936)0.970.58:0.01609,(((((((144150:0.00746,335016:0.01006)1.000.1066:0.02864,(556912:0.00628,549688:0.01936)0.999.313:0.01917)0.905.38:0.01261,(585095:0.03601,((587746:0.02017,((191331:0.00888,228859:0.01646)0.761.22:0.00537,(548839:0.01581,546411:0.00877)0.943.64:0.00947)0.995.128:0.01503)0.521.7:0.0027,((564021:0.01113,104678:0.00894)1.000.1067:0.04931,(548041:0.01441,573531:0.01033)0.215.5:0.00478)0.570.8:0.00428)0.995.129:0.02057)0.896.37:0.00866)1.000.1068:0.05563,((542396:0.0112,255102:0.00638)0.903.42:0.01151,(36814:0.01967,10762:0.01285)0.999.314:0.02551)1.000.1069:0.04248)0.999.315:0.02653,(381777:0.0201,542670:0.01351)1.000.1070:0.05622)0.912.31:0.01552,((568881:0.03349,565064:0.03169)1.000.1071:0.08661,((((346925:0.01453,(253712:0.0186,147174:0.01479)0.751.7:0.0043)0.955.57:0.008,(560172:0.02938,547351:0.03295)0.991.98:0.01488)0.999.316:0.02176,(((552030:0.02526,((76068:0.02534,(77057:0.01387,(241976:0.01278,253827:0.00993)0.999.317:0.01266)0.899.40:0.00687)0.573.5:0.00274,(((((575779:0.0133,(321022:0.00507,(247742:0.00996,588742:0.01015)0.408.11:0.00316)0.934.42:0.00702)0.812.23:0.00141,((89599:0.01726,532393:0.00878)0.257.3:0.00011,(581532:0.04192,144627:0.00352)0.848.27:0.0102)0.250.2:0.00083)0.903.43:0.00247,((237260:0.0375,(213030:0.02703,535437:0.03688)0.998.199:0.02531)0.979.69:0.01466,333757:0.00411)0.913.43:0.01108)0.727.11:0.00091,(328900:0.01003,583247:0.00688)0.854.22:0.01039)0.924.37:0.00494,593973:0.01151)0.996.137:0.01013)0.208.4:0.00448)0.057.5:0.00444,((264619:0.02393,572529:0.01629)0.995.130:0.01307,((555867:0.01634,267944:0.01702)0.666.8:0.00608,(512855:0.02546,(244297:0.02762,271790:0.01164)0.999.318:0.02601)0.828.13:0.00959)0.827.16:0.00481)0.991.99:0.01456)1.000.1072:0.03759,((560191:0.02962,535350:0.01828)1.000.1073:0.0407,(((((240393:0.00781,540681:0.02279)0.961.57:0.00826,(((550245:0.00321,(206015:0.0367,323943:0.00512)1.000.1074:0.03144)0.994.127:0.01324,48147:0.0108)0.537.5:0.00294,(((240726:0.0114,552802:0.0181)0.111.2:0.00198,534301:0.01382)0.790.15:0.00243,222237:0.00617)0.978.81:0.00722)0.707.7:0.00054)1.000.1075:0.02698,(((((256931:0.01025,589212:0.02232)0.990.86:0.01183,(192369:0.01861,60598:0.01514)0.878.41:0.00646)0.840.17:0.00291,((205913:0.00641,535087:0.01418)0.946.48:0.01016,(581516:0.03421,578972:0.03324)1.000.1076:0.02258)0.776.20:0.00564)0.143.5:0.00275,38614:0.02419)0.395.6:0.00206,333181:0.01985)0.902.27:0.01074)0.916.44:0.00962,(168547:0.01434,(214844:0.03624,308535:0.00556)0.943.65:0.0066)1.000.1077:0.02781)0.917.45:0.00842,(255111:0.02246,((((63257:0.0267,(247387:0.00967,((55388:0.00644,252578:0.02486)0.892.29:0.00404,246742:0.01281)0.940.51:0.00476)0.889.33:0.00385)0.401.5:0.00135,(226074:0.01678,302892:0.01376)0.090.2:0.00279)0.946.49:0.00574,248449:0.02791)0.981.71:0.01085,(238562:0.01352,103930:0.02222)0.885.37:0.00814)0.975.73:0.01243)1.000.1078:0.05176)0.843.18:0.00367)0.868.38:0.00647)0.878.42:0.00895)0.888.25:0.01365,((262091:0.02638,(254522:0.02018,(367695:0.0053,546729:0.00478)0.361.3:0.01842)0.357.6:0.00943)0.921.47:0.01582,(586588:0.02504,(182959:0.03409,(558961:0.01559,509231:0.0303)0.291.9:0.00933)0.362.2:0.01035)0.548.6:0.00984)1.000.1079:0.03507)0.994.128:0.02381)0.913.44:0.01296)0.661.6:0.00957,(((23625:0.09104,(571463:0.07229,(256201:0.05315,(317973:0.01128,581115:0.02026)0.997.164:0.03242)0.996.138:0.03567)0.967.60:0.02327)0.990.87:0.02367,(((150819:0.01058,71658:0.02536)1.000.1080:0.04177,(((148800:0.02267,264210:0.01997)0.992.120:0.01901,((328791:0.0181,258754:0.0252)0.996.139:0.01531,((170623:0.00459,334324:0.01475)1.000.1081:0.03352,(235691:0.02782,136925:0.02366)0.678.6:0.01005)0.695.5:0.00689)0.965.56:0.01304)1.000.1082:0.04029,(10703:0.10104,((348822:0.02428,((261435:0.03972,(165369:0.01295,(195967:0.01988,(564760:0.02079,236923:0.02154)0.244.2:0.00473)0.874.31:0.00545)1.000.1083:0.02446)0.919.38:0.00675,((588965:0.03125,(163996:0.01962,344633:0.0154)0.955.58:0.01145)0.999.319:0.02181,(136770:0.01264,334901:0.02205)0.998.200:0.01528)0.897.33:0.01122)0.068.2:0.00415)0.342.5:0.00659,(558364:0.02821,(((344456:0.02539,296500:0.0408)0.981.72:0.0139,((278639:0.0078,(537463:0.02342,341075:0.00264)0.349.3:0.01753)0.999.320:0.01528,(((593921:0.00805,(565517:0.02506,(((140293:0.01948,(537008:0.014,112279:0.01648)0.960.68:0.00613)0.456.6:0.00015,(576102:0.00983,571382:0.02208)0.976.70:0.01554)0.853.26:0.00413,10702:0.01376)0.510.3:0.00485)0.909.39:0.0059)1.000.1084:0.03624,((((268536:0.00866,(550835:0.00608,111722:0.02447)0.463.7:0.00394)0.999.321:0.02275,((543054:0.01729,(546261:0.01783,559828:0.00979)0.992.121:0.01145)0.972.67:0.01114,(361919:0.01626,355585:0.02426)0.967.61:0.00849)0.843.19:0.00335)1.000.1085:0.01662,(178175:0.01216,(203027:0.00652,154831:0.00014)1.000.1086:0.01912)0.973.65:0.00857)0.102.3:0.00244,162701:0.02362)0.885.38:0.00457)0.989.92:0.01133,((349173:0.00898,562069:0.0186)0.998.201:0.01557,(159917:0.01685,563535:0.02543)0.923.40:0.00972)0.975.74:0.00979)0.884.39:0.00336)0.259.8:0.00416)0.941.60:0.01198,(((((325676:0.0242,((325961:0.01179,((362677:0.01629,(165188:0.01424,(253113:0.0123,357893:0.00357)0.952.56:0.00702)0.883.33:0.00455)0.523.5:0.00195,309636:0.01006)0.685.4:0.00397)1.000.1087:0.0158,(((249141:0.0097,(250580:0.01593,321652:0.01505)0.762.22:0.00707)0.977.71:0.00879,585877:0.01065)0.992.122:0.00826,(315157:0.01124,(270803:0.01784,(246832:0.01266,270541:0.0104)0.903.44:0.00554)0.971.66:0.00729)0.555.5:0.00286)0.980.66:0.0093)0.343.6:0.00546)0.421.5:0.00284,(107311:0.01768,(200746:0.01975,250443:0.02215)0.469.4:0.00312)0.892.30:0.00575)0.877.32:0.00525,(((259833:0.02669,572190:0.00573)0.979.70:0.00918,559699:0.00892)0.868.39:0.00374,308197:0.0179)0.526.4:0.00568)0.790.16:0.00383,(236118:0.00966,(273351:0.01689,263721:0.00663)0.987.116:0.00874)0.850.25:0.00462)0.289.2:0.00336,338555:0.02658)0.944.39:0.00533)0.297:0.00586)0.807.11:0.00458)0.422.7:0.00684)0.040.3:0.01095)0.700.11:0.0107)0.992.123:0.02122,(((((((329145:0.01066,(195498:0.01198,(312636:0.004,(572150:0.00982,560979:0.00752)0.874.32:0.00159)0.728.12:0.01105)0.738.18:0.00078)0.952.57:0.00321,544098:0.01043)0.485.2:0.05217,(((159572:0.02095,(168400:0.02033,569799:0.02295)0.656.3:0.00567)0.906.37:0.00753,((341251:0.01278,279076:0.01875)0.957.76:0.00973,(274528:0.03942,((225442:0.02232,267841:0.01795)1.000.1088:0.03459,(240725:0.02638,((208701:0.01527,254949:0.00938)0.935.41:0.00727,((484709:0.0189,(238254:0.01923,258814:0.01299)0.903.45:0.00563)0.786.21:0.00239,(217748:0.01987,(237497:0.01207,586804:0.03266)0.967.62:0.00801)0.970.59:0.009)0.979.71:0.0112)1.000.1089:0.0263)0.995.131:0.01779)0.897.34:0.01204)0.222.6:0.00576)0.865.32:0.00711)0.977.72:0.00936,141905:0.0398)0.736.15:0.00727)0.998.202:0.02725,(((((254628:0.02869,(198233:0.03528,236868:0.02918)0.749.13:0.00954)0.642.10:0.0101,315558:0.01631)1.000.1090:0.03845,(((((560806:0.04791,(((591647:0.01004,(257041:0.01533,590356:0.022)0.973.66:0.01057)0.946.50:0.01127,((259881:0.02124,237963:0.02004)0.940.52:0.00593,554552:0.01806)0.101.3:0.0015)0.935.42:0.00723,(572691:0.03667,350825:0.02047)0.999.322:0.02305)0.924.38:0.00763)0.596.6:0.00398,(114327:0.02089,((368942:0.02919,(364646:0.01343,332361:0.00595)0.999.323:0.01654)0.964.77:0.00882,244127:0.05212)0.402.4:0.00252)0.279.5:0.00366)0.999.324:0.0188,((((((359365:0.01299,220256:0.0203)0.859.33:0.01152,539276:0.00335)0.917.46:0.00953,553302:0.0121)1.000.1091:0.02256,((218567:0.02672,253567:0.03749)0.579.4:0.00939,(((219369:0.01291,(113261:0.00505,562536:0.00921)0.958.63:0.00787)0.962.57:0.00555,(273678:0.02105,((539862:0.0061,360433:0.02026)0.978.82:0.01749,(225155:0.00425,114170:0.00531)0.888.26:0.00661)0.940.53:0.00014)0.787.14:0.00654)0.993.100:0.01141,((((249926:0.01062,(331980:0.00923,156870:0.02004)0.644.8:0.00774)0.316.6:0.00193,(260335:0.02376,(226704:0.01736,586966:0.00835)0.428.5:0.00264)1.000.1092:0.01832)0.944.40:0.0053,(545917:0.01746,368706:0.02053)0.972.68:0.00816)0.862.38:0.00256,251592:0.02065)0.957.77:0.00588)0.906.38:0.00559)0.991.100:0.01347)0.971.67:0.01366,(576469:0.02719,565643:0.02539)0.998.203:0.02264)0.870.34:0.00547,(137235:0.0081,539496:0.01113)0.997.165:0.01592)0.987.117:0.01286)0.974.71:0.01284,(((((82906:0.01383,113748:0.01551)0.963.65:0.00663,(10645:0.00814,362520:0.01045)0.998.204:0.01254)0.946.51:0.00463,((347524:0.01059,588131:0.01254)0.979.72:0.00808,(((((108677:0.00714,355920:0.01314)0.854.23:0.00338,(145144:0.00868,346923:0.01789)0.997.166:0.01208)0.880.29:0.00253,(329035:0.01288,535402:0.01894)0.803.15:0.00192)0.805.21:0.00187,579806:0.00884)0.886.31:0.00316,(136914:0.01487,326414:0.01356)0.938.52:0.00603)0.545.5:0.00191)0.885.39:0.00281)0.136.2:0.00223,(10657:0.01917,(245900:0.01764,((222900:0.01203,71817:0.00911)0.957.78:0.00782,551380:0.03017)0.095.3:0.00326)0.975.75:0.00759)0.746.21:0.00293)0.924.39:0.0062,203878:0.02708)0.939.52:0.00755)0.286.7:0.00373,(331020:0.01413,(((241205:0.0174,594373:0.00869)0.313.9:0.00411,(137363:0.01688,256765:0.01772)0.734.12:0.00484)0.804.17:0.00397,542445:0.01778)0.851.27:0.00339)0.997.167:0.01243)0.780.12:0.00428)0.308.4:0.00749,70239:0.08063)0.940.54:0.01503,(((154217:0.01774,45556:0.00423)1.000.1093:0.08063,543094:0.03475)0.954.51:0.01513,(561765:0.00577,(176741:0.02085,(180089:0.01236,522005:0.02022)0.071.3:0.00691)0.981.73:0.01279)1.000.1094:0.05584)0.557.2:0.01154)0.800.18:0.00835)0.885.40:0.00939,(((((((236963:0.00894,(240744:0.01028,(197062:0.01238,112562:0.0132)0.747.16:0.00244)0.843.20:0.00723)0.985.68:0.01741,(204177:0.02308,204372:0.01459)1.000.1095:0.01932)0.025.2:0.00582,(206277:0.03053,216600:0.01973)0.286.8:0.00887)0.985.69:0.01501,((239367:0.02262,542734:0.02718)0.382.5:0.00471,((226488:0.01057,256915:0.03517)0.876.34:0.00933,(578392:0.0153,(224129:0.01612,574674:0.01653)0.891.36:0.00623)0.892.31:0.00632)0.987.118:0.01255)0.974.72:0.01207)0.981.74:0.01155,(((240788:0.01618,209489:0.01775)0.976.71:0.00985,((578408:0.02079,570453:0.02777)0.994.129:0.01513,((253246:0.0265,(74929:0.03181,(248757:0.02094,(152024:0.01512,543753:0.01392)0.958.64:0.00795)0.959.52:0.00792)0.725.13:0.00291)0.952.58:0.01029,238968:0.02441)0.791.22:0.00568)0.950.74:0.00895)0.901.33:0.00802,(((546642:0.01197,((180118:0.01196,(236984:0.00775,10652:0.02095)0.893.32:0.00339)0.353.6:0.00179,533367:0.00567)0.998.205:0.01347)0.741.17:0.00679,((236312:0.01498,345106:0.00338)0.998.206:0.01325,((313196:0.01746,(353947:0.02094,572137:0.01896)0.890.36:0.00644)0.960.69:0.00779,((279253:0.01387,(530479:0.0105,(244128:0.00629,173727:0.01923)0.292.10:0.00384)0.901.34:0.00523)0.082.3:0.00238,(333636:0.0231,(10682:0.00772,578632:0.01932)1.000.1096:0.02479)0.841.21:0.00511)0.919.39:0.00524)0.913.45:0.0049)0.050.3:0.0045)0.985.70:0.01796,((253255:0.03065,553485:0.02566)0.990.88:0.01289,(((((239581:0.02897,((303295:0.00578,(566634:0.00922,281524:0.01462)0.904.45:0.005)0.865.33:0.00342,((((279206:0.0016,(547658:0.0061,136820:0.01175)0.872.45:0.00201)0.868.40:0.00156,(141267:0.00912,(305498:0.00825,342714:0.006)0.847.25:0.00166)0.979.73:0.00547)0.883.34:0.00175,((218845:0.00095,217328:0.03331)0.987.119:0.00254,(114341:0.02232,((220702:0.00511,76933:0.0284)0.760.23:0.00072,137564:0.03339)0.842.25:0.00164)0.469.5:0.00015)0.886.32:0.0022)0.475.2:0.00238,(99343:0.0058,278622:0.00316)0.811.19:0.00465)0.683.11:0.0014)0.512.7:0.00281)0.861.34:0.0017,(331288:0.00573,352023:0.0055)0.862.39:0.00312)0.899.41:0.00261,((113379:0.00594,268169:0.00279)0.909.40:0.01114,(313833:0.00921,(156882:0.00466,154207:0.01665)0.545.6:0.00521)0.210.5:0.00291)0.934.43:0.00494)0.839.28:0.00195,(345643:0.01235,534810:0.01841)0.971.68:0.007)0.959.53:0.00732,(((242637:0.00628,283637:0.01983)0.985.71:0.00937,((577309:0.01742,(30991:0.01161,305616:0.00849)0.806.15:0.00763)0.852.38:0.00284,(240045:0.01893,564418:0.01016)0.925.34:0.00633)0.963.66:0.00638)0.991.101:0.01166,(113228:0.02153,(533124:0.01266,(238764:0.01521,553358:0.01386)0.112.2:0.00342)0.387.8:0.00494)0.139.3:0.00016)0.962.58:0.00672)0.911.40:0.01029)0.990.89:0.01904)0.997.168:0.02204)0.928.43:0.01002)0.615.3:0.00774,((246356:0.00749,221219:0.01129)1.000.1097:0.02759,(((540592:0.00674,((563875:0.01416,351128:0.00014)0.870.35:0.00223,300085:0.00832)0.563.6:0.00107)0.965.57:0.00615,((564949:0.00893,((565027:0.00805,((239539:0.00998,586306:0.01022)0.564.6:0.0022,309226:0.00962)0.079:0.00349)0.796.17:0.00164,256241:0.00959)0.867.29:0.00232)0.990.90:0.00682,(565375:0.02088,(593659:0.01391,547926:0.0118)0.321.5:0.00072)0.902.28:0.00324)0.998.207:0.01629)0.943.66:0.00734,(207878:0.03438,256412:0.02383)0.993.101:0.02499)0.996.140:0.02354)0.376.7:0.00672)0.991.102:0.01959,(((((135973:0.01307,(253828:0.01892,(((325716:0.01072,556903:0.01453)0.267.4:0.00213,(304479:0.01245,(585473:0.01526,85080:0.01622)0.962.59:0.00624)0.709.7:0.00267)0.920.33:0.00632,(222114:0.00909,245344:0.01773)0.173.5:0.0037)0.649.6:0.00248)0.995.132:0.00986)0.809.15:0.00309,327106:0.00562)1.000.1098:0.02689,(263386:0.04303,(152915:0.01154,((246426:0.01045,570862:0.00799)1.000.1099:0.0221,(513263:0.01741,546538:0.01897)0.891.37:0.00419)0.135.2:0.00173)0.944.41:0.00786)0.952.59:0.01044)0.938.53:0.01302,259971:0.03641)0.430.6:0.00332,(314203:0.01655,10651:0.03346)0.977.73:0.01461)1.000.1100:0.06362)0.996.141:0.01982)0.995.133:0.02148,((254386:0.00921,(219204:0.01819,136119:0.03827)0.908.43:0.00745)1.000.1101:0.04022,((236608:0.02035,(264188:0.01585,246618:0.0189)0.459.2:0.0098)0.999.325:0.03088,(511782:0.01682,243003:0.02957)1.000.1102:0.03943)0.999.326:0.02849)0.994.130:0.02263)0.510.4:0.00538,(((220025:0.01142,10596:0.02007)1.000.1103:0.05657,(((148517:0.00967,(((582110:0.0041,234490:0.01611)0.757.22:0.0175,(113159:0.00631,242301:0.00937)0.749.14:0.00463)0.757.23:0.00322,571531:0.02641)0.964.78:0.0066)0.843.21:0.00158,(585303:0.03487,(179879:0.0088,169586:0.0177)0.745.18:0.00077)0.773.18:0.00081)0.953.51:0.02934,((580197:0.00383,552338:0.01294)1.000.1104:0.03777,(575313:0.01928,(326656:0.00575,(((((((251842:0.01573,265680:0.00974)0.742.21:0.00074,(140591:0.00929,10633:0.0111)0.969.66:0.00395)0.927.51:0.00643,172129:0.01475)0.747.17:0.00128,133876:0.01206)0.554.6:0.00366,((137984:0.01437,336745:0.00916)0.784.17:0.00361,584276:0.02009)0.961.58:0.00657)0.726.12:0.00109,((511742:0.03174,240506:0.00839)0.128.4:0.00569,237206:0.02699)0.957.79:0.00844)0.949.52:0.00591,((250244:0.01673,((((337571:0.00705,(221410:0.01004,65455:0.01729)0.268.4:0.00242)0.953.52:0.00578,(236994:0.01817,141394:0.02567)1.000.1105:0.01952)0.969.67:0.00632,(254471:0.01429,278697:0.01498)0.978.83:0.01028)0.986.81:0.00999,501815:0.01168)0.894.35:0.00404)0.879.40:0.00281,(104020:0.05532,(203689:0.00857,237842:0.01548)0.598.5:0.00179)0.859.34:0.00219)0.861.35:0.00173)0.967.63:0.0056)0.921.48:0.00496)0.985.72:0.01212)0.979.74:0.01772)0.494.7:0.00938)0.997.169:0.02849,((244674:0.01883,((159288:0.01895,144289:0.0188)1.000.1106:0.02764,(100929:0.02566,266085:0.01453)0.853.27:0.00473)0.574.3:0.01149)1.000.1107:0.06281,(((407306:0.01715,137768:0.00261)1.000.1108:0.04266,(144097:0.03344,268363:0.02545)0.989.93:0.02297)0.948.58:0.01343,(((144578:0.00405,141222:0.00214)0.994.131:0.02245,(247812:0.00797,257469:0.00997)0.751.8:0.00916)1.000.1109:0.03644,(227986:0.03784,(230200:0.03104,228943:0.01367)0.965.58:0.01558)0.997.170:0.028)0.950.75:0.01326)0.992.124:0.01909)0.884.40:0.01274)0.197.4:0.00554)0.996.142:0.01829)0.974.73:0.01246)0.913.46:0.0091,((((((113721:0.01862,143335:0.00738)0.312.3:0.00237,244045:0.03245)0.922.45:0.00456,((((146553:0.01438,532352:0.00838)0.349.4:0.00227,142149:0.01272)0.754.16:0.00124,(((220531:0.01371,((((((102276:0.00883,(552927:0.01175,592983:0.00609)0.890.37:0.00687)0.728.13:0.00569,591167:0.02045)0.763.20:0.00966,223785:0.00028)0.962.60:0.01042,511698:0.01288)0.886.33:0.00156,(582001:0.00327,104157:0.01108)0.974.74:0.00399)0.961.59:0.00459,(166514:0.00685,251985:0.00281)0.929.54:0.00368)0.269.3:0.00341)0.987.120:0.00719,(149023:0.00999,((527719:0.00675,(255281:0.01105,571587:0.01922)0.406.3:0.00265)0.994.132:0.01092,143933:0.00774)0.232.6:0.0031)0.979.75:0.00665)0.858.21:0.0024,326474:0.01571)0.910.41:0.00658)0.822.20:0.00279,((143994:0.01785,(109573:0.0118,590693:0.01696)0.980.67:0.00822)0.963.67:0.00557,((590579:0.00761,202514:0.01466)0.998.208:0.01367,(((242051:0.0132,160740:0.01951)0.925.35:0.00516,((257690:0.01012,(592942:0.00633,(224061:0.00879,(108346:0.00757,223706:0.01682)0.854.24:0.00425)0.893.33:0.00307)0.878.43:0.0022)0.855.25:0.00199,(590664:0.02269,((105662:0.00718,239068:0.01685)0.782.21:0.00511,(149523:0.01095,10720:0.01156)0.803.16:0.00413)0.918.51:0.00363)0.740.21:0.00111)0.918.52:0.00341)0.882.34:0.0036,(((337418:0.00326,163641:0.01707)0.757.24:0.01781,(((142841:0.01597,(251258:0.00078,23462:0.01268)0.871.43:0.01013)0.876.35:0.00244,((10733:0.0058,10714:0.00604)0.913.47:0.00502,(236914:0.00949,90570:0.00734)0.131.4:0.00541)0.720.11:0.00249)0.982.87:0.00788,258079:0.00554)0.949.53:0.0046)0.932.47:0.00872,(142651:0.01163,128993:0.01456)0.993.102:0.01122)0.940.55:0.00446)0.755.20:0.00321)0.304.4:0.00176)0.919.40:0.00014)0.888.27:0.0032)0.897.35:0.00382,(534035:0.03529,(141881:0.01395,549687:0.00428)0.939.53:0.00544)0.979.76:0.01048)0.999.327:0.02091,(((298732:0.00539,(271536:0.00344,236199:0.02933)0.983.84:0.01294)0.918.53:0.01222,(538924:0.00498,(145062:0.00958,276490:0.01653)0.604.6:0.00737)1.000.1110:0.02842)0.561.8:0.0041,(((((108505:0.01949,(10668:0.004,(550496:0.01042,(153819:0.0015,287305:0.02155)0.988.108:0.00665)0.950.76:0.00619)0.157.3:0.00731)0.852.39:0.00765,(41059:0.00818,(10672:0.00855,103899:0.01701)0.674.7:0.01005)0.982.88:0.00879)0.871.44:0.0028,((575070:0.00697,(235638:0.02159,(159422:0.0119,343915:0.00873)0.339.4:0.00813)0.883.35:0.00424)0.999.328:0.01131,(525862:0.00129,210461:0.05174)0.996.143:0.01496)0.647.4:0.00501)0.781.19:0.00441,((((302024:0.00562,554911:0.01553)0.980.68:0.00545,(((283864:0.01642,536485:0.01022)0.864.38:0.00305,338040:0.00767)0.373.3:0.00731,580913:0.00716)0.646.6:0.00016)0.390.4:0.01816,263005:0.01312)0.920.34:0.00362,(252121:0.02848,10662:0.01938)0.345.5:0.00552)0.910.42:0.00463)0.133.2:0.00365,((81506:0.01424,559852:0.02229)0.891.38:0.01144,(511753:0.01247,((247130:0.00894,449515:0.01321)0.439.9:0.01444,((((237074:0.0239,(286079:0.01117,545048:0.01178)0.883.36:0.0027)0.928.44:0.00351,(74843:0.00778,(289410:0.00734,266544:0.011)0.867.30:0.00235)0.637.8:0.00155)0.935.43:0.00867,(539100:0.00967,(552396:0.01455,329004:0.01295)0.283.2:0.00384)0.906.39:0.00442)0.870.36:0.00198,(110229:0.00815,78716:0.00604)0.978.84:0.00698)0.932.48:0.00557)0.997.171:0.01682)0.390.5:0.00487)0.771.18:0.00465)0.945.49:0.0076)0.869.41:0.00934)0.994.133:0.02379,(((261919:0.00392,591266:0.0042)1.000.1111:0.04807,((231902:0.01711,(149878:0.00654,(234196:0.01073,233254:0.00595)0.841.22:0.00799)0.985.73:0.01421)0.998.209:0.0226,(136482:0.0254,233453:0.02575)1.000.1112:0.04027)0.933.47:0.01075)0.131.5:0.00752,(253083:0.01523,336493:0.02032)0.999.329:0.02426)1.000.1113:0.05384)0.646.7:0.00986)0.891.39:0.01126)0.897.36:0.00827)0.748.18:0.00432)0.996.144:0.01567)0.566.7:0.00832)0.870.37:0.00863)0.834.27:0.01007)0.811.20:0.00444)0.990.91:0.01559,((((((95866:0.00399,556167:0.00967)0.999.330:0.03835,((106690:0.00906,(255924:0.00972,515827:0.01058)0.982.89:0.01)0.822.21:0.01187,(567970:0.02405,(143463:0.02198,(225464:0.00887,(576066:0.0283,513019:0.01335)0.107.3:0.0045)0.988.109:0.01389)0.732.10:0.00538)0.526.5:0.00724)0.998.210:0.02834)1.000.1114:0.07234,((((141160:0.02711,(144219:0.00222,(147755:0.0078,149003:0.00642)0.974.75:0.00831)0.395.7:0.00355)1.000.1115:0.05813,(522127:0.01684,(537538:0.02259,(325820:0.01428,513979:0.01952)0.995.134:0.01564)0.676.8:0.00785)0.566.8:0.01339)1.000.1116:0.05182,((144494:0.01134,(149384:0.01337,(164433:0.01425,221488:0.01045)1.000.1117:0.03976)0.473.5:0.02468)1.000.1118:0.11181,(317953:0.01255,560472:0.01616)0.996.145:0.03548)1.000.1119:0.05789)0.999.331:0.04276,((560552:0.09208,(((99903:0.06471,((((511393:0.01907,(252875:0.02256,255842:0.01429)0.964.79:0.01126)0.993.103:0.02062,(88595:0.01799,(547407:0.01607,247813:0.02452)0.889.34:0.00571)0.981.75:0.01667)0.921.49:0.01444,(575810:0.03406,(91023:0.02389,(99743:0.01487,101263:0.02938)0.342.6:0.00773)0.999.332:0.02059)0.326.9:0.00811)0.983.85:0.01548,((240724:0.01397,588053:0.03556)0.450.6:0.00558,(213866:0.02611,((((269562:0.02179,((591641:0.04163,99883:0.00453)0.999.333:0.01592,589318:0.02247)0.991.103:0.011)0.386.4:0.00418,((151437:0.00939,(256995:0.02411,(564034:0.00932,556862:0.0108)1.000.1120:0.02661)0.832.19:0.00394)0.953.53:0.00706,(509234:0.02214,363429:0.01797)0.995.135:0.01189)0.895.40:0.00485)0.272.3:0.00365,577176:0.03135)0.617.4:0.00418,149719:0.02062)0.661.7:0.00173)0.849.19:0.0088)0.770.14:0.01433)1.000.1121:0.03538)0.678.7:0.01358,((((41844:0.02554,(239595:0.02789,203705:0.02968)0.768.21:0.01622)1.000.1122:0.05784,((258229:0.01207,(30516:0.00652,263133:0.00463)0.881.34:0.01202)1.000.1123:0.09213,(72373:0.06745,(253338:0.01318,114305:0.00921)0.999.334:0.04711)1.000.1124:0.07886)0.522.6:0.02188)0.982.90:0.02793,((91225:0.01347,142961:0.01627)1.000.1125:0.09838,(((141738:0.02303,60815:0.03315)0.743.24:0.00885,(241447:0.02045,(321779:0.02699,222624:0.03391)0.996.146:0.02139)0.999.335:0.02655)0.913.48:0.01173,((2677:0.01456,224048:0.01063)1.000.1126:0.04897,((152291:0.03149,(214114:0.0025,360408:0.00186)1.000.1127:0.04268)0.998.211:0.02475,(586390:0.06405,(60814:0.01387,(563907:0.0162,114547:0.01302)0.981.76:0.01179)0.982.91:0.01608)0.283.3:0.00856)0.414.4:0.00238)0.970.60:0.00878)0.980.69:0.01477)0.992.125:0.01978)0.140.2:0.00961,((152302:0.03733,(373685:0.00697,(310286:0.01464,83562:0.02022)0.977.74:0.01152)1.000.1128:0.03151)1.000.1129:0.03939,((408758:0.00871,569201:0.00922)1.000.1130:0.02893,(((196940:0.02234,(172221:0.01491,145701:0.01149)0.907.36:0.00535)0.374.4:0.0048,(565419:0.01472,(170662:0.02313,149135:0.00475)0.981.77:0.01263)0.934.44:0.0093)0.945.50:0.00815,(565948:0.03085,((147632:0.01035,561793:0.01608)1.000.1131:0.0202,(578382:0.02196,581318:0.03684)0.972.69:0.01399)0.222.7:0.00343)0.433.5:0.00351)0.960.70:0.01038)0.998.212:0.03624)0.992.126:0.02811)0.916.45:0.00782)0.986.82:0.01698,(((((((253625:0.01556,554668:0.01635)0.881.35:0.00486,((114215:0.00822,(154451:0.02061,140324:0.00411)0.206.3:0.01593)1.000.1132:0.03759,(62487:0.01344,139517:0.01619)0.110:0.00154)0.949.54:0.00993)0.991.104:0.0203,(((330672:0.01377,(105775:0.02177,100508:0.00285)0.916.46:0.00976)0.970.61:0.01749,(320692:0.01196,(331718:0.01829,337360:0.0149)0.096.6:0.00198)1.000.1133:0.02562)1.000.1134:0.03516,((304760:0.02957,579287:0.01599)0.999.336:0.02394,(((140313:0.00151,62692:0.00774)0.996.147:0.01319,(253662:0.01292,(587149:0.03359,(((278678:0.00551,314591:0.02076)0.972.70:0.00741,278854:0.00659)0.191.2:0.00172,201254:0.01484)0.919.41:0.00774)0.828.14:0.00817)0.990.92:0.00994)0.399.5:0.0036,((237373:0.01016,(314015:0.00319,579339:0.00233)0.746.22:0.02232)0.811.21:0.00334,(352504:0.026,((355541:0.01773,307759:0.01426)0.548.7:0.00676,(332368:0.00989,334417:0.0165)0.300.2:0.00319)0.747.18:0.00843)0.200.7:0.00186)0.907.37:0.00553)0.974.76:0.00974)0.888.28:0.01236)0.599.6:0.01114)0.992.127:0.02888,((562556:0.01108,593267:0.00268)0.051.5:0.08436,80875:0.04041)0.413.3:0.03103)1.000.1135:0.05403,(((275952:0.01858,290158:0.01594)1.000.1136:0.06693,(351651:0.01665,241269:0.03247)1.000.1137:0.06329)0.969.68:0.02508,(((576168:0.00772,566078:0.0325)0.150.3:0.15072,(((2632:0.00679,2631:0.00015)1.000.1138:0.13194,2626:0.08089)0.919.42:0.02213,(201378:0.31944,17215:0.07421)0.999.337:0.06007)0.464.4:0.008)0.778.21:0.00297,((220753:0.01506,159936:0.02249)1.000.1139:0.11914,((((((((((320231:0.01308,103624:0.02969)0.976.72:0.01604,((234079:0.01743,525955:0.02018)0.671.8:0.00493,((256454:0.01103,101215:0.01076)0.836.22:0.00222,(81966:0.00649,28868:0.01636)1.000.1140:0.02029)0.666.9:0.00384)0.986.83:0.01729)1.000.1141:0.07294,(((190481:0.03123,342087:0.00769)1.000.1142:0.02546,(583651:0.03079,(66632:0.02511,573049:0.01662)0.960.71:0.01466)0.986.84:0.01999)0.906.40:0.01024,((230500:0.03346,(((232905:0.00357,(172515:0.00917,576998:0.01803)0.939.54:0.00767)0.995.136:0.01186,(91513:0.00888,578477:0.02039)0.267.5:0.00175)0.971.69:0.01043,(233819:0.0194,(593420:0.0235,590041:0.00029)0.840.18:0.02393)0.931.43:0.00683)0.264.4:0.00434)1.000.1143:0.02525,((141856:0.00365,576960:0.00179)0.907.38:0.01665,(141750:0.02412,(539275:0.02019,(575472:0.01853,351903:0.01718)0.830.20:0.0075)0.925.36:0.00738)0.980.70:0.01247)1.000.1144:0.0555)0.965.59:0.01339)0.932.49:0.01576)0.251.4:0.01668,(240929:0.06457,(187086:0.06522,(426848:0.05333,(254309:0.01917,544037:0.01914)0.999.338:0.02924)0.702.10:0.01485)0.963.68:0.02388)0.980.71:0.01863)0.715.10:0.00498,(((140681:0.03401,(61105:0.03838,(278327:0.02364,319175:0.013)0.994.134:0.02359)1.000.1145:0.03418)0.991.105:0.0146,((248998:0.04593,561016:0.02535)1.000.1146:0.03354,589627:0.04112)0.928.45:0.01102)0.965.60:0.00901,((((560135:0.02125,(337438:0.02043,539543:0.03149)0.602.4:0.01014)0.998.213:0.03325,(79684:0.05238,133610:0.03131)1.000.1147:0.03077)0.942.53:0.01353,((192107:0.04023,((146541:0.03591,((200540:0.01651,542989:0.0004)0.999.339:0.02025,(450857:0.02283,356497:0.01602)0.188.5:0.00507)0.891.40:0.01318)0.997.172:0.02153,(575686:0.01913,(82792:0.00158,(339261:0.02331,118250:0.00164)0.954.52:0.0041)0.935.44:0.0034)1.000.1148:0.0276)0.925.37:0.01155)1.000.1149:0.02598,(100077:0.02405,(130262:0.02918,(278158:0.01604,80562:0.0351)0.370.6:0.01055)0.855.26:0.00825)0.975.76:0.01325)0.915.40:0.01016)0.940.56:0.01226,((((254573:0.01342,511108:0.01355)1.000.1150:0.06237,(252859:0.04837,(113227:0.04159,(562142:0.01169,537682:0.012)0.998.214:0.02708)0.986.85:0.02825)0.852.40:0.01512)0.812.24:0.00429,((541389:0.00741,152557:0.00831)1.000.1151:0.0606,((268391:0.00604,(324212:0.03111,107505:0.01777)0.922.46:0.00749)1.000.1152:0.05961,((95582:0.01961,146507:0.01684)1.000.1153:0.02759,(545820:0.02662,(156530:0.00861,(232542:0.01233,154321:0.01382)0.999.340:0.01611)1.000.1154:0.02585)0.984.74:0.01378)0.954.53:0.00944)0.458.4:0.00691)0.975.77:0.0132)0.979.77:0.01548,(((165490:0.01442,((122966:0.00421,101620:0.01427)0.353.7:0.00218,232612:0.01065)0.923.41:0.0058)1.000.1155:0.02346,((160898:0.02678,(583037:0.01421,560419:0.01422)1.000.1156:0.02023)0.502.5:0.00433,(578324:0.01876,(((591187:0.02589,(161664:0.00316,549336:0.0192)0.881.36:0.00159)0.785.18:0.00325,(246595:0.01231,582159:0.01224)0.967.64:0.00683)1.000.1157:0.01767,((548838:0.00252,242779:0.02494)0.770.15:0.00156,118269:0.00157)0.870.38:0.01393)0.988.110:0.0135)0.908.44:0.00724)1.000.1158:0.0366)0.996.148:0.01909,(228608:0.07669,233682:0.03251)0.999.341:0.01831)0.979.78:0.01046)0.732.11:0.00484)0.898.42:0.00677)0.964.80:0.00922)0.960.72:0.01101,(((556891:0.0274,556785:0.02512)1.000.1159:0.05997,((((580004:0.03663,(((78228:0.02073,(((302349:0.01707,(286813:0.00015,(77676:0.01017,109300:0.02042)0.113.4:0.00281)0.952.60:0.00555)0.063:0.00254,299050:0.01568)0.966.59:0.00679,((((239029:0.00442,345451:0.01078)0.875.36:0.00218,(361177:0.01797,(250747:0.00564,150212:0.00711)0.995.137:0.0093)0.198.3:0.00183)0.740.22:0.0018,518689:0.01479)0.865.34:0.00158,(((191828:0.00914,576486:0.01269)0.473.6:0.00341,591430:0.00477)0.871.45:0.0074,249994:0.00897)0.905.39:0.00296)0.942.54:0.00397)0.748.19:0.00137)0.190.3:0.00562,(216031:0.01721,529450:0.00333)0.997.173:0.0197)0.993.104:0.01589,((261907:0.01545,237287:0.01654)0.999.342:0.01705,(145981:0.01558,(330803:0.01808,((144996:0.01551,143730:0.01084)0.856.27:0.00838,88682:0.00424)0.856.28:0.00404)0.980.72:0.00772)0.904.46:0.00633)0.971.70:0.01539)0.920.35:0.00744)0.322.3:0.00532,(((579831:0.02324,(((((((256063:0.00121,24630:0.03387)0.775.15:0.00117,(70738:0.01105,(250190:0.00474,245153:0.00408)0.800.19:0.00128)0.351.4:0.00159)0.902.29:0.00464,578850:0.00803)1.000.1160:0.02943,((((((355477:0.01383,579719:0.00749)0.884.41:0.00256,304863:0.01734)0.987.121:0.00956,(229999:0.01137,(105331:0.0003,230145:0.02531)0.874.33:0.00446)0.997.174:0.0138)0.998.215:0.01797,(551808:0.01115,188334:0.01572)0.976.73:0.00818)0.138.4:0.00731,230843:0.01731)0.999.343:0.02055,((((348941:0.01948,233439:0.0081)0.914.33:0.0044,171189:0.01751)0.902.30:0.003,((341072:0.00733,(170720:0.01366,588375:0.00735)0.735.14:0.00154)0.994.135:0.00824,(((337899:0.01841,(540127:0.0184,((((118817:0.00334,343544:0.01033)0.938.54:0.00428,(233858:0.01391,228487:0.00498)0.934.45:0.00398)0.755.21:0.00081,341186:0.00471)0.970.62:0.00823,308318:0.0073)0.756.23:0.00125)0.482.6:0.00164)0.948.59:0.00459,(354757:0.00016,582330:0.02502)0.987.122:0.00669)0.932.50:0.00563,563825:0.01518)0.965.61:0.0067)0.816.21:0.00321)0.833.16:0.0022,(((272995:0.00766,(355726:0.00501,318365:0.01966)0.539.5:0.0032)1.000.1161:0.03056,(312094:0.03055,(314108:0.02197,573052:0.00419)0.989.94:0.00971)0.904.47:0.00767)0.439.10:0.00201,548700:0.02193)0.889.35:0.0034)0.992.128:0.01529)0.941.61:0.01243)0.948.60:0.01262,((325922:0.01731,(223113:0.03707,61970:0.00987)0.995.138:0.01741)0.959.54:0.01037,(((((71957:0.01635,249740:0.00302)0.985.74:0.01488,((530224:0.00291,570358:0.00092)1.000.1162:0.00619,(2375:0.02244,469897:0.01)0.876.36:0.00651)0.994.136:0.01634)0.984.75:0.01471,(2371:0.04263,((((465094:0.01213,95610:0.00723)0.914.34:0.00396,314128:0.00557)0.734.13:0.00015,2383:0.00014)0.992.129:0.01239,(125969:0.02526,((362663:0.01439,((18298:0.00235,((2394:0.00356,(59534:0.00559,149379:0.01517)0.878.44:0.0018)0.902.31:0.00446,165694:0.00079)0.958.65:0.00705)0.890.38:0.00015,((126521:0.00105,56208:0.00079)0.997.175:0.00818,2370:0.00307)0.947.55:0.00399)0.956.56:0.00495)0.655.7:0.00518,((249209:0.00154,162320:0.00104)0.980.73:0.00701,(244595:0.00184,167376:0.00028)0.991.106:0.00732)0.639.7:0.00258)0.991.107:0.01405)0.999.344:0.01926)0.905.40:0.00829)0.935.45:0.01098)0.999.345:0.01918,((556232:0.01125,112578:0.00029)0.999.346:0.01836,(((((((291322:0.02983,521447:0.00916)0.852.41:0.00366,((362502:0.009,(520926:0.01463,(279056:0.01271,521135:0.01376)0.998.216:0.01825)0.708.9:0.00966)0.980.74:0.01183,((244229:0.01807,514212:0.008)0.428.6:0.0028,((245789:0.00163,531545:0.01773)1.000.1163:0.00103,512634:0.01285)0.983.86:0.0154)0.193.3:0.00841)0.975.78:0.00935)0.856.29:0.00465,(179042:0.01651,(290356:0.01833,296455:0.00724)0.993.105:0.0155)0.953.54:0.01075)0.857.26:0.0043,(107484:0.0142,(188751:0.01487,148678:0.01923)1.000.1164:0.01432)0.868.41:0.00474)0.851.28:0.00969,(215484:0.02416,580595:0.00876)0.770.16:0.01725)0.932.51:0.0086,(154979:0.0236,(200923:0.02177,(541179:0.01703,(584759:0.00798,(130888:0.01586,((364018:0.00204,272207:0.00042)0.767.33:0.01093,(561099:0.0092,579933:0.0099)0.757.25:0.00086)0.831.13:0.00259)0.906.41:0.01043)0.934.46:0.00622)1.000.1165:0.02788)0.927.52:0.00994)0.999.347:0.02323)0.982.92:0.01211,(((111856:0.01319,(143254:0.01087,143021:0.01006)0.974.77:0.00685)0.946.52:0.00519,((509372:0.00245,(((272591:0.00116,((320892:0.00712,269476:0.00813)0.966.60:0.00629,21369:0.02194)0.834.28:0.0076)0.964.81:0.00659,(114404:0.01518,(141407:0.00921,60733:0.01499)0.299.4:0.00497)0.753.27:0.00132)0.766.20:0.00417,(161600:0.01885,542078:0.00835)0.932.52:0.00535)0.053.2:0.00715)0.935.46:0.00329,100505:0.00891)0.993.106:0.00945)0.138.5:0.00321,(((151202:0.02358,(((((467930:0.0208,243237:0.00078)0.971.71:0.0054,((442300:0.01071,355611:0.01048)0.958.66:0.00905,(411531:0.00829,428078:0.00734)0.797.17:0.0113)0.892.32:0.00793)0.988.111:0.0109,(39294:0.01207,((580196:0.01725,(154970:0.00942,161828:0.00844)0.172.3:0.00315)0.985.75:0.01074,(223631:0.00493,(58543:0.02053,551252:0.01012)0.770.17:0.0057)0.175.4:0.00709)0.947.56:0.00763)0.603.6:0.00298)0.986.86:0.00916,(248981:0.00586,(499875:0.0106,(248591:0.00891,240042:0.00431)0.998.217:0.01468)0.844.20:0.0057)0.991.108:0.01063)0.742.22:0.00101,509696:0.01171)0.933.48:0.00473)0.942.55:0.00689,((151904:0.01484,(59360:0.01182,574039:0.01686)0.592.10:0.00286)0.534.5:0.00257,((2325:0.01302,(20785:0.00925,544271:0.00715)0.883.37:0.00417)0.995.139:0.01096,((263519:0.01146,362231:0.00422)0.995.140:0.01102,(142692:0.01451,141821:0.00189)0.986.87:0.01276)0.996.149:0.01142)0.993.107:0.01338)0.948.61:0.0063)0.720.12:0.00532,(186502:0.00833,((275404:0.00495,426016:0.00123)0.983.87:0.02083,(((253914:0.00939,(249417:0.01551,156296:0.01137)0.726.13:0.00284)0.929.55:0.00354,(((99774:0.00785,257152:0.0113)0.818.11:0.00327,334370:0.02537)0.778.22:0.00078,((((((142260:0.01223,142257:0.02871)0.406.4:0.00299,(((2322:0.01082,(((143427:0.00448,341862:0.00182)0.831.14:0.00078,(555401:0.00707,363931:0.027)0.993.108:0.00016)0.860.36:0.0025,2321:0.00725)0.830.21:0.0029)0.770.18:0.00661,52052:0.01959)0.898.43:0.00231,((325093:0.01607,324969:0.00639)0.969.69:0.00566,177323:0.0089)0.145.2:0.00414)0.973.67:0.00399)0.874.34:0.00014,(((172143:0.01158,(544153:0.01072,567785:0.01799)0.735.15:0.01208)0.346.6:0.00934,218593:0.00267)0.405.5:0.00988,((140859:0.01315,(36481:0.02253,217831:0.01247)0.825.17:0.00469)0.369.4:0.00013,(578270:0.00703,200551:0.00488)0.931.44:0.00156)0.485.3:0.00015)0.988.112:0.00312)0.793.15:0.00083,((70404:0.00529,548887:0.01988)0.471.6:0.00015,(161640:0.00585,372941:0.03138)0.720.13:0.00089)0.224.3:0.00078)0.965.62:0.00388,((255909:0.01032,270711:0.00468)1.000.1166:0.00014,548921:0.01207)0.302.6:0.00717)0.874.35:0.00016,((((329653:0.00016,594081:0.03259)0.229.4:0.01169,(577092:0.00582,471197:0.0096)0.871.46:0.00211)0.784.18:0.00314,(211062:0.0101,574686:0.00263)0.752.21:0.00423)0.958.67:0.00448,((((((248249:0.00513,((421495:0.00225,((141129:0.01121,71577:0.01309)0.899.42:0.00333,341814:0.01552)0.947.57:0.00466)0.996.150:0.00682,(144140:0.00959,(389860:0.00649,557941:0.0103)0.023.4:0.00074)0.866.27:0.00199)0.884.42:0.00327)0.758.21:0.00164,(77276:0.01203,((515258:0.00565,277429:0.0086)0.956.57:0.00482,140763:0.02712)0.894.36:0.00275)0.964.82:0.00522)0.985.76:0.00672,(139119:0.00634,(250428:0.00548,468196:0.01201)0.617.5:0.00014)0.989.95:0.00679)0.985.77:0.00697,((((329380:0.00474,2295:0.00858)0.911.41:0.00783,(212836:0.01192,109013:0.01825)0.870.39:0.00709)0.912.32:0.00421,(579829:0.00477,(((141253:0.00992,(540329:0.00925,339084:0.01935)0.641.7:0.00202)0.954.54:0.00572,((266040:0.01001,489865:0.01168)0.987.123:0.00944,((586494:0.00792,(101031:0.00348,413256:0.01204)0.512.8:0.00539)0.999.348:0.01455,(469382:0.00883,((348328:0.00257,(139704:0.01093,(445836:0.02658,325594:0.00468)0.865.35:0.00016)0.959.55:0.00177)0.891.41:0.0065,(274547:0.02997,(491172:0.01311,238087:0.03968)0.109.4:0.00476)0.917.47:0.01136)0.949.55:0.01136)0.732.12:0.00367)0.919.43:0.00565)0.780.13:0.0015)0.978.85:0.00548,((((552287:0.00395,263605:0.00172)0.942.56:0.00389,(397003:0.01348,(140984:0.01763,549166:0.00015)0.998.218:0.01476)0.729.12:0.00083)0.890.39:0.00181,(248774:0.00398,382914:0.00792)0.926.53:0.00294)0.903.46:0.00247,169102:0.00633)0.915.41:0.0061)0.961.60:0.00395)0.977.75:0.00555)0.852.42:0.00153,152630:0.00899)0.972.71:0.00637)0.914.35:0.00356,2299:0.03864)0.051.6:0.00242,100337:0.01159)0.755.22:0.00117)0.867.31:0.00159)0.796.18:0.00078)0.750.16:0.00074)0.910.43:0.0055,((((335211:0.00716,(114965:0.01131,241587:0.00425)0.884.43:0.00177)0.757.26:0.00077,(227962:0.00495,(71114:0.02062,(249853:0.00569,(244921:0.01204,2309:0.00235)0.919.44:0.00309)0.712.6:0.00288)0.943.67:0.00825)0.984.76:0.00559)1.000.1167:0.01162,(((538912:0.01768,(571350:0.00475,(273658:0.00221,(148060:0.02111,(247605:0.00249,(103036:0.01576,197932:0.01392)0.966.61:0.00665)0.972.72:0.00693)0.941.62:0.00492)0.978.86:0.00566)0.960.73:0.00394)0.740.23:0.00462,(99937:0.00471,559612:0.00871)0.982.93:0.00566)0.961.61:0.00564,509233:0.01336)0.636.3:0.00175)0.764.20:0.00152,138799:0.043)0.496.5:0.00474)0.796.19:0.0046)0.995.141:0.00779)0.443.4:0.01514)0.932.53:0.00633)0.874.36:0.00654)0.317.3:0.00582)0.999.349:0.02374)0.054.2:0.01,((103127:0.01203,348550:0.03214)0.999.350:0.01898,(237293:0.03614,313507:0.03571)0.296.4:0.00638)0.736.16:0.00726)0.484.5:0.00016)0.976.74:0.01289)0.425.2:0.00621,(((((((568291:0.00528,326705:0.00504)0.799.15:0.00645,((576289:0.00931,338270:0.01766)0.732.13:0.00237,(156672:0.00807,257628:0.00553)0.864.39:0.00153)0.979.79:0.00688)0.998.219:0.0185,((129412:0.00754,132493:0.01224)1.000.1168:0.03504,(139216:0.00974,159390:0.00465)0.998.220:0.02073)1.000.1169:0.02912)0.946.53:0.00583,(((346435:0.01662,(((550030:0.01542,(278201:0.03312,366927:0.00249)0.983.88:0.00583)0.725.14:0.00264,((279468:0.00634,(160724:0.00791,(244377:0.00587,(97808:0.00736,318524:0.0256)0.970.63:0.00861)0.979.80:0.00874)0.770.19:0.00151)0.615.4:0.00249,(341915:0.012,253031:0.00412)0.966.62:0.0039)0.813.17:0.00178)0.859.35:0.00643,(((241828:0.00076,500244:0.01224)0.903.47:0.00313,(((82527:0.00158,77007:0.00741)0.964.83:0.00808,(252315:0.00132,544559:0.0012)0.949.56:0.00712)0.942.57:0.00708,(242002:0.00099,(154376:0.00158,102953:0.0345)0.866.28:0.0017)0.914.36:0.00354)1.000.1170:0.01758)0.996.151:0.01018,61974:0.00361)0.995.142:0.01302)0.907.39:0.00793)0.944.42:0.00531,((261222:0.03847,(201647:0.00351,61980:0.00155)0.994.137:0.01242)1.000.1171:0.01506,(107403:0.01429,(268486:0.02569,(139680:0.01191,(114546:0.00621,218335:0.00119)0.367.3:0.00322)0.944.43:0.01053)0.941.63:0.00988)0.957.80:0.00915)0.841.23:0.00257)0.974.78:0.00676,((((343374:0.0135,(200830:0.03665,154938:0.00209)0.987.124:0.01142)0.301.7:0.00611,(353715:0.0335,333131:0.02632)0.940.57:0.01046)0.879.41:0.00615,335445:0.02731)0.579.5:0.00525,(188893:0.00598,154890:0.00807)0.756.24:0.03142)0.968.62:0.00751)0.632.3:0.00176)0.991.109:0.01071,(130810:0.03808,((126934:0.00122,160074:0.00029)0.792.16:0.01429,(310620:0.00954,(42542:0.00665,(193530:0.00596,(165148:0.00101,340255:0.00453)0.999.351:0.01576)0.928.46:0.00636)0.929.56:0.00552)0.881.37:0.00392)0.931.45:0.00536)1.000.1172:0.02481)0.843.22:0.00325,((586530:0.01995,(176649:0.01534,(182252:0.03164,((145264:0.00729,(289585:0.01091,188913:0.00206)0.848.28:0.00259)0.965.63:0.00612,49407:0.01673)0.972.73:0.00576)0.810.18:0.00193)0.969.70:0.00799)0.997.176:0.01572,((((541534:0.01768,312481:0.00875)0.675.3:0.00465,(((67316:0.02579,58618:0.00617)0.789.19:0.00473,(((67320:0.00979,172234:0.00156)0.906.42:0.00356,543250:0.006)0.964.84:0.00014,(540893:0.00717,(((259268:0.00485,(207314:0.01982,149899:0.00411)0.882.35:0.00016)0.864.40:0.00157,146214:0.0066)0.741.18:0.00011,67319:0.02669)1.000.1173:0.00317)0.831.15:0.00159)0.977.76:0.00783)0.304.5:0.00369,122067:0.01425)0.969.71:0.00526)0.882.36:0.01553,356898:0.01958)0.553.5:0.00504,287759:0.03208)0.990.93:0.01169)0.949.57:0.0079)0.813.18:0.00564,(((147529:0.01154,213627:0.01264)0.882.37:0.00407,(((555637:0.04086,267240:0.02332)0.752.22:0.00182,((135966:0.01261,279641:0.01073)0.870.40:0.00333,165728:0.03688)0.988.113:0.00908)0.977.77:0.00539,((((542229:0.00711,162582:0.02268)0.892.33:0.00699,570839:0.01848)0.988.114:0.012,(221580:0.00931,(536363:0.0196,140527:0.02303)0.647.5:0.0059)0.753.28:0.00359)0.953.55:0.00634,(148831:0.00458,((((((143925:0.01894,((109136:0.01623,318306:0.00078)0.888.29:0.00701,119865:0.00108)0.361.4:0.00232)0.936.44:0.00489,587431:0.00079)0.844.21:0.01059,511603:0.00377)0.977.78:0.00872,((89871:0.00732,(243681:0.01251,299765:0.01807)0.979.81:0.00983)0.953.56:0.00721,351824:0.00737)0.974.79:0.01148)0.611.5:0.00662,(301867:0.01145,(512157:0.02133,257756:0.01635)0.987.125:0.01154)0.993.109:0.0126)0.998.221:0.01544,146853:0.00812)0.581.3:0.00541)0.905.41:0.0044)0.959.56:0.00623)0.891.42:0.00285)0.984.77:0.0117,((((((((143728:0.00784,547999:0.00244)0.731.12:0.0024,535560:0.02264)0.730.11:0.00123,(246099:0.01178,545908:0.02185)0.947.58:0.00677)0.983.89:0.01252,(279541:0.01682,(430903:0.00014,544761:0.00015)0.371.4:0.0207)0.408.12:0.00441)0.986.88:0.0121,(166621:0.03967,190536:0.04738)0.999.352:0.0178)0.819.20:0.0031,((((((145434:0.04373,(((150229:0.02538,511434:0.00267)0.938.55:0.0353,((310761:0.01864,(190052:0.02457,(86285:0.00304,107130:0.03478)0.734.14:0.0014)0.996.152:0.01833)0.954.55:0.00754,(160771:0.02177,356775:0.0364)0.891.43:0.00734)0.877.33:0.00655)0.213.4:0.00698,((150224:0.01122,142668:0.0119)0.999.353:0.01141,272395:0.00486)0.956.58:0.00734)0.954.56:0.00993)0.951.55:0.00713,(278231:0.01796,572926:0.02151)0.996.153:0.01591)0.767.34:0.00522,(((39004:0.01108,20776:0.03166)0.914.37:0.00677,(321124:0.01199,136218:0.0022)0.779.10:0.00937)0.926.54:0.00677,341525:0.01087)0.938.56:0.00687)0.885.41:0.00527,((((549936:0.00847,(153568:0.0081,187417:0.01125)0.947.59:0.00526)0.878.45:0.00391,(((91580:0.00978,(586054:0.01091,159349:0.00278)0.999.354:0.01864)0.938.57:0.00682,519516:0.01096)0.547.5:0.00384,544563:0.00612)0.997.177:0.01121)0.998.222:0.02328,((((270202:0.0122,(179177:0.00014,570779:0.00476)0.928.47:0.01048)0.945.51:0.0077,(259037:0.00771,512567:0.0258)0.959.57:0.00658)0.800.20:0.00553,((206791:0.0122,(199798:0.00939,191855:0.01358)0.918.54:0.00613)0.977.79:0.00932,((144373:0.0183,136336:0.02211)0.994.138:0.01421,249208:0.00916)0.868.42:0.00394)0.987.126:0.01209)0.971.72:0.00976,(284011:0.02379,586182:0.0167)0.970.64:0.00778)0.065.3:0.00665)0.986.89:0.0142,(((552860:0.04666,((569055:0.01993,104306:0.01272)1.000.1174:0.02915,((270683:0.03075,((((263680:0.01022,586123:0.02068)1.000.1175:0.01598,344393:0.004)0.998.223:0.01299,(141336:0.01094,328692:0.0035)0.151.4:0.00373)0.854.25:0.00347,237103:0.00457)0.992.130:0.01034)0.897.37:0.00398,(((198682:0.02736,(140396:0.00431,(136017:0.02073,(139623:0.0116,183437:0.01338)0.475.3:0.00412)0.995.143:0.01052)0.990.94:0.01216)0.871.47:0.00363,(66035:0.03761,(346167:0.00298,100411:0.01033)0.166.5:0.00271)0.930.47:0.00469)0.007.3:0.00259,(306527:0.01011,((363466:0.00845,(329636:0.00281,(363062:0.00972,140442:0.01476)0.945.52:0.00474)0.990.95:0.00632)0.905.42:0.00242,((217732:0.00324,(139292:0.00597,((561872:0.00622,582005:0.01254)0.874.37:0.00185,(312668:0.00637,100173:0.00093)0.799.16:0.00078)0.917.48:0.00238)0.977.80:0.00945)0.905.43:0.00707,(105219:0.00416,100568:0.00796)1.000.1176:0.01693)0.963.69:0.00395)0.998.224:0.00952)0.917.49:0.00441)0.941.64:0.00496)0.984.78:0.01033)0.979.82:0.00961)0.969.72:0.01286,(156104:0.02149,540288:0.03338)0.927.53:0.00782)0.859.36:0.00435,(((((((205298:0.00144,(193970:0.00733,186193:0.0177)0.668.4:0.01058)0.999.355:0.01088,(((238195:0.00777,(547764:0.00217,238825:0.02988)0.958.68:0.00613)0.965.64:0.00604,(136335:0.00763,(567350:0.00472,86570:0.00596)0.761.23:0.00241)0.839.29:0.00999)0.859.37:0.00292,((144735:0.00261,61123:0.02269)0.847.26:0.00304,293136:0.02394)0.920.36:0.0029)0.796.20:0.00169)0.944.44:0.00807,330503:0.0168)0.408.13:0.00563,69822:0.02145)0.957.81:0.01033,(187349:0.03143,288165:0.02036)0.196.5:0.00811)0.352.3:0.00398,((564481:0.01843,(139202:0.03612,(277651:0.01072,105314:0.03853)0.980.75:0.01383)0.315.3:0.006)0.966.63:0.01042,((83581:0.00654,538619:0.00756)0.873.30:0.01978,249383:0.02881)0.983.90:0.01109)0.778.23:0.00587)0.863.31:0.00395,(161450:0.02823,305966:0.02386)0.838.25:0.00761)0.158:0.00226)0.552.6:0.00399)0.841.24:0.00809)0.915.42:0.00887,(264786:0.02624,(306834:0.02928,348807:0.00508)0.866.29:0.00725)1.000.1177:0.04604)0.168.8:0.00501,2403:0.05534)0.978.87:0.01169)0.981.78:0.0093,((587677:0.01128,((61968:0.01427,142804:0.00311)0.623.3:0.00015,283302:0.00898)0.906.43:0.00518)0.981.79:0.01432,(167133:0.0377,150087:0.0492)0.923.42:0.01456)0.967.65:0.0119)0.998.225:0.01738,((278244:0.02013,((((263786:0.01838,346644:0.01405)0.373.4:0.00136,278872:0.01626)0.787.15:0.00673,(566298:0.01804,(556650:0.00519,(((((238821:0.01645,((279132:0.03014,104868:0.04132)0.085.4:0.01262,(126861:0.00246,179212:0.02265)0.907.40:0.00667)0.841.25:0.00492)0.773.19:0.00162,(254152:0.02629,179670:0.01838)0.461.7:0.00282)0.444.8:0.00018,((((322445:0.02121,(((328057:0.01314,(110266:0.01049,(249295:0.0184,(185297:0.01617,(341938:0.00675,(108543:0.00749,317837:0.00522)0.879.42:0.00353)0.892.34:0.00296)0.738.19:0.0051)0.819.21:0.0022)0.999.356:0.01023)0.878.46:0.00294,(((308741:0.00161,171297:0.02578)0.780.14:0.00078,((152380:0.00796,((360101:0.00569,145993:0.00665)0.904.48:0.00554,549640:0.00859)0.906.44:0.00014)0.906.45:0.00613,(99754:0.01684,((72771:0.01538,394758:0.01016)0.921.50:0.01384,147248:0.0123)0.206.4:0.00014)0.113.5:0.00484)0.835.19:0.00397)0.613.4:0.00014,(61113:0.03978,566549:0.00913)0.935.47:0.00866)0.968.63:0.00518)0.746.23:0.00172,(279125:0.01129,543469:0.00458)0.746.24:0.00701)0.978.88:0.00763)0.859.38:0.00241,(244757:0.01076,177653:0.01674)0.991.110:0.01054)0.964.85:0.00578,((((((25577:0.00381,(108084:0.0049,2444:0.0132)0.341.4:0.00553)0.994.139:0.01072,(169149:0.00974,(2313:0.04038,342471:0.00885)0.859.39:0.00245)0.961.62:0.00785)0.926.55:0.00734,558722:0.01069)0.985.78:0.00989,534732:0.02397)0.910.44:0.0044,(93188:0.01593,(226929:0.02853,((183892:0.00903,175223:0.0159)0.890.40:0.00257,((212182:0.00246,278042:0.01609)0.997.178:0.00996,175076:0.01797)0.017.3:0.00155)0.942.58:0.00536)0.974.80:0.0098)0.896.38:0.00758)0.970.65:0.00796,362896:0.00844)0.802.16:0.00188)0.954.57:0.00456,(578211:0.01305,(542026:0.01493,(100807:0.00078,581424:0.03143)0.808.16:0.0137)1.000.1178:0.02078)0.963.70:0.00789)0.929.57:0.00461)0.972.74:0.00563,(((((2450:0.01821,(81033:0.00082,2446:0.02353)0.968.64:0.00768)0.982.94:0.00651,277351:0.00592)0.920.37:0.00321,145548:0.00231)0.982.95:0.00566,((((153340:0.02272,262661:0.004)0.894.37:0.00229,((((140098:0.01957,((552203:0.00729,146986:0.00904)0.994.140:0.00987,265965:0.01497)0.803.17:0.00399)0.994.141:0.01207,((93722:0.01088,590020:0.00966)0.999.357:0.01582,((359870:0.00134,((100882:0.00802,(358171:0.02518,554611:0.0178)0.950.77:0.00579)0.984.79:0.00836,102971:0.00685)0.882.38:0.0042)0.794.21:0.01364,(93747:0.01251,(112379:0.01114,((((208205:0.00099,(245619:0.00016,265850:0.0062)0.962.61:0.02155)0.234:0.00267,140533:0.00538)0.939.55:0.00472,(560027:0.00474,357986:0.01821)0.950.78:0.00427)0.979.83:0.00591,((192374:0.00594,(93990:0.0247,((100578:0.01242,((154706:0.01578,260191:0.01674)0.999.358:0.01466,(278937:0.00748,149971:0.01161)0.937.50:0.00531)0.740.24:0.006)0.901.35:0.00327,(585873:0.02099,510003:0.01724)0.944.45:0.00807)0.535.6:0.00467)1.000.1179:0.01164)0.957.82:0.00518,((593254:0.01025,560943:0.01233)0.958.69:0.00349,(73211:0.00954,139263:0.00081)0.392.5:0.00403)0.429.2:0.00083)0.866.30:0.00157)0.987.127:0.00794)0.955.59:0.00556)0.168.9:0.00016)0.892.35:0.00403)0.620.2:0.00248)0.753.29:0.00068,(243677:0.01359,573111:0.0085)0.909.41:0.00604)0.939.56:0.00443,236001:0.00558)0.950.79:0.00314)0.965.65:0.00392,(245177:0.02643,((226794:0.00203,194872:0.00476)0.955.60:0.00469,((141174:0.00245,578421:0.01686)0.984.80:0.00566,142220:0.00151)0.812.25:0.00611)0.024.3:0.00091)0.956.59:0.00345)0.776.21:0.00157,((231727:0.0074,585081:0.01361)0.854.26:0.0045,332554:0.00751)1.000.1180:0.01719)0.614.4:0.00015)0.971.73:0.00556,((138738:0.01353,(353843:0.01529,(317388:0.00491,2452:0.01238)0.987.128:0.00646)0.915.43:0.00397)0.345.6:0.00243,(((539891:0.0088,((2463:0.00087,246587:0.02151)0.900.38:0.00986,(353655:0.00561,((267629:0.00748,550326:0.00328)0.997.179:0.01328,557116:0.00559)0.278.4:0.00298)0.995.144:0.01143)0.970.66:0.0059)0.919.45:0.00253,((252158:0.00586,244501:0.02649)0.960.74:0.00903,295470:0.01021)0.897.38:0.00331)0.079.2:0.0008,(258097:0.00548,572964:0.00612)0.932.54:0.00467)0.986.90:0.00782)0.944.46:0.00372)0.959.58:0.00399)0.987.129:0.00695,(336602:0.00812,345475:0.0134)0.929.58:0.00356)0.884.44:0.0024)0.893.34:0.00262)0.949.58:0.00534)0.953.57:0.00621,103103:0.02661)0.989.96:0.00963)0.928.48:0.00551,145257:0.02905)0.934.47:0.00574)0.755.23:0.00361)0.992.131:0.01343)0.919.46:0.00673)0.855.27:0.00519,(337128:0.03992,(521453:0.01315,569609:0.02869)0.642.11:0.0077)0.849.20:0.01318)0.937.51:0.01012)0.999.359:0.01706,(249329:0.01883,(311773:0.01461,(544791:0.0152,302405:0.00683)0.954.58:0.00862)0.998.226:0.01598)0.998.227:0.01621)0.622.4:0.00242,(((((552208:0.01068,((152452:0.01252,(238400:0.02795,539572:0.00779)0.996.154:0.01593)0.515.8:0.00606,224838:0.00878)0.999.360:0.0174)0.997.180:0.01445,(((548650:0.01446,179639:0.01287)0.735.16:0.00059,((150157:0.01121,148990:0.00563)0.765.20:0.00167,308348:0.01973)0.761.24:0.00338)0.974.81:0.00769,((252313:0.00714,(((114438:0.00824,((((358831:0.01057,205919:0.0045)0.267.6:0.00093,((148858:0.02651,((278366:0.02085,(((154768:0.00836,(70094:0.01233,179748:0.0069)0.977.81:0.01456)0.997.181:0.00628,250970:0.00517)0.877.34:0.00537,(2355:0.0182,81979:0.01102)0.742.23:0.00058)0.988.115:0.00687)0.847.27:0.00206,184550:0.00457)0.897.39:0.00402)0.940.58:0.00429,366819:0.02175)0.948.62:0.00381)0.768.22:0.00086,(((109042:0.01357,61127:0.03799)0.962.62:0.01012,249428:0.01291)0.968.65:0.00847,((58630:0.00014,2358:0.00517)0.881.38:0.0026,(169324:0.00576,(((73960:0.04177,(((552292:0.00014,216158:0.004)0.991.111:0.00591,535272:0.00989)0.544.6:0.0066,((((((173295:0.00392,(562638:0.01065,150237:0.01826)0.466.8:0.00379)0.952.61:0.00511,542003:0.01042)0.985.79:0.00673,(189519:0.00106,106077:0.00043)0.866.31:0.00156)0.830.22:0.00171,(207908:0.00472,39685:0.01132)0.959.59:0.00014)0.750.17:0.00015,58642:0.00313)0.882.39:0.00015,230495:0.00153)0.279.6:0.00078)0.836.23:0.00183)0.916.47:0.00353,(233273:0.00324,(((152840:0.00881,580835:0.00781)0.913.49:0.00494,61125:0.01162)0.992.132:0.00468,58631:0.00032)0.768.23:0.00476)0.887.33:0.00264)0.986.91:0.00739,114689:0.02764)0.955.61:0.00624)0.998.228:0.01225)0.992.133:0.00943)0.776.22:0.00149)0.990.96:0.00647,(360184:0.00626,113326:0.01849)0.994.142:0.00974)0.417.6:0.00193)0.297.2:0.00173,(574735:0.00474,140556:0.01861)0.407.5:0.01167)0.923.43:0.00355,35881:0.0162)0.906.46:0.00351)1.000.1181:0.01523,568108:0.00633)0.981.80:0.00911)0.696.9:0.004)0.867.32:0.00198,((207800:0.00555,(247974:0.00317,(553046:0.01412,277737:0.02278)0.716.5:0.00043)0.854.27:0.01302)0.906.47:0.00545,237065:0.01924)0.147.2:0.00328)0.859.40:0.00317,((176313:0.02417,(154027:0.01314,292328:0.01803)0.906.48:0.00565)0.547.6:0.00296,(18143:0.02183,(362103:0.02216,361407:0.01196)1.000.1182:0.02502)0.995.145:0.01916)0.985.80:0.01261)0.832.20:0.00392,336572:0.08758)0.979.84:0.01073)0.867.33:0.00654)1.000.1183:0.02875,(((279316:0.00201,255704:0.02688)0.488.8:0.08514,(104416:0.01026,(279631:0.01234,518109:0.02941)0.937.52:0.01792)1.000.1184:0.03888)0.620.3:0.01383,(61850:0.01528,(100771:0.0355,100925:0.01861)0.306.5:0.00767)1.000.1185:0.04913)0.682.4:0.01251)0.981.81:0.01545,((239868:0.02527,339642:0.0152)1.000.1186:0.04559,(((((((149305:0.01968,(145647:0.00988,192694:0.02968)0.890.41:0.0084)0.979.85:0.01115,147249:0.02207)0.266.4:0.00693,((147083:0.01232,(145909:0.02838,(175321:0.02816,147524:0.01404)0.681.11:0.00371)0.289.3:0.00112)0.885.42:0.00339,((149599:0.02144,(198687:0.01809,197654:0.03527)0.391.6:0.00292)1.000.1187:0.02866,(174760:0.00992,(176430:0.03148,147836:0.01703)0.580.4:0.0034)0.982.96:0.01103)0.859.41:0.00401)0.931.46:0.00776)1.000.1188:0.07841,((150656:0.01849,(552008:0.03444,141251:0.0125)0.919.47:0.00682)1.000.1189:0.06732,(529455:0.04453,(165396:0.01709,(142313:0.03008,(199809:0.02316,257992:0.02386)0.961.63:0.01248)0.839.30:0.01591)0.972.75:0.01661)0.984.81:0.02472)1.000.1190:0.03394)0.940.59:0.01395,2240:0.04491)0.562.6:0.00637,(2237:0.01337,(((514494:0.01127,(140397:0.0191,2243:0.01206)0.880.30:0.00223)0.858.22:0.00399,(93927:0.02641,(101540:0.01253,525720:0.00795)0.988.116:0.00844)0.746.25:0.00085)0.928.49:0.00557,(2235:0.01575,36262:0.01718)0.998.229:0.01402)0.825.18:0.00559)1.000.1191:0.03766)1.000.1192:0.0575,(((((556300:0.0306,577339:0.00244)0.492.5:0.00062,264348:0.05539)0.812.26:0.01197,(((213800:0.01873,((384131:0.00787,((248648:0.01704,527641:0.01916)0.962.63:0.009,(((391969:0.0064,579506:0.00763)0.317.4:0.00014,512790:0.01425)0.457.4:0.0199,462011:0.01612)0.841.26:0.00588)0.725.15:0.00135)0.787.16:0.00487,(((110654:0.01142,((215211:0.00365,337613:0.00215)0.958.70:0.00695,(149122:0.01753,353686:0.00588)0.945.53:0.00532)0.985.81:0.01121)0.987.130:0.01161,95871:0.01203)0.895.41:0.01033,(525635:0.00865,(561710:0.00507,345958:0.01154)0.800.21:0.00352)0.996.155:0.0148)0.676.9:0.00262)0.863.32:0.00522)0.954.59:0.00603,(((((123752:0.01237,183535:0.0089)0.879.43:0.00173,(((335772:0.00833,277065:0.00751)0.926.56:0.00694,335481:0.00959)0.943.68:0.01082,250944:0.00548)0.979.86:0.0119)0.948.63:0.00429,(218840:0.0176,253429:0.0192)0.985.82:0.00959)0.575.7:0.00196,(255020:0.003,((((240678:0.01289,(((573326:0.00566,328172:0.01579)0.868.43:0.01108,(514346:0.00719,(345921:0.02261,106749:0.02195)0.802.17:0.00174)0.870.41:0.00207)0.990.97:0.00868,(((115159:0.00962,578292:0.02536)0.914.38:0.0128,191237:0.02812)0.785.19:0.00698,469921:0.00093)0.936.45:0.00547)0.954.60:0.00393)0.968.66:0.00475,(((149555:0.00746,(102388:0.01083,(229949:0.00945,243201:0.00724)0.965.66:0.00549)0.979.87:0.00347)0.798.13:0.00124,(((220303:0.00362,413557:0.0123)0.912.33:0.00342,((146939:0.01308,262464:0.01088)0.968.67:0.01302,(155194:0.01461,(536629:0.00725,219189:0.00335)0.983.91:0.00649)0.952.62:0.00915)0.761.25:0.00422)0.749.15:0.00191,(105406:0.01286,136774:0.00982)0.975.79:0.00527)0.865.36:0.00264)0.922.47:0.00235,((185444:0.00807,(((248871:0.00094,((145094:0.00422,100187:0.00994)0.784.19:0.00512,56180:0.00677)0.931.47:0.00699)0.986.92:0.00627,464073:0.00939)0.905.44:0.00015,153040:0.00636)0.970.67:0.0047)0.704.14:0.00075,(147462:0.00014,414506:0.00686)0.927.54:0.00234)0.876.37:0.00157)0.861.36:0.00158)0.985.83:0.00548,(((150743:0.02141,201471:0.01204)0.843.23:0.00661,214748:0.00028)0.909.42:0.00015,208388:0.0011)0.941.65:0.00014)0.876.38:0.00156,147046:0.01204)0.768.24:0.00173)0.914.39:0.00245)0.929.59:0.00466,(345646:0.00619,(46641:0.01105,(72435:0.00766,(31516:0.01011,414371:0.00743)1.000.1193:0.06833)0.802.18:0.00813)0.746.26:0.01026)0.990.98:0.01354)0.998.230:0.01178)0.950.80:0.0079,((286716:0.00791,(137197:0.02008,142000:0.00745)0.990.99:0.01601)0.873.31:0.01456,(204955:0.03176,581231:0.05689)0.106.3:0.01223)0.862.40:0.00591)0.997.182:0.02177)0.999.361:0.03627,(((((110915:0.00925,28957:0.01365)0.995.146:0.01762,151755:0.04624)0.924.40:0.00845,(526477:0.0233,520418:0.01142)0.980.76:0.01099)0.933.49:0.01203,510360:0.01372)1.000.1194:0.02763,((137948:0.0261,(2256:0.0262,544536:0.02744)0.539.6:0.0045)1.000.1195:0.02756,((288740:0.02154,337303:0.01149)0.922.48:0.00647,(568419:0.01489,(385877:0.01806,154777:0.01826)0.971.74:0.00792)0.945.54:0.00848)1.000.1196:0.01922)0.895.42:0.00627)0.984.82:0.01562)0.567.6:0.01281,2251:0.04008)0.597.8:0.01404)0.882.40:0.01233)0.729.13:0.00636)0.973.68:0.01494)0.959.60:0.01383,((364221:0.09111,(136955:0.01013,196863:0.02333)1.000.1197:0.05915)0.314.6:0.01234,(((162765:0.03218,561026:0.03321)0.899.43:0.0108,((242251:0.03106,(572612:0.00807,(102122:0.01309,340884:0.01897)1.000.1198:0.03818)0.991.112:0.01295)0.908.45:0.0068,(58645:0.0241,168147:0.01767)0.365.4:0.00733)0.961.64:0.01061)1.000.1199:0.03669,((((62693:0.00331,(80707:0.01145,(131164:0.01492,(167189:0.00788,69811:0.02291)0.858.23:0.02155)0.867.34:0.01391)0.258.3:0.00447)0.995.147:0.01401,564878:0.0195)0.854.28:0.0066,(163961:0.00922,(278006:0.01914,((((293440:0.01929,(326618:0.02561,(277616:0.00452,(327084:0.01458,(308722:0.01691,195676:0.00988)0.057.6:0.00243)0.933.50:0.00741)0.965.67:0.00739)0.979.88:0.01207)0.720.14:0.00931,260320:0.00044)0.997.183:0.0152,((556423:0.01237,((((450976:0.00273,(144270:0.01206,(439462:0.00958,(160084:0.00235,555540:0.00552)0.762.23:0.00176)0.757.27:0.00329)0.920.38:0.00254)0.962.64:0.00477,((376387:0.00171,(575104:0.00985,228555:0.01555)0.972.76:0.00705)0.837.17:0.00394,443793:0.00805)0.285.4:0.00322)0.855.28:0.00299,(20456:0.01884,533497:0.01843)1.000.1200:0.01502)0.976.75:0.00687,325694:0.00591)0.997.184:0.01229)0.807.12:0.00425,(237434:0.01263,351352:0.03217)0.979.89:0.0172)0.909.43:0.00507)0.915.44:0.00921,((140107:0.01175,(136269:0.02344,547896:0.00536)0.975.80:0.00949)0.999.362:0.01879,((144059:0.00018,(103229:0.00849,328847:0.02056)0.996.156:0.01434)0.963.71:0.01205,179461:0.01617)0.965.68:0.00976)0.966.64:0.01165)0.964.86:0.0101)0.978.89:0.00999)0.907.41:0.00523)0.926.57:0.00683,(264094:0.02133,106103:0.01986)0.820.9:0.00953)0.953.58:0.01792)1.000.1201:0.03098)0.664.3:0.0078)0.497.7:0.00417)0.957.83:0.0128,(((((((((108748:0.0418,((226667:0.01033,((270553:0.01436,(220351:0.01782,139833:0.00453)0.968.68:0.00774)0.979.90:0.00924,(276338:0.00629,(580179:0.01348,541603:0.00476)0.992.134:0.01437)0.916.48:0.00764)0.827.17:0.00237)0.856.30:0.00337,((327668:0.007,227404:0.01089)1.000.1202:0.02661,138582:0.03947)0.965.69:0.00832)0.956.60:0.00916)0.181.4:0.00491,(342042:0.03452,509721:0.05165)0.983.92:0.01837)0.541.6:0.00825,(540416:0.01745,(161206:0.03158,(51657:0.00774,238065:0.00654)0.954.61:0.00803)0.994.143:0.0187)1.000.1203:0.02546)0.852.43:0.0111,240290:0.04878)0.215.6:0.00875,(((315129:0.00962,336805:0.0111)0.878.47:0.00557,(542702:0.01538,(240737:0.01169,549338:0.01383)0.992.135:0.01311)0.958.71:0.01403)0.993.110:0.01827,(61849:0.0075,(277011:0.00625,(214670:0.01029,99342:0.02585)0.533.2:0.00821)1.000.1204:0.03472)1.000.1205:0.03544)0.963.72:0.01199)0.979.91:0.01383,(1894:0.01624,((545501:0.00734,510458:0.0191)0.999.363:0.01612,174634:0.00513)0.727.12:0.00822)0.976.76:0.00919)0.967.66:0.00889,((222737:0.09935,((((((197441:0.00481,165115:0.00363)0.868.44:0.03882,129618:0.03813)0.958.72:0.0177,(208843:0.02859,(113167:0.02005,((233403:0.02061,(170335:0.00014,258615:0.01143)0.933.51:0.01321)0.988.117:0.01137,347862:0.02258)0.631.6:0.0054)0.977.82:0.01803)0.962.65:0.01821)0.369.5:0.00651,(((120762:0.00318,(107252:0.00252,(123567:0.0126,((574792:0.00465,(187121:0.0147,(70461:0.00425,287576:0.0009)0.944.47:0.00498)0.905.45:0.00411)0.774.11:0.00089,366044:0.00362)0.828.15:0.00199)0.999.364:0.00034)0.773.20:0.00129)0.997.185:0.01323,288734:0.01287)0.561.9:0.00842,367384:0.00845)1.000.1206:0.08189)0.824.14:0.01243,((355683:0.00748,297581:0.00918)1.000.1207:0.05215,(113220:0.00823,59562:0.00245)1.000.1208:0.0772)0.519.4:0.01567)0.948.64:0.01553,(114637:0.00383,78000:0.01399)1.000.1209:0.05813)0.999.365:0.0388)0.968.69:0.02244,((57463:0.02793,550796:0.00898)1.000.1210:0.04723,((563763:0.01358,(247247:0.01308,((511857:0.00853,99250:0.00884)0.885.43:0.00235,354185:0.00992)0.844.22:0.00507)0.978.90:0.01041)0.490.4:0.00561,(((241633:0.01203,237913:0.00383)1.000.1211:0.01776,570910:0.00198)0.901.36:0.01449,(551164:0.00399,(334416:0.01328,(240140:0.0156,336950:0.00773)0.171.2:0.00162)0.814.11:0.00333)0.991.113:0.01)0.678.8:0.00505)1.000.1212:0.03829)0.681.12:0.02051)0.796.21:0.00804)0.977.83:0.01309,((((((139032:0.05937,((1914:0.01137,281862:0.01976)0.943.69:0.00832,(525962:0.01748,(382628:0.00775,(584987:0.02589,(509390:0.01274,571300:0.00336)0.937.53:0.00436)0.726.14:0.00674)0.960.75:0.00675)0.999.366:0.02086)0.273.3:0.01192)0.994.144:0.02237,(205413:0.04368,(516782:0.03543,(554622:0.01886,(301911:0.01439,281446:0.01471)0.773.21:0.00352)0.468.4:0.0057)1.000.1213:0.03027)0.867.35:0.01174)0.778.24:0.00935,((160787:0.01015,325319:0.00971)0.989.97:0.01265,(314028:0.03075,546673:0.00619)1.000.1214:0.04183)0.966.65:0.01379)0.873.32:0.00801,((215410:0.04363,533553:0.0494)0.991.114:0.02526,((566383:0.04271,592485:0.02527)1.000.1215:0.03825,((312412:0.06398,(554211:0.0041,302703:0.01765)0.982.97:0.01751)0.998.231:0.03566,(326247:0.01744,313150:0.02595)1.000.1216:0.06015)0.365.5:0.01612)0.861.37:0.01329)0.317.5:0.01009)0.937.54:0.00864,((((((66950:0.00898,((149805:0.0068,277035:0.02604)0.994.145:0.01172,((566043:0.00186,325619:0.03124)0.887.34:0.00229,308796:0.01722)0.978.91:0.00779)0.711.8:0.00773)0.341.5:0.00502,566344:0.01539)1.000.1217:0.04341,((((545634:0.0073,(244201:0.02316,251280:0.00954)0.878.48:0.00392)0.182.4:0.0041,(101390:0.02636,312659:0.00292)0.303.3:0.00454)0.995.148:0.01452,(((273970:0.01992,(246041:0.01502,561608:0.01526)0.318.4:0.00508)1.000.1218:0.02192,(((258102:0.01003,532482:0.00556)0.922.49:0.00391,519566:0.01213)0.913.50:0.00436,(1910:0.01477,237093:0.01288)0.933.52:0.00525)0.965.70:0.00865)0.959.61:0.00794,(59897:0.02359,(247849:0.018,108426:0.00471)0.145.3:0.00414)0.918.55:0.00649)0.945.55:0.00783)0.481.4:0.00633,((577189:0.01149,(114657:0.01289,534403:0.00016)0.993.111:0.01131)0.982.98:0.01274,((342229:0.03025,334161:0.02548)0.998.232:0.02327,(560122:0.04218,((157758:0.01741,349092:0.01129)0.753.30:0.00844,(57446:0.02578,586489:0.01448)0.480.7:0.00614)0.848.29:0.00627)0.124.3:0.00272)0.975.81:0.012)0.751.9:0.00521)0.605.7:0.01207)0.997.186:0.02235,(((141722:0.02095,(317871:0.02007,((546781:0.00236,(146275:0.0002,559454:0.0188)1.000.1219:0.0274)0.999.367:0.01641,261776:0.01594)0.989.98:0.01336)0.975.82:0.01044)1.000.1220:0.02425,((257614:0.05222,(531670:0.04381,(145206:0.01757,70875:0.02129)1.000.1221:0.05087)0.987.131:0.02082)0.262.4:0.00668,(458762:0.01349,565691:0.01394)0.985.84:0.01302)0.965.71:0.01214)0.949.59:0.00776,((1907:0.01857,(((246533:0.01927,283400:0.00242)0.985.85:0.00697,(49501:0.01424,582798:0.00529)0.954.62:0.00492)0.689.6:0.0007,(329166:0.01251,((282325:0.00953,1918:0.00991)0.807.13:0.00359,(582590:0.00524,(136857:0.01204,(577518:0.0122,75260:0.01381)0.271.3:0.00472)0.828.16:0.00491)0.993.112:0.01088)0.837.18:0.00232)0.869.42:0.00771)0.601.2:0.00451)0.916.49:0.00731,508913:0.05554)0.986.93:0.01344)0.747.19:0.00388)0.942.59:0.00745,((((((424534:0.00645,155965:0.00473)0.882.41:0.01029,(231844:0.00698,284431:0.00722)0.989.99:0.01469)1.000.1222:0.02975,((150297:0.01359,(144918:0.02071,76102:0.01542)1.000.1223:0.03725)0.280.5:0.00366,(553039:0.00014,509023:0.00274)1.000.1224:0.01636)0.923.44:0.01016)0.925.38:0.01139,((144088:0.00781,150133:0.02318)0.997.187:0.01552,(228557:0.00681,(232382:0.00499,518407:0.0114)0.867.36:0.00376)0.920.39:0.0068)0.710.9:0.00613)0.999.368:0.0223,565959:0.04034)0.905.46:0.0078,(279217:0.00614,(279003:0.00971,277641:0.01754)0.968.70:0.00529)0.989.100:0.07089)0.964.87:0.00984)0.996.157:0.01505,((((152831:0.01879,((540949:0.00985,533148:0.009)0.998.233:0.01238,(272734:0.01993,(202437:0.01133,537712:0.02252)0.971.75:0.00838)0.830.23:0.00444)0.980.77:0.01165)0.591.7:0.00731,239867:0.00853)1.000.1225:0.0319,(((306043:0.00801,(281260:0.01518,(((232933:0.00935,354348:0.01598)0.861.38:0.00573,(237249:0.02862,(91285:0.01095,544483:0.01545)0.987.132:0.00172)0.906.49:0.00188)0.761.26:0.00016,(552076:0.0119,551671:0.00842)0.862.41:0.00365)0.861.39:0.00013)0.995.149:0.01252)0.994.146:0.01413,247883:0.02283)0.125.2:0.00438,((551607:0.02021,(551093:0.03569,592509:0.00506)0.871.48:0.00955)0.999.369:0.01955,((141996:0.01204,(213042:0.00316,(((((133272:0.00459,1896:0.00579)0.101.4:0.00175,(578372:0.0167,549388:0.01325)0.942.60:0.00457)0.977.84:0.00543,80197:0.02531)0.952.63:0.00324,283919:0.00683)0.901.37:0.00231,(1911:0.01626,(284946:0.00737,286918:0.00328)0.772.18:0.00082)0.740.25:0.00086)0.948.65:0.00549)0.997.188:0.00182)0.075.4:0.00015,(590874:0.00955,(279548:0.00633,(((((1909:0.01256,511725:0.00563)0.594.7:0.00014,227318:0.01631)0.881.39:0.00721,((547826:0.00784,558606:0.01945)0.943.70:0.00833,(583211:0.02615,569229:0.01947)0.971.76:0.00946)0.407.6:0.00646)0.953.59:0.00663,((586380:0.00812,148262:0.00976)0.890.42:0.00222,((282750:0.01447,548669:0.00478)0.835.20:0.00182,(563011:0.01293,(540878:0.01988,106142:0.01511)0.373.5:0.00102)0.961.65:0.00436)0.751.10:0.00076)0.877.35:0.00243)0.966.66:0.00399,(((184224:0.00554,572774:0.0081)0.731.13:0.00116,(547745:0.01009,27268:0.01158)0.997.189:0.01134)0.708.10:0.00096,(285668:0.01877,(109168:0.00841,(52484:0.00728,16567:0.00842)0.744.15:0.00342)0.998.234:0.01034)0.867.37:0.00285)0.611.6:0.00108)0.943.71:0.00367)0.920.40:0.00349)0.905.47:0.00319)0.732.14:0.0063)0.616.6:0.00395)1.000.1226:0.02589)0.838.26:0.00808,511739:0.05157)0.447.5:0.00411)0.907.42:0.00504)0.837.19:0.00613,((((201544:0.01569,254208:0.0184)0.990.100:0.01726,(250662:0.02266,(250044:0.00725,(543701:0.01486,250289:0.01584)0.033.3:0.00309)0.840.19:0.00842)1.000.1227:0.03295)1.000.1228:0.04602,(282451:0.02887,(260327:0.02627,(295119:0.01167,((337337:0.00632,547346:0.00905)0.187.4:0.00203,(175084:0.0157,114375:0.01615)0.605.8:0.00379)0.958.73:0.00701)0.506.3:0.00431)0.996.158:0.02162)0.946.54:0.01027)0.981.82:0.01318,((((((533883:0.00412,156416:0.01912)0.966.67:0.00893,((201885:0.01311,581539:0.01645)0.981.83:0.00866,(283513:0.04604,(540386:0.00897,511119:0.03215)0.998.235:0.01879)0.977.85:0.01086)0.919.48:0.00805)0.985.86:0.01387,273337:0.02783)0.443.5:0.00532,((2476:0.01591,254232:0.02811)0.987.133:0.01303,((252310:0.01065,(252709:0.00864,251747:0.00935)0.787.17:0.00754)0.990.101:0.01755,169882:0.0391)0.854.29:0.00993)0.714.7:0.00677)0.936.46:0.01002,((251268:0.02424,533502:0.00746)0.955.62:0.00786,(((1912:0.01836,100286:0.00938)0.952.64:0.00728,((551394:0.02089,539886:0.01309)0.895.43:0.00654,(275358:0.0191,(159495:0.00638,159207:0.00198)0.980.78:0.01134)0.978.92:0.00874)0.968.71:0.00766)0.995.150:0.01458,(249460:0.00943,(253987:0.00781,237297:0.01988)0.156.3:0.00451)0.995.151:0.0146)0.935.48:0.00945)0.858.24:0.00672)0.742.24:0.0019,(((562395:0.0218,513370:0.01996)1.000.1229:0.04393,(556399:0.02995,258028:0.04526)0.917.50:0.00824)0.285.5:0.00633,(565133:0.01222,149156:0.01136)1.000.1230:0.02397)0.729.14:0.00676)0.755.24:0.00565)0.954.63:0.00774)0.986.94:0.01383)0.953.60:0.00806,(((((((125230:0.02414,(253550:0.02107,364844:0.01135)0.396:0.00952)1.000.1231:0.11589,(((291681:0.00294,(288119:0.02699,295551:0.00404)0.724.11:0.01345)0.955.63:0.01519,(((578937:0.0177,354389:0.03316)0.944.48:0.00644,((339060:0.01427,(105037:0.00823,293284:0.02196)0.949.60:0.00784)0.972.77:0.00746,((((((346395:0.02005,207160:0.00443)0.387.9:0.00292,(341624:0.00014,310175:0.00321)1.000.1232:0.01399)0.874.38:0.00255,334177:0.01397)0.934.48:0.00346,274019:0.01824)0.887.35:0.00344,269732:0.02135)0.940.60:0.00594,241034:0.01795)0.867.38:0.0042)0.857.27:0.00438)1.000.1233:0.02344,(84178:0.02212,(99953:0.01449,(240168:0.00196,107241:0.01853)0.938.58:0.00799)0.973.69:0.00916)0.975.83:0.01234)0.956.61:0.01234)0.998.236:0.03358,(((347173:0.00383,((341051:0.01566,288259:0.02984)1.000.1234:0.03668,301779:0.00932)0.795.17:0.0008)0.927.55:0.00851,((298588:0.01658,(287667:0.01424,319786:0.01136)0.961.66:0.01026)1.000.1235:0.02654,297608:0.0231)0.615.5:0.00848)0.337.2:0.01034,((303063:0.00286,(310931:0.01432,290386:0.00574)0.999.370:0.01098)0.413.4:0.00347,288500:0.00843)1.000.1236:0.04228)0.803.18:0.01883)1.000.1237:0.05171)0.729.15:0.01957,((592533:0.011,(144997:0.00924,149981:0.01873)0.725.16:0.00596)1.000.1238:0.06637,((518257:0.02594,(268892:0.02082,(548981:0.00845,(109173:0.00328,109181:0.00028)0.499.5:0.00604)0.550.8:0.00289)0.514.3:0.00499)1.000.1239:0.03907,((181820:0.02383,559587:0.00423)0.996.159:0.01838,(568069:0.02398,311025:0.03039)0.984.83:0.0182)0.776.23:0.01523)0.968.72:0.02148)0.744.16:0.01416)0.998.237:0.03797,(((((((576731:0.00203,348878:0.0012)0.995.152:0.02123,((((295015:0.01096,((297567:0.02731,313009:0.0138)0.892.36:0.01007,340727:0.01642)0.340.5:0.00181)0.999.371:0.01602,(560795:0.00715,321439:0.00673)0.891.44:0.04118)0.917.51:0.00817,49015:0.01787)0.919.49:0.01055,298213:0.01004)0.988.118:0.02054)1.000.1240:0.07258,(((295143:0.01483,(317109:0.00126,293189:0.00317)0.957.84:0.00382)0.804.18:0.03501,((294909:0.05484,((164603:0.00816,((309629:0.03382,274244:0.00173)0.970.68:0.00331,185187:0.00895)0.455.4:0.01118)0.949.61:0.01623,(328822:0.02537,288628:0.02702)0.924.41:0.01133)0.898.44:0.01783)0.929.60:0.0199,((112848:0.01108,74314:0.00944)1.000.1241:0.03858,((351182:0.02142,343616:0.02497)1.000.1242:0.03435,(578799:0.03014,208415:0.03416)0.659.6:0.00935)0.379.11:0.00959)0.614.5:0.017)0.707.8:0.01494)0.982.99:0.02016,(175554:0.069,((((((((((263057:0.00544,162639:0.00363)0.725.17:0.06379,(396239:0.00764,160191:0.01405)1.000.1243:0.03333)0.628.9:0.01584,515707:0.05212)0.899.44:0.00855,(((376333:0.01501,(279387:0.01402,435430:0.01335)0.906.50:0.00704)0.932.55:0.0055,((315412:0.01566,(312121:0.02059,314284:0.03424)0.466.9:0.00424)0.987.134:0.01729,457168:0.00919)0.905.48:0.01531)0.683.12:0.00303,(355032:0.02494,278675:0.01771)0.849.21:0.00483)0.913.51:0.00501)0.959.62:0.00771,(((375076:0.03076,(278171:0.016,277649:0.00809)1.000.1244:0.0241)0.981.84:0.01243,((279442:0.01401,(392238:0.01613,388276:0.02657)0.586.5:0.00358)0.972.78:0.00837,(((396217:0.01478,((169379:0.01501,(175413:0.0037,(196985:0.023,199177:0.02064)0.872.46:0.00605)0.813.19:0.01178)0.639.8:0.00774,216194:0.01426)0.897.40:0.01088)0.881.40:0.00815,(278000:0.01652,(277842:0.01947,407993:0.01541)0.990.102:0.00843)0.995.153:0.01403)0.885.44:0.00758,(279248:0.01895,(172976:0.02909,((206324:0.03364,(212911:0.00517,(342392:0.01949,270051:0.00697)1.000.1245:0.01892)0.879.44:0.00225)0.697.7:0.00367,274125:0.03387)0.969.73:0.00869)0.303.4:0.00687)0.971.77:0.00912)0.993.113:0.01419)0.977.86:0.01104)0.543.3:0.00333,(297675:0.01079,311186:0.01751)1.000.1246:0.03172)0.480.8:0.00177)0.992.136:0.01542,(268529:0.02043,(268155:0.00705,(177014:0.01927,262102:0.03071)0.841.27:0.01515)0.923.45:0.01828)0.987.135:0.02285)0.959.63:0.01679,((276218:0.02881,211443:0.02597)0.899.45:0.00616,(386135:0.031,(345791:0.01064,521254:0.00985)0.999.372:0.03397)0.896.39:0.013)0.743.25:0.00925)0.999.373:0.0333,((((214715:0.01035,207424:0.01027)1.000.1247:0.01951,(212074:0.01202,259536:0.00976)0.981.85:0.01103)0.996.160:0.01563,(((((188969:0.01851,185300:0.00918)0.999.374:0.01665,((194048:0.02577,((210056:0.00473,210068:0.02576)1.000.1248:0.01131,215550:0.00185)0.912.34:0.00303)0.929.61:0.0071,(287014:0.0073,177425:0.02357)0.815.14:0.00305)0.964.88:0.00694)0.720.15:0.00134,303491:0.01932)0.985.87:0.01,(233896:0.02735,(((279579:0.02722,(175646:0.00046,171617:0.01119)0.999.375:0.02866)0.971.78:0.01153,(181605:0.01727,185948:0.00183)0.818.12:0.00545)0.706.11:0.00577,97151:0.03103)0.905.49:0.00633)0.997.190:0.01411)0.814.12:0.00218,(285936:0.02855,321735:0.02577)0.905.50:0.00756)0.889.36:0.00535)0.928.50:0.00558,((((264534:0.01758,(348350:0.01105,(176600:0.03643,(329371:0.03118,277259:0.00995)0.435.7:0.00356)0.934.49:0.01056)0.869.43:0.00461)0.964.89:0.01953,(((((277818:0.02003,(199707:0.02037,(194225:0.00943,423455:0.01214)0.833.17:0.00342)0.499.6:0.00674)0.933.53:0.00892,(233977:0.02349,(((((210665:0.00298,337680:0.01582)0.864.41:0.00438,((329502:0.03456,(216363:0.00371,209216:0.02831)0.816.22:0.00082)0.975.84:0.00794,197438:0.04055)0.946.55:0.00593)0.987.136:0.01362,(216517:0.03579,(192206:0.01105,(193807:0.0132,(217096:0.01164,206367:0.01463)0.988.119:0.01151)0.820.10:0.0086)0.915.45:0.01306)0.853.28:0.00442)0.967.67:0.01009,279298:0.02447)0.552.7:0.00412,278501:0.00675)0.852.44:0.01013)0.990.103:0.01319)0.700.12:0.00307,(441329:0.00568,357305:0.00318)0.995.154:0.01293)0.949.62:0.00774,(234036:0.01447,(205160:0.03876,207962:0.00588)1.000.1249:0.0227)0.999.376:0.01678)0.994.147:0.01099,(174420:0.01976,(182945:0.01157,205981:0.01108)0.873.33:0.00387)0.999.377:0.01439)0.963.73:0.00778)0.159.4:0.00252,((((442846:0.02702,((453275:0.01567,261177:0.01812)0.977.87:0.00834,((517722:0.00275,((((98087:0.00516,391644:0.00951)0.938.59:0.01192,259625:0.01589)0.934.50:0.00715,(405046:0.0121,(96673:0.00822,387661:0.00952)0.882.42:0.00374)0.996.161:0.01078)0.822.22:0.00365,432085:0.02372)0.997.191:0.01192)0.977.88:0.00738,192494:0.01601)0.895.44:0.0032)0.936.47:0.00474)0.459.3:0.0041,(333695:0.00905,279774:0.01515)0.993.114:0.01477)0.732.15:0.00108,(278712:0.01872,(189730:0.019,213945:0.00633)0.996.162:0.01167)0.956.62:0.007)0.970.69:0.00905,(((214236:0.02575,(278873:0.03032,339503:0.06171)0.346.7:0.01256)0.185.3:0.00433,180290:0.03861)0.523.6:0.00829,(((302196:0.04219,(291153:0.01593,(309676:0.01158,294992:0.01338)0.026.2:0.00661)1.000.1250:0.03697)0.855.29:0.00668,(45604:0.03806,(325899:0.02522,353668:0.01349)1.000.1251:0.03655)0.933.54:0.01287)0.452.9:0.00876,(232423:0.03504,447761:0.03767)0.971.79:0.01562)0.813.20:0.00514)0.900.39:0.00758)0.909.44:0.00605)0.738.20:0.00347,((((((131498:0.01848,(420373:0.01254,390896:0.01825)0.920.41:0.00765)0.995.155:0.01466,((((404629:0.01751,180405:0.02298)0.998.238:0.0138,((176637:0.02159,447141:0.00972)0.995.156:0.01072,((((313410:0.02625,300665:0.02325)0.940.61:0.00912,350149:0.01046)0.859.42:0.00509,((214904:0.00495,(191199:0.01692,(188940:0.03081,206189:0.00358)0.959.64:0.0053)0.954.64:0.01699)0.998.239:0.01767,(292867:0.01817,(450849:0.0184,295975:0.01315)0.997.192:0.02008)0.877.36:0.00711)0.998.240:0.02)0.903.48:0.0068,(518896:0.03398,(((394270:0.01405,(169878:0.00016,269328:0.00266)0.993.115:0.01013)0.993.116:0.01432,466981:0.01495)0.732.16:0.00592,((189237:0.00787,278889:0.00799)0.491.3:0.0096,((131594:0.01564,322991:0.02089)0.174.5:0.01042,(332666:0.0222,276861:0.02141)0.802.19:0.00612)0.993.117:0.01659)0.980.79:0.00954)0.754.17:0.00343)0.908.46:0.00645)0.904.49:0.00524)0.350.6:0.00163)0.340.6:0.00224,(177453:0.02228,(292591:0.01458,291109:0.02122)0.566.9:0.0048)0.998.241:0.01312)0.754.18:0.00015,(((286179:0.02337,190976:0.02018)0.949.63:0.00758,((((178306:0.02159,(((((175658:0.01532,(196085:0.01896,198966:0.0089)0.739.20:0.00107)0.967.68:0.00881,181473:0.01157)0.854.30:0.00198,178596:0.02522)0.854.31:0.00176,(192522:0.00092,129623:0.01366)1.000.1252:0.00092)0.731.14:0.00364,188845:0.01318)0.926.58:0.0079)1.000.1253:0.02312,((((135567:0.00326,196240:0.02774)1.000.1254:0.00442,193038:0.02781)0.921.51:0.01263,((189531:0.01799,(179192:0.00266,267565:0.01919)0.998.242:0.02505)0.996.163:0.01663,(((198418:0.00265,259557:0.02377)0.977.89:0.00028,185723:0.01525)0.563.7:0.00878,((132313:0.0018,322067:0.03879)0.320.4:0.00056,271846:0.0145)0.998.243:0.01597)0.497.8:0.00791)0.935.49:0.00734)0.952.65:0.0074,((277717:0.01032,(183478:0.01898,185326:0.0019)0.933.55:0.00014)0.810.19:0.01035,194978:0.01709)0.973.70:0.00652)0.382.6:0.00195)0.921.52:0.00703,(290178:0.02161,(((377256:0.01056,((184222:0.01176,336035:0.00841)0.910.45:0.01916,135808:0.03732)0.229.5:0.00226)0.734.15:0.00102,174003:0.04045)0.949.64:0.01007,377508:0.00252)0.740.26:0.00187)0.900.40:0.00393)0.985.88:0.00909,((((182382:0.00664,204111:0.02356)0.896.40:0.00349,(184698:0.03882,211743:0.01522)0.743.26:0.00129)0.971.80:0.00518,209377:0.03924)0.995.157:0.0004,(((178620:0.01789,192712:0.0269)0.442.2:0.00118,215756:0.01101)0.525.4:0.00016,214699:0.02448)0.663.5:0.01117)0.177:0.01528)0.715.11:0.00345)0.453.3:0.00579,(177669:0.06469,260460:0.04692)0.995.158:0.02561)0.817.22:0.01023)0.785.20:0.00503)0.953.61:0.00671,((277967:0.01839,((270707:0.02577,(276206:0.01215,278912:0.00557)0.726.15:0.01092)0.991.115:0.01049,(393026:0.00229,264496:0.01272)0.765.21:0.01603)0.838.27:0.00288)0.996.164:0.0098,(430219:0.04263,(402704:0.01464,214392:0.028)0.564.7:0.00637)0.945.56:0.00819)0.437.4:0.00014)0.930.48:0.00315,((376658:0.01867,(183770:0.02547,(176935:0.02769,(190816:0.0246,186903:0.00292)0.427.4:0.00307)0.885.45:0.0037)0.991.116:0.01273)0.931.48:0.00501,(279389:0.02754,278882:0.01549)0.959.65:0.00644)0.873.34:0.00244)0.804.19:0.00228,((((285705:0.00537,(195005:0.01625,204144:0.01444)0.871.49:0.01854)0.996.165:0.00652,211335:0.01168)0.478:0.00021,(259623:0.01551,206173:0.0059)1.000.1255:0.01683)0.985.89:0.01276,(229150:0.00639,(322870:0.01669,264657:0.00994)0.888.30:0.00407)1.000.1256:0.02708)0.967.69:0.01138)0.902.32:0.00667,((447694:0.02275,413168:0.00281)0.776.24:0.00848,(289213:0.02163,(259051:0.00636,(466448:0.00854,264603:0.00292)0.791.23:0.00582)0.670:0.00184)0.269.4:0.00168)0.995.159:0.01293)0.367.4:0.00225)0.791.24:0.0032)0.827.18:0.00914)0.242.6:0.00739,(305143:0.01876,(276141:0.03032,(306275:0.01469,99034:0.00411)0.986.95:0.01474)0.857.28:0.00403)1.000.1257:0.03637)0.897.41:0.0079,(298057:0.08139,(344879:0.03157,(((291410:0.00318,(576221:0.01717,336002:0.01885)0.974.82:0.01128)0.960.76:0.0229,(298229:0.00672,(295848:0.01127,351900:0.02547)0.502.6:0.00316)0.979.92:0.01146)0.983.93:0.01703,((534929:0.00896,(((325655:0.00958,313650:0.0135)0.812.27:0.00479,(100296:0.00648,259819:0.01681)0.452.10:0.00319)0.802.20:0.00223,550070:0.01071)0.698.9:0.00344)0.999.378:0.02109,(((271502:0.00016,344142:0.00172)1.000.1258:0.03831,(277262:0.00315,272852:0.00278)0.995.160:0.02218)0.996.166:0.01835,(((((272858:0.01751,274052:0.00258)0.773.22:0.01145,(267153:0.01624,310093:0.0057)1.000.1259:0.03802)0.940.62:0.00971,276889:0.05496)0.954.65:0.0121,((355949:0.00624,583635:0.0193)0.920.42:0.00445,(((107308:0.01123,307011:0.01135)0.925.39:0.00385,((336446:0.00782,(255232:0.00323,262779:0.01206)0.788.17:0.00207)0.765.22:0.00118,(572335:0.00809,592982:0.00711)0.945.57:0.00922)0.813.21:0.00175)0.962.66:0.00415,299120:0.0211)0.855.30:0.00165)0.848.30:0.00415)0.927.56:0.01293,(338022:0.00761,(539073:0.00675,341053:0.00921)0.879.45:0.00455)1.000.1260:0.02436)0.781.20:0.00633)0.891.45:0.01079)0.939.57:0.00876)0.203.2:0.0072)1.000.1261:0.03149)0.907.43:0.01057)0.379.12:0.01092)0.959.66:0.01953)0.993.118:0.02134)0.767.35:0.00686,((344178:0.11322,(299921:0.044,(349490:0.0035,302042:0.00103)0.731.15:0.00486)1.000.1262:0.04388)0.873.35:0.02048,(304823:0.05618,308259:0.03274)1.000.1263:0.03624)0.982.100:0.01741)0.998.244:0.03116,(((((160421:0.01823,371274:0.02715)1.000.1264:0.06932,(559138:0.02906,(((344827:0.02088,153657:0.04773)0.989.101:0.02504,((166787:0.00513,(515828:0.01455,525518:0.01072)0.000.59:0.00735)0.443.6:0.0499,(30861:0.04919,((((130663:0.06469,129213:0.00044)0.999.379:0.01946,(236480:0.01437,247202:0.01645)0.887.36:0.00311)0.871.50:0.00303,((((((((((348210:0.02083,(184209:0.03598,469842:0.0086)0.923.46:0.00427)0.722.11:0.00157,(360785:0.01713,357833:0.01583)0.772.19:0.00098)0.932.56:0.00444,(196080:0.01782,(174194:0.01441,(175617:0.00484,563616:0.00078)0.995.161:0.0122)0.487.4:0.00302)0.999.380:0.01086)0.756.25:0.0035,(((((((((240992:0.01088,158906:0.02135)0.989.102:0.01674,(158839:0.02004,512957:0.01923)0.916.50:0.01334)0.994.148:0.01676,((((469778:0.02135,168571:0.02492)0.864.42:0.00454,(533830:0.03894,((356051:0.00723,433947:0.0059)0.993.119:0.01387,((271214:0.00967,(213284:0.02269,196778:0.03227)1.000.1265:0.01976)0.480.9:0.00699,(356687:0.00123,(171559:0.03502,(((577170:0.03806,176506:0.0362)0.645.14:0.00137,(193178:0.01065,349524:0.00873)0.805.22:0.01101)0.890.43:0.00014,(556126:0.02273,(271150:0.02433,(244304:0.00615,356052:0.02633)0.740.27:0.02095)0.081.2:0.00226)0.969.74:0.01498)0.383.2:0.00077)0.808.17:0.00076)0.998.245:0.01292)0.996.167:0.01562)0.950.81:0.01008)0.984.84:0.01206)0.334.2:0.00673,((205290:0.0167,352963:0.01273)0.949.65:0.01051,(((187179:0.00427,301746:0.00502)0.996.168:0.01848,208972:0.02071)0.881.41:0.00904,((540982:0.01103,568757:0.01263)0.925.40:0.00759,(130228:0.01574,(572727:0.00123,271259:0.01905)0.997.193:0.01344)0.798.14:0.00935)0.807.14:0.0088)0.175.5:0.00487)0.982.101:0.01024)0.972.79:0.00851,((324055:0.00014,261878:0.0028)1.000.1266:0.03903,((172821:0.00956,158667:0.0298)0.125.3:0.01206,((158651:0.02804,158706:0.01681)0.843.24:0.01232,231302:0.08889)0.978.93:0.017)0.996.169:0.02108)0.922.50:0.01375)0.822.23:0.00842)0.471.7:0.00725,(233549:0.00719,564649:0.00978)0.868.45:0.0073)0.944.49:0.00934,242308:0.03084)0.096.7:0.00646,256830:0.03011)0.172.4:0.00671,(287763:0.00705,415631:0.00015)1.000.1267:0.0226)0.960.77:0.00847,((((((((((343171:0.04287,(567752:0.0155,310432:0.01895)0.997.194:0.02237)0.983.94:0.01796,(((273149:0.02065,355627:0.00184)0.865.37:0.00015,268046:0.00272)1.000.1268:0.05982,(((((251373:0.01801,((100912:0.01097,(242857:0.01876,(565400:0.01706,312608:0.00331)0.970.70:0.00876)0.931.49:0.00666)0.870.42:0.00634,294705:0.01699)0.448.7:0.00896)0.996.170:0.02375,(297018:0.03432,294517:0.02322)0.139.4:0.01592)1.000.1269:0.05348,((266559:0.0083,324275:0.00096)0.998.246:0.03505,((513699:0.00025,265102:0.01154)0.888.31:0.01016,268454:0.00441)0.998.247:0.03054)0.894.38:0.02431)0.968.73:0.01752,555204:0.04232)0.583.8:0.01029,321621:0.04708)0.832.21:0.00854)0.155.5:0.01117)1.000.1270:0.04044,(((((337349:0.01675,316250:0.05105)0.989.103:0.01638,(272800:0.02748,(264746:0.01134,325779:0.01149)0.937.55:0.0074)0.869.44:0.00644)0.972.80:0.01987,270923:0.02038)1.000.1271:0.03694,(((((((((267490:0.02323,586099:0.02108)0.857.29:0.0044,((((((143016:0.00014,265791:0.02133)0.864.43:0.01206,106932:0.01372)0.655.8:0.00278,2124:0.01776)0.862.42:0.00267,216511:0.00334)0.990.104:0.01217,(103209:0.0131,535115:0.01624)0.814.13:0.00764)0.953.62:0.00807,(69910:0.0303,(((251069:0.00941,594511:0.00936)0.992.137:0.00974,(584509:0.03148,583661:0.03073)0.679.9:0.00288)0.942.61:0.0038,84913:0.00653)0.858.25:0.00553)0.750.18:0.00475)0.723.11:0.00564)0.938.60:0.00852,((312051:0.01614,314731:0.04162)1.000.1272:0.02325,336924:0.01832)0.712.7:0.00845)0.731.16:0.00275,(103752:0.03375,(351813:0.01958,(314206:0.02286,165462:0.02396)0.977.90:0.01554)0.962.67:0.01254)0.936.48:0.01189)0.772.20:0.00464,((((107586:0.02347,(((290384:0.04501,(301030:0.0166,(327684:0.0052,(295175:0.01711,288876:0.01751)0.947.60:0.00537)0.964.90:0.00632)0.972.81:0.00743)0.840.20:0.00274,(306803:0.00291,344230:0.01262)0.000.60:0.00015)0.066.2:0.00227,351472:0.03086)0.972.82:0.02336)0.983.95:0.0097,((((103448:0.02043,102929:0.02424)0.641.8:0.00823,534756:0.03641)0.907.44:0.00696,2142:0.03899)0.475.4:0.00423,(((557101:0.01197,346503:0.02244)0.931.50:0.00658,(297847:0.02991,(((297936:0.01372,340238:0.00848)0.999.381:0.01585,348924:0.01364)0.881.42:0.00426,326333:0.012)0.949.66:0.00629)0.187.5:0.00192)0.894.39:0.00658,(((((105738:0.01827,212167:0.01184)0.956.63:0.00741,((85799:0.01022,84248:0.00927)0.995.162:0.01077,(86315:0.01498,351351:0.0112)0.906.51:0.00505)0.972.83:0.00685)0.998.248:0.01566,((((337278:0.02023,304053:0.00157)0.743.27:0.00015,((213152:0.01629,(346189:0.01787,332661:0.0208)0.804.20:0.00747)0.878.49:0.00311,43707:0.00337)0.876.39:0.01225)0.980.80:0.00565,(((169522:0.00842,162479:0.00745)0.947.61:0.00905,(106533:0.00912,168200:0.00749)0.958.74:0.0083)0.736.17:0.00114,351650:0.00195)0.980.81:0.00951)0.734.16:0.00231,329927:0.02574)0.922.51:0.00506)0.644.9:0.00543,((328246:0.0057,(307197:0.01529,2170:0.02756)0.927.57:0.00612)0.987.137:0.00924,(((((105764:0.01087,(82965:0.01807,(269318:0.00496,(259544:0.01307,103100:0.00716)0.999.382:0.0121)0.675.4:0.00844)0.802.21:0.00421)0.969.75:0.00895,100068:0.02581)0.921.53:0.00641,(584296:0.02737,84905:0.00655)0.867.39:0.00649)0.988.120:0.01067,213576:0.01408)0.878.50:0.00432,(((353168:0.02541,575799:0.05731)0.968.74:0.01307,((((102311:0.00857,2088:0.01207)0.662.8:0.00287,(46597:0.01138,(149030:0.01543,549984:0.01267)0.606.8:0.00169)0.938.61:0.00405)0.971.81:0.00441,541984:0.00196)0.904.50:0.00735,(211784:0.00901,592852:0.01856)0.139.5:0.00216)0.873.36:0.00455)0.994.149:0.01365,((((300289:0.00445,328946:0.03717)0.900.41:0.02034,(292202:0.02867,(353551:0.00016,(301119:0.00867,292234:0.01608)0.999.383:0.0232)0.979.93:0.00872)0.935.50:0.00839)0.852.45:0.00665,((((100696:0.01905,543175:0.00355)0.879.46:0.00171,575677:0.01595)0.982.102:0.00509,((289723:0.03108,296201:0.00346)0.967.70:0.0003,82695:0.01703)0.131.6:0.00302)0.869.45:0.00289,290880:0.01324)0.899.46:0.00736)0.943.72:0.00576,(562683:0.01932,((2074:0.01665,542145:0.00892)0.619.6:0.00514,((106191:0.00014,312373:0.01317)0.538.4:0.01207,58157:0.01162)0.287.3:0.00292)0.954.66:0.00639)0.760.24:0.00444)0.925.41:0.00607)0.971.82:0.00806)0.957.85:0.00659)0.962.68:0.00756)0.939.58:0.00671,((83419:0.02759,(339973:0.02472,(109358:0.01287,82967:0.02727)0.993.120:0.01398)0.949.67:0.00897)0.767.36:0.00323,(((310111:0.01254,(2114:0.00411,169258:0.00105)0.924.42:0.00521)0.947.62:0.00645,((165485:0.00016,273823:0.00245)0.993.121:0.01292,(((255234:0.01573,(164502:0.00137,(246608:0.00641,(106618:0.01337,104591:0.01062)0.734.17:0.00479)0.991.117:0.01014)0.890.44:0.00201)0.998.249:0.01104,(150858:0.01472,(164946:0.00682,512265:0.01599)0.944.50:0.0064)0.943.73:0.00626)0.957.86:0.00734,(((535033:0.01803,593100:0.0037)0.448.8:0.00233,576727:0.00725)0.761.27:0.0013,(553043:0.01095,547285:0.01924)0.996.171:0.01163)0.790.17:0.01812)0.911.42:0.00624)0.650.3:0.00565)0.951.56:0.00655,((162812:0.0293,(322023:0.02031,268455:0.02231)0.875.37:0.00858)0.910.46:0.00804,((2113:0.01277,2141:0.01319)0.920.43:0.00739,(313890:0.01954,(106839:0.00014,552400:0.02181)0.892.37:0.01271)0.749.16:0.00433)0.991.118:0.01314)0.853.29:0.00364)0.867.40:0.00505)0.939.59:0.0069)0.828.17:0.00685)0.842.26:0.0052)0.952.66:0.00805)0.842.27:0.00232,((340840:0.00125,172998:0.00863)1.000.1273:0.0157,(326292:0.0025,(274213:0.01717,209309:0.00857)0.919.50:0.00316)0.993.122:0.0113)0.966.68:0.00777)0.956.64:0.00667,210346:0.0214)0.438.3:0.0036)0.977.91:0.01052,(((214336:0.01545,(2129:0.00565,318637:0.02079)0.991.119:0.0144)0.998.250:0.02088,(((510554:0.01348,(335492:0.02292,167679:0.02172)0.997.195:0.01745)0.956.65:0.0099,((329127:0.01689,313131:0.02446)0.992.138:0.01395,(339847:0.02764,310381:0.02323)0.991.120:0.01612)0.637.9:0.00568)0.898.45:0.00898,((((213819:0.00625,214885:0.01477)0.406.5:0.00659,(102469:0.01172,2111:0.01898)0.585.7:0.0088)0.991.121:0.0154,(171639:0.01467,287773:0.006)0.989.104:0.01265)0.677.5:0.0062,(212051:0.02297,212161:0.03765)0.443.7:0.0078)0.801.16:0.0064)0.954.67:0.01102)0.922.52:0.00627,303058:0.05971)0.794.22:0.00312)0.979.94:0.01171,(31669:0.01122,(((306007:0.00566,((293801:0.03638,289936:0.03051)1.000.1274:0.00083,302092:0.00218)0.906.52:0.0204)1.000.1275:0.02187,298018:0.01162)0.999.384:0.0292,(73430:0.01516,(317995:0.02322,216520:0.00079)0.929.62:0.00549)0.957.87:0.01253)0.905.51:0.00726)0.966.69:0.01)0.964.91:0.01351,(71561:0.00014,70217:0.00014)0.452.11:0.03542)0.992.139:0.02186,353376:0.04572)0.589.5:0.01095)0.984.85:0.01962,((((229228:0.00154,214549:0.00096)1.000.1276:0.05064,(158775:0.01462,(510136:0.01202,272955:0.0053)0.372.5:0.00784)1.000.1277:0.04167)1.000.1278:0.03278,((331015:0.02932,301320:0.02076)1.000.1279:0.04192,(301235:0.06946,((55854:0.00669,(92629:0.02841,127545:0.0131)0.493.3:0.00457)1.000.1280:0.03602,(190233:0.03231,114813:0.04292)1.000.1281:0.03039)0.910.47:0.01248)0.863.33:0.01117)0.641.9:0.00831)0.951.57:0.01045,(((585810:0.04977,(210764:0.00716,(326121:0.00566,165467:0.00926)0.941.66:0.0074)1.000.1282:0.03664)0.998.251:0.03026,(((353232:0.02756,(352418:0.01161,305603:0.02647)1.000.1283:0.05138)0.995.163:0.02756,(((2172:0.02117,170169:0.02427)0.985.90:0.01631,((267845:0.00645,578880:0.01229)1.000.1284:0.02502,(538772:0.01612,352441:0.0058)1.000.1285:0.03629)0.988.121:0.0202)0.615.6:0.01577,582218:0.06183)0.718.10:0.01065)0.568.6:0.00922,((((580286:0.03104,101192:0.02151)0.999.385:0.03723,(83758:0.02656,591971:0.02186)1.000.1286:0.04057)0.843.25:0.01289,100680:0.02431)0.034.2:0.00756,((104511:0.01151,592633:0.02112)1.000.1287:0.03518,((548932:0.0097,(309214:0.0044,211557:0.02694)0.804.21:0.01193)0.941.67:0.00592,(552281:0.01772,592631:0.02762)0.548.8:0.00498)0.994.150:0.01696)0.351.5:0.01081)0.994.151:0.02182)0.388.4:0.00518)0.960.78:0.00749,(((334051:0.00339,322826:0.00719)1.000.1288:0.03491,(327603:0.05667,(310889:0.00954,589648:0.01294)0.991.122:0.0213)0.792.17:0.01308)0.960.79:0.01406,(((258507:0.00074,317718:0.00367)0.000.61:0.04912,(260187:0.00186,317607:0.00014)1.000.1289:0.05066)0.969.76:0.02063,(((269567:0.00152,510667:0.01966)0.873.37:0.0044,321154:0.00221)1.000.1290:0.04229,(((46017:0.01069,299872:0.01322)1.000.1291:0.04123,((((166308:0.00344,270523:0.002)0.872.47:0.00416,178558:0.03251)1.000.1292:0.03722,(277624:0.02199,(332624:0.01223,288622:0.01729)1.000.1293:0.04352)0.975.85:0.01185)0.603.7:0.00321,(293968:0.04487,290589:0.03077)0.999.386:0.02314)0.926.59:0.0077)0.912.35:0.0081,((((((301070:0.02444,(279066:0.01402,432904:0.0058)0.996.172:0.01442)0.973.71:0.01295,(26715:0.04292,588947:0.04189)0.964.92:0.01513)0.959.67:0.01334,((((71799:0.01585,(460438:0.01506,295103:0.04012)0.990.105:0.01827)0.966.70:0.01092,((92942:0.02978,(557665:0.00251,561578:0.02444)0.996.173:0.0147)0.999.387:0.01916,(68339:0.02111,532929:0.02509)0.663.6:0.00722)0.804.22:0.00629)0.727.13:0.00538,((71907:0.03296,(2181:0.02044,192814:0.02915)1.000.1294:0.03356)0.250.3:0.00606,(590846:0.0193,(221543:0.00441,210076:0.00264)0.133.3:0.02279)1.000.1295:0.03533)0.964.93:0.00993)0.958.75:0.01233,(543900:0.01571,((459530:0.02498,2182:0.0018)0.664.4:0.00758,181627:0.03646)0.994.152:0.0165)0.998.252:0.01973)0.973.72:0.01321)0.994.153:0.01745,((291055:0.05549,(291091:0.02845,((42406:0.00519,296325:0.00097)0.999.388:0.0227,295322:0.03904)0.995.164:0.02158)0.923.47:0.0121)0.828.18:0.01071,(290731:0.04595,((2082:0.05127,((290403:0.02012,(543032:0.0201,74407:0.01325)0.985.91:0.01745)0.997.196:0.02288,(307519:0.0112,(303112:0.00877,319810:0.02775)0.988.122:0.01071)0.990.106:0.01745)0.364.6:0.00628)0.739.21:0.0108,(((213877:0.00951,(276679:0.01754,294605:0.02326)0.820.11:0.00755)0.822.24:0.00437,587640:0.02101)0.998.253:0.02512,(2051:0.02462,303045:0.03583)0.990.107:0.01717)0.646.8:0.00947)0.874.39:0.01606)0.158.2:0.01111)0.869.46:0.00459)0.899.47:0.00487,(((24916:0.00485,529733:0.01815)1.000.1296:0.04018,((28413:0.00829,46566:0.02417)0.930.49:0.01527,48550:0.00127)1.000.1297:0.03326)0.942.62:0.01865,(185006:0.0525,(((288807:0.03375,(301251:0.00839,24907:0.0031)0.991.123:0.02132)0.952.67:0.01576,351935:0.03892)0.995.165:0.02043,((((((226618:0.04025,(2208:0.01228,570979:0.01909)1.000.1298:0.03625)0.922.53:0.00826,((577260:0.01657,196997:0.02732)0.899.48:0.01084,((150436:0.00602,(583107:0.01943,73666:0.00016)0.801.17:0.01068)0.960.80:0.01215,(239506:0.00575,(70671:0.0029,88035:0.02404)0.386.5:0.0144)1.000.1299:0.02823)0.960.81:0.01472)0.969.77:0.00985)0.850.26:0.00243,(((2176:0.03682,470823:0.0333)0.874.40:0.01321,(406442:0.00794,227285:0.00766)0.999.389:0.02283)0.816.23:0.00809,526576:0.03974)0.988.123:0.01218)0.884.45:0.00523,((324283:0.04155,76393:0.00182)1.000.1300:0.0281,(300879:0.04647,(214997:0.01786,(427400:0.03683,115051:0.0242)0.994.154:0.01587)0.995.166:0.01763)0.894.40:0.0084)0.993.123:0.01511)0.604.7:0.00333,(172962:0.06589,(45645:0.02208,532681:0.02034)0.979.95:0.01629)0.913.52:0.00885)0.952.68:0.00888,((((((((177351:0.00601,(527574:0.0338,510921:0.03237)0.850.27:0.00078)0.891.46:0.00045,334729:0.0238)0.055.3:0.00525,((297631:0.00015,287991:0.02677)0.905.52:0.00318,(522959:0.00719,513915:0.0154)0.814.14:0.00215)0.990.108:0.01374)0.980.82:0.01295,326403:0.01519)1.000.1301:0.02883,328035:0.00569)1.000.1302:0.02201,((((327728:0.00447,((515798:0.0265,(515381:0.01926,516274:0.01822)1.000.1303:0.00203)1.000.1304:0.00068,316961:0.00467)1.000.1305:0.00973)0.996.174:0.00687,(342638:0.00818,198502:0.02265)0.802.22:0.00402)0.957.88:0.01158,(((588929:0.01879,(520962:0.02521,(530725:0.02075,524963:0.01704)0.937.56:0.00357)0.846.21:0.00125)0.895.45:0.00903,((539050:0.02065,560836:0.02862)0.796.22:0.02226,((72142:0.01758,(582363:0.01876,181943:0.03009)0.813.22:0.00364)0.795.18:0.00162,121883:0.00553)0.681.13:0.00172)0.784.20:0.00158)1.000.1306:0.0095,(((((521356:0.00734,(527333:0.02306,(296442:0.02066,184464:0.01248)0.772.21:0.00744)0.722.12:0.0098)0.384.4:0.01488,(512004:0.02771,329650:0.01642)0.951.58:0.00156)0.752.23:0.00014,((302357:0.02001,(299882:0.00729,(325802:0.01836,(338889:0.00659,(313753:0.03938,305546:0.00183)1.000.1307:0.02675)0.989.105:0.01435)0.999.390:0.01731)0.999.391:0.02113)0.936.49:0.01032,309068:0.01133)0.728.14:0.02815)0.980.83:0.00682,294453:0.01535)0.359:0.00083,((((303304:0.00527,304166:0.03732)0.990.109:0.01654,(44821:0.03259,195574:0.03221)0.950.82:0.01158)0.957.89:0.01252,(294196:0.0072,307154:0.00384)0.997.197:0.0195)0.548.9:0.00601,(186364:0.00546,298451:0.031)0.993.124:0.01092)0.856.31:0.00409)0.899.49:0.0028)0.991.124:0.00616)0.502.7:0.00016,516259:0.0075)0.991.125:0.01408)0.909.45:0.01074,(((((104043:0.01416,(227529:0.02224,((244669:0.01533,(((528418:0.01753,((570119:0.00015,(559347:0.01783,(521268:0.0346,471122:0.00269)0.736.18:0.00995)1.000.1308:0.01535)1.000.1309:0.01624,(108140:0.01556,(555099:0.00234,88701:0.00078)0.779.11:0.00798)0.684.11:0.02359)0.487.5:0.005)0.995.167:0.01428,60257:0.03179)0.527.4:0.00754,(154641:0.04386,(56320:0.02431,2173:0.04797)0.752.24:0.01343)0.857.30:0.00445)0.963.74:0.00872)1.000.1310:0.02011,471149:0.01841)0.865.38:0.00375)0.901.38:0.00768)0.553.6:0.00593,68416:0.03632)0.935.51:0.00959,(99412:0.03759,2216:0.03402)0.863.34:0.00963)0.995.168:0.01474,(((250493:0.00093,2217:0.0016)0.996.175:0.01284,245050:0.01648)0.975.86:0.01081,(((2196:0.03274,102425:0.02132)0.971.83:0.02098,((63123:0.0003,(288721:0.01634,106406:0.01877)0.998.254:0.0118)0.858.26:0.00765,(72060:0.00081,516705:0.00957)0.806.16:0.01018)0.853.30:0.00477)0.822.25:0.01572,(374568:0.01728,(361926:0.00156,536919:0.0019)0.969.78:0.00739)0.648.7:0.00543)0.999.392:0.0245)0.979.96:0.01054)0.927.58:0.00643,((114909:0.00014,50936:0.04619)0.999.393:0.01693,(544531:0.02436,68338:0.05487)0.920.44:0.0077)0.542.5:0.00391)0.981.86:0.00926)0.918.56:0.00651,(((321459:0.05635,204190:0.02691)0.970.71:0.02252,31856:0.04522)0.929.63:0.00933,(71710:0.00096,292177:0.00713)1.000.1311:0.04763)0.940.63:0.00759)0.992.140:0.01191)0.820.12:0.00486)0.986.96:0.01381)0.537.6:0.0034)0.860.37:0.00786)0.238.4:0.00458,(((323411:0.0016,296766:0.03519)1.000.1312:0.04224,((275816:0.00114,267768:0.02776)1.000.1313:0.05158,(289313:0.00177,510569:0.03403)0.272.4:0.04873)0.640.6:0.01017)0.729.16:0.01154,(559528:0.01343,(205607:0.03767,147308:0.04392)0.529.4:0.00465)0.999.394:0.02452)0.836.24:0.01005)0.748.20:0.00563)0.879.47:0.00523)0.824.15:0.00611)0.901.39:0.00765)0.280.6:0.00323)0.372.6:0.00463)0.890.45:0.00725)1.000.1314:0.03832)0.833.18:0.0134,(((547660:0.04656,321723:0.04084)0.969.79:0.01913,(548524:0.03916,259666:0.05381)0.832.22:0.01155)0.995.169:0.01973,(552964:0.06876,(484338:0.08962,(((181245:0.04316,(((584911:0.024,(((343453:0.03531,20534:0.01954)0.979.97:0.01766,277626:0.0534)0.937.57:0.01462,(539347:0.02736,((350691:0.00417,338456:0.00697)1.000.1315:0.03785,(244198:0.02031,(92743:0.02329,28952:0.02653)0.852.46:0.01123)1.000.1316:0.02554)0.973.73:0.01504)0.677.6:0.00704)0.836.25:0.00915)0.971.84:0.01126,300379:0.02124)0.323.4:0.00604,((259959:0.00591,174490:0.02649)0.852.47:0.0386,297647:0.01367)0.901.40:0.009)0.965.72:0.00854)0.598.6:0.00528,((73937:0.00101,518773:0.00453)0.735.17:0.01942,((446840:0.02472,(130336:0.00599,323303:0.00256)0.989.106:0.00999)0.918.57:0.01045,(301956:0.0229,268604:0.00505)0.880.31:0.01426)0.506.4:0.0088)0.991.126:0.01599)0.655.9:0.00505,(526582:0.06923,46419:0.02787)0.744.17:0.01212)0.896.41:0.01712)0.447.6:0.01278)0.394.3:0.00642)0.971.85:0.01777)0.933.56:0.01186,(34809:0.06552,((((459896:0.10784,((308029:0.00129,302334:0.01311)0.233.3:0.00691,355249:0.0172)1.000.1317:0.03902)0.309.6:0.00918,278515:0.04535)0.899.50:0.01242,((184778:0.0009,362169:0.00014)0.842.28:0.02367,(358706:0.00486,183884:0.00264)0.734.18:0.01666)0.999.395:0.01809)0.810.20:0.00801,(289258:0.05872,(85059:0.05927,((((569237:0.01671,(307953:0.01898,(334154:0.00648,551508:0.00932)0.883.38:0.00652)0.749.17:0.00212)1.000.1318:0.0165,(((581039:0.0067,(288263:0.01394,293135:0.03061)0.957.90:0.01042)1.000.1319:0.02254,(300691:0.01167,(539244:0.01677,302959:0.01381)0.977.92:0.01209)1.000.1320:0.0271)0.336.9:0.00528,((320615:0.00653,555046:0.00772)0.986.97:0.01033,((306182:0.00718,(308932:0.01232,304289:0.01913)0.975.87:0.00699)0.990.110:0.00903,299230:0.01654)0.826.14:0.00474)0.999.396:0.02113)0.761.28:0.00388)0.153.2:0.00651,((317779:0.02529,292651:0.04774)0.956.66:0.01354,316231:0.02952)0.833.19:0.01185)0.851.29:0.00848,(326540:0.00747,290809:0.01069)1.000.1321:0.06166)0.336.10:0.02199)0.944.51:0.0176)0.894.41:0.00837)0.960.82:0.01245)0.793.16:0.00421)0.970.72:0.01209,(((549284:0.01321,(204894:0.00581,340817:0.00942)0.881.43:0.00781)1.000.1322:0.05774,(291253:0.02068,(344506:0.00556,564445:0.01616)0.941.68:0.00954)1.000.1323:0.03998)0.995.170:0.03004,((260334:0.01727,580146:0.00594)0.043.2:0.05934,(139337:0.05084,259929:0.04847)0.933.57:0.02125)0.958.76:0.01516)0.925.42:0.01293)0.804.23:0.01007,((342245:0.0065,(347016:0.00256,287509:0.01601)0.891.47:0.00761)1.000.1324:0.05657,((299307:0.02449,((292861:0.01717,288253:0.00628)0.903.49:0.00014,298225:0.01892)0.999.397:0.0295)0.996.176:0.02507,(344007:0.04567,(291877:0.03736,(291030:0.00674,346845:0.01575)0.646.9:0.00363)0.999.398:0.02884)0.998.255:0.02762)0.986.98:0.0211)0.936.50:0.0114)0.121:0.00571,((((344306:0.00225,(324200:0.01933,558445:0.04885)0.990.111:0.01379)0.889.37:0.00353,(331181:0.0124,(298524:0.01325,290839:0.00765)0.943.74:0.00551)0.949.68:0.00606)0.156.4:0.00305,310376:0.00991)0.961.67:0.00815,289343:0.03118)1.000.1325:0.04558)0.998.256:0.0247,((509560:0.02759,347138:0.1075)0.788.18:0.02518,(537070:0.01603,((((339240:0.00194,523552:0.00676)0.880.32:0.0016,(((300787:0.00558,(303214:0.00721,300412:0.00617)0.888.32:0.00285)0.924.43:0.00111,(294949:0.00912,319354:0.00304)0.870.43:0.00206)0.895.46:0.00159,(((298545:0.0049,531310:0.02134)0.976.77:0.00885,(((312082:0.00323,509073:0.01459)0.989.107:0.0082,(288028:0.01198,300384:0.01236)0.944.52:0.00794)0.940.64:0.0044,288889:0.01388)0.746.27:0.0011)0.965.73:0.00731,(298773:0.01005,(290263:0.00958,(326382:0.0083,531519:0.00642)0.821.13:0.01039)0.902.33:0.00401)0.760.25:0.00368)0.870.44:0.00266)0.888.33:0.00162)0.729.17:0.00064,(355033:0.02928,((294848:0.01942,((324587:0.00774,321527:0.02584)0.948.66:0.00503,(317802:0.00515,290252:0.02539)0.000.62:0.00141)0.730.12:0.00168)0.985.92:0.02352,303068:0.00459)0.995.171:0.01659)0.921.54:0.00722)0.993.125:0.01628,(288702:0.02193,295156:0.03045)0.755.25:0.00756)0.970.73:0.01097)0.386.6:0.01015)1.000.1326:0.03317)0.685.5:0.01428,(349088:0.01526,(349786:0.05856,180141:0.04533)0.628.10:0.00935)0.984.86:0.01638)0.970.74:0.01297)0.791.25:0.00353,((322715:0.03513,555222:0.00029)0.840.21:0.00767,((264926:0.00515,(361496:0.03761,(197016:0.03842,303158:0.02833)0.894.42:0.00976)0.324:0.00489)0.981.87:0.00709,(570110:0.00029,169036:0.01403)0.487.6:0.00834)0.965.74:0.00731)0.548.10:0.00312)0.884.46:0.00497)0.721.9:0.00535,(296470:0.02854,178148:0.00585)0.995.172:0.0114)0.838.28:0.00266,((((534976:0.01859,(197052:0.0209,127309:0.00644)0.800.22:0.00326)0.974.83:0.00014,139815:0.00987)0.930.50:0.00284,469834:0.00569)0.971.86:0.00836,(167401:0.00429,567197:0.00409)0.714.8:0.00093)0.997.198:0.01013)0.957.91:0.00712,564444:0.03301)0.873.38:0.00385,((183618:0.03473,307742:0.16293)1.000.1327:0.07852,(195926:0.0256,191403:0.0462)0.971.87:0.02436)0.981.88:0.0168)0.894.43:0.00444,536369:0.04896)0.898.46:0.00437,(((((115985:0.00883,((71331:0.01838,((119570:0.00556,363625:0.02452)0.988.124:0.00048,469817:0.00567)0.982.103:0.01269)0.881.44:0.0047,469686:0.00078)0.791.26:0.00078)0.757.28:0.00636,1994:0.00104)0.897.42:0.00015,((469746:0.01188,((318215:0.03533,356164:0.03248)0.795.19:0.00152,256779:0.00106)0.913.53:0.0032)0.978.94:0.00713,(195781:0.01137,(524318:0.01122,(327163:0.00533,136529:0.00338)1.000.1328:0.01895)0.622.5:0.0017)0.915.46:0.00343)0.886.34:0.00394)0.997.199:0.0112,((531722:0.01292,521927:0.01413)0.943.75:0.00801,((469677:0.00198,(368270:0.01297,290660:0.01136)0.822.26:0.00078)0.879.48:0.00015,(357582:0.00806,362168:0.01967)0.999.399:0.00882)0.756.26:0.00117)0.955.64:0.00573)0.860.38:0.00271,(234488:0.01311,279192:0.00189)0.990.112:0.00846)0.839.31:0.00198)0.913.54:0.00377)1.000.1329:0.01711,((202014:0.01501,221758:0.01056)0.927.59:0.00055,(236772:0.00239,547506:0.04791)1.000.1330:0.00016)0.984.87:0.01166)0.944.53:0.01221)0.931.51:0.01229)0.979.98:0.01648)0.491.4:0.01102,(102227:0.02786,(156012:0.01641,225388:0.02237)0.300.3:0.01066)1.000.1331:0.04324)0.935.52:0.01263)0.610.5:0.00722)0.528.6:0.00867,(149371:0.02175,142777:0.02563)1.000.1332:0.05472)0.512.9:0.00786,(((((110974:0.03259,(((227654:0.02339,114455:0.0186)0.705.3:0.00616,530785:0.01384)0.754.19:0.00339,202454:0.02036)0.368.7:0.00544)0.992.141:0.01612,((227900:0.01333,106291:0.02817)1.000.1333:0.02654,((248514:0.016,(((529756:0.01153,((98890:0.02427,147764:0.01708)0.143.6:0.0052,369543:0.0171)0.842.29:0.00374)0.948.67:0.00573,524197:0.01009)0.960.83:0.00564,(363898:0.02465,((203234:0.02092,535685:0.01519)0.710.10:0.0079,147708:0.0308)0.838.29:0.00526)0.901.41:0.00576)0.827.19:0.00609)0.938.62:0.00836,239708:0.01643)0.989.108:0.01604)0.953.63:0.01475)0.976.78:0.01245,((45562:0.00314,576166:0.024)1.000.1334:0.05634,(((((572688:0.00667,(314461:0.01198,((237787:0.01324,528126:0.00707)0.902.34:0.00287,324492:0.00957)0.830.24:0.00238)0.982.104:0.00625)0.916.51:0.00238,221775:0.03347)0.769.17:0.00107,(156589:0.01686,((305622:0.02751,((((513931:0.01619,522470:0.01576)0.402.5:0.00256,526020:0.01484)0.972.84:0.00686,224633:0.00119)1.000.1335:0.01879,355592:0.00839)1.000.1336:0.02487)0.473.7:0.00666,(111183:0.00934,((109165:0.00903,(570409:0.00096,551763:0.00264)0.872.48:0.00186)1.000.1337:0.01388,(109162:0.00097,1928:0.00388)0.870.45:0.00469)0.960.84:0.00793)0.997.200:0.01308)0.942.63:0.00623)0.812.28:0.00163)0.836.26:0.00271,((220350:0.01112,((524103:0.01277,(107687:0.0003,315404:0.01651)0.399.6:0.00088)0.957.92:0.00571,(106462:0.00545,(358687:0.01268,523242:0.01739)0.310.5:0.00218)0.168.10:0.00294)0.989.109:0.00941)0.919.51:0.00366,((218156:0.00821,(223582:0.00995,527194:0.00813)0.987.138:0.00714)0.995.173:0.01247,(((528401:0.01286,516137:0.01457)0.366.3:0.00836,278734:0.01067)0.336.11:0.00648,(219611:0.0287,154097:0.00107)0.939.60:0.01419)0.959.68:0.00841)0.967.71:0.00575)0.945.58:0.00402)0.787.18:0.00312,((220506:0.00193,105448:0.0003)0.833.20:0.02697,(549827:0.01267,546351:0.03363)1.000.1338:0.03932)0.343.7:0.00487)1.000.1339:0.03892)0.201.4:0.00256)0.755.26:0.00552,(77707:0.08626,((148392:0.0274,535826:0.02972)1.000.1340:0.05514,((((135574:0.01656,((257899:0.01763,(149530:0.01845,146492:0.00561)0.973.74:0.01051)0.983.96:0.01482,(142402:0.01197,(143775:0.03456,149037:0.02187)0.518.2:0.00657)1.000.1341:0.03152)0.998.257:0.02251)0.603.8:0.00615,(143532:0.00595,(39177:0.02862,135575:0.01906)0.476.4:0.00703)0.996.177:0.01818)0.995.174:0.02258,((((135719:0.02098,(243363:0.02265,150115:0.01488)1.000.1342:0.03278)0.990.113:0.01862,((78378:0.0187,77928:0.02107)0.999.400:0.02924,144163:0.02089)0.900.42:0.01009)0.778.25:0.00556,((114458:0.02071,114426:0.00826)0.987.139:0.01172,((148689:0.01462,(143965:0.0057,(114468:0.00641,135754:0.01235)0.686.6:0.00433)0.971.88:0.00985)1.000.1343:0.02625,(((78150:0.01156,135576:0.03032)0.984.88:0.01106,(146610:0.01913,114539:0.02079)0.986.99:0.00818)0.899.51:0.00478,(143852:0.007,(141613:0.00534,149020:0.01843)0.909.46:0.0088)0.997.201:0.01448)0.480.10:0.00533)0.858.27:0.00758)0.730.13:0.00413)0.942.64:0.00985,((148038:0.03623,(143053:0.01131,135739:0.01632)1.000.1344:0.05603)0.309.7:0.01059,144220:0.02005)1.000.1345:0.02944)0.872.49:0.00668)0.989.110:0.01383,(((38316:0.00274,559588:0.00086)1.000.1346:0.03653,78147:0.01764)0.997.202:0.02798,((191131:0.01055,186572:0.0112)0.851.30:0.00997,(191537:0.01827,363476:0.01155)0.986.100:0.0122)1.000.1347:0.06093)0.728.15:0.00662)0.818.13:0.01213)0.982.105:0.02274)0.806.17:0.00993)0.911.43:0.00543,((1950:0.05106,(((1934:0.04855,(470812:0.07357,(63166:0.02697,(471139:0.00235,340509:0.01808)0.778.26:0.00266)0.990.114:0.04204)0.996.178:0.02736)0.652.7:0.00562,((1941:0.01836,(268540:0.06553,(569151:0.04113,(115278:0.00467,(564437:0.01761,(241488:0.01476,522507:0.0027)0.598.7:0.00173)0.388.5:0.00774)0.982.106:0.01332)0.975.88:0.01376)0.997.203:0.01704)0.895.47:0.0098,556500:0.03163)0.998.258:0.01622)0.923.48:0.00603,(((550226:0.00198,(30493:0.00393,567158:0.0045)0.913.55:0.00557)0.922.54:0.01099,(131765:0.01567,(437599:0.00621,385147:0.01022)0.999.401:0.02068)0.939.61:0.01232)1.000.1348:0.03327,(236176:0.00974,147743:0.01637)1.000.1349:0.03365)0.988.125:0.01635)0.824.16:0.00468)1.000.1350:0.02639,((((((114716:0.00648,77992:0.01638)1.000.1351:0.02996,((19082:0.00268,560962:0.00016)1.000.1352:0.02778,(((34893:0.0097,(((112819:0.01174,97947:0.02507)0.849.22:0.0061,(166153:0.01548,(149284:0.00823,144335:0.00716)0.973.75:0.0095)0.959.69:0.01219)0.988.126:0.01505,((146620:0.01597,(113096:0.02619,78297:0.01493)0.840.22:0.00583)0.989.111:0.01458,(142913:0.01027,113389:0.03333)0.987.140:0.01486)0.235.3:0.00682)0.835.21:0.0028)0.677.7:0.00229,254492:0.01959)0.965.75:0.00642,(368029:0.01507,(101861:0.00742,360716:0.00741)1.000.1353:0.03417)0.647.6:0.00664)0.987.141:0.01135)0.996.179:0.01629)0.362.3:0.0055,(144765:0.09402,(201673:0.03898,149277:0.01517)0.987.142:0.02495)0.784.21:0.0156)0.976.79:0.01098,((((230553:0.03345,157481:0.01961)1.000.1354:0.0345,((28974:0.01942,306330:0.02225)0.696.10:0.00998,(((521001:0.00578,(513985:0.0104,520075:0.01455)0.970.75:0.00832)0.230.3:0.00245,31170:0.00418)1.000.1355:0.02495,((291090:0.00942,335156:0.00532)0.843.26:0.00111,186233:0.03071)0.471.8:0.01022)1.000.1356:0.01987)0.528.7:0.00471)0.918.58:0.00619,(525992:0.01505,531353:0.0218)0.964.94:0.00759)0.978.95:0.01137,(199622:0.05152,470806:0.00329)0.937.58:0.00647)0.934.51:0.01019)0.979.99:0.01314,(249661:0.01291,(113333:0.01551,570230:0.0112)0.846.22:0.00738)0.897.43:0.01288)0.979.100:0.01307,(109738:0.00943,114497:0.0066)1.000.1357:0.0203)0.962.69:0.01564)0.956.67:0.01948)0.951.59:0.00629)0.990.115:0.01321,(((102513:0.01525,(294080:0.05473,(288856:0.0152,((((544305:0.0192,(165982:0.01127,2037:0.01276)0.997.204:0.01595)0.635.4:0.00387,((1969:0.00555,1923:0.01209)0.996.180:0.01352,(162060:0.01215,1942:0.01674)0.952.69:0.00622)0.988.127:0.00966)0.962.70:0.0086,((341926:0.00244,300463:0.00448)0.789.20:0.00116,(331693:0.03073,302627:0.01893)0.769.18:0.00164)0.882.43:0.0039)0.917.52:0.00543,(339564:0.00096,302194:0.00199)1.000.1358:0.01892)0.996.181:0.01529)0.668.5:0.00921)0.982.107:0.013)0.836.27:0.00794,(290027:0.01889,300752:0.05518)0.158.3:0.00478)1.000.1359:0.04841,(345001:0.05442,((((110885:0.02804,150801:0.02992)1.000.1360:0.02728,(((385574:0.01222,254175:0.01452)1.000.1361:0.04512,((576176:0.02222,150369:0.01565)0.999.402:0.02857,(143491:0.05394,(135845:0.02287,81498:0.04242)0.975.89:0.02889)0.949.69:0.02018)0.982.108:0.02272)0.971.89:0.01909,(((101957:0.01184,536271:0.024)0.696.11:0.00425,(235348:0.0024,(277821:0.00769,279117:0.01488)0.151.5:0.00186)1.000.1362:0.02264)0.983.97:0.01069,((590243:0.00586,((459820:0.0178,1975:0.02838)0.995.175:0.01442,441814:0.0048)0.867.41:0.00677)0.331.9:0.00426,(((460594:0.00646,541485:0.01643)0.956.68:0.00744,((538931:0.03061,196464:0.0138)0.768.25:0.00095,525517:0.00793)0.936.51:0.00442)0.913.56:0.00299,(((532714:0.00401,238617:0.0284)0.990.116:0.01327,572608:0.00233)0.986.101:0.03158,(134018:0.00747,(55494:0.00426,(((356639:0.00371,136506:0.01028)0.583.9:0.00305,((41939:0.00833,594682:0.00495)0.242.7:0.00287,326517:0.01569)0.521.8:0.00304)0.855.31:0.00364,(79914:0.01085,(242141:0.01502,337068:0.01846)0.967.72:0.00722)0.769.19:0.00136)0.990.117:0.00711)0.973.76:0.00537)0.425.3:0.00143)0.941.69:0.00503)0.915.47:0.0047)0.967.73:0.00788)0.919.52:0.00565)0.865.39:0.00531)0.866.32:0.00643,((547303:0.01739,558226:0.00259)0.847.28:0.01041,292037:0.08119)0.890.46:0.01734)0.945.59:0.00997,99288:0.06992)0.922.55:0.01272)0.319.2:0.00956)0.977.93:0.01618)0.752.25:0.00934)0.957.93:0.02116,(((202195:0.02547,149386:0.01556)0.528.8:0.00982,(148852:0.03148,((167008:0.00734,218475:0.00938)0.980.84:0.01214,(203506:0.01634,104105:0.00862)0.998.259:0.02116)0.780.15:0.00312)0.928.51:0.01677)1.000.1363:0.05147,((143744:0.02665,280540:0.0172)1.000.1364:0.02409,((271588:0.00394,257527:0.00698)1.000.1365:0.03235,(579376:0.02328,141173:0.01017)0.996.182:0.02892)0.785.21:0.00908)0.986.102:0.02006)0.989.112:0.02566)0.953.64:0.01338,((361803:0.05246,(((218502:0.01066,(148980:0.03186,(227458:0.01884,(148546:0.02555,148358:0.00729)0.950.83:0.01381)0.805.23:0.00841)0.996.183:0.01684)0.274.6:0.00753,145246:0.01743)0.996.184:0.02935,((114774:0.04686,114798:0.04142)0.938.63:0.01856,((579089:0.00016,22095:0.00268)0.999.403:0.03542,(78232:0.02173,(114668:0.00849,142693:0.02718)0.911.44:0.01277)0.859.43:0.01976)0.995.176:0.03193)0.999.404:0.03907)1.000.1366:0.05206)0.983.98:0.0217,(((248049:0.00626,(150318:0.00357,254517:0.00665)0.536.4:0.00612)1.000.1367:0.0866,(143966:0.01387,(238682:0.02099,150102:0.01286)0.972.85:0.02075)0.999.405:0.06296)0.619.7:0.0231,((141132:0.02279,(142488:0.00981,(221818:0.01073,(226064:0.00675,173359:0.00836)0.962.71:0.00924)0.930.51:0.00749)1.000.1368:0.03822)0.996.185:0.02543,(((((159032:0.03989,((353174:0.01616,349669:0.02046)0.199.3:0.00635,(290969:0.01258,305635:0.03209)0.905.53:0.00671)0.998.260:0.02679)0.995.177:0.02715,((((346144:0.03333,(((356283:0.01202,(300827:0.0135,330276:0.01663)0.828.19:0.00466)0.977.94:0.01149,(354544:0.0109,(((154077:0.00624,(564659:0.00078,169164:0.02231)0.923.49:0.00545)0.320.5:0.00172,((101318:0.00092,99341:0.02042)0.895.48:0.00389,101833:0.00386)0.948.68:0.00318)0.909.47:0.00502,289243:0.00674)0.975.90:0.00912)0.996.186:0.01379)0.982.109:0.00828,345156:0.01821)0.941.70:0.00516)0.937.59:0.00623,(((312372:0.0077,578287:0.00239)0.935.53:0.01117,(347952:0.0039,302316:0.02372)0.940.65:0.00781)0.996.187:0.01237,(332290:0.02097,301331:0.02294)0.796.23:0.00226)0.987.143:0.01005)0.906.53:0.00717,(((578661:0.01608,(163265:0.00575,(162741:0.00741,583826:0.0082)0.938.64:0.01052)0.513.3:0.0069)0.985.93:0.01433,((327771:0.00093,(253223:0.00626,151340:0.00305)0.854.32:0.0017)0.998.261:0.00874,207769:0.00169)0.883.39:0.04093)0.944.54:0.01073,(567010:0.01441,(103842:0.00362,(110080:0.0037,561070:0.02259)1.000.1369:0.02134)0.063.2:0.00777)1.000.1370:0.0452)0.494.8:0.00474)0.924.44:0.00671,((((296444:0.01103,303021:0.00603)0.993.126:0.0118,(296553:0.0088,(((348911:0.02177,319040:0.02156)0.987.144:0.01209,301040:0.00889)0.445.2:0.02469,298447:0.01046)0.965.76:0.00747)0.922.56:0.005)0.918.59:0.00562,319120:0.03215)1.000.1371:0.03569,((302419:0.02813,(291143:0.00474,308203:0.03341)0.995.178:0.01668)0.963.75:0.00864,((((153082:0.02916,317960:0.00986)0.999.406:0.01698,((539412:0.0075,564276:0.01547)1.000.1372:0.02632,(((((((((((312227:0.02036,(298928:0.01512,297914:0.01409)0.973.77:0.00917)0.978.96:0.00909,(303299:0.01447,326906:0.01455)0.954.68:0.00659)0.869.47:0.00214,(354566:0.01315,(293304:0.02431,296079:0.01195)0.838.30:0.00304)0.966.71:0.00725)0.987.145:0.0112,(330732:0.03637,296167:0.02041)0.939.62:0.00553)0.323.5:0.00014,320847:0.03721)0.935.54:0.00541,(294910:0.02876,(327764:0.00887,(291060:0.00904,290750:0.00835)0.558.7:0.00418)0.922.57:0.00632)0.513.4:0.00829)0.813.23:0.00492,317979:0.01673)0.970.76:0.01177,(209416:0.00615,((208980:0.00883,588581:0.01057)0.964.95:0.00495,((334390:0.00761,(155658:0.0071,(275098:0.00433,(318842:0.00802,235376:0.00857)0.957.94:0.00492)0.957.95:0.00444)0.397.8:0.00158)0.897.44:0.00343,103382:0.00433)0.316.7:0.00179)0.951.60:0.00562)0.984.89:0.01026)0.521.9:0.00466,(((533916:0.00945,(((578988:0.01225,(292966:0.02838,290855:0.01578)1.000.1373:0.03793)0.741.19:0.00116,101512:0.00748)0.877.37:0.00305,107513:0.02085)0.898.47:0.00769)0.659.7:0.00518,155291:0.02433)0.979.101:0.0117,((((578586:0.02211,588984:0.02115)0.969.80:0.01228,(216385:0.0051,100923:0.00903)0.873.39:0.00426)0.988.128:0.01302,1883:0.01169)1.000.1374:0.02346,(578259:0.02382,(265114:0.00793,((579752:0.01012,512831:0.01289)0.834.29:0.00272,((554962:0.00685,(318523:0.00567,307136:0.00872)0.891.48:0.00576)0.756.27:0.00501,(258501:0.01343,(((313743:0.02379,271256:0.00471)0.833.21:0.00427,(591242:0.01218,578420:0.01501)0.333.4:0.00461)0.862.43:0.00914,(((276250:0.01337,(325302:0.00119,343180:0.00091)0.011.2:0.00384)0.670.2:0.01165,267496:0.05217)0.733.16:0.00311,273956:0.00736)0.694.5:0.00478)0.826.15:0.00622)0.785.22:0.00298)0.735.18:0.0034)0.971.90:0.0124)0.913.57:0.01117)0.939.63:0.01155)0.997.205:0.01667)0.903.50:0.00666)0.715.12:0.0065,(((237268:0.04539,((155468:0.01272,300705:0.01032)0.988.129:0.00943,(295171:0.02232,352931:0.01465)0.848.31:0.00604)0.857.31:0.00498)0.721.10:0.01057,247308:0.02301)0.957.96:0.00932,((169316:0.00418,338019:0.03329)0.999.407:0.02261,(214424:0.03082,(291940:0.02924,314825:0.03762)0.126.6:0.00997)0.468.5:0.00869)0.843.27:0.002)0.931.52:0.00476)0.988.130:0.01866,(303347:0.02797,(327256:0.01008,((103498:0.01437,(83719:0.00786,316811:0.00527)0.806.18:0.00233)0.747.20:0.00195,(570921:0.01294,(155279:0.01351,584356:0.00417)0.731.17:0.00382)0.319.3:0.00202)0.362.4:0.0049)0.929.64:0.01459)0.865.40:0.01026)0.482.7:0.00534)0.947.63:0.00869)0.441.2:0.00634,(((578597:0.04913,(((563560:0.00939,(207007:0.01234,328878:0.01751)0.955.65:0.00797)0.940.66:0.00655,((542179:0.01439,156519:0.01686)0.998.262:0.01367,((308518:0.00974,(290891:0.01297,(293725:0.02932,(38895:0.00161,(299841:0.02813,293095:0.01652)0.722.13:0.00532)0.730.14:0.00082)0.942.65:0.00363)0.698.10:0.00087)0.914.40:0.00594,315117:0.0067)1.000.1375:0.02612)0.975.91:0.00887)0.550.9:0.01047,342684:0.07605)0.629.9:0.01536)0.989.113:0.01472,(((272931:0.06816,261103:0.04737)0.898.48:0.01654,250872:0.11382)0.244.3:0.00859,(549765:0.02864,104352:0.01191)1.000.1376:0.02621)0.728.16:0.00397)0.998.263:0.01682,((((303206:0.09083,((207296:0.01276,(165532:0.01622,582555:0.00882)0.540:0.00225)0.996.188:0.01725,(((300918:0.01519,(((210223:0.00249,249585:0.00798)0.957.97:0.00345,538956:0.01759)0.955.66:0.00716,(334792:0.00307,(((557769:0.01173,(297690:0.00715,(332438:0.00548,241265:0.01112)0.845.22:0.00012)0.958.77:0.00506)0.993.127:0.01072,(553911:0.00906,(355522:0.00849,((152620:0.00236,(84869:0.00429,(314419:0.00472,581721:0.00496)0.999.408:0.01024)0.857.32:0.00225)1.000.1377:0.00014,(593405:0.00845,238999:0.0043)0.940.67:0.00069)0.960.85:0.00455)0.849.23:0.00266)0.924.45:0.00418)0.889.38:0.00275,(((211997:0.00818,((356513:0.00829,99392:0.00542)0.731.18:0.00055,((356316:0.01531,586322:0.01246)0.999.409:0.01348,103216:0.00626)0.928.52:0.0005)0.928.53:0.0005)0.996.189:0.00584,315301:0.00475)0.910.48:0.00056,338247:0.02799)0.807.15:0.00077)0.515.9:0.00169)0.975.92:0.00726)0.991.127:0.00958)0.928.54:0.00799,((((((173669:0.01977,((163934:0.01045,(242056:0.00487,152221:0.00302)1.000.1378:0.01663)0.954.69:0.0059,((243372:0.00546,551088:0.00527)0.937.60:0.00513,(161918:0.00654,(((243105:0.01062,(342689:0.01599,561887:0.01383)0.957.98:0.00815)0.979.102:0.00813,(335200:0.01354,166526:0.01131)0.851.31:0.00364)0.952.70:0.00572,(((338550:0.01424,166121:0.00891)0.980.85:0.00802,((512861:0.00968,554618:0.0137)0.964.96:0.00671,(107319:0.02133,107522:0.0121)0.898.49:0.00319)0.971.91:0.00649)0.608.4:0.00016,((154297:0.01523,((((216099:0.01445,577030:0.00821)0.938.65:0.00405,(563403:0.01035,(171686:0.01398,169577:0.01469)0.439.11:0.00371)0.919.53:0.00376)0.971.92:0.00551,((264591:0.00964,(556424:0.00894,101760:0.01443)0.607.9:0.00324)0.982.110:0.00768,((100279:0.01806,172891:0.00109)0.970.77:0.00865,84953:0.04288)0.876.40:0.0029)0.858.28:0.00173)0.826.16:0.00472,215985:0.03089)0.496.6:0.00273)0.807.16:0.00269,(237728:0.00539,(((312550:0.01664,548324:0.00078)0.897.45:0.01011,576416:0.00486)0.857.33:0.00128,559781:0.00425)0.957.99:0.00599)0.971.93:0.00778)0.927.60:0.0044)0.825.19:0.00278)0.955.67:0.00811)0.933.58:0.00549)0.958.78:0.00655)0.889.39:0.00425)0.955.68:0.00692,239886:0.01775)0.418.6:0.00303,101584:0.01848)0.842.30:0.00579,156411:0.01903)0.963.76:0.00974,207085:0.01562)0.971.94:0.01202,(249825:0.0161,209931:0.0193)1.000.1379:0.0259)0.822.27:0.00301)0.940.68:0.00985,261406:0.06734)0.550.10:0.01337)0.841.28:0.01186)0.992.142:0.02341,((110799:0.01064,547076:0.01562)1.000.1380:0.04545,(167267:0.01565,(101991:0.02485,318963:0.02896)0.947.64:0.01222)0.648.8:0.0132)0.999.410:0.03531)0.971.95:0.01273,(((587501:0.00914,553019:0.01971)0.966.72:0.00015,276301:0.00106)1.000.1381:0.04997,((300348:0.03578,(308981:0.01018,340349:0.0395)0.898.50:0.01434)0.997.206:0.02055,(((1956:0.00307,294350:0.00147)1.000.1382:0.02364,((273097:0.01861,(581753:0.01008,327390:0.01711)0.860.39:0.00652)0.893.35:0.00994,(586944:0.0173,268148:0.03486)0.984.90:0.01446)0.999.411:0.03387)0.938.66:0.01088,(256305:0.03326,((99573:0.00865,591233:0.00923)1.000.1383:0.03753,((356053:0.00241,352360:0.00092)0.965.77:0.00754,549508:0.00701)1.000.1384:0.02142)0.533.3:0.01475)0.982.111:0.01523)0.608.5:0.00443)0.983.99:0.01249)0.972.86:0.00909)0.820.13:0.00241,((55917:0.00479,43783:0.00297)0.906.54:0.06258,(47049:0.02915,(21213:0.02414,(205678:0.01381,(289309:0.03142,520917:0.01352)0.908.47:0.0089)0.693.7:0.00903)0.998.264:0.02012)0.953.65:0.01401)0.998.265:0.02154)0.731.19:0.00505)0.966.73:0.0145)0.997.207:0.01678,(((((291293:0.00573,314276:0.03043)0.387.10:0.00824,(288652:0.03206,(327330:0.00788,(302775:0.01702,300161:0.0158)0.953.66:0.01408)0.948.69:0.00926)0.988.131:0.02126)0.959.70:0.01,301037:0.02477)0.972.87:0.0088,((((302016:0.0121,297430:0.00788)0.877.38:0.00595,(296205:0.01433,(336579:0.00885,307692:0.00931)0.787.19:0.00358)0.913.58:0.00476)0.808.18:0.00499,(335713:0.01933,300177:0.00866)0.341.6:0.00422)0.957.100:0.00555,((((330846:0.00883,297508:0.01009)0.803.19:0.00205,287662:0.01593)0.982.112:0.00684,(((289466:0.01101,((523743:0.00554,527187:0.01922)0.672.3:0.00627,(294484:0.00355,(306627:0.00081,517825:0.02967)0.737.10:0.00081)0.754.20:0.00182)0.814.15:0.00337)0.912.36:0.00626,291169:0.01902)0.921.55:0.00547,(515043:0.01267,(((297758:0.02077,337284:0.04684)0.523.7:0.00016,(330660:0.00833,517244:0.00539)0.918.60:0.00275)0.980.86:0.00474,(356132:0.00403,(((297373:0.00113,590764:0.01066)0.963.77:0.0059,546281:0.01626)0.964.97:0.00555,349018:0.0153)0.849.24:0.00216)0.763.21:0.00087)0.781.21:0.00094)0.952.71:0.00542)0.719.6:0.00075)0.956.69:0.00382,(((((301116:0.00784,(((299917:0.00902,((295033:0.00803,297887:0.02446)0.679.10:0.00426,(348627:0.0116,293211:0.0107)0.732.17:0.00104)0.876.41:0.00277)0.224.4:0.00198,(294315:0.01194,342089:0.00602)0.920.45:0.00441)0.984.91:0.00691,332097:0.00693)0.374.5:0.00108)0.948.70:0.005,342651:0.00423)0.986.103:0.0067,((295112:0.01111,(299501:0.00533,328258:0.01464)0.851.32:0.00014)0.963.78:0.00529,(301941:0.01488,298314:0.00361)0.883.40:0.00183)0.936.52:0.00016)0.959.71:0.00614,(332604:0.00695,291804:0.01139)0.073.6:0.00221)0.883.41:0.00486,301613:0.0104)0.886.35:0.00233)0.350.7:0.00082)0.730.15:0.00121)1.000.1385:0.02102,(131776:0.04414,234533:0.03665)0.960.86:0.01676)0.956.70:0.00939)0.929.65:0.00692)0.944.55:0.0072)0.985.94:0.01303)0.574.4:0.01149)0.955.69:0.01262,292562:0.05088)0.772.22:0.00889,(((253230:0.00404,99097:0.01401)0.916.52:0.00442,(155511:0.00292,258985:0.01412)0.772.23:0.00179)1.000.1386:0.05638,(300128:0.00724,(300082:0.01357,300027:0.00435)0.986.104:0.01297)1.000.1387:0.03873)0.973.78:0.02084)0.932.57:0.01278,(((218868:0.01351,327454:0.0106)0.995.179:0.01794,(137403:0.00764,(((570164:0.01054,569212:0.01837)0.920.46:0.01144,(129393:0.01415,546510:0.01814)0.972.88:0.00639)0.925.43:0.00613,99598:0.02192)0.110.2:0.00187)0.970.78:0.00973)1.000.1388:0.03254,(((109167:0.0033,109166:0.00167)0.888.34:0.00464,109613:0.00502)1.000.1389:0.04932,((544376:0.02692,(((136816:0.01294,28192:0.01621)0.877.39:0.00537,(189900:0.01773,(256827:0.00589,241348:0.01642)0.905.54:0.0053)0.772.24:0.00523)0.199.4:0.00294,108432:0.0227)0.294.6:0.00256)0.518.3:0.0049,((((587311:0.00091,337880:0.00468)0.973.79:0.00419,559276:0.03459)0.984.92:0.00713,(347798:0.03653,171842:0.00422)0.973.80:0.00587)0.955.70:0.00628,1884:0.02109)0.691.8:0.00245)0.960.87:0.01238)0.096.8:0.00499)0.396.2:0.00396)0.900.43:0.01459)0.998.266:0.03237)0.884.47:0.0158)0.962.72:0.0121)0.852.48:0.01082)0.979.103:0.02263,1878:0.09026)0.787.20:0.01398,((((190026:0.03335,418501:0.00029)0.516.5:0.0566,((248766:0.01414,575960:0.00778)0.992.143:0.02024,(((((144084:0.01955,(((145633:0.00676,(((((148721:0.01016,143794:0.00409)0.935.55:0.00428,150138:0.01115)0.944.56:0.0036,(144661:0.00403,144499:0.01033)0.959.72:0.00911)0.661.8:0.00014,144753:0.01078)0.988.132:0.00015,(((147588:0.0102,147501:0.00721)0.890.47:0.00315,150253:0.00844)0.874.41:0.00185,149802:0.00663)0.926.60:0.00377)0.819.22:0.00533)0.966.74:0.00638,(149879:0.01108,(147729:0.0085,147387:0.00552)0.966.75:0.00612)0.977.95:0.00712)0.903.51:0.0045,(((200373:0.0156,145163:0.01123)0.849.25:0.00555,150021:0.01793)0.959.73:0.00748,(((147232:0.01415,(147603:0.00349,147907:0.02272)0.910.49:0.00657)0.766.21:0.00303,146402:0.01461)0.564.8:0.00345,(((149800:0.00758,147312:0.0122)0.914.41:0.00374,(145895:0.0082,(145456:0.01315,(149452:0.00527,147203:0.00614)0.187.6:0.00352)0.916.53:0.0052)0.923.50:0.00373)0.742.25:0.00153,147906:0.01011)0.403.5:0.00537)0.859.44:0.00218)0.937.61:0.00549)0.470.6:0.00917)1.000.1390:0.02374,((148704:0.02175,144258:0.03094)0.798.15:0.00304,(203209:0.01885,((233478:0.01899,167498:0.01815)0.463.8:0.00367,((((((231096:0.01814,((303036:0.00675,(298464:0.01592,333235:0.00525)0.737.11:0.00809)0.958.79:0.00989,523051:0.00309)0.996.190:0.01232)0.952.72:0.00876,(230364:0.02271,(((515059:0.02863,(190844:0.00666,(520462:0.02035,518598:0.01854)0.954.70:0.00639)0.912.37:0.00513)0.839.32:0.00466,((353194:0.00095,324379:0.00533)0.786.22:0.00189,340829:0.03171)0.083.4:0.01578)0.941.71:0.00623,(273361:0.00439,243150:0.00099)0.900.44:0.0235)0.508.7:0.00382)0.358.3:0.00317)0.920.47:0.00577,323400:0.02193)0.954.71:0.00587,((157566:0.00542,(175375:0.01775,234443:0.00713)0.978.97:0.00732)0.903.52:0.00408,(171772:0.01593,232099:0.00831)0.844.23:0.0033)0.806.19:0.00373)0.353.8:0.001,469639:0.01132)0.911.45:0.00453,((185451:0.02242,(104780:0.00552,(138006:0.0036,168181:0.00093)0.815.15:0.00799)0.933.59:0.00323)0.945.60:0.0044,((107044:0.00673,167699:0.00872)0.674.8:0.00394,((337670:0.03121,(109956:0.00155,335530:0.0058)0.948.71:0.0041)0.944.57:0.00428,(179688:0.0096,109541:0.00537)0.919.54:0.00521)0.746.28:0.00139)0.580.5:0.00453)0.876.42:0.00362)0.073.7:0.00184)0.980.87:0.01025)0.962.73:0.00742)0.872.50:0.00642)0.989.114:0.01209,(((214981:0.01098,569158:0.00467)0.986.105:0.01109,(((233805:0.01359,(355255:0.01389,216599:0.00078)0.997.208:0.01056)0.961.68:0.00487,565304:0.01979)0.175.6:0.00015,204176:0.00486)0.936.53:0.00565)0.999.412:0.01931,(149950:0.02726,(144995:0.02206,(202614:0.02225,200982:0.01666)0.450.7:0.01059)0.553.7:0.00879)0.998.267:0.02483)0.935.56:0.01009)0.860.40:0.00392,((77794:0.0285,(146571:0.00757,143926:0.0155)0.999.413:0.02335)0.987.146:0.01469,((((148007:0.02577,(144647:0.00989,202261:0.02824)0.874.42:0.00404)0.872.51:0.0034,150403:0.00923)0.976.80:0.01067,366302:0.01547)0.396.3:0.00479,146656:0.01352)0.996.191:0.01573)0.939.64:0.00789)0.847.29:0.0115,(149167:0.01278,(135820:0.01742,143922:0.01245)0.987.147:0.01467)1.000.1391:0.03892)0.559.5:0.00575)0.991.128:0.02135)0.966.76:0.0178,(1879:0.01491,569030:0.01831)0.999.414:0.02899)1.000.1392:0.03595,((108457:0.01809,((206450:0.03303,((114806:0.0265,144582:0.01635)0.995.180:0.02343,((338506:0.03568,(584405:0.00654,323552:0.01536)1.000.1393:0.04641)1.000.1394:0.07212,(288346:0.01734,(16915:0.01416,293970:0.03397)0.960.88:0.01615)0.994.155:0.03103)0.951.61:0.02165)0.992.144:0.02899)0.997.209:0.02326,(195578:0.01831,573374:0.01399)0.319.4:0.00616)0.748.21:0.01166)1.000.1395:0.04305,(291587:0.03058,((((212713:0.02153,289776:0.03925)0.995.181:0.01885,(587276:0.01485,(254135:0.02155,(248112:0.00044,((83084:0.009,345195:0.01349)0.970.79:0.00858,((316412:0.01022,((274261:0.00403,320791:0.01263)0.996.192:0.00936,(318202:0.01322,306219:0.00941)0.665.6:0.00272)0.987.148:0.00786)0.520.6:0.00146,(307832:0.00441,103074:0.02242)0.414.5:0.00456)0.350.8:0.00565)0.647.7:0.00629)0.709.8:0.00906)0.718.11:0.00642)0.999.415:0.01948)0.884.48:0.00742,(295364:0.01839,(300869:0.00014,312256:0.01333)1.000.1396:0.0391)0.980.88:0.01243)0.896.42:0.00808,580964:0.02284)1.000.1397:0.04782)0.994.156:0.02427)0.981.89:0.01536)0.792.18:0.01181)0.106.4:0.01032,(((((((558876:0.00285,(509797:0.00587,143645:0.01536)0.931.53:0.00835)0.926.61:0.01301,(99570:0.00588,243406:0.01228)0.999.416:0.02191)1.000.1398:0.02237,((((340420:0.00296,(148286:0.01111,160369:0.02429)0.774.12:0.00783)0.346.8:0.00324,147090:0.00999)0.999.417:0.01823,(509437:0.01939,(105877:0.00745,213433:0.0173)0.954.72:0.0071)0.908.48:0.01027)0.414.6:0.0052,(((567379:0.01088,(114259:0.0179,534938:0.00405)0.943.76:0.00364)0.756.28:0.00124,(549760:0.00613,105833:0.00311)0.936.54:0.00366)1.000.1399:0.01322,((79360:0.02811,((79430:0.00607,79194:0.02087)0.623.4:0.01066,(79223:0.03663,258089:0.01746)0.690.7:0.00725)0.151.6:0.00014)0.887.37:0.00277,(512662:0.01119,(577375:0.00877,2475:0.01044)0.317.6:0.00157)0.800.23:0.00175)0.976.81:0.00744)0.886.36:0.00324)1.000.1400:0.02433)0.944.58:0.01225,((157385:0.02288,(565660:0.02682,554336:0.01031)0.998.268:0.01856)0.975.93:0.01195,(566082:0.03991,(((((351676:0.02489,513117:0.02737)1.000.1401:0.02386,(319486:0.00473,(151832:0.02481,((81597:0.01974,518764:0.02971)0.873.40:0.00432,(540867:0.01012,511060:0.01107)0.911.46:0.0038)0.883.42:0.00313)0.570.9:0.00266)0.996.193:0.01391)0.986.106:0.01205,534571:0.0428)0.872.52:0.00505,((558868:0.01625,541446:0.02098)1.000.1402:0.02969,(((66951:0.02246,(80463:0.04604,(571923:0.02137,211941:0.00476)0.892.38:0.00413)0.969.81:0.01061)0.993.128:0.01644,(((533495:0.00549,(540824:0.01279,(313117:0.0097,66165:0.00723)0.536.5:0.00133)0.887.38:0.0021)0.731.20:0.00578,161604:0.01994)0.547.7:0.00244,(562180:0.01112,310915:0.02148)0.962.74:0.01016)1.000.1403:0.03054)0.956.71:0.01082,(239593:0.01217,(((((145424:0.00879,509493:0.0141)0.874.43:0.00297,(151737:0.00539,146292:0.02069)0.966.77:0.00656)0.890.48:0.00582,((157434:0.01588,(267968:0.00528,130615:0.01726)0.182.5:0.0099)0.503.7:0.00472,(230769:0.01036,151096:0.02027)0.975.94:0.01)0.909.48:0.00671)0.975.95:0.00807,(547090:0.01133,544371:0.0269)0.978.98:0.00853)0.599.7:0.00251,535284:0.01927)0.953.67:0.00817)0.986.107:0.01256)0.973.81:0.01347)0.954.73:0.00856)0.891.49:0.00382,((158192:0.02866,560267:0.01092)0.991.129:0.0192,(((315021:0.00741,57445:0.0272)0.981.90:0.00947,(570855:0.02318,353370:0.01594)0.890.49:0.00267)0.778.27:0.0054,(267276:0.01398,159438:0.0193)0.937.62:0.00862)0.654.5:0.00359)0.972.89:0.00749)0.589.6:0.00397)0.918.61:0.01101)0.861.40:0.01092)0.772.25:0.00727,(249734:0.09956,(559789:0.05286,(245007:0.02897,(107777:0.01505,56432:0.00921)1.000.1404:0.025)0.995.182:0.01967)0.971.96:0.01441)0.600.4:0.00238)0.943.77:0.00614,((320804:0.01542,72909:0.00522)1.000.1405:0.03529,(109319:0.03185,214867:0.00174)1.000.1406:0.04284)0.993.129:0.01673)0.399.7:0.00584,((134023:0.06745,((114514:0.00262,558040:0.01515)1.000.1407:0.05202,(((202388:0.01184,148765:0.00788)0.851.33:0.00399,(219599:0.02184,142695:0.01408)0.257.4:0.00479)0.994.157:0.0184,(77751:0.02093,135753:0.01807)1.000.1408:0.03131)0.998.269:0.02615)0.997.210:0.02859)0.961.69:0.01797,((((258955:0.03155,(165659:0.01143,221828:0.01574)0.995.183:0.02014)1.000.1409:0.04083,(((537419:0.0209,270605:0.01002)0.996.194:0.0235,((312838:0.01667,((167634:0.01194,((343959:0.01429,291869:0.02994)1.000.1410:0.03748,((106042:0.00618,((248651:0.00393,212612:0.0049)0.192.4:0.00275,342978:0.02058)0.698.11:0.00719)0.954.74:0.01212,(293605:0.0097,(262328:0.0049,295940:0.00999)0.352.4:0.00452)0.953.68:0.01615)0.997.211:0.02398)0.996.195:0.01861)0.223.7:0.00677,301023:0.01114)1.000.1411:0.03564)0.947.65:0.01098,(148295:0.03123,114648:0.02821)0.993.130:0.01684)0.516.6:0.00426)0.943.78:0.01624,(112800:0.01095,97753:0.02223)0.994.158:0.0206)0.999.418:0.0307)0.997.212:0.03719,((509329:0.02742,((580870:0.00462,571023:0.00693)1.000.1412:0.04573,187075:0.03486)0.999.419:0.03133)0.953.69:0.01642,(557739:0.00924,222751:0.00635)1.000.1413:0.03669)0.355.6:0.01129)0.995.184:0.02624,(((278831:0.01952,527083:0.00536)0.961.70:0.02044,(252161:0.03565,277705:0.01804)0.897.46:0.0128)0.900.45:0.01231,((140053:0.02658,(272261:0.01972,526392:0.0137)0.757.29:0.0081)0.990.118:0.018,(81010:0.02337,((79827:0.00708,62699:0.01213)0.858.29:0.00574,(150564:0.0248,(56341:0.0119,80791:0.01843)0.998.270:0.01771)0.833.22:0.01045)0.808.19:0.01016)0.999.420:0.04114)0.352.5:0.0096)1.000.1414:0.04219)0.895.49:0.02091)0.971.97:0.01344)0.705.4:0.01111)0.938.67:0.00882)0.947.66:0.01344)0.982.113:0.02595,((513523:0.062,232505:0.06395)0.571.7:0.01168,(((((((((578548:0.0248,543813:0.02721)1.000.1415:0.03389,((334320:0.01382,(251049:0.01641,(103093:0.02844,((574553:0.00759,(((219243:0.00759,(244639:0.02013,332717:0.01307)0.695.6:0.00569)0.425.4:0.00098,351083:0.02159)0.239.3:0.00358,142131:0.00442)0.949.70:0.00648)0.969.82:0.00679,(240552:0.01231,533024:0.02845)0.275:0.0029)0.918.62:0.00514)0.698.12:0.00766)0.983.100:0.01708)0.997.213:0.01993,(361111:0.02151,(231977:0.01281,(226802:0.00696,140883:0.01605)0.968.75:0.00944)0.682.5:0.00503)1.000.1416:0.04448)0.994.159:0.02153)0.809.16:0.01384,(2483:0.01811,(553113:0.00567,(179289:0.01438,539235:0.00885)0.912.38:0.01312)0.999.421:0.02426)0.867.42:0.01187)1.000.1417:0.04372,277727:0.08307)0.998.271:0.02573,(583756:0.08023,((144069:0.01777,(222673:0.00332,549972:0.00909)0.887.39:0.00331)0.957.101:0.01072,((304078:0.013,(113559:0.0067,(284644:0.00729,223126:0.01177)0.539.7:0.00398)0.951.62:0.0058)0.996.196:0.01968,113674:0.03489)0.865.41:0.01697)0.671.9:0.01562)0.926.62:0.00802)0.261.5:0.00379,((2488:0.01725,348830:0.00089)1.000.1418:0.02768,((309050:0.00907,593017:0.00959)0.999.422:0.02023,(((82023:0.01375,326466:0.01134)0.954.75:0.00719,536400:0.0053)0.752.26:0.00557,113309:0.0098)0.996.197:0.01673)0.999.423:0.01869)0.273.4:0.00792)0.993.131:0.02001,((257151:0.04143,((((((144042:0.01169,((81939:0.01525,167598:0.00486)0.898.51:0.00321,197142:0.0229)0.746.29:0.00322)0.924.46:0.00279,(527601:0.01348,584477:0.01233)0.849.26:0.00246)0.941.72:0.00735,525642:0.00682)0.818.14:0.00313,135933:0.01853)0.980.89:0.01322,(307869:0.02562,593617:0.0084)1.000.1419:0.07316)0.911.47:0.01302,(227169:0.01415,279186:0.03242)0.312.4:0.00846)0.978.99:0.01367)0.973.82:0.01463,(419525:0.0244,99927:0.02583)0.999.424:0.02405)0.950.84:0.01119)0.972.90:0.01214,((56235:0.02958,(332533:0.00482,238434:0.00451)0.873.41:0.01153)1.000.1420:0.02945,(((370124:0.00988,(160802:0.02438,(305278:0.01542,(321863:0.02335,(248306:0.00646,405895:0.01159)0.898.52:0.0053)0.002:0.00106)0.942.66:0.00664)0.953.70:0.00635)0.967.74:0.00804,(((236952:0.01787,99619:0.00881)0.999.425:0.01442,(249532:0.02236,(((201498:0.00662,(366805:0.00832,257729:0.00491)0.888.35:0.00158)0.926.63:0.00014,193898:0.01243)0.893.36:0.00158,235879:0.00512)0.873.42:0.01035)0.971.98:0.00865)0.862.44:0.00343,(2489:0.02284,(141304:0.01727,81220:0.01425)0.983.101:0.01107)0.968.76:0.00919)0.772.26:0.00401)0.905.55:0.0058,(((((((((198091:0.00168,(((40435:0.05946,278016:0.00696)1.000.1421:0.01164,((570755:0.03246,594449:0.00015)0.880.33:0.0009,(43819:0.0332,465948:0.02154)0.806.20:0.00042)0.814.16:0.00959)0.926.64:0.0045,44075:0.01794)0.975.96:0.00623)0.974.84:0.00679,237550:0.02246)0.996.198:0.01117,242140:0.00438)0.933.60:0.00367,((517388:0.01328,(163090:0.00951,((199804:0.00709,283608:0.00412)0.849.27:0.00322,248251:0.01462)0.955.71:0.01293)0.680.6:0.00399)0.862.45:0.00263,306541:0.0134)0.945.61:0.00566)0.988.133:0.0112,(((((520492:0.00188,585420:0.04325)0.925.44:0.00384,160886:0.00965)0.872.53:0.0031,(141107:0.0072,(549127:0.00508,(404148:0.00613,279271:0.00108)0.323.6:0.00892)0.088.2:0.00594)0.787.21:0.00721)0.917.53:0.00316,((((284420:0.00783,513757:0.00724)0.948.72:0.00397,(212159:0.0091,(142324:0.01274,397119:0.00836)1.000.1422:0.00648)0.971.99:0.00014)0.949.71:0.00451,70559:0.00852)0.925.45:0.00489,282928:0.01742)0.909.49:0.00327)0.979.104:0.00014,161490:0.00786)0.450.8:0.0039)0.931.54:0.00605,586032:0.03207)0.995.185:0.01378,((68384:0.01746,(74255:0.00369,42382:0.00275)0.982.114:0.01046)1.000.1423:0.03487,(((357684:0.01701,(238897:0.0104,2485:0.05148)0.948.73:0.0091)1.000.1424:0.01876,206581:0.00506)0.950.85:0.00718,(((108062:0.00519,19131:0.01559)0.992.145:0.01164,258308:0.0307)0.181.5:0.00304,(212927:0.00862,(167100:0.00014,144456:0.02181)0.988.134:0.005)0.872.54:0.00307)0.905.56:0.00558)0.415.3:0.0021)0.989.115:0.01041)0.841.29:0.00433,((527168:0.0242,(299710:0.01714,((587000:0.00785,78203:0.00881)0.972.91:0.0076,93552:0.00582)0.902.35:0.00596)0.969.83:0.01199)0.939.65:0.01163,((140230:0.02755,((510706:0.0144,(558369:0.00809,204990:0.01097)0.928.55:0.00639)0.981.91:0.01015,(172404:0.01296,169934:0.01007)0.836.28:0.00769)0.895.50:0.0072)0.939.66:0.00847,(((586743:0.00606,((184347:0.0129,(561874:0.00334,(150268:0.01419,(((574254:0.01052,547664:0.01014)0.890.50:0.00713,512577:0.00584)0.048.2:0.00338,164618:0.00903)0.961.71:0.00811)0.848.32:0.00353)0.886.37:0.00271)0.927.61:0.00528,(543870:0.00083,(154665:0.03868,147488:0.08441)0.996.199:0.02749)0.835.22:0.00248)0.925.46:0.00592)1.000.1425:0.02552,43436:0.00029)0.975.97:0.02568,((442126:0.01559,((293198:0.00801,46477:0.00091)0.924.47:0.0055,(207629:0.05349,(155740:0.00032,237157:0.04302)0.946.56:0.00398)0.967.75:0.00607)0.943.79:0.00611)0.810.21:0.00561,(((570493:0.01994,(348866:0.0089,560815:0.0121)0.952.73:0.00537)0.927.62:0.00557,(178057:0.01025,573731:0.02753)0.968.77:0.01222)0.752.27:0.00684,(((26308:0.01508,(263070:0.00158,(((100180:0.01206,184233:0.03144)0.863.35:0.01561,276940:0.03037)0.205.3:0.00731,(593826:0.00028,109982:0.00972)0.895.51:0.00102)0.773.23:0.00175)0.525.5:0.00172)0.999.426:0.01593,(((22691:0.00014,(144239:0.03317,175904:0.01359)0.962.75:0.01921)0.621.7:0.00016,417695:0.03545)0.871.51:0.00015,(196049:0.00552,192089:0.04855)0.917.54:0.00487)0.999.427:0.02337)0.999.428:0.02049,((564708:0.01157,(189569:0.017,(292196:0.01029,291105:0.01226)0.812.29:0.00878)0.798.16:0.00962)0.336.12:0.00543,((149645:0.00223,(303330:0.0107,518547:0.00281)0.924.48:0.00845)0.905.57:0.0058,(471002:0.02984,277106:0.0463)0.806.21:0.00921)0.094.3:0.00391)0.857.34:0.00659)0.388.6:0.00479)0.919.55:0.00613)1.000.1426:0.02568)0.958.80:0.01422)0.996.200:0.02035)0.982.115:0.01068)0.986.108:0.01087,((((163173:0.00813,500109:0.0045)0.946.57:0.00459,(((548392:0.0089,((536252:0.00379,46956:0.01596)0.058.3:0.00106,(365158:0.01077,(140496:0.00799,(296702:0.01247,353839:0.00976)0.870.46:0.00155)0.950.86:0.00408)0.911.48:0.00313)0.976.82:0.00488)0.958.81:0.00494,((223783:0.00884,(250357:0.00797,(158861:0.00679,243196:0.00934)0.947.67:0.00441)0.852.49:0.01186)0.918.63:0.00746,544883:0.00016)0.956.72:0.01383)0.716.6:0.00079,(((((268993:0.01606,546342:0.02536)0.971.100:0.0057,382125:0.02161)0.941.73:0.00487,(((165006:0.00765,353143:0.01576)0.902.36:0.00363,581108:0.00358)0.568.7:0.00547,(560742:0.00802,(168491:0.00733,(205748:0.00501,204906:0.016)0.752.28:0.02048)0.824.17:0.00154)0.983.102:0.00573)0.979.105:0.01024)0.959.74:0.00556,((140918:0.01142,238033:0.03471)0.966.78:0.01649,((((217769:0.00858,(343772:0.00638,(550032:0.00849,534367:0.00509)0.822.28:0.00254)0.927.63:0.00344)0.848.33:0.00931,(((222163:0.00997,587516:0.0238)0.974.85:0.00014,(267158:0.00014,542660:0.01995)0.863.36:0.00706)0.944.59:0.00811,156401:0.00323)0.989.116:0.00738)0.866.33:0.00015,(324731:0.00981,106893:0.00431)0.868.46:0.00426)0.891.50:0.00395,(221497:0.00379,(348099:0.00748,156920:0.01551)0.971.101:0.0061)0.994.160:0.00741)0.964.98:0.00691)0.502.8:0.00205)0.671.10:0.00071,565163:0.00593)0.980.90:0.00601)1.000.1427:0.01561)0.873.43:0.00688,347195:0.01071)0.146.2:0.00284,223975:0.01656)0.986.109:0.01177)0.958.82:0.01078)1.000.1428:0.02735)1.000.1429:0.02727)0.553.8:0.00518,(40351:0.06932,((142943:0.01321,(176843:0.02304,594313:0.00772)0.031:0.00829)1.000.1430:0.04291,((576633:0.02038,((543271:0.0098,(275936:0.01638,374206:0.01549)0.969.84:0.00999)0.994.161:0.01849,(415279:0.02107,547205:0.00108)1.000.1431:0.02329)0.617.6:0.00501)0.992.146:0.02412,170857:0.08728)0.991.130:0.02209)0.974.86:0.02168)0.986.110:0.01731)0.815.16:0.00704)0.988.135:0.02077)0.228:0.01119,(((((58606:0.06259,155601:0.03899)0.187.7:0.0138,((((((((331118:0.02257,269319:0.00161)0.976.83:0.00726,(307533:0.01658,160672:0.01225)0.391.7:0.00284)0.968.78:0.01014,((2610:0.0018,105344:0.00257)0.967.76:0.00665,((2613:0.00687,539298:0.0107)0.961.72:0.00529,115015:0.0072)0.913.59:0.00665)0.998.272:0.01694)1.000.1432:0.02403,(514663:0.0237,531607:0.02659)0.996.201:0.01862)0.849.28:0.0086,(140034:0.03862,183122:0.01409)0.937.63:0.0084)0.938.68:0.00584,((553854:0.02866,((24593:0.00418,130535:0.00056)0.760.26:0.01794,(570390:0.02025,555312:0.00921)0.985.95:0.01054)0.147.3:0.00502)0.704.15:0.00955,(327267:0.01696,((42147:0.00583,318195:0.00633)0.942.67:0.00712,591576:0.01986)0.946.58:0.00699)0.306.6:0.00465)0.941.74:0.0094)0.958.83:0.00716,((278764:0.02481,562755:0.00908)0.983.103:0.01297,((278317:0.00345,547353:0.00236)0.999.429:0.02395,(567241:0.0109,277932:0.01533)0.695.7:0.01147)0.479.11:0.01056)0.951.63:0.01028)0.913.60:0.00767,((((((((346430:0.00101,(156351:0.00975,537859:0.02303)0.462.4:0.00152)0.996.202:0.00944,((144558:0.00028,(350649:0.00014,2611:0.04335)0.958.84:0.00511)0.910.50:0.00078,330914:0.00457)0.884.49:0.0045)0.941.75:0.00441,(176767:0.00464,(93459:0.0045,278695:0.02183)0.349.5:0.00359)0.493.4:0.00797)0.998.273:0.01013,((183918:0.01634,129514:0.00835)0.950.87:0.00482,110611:0.00398)0.904.51:0.00248)0.986.111:0.0089,(399883:0.00594,(556145:0.01777,221747:0.00246)0.710.11:0.0063)0.874.44:0.00623)0.995.186:0.01166,(139749:0.00128,(318649:0.01892,211176:0.0122)0.977.96:0.00807)0.960.89:0.00687)0.651.9:0.00247,((549320:0.01129,255596:0.00249)0.817.23:0.00209,(298589:0.00555,(342719:0.03902,411790:0.00358)0.397.9:0.00635)0.890.51:0.01279)0.991.131:0.01105)0.987.149:0.00927,105029:0.00632)0.998.274:0.0203)1.000.1433:0.03574)0.999.430:0.03123,(((((19106:0.05454,(232028:0.03943,(161860:0.02027,(278338:0.01502,58609:0.01916)0.427.5:0.00361)0.999.431:0.02248)0.923.51:0.01665)0.529.5:0.00354,(352793:0.05038,58619:0.04598)0.970.80:0.01276)0.638.4:0.00938,((((376180:0.05915,(340277:0.03617,255630:0.01292)1.000.1434:0.05202)0.863.37:0.01281,155474:0.03825)0.889.40:0.00552,(534791:0.05088,(((((156524:0.033,((206913:0.01401,278040:0.00433)0.988.136:0.01117,(152561:0.01255,308220:0.01204)0.851.34:0.00015)0.934.52:0.03958)0.992.147:0.01642,(571975:0.05168,200444:0.02272)0.546.3:0.00569)0.923.52:0.00948,((320514:0.01145,(583611:0.0078,(576352:0.00461,356711:0.01095)0.923.53:0.00416)0.949.72:0.00622)1.000.1435:0.03464,((((322937:0.00698,(145556:0.01362,136283:0.00645)0.892.39:0.00398)0.971.102:0.00985,(99024:0.00682,(239948:0.01274,202962:0.01975)0.868.47:0.00683)0.915.48:0.0042)0.949.73:0.00737,(102384:0.00873,293250:0.01646)0.973.83:0.00982)1.000.1436:0.02407,((((215176:0.00585,277421:0.0121)0.978.100:0.01154,(511354:0.03787,(113510:0.00486,2589:0.02015)0.838.31:0.00224)0.643.4:0.00331)1.000.1437:0.03418,(342492:0.02725,252466:0.01822)0.964.99:0.01005)0.303.5:0.00571,(((278904:0.00549,(58615:0.01045,278936:0.01556)0.999.432:0.01829)0.996.203:0.01283,279603:0.00716)0.975.98:0.01156,(((58647:0.00726,(79778:0.01846,248079:0.02111)0.507.5:0.00325)0.982.116:0.00744,(535290:0.0213,((579336:0.00887,332966:0.0107)1.000.1438:0.02293,150898:0.0211)0.906.55:0.00689)0.851.35:0.00275)0.108.2:0.00014,(183196:0.01744,303670:0.0216)0.000.63:0.00298)0.919.56:0.00552)0.826.17:0.01134)0.963.79:0.01401)0.807.17:0.01058)0.899.52:0.00815)0.863.38:0.00736,(79737:0.02263,555927:0.01519)1.000.1439:0.05682)0.493.5:0.00443,((363114:0.02182,((584851:0.01129,586779:0.0196)0.972.92:0.00967,(168787:0.0195,66996:0.0153)0.818.15:0.00789)0.891.51:0.01068)0.874.45:0.00645,(252068:0.02739,(276351:0.01696,(274907:0.00632,327877:0.01952)0.753.31:0.00147)0.997.214:0.01519)0.934.53:0.00811)1.000.1440:0.02466)0.973.84:0.01057)0.569.4:0.00262)0.935.57:0.00509,(((2588:0.04648,(356218:0.01678,555186:0.01034)1.000.1441:0.03877)0.360.4:0.00923,((160192:0.01087,275754:0.01544)1.000.1442:0.03306,(201988:0.01742,(270590:0.04535,323057:0.01547)0.341.7:0.00518)1.000.1443:0.03243)0.976.84:0.01661)0.987.150:0.01524,((550726:0.02,((334363:0.00688,589134:0.01508)0.757.30:0.00405,328531:0.01273)0.789.21:0.01323)1.000.1444:0.03279,(90078:0.04196,(357873:0.0218,140328:0.02642)1.000.1445:0.03812)0.942.68:0.01344)0.907.45:0.00822)0.803.20:0.00237)0.878.51:0.00494)0.842.31:0.00788,((228648:0.0741,((288687:0.02182,221302:0.02321)0.990.119:0.01437,(209792:0.05987,(250749:0.03152,239504:0.01772)1.000.1446:0.03449)0.179.2:0.00458)0.989.117:0.01625)0.317.7:0.00764,(2577:0.03603,(154264:0.03834,153017:0.06799)0.962.76:0.02203)1.000.1447:0.03463)0.978.101:0.01527)0.963.80:0.00744,((302378:0.04918,204028:0.02002)1.000.1448:0.05551,((356575:0.0241,315126:0.03213)1.000.1449:0.07118,(533696:0.06229,((146310:0.02506,(407614:0.02229,(27784:0.01645,2585:0.0054)0.778.28:0.01152)0.205.4:0.00568)1.000.1450:0.05262,((315090:0.01668,(169235:0.02069,148599:0.00903)0.665.7:0.00327)0.955.72:0.01644,(319915:0.05572,342865:0.02451)0.964.100:0.01653)0.226.2:0.00898)0.960.90:0.01369)0.892.40:0.01559)0.992.148:0.02802)0.880.34:0.00846)0.839.33:0.00661)0.936.55:0.00933,((((58624:0.00014,2599:0.00079)1.000.1451:0.07828,(58626:0.04003,198207:0.05451)0.968.79:0.02329)0.998.275:0.02832,((((318753:0.0048,227919:0.0137)0.876.43:0.01957,(((42709:0.02271,(((153734:0.01073,566830:0.01392)0.894.44:0.00664,((132838:0.02736,548681:0.01918)0.735.19:0.00334,((126906:0.01286,(161173:0.01294,137491:0.00318)1.000.1452:0.01518)0.865.42:0.00484,((561097:0.02047,572203:0.00397)0.964.101:0.00574,172112:0.00025)0.680.7:0.00425)0.835.23:0.00209)0.752.29:0.00117)0.861.41:0.00469,(336236:0.0121,306350:0.0144)0.795.20:0.00598)0.902.37:0.00427)0.723.12:0.00152,(((153137:0.00794,83725:0.01859)0.808.20:0.00494,247967:0.02845)0.852.50:0.00613,567720:0.01128)0.971.103:0.00806)1.000.1453:0.0305,((77550:0.02377,((267557:0.00655,579386:0.01548)0.959.75:0.00971,(327621:0.008,551605:0.01983)0.180:0.00532)0.997.215:0.01731)0.956.73:0.01421,((250036:0.0186,2574:0.02234)1.000.1454:0.02487,(((((140348:0.00428,(((((448094:0.003,46108:0.01037)0.870.47:0.00266,82222:0.00483)0.827.20:0.00194,512199:0.02632)0.924.49:0.01458,2603:0.03333)0.971.104:0.00889,((42223:0.0143,((164255:0.0077,(127012:0.00329,(2604:0.02282,288167:0.0003)0.869.48:0.00156)0.969.85:0.01334)0.884.50:0.00182,432646:0.01772)0.924.50:0.00497)0.974.87:0.00931,(83153:0.01352,(434806:0.00455,418749:0.02853)0.587.5:0.00061)0.845.23:0.01407)0.250.4:0.00346)0.530.5:0.00561)0.980.91:0.01061,235307:0.00355)0.975.99:0.00868,(259708:0.00223,270190:0.02214)0.973.85:0.00665)0.960.91:0.00928,2605:0.00963)0.986.112:0.01368,(((593226:0.02427,(578726:0.02202,241717:0.00342)0.962.77:0.00602)0.968.80:0.00732,(267978:0.01669,(413415:0.01333,(224467:0.02075,488159:0.00331)0.925.47:0.00285)0.988.137:0.00644)0.875.38:0.00311)0.749.18:0.0322,(((359824:0.02029,309496:0.00175)0.490.5:0.00201,303779:0.00276)0.999.433:0.01623,(((((212044:0.02629,211042:0.00896)0.950.88:0.00679,(151061:0.00268,(449027:0.00969,74081:0.01088)0.873.44:0.01355)0.942.69:0.00542)0.789.22:0.00416,513548:0.00889)0.880.35:0.00237,(207271:0.00672,209617:0.0322)0.863.39:0.00249)0.565.3:0.00318,(137870:0.00776,47791:0.01957)0.987.151:0.01655)0.945.62:0.01074)0.787.22:0.01446)0.987.152:0.0159)0.730.16:0.00666)1.000.1455:0.0472)0.432.4:0.00632)0.964.102:0.01048)0.977.97:0.01049,((356181:0.01164,(99405:0.01298,330039:0.0075)0.944.60:0.00905)0.982.117:0.01049,((567966:0.01135,549650:0.01495)0.248.2:0.00502,(249391:0.01993,107814:0.025)0.928.56:0.00827)0.891.52:0.00511)0.916.54:0.00745)0.999.434:0.02528,((277776:0.01923,257862:0.01932)1.000.1456:0.06528,(((58635:0.10015,(58634:0.08201,((189580:0.02301,169084:0.02258)1.000.1457:0.05146,(((334587:0.00096,(534971:0.02351,(563166:0.00318,575495:0.01777)0.741.20:0.00089)0.733.17:0.00846)1.000.1458:0.02736,((((((200864:0.01132,111136:0.01187)0.843.28:0.00357,((141442:0.00847,593976:0.00793)0.957.102:0.009,(327093:0.01338,205255:0.02831)0.724.12:0.0104)0.953.71:0.00963)0.989.118:0.01156,((251656:0.01411,(220782:0.01013,(256413:0.01353,(511876:0.01087,512079:0.01529)0.652.8:0.00601)0.405.6:0.00769)0.971.105:0.01073)0.995.187:0.01801,((103998:0.0284,221084:0.00404)0.995.188:0.01458,(((546606:0.00923,351446:0.00864)0.906.56:0.00383,(205855:0.0075,151819:0.0117)0.592.11:0.00288)0.745.19:0.00252,523189:0.01262)1.000.1459:0.01344)0.835.24:0.00457)0.685.6:0.00824)0.530.6:0.00594,(((217925:0.0167,(114481:0.02305,560704:0.00425)1.000.1460:0.0517)0.745.20:0.00864,(156963:0.01853,239396:0.00524)0.979.106:0.01333)0.230.4:0.00508,575014:0.03173)0.998.276:0.01595)0.775.16:0.0079,(254014:0.01094,317507:0.02123)0.996.204:0.01754)0.937.64:0.01256,254579:0.03697)0.967.77:0.01341)0.397.10:0.00867,((578102:0.03354,559898:0.0146)0.999.435:0.02333,(((245713:0.01458,(247445:0.01084,532331:0.01194)0.990.120:0.0115)0.996.205:0.01338,(((136472:0.00543,(141044:0.01168,550164:0.02904)0.998.277:0.01856)0.940.69:0.006,582988:0.00014)0.934.54:0.00467,(((571372:0.01278,(248117:0.01438,245770:0.02172)0.590.5:0.00902)0.933.61:0.01049,(249693:0.01177,242737:0.02519)0.875.39:0.00761)0.920.48:0.00956,(307763:0.01578,((243349:0.01331,((112959:0.00928,266549:0.01149)0.976.85:0.01226,113298:0.01904)0.912.39:0.00438)0.962.78:0.00702,(577085:0.01097,(65682:0.01759,136478:0.01407)0.982.118:0.00837)0.965.78:0.0101)0.743.28:0.00194)0.888.36:0.00834)0.997.216:0.01838)0.858.30:0.0062)0.990.121:0.01157,((348223:0.01937,(327525:0.01275,159655:0.00675)0.926.65:0.00862)1.000.1461:0.02918,((345298:0.00188,(541511:0.00965,561275:0.0152)0.939.67:0.00305)0.317.8:0.00887,(562144:0.02711,(((511388:0.00786,527565:0.02189)0.744.18:0.00409,142125:0.01339)0.991.132:0.0066,((143187:0.01869,554920:0.00426)0.621.8:0.00243,594352:0.00631)0.904.52:0.00365)0.353.9:0.00094)0.989.119:0.01192)0.842.32:0.0077)0.175.7:0.00135)0.973.86:0.01357)0.978.102:0.01299)1.000.1462:0.03644)0.972.93:0.0202)0.584.4:0.01241)0.133.4:0.00774,(32271:0.12111,(((146568:0.01052,(112685:0.01452,227860:0.02471)0.998.278:0.01894)1.000.1463:0.03993,(115032:0.04001,135189:0.02187)0.975.100:0.02685)1.000.1464:0.06349,(58625:0.08264,(58628:0.08267,(58632:0.05372,58627:0.02479)1.000.1465:0.052)0.615.7:0.02194)0.958.85:0.02511)0.253.2:0.01504)0.962.79:0.01546)0.367.5:0.00747,(((((77560:0.00029,(580222:0.00161,568167:0.00972)0.888.37:0.00626)0.242.8:0.00291,(20151:0.00178,(144973:0.01077,188216:0.00729)0.973.87:0.00635)1.000.1466:0.01322)0.991.133:0.0128,(130977:0.01667,(139268:0.00315,539293:0.00896)0.983.104:0.01382)0.997.217:0.01613)0.991.134:0.02,(112775:0.02166,(137145:0.00329,95713:0.00883)0.852.51:0.01725)0.960.92:0.01223)1.000.1467:0.04774,((150422:0.05987,(244833:0.03761,((((555655:0.02372,581547:0.03037)0.714.9:0.0048,(((146467:0.00562,(220913:0.02009,(443310:0.00655,339418:0.00752)0.875.40:0.00989)0.180.2:0.00693)0.999.436:0.02177,((259377:0.00014,269473:0.0127)1.000.1468:0.02561,((223665:0.00485,127534:0.00583)0.904.53:0.00416,((40101:0.00641,510901:0.00592)0.991.135:0.01229,217969:0.02821)0.929.66:0.00851)0.964.103:0.01219)0.283.4:0.00723)0.963.81:0.01033,(161920:0.03902,269355:0.00953)0.965.79:0.01099)0.991.136:0.01236)0.572.4:0.00815,489354:0.02804)0.999.437:0.02797,((245429:0.01237,(162617:0.01189,381082:0.00769)0.995.189:0.01827)1.000.1469:0.02359,(510104:0.01557,(353158:0.0043,588867:0.00918)0.993.132:0.01447)0.111.3:0.00504)1.000.1470:0.04714)0.412.3:0.01333)0.998.279:0.03086)1.000.1471:0.04932,(((153026:0.02079,(223870:0.01901,(152312:0.01326,508075:0.01271)0.998.280:0.01815)0.807.18:0.00664)0.844.24:0.00394,(2594:0.00499,448050:0.01911)0.368.8:0.00258)1.000.1472:0.07901,(((533548:0.01294,436560:0.00818)0.876.44:0.00426,(((544631:0.01668,(510721:0.0151,(99418:0.01285,(362966:0.00705,170823:0.01596)0.961.73:0.00025)0.867.43:0.00537)0.789.23:0.00378)0.837.20:0.0044,392770:0.00166)0.979.107:0.01047,256521:0.01318)0.773.24:0.00198)0.740.28:0.00622,140715:0.00849)0.999.438:0.04492)0.998.281:0.04091)0.633.3:0.01247)0.985.96:0.02313)0.598.8:0.00837)0.907.46:0.0054)0.843.29:0.00354)0.970.81:0.00862,(((239946:0.05896,(276048:0.0603,((167177:0.01961,(178196:0.01832,549152:0.01305)0.801.18:0.0095)1.000.1473:0.0665,(225690:0.01566,58613:0.00148)0.851.36:0.05333)0.966.79:0.02197)0.877.40:0.00987)0.999.439:0.02741,((2568:0.00029,58649:0.00016)0.297.3:0.02406,195039:0.04855)0.974.88:0.02395)0.966.80:0.0239,2573:0.05668)0.994.162:0.0196)0.530.7:0.00455)0.840.23:0.00592,2576:0.11813)0.997.218:0.02327)0.924.51:0.01,(181596:0.11514,((((2534:0.00088,335025:0.00028)1.000.1474:0.05678,((275264:0.00773,229271:0.00441)1.000.1475:0.09894,((440170:0.00976,(334860:0.00497,268864:0.00417)0.877.41:0.00903)1.000.1476:0.06047,((329138:0.02585,538346:0.01087)0.995.190:0.01906,((231673:0.03831,(523672:0.01219,(146854:0.01577,248300:0.00647)0.926.66:0.00516)0.772.27:0.00478)0.996.206:0.01784,((34106:0.01587,258576:0.0151)0.988.138:0.01525,(((325671:0.02888,(273341:0.00589,557079:0.0061)0.999.440:0.02146)0.991.137:0.01403,244167:0.00973)0.707.9:0.00403,204415:0.01423)0.835.25:0.00474)0.954.76:0.0071)0.191.3:0.00278)0.874.46:0.00859)0.999.441:0.04069)0.580.6:0.01822)0.991.138:0.02523,(((2538:0.00015,276627:0.01481)0.999.442:0.01711,(527241:0.00797,(52545:0.02188,109536:0.01379)0.778.29:0.00176)0.156.5:0.00498)1.000.1477:0.04716,(((((278351:0.02633,232874:0.02475)1.000.1478:0.03075,(246616:0.01835,(((348061:0.00536,(19837:0.01019,572749:0.01562)0.794.23:0.00437)1.000.1479:0.01775,(((252788:0.01495,191465:0.00636)0.971.106:0.00785,249818:0.01849)0.455.5:0.00244,(255371:0.01265,(230351:0.0173,(232231:0.01207,(228243:0.01422,263176:0.03447)0.960.93:0.01029)0.896.43:0.00856)0.935.58:0.01239)0.842.33:0.00298)0.875.41:0.00362)0.942.70:0.00712,553616:0.02051)0.930.52:0.00656)0.992.149:0.01938)0.998.282:0.02543,(((348204:0.01068,(538448:0.01069,230258:0.01993)0.958.86:0.0105)1.000.1480:0.06282,(579010:0.01655,274126:0.0364)1.000.1481:0.03727)0.491.5:0.01407,(576442:0.01857,214395:0.00728)1.000.1482:0.0299)0.965.80:0.01939)0.124.4:0.0083,((256252:0.0762,((((340563:0.01053,((353774:0.016,161382:0.00669)0.938.69:0.0071,586760:0.0141)0.312.5:0.00398)1.000.1483:0.02194,((((132699:0.00467,561978:0.0071)0.971.107:0.0072,567942:0.01435)0.941.76:0.00482,(241706:0.0244,513355:0.01495)0.000.64:0.00223)0.849.29:0.00314,319273:0.00226)1.000.1484:0.0393)0.992.150:0.01905,(316764:0.02727,(299993:0.02151,(200498:0.0211,590902:0.00613)0.754.21:0.005)0.985.97:0.01113)1.000.1485:0.04624)0.813.24:0.00341,(((((((159352:0.01635,251895:0.01661)1.000.1486:0.03248,((136195:0.0054,255565:0.01777)1.000.1487:0.03151,(194491:0.02388,568024:0.02644)0.987.153:0.01613)0.938.70:0.01834)0.989.120:0.02031,(250836:0.03657,237153:0.05611)0.700.13:0.00926)0.749.19:0.01137,(590838:0.04301,(228517:0.02034,(115129:0.00326,184685:0.01543)0.779.12:0.02256)0.988.139:0.01712)0.981.92:0.01627)0.970.82:0.01205,(278585:0.02252,228176:0.02226)1.000.1488:0.03217)0.838.32:0.00311,((262076:0.0223,242491:0.01042)1.000.1489:0.03334,((295985:0.01909,(2546:0.04322,273031:0.0047)0.915.49:0.01245)1.000.1490:0.03358,((584337:0.02621,((255544:0.03037,(256191:0.01542,((236361:0.02073,219002:0.02047)0.505.6:0.00762,(566103:0.03277,259026:0.00168)0.994.163:0.01439)0.512.10:0.00369)0.981.93:0.0119)1.000.1491:0.03305,(586573:0.0077,(579046:0.01676,(263575:0.00183,2550:0.05737)0.830.25:0.00231)0.997.219:0.01232)0.990.122:0.01614)0.938.71:0.01181)0.750.19:0.00766,29419:0.019)1.000.1492:0.02672)0.944.61:0.01234)0.981.94:0.01034)0.828.20:0.00682,((((240609:0.06016,(211973:0.02161,270239:0.01341)0.990.123:0.01955)0.134.3:0.00527,(268688:0.01791,(109301:0.01986,(263421:0.02978,244393:0.02047)1.000.1493:0.0313)0.987.154:0.0167)1.000.1494:0.02652)0.954.77:0.01028,(((168111:0.00614,144512:0.01335)0.736.19:0.0071,(249508:0.02016,(245374:0.02795,150023:0.02154)0.475.5:0.00558)1.000.1495:0.02721)0.999.443:0.0219,(570841:0.04181,344629:0.04427)0.978.103:0.0139)0.913.61:0.00706)0.540.2:0.00765,(((573306:0.01678,142556:0.00706)1.000.1496:0.04024,583745:0.05857)0.665.8:0.00965,(182208:0.02002,(294530:0.02423,315083:0.01825)0.999.444:0.02327)0.972.94:0.013)0.965.81:0.01226)0.970.83:0.01228)0.821.14:0.00622)0.765.23:0.00854)0.537.7:0.00391,((63062:0.0358,((101586:0.01752,35418:0.01209)0.987.155:0.01373,(241532:0.02139,((255567:0.00626,574719:0.01005)0.987.156:0.01381,((70825:0.01655,186116:0.00832)0.999.445:0.01649,21302:0.00016)0.804.24:0.01933)0.967.78:0.01)0.218.6:0.00821)1.000.1497:0.04242)0.955.73:0.01454,((30366:0.01529,29244:0.00805)1.000.1498:0.05922,112991:0.0961)0.976.86:0.03083)1.000.1499:0.04885)0.213.5:0.00604)0.211.4:0.00979,(232619:0.06519,(274068:0.00785,228572:0.00458)1.000.1500:0.05045)0.776.25:0.02014)0.795.21:0.00919)0.497.9:0.01)0.999.446:0.03143,(((245356:0.03454,(320742:0.0171,(225059:0.0153,538684:0.02013)0.993.133:0.01569)0.877.42:0.01137)1.000.1501:0.05057,((2554:0.0045,327821:0.0199)1.000.1502:0.089,(55727:0.03146,(210280:0.00871,533454:0.01259)0.835.26:0.01184)0.999.447:0.04554)0.068.3:0.01552)0.900.46:0.01762,(((321867:0.01141,239523:0.02326)0.965.82:0.00683,250647:0.00727)0.927.64:0.09061,((((140407:0.01417,(239831:0.00206,240677:0.00719)0.987.157:0.0148)1.000.1503:0.04409,((((((273018:0.02679,578107:0.01999)1.000.1504:0.03382,(47356:0.01541,345271:0.02946)0.982.119:0.01705)0.986.113:0.0145,((557173:0.04933,(((146138:0.01539,222507:0.01021)0.920.49:0.00564,(241640:0.01015,(242277:0.00588,(249252:0.02334,317901:0.00449)0.368.9:0.00694)0.108.3:0.00525)0.621.9:0.00232)0.963.82:0.00705,(((26459:0.01729,(((332655:0.01229,189508:0.01317)0.841.30:0.00194,(511346:0.00555,(219534:0.01826,((186736:0.00335,96745:0.01036)0.000.65:0.00232,356235:0.01168)0.965.83:0.00502)0.664.5:0.00555)0.673.4:0.00091)0.808.21:0.00125,179407:0.01861)0.444.9:0.00275)0.899.53:0.00347,256813:0.0138)0.806.22:0.00301,582864:0.01396)0.971.108:0.00788)0.951.64:0.00762)0.909.50:0.00831,(((225192:0.01574,161257:0.03504)0.911.49:0.0062,(204934:0.0061,161479:0.00758)0.771.19:0.0044)0.980.92:0.01011,547060:0.00315)1.000.1505:0.03844)0.755.27:0.00632)0.917.55:0.01367,(327512:0.02043,511866:0.02061)0.812.30:0.01213)0.999.448:0.02944,(((203293:0.00867,177869:0.06534)0.973.88:0.01073,((81845:0.00792,314149:0.00706)0.731.21:0.00075,((82069:0.00015,68865:0.02356)1.000.1506:0.02368,((82360:0.0041,(356980:0.01058,525495:0.00554)0.857.35:0.00153)0.908.49:0.00708,(((((564144:0.00786,246243:0.00455)0.735.20:0.0009,225377:0.02438)0.940.70:0.00302,309523:0.00237)0.730.17:0.00408,580493:0.0065)0.962.80:0.0039,325652:0.01192)0.995.191:0.00872)0.814.17:0.00016)0.271.4:0.0016)0.971.109:0.0068)0.983.105:0.01419,(150281:0.0171,(((555333:0.00499,233334:0.01586)0.444.10:0.00188,((82223:0.00359,574458:0.01908)0.739.22:0.00637,(110665:0.02094,(561510:0.00252,(509873:0.00738,368451:0.00469)0.863.40:0.0047)0.794.24:0.00126)0.972.95:0.0053)0.993.134:0.01116)0.690.8:0.00093,(((((558682:0.03364,570934:0.00156)0.770.20:0.00328,((306684:0.00386,(540217:0.02032,525670:0.00561)0.969.86:0.01057)0.887.40:0.00542,(219059:0.01428,353659:0.00209)0.986.114:0.00914)0.830.26:0.00321)0.395.8:0.00381,575954:0.01931)0.634.6:0.00441,((282480:0.00899,328449:0.02154)0.966.81:0.00544,((500937:0.00827,(211909:0.00124,498304:0.01592)0.852.52:0.00956)0.780.16:0.00161,(154846:0.00316,549981:0.01518)0.915.50:0.00346)0.951.65:0.00463)0.982.120:0.00668)0.952.74:0.00584,((143769:0.01231,((510619:0.01155,122683:0.02071)0.626.7:0.00191,((160173:0.0087,181908:0.00842)0.927.65:0.0033,528336:0.0149)0.700.14:0.0017)0.914.42:0.00408)0.161.3:0.00276,(560204:0.00735,360124:0.01448)0.963.83:0.00653)0.891.53:0.00015)0.950.89:0.0047)0.981.95:0.01023)0.993.135:0.01616)0.845.24:0.0104)0.322.4:0.00633,(((329517:0.00176,(((567683:0.00184,347661:0.00763)0.997.220:0.01146,((246781:0.00378,307581:0.0089)0.689.7:0.00188,330540:0.00497)0.845.25:0.00212)0.952.75:0.00551,572138:0.01131)0.863.41:0.0067)1.000.1507:0.0348,2508:0.12149)0.625.8:0.01717,((((((72978:0.04244,(160651:0.01248,256584:0.02574)0.675.5:0.00901)0.904.54:0.00677,((270708:0.02139,(216180:0.00995,(193234:0.00978,557323:0.01088)0.993.136:0.0109)0.999.449:0.01697)0.940.71:0.00543,((107742:0.025,(363703:0.00458,534551:0.02521)0.995.192:0.0153)1.000.1508:0.03896,(209422:0.02356,(395698:0.0108,((240659:0.01354,160010:0.01107)0.993.137:0.01255,(218287:0.00349,161009:0.01187)0.758.22:0.00133)0.867.44:0.00337)0.995.193:0.01602)0.991.139:0.01825)0.865.43:0.00675)0.751.11:0.00269)0.991.140:0.01267,135386:0.03197)0.744.19:0.00245,((((154287:0.01975,((590811:0.01671,(((578831:0.02682,277886:0.01968)0.873.45:0.00383,((232992:0.02299,156950:0.00966)0.951.66:0.00522,257780:0.00013)0.895.52:0.01631)0.941.77:0.00553,(247882:0.03441,350591:0.00825)0.949.74:0.00814)0.919.57:0.00546)0.213.6:0.00415,568143:0.0329)0.590.6:0.00263)0.999.450:0.01947,(((((((241705:0.01171,344260:0.00608)0.992.151:0.01152,(248072:0.00553,583654:0.00174)0.044.3:0.00951)0.974.89:0.00734,((((244018:0.01731,242455:0.01939)0.302.7:0.00612,(268617:0.01045,109907:0.0041)0.994.164:0.00899)0.759.14:0.0013,541963:0.01586)0.825.20:0.00324,(((((245652:0.00915,238692:0.00567)0.957.103:0.00677,(222635:0.0129,211598:0.02596)0.667.5:0.00618)0.866.34:0.00459,551319:0.00723)0.257.5:0.0031,((321965:0.00239,153542:0.02692)0.905.58:0.00237,((213892:0.00821,204470:0.01101)0.994.165:0.00824,(247873:0.01814,((((328627:0.01451,(396799:0.00668,(((521806:0.01238,551895:0.01423)0.816.24:0.00188,((362733:0.0104,563615:0.0102)0.472:0.00341,47339:0.02731)0.855.32:0.00016)0.870.48:0.00255,363617:0.01731)0.906.57:0.00303)0.789.24:0.00354)0.886.38:0.00016,510591:0.00971)0.757.31:0.004,250672:0.02448)0.922.58:0.00169,(255322:0.01955,358280:0.01952)0.889.41:0.01102)0.993.138:0.00742)0.049:0.00069)0.750.20:0.0008)0.851.37:0.00197)0.947.68:0.00492,317096:0.02136)0.547.8:0.00333)0.872.55:0.00311)0.950.90:0.00427,(((145905:0.01742,(556825:0.01669,(280545:0.01397,545016:0.00933)0.859.45:0.00398)0.993.139:0.0098)0.935.59:0.00633,(286368:0.01092,113866:0.01965)0.735.21:0.00228)0.739.23:0.00111,(305064:0.01433,(213634:0.01432,541215:0.013)0.962.81:0.00834)0.989.121:0.00807)0.761.29:0.00092)0.861.42:0.0021,((560165:0.01244,(183639:0.01783,510807:0.02383)0.971.110:0.01111)0.977.98:0.00811,560820:0.00483)0.877.43:0.00276)0.923.54:0.00398,(((236776:0.00969,105552:0.01142)0.760.27:0.00359,(558805:0.00416,567628:0.01908)0.903.53:0.0028)0.917.56:0.00467,((208800:0.02653,((592173:0.01136,509090:0.00562)0.241:0.00283,565399:0.01324)0.856.32:0.00459)0.930.53:0.00468,(551005:0.00028,510666:0.00106)0.866.35:0.00401)0.969.87:0.00628)0.999.451:0.01387)0.948.74:0.006,((((((151174:0.02419,32581:0.02796)0.993.140:0.01148,(346596:0.01099,138295:0.01172)0.830.27:0.00455)0.937.65:0.00847,248199:0.01852)0.942.71:0.00637,554086:0.00994)0.993.141:0.01091,(583688:0.00316,((540510:0.0082,578565:0.01192)0.944.62:0.00478,(((((367060:0.0042,((220305:0.01026,204795:0.01468)0.763.22:0.00506,(553480:0.00875,137753:0.00165)0.873.46:0.00625)0.889.42:0.00397)0.746.30:0.00066,(((143991:0.01164,(534493:0.01473,210176:0.00651)0.409.3:0.00252)0.728.17:0.00053,(123738:0.00298,(594040:0.01503,(154761:0.01498,532743:0.00346)0.998.283:0.01626)0.941.78:0.00793)0.986.115:0.00982)0.814.18:0.00334,(81764:0.01864,152427:0.02408)0.966.82:0.00855)0.745.21:0.0012)0.999.452:0.00798,(114276:0.00508,167066:0.00547)0.602.5:0.00015)0.475.6:0.00014,(564288:0.0145,311656:0.00363)0.959.76:0.01574)0.395.9:0.00078,(141786:0.00225,136598:0.0214)0.870.49:0.00249)0.889.43:0.00218)0.744.20:0.00096)0.907.47:0.00381)1.000.1509:0.01953,(((325365:0.01551,(254256:0.02656,160560:0.0174)0.653.3:0.00282)0.984.93:0.00769,(((555876:0.00932,(250514:0.01358,558503:0.00973)0.744.21:0.00495)0.928.57:0.00669,(356069:0.02806,113307:0.00831)0.994.166:0.00979)0.879.49:0.0029,(((440021:0.02226,((((55504:0.00528,211087:0.0109)0.963.84:0.00659,(142330:0.00734,251675:0.00093)0.876.45:0.00158)0.861.43:0.00157,540458:0.00248)0.763.23:0.00074,202525:0.02973)0.987.158:0.0079)0.434.7:0.00413,((357994:0.00347,535998:0.01352)0.974.90:0.00697,252655:0.00486)0.478.2:0.00462)0.943.80:0.0055,(((42740:0.01678,((152361:0.00777,((78422:0.01702,(548715:0.01272,93798:0.01273)0.785.23:0.00433)0.999.453:0.0169,(553534:0.01529,((250223:0.00565,351981:0.00954)0.982.121:0.00808,(144976:0.01211,(140614:0.00288,246952:0.01365)0.947.69:0.0045)0.839.34:0.00015)0.932.58:0.00804)0.920.50:0.00542)0.905.59:0.00432)0.913.62:0.00407,(248363:0.00967,254621:0.02388)0.998.284:0.01319)0.947.70:0.00766)0.799.17:0.00235,(535708:0.02016,136597:0.01182)0.778.30:0.00569)0.947.71:0.00646,(250277:0.01769,(540472:0.02417,(215823:0.00621,(240838:0.02042,242284:0.00611)0.956.74:0.00765)0.901.42:0.00454)0.110.3:0.0017)0.914.43:0.00409)0.786.23:0.00352)0.911.50:0.00315)0.970.84:0.00688)0.944.63:0.00605,(235802:0.00562,(544834:0.02137,(250093:0.02506,(445860:0.01247,221611:0.01507)0.671.11:0.00256)0.505.7:0.00015)0.119:0.00182)1.000.1510:0.01211)0.868.48:0.00294)0.670.3:0.00414)0.605.9:0.00195)0.982.122:0.01233,((((((168695:0.00715,233375:0.00563)0.744.22:0.00078,550271:0.0017)0.877.44:0.00316,(((252642:0.01293,58633:0.01107)0.985.98:0.00872,((58620:0.00972,218242:0.0063)0.946.59:0.00484,((58621:0.00083,584085:0.01209)0.992.152:0.00952,134254:0.02716)0.131.7:0.00337)0.880.36:0.00504)0.554.7:0.00568,((164755:0.01092,141868:0.01042)0.952.76:0.00443,((82843:0.01293,99395:0.01082)0.850.28:0.00307,182778:0.00866)0.977.99:0.00638)0.984.94:0.00917)0.337.3:0.00326)0.998.285:0.02481,(558160:0.025,557824:0.03751)1.000.1511:0.02854)0.511.6:0.00697,((209940:0.01388,(212054:0.0092,(271833:0.011,549222:0.00612)0.504.5:0.00435)0.961.74:0.0109)1.000.1512:0.02596,(542274:0.01923,(343941:0.01703,238970:0.02261)1.000.1513:0.02147)0.984.95:0.01503)0.926.67:0.00945)0.540.3:0.00847,(((172843:0.0068,(152017:0.00994,(222632:0.00534,156035:0.0074)0.871.52:0.00269)0.998.286:0.01002)0.287.4:0.00393,(222536:0.00468,341142:0.01019)0.883.43:0.00376)0.974.91:0.00672,(154794:0.00404,(((221734:0.00605,(572326:0.01174,562783:0.00652)0.725.18:0.00116)0.992.153:0.00588,509171:0.00269)0.735.22:0.00076,155441:0.00832)0.984.96:0.00507)0.887.41:0.00237)1.000.1514:0.02588)0.948.75:0.01825)0.678.9:0.00109,(103987:0.01051,(512264:0.01263,517065:0.00571)0.907.48:0.00941)1.000.1515:0.01915)0.890.52:0.00815)0.867.45:0.00653,((((229576:0.01317,96454:0.01124)1.000.1516:0.02061,(58374:0.02833,42289:0.02114)0.674.9:0.00875)0.878.52:0.01028,(((155928:0.00988,(203393:0.00522,29854:0.00191)0.263.7:0.00016)1.000.1517:0.00999,130882:0.0042)0.961.75:0.01945,(((((180406:0.00251,127471:0.0188)0.981.96:0.0154,(70346:0.01329,576145:0.01013)0.939.68:0.00987)0.999.454:0.01717,(548506:0.00871,(119875:0.0052,52719:0.00766)0.952.77:0.00699)0.894.45:0.00727)0.981.97:0.00784,(142029:0.00663,104742:0.00622)0.743.29:0.00519)0.583.10:0.00593,((61843:0.0092,188772:0.02116)0.741.21:0.00178,((((358964:0.00352,206279:0.00097)0.981.98:0.00234,241209:0.01015)0.897.47:0.0027,129332:0.01724)0.861.44:0.00547,(190531:0.0119,(153532:0.0148,534228:0.01591)0.971.111:0.009)0.775.17:0.00595)0.921.56:0.00577)0.989.122:0.00887)0.993.142:0.01088)0.845.26:0.00628)0.755.28:0.00727,333860:0.02724)0.997.221:0.01471)0.932.59:0.0069,((97049:0.01144,(152842:0.01329,(264742:0.02136,156149:0.00476)0.429.3:0.00862)0.944.64:0.00915)0.997.222:0.01679,(99400:0.0059,((((558023:0.0112,557002:0.02275)0.674.10:0.00639,((241123:0.00479,250900:0.01456)0.991.141:0.01083,354739:0.0226)0.979.108:0.00968)1.000.1518:0.01773,534616:0.00672)0.234.2:0.00256,(142837:0.02151,228372:0.02041)0.862.46:0.00596)0.939.69:0.00722)1.000.1519:0.03226)0.765.24:0.00227)0.697.8:0.01144)0.864.44:0.00644)0.983.106:0.01452)0.950.91:0.01814,(224117:0.02058,(203786:0.02594,258047:0.02196)0.979.109:0.02844)1.000.1520:0.06504)0.904.55:0.02112,((320104:0.05643,234598:0.12106)0.086.3:0.01227,(348170:0.01189,(246967:0.01316,576653:0.00345)1.000.1521:0.0212)1.000.1522:0.05149)0.942.72:0.01445)0.940.72:0.01449)0.925.48:0.02193)0.987.159:0.01972)0.388.7:0.01145)0.994.167:0.02554)0.361.5:0.00756)0.773.25:0.01391)0.990.124:0.0229)1.000.1523:0.0579)0.768.26:0.01677,((((2649:0.07001,((268113:0.01179,(2648:0.00734,(2650:0.0064,536800:0.01374)0.952.78:0.00412)0.986.116:0.00946)0.998.287:0.02183,(153143:0.01191,(570489:0.01052,255558:0.02752)0.952.79:0.01219)0.468.6:0.0078)0.500.10:0.01529)0.896.44:0.01384,((((249217:0.00479,136516:0.00788)1.000.1524:0.04942,(592121:0.00728,586741:0.01441)0.783.22:0.01323)0.956.75:0.01024,((((((324940:0.00956,(((200167:0.01915,(542858:0.03067,314441:0.00366)0.761.30:0.00093)0.831.16:0.00355,552214:0.01897)0.876.46:0.00416,(((161235:0.0139,200395:0.0154)0.953.72:0.0061,(534411:0.01888,(2644:0.01832,(203228:0.01121,551092:0.00275)0.778.31:0.00393)0.905.60:0.00265)0.731.22:0.00552)0.633.4:0.00593,(218378:0.01454,218096:0.00973)1.000.1525:0.01988)0.267.7:0.00219)0.306.7:0.0037)1.000.1526:0.01656,259356:0.00975)0.988.140:0.00773,(343773:0.0148,317840:0.01545)0.391.8:0.00072)0.949.75:0.0035,202685:0.01712)0.029:0.00078,224603:0.02251)0.732.18:0.0333,(532169:0.02702,(202481:0.0089,(201362:0.01342,248702:0.00572)0.994.168:0.01238)0.993.143:0.01945)0.972.96:0.01259)1.000.1527:0.03541)0.891.54:0.00906,((562262:0.02921,(251756:0.00815,(60166:0.00732,577399:0.00535)0.289.4:0.00235)0.009.6:0.0046)1.000.1528:0.03497,(172643:0.01921,164768:0.01427)1.000.1529:0.02846)0.910.51:0.01287)0.306.8:0.00495)0.864.45:0.00583,((161044:0.01871,(574540:0.02507,571736:0.0161)0.658.8:0.00577)1.000.1530:0.04606,(351788:0.0257,(257002:0.01651,437088:0.03055)0.998.288:0.01558)1.000.1531:0.02758)0.971.112:0.02235)0.945.63:0.01505,(60618:0.08566,((279426:0.00486,536032:0.0473)0.930.54:0.00818,(2637:0.00427,319658:0.02248)0.665.9:0.01179)0.926.68:0.01981)0.646.10:0.02265)0.998.289:0.0293)1.000.1532:0.03832,(((566742:0.02416,164804:0.01824)0.991.142:0.02866,(146912:0.00493,(158864:0.01477,2647:0.02225)0.930.55:0.01231)1.000.1533:0.17199)0.952.80:0.04708,((((540017:0.05251,(160290:0.02978,101281:0.01915)0.986.117:0.01731)0.892.41:0.0126,((549181:0.00098,160670:0.00624)0.877.45:0.04718,25315:0.08367)0.876.47:0.01707)0.995.194:0.02283,(((217425:0.02305,442850:0.01482)1.000.1534:0.03208,(144037:0.02081,515456:0.02925)0.113.6:0.00666)0.657.4:0.01304,((219958:0.01132,(202747:0.02053,((201304:0.01347,144843:0.01228)0.935.60:0.00542,142737:0.01255)0.949.76:0.00655)0.947.72:0.00893)1.000.1535:0.04687,239397:0.05566)0.982.123:0.02206)0.536.6:0.01061)0.942.73:0.02033,49950:0.0565)0.867.46:0.03097)0.997.223:0.03911)0.958.87:0.017)0.046.2:0.00764)0.342.7:0.01148,((38651:0.02101,(203568:0.00868,51942:0.02682)0.998.290:0.0247)1.000.1536:0.0366,(347501:0.04063,(144442:0.02241,((347487:0.01237,(545587:0.00476,(((2674:0.01109,39385:0.01103)0.923.55:0.00363,568732:0.02785)0.961.76:0.0043,(343158:0.00864,112750:0.01613)0.985.99:0.00649)0.865.44:0.00281)0.443.8:0.00399)1.000.1537:0.01993,((300257:0.00645,(2700:0.01542,(191371:0.00851,(18641:0.00765,356467:0.0044)0.968.81:0.00887)0.801.19:0.00417)0.437.5:0.00439)0.721.11:0.00172,(143787:0.00466,(((2692:0.00015,540816:0.00909)1.000.1538:0.01538,(192472:0.01534,((2690:0.0057,143100:0.00569)0.994.169:0.00861,538426:0.00749)0.918.64:0.0026)0.921.57:0.0146)0.955.74:0.00644,(345873:0.01377,226020:0.01307)0.800.24:0.00168)0.939.70:0.00706)0.966.83:0.00624)0.909.51:0.00758)0.245.6:0.00657)1.000.1539:0.03748)0.984.97:0.02139)1.000.1540:0.0878)0.983.107:0.0214)0.903.54:0.01534)0.896.45:0.01207,(((565473:0.01289,(164927:0.01309,53652:0.01102)0.921.58:0.00912)1.000.1541:0.05191,(((558716:0.0112,(199479:0.02099,(523998:0.00643,(543260:0.00865,31738:0.00483)0.221.5:0.00087)0.987.160:0.00893)0.947.73:0.00751)0.792.19:0.00265,(581301:0.00774,171914:0.00917)0.863.42:0.00411)1.000.1542:0.04234,(158348:0.04475,(101254:0.01163,(378810:0.00222,575437:0.00504)0.946.60:0.00637)0.987.161:0.01467)0.072.2:0.00929)0.957.104:0.01271)0.997.224:0.02331,(((346244:0.02162,266775:0.02317)1.000.1543:0.03976,(((583042:0.02725,149924:0.01117)1.000.1544:0.07389,(273907:0.06631,(240577:0.04347,250805:0.03167)0.611.7:0.00974)0.963.85:0.01659)0.152.5:0.00866,(529594:0.01005,(513424:0.02736,(566342:0.03415,162879:0.01427)0.853.31:0.00599)0.650.4:0.00601)0.997.225:0.02039)0.915.51:0.01084)0.986.118:0.02875,((554139:0.08123,(((108876:0.0691,(((576577:0.02031,(337870:0.00467,347223:0.01759)0.700.15:0.01306)1.000.1545:0.02994,(((272826:0.01462,542779:0.01817)0.982.124:0.01181,(((121667:0.0151,((118980:0.00406,159949:0.00875)0.995.195:0.00993,((343475:0.0025,118018:0.00236)0.945.64:0.00738,(228552:0.03242,337085:0.00621)1.000.1546:0.02562)0.799.18:0.00814)0.890.53:0.00557)0.672.4:0.00772,104828:0.02134)1.000.1547:0.03386,((550423:0.0033,539047:0.00014)0.954.78:0.04396,(105774:0.01557,159830:0.01344)0.991.143:0.01659)0.768.27:0.00807)0.448.9:0.00637)0.960.94:0.0101,((((234454:0.00975,583805:0.0111)0.710.12:0.00173,121027:0.01242)0.999.455:0.02056,((589064:0.01555,328031:0.01796)0.994.170:0.01341,(3654:0.01419,(((570167:0.01394,68979:0.00754)0.994.171:0.00823,(107060:0.02385,(566226:0.0167,(592334:0.00491,542423:0.01956)0.995.196:0.01095)0.911.51:0.00414)0.893.37:0.00351)0.867.47:0.00158,322984:0.01001)0.960.95:0.00561)0.854.33:0.00339)0.994.172:0.01892)1.000.1548:0.02337,(((127240:0.00028,332828:0.00238)0.736.20:0.00875,101444:0.0187)1.000.1549:0.01875,591448:0.00579)1.000.1550:0.0986)0.583.11:0.0062)0.996.207:0.02035)0.907.49:0.01299,(91672:0.02495,568134:0.02681)0.992.154:0.02022)0.940.73:0.01747)0.976.87:0.02779,((276241:0.01858,255266:0.01308)1.000.1551:0.0469,((((79412:0.02844,(159771:0.01284,514613:0.01162)0.863.43:0.00561)1.000.1552:0.04372,(137211:0.01917,161026:0.03233)1.000.1553:0.04468)0.976.88:0.01731,(160411:0.00826,127630:0.00294)1.000.1554:0.04268)0.089:0.00819,(515331:0.01845,(108893:0.01416,579754:0.01458)0.986.119:0.01394)0.988.141:0.01764)1.000.1555:0.03847)0.941.79:0.02401)0.937.66:0.01763,((553223:0.00907,581203:0.02004)1.000.1556:0.08628,(102419:0.00627,590482:0.01989)1.000.1557:0.07013)0.799.19:0.01872)0.982.125:0.02292)0.998.291:0.03455,(551134:0.01626,((562251:0.02607,(26077:0.00707,532343:0.01243)0.990.125:0.01189)0.988.142:0.0144,(261859:0.01867,246605:0.01831)0.937.67:0.00928)0.708.11:0.01021)1.000.1558:0.05031)0.076.2:0.00948)0.914.44:0.01281)0.991.144:0.01765)0.987.162:0.02124,(((((166199:0.05646,(214125:0.04207,((195968:0.02802,114772:0.02047)0.998.292:0.02686,(573829:0.01591,(238702:0.03371,542899:0.01569)0.652.9:0.00945)0.998.293:0.02809)0.993.144:0.03062)0.966.84:0.02085)0.962.82:0.01595,(571567:0.0522,((166827:0.01993,(1393:0.03176,(129123:0.00726,(290466:0.01448,(67904:0.02005,(247874:0.00421,(((30085:0.00973,(60559:0.0043,(25164:0.01933,(63588:0.01526,(((71171:0.00463,106962:0.00871)0.732.19:0.00088,27993:0.01601)0.959.77:0.00396,111711:0.0078)0.907.50:0.00263)0.985.100:0.0082)0.731.23:0.00123)0.988.143:0.00864)0.942.74:0.00475,((((165994:0.00717,1324:0.01429)0.231.3:0.00029,171944:0.0112)0.831.17:0.0025,1345:0.00393)0.875.42:0.00171,313176:0.0115)0.938.72:0.00669)0.827.21:0.00186,17962:0.01063)0.361.6:0.00176)0.319.5:0.00388)0.668.6:0.00352)0.921.59:0.00569)0.776.26:0.00895)0.990.126:0.01961)1.000.1559:0.0335,(540517:0.03395,20463:0.039)0.786.24:0.01254)0.681.14:0.00883)0.996.208:0.02357)0.997.226:0.03147,((568744:0.04789,((163551:0.02451,593108:0.00706)1.000.1560:0.05812,((139442:0.04014,(252255:0.03523,((((146805:0.00603,139395:0.00425)0.971.113:0.01357,(((167964:0.01297,332198:0.03484)0.986.120:0.01317,138099:0.00764)0.837.21:0.00359,138901:0.02077)1.000.1561:0.03495)0.999.456:0.02846,546802:0.02188)0.566.10:0.01211,307654:0.02815)0.876.48:0.0143)0.929.67:0.01374)0.876.49:0.01241,((139285:0.0016,250755:0.02013)1.000.1562:0.03252,((285250:0.00728,591329:0.0142)1.000.1563:0.04395,(((138898:0.00127,235373:0.00408)1.000.1564:0.03579,((147332:0.0095,(224771:0.01382,219134:0.0223)0.982.126:0.00997)0.864.46:0.00369,((115072:0.00016,298891:0.00178)0.969.88:0.00754,286584:0.01433)0.783.23:0.00416)0.997.227:0.02103)0.902.38:0.00813,((171712:0.01133,(1408:0.02007,(380491:0.01443,1410:0.01772)0.564.9:0.00584)0.975.101:0.01767)1.000.1565:0.02905,(((((279558:0.01125,(97958:0.00377,(95404:0.00266,243604:0.01488)0.986.121:0.00882)1.000.1566:0.02315)0.995.197:0.01756,((139362:0.00434,(250725:0.0305,242608:0.02295)0.374.6:0.01339)0.972.97:0.01405,(((159842:0.03156,158194:0.01833)0.310.6:0.01667,((160732:0.01978,152911:0.01625)0.509.5:0.00555,(172829:0.03563,(160687:0.01849,(461798:0.01437,168505:0.01078)0.739.24:0.00469)0.654.6:0.00616)1.000.1567:0.02407)0.791.27:0.00866)0.999.457:0.02332,(1415:0.00015,171820:0.02844)0.601.3:0.01646)0.987.163:0.01424)0.937.68:0.00756)0.900.47:0.01145,(260458:0.0159,(389682:0.00871,405062:0.01205)0.974.92:0.01117)0.906.58:0.00609)0.991.145:0.01602,(((170011:0.01918,(((148663:0.00011,169628:0.0003)0.839.35:0.02195,(365101:0.01482,263628:0.04005)0.466.10:0.00331)0.983.108:0.01111,(114526:0.01461,(282134:0.03842,(138316:0.02961,366545:0.01136)0.971.114:0.01852)0.942.75:0.01044)0.939.71:0.0091)0.397.11:0.00547)0.508.8:0.00621,344057:0.01568)0.959.78:0.00731,((((518775:0.00086,249433:0.00532)0.996.209:0.01126,(146327:0.00622,369026:0.01225)0.979.110:0.01018)0.410.5:0.00825,453803:0.03534)0.885.46:0.00715,(((276339:0.00794,155693:0.01535)0.848.34:0.00295,(565195:0.01156,(486153:0.01879,(139376:0.00135,517000:0.0145)0.944.65:0.01444)0.980.93:0.01156)0.929.68:0.00845)0.965.84:0.00777,556641:0.01038)0.584.5:0.00772)0.873.47:0.00695)0.890.54:0.00469)0.571.8:0.00489,264010:0.02694)0.939.72:0.00698)0.799.20:0.00286)0.958.88:0.01225)0.984.98:0.01863)0.909.52:0.01016)0.830.28:0.01167)0.451.4:0.00861)0.999.458:0.05216,((562787:0.01252,541456:0.00729)1.000.1568:0.03453,((159252:0.01212,556010:0.00367)1.000.1569:0.06951,((266995:0.00685,(86556:0.00407,274011:0.01583)0.863.44:0.00276)1.000.1570:0.02954,(136712:0.00014,328645:0.00911)0.993.145:0.0144)0.992.155:0.0203)0.843.30:0.01611)1.000.1571:0.03713)0.985.101:0.02837)1.000.1572:0.08092,(350334:0.08455,(((((259192:0.01858,(212819:0.02271,((240890:0.01106,237047:0.01872)1.000.1573:0.02772,(100478:0.01496,(82251:0.00747,82212:0.004)0.329.5:0.00359)1.000.1574:0.02358)0.951.67:0.00794)0.974.93:0.01096)0.962.83:0.00786,(219973:0.00447,113407:0.00543)0.995.198:0.0116)0.855.33:0.00513,((((114146:0.03183,((560811:0.00608,112960:0.01716)1.000.1575:0.0223,(250371:0.02792,(112876:0.0111,(516178:0.01271,(516051:0.00524,560752:0.00785)0.991.146:0.01333)0.985.102:0.01562)1.000.1576:0.02507)0.980.94:0.01172)0.622.6:0.00495)0.440.5:0.00699,(((113293:0.0157,(113699:0.01643,((250985:0.01649,(249948:0.02406,(254514:0.00158,(240654:0.01661,247730:0.00942)0.963.86:0.00979)0.960.96:0.0065)0.893.38:0.00379)0.425.5:0.00376,65681:0.01046)0.774.13:0.00717)0.083.5:0.00649)1.000.1577:0.02674,(89632:0.04631,(((518957:0.01678,(136707:0.01031,(325238:0.00384,526802:0.01103)0.990.127:0.01001)0.936.56:0.00645)0.977.100:0.01067,(467362:0.02193,(511514:0.00201,111580:0.02253)0.803.21:0.009)0.723.13:0.00182)0.523.8:0.00371,(246196:0.02762,(112986:0.01065,227543:0.02284)0.965.85:0.0116)0.924.52:0.00543)0.914.45:0.00641)0.986.122:0.01623)0.456.7:0.01246,(573327:0.00849,(105839:0.0072,156895:0.00942)0.985.103:0.01259)0.994.173:0.01705)0.952.81:0.01469)0.920.51:0.00806,(((((362018:0.04777,(207957:0.02463,(572428:0.00845,234041:0.01654)0.887.42:0.00589)0.998.294:0.02475)0.971.115:0.01348,(((278836:0.03724,(243247:0.01245,340252:0.01566)0.891.55:0.00544)0.957.105:0.00891,(81623:0.03143,(142105:0.01457,76374:0.02141)0.357.7:0.00768)0.980.95:0.01734)1.000.1578:0.0239,(590202:0.03738,(325626:0.06197,((((593891:0.04307,152039:0.01131)0.759.15:0.00176,567077:0.00973)0.996.210:0.00999,316787:0.00619)0.935.61:0.00691,(591271:0.02538,81666:0.00599)0.991.147:0.00989)0.940.74:0.01348)0.985.104:0.01735)0.960.97:0.01239)0.845.27:0.00946)0.669.7:0.00393,(((265561:0.01758,233155:0.01289)0.969.89:0.00967,((560457:0.01576,(113777:0.01778,565312:0.00733)0.968.82:0.01273)0.987.164:0.01677,(329882:0.04725,588566:0.03818)0.839.36:0.00701)0.953.73:0.01035)0.936.57:0.00862,211950:0.02879)0.996.211:0.01654)0.537.8:0.00464,112987:0.04797)0.788.19:0.00274,(((114406:0.01985,366191:0.02162)0.999.459:0.02268,((557467:0.00506,((329566:0.00778,((105150:0.02112,((114310:0.01027,111803:0.01143)0.697.9:0.00447,263767:0.00155)0.787.23:0.00168)0.886.39:0.00393,112916:0.0074)0.712.8:0.01006)0.942.76:0.01043,(551366:0.01628,360530:0.01284)0.356.4:0.00017)0.886.40:0.00373)0.997.228:0.01409,((((172910:0.00157,172222:0.00078)0.953.74:0.00525,384470:0.00443)0.393.5:0.0036,((168856:0.00857,168032:0.01935)0.886.41:0.00441,(167878:0.01177,167426:0.00626)0.940.75:0.00568)0.711.9:0.0035)1.000.1579:0.03165,(379634:0.02702,325203:0.01678)0.913.63:0.0112)0.991.148:0.01613)0.687.10:0.00903)0.076.3:0.00186,(289924:0.02898,225306:0.02609)0.991.149:0.01375)0.604.8:0.00675)0.953.75:0.00669)0.792.20:0.01011,(384373:0.01416,(136683:0.02235,114101:0.01551)0.690.9:0.00651)0.998.295:0.02043)0.881.45:0.00795)1.000.1580:0.05445,((166488:0.04165,((243927:0.01296,308062:0.02195)0.963.87:0.00981,(254529:0.02662,(325972:0.00407,104333:0.02118)0.998.296:0.02075)0.935.62:0.01139)0.936.58:0.00912)0.161.4:0.00682,((((253524:0.00937,(75389:0.01305,217365:0.008)0.923.56:0.00787)0.761.31:0.00387,((246136:0.01377,244604:0.01268)0.958.89:0.00716,257417:0.01574)0.894.46:0.0035)0.585.8:0.00174,((246796:0.01227,(219707:0.01742,(249854:0.00579,240353:0.00846)0.959.79:0.0044)0.803.22:0.01566)0.952.82:0.00561,255641:0.02575)0.913.64:0.00391)0.998.297:0.02079,((79189:0.02143,((336425:0.00908,(580291:0.01493,(219517:0.01545,(((((245557:0.00564,360943:0.00559)0.871.53:0.00431,(172879:0.01382,138605:0.01825)0.990.128:0.01184)0.352.6:0.00632,79951:0.04543)0.979.111:0.00962,((((350957:0.01296,255407:0.02117)0.847.30:0.00213,524341:0.01516)0.837.22:0.00181,(586438:0.02058,210111:0.03554)1.000.1581:0.04557)0.810.22:0.0046,209613:0.01407)0.940.76:0.00503)0.986.123:0.00951,(314599:0.03145,(114629:0.00876,(558730:0.01585,(555339:0.00757,509573:0.02239)0.980.96:0.01087)1.000.1582:0.01559)0.375.5:0.01708)0.988.144:0.01358)0.879.50:0.0046)0.580.7:0.00317)0.933.62:0.00517)0.920.52:0.00611,((580796:0.01102,((252798:0.02495,((317381:0.0067,337662:0.01565)0.935.63:0.00439,(339480:0.00792,249280:0.00999)0.992.156:0.00994)0.714.10:0.00531)0.790.18:0.00648,75259:0.01691)0.938.73:0.00497)0.977.101:0.00879,90194:0.00613)0.909.53:0.0042)0.917.57:0.00423)0.348.4:0.00253,(279650:0.02346,(75274:0.0147,355297:0.02206)0.911.52:0.00777)0.922.59:0.00704)1.000.1583:0.0289)0.722.14:0.01354)0.477.5:0.02009)0.988.145:0.02695,(((136926:0.01043,(99079:0.01744,(269262:0.03469,373545:0.01654)0.737.12:0.00387)0.961.77:0.00771)1.000.1584:0.03911,((241659:0.01349,554991:0.01579)1.000.1585:0.03636,((273255:0.00562,244505:0.01932)0.981.99:0.01189,((((226129:0.02631,(588644:0.00993,(582373:0.01931,321448:0.00617)0.919.58:0.00499)0.923.57:0.00449)0.981.100:0.00892,(565523:0.01973,(259956:0.00258,359471:0.00277)0.974.94:0.00833)0.914.46:0.00671)0.334.3:0.00227,113948:0.01737)0.995.199:0.01309,(268219:0.02147,(11593:0.02053,277772:0.01658)0.881.46:0.00656)0.310.7:0.00253)0.939.73:0.00887)1.000.1586:0.04537)0.828.21:0.01517)0.982.127:0.01988,(((317743:0.01055,247892:0.03955)0.962.84:0.01183,((356019:0.00165,(340123:0.00844,340322:0.01583)0.947.74:0.00551)1.000.1587:0.04518,((((586289:0.02867,(((((112749:0.00015,362150:0.0085)0.861.45:0.0227,511651:0.01391)0.885.47:0.00305,(341873:0.01148,510532:0.01558)0.987.165:0.00658)0.908.50:0.00469,(249289:0.01365,248455:0.01334)0.931.55:0.00673)0.986.124:0.01315,(((113201:0.01257,(308997:0.00529,332252:0.0049)0.733.18:0.0075)1.000.1588:0.0131,588995:0.00015)0.686.7:0.01653,(239710:0.02345,225607:0.01789)0.468.7:0.00986)0.996.212:0.0156)0.977.102:0.01267)0.869.49:0.0099,(579054:0.02842,312327:0.01939)0.486.3:0.00887)0.879.51:0.00621,((344838:0.02184,511297:0.01994)0.564.10:0.00532,((354981:0.01587,310862:0.01676)0.879.52:0.00612,(510548:0.00411,197816:0.01632)0.958.90:0.02383)0.583.12:0.00447)0.264.5:0.00256)0.809.17:0.00752,(113234:0.01026,(113916:0.00998,562350:0.01627)0.993.146:0.0131)0.979.112:0.01335)1.000.1589:0.02503)0.037.2:0.01025)0.999.460:0.03182,(((346977:0.0215,579069:0.01048)0.999.461:0.02232,((((240326:0.01508,322198:0.0117)0.990.129:0.0098,(208663:0.01433,(113023:0.00948,(569321:0.02149,270789:0.01674)0.040.4:0.00641)0.636.4:0.00709)1.000.1590:0.02153)0.885.48:0.00471,(((((182109:0.01218,(552061:0.0157,11278:0.02009)0.722.15:0.00355)0.999.462:0.01764,((237565:0.02241,304950:0.02017)0.544.7:0.00469,(277812:0.01815,542631:0.02849)0.996.213:0.01563)0.965.86:0.00957)0.646.11:0.0046,(160916:0.01896,589761:0.0143)0.936.59:0.00697)0.919.59:0.00567,(582235:0.0329,13726:0.02843)1.000.1591:0.02035)0.777.17:0.00164,((565653:0.00926,((60181:0.01077,581097:0.00537)0.915.52:0.00778,(557462:0.01447,257775:0.03094)0.766.22:0.00387)0.789.25:0.00364)1.000.1592:0.02592,((277923:0.01791,530814:0.02067)0.985.105:0.01158,((227014:0.00107,533069:0.01263)0.936.60:0.02442,((89477:0.01208,558725:0.011)0.992.157:0.01102,(276624:0.01029,(148154:0.02084,(279648:0.0078,147735:0.00986)0.869.50:0.00514)0.909.54:0.00435)0.942.77:0.00505)0.994.174:0.01202)0.985.106:0.01122)0.846.23:0.0024)0.950.92:0.00851)0.929.69:0.00724)0.661.9:0.00331,((342606:0.01001,(585141:0.01982,356469:0.0093)0.998.298:0.01399)0.948.76:0.00665,(267176:0.01531,543227:0.03246)0.992.158:0.01536)0.985.107:0.01037)0.988.146:0.01508)0.969.90:0.01938,(((113624:0.01068,254895:0.02394)0.987.166:0.01641,(217803:0.02539,(443417:0.01336,(218620:0.02113,113279:0.00162)0.977.103:0.01233)0.963.88:0.00859)0.860.41:0.00649)1.000.1593:0.0258,(565702:0.04683,((((592737:0.00732,255390:0.02682)0.742.26:0.00288,(113420:0.01979,250850:0.00514)0.189.2:0.0029)0.958.91:0.0081,250646:0.01541)1.000.1594:0.03218,((579227:0.00829,555912:0.00551)0.936.61:0.00581,(354630:0.00339,(527979:0.02391,509776:0.00738)0.097.5:0.00162)0.996.214:0.01118)0.996.215:0.01756)0.913.65:0.01004)0.968.83:0.01236)1.000.1595:0.03379)0.457.5:0.01)0.975.102:0.01639)0.931.56:0.01912)0.967.79:0.01818)0.542.6:0.01306)0.931.57:0.0137,((((148419:0.0014,((143743:0.01421,148863:0.00449)0.528.9:0.01359,141484:0.00216)0.742.27:0.00804)0.995.200:0.01954,((((172833:0.00813,225213:0.01212)1.000.1596:0.01802,(((((170934:0.0181,172538:0.00849)0.970.85:0.0083,(227403:0.00895,218852:0.01736)0.747.21:0.00358)0.779.13:0.00113,(172708:0.00595,168966:0.00636)0.780.17:0.00358)0.942.78:0.00405,(218346:0.01177,173574:0.00538)0.740.29:0.00325)0.982.128:0.00883,(144032:0.01741,148046:0.0067)1.000.1597:0.02348)0.896.46:0.00701)0.586.6:0.00518,((148440:0.01155,141110:0.01852)0.986.125:0.01355,(142085:0.00103,(144541:0.00479,142099:0.01318)0.881.47:0.0075)0.927.66:0.02736)0.948.77:0.00592)0.807.19:0.00389,(141091:0.02234,(141065:0.00915,(147943:0.01576,143792:0.01204)0.530.8:0.00437)0.977.104:0.01092)0.941.80:0.0083)0.894.47:0.00799)1.000.1598:0.08249,(((100224:0.00039,288272:0.0023)0.984.99:0.01211,(298129:0.01214,3580:0.01255)0.252:0.00483)0.996.216:0.02911,(278133:0.03489,(3591:0.05026,((((288106:0.01255,(292815:0.00606,3581:0.01676)0.974.95:0.007)0.996.217:0.00984,(333463:0.01779,293363:0.01539)0.989.123:0.01231)0.839.37:0.00267,((580653:0.0043,300420:0.00789)0.738.21:0.00124,((273321:0.00796,288501:0.01748)0.911.53:0.00337,104029:0.0071)0.977.105:0.00687)0.433.6:0.00192)0.917.58:0.00634,(170693:0.03205,576830:0.01485)0.543.4:0.0052)0.960.98:0.01024)0.984.100:0.01297)0.965.87:0.01725)0.996.218:0.04261)1.000.1599:0.08495,(15719:0.06564,(225636:0.0085,145889:0.00606)1.000.1600:0.12754)1.000.1601:0.11178)0.969.91:0.03449)0.971.116:0.02007)0.471.9:0.01398,(((261146:0.02451,(136393:0.00642,(136727:0.02427,(24509:0.00944,518321:0.01108)0.866.36:0.00361)0.750.21:0.00161)1.000.1602:0.02218)0.954.79:0.02193,(585039:0.03081,(561622:0.04229,(266522:0.00746,(238623:0.01443,((243968:0.00812,(243874:0.01359,223032:0.00705)0.985.108:0.01327)0.482.8:0.00077,568429:0.02041)0.852.53:0.00174)0.733.19:0.00083)0.771.20:0.00285)1.000.1603:0.0215)0.930.56:0.0123)1.000.1604:0.08298,((548387:0.02658,(((316722:0.01045,331621:0.00398)0.768.28:0.00159,239596:0.01449)0.482.9:0.00068,250058:0.0057)1.000.1605:0.03979)0.697.10:0.0096,((317416:0.01953,566144:0.00882)1.000.1606:0.03351,(159470:0.01087,138844:0.02294)0.990.130:0.01783)0.384.5:0.01322)1.000.1607:0.09193)0.987.167:0.03103)0.827.22:0.01077)0.936.62:0.00634,((((96419:0.01498,513258:0.01678)1.000.1608:0.08281,(202214:0.02742,578115:0.02422)1.000.1609:0.08501)1.000.1610:0.04479,(((106589:0.04127,261876:0.02253)1.000.1611:0.06867,(136863:0.0451,(105224:0.02412,112449:0.01322)1.000.1612:0.02811)1.000.1613:0.06189)1.000.1614:0.05303,(556661:0.14598,254942:0.10932)0.757.32:0.02444)0.985.109:0.02787)0.994.175:0.0261,((((((272615:0.04046,(164866:0.02191,208727:0.02846)1.000.1615:0.04651)0.991.150:0.01847,((288743:0.01839,350305:0.01845)1.000.1616:0.0564,(576754:0.04504,(154700:0.02304,528972:0.03508)0.955.75:0.01484)0.971.117:0.01653)0.000.66:0.0107)0.877.46:0.01009,(61498:0.02108,(576700:0.03321,61307:0.01343)0.923.58:0.00612)1.000.1617:0.03738)0.911.54:0.00921,(((65675:0.00969,19035:0.0217)0.973.89:0.01206,(225631:0.00665,556480:0.0402)0.745.22:0.01527)1.000.1618:0.06912,((148869:0.0235,571249:0.03028)1.000.1619:0.0297,(137499:0.04844,(((50582:0.05132,((577040:0.0191,(162674:0.02299,153947:0.02352)0.074.5:0.00366)1.000.1620:0.03102,((239453:0.02793,555087:0.02053)1.000.1621:0.01969,(159549:0.05627,(10543:0.07974,(((366100:0.00397,29910:0.00513)0.995.201:0.01404,(260396:0.0182,(259809:0.02303,545705:0.0167)0.879.53:0.00693)0.998.299:0.01633)0.995.202:0.01344,(550666:0.01205,(238619:0.01462,159650:0.01469)0.515.10:0.00841)0.999.463:0.02782)0.702.11:0.00895)0.396.4:0.0037)0.713.7:0.009)0.984.101:0.01546)0.675.6:0.00909)0.115.3:0.00645,(156555:0.02536,586450:0.00702)1.000.1622:0.03398)0.341.8:0.0051,(242950:0.01173,(226009:0.01454,242897:0.02883)0.109.5:0.00325)1.000.1623:0.03217)0.079.3:0.00779)0.854.34:0.00995)0.530.9:0.00725)0.159.5:0.0074)0.988.147:0.02867,(32172:0.097,(591098:0.02111,226370:0.03339)1.000.1624:0.04123)0.979.113:0.0307)1.000.1625:0.10223,((((592094:0.02779,509050:0.01702)0.932.60:0.01143,(61847:0.0583,(304435:0.03291,(257532:0.02633,70520:0.035)0.987.168:0.01243)0.770.21:0.00941)0.909.55:0.00888)1.000.1626:0.05089,(541526:0.11975,((171751:0.02596,(((249848:0.00788,527795:0.01526)1.000.1627:0.02922,((((154652:0.00154,170025:0.0212)0.682.6:0.0128,((109302:0.01713,539329:0.00472)1.000.1628:0.01742,(169350:0.02628,546424:0.00237)0.860.42:0.01324)0.435.8:0.00341)0.974.96:0.00886,(377046:0.01889,((510341:0.01206,344235:0.02552)0.997.229:0.01457,542476:0.02066)0.777.18:0.00447)0.898.53:0.00454)0.070.3:0.00326,(241933:0.03884,(160976:0.02731,243596:0.05615)0.666.10:0.00953)0.971.118:0.00862)0.771.21:0.00658)0.850.29:0.0067,(((218039:0.01554,395003:0.02085)0.948.78:0.00734,((251942:0.02036,(238907:0.00992,(447292:0.00921,(237236:0.01985,260694:0.01584)0.918.65:0.00466)0.995.203:0.01064)0.898.54:0.00505)0.062:0.00321,130540:0.01645)0.911.55:0.00614)0.578.3:0.00562,(275259:0.02992,((269229:0.0044,561014:0.01698)1.000.1629:0.01572,(265533:0.02061,(133316:0.02792,112300:0.01783)0.953.76:0.00918)0.821.15:0.00336)0.723.14:0.00259)0.868.49:0.00275)0.830.29:0.00365)0.934.55:0.00618)0.981.101:0.01291,(1687:0.05613,(((134526:0.00322,95059:0.00299)0.997.230:0.01228,(((276838:0.02022,535000:0.00727)0.120.3:0.00115,73076:0.01698)0.265.4:0.00263,((564233:0.00196,553953:0.00603)0.980.97:0.00891,271962:0.00563)0.204.3:0.00505)0.930.57:0.00742)1.000.1630:0.02565,(576545:0.04254,(53679:0.02276,89676:0.02517)0.988.148:0.01412)0.803.23:0.00406)0.080.3:0.00272)0.906.59:0.00844)0.999.464:0.03648)0.582.5:0.01517)0.921.60:0.01462,((260503:0.00739,(273717:0.01499,277194:0.01552)0.883.44:0.00503)1.000.1631:0.12775,((((417474:0.00793,(94416:0.00554,539125:0.01183)0.976.89:0.01446)1.000.1632:0.07145,(((135579:0.02641,447372:0.00798)0.999.465:0.0255,((581048:0.00462,(23652:0.01203,316477:0.00769)0.978.104:0.00851)0.996.219:0.01976,(135547:0.02483,278595:0.02279)0.878.53:0.01207)0.966.85:0.01487)1.000.1633:0.06961,((104126:0.03432,217004:0.02808)0.970.86:0.02296,((199167:0.00725,(241732:0.02827,60591:0.01089)0.908.51:0.01059)1.000.1634:0.0536,(567115:0.00781,(260316:0.02321,94589:0.02753)0.901.43:0.01227)0.970.87:0.01756)1.000.1635:0.0396)0.959.80:0.01908)0.679.11:0.01189)1.000.1636:0.04539,(((((249817:0.01585,239492:0.01674)0.034.3:0.00151,(250278:0.01083,256432:0.02599)0.727.14:0.00266)0.983.109:0.01186,(531378:0.01424,(248280:0.02018,509120:0.00734)0.555.6:0.00661)0.552.8:0.00093)0.988.149:0.01605,((176150:0.01554,576679:0.03903)0.965.88:0.01213,((312378:0.00744,240747:0.01479)0.885.49:0.00516,(218152:0.01526,156651:0.0091)0.993.147:0.0119)0.997.231:0.01711)0.564.11:0.00791)1.000.1637:0.04887,(220050:0.07756,((114382:0.01983,276586:0.00841)0.992.159:0.01629,(312923:0.02945,(236940:0.02629,330280:0.01532)0.897.48:0.0093)0.851.38:0.00725)1.000.1638:0.05531)0.833.23:0.021)0.613.5:0.00912)0.939.74:0.01045,(549436:0.06674,((223649:0.02153,236485:0.02824)0.995.204:0.0181,((333765:0.0228,114552:0.01479)0.354.6:0.00453,((((242418:0.02372,114801:0.01411)0.907.51:0.00387,(142490:0.01579,((218590:0.01982,141207:0.02722)0.951.68:0.00524,(223811:0.01008,((223647:0.01715,253909:0.02093)0.770.22:0.00076,253597:0.01474)0.765.25:0.00082)0.749.20:0.00334)0.751.12:0.00076)0.975.103:0.00464)0.480.11:0.00203,(251556:0.01795,104704:0.0155)0.937.69:0.00512)0.382.7:0.00232,114701:0.01325)0.971.119:0.00818)0.930.58:0.00816)1.000.1639:0.06494)0.800.25:0.01252)0.998.300:0.03368)1.000.1640:0.05551)0.967.80:0.01575)0.897.49:0.01063)0.992.160:0.01784)0.737.13:0.00332,((((((273635:0.01571,(546446:0.02215,(136988:0.01275,269959:0.01565)0.246.3:0.00343)0.903.55:0.00496)0.733.20:0.00219,260370:0.01395)1.000.1641:0.02812,(105296:0.02238,((533590:0.01267,(((578422:0.00798,538794:0.01389)0.434.8:0.00087,((257902:0.00758,75874:0.01129)0.337.4:0.00286,263398:0.01802)0.666.11:0.00195)0.680.8:0.00053,(514003:0.00934,261386:0.00705)0.976.90:0.00789)0.468.8:0.00292)1.000.1642:0.02939,(239257:0.01784,(156874:0.00505,(236002:0.02336,439693:0.01507)0.777.19:0.00541)0.899.54:0.00316)1.000.1643:0.03594)0.957.106:0.01095)0.912.40:0.01385)1.000.1644:0.03444,(328957:0.04769,(((((311659:0.0256,250861:0.027)0.741.22:0.00466,593321:0.0063)0.784.22:0.00422,(546975:0.02619,((534321:0.01437,251451:0.00549)0.940.77:0.00233,213599:0.01447)0.924.53:0.00014)0.903.56:0.00361)0.991.151:0.0098,((((576480:0.00791,(240574:0.00592,590993:0.00696)0.692.4:0.00507)0.834.30:0.00417,(239170:0.0101,220959:0.03397)0.975.104:0.01088)1.000.1645:0.02639,152331:0.03536)0.968.84:0.01018,(((208694:0.01843,251452:0.02719)1.000.1646:0.02692,((512689:0.01699,(579647:0.00929,(306708:0.02622,250822:0.00767)0.884.51:0.00315)0.555.7:0.00191)0.932.61:0.00454,(544639:0.00211,29377:0.0223)0.997.232:0.01027)0.797.18:0.00172)0.932.62:0.0036,(237286:0.01503,((255269:0.00998,248104:0.01858)0.977.106:0.00813,(327042:0.00439,255582:0.01782)1.000.1647:0.01754)0.877.47:0.00259)0.758.23:0.00121)0.845.28:0.00224)0.933.63:0.00528)0.983.110:0.01299,279508:0.02626)1.000.1648:0.04664)1.000.1649:0.03567)0.999.466:0.02368,((((254851:0.03036,520644:0.02719)1.000.1650:0.07607,((((297788:0.01145,(256652:0.01709,315699:0.0157)0.932.63:0.01049)1.000.1651:0.04201,(252185:0.02331,(97371:0.01883,271398:0.01435)0.946.61:0.00938)0.989.124:0.02288)1.000.1652:0.04372,((((((((586051:0.01117,548129:0.01047)0.999.467:0.01642,((((200479:0.01113,341275:0.00253)1.000.1653:0.03171,((258009:0.02717,(((304627:0.01098,(142005:0.01772,((211684:0.01386,(254041:0.01362,570191:0.01166)0.667.6:0.00379)0.627.6:0.00296,(578362:0.01953,587792:0.01045)0.967.81:0.00712)0.863.45:0.00278)0.911.56:0.0036)0.901.44:0.00289,((512598:0.0144,535721:0.00984)0.346.9:0.00192,(156857:0.01774,(310568:0.0081,(175459:0.01017,(555684:0.00628,237962:0.00692)0.514.4:0.00113)0.735.23:0.00073)0.736.21:0.00079)0.765.26:0.00066)0.923.59:0.00325)0.743.30:0.00081,((317166:0.00321,575389:0.01312)0.981.102:0.0056,(((((537397:0.00539,509823:0.01899)0.880.37:0.00274,359433:0.00976)0.978.105:0.0071,(254954:0.00335,(195246:0.01271,(337205:0.00812,95448:0.01246)0.856.33:0.00333)0.995.205:0.00956)0.712.9:0.00188)0.980.98:0.00897,(234743:0.02106,550406:0.00828)0.928.58:0.00672)0.984.102:0.00836,(511538:0.01317,541146:0.01628)0.628.11:0.00314)0.896.47:0.00291)0.967.82:0.00654)0.754.22:0.00164)0.374.7:0.00084,((245217:0.00789,(321058:0.00576,224143:0.00666)0.830.30:0.00208)0.948.79:0.00799,156808:0.00941)0.895.53:0.0027)0.526.6:0.00469)0.430.7:0.00796,(584106:0.01791,(155852:0.01344,551640:0.00869)0.925.49:0.0064)0.868.50:0.00401)0.109.6:0.00587,252636:0.03799)0.900.48:0.00754)1.000.1654:0.03093,(((321505:0.01489,(((((265935:0.03528,((325556:0.0009,((200645:0.00969,172055:0.01196)0.901.45:0.00443,(136924:0.01054,352218:0.00508)0.669.8:0.00014)0.922.60:0.00587)0.056.3:0.00768,260189:0.00976)0.706.12:0.00649)0.902.39:0.00598,(161147:0.01965,246465:0.01762)0.992.161:0.01065)0.898.55:0.00498,(554828:0.0249,309927:0.01521)1.000.1655:0.03869)0.888.38:0.00337,151522:0.01466)0.995.206:0.01342,(586725:0.01778,(1813:0.01004,(100019:0.02224,(297309:0.00873,235765:0.01412)0.717.10:0.00088)0.997.233:0.01092)0.139.6:0.01624)0.329.6:0.00224)0.798.17:0.00341)0.950.93:0.00777,542998:0.01389)0.595.3:0.00323,202031:0.00558)1.000.1656:0.02665)0.983.111:0.02295,(((((((249598:0.00851,353698:0.00574)0.990.131:0.01564,(584482:0.01313,558300:0.02092)0.935.64:0.01037)0.999.468:0.02704,((251722:0.01581,543361:0.01699)1.000.1657:0.05703,(255794:0.01454,(566535:0.02043,(((195316:0.00552,590747:0.00994)0.885.50:0.00248,(1745:0.00421,247203:0.00745)0.983.112:0.01112)0.683.13:0.00189,(36828:0.00601,((588240:0.01239,(347625:0.00626,367606:0.02393)0.986.126:0.01061)0.677.8:0.00843,(339544:0.00163,225472:0.00659)0.046.3:0.01579)0.903.57:0.00573)0.918.66:0.00396)0.981.103:0.01093)0.514.5:0.00464)0.998.301:0.03206)0.979.114:0.01853)0.957.107:0.01368,(3696:0.02203,210897:0.01246)1.000.1658:0.09339)0.861.46:0.00961,(((588464:0.00706,(((((((573607:0.02684,238781:0.04366)0.236.3:0.00548,(307883:0.0068,242038:0.06571)1.000.1659:0.05324)0.971.120:0.01367,(340160:0.0211,(533587:0.00909,(204614:0.00806,364090:0.00695)0.833.24:0.0049)0.969.92:0.00714)0.825.21:0.01098)0.902.40:0.00646,(144752:0.01366,(((256992:0.00553,205994:0.0126)0.852.54:0.00164,277931:0.01284)0.738.22:0.00077,(368049:0.00237,3697:0.02107)0.785.24:0.01978)0.997.234:0.0126)0.993.148:0.00962)0.676.10:0.00256,((193986:0.00572,((511024:0.01487,566753:0.01313)0.146.3:0.00612,((354865:0.00158,((590281:0.00763,509431:0.01089)0.554.8:0.0084,(572890:0.0056,306532:0.01587)0.933.64:0.00774)0.889.44:0.00159)0.397.12:0.00765,328633:0.00428)0.968.85:0.00504)0.777.20:0.00108)0.916.55:0.02144,(((584174:0.03537,340198:0.00821)0.955.76:0.00701,21414:0.01266)0.523.9:0.01221,311802:0.01026)0.730.18:0.00108)0.960.99:0.00451)0.916.56:0.00414,587975:0.00873)0.904.56:0.00239,(((((112292:0.02391,113626:0.0055)0.978.106:0.00728,(152150:0.01292,254353:0.02137)0.808.22:0.00563)0.866.37:0.00468,((509991:0.01691,(223948:0.01658,279384:0.01601)0.990.132:0.01079)0.876.50:0.00907,((208380:0.01305,549105:0.02687)1.000.1660:0.04415,(((260501:0.01218,(575010:0.02039,(235699:0.01212,219881:0.01546)0.962.85:0.0061)0.464.5:0.00427)0.938.74:0.03381,((((((238020:0.01573,(151008:0.02456,242304:0.00353)0.967.83:0.00655)0.723.15:0.00229,593400:0.03563)0.357.8:0.01051,(((234366:0.02021,578757:0.0186)0.927.67:0.00958,(580740:0.01311,(540520:0.02213,562772:0.00741)0.929.70:0.00637)0.919.60:0.00712)0.755.29:0.00175,130368:0.01617)0.894.48:0.00352)0.860.43:0.00262,(((((((590533:0.01086,(510646:0.00925,(239226:0.00855,113400:0.01055)0.582.6:0.01225)0.993.149:0.01028)0.936.63:0.00502,(113347:0.0009,130135:0.00114)1.000.1661:0.02047)0.941.81:0.00409,594110:0.01056)0.970.88:0.00498,((1749:0.00853,(114283:0.00607,552051:0.03665)0.983.113:0.0056)0.879.54:0.00185,(197417:0.02798,(160452:0.0008,86817:0.01131)0.988.150:0.00712)0.993.150:0.00761)0.862.47:0.0016)0.976.91:0.00782,((51292:0.00783,(340207:0.02862,565019:0.04008)1.000.1662:0.03514)0.581.4:0.00601,((579734:0.01384,(401741:0.02061,(113849:0.011,352078:0.01575)0.953.77:0.00557)0.908.52:0.00375)0.933.65:0.01101,(113117:0.00512,244820:0.03248)0.993.151:0.01122)0.946.62:0.00685)0.851.39:0.00289)0.957.108:0.00482,513801:0.01742)0.825.22:0.0016,((532693:0.00728,((226437:0.00092,211753:0.00275)0.793.17:0.00174,221008:0.01876)0.751.13:0.00075)0.987.169:0.00854,252860:0.00512)0.974.97:0.00549)0.865.45:0.00428)0.928.59:0.00666,(509757:0.01872,(592028:0.01805,((81698:0.00492,(244149:0.02281,113715:0.00666)0.008.2:0.01382)0.974.98:0.00609,(56714:0.01572,142878:0.00762)0.922.61:0.00409)0.882.44:0.00455)0.883.45:0.00486)0.762.24:0.00541)0.968.86:0.00804,((228891:0.02085,164562:0.02271)0.957.109:0.00648,((((141698:0.00953,238914:0.02913)0.985.110:0.01253,(171397:0.01294,103211:0.0149)0.984.103:0.01151)0.906.60:0.00745,(((240332:0.0063,55946:0.00648)0.989.125:0.01286,(244863:0.0092,254706:0.00875)1.000.1663:0.02519)0.999.469:0.01553,(570160:0.015,156358:0.01224)0.844.25:0.01052)0.917.59:0.00893)0.868.51:0.00744,((213767:0.01251,(105011:0.00727,510284:0.01173)0.805.24:0.0053)0.886.42:0.00325,((353816:0.01025,86097:0.0088)0.977.107:0.00758,(509916:0.01462,216444:0.02239)0.930.59:0.00768)0.953.78:0.00509)0.861.47:0.00298)0.985.111:0.01049)0.751.14:0.00486)0.990.133:0.01279)0.995.207:0.02028,((169436:0.00593,(568934:0.00789,(169877:0.00206,(549177:0.01304,171427:0.01814)0.999.470:0.01396)0.889.45:0.02364)0.847.31:0.00361)0.990.134:0.01128,213061:0.0093)1.000.1664:0.03278)0.992.162:0.02329)0.980.99:0.01569)0.229.6:0.00454)0.152.6:0.00544,((218151:0.00726,256444:0.01421)0.961.78:0.00932,(((247010:0.0082,220006:0.01673)0.998.302:0.01611,(1791:0.01938,(576793:0.01859,78630:0.03461)0.304.6:0.00403)0.962.86:0.01058)0.853.32:0.01205,548626:0.02999)0.056.4:0.00226)0.917.60:0.00663)0.999.471:0.01854,327784:0.00596)0.283.5:0.00717)0.901.46:0.00341)0.987.170:0.01496,251244:0.07227)1.000.1665:0.03817,((237774:0.01838,((238966:0.02673,582048:0.00446)0.999.472:0.01534,336301:0.01465)0.895.54:0.00648)1.000.1666:0.03721,((357085:0.00855,258006:0.00703)0.976.92:0.01417,((((((((250719:0.01821,556648:0.02428)0.991.152:0.01349,314978:0.01429)0.881.48:0.00503,136325:0.01871)0.884.52:0.00598,253420:0.01574)0.800.26:0.00551,136430:0.02301)0.937.70:0.00609,(214023:0.01189,582008:0.03278)0.292.11:0.00233)0.997.235:0.0102,355815:0.00167)0.854.35:0.00317,252725:0.03139)1.000.1667:0.03058)1.000.1668:0.04941)0.686.8:0.01455)0.927.68:0.00845)1.000.1669:0.04023,((((155327:0.0204,213048:0.04211)1.000.1670:0.03451,(((240236:0.02363,224515:0.02592)0.689.8:0.01041,(((211929:0.00148,(223186:0.00623,225201:0.01101)0.952.83:0.00512)1.000.1671:0.02382,(((405751:0.02404,244850:0.02483)0.723.16:0.00682,(241341:0.03087,(155408:0.01995,556931:0.01246)0.641.10:0.00399)0.615.8:0.00299)0.965.89:0.01168,(215656:0.01628,(((((249932:0.01135,241044:0.01036)0.777.21:0.00126,(560258:0.01536,318060:0.00456)0.780.18:0.00131)0.778.32:0.00254,((286989:0.01335,(540065:0.00485,(204114:0.01671,(513940:0.0164,208519:0.0114)0.737.14:0.00917)0.987.171:0.00928)1.000.1672:0.01077)0.878.54:0.00224,(215337:0.01744,539070:0.00576)0.978.107:0.00732)0.872.56:0.00322)0.384.6:0.00195,221964:0.01758)0.774.14:0.00537,(212861:0.01774,(136313:0.01846,226137:0.01238)0.882.45:0.00472)0.881.49:0.0056)0.465.4:0.00182)0.787.24:0.00406)0.155.6:0.00589)0.492.6:0.00495,((((((((((109694:0.0162,(211355:0.02874,218366:0.00264)0.856.34:0.00222)0.650.5:0.00378,206267:0.01165)0.998.303:0.01748,(250258:0.01529,(535450:0.02052,(313233:0.00924,284815:0.01801)0.999.473:0.02196)0.880.38:0.00679)0.813.25:0.00563)0.940.78:0.00623,(((152830:0.01783,280477:0.02433)0.976.93:0.0123,(156032:0.01076,557465:0.00662)0.996.220:0.00725)0.895.55:0.01013,238801:0.01486)0.996.221:0.00878)0.874.47:0.00435,564102:0.01915)0.873.48:0.00443,((143307:0.01474,((267261:0.04007,540833:0.01929)0.227.6:0.00481,590862:0.01633)0.901.47:0.01133)0.637.10:0.00324,(223904:0.01151,(159436:0.01516,(63241:0.02229,250674:0.00108)0.998.304:0.0202)0.915.53:0.00676)0.883.46:0.00638)0.959.81:0.00631)0.962.87:0.00845,((151957:0.01827,((246248:0.01959,531776:0.01487)0.923.60:0.00487,((235525:0.00926,532058:0.0138)0.921.61:0.00447,((206159:0.00468,(204592:0.02119,160645:0.0083)0.885.51:0.00416)0.903.58:0.00322,217472:0.02988)0.994.176:0.0102)0.810.23:0.00319)0.825.23:0.00422)0.780.19:0.00695,(151613:0.02501,(223221:0.02239,155429:0.016)0.996.222:0.01619)0.990.135:0.01398)0.367.6:0.00423)0.726.16:0.00107,((((572096:0.01004,(559379:0.01085,(((168599:0.0107,(170495:0.00016,212239:0.02025)0.999.474:0.01854)0.254.7:0.0158,((((541280:0.00447,((229694:0.00096,351370:0.01121)0.680.9:0.0186,(3735:0.01547,347619:0.01544)0.931.58:0.00481)0.967.84:0.00545)0.966.86:0.00544,((((278437:0.01463,513331:0.0096)0.984.104:0.0131,(335716:0.01452,(546864:0.01571,545180:0.03012)0.872.57:0.00511)0.638.5:0.00312)0.977.108:0.00638,45591:0.01594)0.951.69:0.0043,151114:0.03435)0.389.4:0.00159)0.413.5:0.00231,3730:0.0098)0.922.62:0.01096,(334337:0.00852,111942:0.00608)0.912.41:0.00015)0.945.65:0.00485)0.925.50:0.00568,((155420:0.01909,296370:0.02378)0.188.6:0.00015,160987:0.00232)0.892.42:0.00948)0.930.60:0.00321)0.861.48:0.00969)0.960.100:0.00716,((221248:0.01571,562598:0.0129)0.971.121:0.01046,39070:0.00569)0.948.80:0.00566)0.994.177:0.0137,((((530449:0.02585,(232738:0.02267,(245135:0.01988,(344737:0.00714,207107:0.00961)0.731.24:0.00253)0.998.305:0.01481)0.492.7:0.00623)0.637.11:0.00166,244442:0.02598)0.921.62:0.00469,(((511685:0.03176,(553417:0.01835,(220544:0.01407,245829:0.01134)0.945.66:0.00628)0.963.89:0.00913)0.666.12:0.00708,(((((((568253:0.01546,156636:0.01382)0.902.41:0.00586,154418:0.02154)0.867.48:0.00299,(511274:0.01235,534348:0.00183)0.908.53:0.00309)0.988.151:0.00656,28064:0.00538)0.744.23:0.00117,(225706:0.01229,113079:0.03731)0.647.8:0.00304)0.955.77:0.00607,573448:0.03019)0.925.51:0.00277,566300:0.01074)0.569.5:0.00298)0.930.61:0.00544,(563496:0.0127,41797:0.01514)0.783.24:0.00434)0.673.5:0.00111)0.838.33:0.00709,((154268:0.01943,112964:0.02368)0.988.152:0.01387,(((513998:0.01809,(((((156545:0.0084,(112195:0.00319,216773:0.00416)0.981.104:0.00672)0.994.178:0.00911,77390:0.01881)0.850.30:0.00185,151539:0.00935)0.933.66:0.00365,510380:0.01258)0.789.26:0.00164,575808:0.01991)0.940.79:0.00643)0.546.4:0.00014,((180012:0.01039,538729:0.00529)0.929.71:0.0176,(545339:0.01027,(136303:0.0078,255562:0.00434)0.971.122:0.00934)0.906.61:0.00246)0.845.29:0.00156)0.909.56:0.00271,(((255143:0.02234,((144515:0.00685,534446:0.01088)0.967.85:0.00548,209825:0.00592)0.964.104:0.00376)0.319.6:0.00832,182111:0.02562)0.880.39:0.00177,((257823:0.01693,538131:0.0035)0.958.92:0.0077,570719:0.00907)0.736.22:0.00517)0.873.49:0.00175)0.956.76:0.00677)0.939.75:0.00968)0.954.80:0.00941)0.999.475:0.01835,((566462:0.01599,((309897:0.02004,568935:0.00394)0.755.30:0.00081,(222403:0.0091,223898:0.0345)0.916.57:0.00981)0.986.127:0.00669)0.650.6:0.00354,(257397:0.02717,251273:0.02442)0.956.77:0.00663)0.999.476:0.01662)0.986.128:0.00964)0.973.90:0.00674,351826:0.02234)0.973.91:0.00835,(((238672:0.024,((294313:0.01254,(145786:0.02685,(531025:0.01079,279208:0.01538)0.683.14:0.00101)0.780.20:0.00456)0.960.101:0.00839,(77412:0.01862,226506:0.02439)0.779.14:0.0045)0.930.62:0.00987)0.862.48:0.00337,(((153867:0.02361,((278816:0.0153,543955:0.01192)0.928.60:0.00313,111986:0.01159)0.767.37:0.00476)0.936.64:0.01308,(((277669:0.0061,(155428:0.02304,575937:0.01054)0.810.24:0.00312)0.860.44:0.00183,(291945:0.02372,((223625:0.01554,(((512197:0.01453,154494:0.0168)0.486.4:0.0029,570222:0.01076)0.762.25:0.00552,(566139:0.00932,154234:0.02556)0.800.27:0.00816)0.731.25:0.00109)0.805.25:0.00085,(((221695:0.01555,211201:0.01725)0.975.105:0.00825,220721:0.00496)0.960.102:0.00483,300472:0.01135)0.899.55:0.00029)0.630.4:0.01495)0.374.8:0.00324)0.948.81:0.00828,(114172:0.01462,114214:0.03589)0.870.50:0.01553)0.901.48:0.00307)0.987.172:0.01026,136431:0.01937)0.485.4:0.00416)0.936.65:0.00491,((((((572434:0.01753,(243138:0.01948,(((137302:0.0101,244133:0.01859)0.977.109:0.00755,(218646:0.01113,223415:0.01357)0.848.35:0.00415)0.712.10:0.00182,235675:0.00765)0.832.23:0.00271)0.895.56:0.00669)0.849.30:0.0038,510555:0.01639)0.618.7:0.00441,(((226552:0.00804,254817:0.0047)0.889.46:0.00826,(248716:0.00517,280233:0.02701)0.908.54:0.00715)0.992.163:0.0124,245527:0.01458)0.939.76:0.00581)0.726.17:0.00153,(((215113:0.01571,155297:0.01096)0.859.46:0.00533,(257132:0.00949,316260:0.01373)0.930.63:0.00596)0.945.67:0.00015,236797:0.00438)0.915.54:0.01226)0.878.55:0.00553,(221826:0.02493,(((216553:0.0167,227524:0.02653)0.980.100:0.01542,237546:0.01326)0.121.2:0.00435,(303986:0.01789,223452:0.00528)0.777.22:0.01512)0.798.18:0.00166)0.975.106:0.00767)0.635.5:0.00251,153165:0.05367)0.973.92:0.01044)0.933.67:0.00545)0.987.173:0.00863)0.897.50:0.00612)0.945.68:0.01031,(212293:0.04612,211129:0.028)0.998.306:0.02898)0.973.93:0.01207)0.997.236:0.01932,(((539508:0.00842,(244025:0.00832,553285:0.01064)0.946.63:0.00621)0.996.223:0.02167,(128109:0.01185,566891:0.0145)0.993.152:0.01861)0.863.46:0.01805,((548651:0.06679,(((257405:0.03285,581434:0.01272)1.000.1673:0.02146,186978:0.00969)0.313.10:0.00405,236707:0.02917)0.956.78:0.011)1.000.1674:0.0272,((544603:0.00867,((161093:0.00277,583370:0.0075)0.983.114:0.00652,347057:0.00448)0.954.81:0.00565)1.000.1675:0.01998,((1740:0.02574,(((222528:0.01164,113257:0.00862)0.901.49:0.00605,212863:0.02437)0.979.115:0.00882,((246947:0.0367,((218628:0.01329,513426:0.01297)0.947.75:0.00401,(238014:0.02355,250533:0.01026)0.911.57:0.00174)1.000.1676:0.00064)1.000.1677:0.00239,255838:0.01063)0.620.4:0.0064)0.997.237:0.014)0.922.63:0.0051,(138920:0.01283,((16412:0.00923,278051:0.03211)0.935.65:0.00417,540569:0.01682)0.998.307:0.01217)0.936.66:0.00554)0.996.224:0.01834)0.999.477:0.03225)0.998.308:0.02324)0.912.42:0.01526)0.981.105:0.01589,((((161597:0.02608,256750:0.02514)0.997.238:0.01864,((220785:0.02704,238997:0.03362)0.922.64:0.008,(((154876:0.0128,524824:0.00306)0.994.179:0.01343,(569539:0.01623,529365:0.00888)0.997.239:0.01594)0.981.106:0.01503,((((241551:0.01825,511513:0.00657)0.912.43:0.00672,(((113331:0.0134,528713:0.01742)0.122.3:0.00102,(534995:0.01335,114356:0.01558)0.756.29:0.00127)0.827.23:0.00121,((536567:0.02806,151955:0.01788)0.925.52:0.00381,((114163:0.00732,(239890:0.00447,137193:0.01827)0.963.90:0.00747)0.970.89:0.01026,(((279613:0.01006,559586:0.01793)0.952.84:0.00596,(113763:0.00793,114349:0.01579)0.994.180:0.01137)0.957.110:0.00608,(239344:0.00878,(((113212:0.01515,203969:0.01173)0.922.65:0.00382,(539027:0.01907,(153350:0.00752,153088:0.01246)0.946.64:0.00424)0.858.31:0.00169)0.856.35:0.00501,560615:0.01488)0.849.31:0.00633)0.842.34:0.00207)0.880.40:0.00263)0.809.18:0.00016)0.896.48:0.00166)0.849.32:0.00313)0.999.478:0.02303,((225293:0.00838,533156:0.01887)0.370.7:0.0053,(210409:0.01706,558328:0.02659)0.996.225:0.0154)0.277.4:0.00541)0.974.99:0.01278,(244154:0.02786,(((222398:0.01792,61838:0.01238)0.996.226:0.01169,(((224300:0.02926,(255186:0.00897,556383:0.00484)0.898.56:0.00299)0.678.10:0.00146,(212596:0.03351,590477:0.0041)0.978.108:0.00579)0.999.479:0.01572,(210872:0.04983,((328506:0.02214,(248865:0.00822,(244404:0.02169,244412:0.00727)0.949.77:0.00608)0.864.47:0.00166)0.746.31:0.00089,(((279297:0.01943,151018:0.02746)0.907.52:0.00591,((154822:0.00828,(514746:0.00508,(204462:0.0127,563862:0.00855)0.172.5:0.00289)0.897.51:0.00339)0.961.79:0.00627,(245411:0.00868,(3732:0.01877,255584:0.0292)0.884.53:0.00333)0.664.6:0.00486)0.915.55:0.00015)0.939.77:0.00351,(39712:0.01995,(537147:0.01376,(557785:0.00834,(549552:0.01506,(156700:0.00627,320175:0.01449)0.950.94:0.01672)0.927.69:0.00391)0.894.49:0.00463)0.038:0.00274)0.998.309:0.00918)0.861.49:0.00159)0.912.44:0.00532)1.000.1678:0.02714)0.954.82:0.00767)0.196.6:0.00187,((((577299:0.00528,78648:0.02283)0.351.6:0.00434,547632:0.01452)0.710.13:0.00373,278511:0.00682)0.196.7:0.00364,((156406:0.00437,(156650:0.00695,575745:0.01548)1.000.1679:0.01724)0.449.6:0.00274,136474:0.00662)0.920.53:0.00331)0.849.33:0.0029)0.976.94:0.01278)0.998.310:0.01434)0.848.36:0.00917)0.736.23:0.00291)0.977.110:0.01388)1.000.1680:0.05669,(((((567172:0.00881,(199494:0.01766,561841:0.00993)0.979.116:0.00956)1.000.1681:0.02764,(((316355:0.01495,(209523:0.01307,334056:0.011)0.934.56:0.00646)1.000.1682:0.01858,(((((((580473:0.02129,579143:0.01748)0.984.105:0.01168,((110461:0.0196,137789:0.01819)0.916.58:0.00589,((139328:0.00721,(244715:0.01112,553939:0.00365)0.896.49:0.00268)0.996.227:0.0215,(553759:0.01069,(299684:0.01029,240983:0.0239)0.572.5:0.00574)0.938.75:0.00675)0.940.80:0.00881)0.982.129:0.00943)0.836.29:0.00372,(257077:0.01025,(161914:0.02064,(200953:0.02613,562807:0.01195)0.914.47:0.0053)1.000.1683:0.0132)0.804.25:0.00481)0.824.18:0.00264,(519245:0.02198,(((((138346:0.00866,(245809:0.01773,250526:0.01545)0.379.13:0.00243)0.976.95:0.00756,572184:0.0092)0.736.24:0.00014,112776:0.01708)0.958.93:0.00543,594169:0.01695)0.751.15:0.00433,3667:0.01222)0.873.50:0.00426)0.931.59:0.00514)0.980.101:0.00817,(((390023:0.00649,287625:0.00357)0.756.30:0.00095,(259327:0.0035,205900:0.01247)0.967.86:0.00476)0.948.82:0.00593,356083:0.00158)0.905.61:0.00329)0.996.228:0.00971,((589802:0.00902,136281:0.02616)0.993.153:0.01213,(((((60816:0.02334,((60599:0.01987,(154141:0.02105,3677:0.01984)0.968.87:0.00858)0.969.93:0.00762,((155709:0.01836,(106955:0.01082,195362:0.02898)0.880.41:0.0065)0.302.8:0.00154,564318:0.02995)0.876.51:0.00513)0.994.181:0.01348)0.956.79:0.00855,(105319:0.02899,(((568755:0.02814,220379:0.02142)0.981.107:0.01134,(((138184:0.00922,566933:0.01366)0.989.126:0.01198,(272411:0.01279,553792:0.02394)0.958.94:0.00863)0.136.3:0.0034,((250740:0.00103,110208:0.00343)0.992.164:0.01583,153964:0.01691)0.139.7:0.00703)0.939.78:0.00947)0.815.17:0.00447,(145554:0.02452,(75670:0.01287,66935:0.01243)0.330.5:0.00089)0.580.8:0.01922)0.905.62:0.00774)0.872.58:0.0052)0.935.66:0.00554,155976:0.02238)0.923.61:0.00376,(339795:0.01038,(322588:0.01779,583760:0.0247)0.992.165:0.01211)0.964.105:0.00801)0.297.4:0.00107,272782:0.02163)0.934.57:0.0071)0.842.35:0.00481)0.180.3:0.00326,(((((((254383:0.00237,(((213933:0.0234,152545:0.02271)0.916.59:0.00781,156065:0.00489)0.943.81:0.00511,241718:0.00921)0.970.90:0.00494)0.742.28:0.00811,(537224:0.01042,541300:0.00204)0.981.108:0.00646)0.969.94:0.00644,(229514:0.01887,513782:0.00653)0.902.42:0.00439)0.635.6:0.00331,((210745:0.0129,160412:0.01288)0.882.46:0.00276,(26745:0.00584,190912:0.02377)0.885.52:0.0024)0.966.87:0.00542)0.954.83:0.00668,((145959:0.02191,263285:0.01856)0.368.10:0.00408,(105453:0.02019,((579706:0.00421,((241889:0.00883,534479:0.01244)0.928.61:0.00442,(250451:0.01044,((279551:0.00834,217695:0.01051)0.996.229:0.008,((237058:0.01815,541524:0.00662)0.061:0.00413,(((348120:0.00499,((532631:0.01205,257240:0.01185)0.878.56:0.00248,231211:0.01947)0.518.4:0.00208)0.842.36:0.0023,(((131970:0.0104,594397:0.00532)0.970.91:0.00644,343928:0.01032)0.786.25:0.00176,(516306:0.02321,(225571:0.0128,204993:0.01565)0.997.240:0.01811)0.592.12:0.00696)0.910.52:0.00498)0.749.21:0.00119,(593316:0.00741,589751:0.02839)0.933.68:0.00327)0.913.66:0.00245)0.957.111:0.00433)0.872.59:0.00209)0.915.56:0.00381)0.966.88:0.00576)0.984.106:0.01107,103580:0.03004)0.839.38:0.0078)0.133.5:0.00423)0.921.63:0.00563)0.844.26:0.00288,((((241567:0.03613,253770:0.01281)0.890.55:0.00761,338879:0.02532)0.916.60:0.00542,540593:0.04769)0.209.4:0.00129,(529964:0.00886,526228:0.01172)0.421.6:0.02187)0.989.127:0.0144)0.842.37:0.0036,((((513591:0.00628,(154125:0.01202,(221341:0.0142,240440:0.01891)0.617.7:0.0065)0.884.54:0.00421)0.866.38:0.00446,((((533572:0.0285,575896:0.01198)0.816.25:0.00737,(100169:0.01499,534921:0.00357)0.065.4:0.00485)0.344.3:0.00648,564195:0.02655)1.000.1684:0.03189,((((220981:0.00246,(222843:0.0124,542299:0.01305)0.768.29:0.00082)0.977.111:0.00488,534914:0.00984)0.886.43:0.00235,136323:0.01429)0.834.31:0.00216,(332934:0.0162,574961:0.01479)0.946.65:0.0046)0.044.4:0.00145)0.876.52:0.00249)0.541.7:0.00058,160999:0.02476)0.992.166:0.01181,((203801:0.0358,578443:0.02818)1.000.1685:0.03045,(242626:0.05142,150160:0.03735)0.100.5:0.00744)0.976.96:0.0139)0.878.57:0.00462)0.811.22:0.00202)0.855.34:0.00373)0.929.72:0.00644,((((((545323:0.01291,155041:0.02175)0.812.31:0.00392,((110291:0.00679,543818:0.0123)0.863.47:0.00346,(245506:0.00668,(544353:0.02999,152363:0.00929)0.929.73:0.00581)0.900.49:0.00473)0.776.27:0.00177)0.939.79:0.00442,(((214565:0.00674,(369392:0.02535,223220:0.00289)0.989.128:0.01116)0.892.43:0.00373,(226313:0.00732,155208:0.01461)0.382.8:0.00317)0.954.84:0.00618,((198925:0.00264,215356:0.01799)0.834.32:0.00156,(514228:0.01011,152963:0.03072)0.809.19:0.00541)0.975.107:0.00645)0.875.43:0.00304)0.882.47:0.00299,(161275:0.01726,(102185:0.00975,(((305083:0.00553,350446:0.00902)0.954.85:0.00322,(((224203:0.02318,235831:0.00461)0.964.106:0.00486,(152086:0.01157,304016:0.0104)0.759.16:0.0008)0.863.48:0.00241,(541749:0.00835,154988:0.0209)0.919.61:0.00473)0.771.22:0.00153)0.583.13:0.00073,278495:0.01369)0.910.53:0.00255)0.881.50:0.00239)0.818.16:0.00335)0.876.53:0.00292,(((210615:0.02772,141801:0.07414)0.979.117:0.01667,206392:0.02306)0.719.7:0.00147,(114075:0.02204,((151094:0.00999,(327098:0.00606,584693:0.03457)0.993.154:0.01471)0.571.9:0.00671,235290:0.0317)0.478.3:0.00382)0.347.3:0.00245)0.873.51:0.00267)0.981.109:0.00872,(((101119:0.01434,((135106:0.01181,(566511:0.004,(562190:0.01497,((278235:0.00528,525931:0.01942)0.925.53:0.0046,551710:0.00392)0.870.51:0.00312)0.966.89:0.00392)0.961.80:0.00556)0.554.9:0.00422,574911:0.01883)1.000.1686:0.01533)0.915.57:0.00548,((570620:0.01486,(101450:0.00242,343210:0.01187)0.471.10:0.01812)0.992.167:0.01396,561042:0.02825)0.732.20:0.00598)0.950.95:0.00988,(((139346:0.00915,581422:0.01635)0.999.480:0.02084,(254271:0.01582,(216341:0.01903,278026:0.01816)0.812.32:0.0051)0.545.7:0.00637)0.967.87:0.00988,((((78832:0.00641,240295:0.04892)0.025.3:0.00622,(226099:0.00418,210677:0.03331)0.963.91:0.00865)0.951.70:0.0068,134417:0.01014)0.992.168:0.0104,((210270:0.00229,223529:0.01378)0.957.112:0.00503,(583954:0.00858,209415:0.01622)0.842.38:0.00244)0.979.118:0.00762)0.930.64:0.00673)0.882.48:0.00524)0.707.10:0.00373)0.991.153:0.01069)0.902.43:0.01059)0.857.36:0.00622,(241307:0.02482,(211735:0.01307,101141:0.01072)0.452.12:0.00392)0.993.155:0.01262)0.893.39:0.00596,((534931:0.01859,(220327:0.01075,211403:0.02109)0.999.481:0.01766)0.997.241:0.02293,((535547:0.03005,((3678:0.00556,588882:0.01185)1.000.1687:0.02335,(199540:0.01623,268134:0.01425)1.000.1688:0.02589)0.998.311:0.02185)0.401.6:0.00286,((549259:0.01672,576977:0.02029)0.260.5:0.0056,272414:0.04173)0.920.54:0.00548)0.999.482:0.0221)0.959.82:0.016)1.000.1689:0.04072,(237818:0.03438,247708:0.01943)0.981.110:0.01907)0.996.230:0.02587)0.394.4:0.00889,((136128:0.01149,(159622:0.00563,147271:0.00321)0.881.51:0.0031)0.983.115:0.01171,((253443:0.00642,(240752:0.01123,(79020:0.01458,27253:0.00149)0.967.88:0.00697)0.728.18:0.00385)1.000.1690:0.01664,((73852:0.02344,(569274:0.02225,(533240:0.00636,((156453:0.00851,(578427:0.00865,77218:0.014)0.529.6:0.00326)0.613.6:0.00316,3703:0.01423)0.584.6:0.00292)0.962.88:0.00734)0.301.8:0.00435)0.861.50:0.00433,((587893:0.00985,242022:0.00029)0.274.7:0.00876,((246440:0.01025,112971:0.01724)0.821.16:0.0058,(61819:0.02241,239700:0.01977)0.619.8:0.00446)0.883.47:0.00662)0.833.25:0.00394)0.887.43:0.00496)0.849.34:0.00766)1.000.1691:0.04296)0.949.78:0.01558)0.995.208:0.02243)0.928.62:0.01275,(204097:0.03957,((160588:0.00719,((17129:0.01024,(174765:0.00449,440284:0.01541)0.737.15:0.0021)0.998.312:0.01269,(((567564:0.02014,113973:0.00752)0.859.47:0.00297,((139322:0.01158,(105472:0.0064,(((510240:0.00235,333617:0.00585)0.168.11:0.00371,((1815:0.00487,113500:0.01504)0.612.9:0.00176,(582320:0.00875,(572987:0.01388,(548878:0.00875,(549496:0.0119,155016:0.00559)0.372.7:0.00244)0.878.58:0.004)0.441.3:0.0056)0.881.52:0.00156)0.776.28:0.00078)0.975.108:0.00486,559480:0.00782)0.692.5:0.0098)0.929.74:0.00454)0.135.3:0.00513,571433:0.03412)0.877.48:0.01025)0.674.11:0.00696,(((210543:0.00984,346716:0.00581)0.951.71:0.00016,((314771:0.00997,112801:0.01755)0.786.26:0.00579,(123625:0.00937,328752:0.01062)0.958.95:0.00496)0.887.44:0.00182)0.914.48:0.00446,(88422:0.01207,577038:0.01147)0.914.49:0.00357)0.752.30:0.00624)0.989.129:0.01317)0.925.54:0.0121)0.994.182:0.01488,(((((162343:0.00702,252501:0.03032)0.992.169:0.01226,(162661:0.01514,(180033:0.01976,292370:0.01759)0.648.9:0.00647)0.888.39:0.00435)0.971.123:0.00649,((((((((((348270:0.00924,368658:0.01333)0.899.56:0.00298,561393:0.03195)0.852.55:0.00204,(582150:0.01027,565168:0.0112)0.502.9:0.00167)0.632.4:0.00016,(((324293:0.00161,582243:0.01318)0.872.60:0.00313,(88876:0.01283,136995:0.01182)0.769.20:0.00093)0.889.47:0.00474,((((274731:0.01816,(1818:0.00943,138750:0.01118)0.858.32:0.00159)0.847.32:0.007,257461:0.00794)0.413.6:0.00159,343086:0.0032)0.942.79:0.00381,(121575:0.00556,370090:0.00779)0.911.58:0.00347)0.867.49:0.00313)0.202.3:0.00078)0.925.55:0.00225,((556094:0.00595,(((325121:0.01925,((142821:0.0083,592157:0.00907)0.504.6:0.00015,(139075:0.01171,138743:0.02124)0.977.112:0.00737)0.975.109:0.00628)0.906.62:0.00252,(213076:0.01742,((509442:0.00409,1807:0.00972)0.371.5:0.00093,327577:0.00176)0.750.22:0.00076)0.736.25:0.00093)0.962.89:0.00766,584801:0.0109)0.176.3:0.001)0.829.22:0.00149,((530540:0.00161,(309159:0.0089,(576508:0.03878,((153308:0.00501,305400:0.01297)0.922.66:0.00683,((((310002:0.00879,572934:0.00486)0.942.80:0.00324,333533:0.01394)0.466.11:0.00554,524142:0.0041)0.788.20:0.00431,((89318:0.01069,(245895:0.01117,(312487:0.00319,328292:0.00478)1.000.1692:0.00549)0.504.7:0.00388)0.943.82:0.00615,(508833:0.01219,(557143:0.0071,(278441:0.00989,247003:0.01118)0.506.5:0.00464)0.596.7:0.00845)0.833.26:0.00379)0.852.56:0.00234)0.811.23:0.0022)0.880.42:0.00207)0.387.11:0.0016)0.959.83:0.00329)0.969.95:0.00316,(361764:0.0049,((510265:0.0187,(1820:0.00944,(537821:0.00314,319263:0.00251)0.851.40:0.00016)0.990.136:0.00465)0.926.69:0.0102,(136866:0.00489,(238873:0.00394,247753:0.00858)0.784.23:0.00314)0.844.27:0.00015)0.690.10:0.00156)0.860.45:0.00148)0.937.71:0.00236)0.883.48:0.00157)0.700.16:0.00016,(242266:0.00738,585214:0.02699)0.782.22:0.00562)0.965.90:0.00356,((104490:0.0055,(93865:0.01482,(550219:0.00903,340199:0.00616)0.562.7:0.00015)0.974.100:0.00515)0.849.35:0.00165,((138816:0.0267,(527683:0.01274,((352324:0.00467,212676:0.01913)0.133.6:0.00016,359837:0.00977)0.827.24:0.0052)0.877.49:0.00257)0.942.81:0.00454,128220:0.01137)0.888.40:0.00255)0.946.66:0.00456)0.802.23:0.00022,193558:0.01287)0.915.58:0.00748,250708:0.02282)0.203.3:0.002,(254728:0.01831,(28670:0.01865,362116:0.01192)0.995.209:0.01529)0.985.112:0.01281)0.876.54:0.00433)0.856.36:0.0033,(161103:0.01995,(51476:0.01391,591433:0.02653)0.790.19:0.00642)0.953.79:0.008)0.704.16:0.00511,(((((209723:0.03597,214042:0.05203)0.989.130:0.01685,((217420:0.04969,204976:0.02108)0.984.107:0.01973,(213329:0.02749,(226288:0.02665,((257702:0.01711,226487:0.01852)0.889.48:0.01234,250365:0.04083)0.320.6:0.02148)0.325.5:0.00511)0.397.13:0.00642)0.952.85:0.00724)0.935.67:0.0083,(((193841:0.02691,(361862:0.01382,(592891:0.0159,((((586102:0.00926,75044:0.02724)0.985.113:0.0064,(18847:0.0112,544080:0.00837)0.817.24:0.01289)0.859.48:0.00186,(247121:0.01608,((((240630:0.0123,510188:0.01226)0.468.9:0.00167,(564241:0.03608,((555182:0.00619,534744:0.01455)0.248.3:0.00252,(512576:0.01262,587357:0.00733)0.918.67:0.00372)0.492.8:0.00131)0.965.91:0.00513)0.965.92:0.00569,113782:0.00341)0.990.137:0.00772,(((586374:0.01205,(551255:0.01412,510034:0.0077)0.434.9:0.00321)0.897.52:0.00373,(203749:0.02383,251649:0.00666)0.862.49:0.00678)0.789.27:0.00193,((256297:0.01463,(212726:0.03392,(141140:0.00873,364386:0.00909)0.171.3:0.00227)0.868.52:0.00173)0.999.483:0.009,279390:0.0129)0.898.57:0.00375)0.784.24:0.00115)0.718.12:0.0008)0.908.55:0.00253)0.982.130:0.00652,(141984:0.02156,(1857:0.01363,245005:0.00632)0.986.129:0.00806)0.994.183:0.01025)0.792.21:0.00181)0.961.81:0.00809)0.759.17:0.00532)0.910.54:0.00421,326559:0.01981)1.000.1693:0.02655,(362913:0.00093,(139345:0.00482,323276:0.00321)0.991.154:0.00936)1.000.1694:0.02084)0.542.7:0.00913)0.904.57:0.00599,((((268255:0.01016,250865:0.02101)0.168.12:0.00206,((236775:0.01454,(144789:0.01623,162113:0.01071)0.939.80:0.00707)0.989.131:0.00934,(205244:0.01736,(((129839:0.00913,221631:0.02441)0.982.131:0.00552,((559897:0.00643,(342790:0.00716,113270:0.00957)0.949.79:0.00469)0.976.97:0.00393,559777:0.00409)0.850.31:0.0079)0.763.24:0.0063,153556:0.00015)0.919.62:0.00498)0.982.132:0.00853)0.907.53:0.00499)0.893.40:0.00249,((1847:0.0093,126401:0.01905)0.969.96:0.00639,(((((((321704:0.00318,128094:0.01369)0.990.138:0.00652,(129976:0.00413,324175:0.02417)0.895.57:0.00138)0.942.82:0.00504,((84930:0.00537,(((310755:0.01433,341778:0.01678)0.772.28:0.00114,(305728:0.01801,(255687:0.01525,(178504:0.0044,(324678:0.00014,583119:0.00633)0.793.18:0.00662)0.697.11:0.00384)0.795.22:0.00234)0.800.28:0.00138)0.756.31:0.00042,350966:0.02063)0.954.86:0.00719)0.063.3:0.00552,254729:0.01591)0.911.59:0.00508)0.962.90:0.00681,(((565780:0.0068,((90452:0.02424,((538113:0.01425,255089:0.0151)0.746.32:0.00223,314799:0.01956)0.474.2:0.00114)0.778.33:0.00355,(540282:0.00646,309002:0.01279)0.957.113:0.0076)0.912.45:0.00309)0.937.72:0.00316,((251031:0.01455,((113404:0.00884,511023:0.00974)1.000.1695:0.00014,(23209:0.01334,336433:0.01318)0.832.24:0.00634)0.897.53:0.00163)0.269.5:0.00262,(579733:0.01658,(589598:0.00563,52982:0.01146)0.974.101:0.00513)0.821.17:0.00164)0.673.6:0.00143)0.932.64:0.0045,(((((((291406:0.01517,347577:0.00265)0.379.14:0.00232,553968:0.0055)0.940.81:0.00324,((331686:0.0071,250320:0.00794)0.815.18:0.00185,540129:0.00295)0.907.54:0.00601)0.935.68:0.00316,(508941:0.01435,(258715:0.0081,((566106:0.00623,352422:0.00771)0.896.50:0.00315,((89660:0.00156,1852:0.01288)0.876.55:0.00156,((562471:0.02129,365485:0.00345)0.971.124:0.00398,114322:0.01061)0.734.19:0.00014)0.818.17:0.00257)0.976.98:0.00533)0.740.30:0.00068)0.860.46:0.00158)0.964.107:0.00414,(89580:0.01009,360823:0.00621)0.941.82:0.00316)0.716.7:0.00155,355862:0.0077)0.857.37:0.00175,(245188:0.02082,113217:0.00014)0.445.3:0.00043)0.863.49:0.00447)0.421.7:0.00757)0.987.174:0.01014,258402:0.00772)0.910.55:0.00382,(88072:0.02166,(76119:0.01563,(((((279512:0.01301,541741:0.0144)0.970.92:0.00538,((155954:0.0095,136848:0.00589)0.908.56:0.00688,327611:0.00426)0.888.41:0.00289)0.371.6:0.00195,((571492:0.0024,156776:0.01484)0.315.4:0.00156,591810:0.01266)0.914.50:0.00396)0.420.8:0.00091,(328111:0.01332,((238869:0.00157,306180:0.01212)0.960.103:0.01604,514381:0.00917)0.942.83:0.0037)0.914.51:0.00256)0.892.44:0.00305,((361264:0.01517,(242275:0.01382,(587313:0.01572,(((((564526:0.01337,(((114260:0.00089,284022:0.00014)0.997.242:0.00899,346493:0.00329)0.857.38:0.00157,((324295:0.00318,76925:0.01598)0.864.48:0.0051,((362645:0.01111,277357:0.00893)0.872.61:0.00229,151517:0.01706)0.756.32:0.00083)0.848.37:0.00235)0.890.56:0.00361)0.934.58:0.00268,(510247:0.01094,(223520:0.00742,(103326:0.01108,207263:0.00608)0.778.34:0.00156)0.851.41:0.00476)0.867.50:0.00486)0.802.24:0.00149,(((316315:0.01428,(319420:0.00853,(75679:0.01826,(308898:0.00325,220968:0.00564)0.279.7:0.00231)0.758.24:0.00323)0.497.10:0.00166)0.938.76:0.00331,220395:0.01322)0.699.6:0.0014,((313245:0.00161,318669:0.00314)0.745.23:0.00691,(((209061:0.0082,550370:0.00693)0.935.69:0.00442,207042:0.0103)0.524.4:0.00361,(273309:0.01997,77025:0.0201)0.440.6:0.0034)0.799.21:0.0016)0.927.70:0.00277)0.350.9:0.005)0.788.21:0.00236,(((306018:0.00161,1833:0.00823)0.806.23:0.0065,511954:0.01465)0.762.26:0.0039,113191:0.01134)0.952.86:0.00483)0.736.26:0.00108,(218467:0.00797,215609:0.00908)0.908.57:0.0077)0.986.130:0.00477)0.841.31:0.00162)0.771.23:0.00081)0.935.70:0.00235,556941:0.01525)0.968.88:0.00016)0.568.8:0.00234)0.217.2:0.00155)0.909.57:0.00773)0.699.7:0.00224,77004:0.01919)0.698.13:0.00433)0.757.33:0.00035)0.852.57:0.00177,((((255091:0.00421,142411:0.01671)0.965.93:0.00828,114093:0.00196)0.980.102:0.00761,((252534:0.01298,249949:0.01786)0.971.125:0.01089,(273538:0.01294,260531:0.02624)0.536.7:0.0069)0.947.76:0.00874)0.935.71:0.00543,(((242816:0.01768,254245:0.01287)0.794.25:0.0038,(276648:0.01566,(((((102133:0.01736,159840:0.00967)0.945.69:0.00638,((547672:0.0085,587581:0.01489)0.987.175:0.00968,248498:0.01319)0.804.26:0.00531)0.924.54:0.0071,(569879:0.02442,159723:0.00695)0.823.21:0.00222)0.977.113:0.01052,(((248253:0.01556,(240765:0.01558,238131:0.01121)0.853.33:0.00326)0.703.12:0.00736,(551614:0.02274,(586657:0.00795,60163:0.01081)0.991.155:0.01107)0.675.7:0.00506)1.000.1696:0.01939,(((235415:0.00967,317900:0.01395)0.206.5:0.00278,322030:0.0173)0.636.5:0.00131,((102608:0.02684,(558733:0.00244,547736:0.00887)0.881.53:0.00321)0.209.5:0.00287,263943:0.0121)0.771.24:0.00122)0.505.8:0.00181)0.976.99:0.01265)0.959.84:0.00892,(238448:0.01055,324810:0.0176)0.913.67:0.00579)0.799.22:0.00528)0.896.51:0.00442)0.990.139:0.00805,(169706:0.02871,(((249325:0.00918,((334113:0.00598,(557097:0.00839,194730:0.01439)0.531.7:0.00641)0.602.6:0.00309,(552050:0.01433,319233:0.00998)0.917.61:0.00392)0.904.58:0.00295)0.765.27:0.00082,343921:0.01146)0.906.63:0.0036,(537715:0.01397,(113311:0.00692,183276:0.01535)0.451.5:0.01182)0.916.61:0.0046)0.998.313:0.01782)0.035:0.00353)0.785.25:0.00166)0.861.51:0.00176)0.946.67:0.00502)0.903.59:0.00472,(567399:0.00368,565220:0.00473)1.000.1697:0.03127)0.993.156:0.01054)0.759.18:0.00208)0.989.132:0.01895)1.000.1698:0.03807)0.497.11:0.01302)0.870.52:0.01034,((((532951:0.02883,((333448:0.00818,((222747:0.01844,(63234:0.01163,545706:0.01031)0.961.82:0.00997)0.956.80:0.00647,584529:0.01183)0.999.484:0.01018)0.736.27:0.00173,((556670:0.01703,553178:0.009)0.861.52:0.00878,173285:0.00967)0.853.34:0.0023)0.878.59:0.00309)0.969.97:0.00575,(237457:0.0177,(226421:0.01715,564867:0.01482)0.963.92:0.00727)0.716.8:0.00115)0.206.6:0.00042,((590326:0.01437,(583865:0.00912,63239:0.01394)0.866.39:0.00338)0.966.90:0.00616,((170193:0.0202,(252937:0.01547,(((252244:0.01916,235692:0.01266)0.308.5:0.00481,(78424:0.01377,543083:0.01437)0.982.133:0.00015)0.571.10:0.00324,(142591:0.01517,113738:0.01691)0.930.65:0.00446)0.773.26:0.00084)0.852.58:0.00014)0.787.25:0.00151,(272259:0.01111,509593:0.02776)0.926.70:0.00555)0.841.32:0.00156)0.144.3:0.00265)1.000.1699:0.04585,(((520382:0.00327,154601:0.01702)0.933.69:0.03953,(242979:0.02051,160549:0.03173)0.081.3:0.00809)0.999.485:0.03419,((((262954:0.01117,513969:0.00312)1.000.1700:0.02539,244907:0.02445)0.995.210:0.01906,((512374:0.02586,(342871:0.00809,323947:0.0111)1.000.1701:0.04081)0.996.231:0.01999,576767:0.06083)0.802.25:0.00703)0.147.4:0.00487,(259584:0.01215,279230:0.02847)1.000.1702:0.02987)0.938.77:0.01531)1.000.1703:0.04107)0.954.87:0.01602)0.956.81:0.01112,(((276636:0.0181,((218783:0.00661,133019:0.01001)0.993.157:0.01279,(220849:0.01656,133702:0.01642)0.881.54:0.00429)0.907.55:0.0046)1.000.1704:0.04195,(270648:0.02782,((248553:0.00382,(213210:0.04288,(238910:0.01058,317628:0.008)0.940.82:0.00407)0.550.11:0.00037)0.998.314:0.02342,(239746:0.01276,253652:0.01289)0.999.486:0.02249)0.969.98:0.01945)1.000.1705:0.06134)0.562.8:0.01409,((267584:0.02356,590912:0.01583)1.000.1706:0.0307,((97685:0.02538,(562230:0.01502,587607:0.01903)0.994.184:0.01523)0.970.93:0.01087,(206727:0.03276,347415:0.02442)0.565.4:0.00805)0.989.133:0.01493)0.995.211:0.02424)0.937.73:0.01763)0.862.50:0.01105,(((((244081:0.00895,574623:0.01411)1.000.1707:0.02409,(204817:0.01779,553535:0.02535)1.000.1708:0.02316)0.972.98:0.01875,(((329074:0.01945,(138615:0.0126,362561:0.01264)0.884.55:0.01216)1.000.1709:0.04083,(251109:0.04917,(255335:0.00858,275375:0.01106)0.994.185:0.01822)0.068.4:0.00931)0.988.153:0.01559,(((508870:0.00887,(239505:0.00834,247920:0.01714)0.791.28:0.00361)0.941.83:0.00702,(277687:0.02484,537795:0.00889)0.974.102:0.01047)1.000.1710:0.03036,(240534:0.0115,(207151:0.0031,252933:0.01927)0.769.21:0.00468)1.000.1711:0.04838)0.880.43:0.00779)0.925.56:0.01181)0.944.66:0.01554,(((98520:0.01862,593606:0.01467)0.356.5:0.00431,(((1710:0.02982,(254631:0.02135,(572063:0.02031,243375:0.01602)0.912.46:0.00795)0.870.53:0.00394)0.945.70:0.00642,(251390:0.03057,308578:0.01702)0.656.4:0.00489)0.936.67:0.00636,((145520:0.02843,(((((318148:0.02342,(((513960:0.01965,(349855:0.03308,(267857:0.01355,(535394:0.01884,((303596:0.01719,591217:0.01824)0.847.33:0.00642,(548591:0.02394,511412:0.01634)0.904.59:0.0086)0.167.3:0.00197)0.812.33:0.00292)0.990.140:0.01043)0.749.22:0.00243)0.848.38:0.00281,((559646:0.01926,538317:0.01891)0.941.84:0.00701,((((311833:0.00782,319376:0.01031)0.935.72:0.00437,(240770:0.00243,263001:0.02162)0.392.6:0.00654)0.895.58:0.00442,(329437:0.00822,581765:0.02487)0.879.55:0.00488)0.199.5:0.00364,(258891:0.02849,(148861:0.0144,338279:0.02515)0.047.2:0.00728)0.866.40:0.00475)0.702.12:0.00482)0.215.7:0.00473)0.123.2:0.00169,1717:0.01993)0.472.2:0.00274)0.882.49:0.00406,589584:0.00756)0.787.26:0.00192,267499:0.01153)0.394.5:0.0017,587824:0.02435)0.682.7:0.00507,266442:0.01762)0.665.10:0.00159)0.861.53:0.00279,(154004:0.02358,508895:0.01379)0.965.94:0.00694)0.912.47:0.00431)0.912.48:0.00839)0.998.315:0.0244,(581733:0.02787,513563:0.0221)1.000.1712:0.04432)0.852.59:0.01018)0.430.8:0.00846,((256935:0.01765,95646:0.0163)0.941.85:0.01215,(249658:0.02038,278012:0.01383)0.816.26:0.00936)1.000.1713:0.04726)0.251.5:0.009)0.966.91:0.01184,(((((146478:0.01511,581273:0.008)0.940.83:0.00746,((63243:0.0131,(259001:0.00544,262036:0.01567)0.921.64:0.00546)0.996.232:0.01604,(571698:0.01304,537353:0.02219)0.541.8:0.00194)0.644.10:0.00552)1.000.1714:0.03709,((272699:0.01072,(262506:0.01101,(558378:0.00921,(((((1716:0.01447,(83489:0.01257,541520:0.02933)0.874.48:0.0022)0.988.154:0.01215,347806:0.00352)0.117.6:0.01878,(318951:0.01056,(574651:0.0059,(310379:0.0063,578054:0.00411)0.469.6:0.00202)0.367.7:0.00979)0.597.9:0.00711)0.574.5:0.00452,(138946:0.02689,(583825:0.0104,556154:0.01373)0.997.243:0.01337)0.668.7:0.00383)0.972.99:0.00538,(249721:0.01281,(200496:0.00616,((563302:0.0072,556802:0.0086)0.985.114:0.0106,(174936:0.00678,571889:0.01769)0.929.75:0.0039)0.768.30:0.00214)0.900.50:0.00275)0.750.23:0.00085)0.874.49:0.00265)0.929.76:0.00435)0.869.51:0.00308)0.386.7:0.00218,524760:0.01166)0.999.487:0.02407)0.995.212:0.023,((((324084:0.01159,278792:0.01726)0.665.11:0.00947,(((113021:0.01624,150710:0.00683)0.979.119:0.00864,((76467:0.02123,237934:0.02175)0.517.5:0.00512,((((348986:0.0128,((218370:0.00992,((223175:0.00827,(209467:0.00938,219348:0.007)0.907.56:0.0118)0.960.104:0.00737,(250693:0.01746,(141653:0.01387,((542461:0.01308,340009:0.00988)0.953.80:0.00604,551166:0.01666)0.728.19:0.00104)0.713.8:0.0051)0.880.44:0.00343)0.299.5:0.00338)0.826.18:0.00388,(509505:0.01776,(238670:0.01449,249729:0.01071)0.564.12:0.00456)0.828.22:0.00341)0.942.84:0.0049)0.864.49:0.00186,((111806:0.01493,((((113723:0.01367,((((112856:0.01155,(151990:0.00926,157110:0.01186)0.122.4:0.00208)0.869.52:0.00211,(581845:0.00348,(((574535:0.02261,(245720:0.00926,587656:0.01007)0.868.53:0.00629)0.743.31:0.00122,586752:0.00238)0.899.57:0.0024,509980:0.01589)0.984.108:0.00691)0.828.23:0.00232)0.755.31:0.00075,(274632:0.01151,(237284:0.00393,144956:0.00757)0.920.55:0.00477)0.888.42:0.00345)0.761.32:0.00185,563979:0.0088)0.789.28:0.00166)0.879.56:0.00237,(572204:0.00863,(248908:0.0038,(251344:0.0138,244993:0.01473)0.751.16:0.00266)0.978.109:0.00623)0.833.27:0.00163)0.968.89:0.00457,272869:0.00518)0.158.4:0.00602,509899:0.00482)0.985.115:0.00825)0.999.488:0.01715,(224859:0.01016,160986:0.00916)0.927.71:0.00495)0.704.17:0.00171)0.989.134:0.00714,(1718:0.02465,542433:0.01636)0.935.73:0.00532)0.341.9:0.00014,(136569:0.01654,586275:0.01166)1.000.1715:0.02051)0.961.83:0.00595)0.324.2:0.00353)0.898.58:0.00799,(143029:0.01756,(565231:0.00866,252625:0.00954)0.997.244:0.01453)0.859.49:0.00783)0.943.83:0.00971)0.999.489:0.03024,(((326537:0.01054,217242:0.01023)0.226.3:0.01079,(148437:0.01325,537719:0.02599)0.608.6:0.0082)1.000.1716:0.04321,((208251:0.00421,100786:0.00559)0.995.213:0.02474,((311763:0.01642,204439:0.00178)0.919.63:0.01827,((((589261:0.00618,(251439:0.03217,1728:0.01198)0.839.39:0.00414)1.000.1717:0.0114,225918:0.01112)0.111.4:0.00014,(251805:0.01576,194029:0.01285)0.000.67:0.00359)0.884.56:0.00364,522476:0.00804)0.887.45:0.00295)0.986.131:0.01791)0.999.490:0.03939)0.993.158:0.02987)0.769.22:0.00629,(169792:0.08633,((269664:0.02423,(241435:0.00727,332316:0.00712)0.940.84:0.00729)0.972.100:0.0108,((((((213078:0.02117,351025:0.01972)1.000.1718:0.0423,(((168493:0.03273,594049:0.01065)1.000.1719:0.01478,((535475:0.02006,(259756:0.00499,(563943:0.00473,155783:0.0272)0.969.99:0.00982)0.997.245:0.01404)0.996.233:0.01201,((351120:0.0177,542627:0.02445)0.701.6:0.00668,(550228:0.01736,532228:0.01706)0.962.91:0.0121)0.764.21:0.00348)0.350.10:0.00274)0.749.23:0.00131,((574895:0.01599,((187788:0.01729,(590289:0.01447,(311554:0.01343,186035:0.00778)0.984.109:0.00721)0.981.111:0.00737)0.953.81:0.00498,251889:0.02311)0.814.19:0.00326)0.684.12:0.00313,319170:0.0195)0.499.7:0.00155)0.839.40:0.00442)0.895.59:0.00667,((558085:0.00947,263572:0.0217)0.985.116:0.00902,(310445:0.01474,(259892:0.01354,(354311:0.01738,147802:0.01425)0.526.7:0.00899)0.914.52:0.00665)0.769.23:0.0022)1.000.1720:0.02103)0.976.100:0.00964,((557689:0.02986,(543301:0.0146,(185747:0.01571,558089:0.007)0.995.214:0.01378)0.810.25:0.00944)0.982.134:0.0155,(76549:0.0198,((155120:0.00958,((224421:0.02569,223577:0.00065)0.982.135:0.00883,(113960:0.01439,313254:0.01964)0.253.3:0.00254)0.864.50:0.00459)0.777.23:0.00984,(217972:0.01654,223804:0.00296)0.894.50:0.00279)0.823.22:0.00328)0.996.234:0.02022)0.983.116:0.01185)0.848.39:0.00265,(139153:0.02176,(204709:0.01844,(73910:0.02344,326715:0.02286)0.993.159:0.01558)0.294.7:0.00286)0.987.176:0.0093)0.932.65:0.00538,((161000:0.00637,((191682:0.02156,537009:0.01178)0.885.53:0.00342,(561492:0.01487,159731:0.01479)0.940.85:0.00493)0.984.110:0.00962)1.000.1721:0.0325,((363109:0.00745,264224:0.02562)1.000.1722:0.02678,(((98176:0.01788,347056:0.02509)0.970.94:0.013,(328268:0.02271,259485:0.02526)0.880.45:0.0088)0.980.103:0.01101,(55462:0.02133,(512134:0.00292,582307:0.0183)0.000.68:0.00495)1.000.1723:0.03354)0.386.8:0.00305)0.994.186:0.01602)0.803.24:0.00693)0.928.63:0.01005)0.999.491:0.02632)0.924.55:0.01241)0.964.108:0.01239)0.855.35:0.01455,(289370:0.0841,(((170960:0.01615,(144303:0.0119,142455:0.01609)0.581.5:0.00613)1.000.1724:0.02987,((((((149184:0.00015,132450:0.01586)0.502.10:0.00316,(317690:0.00337,146600:0.01702)0.655.10:0.00233)0.941.86:0.00529,(((1733:0.01624,556025:0.013)0.590.7:0.00635,76766:0.01311)0.670.4:0.0009,(144899:0.00614,1732:0.02003)0.923.62:0.00418)0.893.41:0.00446)0.718.13:0.00308,((553643:0.0087,1736:0.00349)0.948.83:0.01001,(149516:0.01153,373070:0.01299)0.966.92:0.00918)0.789.29:0.00268)0.893.42:0.00496,319440:0.05157)0.623.5:0.00374,(((433880:0.02034,133176:0.01671)0.421.8:0.00516,542134:0.02332)0.128.5:0.00394,(592179:0.00966,554680:0.01318)0.992.170:0.01237)0.919.64:0.00651)0.797.19:0.00772)0.271.5:0.00982,147757:0.0251)1.000.1725:0.05988)1.000.1726:0.05624)0.763.25:0.01029)0.801.20:0.00357)0.250.5:0.00651,(272230:0.11039,(569418:0.01079,250563:0.00093)0.100.6:0.07496)0.918.68:0.01816)0.808.23:0.01224)0.691.9:0.00861,(((279844:0.01439,510853:0.01015)0.999.492:0.02549,(276607:0.01413,(557031:0.04282,242908:0.01902)0.894.51:0.008)0.980.104:0.01857)1.000.1727:0.07,((((200798:0.02742,((584184:0.01115,206544:0.01119)0.930.66:0.00503,(((172525:0.01449,(1530:0.00815,(46856:0.0055,27466:0.00472)0.739.25:0.00785)0.958.96:0.00506)0.180.4:0.00173,336962:0.01279)0.898.59:0.00288,(99703:0.00575,188164:0.01596)0.838.34:0.00171)0.779.15:0.00336)0.752.31:0.00084)1.000.1728:0.051,(((150698:0.01708,279748:0.00336)1.000.1729:0.04458,((329091:0.02192,514080:0.00723)0.998.316:0.01949,(350705:0.00775,((264255:0.00909,582662:0.02096)0.805.26:0.00523,(277016:0.0207,(347334:0.01476,368981:0.02091)0.122.5:0.00323)0.259.9:0.0024)0.943.84:0.00698)0.920.56:0.01105)0.732.21:0.01123)1.000.1730:0.05109,((((((570897:0.02278,(512051:0.017,(155453:0.01692,(558620:0.01963,(582971:0.01008,563430:0.01325)0.639.9:0.00949)0.961.84:0.01185)0.237.4:0.0023)0.919.65:0.00521)0.845.30:0.00567,540847:0.01604)0.971.126:0.01359,(((514108:0.02236,508980:0.01071)0.822.29:0.0044,549690:0.03916)0.908.58:0.00615,565568:0.01136)0.999.493:0.01664)0.764.22:0.00478,((((167800:0.01702,(63221:0.01635,(249639:0.00254,51852:0.00738)1.000.1731:0.0186)0.650.7:0.0037)0.893.43:0.00544,(577111:0.0181,63218:0.0163)0.867.51:0.00625)0.962.92:0.00746,171789:0.01747)0.455.6:0.00564,(594158:0.01781,(279490:0.01388,585687:0.02313)0.960.105:0.00998)0.929.77:0.00624)0.842.39:0.00371)1.000.1732:0.0262,((((558025:0.01703,((205738:0.0009,245348:0.00105)0.728.20:0.00098,143270:0.03537)1.000.1733:0.02483)0.080.4:0.00623,536366:0.02414)0.977.114:0.00889,(346934:0.02399,566145:0.0147)0.924.56:0.00722)0.685.7:0.00407,((101911:0.02442,157958:0.01286)1.000.1734:0.03273,(526213:0.03809,(593032:0.03212,546628:0.02862)0.970.95:0.01707)0.740.31:0.00998)0.783.25:0.00549)0.983.117:0.01387)0.923.63:0.00928,((((34433:0.0049,149171:0.00265)0.996.235:0.01381,(159929:0.01141,((138089:0.01279,572073:0.01769)0.724.13:0.00284,(590639:0.01365,(534434:0.0294,(555144:0.02959,152196:0.01569)0.645.15:0.00657)0.616.7:0.00228)1.000.1735:0.02682)1.000.1736:0.02063)0.166.6:0.00444)0.849.36:0.00641,(169793:0.00908,143990:0.00777)0.954.88:0.00938)0.976.101:0.01295,(1513:0.01834,(304122:0.00015,564205:0.02493)0.993.160:0.02196)1.000.1737:0.05384)0.858.33:0.00901)0.958.97:0.02089)0.902.44:0.02232)1.000.1738:0.05062,(((157746:0.00177,(((203678:0.00344,((347059:0.0081,311765:0.01208)0.917.62:0.00481,(146603:0.01077,159816:0.01126)0.720.16:0.00089)0.953.82:0.00384)1.000.1739:0.01494,162578:0.00014)1.000.1740:0.00014,184252:0.00413)0.929.78:0.0234)0.981.112:0.02491,((143919:0.0187,(221094:0.00273,28578:0.02425)0.933.70:0.00803)0.988.155:0.01002,(199656:0.0054,34533:0.00315)0.816.27:0.00093)0.894.52:0.01246)1.000.1741:0.09528,((210132:0.03333,123055:0.02009)0.999.494:0.03312,((241116:0.01675,(142598:0.01248,(76201:0.02489,203943:0.0074)0.967.89:0.00454)0.849.37:0.00013)1.000.1742:0.03519,(((104647:0.02937,((1443:0.01122,((192761:0.01281,241564:0.00265)0.784.25:0.00226,((152496:0.00617,542056:0.01046)0.829.23:0.00407,((246040:0.02183,1449:0.00567)0.824.19:0.00314,541711:0.01457)0.746.33:0.00078)0.880.46:0.00217)0.971.127:0.00487)0.509.6:0.00288,279215:0.03305)0.925.57:0.00611)0.822.30:0.00593,(537196:0.04339,(237590:0.01646,(237055:0.01428,(260626:0.02015,(536233:0.00786,316621:0.02947)0.917.63:0.00501)0.726.18:0.0027)0.883.49:0.00636)0.768.31:0.0089)0.928.64:0.01149)1.000.1743:0.02602,((269173:0.01414,(136664:0.00997,136585:0.02246)0.998.317:0.01655)0.810.26:0.00568,(((276859:0.01718,159592:0.01044)0.920.57:0.00447,(511523:0.01173,102418:0.0157)0.841.33:0.00285)0.210.6:0.00197,((246188:0.02056,257467:0.02634)0.930.67:0.013,((106808:0.01199,(332256:0.00796,159639:0.01043)0.890.57:0.00876)0.769.24:0.00717,(((((170041:0.00391,1468:0.00029)1.000.1744:0.00497,(264343:0.00901,(518198:0.01567,(((110103:0.02316,1456:0.00669)0.400.7:0.00154,(109789:0.00904,111105:0.00406)0.868.54:0.00225)0.715.13:0.00029,(171435:0.00092,1490:0.00397)0.771.25:0.00809)0.869.53:0.00158)0.742.29:0.00067)0.868.55:0.00313)0.060:0.00411,((((321513:0.0236,589856:0.01504)0.916.62:0.00593,(139106:0.00971,141312:0.02962)0.776.29:0.00289)0.941.87:0.01161,242147:0.0135)0.945.71:0.00465,((570844:0.00939,173004:0.01178)0.768.32:0.00127,240878:0.01304)0.903.60:0.00406)0.912.49:0.00543)0.908.59:0.00835,102911:0.04328)0.923.64:0.01079,((113562:0.00091,524254:0.00016)0.862.51:0.00269,((583840:0.02211,1463:0.00459)0.844.28:0.00768,1473:0.00664)0.822.31:0.01071)0.999.495:0.02153)0.960.106:0.01069)0.978.110:0.01399)0.344.4:0.00016)0.879.57:0.00918)0.735.24:0.0057)0.990.141:0.01817)1.000.1745:0.0346)0.853.35:0.01603)0.983.118:0.02459)0.907.57:0.00987,((((589700:0.00816,((244996:0.01559,242324:0.01381)0.458.5:0.00271,(551656:0.00923,(138343:0.01468,252035:0.01127)0.964.109:0.00592)0.915.59:0.00538)0.990.142:0.00865)0.785.26:0.00419,543239:0.02509)0.280.7:0.00413,586974:0.00981)1.000.1746:0.04873,(63250:0.02024,(28363:0.02594,(138554:0.01509,538206:0.0213)0.669.9:0.00707)0.042.2:0.00505)1.000.1747:0.05899)0.978.111:0.02046)0.796.24:0.0152)0.969.100:0.01526)0.981.113:0.01134,((338777:0.01086,259879:0.01521)1.000.1748:0.09017,((253719:0.05306,(((246264:0.02065,574286:0.01635)1.000.1749:0.04938,(105731:0.02321,(271769:0.02586,273634:0.0132)0.650.8:0.00669)1.000.1750:0.04134)0.995.215:0.02647,((((98326:0.01684,140983:0.0152)0.986.132:0.01107,((540822:0.02322,274620:0.02373)0.898.60:0.0062,339589:0.03088)0.395.10:0.00272)0.771.26:0.00777,(((((151879:0.01053,567584:0.01721)0.929.79:0.00484,(246577:0.02947,215896:0.02449)0.686.9:0.00621)0.997.246:0.01836,(275593:0.01785,262293:0.01522)0.591.8:0.00802)0.848.40:0.00455,269228:0.01792)0.886.44:0.00493,(((161392:0.01506,513991:0.01784)0.924.57:0.00679,(260159:0.02343,(((559017:0.00394,154348:0.00162)0.764.23:0.0202,154118:0.00471)0.799.23:0.0299,125448:0.01351)1.000.1751:0.02105)0.187.8:0.00214)0.417.7:0.00263,304240:0.04308)0.624.7:0.00352)0.916.63:0.00703)1.000.1752:0.04704,(540009:0.08973,(28255:0.04336,573924:0.02394)0.995.216:0.02603)0.980.105:0.01992)0.984.111:0.01992)0.819.23:0.01267)0.907.58:0.00817,((221993:0.01713,(235482:0.01133,((222448:0.03034,63111:0.0112)0.146.4:0.00191,(274944:0.01536,269081:0.00644)0.898.61:0.00421)0.936.68:0.00617)0.940.86:0.00653)1.000.1753:0.11491,(155196:0.03305,((146139:0.0218,(105663:0.01701,327363:0.01272)0.947.77:0.00785)0.936.69:0.00927,(100278:0.02926,(572965:0.01415,583062:0.01011)0.984.112:0.01243)0.919.66:0.00819)0.998.318:0.01982)0.996.236:0.03192)0.976.102:0.01823)0.742.30:0.00886)0.581.6:0.007)0.817.25:0.00328)0.914.53:0.00765,(((247675:0.1028,(268940:0.00778,151277:0.01865)1.000.1754:0.06837)0.996.237:0.03008,((541004:0.0119,265819:0.01095)1.000.1755:0.06825,(275968:0.04309,((554824:0.01544,254018:0.01437)0.997.247:0.02336,(((132726:0.0461,((263387:0.01122,591704:0.02891)0.963.93:0.01301,((154578:0.0146,539167:0.00881)1.000.1756:0.02609,(582305:0.01044,546200:0.01596)0.936.70:0.00931)0.999.496:0.02072)0.982.136:0.01591)0.919.67:0.01044,((269047:0.01855,7633:0.02339)0.999.497:0.01772,(306314:0.04195,(254500:0.00348,(238328:0.01541,(514019:0.01107,536891:0.00014)0.407.7:0.02456)0.862.52:0.0094)0.999.498:0.01889)0.919.68:0.0088)0.336.13:0.00508)0.769.25:0.00362,(66921:0.03419,(((540736:0.00881,539077:0.00863)0.996.238:0.01648,(552017:0.00648,568248:0.01882)0.961.85:0.00879)0.994.187:0.01762,(574695:0.03522,541641:0.02341)0.990.143:0.01816)0.996.239:0.024)0.939.81:0.01071)0.971.128:0.0162)0.989.135:0.01881)1.000.1757:0.05784)0.905.63:0.01088)0.964.110:0.01878,(((((74751:0.10921,((30354:0.02785,239678:0.0116)1.000.1758:0.08304,(255060:0.06562,((152394:0.00871,(247643:0.00567,103366:0.0162)0.840.24:0.00453)1.000.1759:0.03478,(39225:0.01752,(143475:0.01868,82144:0.01947)0.946.68:0.00753)0.998.319:0.02894)1.000.1760:0.05132)0.195.2:0.01204)0.270.3:0.01688)0.995.217:0.03327,(553102:0.09933,((589068:0.02715,(513329:0.01523,(582161:0.00844,551137:0.00606)0.988.156:0.0049)0.803.25:0.01261)1.000.1761:0.09802,(253575:0.01283,237283:0.03282)1.000.1762:0.05199)0.124.5:0.02128)1.000.1763:0.05249)0.439.12:0.01352,(63284:0.01932,591575:0.03746)1.000.1764:0.1195)0.947.78:0.01543,(((221640:0.02667,351571:0.01995)0.967.90:0.01833,((158359:0.01485,(227606:0.01332,(283688:0.01101,(298479:0.01396,350822:0.01066)0.087.2:0.00109)0.970.96:0.00906)1.000.1765:0.01867)0.994.188:0.01689,((78123:0.00586,(152729:0.00648,562766:0.02564)1.000.1766:0.04553)0.753.32:0.0022,563413:0.01335)0.995.218:0.01403)0.998.320:0.03017)1.000.1767:0.06746,((((((247500:0.01778,567353:0.03345)0.999.499:0.0268,((228206:0.0257,161669:0.02491)0.995.219:0.01925,(553862:0.00901,(541341:0.03281,525404:0.01402)0.687.11:0.00751)0.999.500:0.02294)0.888.43:0.00723)1.000.1768:0.04776,((((547431:0.02,511928:0.02967)1.000.1769:0.04076,(512989:0.05761,(348335:0.02529,(252204:0.02405,((33594:0.01853,567951:0.01047)0.984.113:0.01224,((543219:0.00504,(286220:0.00933,195747:0.01912)0.915.60:0.00705)0.987.177:0.01376,(330990:0.01546,565629:0.0144)0.271.6:0.004)0.965.95:0.00755)0.942.85:0.00801)0.984.114:0.01373)0.981.114:0.02252)0.553.9:0.01774)1.000.1770:0.02932,((229278:0.03083,538765:0.01576)0.990.144:0.01627,(324346:0.01443,((148962:0.02813,513149:0.01614)0.999.501:0.01876,(535561:0.0229,(130730:0.0299,(255057:0.01583,535361:0.01303)0.614.6:0.00679)0.991.156:0.01379)0.100.7:0.00216)0.712.11:0.00563)0.984.115:0.01412)1.000.1771:0.03409)0.415.4:0.00574,(((581317:0.0069,(267092:0.00366,264692:0.02793)0.505.9:0.00648)0.589.7:0.00704,(332707:0.01226,159595:0.0305)0.906.64:0.00858)1.000.1772:0.06934,(((((164821:0.0125,315579:0.01366)1.000.1773:0.02516,((572696:0.01618,558534:0.01503)0.971.129:0.01085,(168703:0.03109,(568893:0.00807,529621:0.01814)0.994.189:0.01393)0.800.29:0.00736)0.964.111:0.0131)1.000.1774:0.03139,(562684:0.03667,552555:0.01699)1.000.1775:0.03853)0.609.5:0.00716,((63782:0.02306,555839:0.0213)0.998.321:0.02421,(((28793:0.02611,242108:0.01885)0.971.130:0.01344,(45661:0.01375,246491:0.02911)0.997.248:0.01935)0.999.502:0.0238,((204723:0.01984,(223449:0.01036,581655:0.01903)0.700.17:0.00739)0.999.503:0.0304,(((240877:0.02208,(556217:0.00479,(((586031:0.00662,(241451:0.02067,277712:0.01105)0.897.54:0.00783)0.684.13:0.00189,((185827:0.00971,((570966:0.01864,144091:0.01487)0.890.58:0.01003,252958:0.01409)0.694.6:0.00248)0.908.60:0.00396,((259155:0.0133,587599:0.00764)0.969.101:0.00653,(211083:0.00766,553028:0.01736)0.873.52:0.00266)0.824.20:0.00014)0.980.106:0.00654)0.913.68:0.00418,22777:0.01947)0.885.54:0.00688)0.678.11:0.00512)0.419.5:0.00308,540716:0.0234)1.000.1776:0.03673,(314857:0.06476,(243244:0.00663,113334:0.02436)1.000.1777:0.03495)0.996.240:0.02991)0.740.32:0.01461)0.813.26:0.01228)0.982.137:0.01709)0.628.12:0.00918)0.996.241:0.02177,(171876:0.04841,(231901:0.02124,129513:0.01768)0.999.504:0.03023)0.917.64:0.01009)0.962.93:0.01251)0.724.14:0.01187)0.973.94:0.01824)0.888.44:0.00841,((((536135:0.01535,(63773:0.02446,526723:0.01591)0.431.8:0.00586)0.994.190:0.01466,563562:0.02317)0.806.24:0.00627,290797:0.01365)1.000.1778:0.09216,(580844:0.02379,(510686:0.01513,279399:0.01123)0.976.103:0.01162)1.000.1779:0.04737)0.515.11:0.00693)0.479.12:0.0066,((((274093:0.01377,252783:0.02067)0.984.116:0.01227,((131965:0.01452,(133676:0.01197,259885:0.01164)0.998.322:0.01555)0.938.78:0.00615,(158922:0.01631,242481:0.0346)0.621.10:0.00413)0.983.119:0.00954)0.887.46:0.00714,((586441:0.02192,(590821:0.02308,(266329:0.00874,536066:0.01229)0.039.2:0.00329)0.961.86:0.0107)0.992.171:0.01661,(32598:0.032,540069:0.02399)0.971.131:0.01466)0.502.11:0.00264)1.000.1780:0.05751,(172911:0.01639,260737:0.02042)1.000.1781:0.05131)0.965.96:0.01666)0.995.220:0.01829,((((592392:0.00351,106114:0.01314)1.000.1782:0.02357,536025:0.02889)1.000.1783:0.04988,(242733:0.07072,103111:0.09027)0.999.505:0.03578)0.993.161:0.0205,(((585963:0.01672,251965:0.01929)1.000.1784:0.06701,(((547820:0.01875,564815:0.01565)0.999.506:0.0326,((204759:0.01888,254199:0.02347)1.000.1785:0.04924,(277175:0.01971,539574:0.01197)1.000.1786:0.04063)0.487.7:0.01255)0.926.71:0.0138,(252178:0.08331,((273867:0.02649,256848:0.02617)0.940.87:0.0176,(357229:0.03472,(405181:0.03847,(258358:0.01282,523166:0.01045)0.639.10:0.01434)0.968.90:0.01541)0.998.323:0.02818)0.872.62:0.0236)0.976.104:0.01909)0.696.12:0.00747)0.996.242:0.02584,582962:0.08165)0.955.78:0.01663)0.981.115:0.01532)0.840.25:0.00783)0.967.91:0.01428)0.552.9:0.00388,(((536466:0.04789,((101576:0.00492,111938:0.0173)0.973.95:0.01432,151718:0.00582)0.999.507:0.0481)1.000.1787:0.11149,(((((((((((105129:0.02229,278462:0.02227)0.998.324:0.02671,(270455:0.02511,(310512:0.01529,(341545:0.00835,278788:0.01377)0.974.103:0.00819)0.832.25:0.0097)1.000.1788:0.03656)0.995.221:0.02431,(((((241161:0.01664,135899:0.02853)1.000.1789:0.04988,((256775:0.01937,77840:0.0174)0.997.249:0.01922,(247779:0.02459,143040:0.02782)0.973.96:0.01661)1.000.1790:0.03959)0.990.145:0.02937,(((155355:0.00482,334374:0.00559)0.928.65:0.00597,(545467:0.01024,353683:0.01417)0.973.97:0.00922)0.860.47:0.00846,((514012:0.02522,((41135:0.00614,250508:0.01968)1.000.1791:0.01669,234581:0.01365)0.831.18:0.00157)0.917.65:0.00381,234675:0.00821)0.953.83:0.00694)0.995.222:0.01943)0.688.6:0.00968,((238901:0.01055,257229:0.00455)0.951.72:0.06079,((137292:0.01538,543373:0.02278)1.000.1792:0.03269,((244979:0.03439,(229594:0.016,((231268:0.00233,119942:0.00775)0.955.79:0.00834,152827:0.02321)0.938.79:0.00506)0.842.40:0.00331)0.999.508:0.03699,((154577:0.01662,159413:0.01324)0.872.63:0.00346,(161645:0.01543,91973:0.00944)0.980.107:0.00822)1.000.1793:0.0374)0.921.65:0.02074)0.321.6:0.01008)0.614.7:0.00712)0.979.120:0.01387,((230340:0.00264,(230425:0.0033,234599:0.00978)0.970.97:0.00715)1.000.1794:0.05804,153206:0.03812)0.984.117:0.0168)0.451.6:0.00471)0.939.82:0.01711,((577212:0.0119,(111504:0.02769,305871:0.01361)0.746.34:0.00509)1.000.1795:0.03931,((1556:0.01443,588078:0.01687)0.661.10:0.01464,(578464:0.02468,147284:0.01982)0.557.3:0.00857)0.999.509:0.04259)1.000.1796:0.05454)0.997.250:0.02428,(((((1559:0.00533,(89313:0.00455,(142233:0.0153,(1549:0.00376,557073:0.00637)0.929.80:0.00596)0.802.26:0.00411)0.962.94:0.00572)0.822.32:0.00176,(((350772:0.01356,153174:0.01129)0.963.94:0.00614,(231447:0.00503,208430:0.00092)1.000.1797:0.02804)0.948.84:0.00416,(567922:0.00158,(246008:0.02344,136998:0.00437)0.817.26:0.0015)0.883.50:0.00365)0.948.85:0.00372)0.977.115:0.0069,((((144058:0.00474,555078:0.00015)0.986.133:0.00866,(526112:0.01409,((100737:0.01197,(220491:0.00637,584845:0.01292)0.685.8:0.01234)0.934.59:0.00595,210320:0.01034)0.918.69:0.00651)0.970.98:0.00799)0.996.243:0.00936,216828:0.00338)0.991.157:0.01258,(167091:0.02021,(144437:0.02482,(554044:0.00015,553713:0.00497)1.000.1798:0.02458)0.715.14:0.00601)0.703.13:0.00457)0.868.56:0.00404)0.873.53:0.0057,((18774:0.02077,96609:0.0229)0.890.59:0.00674,(245910:0.02233,(((114788:0.0377,242264:0.02543)0.163.4:0.01336,327499:0.03335)0.948.86:0.0079,(38097:0.02697,566392:0.01422)0.332.4:0.00496)0.867.52:0.00302)0.948.87:0.00686)0.746.35:0.00548)0.938.80:0.00685,(237121:0.02599,106739:0.03028)0.799.24:0.00446)0.997.251:0.02092)0.864.51:0.01177,(9675:0.01652,513750:0.02021)0.991.158:0.02007)1.000.1799:0.06687,((((274373:0.02324,511517:0.02006)0.992.172:0.01806,(260358:0.00015,301652:0.03846)0.926.72:0.02943)1.000.1800:0.06244,(534533:0.07373,((536133:0.00506,285033:0.01643)1.000.1801:0.06545,576499:0.08424)0.476.5:0.0138)0.919.69:0.01868)0.850.32:0.02987,((((((((159582:0.0177,(269422:0.02084,549372:0.01562)0.139.8:0.00725)0.999.510:0.02848,((((148371:0.00854,544574:0.02694)1.000.1802:0.03298,(((((55975:0.01595,565073:0.01644)0.999.511:0.01884,(((((319505:0.00419,((536890:0.00893,155636:0.02964)0.644.11:0.00029,170976:0.0047)0.975.110:0.01012)0.757.34:0.00061,((((247354:0.00884,526724:0.00729)0.930.68:0.00379,(46876:0.03066,535322:0.00386)1.000.1803:0.01908)0.912.50:0.00356,((104164:0.00554,510823:0.00647)0.901.50:0.00236,(332679:0.01718,((589956:0.0113,568448:0.00313)0.942.86:0.00395,((562639:0.01431,(512867:0.00157,(546058:0.01289,239291:0.02626)0.024.4:0.00835)0.997.252:0.00999)0.338.5:0.00091,(552512:0.01312,(106097:0.01322,528651:0.00733)0.798.19:0.00382)0.857.39:0.00202)0.972.101:0.00355)0.349.6:0.00147)0.745.24:0.00081)0.935.74:0.00576)0.116.9:0.00087,105709:0.00717)0.899.58:0.00475)0.963.95:0.00562,(111348:0.03624,555601:0.00397)0.997.253:0.00961)0.835.27:0.00301,(((251189:0.00867,562802:0.01511)0.676.11:0.00467,((532430:0.00317,(101704:0.01361,(140683:0.00857,540464:0.01353)0.864.52:0.00647)0.993.162:0.0083)0.665.12:0.00251,153529:0.02951)0.948.88:0.00402)0.599.8:0.00101,540412:0.01991)0.966.93:0.00746)1.000.1804:0.01999,((142471:0.01152,(341698:0.01519,(255112:0.00296,527033:0.0119)0.988.157:0.00777)0.864.53:0.00242)0.999.512:0.01883,((249874:0.02017,62479:0.0261)0.877.50:0.01054,(252550:0.0203,(235471:0.01629,(153007:0.01182,249520:0.01868)0.779.16:0.00338)0.936.71:0.00925)0.334.4:0.00682)0.951.73:0.0095)0.356.6:0.00396)0.992.173:0.01276)0.996.244:0.01641,((564853:0.01653,((580227:0.00811,(585845:0.01846,161217:0.01006)0.659.8:0.00493)0.104:0.00345,155518:0.01436)0.983.120:0.01616)1.000.1805:0.03828,((((267867:0.00584,152048:0.02048)0.753.33:0.00621,362235:0.01151)0.989.136:0.00816,(324239:0.00843,(134657:0.00343,535743:0.01537)0.899.59:0.00457)0.662.9:0.00234)0.866.41:0.0218,(329673:0.02404,(((570408:0.01164,247197:0.00096)0.964.112:0.00856,(151337:0.01807,152550:0.02415)0.328.4:0.00432)0.891.56:0.00684,235366:0.02429)0.835.28:0.01126)0.994.191:0.01614)0.995.223:0.01748)0.864.54:0.0078)0.211.5:0.00499,(((533261:0.014,560235:0.02153)0.905.64:0.00546,(154124:0.01457,223134:0.02538)0.573.6:0.00591)0.996.245:0.01612,585906:0.05349)0.655.11:0.00585)1.000.1806:0.02756,((586112:0.00422,517793:0.00862)0.817.27:0.03768,(200743:0.02067,(565525:0.01614,510838:0.00866)0.727.15:0.00296)0.998.325:0.0183)0.931.60:0.01171)0.584.7:0.00896)0.453.4:0.00775,(((264509:0.02637,((((159169:0.01962,(237007:0.02654,524039:0.00261)0.967.92:0.01031)1.000.1807:0.02276,((347995:0.01959,(569018:0.01737,(256408:0.01254,38408:0.01494)0.099.2:0.0076)0.784.26:0.00498)0.864.55:0.0061,(346553:0.01775,(587060:0.02579,586538:0.00378)0.955.80:0.00793)0.984.118:0.01059)0.899.60:0.00699)0.767.38:0.00419,45741:0.02245)0.839.41:0.00364,(323497:0.0179,237232:0.0303)0.822.33:0.00554)0.997.254:0.01542)0.634.7:0.00256,308686:0.01442)0.979.121:0.01206,584159:0.04245)0.977.116:0.01115)0.982.138:0.01014,562351:0.03138)0.968.91:0.00882)0.880.47:0.00966,(((301351:0.01983,299422:0.01176)1.000.1808:0.03058,((291813:0.01842,(301881:0.01335,300911:0.00628)0.733.21:0.01608)0.975.111:0.00947,((334614:0.01884,(299809:0.01138,(297340:0.01625,300484:0.0233)0.964.113:0.00565)0.981.116:0.00835)0.913.69:0.00398,(295498:0.00361,302501:0.02326)0.739.26:0.01812)0.239.4:0.00382)0.939.83:0.01043)1.000.1809:0.03707,((287314:0.00118,((561987:0.02706,533891:0.03586)0.764.24:0.01483,192963:0.02291)0.869.54:0.00613)0.850.33:0.00308,178399:0.06146)0.894.53:0.01585)1.000.1810:0.06545)0.944.67:0.01131,((((540160:0.01894,523568:0.0052)0.974.104:0.01129,((113378:0.01271,((509900:0.01536,(102890:0.01132,(254951:0.01121,(570097:0.00899,564314:0.00754)0.742.31:0.00491)0.786.27:0.00173)0.953.84:0.00566)0.833.28:0.00376,(319751:0.01737,((556878:0.00546,200446:0.01696)0.929.81:0.00367,248041:0.00423)0.827.25:0.00327)0.270.4:0.00481)0.810.27:0.01034)0.997.255:0.0148,(91270:0.0179,(510359:0.0168,512820:0.00924)0.942.87:0.00909)0.913.70:0.0085)0.998.326:0.01704)0.882.50:0.00986,((537149:0.00177,549833:0.01388)1.000.1811:0.03526,(35008:0.0192,(((314513:0.02095,591617:0.00906)0.275.2:0.00685,246208:0.01653)0.938.81:0.00863,102039:0.01142)0.244.4:0.00498)1.000.1812:0.05452)0.877.51:0.01148)1.000.1813:0.02626,525126:0.0695)0.472.3:0.01153)0.918.70:0.01026,519950:0.04057)0.962.95:0.01992,((((231528:0.01346,(123679:0.00451,97376:0.00478)0.967.93:0.00742)1.000.1814:0.0247,(570886:0.03298,((517031:0.01799,(243512:0.01483,(519253:0.02112,221896:0.00784)0.234.3:0.00454)0.963.96:0.00746)0.980.108:0.00946,((512172:0.01113,(509040:0.01941,587216:0.01087)0.712.12:0.00578)0.984.119:0.01287,((244496:0.01253,(267722:0.0128,578508:0.01856)0.985.117:0.00779)0.954.89:0.00594,(256945:0.01469,(589110:0.01702,146126:0.01106)0.367.8:0.00621)0.822.34:0.00283)0.996.246:0.01465)0.984.120:0.01187)0.981.117:0.01018)0.768.33:0.00303)0.874.50:0.0051,(((320201:0.02137,155782:0.04692)1.000.1815:0.03982,(((((593896:0.00797,551791:0.01211)0.903.61:0.00475,(221946:0.00637,(252964:0.01277,((154899:0.0165,225353:0.00249)0.525.6:0.00155,((226992:0.00662,(216556:0.01081,(227051:0.01141,256946:0.01159)0.000.69:0.0036)0.840.26:0.00016)0.961.87:0.00452,(((206493:0.02834,217322:0.0265)0.490.6:0.00589,(((154102:0.00073,238853:0.01283)0.965.97:0.00531,217594:0.01022)0.514.6:0.00282,(352632:0.00815,241683:0.01672)0.398.3:0.00253)0.854.36:0.00345)0.766.23:0.00364,((((((573812:0.00639,533444:0.01458)0.452.13:0.00396,(243516:0.01459,153622:0.00473)0.918.71:0.00803)0.817.28:0.00368,(222701:0.00824,((246641:0.0176,(226045:0.00454,((((220494:0.00578,220859:0.00821)0.742.32:0.00099,214730:0.00718)0.855.36:0.00237,((((100320:0.00474,(225313:0.00789,3046:0.01001)0.755.32:0.00464)0.779.17:0.0022,226369:0.0092)0.479.13:0.00174,225633:0.00594)0.851.42:0.0031,245203:0.02043)0.826.19:0.00175)0.985.118:0.005,214739:0.01274)0.672.5:0.00193)0.895.60:0.00645)0.968.92:0.00743,338457:0.00798)0.973.98:0.00779)0.828.24:0.00503)0.473.8:0.00218,547960:0.02244)0.856.37:0.00201,(((335630:0.00735,515684:0.009)0.384.7:0.00376,(340507:0.01786,559200:0.0148)0.854.37:0.00767)0.972.102:0.008,(225899:0.0074,((545110:0.01373,(583508:0.01663,(1605:0.00948,(538394:0.01954,544067:0.00808)0.694.7:0.00192)1.000.1816:0.0017)0.858.34:0.01128)0.811.24:0.00234,(240862:0.00152,(82424:0.01043,584155:0.00409)0.705.5:0.00476)0.333.5:0.00597)0.982.139:0.00602)0.900.51:0.00311)0.945.72:0.00419)0.927.72:0.0038,(213519:0.01264,(320988:0.00703,582528:0.00932)0.593.5:0.00241)0.739.27:0.00092)0.902.45:0.00262)0.952.87:0.00015)0.987.178:0.00587)0.296.5:0.00426)0.873.54:0.00296)0.861.54:0.00194)0.858.35:0.00209,((212840:0.0097,(((248485:0.01119,309788:0.00156)0.871.54:0.00356,256977:0.00919)0.951.74:0.00498,256227:0.01591)0.458.6:0.0021)0.979.122:0.00704,(((209352:0.02209,((254530:0.00568,221749:0.00858)0.982.140:0.00714,((245490:0.00881,((255611:0.00315,222501:0.02216)0.488.9:0.00014,152808:0.01096)0.944.68:0.00315)0.756.33:0.00073,205456:0.00568)0.966.94:0.00488)0.767.39:0.00427)0.997.256:0.01109,544465:0.00759)0.239.5:0.00163,283225:0.00257)0.999.513:0.01449)0.957.114:0.00461)0.910.56:0.00347,(253005:0.01025,(((346909:0.00711,512364:0.01029)0.383.3:0.00266,240809:0.00653)0.816.28:0.00762,(539636:0.00552,153320:0.0063)0.895.61:0.0018)0.958.98:0.01292)0.851.43:0.00752)0.820.14:0.00779,((243353:0.00343,((206632:0.00599,(201417:0.00873,252778:0.02326)0.906.65:0.00433)0.891.57:0.00461,(237783:0.01846,(244491:0.00848,136502:0.01673)0.999.514:0.01781)0.840.27:0.00362)1.000.1817:0.02111)0.971.132:0.01237,235848:0.01024)0.829.24:0.00785)0.680.10:0.00476)0.947.79:0.01071,((224503:0.00966,106151:0.0437)1.000.1818:0.05587,((((236458:0.01421,((354713:0.03602,(142335:0.02398,(160871:0.0267,((152506:0.00723,207125:0.0072)0.493.6:0.00187,512523:0.00901)0.986.134:0.00856)0.827.26:0.00475)0.617.8:0.00466)0.862.53:0.0043,214306:0.01655)0.948.89:0.005)0.950.96:0.00773,((249859:0.01809,358664:0.01043)0.781.22:0.00409,538238:0.04784)0.879.58:0.00487)0.814.20:0.00536,((270180:0.00807,(((221511:0.00237,588961:0.01474)0.891.58:0.0042,205420:0.00528)0.131.8:0.00614,(220364:0.01285,535268:0.01606)0.406.6:0.00234)0.991.159:0.01064)0.952.88:0.00709,(579861:0.01031,((560184:0.01075,(251499:0.01352,104784:0.0063)0.742.33:0.00061)0.754.23:0.00079,356605:0.01335)0.935.75:0.00305)0.005:0.00651)0.943.85:0.00464)0.908.61:0.00326,((512674:0.00678,(553562:0.02428,(578410:0.01082,(511873:0.00276,410272:0.00366)0.970.99:0.00804)0.995.224:0.01405)0.614.8:0.00391)1.000.1819:0.03175,((541789:0.00556,(216255:0.0014,(115085:0.00279,248216:0.01304)1.000.1820:0.02585)0.784.27:0.02486)0.581.7:0.01045,511817:0.06559)0.976.105:0.01188)0.909.58:0.00605)0.942.88:0.00904)0.719.8:0.00667)0.966.95:0.01069)0.977.117:0.01593,564160:0.04458)0.085.5:0.0156)0.394.6:0.01777,253042:0.10236)0.987.179:0.029,((((((134130:0.0077,21798:0.03528)1.000.1821:0.03116,(99144:0.0229,(560147:0.01157,(337426:0.00614,526598:0.01381)0.797.20:0.00194)1.000.1822:0.02664)0.907.59:0.01642)0.968.93:0.01615,(110302:0.02007,578036:0.01665)1.000.1823:0.07688)0.971.133:0.01629,(((574533:0.02872,(159970:0.01756,570490:0.01724)0.939.84:0.00963)0.820.15:0.00634,550907:0.07282)0.610.6:0.00788,(((((239430:0.03961,((237237:0.01344,(131444:0.00501,246713:0.00549)0.908.62:0.00249)0.262.5:0.02456,((254690:0.00783,579528:0.02417)0.997.257:0.02373,250649:0.05276)0.965.98:0.00944)0.842.41:0.00539)0.978.112:0.01096,(((((((331129:0.01224,(235693:0.01231,536090:0.00061)0.988.158:0.00955)0.975.112:0.00749,222261:0.00771)0.760.28:0.00229,(238183:0.02669,222914:0.00644)0.747.22:0.00169)0.985.119:0.00672,(((((149969:0.01462,(590842:0.01545,589297:0.02625)0.628.13:0.00298)0.307.5:0.00388,111959:0.00904)0.837.23:0.0025,(578045:0.01408,161448:0.0119)0.871.55:0.00407)0.866.42:0.0016,(((308202:0.01417,(((210788:0.01093,217650:0.00409)0.903.62:0.00333,(218224:0.0078,(212455:0.00842,217739:0.02752)0.881.55:0.00936)0.913.71:0.00399)0.115.4:0.00123,222219:0.00363)0.296.6:0.00341)0.849.38:0.00215,152799:0.01261)0.893.44:0.00527,137061:0.00732)0.698.14:0.00085)0.507.6:0.00083,(((539821:0.005,(151664:0.01811,241839:0.01087)0.997.258:0.01122)0.967.94:0.00614,((223808:0.00576,353925:0.01291)0.875.44:0.00627,583875:0.00669)0.916.64:0.00345)0.759.19:0.001,242989:0.01047)0.945.73:0.00319)0.887.47:0.00341)0.987.180:0.0107,((235541:0.00839,590872:0.01503)0.941.88:0.00625,(217929:0.00983,(107428:0.02676,((220892:0.03604,(113303:0.01641,(110756:0.01239,569190:0.0068)0.806.25:0.00372)0.415.5:0.00406)0.102.4:0.00208,(217883:0.00881,221568:0.02729)0.810.28:0.00867)0.914.54:0.00631)0.990.146:0.00971)0.947.80:0.00762)0.924.58:0.00577)0.862.54:0.00708,(221251:0.01537,(250253:0.02124,((220118:0.0126,223550:0.00803)0.995.225:0.01501,219094:0.01247)0.964.114:0.00986)0.798.20:0.0078)0.873.55:0.00494)0.986.135:0.00985,(((241999:0.0098,(((551837:0.00714,510956:0.00852)0.830.31:0.00551,114948:0.02457)0.988.159:0.00732,220193:0.01015)0.429.4:0.00572)0.893.45:0.00365,(82263:0.01107,238929:0.02461)0.765.28:0.0049)0.915.61:0.00417,((((155827:0.01209,(221692:0.01347,151165:0.00349)0.520.7:0.0106)1.000.1824:0.0155,(((218032:0.00546,114952:0.00659)0.965.99:0.00923,153239:0.00952)0.977.118:0.01287,(224152:0.01011,(248805:0.005,585431:0.02596)0.915.62:0.00644)0.866.43:0.00395)0.977.119:0.01056)0.819.24:0.00202,(251275:0.02302,(155778:0.00707,513679:0.01887)0.926.73:0.00743)0.989.137:0.00941)0.150.4:0.00361,132722:0.01667)0.633.5:0.00486)0.733.22:0.00109)0.965.100:0.00646)0.761.33:0.00145,(206560:0.01002,(215142:0.01817,256933:0.02)0.992.174:0.0125)0.807.20:0.00598)0.766.24:0.00158,(((144796:0.00775,218456:0.00641)0.921.66:0.00541,(590252:0.01238,248808:0.02612)0.933.71:0.00909)0.838.35:0.0032,(((527561:0.02256,589290:0.00942)0.966.96:0.00931,(253619:0.03182,(((200868:0.01333,(237754:0.02274,204465:0.01629)0.799.25:0.00692)0.958.99:0.0073,((((((543327:0.01322,125736:0.00626)0.786.28:0.00203,192386:0.045)0.988.160:0.01072,(541345:0.00226,575223:0.00971)0.980.109:0.01103)0.697.12:0.00514,81873:0.01174)1.000.1825:0.01562,((259898:0.01665,509177:0.01159)0.962.96:0.00633,(570808:0.01772,161263:0.01108)0.236.4:0.00277)0.853.36:0.00244)0.783.26:0.00252,(155970:0.01181,(577355:0.01867,(586002:0.00742,339617:0.01238)0.982.141:0.01022)0.868.57:0.00416)0.925.58:0.00383)0.962.97:0.00661)0.933.72:0.00587,(247797:0.01043,255540:0.02606)0.982.142:0.01436)0.848.41:0.00591)0.905.65:0.00566)0.985.120:0.01317,((255118:0.03453,333383:0.01985)0.973.99:0.01266,(248066:0.012,257782:0.02474)0.984.121:0.00747)0.678.12:0.00441)0.778.35:0.00298)0.267.8:0.00015)0.950.97:0.00512,(((218434:0.00782,113158:0.01384)0.983.121:0.01041,(239159:0.00752,(251458:0.01543,253183:0.02768)0.982.143:0.01123)0.990.147:0.01277)0.934.60:0.00671,((583421:0.01021,(((137100:0.00442,355792:0.00853)0.663.7:0.00454,(570407:0.00492,553068:0.00714)0.942.89:0.00388)0.965.101:0.00686,((218116:0.00525,240895:0.0101)0.971.134:0.00544,(512539:0.01655,242520:0.01209)0.900.52:0.00299)0.445.4:0.00163)1.000.1826:0.01711)0.966.97:0.00851,(567812:0.01728,(257941:0.02353,(((512717:0.01639,(143418:0.00648,(((248211:0.0114,249432:0.00063)0.882.51:0.00234,(114975:0.01173,225894:0.03028)0.969.102:0.00826)0.316.8:0.00187,50109:0.01892)0.913.72:0.00348)0.529.7:0.00488)1.000.1827:0.02949,321753:0.01474)0.971.135:0.01204,249781:0.01522)0.331.10:0.00598)0.896.52:0.00811)0.954.90:0.00832)0.980.110:0.01033)0.897.55:0.0042)0.983.122:0.01241)0.779.18:0.01008)0.798.21:0.0153,((318309:0.01374,513658:0.00856)1.000.1828:0.02145,(156301:0.0243,587410:0.01676)0.994.192:0.0187)0.606.9:0.01196)0.949.80:0.01654,(((153452:0.01209,355710:0.01332)0.993.163:0.01924,(154781:0.02223,152076:0.04933)0.928.66:0.01295)1.000.1829:0.06671,(137476:0.0152,((541042:0.0075,158564:0.00871)0.869.55:0.00291,124030:0.00767)0.982.144:0.01175)0.998.327:0.031)0.977.120:0.01963)0.999.515:0.03082)0.871.56:0.01446)0.947.81:0.0187)1.000.1830:0.05249,(((((63828:0.01963,63827:0.01205)1.000.1831:0.0304,549407:0.07534)0.974.105:0.01862,(((238891:0.02118,344975:0.01835)1.000.1832:0.06628,(((227938:0.01337,229260:0.01408)1.000.1833:0.01938,(((57459:0.01623,564018:0.01485)0.836.30:0.00415,((61852:0.00232,61853:0.02181)0.984.122:0.0078,114335:0.01627)0.924.59:0.00395)0.794.26:0.0025,(146032:0.00643,(240419:0.03159,548455:0.01599)0.937.74:0.00586)0.948.90:0.00632)0.724.15:0.00807)1.000.1834:0.05151,(158351:0.03896,(160321:0.02111,113883:0.02222)0.993.164:0.0177)0.849.39:0.01323)1.000.1835:0.04369)0.957.115:0.01719,(258236:0.02777,149984:0.03039)0.960.107:0.0146)0.576.6:0.00728)0.881.56:0.00998,((586950:0.0429,((85524:0.01868,534591:0.01707)0.998.328:0.02944,((564774:0.01689,539474:0.01146)1.000.1836:0.07042,(326916:0.10302,(327984:0.04249,((((294058:0.0235,(343926:0.01146,(294299:0.01892,((315802:0.00978,288985:0.0159)0.991.160:0.01338,355139:0.00654)0.951.75:0.00873)0.881.57:0.00717)0.834.33:0.00444)1.000.1837:0.02762,(((301864:0.00938,258636:0.01717)0.986.136:0.01259,(306305:0.02546,294881:0.02678)0.936.72:0.01215)0.964.115:0.00933,((349658:0.00958,554127:0.02179)1.000.1838:0.04083,((((288354:0.02187,(292730:0.01775,294166:0.01233)0.873.56:0.00489)0.940.88:0.0058,258180:0.04204)0.955.81:0.00755,((((301719:0.01058,((287526:0.00597,(((((301621:0.01555,296570:0.02162)0.750.24:0.00847,(290862:0.01035,293920:0.01888)0.961.88:0.00764)0.875.45:0.00286,(298808:0.01115,295515:0.01272)0.527.5:0.00804)0.936.73:0.00522,296097:0.00832)0.922.67:0.00603,296615:0.01917)0.856.38:0.00392)0.576.7:0.00335,((((319237:0.00539,293419:0.01839)0.286.9:0.00102,(308390:0.00533,294845:0.01032)0.771.27:0.00098)0.757.35:0.00162,(293852:0.02027,((297614:0.00696,288689:0.01028)0.925.59:0.00405,((303126:0.02318,(291085:0.01906,298859:0.0146)1.000.1839:0.00014)0.927.73:0.00353,293925:0.01007)0.886.45:0.00183)0.751.17:0.0009)0.861.55:0.00384)0.778.36:0.00198,294205:0.01414)0.882.52:0.00379)0.736.28:0.00536)0.904.60:0.00509,301052:0.01615)0.839.42:0.00325,291292:0.02068)0.945.74:0.00865,((291133:0.01717,(((((((293093:0.02064,296368:0.00742)0.640.7:0.00766,296495:0.01237)0.611.8:0.00439,305041:0.01137)0.961.89:0.00812,295365:0.00704)0.949.81:0.00584,((355540:0.00396,(339305:0.01792,297691:0.01889)0.932.66:0.00603)0.990.148:0.0091,((296121:0.00805,326635:0.01353)0.912.51:0.00177,(298910:0.00556,346528:0.0114)0.977.121:0.00917)0.829.25:0.00376)0.987.181:0.00868)0.754.24:0.00092,((313967:0.00897,297855:0.02773)0.718.14:0.00099,(344843:0.0153,((293151:0.03241,((355681:0.00909,300897:0.00448)0.865.46:0.00345,300726:0.01368)0.846.24:0.00177)0.853.37:0.01295,315248:0.01375)0.939.85:0.00687)0.924.60:0.00395)0.984.123:0.00742)0.958.100:0.00719,355717:0.01449)0.851.44:0.00283)0.995.226:0.01243,(300038:0.02234,(340709:0.01055,302547:0.00663)0.932.67:0.00461)0.745.25:0.00688)0.974.106:0.00819)0.987.182:0.01117)0.915.63:0.00645,(312384:0.00995,288046:0.01788)0.988.161:0.01801)0.945.75:0.01355)0.584.8:0.0075)0.997.259:0.01871)0.717.11:0.00892,(263256:0.00784,522702:0.00945)1.000.1840:0.03431)0.950.98:0.01633,584326:0.01712)1.000.1841:0.04334)0.626.8:0.02059)0.986.137:0.02616)0.730.19:0.02056)0.991.161:0.02484)0.975.113:0.01454,((251544:0.03404,(203309:0.02951,63771:0.02323)0.339.5:0.00424)0.996.247:0.01502,(70013:0.03481,314573:0.01924)0.976.106:0.01326)0.742.34:0.00564)1.000.1842:0.02636)0.938.82:0.01128,(331675:0.04589,((83178:0.01736,329759:0.01066)0.994.193:0.01402,(337485:0.01564,(327948:0.0198,(((594250:0.01359,570802:0.00737)0.938.83:0.00744,548553:0.01089)0.966.98:0.00689,(544659:0.01667,558910:0.00805)0.948.91:0.00675)0.947.82:0.00566)0.576.8:0.00015)0.870.54:0.00732)0.491.6:0.01416)1.000.1843:0.06013)0.633.6:0.01462)0.515.12:0.01248,((((79161:0.0266,552105:0.02109)1.000.1844:0.03225,((131533:0.0142,542252:0.01471)1.000.1845:0.04246,((276436:0.04273,(299835:0.00935,(323780:0.03563,(243625:0.00983,274447:0.01078)0.910.57:0.00596)0.527.6:0.0072)1.000.1846:0.04226)0.331.11:0.01211,(((349140:0.0013,100123:0.01331)1.000.1847:0.03197,(552574:0.05222,(577380:0.00956,575335:0.01867)0.974.107:0.00913)0.770.23:0.013)1.000.1848:0.03626,(32071:0.05321,(306087:0.02015,(245431:0.01432,(368928:0.00335,31745:0.01147)0.980.111:0.01662)0.997.260:0.02463)0.997.261:0.03217)0.992.175:0.02327)0.918.72:0.02064)0.625.9:0.01548)0.800.30:0.01117)0.938.84:0.01266,(((584283:0.02491,((573751:0.05377,(289836:0.01204,(290815:0.01323,(148459:0.00925,288148:0.0082)0.135.4:0.00221)0.760.29:0.00281)0.897.56:0.00917)0.997.262:0.02379,((553112:0.00625,568204:0.01567)1.000.1849:0.04029,(287575:0.03512,557929:0.01972)0.982.145:0.02048)0.997.263:0.03421)0.991.162:0.02992)1.000.1850:0.05513,(79502:0.02346,(255593:0.02713,(563388:0.01255,111096:0.01962)0.988.162:0.01227)0.953.85:0.01168)0.864.56:0.00745)0.994.194:0.02203,((265874:0.01566,(313926:0.022,275456:0.01821)0.225.7:0.0057)1.000.1851:0.04662,((237639:0.01751,111528:0.01276)1.000.1852:0.06122,(308950:0.02501,535916:0.00733)0.982.146:0.03143)1.000.1853:0.06882)0.567.7:0.01352)0.816.29:0.01127)0.980.112:0.02387,130443:0.11445)1.000.1854:0.04635)0.995.227:0.0274,(((203339:0.01488,562517:0.02433)1.000.1855:0.08387,(63785:0.02499,(63781:0.00195,532990:0.00436)0.999.516:0.02087)0.999.517:0.04289)0.975.114:0.02131,((293434:0.01853,312632:0.01329)1.000.1856:0.12506,((158932:0.04374,(112346:0.01661,(2940:0.01506,(3027:0.01635,19114:0.0146)0.774.15:0.00371)0.970.100:0.00749)1.000.1857:0.03068)1.000.1858:0.03472,(((100535:0.0037,542808:0.0354)1.000.1859:0.03533,(((24341:0.01206,579085:0.0303)1.000.1860:0.02981,((547579:0.01903,471188:0.02048)0.994.195:0.01427,(2920:0.00582,(580742:0.01603,544430:0.02752)0.965.102:0.01502)0.952.89:0.0102)0.590.8:0.00594)0.960.108:0.01177,(320922:0.02687,152262:0.03355)0.999.518:0.02558)0.897.57:0.00702)0.962.98:0.0085,(((584073:0.0364,(2935:0.02364,(217851:0.0141,89521:0.02559)0.999.519:0.02121)0.940.89:0.01516)0.888.45:0.01674,(543064:0.06127,((249365:0.02724,(583949:0.00522,24629:0.01441)0.940.90:0.009)0.927.74:0.0101,((152689:0.00684,83246:0.01658)0.997.264:0.0181,(239522:0.02216,253897:0.02253)0.989.138:0.01985)0.938.85:0.01135)0.652.10:0.00707)0.993.165:0.01991)0.992.176:0.01895,(535088:0.03488,(171324:0.03466,552935:0.02798)0.995.228:0.0204)1.000.1861:0.03714)0.589.8:0.00971)0.930.69:0.01178)1.000.1862:0.09508)0.991.163:0.03047)0.773.27:0.01447)0.926.74:0.02263,(89297:0.08827,(((211556:0.02371,317758:0.02988)1.000.1863:0.07562,((204198:0.01955,239745:0.02815)1.000.1864:0.14785,(((((264161:0.00705,(553831:0.01737,(552101:0.0032,538271:0.01085)0.922.68:0.00668)0.898.62:0.00569)0.962.99:0.01033,(266103:0.0337,542208:0.0103)0.993.166:0.01672)1.000.1865:0.03425,(65679:0.02708,592150:0.03456)0.303.6:0.00787)0.976.107:0.0222,((226121:0.00419,(233789:0.01063,331078:0.0151)0.877.52:0.00693)1.000.1866:0.05624,(279473:0.0312,(137070:0.00992,(154140:0.00546,143416:0.01644)0.669.10:0.00623)0.766.25:0.015)1.000.1867:0.04652)0.615.9:0.01903)0.995.229:0.03988,((413816:0.02421,539754:0.01367)1.000.1868:0.07867,(138086:0.04437,(102531:0.02384,(34401:0.02123,153882:0.00941)0.037.3:0.00557)0.995.230:0.0225)1.000.1869:0.03738)0.659.9:0.01802)0.999.520:0.05006)0.583.14:0.02217)0.917.66:0.02173,(((((63198:0.01757,(593533:0.02266,(509494:0.01995,509258:0.01391)0.864.57:0.01078)0.965.103:0.02016)0.559.6:0.00901,((131589:0.00776,512654:0.00575)1.000.1870:0.03018,(176040:0.02624,517884:0.03111)0.945.76:0.01761)0.743.32:0.0083)1.000.1871:0.03585,(204818:0.0274,534717:0.04209)0.997.265:0.03191)0.999.521:0.04099,(274405:0.06936,(266325:0.00463,263328:0.00765)1.000.1872:0.06203)0.793.19:0.01522)0.990.149:0.03152,(236704:0.11353,(((277154:0.02229,257843:0.0149)0.992.177:0.02335,(271229:0.01669,258656:0.02658)0.998.329:0.02603)0.992.178:0.02465,((272472:0.01406,266542:0.02752)1.000.1873:0.05409,((((271294:0.01632,(564252:0.01616,511521:0.0193)0.163.5:0.00508)0.926.75:0.00522,1665:0.0267)0.202.4:0.00085,(258248:0.01782,266385:0.01747)0.893.46:0.00415)0.802.27:0.00206,264190:0.02237)1.000.1874:0.0345)0.302.9:0.01118)0.795.23:0.02053)0.961.90:0.02208)0.659.10:0.01641)0.989.139:0.02476)0.977.122:0.01842)0.741.23:0.00669)0.986.138:0.01724,((((((171614:0.02357,329424:0.019)1.000.1875:0.09832,(346712:0.02967,242995:0.02271)1.000.1876:0.12756)0.998.330:0.04899,(309328:0.0335,537833:0.02757)1.000.1877:0.05746)0.730.20:0.00484,((274779:0.016,324136:0.01152)1.000.1878:0.07181,(((269582:0.01865,546202:0.02146)0.998.331:0.02697,((189073:0.01037,256635:0.01457)0.998.332:0.02346,(73086:0.02224,509051:0.01885)1.000.1879:0.04496)0.634.8:0.00989)1.000.1880:0.03188,((537364:0.01428,(10701:0.03643,(149255:0.01571,(10700:0.00681,(148034:0.01084,(144663:0.00733,(359089:0.01218,(150077:0.00968,144770:0.0097)0.872.64:0.00254)0.545.8:0.0019)0.736.29:0.00112)0.855.37:0.0043)0.472.4:0.00868)0.996.248:0.0179)0.531.8:0.01029)1.000.1881:0.13294,(579236:0.02971,555348:0.02139)1.000.1882:0.08708)0.914.55:0.01519)0.895.62:0.01301)0.930.70:0.01287)0.997.266:0.01965,((((527968:0.10075,((565666:0.01331,256416:0.01328)1.000.1883:0.04175,(353402:0.03149,535979:0.01954)1.000.1884:0.03608)1.000.1885:0.05613)0.982.147:0.02073,((((194829:0.00929,(537094:0.02124,102449:0.01135)0.900.53:0.00547)0.998.333:0.02641,((255816:0.00893,1683:0.01989)1.000.1886:0.02267,(444016:0.00536,((206446:0.02103,335981:0.01988)0.191.4:0.00014,(273090:0.01683,586090:0.00379)0.768.34:0.01103)0.943.86:0.00508)0.990.150:0.01014)0.843.31:0.00463)1.000.1887:0.04402,((203049:0.01995,(((201565:0.01245,157415:0.01472)0.776.30:0.02095,(226833:0.00378,542545:0.01129)0.877.53:0.01813)1.000.1888:0.01621,(((200259:0.01247,(246657:0.01295,(577666:0.0168,592048:0.01313)0.736.30:0.00539)0.893.47:0.00537)0.906.66:0.00501,(563138:0.01596,107581:0.01775)0.879.59:0.01232)0.911.60:0.00511,(202452:0.01299,(236143:0.00637,(587985:0.01057,204400:0.0277)0.725.19:0.00352)0.939.86:0.00673)0.989.140:0.01026)0.991.164:0.01263)0.592.13:0.00539)0.745.26:0.00974,((510941:0.0173,156688:0.00106)1.000.1889:0.00014,203080:0.01136)0.952.90:0.00932)0.972.103:0.02108)1.000.1890:0.07923,(237562:0.0244,557224:0.02089)1.000.1891:0.05478)0.039.3:0.01641)0.180.5:0.00687,(((274014:0.02713,545349:0.02302)1.000.1892:0.08547,(((217349:0.00735,545813:0.01347)0.979.123:0.01353,(351437:0.02041,588740:0.02712)0.947.83:0.01277)1.000.1893:0.04305,(315617:0.05491,(513469:0.0254,587403:0.0313)0.142.5:0.00826)0.988.163:0.03215)0.981.118:0.03399)0.752.32:0.01977,((85294:0.01165,550928:0.02217)1.000.1894:0.06307,((137533:0.00646,583292:0.02373)1.000.1895:0.05788,(338211:0.03613,((102613:0.01443,535393:0.01499)0.994.196:0.01627,(584707:0.014,237785:0.0155)0.990.151:0.0146)0.982.148:0.01745)0.999.522:0.03414)1.000.1896:0.05178)0.980.113:0.02691)0.395.11:0.00796)0.880.48:0.01016,((((((579975:0.01283,161200:0.01029)1.000.1897:0.04832,((((270621:0.02478,589980:0.01875)0.956.82:0.009,(((148269:0.02835,((339005:0.00308,160383:0.00338)1.000.1898:0.04726,332383:0.03092)0.646.12:0.00685)0.958.101:0.00837,193866:0.01542)0.972.104:0.01001,(137332:0.00979,(242702:0.00986,2788:0.01533)0.934.61:0.00591)1.000.1899:0.02438)0.867.53:0.00628)0.378.9:0.00456,548511:0.03141)0.999.523:0.01844,((518864:0.01359,333159:0.0464)0.842.42:0.0054,((244050:0.00982,336106:0.0036)0.316.9:0.02774,(318107:0.01646,536150:0.00692)1.000.1900:0.02446)0.928.67:0.01155)1.000.1901:0.03509)0.462.5:0.00629)0.332.5:0.00583,((242961:0.00167,564055:0.00985)1.000.1902:0.05905,(266276:0.0161,(270915:0.01725,266754:0.02532)0.340.7:0.00472)0.985.121:0.018)0.172.6:0.00921)1.000.1903:0.05014,(((201142:0.01077,290327:0.03345)0.987.183:0.01478,(577122:0.03343,560658:0.00774)0.996.249:0.02288)0.345.7:0.00653,((352790:0.00015,546338:0.02081)0.044.5:0.07805,((305954:0.01991,509346:0.02584)0.281.2:0.00872,((227921:0.02398,(327521:0.00449,540414:0.0362)0.999.524:0.019)1.000.1904:0.02749,((40888:0.00757,307010:0.00655)1.000.1905:0.03297,((416173:0.00403,(342856:0.01029,(60573:0.00982,(339256:0.01241,533628:0.02337)0.938.86:0.00476)0.872.65:0.00392)0.454.4:0.00217)0.863.50:0.00253,(332638:0.01704,139290:0.01258)0.952.91:0.00529)0.727.16:0.00106)0.199.6:0.00727)0.933.73:0.00966)1.000.1906:0.03936)0.474.3:0.01214)0.999.525:0.04314)0.814.21:0.01835,(39505:0.17401,((132935:0.00174,564222:0.00312)1.000.1907:0.06479,512263:0.06611)1.000.1908:0.08423)0.958.102:0.03701)0.941.89:0.02179,((((((((551026:0.0158,2799:0.01047)1.000.1909:0.05345,(((511983:0.02959,(573216:0.02109,525666:0.009)1.000.1910:0.04163)0.877.54:0.01545,(200064:0.03413,(314228:0.01362,255612:0.00936)0.959.85:0.01313)1.000.1911:0.05283)0.598.9:0.01119,((559075:0.01669,542907:0.0157)1.000.1912:0.03262,((591476:0.01894,(72455:0.00935,178940:0.0171)0.868.58:0.00462)0.992.179:0.01736,(581335:0.01509,514874:0.01451)1.000.1913:0.05661)0.483.7:0.01134)0.998.334:0.01984)0.299.6:0.00266)0.943.87:0.00975,((537288:0.03842,(((((555797:0.02833,(249219:0.02543,(267389:0.00672,512846:0.00402)1.000.1914:0.03978)0.913.73:0.0115)0.918.73:0.00609,((558568:0.01932,555142:0.01371)0.917.67:0.00827,(((510851:0.01801,342870:0.05977)0.291.10:0.0052,(154335:0.03403,(226857:0.00868,159441:0.01808)0.896.53:0.00807)0.999.526:0.02079)0.851.45:0.00559,112006:0.01744)0.719.9:0.00103)0.898.63:0.00514)0.868.59:0.00697,(((536385:0.01673,(100723:0.02531,155022:0.02793)0.969.103:0.01737)0.957.116:0.01111,(560786:0.02101,66932:0.02618)0.548.11:0.01029)0.945.77:0.0088,(154653:0.02535,(((270829:0.02948,(((558390:0.02057,582509:0.01182)1.000.1915:0.03113,(583233:0.02337,((577182:0.01934,(242039:0.01244,549290:0.01872)0.936.74:0.00356)0.431.9:0.00173,(((244548:0.02475,550316:0.02726)0.442.3:0.00475,(224266:0.00773,(27673:0.0143,86701:0.00513)0.999.527:0.02096)0.958.103:0.00902)0.247.4:0.00333,246506:0.01182)0.931.61:0.00868)0.587.6:0.01098)0.993.167:0.0162)0.999.528:0.02083,346710:0.03077)0.347.4:0.00889)0.478.4:0.00652,(((590779:0.01628,(270592:0.02274,524812:0.01762)0.612.10:0.00777)1.000.1916:0.02435,((588256:0.02115,((327231:0.01838,(306565:0.01483,558165:0.01777)0.775.18:0.00812)0.965.104:0.01054,(540250:0.0119,(321503:0.01713,337713:0.02075)0.935.76:0.0093)0.596.8:0.00348)0.967.95:0.00872)0.843.32:0.00419,(323269:0.01779,143613:0.0211)0.141.3:0.00467)0.991.165:0.01115)0.938.87:0.00539,(((510435:0.00827,2818:0.01241)0.879.60:0.02265,(((274382:0.01751,(555016:0.01918,(579255:0.01721,(579998:0.0272,311841:0.00678)0.992.180:0.01701)0.374.9:0.005)0.973.100:0.0109)0.853.38:0.00325,516943:0.01659)0.984.124:0.01004,((364773:0.01189,258197:0.0135)0.984.125:0.01046,(147997:0.01541,(566761:0.01383,518971:0.0112)0.997.267:0.01079)0.988.164:0.00968)0.534.6:0.00341)0.938.88:0.00679)0.996.250:0.02075,544225:0.05885)0.893.48:0.00634)0.950.99:0.00952)1.000.1917:0.02568,(543923:0.01062,(265831:0.01211,562401:0.01198)0.937.75:0.00473)0.782.23:0.00267)0.993.168:0.01445)0.999.529:0.01761)0.890.60:0.00419)0.943.88:0.00601,((582419:0.03076,((153472:0.01965,(323064:0.02874,50882:0.02831)0.435.9:0.00546)0.890.61:0.00738,(((579256:0.02477,544469:0.02884)0.862.55:0.00742,(225493:0.02082,509843:0.01924)0.953.86:0.00746)0.833.29:0.00256,(537953:0.00986,((113272:0.01654,(213225:0.00149,200097:0.0048)0.986.139:0.00827)0.958.104:0.0076,589011:0.01744)0.440.7:0.0048)0.996.251:0.01373)0.785.27:0.00522)0.939.87:0.0103)0.907.60:0.00618,(170286:0.02713,272697:0.02794)0.462.6:0.01013)0.190.4:0.00438)0.757.36:0.00584,(353483:0.0163,(150611:0.02029,259028:0.02714)0.067.2:0.00679)0.839.43:0.0038)0.855.38:0.00885)0.908.63:0.00508,((((((((258872:0.00452,558104:0.01443)0.992.181:0.0132,(200911:0.01076,570362:0.00807)1.000.1918:0.02578)0.901.51:0.00892,(((317173:0.03222,((((337594:0.01194,288336:0.02031)0.556.4:0.0045,593766:0.0208)0.028.3:0.00286,348802:0.02434)0.999.530:0.02675,271620:0.01774)0.230.5:0.00747)0.978.113:0.01505,(((552149:0.01189,520693:0.01093)0.997.268:0.01399,((20894:0.0209,83410:0.01151)0.864.58:0.00742,(342256:0.01496,533036:0.01406)0.953.87:0.00783)0.961.91:0.00857)0.952.92:0.00898,(563596:0.0264,(337297:0.00671,578134:0.00806)1.000.1919:0.03174)0.570.10:0.00865)0.978.114:0.01099)0.953.88:0.00945,200049:0.04457)0.882.53:0.00783)0.929.82:0.00999,106057:0.03974)0.867.54:0.00553,571201:0.04185)0.958.105:0.00821,(355875:0.03344,(165216:0.00078,237911:0.03097)0.880.49:0.02888)0.905.66:0.00956)0.710.14:0.00102,((179213:0.0315,(532921:0.02609,(535958:0.04082,(548121:0.02387,(313623:0.00659,151305:0.00968)0.912.52:0.00697)0.933.74:0.00744)1.000.1920:0.02139)0.999.531:0.01668)0.783.27:0.00427,342344:0.02924)0.599.9:0.00878)0.960.109:0.00738,(((((240923:0.02016,2800:0.02825)1.000.1921:0.02975,((2793:0.0123,583343:0.01386)0.996.252:0.01612,(((2787:0.01668,395015:0.00841)0.728.21:0.01003,((((159318:0.02116,2782:0.01745)0.956.83:0.00718,(440418:0.00597,198796:0.01694)0.319.7:0.00343)0.793.20:0.00441,591115:0.01315)0.983.123:0.01031,((399640:0.00468,((459625:0.02426,(444204:0.00499,(448778:0.01258,(2770:0.01831,200748:0.01973)0.904.61:0.00281)0.195.3:0.00414)0.999.532:0.02282)0.998.335:0.01805,40268:0.01301)0.730.21:0.00106)0.424:0.00292,(233369:0.00451,((235788:0.02227,(297164:0.01978,229328:0.00821)0.666.13:0.00497)0.982.149:0.00905,(190032:0.01288,2777:0.01784)0.962.100:0.00993)0.507.7:0.00312)0.148.3:0.00614)0.973.101:0.01097)0.996.253:0.01764)0.984.126:0.01785,(2783:0.02173,(2785:0.02025,387035:0.01413)0.989.141:0.01242)0.904.62:0.00623)0.914.56:0.00725)0.973.102:0.01308)0.994.197:0.02177,((557601:0.00015,172922:0.00597)1.000.1922:0.03077,(((247225:0.01094,82890:0.02472)0.997.269:0.02498,((107148:0.03724,43467:0.0218)0.521.10:0.00789,((589642:0.02467,(558514:0.03305,((352654:0.00674,499852:0.01097)0.788.22:0.00284,2805:0.02015)0.515.13:0.01277)0.842.43:0.00404)0.991.166:0.01652,2801:0.04061)0.549.5:0.00913)0.963.97:0.01306)0.382.9:0.00666,(529264:0.03849,(566185:0.02718,(319990:0.02098,545203:0.00823)0.938.89:0.01951)0.510.5:0.00552)0.940.91:0.01071)0.941.90:0.01434)1.000.1923:0.03343)0.122.6:0.00721,(323488:0.08797,((246979:0.00758,238557:0.0232)1.000.1924:0.03164,((269075:0.01151,542353:0.01516)1.000.1925:0.04448,(512628:0.02365,(173437:0.01596,(((2833:0.02015,(159626:0.0194,536123:0.01801)0.390.6:0.00499)0.334.5:0.00773,(((543167:0.01937,(235890:0.02526,(258652:0.01613,(269780:0.00803,(200848:0.00918,154324:0.00796)0.908.64:0.00405)0.860.48:0.0067)0.840.28:0.00585)0.939.88:0.00835)0.283.6:0.00784,182611:0.02762)0.902.46:0.00726,167808:0.01601)0.998.336:0.02614)0.935.77:0.0154,(136912:0.01992,99819:0.02548)0.983.124:0.01221)0.967.96:0.00985)0.789.30:0.00675)0.987.184:0.01425)0.892.45:0.01128)0.926.76:0.01483)0.964.116:0.01732)0.988.165:0.01568,(((537024:0.01838,(2813:0.03068,(573455:0.00674,531452:0.00731)1.000.1926:0.02115)0.561.10:0.00418)0.994.198:0.01908,(521136:0.05064,(341990:0.01952,(337295:0.02269,545112:0.00394)0.198.4:0.00535)1.000.1927:0.02235)1.000.1928:0.0367)0.388.8:0.0063,(168811:0.00847,(263866:0.00356,272893:0.01438)0.947.84:0.00939)1.000.1929:0.05296)0.293.2:0.00627)0.979.124:0.01074)0.914.57:0.00645)0.721.12:0.00682)0.854.38:0.00339,(288230:0.0088,264273:0.01074)1.000.1930:0.10501)0.857.40:0.00449,((((142901:0.02065,(267225:0.01307,555396:0.02331)0.986.140:0.01037)0.994.199:0.00961,(214356:0.02344,201662:0.02656)0.921.67:0.00731)0.973.103:0.00777,(((153928:0.01476,(152343:0.00511,(41301:0.00776,30167:0.0073)0.439.13:0.01879)0.400.8:0.00894)0.997.270:0.02031,(223634:0.01607,(51628:0.01243,(145730:0.00779,200304:0.01928)0.864.59:0.00411)0.708.12:0.00454)1.000.1931:0.03027)0.991.167:0.01327,(245832:0.02036,577081:0.00432)1.000.1932:0.02752)0.888.46:0.00621)0.931.62:0.00553,((((513590:0.01845,(252268:0.01233,(579129:0.00188,586939:0.02121)0.998.337:0.01363)0.966.99:0.01008)0.984.127:0.0124,(204219:0.03233,431182:0.02785)0.439.14:0.00673)1.000.1933:0.02574,((2823:0.0869,((331408:0.0366,(202072:0.02244,332146:0.00589)0.975.115:0.0139)1.000.1934:0.03364,((307648:0.00959,321170:0.02194)0.997.271:0.01968,(28765:0.04406,(((245810:0.01589,(244840:0.00986,(329544:0.01086,(238799:0.02105,208708:0.01212)0.962.101:0.00798)0.898.64:0.00659)0.986.141:0.01566)0.998.338:0.02152,((139405:0.00823,172675:0.01147)1.000.1935:0.0233,(575719:0.01854,(513929:0.01678,568748:0.02185)0.240.5:0.00743)0.919.70:0.00874)0.787.27:0.00394)0.624.8:0.00503,2826:0.03643)0.756.34:0.008)0.998.339:0.01884)0.984.128:0.01847)0.453.5:0.00706)0.761.34:0.01505,216224:0.0478)0.812.34:0.01133)0.332.6:0.00842,((512816:0.04547,(219667:0.01309,235394:0.0161)1.000.1936:0.03193)0.410.6:0.01157,(((588739:0.00793,560957:0.01021)1.000.1937:0.04166,((((583602:0.00751,35556:0.01618)0.835.29:0.0042,(530010:0.01059,241588:0.03814)0.833.30:0.00027)0.763.26:0.00616,285747:0.00328)0.779.19:0.00266,208621:0.0219)0.119.2:0.01892)0.858.36:0.01399,((252536:0.01871,513639:0.01895)1.000.1938:0.03477,((304581:0.00015,33376:0.01834)0.940.92:0.00708,273690:0.02281)1.000.1939:0.0255)0.978.115:0.01335)0.235.4:0.0043)0.368.11:0.00762)0.973.104:0.01149)0.987.185:0.01361)0.969.104:0.01065,((245639:0.00872,200461:0.01313)1.000.1940:0.03176,(((200805:0.00961,(201428:0.02812,((202004:0.00014,549345:0.00106)0.998.340:0.01399,(203055:0.01587,(278832:0.01173,202625:0.01748)0.991.168:0.01151)0.834.34:0.00202)0.489.2:0.00632)0.986.142:0.01408)0.957.117:0.00899,202179:0.01304)0.533.4:0.06683,((200509:0.01167,((170512:0.01192,344848:0.00185)0.984.129:0.00955,201604:0.01926)0.777.24:0.00916)0.999.533:0.0334,(555156:0.04843,(568521:0.02439,(348461:0.01756,533316:0.00777)0.422.8:0.00576)0.875.46:0.00804)1.000.1941:0.05909)0.988.166:0.03237)0.987.186:0.02183)0.999.534:0.02386)0.993.169:0.02514,(((((2853:0.00147,251126:0.02546)0.874.51:0.05161,(199994:0.03715,534439:0.03954)0.998.341:0.0394)0.972.105:0.02317,(((585391:0.03859,((251778:0.02371,565226:0.01698)0.988.167:0.01705,((533695:0.01042,(278198:0.0087,560315:0.01112)0.873.57:0.01111)0.996.254:0.01399,(248342:0.01939,(558118:0.01612,(((((248724:0.00911,(105215:0.00516,207109:0.01411)0.991.169:0.00772)0.826.20:0.00161,(156144:0.00827,2871:0.00745)0.925.60:0.0025)0.929.83:0.00397,85331:0.00947)0.661.11:0.0019,256640:0.01596)0.704.18:0.00425,551587:0.01575)0.851.46:0.00304)0.985.122:0.01095)0.201.5:0.00436)0.983.125:0.01224)0.190.5:0.00616)0.962.102:0.01451,(548264:0.05751,((199905:0.00325,46917:0.01621)1.000.1942:0.02267,((83497:0.01218,573487:0.01101)0.988.168:0.01165,((255306:0.02081,(584125:0.01325,537762:0.01609)0.723.17:0.009)0.991.170:0.01234,526316:0.02084)0.389.5:0.00289)0.961.92:0.00917)0.969.105:0.01516)0.956.84:0.01358)0.999.535:0.0193,(((((237767:0.01213,576162:0.00932)1.000.1943:0.03323,239348:0.03484)0.713.9:0.01266,(((308618:0.00338,(518487:0.00802,530856:0.01811)0.895.63:0.00363)0.798.22:0.04302,(581457:0.05533,((577627:0.01659,(252618:0.009,(201410:0.01198,311201:0.00923)0.687.12:0.0075)0.990.152:0.01631)0.997.272:0.02747,((((((((385507:0.01919,135215:0.01124)0.860.49:0.00161,83894:0.01695)0.996.255:0.00727,201328:0.02238)0.996.256:0.00079,85208:0.03578)0.962.103:0.00453,(325188:0.00343,173765:0.00666)0.889.49:0.0207)0.739.28:0.00443,238659:0.00516)0.977.123:0.00711,(82559:0.0225,339412:0.00769)0.958.106:0.00608)0.991.171:0.0136,((314251:0.00568,511720:0.01206)0.994.200:0.01244,((288195:0.00639,(190209:0.03328,140014:0.00539)0.999.536:0.02205)0.067.3:0.00288,247503:0.00492)0.988.169:0.01278)0.874.52:0.01279)1.000.1944:0.02843)0.998.342:0.02231)0.714.11:0.00991)0.952.93:0.01256,((2859:0.06146,(2860:0.01678,(139631:0.00971,62476:0.02441)0.652.11:0.00755)1.000.1945:0.02728)0.868.60:0.01318,(((235836:0.02251,542264:0.01974)1.000.1946:0.02425,((200271:0.02052,589189:0.01061)1.000.1947:0.02307,((532634:0.01729,528880:0.01731)0.906.67:0.00478,(((517269:0.00628,(522800:0.00629,2856:0.01256)0.943.89:0.00864)0.925.61:0.00638,(500211:0.00784,2861:0.01798)0.971.136:0.00981)0.747.23:0.0041,(530825:0.0159,509879:0.00661)0.994.201:0.00973)0.763.27:0.00182)1.000.1948:0.02415)0.930.71:0.00851)0.945.78:0.01093,((275643:0.0089,((143227:0.01184,246619:0.02903)0.997.273:0.01287,2868:0.01842)0.723.18:0.00114)1.000.1949:0.03911,(2862:0.03628,(50220:0.02271,368930:0.01556)1.000.1950:0.04277)0.977.124:0.01145)0.914.58:0.01253)1.000.1951:0.02862)0.981.119:0.01791)0.443.9:0.00786)0.831.19:0.00907,(588567:0.04007,(533542:0.01901,509799:0.02631)0.998.343:0.01666)0.978.116:0.0159)0.017.4:0.01089,(308302:0.01113,(374822:0.01167,88168:0.00613)0.976.108:0.01198)1.000.1952:0.05055)0.681.15:0.00583)0.900.54:0.01199)0.893.49:0.01321,((((509806:0.02101,(2838:0.01045,(584866:0.02975,(197419:0.01384,591299:0.00888)0.762.27:0.00458)0.372.8:0.00278)1.000.1953:0.02096)0.818.18:0.00664,314184:0.01785)1.000.1954:0.0339,(((2845:0.00359,529704:0.00536)1.000.1955:0.04128,528592:0.03475)0.913.74:0.00836,((415267:0.0189,(236044:0.00898,(243965:0.00726,583403:0.00892)0.900.55:0.00941)0.914.59:0.00999)0.998.344:0.0257,((86309:0.02285,(89643:0.00626,524712:0.00608)0.993.170:0.01474)0.982.150:0.01452,((226030:0.0122,568330:0.01149)0.932.68:0.00744,(2840:0.01067,521367:0.01868)0.982.151:0.01047)0.853.39:0.01033)0.923.65:0.01216)0.995.231:0.02093)0.982.152:0.01729)0.942.90:0.02172,(158129:0.00686,105661:0.01926)1.000.1956:0.11358)0.134.4:0.00495)0.998.345:0.02885,((((240110:0.0016,592092:0.01013)1.000.1957:0.03296,(326719:0.07856,(585568:0.02796,331911:0.02327)0.994.202:0.02444)0.571.11:0.01799)1.000.1958:0.04896,(((549231:0.00349,(558858:0.00932,517654:0.01085)0.907.61:0.00455)0.984.130:0.00999,242568:0.00058)1.000.1959:0.09363,(((((304737:0.0239,243139:0.0131)0.991.172:0.01607,(201735:0.02205,242839:0.00575)0.993.171:0.0184)0.999.537:0.0219,(((277693:0.01324,328112:0.02245)1.000.1960:0.04074,((201913:0.02032,519588:0.01072)0.998.346:0.02037,((((248077:0.01237,268742:0.00493)0.744.24:0.00453,(201123:0.01127,(218719:0.02201,278656:0.0146)0.916.65:0.00814)0.923.66:0.005)0.816.30:0.01408,332967:0.00758)0.306.9:0.00202,249544:0.01808)1.000.1961:0.02322)0.984.131:0.01758)0.999.538:0.02659,(((202781:0.04238,((594164:0.02595,((515612:0.01116,(200550:0.0054,((200073:0.00231,201778:0.00107)1.000.1962:0.01805,517456:0.00372)0.967.97:0.00487)0.993.172:0.01099)0.994.203:0.01407,(200854:0.01117,((585097:0.02203,531410:0.02229)0.886.46:0.0114,545782:0.01322)0.556.5:0.00423)0.979.125:0.01041)0.999.539:0.01737)0.915.64:0.00718,(((202207:0.0275,(528610:0.02845,(251057:0.0123,(242145:0.01135,200885:0.02199)0.733.23:0.00976)0.601.4:0.00642)0.364.7:0.00408)0.995.232:0.01227,(242138:0.01963,591358:0.02835)0.970.101:0.01477)0.070.4:0.00499,509595:0.02359)0.897.58:0.00463)0.771.28:0.00534)0.979.126:0.01681,((359924:0.01289,252286:0.01936)0.999.540:0.02937,((235865:0.01109,336869:0.00742)1.000.1963:0.03628,(416617:0.01622,524860:0.02029)0.997.274:0.02648)0.991.173:0.02019)0.851.47:0.00481)0.994.204:0.01593,((((592082:0.02059,581403:0.00701)1.000.1964:0.03242,((257237:0.01453,(572280:0.02735,588684:0.01349)0.960.110:0.00772)0.957.118:0.00992,(200837:0.03129,278955:0.01834)0.995.233:0.019)0.947.85:0.01039)0.985.123:0.01345,(527118:0.0092,199912:0.00963)0.999.541:0.01587)0.588.4:0.00411,202670:0.01728)0.977.125:0.0105)0.974.108:0.01474)0.812.35:0.00539)0.972.106:0.01519,((((((540991:0.01819,(321885:0.01073,255557:0.0218)0.921.68:0.00577)0.973.105:0.01138,(216364:0.02906,324960:0.03313)0.949.82:0.01006)0.869.56:0.0101,((397840:0.01005,(278884:0.01651,223042:0.04346)0.974.109:0.01451)0.477.6:0.00665,201687:0.01236)0.997.275:0.02322)1.000.1965:0.02467,(576167:0.02387,253376:0.01967)1.000.1966:0.04129)0.794.27:0.01016,(((202686:0.01674,202759:0.01813)0.991.174:0.01683,((205038:0.01683,246400:0.01926)1.000.1967:0.03348,243365:0.04279)0.080.5:0.00654)0.999.542:0.02117,(((((((((320812:0.03782,(325828:0.02008,279355:0.03501)0.965.105:0.02038)0.828.25:0.01196,(238600:0.02957,308368:0.03387)0.154.2:0.00829)0.891.59:0.00663,(250454:0.02815,(251955:0.00544,214051:0.00541)0.791.29:0.00993)0.972.107:0.01346)0.368.12:0.00998,(((316663:0.01706,(236629:0.02579,563036:0.01164)0.833.31:0.00737)1.000.1968:0.04773,(251736:0.02109,244582:0.01025)0.309.8:0.00674)0.749.24:0.00936,580792:0.03966)0.934.62:0.0125)0.269.6:0.00768,(551517:0.0113,(254402:0.02227,277743:0.00919)0.279.8:0.0079)1.000.1969:0.03484)0.955.82:0.01063,(202050:0.00908,254636:0.02516)0.996.257:0.01907)0.972.108:0.01446,(244080:0.02069,253174:0.03919)0.987.187:0.01928)0.969.106:0.00968,(((242459:0.02041,307453:0.02921)0.406.7:0.00642,((512722:0.01802,587032:0.03133)0.980.114:0.01139,((582011:0.018,252476:0.01251)1.000.1970:0.03672,(237881:0.03255,307906:0.03245)0.959.86:0.01263)0.977.126:0.01445)0.856.39:0.00716)0.819.25:0.00667,(((249810:0.04475,203044:0.03115)0.990.153:0.02289,(240854:0.02234,(246187:0.01871,567145:0.01494)0.829.26:0.00996)1.000.1971:0.03771)0.427.6:0.00432,(250620:0.03128,250705:0.02453)0.906.68:0.01031)0.983.126:0.01288)0.260.6:0.00676)0.961.93:0.00685,((((219131:0.01109,251291:0.01315)0.964.117:0.01212,((219963:0.02038,(238979:0.02662,593280:0.02009)0.909.59:0.00836)0.917.68:0.01024,(561767:0.02013,(253499:0.02205,581157:0.02141)0.624.9:0.00467)0.232.7:0.0084)0.974.110:0.00988)0.980.115:0.01353,(((362115:0.01755,235969:0.01346)1.000.1972:0.01405,((312527:0.00694,244946:0.03058)0.896.54:0.00662,(258765:0.02517,220916:0.02211)0.684.14:0.00302)0.835.30:0.00219)0.885.55:0.00389,216983:0.0309)0.852.60:0.00329)0.891.60:0.00429,(203365:0.03351,((((237749:0.0413,(201554:0.01886,(221204:0.00884,219582:0.01739)0.995.234:0.01395)0.957.119:0.00734)0.980.116:0.01003,((247522:0.0272,572242:0.03684)0.944.69:0.01042,((221442:0.01195,202096:0.02019)0.987.188:0.01237,579057:0.01913)0.412.4:0.00402)0.774.16:0.00389)0.601.5:0.00205,(212306:0.01664,532312:0.0092)0.998.347:0.01822)0.802.28:0.00458,(238922:0.03616,(333924:0.03456,329570:0.00786)0.962.104:0.0097)0.992.182:0.01837)0.900.56:0.00884)0.918.74:0.00517)0.612.11:0.00065)0.922.69:0.00486)0.930.72:0.00935)0.675.8:0.00942,(239490:0.01918,(202015:0.02552,200727:0.0135)0.899.61:0.00808)0.999.543:0.02313)0.892.46:0.00949)1.000.1973:0.02968,586523:0.06146)0.948.92:0.02026)0.978.117:0.02258)0.988.170:0.02036,(340209:0.10423,(544572:0.06731,((202653:0.00216,160585:0.00086)1.000.1974:0.02866,((578132:0.04494,(20269:0.01685,(200402:0.00833,201392:0.00076)0.766.26:0.00307)1.000.1975:0.03952)0.997.276:0.02535,(((((2879:0.01242,2881:0.01501)0.348.5:0.0227,((512214:0.01376,(((549261:0.00497,((561727:0.01457,547217:0.01282)0.829.27:0.00297,344331:0.00945)0.378.10:0.00762)0.951.76:0.00637,556485:0.01088)0.858.37:0.00228,224453:0.00729)0.936.75:0.0037)0.483.8:0.00539,((235318:0.01834,221502:0.00151)0.785.28:0.00383,((186517:0.00343,500792:0.01169)0.964.118:0.00931,336484:0.02005)0.670.5:0.00595)0.988.171:0.01049)0.904.63:0.00559)0.980.117:0.00947,(20352:0.01836,2885:0.01121)1.000.1976:0.03189)0.917.69:0.00782,(((51778:0.01191,(345379:0.0109,328853:0.02357)0.949.83:0.00699)0.826.21:0.00236,(((33768:0.0192,53180:0.01707)0.975.116:0.00891,((217401:0.00394,24367:0.00547)0.894.54:0.00574,(2877:0.01867,(250059:0.00358,533332:0.02125)1.000.1977:0.01972)0.990.154:0.0148)0.905.67:0.00448)0.796.25:0.00171,((534483:0.022,(246083:0.01267,170729:0.01702)0.747.24:0.00221)0.998.348:0.01618,((240933:0.01248,(222936:0.01096,35108:0.02328)0.886.47:0.01027)0.993.173:0.01515,((224547:0.00811,(534352:0.01563,520144:0.01349)0.736.31:0.00299)0.666.14:0.00448,201365:0.0229)0.060.2:0.01039)0.017.5:0.00355)1.000.1978:0.01701)0.888.47:0.00261)0.989.142:0.01024,((541286:0.00471,220676:0.01615)0.865.47:0.00488,155587:0.02871)0.920.58:0.00653)0.751.18:0.00806)0.721.13:0.00721,(215283:0.02164,249037:0.0131)0.998.349:0.0232)0.315.5:0.00696)0.949.84:0.01071)0.995.235:0.02381)1.000.1979:0.05616)0.980.118:0.02027)0.971.137:0.01645)0.752.33:0.01519)1.000.1980:0.05431,((((((250460:0.00357,239076:0.00392)0.999.544:0.03316,(554768:0.0712,(356606:0.00773,(569736:0.00289,44762:0.03347)0.234.4:0.00598)1.000.1981:0.03779)0.900.57:0.02619)0.966.100:0.02086,(302876:0.01617,261113:0.03025)0.997.277:0.02843)0.999.545:0.04427,(((331993:0.0393,((509247:0.0125,(252359:0.0111,311526:0.02469)0.375.6:0.00526)0.875.47:0.00983,(240932:0.01053,(135881:0.02617,239668:0.03135)0.812.36:0.00684)0.992.183:0.0152)0.998.350:0.02631)0.895.64:0.01628,(((251688:0.01876,548530:0.01786)0.999.546:0.02309,(((124913:0.00247,202117:0.00549)0.998.351:0.01539,(533393:0.01566,(173711:0.00647,279241:0.00898)0.981.120:0.00847)0.880.50:0.00925)1.000.1982:0.01901,((361409:0.0117,(510050:0.00922,272431:0.0211)0.444.11:0.00946)0.992.184:0.01339,(146588:0.02606,294838:0.02533)0.933.75:0.01045)0.535.7:0.00626)0.929.84:0.00894)0.000.70:0.00433,((576164:0.01329,(203245:0.00697,568919:0.00014)0.983.127:0.01614)1.000.1983:0.0351,(160569:0.01933,151370:0.0095)0.998.352:0.0196)0.999.547:0.02052)0.918.75:0.01189)0.748.22:0.01072,(((231309:0.00323,346138:0.02097)1.000.1984:0.02635,(91792:0.00992,341049:0.03001)0.996.258:0.01757)0.400.9:0.00704,351708:0.08874)0.795.24:0.0083)0.695.8:0.01635)0.979.127:0.023,(237452:0.06885,(340702:0.03982,(135862:0.09139,115016:0.08819)0.438.4:0.0185)0.987.189:0.03206)0.996.259:0.02781)1.000.1985:0.05156,((((219735:0.02501,(548781:0.00087,2794:0.00984)0.976.109:0.01082)0.999.548:0.04032,(517627:0.05702,(((((576923:0.04,(260280:0.01522,(258862:0.01459,274232:0.01721)0.818.19:0.00796)0.949.85:0.00715)0.755.33:0.00348,(239299:0.02656,((571999:0.01814,((279053:0.00936,((549097:0.01127,554003:0.0296)0.573.7:0.00401,((198875:0.01134,(571346:0.02675,(218062:0.00834,585249:0.03028)0.936.76:0.01166)0.992.185:0.01506)0.606.10:0.00597,551072:0.01674)0.310.8:0.00464)0.929.85:0.00863)0.846.25:0.0054,((63101:0.0154,403814:0.01697)0.852.61:0.00178,(549187:0.01488,135609:0.02237)0.422.9:0.00236)0.892.47:0.00515)0.988.172:0.00913)0.967.98:0.0089,(223588:0.02393,454105:0.01472)0.999.549:0.02044)0.564.13:0.00392)0.143.7:0.00239)0.941.91:0.00465,(((((267303:0.01156,414839:0.01709)0.998.353:0.01823,(268863:0.03755,276082:0.01392)0.180.6:0.00624)0.761.35:0.00535,(265777:0.00757,(269597:0.01927,132242:0.01799)0.855.39:0.0035)0.995.236:0.01292)0.831.20:0.00355,(243591:0.01877,(568912:0.01751,(262105:0.01011,532739:0.02645)0.542.8:0.00393)0.952.94:0.00857)0.980.119:0.00797)0.891.61:0.00384,(((240360:0.02915,277297:0.01965)0.935.78:0.00793,((269210:0.02217,(538614:0.02405,(264090:0.00586,319618:0.01687)0.999.550:0.01647)0.966.101:0.00843)0.944.70:0.00882,(201078:0.02856,137395:0.03273)0.375.7:0.00601)0.097.6:0.00369)0.374.10:0.00265,(264998:0.01914,(276723:0.01879,258649:0.02102)0.939.89:0.00924)0.736.32:0.00362)0.939.90:0.0058)0.542.9:0.00148)0.926.77:0.00648,(132098:0.02199,550342:0.01526)0.997.278:0.02069)0.966.102:0.01216,((558888:0.04579,(((239622:0.0334,256257:0.02061)0.985.124:0.01494,(350673:0.0727,(544394:0.00395,200413:0.01974)1.000.1986:0.06221)0.590.9:0.01302)0.937.76:0.00893,((276403:0.03506,((352115:0.03245,(238358:0.02692,264857:0.02369)0.155.7:0.00666)0.975.117:0.01048,(2842:0.03156,326426:0.01639)0.894.55:0.00674)0.877.55:0.00437)0.998.354:0.01751,(576295:0.01222,(545079:0.0056,548566:0.00364)0.943.90:0.00583)1.000.1987:0.03049)0.822.35:0.00535)0.991.175:0.0126)0.938.90:0.00769,273744:0.03073)0.734.20:0.00635)0.295.4:0.00507)0.943.91:0.01894)1.000.1988:0.03409,((161593:0.02165,256308:0.02148)1.000.1989:0.07437,((((156060:0.01687,155934:0.02094)0.968.94:0.01095,(181593:0.01581,588933:0.03885)0.906.69:0.01613)0.952.95:0.0112,((((((65683:0.02229,(((550231:0.01261,216191:0.00734)0.950.100:0.01242,216375:0.01257)0.450.9:0.00741,(250291:0.00441,214965:0.01197)0.994.205:0.01808)0.349.7:0.01334)0.366.4:0.00739,(225058:0.00995,(112280:0.00955,111273:0.01938)0.922.70:0.01226)1.000.1990:0.02467)0.342.8:0.00504,(35667:0.02553,109707:0.0284)0.923.67:0.00725)0.406.8:0.00292,(556144:0.02149,((219383:0.0176,(245617:0.00814,543366:0.00884)0.929.86:0.00422)1.000.1991:0.01323,(((235601:0.00977,(583790:0.0193,245031:0.01481)0.567.8:0.00322)0.921.69:0.00627,555571:0.01442)0.330.6:0.00284,((((286652:0.02063,(574158:0.01436,(217403:0.00805,102644:0.01275)0.950.101:0.00489)0.741.24:0.00098)0.964.119:0.00317,223251:0.00644)0.893.50:0.00015,240455:0.02024)0.969.107:0.00594,((44216:0.02309,(203959:0.01519,547110:0.01187)0.664.7:0.0066)0.995.237:0.01439,(588682:0.03097,((227320:0.0268,205811:0.02473)0.982.153:0.01235,(281308:0.01975,(215591:0.00584,238340:0.02416)1.000.1992:0.02611)0.648.10:0.00016)0.983.128:0.01015)0.908.65:0.00446)0.971.138:0.00833)0.862.56:0.00269)0.804.27:0.00232)0.983.129:0.00874)0.000.71:0.00055)0.970.102:0.00855,568391:0.01626)0.266.5:0.00535,211992:0.01569)0.984.132:0.01062)0.956.85:0.00815,((((((136324:0.01394,534600:0.00429)1.000.1993:0.02413,(572899:0.00525,55354:0.00807)0.926.78:0.00518)0.979.128:0.00857,(255902:0.03415,587131:0.02339)0.941.92:0.0066)0.842.44:0.00202,(244874:0.02792,559033:0.02206)0.951.77:0.00883)0.925.62:0.00598,((224072:0.01086,(513586:0.0214,223655:0.0137)0.556.6:0.00557)0.995.238:0.01851,(253931:0.02004,(575940:0.00663,324022:0.00646)0.828.26:0.02324)1.000.1994:0.02713)0.266.6:0.00619)0.963.98:0.01004,(523807:0.03124,(220924:0.01036,210216:0.03843)1.000.1995:0.03528)0.786.29:0.01918)0.958.107:0.01311)0.935.79:0.01685)1.000.1996:0.08936)0.934.63:0.01792,(((273516:0.01237,262812:0.02116)0.992.186:0.02282,(221149:0.02265,262361:0.01554)0.999.551:0.02401)1.000.1997:0.0528,(((237389:0.00908,269295:0.01355)1.000.1998:0.02982,((565879:0.01688,(560755:0.03955,543377:0.02492)0.981.121:0.0196)0.583.15:0.01123,245089:0.02006)0.834.35:0.01187)0.999.552:0.03956,(((338655:0.02179,202239:0.01448)1.000.1999:0.04996,((237582:0.01022,86075:0.01454)1.000.2000:0.03906,((273062:0.03106,242797:0.02864)0.854.39:0.01341,(254731:0.05027,((518323:0.01987,(191128:0.01846,296920:0.03224)0.269.7:0.00664)0.487.8:0.0065,(266224:0.02425,(271883:0.01913,278104:0.03129)0.961.94:0.01125)0.894.56:0.01017)0.809.20:0.01081)0.627.7:0.00844)0.982.154:0.02378)0.903.63:0.01151)0.516.7:0.00871,((150640:0.02344,508778:0.01586)1.000.2001:0.02988,((((574380:0.01689,(203322:0.03225,311487:0.02596)0.735.25:0.0014)0.355.7:0.00399,(328483:0.03788,573896:0.01648)1.000.2002:0.03281)0.881.58:0.00732,257221:0.03011)0.841.34:0.00466,((545808:0.02052,556620:0.0084)1.000.2003:0.03691,327743:0.01547)0.900.58:0.01142)1.000.2004:0.0537)0.995.239:0.01967)0.993.174:0.02924)0.999.553:0.03443)0.457.6:0.01049)0.989.143:0.01901)0.999.554:0.04829)0.988.173:0.02955)0.904.64:0.01572)0.894.57:0.01283)0.921.70:0.00961,(262268:0.04613,(131205:0.0174,266422:0.01714)1.000.2005:0.06088)0.995.240:0.03974)1.000.2006:0.05131)0.663.8:0.00686)0.995.241:0.0124)0.789.31:0.00348)0.597.10:0.00951)0.990.155:0.01544,(((246054:0.00657,568220:0.02058)1.000.2007:0.08845,((50283:0.0389,(((540687:0.01022,101299:0.01659)0.960.111:0.01171,(11924:0.01141,286244:0.02333)0.999.555:0.02585)1.000.2008:0.02954,(162513:0.02648,(97060:0.01446,281663:0.01707)0.973.106:0.01572)0.755.34:0.00907)0.995.242:0.0243)0.125.4:0.00541,(((559072:0.01834,538538:0.01399)0.138.6:0.00619,(514985:0.02063,(((142715:0.01992,154687:0.00807)0.972.109:0.00684,545186:0.01243)0.660.5:0.00234,156539:0.01074)0.783.28:0.00182)0.603.9:0.00742)1.000.2009:0.0333,(584680:0.03111,164797:0.02498)0.946.69:0.01099)0.848.42:0.00531)1.000.2010:0.04643)1.000.2011:0.05823,((51617:0.02273,11591:0.01675)0.999.556:0.0265,((((143031:0.07032,(227819:0.03571,(((315006:0.05022,((232800:0.0071,351744:0.01133)1.000.2012:0.07444,((51091:0.03107,((11375:0.04476,((((((182894:0.02638,302545:0.02698)0.169.4:0.00226,(130587:0.02263,(263695:0.01878,133178:0.05446)0.266.7:0.01007)0.967.99:0.01173)0.935.80:0.00626,((((132749:0.02425,(231510:0.00093,168047:0.00654)0.994.206:0.01058)0.347.5:0.00362,((297831:0.01979,(295165:0.00724,295186:0.00801)0.859.50:0.00375)1.000.2013:0.02669,((252801:0.00478,(175843:0.01432,(245324:0.00096,199727:0.02201)0.998.355:0.01195)0.987.190:0.00822)0.595.4:0.00015,(195061:0.0178,(173965:0.02244,198774:0.02361)0.755.35:0.00169)0.910.58:0.0049)0.950.102:0.00445)0.814.22:0.00475)0.838.36:0.00684,((419024:0.05247,((373939:0.02117,267123:0.00309)0.978.118:0.01311,(188581:0.03357,332547:0.02514)0.864.60:0.01429)0.699.8:0.01056)0.967.100:0.01147,268417:0.04118)0.983.130:0.01238)0.980.120:0.01442,((230578:0.0015,364198:0.00043)1.000.2014:0.03185,((272555:0.02969,512048:0.02504)0.992.187:0.01732,((560141:0.01012,184351:0.00215)0.999.557:0.02774,(146867:0.01988,351639:0.02833)0.936.77:0.01535)0.791.30:0.01778)0.359.2:0.00773)0.842.45:0.00825)0.963.99:0.00879)0.961.95:0.00734,(11365:0.04974,((((11367:0.0049,585313:0.01501)1.000.2015:0.01917,(((470150:0.01344,334224:0.00308)0.983.131:0.01115,(((414949:0.00014,145636:0.00336)0.747.25:0.00782,(300654:0.04742,287705:0.0195)0.466.12:0.01468)0.988.174:0.01125,545018:0.00298)0.836.31:0.00212)0.970.103:0.00969,((187312:0.00029,572680:0.00253)0.977.127:0.00791,(585054:0.04921,((196712:0.06995,470146:0.00304)0.344.5:0.00092,(196639:0.01453,176814:0.01534)1.000.2016:0.04174)0.270.5:0.00014)0.994.207:0.02035)0.992.188:0.01301)0.993.175:0.01408)0.960.112:0.01226,(((426227:0.01623,100258:0.0258)0.992.189:0.01221,(((269678:0.02679,((584954:0.00344,263650:0.00014)0.700.18:0.00105,262384:0.0303)0.982.155:0.01004)0.821.18:0.00428,(((551179:0.01095,521275:0.00324)0.975.118:0.00701,(((27160:0.02526,29393:0.00576)0.528.10:0.00326,266446:0.0118)0.928.68:0.00672,(530138:0.00648,(((566154:0.01588,170281:0.02052)0.901.52:0.0076,(582069:0.01547,((11379:0.01596,11380:0.00316)1.000.2017:0.02941,(484296:0.02061,529659:0.00273)0.998.356:0.01233)0.843.33:0.0038)0.908.66:0.00853)0.933.76:0.00538,(87668:0.03019,469663:0.01825)0.974.111:0.00742)0.957.120:0.00715)0.176.4:0.00265)0.198.5:0.00264)0.206.7:0.00197,((163566:0.00248,161878:0.01084)0.972.110:0.00729,111683:0.00529)0.842.46:0.01313)0.846.26:0.00396)0.990.156:0.01237,461524:0.02929)0.951.78:0.00754)0.768.35:0.00667,(301004:0.02184,139221:0.01442)0.730.22:0.0098)1.000.2018:0.03176)1.000.2019:0.02397,(((231940:0.00145,560981:0.00016)1.000.2020:0.03865,(125665:0.01334,(191823:0.00847,324824:0.01171)0.967.101:0.00732)0.987.191:0.0116)0.519.5:0.00976,(594084:0.03652,(471157:0.01613,576959:0.01311)0.996.260:0.01525)0.477.7:0.00171)0.979.129:0.01083)0.917.70:0.00856)0.957.121:0.00808)0.903.64:0.00484,(546227:0.01381,(136127:0.01486,175508:0.00299)0.305.4:0.00286)1.000.2021:0.02893)0.860.50:0.00611,(89460:0.00983,(175907:0.01448,11372:0.00906)0.997.279:0.01477)0.872.66:0.00922)0.966.103:0.01099)0.697.13:0.00404,(187322:0.00225,231565:0.00107)0.937.77:0.02412)0.956.86:0.00734)0.797.21:0.00643,((32126:0.00266,292011:0.00167)1.000.2022:0.0517,(147031:0.00142,337884:0.03234)0.997.280:0.01948)0.799.26:0.00652)0.982.156:0.01817)0.975.119:0.01715)0.826.22:0.00768,(593352:0.01855,(257095:0.01286,(235127:0.0133,181468:0.01491)0.390.7:0.00157)0.857.41:0.00506)0.998.357:0.01479)0.991.176:0.01427,(((147370:0.02061,128261:0.01346)0.536.8:0.0071,((580258:0.01757,541074:0.00342)0.946.70:0.00468,(135069:0.03619,565593:0.01446)0.494.9:0.00016)0.438.5:0.00303)0.582.7:0.00263,(200827:0.02718,(143800:0.02625,(78283:0.01316,142444:0.01807)0.717.12:0.00245)0.650.9:0.00586)0.847.34:0.00825)0.987.192:0.01348)0.615.10:0.00975)0.508.9:0.01074)0.997.281:0.02327,((135349:0.01991,78335:0.02876)1.000.2023:0.04112,(76995:0.0094,569953:0.0187)0.911.61:0.0125)0.671.12:0.0133)0.231.4:0.0107,(559533:0.03276,(94574:0.02528,75442:0.01408)0.531.9:0.00718)0.604.9:0.00542)1.000.2024:0.05306,((((66768:0.03832,((236583:0.01708,236596:0.02116)0.999.558:0.02068,(((236024:0.03004,(((349634:0.02427,(((114136:0.01944,305277:0.00309)0.727.17:0.00139,(((11359:0.01048,11358:0.01653)0.862.57:0.00467,574032:0.00909)0.852.62:0.0109,582337:0.01442)0.905.68:0.00385)0.959.87:0.00528,((((401635:0.00623,(371783:0.01299,254049:0.01169)0.938.91:0.00631)0.866.44:0.00271,252569:0.01106)0.990.157:0.00831,((((((149649:0.00014,(247938:0.0088,(72508:0.00752,500312:0.00563)0.868.61:0.00158)0.930.73:0.00576)0.857.42:0.0048,529086:0.00658)0.640.8:0.00251,(239050:0.004,(570893:0.01241,252979:0.00837)0.881.59:0.00279)0.968.95:0.00477)0.712.13:0.00096,566193:0.00977)0.812.37:0.0013,(251135:0.01418,203418:0.02819)0.193.4:0.00163)0.841.35:0.00182,(190707:0.0078,589372:0.01565)0.996.261:0.01086)0.902.47:0.00388)0.970.104:0.006,(((154017:0.01318,((250637:0.00515,(149358:0.01454,267851:0.01478)0.776.31:0.00676)0.984.133:0.01145,((248034:0.00908,452924:0.01752)0.883.51:0.00279,(125555:0.03475,(111050:0.00402,239059:0.02807)0.863.51:0.00682)0.910.59:0.00453)0.844.29:0.00486)0.893.51:0.00438)0.878.60:0.00297,(533434:0.00977,580290:0.00978)0.988.175:0.00889)0.615.11:0.00389,(159851:0.02084,((146359:0.0065,361249:0.02035)1.000.2025:0.02687,(251712:0.03865,(314357:0.0208,(223764:0.00736,538427:0.01224)0.275.3:0.01438)1.000.2026:0.0532)0.951.79:0.01388)0.040.5:0.00834)0.842.47:0.00805)0.879.61:0.00015)0.932.69:0.00289)0.815.19:0.0018)0.998.358:0.0119,(((223672:0.01878,138529:0.04719)0.981.122:0.01126,(24067:0.01995,(200245:0.01449,(214033:0.01348,223743:0.00708)0.834.36:0.00401)0.874.53:0.00619)0.912.53:0.00438)0.843.34:0.00215,217548:0.00772)0.924.61:0.00371)0.927.75:0.00382,(((244375:0.01718,(243412:0.00282,580173:0.03475)0.878.61:0.00267)0.972.111:0.0078,513327:0.01004)0.524.5:0.0053,88335:0.02012)0.979.130:0.00942)0.481.5:0.00287)0.824.21:0.00223,(((90198:0.02515,(255859:0.01625,19731:0.00921)0.767.40:0.00727)0.936.78:0.00649,217448:0.01305)0.737.16:0.00184,250264:0.00687)0.944.71:0.00525)0.958.108:0.00519,(593430:0.02895,((325303:0.02385,(((256705:0.02102,538368:0.01964)0.825.24:0.00866,((544147:0.01379,583293:0.00963)0.836.32:0.00783,(251034:0.01774,(209882:0.00655,(509881:0.03074,561981:0.02266)0.861.56:0.00732)0.901.53:0.00394)0.991.177:0.0101)0.978.119:0.00969)0.933.77:0.00683,((((89763:0.0115,252993:0.0212)0.331.12:0.00245,(151564:0.00664,113997:0.0086)1.000.2027:0.0347)0.465.5:0.00305,550165:0.01076)0.839.44:0.00177,238013:0.02233)0.408.14:0.00195)0.907.62:0.00472)0.726.19:0.00274,235293:0.02778)0.860.51:0.00337)0.921.71:0.00647)0.835.31:0.00354)0.513.5:0.00776)0.998.359:0.02382,(((235064:0.06086,(225982:0.00085,214806:0.00095)1.000.2028:0.0436)0.946.71:0.018,((((239214:0.02438,(318193:0.01386,(583767:0.00887,((547544:0.00985,(250652:0.01197,322589:0.0261)1.000.2029:0.02178)0.749.25:0.00099,(534856:0.01772,(591906:0.01142,368520:0.00879)0.807.21:0.00219)0.924.62:0.00326)0.847.35:0.00179)0.312.6:0.00148)0.886.48:0.00408)0.955.83:0.00747,((((243579:0.0246,(585888:0.01147,240349:0.01823)0.993.176:0.01153)0.939.91:0.00802,584366:0.02496)0.982.157:0.01019,((588492:0.00743,(552857:0.00482,362794:0.00891)0.996.262:0.01208)0.955.84:0.01345,(((250268:0.02678,239437:0.01033)0.897.59:0.01704,586961:0.03137)0.960.113:0.00679,((513760:0.02863,(219818:0.00399,103760:0.01247)0.865.48:0.00381)0.859.51:0.00198,233990:0.00826)0.871.57:0.00284)0.293.3:0.00248)0.876.56:0.00473)0.950.103:0.00406,((583887:0.00557,(550664:0.01227,255525:0.01564)0.889.50:0.00318)0.999.559:0.0149,(282759:0.03306,(254092:0.01742,(562566:0.03874,(240169:0.0199,(584642:0.01512,121120:0.01452)0.977.128:0.0073)0.731.26:0.00302)0.864.61:0.00565)0.861.57:0.00321)0.661.12:0.00265)0.959.88:0.00467)0.741.25:0.00081)0.783.29:0.00227,(((249770:0.01377,((240881:0.01805,547206:0.01677)0.742.35:0.00074,216189:0.01879)0.632.5:0.00262)0.000.72:0.00014,(572569:0.0288,509116:0.01071)0.334.6:0.00272)0.977.129:0.00726,((((((240793:0.00472,((240172:0.01049,(314789:0.00405,589383:0.00553)0.954.91:0.01587)0.949.86:0.01185,566165:0.01997)0.169.5:0.00569)0.989.144:0.00974,(27571:0.01591,305385:0.0219)0.844.30:0.00221)0.925.63:0.0027,251277:0.02419)0.747.26:0.00105,((255658:0.01233,(582187:0.02607,593325:0.01552)0.908.67:0.00504)0.993.177:0.00896,356027:0.0122)0.690.11:0.0032)0.908.68:0.0028,(235807:0.01351,((227361:0.03022,236252:0.0119)0.397.14:0.00395,(368831:0.00698,141167:0.00457)0.979.131:0.00918)0.623.6:0.00402)0.919.71:0.00412)0.918.76:0.00265,(((((238666:0.0204,572125:0.01204)0.975.120:0.00749,((253490:0.02171,573278:0.00853)0.975.121:0.00727,(247724:0.01428,(((256786:0.01865,248299:0.01198)0.414.7:0.00206,563703:0.01206)0.539.8:0.00417,(591747:0.0151,132770:0.01192)0.615.12:0.00395)0.548.12:0.00405)1.000.2030:0.01141)0.160.3:0.00187)0.984.134:0.00818,(534057:0.03555,(136780:0.01283,560846:0.00753)0.883.52:0.00465)0.414.8:0.00227)0.750.25:0.00335,(((239095:0.02393,563350:0.01614)0.850.34:0.00251,(244063:0.01655,(((542025:0.00807,(210666:0.04102,((368027:0.01201,78720:0.01194)0.721.14:0.00078,350375:0.02496)0.895.65:0.00316)0.743.33:0.00064)1.000.2031:0.01274,593718:0.02505)0.911.62:0.00315,(239952:0.00881,572494:0.0151)0.998.360:0.01069)0.249.4:0.00014)0.968.96:0.00481)0.862.58:0.00243,(258935:0.01592,539978:0.01789)0.928.69:0.00478)0.177.2:0.00445)0.089.2:0.00016,(((573529:0.01665,(252595:0.02081,217980:0.01903)0.970.105:0.00551)0.859.52:0.00159,(254009:0.02168,((248114:0.01594,332167:0.01059)0.609.6:0.00212,243586:0.02225)0.881.60:0.00395)0.936.79:0.00411)0.919.72:0.00272,251112:0.01597)0.940.93:0.00358)0.946.72:0.00234)0.000.73:0.00014)0.922.71:0.00341)0.927.76:0.00535,((253677:0.02609,(134038:0.02081,253774:0.01738)0.207.2:0.008)0.985.125:0.0153,247772:0.08593)0.745.27:0.01268)0.997.282:0.02327)0.975.122:0.01497,((541327:0.02879,(533016:0.02371,((512663:0.01756,(216558:0.01425,370425:0.00862)0.722.16:0.00062)0.955.85:0.00541,253296:0.03047)0.405.7:0.00054)0.962.105:0.0079)0.991.178:0.01635,(91949:0.03423,577551:0.02392)0.990.158:0.0196)0.698.15:0.00391)0.972.112:0.01345)0.955.86:0.01152,((565173:0.0149,(((580190:0.01373,((242600:0.02306,((137813:0.01572,(237357:0.01605,(168709:0.01725,166076:0.00878)0.598.10:0.00665)0.985.126:0.01211)0.997.283:0.01782,(((105479:0.01511,((241104:0.01741,557397:0.02297)0.984.135:0.01239,155707:0.02855)0.957.122:0.00934)0.931.63:0.00587,257807:0.0056)0.854.40:0.00956,(544140:0.00943,250781:0.01937)0.997.284:0.01592)0.867.55:0.00874)0.213.7:0.00585)0.979.132:0.01197,592796:0.00671)0.870.55:0.00285)0.858.38:0.0089,(251877:0.02309,104603:0.00578)0.997.285:0.04814)0.998.361:0.03164,160994:0.01562)0.901.54:0.01337)1.000.2032:0.06378,(337942:0.00907,591294:0.00811)1.000.2033:0.04557)0.928.70:0.00923)0.792.22:0.01563,(144186:0.02004,535482:0.01348)1.000.2034:0.04235)0.873.58:0.01143)0.890.62:0.009)0.432.5:0.01499)1.000.2035:0.03097)0.966.104:0.01264,((((581342:0.04035,(14307:0.01213,(14306:0.02887,559966:0.00181)0.995.243:0.01576)1.000.2036:0.04315)0.450.10:0.01737,((525208:0.03873,(14304:0.02719,((72053:0.03652,(242318:0.01195,14302:0.01108)0.893.52:0.00734)0.709.9:0.0062,(239530:0.01021,69790:0.02628)0.818.20:0.00675)0.709.10:0.00277)0.567.9:0.00947)0.512.11:0.00868,((((225174:0.01695,534562:0.00984)0.517.6:0.00427,14292:0.00393)0.999.560:0.02193,(200257:0.02463,(574532:0.00285,576104:0.01295)0.796.26:0.0035)0.974.112:0.01145)1.000.2037:0.0409,(288770:0.05226,14300:0.01094)0.017.6:0.04297)0.962.106:0.01554)0.995.244:0.02395)1.000.2038:0.04873,(115290:0.10422,(471315:0.02591,(((316746:0.02939,201587:0.02446)1.000.2039:0.04427,(243756:0.05859,(14313:0.01704,((239617:0.01093,223575:0.00311)0.899.62:0.0092,(14314:0.00945,(49272:0.0177,170629:0.01944)0.957.123:0.00477)0.972.113:0.00751)0.207.3:0.00802)1.000.2040:0.0446)0.958.109:0.01513)0.982.158:0.01528,((14311:0.03077,(247996:0.03553,(153243:0.01277,248744:0.01137)0.927.77:0.00944)0.984.136:0.01411)0.974.113:0.01043,320426:0.05419)0.522.7:0.0047)0.992.190:0.02122)0.945.79:0.01702)1.000.2041:0.03623)0.944.72:0.01611,((260188:0.06365,(106466:0.03258,(160330:0.04418,(134010:0.01549,63105:0.0229)0.972.114:0.01189)0.975.123:0.01538)0.989.145:0.02823)1.000.2042:0.09649,(272785:0.019,267027:0.00816)1.000.2043:0.11904)0.358.4:0.01807)0.991.179:0.02785)0.931.64:0.00895,(((((567823:0.01065,66918:0.02095)0.996.263:0.02468,(1659:0.02544,((249738:0.01334,(563306:0.01632,548557:0.00979)0.962.107:0.00972)1.000.2044:0.0367,(312387:0.00899,1654:0.00094)0.906.70:0.02074)0.207.4:0.00541)0.995.245:0.01808)1.000.2045:0.04231,((326154:0.01237,22245:0.03184)1.000.2046:0.02946,(((218557:0.01884,(223917:0.01076,83693:0.01602)0.737.17:0.00134)0.758.25:0.00469,(294081:0.02549,84679:0.02492)0.942.91:0.00776)0.685.9:0.00506,(((179868:0.0133,((238249:0.00559,(106480:0.02015,(248210:0.00852,19248:0.00956)0.839.45:0.00135)0.725.20:0.00015)0.820.16:0.00015,(587592:0.0109,586929:0.00264)0.968.97:0.00879)0.704.19:0.00078)0.839.46:0.00243,13713:0.01538)0.894.58:0.00333,85667:0.00553)0.956.87:0.00522)0.993.178:0.02044)1.000.2047:0.05783)0.917.71:0.0187,(((256422:0.01458,318438:0.02323)0.931.65:0.01334,((574876:0.01072,((509179:0.01686,(558886:0.01358,((136276:0.00481,289509:0.0177)0.823.23:0.00623,(574772:0.01233,112012:0.00818)0.747.27:0.00566)0.999.561:0.01711)0.673.7:0.0043)1.000.2048:0.01849,518048:0.02217)0.783.30:0.00503)0.635.7:0.0062,((((155824:0.02006,(101372:0.01555,535993:0.02931)0.925.64:0.00964)1.000.2049:0.0203,(575107:0.02602,227521:0.02776)0.827.27:0.00838)0.694.8:0.00563,554294:0.01506)0.951.80:0.00806,352371:0.0179)0.996.264:0.01581)0.427.7:0.00752)1.000.2050:0.10639,((594307:0.00584,295252:0.00798)1.000.2051:0.09388,((246550:0.00466,(90217:0.01161,308650:0.0049)0.453.6:0.0176)0.994.208:0.02219,(329801:0.02196,(500138:0.00795,(517876:0.01345,525035:0.00919)0.101.5:0.00578)0.993.179:0.01964)0.991.180:0.02526)0.941.93:0.03123)0.999.562:0.04273)1.000.2052:0.04196)0.944.73:0.02087,(((((((((((282718:0.01035,66917:0.0274)1.000.2053:0.09884,((247969:0.00177,243587:0.00457)1.000.2054:0.07895,((362406:0.00597,237688:0.00986)0.932.70:0.00565,(1435:0.02747,(238881:0.02824,569984:0.03119)0.719.10:0.00423)0.857.43:0.01048)0.995.246:0.02474)0.917.72:0.02346)0.868.62:0.0164,(((238196:0.02332,((221405:0.01684,535432:0.02887)0.988.176:0.01214,((585940:0.02571,(247439:0.00334,369894:0.00453)1.000.2055:0.01561)0.880.51:0.00539,(549912:0.01513,224566:0.03742)0.601.6:0.00636)0.980.121:0.01119)0.940.94:0.00786)0.369.6:0.00489,(100171:0.02636,((296956:0.01453,247987:0.01144)0.983.132:0.01213,(141847:0.0109,112133:0.01003)1.000.2056:0.02946)0.855.40:0.00822)0.900.59:0.00385)0.999.563:0.02852,(550111:0.03256,((319724:0.01635,32254:0.01133)0.930.74:0.01128,(321601:0.02524,(238705:0.00016,570974:0.03951)0.912.54:0.01966)0.711.10:0.0079)0.996.265:0.02185)1.000.2057:0.03752)0.995.247:0.03274)0.923.68:0.02594,((268685:0.01887,537376:0.02646)1.000.2058:0.07686,(208525:0.04288,((163720:0.01573,155350:0.01035)1.000.2059:0.04822,(((((247640:0.01041,542039:0.01961)0.991.181:0.01225,576477:0.00903)0.827.28:0.00239,((1430:0.01993,537151:0.01348)0.909.60:0.00416,(548377:0.01596,(334992:0.02185,(143747:0.00638,565587:0.01128)0.946.73:0.00516)0.941.94:0.00547)0.723.19:0.00057)0.995.248:0.00776)0.858.39:0.00294,568807:0.02804)0.846.27:0.00441,((250645:0.01412,(352273:0.01728,297697:0.01739)1.000.2060:0.05153)0.985.127:0.01673,(153870:0.00219,519788:0.01723)1.000.2061:0.033)0.814.23:0.00381)0.999.564:0.02596)0.898.65:0.01524)0.955.87:0.0187)0.636.6:0.01166)0.782.24:0.01243,((((243552:0.01588,(235803:0.01051,241653:0.00683)0.965.106:0.00813)0.669.11:0.00848,(255679:0.0215,(239599:0.01067,254133:0.00851)0.987.193:0.01237)0.719.11:0.00797)0.987.194:0.01818,((249907:0.00539,548630:0.01502)0.998.362:0.02219,(246743:0.01864,(237351:0.01098,551234:0.01028)0.923.69:0.0053)0.846.28:0.00713)1.000.2062:0.03732)0.969.108:0.02499,((243245:0.01748,88820:0.0103)1.000.2063:0.05312,((347509:0.02486,306910:0.01937)0.084.5:0.0046,(537856:0.03033,(487233:0.02278,512121:0.02416)1.000.2064:0.02539)0.879.62:0.00909)0.980.122:0.0193)0.999.565:0.03307)0.993.180:0.02902)0.872.67:0.01441,(159278:0.07235,(((169378:0.01417,(46324:0.01456,450088:0.03489)0.961.96:0.00788)0.798.23:0.00385,(552379:0.02013,162210:0.01725)0.452.14:0.00574)1.000.2065:0.08497,(369522:0.05186,(590848:0.05134,(249601:0.02935,((584471:0.00982,((221811:0.02027,550218:0.03753)0.963.100:0.01029,225301:0.01429)0.939.92:0.00852)0.973.107:0.01494,589390:0.01707)0.862.59:0.02326)0.949.87:0.01367)0.200.8:0.00665)1.000.2066:0.09667)0.870.56:0.02521)0.949.88:0.01999)1.000.2067:0.03712,(((238017:0.02353,(((156900:0.01724,56514:0.01048)0.416.6:0.00825,((245862:0.01152,369447:0.02022)0.782.25:0.00694,(100933:0.01296,(113767:0.02308,282328:0.00863)0.997.286:0.01776)0.446.7:0.00481)0.970.106:0.01085)0.749.26:0.00681,241676:0.03451)0.393.6:0.00339)1.000.2068:0.0967,(239967:0.12391,((212739:0.01342,(240556:0.00882,(235372:0.00909,253007:0.00025)0.720.17:0.00911)0.961.97:0.00955)1.000.2069:0.08732,(((580909:0.02185,571106:0.02334)1.000.2070:0.03574,((341554:0.01398,(242750:0.02412,255590:0.01624)0.811.25:0.00427)0.999.566:0.03082,((((165179:0.01312,537729:0.03649)1.000.2071:0.05239,(150628:0.01681,(145137:0.02073,114578:0.01341)0.614.9:0.00965)1.000.2072:0.03796)0.999.567:0.03438,(566264:0.01968,107878:0.01802)0.987.195:0.01617)0.995.249:0.0178,((340964:0.01326,535583:0.02134)1.000.2073:0.0409,(((95632:0.0274,(98073:0.01887,(26128:0.0212,153837:0.0105)0.664.8:0.00665)0.705.6:0.0055)0.986.143:0.01433,((17020:0.01845,(100865:0.01545,543306:0.03058)0.865.49:0.00533)0.986.144:0.01208,(142798:0.03798,((258263:0.01113,(206186:0.00728,(277519:0.00842,289202:0.01048)0.990.159:0.01335)0.503.8:0.00698)0.965.107:0.01833,((((577671:0.00093,((319292:0.01157,266399:0.00649)0.883.53:0.01395,(352046:0.02258,325545:0.00493)0.546.5:0.0098)0.945.80:0.00592)0.933.78:0.01115,259908:0.01352)0.858.40:0.00579,275085:0.00939)0.978.120:0.01037,(39443:0.01632,((423647:0.01268,133464:0.01869)0.847.36:0.00155,(169901:0.01439,129972:0.0254)0.985.128:0.0073)0.901.55:0.00328)0.953.89:0.00909)0.962.108:0.01751)1.000.2074:0.04112)0.904.65:0.01574)0.986.145:0.02155)0.590.10:0.00751,((((((587488:0.01532,232777:0.02097)0.756.35:0.00833,(72347:0.01415,(242796:0.0127,21239:0.01927)0.893.53:0.00643)0.932.71:0.00752)0.746.36:0.00425,34261:0.01457)0.940.95:0.00517,(536952:0.01285,(446772:0.00695,(360054:0.01411,((257264:0.00538,437281:0.02895)0.866.45:0.00332,113976:0.01253)0.947.86:0.00525)0.751.19:0.00232)0.972.115:0.00698)0.794.28:0.00372)0.886.49:0.0049,(351745:0.01917,(134305:0.01328,149506:0.01405)0.940.96:0.00576)0.990.160:0.01197)0.974.114:0.00695,((((249103:0.01286,219538:0.02104)0.899.63:0.00562,((529843:0.01108,(151868:0.01428,((284812:0.0113,593890:0.00133)0.566.11:0.00609,372002:0.02162)0.984.137:0.00926)0.969.109:0.00747)0.547.9:0.00348,((525269:0.01726,((93147:0.02173,(292471:0.01764,70379:0.00531)0.064.3:0.00181)0.999.568:0.01652,(65419:0.01505,(145098:0.01558,(65418:0.00942,500117:0.02017)0.543.5:0.00407)0.882.54:0.00715)0.704.20:0.00697)0.910.60:0.01078)0.980.123:0.00968,194242:0.03168)0.936.80:0.00343)0.998.363:0.01196)0.966.105:0.00888,(136076:0.01312,(253988:0.02404,(22963:0.00666,344092:0.01734)0.901.56:0.00726)0.998.364:0.0163)0.995.250:0.01563)0.939.93:0.00649,((308780:0.03868,(101844:0.02186,449146:0.01813)1.000.2075:0.02263)0.999.569:0.0189,((240941:0.01819,65697:0.0167)0.841.36:0.00689,(520961:0.0286,141249:0.01448)0.091.3:0.00364)0.979.133:0.0142)0.738.23:0.00046)0.955.88:0.00556)0.983.133:0.01421)0.821.19:0.00389)0.704.21:0.01137)0.998.365:0.02812)0.788.23:0.01568)0.970.107:0.0246,((((((91512:0.01858,532665:0.02347)1.000.2076:0.02355,((518809:0.0374,(107654:0.01119,513864:0.03353)1.000.2077:0.01934)0.993.181:0.01464,(159654:0.02256,566066:0.03128)0.987.196:0.01356)0.728.22:0.00299)0.898.66:0.01178,((156885:0.01825,63207:0.04022)1.000.2078:0.03522,(((248392:0.01035,572143:0.01793)0.809.21:0.00793,89159:0.03284)0.613.7:0.00803,((321742:0.01401,546081:0.0182)0.567.10:0.00744,(((566086:0.01273,(152373:0.0078,245146:0.00862)0.992.191:0.00844)0.911.63:0.00482,557742:0.01751)0.871.58:0.00352,571651:0.01774)0.855.41:0.00503)0.978.121:0.00962)0.701.7:0.0096)0.694.9:0.00348)0.936.81:0.00537,(((377511:0.02007,234307:0.03966)1.000.2079:0.03219,((245400:0.01062,238281:0.01483)1.000.2080:0.03194,(333899:0.00783,336607:0.02204)0.996.266:0.01918)0.785.29:0.01247)0.449.7:0.00719,((251050:0.02667,(111978:0.02385,(251920:0.02862,((168747:0.01003,574969:0.01106)0.930.75:0.00505,(536464:0.01489,549319:0.01773)0.870.57:0.00374)0.862.60:0.00505)0.985.129:0.01058)0.551.4:0.00687)0.511.7:0.00064,((316001:0.01466,(257656:0.01805,156783:0.03344)0.886.50:0.00992)0.858.41:0.00243,(279572:0.00866,244425:0.02635)0.853.40:0.00391)0.874.54:0.00363)0.990.161:0.01215)0.908.69:0.00417)0.960.114:0.00904,(559880:0.01528,515209:0.03095)0.999.570:0.02361)0.929.87:0.00889,(65695:0.02391,520918:0.01857)1.000.2081:0.03217)0.996.267:0.02455)1.000.2082:0.05787)0.819.26:0.02457)0.988.177:0.03492)0.983.134:0.02101,((((((242431:0.04735,(((255765:0.02472,((534444:0.04263,(86409:0.02293,534443:0.02159)0.999.571:0.02411)0.737.18:0.00509,(144673:0.01716,142691:0.02263)1.000.2083:0.03054)0.825.25:0.00578)0.951.81:0.01254,(((141911:0.05509,(590745:0.00015,37981:0.01381)0.204.4:0.0027)0.962.109:0.00746,(562734:0.0173,537633:0.01304)0.864.62:0.00503)0.950.104:0.00707,256207:0.03524)0.722.17:0.00101)0.947.87:0.01663,(11188:0.01825,562277:0.02711)0.897.60:0.00894)0.993.182:0.02678)1.000.2084:0.12765,159241:0.10772)0.987.197:0.04231,((108884:0.04894,(144617:0.00809,(352496:0.01503,581615:0.01359)0.995.251:0.02002)1.000.2085:0.04446)0.999.572:0.03716,243744:0.08962)0.447.7:0.01907)0.838.37:0.01496,(((264825:0.0106,523133:0.03141)0.998.366:0.03412,((238795:0.03189,279458:0.03685)1.000.2086:0.04849,((241608:0.02663,65673:0.01588)1.000.2087:0.04271,(237494:0.02291,216289:0.03309)1.000.2088:0.05909)0.135.5:0.01393)0.996.268:0.0324)1.000.2089:0.08374,510313:0.08634)0.326.10:0.01276)1.000.2090:0.06985,((250492:0.18412,((205209:0.03255,200752:0.01749)1.000.2091:0.07588,(206828:0.01217,(225432:0.02565,202068:0.02212)0.994.209:0.02309)1.000.2092:0.06563)0.998.367:0.05139)1.000.2093:0.0851,((512414:0.12167,(((220280:0.03696,540167:0.0131)1.000.2094:0.10764,((108824:0.03481,(13864:0.0286,577547:0.02759)0.729.18:0.0065)1.000.2095:0.07185,((278524:0.01575,346434:0.02294)1.000.2096:0.05377,((223552:0.01744,556248:0.01291)1.000.2097:0.05704,(224443:0.01914,284266:0.02468)0.997.287:0.02693)0.989.146:0.0251)0.999.573:0.04453)0.638.6:0.02898)0.788.24:0.02454,(537245:0.1072,((535062:0.01951,528049:0.01546)1.000.2098:0.1,(99699:0.01782,35504:0.02602)1.000.2099:0.09245)0.669.12:0.02758)0.977.130:0.02887)0.931.66:0.01457)0.992.192:0.02646,(((202085:0.01898,(243448:0.00311,(201852:0.0035,201200:0.02066)0.769.26:0.00179)0.681.16:0.00135)1.000.2100:0.14215,112352:0.11129)0.941.95:0.0341,(((108872:0.10988,541294:0.08217)0.971.139:0.03724,(65700:0.10255,((218322:0.01612,529088:0.02299)1.000.2101:0.04687,(225148:0.02058,106562:0.02925)1.000.2102:0.04903)0.979.134:0.03167)0.975.124:0.03342)1.000.2103:0.11277,569405:0.18282)0.983.135:0.03929)0.984.138:0.03113)0.568.9:0.0143)0.997.288:0.03944)0.961.98:0.02438,((237518:0.18377,(254839:0.1879,(534156:0.07225,((256579:0.00588,(316970:0.0104,236710:0.01266)0.843.35:0.00463)1.000.2104:0.04998,(((263809:0.01615,200595:0.01001)0.993.183:0.0165,(590013:0.03484,(203106:0.02772,(202038:0.01659,(550488:0.01134,(202478:0.04305,201574:0.01958)0.505.10:0.00564)0.927.78:0.00603)0.917.73:0.00769)0.864.63:0.00801)0.112.3:0.00609)1.000.2105:0.03305,(260329:0.00951,(202078:0.03147,202711:0.01316)0.484.6:0.00389)0.996.269:0.02801)0.641.11:0.01474)0.992.193:0.03536)0.989.147:0.0408)0.921.72:0.02768)0.991.182:0.03962,((113245:0.01825,142268:0.02553)1.000.2106:0.12937,((202139:0.02644,509215:0.03022)1.000.2107:0.13985,(((((230087:0.01567,200037:0.01892)1.000.2108:0.06669,(199942:0.01571,255464:0.01589)1.000.2109:0.06956)0.999.574:0.05734,((244338:0.00885,206209:0.03159)1.000.2110:0.14763,201986:0.15016)0.320.7:0.03263)0.639.11:0.02012,(((264265:0.00908,(((103467:0.00663,510976:0.00769)0.616.8:0.00328,141967:0.00724)0.953.90:0.01001,(271039:0.00862,10566:0.01724)0.711.11:0.00769)0.930.76:0.00844)0.997.289:0.0403,((81179:0.00492,340415:0.02003)0.962.110:0.02718,(80684:0.02724,327380:0.02251)0.891.62:0.01207)1.000.2111:0.05347)1.000.2112:0.15212,(((200590:0.00182,(201331:0.01072,201680:0.02006)0.949.89:0.01564)1.000.2113:0.09286,201519:0.08034)0.950.105:0.03231,(200616:0.12076,(202937:0.01087,(202730:0.00391,(201044:0.0183,203202:0.0057)0.918.77:0.00467)0.924.63:0.00873)1.000.2114:0.08847)0.943.92:0.02716)1.000.2115:0.13487)0.968.98:0.04313)0.887.48:0.02971,((200948:0.11414,((225404:0.02207,210172:0.00941)1.000.2116:0.16457,269947:0.06602)0.557.4:0.03149)0.997.290:0.05985,((593853:0.06148,((259108:0.04508,(10560:0.02132,42849:0.01257)0.974.115:0.02089)1.000.2117:0.07923,582694:0.05863)0.291.11:0.01748)0.999.575:0.08471,(258996:0.1093,(253554:0.09338,136414:0.06548)0.995.252:0.03817)0.967.102:0.03851)0.999.576:0.05821)0.935.81:0.03658)0.997.291:0.04137)0.988.178:0.03569)0.979.135:0.03587)0.998.368:0.03162)0.482.10:0.01565)1.000.2118:0.04983)0.509.7:0.01053,(((((161537:0.02677,(223797:0.02895,(136311:0.02594,78594:0.02654)0.625.10:0.01389)0.869.57:0.01032)1.000.2119:0.08635,(((1282:0.09069,528078:0.0242)0.984.139:0.02206,((((247875:0.01074,(85874:0.00606,508084:0.00699)0.876.57:0.00607)0.954.92:0.01659,((521078:0.00584,(516783:0.00366,(90766:0.01817,137866:0.0062)0.575.8:0.00238)0.931.67:0.00492)0.987.198:0.01305,((566611:0.00991,521433:0.01032)0.975.125:0.01317,(542927:0.00497,(523561:0.0178,527277:0.00661)0.891.63:0.00591)1.000.2120:0.0355)0.349.8:0.00666)0.930.77:0.01267)1.000.2121:0.04176,((213308:0.02401,509610:0.00827)0.999.577:0.0226,((565457:0.00669,418721:0.01009)1.000.2122:0.03208,(549954:0.01627,534516:0.01349)0.987.199:0.01508)0.998.369:0.0269)0.889.51:0.01214)0.994.210:0.03289,((516459:0.01707,573695:0.00924)1.000.2123:0.04281,(349986:0.02935,113628:0.03583)0.991.183:0.02353)0.995.253:0.02516)0.600.5:0.01033)0.998.370:0.0282,(245517:0.11786,(((251486:0.04721,((307846:0.02796,584457:0.01793)1.000.2124:0.04159,(253655:0.02113,(101617:0.03959,159054:0.03037)0.257.6:0.00975)0.893.54:0.0147)0.912.55:0.02245)0.881.61:0.01077,(1277:0.0463,(529982:0.00549,(529380:0.0191,72831:0.01396)0.972.116:0.01057)1.000.2125:0.05228)0.990.162:0.02396)0.997.292:0.02474,((235028:0.01942,(73282:0.01626,254288:0.0077)0.869.58:0.00845)1.000.2126:0.05525,((28650:0.01891,541046:0.02317)1.000.2127:0.0899,((((248841:0.01903,329622:0.00805)0.996.270:0.02462,(1281:0.01529,(113942:0.01831,(103239:0.01786,37175:0.01121)0.234.5:0.00425)0.978.122:0.0115)0.995.254:0.01918)0.376.8:0.0142,(((579215:0.0032,238495:0.01261)0.938.92:0.00811,544460:0.02362)1.000.2128:0.03694,((240467:0.01686,(253437:0.01262,(241781:0.00558,65689:0.0087)0.147.5:0.00182)0.983.136:0.01488)0.999.578:0.02408,((241985:0.01436,269768:0.00492)0.986.146:0.01218,(((563358:0.01565,559278:0.01292)0.177.3:0.0027,(510471:0.01089,(238766:0.01002,(220353:0.00709,((225041:0.0074,(289530:0.01314,588861:0.0105)0.959.89:0.00799)0.941.96:0.00671,217917:0.00854)0.925.65:0.00459)0.781.23:0.00573)0.955.89:0.00686)0.384.8:0.00369)0.858.42:0.00554,(545548:0.02051,253621:0.02766)0.800.31:0.00309)0.744.25:0.00821)0.998.371:0.0239)0.972.117:0.02294)0.934.64:0.01869)0.955.90:0.02078,((273949:0.01862,591650:0.01947)1.000.2129:0.06399,579686:0.08741)0.856.40:0.02197)0.175.8:0.01015)0.474.4:0.01572)1.000.2130:0.04335)0.995.255:0.03236)1.000.2131:0.05751)0.974.116:0.02628)0.986.147:0.02482,(531464:0.08098,(551024:0.00862,239002:0.03549)1.000.2132:0.05387)0.969.110:0.01696)0.875.48:0.00664,(((((212287:0.01174,(242804:0.02498,225131:0.01172)0.331.13:0.01146)1.000.2133:0.1022,((544441:0.02343,(494235:0.01606,551069:0.01353)0.146.5:0.00573)1.000.2134:0.02959,(329466:0.08555,(282711:0.00678,(((82007:0.017,320629:0.01878)0.952.96:0.01361,(((522273:0.01407,(566107:0.00966,134616:0.0163)0.964.120:0.00492)0.930.78:0.00416,560690:0.01116)0.968.99:0.00362,(206278:0.01711,(113054:0.01495,212027:0.01343)0.793.21:0.0055)0.758.26:0.00098)0.918.78:0.00446)0.956.88:0.0078,((206963:0.02658,((593363:0.00518,(533602:0.01131,(256904:0.00653,549321:0.01373)0.957.124:0.00653)0.842.48:0.00604)1.000.2135:0.02149,512939:0.01713)0.629.10:0.00334)0.731.27:0.0012,(112929:0.01552,(113232:0.01948,(218985:0.0031,284171:0.01033)0.854.41:0.00428)0.597.11:0.00717)0.910.61:0.00671)0.959.90:0.00609)0.940.97:0.00752)0.914.60:0.01143)0.948.93:0.01295)0.846.29:0.01561)1.000.2136:0.03418,(166792:0.08079,((217784:0.026,100475:0.01176)1.000.2137:0.07877,((215100:0.02356,565783:0.02942)1.000.2138:0.07769,((204816:0.01637,130709:0.00898)1.000.2139:0.10655,((246345:0.00934,250890:0.01527)1.000.2140:0.07611,(((156599:0.01605,104930:0.0284)1.000.2141:0.04862,(((238971:0.01353,316520:0.0075)1.000.2142:0.0618,((559181:0.00089,105343:0.01888)1.000.2143:0.04812,(104491:0.01972,513165:0.02379)1.000.2144:0.05007)0.968.100:0.02759)0.962.111:0.02091,((((321040:0.02001,(((560616:0.00726,509188:0.01243)0.997.293:0.01748,581736:0.02182)0.949.90:0.00686,((557636:0.01838,(537728:0.03089,580677:0.01381)0.767.41:0.00685)0.979.136:0.01099,(254105:0.02815,(240228:0.01208,592361:0.00937)0.929.88:0.00408)0.972.118:0.00746)0.879.63:0.00421)0.933.79:0.00686)1.000.2145:0.02713,(100738:0.07212,(160268:0.01152,224491:0.0228)0.998.372:0.03228)0.873.59:0.01221)0.839.47:0.01782,217400:0.0561)0.397.15:0.01151,159894:0.06492)0.429.5:0.00868)0.657.5:0.01)0.734.21:0.01157,(((218887:0.01509,(226220:0.00941,(160354:0.03734,160289:0.00504)0.335.2:0.00345)1.000.2146:0.02406)0.229.7:0.01097,154958:0.02197)0.912.56:0.00967,((226026:0.01684,(236930:0.02102,217959:0.01308)0.883.54:0.00412)0.998.373:0.02471,((222240:0.0246,525657:0.03182)0.999.579:0.02525,(520338:0.04373,(((181846:0.00984,161567:0.01054)0.969.111:0.00912,(244455:0.01402,(257714:0.03868,(222810:0.00481,285767:0.02938)0.962.112:0.00895)0.967.103:0.01155)0.478.5:0.00383)0.990.163:0.01275,(285320:0.00397,250185:0.00805)0.894.59:0.01819)1.000.2147:0.03316)0.909.61:0.0144)0.931.68:0.01219)0.956.89:0.01723)0.920.59:0.01067)0.990.164:0.02252)0.991.184:0.03821)0.995.256:0.03376)0.955.91:0.02298)0.983.137:0.02561)0.497.12:0.00521)0.941.97:0.01148,((((246833:0.02279,113170:0.03002)1.000.2148:0.0365,(((247231:0.01387,494253:0.02025)0.731.28:0.00739,(236377:0.01038,(((325796:0.00671,161451:0.01237)0.984.140:0.00646,204422:0.00553)0.770.24:0.00111,184881:0.0106)0.881.62:0.00385)0.709.11:0.00585)0.828.27:0.0039,239601:0.00746)1.000.2149:0.06558)0.166.7:0.01,(((250099:0.02344,186887:0.01409)0.944.74:0.01159,(225605:0.01388,(251412:0.01988,(242432:0.01606,219565:0.00824)0.950.106:0.00815)0.964.121:0.00828)0.997.294:0.01785)0.997.295:0.02227,((((223752:0.00681,(((114052:0.00624,261663:0.01396)0.866.46:0.00309,113599:0.01761)0.995.257:0.00869,114334:0.00125)0.258.4:0.00142)0.616.9:0.00214,113276:0.0122)0.849.40:0.00727,(494503:0.00746,(112836:0.00372,213333:0.01444)0.496.7:0.002)0.632.6:0.00315)0.860.52:0.00538,235741:0.01741)1.000.2150:0.04427)0.965.108:0.013)1.000.2151:0.04727,(((((249501:0.02694,((510580:0.02993,(241504:0.01473,567113:0.01639)0.880.52:0.00345)0.541.9:0.00433,((65691:0.01362,((559741:0.02222,569476:0.00977)0.777.25:0.00411,568917:0.00916)0.900.60:0.00351)0.141.4:0.00333,206654:0.01771)0.297.5:0.00211)0.488.10:0.00503)1.000.2152:0.03968,(225280:0.01024,273805:0.02204)0.988.179:0.01564)0.972.119:0.01596,((238509:0.0196,(144409:0.01565,226939:0.01557)0.717.13:0.00083)1.000.2153:0.03864,(78515:0.01111,587291:0.02119)1.000.2154:0.05631)0.650.10:0.00898)0.986.148:0.01985,(((583310:0.00688,60171:0.01182)0.408.15:0.05952,353248:0.04485)1.000.2155:0.04586,(((60159:0.01175,592776:0.00482)0.999.580:0.02982,((583206:0.01042,((143212:0.01229,(536088:0.01221,101541:0.02681)0.943.93:0.0047)0.828.28:0.00162,106084:0.01269)0.936.82:0.00348)0.960.115:0.00808,(239300:0.02475,560792:0.02538)0.216.2:0.00817)0.998.374:0.03312)1.000.2156:0.06086,(239349:0.02675,(329203:0.0056,577544:0.01566)1.000.2157:0.03988)1.000.2158:0.03723)0.889.52:0.01588)0.195.4:0.0084)0.988.180:0.0173,(229483:0.03093,537694:0.02935)0.997.296:0.02385)0.969.112:0.01728)0.256.4:0.00276)0.984.141:0.01804,(((203315:0.00982,567811:0.0168)1.000.2159:0.08248,(543362:0.03652,(511930:0.02179,(512873:0.02155,(210026:0.01484,213345:0.02352)0.988.181:0.01252)0.924.64:0.01056)0.925.66:0.01679)0.966.106:0.02147)0.922.72:0.02421,((((108878:0.01415,564046:0.01136)1.000.2160:0.06449,569497:0.09757)0.983.138:0.02259,(((112268:0.0221,13754:0.02254)0.978.123:0.01508,((241502:0.01247,540711:0.02771)1.000.2161:0.03259,((103329:0.04,((559468:0.01109,((576457:0.01985,((13753:0.01635,570070:0.0055)0.986.149:0.0108,((171264:0.01747,(556998:0.01006,253385:0.01204)0.837.24:0.00293)1.000.2162:0.02538,(554951:0.00606,546178:0.01141)0.981.123:0.02365)0.571.12:0.00722)0.843.36:0.00861)0.530.10:0.00238,558845:0.01766)0.801.21:0.0028)0.864.64:0.0042,(273086:0.00802,(567018:0.02472,(593949:0.00549,553650:0.00996)0.778.37:0.00702)0.201.6:0.0027)1.000.2163:0.02503)0.490.7:0.00533)0.830.32:0.0063,(((590414:0.00704,249939:0.00374)0.762.28:0.00083,(106593:0.02419,237866:0.012)0.509.8:0.002)1.000.2164:0.02438,101990:0.06692)0.910.62:0.01327)0.998.375:0.02772)0.996.271:0.01922)1.000.2165:0.02427,((((558878:0.05857,260023:0.04618)0.393.7:0.01222,(562465:0.01203,((256176:0.01134,104227:0.01972)1.000.2166:0.04298,(100538:0.01292,257894:0.01941)0.702.13:0.00583)0.738.24:0.00617)1.000.2167:0.07555)0.436.7:0.00961,(111716:0.05183,(13752:0.08094,((587936:0.01415,136672:0.02011)1.000.2168:0.07093,(((((336039:0.00797,557611:0.01128)0.505.11:0.00388,538252:0.00643)0.757.37:0.00161,536270:0.02615)0.977.131:0.00715,135070:0.0276)0.965.109:0.00742,258174:0.00982)1.000.2169:0.03182)0.849.41:0.01438)0.998.376:0.03288)1.000.2170:0.0533)0.794.29:0.00439,(((100849:0.02179,587703:0.02127)0.998.377:0.02,((580967:0.03254,589266:0.00962)1.000.2171:0.03445,(578581:0.03256,100694:0.01875)0.992.194:0.01309)0.954.93:0.01073)0.645.16:0.00931,(((581015:0.01349,((250510:0.01103,580543:0.00776)0.881.63:0.00691,573945:0.00893)0.879.64:0.00323)1.000.2172:0.03747,(((238932:0.0027,591507:0.0054)0.896.55:0.00473,(535516:0.01386,240741:0.02546)0.029.2:0.00672)1.000.2173:0.02806,(101670:0.00926,(111871:0.01348,569652:0.01932)0.037.4:0.00532)1.000.2174:0.04872)0.974.117:0.01588)0.072.3:0.0072,(111406:0.00462,561898:0.03097)1.000.2175:0.10075)0.884.57:0.00809)1.000.2176:0.03263)0.946.74:0.01043)0.378.11:0.00543)0.982.159:0.02065,((((((224175:0.01373,570224:0.02761)1.000.2177:0.03556,((159156:0.02256,261089:0.02197)1.000.2178:0.04431,((564029:0.03493,(56382:0.00229,572867:0.00403)0.993.184:0.01609)1.000.2179:0.03391,(253195:0.05286,(563464:0.01007,(267006:0.01382,552108:0.01957)0.723.20:0.00632)1.000.2180:0.06069)0.194.2:0.00702)0.545.9:0.01186)1.000.2181:0.04744)0.621.11:0.00798,(513763:0.0474,((167201:0.01292,559475:0.0258)0.943.94:0.00903,((561960:0.01601,269686:0.01807)0.937.78:0.00711,(169163:0.02919,(31890:0.02332,512765:0.02386)0.952.97:0.00757)0.459.4:0.00369)0.479.14:0.00647)0.593.6:0.00877)0.999.581:0.02251)0.999.582:0.02901,((((((((160026:0.02964,(257219:0.01813,102867:0.0398)0.997.297:0.01682)0.758.27:0.00554,105046:0.02998)0.910.63:0.00906,((269946:0.01799,263318:0.01626)0.891.64:0.01065,((258100:0.01863,265942:0.01188)0.565.5:0.00324,(153904:0.01126,587486:0.03098)0.868.63:0.00237)1.000.2182:0.03076)0.791.31:0.01366)0.922.73:0.00878,(219464:0.01737,13747:0.00908)1.000.2183:0.0313)0.720.18:0.00462,(((508953:0.00525,508819:0.0137)0.902.48:0.00393,((111377:0.01305,(273645:0.01899,112030:0.01045)0.955.92:0.0032)0.904.66:0.00334,(562320:0.00287,541331:0.01392)0.780.21:0.00665)0.984.142:0.00775)1.000.2184:0.02305,((((105090:0.0294,549113:0.01588)0.999.583:0.02087,(135968:0.02227,((160017:0.01745,(111801:0.01632,132386:0.0199)0.886.51:0.00663)0.471.11:0.00306,(264887:0.02164,274642:0.01325)0.437.6:0.00256)0.769.27:0.00454)0.995.258:0.01438)0.913.75:0.00647,(277282:0.03019,406679:0.01121)0.998.378:0.01734)0.670.6:0.00383,((112044:0.01367,((551455:0.00177,111379:0.01036)0.171.4:0.00391,511891:0.01753)0.975.126:0.00647)0.463.9:0.00322,(277499:0.01997,(107392:0.0299,534610:0.03054)0.880.53:0.00861)0.976.110:0.01318)0.257.7:0.00261)0.222.8:0.00139)0.810.29:0.00748)0.819.27:0.00533,(((260680:0.00185,547318:0.00516)0.570.11:0.019,(13748:0.01867,217608:0.02893)0.976.111:0.01424)0.977.132:0.01178,((461677:0.01276,143078:0.02572)0.990.165:0.01523,(143037:0.02938,153175:0.0244)1.000.2185:0.04335)1.000.2186:0.02459)0.908.70:0.00559)0.849.42:0.00487,(63106:0.02424,260311:0.0298)0.949.91:0.00693)0.955.93:0.01022,(((227439:0.01493,13743:0.00482)1.000.2187:0.02405,((573684:0.05637,223996:0.04442)0.993.185:0.02858,(74169:0.01873,(148986:0.01695,113904:0.01538)0.921.73:0.00858)1.000.2188:0.03711)0.768.36:0.01014)0.994.211:0.01745,(63128:0.01646,(545811:0.02279,260482:0.03936)0.967.104:0.01149)0.231.5:0.00846)0.990.166:0.01458)0.999.584:0.02392)0.529.8:0.01358,(559658:0.01627,((549034:0.01375,535141:0.0196)0.989.148:0.01816,(111953:0.00955,409759:0.01409)0.989.149:0.01462)0.905.69:0.00869)0.999.585:0.03058)0.925.67:0.01119,(542071:0.09393,(163063:0.03343,(199608:0.0124,569338:0.01728)1.000.2189:0.02917)1.000.2190:0.03527)0.229.8:0.01058)0.995.259:0.02333)0.809.22:0.02127)0.982.160:0.01927)0.444.12:0.00968)0.415.6:0.00565,(((325009:0.00881,(277153:0.01338,272484:0.01967)0.763.28:0.00119)1.000.2191:0.14857,(((581116:0.00695,538902:0.00743)0.868.64:0.00334,(582684:0.00676,(544664:0.00124,590831:0.01739)0.872.68:0.00426)0.935.82:0.00568)1.000.2192:0.05488,((337102:0.01985,(163552:0.00399,(563442:0.01854,69877:0.01271)0.968.101:0.00794)0.994.212:0.01636)0.575.9:0.00768,(551159:0.11436,((150701:0.02351,(81557:0.01099,160746:0.00782)0.995.260:0.01639)0.982.161:0.0116,((582257:0.04773,((357330:0.02729,263276:0.02213)1.000.2193:0.03229,(26613:0.02861,((542211:0.01503,573491:0.01085)0.449.8:0.0043,(509986:0.00926,245129:0.0187)0.435.10:0.00345)0.993.186:0.01825)0.923.70:0.0169)0.924.65:0.01099)0.586.7:0.00841,(((324544:0.00703,(((312421:0.00422,161988:0.01521)0.898.67:0.00305,(113898:0.00342,533078:0.00656)0.915.65:0.00371)0.842.49:0.00186,(262783:0.01114,345978:0.00982)0.546.6:0.00185)0.967.105:0.00907)1.000.2194:0.03515,(349891:0.03165,(534526:0.01823,(193891:0.01076,552001:0.01345)0.855.42:0.00949)0.938.93:0.00844)0.994.213:0.0198)0.949.92:0.01114,((177331:0.01193,342274:0.01219)1.000.2195:0.02614,591501:0.03418)0.981.124:0.01514)0.714.12:0.0075)0.967.106:0.00991)0.367.9:0.00208)0.377.3:0.00491)0.994.214:0.02368)0.945.81:0.02534)0.997.298:0.04048,(((584240:0.00903,(549616:0.0181,253990:0.02516)0.793.22:0.01087)1.000.2196:0.0628,((566274:0.02633,((239489:0.02217,44024:0.01725)0.029.3:0.00659,(586223:0.03858,(215669:0.02301,65688:0.01033)0.876.58:0.00714)0.684.15:0.01172)0.995.261:0.02373)1.000.2197:0.07766,(102361:0.02222,531244:0.01605)1.000.2198:0.04976)1.000.2199:0.04729)0.865.50:0.02065,((((71233:0.00263,111913:0.01218)1.000.2200:0.04521,(((55534:0.01716,267008:0.02084)1.000.2201:0.05194,(163647:0.05381,(538177:0.0673,(255643:0.04404,324201:0.03308)0.336.14:0.01362)0.997.299:0.03423)0.968.102:0.01774)0.236.5:0.00908,(220036:0.01399,260028:0.0108)1.000.2202:0.05556)0.277.5:0.0111)0.898.68:0.01293,((((169619:0.01296,574788:0.01017)1.000.2203:0.02931,(((81161:0.03354,66949:0.0135)1.000.2204:0.03002,(((71882:0.02071,569732:0.01042)0.965.110:0.01036,(103240:0.01166,103391:0.01492)1.000.2205:0.02461)1.000.2206:0.02541,((((52017:0.02742,(66912:0.03218,65481:0.00581)1.000.2207:0.03357)0.994.215:0.01359,(210067:0.01046,162849:0.0229)0.485.5:0.0039)0.839.48:0.00421,((262429:0.02283,171902:0.01644)0.968.103:0.00855,(235963:0.00898,146618:0.01321)1.000.2208:0.01638)0.906.71:0.00657)0.969.113:0.00982,263961:0.01626)1.000.2209:0.03203)0.977.133:0.01566)0.983.139:0.01672,(185193:0.01964,(100980:0.02286,576076:0.01365)0.972.120:0.00968)1.000.2210:0.02273)0.969.114:0.01073)0.973.108:0.01269,(549880:0.03678,(200559:0.00734,533160:0.01496)0.158.5:0.00283)0.965.111:0.01485)0.967.107:0.00917,(166011:0.04954,(((199390:0.02614,151229:0.0089)0.929.89:0.00583,170927:0.01466)0.758.28:0.00113,(((170827:0.00794,518280:0.00878)0.900.61:0.00293,192578:0.01274)0.758.29:0.00125,(558101:0.0132,(103642:0.01899,104347:0.0132)0.931.69:0.00654)0.275.4:0.00228)0.958.110:0.00451)1.000.2211:0.02459)0.976.112:0.00973)0.755.36:0.00554)0.621.12:0.00822,((((159186:0.03035,252718:0.02866)0.999.586:0.02395,(153171:0.04857,((237261:0.02772,(248343:0.02023,(443913:0.02044,(571797:0.02585,162958:0.01061)0.278.5:0.00483)0.967.108:0.00971)0.980.124:0.0125)0.931.70:0.00556,((((252777:0.03254,(536506:0.01749,238582:0.00726)0.998.379:0.01846)0.988.182:0.01873,((560750:0.05569,235843:0.04731)0.735.26:0.00959,((((237669:0.00655,237372:0.02866)1.000.2212:0.042,((159294:0.01607,62480:0.01028)0.981.125:0.01699,(571883:0.02197,550014:0.02177)0.993.187:0.01517)0.553.10:0.01309)0.289.5:0.00281,((245606:0.0066,239971:0.02044)0.191.5:0.02698,((334183:0.01563,333224:0.00758)0.989.150:0.01523,(573706:0.00826,554465:0.0115)0.987.200:0.01515)1.000.2213:0.0258)0.937.79:0.01369)0.888.48:0.01685,245943:0.0649)0.925.68:0.01548)1.000.2214:0.05972)0.534.7:0.009,((((((241036:0.01174,159389:0.01284)1.000.2215:0.03955,539560:0.01489)0.163.6:0.00281,162071:0.01615)0.992.195:0.01094,(151554:0.01865,548208:0.00748)0.538.5:0.00226)0.909.62:0.00761,(543473:0.02322,(424925:0.00703,(166485:0.01903,101754:0.00983)0.993.188:0.01466)0.369.7:0.00494)0.968.104:0.01358)0.853.41:0.0077,70228:0.01123)0.952.98:0.00808)0.166.8:0.00327,(2712:0.03563,246364:0.0386)0.840.29:0.00795)1.000.2216:0.0307)0.969.115:0.01636)0.948.94:0.01096)0.914.61:0.00756,(((214216:0.0366,((235905:0.01516,594126:0.01222)0.995.262:0.02848,(274279:0.01721,(242454:0.00795,255346:0.01964)0.637.12:0.0055)0.978.124:0.01714)1.000.2217:0.05145)1.000.2218:0.04591,((((570248:0.02588,536403:0.0242)0.941.98:0.01002,((560466:0.0055,((509572:0.01229,538643:0.01009)0.168.13:0.00439,553966:0.0185)0.867.56:0.00394)1.000.2219:0.01938,((111450:0.01526,(171074:0.01971,521924:0.02824)0.985.130:0.01219)0.813.27:0.00548,(575684:0.02063,(548187:0.02998,298691:0.01965)0.056.5:0.00132)0.884.58:0.00325)0.651.10:0.0055)0.988.183:0.01541)0.978.125:0.02171,((74684:0.04667,((208234:0.05924,((((133917:0.0382,((300528:0.01221,274010:0.01471)1.000.2220:0.03809,(569482:0.02177,567453:0.0161)0.968.105:0.0149)1.000.2221:0.05832)0.734.22:0.01087,((156671:0.01075,246386:0.00997)0.298.2:0.00334,(554498:0.00965,544935:0.02565)0.884.59:0.00715)1.000.2222:0.03216)0.984.143:0.02008,(419583:0.02574,547418:0.01847)1.000.2223:0.06962)0.784.28:0.00981,((2724:0.02215,(277874:0.03435,((((158754:0.01122,((510501:0.00902,69761:0.00947)1.000.2224:0.0176,(264526:0.01472,(575995:0.02481,(195376:0.01179,(102925:0.0067,579753:0.01561)0.849.43:0.00748)0.999.587:0.01634)0.901.57:0.00538)0.940.98:0.00512)0.960.116:0.00802)1.000.2225:0.03095,(((66911:0.01416,(312944:0.02641,102061:0.00845)1.000.2226:0.03084)0.693.8:0.00562,142634:0.01406)0.894.60:0.00445,90530:0.02545)0.966.107:0.00817)0.712.14:0.00399,(((21437:0.03067,65476:0.01492)0.999.588:0.01406,(((((2742:0.01704,47263:0.02591)0.988.184:0.01189,(105928:0.01546,(2741:0.01364,77298:0.00699)0.992.196:0.00981)0.998.380:0.00903)0.239.6:0.00241,2745:0.00182)0.999.589:0.02178,(310689:0.01208,83666:0.00504)1.000.2227:0.02658)0.222.9:0.00406,((143650:0.01627,563365:0.00749)1.000.2228:0.03126,(284052:0.02257,511535:0.03464)0.991.185:0.01693)0.901.58:0.00679)0.714.13:0.00628)0.714.14:0.00543,(226739:0.03445,(((251739:0.01032,241403:0.01928)0.887.49:0.00361,(259453:0.00806,(264388:0.02313,(133577:0.00584,558598:0.01189)0.946.75:0.00695)0.992.197:0.01403)0.241.2:0.00628)1.000.2229:0.02754,(562917:0.02628,(((((42924:0.00303,((313405:0.02668,218815:0.02383)0.387.12:0.00714,224766:0.01247)0.000.74:0.00434)0.962.113:0.00455,535267:0.00853)0.960.117:0.0072,535959:0.01891)0.901.59:0.00596,(1299:0.00867,109903:0.02953)0.839.49:0.00678)0.728.23:0.00281,(((278445:0.02309,557916:0.02691)0.972.121:0.01173,542042:0.00429)0.580.9:0.00664,((559277:0.02175,281467:0.01525)0.899.64:0.00839,(2738:0.00856,(2736:0.04181,(49666:0.01191,((((544051:0.01684,588737:0.00883)0.950.107:0.02179,552658:0.01493)0.953.91:0.00543,((226163:0.0323,34771:0.01676)0.881.64:0.00469,(2737:0.0041,(538601:0.0204,133965:0.02301)0.858.43:0.01071)0.979.137:0.00392)0.890.63:0.0113)0.302.10:0.0026,513670:0.00937)0.791.32:0.00406)0.913.76:0.00637)0.275.5:0.0114)0.857.44:0.01171)0.984.144:0.01313)0.981.126:0.01313)0.901.60:0.0102)0.990.167:0.02886)0.994.216:0.02241)0.904.67:0.00912)0.345.8:0.00344)0.948.95:0.00707,(2744:0.02425,(544167:0.03211,549137:0.03417)0.850.35:0.00892)0.871.59:0.00452)0.998.381:0.01293)0.487.9:0.00384)0.900.62:0.00664,(((194108:0.00104,227415:0.03226)0.999.590:0.01912,((112889:0.02174,(143935:0.00769,(149727:0.02032,332648:0.01185)0.963.101:0.00674)0.907.63:0.00637)0.951.82:0.00731,(114372:0.02255,209842:0.01663)0.871.60:0.00385)0.436.8:0.00451)0.991.186:0.02021,((66944:0.01612,564996:0.00786)1.000.2230:0.02218,(44630:0.02702,85265:0.02916)0.999.591:0.02733)0.985.131:0.01643)0.994.217:0.02173)0.757.38:0.00496)0.667.7:0.00756)0.994.218:0.01967,((99031:0.01739,(563567:0.00623,558453:0.02467)0.928.71:0.0051)1.000.2231:0.04087,(529542:0.04138,((573498:0.01577,(565564:0.01672,529600:0.00512)0.894.61:0.00536)0.964.122:0.00778,(67539:0.03005,(((562167:0.01351,(568335:0.00766,(575415:0.02368,22845:0.01903)0.857.45:0.00349)0.948.96:0.00598)0.369.8:0.00556,139125:0.01227)0.524.6:0.00613,((555650:0.0187,(430848:0.00983,(559525:0.01253,278709:0.01897)0.755.37:0.00375)0.000.75:0.00107)0.994.219:0.01364,(160945:0.01312,(154569:0.01625,89830:0.01813)0.834.37:0.00798)0.980.125:0.00928)0.984.145:0.01419)0.252.2:0.00803)0.982.162:0.00985)0.969.116:0.01219)0.727.18:0.00899)0.913.77:0.0081)0.728.24:0.00905)0.972.122:0.01166,(((219143:0.04192,162841:0.01381)0.984.146:0.01659,((534587:0.04201,((590201:0.04024,(353087:0.02842,274885:0.02661)1.000.2232:0.02535)0.995.263:0.02761,(((323817:0.02775,552531:0.01039)0.993.189:0.014,(319179:0.02968,(72863:0.0172,581965:0.0245)0.477.8:0.00488)0.842.50:0.00824)0.978.126:0.01281,(351323:0.01765,(269159:0.01075,244008:0.02651)0.995.264:0.01803)1.000.2233:0.05641)0.919.73:0.00938)0.283.7:0.00964)0.934.65:0.0105,(182748:0.00733,(1295:0.00648,(1296:0.01555,103430:0.01247)0.065.5:0.00294)0.909.63:0.00401)0.997.300:0.01687)0.979.138:0.01892)0.803.26:0.01085,(((260455:0.0135,((208032:0.02473,555305:0.01378)0.301.9:0.00445,(563339:0.02449,552873:0.02736)0.957.125:0.01136)0.960.118:0.01333)0.737.19:0.0054,(348818:0.03405,(570460:0.03921,100356:0.02098)0.061.2:0.01156)0.751.20:0.00797)0.899.65:0.00962,(((581297:0.00665,114374:0.00924)0.999.592:0.02749,((75029:0.01746,551071:0.01663)0.643.5:0.00695,((174689:0.00901,((171290:0.00487,106531:0.01957)0.935.83:0.00409,272224:0.05039)0.773.28:0.00154)0.958.111:0.00464,13740:0.00792)0.998.382:0.02045)0.999.593:0.01803)0.273.5:0.01019,(350011:0.01631,112747:0.03924)0.839.50:0.00793)0.829.28:0.01113)0.755.38:0.00472)0.388.9:0.00738)0.987.201:0.0176)0.972.123:0.01405,(((211661:0.01954,143557:0.01767)1.000.2234:0.03662,(((311252:0.00441,241276:0.01287)1.000.2235:0.03124,((203941:0.01661,143216:0.00571)0.970.108:0.00821,((572435:0.01871,594201:0.02727)0.968.106:0.00991,(232744:0.01582,(2751:0.01048,542855:0.012)1.000.2236:0.01351)0.677.9:0.00342)0.881.65:0.00443)0.559.7:0.01005)0.997.301:0.0309,((65692:0.01718,587145:0.00899)0.923.71:0.01641,(213540:0.01385,(25243:0.0128,203363:0.01651)0.999.594:0.02027)0.955.94:0.01406)0.993.190:0.02669)1.000.2237:0.08278)0.999.595:0.03154,((246089:0.03197,559629:0.02031)1.000.2238:0.0408,((82558:0.00552,(250810:0.02291,557083:0.00762)0.551.5:0.00604)0.991.187:0.02095,(144633:0.0269,(23428:0.00675,114295:0.01391)0.999.596:0.01711)0.948.97:0.01374)0.986.150:0.02709)0.993.191:0.02632)0.981.127:0.01713)0.983.140:0.0144)0.566.12:0.01164,((((((235632:0.0034,159501:0.00951)0.995.265:0.02272,252689:0.02557)1.000.2239:0.06862,((159154:0.02783,(258151:0.0273,((88752:0.00867,551191:0.01253)1.000.2240:0.05018,(568011:0.01308,523452:0.0386)0.935.84:0.01311)0.075.5:0.009)0.270.6:0.00567)0.950.108:0.01152,248616:0.02353)0.988.185:0.01664)0.226.4:0.01267,(((248283:0.01326,200324:0.00406)1.000.2241:0.06542,((243950:0.03241,154220:0.01672)0.883.55:0.01366,(219899:0.02649,(223701:0.01785,214902:0.01914)0.769.28:0.00662)0.955.95:0.01255)0.434.10:0.01247)1.000.2242:0.0605,((((103839:0.01066,(113259:0.01047,576611:0.01605)0.837.25:0.00601)1.000.2243:0.06662,((330071:0.01354,535410:0.01253)1.000.2244:0.06369,(((551360:0.03215,72480:0.01143)0.922.74:0.0074,((582099:0.00982,(513560:0.01562,(539201:0.01101,509673:0.01537)0.659.11:0.00561)0.841.37:0.00492)0.872.69:0.00298,(60174:0.02533,256687:0.02698)0.935.85:0.00891)0.976.113:0.01629)1.000.2245:0.07183,(131339:0.03307,(249472:0.0115,(114049:0.00627,251267:0.02291)0.936.83:0.01063)1.000.2246:0.03596)0.993.192:0.02197)0.918.79:0.02118)0.986.151:0.02688)0.967.109:0.02204,(((330804:0.00451,(113017:0.0044,65453:0.02561)0.831.21:0.00693)1.000.2247:0.03792,(308698:0.01438,(533646:0.00437,248412:0.02532)0.574.6:0.00594)0.977.134:0.01444)0.997.302:0.02323,((583735:0.06518,((251353:0.09833,(114246:0.02483,540154:0.00981)1.000.2248:0.0548)0.956.90:0.0205,((537911:0.03861,576689:0.04692)0.959.91:0.02745,((71617:0.02033,((584629:0.01213,(53925:0.01259,251306:0.02051)0.198.6:0.00328)0.911.64:0.0071,576742:0.00847)0.984.147:0.01147)1.000.2249:0.02879,(((47497:0.01231,60582:0.00872)1.000.2250:0.05032,(540671:0.01439,361865:0.01094)1.000.2251:0.06258)0.968.107:0.02222,(((270948:0.02454,2727:0.02303)1.000.2252:0.03531,(584712:0.0269,591318:0.00926)0.998.383:0.02734)0.999.597:0.03593,(207768:0.02859,30131:0.0242)0.999.598:0.03935)0.169.6:0.00664)0.981.128:0.01863)0.902.49:0.01495)0.935.86:0.02018)0.678.13:0.01052)0.437.7:0.00936,((336578:0.00248,310277:0.00815)1.000.2253:0.04745,(44185:0.01625,257781:0.01638)1.000.2254:0.03179)0.989.151:0.01659)0.492.9:0.00763)0.835.32:0.00947)0.896.56:0.01896,(240136:0.011,538633:0.01527)1.000.2255:0.07717)0.989.152:0.0221)0.974.118:0.01824)0.513.6:0.00548,38185:0.05025)0.996.272:0.02192,((((581284:0.01849,(248137:0.00667,582447:0.00948)0.722.18:0.001)0.994.220:0.01456,(235577:0.01896,167645:0.00674)0.948.98:0.01233)1.000.2256:0.04435,(((202701:0.00669,264138:0.01599)0.987.202:0.00768,(168417:0.0066,539981:0.02033)0.860.53:0.00228)0.917.74:0.00351,(324734:0.01099,155786:0.03584)0.975.127:0.00883)0.883.56:0.00431)0.845.31:0.0054,((((533512:0.0128,(193982:0.01163,163074:0.01503)0.991.188:0.01229)0.772.29:0.01298,(515557:0.02905,(155448:0.02051,(104811:0.02083,63984:0.02002)1.000.2257:0.02705)0.630.5:0.00688)1.000.2258:0.02634)0.366.5:0.00466,((191839:0.01661,296953:0.02843)0.960.119:0.01167,546442:0.06373)0.734.23:0.00873)0.986.152:0.01322,(159576:0.0211,(587139:0.0273,142371:0.01115)0.898.69:0.00419)0.265.5:0.00494)0.623.7:0.0061)1.000.2259:0.04974)0.634.9:0.01184)0.876.59:0.00785)0.987.203:0.01585,(((184557:0.0115,(550672:0.01484,533639:0.00863)0.277.6:0.0111)0.845.32:0.00978,(211551:0.02201,(173719:0.01355,102457:0.03065)0.940.99:0.01097)0.996.273:0.01854)0.991.189:0.01444,(((322996:0.02438,272558:0.01287)0.959.92:0.00822,(559886:0.03373,(243734:0.02437,248966:0.01899)0.790.20:0.00618)0.882.55:0.00398)0.826.23:0.00734,((233441:0.02682,(587238:0.06347,(509710:0.01427,(241530:0.00845,590274:0.01948)0.871.61:0.00679)0.998.384:0.02327)0.756.36:0.02051)0.943.95:0.01769,((161161:0.01906,274220:0.0074)1.000.2260:0.044,(((304270:0.00584,(237462:0.00607,574527:0.00619)0.754.25:0.00588)0.760.30:0.00876,(312328:0.00713,260014:0.01017)0.957.126:0.0126)1.000.2261:0.03414,594219:0.04486)0.451.7:0.00784)0.650.11:0.00889)0.834.38:0.00498)1.000.2262:0.03336)1.000.2263:0.0342)0.989.153:0.01277)0.852.63:0.00611)0.971.140:0.01259)0.998.385:0.02434)1.000.2264:0.04189)0.939.94:0.01901,((((41280:0.00737,570087:0.01703)1.000.2265:0.10851,(217524:0.02122,(512694:0.01793,246357:0.00913)0.393.8:0.00482)1.000.2266:0.04723)0.983.141:0.02577,(81510:0.02594,((230135:0.01077,251851:0.01971)1.000.2267:0.0632,530796:0.01993)0.240.6:0.00952)1.000.2268:0.11828)0.981.129:0.02835,((((257305:0.05628,(568257:0.02328,(12396:0.02021,569727:0.01509)0.809.23:0.00829)1.000.2269:0.03159)0.851.48:0.00865,((224753:0.01927,575067:0.03925)0.935.87:0.00737,145916:0.00919)0.744.26:0.04198)0.954.94:0.01491,(269023:0.01761,241183:0.0337)1.000.2270:0.04207)1.000.2271:0.08315,(184899:0.05128,((258809:0.03191,(111995:0.02889,262115:0.02542)0.998.386:0.02531)0.978.127:0.02428,(143458:0.00425,(384169:0.01454,120514:0.02398)0.514.7:0.00883)1.000.2272:0.03472)0.186.6:0.01568)1.000.2273:0.1441)0.135.6:0.01829)0.341.10:0.00912)0.949.93:0.0155,((255244:0.06588,(145976:0.04191,(212594:0.01485,239073:0.02442)0.994.221:0.0245)0.413.7:0.01419)1.000.2274:0.04316,((274043:0.04177,(139838:0.01956,567096:0.02313)0.516.8:0.01467)1.000.2275:0.11238,((((114339:0.00531,211351:0.01471)0.605.10:0.00384,216644:0.01939)0.995.266:0.01815,((247114:0.00829,255961:0.01704)0.931.71:0.0111,(238291:0.01321,246082:0.02755)0.951.83:0.00797)0.940.100:0.01215)1.000.2276:0.04882,(559239:0.05609,(161224:0.01235,226892:0.0067)0.326.11:0.06974)0.920.60:0.02767)0.999.599:0.03512)0.980.126:0.02517)0.920.61:0.0156)0.782.26:0.0168,((27351:0.10704,(261806:0.02367,573839:0.02089)1.000.2277:0.08516)0.613.8:0.01582,(((217919:0.00876,558714:0.00921)1.000.2278:0.12548,((((559236:0.0151,349253:0.00804)0.998.387:0.05338,(262018:0.0096,532900:0.01918)1.000.2279:0.05388)1.000.2280:0.06094,((159265:0.00028,137052:0.00275)0.947.88:0.08781,568588:0.06637)1.000.2281:0.05968)0.993.193:0.03103,((284592:0.00652,(344869:0.01195,2660:0.00429)0.247.5:0.00595)1.000.2282:0.11254,((155382:0.02229,71654:0.02199)1.000.2283:0.03611,(553681:0.03405,(522383:0.02861,254219:0.01635)0.989.154:0.01726)0.979.139:0.02176)0.999.600:0.04428)0.964.123:0.01937)0.729.19:0.01281)1.000.2284:0.07052,((((1688:0.0036,563699:0.01525)0.922.75:0.01174,(32158:0.01299,(202958:0.01091,(160707:0.01297,246314:0.00558)0.895.66:0.00504)0.996.274:0.015)0.114.2:0.01498)1.000.2285:0.1419,((549905:0.0399,160462:0.05208)0.722.19:0.01168,(160216:0.03809,(((547075:0.02961,511355:0.01591)0.998.388:0.01943,251185:0.02038)0.877.56:0.00683,269354:0.02048)0.995.267:0.02209)0.141.5:0.00635)1.000.2286:0.06404)1.000.2287:0.05838,(162883:0.01526,588200:0.02197)1.000.2288:0.12724)0.250.6:0.02671)0.965.112:0.01976)0.823.24:0.01889)0.708.13:0.00553)0.698.16:0.00691)0.940.101:0.01249,(((((((29289:0.07633,(90270:0.00826,537588:0.0325)0.980.127:0.01738)1.000.2289:0.0438,((((((((362415:0.07217,(14244:0.01803,589048:0.01737)0.991.190:0.01241)0.981.130:0.01212,(((113661:0.02598,(266069:0.02206,249139:0.00733)0.839.51:0.01168)1.000.2290:0.08242,(((85846:0.01175,((236283:0.01146,(278979:0.02101,350940:0.01844)0.960.120:0.0124)1.000.2291:0.02864,(279154:0.00955,515398:0.00315)1.000.2292:0.03414)0.966.108:0.02055)1.000.2293:0.05605,((310026:0.0338,(((104179:0.01349,(60603:0.02022,((170026:0.00491,442217:0.01611)0.931.72:0.0083,(141863:0.00982,337073:0.00716)0.912.57:0.00466)0.689.9:0.00092)0.977.135:0.01649)0.985.132:0.01212,((67640:0.01292,((279224:0.00812,235610:0.01183)0.446.8:0.00079,(343782:0.0015,(313235:0.01697,586053:0.01237)0.967.110:0.00767)0.785.30:0.00259)0.935.88:0.00479)0.967.111:0.01031,(583876:0.01218,(102801:0.01504,(79318:0.02803,109556:0.00916)0.969.117:0.00755)0.994.222:0.01712)0.950.109:0.01457)0.989.155:0.01703)0.898.70:0.00381,((322850:0.00563,547804:0.01776)1.000.2294:0.03166,(((587712:0.04123,((((((313791:0.00206,66195:0.00014)0.502.12:0.00603,14175:0.00014)0.732.22:0.01933,(108507:0.00093,527402:0.01667)0.981.131:0.01066)0.991.191:0.01785,((410501:0.0316,(51973:0.00015,544872:0.00014)0.979.140:0.0206)0.935.89:0.01493,((73998:0.01758,(247984:0.01725,(357987:0.00028,14179:0.00204)0.930.79:0.01937)0.907.64:0.00847)1.000.2295:0.08257,((60259:0.00084,14177:0.00568)0.988.186:0.01909,14178:0.02564)0.996.275:0.03076)0.999.601:0.04625)0.952.99:0.01282)0.998.389:0.02114,((((301700:0.01142,((332025:0.01323,313844:0.01266)0.996.276:0.02048,(267447:0.02751,(344780:0.0788,325608:0.03856)0.957.127:0.0137)0.953.92:0.01085)0.976.114:0.01276)0.961.99:0.00989,(138368:0.02731,(301484:0.02482,(273357:0.02497,270382:0.00903)0.931.73:0.0075)0.967.112:0.01175)0.011.3:0.00641)0.967.113:0.01263,(((290375:0.02381,269174:0.03066)0.956.91:0.00695,(((347553:0.00629,510994:0.01214)0.745.28:0.00079,(350385:0.00879,344394:0.00633)0.860.54:0.00157)0.771.29:0.00395,((173883:0.01293,561500:0.01861)0.168.14:0.01828,((273775:0.01199,(((261590:0.01848,259527:0.01792)0.796.27:0.00229,96258:0.00816)0.946.76:0.00106,(522977:0.03376,584060:0.00441)0.000.76:0.0003)0.821.20:0.00094)0.761.36:0.00015,((303273:0.0116,297143:0.01084)0.786.30:0.00281,357984:0.00831)0.718.15:0.00356)1.000.2296:0.00073)0.125.5:0.00309)0.765.29:0.00105)0.782.27:0.00204,((((545932:0.01319,(312221:0.01108,135311:0.03658)1.000.2297:0.01716)0.915.66:0.00318,(((102022:0.01038,75534:0.01339)0.969.118:0.01065,((105998:0.00858,239244:0.00692)0.347.6:0.00384,(14227:0.00042,(89770:0.00014,532521:0.00042)0.856.41:0.00014)0.969.119:0.00014)0.221.6:0.00557)1.000.2298:0.04039,(291844:0.01738,((318982:0.00398,(197510:0.03013,(295560:0.02063,((312781:0.01605,344553:0.02976)0.963.102:0.02269,(326083:0.01863,289373:0.02426)0.980.128:0.00939)0.921.74:0.00907)0.137.4:0.00014)0.865.51:0.00089)0.922.76:0.0041,(332962:0.01402,193066:0.02071)1.000.2299:0.03954)0.956.92:0.01324)0.961.100:0.00748)0.781.24:0.00176)0.995.268:0.00814,(525086:0.01278,((((182202:0.01126,577153:0.02679)1.000.2300:0.00015,((40074:0.0044,(338957:0.03233,553388:0.03087)0.873.60:0.0023)0.172.7:0.0107,291737:0.00736)0.346.10:0.00081)1.000.2301:0.007,554685:0.01106)0.906.72:0.0131,267139:0.00739)0.893.55:0.00157)0.932.72:0.00374)0.924.66:0.0033,((((511831:0.00382,300356:0.0161)0.953.93:0.00016,(((162021:0.07774,306508:0.01453)0.347.7:0.00955,14207:0.00486)0.999.602:0.0111,14202:0.00092)0.778.38:0.00648)0.996.277:0.01013,(298875:0.05261,337083:0.01438)0.243.4:0.0133)0.903.65:0.00401,(291183:0.00751,((289214:0.0054,(321137:0.00814,583305:0.00653)0.769.29:0.0015)0.795.25:0.0044,(308502:0.02477,154631:0.01601)0.268.5:0.00039)0.820.17:0.00078)0.899.66:0.00167)0.259.10:0.00171)0.983.142:0.01088)0.771.30:0.01105)1.000.2302:0.01774,((136314:0.01752,((104561:0.0178,(202999:0.00095,14194:0.00125)1.000.2303:0.02039)0.993.194:0.01184,((576230:0.01133,(104867:0.00129,(577685:0.0065,458548:0.00305)0.883.57:0.01944)0.966.109:0.01179)0.744.27:0.00615,76251:0.02272)0.955.96:0.00871)0.969.120:0.00712)0.984.148:0.01177,99507:0.01577)0.977.136:0.01224)0.993.195:0.01448)0.888.49:0.00857,(14182:0.02646,(533896:0.00344,14180:0.00972)0.973.109:0.01018)1.000.2304:0.02404)0.864.65:0.0155)0.973.110:0.01773,(((((582632:0.00478,(72205:0.04236,89166:0.01097)0.987.204:0.00826)0.961.101:0.00671,(((141051:0.00244,62667:0.02297)0.987.205:0.00769,(19154:0.00736,40951:0.00629)0.022.2:0.00178)0.918.80:0.00373,140290:0.01391)0.941.99:0.00672)1.000.2305:0.02878,(58790:0.01987,((339489:0.00313,((164378:0.02826,45283:0.00122)0.954.95:0.00584,562956:0.00278)0.733.24:0.00156)0.636.7:0.00182,(106464:0.00991,(176156:0.00885,210652:0.00785)0.736.33:0.00112)0.992.198:0.01193)0.929.90:0.00532)0.926.79:0.00618)0.637.13:0.00379,(543786:0.03235,((138025:0.00441,(14169:0.00373,305185:0.01929)0.935.90:0.00708)0.920.62:0.00511,((211906:0.00167,570234:0.00014)0.908.71:0.00363,224798:0.02434)0.619.9:0.00381)0.991.192:0.01199)0.954.96:0.00907)0.609.7:0.00415,(161419:0.00522,(348824:0.00321,235642:0.01114)0.200.9:0.00568)0.999.603:0.01659)0.670.7:0.00586)0.901.61:0.00832,(551148:0.00923,112412:0.00929)1.000.2306:0.04181)0.402.6:0.00742)0.880.54:0.00977)0.869.59:0.00398)0.953.94:0.00875,(293386:0.00764,(516702:0.02457,557019:0.00819)0.889.53:0.00624)0.995.269:0.01399)0.865.52:0.00513)0.892.48:0.00563,((((((229659:0.00649,391328:0.01857)0.977.137:0.01019,((242873:0.02116,((289611:0.0156,213337:0.01761)1.000.2307:0.03267,((330492:0.01932,294860:0.01067)0.999.604:0.01815,(((271320:0.01664,(557370:0.00991,325246:0.0286)0.701.8:0.005)0.880.55:0.00707,((329497:0.01484,340751:0.01633)0.620.5:0.00427,(330418:0.01141,584340:0.03239)0.511.8:0.00797)0.888.50:0.00783)0.999.605:0.01911,(303248:0.03755,(352027:0.01454,291568:0.02662)0.698.17:0.00789)0.915.67:0.00755)0.178.5:0.0029)0.389.6:0.00585)0.268.6:0.00596)0.990.168:0.01226,((108712:0.00978,549514:0.00737)0.864.66:0.00409,108469:0.00958)0.867.57:0.00513)1.000.2308:0.01851)0.974.119:0.01006,(135738:0.03096,((135736:0.0141,((150434:0.00952,149904:0.01712)0.841.38:0.00727,((583465:0.00131,41514:0.00232)1.000.2309:0.04077,(51211:0.00091,557481:0.0065)1.000.2310:0.03308)0.267.9:0.00668)0.992.199:0.01543)1.000.2311:0.03387,((82172:0.02827,(202115:0.01434,95791:0.03222)0.525.7:0.00681)0.810.30:0.00535,((150105:0.03706,77744:0.00931)0.561.11:0.00655,((144180:0.01286,((144158:0.01282,77831:0.00159)0.918.81:0.00768,(579712:0.02375,149597:0.01886)0.923.72:0.00917)0.996.278:0.01875)0.955.97:0.01414,(78301:0.02331,114804:0.01951)0.872.70:0.00803)0.176.5:0.0045)0.840.30:0.00993)0.256.5:0.01059)0.990.169:0.01796)0.956.93:0.0105)0.452.15:0.00399,(((299957:0.01105,(344940:0.00404,310757:0.00492)0.906.73:0.00408)1.000.2312:0.0241,(((146485:0.03086,214036:0.00913)0.732.23:0.00414,(((43267:0.00014,331417:0.04164)0.735.27:0.01519,43339:0.03089)0.670.8:0.00735,((((308321:0.00014,268884:0.00382)0.999.606:0.01292,274787:0.00264)0.957.128:0.00735,((344702:0.0088,(306730:0.02618,288505:0.0398)1.000.2313:0.03243)0.972.124:0.01472,((340436:0.0044,537649:0.00612)0.964.124:0.00609,291948:0.01743)0.847.37:0.00213)0.983.143:0.00938)1.000.2314:0.01768,173642:0.01341)0.879.65:0.0039)0.946.77:0.00865)0.983.144:0.0134,68350:0.03767)0.748.23:0.00687)0.453.7:0.0023,((((((14145:0.00661,((66192:0.00707,328267:0.00754)0.995.270:0.01201,(455325:0.00385,188169:0.0374)0.972.125:0.01107)1.000.2315:0.01317)0.747.28:0.00078,(42984:0.0103,591847:0.00607)0.974.120:0.00476)0.897.61:0.0031,((98253:0.01537,66939:0.0067)0.999.607:0.01418,(((333586:0.00666,(112263:0.00698,555860:0.00776)0.984.149:0.00683)0.975.128:0.00683,((586356:0.00707,582846:0.02402)0.933.80:0.00383,(546401:0.01323,93469:0.00921)0.887.50:0.00325)0.861.58:0.00165)0.867.58:0.00327,104585:0.00648)0.883.58:0.0033)0.622.7:0.0032)0.936.84:0.00421,(52379:0.01357,(((563443:0.02883,92717:0.01724)1.000.2316:0.02743,250414:0.00498)0.919.74:0.00956,210811:0.00555)0.965.113:0.01029)0.954.97:0.00876)0.886.52:0.00532,((112802:0.01788,102595:0.01379)0.335.3:0.00953,141366:0.03301)0.880.56:0.00649)0.407.8:0.00455,((14157:0.03761,287954:0.00871)0.969.121:0.01017,((((((158826:0.02588,(351498:0.01936,(59564:0.02046,297744:0.02733)0.664.9:0.00446)0.999.608:0.01792)0.907.65:0.00722,(((((522255:0.02129,(589662:0.02751,242984:0.01256)0.677.10:0.00884)0.972.126:0.01061,(271324:0.02519,(356624:0.01031,312858:0.01758)0.967.114:0.01109)0.990.170:0.0113)0.936.85:0.0091,((102610:0.00859,224206:0.02268)1.000.2317:0.01993,(78236:0.01549,538259:0.02075)0.994.223:0.01598)0.822.36:0.0073)0.863.52:0.00344,(289153:0.02507,(417981:0.01088,187802:0.02989)0.984.150:0.01138)0.995.271:0.01288)0.930.80:0.00532,(((350842:0.01663,295196:0.0128)0.968.108:0.00913,((564617:0.00884,167176:0.00705)0.989.156:0.00827,(539124:0.00616,(((((289966:0.02685,(549217:0.00653,(317655:0.01504,324857:0.01387)0.864.67:0.00267)0.740.33:0.00159)0.781.25:0.00124,(347636:0.01435,346974:0.00242)0.199.7:0.00103)0.991.193:0.00743,((104272:0.006,265281:0.01162)0.935.91:0.00448,300285:0.01472)0.970.109:0.00331)0.771.31:0.00074,((303417:0.00861,302645:0.02892)1.000.2318:0.03436,(336615:0.02048,(289265:0.02183,287814:0.02621)0.961.102:0.01229)0.989.157:0.01952)0.987.206:0.01591)0.964.125:0.00436,(324077:0.00395,331227:0.01112)0.921.75:0.00313)0.733.25:0.00147)0.955.98:0.00576)0.931.74:0.00861)0.934.66:0.00802,((216010:0.01533,((110579:0.01176,302577:0.01054)0.981.132:0.00765,(212304:0.01076,((145887:0.01566,591439:0.00974)0.931.75:0.00591,208539:0.00766)0.788.25:0.00285)0.524.7:0.00248)0.820.18:0.00283)0.908.72:0.00325,33133:0.03529)0.334.7:0.00189)0.986.153:0.01097)0.634.10:0.0012)0.912.58:0.00405,(20310:0.01014,(36792:0.00381,293133:0.01077)0.999.609:0.01344)0.981.133:0.00999)0.902.50:0.00381,484443:0.04099)0.894.62:0.00414,(((147130:0.01782,((((441376:0.01216,191389:0.0149)0.773.29:0.0087,((229905:0.00078,306545:0.04944)0.923.73:0.00251,421755:0.0044)0.950.110:0.00829)0.838.38:0.00651,302433:0.01449)0.705.7:0.00356,(207340:0.01581,(332564:0.00399,(324364:0.02353,294038:0.02146)0.998.390:0.02218)0.915.68:0.00575)0.750.26:0.00394)0.822.37:0.00301)0.825.26:0.00383,((((578735:0.02068,((14156:0.01596,14150:0.02128)1.000.2319:0.03175,(14152:0.00982,576991:0.01465)0.941.100:0.00893)0.922.77:0.00919)0.965.114:0.01183,((107594:0.01063,320783:0.00397)1.000.2320:0.0288,(50860:0.0226,534031:0.04164)0.990.171:0.02027)0.906.74:0.01018)1.000.2321:0.02548,(41364:0.0009,552342:0.00014)1.000.2322:0.05361)0.992.200:0.01812,((589746:0.00093,248474:0.00028)0.890.64:0.02998,(459464:0.03377,((((544495:0.0128,306820:0.00864)0.872.71:0.00189,((317647:0.01137,((316143:0.00701,583497:0.00613)0.989.158:0.00854,592626:0.0016)0.990.172:0.00632)0.651.11:0.00182,((553612:0.00212,559598:0.00956)0.870.58:0.00336,(584895:0.01057,318883:0.00384)0.968.109:0.00595)0.993.196:0.01009)0.981.134:0.00522)0.828.29:0.00165,335363:0.00995)0.937.80:0.00629,325334:0.00963)0.994.224:0.01578)0.710.15:0.01032)0.052.2:0.00265)0.919.75:0.0059)0.973.111:0.00773,(((340684:0.00014,439876:0.01322)0.841.39:0.03912,(((512871:0.00953,(307485:0.00809,348479:0.00499)0.946.78:0.0081)0.768.37:0.00593,(316582:0.01351,(112328:0.00927,575933:0.0054)0.998.391:0.01512)0.490.8:0.00281)0.793.23:0.00694,((((323483:0.00225,350588:0.00687)0.907.66:0.00356,508773:0.01661)0.953.95:0.00744,(150089:0.02777,((109624:0.00641,((561915:0.00174,(591989:0.01185,(520966:0.0063,296033:0.00637)0.785.31:0.0027)0.682.8:0.00014)0.862.61:0.00198,(18066:0.00015,(569504:0.01174,151714:0.00152)0.970.110:0.00411)0.998.392:0.00642)0.921.76:0.00288)0.761.37:0.00075,(((552123:0.00265,585998:0.01599)0.780.22:0.00216,593374:0.00435)0.879.66:0.00238,((44410:0.00308,(582845:0.00316,538533:0.00789)1.000.2323:0.00335)0.236.6:0.00759,((((243911:0.0055,560706:0.00805)0.653.4:0.00155,((533404:0.00859,166398:0.00193)0.782.28:0.00115,527606:0.02232)0.962.114:0.00415)0.626.9:0.00015,589170:0.00626)0.863.53:0.00157,289968:0.00627)0.955.99:0.00319)0.574.7:0.00112)0.984.151:0.00518)0.923.74:0.00392)0.828.30:0.00333)1.000.2324:0.02403,((165401:0.01082,(((461939:0.00184,109334:0.00255)0.860.55:0.00182,495461:0.00558)1.000.2325:0.0345,(148228:0.01637,236437:0.01141)1.000.2326:0.02161)0.870.59:0.00568)0.989.159:0.01028,(76547:0.01984,(46159:0.00252,14163:0.01293)0.998.393:0.01389)0.926.80:0.00605)0.916.66:0.00514)0.951.84:0.0078)1.000.2327:0.03503)0.542.10:0.0059,(456006:0.00796,((((319547:0.04486,((351583:0.00947,(525024:0.02621,(343104:0.0112,304402:0.00014)0.790.21:0.00262)0.749.27:0.0047)0.816.31:0.00377,524842:0.01428)0.930.81:0.00482)0.552.10:0.00142,(((295362:0.01838,(239555:0.01624,((293697:0.01671,((320853:0.012,293265:0.01645)0.977.138:0.0096,298641:0.01887)0.139.9:0.00271)0.944.75:0.00784,(((275808:0.00014,565221:0.00926)0.987.207:0.00791,((323615:0.00391,512635:0.00481)0.967.115:0.01052,(334088:0.00185,264144:0.00703)0.924.67:0.00289)0.858.44:0.00444)0.988.187:0.01064,((347080:0.00337,(110083:0.00092,295532:0.04924)0.783.31:0.00078)0.987.208:0.00763,(309179:0.01694,288422:0.01979)0.924.68:0.00524)0.978.128:0.00698)0.965.115:0.00773)0.986.154:0.01095)0.880.57:0.0027)0.945.82:0.00482,582284:0.01531)0.922.78:0.0047,(578588:0.00803,320741:0.00481)0.780.23:0.00548)0.863.54:0.00477)0.379.15:0.00225,36226:0.01399)0.992.201:0.00881,210945:0.01545)0.833.32:0.00257)0.997.303:0.01395)0.779.20:0.00401)0.994.225:0.01239)0.820.19:0.00247,449190:0.04353)0.954.98:0.00663)0.909.64:0.00704)0.962.115:0.00691)0.717.14:0.00501)1.000.2328:0.02674,14148:0.03446)1.000.2329:0.02401,((((14135:0.00675,245248:0.00595)0.993.197:0.01249,(267846:0.00928,(371743:0.0141,(583445:0.00824,(85424:0.00343,(155716:0.00207,275236:0.00157)0.075.6:0.00452)0.906.75:0.00302)0.924.69:0.00454)0.215.8:0.00216)0.985.133:0.00894)0.992.202:0.0124,((584294:0.01015,219207:0.0111)1.000.2330:0.03463,((41951:0.00536,(534793:0.01876,568270:0.00291)0.377.4:0.00159)0.999.610:0.01865,(166705:0.00466,552117:0.00861)1.000.2331:0.02725)0.997.304:0.01814)0.824.22:0.00637)0.706.13:0.00313,(((((544542:0.01105,532885:0.0104)1.000.2332:0.03124,((((572909:0.00364,(370067:0.00714,348558:0.01309)0.601.7:0.01063)0.885.56:0.00269,(166259:0.00179,516206:0.00107)0.916.67:0.00365)0.555.8:0.00241,344556:0.00845)0.991.194:0.01201,(255423:0.01264,(103172:0.00627,14137:0.00489)0.694.10:0.00342)0.983.145:0.0091)0.998.394:0.01887)0.900.63:0.00953,(14204:0.02218,345543:0.00916)1.000.2333:0.02122)0.717.15:0.00228,((342975:0.02128,49266:0.03025)0.950.111:0.00691,(567760:0.01961,(529003:0.02709,(157625:0.01879,336842:0.00118)0.767.42:0.0139)0.972.127:0.00934)0.284.7:0.00172)0.216.3:0.00285)0.968.110:0.00686,(344298:0.0119,(340616:0.02085,(110063:0.01241,562002:0.01816)0.377.5:0.00264)0.982.163:0.0095)1.000.2334:0.02885)0.799.27:0.00422)0.883.59:0.00669)0.992.203:0.01211)0.975.129:0.01298)0.665.13:0.01705,(591869:0.0387,(28721:0.01091,278481:0.03419)1.000.2335:0.03792)0.973.112:0.02088)0.995.272:0.01962)0.412.5:0.00659,((((210515:0.02338,(142175:0.00376,572646:0.02087)0.997.305:0.01525)0.887.51:0.01583,((136382:0.01164,115012:0.0077)1.000.2336:0.03467,(14258:0.0128,((568699:0.01265,108707:0.01942)0.875.49:0.00612,((528290:0.01541,(535477:0.00262,527633:0.00101)0.866.47:0.01087)0.623.8:0.00621,((112895:0.00649,((99748:0.00349,312821:0.00733)0.976.115:0.00644,(209844:0.00767,(157172:0.00171,85712:0.01957)0.960.121:0.00571)0.963.103:0.0062)0.856.42:0.00241)0.917.75:0.00307,(34493:0.02021,360415:0.00879)0.941.101:0.00421)0.914.62:0.00361)0.873.61:0.00468)0.336.15:0.00703)0.998.395:0.01934)0.963.104:0.01382)0.280.8:0.01111,(257832:0.00791,191385:0.0081)1.000.2337:0.03375)0.995.273:0.02204,(((553089:0.01772,(217815:0.01641,(251691:0.01324,585584:0.01471)1.000.2338:0.02818)0.986.155:0.01458)0.985.134:0.01483,(((((522677:0.0039,190629:0.00654)0.977.139:0.00954,(148745:0.00963,558630:0.02478)0.531.10:0.00304)1.000.2339:0.01881,(((586378:0.00771,(530341:0.00071,520888:0.01488)0.773.30:0.00366)1.000.2340:0.02964,(((151594:0.01153,(517458:0.01247,(14266:0.00827,533912:0.00814)0.864.68:0.00635)0.902.51:0.00595)0.965.116:0.00684,((321681:0.01584,(14263:0.00498,88113:0.00324)0.941.102:0.00505)0.733.26:0.00111,558311:0.00718)0.977.140:0.00624)0.891.65:0.00341,312767:0.01235)0.863.55:0.0032)0.898.71:0.0039,((560842:0.01268,(14265:0.00689,510178:0.02807)0.722.20:0.00816)0.456.8:0.00453,(543440:0.01847,(326473:0.00884,551545:0.01446)0.967.116:0.00684)0.807.22:0.0029)0.832.26:0.0048)0.681.17:0.00236)0.748.24:0.00321,577239:0.01896)0.998.396:0.01569,(541364:0.02304,(14269:0.02649,(((246996:0.0098,550421:0.01232)0.496.8:0.01071,((240034:0.01351,(164133:0.00462,256589:0.01409)0.901.62:0.00454)0.980.129:0.01225,((((312984:0.00991,(284303:0.01392,((72660:0.02399,99474:0.00642)1.000.2341:0.01303,101262:0.00945)0.961.103:0.00585)0.966.110:0.00894)0.997.306:0.01774,285855:0.00878)0.996.279:0.01798,523738:0.00408)0.759.20:0.00889,179709:0.00292)0.996.280:0.01595)0.959.93:0.01091)0.925.69:0.00718,(548585:0.00947,184280:0.01109)0.961.104:0.00824)0.925.70:0.00963)0.996.281:0.01822)0.587.7:0.00701)0.689.10:0.01016)0.198.7:0.01074,((((505646:0.00259,160284:0.00067)0.892.49:0.06012,(504691:0.00346,(376401:0.0049,259604:0.01799)0.870.60:0.01291)0.995.274:0.02799)1.000.2342:0.03014,(494938:0.02752,70787:0.02091)1.000.2343:0.05665)0.790.22:0.00744,(((311771:0.00113,259078:0.01512)1.000.2344:0.04379,(((495052:0.02028,(198893:0.0149,68341:0.00411)0.815.20:0.02032)0.941.103:0.0123,(521628:0.00974,(249152:0.01957,(14275:0.01067,((508117:0.00014,179165:0.00768)0.501.7:0.0065,(184255:0.00267,(186999:0.00465,493044:0.01499)0.643.6:0.00445)0.959.94:0.0114)0.984.152:0.01001)0.998.397:0.01922)0.877.57:0.01041)0.998.398:0.02261)0.834.39:0.00954,((75475:0.00611,146014:0.01479)0.999.611:0.02692,(14281:0.0248,(((268507:0.0019,464237:0.00289)0.745.29:0.02414,((377279:0.01301,(386953:0.00354,266777:0.0046)0.776.32:0.00085)0.775.19:0.03807,(((583324:0.02717,(185686:0.0003,(((516352:0.01009,14278:0.00435)0.774.17:0.00296,14277:0.02524)0.993.198:0.00991,192451:0.01026)0.872.72:0.00218)0.228.2:0.00577)0.998.399:0.01277,494937:0.00402)0.996.282:0.01541,(14280:0.02826,((83175:0.00355,(524552:0.00656,521795:0.02308)0.881.66:0.00016)0.765.30:0.00014,180390:0.02178)0.975.130:0.00963)0.983.146:0.01592)0.992.204:0.0187)0.977.141:0.01668)0.774.18:0.00653,(364655:0.04421,((14283:0.0382,14282:0.00015)0.769.30:0.0279,567933:0.01246)0.941.104:0.01213)0.515.14:0.01012)0.634.11:0.00668)0.502.13:0.00929)0.971.141:0.02104)0.880.58:0.01295)0.942.92:0.01009,((((446471:0.02892,471282:0.00451)1.000.2345:0.04026,(523577:0.02909,254577:0.01523)0.999.612:0.03029)0.983.147:0.02332,(((458383:0.01285,(425038:0.00198,275891:0.00972)0.882.56:0.00715)1.000.2346:0.07627,((325610:0.0033,(561616:0.00585,514738:0.01575)0.185.4:0.00219)0.998.400:0.02376,((14271:0.00837,((590651:0.00572,523124:0.01662)0.957.129:0.00719,521161:0.01707)0.995.275:0.01732)0.590.11:0.00905,((99351:0.01523,239761:0.01121)0.949.94:0.01027,((540075:0.00713,570444:0.01661)0.939.95:0.00639,162924:0.02118)0.173.6:0.00512)0.780.24:0.00711)1.000.2347:0.02597)0.999.613:0.03104)0.906.76:0.01994,((208044:0.02152,236547:0.0197)1.000.2348:0.05138,((((14288:0.01084,494913:0.02194)0.244.5:0.00334,(175062:0.0123,448819:0.00642)0.868.65:0.00175)0.952.100:0.00381,368769:0.0042)0.800.32:0.02605,((((((14284:0.03693,469655:0.0003)0.928.72:0.00623,360880:0.01423)0.066.3:0.00286,(14285:0.02531,521979:0.00699)0.732.24:0.00031)0.922.79:0.00836,((134786:0.0027,219668:0.0031)0.908.73:0.00873,(((((180838:0.00822,185333:0.00362)0.268.7:0.00016,180355:0.00164)0.999.614:0.00588,198874:0.00834)0.683.15:0.00236,((84198:0.00014,469646:0.00152)0.921.77:0.00025,557067:0.00028)0.998.401:0.00782)0.045.4:0.00294,(253709:0.00594,(516667:0.00579,357089:0.01006)0.853.42:0.00327)0.776.33:0.00223)0.955.100:0.00376)0.748.25:0.01541)0.993.199:0.01396,(((497711:0.00493,363649:0.00469)0.928.73:0.00462,((432522:0.00361,(409667:0.01453,(420664:0.00805,(429411:0.01769,(418100:0.00666,432303:0.00938)0.744.28:0.00087)0.978.129:0.00642)0.605.11:0.00323)0.849.44:0.00318)0.943.96:0.00389,(505462:0.00577,366401:0.00616)0.872.73:0.00198)0.843.37:0.00147)0.937.81:0.00547,(((309744:0.00588,423311:0.01196)0.997.307:0.00971,(((431949:0.00441,(225409:0.00263,224222:0.00156)0.269.8:0.00566)0.854.42:0.0019,(572481:0.01383,(357533:0.0184,367454:0.00029)0.917.76:0.00434)0.911.65:0.00298)0.959.95:0.00556,367842:0.01412)0.986.156:0.00844)0.896.57:0.00339,212128:0.04021)0.598.11:0.00166)0.877.58:0.00469)0.957.130:0.01282,(239292:0.01006,515662:0.00917)0.999.615:0.018)0.535.8:0.00693)1.000.2349:0.07274)0.532.7:0.01378)0.652.12:0.01267)0.498.5:0.01047,(132546:0.01884,(500281:0.01318,(92862:0.01664,470805:0.00694)0.923.75:0.01081)0.951.85:0.00818)1.000.2350:0.0385)0.973.113:0.01575)0.870.61:0.00795)0.995.276:0.02357)0.957.131:0.01595)0.975.131:0.0113)0.931.76:0.00783,((((584822:0.02234,33448:0.04363)0.907.67:0.00474,556596:0.0278)0.802.29:0.0101,((275937:0.00834,256125:0.00939)0.966.111:0.00992,((590570:0.00566,591796:0.01248)0.995.277:0.01537,180011:0.04406)0.403.6:0.0055)0.685.10:0.00966)0.998.402:0.01398,(338939:0.004,(176096:0.01928,557604:0.01022)0.957.132:0.00857)0.999.616:0.01672)0.750.27:0.00915)1.000.2351:0.03225,((99126:0.04502,(307463:0.031,518452:0.01267)0.214.2:0.0048)0.915.69:0.01205,(((((214750:0.00893,157142:0.00398)1.000.2352:0.09631,(((522312:0.00014,548330:0.00356)0.362.5:0.01483,(((79549:0.01318,(79463:0.0055,79098:0.0065)0.853.43:0.00264)0.841.40:0.00077,271555:0.00043)0.858.45:0.00476,(28584:0.00538,172960:0.0149)0.957.133:0.00535)0.907.68:0.00429)0.956.94:0.00906,(((14253:0.0043,237359:0.00212)1.000.2353:0.02019,((335896:0.02366,302875:0.02609)0.940.102:0.01399,((46867:0.016,34061:0.0204)0.985.135:0.01584,(311974:0.02295,274259:0.03943)0.871.62:0.00945)0.929.91:0.01123)0.836.33:0.01129)0.974.121:0.01268,259130:0.01741)0.905.70:0.00933)0.709.12:0.00871)1.000.2354:0.03144,89746:0.02135)0.221.7:0.00809,((148988:0.04144,469664:0.00332)1.000.2355:0.03276,(135410:0.08746,((264374:0.00123,543902:0.00157)1.000.2356:0.05461,(350699:0.01749,(553489:0.016,149723:0.00912)0.758.30:0.00466)0.968.111:0.0123)0.998.403:0.02733)0.650.12:0.01226)1.000.2357:0.04738)0.212.6:0.00452,((554177:0.00513,276910:0.00927)1.000.2358:0.05899,(516398:0.02269,(103303:0.00467,514900:0.00679)0.756.37:0.01762)0.953.96:0.014)0.821.21:0.00982)0.999.617:0.02574)0.670.9:0.01289)0.722.21:0.01862,(336463:0.00672,257609:0.01323)1.000.2359:0.11124)0.547.10:0.01573,(314041:0.0508,(555479:0.01696,(235339:0.02413,545286:0.01258)0.941.105:0.00801)0.999.618:0.02957)0.930.82:0.0135)0.785.32:0.01176,((((((170514:0.01146,(146312:0.01727,146278:0.00983)0.956.95:0.00708)0.941.106:0.00871,((16084:0.00799,219683:0.00655)0.865.53:0.00196,(131000:0.00701,108483:0.00901)0.815.21:0.00149)0.576.9:0.00386)0.000.77:0.00289,(577588:0.02321,(137922:0.01218,554692:0.0095)0.969.122:0.00568)0.214.3:0.00187)0.971.142:0.01184,((571852:0.02481,(585277:0.00795,43679:0.0142)0.893.56:0.0058)0.997.308:0.02093,(243038:0.03853,(101117:0.00551,16082:0.02196)0.979.141:0.01335)0.972.128:0.01594)0.503.9:0.01265)1.000.2360:0.05336,((((22668:0.01245,(133349:0.01752,((131760:0.04044,395382:0.00092)0.000.78:0.00839,16195:0.007)0.932.73:0.00579)0.731.29:0.00919)1.000.2361:0.09488,(((((((((26148:0.00873,308659:0.00395)0.808.24:0.01528,((104736:0.02981,((170506:0.00932,16159:0.01005)0.973.114:0.00853,(16164:0.03155,471261:0.02994)0.982.164:0.01257)0.558.8:0.00319)0.903.66:0.005,((518518:0.02592,(((368281:0.01552,192526:0.034)0.430.9:0.00556,(((591228:0.01271,(113808:0.00089,250893:0.00014)0.944.76:0.00335)0.941.107:0.00572,280295:0.01936)0.187.9:0.01897,(365437:0.02358,(295330:0.02244,533504:0.01386)0.998.404:0.0207)0.999.619:0.02122)0.949.95:0.00963)0.809.24:0.00396,(((((546025:0.00733,569919:0.01563)0.975.132:0.00813,263693:0.02515)0.918.82:0.00504,((111782:0.0048,(102321:0.01218,138256:0.0129)0.943.97:0.00891)0.948.99:0.00951,(16129:0.0096,571639:0.01264)1.000.2362:0.02193)0.953.97:0.00731)0.637.14:0.00119,(102839:0.00179,586854:0.00295)0.858.46:0.00979)0.578.4:0.00385,16121:0.00951)1.000.2363:0.02704)0.293.4:0.00665)0.983.148:0.012,(578922:0.02955,((275556:0.00824,(250479:0.00703,111248:0.03768)0.863.56:0.00674)0.757.39:0.00725,(102184:0.01241,150000:0.00921)0.890.65:0.00534)0.932.74:0.0134)0.980.130:0.01402)0.691.10:0.00511)0.392.7:0.00359)0.750.28:0.00306,((((180930:0.00119,114760:0.03879)0.468.10:0.00444,147444:0.03326)0.997.309:0.01075,((114950:0.00731,(104248:0.02949,75821:0.00873)0.473.9:0.0043)0.997.310:0.01297,16182:0.02014)0.940.103:0.00698)0.513.7:0.00257,((134335:0.03024,(((29618:0.00869,60741:0.00029)0.831.22:0.00327,287160:0.00178)0.855.43:0.0055,513952:0.00358)0.991.195:0.00824)0.883.60:0.00486,(248695:0.00823,79436:0.00263)0.968.112:0.00469)0.862.62:0.00285)0.991.196:0.00946)0.914.63:0.00373,(((541432:0.01741,(589921:0.02359,(574742:0.01002,(183481:0.02912,571847:0.00827)0.853.44:0.01082)0.900.64:0.0059)0.815.22:0.00524)0.043.3:0.00286,551963:0.01776)0.964.126:0.00734,(((157392:0.02112,((((201367:0.03505,16144:0.00014)0.997.311:0.0131,(577675:0.01645,(254078:0.02585,(137771:0.00039,543607:0.02488)0.997.312:0.01348)0.877.59:0.00873)0.561.12:0.00693)1.000.2364:0.02124,(16138:0.00947,(246458:0.00369,587008:0.00569)0.747.29:0.00785)0.871.63:0.00268)0.841.41:0.00201,20034:0.02006)0.819.28:0.00356)0.814.24:0.00215,(16145:0.00277,(224769:0.02073,104466:0.01439)0.606.11:0.00322)1.000.2365:0.02162)0.937.82:0.00539,(562096:0.02772,((248740:0.02051,96743:0.0364)0.978.130:0.00659,34959:0.0078)0.986.157:0.0099)0.980.131:0.00878)0.774.19:0.00281)0.771.32:0.0037)0.942.93:0.00658,(429862:0.01396,(104893:0.00631,((563768:0.01828,35306:0.01735)0.946.79:0.00799,(37089:0.01592,580821:0.03611)0.648.11:0.00562)0.119.3:0.00764)0.928.74:0.00918)0.998.405:0.01627)0.851.49:0.00644,(60611:0.01686,((((60609:0.02019,16163:0.0024)0.858.47:0.0023,252720:0.0071)0.937.83:0.0054,(23517:0.00383,(587789:0.00354,578057:0.02418)0.575.10:0.00411)0.978.131:0.01034)0.168.15:0.00413,36004:0.01766)0.984.153:0.0103)0.988.188:0.01013)0.405.8:0.00336,(137941:0.0137,220242:0.00891)0.991.197:0.00934)0.927.79:0.00864,((((((108384:0.01826,209150:0.01967)0.843.38:0.00235,(16199:0.00902,177051:0.01108)0.499.8:0.00156)0.706.14:0.00272,(167430:0.00568,27748:0.00467)1.000.2366:0.01211)0.967.117:0.00563,((275472:0.00843,(((224105:0.00629,364085:0.01641)0.999.620:0.01931,(((((339996:0.00345,291805:0.02467)0.962.116:0.00393,(((171437:0.03028,168712:0.01863)0.240.7:0.00056,164332:0.01881)0.465.6:0.00043,169629:0.01667)0.903.67:0.00831)0.904.68:0.00242,((319567:0.03805,(346780:0.01334,293992:0.06184)0.931.77:0.01081)0.990.173:0.01264,(((37589:0.00014,534306:0.01029)0.441.4:0.00381,298080:0.01949)0.911.66:0.0067,(((326915:0.01029,289622:0.00899)0.939.96:0.00015,290241:0.00903)0.766.27:0.01928,((237935:0.02995,(199214:0.02456,248563:0.00775)0.991.198:0.01701)0.889.54:0.0075,249839:0.01054)0.993.200:0.01001)0.199.8:0.00261)0.770.25:0.00474)0.982.165:0.01089)0.989.160:0.0077,(((300755:0.00955,564094:0.01103)0.883.61:0.01092,((76200:0.00107,437215:0.00056)0.780.25:0.00014,291025:0.00425)1.000.2367:0.02153)0.919.76:0.00502,(((267161:0.01716,(304779:0.01052,323142:0.043)0.980.132:0.01223)1.000.2368:0.01881,(296166:0.01381,138056:0.01157)0.865.54:0.00206)0.317.9:0.00174,(((290669:0.03572,336187:0.02108)0.968.113:0.01211,((16202:0.00028,295640:0.03406)0.751.21:0.00317,(288118:0.02413,(294641:0.02068,(291774:0.01547,295643:0.03355)0.978.132:0.01269)0.467.5:0.00014)0.908.74:0.00016)0.839.52:0.00321)0.445.5:0.01,(((282549:0.02222,(287388:0.00275,299371:0.02208)0.000.79:0.00926)1.000.2369:0.02107,300929:0.01313)0.388.10:0.02478,(295190:0.08803,(313503:0.03035,207741:0.0612)0.911.67:0.01407)0.394.7:0.01401)0.622.8:0.00326)0.427.8:0.00288)0.986.158:0.00893)0.882.57:0.00316)0.855.44:0.00178,(((317249:0.00584,524580:0.00449)0.912.59:0.01197,(555562:0.03949,((541328:0.00845,296268:0.01559)0.962.117:0.00505,(529116:0.00836,(289672:0.02567,((103132:0.00897,523511:0.01085)0.755.39:0.0143,(16267:0.0003,((101362:0.00692,95300:0.00887)0.982.166:0.00013,49549:0.02352)0.773.31:0.00234)0.784.29:0.00234)0.866.48:0.00156)0.965.117:0.00312)0.910.64:0.00734)0.861.59:0.00457)0.704.22:0.00356)0.982.167:0.00661,(((33176:0.01655,((302631:0.02777,297938:0.0064)0.849.45:0.00969,311546:0.0094)0.992.205:0.01418)0.466.13:0.00189,((103941:0.00477,16240:0.02962)0.586.8:0.00471,91168:0.00664)0.014.2:0.00294)0.821.22:0.00283,((352427:0.01674,297378:0.02189)0.044.6:0.00727,291134:0.01128)0.958.112:0.0181)0.966.112:0.0066)0.952.101:0.00399)0.973.115:0.0063)0.856.43:0.00472,(16220:0.00582,158664:0.01081)1.000.2370:0.01758)0.957.134:0.00581)0.829.29:0.00285,192270:0.01939)0.882.58:0.00277)0.221.8:0.00016,(100886:0.01081,81682:0.01711)0.809.25:0.00679)0.964.127:0.00722,(264826:0.02688,((361428:0.00627,173110:0.01028)0.539.9:0.00461,(206671:0.00826,180489:0.02052)0.874.55:0.00785)0.957.135:0.00817)1.000.2371:0.02436)0.355.8:0.00179)0.964.128:0.00822,(582154:0.03041,(71518:0.02383,((99654:0.0027,563918:0.00316)1.000.2372:0.03463,16224:0.01677)0.975.133:0.01231)0.588.5:0.00324)0.392.8:0.00436)0.942.94:0.01236)0.996.283:0.02441,((31072:0.01102,541232:0.01583)0.987.209:0.01878,(547837:0.03618,(30766:0.04542,((568904:0.04012,562299:0.00015)0.822.38:0.00939,49771:0.00479)0.997.313:0.01813)0.878.62:0.00801)0.761.38:0.00916)1.000.2373:0.03117)0.280.9:0.011,(((565913:0.00094,327095:0.00581)0.999.621:0.02059,((542027:0.00636,572787:0.00718)0.999.622:0.01963,(564401:0.01985,16108:0.00467)0.752.34:0.02083)0.964.129:0.01591)0.477.9:0.0165,((46414:0.0301,(130030:0.00639,(220054:0.01618,((554614:0.02585,548100:0.01113)0.468.11:0.00319,(256611:0.00094,567007:0.00275)0.993.201:0.01143)0.858.48:0.00382)0.540.4:0.00665)1.000.2374:0.02545)0.845.33:0.00629,((151986:0.00617,137800:0.01784)1.000.2375:0.01594,((138500:0.02271,32182:0.0276)0.732.25:0.01001,584933:0.02824)0.123.3:0.00695)0.872.74:0.00928)0.918.83:0.0128)0.990.174:0.02336)0.992.206:0.02473)0.670.10:0.0112,(((((199886:0.01472,16087:0.01328)1.000.2376:0.06101,(((534780:0.04983,109644:0.03635)0.880.59:0.01644,(248069:0.05061,(210805:0.03382,((554995:0.00158,251338:0.00721)0.770.26:0.00167,(398425:0.01098,106936:0.01697)0.872.75:0.00508)0.999.623:0.02457)0.949.96:0.01077)0.751.22:0.00955)0.762.29:0.01078,(523387:0.01839,519584:0.00339)1.000.2377:0.02667)0.785.33:0.01021)0.952.102:0.01064,((196295:0.1023,((((204369:0.00962,561826:0.00779)1.000.2378:0.04166,(555547:0.03277,206881:0.04003)0.991.199:0.02077)0.991.200:0.01678,(((336601:0.03536,110797:0.00339)1.000.2379:0.04726,((((101672:0.01464,111250:0.01787)0.458.7:0.00272,(((110910:0.00273,(176138:0.01856,109670:0.01295)0.535.9:0.00458)0.926.81:0.00826,33347:0.03973)0.902.52:0.00623,(((105672:0.01375,(205022:0.00217,219113:0.02365)0.170.3:0.01445)0.999.624:0.01872,(((231376:0.00099,(188183:0.02028,178930:0.00973)0.991.201:0.01035)0.984.154:0.00819,570052:0.00717)0.897.62:0.00521,166629:0.02114)0.853.45:0.00305)0.810.31:0.00355,(313226:0.0228,(66191:0.01396,557576:0.04532)0.877.60:0.00459)0.869.60:0.01275)0.188.7:0.00393)0.300.4:0.00522)0.971.143:0.01119,((325254:0.02106,(((368422:0.01772,(184040:0.00283,224786:0.00311)0.983.149:0.00792)0.981.135:0.00843,(542440:0.02222,(177600:0.02298,549577:0.02344)0.567.11:0.0059)0.832.27:0.00628)0.493.7:0.00443,110605:0.02494)0.982.168:0.01412)0.916.68:0.00819,(107464:0.02436,(329761:0.00779,(297728:0.00149,145236:0.00273)0.991.202:0.00966)0.913.78:0.0081)0.958.113:0.01033)0.908.75:0.00606)0.874.56:0.00744,(320073:0.05873,(344860:0.0266,569463:0.02562)1.000.2380:0.0383)0.985.136:0.01958)0.961.105:0.01083)1.000.2381:0.02652,((584437:0.02489,513057:0.02182)0.995.278:0.01321,((((16130:0.00864,(563636:0.01322,(549867:0.01081,538197:0.02619)0.772.30:0.00554)0.946.80:0.00862)0.903.68:0.00475,((14134:0.00402,(161457:0.01237,((217463:0.006,(174127:0.01165,547472:0.00845)0.867.59:0.00206)0.798.24:0.00183,(83754:0.01418,(16131:0.01144,580071:0.01942)0.831.23:0.00377)0.885.57:0.00271)0.870.62:0.00302)0.836.34:0.00231)0.896.58:0.0025,316042:0.03538)0.099.3:0.00242)1.000.2382:0.02044,(((99789:0.01187,16086:0.0103)0.996.284:0.01571,(297732:0.01753,339645:0.01766)0.965.118:0.00905)0.887.52:0.00529,((101433:0.0047,177179:0.0278)1.000.2383:0.01449,552235:0.00078)1.000.2384:0.03133)0.983.150:0.01253)0.239.7:0.0046,((((350834:0.01337,274619:0.0249)0.362.6:0.00604,(302115:0.02139,295440:0.02434)0.881.67:0.01019)0.334.8:0.00194,217109:0.02336)0.919.77:0.004,((231952:0.00014,319455:0.04154)0.985.137:0.01337,544392:0.05137)0.960.122:0.00737)0.967.118:0.00817)0.981.136:0.01163)0.450.11:0.0054)0.860.56:0.00861)0.493.8:0.00296,((586485:0.068,(369177:0.01682,213762:0.00614)1.000.2385:0.04591)0.192.5:0.01211,(((((174358:0.02275,157545:0.01345)0.857.46:0.00641,(324015:0.02787,((((538994:0.02304,440684:0.01953)0.519.6:0.00185,(212758:0.0102,(157989:0.02409,(288525:0.01699,554511:0.01698)0.848.43:0.00716)0.840.31:0.00238)0.349.9:0.00138)0.927.80:0.00634,347523:0.01785)1.000.2386:0.02684,((229452:0.01584,(418334:0.01854,(216356:0.00683,323025:0.02877)0.980.133:0.00991)0.937.84:0.00751)0.988.189:0.01512,((197539:0.0381,(368252:0.02082,389371:0.0047)0.760.31:0.00338)0.985.138:0.02258,(((531174:0.00267,528434:0.02014)0.980.134:0.00671,((292395:0.03229,297458:0.031)0.925.71:0.00851,(525729:0.0182,511624:0.0287)0.849.46:0.00446)0.560.3:0.00016)0.897.63:0.00436,560042:0.01495)0.705.8:0.01095)0.997.314:0.02767)0.965.119:0.01263)0.993.202:0.0159)0.701.9:0.00702)0.997.315:0.01944,(562503:0.02574,69942:0.02163)0.997.316:0.02046)1.000.2387:0.03598,((187767:0.03087,(380753:0.01462,(205846:0.00839,324112:0.01342)0.615.13:0.00355)0.997.317:0.01884)0.999.625:0.03016,((149668:0.01496,220301:0.03426)1.000.2388:0.04444,((((552928:0.03916,(172163:0.00306,234054:0.01882)1.000.2389:0.03238)1.000.2390:0.02869,110842:0.03645)0.982.169:0.0163,(16272:0.04628,215086:0.01916)1.000.2391:0.03144)0.440.8:0.01094,227950:0.0258)0.759.21:0.01005)0.972.129:0.0144)0.445.6:0.01235)0.958.114:0.01935,(((248103:0.0215,(78317:0.02873,78174:0.00833)0.225.8:0.00382)1.000.2392:0.07387,((144142:0.0294,142190:0.02454)0.351.7:0.00579,(149631:0.03303,(169989:0.03364,143863:0.0244)0.988.190:0.01701)0.359.3:0.00462)1.000.2393:0.06097)1.000.2394:0.05695,(416837:0.06875,((18245:0.02758,237802:0.03625)1.000.2395:0.06225,(148219:0.03568,146392:0.02809)0.995.279:0.02442)0.849.47:0.0128)0.417.8:0.0169)0.963.105:0.02244)1.000.2396:0.03635)0.926.82:0.01108)0.126.7:0.00602)0.972.130:0.01222,(110327:0.03451,(((99414:0.00641,((296038:0.00917,(339673:0.00594,293977:0.01276)1.000.2397:0.03085)0.900.65:0.00513,(336665:0.07753,117121:0.00515)0.767.43:0.00136)0.966.113:0.00679)0.910.65:0.00435,(363519:0.00832,410242:0.00897)0.751.23:0.00181)1.000.2398:0.03404,((297730:0.01299,(183822:0.00259,348609:0.00014)0.971.144:0.00971)1.000.2399:0.05913,(((((((364765:0.02652,(357890:0.00239,363631:0.0102)1.000.2400:0.03198)0.868.66:0.00526,135426:0.01145)0.498.6:0.00401,135425:0.01278)0.656.5:0.00855,((((((23926:0.00892,583106:0.00693)1.000.2401:0.0242,(81242:0.02486,104874:0.02792)0.592.14:0.0056)0.935.92:0.00644,78366:0.0211)0.214.4:0.00349,99858:0.01688)0.434.11:0.00605,((78152:0.01909,(77725:0.01607,(578447:0.00395,142575:0.00731)0.998.406:0.01547)0.920.63:0.00862)0.958.115:0.0086,((583216:0.0256,(77906:0.01024,(77865:0.00817,150315:0.01791)0.986.159:0.01274)0.873.62:0.00677)0.633.7:0.00707,589553:0.01008)1.000.2402:0.02737)0.890.66:0.00578)0.995.280:0.02106,(270505:0.05063,(77796:0.00506,(368813:0.01724,366568:0.01557)0.989.161:0.01296)0.963.106:0.0103)0.999.626:0.0233)0.994.226:0.02471)0.736.34:0.01529,((((297963:0.01162,((351714:0.0132,334145:0.00535)0.943.98:0.00855,287808:0.01373)0.668.8:0.01016)0.839.53:0.00488,(191842:0.02647,((((344805:0.00839,(320030:0.00736,153989:0.01084)0.988.191:0.00818)0.970.111:0.00711,(261542:0.01497,268221:0.00995)0.857.47:0.0061)0.949.97:0.00659,(265260:0.00264,511896:0.00367)1.000.2403:0.03545)0.966.114:0.00693,(((344227:0.00236,345360:0.00437)0.995.281:0.01689,(269819:0.01086,(538647:0.00209,510387:0.03616)0.919.78:0.00579)0.980.135:0.01242)0.961.106:0.00858,((((((((318326:0.00461,(298712:0.01395,290761:0.0057)0.791.33:0.00141)0.552.11:0.00177,((299745:0.01181,292205:0.00744)0.928.75:0.00727,(290122:0.00753,339639:0.00912)0.429.6:0.00987)0.922.80:0.00382)0.909.65:0.00421,296695:0.00704)0.826.24:0.00387,(331014:0.01191,326224:0.02002)0.747.30:0.00244)0.897.64:0.0046,303296:0.01638)0.705.9:0.00161,(346374:0.02842,330496:0.01791)0.844.31:0.00212)0.852.64:0.00181,((214441:0.00078,300655:0.01241)0.904.69:0.00169,(194211:0.03054,336953:0.01477)0.987.210:0.01236)0.734.24:0.00099)0.939.97:0.00375,571720:0.01369)0.137.5:0.00212)0.544.8:0.00128)0.920.64:0.00522)0.998.407:0.01556)0.991.203:0.01198,((333033:0.03214,(312520:0.05491,((326804:0.02717,(341717:0.00982,296723:0.01986)0.998.408:0.01439)0.899.67:0.00993,(355003:0.02467,291601:0.02522)0.996.285:0.01921)0.995.282:0.02055)0.934.67:0.00993)1.000.2404:0.02808,(((279716:0.02405,((571666:0.01177,(83830:0.01718,493341:0.00724)1.000.2405:0.0257)0.887.53:0.01028,(252834:0.02107,((114756:0.01364,(321748:0.00606,227245:0.01111)0.925.72:0.01293)1.000.2406:0.03041,246299:0.01526)0.117.7:0.00462)0.936.86:0.00675)0.937.85:0.00748)0.998.409:0.02217,(270625:0.0066,(315261:0.00845,110551:0.00851)0.948.100:0.00584)1.000.2407:0.02037)0.026.3:0.00597,(((323826:0.01719,(((((305371:0.02455,((110359:0.01765,(((109953:0.00852,352644:0.00185)0.868.67:0.00677,(102625:0.00936,((((303837:0.00835,333272:0.01185)0.268.8:0.00107,((355534:0.00839,346478:0.01096)0.581.8:0.00015,297242:0.01786)0.891.66:0.00418)0.880.60:0.00176,(164629:0.00081,235597:0.00236)0.762.30:0.00467)0.861.60:0.00191,(355800:0.01183,(109496:0.00155,303828:0.00907)0.927.81:0.00382)0.914.64:0.00523)0.917.77:0.00366)0.896.59:0.00346)0.837.26:0.00519,(258613:0.00551,(168506:0.01439,(322078:0.00312,(299525:0.0035,153332:0.00396)0.965.120:0.00016)0.864.69:0.00625)0.749.28:0.00012)0.854.43:0.00303)0.918.84:0.00381)0.983.151:0.00715,(305545:0.00745,(273272:0.00918,(314921:0.01065,(319115:0.00281,343651:0.00961)0.953.98:0.00502)0.922.81:0.00565)0.659.12:0.0058)0.948.101:0.00473)0.750.29:0.00075)0.952.103:0.00315,(321445:0.00421,(109894:0.013,583213:0.02419)0.946.81:0.01093)0.996.286:0.00941)0.549.6:0.00066,((100706:0.00888,(339691:0.00652,353910:0.0055)0.485.6:0.00015)0.923.76:0.00233,((290585:0.00555,321843:0.00869)0.995.283:0.00849,(330873:0.00969,304883:0.00941)0.873.63:0.00404)0.824.23:0.0033)0.916.69:0.00238)1.000.2408:0.01583,(301715:0.00759,((325168:0.00251,288312:0.00768)0.927.82:0.00366,291491:0.01541)0.707.11:0.00096)0.858.49:0.00397)0.134.5:0.00283,(339693:0.02176,289781:0.02321)1.000.2409:0.03767)0.771.33:0.00449)0.919.79:0.00779,((309501:0.00742,(335015:0.02104,565528:0.01786)1.000.2410:0.02435)0.986.160:0.0113,587962:0.02166)0.988.192:0.01373)0.259.11:0.00665,((322595:0.00705,16466:0.00677)0.415.7:0.03279,(562970:0.00669,101427:0.00623)0.378.12:0.00288)0.995.284:0.01507)0.967.119:0.01039)1.000.2411:0.0263)0.360.5:0.00709)0.929.92:0.00575,(((((((((173597:0.02382,292643:0.01021)0.772.31:0.00451,(208807:0.00473,185339:0.03603)0.648.12:0.01821)0.871.64:0.00529,(346840:0.01324,(((355503:0.00624,(288637:0.01601,((((296709:0.01285,349752:0.00815)0.854.44:0.00317,((297554:0.02743,(291839:0.03048,300646:0.00819)0.951.86:0.00299)0.750.30:0.00277,(320289:0.00735,305293:0.00711)0.436.9:0.00178)0.516.9:0.00237)0.927.83:0.00474,292215:0.01636)0.391.9:0.00143,300443:0.01922)0.875.50:0.00385)0.923.77:0.00361)0.745.30:0.00099,529918:0.01559)0.907.69:0.00607,((291097:0.01438,(302525:0.00456,297295:0.01394)0.511.9:0.00207)0.975.134:0.01093,(295935:0.02949,297601:0.02791)0.924.70:0.00906)0.107.4:0.00399)0.999.627:0.01733)0.946.82:0.00774)0.640.9:0.00297,((290559:0.01204,(299042:0.01028,296717:0.00532)0.303.7:0.00276)0.959.96:0.0057,((167946:0.00392,((99808:0.00872,593474:0.00707)0.868.68:0.00652,309781:0.00816)0.932.75:0.00645)0.886.53:0.00015,((340516:0.01197,112382:0.00626)0.456.9:0.00168,((((319333:0.01401,112374:0.0072)0.945.83:0.00262,(((300390:0.00453,(592156:0.00789,(107443:0.00824,585991:0.01312)0.878.63:0.00467)0.930.83:0.00239)0.924.71:0.00014,45569:0.00555)0.529.9:0.00147,111389:0.00874)0.927.84:0.00285)0.192.6:0.0026,172652:0.0031)0.772.32:0.00319,((((289817:0.01272,(348946:0.0099,111789:0.00357)0.632.7:0.00197)0.774.20:0.00237,((102387:0.01406,(324374:0.00079,(266243:0.01353,350328:0.00161)0.818.21:0.0167)0.758.31:0.00129)0.939.98:0.00281,((((338067:0.0045,(167555:0.00636,(262286:0.01209,104515:0.00627)0.783.32:0.00169)0.419.6:0.00164)0.942.95:0.00155,135173:0.02034)0.868.69:0.00077,((353984:0.02601,330542:0.01025)0.754.26:0.00662,100045:0.00905)0.928.76:0.00059)0.776.34:0.00158,311103:0.00605)0.893.57:0.0023)0.815.23:0.00157)0.916.70:0.0056,(((345307:0.00701,111130:0.00904)0.837.27:0.00227,104660:0.00641)0.948.102:0.00487,101011:0.01988)0.780.26:0.00078)0.736.35:0.00106,(111724:0.00883,111550:0.00354)0.871.65:0.00329)0.931.78:0.00305)0.890.67:0.00177)0.910.66:0.00155)0.764.25:0.00289)0.810.32:0.00346)0.893.58:0.00426,(((341416:0.00452,258791:0.01588)0.364.8:0.02948,((272425:0.01402,(320258:0.01967,(274592:0.01414,(318414:0.00741,322214:0.00808)0.896.60:0.0037)0.968.114:0.00931)0.625.11:0.00213)0.983.152:0.01039,(393699:0.02078,((316685:0.00091,263891:0.00839)1.000.2412:0.02976,((111455:0.01389,102847:0.00701)0.569.6:0.00623,((312557:0.00172,262314:0.00438)0.997.318:0.01561,(112201:0.01288,356192:0.01125)0.725.21:0.00733)0.878.64:0.00572)0.912.60:0.01007)0.086.4:0.00344)0.517.7:0.006)0.935.93:0.00689)0.799.28:0.00639,520004:0.02104)0.798.25:0.00688)0.955.101:0.00793,(566434:0.02947,((570930:0.00278,197427:0.00307)0.743.34:0.01156,569681:0.01114)0.295.5:0.00651)0.959.97:0.00895)0.947.89:0.00862,((270638:0.00769,592660:0.016)0.698.18:0.00444,(180400:0.01183,356309:0.01763)0.837.28:0.00335)0.999.628:0.01886)0.162.2:0.00592,(((((463007:0.00804,212288:0.01496)0.925.73:0.00413,325749:0.02489)0.870.63:0.00356,((((347271:0.00407,294358:0.01511)0.823.25:0.0023,(304460:0.0144,298190:0.00837)0.154.3:0.00942)0.760.32:0.00138,310222:0.01841)0.300.5:0.00316,((318442:0.02535,((301218:0.01921,344639:0.00893)0.988.193:0.01013,350906:0.00801)0.712.15:0.00536)0.909.66:0.00504,336508:0.01317)0.744.29:0.00143)0.975.135:0.00834)0.892.50:0.00323,(292758:0.01361,(343258:0.01991,(291773:0.02473,338304:0.00779)0.766.28:0.00684)0.682.9:0.00569)0.939.99:0.00571)0.944.77:0.00485,((((168046:0.01866,311772:0.01061)0.468.12:0.01042,194970:0.00586)0.777.26:0.00133,577684:0.01265)0.996.287:0.01313,(113972:0.0456,(149112:0.01115,(326523:0.00884,343949:0.02454)0.682.10:0.0061)0.824.24:0.00593)0.731.30:0.00616)0.385.8:0.00249)0.906.77:0.00931)0.857.48:0.00989,(317515:0.00518,297498:0.02103)1.000.2413:0.02536)0.904.70:0.00784)0.805.27:0.01013)0.989.162:0.01589,(362955:0.03993,((((((((((112186:0.01682,(259158:0.02091,(277425:0.00945,574618:0.0143)1.000.2414:0.02575)0.597.12:0.00567)0.656.6:0.00288,(365297:0.01282,195998:0.01338)0.930.84:0.00476)0.982.170:0.00684,((((329283:0.0024,(348940:0.00633,338997:0.00993)0.972.131:0.0063)0.363.7:0.00014,299456:0.01922)0.990.175:0.00701,295527:0.00405)0.948.103:0.00331,(148368:0.00521,540910:0.01103)1.000.2415:0.01813)0.769.31:0.00078)0.877.61:0.00216,((190409:0.00358,218710:0.00685)0.829.30:0.01829,((323234:0.03054,299885:0.01256)0.985.139:0.01243,291690:0.01983)0.414.9:0.0054)0.965.121:0.00935)0.857.49:0.0029,((331156:0.02005,(292296:0.02442,((352091:0.01288,299306:0.02274)0.954.99:0.00955,((177039:0.00354,552988:0.00328)0.731.31:0.00657,(329282:0.00914,298178:0.02203)0.850.36:0.01176)0.080.6:0.00644)0.207.5:0.00463)0.755.40:0.00297)0.872.76:0.00462,((((353460:0.00942,304071:0.02346)0.976.116:0.00936,(((194833:0.003,289639:0.0078)0.993.203:0.01288,339644:0.01086)0.724.16:0.00603,(510750:0.01574,290542:0.01652)0.813.28:0.00802)0.884.60:0.00945)0.859.53:0.00337,(((((((334849:0.00666,((301205:0.00803,299783:0.00488)0.796.28:0.0034,(347369:0.01067,(327560:0.00371,(302410:0.01056,297722:0.02182)0.796.29:0.00161)0.803.27:0.00188)0.744.30:0.00158)0.959.98:0.00505)0.754.27:0.00784,(301374:0.00087,(315775:0.02636,296567:0.01509)0.873.64:0.0084)0.936.87:0.00568)0.794.30:0.00179,340508:0.01493)0.837.29:0.00116,343551:0.01002)0.964.130:0.00634,(131845:0.01197,576853:0.00278)0.947.90:0.00489)0.316.10:0.00457,(333999:0.014,302450:0.01653)0.664.10:0.00156)0.749.29:0.00089,((301206:0.00597,289855:0.01844)0.914.65:0.00405,512522:0.01617)0.714.15:0.00159)0.745.31:0.00106)0.973.116:0.00531,((339530:0.02546,(((308849:0.00518,107159:0.00311)0.758.32:0.00077,522927:0.00388)0.898.72:0.00095,((310662:0.00471,318530:0.01073)0.941.108:0.00367,((((311412:0.00661,(350400:0.00621,((310611:0.00645,((110428:0.01545,109800:0.00988)0.966.115:0.01106,((172751:0.01882,(260907:0.00798,104021:0.00976)0.883.62:0.00237)0.876.60:0.00163,105492:0.00865)0.868.70:0.00158)0.785.34:0.00129)0.478.6:0.00272,((299039:0.00359,(325578:0.00365,(209059:0.01124,352369:0.00463)0.931.79:0.00593)0.884.61:0.00584)0.951.87:0.00439,581109:0.00916)0.507.8:0.00309)0.850.37:0.00224)0.900.66:0.00314)0.530.11:0.00015,(589903:0.00777,(((251961:0.0025,106522:0.00372)0.891.67:0.00262,(249189:0.01282,511820:0.00923)0.783.33:0.00182)0.949.98:0.00436,301323:0.01216)0.949.99:0.00331)0.909.67:0.00312)0.814.25:0.00161,(((326242:0.01132,(111024:0.01199,((329788:0.00216,352706:0.00686)0.376.9:0.00273,170255:0.0166)0.473.10:0.00215)0.840.32:0.00681)0.861.61:0.00423,((251395:0.00315,297662:0.00538)0.874.57:0.00157,(((308383:0.0046,111415:0.01984)0.986.161:0.00867,346231:0.00603)0.963.107:0.00505,(580224:0.01563,(24765:0.00583,262016:0.01293)0.362.7:0.00169)0.975.136:0.00649)0.969.123:0.00011)0.739.29:0.00062)0.880.61:0.00201,((112105:0.00397,(111293:0.01056,(167408:0.00838,561916:0.00589)0.326.12:0.00156)0.962.118:0.00443)0.721.15:0.00155,(100862:0.00479,208891:0.01859)0.749.30:0.0007)0.831.24:0.00207)0.428.7:0.00155)0.909.68:0.00313,301010:0.00482)0.841.42:0.00282)0.997.319:0.00015)0.989.163:0.01178)0.988.194:0.01147,312592:0.00798)0.992.207:0.0098)0.483.9:0.00015)0.930.85:0.00506)0.977.142:0.00887,298408:0.01422)0.944.78:0.00661,(174033:0.01692,301002:0.00959)0.878.65:0.00492)0.979.142:0.00671,307420:0.01409)0.925.74:0.00479,((((323155:0.01413,256619:0.00216)0.990.176:0.01093,(((344489:0.01069,(344129:0.0087,589931:0.0139)0.093.2:0.00233)0.992.208:0.01196,((((339796:0.00767,310192:0.00757)0.939.100:0.00638,((258394:0.01698,336198:0.01014)0.987.211:0.00804,332625:0.00382)0.559.8:0.00489)0.893.59:0.00637,(311096:0.00502,321950:0.00977)0.882.59:0.00411)0.264.6:0.0047,(577468:0.00409,590015:0.03756)1.000.2416:0.02012)0.800.33:0.00392)0.929.93:0.00445,573551:0.0354)0.959.99:0.00751)0.663.9:0.0065,(((542417:0.01864,(299009:0.01372,311009:0.01448)0.930.86:0.00784)0.998.410:0.01228,(287880:0.01058,302105:0.01232)0.997.320:0.01452)0.437.8:0.0049,313089:0.04323)0.909.69:0.00552)0.974.122:0.01072,(274560:0.02312,149085:0.02234)0.529.10:0.00678)0.866.49:0.00404)0.961.107:0.00612,(((360186:0.02416,((133222:0.00943,524935:0.01998)1.000.2417:0.02711,361679:0.01041)0.175.9:0.00259)0.555.9:0.0013,(462052:0.03399,(587142:0.00755,(191866:0.03179,((273437:0.01571,(336747:0.02014,242174:0.0225)0.258.5:0.0044)0.988.195:0.01158,(323118:0.01098,306785:0.00692)0.987.212:0.01323)0.888.51:0.00389)0.999.629:0.0205)0.998.411:0.01835)0.168.16:0.00129)0.938.94:0.0064,359509:0.01265)0.970.112:0.00766)0.911.68:0.00537)0.914.66:0.01017)0.941.109:0.01453,(560873:0.01031,369994:0.01506)0.997.321:0.01857)0.940.104:0.01321)0.902.53:0.01615)0.985.140:0.01716)0.997.322:0.02195)0.856.44:0.00575)0.983.153:0.0148,(((((556164:0.05287,50158:0.02435)1.000.2418:0.03983,((((((219192:0.01921,((138311:0.01294,100155:0.00322)0.144.4:0.00175,((223000:0.00611,(216349:0.00112,((114917:0.01215,509371:0.01726)0.749.31:0.00245,590008:0.00979)0.937.86:0.00476)0.984.155:0.00871)0.972.132:0.00808,81208:0.00279)0.964.131:0.00493)0.925.75:0.00497)0.890.68:0.00273,((((((353886:0.02381,577614:0.02458)0.871.66:0.00441,((((((((((41245:0.03035,572366:0.01087)0.990.177:0.01505,134002:0.02335)0.723.21:0.00582,(153096:0.0104,579868:0.02524)0.830.33:0.00352)0.911.69:0.00488,587360:0.02929)0.902.54:0.00793,591475:0.01169)0.606.12:0.00337,561133:0.02077)0.851.50:0.00305,(((564645:0.01091,(15960:0.01625,(15980:0.0079,15957:0.01428)0.702.14:0.00062)0.818.22:0.00229)1.000.2419:0.01751,(15955:0.01238,109994:0.00372)0.811.26:0.00785)0.649.7:0.00226,200112:0.03182)0.920.65:0.00271)0.865.55:0.00164,((341557:0.01603,(556185:0.01963,(53642:0.01221,554015:0.01831)0.911.70:0.00685)0.943.99:0.00782)0.911.71:0.00386,(315396:0.01889,(((131553:0.01912,((122896:0.0093,338452:0.00622)0.734.25:0.00554,(50120:0.01146,128200:0.00874)0.361.7:0.00276)0.904.71:0.00574)0.846.30:0.00014,16925:0.01852)0.833.33:0.00371,(((560650:0.01677,128058:0.0106)0.837.30:0.00015,(142282:0.00594,(271652:0.00984,(304678:0.01645,268811:0.01059)0.446.9:0.00144)0.980.136:0.00694)0.515.15:0.00239)0.793.24:0.0005,(523910:0.0059,(161992:0.01071,211851:0.02256)1.000.2420:0.01566)0.711.12:0.00605)0.977.143:0.0077)0.910.67:0.00629)0.982.171:0.00766)0.851.51:0.0025)0.140.3:0.00012,(((586020:0.01786,((550979:0.01655,532875:0.0154)0.723.22:0.00758,(142277:0.01583,578409:0.01973)0.843.39:0.00395)1.000.2421:0.02003)0.125.6:0.00309,(97132:0.01795,555754:0.01642)0.967.120:0.00788)0.299.7:0.00235,(144522:0.01641,((560325:0.00739,262840:0.01902)0.933.81:0.00938,(16001:0.02252,(136404:0.0217,131056:0.00309)0.957.136:0.00899)0.890.69:0.00716)0.868.71:0.00868)0.956.96:0.00866)0.817.29:0.00185)0.872.77:0.00642,135577:0.06386)0.816.32:0.00249)0.861.62:0.00191,((133750:0.01352,538658:0.03203)0.967.121:0.0101,(132457:0.00371,(574342:0.01433,565390:0.00603)0.241.3:0.00722)0.796.30:0.00439)0.255.3:0.00334)0.730.23:0.00342,((537375:0.03914,48313:0.01642)0.985.141:0.01655,556214:0.01551)0.959.100:0.01038)0.874.58:0.00422,(142890:0.00757,15969:0.0114)0.946.83:0.00448)0.967.122:0.00541,106370:0.02146)0.871.67:0.00293)0.880.62:0.00506,(105179:0.00904,211744:0.01789)1.000.2422:0.05309)0.883.63:0.00601,((16013:0.03654,((547395:0.0193,164182:0.00951)0.891.68:0.01039,(213313:0.00508,((238697:0.01311,360102:0.01823)0.995.285:0.01249,((((354853:0.0156,218474:0.01164)1.000.2423:0.02219,97792:0.00909)0.870.64:0.00333,257293:0.01313)0.975.137:0.00506,585365:0.00575)0.810.33:0.00221)0.985.142:0.01349)0.835.33:0.00841)0.352.7:0.00764)0.989.164:0.01831,(205562:0.03852,(77496:0.02126,561806:0.0201)1.000.2424:0.02468)0.776.35:0.01172)0.879.67:0.00592)0.945.84:0.01028,(((114115:0.02871,(358030:0.00349,211458:0.01388)0.895.67:0.02011)1.000.2425:0.03354,(172948:0.02082,406604:0.0304)1.000.2426:0.02698)0.676.12:0.00513,(520891:0.03542,(((((224390:0.06349,(518565:0.00477,99206:0.01522)1.000.2427:0.03463)0.870.65:0.01215,(309894:0.00397,554797:0.0154)0.278.6:0.04244)0.246.4:0.00703,(((555096:0.02653,(70302:0.02527,69664:0.03469)0.919.80:0.01039)0.880.63:0.00986,((221124:0.00803,546480:0.01115)1.000.2428:0.03082,(224110:0.02242,218576:0.00728)1.000.2429:0.03952)0.994.227:0.0211)0.673.8:0.01728,97301:0.09139)0.975.138:0.01599)0.332.7:0.00549,((17027:0.01558,(325962:0.01895,53823:0.02014)0.911.72:0.00534)0.051.7:0.00299,((114814:0.00604,74679:0.01171)0.985.143:0.01036,565478:0.02853)0.971.145:0.00999)0.955.102:0.00979)0.935.94:0.00745,(((341912:0.01617,(((316964:0.00851,(263036:0.01202,(299194:0.02009,553970:0.00943)0.768.38:0.00193)0.930.87:0.00389)0.494.10:0.00278,(293847:0.0162,(298237:0.01299,(571954:0.01409,570898:0.00772)0.994.228:0.01036)0.765.31:0.00569)0.836.35:0.00215)0.845.34:0.00285,((248256:0.00622,267134:0.01569)0.944.79:0.00892,((307646:0.00683,(334222:0.01025,305869:0.00273)0.786.31:0.01395)0.995.286:0.01476,(((538279:0.01551,540540:0.03465)0.973.117:0.00879,(99484:0.01864,(100371:0.00845,335224:0.01844)0.877.62:0.00528)0.282.4:0.00553)0.971.146:0.00807,554461:0.01661)0.962.119:0.01074)0.396.5:0.00472)0.787.28:0.00648)0.880.64:0.00716)0.999.630:0.01731,(351462:0.01684,270080:0.01821)0.577.3:0.00422)0.975.139:0.00951,(114488:0.00775,540546:0.0107)0.971.147:0.00679)1.000.2430:0.02426)0.926.83:0.00695)1.000.2431:0.02912)0.760.33:0.00505)0.392.9:0.00611,(279177:0.04878,(32177:0.01957,(159709:0.00646,(257774:0.01192,(589021:0.02515,558397:0.00828)0.025.4:0.00398)0.947.91:0.00771)0.698.19:0.00764)0.998.412:0.0232)1.000.2432:0.02645)0.996.288:0.02274)0.704.23:0.01097,((22883:0.04895,(293092:0.01845,(333352:0.01855,(296649:0.01957,544734:0.01107)0.946.84:0.00795)0.548.13:0.00711)0.999.631:0.0319)0.469.7:0.00998,((295729:0.03138,(108481:0.0439,((562369:0.01911,((268730:0.00461,525102:0.00478)1.000.2433:0.02495,161946:0.0206)0.879.68:0.00519)0.854.45:0.0064,(214651:0.018,((151647:0.01871,(522961:0.00646,(237063:0.01527,155461:0.0124)0.208.5:0.00378)0.794.31:0.00503)0.999.632:0.01873,514445:0.01857)0.782.29:0.00165)0.238.5:0.0038)0.998.413:0.03178)0.961.108:0.01745)1.000.2434:0.06632,((338971:0.00418,574394:0.01426)1.000.2435:0.06168,(399586:0.07786,((114670:0.03591,((((302171:0.01169,(99888:0.00874,319676:0.00972)0.774.21:0.01622)1.000.2436:0.02707,(((296146:0.00742,261161:0.01996)0.936.88:0.00584,(((299807:0.00975,(260290:0.00693,(110924:0.01057,275429:0.01341)0.249.5:0.00089)0.874.59:0.00345)1.000.2437:0.01714,(152282:0.00092,(((300990:0.01187,512683:0.01047)0.862.63:0.0032,168060:0.00872)0.524.8:0.00397,((540599:0.01823,((152014:0.009,261966:0.0112)1.000.2438:0.01388,(154859:0.00859,256393:0.00898)0.890.70:0.01237)0.874.60:0.00193)0.925.76:0.00337,531149:0.00964)0.979.143:0.00015)0.915.70:0.00339)0.775.20:0.00261)0.751.24:0.0012,(290338:0.01679,(135398:0.00772,(134494:0.00922,132950:0.02074)0.162.3:0.00491)1.000.2439:0.02002)0.977.144:0.0111)0.947.92:0.00722)0.979.144:0.00847,130890:0.00878)0.325.6:0.00352)0.799.29:0.00745,(134729:0.00622,(586232:0.01664,276563:0.01421)0.989.165:0.00939)0.956.97:0.00602)0.975.140:0.00859,((78177:0.04671,211234:0.01337)0.748.26:0.00132,(((((512361:0.03337,(109787:0.02895,312490:0.00453)0.994.229:0.01369)0.742.36:0.00529,(240691:0.00873,((523440:0.01536,(((97321:0.01008,298416:0.01644)0.386.9:0.00186,146754:0.02631)0.850.38:0.00239,588534:0.00728)0.843.40:0.00245)0.721.16:0.00187,(310219:0.01087,(214214:0.0177,153187:0.00902)0.853.46:0.00355)0.926.84:0.00405)0.922.82:0.0054)0.891.69:0.003)0.937.87:0.00505,(177704:0.01263,(196517:0.00854,580809:0.01477)0.999.633:0.01636)0.905.71:0.00477)0.982.172:0.00671,(((158451:0.00864,157802:0.02015)0.821.23:0.00297,((230618:0.01627,((361312:0.02187,365231:0.01247)0.880.65:0.00961,228232:0.00015)0.761.39:0.00752)0.917.78:0.00792,((157193:0.02165,(167872:0.01185,(210895:0.02244,((271602:0.01014,173245:0.01569)0.897.65:0.00454,164369:0.01959)0.876.61:0.00334)0.912.61:0.00604)0.673.9:0.00561)0.883.64:0.00401,166074:0.02988)0.740.34:0.00632)0.909.70:0.00598)0.897.66:0.00382,(((201812:0.02882,((231151:0.00572,267634:0.00724)0.820.20:0.0028,(288193:0.01111,258280:0.00784)0.431.10:0.00301)1.000.2440:0.02529)0.698.20:0.00613,((111015:0.0183,354431:0.00582)0.976.117:0.00842,((309873:0.02301,155117:0.00396)0.799.30:0.00236,304031:0.01495)0.973.118:0.00904)0.950.112:0.00758)0.974.123:0.00623,(132122:0.02565,((((((303030:0.01366,(231752:0.00254,179429:0.00667)0.974.124:0.00764)0.205.5:0.00412,(348897:0.02057,152278:0.00248)0.891.70:0.00519)0.973.119:0.00014,232778:0.01103)0.036.3:0.00241,592649:0.01149)0.996.289:0.01227,(((309419:0.00655,317085:0.01357)0.857.50:0.00397,(292672:0.00959,287827:0.0161)0.781.26:0.00412)0.980.137:0.01119,((293840:0.02172,591032:0.01464)0.990.178:0.01132,((155701:0.00669,(239427:0.01508,(347323:0.00246,535372:0.0076)0.869.61:0.00227)0.798.26:0.00768)0.685.11:0.00127,(204636:0.0276,(336867:0.00414,353850:0.01291)0.938.95:0.0083)0.747.31:0.00095)0.743.35:0.00137)0.787.29:0.00604)0.994.230:0.01321)1.000.2441:0.01466,(((511168:0.00885,((135365:0.00539,130407:0.01301)0.255.4:0.00793,48899:0.01988)0.337.5:0.00312)0.939.101:0.00766,((((165721:0.00149,162109:0.00252)0.295.6:0.00101,365712:0.05565)0.991.204:0.0112,49406:0.02811)0.993.204:0.0142,(((((229088:0.00969,(531038:0.01484,(105109:0.0091,(153075:0.00816,(295278:0.01601,271442:0.00809)0.512.12:0.00204)0.926.85:0.00499)0.865.56:0.00301)0.948.104:0.01344)0.901.63:0.004,(303038:0.01111,((((207718:0.01137,205594:0.01863)0.117.8:0.00395,((153800:0.01816,252327:0.00821)0.673.10:0.00015,164696:0.00511)0.999.634:0.01363)0.655.12:0.00165,254600:0.00896)0.935.95:0.00405,509838:0.00014)0.734.26:0.00045)0.902.55:0.00426)0.995.287:0.00704,(305549:0.01222,(((23757:0.01211,523404:0.01254)0.929.94:0.00354,(17598:0.00697,289771:0.0147)0.890.71:0.00172)0.847.38:0.00237,(327439:0.00904,(530615:0.00391,(522274:0.01855,527536:0.01474)0.777.27:0.00089)0.875.51:0.0018)0.874.61:0.00173)0.926.86:0.01337)0.958.116:0.0061)0.907.70:0.00235,(((105813:0.01668,((269956:0.02001,(165623:0.01425,526552:0.00953)0.903.69:0.01452)0.904.72:0.01089,111771:0.01231)0.727.19:0.00083)0.918.85:0.00414,((((565565:0.00057,363430:0.02224)0.187.10:0.014,((568692:0.00736,(((516447:0.00907,(204351:0.00822,300102:0.01663)0.874.62:0.0025)0.766.29:0.001,((((158591:0.00819,((238056:0.00156,(511272:0.0077,(298460:0.0054,522433:0.0045)0.938.96:0.004)0.307.6:0.00178)0.788.26:0.00412,158411:0.0042)0.980.138:0.0003)0.207.6:0.00495,158293:0.00997)0.864.70:0.0016,(((214684:0.01364,313037:0.01191)0.901.64:0.00241,340449:0.0054)0.443.10:0.00016,105349:0.00742)0.955.103:0.00387)0.768.39:0.00069,536989:0.01329)0.885.58:0.00164)0.780.27:0.00077,(518967:0.01381,157744:0.01486)0.391.10:0.01036)0.791.34:0.00074)0.768.40:0.00152,(530129:0.01007,566391:0.01988)0.038.2:0.00337)0.998.414:0.00014)0.773.32:0.00077,((157804:0.03553,146564:0.00703)0.857.51:0.01274,358312:0.013)0.351.8:0.00496)0.849.48:0.00159,(((16003:0.01251,((526490:0.00626,522130:0.0138)0.853.47:0.00248,167609:0.01455)0.279.9:0.00326)0.747.32:0.00079,((184154:0.00331,144196:0.01859)0.973.120:0.00419,345011:0.01525)0.894.63:0.00246)0.954.100:0.00419,(196795:0.01979,(195730:0.0504,511510:0.01756)0.973.121:0.01168)0.992.209:0.01437)0.902.56:0.00373)0.848.44:0.0023)0.862.64:0.00171,335206:0.00508)0.877.63:0.00191)0.848.45:0.00259,(344807:0.00357,(304381:0.01661,317213:0.00426)0.989.166:0.01179)0.999.635:0.01743)0.764.26:0.00555)0.921.78:0.00662)0.909.71:0.00351,(270013:0.03624,(((205067:0.01262,((((250361:0.00719,168698:0.02396)0.949.100:0.00864,(164647:0.01123,((134365:0.02647,534874:0.00433)0.457.7:0.00342,267216:0.02036)0.998.415:0.01405)0.962.120:0.0089)0.658.9:0.007,203737:0.02029)0.869.62:0.00315,(260716:0.03503,208300:0.01291)0.816.33:0.00476)0.525.8:0.00428)0.296.7:0.00319,(((555304:0.01634,272980:0.01359)0.980.139:0.01188,(240560:0.01952,352638:0.03149)0.973.122:0.0125)0.924.72:0.00858,549631:0.04628)0.222.10:0.00736)0.869.63:0.00877,255669:0.02685)0.991.205:0.01386)0.946.85:0.00781)0.953.99:0.00691)0.654.7:0.00665)0.932.76:0.00484)0.991.206:0.00857)0.397.16:0.00076)0.942.96:0.00472,((365319:0.01456,148500:0.02251)0.705.10:0.00489,((297219:0.02305,(155975:0.01087,342431:0.01056)0.570.12:0.00273)0.799.31:0.00396,(155911:0.01349,(270786:0.0179,350056:0.01147)0.587.8:0.00371)0.920.66:0.01298)0.873.65:0.00385)0.990.179:0.00777)0.942.97:0.00551)0.775.21:0.00437)0.770.27:0.00247)0.614.10:0.00899,((207713:0.00394,((351821:0.00777,103804:0.018)0.870.66:0.00296,(338243:0.01033,350719:0.01768)0.883.65:0.00437)0.971.148:0.00662)0.814.26:0.00215,107768:0.00509)0.999.636:0.02404)0.965.122:0.01781)0.992.210:0.0242)0.513.8:0.01514)0.973.123:0.023)0.993.205:0.02493)0.974.125:0.01135,((289546:0.03643,(332742:0.02761,293085:0.0274)0.935.96:0.00979)0.979.145:0.01361,(((((583328:0.03604,(144981:0.01203,(81891:0.03387,81110:0.00904)0.806.26:0.00709)0.945.85:0.01869)1.000.2442:0.05387,(142583:0.02323,135518:0.03121)1.000.2443:0.05074)0.998.416:0.03314,((287835:0.03685,((((291202:0.00608,226682:0.01727)0.997.323:0.01561,((336890:0.01415,213045:0.00042)0.863.57:0.00794,(291484:0.00856,170620:0.01795)0.478.7:0.0043)0.832.28:0.00367)0.235.5:0.00492,(195252:0.01964,288081:0.02155)0.996.290:0.01937)0.585.9:0.00816,((((301561:0.01419,((((95424:0.01098,208305:0.00898)0.802.30:0.00136,170555:0.00589)0.979.146:0.00588,((235071:0.00111,266512:0.0044)0.995.288:0.00016,263044:0.02017)0.986.162:0.00686)0.991.207:0.01017,((363997:0.01107,(21195:0.00724,186654:0.00853)0.855.45:0.00568)0.733.27:0.00722,((580521:0.00705,(186321:0.00939,(198841:0.02581,(170124:0.00332,((198941:0.01728,176386:0.00761)1.000.2444:0.00103,193865:0.02664)1.000.2445:0.00014)0.874.63:0.00122)0.993.206:0.00575)0.899.68:0.00336)0.891.71:0.00174,(187846:0.0025,(158946:0.00164,237438:0.01073)0.767.44:0.00257)0.879.69:0.00236)0.824.25:0.01118)0.958.117:0.00763)0.954.101:0.0062)0.731.32:0.0035,(272926:0.00774,317286:0.01077)0.943.100:0.00556)0.923.78:0.00529,(319881:0.01394,278901:0.01654)0.984.156:0.01168)0.979.147:0.00938,16057:0.02342)0.887.54:0.00539)0.998.417:0.02208)0.977.145:0.01589,((((((336486:0.02323,299331:0.03179)0.966.116:0.01062,((205309:0.02679,149078:0.02601)0.949.101:0.00916,(367864:0.01475,81848:0.03075)1.000.2446:0.02795)0.062.2:0.00437)0.847.39:0.00631,539939:0.0382)0.672.6:0.00566,398138:0.04688)0.953.100:0.00728,(((380392:0.01733,216140:0.02488)1.000.2447:0.03469,(((151267:0.01745,326000:0.00306)0.985.144:0.01128,(332726:0.02269,297021:0.01714)0.195.5:0.00597)1.000.2448:0.03601,(((296863:0.02064,154393:0.00953)0.928.77:0.01242,(298624:0.01792,110477:0.02257)0.441.5:0.00704)1.000.2449:0.04298,(330285:0.0317,(299759:0.02341,273552:0.02817)0.991.208:0.01898)0.983.154:0.01765)0.258.6:0.01668)0.995.289:0.02652)0.865.57:0.01892,(((((183699:0.00617,((179090:0.00103,180108:0.03268)0.179.3:0.00014,510817:0.00086)0.995.290:0.01232)0.999.637:0.01118,(((((((275748:0.02922,((329869:0.00014,290513:0.03894)0.991.209:0.0181,(295667:0.0187,(302287:0.00843,(((356394:0.02043,(291609:0.01161,((349800:0.00895,(301182:0.01425,299280:0.01549)1.000.2450:0.00449)0.842.51:0.00235,(301415:0.01081,165597:0.00402)0.893.60:0.00196)0.891.72:0.00584)0.744.31:0.00015)0.884.62:0.00915,((579278:0.00104,292062:0.01774)0.775.22:0.0162,((51907:0.01197,((111800:0.00882,556788:0.00846)0.874.64:0.00571,((332538:0.00014,587722:0.00592)0.960.123:0.0051,(312967:0.0009,207816:0.01661)0.951.88:0.00722)0.931.80:0.00253)0.869.64:0.00277)0.961.109:0.00411,((433722:0.00712,((321053:0.01006,299555:0.01015)0.911.73:0.0142,327834:0.00713)0.752.35:0.00087)0.631.7:0.00835,(338781:0.00905,296737:0.01395)0.886.54:0.0001)0.102.5:0.00354)0.876.62:0.00261)0.854.46:0.00238)0.290.6:0.00264,(((294626:0.00808,563490:0.01735)0.981.137:0.01258,(302056:0.02315,299836:0.01447)0.912.62:0.00446)0.461.8:0.00907,309922:0.00644)0.536.9:0.00212)0.745.32:0.00081)0.935.97:0.00336)0.645.17:0.00303)0.889.55:0.00433)0.914.67:0.00454,((301325:0.01467,((335011:0.00045,407759:0.00885)0.804.28:0.00468,((295585:0.01209,326444:0.01452)0.560.4:0.023,(298819:0.01353,296591:0.00895)0.824.26:0.00173)0.874.65:0.00639)0.683.16:0.00283)0.753.34:0.00086,289495:0.02104)0.874.66:0.006)0.902.57:0.00503,((582089:0.00754,(302812:0.01451,549499:0.0046)0.990.180:0.01072)0.957.137:0.01202,(186640:0.02196,358295:0.00218)0.764.27:0.00816)0.970.113:0.00712)0.507.9:0.00282,(((344906:0.00544,232257:0.00589)0.907.71:0.01282,(345353:0.0223,(290637:0.02455,(298804:0.01434,321410:0.01769)0.916.71:0.00576)0.738.25:0.00374)1.000.2451:0.01859)0.830.34:0.00591,(273850:0.02888,(314448:0.00123,320155:0.00198)0.950.113:0.015)0.410.7:0.00332)0.911.74:0.00758)0.977.146:0.00798,(((336513:0.02859,297796:0.02601)0.996.291:0.02001,((((196210:0.02577,183187:0.0072)0.042.3:0.00014,182321:0.01456)1.000.2452:0.00107,164025:0.00323)0.945.86:0.0045,((99242:0.00428,(332860:0.01186,317405:0.02066)0.981.138:0.01189)0.867.60:0.00255,(111120:0.01353,(335203:0.01289,302165:0.02194)0.991.210:0.01)0.882.60:0.00315)0.340.8:0.00401)0.933.82:0.00773)0.908.76:0.00557,((155059:0.02425,335010:0.02943)1.000.2453:0.01882,((287826:0.00544,298695:0.01196)0.990.181:0.01129,(((338028:0.00184,309061:0.00549)0.982.173:0.00888,(337524:0.00105,208443:0.01198)0.968.115:0.02253)0.958.118:0.01006,151013:0.0067)0.924.73:0.00529)0.912.63:0.00571)0.851.52:0.00364)0.881.68:0.00371)0.990.182:0.0061,16034:0.01642)0.417.9:0.00164,((525763:0.01462,364621:0.01967)0.903.70:0.00436,((557458:0.0092,((132303:0.01059,578216:0.01755)0.887.55:0.00298,(515774:0.01358,(339994:0.02255,568260:0.01095)0.982.174:0.00841)0.961.110:0.00675)0.344.6:0.00316)0.964.132:0.00627,(((379762:0.01223,(299233:0.01176,(288487:0.00936,(312029:0.006,292360:0.00272)0.813.29:0.00204)0.998.418:0.01046)0.452.16:0.00415)1.000.2454:0.02989,135737:0.02754)0.860.57:0.00443,78376:0.02204)0.913.79:0.00452)0.983.155:0.01077)0.859.54:0.00203)0.958.119:0.00537)0.518.5:0.00197,((180468:0.04879,((((77848:0.01993,(571808:0.00945,(578230:0.01007,78408:0.01712)0.800.34:0.01057)0.943.101:0.01178)1.000.2455:0.05923,(345556:0.03474,(355155:0.01496,298739:0.01471)0.935.98:0.02322)0.999.638:0.0268)0.850.39:0.00763,99182:0.01767)0.985.145:0.01363,(293308:0.00828,(295239:0.0293,594119:0.0266)0.741.26:0.00906)0.985.146:0.01562)0.275.6:0.00357)0.101.6:0.0044,((347804:0.03607,(((338681:0.01043,336032:0.04818)0.995.291:0.02545,(288349:0.02096,345938:0.03986)0.144.5:0.01213)0.985.147:0.01673,(287561:0.01046,328777:0.0207)0.414.10:0.00109)0.767.45:0.00376)0.960.124:0.00857,(272021:0.02115,(((301418:0.01707,326022:0.01926)1.000.2456:0.03488,(300385:0.03905,324219:0.01059)0.195.6:0.00337)0.568.10:0.01044,539506:0.00884)0.962.121:0.00832)0.183.3:0.0034)1.000.2457:0.02228)0.510.6:0.00482)0.986.163:0.0097,(((302160:0.01605,(198512:0.09014,(287529:0.03032,199149:0.03279)0.969.124:0.0173)0.916.72:0.00955)0.971.149:0.01153,(((292243:0.02172,((330252:0.01053,(292001:0.02118,351927:0.03821)0.926.87:0.00663)0.973.124:0.00904,292438:0.02622)0.428.8:0.00357)0.927.85:0.00703,(157081:0.02876,291270:0.01374)1.000.2458:0.02723)0.701.10:0.00121,(((((((((((276522:0.02494,(334771:0.0004,167927:0.00319)0.795.26:0.00546)0.782.30:0.01276,452682:0.00938)0.096.9:0.00149,(258191:0.01004,((269172:0.00818,167034:0.02068)0.717.16:0.00482,((320249:0.01125,232326:0.00639)0.940.105:0.00015,((383973:0.01728,182033:0.0175)0.757.40:0.00085,270993:0.01467)0.855.46:0.00078)0.807.23:0.00078)0.764.28:0.00156)0.882.61:0.00224)0.950.114:0.00418,((269994:0.00445,((((373909:0.0103,271563:0.0094)0.867.61:0.00642,434236:0.00494)0.747.33:0.00312,((16023:0.00015,308523:0.02704)0.846.31:0.00128,(343689:0.00487,261209:0.01228)0.889.56:0.00135)0.062.3:0.0003)0.744.32:0.00078,(((210292:0.0107,(((((177113:0.00364,363858:0.00849)0.864.71:0.00253,209327:0.01705)0.830.35:0.00339,213904:0.0165)0.888.52:0.00262,((((214009:0.00872,355314:0.05344)0.164.2:0.00584,297045:0.01679)0.304.7:0.00578,(300851:0.03248,289306:0.00819)0.946.86:0.00539)0.851.53:0.00014,(231978:0.0031,175735:0.00462)0.451.8:0.00667)0.937.88:0.00379)0.907.72:0.00889,204093:0.00678)0.105.3:0.00372)0.300.6:0.00219,158027:0.01041)0.925.77:0.00418,(((269418:0.00918,271290:0.04068)0.919.81:0.00773,258340:0.02819)0.473.11:0.01374,(320835:0.00638,196191:0.00728)0.976.118:0.00662)0.908.77:0.00393)0.609.8:0.0002)0.879.70:0.00165)0.754.28:0.00014,273706:0.01407)0.763.29:0.00135)0.844.32:0.00169,314542:0.04158)0.930.88:0.00657,(((230479:0.00573,(316911:0.01105,296941:0.00818)0.905.72:0.00367)0.952.104:0.0053,((365161:0.00768,269360:0.01493)0.830.36:0.00764,(288514:0.01843,100135:0.01367)1.000.2459:0.02763)0.987.213:0.0097)0.993.207:0.0103,(((((204126:0.0146,365533:0.0122)0.959.101:0.00635,(131039:0.01712,((158160:0.01428,(176876:0.00209,292685:0.00908)0.843.41:0.01208)0.950.115:0.0042,469744:0.0044)0.753.35:0.00643)0.852.65:0.0019)0.932.77:0.0038,((((180565:0.01457,((((((270303:0.01084,272068:0.01036)1.000.2460:0.00133,435679:0.00548)0.983.156:0.00519,263994:0.0145)0.516.10:0.00158,(((203742:0.00796,(304489:0.00706,360119:0.02344)0.959.102:0.00622)0.936.89:0.00333,(((288124:0.02194,327116:0.00402)0.326.13:0.00221,312719:0.00632)0.981.139:0.00738,(((334935:0.00929,262611:0.00091)1.000.2461:0.01596,549723:0.01182)0.864.72:0.0017,(290698:0.01717,288371:0.01518)0.175.10:0.01733)0.846.32:0.00206)0.951.89:0.00389)0.849.49:0.00185,(296949:0.02615,168716:0.0097)0.903.71:0.00416)0.961.111:0.00508)0.927.86:0.00375,((276273:0.0391,(((146578:0.01825,109481:0.01578)1.000.2462:0.01628,(497935:0.01553,257881:0.00634)0.500.11:0.00417)0.743.36:0.00074,(369389:0.02059,(554372:0.00772,(579562:0.01453,16072:0.00666)0.984.157:0.008)0.893.61:0.00394)0.961.112:0.00708)0.303.8:0.00664)0.901.65:0.00636,579244:0.01356)0.754.29:0.00425)0.950.116:0.00665,581933:0.00571)0.926.88:0.00535)0.316.11:0.00015,(((290601:0.00777,(296424:0.01202,(299718:0.00143,531339:0.01208)0.427.9:0.00227)0.792.23:0.00553)0.896.61:0.00718,(43950:0.00282,317870:0.01982)0.795.27:0.01028)0.634.12:0.0073,110968:0.01103)0.908.78:0.00728)0.958.120:0.00339,(183985:0.01986,(169479:0.00229,230725:0.0009)0.747.34:0.00358)0.712.16:0.00246)0.500.12:0.00181,(((292962:0.02003,193896:0.01576)0.967.123:0.00802,(240271:0.01407,302641:0.01616)0.894.64:0.00406)0.545.10:0.00302,((228787:0.01124,(184072:0.00755,178852:0.01291)0.439.15:0.00356)0.984.158:0.01131,(443620:0.00282,(((((318314:0.00362,(453710:0.00864,274083:0.01697)0.965.123:0.00391)0.866.50:0.00156,(276404:0.00938,(192930:0.01309,(264539:0.00788,268671:0.01524)0.935.99:0.00746)0.948.105:0.00857)0.923.79:0.00761)0.883.66:0.00159,(405713:0.00937,((231593:0.0033,273545:0.00745)0.861.63:0.00246,(172403:0.01007,309302:0.00577)0.553.11:0.00098)0.875.52:0.00319)0.595.5:0.00406)0.919.82:0.00075,231028:0.01838)0.929.95:0.01339,(278609:0.0121,459280:0.00376)0.776.36:0.00285)0.946.87:0.00413)0.984.159:0.01294)0.758.33:0.00232)0.973.125:0.00543)0.612.12:0.00377)0.718.16:0.00165,(((358001:0.01974,(346686:0.00014,181448:0.00349)0.933.83:0.00467)0.717.17:0.0057,((66188:0.01346,(184954:0.01821,(289463:0.0157,(295879:0.00348,308759:0.01155)0.305.5:0.00613)0.944.80:0.01031)0.842.52:0.00442)0.822.39:0.00407,420168:0.01935)0.937.89:0.00534)0.445.7:0.00194,347991:0.03799)0.951.90:0.00571)0.857.52:0.00402,((((95502:0.01342,278758:0.00752)0.959.103:0.00595,(264585:0.00944,(381694:0.01016,(411008:0.00432,(270531:0.01568,454252:0.00307)0.731.33:0.00412)0.724.17:0.0034)0.925.78:0.00414)0.423.7:0.00205)0.860.58:0.00217,(367176:0.02329,(253896:0.01431,((214380:0.00862,(278667:0.01215,422956:0.00948)0.849.50:0.00318)0.858.50:0.00168,(113111:0.01594,((321628:0.01362,543553:0.01231)0.755.41:0.00154,(134961:0.02228,(175377:0.0171,(((234600:0.00881,277228:0.0052)0.889.57:0.00333,375682:0.00407)0.949.102:0.00548,451081:0.00815)0.941.110:0.00442)0.767.46:0.00071)0.857.53:0.0023)0.956.98:0.00492)0.834.40:0.00247)0.988.196:0.0056)0.701.11:0.00291)0.844.33:0.00328)0.830.37:0.00152,(180235:0.0205,289426:0.03597)0.996.292:0.01485)0.833.34:0.00321)0.782.31:0.00537)0.962.122:0.00645)0.869.65:0.00458,((349798:0.01368,(258946:0.03181,264941:0.01364)0.314.7:0.00696)0.986.164:0.01873,270519:0.06005)0.966.117:0.01691)0.981.140:0.01181,276996:0.02386)0.986.165:0.01376,((((270498:0.04301,(170926:0.00977,129459:0.03129)0.234.6:0.00447)0.996.293:0.01219,((320435:0.00842,318370:0.01907)0.867.62:0.0037,(172136:0.01725,(((244692:0.00305,258899:0.01866)0.986.166:0.00742,(190646:0.02989,(416844:0.00931,168054:0.01814)0.591.9:0.00442)0.850.40:0.00314)0.942.98:0.00171,((355178:0.00746,134455:0.01572)0.848.46:0.01156,(((171864:0.01244,265813:0.02168)0.927.87:0.00121,((((267905:0.00664,(335952:0.00349,275543:0.01538)0.981.141:0.00832)0.975.141:0.01013,262672:0.00728)0.954.102:0.00014,344329:0.00907)0.826.25:0.00028,321019:0.02514)0.386.10:0.00039)0.927.88:0.00483,172209:0.00732)0.919.83:0.00234)0.940.106:0.00315)0.871.68:0.00259)0.899.69:0.00556)0.894.65:0.00466)0.836.36:0.00464,((336691:0.02504,196815:0.00585)0.996.294:0.01439,(232406:0.00401,((310321:0.00679,(263255:0.02661,269156:0.00955)0.957.138:0.00541)0.976.119:0.00429,313324:0.03675)0.000.80:0.00072)0.878.66:0.00245)0.974.126:0.00644)0.998.419:0.00923,(358904:0.01786,(288944:0.01565,295102:0.00645)0.908.79:0.00376)0.897.67:0.00357)0.853.48:0.00289)0.912.64:0.00431,(377756:0.01613,(130048:0.02688,213306:0.00625)0.506.6:0.00171)0.922.83:0.02858)0.994.231:0.00996,((((((175497:0.01935,367642:0.01188)0.746.37:0.00501,39555:0.00324)0.873.66:0.00253,296577:0.01265)0.934.68:0.00779,(193182:0.01639,((311798:0.00723,298457:0.0266)0.742.37:0.01622,(((321337:0.00836,548161:0.01664)0.964.133:0.00594,340177:0.01031)0.892.51:0.00396,298036:0.00706)0.972.133:0.01002)0.875.53:0.00883)0.796.31:0.0083)0.998.420:0.01623,((((323903:0.00945,(299755:0.01054,(352081:0.02054,306711:0.01598)0.992.211:0.01272)0.990.183:0.0094)0.878.67:0.00272,(((((287655:0.00817,331416:0.01268)0.845.35:0.00105,(298271:0.00829,((155341:0.004,344258:0.00831)0.213.8:0.00458,584417:0.00613)0.943.102:0.00357)0.879.71:0.00167)0.748.27:0.00321,(((316845:0.00648,((309456:0.01648,(300253:0.00535,(303900:0.01775,320768:0.02295)0.590.12:0.00441)0.942.99:0.00791)0.946.88:0.00014,554883:0.01186)0.788.27:0.00092)0.984.160:0.00795,(303137:0.01555,(343362:0.02695,(299732:0.01938,301124:0.01467)0.023.5:0.00014)0.174.6:0.00014)0.983.157:0.00013)0.838.39:0.00079,(((581554:0.00043,(366486:0.03502,197095:0.0036)0.513.9:0.0004)0.845.36:0.00484,266274:0.0244)0.968.116:0.00441,(336224:0.0155,342658:0.007)0.446.10:0.00234)0.990.184:0.00704)0.875.54:0.00161)0.867.63:0.00209,(298072:0.00559,299222:0.04314)0.988.197:0.01641)0.893.62:0.00209,287790:0.01842)0.795.28:0.0017)0.756.38:0.00084,326785:0.01742)0.859.55:0.00247,(339294:0.00954,303255:0.02555)0.776.37:0.00447)0.908.80:0.00643)0.468.13:0.00382,(((318490:0.00787,((((171303:0.02485,(((165734:0.02126,73606:0.01563)0.985.148:0.00881,548032:0.01398)0.820.21:0.00277,510070:0.01082)0.983.158:0.01011)0.928.78:0.00572,((((((149007:0.01716,((509045:0.01264,205531:0.0222)0.198.8:0.00165,147024:0.02522)0.942.100:0.00557)0.865.58:0.00909,526665:0.01409)0.759.22:0.007,(249174:0.00412,73132:0.01134)0.739.30:0.00177)0.929.96:0.0074,520720:0.01227)0.927.89:0.00507,(((209712:0.01254,(107235:0.01705,49276:0.00566)0.525.9:0.01639)0.989.167:0.01046,(529688:0.0101,(211374:0.00871,(((369909:0.00753,367411:0.00846)0.995.292:0.00043,314582:0.01736)0.913.80:0.00234,335097:0.01758)0.864.73:0.00662)0.999.639:0.01341)0.857.54:0.00486)0.984.161:0.00668,(((167509:0.01154,(306091:0.0101,(433552:0.01544,262677:0.01174)0.953.101:0.00329)0.844.34:0.00269)0.885.59:0.00258,(292576:0.02162,((((274584:0.00058,(261802:0.01692,134096:0.01464)0.927.90:0.00883)0.807.24:0.00503,((135369:0.00947,260741:0.0075)1.000.2463:0.02299,((273724:0.01295,276985:0.01773)0.987.214:0.00903,(278647:0.00354,(230534:0.00078,260333:0.00895)0.878.68:0.0032)0.907.73:0.00285)0.490.9:0.00074)0.465.7:0.00014)0.862.65:0.00233,303454:0.01773)0.932.78:0.00308,(397363:0.00981,((98048:0.01611,(338887:0.01933,185752:0.00863)0.475.7:0.01417)0.955.104:0.00529,407963:0.0037)0.981.142:0.00673)0.605.12:0.00015)0.988.198:0.00865)0.904.73:0.0038)0.953.102:0.00498,((349204:0.00377,300486:0.01652)0.916.73:0.00362,311718:0.00226)0.999.640:0.01252)0.926.89:0.00263)0.800.35:0.00323)0.871.69:0.00322,((36503:0.01385,526225:0.02094)0.946.89:0.005,((((135795:0.02494,316428:0.02719)0.959.104:0.00477,(263106:0.00371,172735:0.00343)0.941.111:0.00435)0.892.52:0.00451,(((278649:0.00399,134301:0.01227)0.998.421:0.00015,258995:0.03164)0.881.69:0.00156,(304013:0.03009,(259535:0.00851,338796:0.03758)0.481.6:0.00059)0.939.102:0.01044)0.772.33:0.00096)0.754.30:0.00094,539145:0.01433)0.677.11:0.0032)0.749.32:0.00103)0.672.7:0.00635)0.837.31:0.00224,112837:0.02313)0.734.27:0.00094,(325211:0.02515,(544996:0.00289,183410:0.02366)0.834.41:0.00554)0.395.12:0.00207)0.909.72:0.0038)0.927.91:0.00634,549950:0.02634)0.702.15:0.00409,((((193632:0.01519,313471:0.03251)1.000.2464:0.04711,301943:0.03467)0.757.41:0.01798,203545:0.03078)0.941.112:0.01955,(320483:0.02977,276511:0.03751)1.000.2465:0.04075)0.829.31:0.00456)0.884.63:0.00479)0.866.51:0.00258)0.988.199:0.00872)0.770.28:0.00503)0.989.168:0.01064,(((50712:0.00694,569306:0.00267)0.931.81:0.00561,(193310:0.06108,((364600:0.02232,302974:0.02847)0.945.87:0.00879,(298079:0.01255,296377:0.03322)0.970.114:0.00752)0.338.6:0.00281)0.910.68:0.00763)0.970.115:0.00877,(((((209475:0.00584,353387:0.02002)0.998.422:0.01254,(164242:0.0026,266715:0.00387)0.970.116:0.00948)0.172.8:0.00504,549281:0.02417)1.000.2466:0.01961,(358150:0.02417,215231:0.0096)0.999.641:0.0207)0.586.9:0.00713,(((288227:0.01314,(324850:0.01578,300628:0.00808)0.967.124:0.00964)1.000.2467:0.0187,(302377:0.00995,((260744:0.01775,(347037:0.00044,262583:0.00666)0.947.93:0.00518)1.000.2468:0.02119,583134:0.01392)0.771.34:0.00191)0.976.120:0.02617)0.966.118:0.00702,((((((((326879:0.00487,299860:0.01931)0.980.140:0.01353,(295004:0.01216,(322956:0.00454,(297278:0.01375,344815:0.01534)0.246.5:0.00596)0.890.72:0.00303)0.863.58:0.00168)0.987.215:0.00404,((38344:0.01679,((352918:0.02557,268937:0.00904)0.764.29:0.00321,530980:0.00957)0.269.9:0.00012)0.930.89:0.00623,(298494:0.02034,278879:0.00866)0.942.101:0.00587)0.778.39:0.00493)0.861.64:0.00015,292551:0.01474)0.889.58:0.00208,(((((353409:0.01013,((293551:0.01032,(331659:0.00629,385780:0.00344)0.898.73:0.00168)0.785.35:0.00111,((294596:0.01451,301354:0.01562)0.422.10:0.00114,291171:0.01294)0.725.22:0.00366)0.716.9:0.00216)0.878.69:0.0029,(319506:0.01622,524900:0.01296)0.866.52:0.00383)0.986.167:0.00015,(((297798:0.00701,347098:0.01233)0.928.79:0.00771,334216:0.01323)0.735.28:0.0041,(291888:0.02231,(288175:0.01555,288547:0.0107)0.370.8:0.00273)0.907.74:0.00491)0.892.53:0.00413)0.887.56:0.00159,((((518314:0.01805,((294448:0.01706,(343498:0.03097,((332805:0.01735,(314421:0.013,((294078:0.02108,288214:0.02259)0.726.20:0.00741,313908:0.00992)0.972.134:0.01221)0.932.79:0.00628)0.080.7:0.00129,288356:0.02331)0.577.4:0.00579)0.754.31:0.0013)0.712.17:0.00085,(304622:0.02575,(336638:0.01171,293631:0.01497)0.873.67:0.00589)0.930.90:0.00479)0.694.11:0.00289)0.874.67:0.00175,296276:0.00928)0.878.70:0.00158,(522027:0.01176,518444:0.01921)0.645.18:0.00359)1.000.2469:0.00014,((332204:0.0048,((323963:0.014,294377:0.01373)0.760.34:0.00088,(((336907:0.01173,293169:0.00542)0.920.67:0.00272,(288405:0.0095,338789:0.01901)0.942.102:0.00407)0.724.18:0.00677,(334705:0.00654,301772:0.02482)0.044.7:0.00568)0.936.90:0.00714)0.877.64:0.00161)0.866.53:0.00048,298302:0.01307)0.433.7:0.00213)0.801.22:0.00082)0.795.29:0.0008,(((298004:0.01094,315892:0.00888)0.262.6:0.00343,(297237:0.01762,291871:0.01122)0.941.113:0.0051)0.754.32:0.00116,(((((410610:0.00346,414135:0.00735)0.992.212:0.00787,295631:0.02403)0.767.47:0.00091,(300979:0.04589,290284:0.0109)0.340.9:0.00447)1.000.2470:0.00015,(521792:0.0299,532491:0.01499)0.886.55:0.00431)0.741.27:0.00391,(422023:0.00775,295184:0.00937)0.975.142:0.00849)0.883.67:0.00251)0.975.143:0.0044)0.882.62:0.00015)0.332.8:0.00158,(((((309070:0.02325,355912:0.01368)0.842.53:0.00184,(((311684:0.01117,292548:0.02035)0.925.79:0.00567,(287430:0.01582,(295144:0.02674,(352092:0.03425,(287666:0.02464,353620:0.00792)0.488.11:0.00474)0.686.10:0.0075)0.885.60:0.00556)0.609.9:0.00622)0.988.200:0.00897,(306892:0.02037,(292354:0.01752,((302694:0.00793,310416:0.03152)0.494.11:0.0032,((294437:0.02031,302511:0.01382)0.913.81:0.00816,(41628:0.01243,(290201:0.01618,344111:0.01772)0.949.103:0.00923)0.959.105:0.0087)0.945.88:0.00721)0.680.11:0.00366)0.860.59:0.00692)0.950.117:0.00756)0.777.28:0.00148)0.870.67:0.00308,(292088:0.01276,323206:0.01543)0.923.80:0.00388)0.803.28:0.00241,(((346413:0.02901,(290974:0.03458,296798:0.01266)0.849.51:0.00955)1.000.2471:0.0403,((((190124:0.02003,(311308:0.02259,289055:0.02252)0.401.7:0.00446)0.819.29:0.00212,((347159:0.00642,301716:0.028)0.930.91:0.00571,((299908:0.02599,315683:0.02096)0.631.8:0.00978,(295472:0.01759,((353524:0.01056,332292:0.02094)0.978.133:0.01073,581463:0.01183)0.627.8:0.00565)0.948.106:0.01419)0.959.106:0.0133)0.999.642:0.01338)0.426.7:0.0007,((329360:0.02403,(291161:0.02714,290636:0.01509)0.989.169:0.01354)0.968.117:0.01169,(301247:0.02382,321963:0.00491)0.993.208:0.01934)1.000.2472:0.02615)0.869.66:0.00317,295095:0.01709)0.745.33:0.00197)0.969.125:0.00954,(293830:0.03445,((290956:0.00523,317218:0.00997)0.999.643:0.01293,(303301:0.02829,306212:0.02791)0.851.54:0.00574)0.956.99:0.0083)0.878.71:0.0047)0.879.72:0.00378)0.798.27:0.00256,((287585:0.02651,(((((312833:0.01491,(((299640:0.0075,(313702:0.00273,527760:0.00551)0.406.9:0.00377)0.774.22:0.00153,((327731:0.00374,(319693:0.02772,319996:0.011)0.800.36:0.00089)0.913.82:0.00016,(515362:0.02286,(525143:0.02303,292171:0.00452)0.678.14:0.00097)0.964.134:0.00396)0.533.5:0.00023)0.934.69:0.00321,529083:0.02763)0.769.32:0.00117)0.924.74:0.00397,(351178:0.00397,523825:0.01663)0.888.53:0.00195)0.878.72:0.00225,((((308324:0.00014,303242:0.01594)0.744.33:0.00317,(293997:0.01269,303173:0.00858)0.400.10:0.00629)0.772.34:0.00081,290685:0.00421)0.946.90:0.00014,288343:0.01913)0.890.73:0.00157)0.871.70:0.00252,((523506:0.00748,528519:0.00995)0.751.25:0.00112,310510:0.00988)0.866.54:0.00156)0.793.25:0.00078,(((356550:0.01755,(291446:0.00852,528754:0.00647)0.865.59:0.0027)0.976.121:0.00711,523355:0.01184)0.861.65:0.00537,((323702:0.01401,299201:0.02954)0.752.36:0.00394,(((328869:0.00472,299275:0.01024)0.907.75:0.00164,(516395:0.01409,290436:0.01238)0.741.28:0.00096)0.884.64:0.00334,((304720:0.01503,307643:0.01597)0.909.73:0.00262,((289065:0.00276,(296962:0.01558,331081:0.00685)0.851.55:0.00167)0.913.83:0.00858,(289917:0.01429,530505:0.00954)0.924.75:0.00303)0.849.52:0.00158)0.843.42:0.00322)0.672.8:0.00014)0.946.91:0.00237)0.485.7:0.00015)0.397.17:0.00066)0.738.26:0.00048,((300417:0.02178,(((290378:0.02505,511180:0.01562)0.781.27:0.006,((297813:0.02942,(329808:0.02255,305394:0.0303)0.807.25:0.009)0.447.8:0.00427,301405:0.03248)0.298.3:0.00175)0.956.100:0.00538,(((349884:0.03819,289882:0.02046)1.000.2473:0.04776,354741:0.00963)0.999.644:0.02156,(312152:0.02335,292061:0.01938)0.971.150:0.01068)0.996.295:0.01257)0.931.82:0.00601)0.188.8:0.00306,((326991:0.02772,(353173:0.02949,302759:0.03956)0.108.4:0.00754)0.961.113:0.00954,(((302257:0.00877,299299:0.0229)0.938.97:0.00621,311163:0.0274)0.798.28:0.00318,299982:0.0135)0.819.30:0.00415)0.974.127:0.00978)0.932.80:0.00616)0.929.97:0.00254)0.761.40:0.00016)0.994.232:0.00961,((((348949:0.04016,316414:0.01231)0.415.8:0.00765,((292688:0.01222,302061:0.02609)0.823.26:0.0053,16559:0.02236)0.914.68:0.00686)0.809.26:0.00335,((((289465:0.01224,302308:0.00993)0.887.57:0.00015,314937:0.01179)1.000.2474:0.00016,(530805:0.00894,(534850:0.01169,461078:0.01265)0.924.76:0.00332)0.985.149:0.00487)0.899.70:0.00234,((521253:0.00943,(25695:0.01591,157652:0.00815)0.583.16:0.00493)0.702.16:0.00179,(296690:0.01723,((269532:0.01795,(431235:0.01389,((((311997:0.01862,301410:0.02025)0.882.63:0.005,((190914:0.0213,(302037:0.01594,299875:0.02055)0.782.32:0.00959)0.983.159:0.01409,((((304825:0.00584,(194735:0.00647,(294604:0.00901,172878:0.00887)0.856.45:0.00158)0.992.213:0.0074)0.757.42:0.00095,(347085:0.0219,(299081:0.01134,(216747:0.01788,362930:0.0091)0.909.74:0.00442)0.689.11:0.00308)0.942.103:0.00642)0.976.122:0.00506,((292816:0.01211,180551:0.00951)0.982.175:0.00639,(295375:0.02113,(364627:0.00688,(154467:0.01858,299135:0.01092)0.856.46:0.00193)0.747.35:0.00105)0.843.43:0.00325)0.941.114:0.00451)0.787.30:0.00187,((586064:0.01303,(292569:0.01512,(262570:0.00752,(((45641:0.0023,272613:0.01575)0.801.23:0.00015,260436:0.00922)0.777.29:0.00119,293718:0.01541)0.957.139:0.00556)0.965.124:0.00553)0.855.47:0.00154)0.982.176:0.00711,(323737:0.0241,((293285:0.00666,287838:0.00356)0.791.35:0.00446,(330247:0.00111,111019:0.00583)0.920.68:0.00281)0.007.4:0.00273)0.984.162:0.00869)0.943.103:0.00365)0.951.91:0.00475)0.756.39:0.00378)0.564.14:0.00159,312796:0.02222)0.994.233:0.00869,(302467:0.00799,(287941:0.01605,295890:0.02489)0.930.92:0.00015)0.984.163:0.00781)0.866.55:0.00346)0.453.8:0.002)0.555.10:0.00255,358596:0.00501)0.993.209:0.00014)0.987.216:0.00716)0.859.56:0.00169)0.321.7:0.00213)0.800.37:0.00141,((((292152:0.00902,(518873:0.00446,526794:0.01679)0.890.74:0.00192)1.000.2475:0.00014,(101572:0.0152,((301722:0.01562,287719:0.00666)0.731.34:0.0008,294295:0.01309)0.759.23:0.00267)0.889.59:0.00159)0.903.72:0.00156,(((297020:0.01449,(45365:0.01033,288007:0.00152)0.956.101:0.00627)0.347.8:0.00518,(312861:0.00407,303172:0.00749)0.846.33:0.00157)0.949.104:0.00324,(556462:0.01257,(348828:0.00835,(352908:0.01194,330360:0.01014)0.110.4:0.0034)0.758.34:0.00358)0.728.25:0.0008)0.796.32:0.00079)0.322.5:0.00015,289958:0.02206)0.939.103:0.00237)0.714.16:0.00048)0.793.26:0.00302,((354054:0.0082,349695:0.03683)0.999.645:0.03047,(338991:0.0262,294744:0.0229)0.900.67:0.01092)0.350.11:0.00717)0.988.201:0.01169)0.744.34:0.00221)0.943.104:0.0068)0.967.125:0.00725)0.821.24:0.0077)0.917.79:0.0038,(((230370:0.02056,(315439:0.02786,580305:0.02909)0.993.210:0.01324)0.755.42:0.00463,(314186:0.01161,296065:0.00773)0.998.423:0.0142)0.713.10:0.00085,(((161109:0.02185,16031:0.0205)0.974.128:0.00973,148791:0.02011)0.679.12:0.00095,((((145883:0.00873,(((296860:0.03938,(298224:0.0102,561716:0.016)0.996.296:0.01493)0.952.105:0.00763,44151:0.01404)0.133.7:0.00395,215468:0.01581)0.986.168:0.00999)0.835.34:0.00196,(295748:0.01109,155593:0.00388)0.997.324:0.00891)0.801.24:0.00158,((543168:0.01376,(298751:0.01662,(107073:0.02013,555041:0.0021)0.852.66:0.00585)0.968.118:0.00762)0.993.211:0.01329,((((295245:0.0302,293727:0.01128)0.865.60:0.00433,((350731:0.01274,(48325:0.01195,(300186:0.02476,((268828:0.00092,582101:0.00544)1.000.2476:0.0348,(352374:0.00927,322194:0.0195)0.907.76:0.00345)0.912.65:0.00519)0.911.75:0.00543)0.952.106:0.00744)0.985.150:0.0075,((310518:0.00834,339035:0.01892)0.681.18:0.0037,(298616:0.01586,368236:0.00786)0.927.92:0.00659)0.694.12:0.00125)0.905.73:0.00345)0.892.54:0.00369,((572461:0.01533,342940:0.01726)0.908.81:0.00661,166466:0.02297)0.906.78:0.00432)0.771.35:0.00238,((272210:0.00581,(16026:0.01934,311629:0.00287)0.897.68:0.00786)0.968.119:0.00748,((255350:0.02855,(135453:0.01139,147964:0.02759)0.120.4:0.01018)0.952.107:0.00988,(143049:0.03648,134357:0.01239)0.168.17:0.00613)0.865.61:0.00431)0.911.76:0.00528)0.929.98:0.00528)0.867.64:0.00358)0.983.160:0.00515,((113382:0.0229,((224826:0.02592,224436:0.01408)0.816.34:0.00265,(143918:0.00762,(144692:0.00862,144056:0.01534)0.851.56:0.00636)0.410.8:0.00224)0.816.35:0.00638)0.867.65:0.00701,113007:0.01882)0.958.121:0.00594)0.903.73:0.00288)0.994.234:0.0103)0.867.66:0.00314)0.905.74:0.00504)0.883.68:0.00623)0.955.105:0.00656,((294594:0.04184,((167685:0.00026,189527:0.01903)0.978.134:0.00784,205241:0.00559)1.000.2477:0.02762)0.884.65:0.01091,(((446691:0.00795,(590547:0.00468,356242:0.01744)0.943.105:0.00731)0.999.646:0.01738,287931:0.03116)0.956.102:0.01138,((369416:0.01773,(346847:0.01947,((293546:0.03199,183479:0.01638)1.000.2478:0.02232,296883:0.0225)0.713.11:0.00162)0.976.123:0.00926)0.981.143:0.00942,(300074:0.02478,384386:0.01069)0.991.211:0.01293)0.750.31:0.00351)0.975.144:0.01106)0.957.140:0.0098)0.947.94:0.00838)0.673.11:0.00632)0.812.38:0.00736,(347758:0.0426,(((311952:0.08751,351200:0.04745)0.885.61:0.01344,(((((((355476:0.03584,((202709:0.05026,(304097:0.02299,288325:0.02646)0.990.185:0.02084)0.949.105:0.01566,((134877:0.03262,320373:0.02991)0.986.169:0.0195,(308224:0.02006,(313898:0.03635,322112:0.01577)0.805.28:0.02075)0.676.13:0.01218)0.879.73:0.01915)0.970.117:0.01543)0.921.79:0.01061,(((180864:0.02908,350154:0.03605)0.990.186:0.02116,((175959:0.02383,(149313:0.00451,146046:0.00182)0.000.81:0.02038)0.999.647:0.02258,(289087:0.06558,297435:0.01831)0.996.297:0.01866)0.887.58:0.00821)0.808.25:0.00493,((((((303835:0.00562,(355197:0.03289,(341349:0.04084,294565:0.03594)0.985.151:0.01601)0.886.56:0.00543)0.994.235:0.01482,(354161:0.02779,188307:0.01797)0.999.648:0.01744)0.976.124:0.01215,296267:0.05019)0.568.11:0.00715,(194648:0.02623,(318760:0.02577,((299302:0.02475,174078:0.00507)0.971.151:0.00762,(183866:0.0135,366333:0.0381)0.613.9:0.00372)0.988.202:0.01516)0.965.125:0.01274)0.753.36:0.01067)0.536.10:0.00481,(((199308:0.01713,(295116:0.06477,183368:0.0197)0.862.66:0.00897)0.889.60:0.01452,(182568:0.00564,191796:0.02109)0.996.298:0.01451)0.927.93:0.00852,((195677:0.03993,(184514:0.00491,359255:0.01616)1.000.2479:0.03405)0.875.55:0.01345,(184983:0.01238,(179057:0.00819,557450:0.02204)0.841.43:0.02338)0.877.65:0.00419)0.953.103:0.00914)0.267.10:0.00162)0.619.10:0.00528,296257:0.0438)0.991.212:0.01624)0.955.106:0.00805)0.938.98:0.00798,(((342688:0.05835,(312175:0.09795,289424:0.11019)0.989.170:0.05276)1.000.2480:0.04746,(267628:0.04278,305086:0.10362)0.936.91:0.01715)1.000.2481:0.03937,(((352604:0.07121,((14010:0.03509,34466:0.01998)1.000.2482:0.04623,316433:0.05066)0.951.92:0.01265)0.697.14:0.00903,((((((((((261111:0.04071,(((99173:0.00651,(293833:0.02862,291890:0.00295)0.943.106:0.00842)0.983.161:0.01101,586700:0.02536)0.834.42:0.0047,(328438:0.00015,563315:0.00165)0.994.236:0.01281)0.998.424:0.01356)0.251.6:0.00093,((291073:0.00622,(518135:0.02161,(531052:0.01159,((225599:0.01619,177788:0.00995)0.839.54:0.00159,((187114:0.02361,137353:0.00716)0.842.54:0.00014,114970:0.00417)0.364.9:0.00082)0.939.104:0.00361)0.857.55:0.00327)0.231.6:0.00164)0.960.125:0.00635,(358834:0.03616,(178970:0.01354,518460:0.00015)0.755.43:0.00586)0.996.299:0.01442)0.941.115:0.00425)0.995.293:0.01004,(352455:0.04341,(((277489:0.00442,((524122:0.00416,369502:0.01115)0.865.62:0.00052,199013:0.00624)0.922.84:0.00394)0.988.203:0.01096,(557137:0.04435,(323463:0.02029,(260496:0.01999,(565563:0.01391,261381:0.03057)0.672.9:0.00675)0.682.11:0.00744)0.927.94:0.01505)0.982.177:0.01458)0.180.7:0.0027,(299398:0.02617,261634:0.0249)0.330.7:0.00802)0.997.325:0.02034)0.945.89:0.00893)0.808.26:0.00403,(((((231128:0.00484,((123419:0.01732,(289402:0.00917,578857:0.00387)0.995.294:0.012)0.095.4:0.00344,319153:0.00837)0.869.67:0.00474)0.459.5:0.00354,(178682:0.01532,184905:0.0103)0.982.178:0.01021)0.882.64:0.00754,((328453:0.03326,(((386087:0.01065,(397065:0.00873,275964:0.01938)0.926.90:0.00393)0.755.44:0.00391,172622:0.00371)0.336.16:0.00159,230936:0.02121)0.866.56:0.00251)0.826.26:0.00188,((((255828:0.00015,310132:0.01912)0.897.69:0.00227,309206:0.02994)0.990.187:0.00395,134206:0.01323)0.850.41:0.00078,(266539:0.00384,271097:0.01659)1.000.2483:0.00014)1.000.2484:0.00282)0.913.84:0.00642)0.758.35:0.00496,(382721:0.00493,199137:0.00546)0.997.326:0.01502)0.914.69:0.00449,(192659:0.02524,((182621:0.01207,265576:0.00479)0.985.152:0.01815,(274455:0.0192,263074:0.01715)0.213.9:0.01013)0.986.170:0.02046)0.934.70:0.00719)0.985.153:0.01116)0.983.162:0.00766,((((260346:0.02506,178106:0.03837)0.999.649:0.03094,(((((((((((343380:0.01631,(325586:0.00536,183767:0.00966)0.949.106:0.00606)0.069:0.00187,((174911:0.01552,((290467:0.00848,303866:0.02057)0.755.45:0.00093,301579:0.0021)0.883.69:0.00732)0.963.108:0.00863,321589:0.03084)0.980.141:0.01111)0.825.27:0.00438,((360925:0.01341,471182:0.02636)0.910.69:0.00529,((300581:0.01684,(297440:0.01336,(311521:0.02626,(300036:0.00724,((36624:0.0052,298171:0.0041)0.998.425:0.01122,(288210:0.0076,((136370:0.01631,333132:0.00936)0.833.35:0.00821,297819:0.01871)0.903.74:0.00543)0.813.30:0.00382)0.926.91:0.00442)0.870.68:0.00355)0.945.90:0.00635)0.822.40:0.00235)0.940.107:0.00516,455746:0.00688)0.920.69:0.00488)0.902.58:0.00468)0.939.105:0.00638,((337551:0.02112,(290079:0.02352,(296062:0.02103,320009:0.01369)0.994.237:0.01933)0.123.4:0.00642)0.954.103:0.00929,(((((377385:0.0082,161899:0.01571)0.986.171:0.00727,(585138:0.0134,359734:0.00562)0.786.32:0.00122)0.941.116:0.00921,(178295:0.01036,((198450:0.01607,198536:0.00672)0.948.107:0.01923,(((207390:0.02228,(298632:0.02478,182428:0.03178)0.936.92:0.00853)0.918.86:0.00504,((190954:0.01328,(293232:0.01772,((301110:0.01976,(291759:0.0225,300725:0.01752)0.947.95:0.01059)0.898.74:0.00651,293835:0.01423)0.855.48:0.00507)0.836.37:0.00575)0.927.95:0.00559,290502:0.02446)0.345.9:0.00455)0.782.33:0.00635,178609:0.00764)0.272.5:0.00361)0.846.34:0.00758)0.759.24:0.00849)0.730.24:0.0067,((522727:0.01927,16853:0.00184)0.819.31:0.01327,(((350041:0.00822,195692:0.02413)0.701.12:0.00627,(333807:0.01553,(298031:0.01614,294370:0.01349)0.590.13:0.00346)0.736.36:0.00261)0.971.152:0.01124,(165318:0.0226,((((525416:0.01343,(294652:0.00818,(291730:0.0175,300058:0.01086)0.733.28:0.00545)0.971.153:0.01692)0.032.2:0.00015,(350240:0.01922,288160:0.00724)0.381.7:0.00273)0.856.47:0.00236,(520439:0.0148,300967:0.02415)0.888.54:0.00267)0.996.300:0.00892,(299066:0.00559,344713:0.02033)0.862.67:0.00172)0.739.31:0.00354)0.868.72:0.00288)0.836.38:0.00243)0.993.212:0.00911)0.880.66:0.00558,((302429:0.01503,((162110:0.00487,270014:0.00661)1.000.2485:0.0214,(268183:0.00897,330352:0.01701)0.388.11:0.00345)0.407.9:0.00473)0.850.42:0.00528,(236097:0.02319,350732:0.02025)0.419.7:0.00559)0.976.125:0.01064)0.522.8:0.00542)0.975.145:0.00808)0.980.142:0.00782,(((305726:0.01347,(261203:0.01813,(353775:0.00091,273600:0.00985)0.913.85:0.01161)0.996.301:0.01218)0.926.92:0.0068,(((274828:0.01702,(260855:0.02034,(274536:0.01488,(577323:0.02731,592305:0.01128)0.086.5:0.0098)0.701.13:0.00713)0.485.8:0.00662)0.625.12:0.00631,341449:0.01709)0.966.119:0.00996,(193230:0.01528,(195449:0.01923,36611:0.00478)0.400.11:0.00296)0.800.38:0.00532)0.937.90:0.00704)0.952.108:0.00922,(180638:0.01948,(293330:0.01836,353964:0.03356)0.593.7:0.00358)0.957.141:0.00916)0.861.66:0.00483)0.886.57:0.00258,((((302693:0.02265,(341089:0.01375,(289745:0.00448,291535:0.02312)0.774.23:0.01614)0.946.92:0.0083)0.991.213:0.01434,(195413:0.01854,(177986:0.02179,(182122:0.01527,(192396:0.01493,352852:0.01303)0.618.8:0.00556)0.652.13:0.00606)0.548.14:0.00364)0.959.107:0.00853)0.159.6:0.00287,(190961:0.01437,(290696:0.00937,(((193517:0.00641,178994:0.0172)0.937.91:0.00801,186717:0.0066)0.972.135:0.00892,(185589:0.01286,287978:0.00895)0.645.19:0.00281)0.685.12:0.00494)0.909.75:0.00475)0.983.163:0.00852)0.379.16:0.00342,((134789:0.02756,(324696:0.00967,(103407:0.00624,(302900:0.01558,(321822:0.00446,325236:0.00535)0.766.30:0.00088)0.957.142:0.00535)0.816.36:0.01581)0.468.14:0.00602)0.999.650:0.02382,((158566:0.0085,(130773:0.00484,189639:0.00329)0.974.129:0.00633)0.981.144:0.00893,(592901:0.00573,336539:0.03734)0.862.68:0.01457)0.999.651:0.02158)0.909.76:0.00495)0.858.51:0.00202)0.883.70:0.00245,(((185090:0.04288,292951:0.02486)0.989.171:0.02196,(((353827:0.03184,(189827:0.02734,(77514:0.01938,(230741:0.00234,193704:0.00016)0.594.8:0.00938)0.796.33:0.0099)0.721.17:0.0033)0.945.91:0.00639,316452:0.02583)0.752.37:0.00396,(179807:0.01398,337465:0.0369)0.938.99:0.0085)0.321.8:0.00387)1.000.2486:0.02278,((288379:0.04335,(317127:0.04484,317892:0.01049)0.991.214:0.02092)0.528.11:0.01161,((317226:0.08696,((300250:0.03002,294199:0.03606)0.974.130:0.01311,(340111:0.02554,(306413:0.02987,508984:0.02597)0.907.77:0.00797)0.867.67:0.00701)0.344.7:0.00195)0.921.80:0.01043,329526:0.0328)0.927.96:0.01218)0.912.66:0.01008)0.884.66:0.00409)0.989.172:0.00738,((578960:0.02372,(302494:0.02443,(28175:0.00014,590175:0.00015)1.000.2487:0.03149)0.803.29:0.00498)0.956.103:0.00884,((195509:0.02577,365124:0.00274)0.784.30:0.01579,((130289:0.01554,196105:0.02125)0.948.108:0.01268,(((369182:0.00858,((367512:0.00212,364859:0.00908)0.187.11:0.00096,525329:0.0196)0.920.70:0.0043)0.998.426:0.01127,((299639:0.00805,(508767:0.01077,527709:0.00097)0.802.31:0.0027)0.950.118:0.00578,307950:0.01989)0.673.12:0.00352)0.819.32:0.00224,(311692:0.00914,(351880:0.01649,204072:0.00973)0.993.213:0.01067)0.868.73:0.00581)0.383.4:0.00384)0.912.67:0.00674)0.892.55:0.00626)0.843.44:0.00286)0.827.29:0.0015,((((97679:0.00767,(369658:0.00806,453756:0.00349)1.000.2488:0.00489)0.996.302:0.01264,((36330:0.014,290922:0.00878)0.974.131:0.00767,(317041:0.00735,328357:0.02758)0.890.75:0.00337)0.895.68:0.00423)0.727.20:0.00217,((289113:0.0048,(329662:0.01245,290852:0.01905)0.558.9:0.00586)0.990.188:0.01125,514797:0.01701)0.992.214:0.0105)0.973.126:0.00747,(((293389:0.02178,(303897:0.0297,(310490:0.00335,198143:0.02884)0.397.18:0.00869)0.697.15:0.00635)0.757.43:0.00582,(((194041:0.00511,(365763:0.02376,(360469:0.01703,(360421:0.01026,191316:0.02376)0.421.9:0.00059)0.863.59:0.00233)0.239.8:0.01216)0.914.70:0.00398,((((361052:0.01218,(175932:0.01215,571472:0.02783)0.399.8:0.00016)0.910.70:0.00416,(((535440:0.00787,192025:0.01627)0.464.6:0.00126,287919:0.01121)0.671.13:0.00181,290362:0.0164)0.121.3:0.00707)0.820.22:0.00092,(((((369814:0.02262,(589313:0.00793,576975:0.01131)0.877.66:0.00527)0.999.652:0.01124,337246:0.00572)0.916.74:0.00458,(231704:0.00601,(158939:0.02951,158969:0.00256)1.000.2489:0.01492)0.526.8:0.00488)0.893.63:0.00278,((((((299762:0.02443,318444:0.0276)0.072.4:0.00329,(291815:0.0108,311712:0.01872)0.345.10:0.0015)0.971.154:0.01021,345312:0.01079)0.886.58:0.00489,183487:0.01014)0.929.99:0.00342,187778:0.01084)0.961.114:0.00364,(208029:0.00986,295456:0.03139)0.862.69:0.00335)0.850.43:0.00181)0.838.40:0.00602,(301090:0.01839,(296363:0.01083,(302675:0.00885,300831:0.01702)0.860.60:0.00353)0.942.104:0.00635)0.927.97:0.00806)0.889.61:0.00332)0.971.155:0.00489,(287495:0.01101,183955:0.01076)0.394.8:0.01157)0.612.13:0.00087)0.946.93:0.00673,304393:0.02361)0.928.80:0.00609)0.558.10:0.00488,((130214:0.03756,(((196862:0.02286,(((194474:0.00642,(194896:0.01088,183267:0.00676)0.996.303:0.0191)0.739.32:0.00916,335518:0.02157)0.758.36:0.01313,363897:0.00284)0.999.653:0.00071)0.136.4:0.00126,(360553:0.02175,564317:0.00356)1.000.2490:0.00842)0.823.27:0.01558,((((((179597:0.00281,199487:0.023)0.309.9:0.00676,187591:0.01474)0.984.164:0.00825,(186565:0.00937,198325:0.00835)0.754.33:0.00117)0.844.35:0.00315,190191:0.01684)0.711.13:0.0004,((((287539:0.00852,340251:0.0161)0.941.117:0.00566,(((((190108:0.01618,(22923:0.01213,328717:0.01166)0.868.74:0.00421)0.598.12:0.00203,((13992:0.01103,184849:0.00539)0.481.7:0.00197,(((((360742:0.00551,193211:0.03085)0.167.4:0.0003,181678:0.00834)0.659.13:0.00125,332741:0.01032)0.000.82:0.00494,294606:0.01673)0.909.77:0.00169,(532431:0.00763,359203:0.02288)0.333.6:0.00514)0.952.109:0.00346)0.746.38:0.00696)0.638.7:0.00934,347115:0.01208)0.951.93:0.0061,((((181927:0.01524,545419:0.00355)0.901.66:0.00827,(572109:0.0144,349520:0.00942)0.827.30:0.00374)0.813.31:0.00318,(((436032:0.00655,(544532:0.01339,583221:0.01044)0.994.238:0.01155)0.826.27:0.00528,((((182308:0.03053,((175988:0.00463,179436:0.01987)0.880.67:0.0035,(188945:0.0154,366727:0.00479)0.677.12:0.00415)0.901.67:0.00296)0.942.105:0.00558,181226:0.02206)0.230.6:0.00168,193210:0.01839)0.904.74:0.00479,(((174296:0.00538,(353502:0.00585,303452:0.0222)0.866.57:0.00355)0.992.215:0.00858,(198951:0.01874,179715:0.00328)0.282.5:0.00175)0.915.71:0.00488,((295586:0.02381,175586:0.02285)1.000.2491:0.00014,(356995:0.00337,356869:0.01735)0.786.33:0.00619)0.293.5:0.00569)0.840.33:0.00197)0.711.14:0.00157)0.861.67:0.0067,(((184238:0.00757,(323778:0.0128,174625:0.02379)0.943.107:0.0066)0.980.143:0.01072,(((((178122:0.00262,(318504:0.0177,(340354:0.01933,335503:0.01158)0.123.5:0.0081)0.978.135:0.01059)0.995.295:0.01352,(296298:0.01029,194439:0.01842)0.966.120:0.00694)0.477.10:0.00836,361345:0.00722)0.771.36:0.0047,362428:0.0318)0.274.8:0.00226,((77371:0.0285,580349:0.00323)0.832.29:0.0056,(526975:0.00447,292228:0.02191)0.856.48:0.00613)0.952.110:0.00316)0.922.85:0.0026)0.446.11:0.00134,(177391:0.00681,((192899:0.00311,194155:0.01172)0.957.143:0.00421,295910:0.00719)0.715.15:0.00588)0.786.34:0.0136)0.984.165:0.0096)0.918.87:0.00439)0.783.34:0.00469,((291179:0.01852,345215:0.01531)0.218.7:0.00909,((303169:0.01427,404292:0.02422)0.879.74:0.00355,296692:0.01767)0.676.14:0.00674)0.716.10:0.00897)0.581.9:0.00353)0.836.39:0.00212,(299455:0.01821,177349:0.01693)0.798.29:0.00438)0.858.52:0.00297)0.953.104:0.00541,((((147991:0.02273,(((512682:0.02065,(134268:0.02394,469888:0.00771)0.303.9:0.02387)0.564.15:0.00479,302244:0.01771)0.474.5:0.00321,341433:0.01283)0.905.75:0.00692)0.874.68:0.00764,(351299:0.02408,(((172308:0.01868,((164906:0.00569,267626:0.00896)0.999.654:0.02131,(290501:0.00989,(111427:0.0261,310999:0.01093)0.102.6:0.0036)0.952.111:0.00846)0.908.82:0.00757)0.858.53:0.00613,102704:0.01935)0.899.71:0.00616,((554160:0.03216,(264113:0.0105,589215:0.01362)0.508.10:0.0019)0.462.7:0.00435,(535690:0.00735,258461:0.00446)0.997.327:0.01257)0.815.24:0.00488)0.819.33:0.00964)0.969.126:0.00915)0.673.13:0.00221,99566:0.02264)0.923.81:0.00504,(((184926:0.02006,((296007:0.01787,(196932:0.01841,(198626:0.02704,176070:0.00758)0.836.40:0.00917)0.653.5:0.00509)0.900.68:0.00432,(193868:0.03841,177005:0.0124)0.295.7:0.00144)0.874.69:0.00959)0.836.41:0.0029,(((((178449:0.0119,(185845:0.01519,(309092:0.00971,188588:0.00721)0.768.41:0.00291)0.968.120:0.01022)0.848.47:0.0026,(((197222:0.00571,181330:0.01203)0.883.71:0.00348,(197208:0.00635,322652:0.02829)0.746.39:0.00083)0.977.147:0.00538,360033:0.01832)0.396.6:0.00015)0.187.12:0.00504,321453:0.00864)0.911.77:0.0063,((((((((193685:0.01551,187565:0.01904)0.941.118:0.00399,193431:0.00698)0.888.55:0.00729,177848:0.02798)0.848.48:0.00186,178221:0.01376)0.735.29:0.00071,190956:0.02198)0.804.29:0.00078,(((((115149:0.01513,115104:0.01227)0.911.78:0.00623,((((187002:0.00896,177530:0.01337)0.502.14:0.00089,197502:0.00898)0.844.36:0.00015,196200:0.01053)0.843.45:0.00372,367264:0.02658)0.960.126:0.00091)0.999.655:0.00015,193337:0.01166)0.936.93:0.00015,((183040:0.00616,190808:0.01254)0.762.31:0.00225,190938:0.00371)0.397.19:0.00226)0.415.9:0.00077,(360182:0.01958,177101:0.02102)0.917.80:0.00014)0.923.82:0.00153)0.506.7:0.00233,(((216807:0.00799,174328:0.01702)0.819.34:0.00569,((517282:0.02773,(529295:0.00849,518016:0.02016)0.910.71:0.00418)0.771.37:0.00109,(312986:0.0177,(576954:0.00433,((324163:0.00921,(192937:0.0147,185969:0.0144)0.999.656:0.00414)0.732.26:0.00095,199065:0.01629)0.896.62:0.00298)0.825.28:0.00164)0.803.30:0.00147)0.964.135:0.00401)0.357.9:0.00336,((183564:0.01226,179734:0.00931)0.751.26:0.00203,(185563:0.01267,((196082:0.00824,183824:0.01279)0.461.9:0.01232,(174763:0.02457,(187385:0.01679,(198532:0.01261,(193842:0.03257,(187021:0.02153,(174093:0.00315,146226:0.0165)0.988.204:0.01142)0.936.94:0.00838)0.942.106:0.00589)0.949.107:0.00698)0.563.8:0.00289)0.981.145:0.00857)0.976.126:0.00665)0.924.77:0.00703)0.857.56:0.00752)0.956.104:0.00468)0.913.86:0.0025,((532203:0.0147,189726:0.00834)0.975.146:0.00026,(365047:0.02007,165102:0.01657)0.850.44:0.00156)0.788.28:0.00291)0.727.21:0.00106)0.355.9:0.00291,(181675:0.01247,174624:0.02327)0.915.72:0.00581)0.686.11:0.00358)0.717.18:0.00065,(293153:0.01934,((((585201:0.00172,291733:0.00806)0.850.45:0.00629,(337746:0.00481,291392:0.01182)0.882.65:0.01028)0.978.136:0.00488,((131571:0.00992,13995:0.01139)0.910.72:0.0034,((71965:0.00384,(317113:0.00368,408744:0.0158)0.754.34:0.0037)0.936.95:0.00485,519761:0.01313)0.928.81:0.0029)0.862.70:0.00152)0.792.24:0.00014,(449983:0.00561,517348:0.0143)0.959.108:0.00461)0.946.94:0.00577)0.901.68:0.00569)0.366.6:0.00281)0.913.87:0.00513)0.862.71:0.0026,(((290857:0.02543,((173917:0.01202,(356581:0.02064,181003:0.01261)0.955.107:0.02229)0.941.119:0.00609,((191950:0.00963,544925:0.01972)0.925.80:0.00954,197050:0.03087)0.457.8:0.00219)0.972.136:0.01168)0.574.8:0.0048,(514922:0.00515,526948:0.01337)0.998.427:0.01938)0.923.83:0.0063,((312589:0.04379,179866:0.02081)0.159.7:0.00192,(149178:0.01111,((199290:0.01544,190610:0.01183)0.911.79:0.00502,(((((237138:0.00951,585057:0.00476)1.000.2492:0.00467,((184636:0.0105,184327:0.01077)0.603.10:0.00551,171873:0.00434)0.888.56:0.00197)0.863.60:0.00125,((162651:0.0114,370338:0.02242)0.238.6:0.00131,(((361186:0.0155,365869:0.01664)0.623.9:0.0033,359787:0.01431)0.945.92:0.00463,(366237:0.01817,368054:0.00364)0.798.30:0.00268)0.946.95:0.0003)0.049.2:0.0025)1.000.2493:0.00078,183145:0.0198)0.975.147:0.00409,(((193852:0.00426,336133:0.02592)0.422.11:0.00448,176519:0.01974)0.384.9:0.00271,187271:0.02389)0.892.56:0.00334)0.900.69:0.00294)0.491.7:0.01398)0.504.8:0.00638)0.647.9:0.00354)0.871.71:0.00236)1.000.2494:0.01118)0.949.108:0.00572,(571081:0.01899,((563161:0.03473,557619:0.00285)0.985.154:0.00932,302321:0.00903)0.824.27:0.0021)0.994.239:0.01247)0.914.71:0.00509)0.949.109:0.00642)0.946.96:0.00737,(510622:0.0286,521442:0.00335)0.000.83:0.00932)0.676.15:0.00602)0.912.68:0.00462)0.694.13:0.00419)0.934.71:0.00413,((((((178173:0.02,(189189:0.02453,((263339:0.0021,277760:0.00492)0.883.72:0.00097,569818:0.0116)0.955.108:0.00738)0.952.112:0.00683)0.896.63:0.00831,(195934:0.01466,192504:0.02192)0.949.110:0.0078)0.935.100:0.00447,((270908:0.03855,((548361:0.04709,((((339718:0.0148,((402626:0.00804,336277:0.0111)1.000.2495:0.03678,(274387:0.02109,308766:0.0092)0.915.73:0.00898)0.745.34:0.00612)0.555.11:0.00553,(277946:0.01281,272790:0.01188)0.934.72:0.00601)0.913.88:0.00566,(((((319134:0.00348,(175893:0.01972,(356657:0.01617,(329678:0.01178,261710:0.00465)0.904.75:0.00635)0.899.72:0.00778)0.982.179:0.01063)0.522.9:0.00566,190057:0.01555)0.945.93:0.01238,(((((432782:0.0059,278545:0.00067)0.866.58:0.00932,(279438:0.0102,(278208:0.00734,450492:0.00651)0.990.189:0.00934)0.990.190:0.00958)0.729.20:0.00119,290089:0.00561)0.956.105:0.00321,(((((((415491:0.00163,96194:0.01727)0.984.166:0.00562,436548:0.00836)0.903.75:0.00247,437135:0.003)0.740.35:0.00157,(178606:0.01578,279277:0.01043)0.740.36:0.00016)0.984.167:0.00609,270504:0.00577)0.843.46:0.00188,((198513:0.02673,231529:0.00327)0.853.49:0.00884,(459276:0.00673,447776:0.00244)0.916.75:0.00346)0.906.79:0.00294)0.523.10:0.00238,264334:0.0106)0.895.69:0.00266)0.769.33:0.00121,((275096:0.00682,(434658:0.00329,(385545:0.00315,((314110:0.0155,132048:0.02922)0.990.191:0.01437,129358:0.00613)0.518.6:0.00286)0.945.94:0.00483)0.829.32:0.0031)0.750.32:0.00273,260890:0.0069)0.987.217:0.00776)0.804.30:0.0019)0.963.109:0.00666,((232616:0.00993,((233691:0.00867,(((((((276816:0.02642,274339:0.00446)0.303.10:0.00656,436138:0.00914)0.845.37:0.00203,138338:0.01439)0.749.33:0.00096,(276312:0.01357,129537:0.00359)0.987.218:0.00678)0.922.86:0.00343,((259477:0.02396,353923:0.01725)0.982.180:0.00014,259910:0.01445)0.941.120:0.00187)0.743.37:0.00072,180097:0.03497)0.902.59:0.00238,((278753:0.00289,162963:0.00676)0.710.16:0.00395,377946:0.01891)0.947.96:0.00526)0.730.25:0.00198)0.951.94:0.00541,((320181:0.00977,(((135689:0.01452,(351405:0.01507,(333837:0.01738,308400:0.02161)0.431.11:0.00364)0.922.87:0.005)0.743.38:0.00092,((326212:0.01325,305173:0.03424)0.977.148:0.01538,335177:0.00838)0.780.28:0.02175)0.992.216:0.01208,334697:0.01588)0.947.97:0.00664)0.865.63:0.0038,314922:0.03409)0.966.121:0.0054)0.773.33:0.00438)0.842.55:0.00336,((260756:0.0147,390820:0.0132)0.718.17:0.00112,259614:0.03467)0.735.30:0.00455)0.658.10:0.00521)0.964.136:0.00748,(((306062:0.01162,(301753:0.00476,(69988:0.00095,353195:0.00435)0.979.148:0.00507)0.757.44:0.00014)0.453.9:0.00182,(302512:0.01634,(19873:0.01257,24175:0.0111)0.868.75:0.00687)0.988.205:0.01002)0.951.95:0.00509,288438:0.01268)0.792.25:0.00378)0.566.13:0.00367)0.976.127:0.00849,((((173218:0.01873,233583:0.00235)0.964.137:0.01688,(318481:0.03624,((335267:0.01649,(184966:0.01188,209907:0.01089)0.958.122:0.00776)0.998.428:0.01587,(579750:0.01969,((301012:0.00899,((296187:0.02986,(453109:0.0149,((229810:0.00294,163997:0.01283)0.840.34:0.02367,(136722:0.00633,266184:0.00343)0.801.25:0.00177)0.997.328:0.01537)0.547.11:0.00372)0.936.96:0.00855,400150:0.01475)0.799.32:0.00347)0.824.28:0.00496,(110693:0.0049,535209:0.00275)0.988.206:0.01065)0.961.115:0.00662)0.923.84:0.00612)0.758.37:0.00157)0.767.48:0.0039)0.374.11:0.00093,(((229967:0.00122,264889:0.00577)0.935.101:0.0462,(131501:0.02724,196735:0.01467)0.990.192:0.013)0.980.144:0.00981,(190047:0.03774,(269815:0.02463,((328137:0.01096,(196974:0.01746,305803:0.02023)0.304.8:0.00638)0.932.81:0.00602,(174366:0.00797,192585:0.02571)0.359.4:0.00438)0.971.156:0.00846)0.953.105:0.00891)0.963.110:0.00854)0.872.78:0.00641)0.999.657:0.01357,((((((173417:0.00016,(97315:0.0149,450198:0.00232)0.930.93:0.00242)1.000.2496:0.00014,269003:0.00557)0.914.72:0.02518,(319219:0.02396,(259916:0.04363,((230208:0.00517,((233411:0.00737,((285281:0.01023,263834:0.02718)0.995.296:0.01283,(262625:0.01977,351283:0.00178)0.000.84:0.00515)0.081.4:0.00395)0.779.21:0.00301,272905:0.00944)0.973.127:0.00562)0.576.10:0.0003,(186526:0.01136,(343776:0.00725,195547:0.01373)0.958.123:0.00642)0.817.30:0.0012)0.977.149:0.01293)0.999.658:0.02192)0.988.207:0.01568)0.803.31:0.00678,(157957:0.03917,(((356353:0.02637,289030:0.0083)0.987.219:0.01567,287598:0.02064)0.967.126:0.00835,297808:0.01745)0.910.73:0.00833)0.831.25:0.00447)0.930.94:0.00633,((((135814:0.00119,263396:0.01638)0.781.28:0.00166,(129981:0.01444,310407:0.01091)1.000.2497:0.01349)1.000.2498:0.02744,((((450047:0.00183,271815:0.02761)0.996.304:0.00987,(136452:0.00408,276348:0.00372)0.820.23:0.02983)0.964.138:0.00658,(259218:0.00904,255362:0.00156)1.000.2499:0.0066)0.286.10:0.00443,270244:0.01569)0.998.429:0.02126)0.747.36:0.00698,(((284761:0.02455,129575:0.037)1.000.2500:0.02453,((342948:0.02482,261741:0.01794)0.945.95:0.01371,((((180288:0.03353,((133505:0.00815,271537:0.01923)0.905.76:0.00447,270673:0.00363)0.852.67:0.00279)0.918.88:0.00786,239978:0.00692)1.000.2501:0.00081,338813:0.01625)0.419.8:0.02348,(321220:0.02186,(353657:0.01132,322564:0.02621)0.826.28:0.00584)0.951.96:0.00663)0.174.7:0.00278)0.986.172:0.01575)0.893.64:0.00832,(((29205:0.00084,318470:0.00091)0.933.84:0.00438,(174055:0.02135,(275366:0.0239,(188786:0.01687,((174272:0.0305,129885:0.00895)0.994.240:0.01123,264461:0.02308)0.873.68:0.00527)0.976.128:0.01059)0.972.137:0.00883)1.000.2502:0.02076)0.628.14:0.0019,((((437151:0.00188,(((235157:0.00438,331878:0.02784)0.770.29:0.00349,333576:0.01734)0.930.95:0.00563,266778:0.01598)0.337.6:0.00457)0.997.329:0.01363,(((185989:0.01987,130076:0.01528)0.860.61:0.0071,168099:0.00919)0.917.81:0.01028,((230719:0.02887,130500:0.00014)0.974.132:0.00239,(322121:0.02064,(135493:0.02442,273921:0.02593)0.146.6:0.00381)0.959.109:0.00636)0.228.3:0.00027)0.914.73:0.00542)0.965.126:0.00574,(((267569:0.01278,231137:0.01334)0.475.8:0.0024,((264710:0.01066,(275743:0.01379,556312:0.01167)0.754.35:0.00446)0.939.106:0.00656,((308885:0.00178,(293754:0.02395,(298723:0.01766,((312226:0.01154,129394:0.021)0.196.8:0.00232,344400:0.01467)0.905.77:0.0042)0.658.11:0.00237)0.932.82:0.00136)0.850.46:0.00182,253922:0.00043)0.827.31:0.01598)0.838.41:0.0061)0.534.8:0.00273,((136486:0.02168,449164:0.00973)0.419.9:0.00599,(174847:0.0304,((181733:0.01421,(179964:0.01344,180816:0.01689)0.870.69:0.00235)0.627.9:0.00295,(182079:0.02028,(180944:0.02222,202419:0.01051)0.970.118:0.00693)0.144.6:0.00161)0.904.76:0.00536)0.898.75:0.00469)0.940.108:0.00603)0.912.69:0.00348)0.627.10:0.00439,(347885:0.01843,327266:0.01504)0.799.33:0.00764)0.346.11:0.00058)0.901.69:0.0039)1.000.2503:0.01858)0.207.7:0.00271)0.817.31:0.00428,234586:0.02723)0.444.13:0.00323)0.902.60:0.00447)0.732.27:0.00254)0.790.23:0.00229,((((141974:0.01986,13966:0.03217)0.988.208:0.01879,(109986:0.01887,(331088:0.01786,155141:0.0129)0.993.214:0.01846)0.943.108:0.00977)0.991.215:0.0198,((547665:0.02072,553822:0.0159)0.919.84:0.00658,(573916:0.03151,236124:0.01806)0.129.5:0.007)0.988.209:0.01354)0.981.146:0.0123,(((((((302407:0.01443,(291267:0.0125,(288037:0.00762,(514124:0.02314,294302:0.01571)0.712.18:0.00091)0.899.73:0.00928)1.000.2504:0.025)0.555.12:0.00459,(553492:0.00673,322604:0.01642)0.655.13:0.00458)0.982.181:0.00968,((295851:0.01051,(288049:0.01732,(570367:0.00415,(295652:0.00556,515639:0.01899)0.847.40:0.00226)0.463.10:0.00163)0.985.155:0.00765)0.818.23:0.00542,(288168:0.01948,(165839:0.01157,575261:0.00583)0.188.9:0.0048)0.913.89:0.00711)0.905.78:0.00484)0.733.29:0.00124,((333341:0.02672,294440:0.00921)0.947.98:0.00645,(293780:0.02389,((291052:0.01305,(292715:0.00388,356580:0.0184)0.956.106:0.00683)0.986.173:0.01079,((294898:0.00896,(298892:0.01371,(334427:0.00955,302777:0.00209)0.420.9:0.00748)0.943.109:0.00582)0.993.215:0.01126,(294262:0.00871,(337495:0.01192,332813:0.00711)0.920.71:0.005)0.744.35:0.00267)0.783.35:0.00216)0.978.137:0.01114)0.822.41:0.00341)0.962.123:0.00614)0.886.59:0.00238,((((290205:0.01466,((271614:0.03253,(353085:0.00758,(270676:0.00996,(298714:0.01508,289736:0.01572)0.861.68:0.0022)0.732.28:0.0022)0.903.76:0.00446)0.982.182:0.00987,(358712:0.01479,(590498:0.00166,293326:0.03059)0.993.216:0.01057)0.655.14:0.00505)0.865.64:0.00319)0.961.116:0.00599,(454156:0.01852,(277947:0.01045,(252823:0.00466,105358:0.00401)0.968.121:0.00586)0.941.121:0.00491)0.847.41:0.00518)0.715.16:0.00248,((((((569774:0.00855,(327887:0.01207,(264507:0.02108,110227:0.00705)0.623.10:0.00154)0.871.72:0.00181)0.862.72:0.02256,311461:0.03384)0.976.129:0.00899,(336286:0.02066,(593977:0.0127,333590:0.01366)0.587.9:0.00722)0.987.220:0.01208)0.896.64:0.00729,(334453:0.00753,301033:0.02752)0.961.117:0.01348)0.667.8:0.00435,(((186237:0.02893,((345319:0.01956,(348863:0.00931,(198511:0.01715,291720:0.02208)0.917.82:0.00525)0.931.83:0.00435)0.952.113:0.00481,293511:0.02987)1.000.2505:0.00104)0.437.9:0.0007,179453:0.00179)0.914.74:0.00353,((193448:0.01776,179905:0.02722)0.969.127:0.01193,192127:0.02099)0.840.35:0.00306)0.985.156:0.00818)0.859.57:0.0032,((298642:0.01043,132359:0.0017)0.981.147:0.00835,((325199:0.02727,298701:0.00916)0.995.297:0.0131,295388:0.03867)0.790.24:0.00213)1.000.2506:0.01881)0.954.104:0.0065)0.850.47:0.00014,(((((305552:0.01936,((348574:0.0112,(((241483:0.00763,274836:0.02324)1.000.2507:0.0178,(336656:0.00269,(298105:0.00257,110682:0.02358)0.889.62:0.00427)0.962.124:0.00597)0.702.17:0.00185,301233:0.01593)0.815.25:0.00154)0.713.12:0.00311,((301399:0.00015,(298446:0.02069,298374:0.02252)0.896.65:0.00401)0.658.12:0.01057,(577133:0.00299,(311746:0.00185,(245051:0.00107,313603:0.03464)0.895.70:0.00156)0.831.26:0.00015)0.983.164:0.00912)1.000.2508:0.02067)0.972.138:0.00633)0.554.10:0.00014,(327925:0.00407,(260161:0.0212,270153:0.00548)0.618.9:0.00324)1.000.2509:0.01067)0.980.145:0.00608,(((336542:0.02013,298584:0.00855)0.999.659:0.01704,(334956:0.0145,288913:0.00809)0.967.127:0.01179)0.894.66:0.00562,((265265:0.02256,(((322439:0.00978,151483:0.00727)0.977.150:0.00977,(276114:0.00858,323347:0.00804)0.874.70:0.00381)0.255.5:0.00173,(172624:0.00014,270612:0.00028)0.998.430:0.00861)0.996.305:0.01044)0.593.8:0.00247,(292857:0.01413,294416:0.0049)0.749.34:0.01762)0.923.85:0.00417)0.888.57:0.00302)0.730.26:0.00084,(((110046:0.022,592365:0.01774)0.958.124:0.00769,(264036:0.01933,(588060:0.01932,111271:0.01568)0.893.65:0.00669)0.999.660:0.01994)0.929.100:0.00546,(170699:0.0155,328972:0.00725)1.000.2510:0.01669)0.982.183:0.00808)0.213.10:0.00208,((296474:0.01571,(297396:0.01659,(((516494:0.03347,336392:0.00274)0.763.30:0.00721,((299115:0.01906,543408:0.01631)0.989.173:0.0107,300075:0.00606)0.768.42:0.00182)0.508.11:0.00165,(288931:0.01268,586920:0.01008)0.954.105:0.00379)0.975.148:0.0076)0.276.3:0.00075)0.851.57:0.00373,(297411:0.00014,((289011:0.00208,((187186:0.01933,314172:0.03292)0.800.39:0.00014,(175963:0.00649,44923:0.00089)1.000.2511:0.0009)0.920.72:0.0145)0.739.33:0.0006,303073:0.01823)1.000.2512:0.0063)0.924.78:0.00346)0.575.11:0.0014)0.999.661:0.00968)0.990.193:0.00647)0.769.34:0.00087,((((((296330:0.00809,(328825:0.00788,525010:0.0111)0.862.73:0.00218)0.855.49:0.00301,293990:0.01305)0.936.97:0.00722,(229348:0.01396,(294491:0.02022,303502:0.00902)0.902.61:0.00412)0.969.128:0.00741)0.360.6:0.00146,(561312:0.00962,(292892:0.0034,297372:0.0213)0.736.37:0.00135)0.989.174:0.00661)0.894.67:0.00289,(337580:0.02033,(68844:0.00997,213401:0.01488)0.883.73:0.00588)0.933.85:0.00641)0.901.70:0.00385,(145049:0.02329,((333426:0.00902,196037:0.03203)0.940.109:0.00666,(((277813:0.01004,216554:0.01622)0.699.9:0.00026,296137:0.0223)0.943.110:0.00442,302902:0.0099)0.970.119:0.00789)0.921.81:0.00655)0.685.13:0.00186)0.889.63:0.00248)0.969.129:0.00401,((295574:0.02253,(301156:0.01423,584137:0.01157)0.837.32:0.00647)0.966.122:0.00852,(((290620:0.05004,((36647:0.01837,((((318580:0.01011,(275824:0.01244,317199:0.01035)0.926.93:0.00651)0.918.89:0.00401,(345986:0.01511,((236801:0.00773,169584:0.01078)0.977.151:0.00927,312849:0.00559)0.984.168:0.00999)0.954.106:0.01016)0.596.9:0.00183,(294661:0.00412,(294535:0.01179,352263:0.01758)0.782.34:0.00738)0.422.12:0.00353)0.965.127:0.0056,(338071:0.02852,347200:0.01379)0.990.194:0.01355)0.542.11:0.00075)0.956.107:0.006,((291199:0.00903,578649:0.01171)0.903.77:0.0053,(432902:0.02112,(294869:0.00906,509070:0.01112)0.979.149:0.00842)0.338.7:0.00187)0.876.63:0.00328)0.879.75:0.00726)0.164.3:0.00015,(291950:0.02124,(335488:0.0075,309534:0.02224)0.695.9:0.00378)0.745.35:0.00095)0.868.76:0.00269,(((299008:0.00615,(561027:0.00651,289743:0.00874)0.858.54:0.00292)0.931.84:0.00536,((292684:0.01838,329804:0.0359)0.723.23:0.00521,(298865:0.00898,302513:0.00446)0.741.29:0.01452)0.945.96:0.00782)0.965.128:0.00879,530379:0.01891)0.884.67:0.0043)0.895.71:0.00367)0.776.38:0.0015)0.663.10:0.00354)0.828.31:0.00209)0.747.37:0.0011)0.408.16:0.00442,(((290605:0.02952,(586290:0.03385,298821:0.01679)0.968.122:0.01688)0.372.9:0.01247,(323189:0.03002,292285:0.02694)0.958.125:0.02043)0.741.30:0.00479,(((559499:0.02071,263820:0.00937)0.994.241:0.01955,343761:0.03126)0.415.10:0.00729,313441:0.01691)0.931.85:0.007)0.999.662:0.01582)0.874.71:0.0045)0.466.14:0.00341,(((((((469890:0.00635,(530243:0.01755,290640:0.01637)0.937.92:0.00639)0.939.107:0.00582,528596:0.0129)0.666.15:0.00754,554606:0.03872)0.681.19:0.00764,(291997:0.01625,(21881:0.02411,(((265262:0.00779,509912:0.01578)0.835.35:0.00528,565338:0.00298)1.000.2513:0.02759,(248923:0.00383,172857:0.01346)0.919.85:0.00499)0.903.78:0.00394)0.923.86:0.00549)0.980.146:0.00975)0.886.60:0.00427,(((((270480:0.03514,135341:0.01968)0.964.139:0.0188,(264974:0.02517,(261354:0.00814,(345484:0.00317,270215:0.01725)0.868.77:0.00226)0.710.17:0.00804)0.997.330:0.02666)0.962.125:0.01238,(331240:0.02747,30587:0.00402)0.938.100:0.00611)0.767.49:0.00461,313741:0.0128)0.996.306:0.01576,(133550:0.05984,(150502:0.0261,142325:0.01785)0.944.81:0.01394)0.902.62:0.00772)0.970.120:0.01016)0.703.14:0.00117,((348306:0.01346,(323101:0.01463,194202:0.00937)0.925.81:0.00463)0.882.66:0.00359,(300024:0.01874,544448:0.02542)0.757.45:0.00527)0.995.298:0.01536)0.954.107:0.00549,(314939:0.03035,(184518:0.01562,(181170:0.03175,((338380:0.00996,530104:0.00386)0.456.10:0.00084,(347363:0.01123,588549:0.01234)0.951.97:0.00695)0.269.10:0.00204)0.640.10:0.00708)0.993.217:0.01467)0.979.150:0.01208)0.574.9:0.0011)0.960.127:0.00536,(276364:0.03562,(295711:0.00444,570049:0.0092)0.995.299:0.01595)0.924.79:0.00693)0.883.74:0.00264,((((((((((((((16477:0.02176,((((182980:0.01908,180450:0.02002)0.993.218:0.01141,369215:0.00874)1.000.2514:0.01213,((174276:0.02522,135428:0.00585)0.999.663:0.01666,((162491:0.00159,186096:0.00028)0.997.331:0.01072,(25934:0.01531,(421538:0.00942,(341227:0.03326,343536:0.00488)0.845.38:0.01172)0.899.74:0.00376)0.841.44:0.00357)0.792.26:0.00426)0.933.86:0.0057)0.833.36:0.00451,(242741:0.02413,328207:0.01596)0.860.62:0.00534)0.840.36:0.00519)0.349.10:0.005,(((530629:0.01104,526046:0.01313)0.966.123:0.00809,306866:0.02159)0.977.152:0.01145,((343474:0.00763,(14027:0.02609,347744:0.01255)0.608.7:0.00709)0.909.78:0.00692,110000:0.02015)0.907.78:0.00678)0.922.88:0.0065)0.884.68:0.00446,(561808:0.00269,(277178:0.0008,23277:0.00577)0.022.3:0.00029)1.000.2515:0.03213)0.058.4:0.00317,((264313:0.00739,(306558:0.00555,207610:0.01191)0.850.48:0.00252)1.000.2516:0.028,(((583372:0.01174,((560236:0.01273,((304099:0.00262,271681:0.00455)0.995.300:0.00786,328729:0.0122)0.987.221:0.00636)0.900.70:0.00604,582515:0.00761)0.948.109:0.00454)0.954.108:0.00747,(310790:0.00485,((591376:0.00014,(40639:0.02251,585104:0.00809)0.997.332:0.00959)1.000.2517:0.02701,(142942:0.02192,((110596:0.0071,(517275:0.00896,249126:0.00158)0.746.40:0.00081)0.778.40:0.00078,527100:0.00905)0.974.133:0.00015)0.966.124:0.00573)0.997.333:0.01168)1.000.2518:0.02288)0.040.6:0.00195,(325356:0.0124,339410:0.00031)0.015:0.01823)1.000.2519:0.03326)1.000.2520:0.03148)0.377.6:0.00226,(312707:0.06357,(184895:0.04103,(314059:0.07209,354258:0.04826)0.290.7:0.0179)0.900.71:0.00985)0.942.107:0.00785)0.935.102:0.004,(((278392:0.00861,((((273383:0.00749,264239:0.0194)1.000.2521:0.01619,(170131:0.01623,406964:0.01718)0.323.7:0.00317)0.833.37:0.00553,(((262443:0.00657,323383:0.03605)0.988.210:0.01435,194838:0.03046)0.895.72:0.00575,(267478:0.01864,(338141:0.00649,233991:0.01778)0.097.7:0.00877)0.980.147:0.00754)0.970.121:0.0091)0.769.35:0.00459,234346:0.03752)0.850.49:0.00708)0.993.219:0.01118,(191098:0.01084,261277:0.02885)0.837.33:0.00837)0.963.111:0.00845,187089:0.01901)0.993.220:0.0118)0.881.70:0.00302,(((362015:0.02791,(((195365:0.00647,(199370:0.00761,192603:0.01755)0.938.101:0.00696)0.398.4:0.00394,207615:0.00769)0.930.96:0.00697,(25534:0.00672,292941:0.01423)0.692.6:0.00723)0.037.5:0.00226)0.882.67:0.00484,((357235:0.00494,(69909:0.01911,312291:0.03271)0.832.30:0.00094)0.965.129:0.01173,322310:0.02839)0.957.144:0.00819)0.920.73:0.0076,((585749:0.00557,331026:0.02748)0.994.242:0.0131,((110858:0.01408,(320378:0.01406,(276882:0.02485,270268:0.00188)0.965.130:0.00852)1.000.2522:0.02311)0.994.243:0.01267,(333951:0.0003,323271:0.00413)1.000.2523:0.02235)0.573.8:0.00503)0.988.211:0.01024)1.000.2524:0.021)0.825.29:0.00179,(((362404:0.02159,(49854:0.01387,(325329:0.0188,306939:0.00928)0.860.63:0.01195)0.706.15:0.00484)0.989.175:0.01238,322505:0.01998)0.712.19:0.00675,((((354963:0.03951,((137258:0.04205,(111355:0.00551,540190:0.00015)0.973.128:0.01433)0.997.334:0.02044,471001:0.03104)0.921.82:0.00947)0.847.42:0.00897,((310619:0.00936,34221:0.01108)1.000.2525:0.02191,(259903:0.02112,23832:0.00168)0.406.10:0.021)0.982.184:0.0109)0.742.38:0.00592,210165:0.03143)0.909.79:0.0052,(((299484:0.0238,(110481:0.01751,316747:0.01069)0.948.110:0.00516)0.843.47:0.0022,(((211900:0.01311,181843:0.03577)0.993.221:0.0225,(164940:0.06456,(272752:0.02534,(166592:0.00014,271531:0.00014)1.000.2526:0.00207)0.939.108:0.07955)0.866.59:0.00787)0.994.244:0.01538,((((333967:0.01342,154191:0.00688)0.880.68:0.00156,(92464:0.01782,36497:0.0131)0.858.55:0.00015)0.901.71:0.00169,532771:0.01814)0.889.64:0.01833,(223610:0.01037,121128:0.00952)0.940.110:0.005)0.881.71:0.00496)0.753.37:0.00173)0.916.76:0.00363,((104956:0.01028,563363:0.00115)1.000.2527:0.02506,(588390:0.01478,((152231:0.00405,578995:0.0203)0.891.73:0.00308,533469:0.01586)0.905.79:0.0036)0.897.70:0.0051)0.749.35:0.0032)0.964.140:0.00825)0.882.68:0.00581)0.874.72:0.00489)0.771.38:0.00166,((((145765:0.00275,(((177772:0.01385,179885:0.0052)0.886.61:0.00413,192126:0.01732)0.911.80:0.00344,((196920:0.02264,363026:0.01945)0.770.30:0.00719,(195105:0.01187,186077:0.00996)0.967.128:0.00636)0.411.6:0.00315)0.767.50:0.00072)0.940.111:0.00016,520892:0.01302)0.900.72:0.00287,(((525474:0.0058,(510026:0.00773,(254376:0.00421,(292399:0.00539,(((300517:0.0095,295804:0.01619)0.986.174:0.00756,(514534:0.0055,316489:0.00412)0.863.61:0.00158)0.881.72:0.00163,518115:0.00182)0.953.106:0.0041)0.913.90:0.00434)0.953.107:0.00521)0.655.15:0.00206)0.781.29:0.00014,(((195493:0.01717,((((162623:0.00792,187447:0.01626)0.870.70:0.0035,102506:0.01269)0.777.30:0.00247,300235:0.00976)0.543.6:0.00177,541602:0.01436)0.743.39:0.00086)0.867.68:0.00157,(((((329313:0.00082,304891:0.01142)0.983.165:0.00687,192647:0.01581)0.351.9:0.00194,(182065:0.0156,(353124:0.01906,(194053:0.00751,181189:0.01274)0.894.68:0.00642)0.742.39:0.00819)0.959.110:0.00703)0.868.78:0.00224,(((176297:0.01761,(197930:0.0093,188023:0.01339)0.546.7:0.00185)0.977.153:0.00714,190597:0.00978)0.747.38:0.00122,(189208:0.02232,190076:0.0175)0.918.90:0.00725)0.980.148:0.00686)0.794.32:0.00216,300222:0.01618)0.917.83:0.00381)0.908.83:0.0039,(531436:0.00517,((288961:0.00193,(((197953:0.01282,109014:0.01662)0.683.17:0.00479,58380:0.00762)0.916.77:0.00015,((((299126:0.02085,308551:0.01146)0.256.6:0.00082,326170:0.00864)0.914.75:0.00369,(355196:0.00404,529905:0.01986)0.767.51:0.00354)0.740.37:0.00121,(199643:0.00756,179472:0.01196)0.954.109:0.00723)0.924.80:0.00387)0.795.30:0.00079)0.950.119:0.00923,518613:0.01586)0.511.10:0.0017)0.967.129:0.00606)0.453.10:0.00083)0.989.176:0.00552,((562038:0.01205,518438:0.01061)0.422.13:0.00466,((516553:0.01104,512204:0.00717)1.000.2528:0.00933,311704:0.01365)0.868.79:0.00381)0.795.31:0.00176)0.757.46:0.00093)0.196.9:0.00087,((((180878:0.01443,(192415:0.00853,184864:0.01438)0.986.175:0.00946)0.987.222:0.00778,198208:0.01081)0.872.79:0.00221,((178071:0.01308,197653:0.0084)0.970.122:0.00634,((113909:0.00804,(191824:0.02688,194924:0.01003)0.836.42:0.00668)0.774.24:0.00338,183155:0.0261)0.850.50:0.00455)0.922.89:0.00692)0.206.8:0.00014,((((518807:0.008,((186882:0.01626,(((198071:0.01628,198176:0.01082)0.890.76:0.00015,(351820:0.0165,((185516:0.0157,512903:0.02531)0.655.16:0.00169,(180861:0.00925,(((181379:0.01542,(358809:0.01002,(193343:0.0073,188098:0.01001)0.936.98:0.00546)0.868.80:0.00271)0.688.7:0.00183,357795:0.03349)0.750.33:0.00115,((175518:0.00904,184910:0.01355)0.936.99:0.00374,(((183969:0.00546,(182994:0.02213,(197066:0.03594,197864:0.03095)0.374.12:0.01013)0.454.5:0.00625)0.962.126:0.00828,311423:0.01427)0.951.98:0.00522,(175635:0.0165,304401:0.00089)0.886.62:0.00258)0.982.185:0.01313)0.650.13:0.00288)0.770.31:0.00131)0.766.31:0.00463)0.632.8:0.0015)0.777.31:0.00186)0.992.217:0.01462,175208:0.02989)0.826.29:0.00298)0.076.4:0.00449,182233:0.01689)0.000.85:0.00056)0.119.4:0.00169,((((302163:0.01413,(297701:0.01025,(290396:0.02752,(333279:0.00445,(343332:0.01273,((342179:0.01762,(312664:0.01121,349849:0.01143)0.882.69:0.0031)0.903.79:0.00532,(336249:0.00701,174067:0.01353)0.909.80:0.00943)0.777.32:0.00213)0.840.37:0.00446)0.880.69:0.00594)0.489.3:0.00423)0.961.118:0.00797)0.933.87:0.00521,(525401:0.00758,532267:0.00352)0.973.129:0.00744)0.961.119:0.00595,((308426:0.00911,(152170:0.01281,201110:0.01205)0.991.216:0.00632)0.913.91:0.00316,(100198:0.00797,355102:0.00266)0.955.109:0.00394)0.981.148:0.0071)0.861.69:0.00229,358798:0.03283)0.940.112:0.00156)0.748.28:0.00793,192252:0.02205)0.820.24:0.00156,(176014:0.01738,343967:0.01233)0.725.23:0.00104)0.937.93:0.00279)1.000.2529:0.01056)0.899.75:0.00491)0.845.39:0.00298,((317818:0.02634,291764:0.03234)1.000.2530:0.01891,(((313201:0.00517,(198426:0.01918,(367241:0.00921,(352304:0.01813,348861:0.01984)0.836.43:0.00357)0.947.99:0.00907)0.747.39:0.00104)0.806.27:0.00115,(177055:0.02153,347344:0.01874)0.852.68:0.00744)0.974.134:0.00015,((511342:0.01202,332840:0.02974)0.282.6:0.00134,(163013:0.012,(336866:0.02144,330857:0.00894)0.935.103:0.00597)0.759.25:0.00083)0.171.5:0.00149)0.993.222:0.01614)0.610.7:0.0049)0.973.130:0.00884,(192404:0.03434,(((178882:0.02684,((193833:0.01586,348227:0.02084)0.989.177:0.0239,(336761:0.01122,(191208:0.0131,190874:0.01965)0.956.108:0.01101)0.937.94:0.01027)0.904.77:0.00435)0.861.70:0.00318,180911:0.01637)0.976.130:0.0098,((192952:0.0069,191000:0.0323)0.934.73:0.00796,178809:0.0208)0.845.40:0.007)0.544.9:0.01045)0.631.9:0.00785)0.973.131:0.01317,(((((149699:0.01104,136044:0.00536)1.000.2531:0.02648,(167727:0.01398,(195096:0.00178,232273:0.00085)0.873.69:0.00256)0.789.32:0.00441)0.976.131:0.00983,358858:0.01075)0.867.69:0.00486,(((296280:0.00644,299741:0.03144)0.620.6:0.00376,290762:0.04)1.000.2532:0.0216,((((565292:0.03483,104573:0.01047)0.997.335:0.01729,(297942:0.03305,(288604:0.01546,(292979:0.0143,29937:0.00126)0.915.74:0.00334)0.847.43:0.00281)0.986.176:0.01038)0.922.90:0.00592,(41867:0.01908,((331906:0.00652,(594513:0.00793,((((44311:0.01227,345753:0.00769)0.848.49:0.00232,356261:0.0239)0.945.97:0.00489,354158:0.0059)0.735.31:0.00113,274176:0.01171)0.922.91:0.00355)0.985.157:0.00957)1.000.2533:0.01933,((276703:0.00014,512512:0.0032)1.000.2534:0.02934,(568181:0.00889,525259:0.01378)0.999.664:0.01869)0.265.6:0.00768)0.934.74:0.00731)0.782.35:0.00404)0.703.15:0.00773,312780:0.03039)0.054.3:0.00498)0.954.110:0.00635)0.844.37:0.00281,((233103:0.01273,273681:0.01778)0.967.130:0.00763,((275224:0.00918,(137650:0.02515,522551:0.01045)0.990.195:0.01459)0.953.108:0.0101,(((349401:0.00155,151675:0.00507)0.714.17:0.01969,(110748:0.00826,312588:0.00706)0.804.31:0.00581)0.729.21:0.0057,(163539:0.02353,572313:0.03096)0.968.123:0.01309)0.966.125:0.01088)0.866.60:0.00587)0.955.110:0.00628)0.996.307:0.01422)0.414.11:0.00403,((((((132281:0.02556,(134553:0.00806,192408:0.02476)0.953.109:0.00916)0.991.217:0.01457,(191325:0.03513,(178411:0.00357,165437:0.00428)0.912.70:0.00789)0.994.245:0.01518)0.885.62:0.00706,(258148:0.00968,((260811:0.01242,(335149:0.01177,193626:0.00405)0.766.32:0.01242)0.822.42:0.00458,(273376:0.01043,(130551:0.00844,266061:0.02646)0.784.31:0.00791)0.826.30:0.0088)0.920.74:0.00769)0.614.11:0.00512)0.993.223:0.0135,((341752:0.02651,(287401:0.00432,(((535601:0.00846,544635:0.02178)0.428.9:0.02159,((184320:0.02943,119257:0.00555)0.921.83:0.00306,338905:0.00994)0.941.122:0.00448)0.403.7:0.00183,364034:0.01138)0.924.81:0.00455)0.972.139:0.00772)0.699.10:0.00271,(((346275:0.01287,((268581:0.03001,340600:0.02422)0.779.22:0.00307,327240:0.00456)0.999.665:0.01461)0.982.186:0.00886,((314424:0.03895,211001:0.00315)0.912.71:0.00015,((326644:0.01633,(276190:0.02843,274509:0.0205)0.988.212:0.01251)0.979.151:0.00823,(263865:0.01291,260491:0.01108)0.704.24:0.00245)0.990.196:0.00549)0.975.149:0.00737)0.977.154:0.00779,(302598:0.01733,350052:0.02574)0.999.666:0.02169)0.845.41:0.00685)0.983.166:0.0104)0.768.43:0.00183,((((181368:0.0178,230202:0.00398)1.000.2535:0.02329,((((((129978:0.00093,132783:0.00184)0.849.53:0.017,(242536:0.00966,264427:0.01187)0.771.39:0.02488)0.877.67:0.00604,(228046:0.0278,428954:0.00833)0.883.75:0.00514)0.852.69:0.0038,(190699:0.02494,(273967:0.00978,320528:0.00673)0.935.104:0.00778)0.982.187:0.011)0.946.97:0.00756,(135470:0.01338,(271804:0.01579,134379:0.01761)0.995.301:0.01353)0.849.54:0.00403)0.826.31:0.00374,((179907:0.01918,(((315974:0.02324,311724:0.0157)0.958.126:0.01269,352799:0.01199)0.948.111:0.00745,130280:0.0029)0.964.141:0.00644)0.876.64:0.00298,(((197600:0.02554,270765:0.02851)0.944.82:0.00707,259154:0.01914)0.838.42:0.00163,231169:0.00328)0.917.84:0.0062)0.963.112:0.00881)0.671.14:0.00622)0.856.49:0.0043,(((258626:0.03153,(164071:0.00359,(274578:0.01,131642:0.02067)0.999.667:0.01451)0.940.113:0.00999)1.000.2536:0.03067,((((133779:0.00579,(187235:0.02251,275349:0.01364)0.921.84:0.00577)0.960.128:0.0064,(((182632:0.01237,(403497:0.00854,433923:0.00332)0.777.33:0.00363)0.453.11:0.00189,(131400:0.03331,193619:0.00446)0.519.7:0.00014)0.978.138:0.00092,192131:0.01533)0.520.8:0.0026)0.200.10:0.00223,266761:0.03284)0.942.108:0.00663,(188402:0.02637,131273:0.0335)0.765.32:0.00812)0.978.139:0.0124)0.967.131:0.00918,((((((269949:0.01035,258283:0.03766)0.935.105:0.01156,267783:0.00399)0.800.40:0.00259,(350741:0.01201,164638:0.00242)0.995.302:0.0057)0.758.38:0.00113,(((333330:0.01893,(185587:0.00838,194848:0.02257)0.964.142:0.01316)0.614.12:0.00014,457371:0.00158)0.995.303:0.00376,(318008:0.00178,136633:0.00392)0.000.86:0.05204)0.988.213:0.00868)0.896.66:0.00337,228762:0.00627)0.940.114:0.0053,((323595:0.04633,292817:0.00874)0.998.431:0.02509,308832:0.00387)0.755.46:0.00891)0.963.113:0.00667)0.873.70:0.00492)0.969.130:0.00863,(((190188:0.0204,((((((574689:0.03307,554732:0.00567)0.971.157:0.00014,176924:0.02707)0.974.135:0.01187,299536:0.0177)0.948.112:0.00435,(((509709:0.00674,514398:0.01542)0.745.36:0.00123,(527327:0.01309,(331116:0.00163,332584:0.01187)0.863.62:0.00204)0.988.214:0.00715)0.799.34:0.00201,(342085:0.01539,(125041:0.00182,(192117:0.01827,294710:0.02307)0.293.6:0.00028)0.448.10:0.00757)0.856.50:0.0021)0.824.29:0.00145)0.885.63:0.00299,(((334661:0.01346,(522609:0.0079,(518431:0.00478,(530279:0.0045,514233:0.0146)0.745.37:0.00085)0.461.10:0.00242)0.931.86:0.00403)0.927.98:0.00553,((297708:0.01572,296767:0.00261)0.996.308:0.01283,(292757:0.00673,266058:0.03077)0.918.91:0.00387)0.836.44:0.0085)0.727.22:0.0015,((121461:0.0015,521515:0.02084)0.316.12:0.00093,189334:0.02912)0.463.11:0.00744)0.989.178:0.00769)0.247.6:0.00291,(296770:0.02362,(295568:0.00562,296042:0.02108)0.825.30:0.0053)0.840.38:0.01014)0.475.9:0.00391)0.905.80:0.00504,(((((193601:0.02222,305247:0.00173)0.998.432:0.01375,((365644:0.02334,(167386:0.01565,(365086:0.01015,308242:0.00469)0.984.169:0.00625)0.440.9:0.00013)0.876.65:0.00016,(368944:0.0037,185864:0.02999)1.000.2537:0.00093)0.954.111:0.00771)0.910.74:0.00555,187991:0.01704)0.732.29:0.01933,(184155:0.01943,289375:0.00909)0.850.51:0.00544)0.777.34:0.00512,(((198528:0.00014,197783:0.02186)1.000.2538:0.01517,(((174153:0.00932,((261334:0.01782,(271856:0.03123,95553:0.00639)0.760.35:0.0012)0.911.81:0.00352,412586:0.01293)0.757.47:0.00132)0.884.69:0.00186,((185588:0.01361,((365125:0.0162,549032:0.02913)0.959.111:0.00825,(548134:0.00557,367991:0.02534)0.084.6:0.00053)1.000.2539:0.01194)0.944.83:0.0059,(((303153:0.00815,316740:0.01579)0.750.34:0.00119,((293128:0.01523,301862:0.01518)0.840.39:0.00245,291317:0.0308)0.918.92:0.00411)0.908.84:0.00756,(134472:0.01786,(268620:0.00986,262308:0.01479)0.937.95:0.00802)1.000.2540:0.02269)0.933.88:0.00706)0.953.110:0.00574)0.686.12:0.00121,((282785:0.02403,(322062:0.02073,190273:0.00901)0.966.126:0.00643)0.972.140:0.00823,(367907:0.02207,((515629:0.00833,518329:0.01646)0.754.36:0.00196,((290934:0.01228,521796:0.00813)0.864.74:0.00192,(290030:0.00444,294460:0.01707)0.772.35:0.00015)0.742.40:0.00765)0.246.6:0.0061)0.747.40:0.00684)0.856.51:0.00445)0.899.76:0.00473)0.988.215:0.00827,((((173900:0.00931,132892:0.00561)0.833.38:0.00459,(((((165241:0.00242,269586:0.02554)0.855.50:0.00853,540710:0.01013)0.904.78:0.00588,331150:0.00294)0.875.56:0.00271,299552:0.0042)0.789.33:0.00419,(316781:0.01993,298468:0.00807)0.988.216:0.0108)0.842.56:0.00615)0.214.5:0.00767,(358447:0.02719,((120561:0.00134,197553:0.01388)0.975.150:0.00775,493045:0.00812)0.914.76:0.0062)0.832.31:0.00326)0.952.114:0.00615,(((105921:0.02297,((146502:0.03463,(147442:0.02352,149725:0.00835)0.741.31:0.00649)0.420.10:0.00643,289927:0.02292)0.851.58:0.00457)0.851.59:0.00233,(((545473:0.02634,(544891:0.01103,534050:0.00387)0.530.12:0.00498)0.989.179:0.01218,((144133:0.00894,344422:0.01764)0.869.68:0.00182,(445575:0.02038,(399538:0.00525,564961:0.0213)0.899.77:0.00276)0.816.37:0.00272)0.957.145:0.00579)0.897.71:0.00878,(176137:0.01579,296872:0.01179)0.986.177:0.01205)0.965.131:0.00567)0.908.85:0.00329,(369032:0.02634,(((((346658:0.00818,325848:0.03063)0.749.36:0.00091,((173593:0.00177,167365:0.00028)1.000.2541:0.00087,(361375:0.02438,365964:0.00652)0.915.75:0.00517)0.848.50:0.00533)0.996.309:0.00949,582385:0.00815)0.785.36:0.00486,(198051:0.00578,177454:0.00876)0.966.127:0.00672)1.000.2542:0.01422,(251835:0.00818,191809:0.00688)0.971.158:0.00646)0.729.22:0.00069)0.885.64:0.00288)0.547.12:0.00131)0.901.72:0.00288)0.723.24:0.00078)0.916.78:0.00452)0.911.82:0.00447,(182723:0.01918,(292618:0.02125,(292739:0.01962,292544:0.01685)0.938.102:0.00974)0.909.81:0.00525)0.935.106:0.00707)0.816.38:0.00285)0.914.77:0.00504)0.985.158:0.00759,((((560505:0.01627,(541114:0.03039,48768:0.0104)0.458.8:0.00914)1.000.2543:0.01975,(((62513:0.00014,514200:0.00914)0.653.6:0.01046,(463843:0.02356,397999:0.02245)0.795.32:0.00798)0.981.149:0.01019,((86715:0.00229,528380:0.00537)0.879.76:0.02155,(533634:0.02293,584202:0.00715)0.994.246:0.01502)0.455.7:0.00714)0.851.60:0.00335)1.000.2544:0.02437,((295304:0.01201,((305213:0.00477,(((300866:0.01822,296693:0.02401)0.848.51:0.00422,298108:0.01173)0.905.81:0.01077,(304606:0.025,290257:0.00763)0.992.218:0.01748)0.990.197:0.01227)0.922.92:0.00542,296790:0.02766)0.868.81:0.01168)0.997.336:0.02374,((((73120:0.01532,(86302:0.0028,(248759:0.01764,524204:0.01337)0.167.5:0.00144)0.941.123:0.00741)1.000.2545:0.02945,(236261:0.0434,73450:0.05077)0.202.5:0.01081)0.769.36:0.01066,71387:0.02119)0.569.7:0.00696,52106:0.02746)0.917.85:0.00677)0.958.127:0.00863)0.978.140:0.01274,((162766:0.01369,(518973:0.02432,104740:0.02324)0.300.7:0.00831)0.805.29:0.01132,((526081:0.01199,553336:0.01137)0.995.304:0.01903,((524676:0.00479,484394:0.00402)1.000.2546:0.02871,(566635:0.00042,470747:0.00535)0.289.6:0.01808)0.493.9:0.00727)0.970.123:0.01432)0.998.433:0.02358)0.546.8:0.00508)0.729.23:0.00341)0.831.27:0.00298,(((542096:0.0256,(582577:0.00924,(((13978:0.00029,580464:0.01004)0.992.219:0.00768,(559797:0.0022,370304:0.00886)0.982.188:0.00641)0.922.93:0.00444,((144179:0.00083,362576:0.00638)0.774.25:0.0055,586940:0.01492)0.993.224:0.00963)0.967.132:0.00688)0.779.23:0.00147)0.992.220:0.00929,192133:0.01286)0.996.310:0.01752,((299128:0.02014,(351986:0.02455,329503:0.01604)0.759.26:0.00754)0.942.109:0.01094,((289248:0.01696,(293886:0.0282,292877:0.00543)0.893.66:0.01885)0.997.337:0.01409,((182190:0.02573,(369101:0.00269,331451:0.03308)0.175.11:0.00443)0.569.8:0.00612,((303460:0.01695,(181537:0.00758,195799:0.02148)0.480.12:0.00444)0.991.218:0.00928,199220:0.01879)0.903.80:0.00577)0.540.5:0.00202)0.853.50:0.00563)0.973.132:0.012)0.578.5:0.00681)0.751.27:0.00379)0.914.78:0.00298)0.871.73:0.00166,((((((568828:0.02419,(270176:0.01918,301406:0.01794)0.337.7:0.01345)0.999.668:0.0285,((327336:0.01961,((((317517:0.0202,342641:0.00533)0.770.32:0.00622,265316:0.00683)0.995.305:0.01492,551915:0.00929)0.123.6:0.00386,292256:0.01027)1.000.2547:0.04955)0.540.6:0.00917,537874:0.01746)0.951.99:0.01181)0.795.33:0.00762,(349971:0.02692,289929:0.0171)0.995.306:0.01798)0.812.39:0.00261,(579505:0.03129,(178051:0.00661,363919:0.01206)0.999.669:0.02078)0.978.141:0.0109)0.976.132:0.0076,(((181929:0.02344,((197368:0.01222,179563:0.0121)0.908.86:0.00401,(187470:0.0119,318651:0.01477)0.739.34:0.00291)0.954.112:0.00614)0.977.155:0.00907,142216:0.01773)0.978.142:0.00949,(((533480:0.02412,(214094:0.00135,(((((107149:0.00275,306615:0.00539)1.000.2548:0.00232,104891:0.00796)0.924.82:0.00983,325421:0.00234)0.922.94:0.00317,((569660:0.01042,243519:0.00238)0.872.80:0.00242,(164693:0.02186,(585377:0.02102,(337355:0.02647,322569:0.00811)0.976.133:0.01379)0.998.434:0.01659)0.977.156:0.00853)0.717.19:0.00015)0.905.82:0.00347,109954:0.01137)0.390.8:0.0045)0.992.221:0.01102)0.450.12:0.00897,(((296697:0.01632,312940:0.00731)0.957.146:0.00663,(292554:0.01827,(332630:0.01022,(296687:0.00298,(346525:0.01144,(297562:0.01286,300584:0.02716)0.903.81:0.00423)0.810.34:0.00993)0.944.84:0.00956)0.862.74:0.00341)0.984.170:0.01115)0.953.111:0.009,(292587:0.01757,(((313917:0.01541,294728:0.00988)0.761.41:0.00453,((295951:0.00998,(295539:0.00327,(332126:0.02595,(((101412:0.00829,((579971:0.01004,354870:0.02143)0.887.59:0.00358,339661:0.00285)0.971.159:0.00395)0.463.12:0.0033,((267605:0.00703,13999:0.0263)0.995.307:0.00995,(560114:0.00689,(106355:0.00368,266869:0.01214)0.314.8:0.00382)0.945.98:0.00973)0.919.86:0.00315)0.757.48:0.00066,(133732:0.00811,(312179:0.011,(336098:0.01002,352578:0.01577)0.840.40:0.00311)0.763.31:0.00237)0.777.35:0.00077)0.946.98:0.00568)0.742.41:0.00117)0.987.223:0.00999)0.300.8:0.00349,(326476:0.01023,295444:0.00788)0.519.8:0.00383)0.278.7:0.00547)0.741.32:0.00747,((303060:0.00721,(291938:0.0046,296902:0.0156)0.745.38:0.00466)0.760.36:0.00145,(292247:0.00747,68843:0.02348)0.706.16:0.00602)0.989.180:0.01173)0.865.65:0.00548)0.863.63:0.006)0.101.7:0.00388)0.969.131:0.00932,((183867:0.02798,193840:0.02797)0.710.18:0.0061,202310:0.02735)0.581.10:0.00302)0.783.36:0.0038)0.822.43:0.00198)0.918.93:0.00239,(((((((195946:0.01798,181367:0.02021)0.974.136:0.00904,(178513:0.01451,((183228:0.019,556341:0.00906)0.865.66:0.00338,(110691:0.01123,(563252:0.01169,536341:0.02327)0.874.73:0.00718)0.996.311:0.01246)0.244.6:0.00698)0.329.7:0.0038)0.999.670:0.01714,(194574:0.01745,((188077:0.02031,(178474:0.012,(305841:0.0073,((((193024:0.01526,184472:0.00359)0.768.44:0.00564,331886:0.01186)0.981.150:0.01217,340987:0.01301)0.886.63:0.00362,305712:0.03255)0.894.69:0.00259)0.723.25:0.00467)0.406.11:0.00173)0.529.11:0.00102,((196543:0.01471,186505:0.01833)0.863.64:0.00297,(311913:0.02711,(187782:0.04033,302437:0.00178)0.777.36:0.00014)0.999.671:0.01737)0.964.143:0.00516)0.801.26:0.00203)0.999.672:0.01513)0.923.87:0.00596,(265301:0.03235,271579:0.02504)0.992.222:0.01575)0.295.8:0.00475,(((129783:0.01089,227682:0.01484)0.980.149:0.00757,(199455:0.01202,(175704:0.02557,181258:0.0158)0.959.112:0.00851)0.972.141:0.01093)0.951.100:0.00732,(190426:0.02322,(((196948:0.0117,(197940:0.01687,189396:0.0059)0.197.5:0.02933)0.809.27:0.0052,(329520:0.03262,((359596:0.03428,(184670:0.03595,(((587558:0.00483,((366175:0.01806,362767:0.01646)0.363.8:0.00674,198740:0.02834)0.000.87:0.00131)0.895.73:0.00182,287951:0.02804)0.821.25:0.00218,((307868:0.03813,183894:0.01272)0.821.26:0.00384,((((310632:0.00651,360092:0.01011)0.380.2:0.00375,180070:0.00807)0.341.11:0.00089,111446:0.0072)0.797.22:0.00143,(188965:0.01193,(195175:0.00882,(200698:0.00538,175261:0.00539)0.864.75:0.00821)0.910.75:0.00541)0.978.143:0.0089)0.961.120:0.00508)0.856.52:0.01246)0.285.6:0.00237)0.890.77:0.00432)0.810.35:0.00731,((332313:0.01275,192536:0.00565)0.966.128:0.00927,((188792:0.00781,(188008:0.03472,180983:0.00568)0.998.435:0.01583)0.977.157:0.01043,290281:0.00539)0.856.53:0.00614)0.946.99:0.00568)0.673.14:0.00138)0.731.35:0.00073)0.873.71:0.00564,(((345806:0.02143,295250:0.02291)0.988.217:0.01798,(267714:0.01894,173939:0.03241)0.889.65:0.01001)0.901.73:0.00789,((370809:0.00119,440643:0.0106)0.999.673:0.01904,110292:0.02677)0.990.198:0.01331)0.410.9:0.00682)0.897.72:0.00472)0.736.38:0.00082)0.995.308:0.01213)0.768.45:0.00189,(((319097:0.0152,186672:0.02194)0.954.113:0.00908,((195792:0.0119,176808:0.02787)0.536.11:0.00015,196625:0.0161)0.974.137:0.00902)0.962.127:0.00885,(198808:0.02114,315880:0.01407)0.994.247:0.01445)0.986.178:0.01089)0.971.160:0.00873,((((307536:0.04234,345354:0.04699)0.698.21:0.00484,(191109:0.02991,326539:0.03683)0.926.94:0.01006)0.888.58:0.00669,508842:0.04146)0.808.27:0.00635,324236:0.04464)0.941.124:0.0083)0.771.40:0.00015)0.759.27:0.00067)0.840.41:0.00203)0.868.82:0.0039,(((302306:0.02382,(((403543:0.00995,360015:0.0073)1.000.2549:0.00058,((184729:0.01212,182506:0.01815)0.993.225:0.01006,(342397:0.01398,192085:0.0166)0.848.52:0.00274)0.935.107:0.01603)0.865.67:0.00624,(178851:0.00597,339852:0.02884)0.946.100:0.00631)0.818.24:0.00181)0.978.144:0.00612,(((((((369718:0.02083,409686:0.0128)0.884.70:0.00551,((115087:0.00602,176370:0.00434)0.885.65:0.01808,(325977:0.00937,188313:0.00389)0.775.23:0.00686)0.999.674:0.01979)0.858.56:0.00423,368416:0.00814)0.968.124:0.00646,((148467:0.0092,332258:0.01176)0.982.189:0.00906,((((((((188818:0.03149,(360962:0.02604,(360796:0.03926,367855:0.00573)0.784.32:0.0056)0.930.97:0.01005)0.773.34:0.00608,(((325181:0.01716,297678:0.03582)0.804.32:0.00625,(355372:0.00728,321610:0.01193)0.867.70:0.0022)0.990.199:0.00866,301510:0.02742)0.935.108:0.00613)0.959.113:0.0057,294961:0.02304)0.873.72:0.00196,144395:0.02029)0.890.78:0.00279,((182653:0.02551,(184373:0.01682,177359:0.02087)0.950.120:0.00608)0.944.85:0.00531,(((180577:0.01783,(357278:0.00747,196508:0.01929)0.997.338:0.008)0.893.67:0.00234,(573252:0.00902,590777:0.01741)0.856.54:0.00082)0.692.7:0.00103,360592:0.03292)0.588.6:0.0021)0.996.312:0.00786)0.520.9:0.00087,((((330750:0.01495,326380:0.01582)0.997.339:0.01592,(106224:0.02089,184872:0.01822)0.483.10:0.00373)0.946.101:0.00801,((44783:0.03037,(360504:0.00967,(113337:0.01581,112766:0.0181)0.994.248:0.0176)0.925.82:0.00766)0.385.9:0.00584,(323531:0.00862,(349074:0.01262,(179319:0.00594,594052:0.01723)0.950.121:0.01277)0.848.53:0.00285)0.150.5:0.00626)0.877.68:0.00464)0.863.65:0.00282,(((296516:0.01237,(174471:0.01055,(195336:0.00777,306431:0.01892)0.995.309:0.0127)0.983.167:0.01085)0.397.20:0.00357,293078:0.02048)0.565.6:0.0069,(289304:0.0095,169077:0.0305)0.684.16:0.00667)0.908.87:0.00668)0.973.133:0.00654)0.877.69:0.00012,(((593369:0.01695,592784:0.00667)0.840.42:0.00914,297920:0.02098)0.893.68:0.00458,(((146448:0.03236,(185281:0.01743,361829:0.00716)0.937.96:0.00422)0.841.45:0.00241,((313144:0.02284,(((27071:0.00717,(331575:0.00745,336380:0.00824)0.683.18:0.00622)0.749.37:0.00399,292294:0.02805)0.696.13:0.00655,(358915:0.00419,465109:0.0207)0.633.8:0.00302)0.878.73:0.00317)0.725.24:0.00073,(210965:0.00766,(71692:0.01651,((((188194:0.01479,(198555:0.01063,191687:0.00484)0.937.97:0.00561)0.983.168:0.01078,182274:0.00698)0.963.114:0.00841,(14030:0.02483,((367776:0.01427,197405:0.02099)0.942.110:0.00914,529009:0.00512)0.390.9:0.00169)1.000.2550:0.00015)0.760.37:0.0004,(198434:0.01434,313557:0.03364)0.843.48:0.00649)0.422.14:0.00948)0.949.111:0.00504)0.934.75:0.00366)0.965.132:0.00535)0.931.87:0.00418,(196302:0.01254,24546:0.0069)0.998.436:0.01498)0.877.70:0.00289)0.871.74:0.00372)0.807.26:0.00529,(((((262301:0.01799,(197765:0.02139,176877:0.02729)0.946.102:0.00804)0.920.75:0.0043,340272:0.00754)0.998.437:0.01948,((194780:0.03097,(161758:0.0008,173379:0.01707)0.396.7:0.00586)1.000.2551:0.02281,68842:0.01348)0.537.9:0.00464)0.724.19:0.00327,(240371:0.00584,176465:0.00306)1.000.2552:0.02211)0.954.114:0.00679,((181233:0.02117,179497:0.00383)0.898.76:0.00267,368315:0.00284)0.267.11:0.00888)0.888.59:0.00415)0.622.9:0.00228)0.279.10:0.00219)0.825.31:0.00392,(((((146345:0.01935,(180386:0.02595,314489:0.02602)0.970.124:0.00993)0.972.142:0.01117,263138:0.01346)0.962.128:0.01001,((351734:0.04179,((191772:0.01261,(183362:0.02731,269103:0.01157)0.904.79:0.00771)0.932.83:0.00869,(355468:0.03611,(((184695:0.00783,183428:0.01608)0.406.12:0.00359,(333775:0.01608,(191205:0.01621,(14035:0.00799,(130947:0.02843,((193576:0.02077,186933:0.00898)0.827.32:0.00706,195356:0.00446)0.226.5:0.0003)0.785.37:0.00089)0.869.69:0.00204)0.866.61:0.00364)0.917.86:0.00575)0.917.87:0.00576,(259056:0.00968,181683:0.02786)0.893.69:0.0054)0.831.28:0.00817)0.426.8:0.00365)0.970.125:0.00942)0.677.13:0.00386,(573969:0.01687,(264774:0.02809,(563572:0.01516,545139:0.02795)0.587.10:0.00475)0.701.14:0.00702)0.886.64:0.00464)0.392.10:0.00359)0.965.133:0.00631,((45623:0.01139,(188527:0.01639,129538:0.00843)0.923.88:0.00015)0.967.133:0.00562,((273617:0.01509,193509:0.00301)0.964.144:0.00796,173277:0.01517)0.724.20:0.00228)1.000.2553:0.01413)0.226.6:0.0013,(195470:0.01109,((260529:0.02008,(((273179:0.01823,258710:0.01472)0.967.134:0.00787,((231398:0.00981,(((342887:0.01383,262970:0.03198)0.789.34:0.00675,(231146:0.00523,352861:0.0325)0.921.85:0.00373)0.877.71:0.00349,191491:0.00281)0.899.78:0.00409)0.711.15:0.00192,(131793:0.00908,(((328536:0.02707,276663:0.01937)0.993.226:0.01695,178026:0.0282)0.183.4:0.00159,196741:0.0263)0.910.76:0.00387)0.960.129:0.00438)0.970.126:0.00691)0.825.32:0.00361,((((((229073:0.00902,(232638:0.01374,(132092:0.02011,(279026:0.00311,133932:0.00309)0.949.112:0.00423)0.861.71:0.00237)0.322.6:0.00316)0.774.26:0.00082,(259818:0.02002,(132970:0.00695,129526:0.01714)0.889.66:0.00296)0.885.66:0.00224)0.847.44:0.00562,(308032:0.01163,233250:0.02182)0.963.115:0.00387)0.862.75:0.00163,134939:0.02395)0.743.40:0.00088,(266896:0.02373,265503:0.02971)0.949.113:0.00799)0.787.31:0.00249,(314520:0.03281,(188674:0.02203,(586030:0.01269,191476:0.01833)0.000.88:0.00538)0.947.100:0.00718)0.952.115:0.00902)0.619.11:0.00777)0.309.10:0.00484)0.553.12:0.00366,(270015:0.0063,((259781:0.01748,((189272:0.02623,261010:0.01908)0.964.145:0.0113,((185854:0.00604,275676:0.02765)0.898.77:0.00773,(186332:0.01369,(274060:0.00394,275799:0.01969)0.995.310:0.01742)0.995.311:0.01558)0.441.6:0.00287)0.945.99:0.00742)0.988.218:0.01091,355924:0.04811)0.049.3:0.00014)0.997.340:0.01141)0.893.70:0.00287)0.973.134:0.00652)0.979.152:0.00755)0.748.29:0.00264,(((307217:0.01662,317120:0.02357)0.898.78:0.00469,(320468:0.00548,359123:0.01202)0.072.5:0.00029)0.138.7:0.00189,317646:0.01)0.995.312:0.01242)0.934.76:0.00331,((((299604:0.0072,(14036:0.00246,(195999:0.00412,364066:0.00271