/*
*
* imagehandler.cpp
*
* the stone rendering part has been coded by Marin Ferecatu - thanks, good job!
*
*/

#include "imagehandler.h"
#include "icons.h"
#include "qglobal.h"
#include "setting.h"
#include <qpixmap.h>
//#include <iostream>

#ifdef USE_XPM
#include WOOD_PIC
#include WOOD2_PIC
#include WOOD3_PIC
#include WOOD4_PIC
#include WOOD5_PIC
#include TABLE_PIC
#include ALT_GHOST_BLACK
#include ALT_GHOST_WHITE
#endif

#ifdef Q_WS_WIN
 #define M_PI 3.141592653
 double drand48() { return rand()*1.0/RAND_MAX; }
#endif


/*
* Static class variables
*/
QPixmap* ImageHandler::woodPixmap1 = NULL;
QPixmap* ImageHandler::woodPixmap2 = NULL;
QPixmap* ImageHandler::woodPixmap3 = NULL;
QPixmap* ImageHandler::woodPixmap4 = NULL;
QPixmap* ImageHandler::woodPixmap5 = NULL;
QPixmap* ImageHandler::tablePixmap = NULL;
QCanvasPixmapArray* ImageHandler::altGhostPixmaps = NULL;
int ImageHandler::classCounter = 0;
int ImageHandler::antialiasingColor = 0;
int ImageHandler::ac1 = 0;
int ImageHandler::ac2 = 0;
int ImageHandler::ac3 = 0;
int ImageHandler::ac4 = 0;
int ImageHandler::ac5 = 0;

/**
* Stone rendering code
* Heavily inspired from Jago and CGoban code
* http://www.igoweb.org/~wms/comp/cgoban
* /http://www.rene-grothmann.de/jago/
**/

void ImageHandler::icopy(int *im, QImage &qim, int w, int h) {
	
	for (int y = 0; y < h; y++) {
		uint *p = (uint *)qim.scanLine(y);
		for(int x = 0; x < w; x++) {
			p[x] = im[y*h + x];
		}
	}
}


void ImageHandler::decideAppearance(WhiteDesc *desc, int size)  {
	double  minStripeW, maxStripeW, theta;
	
	minStripeW = (double)size / 20.0;
	if (minStripeW < 2.5)
		minStripeW = 2.5;
	maxStripeW = (double)size / 5.0;
	if (maxStripeW < 4.0)
		maxStripeW = 4.0;
	
	theta = drand48() * 2.0 * M_PI;
	desc->cosTheta = cos(theta);
	desc->sinTheta = sin(theta);
	desc->stripeWidth = 1.5*minStripeW +
		(drand48() * (maxStripeW - minStripeW));
	
	desc->xAdd = 3*desc->stripeWidth +
		(double)size * 3.0;
	
	desc->stripeMul = 3.0;
	desc->zMul = drand48() * 650.0 + 70.0;
}


double ImageHandler::getStripe(WhiteDesc &white, double bright, double z, int x, int y) {
	double wBright;
	
	bright = bright/256.0;
	
	double wStripeLoc = x*white.cosTheta - y*white.sinTheta +	white.xAdd;
	double wStripeColor = fmod(wStripeLoc + (z * z * z * white.zMul) *
		white.stripeWidth,
		white.stripeWidth) / white.stripeWidth;
	wStripeColor = wStripeColor * white.stripeMul - 0.5;
	if (wStripeColor < 0.0)
		wStripeColor = -2.0 * wStripeColor;
	if (wStripeColor > 1.0)
		wStripeColor = 1.0;
	wStripeColor = wStripeColor * 0.15 + 0.85;
	
	wBright = bright*wStripeColor;
	
	if (wBright > 255)
		wBright = 255;
	if (wBright < 0)
		wBright = 0;
	
	return wBright*255;
}

void ImageHandler::paintBlackStone (QImage &bi, int d) {
	
	const double pixel=0.8,shadow=0.99;
	// board color
	int col = antialiasingColor; //0xecb164;
	int blue=col&0x0000FF,green=(col&0x00FF00)>>8,red=(col&0xFF0000)>>16;
	
	bool Alias=true;
	
	// these are the images
	int *pb=new int[d*d];
	int i, j, g, k;
	double di, dj, d2=(double)d/2.0-5e-1, r=d2-2e-1, f=sqrt(3);
	double x, y, z, xr, xg, hh;
	
	k=0;
	
	bool smallerstones = false;
	if (smallerstones) r-=1;
	
	for (i=0; i<d; i++)
		for (j=0; j<d; j++) {
			di=i-d2; dj=j-d2;
			hh=r-sqrt(di*di+dj*dj);
			if (hh>=0) {
				z=r*r-di*di-dj*dj;
				if (z>0) z=sqrt(z)*f;
				else z=0;
				x=di; y=dj;
				xr=sqrt(6*(x*x+y*y+z*z));
				xr=(2*z-x+y)/xr;
				if (xr>0.9) xg=(xr-0.9)*10;
				else xg=0;
				if (hh>pixel || !Alias) {
					g=(int)(10+10*xr+xg*140);
					pb[k]=(255<<24)|(g<<16)|(g<<8)|g;
				}
				else {
					hh=(pixel-hh)/pixel;
					g=(int)(10+10*xr+xg*140);
					double shade;
					if (di-dj<r/3) shade=1;
					else shade=shadow;
					pb[k]=((255<<24)
						|(((int)((1-hh)*g+hh*shade*red))<<16)
						|(((int)((1-hh)*g+hh*shade*green))<<8)
						|((int)((1-hh)*g+hh*shade*blue)));
				}
			}
			else pb[k]=0;
			k++;
			
		}
		
		// now copy the result in QImages
		icopy(pb, bi, d, d);
		
		// free memory
		delete[] pb;
}

// non shaded white stones
void ImageHandler::paintWhiteStone1 (QImage &wi, int d, bool stripes) {
	
	WhiteDesc desc;
	decideAppearance(&desc, d);
	
	const double pixel=0.8,shadow=0.99;
	// board color
	int col = antialiasingColor; //0xecb164;
	int blue=col&0x0000FF,green=(col&0x00FF00)>>8,red=(col&0xFF0000)>>16;
	
	bool Alias=true;
	
	// these are the images
	int *pw=new int[d*d];
	int i, j, g, k;
	double di, dj, d2=(double)d/2.0-5e-1, r=d2-2e-1, f=sqrt(3);
	double x, y, z, xr, xg, hh;
	
	k=0;
	
	bool smallerstones = false;
	if (smallerstones) r-=1;
	
	for (i=0; i<d; i++)
		for (j=0; j<d; j++) {
			di=i-d2; dj=j-d2;
			hh=r-sqrt(di*di+dj*dj);
			if (hh>=0) {
				z=r*r-di*di-dj*dj;
				if (z>0) z=sqrt(z)*f;
				else z=0;
				x=di; y=dj;
				xr=sqrt(6*(x*x+y*y+z*z));
				xr=(2*z-x+y)/xr;
				if (xr>0.9) xg=(xr-0.9)*10;
				else xg=0;
				if (hh>pixel || !Alias) {
					g=(int)(200+10*xr+xg*45);
					if (stripes)
						g = (int)getStripe(desc, g, xr/7.0, i, j);
					pw[k]=(255<<24)|(g<<16)|((g)<<8)|(g);
				}
				else {
					hh=(pixel-hh)/pixel;
					//g=(int)(10+10*xr+xg*140);
					double shade;
					if (di-dj<r/3) shade=1;
					else shade=shadow;
					g=(int)(200+10*xr+xg*45);
					if (stripes)
						g = (int)getStripe(desc, g, xr/7.0, i, j);
					pw[k]=((255<<24)
						|(((int)((1-hh)*g+hh*shade*red))<<16)
						|(((int)((1-hh)*g+hh*shade*green))<<8)
						|((int)((1-hh)*g+hh*shade*blue)));
				}
			}
			else pw[k]=0;
			k++;
			
		}
		
		// now copy the result in QImages
		icopy(pw, wi, d, d);
		
		// free memory
		delete[] pw;
}

// shaded white stones
void ImageHandler::paintWhiteStone2 (QImage &wi, int d, bool stripes) {

	WhiteDesc desc;
	decideAppearance(&desc, d);

	// the angle from which the shadow starts (measured to the light direction = pi/4)
	// alpha should be in (0, pi)
	const double ALPHA = M_PI/4;
	// how big the shadow is (should be < d/2)
	const double STRIPE = d/5.0;

	double theta;
	const double pixel=0.8, shadow=0.99;
	// board color
	int col = antialiasingColor; //0xecb164;
	int blue=col&0x0000FF,green=(col&0x00FF00)>>8,red=(col&0xFF0000)>>16;

	bool Alias=true;
	
	// these are the images
	int *pw=new int[d*d];
	int i, j, g, k;
	double di, dj, d2=(double)d/2.0-5e-1, r=d2-2e-1, f=sqrt(3);
	double x, y, z, xr, xg, hh;
	
	k=0;
	
	bool smallerstones = false;
	if (smallerstones) r-=1;

	for (i=0; i<d; i++)
		for (j=0; j<d; j++) {
			di=i-d2; dj=j-d2;
			hh=r-sqrt(di*di+dj*dj);
			if (hh>=0) {
				z=r*r-di*di-dj*dj;
				if (z>0) z=sqrt(z)*f;
				else z=0;
				x=di; y=dj;
				xr=sqrt(6*(x*x+y*y+z*z));
				xr=(2*z-x+y)/xr;
				if (xr>0.9) xg=(xr-0.9)*10;
				else xg=0;

				theta = atan2(double(j-d/2), double(i-d/2)) + M_PI - M_PI/4 + M_PI/2;
				bool stripeband = theta > ALPHA && theta < 2*M_PI-ALPHA;

				if (theta > M_PI)
					theta = (2*M_PI-theta);

				double stripe = STRIPE*sin((M_PI/2)*(theta-ALPHA)/(M_PI-ALPHA));
				if (stripe < 1) stripe = 1;
				
				double gmin=(int)(0+10*xr+xg*45);
				double gmax=(int)(200+10*xr+xg*45);
				gmin = gmax - (gmax-gmin)*(1-exp(-1*(theta-ALPHA)/(M_PI-ALPHA)));
				
				if (hh < STRIPE && hh > pixel && stripeband) {
					
					if (hh > stripe) g = (int)gmax;
					else g = int(gmin + (gmax-gmin)*(hh/stripe));
					
					if (stripes)
						g = (int)getStripe(desc, g, xr/7.0, i, j);
					pw[k]=(255<<24)|(g<<16)|((g)<<8)|(g);
				}
				else if (hh>pixel || !Alias) {
					g=(int)(200+10*xr+xg*45);
					if (stripes)
						g = (int)getStripe(desc, g, xr/7.0, i, j);
					pw[k]=(255<<24)|(g<<16)|((g)<<8)|(g);
				}
				else {
					hh=(pixel-hh)/pixel;
					//g=(int)(10+10*xr+xg*140);
					double shade;
					if (di-dj<r/3) shade=1;
					else shade=shadow;
					if(stripeband) g=(int)gmin;
					else g=(int)(200+10*xr+xg*45);
					if (stripes)
						g = (int)getStripe(desc, g, xr/7.0, i, j);
					pw[k]=((255<<24)
								 |(((int)((1-hh)*g+hh*shade*red))<<16)
								 |(((int)((1-hh)*g+hh*shade*green))<<8)
								 |((int)((1-hh)*g+hh*shade*blue)));
				}
			}
			else pw[k]=0;
			k++;
		}

	// now copy the result in QImages
	icopy(pw, wi, d, d);

	// free memory
	delete[] pw;

}

void ImageHandler::paintWhiteStone (QImage &wi, int d, bool noShadow, bool stripes)
{
	if (noShadow)
		paintWhiteStone1 (wi, d, stripes);
	else
		paintWhiteStone2 (wi, d, stripes);
}


// This function generates a realistic wood like board image.
// It is not really used in the code but included only for
// conveninence.
/*
void createwood (QImage &im, int w, int h, int color, bool shadows, int ox, int oy, int d) {

	if (w==0 || h==0) return;

	int *p=new int[w*h];
	int *ps=0;

	if (shadows) ps=new int[w*h];

	int i,j;
	double f=9e-1;
	int col=color;
	int blue=col&0x0000FF,green=(col&0x00FF00)>>8,red=(col&0xFF0000)>>16;
	double r,g,b;
	double x,y,dist;
	
	bool fine=false;
	
	for (i=0; i<h; i++)
		for (j=0; j<w; j++) {	
			if (fine)
				f=((sin(18*(double)j/w)+1)/2
				+(sin(3*(double)j/w)+1)/10
				+0.2*cos(5*(double)i/h)+
				+0.1*sin(11*(double)i/h))
				*20+0.5;
			else
				f=((sin(14*(double)j/w)+1)/2
				+0.2*cos(3*(double)i/h)+
				+0.1*sin(11*(double)i/h))
				*10+0.5;
			f=f-floor(f);
			if (f<2e-1) f=1-f/2;
			else if (f<4e-1) f=1-(4e-1-f)/2;
			else f=1;
			if (i==w-1 || (i==w-2 && j<w-2) || j==0
				|| (j==1 && i>1)) f=f/2;
			if (i==0 || (i==1 && j>1) || j>=w-1
				|| (j==w-2 && i<h-1)) {	
				r=128+red*f/2; g=128+green*f/2; b=128+blue*f/2;
			}
			else {	
				r=red*f; g=green*f; b=blue*f;
			}
			p[w*i+j]=(255<<24)|((int)(r)<<16)|((int)(g)<<8)|(int)(b);
			
			if (shadows) {	f=1;
			y=abs((i-(ox+d/2+(i-ox)/d*(double)d)));
			x=abs((j-(oy+d/2+(j-oy)/d*(double)d)));
			dist=sqrt(x*x+y*y)/d*2;
			if (dist<1.0) f=0.9*dist;
			ps[w*i+j]=(255<<24)|((int)(r*f)<<16)|((int)(g*f)<<8)|(int)(b*f);
			}
		}
		
		if (shadows)
			icopy(ps, im, w, h);
		else
			icopy(p, im, w, h);
}
*/
/**
* end stone rendering code
**/


// end MF

ImageHandler::ImageHandler()
{
    // Load the pixmaps
#ifdef USE_XPM
    if (tablePixmap == NULL)
		tablePixmap = new QPixmap(const_cast<const char**>(table_xpm));
    if (woodPixmap1 == NULL)
		woodPixmap1 = new QPixmap(const_cast<const char**>(wood_xpm));
    if (woodPixmap2 == NULL)
		woodPixmap2 = new QPixmap(const_cast<const char**>(wood2_xpm));
    if (woodPixmap3 == NULL)
		woodPixmap3 = new QPixmap(const_cast<const char**>(wood3_xpm));
    if (woodPixmap4 == NULL)
		woodPixmap4 = new QPixmap(const_cast<const char**>(wood4_xpm));
    if (woodPixmap5 == NULL)
		woodPixmap5 = new QPixmap(const_cast<const char**>(wood5_xpm));
#else
    if (tablePixmap == NULL)
		tablePixmap = new QPixmap(TABLE_PIC);
    if (woodPixmap1 == NULL)
		//	img_wood = QImage(WOOD_PIC);
		woodPixmap1 = new QPixmap(WOOD_PIC);
    if (woodPixmap2 == NULL)
		woodPixmap2 = new QPixmap(WOOD2_PIC);
    if (woodPixmap3 == NULL)
		woodPixmap3 = new QPixmap(WOOD3_PIC);
    if (woodPixmap4 == NULL)
		woodPixmap4 = new QPixmap(WOOD4_PIC);
    if (woodPixmap5 == NULL)
		woodPixmap5 = new QPixmap(WOOD5_PIC);
#endif

    if (tablePixmap == NULL || tablePixmap->isNull())
		qFatal("Could not load pixmaps.");
    
	// get medium color of table pixmap used for antialiasing
	CHECK_PTR(woodPixmap1);
	QImage img_ = woodPixmap1->convertToImage();
	if (img_.valid(0, 0))
		ac1 = img_.pixel(0, 0);
	CHECK_PTR(woodPixmap2);
	img_ = woodPixmap2->convertToImage();
	if (img_.valid(0, 0))
		ac2 = img_.pixel(0, 0);
	CHECK_PTR(woodPixmap3);
	img_ = woodPixmap3->convertToImage();
	if (img_.valid(0, 0))
		ac3 = img_.pixel(0, 0);
	CHECK_PTR(woodPixmap4);
	img_ = woodPixmap4->convertToImage();
	if (img_.valid(0, 0))
		ac4 = img_.pixel(0, 0);
	CHECK_PTR(woodPixmap5);
	img_ = woodPixmap5->convertToImage();
	if (img_.valid(0, 0))
		ac5 = img_.pixel(0, 0);

    stonePixmaps = NULL;
    ghostPixmaps = NULL;
	
    // Init the alternate ghost pixmaps
    if (altGhostPixmaps == NULL)
    {
#ifdef USE_XPM
		QPixmap alt1(const_cast<const char**>(alt_ghost_black_xpm));
		QPixmap alt2(const_cast<const char**>(alt_ghost_white_xpm));
#else
		QPixmap alt1(ALT_GHOST_BLACK);
		QPixmap alt2(ALT_GHOST_WHITE);
#endif
		if (alt1.isNull() || alt2.isNull())
			qFatal("Could not load alt_ghost pixmaps.");
		QList<QPixmap> list;
		list.append(&alt1);
		list.append(&alt2);
		QList<QPoint> hotspots;
		hotspots.append(new QPoint(alt1.width()/2, alt1.height()/2));
		hotspots.append(new QPoint(alt2.width()/2, alt2.height()/2));
		altGhostPixmaps = new QCanvasPixmapArray(list, hotspots);
    }
    CHECK_PTR(altGhostPixmaps);
	
    classCounter ++;
}

ImageHandler::~ImageHandler()
{
    classCounter --;
    if (classCounter == 0)
    {
		delete woodPixmap1;
		woodPixmap1 = NULL;
		delete woodPixmap2;
		woodPixmap2 = NULL;
		delete woodPixmap3;
		woodPixmap3 = NULL;
		delete woodPixmap4;
		woodPixmap4 = NULL;
		delete woodPixmap5;
		woodPixmap5 = NULL;
		delete tablePixmap;
		tablePixmap = NULL;
		delete altGhostPixmaps;
		altGhostPixmaps = NULL;
    }
	
    delete stonePixmaps;
    delete ghostPixmaps;
}

QPixmap* ImageHandler::getBoardPixmap(skinType s)
{
	switch (s)
	{
	case skinLight:
		return woodPixmap1;
	case skinDark:
		return woodPixmap2;
	case skin3:
		return woodPixmap3;
	case skin4:
		return woodPixmap4;
	case skin5:
		return woodPixmap5;
	default:
		return woodPixmap4;
	}
}

void ImageHandler::init(int size)
{
	// Scale the images
	size = size * 9 / 10;
	
	//*******
	bool noShadow = !setting->readBoolEntry("STONES_SHADOW");
	bool stripes = setting->readBoolEntry("STONES_SHELLS");
	QImage ib = QImage(size, size, 32);
	ib.setAlphaBuffer(TRUE);
	paintBlackStone(ib, size);
	
	QImage iw1 = QImage(size, size, 32);
	iw1.setAlphaBuffer(TRUE);
	paintWhiteStone(iw1, size, noShadow, stripes);

	QImage iw2, iw3, iw4, iw5, iw6, iw7, iw8;

	if (stripes)
	{
		iw2 = QImage(size, size, 32);
		iw2.setAlphaBuffer(TRUE);
		paintWhiteStone(iw2, size, noShadow, stripes);
		
		iw3 = QImage(size, size, 32);
		iw3.setAlphaBuffer(TRUE);
		paintWhiteStone(iw3, size, noShadow, stripes);

		iw4 = QImage(size, size, 32);
		iw4.setAlphaBuffer(TRUE);
		paintWhiteStone(iw4, size, noShadow, stripes);

		iw5 = QImage(size, size, 32);
		iw5.setAlphaBuffer(TRUE);
		paintWhiteStone(iw5, size, noShadow, stripes);

		iw6 = QImage(size, size, 32);
		iw6.setAlphaBuffer(TRUE);
		paintWhiteStone(iw6, size, noShadow, stripes);

		iw7 = QImage(size, size, 32);
		iw7.setAlphaBuffer(TRUE);
		paintWhiteStone(iw7, size, noShadow, stripes);

		iw8 = QImage(size, size, 32);
		iw8.setAlphaBuffer(TRUE);
		paintWhiteStone(iw8, size, noShadow, stripes);
	}
	else
	{
		iw2 = iw1;//.copy();
		iw3 = iw1;//.copy();
		iw4 = iw1;//.copy();
		iw5 = iw1;//.copy();
		iw6 = iw1;//.copy();
		iw7 = iw1;//.copy();
		iw8 = iw1;//.copy();
	}

	//*********
	
	
	// Convert images to pixmaps
	QList<QPixmap> list;
	QPixmap *ibp = new QPixmap();
	ibp->convertFromImage(ib, 
		QPixmap::PreferDither | 
		QPixmap::DiffuseAlphaDither | 
		QPixmap::DiffuseDither);
	list.append(ibp);
	
	QPixmap *iw1p = new QPixmap();
	iw1p->convertFromImage(iw1, 
		QPixmap::PreferDither | 
		QPixmap::DiffuseAlphaDither | 
		QPixmap::DiffuseDither);
	list.append(iw1p);

	QPixmap *iw2p, *iw3p, *iw4p, *iw5p, *iw6p, *iw7p, *iw8p;

	if (stripes)
	{
		iw2p = new QPixmap();
		iw2p->convertFromImage(iw2, 
			QPixmap::PreferDither | 
			QPixmap::DiffuseAlphaDither | 
			QPixmap::DiffuseDither);
		list.append(iw2p);
		
		iw3p = new QPixmap();
		iw3p->convertFromImage(iw3, 
			QPixmap::PreferDither | 
			QPixmap::DiffuseAlphaDither | 
			QPixmap::DiffuseDither);
		list.append(iw3p);
		
		iw4p = new QPixmap();
		iw4p->convertFromImage(iw4, 
			QPixmap::PreferDither | 
			QPixmap::DiffuseAlphaDither | 
			QPixmap::DiffuseDither);
		list.append(iw4p);
		
		iw5p = new QPixmap();
		iw5p->convertFromImage(iw5, 
			QPixmap::PreferDither | 
			QPixmap::DiffuseAlphaDither | 
			QPixmap::DiffuseDither);
		list.append(iw5p);
		
		iw6p = new QPixmap();
		iw6p->convertFromImage(iw6, 
			QPixmap::PreferDither | 
			QPixmap::DiffuseAlphaDither | 
			QPixmap::DiffuseDither);
		list.append(iw6p);
		
		iw7p = new QPixmap();
		iw7p->convertFromImage(iw7, 
			QPixmap::PreferDither | 
			QPixmap::DiffuseAlphaDither | 
			QPixmap::DiffuseDither);
		list.append(iw7p);
		
		iw8p = new QPixmap();
		iw8p->convertFromImage(iw8, 
			QPixmap::PreferDither | 
			QPixmap::DiffuseAlphaDither | 
			QPixmap::DiffuseDither);
		list.append(iw8p);
	}
	else
	{
		list.append(iw1p);
		list.append(iw1p);
		list.append(iw1p);
		list.append(iw1p);
		list.append(iw1p);
		list.append(iw1p);
		list.append(iw1p);
	}
	
	// Create the hotspots as middle of the image
	QList<QPoint> hotspots;
	QPoint point(size/2, size/2);
	hotspots.append(&point);
	hotspots.append(&point);
	hotspots.append(&point);
	hotspots.append(&point);
	hotspots.append(&point);
	hotspots.append(&point);
	hotspots.append(&point);
	hotspots.append(&point);
	hotspots.append(&point);
	
	// Assemble the data in the QCanvasPixmapArray
	stonePixmaps = new QCanvasPixmapArray(list, hotspots);
	
	// Do the same for the ghosts
	QImage gb(ib);
	QImage gw1(iw1);
	QImage gw2(iw2);
	QImage gw3(iw3);
	QImage gw4(iw4);
	QImage gw5(iw5);
	QImage gw6(iw6);
	QImage gw7(iw7);
	QImage gw8(iw8);
	ghostImage(&gb);
	ghostImage(&gw1);
	ghostImage(&gw2);
	ghostImage(&gw3);
	ghostImage(&gw4);
	ghostImage(&gw5);
	ghostImage(&gw6);
	ghostImage(&gw7);
	ghostImage(&gw8);
	list.clear();
	list.append(new QCanvasPixmap(gb));
	list.append(new QCanvasPixmap(gw1));
	list.append(new QCanvasPixmap(gw2));
	list.append(new QCanvasPixmap(gw3));
	list.append(new QCanvasPixmap(gw4));
	list.append(new QCanvasPixmap(gw5));
	list.append(new QCanvasPixmap(gw6));
	list.append(new QCanvasPixmap(gw7));
	list.append(new QCanvasPixmap(gw8));
	ghostPixmaps = new QCanvasPixmapArray(list, hotspots);
}

void ImageHandler::rescale(int size, bool smallerStones)
{
	CHECK_PTR(stonePixmaps);
	CHECK_PTR(ghostPixmaps);

	size = size + 2;

	// get antialiasing color
	switch (setting->readIntEntry("SKIN"))
	{
		case 0:
			antialiasingColor = ac1;
			break;
		case 1:
			antialiasingColor = ac2;
			break;
		case 2:
			antialiasingColor = ac3;
			break;
		case 3:
			antialiasingColor = ac4;
			break;
		case 4:
			antialiasingColor = ac5;
			break;
	}

	// Scale the images
	if (smallerStones)
		size = size - 1;

	bool noShadow = !setting->readBoolEntry("STONES_SHADOW");
	bool stripes = setting->readBoolEntry("STONES_SHELLS");
	QImage ib = QImage(size, size, 32);
	ib.setAlphaBuffer(TRUE);
	paintBlackStone(ib, size);

	QImage iw1 = QImage(size, size, 32);
	iw1.setAlphaBuffer(TRUE);
	paintWhiteStone(iw1, size, noShadow, stripes);

	QImage iw2, iw3, iw4, iw5, iw6, iw7, iw8;

	if (stripes)
	{
		iw2 = QImage(size, size, 32);
		iw2.setAlphaBuffer(TRUE);
		paintWhiteStone(iw2, size, noShadow, stripes);
		
		iw3 = QImage(size, size, 32);
		iw3.setAlphaBuffer(TRUE);
		paintWhiteStone(iw3, size, noShadow, stripes);

		iw4 = QImage(size, size, 32);
		iw4.setAlphaBuffer(TRUE);
		paintWhiteStone(iw4, size, noShadow, stripes);

		iw5 = QImage(size, size, 32);
		iw5.setAlphaBuffer(TRUE);
		paintWhiteStone(iw5, size, noShadow, stripes);

		iw6 = QImage(size, size, 32);
		iw6.setAlphaBuffer(TRUE);
		paintWhiteStone(iw6, size, noShadow, stripes);

		iw7 = QImage(size, size, 32);
		iw7.setAlphaBuffer(TRUE);
		paintWhiteStone(iw7, size, noShadow, stripes);

		iw8 = QImage(size, size, 32);
		iw8.setAlphaBuffer(TRUE);
		paintWhiteStone(iw8, size, noShadow, stripes);
	}
	else
	{
		iw2 = iw1;//.copy();
		iw3 = iw1;//.copy();
		iw4 = iw1;//.copy();
		iw5 = iw1;//.copy();
		iw6 = iw1;//.copy();
		iw7 = iw1;//.copy();
		iw8 = iw1;//.copy();
	}

	// Replace the images in the array

	stonePixmaps->setImage(0, new QCanvasPixmap(ib));
	stonePixmaps->image(0)->setOffset(size/2, size/2);
	stonePixmaps->setImage(1, new QCanvasPixmap(iw1));
	stonePixmaps->image(1)->setOffset(size/2, size/2);
	stonePixmaps->setImage(2, new QCanvasPixmap(iw2));
	stonePixmaps->image(2)->setOffset(size/2, size/2);
	stonePixmaps->setImage(3, new QCanvasPixmap(iw3));
	stonePixmaps->image(3)->setOffset(size/2, size/2);
	stonePixmaps->setImage(4, new QCanvasPixmap(iw4));
	stonePixmaps->image(4)->setOffset(size/2, size/2);
	stonePixmaps->setImage(5, new QCanvasPixmap(iw5));
	stonePixmaps->image(5)->setOffset(size/2, size/2);
	stonePixmaps->setImage(6, new QCanvasPixmap(iw6));
	stonePixmaps->image(6)->setOffset(size/2, size/2);
	stonePixmaps->setImage(7, new QCanvasPixmap(iw7));
	stonePixmaps->image(7)->setOffset(size/2, size/2);
	stonePixmaps->setImage(8, new QCanvasPixmap(iw8));
	stonePixmaps->image(8)->setOffset(size/2, size/2);

	// Do the same for the ghosts
	QImage gb(ib);
	QImage gw1(iw1);
	QImage gw2(iw2);
	QImage gw3(iw3);
	QImage gw4(iw4);
	QImage gw5(iw5);
	QImage gw6(iw6);
	QImage gw7(iw7);
	QImage gw8(iw8);
	ghostImage(&gb);
	ghostImage(&gw1);
	ghostImage(&gw2);
	ghostImage(&gw3);
	ghostImage(&gw4);
	ghostImage(&gw5);
	ghostImage(&gw6);
	ghostImage(&gw7);
	ghostImage(&gw8);
	ghostPixmaps->setImage(0, new QCanvasPixmap(gb));
	ghostPixmaps->image(0)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(1, new QCanvasPixmap(gw1));
	ghostPixmaps->image(1)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(2, new QCanvasPixmap(gw2));
	ghostPixmaps->image(2)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(3, new QCanvasPixmap(gw3));
	ghostPixmaps->image(3)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(4, new QCanvasPixmap(gw4));
	ghostPixmaps->image(4)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(5, new QCanvasPixmap(gw5));
	ghostPixmaps->image(5)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(6, new QCanvasPixmap(gw6));
	ghostPixmaps->image(6)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(7, new QCanvasPixmap(gw7));
	ghostPixmaps->image(7)->setOffset(size/2, size/2);
	ghostPixmaps->setImage(8, new QCanvasPixmap(gw8));
	ghostPixmaps->image(8)->setOffset(size/2, size/2);
}

void ImageHandler::ghostImage(QImage *img)
{
    img->setAlphaBuffer(true);
    img->convertDepth(32);

    int w = img->width(),
		h = img->height(),
		x, y;

    for (y=0; y<h; y++)
    {
		uint *line = (uint*)img->scanLine(y);
		for (x=0; x<w; x++)
		{
			if ((x%2 && !(y%2)) || (!(x%2) && y%2))
			{
				line[x] = qRgba(qRed(line[x]), qGreen(line[x]), qBlue(line[x]), 0);
			}
		}
    }
}
