/*
 * CList.h
 * $Id: CList.h,v 1.5 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999-2001 Michael Meissner, Markus Janich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Definition of the CList class 
//  Purpose     : Managment of class providing the interface to a linked
//                list of objects.
//
//  Updates     :      who       |    date    | what
//                ---------------+------------+-----------------------------
//                               |            |

#ifndef __CLIST_H
#define __CLIST_H


// System
///////////
#if _MSC_VER >= 1300
#include <iostream>
#else
#include <iostream.h>
#endif


// Forward
template <class ObjectType>
class CList;


/// List container class 
template <class ObjectType> 
class CListContainer
/*************************/
{
public:

  //constructor
  CListContainer(ObjectType* pTmp)
	: m_pNext(NULL), m_pPrev(NULL), m_pObject(pTmp) {};
  
  //destructor
  ~CListContainer() {};
  
  //methods
  CListContainer* getNext() { return m_pNext; };
  CListContainer* getPrev() { return m_pPrev; };

  ObjectType* getObject() { return m_pObject; };

  friend class CList<ObjectType>;

protected:
  //data
  CListContainer* m_pNext; 
  CListContainer* m_pPrev;

  ObjectType* const m_pObject;

  //methods
  void setNext(CListContainer* pTmp) { m_pNext = pTmp; };
  void setPrev(CListContainer* pTmp) { m_pPrev = pTmp; };
};


/// List class
/**
  * This class provides functionality of a double linked list.
  * You can add and remove objects to the list and have sequential
  * or random access to the objects in the list. 
  * \par NOTE: The list only keeps the pointer to the objects
  *            so all memory management of the objects itself
  *            belongs to you.
  *
  * @author Michael Meissner, Markus Janich
  */

template <class ObjectType>
class CList
/*************************/
{
public:

  /** Default constructor. */
  CList();

  /** Copy constructor. */
  CList(const CList &cSource);
  
  /** Destructor. */
  ~CList();

  
  /** Returns pointer to the first container
    * of the list. */
  CListContainer<ObjectType>* getFirst()  const { return m_pFirst; };
  
  /** Returns pointer to the last container
    * of the list. */
  CListContainer<ObjectType>* getLast()  const { return m_pLast; };

  /** Returns the number of container in the list. */
  int getNumObjects() const { return m_nNumObjects; };

  /** Inserts new object at the beginning of the list. */
  int insertAsFirst(ObjectType *pObj);

  /** Appends the object 'pObj' to the end of the list. */
  int insertAsLast(ObjectType *pObj);

  /** Inserts the object 'pObject' after the
    * container 'pThere' in the list. */
  int insertAfter(CListContainer<ObjectType> *pThere, ObjectType *pObject);

  /** Searches for the element 'pObj' in the list
    * and returns the pointer to the container
    * which holds the element. */
  CListContainer<ObjectType>* find(ObjectType *pObj) const;

  /** Removes the container 'pRemove' from the list. */
  int remove(CListContainer<ObjectType> *pRemove);

  /** Same as before but with different type
    * of parameter. */
  int remove(ObjectType *pObj);

  /** Returns a copy of the list. But this method NOT only copies the
    * the pointer of the objects as the '=' operator does. No, this
    * method makes a real memcpy of the list. */
  CList<ObjectType>* getFullDuplicate() const;

  /** Removes all containers from the list if 'nFlag' = 0 (default).
    * If 'nFlag' = 1 then also the objects are deleted from the heap. */
  void clear(int nFlag=0);

  /* returns a reference of the object of the i-th container.
   * \parNOTE: 'nIndex' starts at 0.*/
  ObjectType &operator [](int nIndex) const;


  /* returns pointer to the i-th container in the list */
  CListContainer<ObjectType> *operator()(int nIndex) const;

  /** concatenates one list with another by appending one to another. */
  const CList &operator+(const CList &cSource);

  /** Copies one list to another.
    * \par NOTE: The objects itself will not be duplicated on the heap. */
  CList &operator=(const CList &cSource);



protected:
  //data
  int m_nNumObjects;

  CListContainer<ObjectType>* m_pFirst; 
  CListContainer<ObjectType>* m_pLast;


    //methods
  void init() {
    setFirst(NULL);
    setLast(NULL);
    setNumObjects(0);
  };
  void setFirst(CListContainer<ObjectType>* pTmp)
  { m_pFirst = pTmp; };
  void setLast(CListContainer<ObjectType>* pTmp) 
  { m_pLast = pTmp; };
  void setNumObjects(int nTmp)
  { m_nNumObjects = nTmp; };

  void increaseNumObjects()
  {
    setNumObjects(getNumObjects() + 1);
    return;
  }
  void decreaseNumObjects()
  {
    setNumObjects(getNumObjects() - 1);
    return;
  }
};


#if defined(__GNUC__)
  #include "CList.cpp"
#endif

#endif
