/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution
*/
#include <qlistview.h>
#include <qlistbox.h>
#include <qapplication.h>
#include <qpainter.h>
#include <qdragobject.h>
#include "mainimpl.h"
#include "git.h"
#include "domain.h"
#include "filelist.h"

ListBoxFileItem::ListBoxFileItem(QListBox* lb, SCRef t, const QColor& c) :
                                 QListBoxText(lb, t), myColor(c) {}

void ListBoxFileItem::paint(QPainter* p) {

	if (myColor != Qt::black) {
		p->save();
		p->setPen(myColor);
		QListBoxText::paint(p);
		p->restore();
	} else
		QListBoxText::paint(p);
}

ListBoxFiles::ListBoxFiles(Domain* dm, Git* g, QListBox* l) :
                           QObject(dm), d(dm), git(g), lb(l) {

	st = &(d->st);

	connect(lb, SIGNAL(currentChanged(QListBoxItem*)),
		this, SLOT(on_currentChanged(QListBoxItem*)));
	connect(lb, SIGNAL(contextMenuRequested(QListBoxItem*,const QPoint&)),
		this, SLOT(on_contextMenuRequested(QListBoxItem*)));
	connect(lb, SIGNAL(mouseButtonPressed(int,QListBoxItem*,const QPoint&)),this,
		   SLOT(on_mouseButtonPressed(int,QListBoxItem*,const QPoint&)));
	connect(lb, SIGNAL(onItem(QListBoxItem*)),this,SLOT(on_onItem(QListBoxItem*)));
	connect(lb, SIGNAL(onViewport()),this,SLOT(on_onViewport()));
	connect(lb, SIGNAL(clicked(QListBoxItem*)),this,SLOT(on_clicked(QListBoxItem*)));
}

void ListBoxFiles::clear() {

	lb->clear();
}

bool ListBoxFiles::isMergeHeader(QListBoxItem* c) const {

	return ((lb->index(c) == 0) && st->isMerge());
}

void ListBoxFiles::update(const RevFile* files) {

	if (!st->diffToSha().isEmpty()) {
		if (lb->paletteBackgroundColor() != QGit::LIGHT_BLUE)
			lb->setPaletteBackgroundColor(QGit::LIGHT_BLUE);
	} else
		lb->unsetPalette();

	if (files)
		insertFiles(files);

	if (lb->currentText() && (lb->currentText() == st->fileName())) {
		lb->setSelected(lb->currentItem(), st->selectItem()); // just a refresh
		return;
	}
	lb->clearSelection();

	if (st->fileName().isEmpty())
		return;

	QListBoxItem* c = lb->findItem(st->fileName(), Qt::ExactMatch); // could be null
	lb->setSelected(c, st->selectItem()); // calls current changed
}

void ListBoxFiles::insertFiles(const RevFile* files) {

	if (!files)
		return;

	lb->clear();

	if (st->isMerge()) {
		QString header((st->allMergeFiles()) ?
				"Click to view only interesting files" :
				"Click to view all merge files");
		new ListBoxFileItem(lb, header, Qt::blue);
	}
	if (files->names.empty())
		return;

	int prevPar = files->mergeParent[0];
	for (uint i = 0; i < files->names.count(); ++i) {

		SCRef status(files->status[i].right(1));
		if (status == QGit::UNKNOWN)
			continue;

		QColor clr = Qt::black;
		if (files->mergeParent[i] != prevPar) {
			prevPar = files->mergeParent[i];
			new ListBoxFileItem(lb, "", clr);
			new ListBoxFileItem(lb, "", clr);
		}
		if (status == QGit::NEW)
			clr = Qt::darkGreen;
		if (status == QGit::DELETED)
			clr = Qt::red;

		new ListBoxFileItem(lb, git->filePath(*files, i), clr);
	}
}

void ListBoxFiles::on_currentChanged(QListBoxItem* item) {

	if (item) {
		if (st->isMerge() && item == lb->firstItem()) { // header clicked
			st->setAllMergeFiles(!st->allMergeFiles());
			st->setSelectItem(true);
			UPDATE_DOMAIN(d);
			return;
		}
		// if we are called by updateFileList() fileName is already updated
		if (st->fileName() != item->text()) { // avoid loops
			st->setFileName(item->text());
			st->setSelectItem(true);
			UPDATE_DOMAIN(d);
		}
	}
	if (!isMergeHeader(item))
		emit fileSelected(item ? item->text() : "");
}

void ListBoxFiles::on_contextMenuRequested(QListBoxItem* item) {

	if (!item)
		return;

	int idx = lb->index(item);
	if (idx == 0 && st->isMerge()) // header clicked
		return;

	emit contextMenu(item->text(), QGit::POPUP_FILE_EV);
}

void ListBoxFiles::on_mouseButtonPressed(int b, QListBoxItem* item, const QPoint&) {

	if (item && b == Qt::LeftButton) {
		d->setReadyToDrag(true);
		dragFileName = item->text();
	}
}

void ListBoxFiles::on_clicked(QListBoxItem*) {

	d->setReadyToDrag(false); // in case of just click without moving
}

void ListBoxFiles::on_onItem(QListBoxItem*) { mouseMoved(); }
void ListBoxFiles::on_onViewport() { mouseMoved(); }

void ListBoxFiles::mouseMoved() {

	if (d->isReadyToDrag()) {

		if (!d->setDragging(true))
			return;

		if (dragFileName.isEmpty())
			dbs("ASSERT in ListBoxFiles::on_onItem: empty drag name");

		QDragObject* drObj = new QTextDrag(dragFileName, lb);
		dragFileName = "";

		drObj->dragCopy(); // do NOT delete drObj. Blocking until drop event

		d->setDragging(false);
	}
}
