/***************************************************************************
                          rsnap.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rsnap.cpp 1998/09/01 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include <qbitmap.h>
#include <qcursor.h>

#include "rappwin.h"
#include "rsnap.h"
#include "rstatuspanel.h"
#include "rbehaviordef.h"
#include "rgraphdef.h"
#include "rgraphic.h"
#include "rconfig.h"
#include "rinputpanel.h"
#include "rlog.h"
#include "rmath.h"

// Constructor:
//
RSnap::RSnap()
{
  setFreeMode();
  cx=cy=0.0;
  cEl=0;
  snapRange=10.0;
  onlyElement=0;
  firstElement=0;
  showPreview=true;
  calcDirection=false;
  direction=0.0;

  lastUsedXPos=0.0;
  lastUsedYPos=0.0;
  lastUsedAngle=0.0;
  lastUsedRadius=10.0;
  lastUsedPosMode=0;

  snapDistance=10.0;
}



// Destructor:
//
RSnap::~RSnap()
{
}



// Get snapped point with current snap function:
//
//  _x/_y: returns snapped position
//  _definitive: clicked not only a move
//  _onlyElement: ptr to only element to snap
//  _showPreview: show preview in graphic
//  _calcDirection: calculate direction for arrow preview
//  _direction: returns the direction (angle)
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapPoint(double* _x, double* _y,
                 bool _definitive,
                 RElement* _onlyElement,
                 bool _showPreview,
                 bool _calcDirection,
                 double* _direction) 
{
  bool ret=false;
  
  if( currentDoc() && currentDoc()->isMouseInside() ) {

    // Coordinates set by keyboard:
    /*if(currentDoc()->keyCoordsEnabled()) {
      double kx, ky;
      currentDoc()->getKeyCoords(&kx, &ky);
      if(_x) *_x=kx;
      if(_y) *_y=ky;
      return true;
    }*/

    recalculateSnapRange();
    onlyElement=_onlyElement;
    showPreview=_showPreview;
    calcDirection=_calcDirection;
    direction=0.0;

    switch(snapMode) {
      case SNAP_FREE:
        ret = snapFree();
        break;

      case SNAP_GRID:
        ret = snapGrid();
        break;
      
      case SNAP_ENDPOINTS:
        ret = snapEndpoints();
        break;

      case SNAP_CENTERS:
        ret = snapCenters();
        break;

      case SNAP_MIDDLES:
        ret = snapMiddles();
        break;

      case SNAP_AUTOINTER:
        ret = snapAutoInter();
        break;

      case SNAP_MANINTER:
        ret = snapManInter(_definitive);
        break;

      case SNAP_DIST:
        if(true) {
          RInputPanel* inputPanel2=RAppWin::getRAppWin()->getInputPanel2();
          if(inputPanel2) {
            snapDistance = inputPanel2->getValue(0);
          }
          ret = snapDist();
        }
        break;

      case SNAP_ONELEMENT:
        ret = snapOnElement();
        break;

      case SNAP_KEYXY:
        if(true) {
          RInputPanel* inputPanel2=RAppWin::getRAppWin()->getInputPanel2();
          if(inputPanel2) {
            if(inputPanel2->isRadioButtonChecked(0)) {
              cx = inputPanel2->getValue(0);
              cy = inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=0;
            }
            else {
              cx = currentDoc()->getRelZeroX() + inputPanel2->getValue(0);
              cy = currentDoc()->getRelZeroY() + inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=1;
            }

            if(_definitive) {
              lastUsedXPos=inputPanel2->getValue(0);
              lastUsedYPos=inputPanel2->getValue(1);
            }
          }
          ret=true;
        }
        break;

      case SNAP_KEYAR:
        if(true) {
          RInputPanel* inputPanel2=RAppWin::getRAppWin()->getInputPanel2();
          if(inputPanel2) {
            if(inputPanel2->isRadioButtonChecked(0)) {
              cx = cos(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              cy = sin(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=0;
            }
            else {
              cx = currentDoc()->getRelZeroX() + cos(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              cy = currentDoc()->getRelZeroY() + sin(inputPanel2->getValue(0)/ARAD) * inputPanel2->getValue(1);
              if(_definitive) lastUsedPosMode=1;
            }
            if(_definitive) {
              lastUsedAngle=inputPanel2->getValue(0);
              lastUsedRadius=inputPanel2->getValue(1);
            }
          }
          ret=true;
        }
        break;
  
      default:
        break;
    }

    onlyElement=0;
    showPreview=true;
    if(_direction) *_direction=direction;
    direction=0.0;
    calcDirection=false;
  
    if(_x) *_x=cx;
    if(_y) *_y=cy;

    if(cx>=DEF_AREAMAX-DEF_MMTOL || 
       cy>=DEF_AREAMAX-DEF_MMTOL ||
       cx<=DEF_AREAMIN+DEF_MMTOL || 
       cy<=DEF_AREAMIN+DEF_MMTOL   ) ret=false;
    
    if(_showPreview) {
      if(ret) {
        currentDoc()->drawMark(cx, cy);
        QString buf;
        /*sprintf(buf, "Abs: X: %.4f / Y: %.4f\n"
                     "Rel: X: %.4f / Y: %.4f\n",
                     cx, cy,
                     cx-currentDoc()->getRelZeroX(),
                     cy-currentDoc()->getRelZeroY() );*/
        buf = "Abs: X: "
            + mtFormatFloat( cx,
                             currentDoc()->getDimensionExactness(),
                             currentDoc()->getDimensionUnit(),
                             currentDoc()->getDimensionFormat(),
                             true )
            + " / Y: "
            + mtFormatFloat( cy,
                             currentDoc()->getDimensionExactness(),
                             currentDoc()->getDimensionUnit(),
                             currentDoc()->getDimensionFormat(),
                             true )
            + "\nRel: X: "
            + mtFormatFloat( cx-currentDoc()->getRelZeroX(),
                             currentDoc()->getDimensionExactness(),
                             currentDoc()->getDimensionUnit(),
                             currentDoc()->getDimensionFormat(),
                             true )
            + " / Y: "
            + mtFormatFloat( cy-currentDoc()->getRelZeroY(),
                             currentDoc()->getDimensionExactness(),
                             currentDoc()->getDimensionUnit(),
                             currentDoc()->getDimensionFormat(),
                             true );

        statusPanel()->setStatus(buf);
      }
      else {
        currentDoc()->delMark();
      }
    }
  }

  return ret;
}



// Stop snapping (delete preview):
//
void
RSnap::stop() 
{
  if(currentDoc()) {
    currentDoc()->delMark();
    cx = cy = DEF_AREAMAX;
    currentDoc()->delPreviewElement();
    currentDoc()->delPreview();
    currentDoc()->resetPreviewElement();
    currentDoc()->resetHighlightElement();
    currentDoc()->delHighlightElement();
    firstElement=0;
  }
}



// Snap to nothing (free):
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapFree()
{
  currentDoc()->enableMark(true);
  currentDoc()->getRealMousePos(&cx, &cy);
  return true;
}



// Snap to grid:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapGrid()
{
  double minDist;
  double dist;
  double startC[2],           // Mousepointers is in this
         stopC[2];            // range of grid units
  double gp[2];               // Grid point X
  double rm[2];               // real mousepos X
  int    i;

  currentDoc()->getRealMousePos(&rm[0], &rm[1]);

  currentDoc()->enableMark(true);

  // enlarge grid to reasonable size:
  //
  double mg[2];
  for( i=0; i<=1; i++ ) {
    mg[i] = currentDoc()->getCurrentGridWidth( i );
  }

  // Get the range, where the mousepointer is (in factor 100 of grid pieces)
  //
  for( i=0; i<=1; ++i ) {
    if(rm[i]>=0.0) {
      startC[i] = 0.0;
      for(stopC[i]=0.0; stopC[i]<=DEF_AREAMAX; stopC[i]+=mg[i]*100.0) {
        if(stopC[i]>rm[i]) break;
        startC[i] = stopC[i];
      }
    }
    else {
      stopC[i] = 0.0;
      for(startC[i]=0.0; startC[i]>=DEF_AREAMIN; startC[i]-=mg[i]*100.0) {
        if(startC[i]<rm[i]) break;
        stopC[i] = startC[i];
      }
    }
  }

  // Get the snapped coordinate:
  //
  for( i=0; i<=1; ++i ) {
    minDist=DEF_AREAMAX;
    for(gp[i]=startC[i]; gp[i]<=stopC[i]+mg[i]; gp[i]+=mg[i]) {
      // Check grid point:
      //
      if(i==0) dist = mtGetDistance(rm[0], rm[1], gp[0], startC[1]);
      else     dist = mtGetDistance(rm[0], rm[1], gp[0], gp[1]);
      if(dist<minDist) {
        minDist = dist;
        if(i==0) cx = gp[0];
        else     cy = gp[1];
      }
      else if(minDist<DEF_AREAMAX) {
        break;
      }
    }
  }

  return true;
}



// Snap to endpoints:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapEndpoints()
{
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //double minDist=DEF_AREAMAX;  // minimum measured distance
  double minDist=snapRange;    // minimum measured distance
  double dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool startPoint=true;       // snapped to startpoint?
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = mtGetDistance(el->getX1(), el->getY1(),
                             rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          startPoint=true;
          snappedElement=el;
          found=true;
        }

        // Check Endpoint of element (if element has an endpoint):
        //
        if(el->hasEndpoint()) {
          dist = mtGetDistance(el->getX2(), el->getY2(),
                               rmx, rmy);
          if(dist<minDist && dist<snapRange) {
            minDist = dist;
            startPoint=false;
            snappedElement=el;
            found=true;
          }
        }
      }
    }
  }
  
  if(found && snappedElement) {
    if(startPoint) {
      cx = snappedElement->getX1();
      cy = snappedElement->getY1();
      if(calcDirection) direction=mtCorrAngle(snappedElement->getDirection1()+180.0);
    }
    else {
      cx = snappedElement->getX2();
      cy = snappedElement->getY2();
      if(calcDirection) direction=mtCorrAngle(snappedElement->getDirection2()+180.0);
    }
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap to Centers:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapCenters()
{
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //double minDist=DEF_AREAMAX;  // minimum measured distance
  double minDist=snapRange;    // minimum measured distance
  double dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
  }
  
  if(found && snappedElement) {
    cx = snappedElement->getCenterX();
    cy = snappedElement->getCenterY();
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap to Middles:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapMiddles()
{
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //double minDist=DEF_AREAMAX;  // minimum measured distance
  double minDist=snapRange;    // minimum measured distance
  double dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
  }
  
  if(found && snappedElement) {
    cx = snappedElement->getMiddleX();
    cy = snappedElement->getMiddleY();
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap to Auto Intersections:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapAutoInter()
{
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);
  
  //double minDist=DEF_AREAMAX;
  double minDist=snapRange;    // minimum measured distance
  double dist;
  double lb, bb, rb, tb;    // Element borders
  RElement* el1;           // element ptr (walks through elements)
  RElement* el2;           // element ptr (walks through elements)
  //RElement* elBak;         // Backup element pos for two loops
  bool  found=false;       // There was a point in catch range
  double ix1, iy1;          // Intersection result 1
  double ix2, iy2;          // Intersection result 2
  bool  ires1, ires2;      // Tells if there are results

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {
    for(el1=currentDoc()->elementFirst(); el1!=0; el1=currentDoc()->elementNext()) {

      /*if(mouseMoveBreak &&
         PeekMessage(&Msg, *np->GetCadWin(), WM_MOUSEMOVE, WM_MOUSEMOVE, PM_NOREMOVE)) {
        cx=ocx;
        cy=ocy;
        break;
      }*/

      el1->getBorders(&lb, &bb, &rb, &tb);

      if( el1->getFlag(E_VISIBLE)        &&
          el1->getElementTyp()!=T_POINT  &&
          //el1->isOnScreen()            &&
          rmx < rb + snapRange           &&
          rmx > lb - snapRange           &&
          rmy < tb + snapRange           &&
          rmy > bb - snapRange              ) {

        currentDoc()->elementNext();
        for(el2=currentDoc()->elementCurrent(); el2!=0; el2=currentDoc()->elementNext()) {
        
          if(el1!=el2) {
            el2->getBorders(&lb, &bb, &rb, &tb);
  
            if( el2->getFlag(E_VISIBLE)        &&
                el2->getElementTyp()!=T_POINT  &&
                //el2->IsOnScreen()            &&
                rmx < rb + snapRange           &&
                rmx > lb - snapRange           &&
                rmy < tb + snapRange           &&
                rmy > bb - snapRange              ) {
  
              ix1=iy1=ix2=iy2=DEF_AREAMAX;
              ires1=ires2=false;
              
              el1->getIntersection(el2, 
                                   &ires1, &ix1, &iy1, 
                                   &ires2, &ix2, &iy2, 
                                   0,
                                   true);
              
              if(ires1) {
                dist = mtGetDistance(rmx, rmy, ix1, iy1);
                if(dist<minDist) {
                  minDist = dist;
                  cx = ix1;
                  cy = iy1;
                  found=true;
                }
              }
  
              if(ires2) {
                dist = mtGetDistance(rmx, rmy, ix2, iy2);
                if(dist<minDist) {
                  minDist = dist;
                  cx = ix2;
                  cy = iy2;
                  found=true;
                }
              }
            }
          }
        }
        currentDoc()->setCurrentElement(el1);
      }
    }
  }
    
  if(!found) {
    cx = rmx;
    cy = rmy;
  }

  return found;

}



// Snap to Manual Intersections:
//
// _definitive: user clicked (not only mouse move)
//
//  return: true: snapped to point
//          false: no point snapped so far
//
bool
RSnap::snapManInter(bool _definitive)
{
  bool ret=true;   // return value
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  cx = DEF_AREAMAX;
  cy = DEF_AREAMAX;
  
  currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
  
  if(currentDoc()->count()>0) {
    // Catch first element for intersection:
    //
    if(!firstElement) {
      currentDoc()->setHighlightFilter(T_LONG);
      currentDoc()->enableMark(false);
      // 1st element is definitive:
      //
      if(_definitive) {
        firstElement = cEl;
      }
      ret=false;
    }
    
    // Catch second element for intersection:
    //
    else {
      currentDoc()->setHighlightFilter(T_LONG, firstElement);
      bool ires1, ires2;
      double ix1, iy1, ix2, iy2;
      ires1=false;
      ires2=false;
      ix1=iy1=ix2=iy2=DEF_AREAMAX;
          
      firstElement->getIntersection(cEl, 
                                    &ires1, &ix1, &iy1,
                                    &ires2, &ix2, &iy2,
                                    0, false);
        
      if(ires1) {
        currentDoc()->enableMark(true);
          
        // Test both intersections:
        //
        if(ires2) {
          if(  mtGetDistance(rmx, rmy, ix1, iy1)
             > mtGetDistance(rmx, rmy, ix2, iy2) ){

            cx = ix2;
            cy = iy2;
          }
          else {
            cx = ix1;
            cy = iy1;
          }
        }
        else {
          cx = ix1;
          cy = iy1;
        }
      }
      else {
        cx = rmx;
        cy = rmy;
        currentDoc()->enableMark(false);
        ret=false;
      }
          
      // 2nd element is definitive
      //
      if(_definitive) {
        firstElement=0;
      }
    }
  }

  return ret;

}



// Snap to Dist:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapDist()
{
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //double minDist=DEF_AREAMAX;  // minimum measured distance
  double minDist=snapRange;    // minimum measured distance
  double dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0;    // snapped element
  bool found=false;           // There was a point in range

  currentDoc()->enableMark(true);

  if(currentDoc()->count()>0) {

    // Get nearest element:
    //
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) && 
         el->getFlag(E_VISIBLE)) {
         
        // Check Startpoint of element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
    
    if(found) {
      // Start- or endpoint:
      //
      // Check Startpoint of element:
      //
      minDist = mtGetDistance(rmx, rmy, 
                              snappedElement->getX1(),
                              snappedElement->getY1());
  
      // Check Endpoint of element:
      //
      dist = mtGetDistance(rmx, rmy, 
                           snappedElement->getX2(),
                           snappedElement->getY2());

      // Endpoint is nearer:
      //
      if(dist<minDist) {
        snappedElement->getDistPoint(&cx, &cy, snapDistance, false);
      }

      // Startpoint nearer:
      //
      else {
        snappedElement->getDistPoint(&cx, &cy, snapDistance, true);
      }
    }
  }

  if(!found) {
    cx = rmx;
    cy = rmy;
  }

  return found;

}



// Snap to a point on an element:
//
//  return: true: snapped to point
//          false: no point to snap in mouse range
//
bool
RSnap::snapOnElement()
{
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  //double minDist=DEF_AREAMAX;  // minimum measured distance
  double minDist=snapRange;    // minimum measured distance
  double dist;                 // measured distance
  RElement* el;               // element ptr (walks through elements)
  RElement* snappedElement=0; // snapped element
  bool found=false;           // snapped to point?

  currentDoc()->enableMark(true);

  // Find element which is next to cursor:
  //
  if(currentDoc()->count()>0) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
      if((onlyElement==0 || el==onlyElement) &&
         el->getFlag(E_VISIBLE)) {

        // Check Distance to this element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist && dist<snapRange) {
          minDist = dist;
          snappedElement=el;
          found=true;
        }
      }
    }
  }

  if(found && snappedElement) {
    snappedElement->getTouchPoint(&cx, &cy, rmx, rmy);
  }
  else {
    cx = rmx;
    cy = rmy;
  }

  return found;
}



// Snap an element:
//
//   _definitive: true: already chosen (clicked): delete highlighting
//   _show: show highlighting
//   _types: snap only types of group (e.g.: T_ALL, T_STRAIGT, T_CURCULAR, T_LINE, T_ARC, ...)
//   _exception: dont snap this element (or 0)
//
bool 
RSnap::snapElement(bool _show,
                   int _types,
                   RElement* _exception)
{
  double rmx, rmy;  // Real mouse pos
  currentDoc()->getRealMousePos(&rmx, &rmy);

  recalculateSnapRange();
  
  bool found=false;    // element found
  double minDist=snapRange;    // minimum measured distance
  double dist;
  RElement* el;        // element ptr (walks through elements)
  
  cEl=0;               // reset snapped element

  //int omx=np->GetCadWin()->GetMX();
  //int omy=np->GetCadWin()->GetMY();

  if(currentDoc()->count()>0) {
    //for(vc=0; vc<np->GetCadGraphic()->GetVNum(); ++vc) {
    for(el=currentDoc()->elementFirst(); el!=0; el=currentDoc()->elementNext()) {
  
      // break on mouse move:
      //if(omx!=np->GetCadWin()->GetMX() ||
      //   omy!=np->GetCadWin()->GetMY()    ) break;
      
      if(el->isInGroup(_types) &&
         el!=_exception        && 
         el->getFlag(E_VISIBLE)   ) {
  
        // Check distance point<->element:
        //
        dist = el->getDistanceToPoint(rmx, rmy);
        if(dist<minDist) {
          minDist = dist;
          cEl = el;
          found=true;
        }
      }
    }

    if(_show) {
      currentDoc()->setHighlightElement(cEl);
      currentDoc()->drawHighlightElement();
    }
  }

  if(found && cEl) return true;
  else             return false;

}



// Recalculate snap range (in real unit)
//
void 
RSnap::recalculateSnapRange()
{
  if(currentDoc()) {
    snapRange = RCONFIG->getSettingInt("CAD:SnapRange")
                / currentDoc()->getZoom();
  }
}


// EOF
























