#
# Copyright (c) 2002, 2003, 2004 Art Haas
#
# This file is part of PythonCAD.
#
# PythonCAD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# PythonCAD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PythonCAD; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# classes for points
#

from __future__ import generators

import math

import entity
import tolerance
import util
import baseobject
import quadtree
import logger

class Point(baseobject.Subpart):
    """A 2-D point Class.

A Point has the following attributes:

x: x-coordinate
y: y-coordinate

A Point object has the following methods:

{get/set}x(): Get/Set the x-coordinate of the Point.
{get/set}y(): Get/Set the y-coordinate of the Point.
{get/set}Coords(): Get/Set both the x and y coordinates of the Point.
move(): Move a Point.
clone(): Return an identical copy of a Point.
inRegion(): Returns True if the point is in some area.
    """
    messages = {
        'moved' : True,
    }
    # member functions

    def __init__(self, x, y=None, **kw):
        """Initialize a Point.

There are two ways to initialize a Point:

Point(xc,yc) - Two arguments, with both arguments being floats
Point((xc,yc)) - A single tuple containing two float objects
        """
        baseobject.Subpart.__init__(self, **kw)
        if isinstance(x, tuple):
            if y is not None:
                raise SyntaxError, "Invalid call to Point()"
            _x, _y = util.tuple_to_two_floats(x)
        elif y is not None:
            _x = x
            if not isinstance(_x, float):
                _x = float(x)
            _y = y
            if not isinstance(_y, float):
                _y = float(y)
        else:
            raise SyntaxError, "Invalid call to Point()."
        self.__x = _x
        self.__y = _y

    def __str__(self):
        return "(%g,%g)" % (self.__x, self.__y)

    def __sub__(self, p):
        """Return the separation between two points.

This function permits the use of '-' to be an easy to read
way to find the distance between two Point objects.
        """
        if not isinstance(p, Point):
            raise TypeError, "Invalid type for Point subtraction: " + str(p)
        _px, _py = p.getCoords()
        return math.hypot((self.__x - _px), (self.__y - _py))

    def __eq__(self, obj):
        """Compare a Point to either another Point or a tuple for equality.
        """
        if not isinstance(obj, (Point,tuple)):
            return False
        if isinstance(obj, Point):
            if obj is self:
                return True
            _x, _y = obj.getCoords()
        else:
            _x, _y = util.tuple_to_two_floats(obj)
        if abs(self.__x - _x) < 1e-10 and abs(self.__y - _y) < 1e-10:
            return True
        return False

    def __ne__(self, obj):
        """Compare a Point to either another Point or a tuple for inequality.
        """
        if not isinstance(obj, (Point,tuple)):
            return True
        if isinstance(obj, Point):
            if obj is self:
                return False
            _x, _y = obj.getCoords()
        else:
            _x, _y = util.tuple_to_two_floats(obj)
        if abs(self.__x - _x) < 1e-10 and abs(self.__y - _y) < 1e-10:
            return False
        return True

    def getValues(self):
        return 'point', entity.Entity.getValues(self), self.__x, self.__y
    
    def getx(self):
        """Return the x-coordinate of a Point.

getx()
        """
        return self.__x

    def setx(self, val):
        """Set the x-coordinate of a Point

setx(val)

The argument "val" must be a float.
        """
        if self.isLocked():
            raise RuntimeError, "Coordinate change not allowed - object locked."
        _v = val
        if not isinstance(_v, float):
            _v = float(val)
        _x = self.__x
        if abs(_x - _v) > 1e-10:
            self.__x = _v
            self.sendMessage('moved', _x, self.__y)
            self.modified()

    x = property(getx, setx, None, "x-coordinate value")

    def gety(self):
        """Return the y-coordinate of a Point.

gety()
        """
        return self.__y

    def sety(self, val):
        """Set the y-coordinate of a Point

sety(val)

The argument "val" must be a float.
        """
        if self.isLocked():
            raise RuntimeError, "Coordinate change not allowed - object locked."
        _v = val
        if not isinstance(_v, float):
            _v = float(val)
        _y = self.__y
        if abs(_y - _v) > 1e-10:
            self.__y = _v
            self.sendMessage('moved', self.__x, _y)
            self.modified()

    y = property(gety, sety, None, "y-coordinate value")

    def getCoords(self):
        """Return the x and y Point coordinates in a tuple.

getCoords()
        """
        return self.__x, self.__y

    def setCoords(self, x, y):
        """Set both the coordinates of a Point.

setCoords(x, y)

Arguments x and y should be float values.
        """
        _x = x
        if not isinstance(_x, float):
            _x = float(x)
        _y = y
        if not isinstance(_y, float):
            _y = float(y)
        _sx = self.__x
        _sy = self.__y
        if abs(_sx - _x) > 1e-10 or abs(_sy - _y) > 1e-10:
            self.__x = _x
            self.__y = _y
            self.sendMessage('moved', _sx, _sy)
            self.modified()
        
    def move(self, dx, dy):
        """Move a Point.

move(dx, dy)

The first argument gives the x-coordinate displacement,
and the second gives the y-coordinate displacement. Both
values should be floats.
        """
        if self.isLocked():
            raise RuntimeError, "Moving not allowed - object locked."
        _dx = dx
        if not isinstance(_dx, float):
            _dx = float(dx)
        _dy = dy
        if not isinstance(_dy, float):
            _dy = float(dy)
        if abs(_dx) > 1e-10 or abs(_dy) > 1e-10:
            _x = self.__x
            _y = self.__y
            self.__x = _x + _dx
            self.__y = _y + _dy
            self.sendMessage('moved', _x, _y)
            self.modified()

    def clone(self):
        """Create an identical copy of a Point.

clone()
        """
        return Point(self.__x, self.__y)

    def inRegion(self, xmin, ymin, xmax, ymax, fully=True):
        """Returns True if the Point is within the bounding values.

inRegion(xmin, ymin, xmax, ymax)

The four arguments define the boundary of an area, and the
function returns True if the Point lies within that area.
Otherwise, the function returns False.
        """
        _xmin = xmin
        if not isinstance(_xmin, float):
            _xmin = float(xmin)
        _ymin = ymin
        if not isinstance(_ymin, float):
            _ymin = float(ymin)
        _xmax = xmax
        if not isinstance(_xmax, float):
            _xmax = float(xmax)
        if _xmax < _xmin:
            raise ValueError, "Illegal values: xmax < xmin"
        _ymax = ymax
        if not isinstance(_ymax, float):
            _ymax = float(ymax)
        if _ymax < _ymin:
            raise ValueError, "Illegal values: ymax < ymin"
        return _xmin < self.__x < _xmax and _ymin < self.__y < _ymax

    def sendsMessage(self, m):
        if m in Point.messages:
            return True
        return baseobject.Subpart.sendsMessage(self, m)

#
# Quadtree Point storage
#

class PointQuadtree(quadtree.Quadtree):
    def __init__(self):
        quadtree.Quadtree.__init__(self)

    def getNodes(self, *args):
        _alen = len(args)
        if _alen != 2:
            raise ValueError, "Expected 2 arguments, got %d" % _alen
        _x, _y = args        
        if not isinstance(_x, float):
            _x = float(args[0])
        if not isinstance(_y, float):
            _y = float(args[1])
        _nodes = [self.getTreeRoot()]
        while len(_nodes):
            _node = _nodes.pop()
            _xmin, _ymin, _xmax, _ymax = _node.getBoundary()        
            if ((_x < _xmin) or
                (_y < _ymin) or
                (_x > _xmax) or
                (_y > _ymax)):
                continue
            if _node.hasSubnodes():
                _xmid = (_xmin + _xmax)/2.0
                _ymid = (_ymin + _ymax)/2.0
                if _x < _xmid:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SWNODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NWNODE)
                else:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SENODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NENODE)
                _nodes.append(_node)
            else:
                yield _node

    def addObject(self, obj):
        if not isinstance(obj, Point):
            raise TypeError, "Invalid Point object: " + `obj`
        if obj in self:
            return
        _x, _y = obj.getCoords()
        _bounds = self.getTreeRoot().getBoundary()
        _xmin = _ymin = _xmax = _ymax = None
        _resize = False
        if _bounds is None: # first node in tree
            _resize = True
            _xmin = _x - 1.0
            _ymin = _y - 1.0
            _xmax = _x + 1.0
            _ymax = _y + 1.0
        else:
            _xmin, _ymin, _xmax, _ymax = _bounds
            if _x < _xmin:
                _xmin = _x - 1.0
                _resize = True
            if _x > _xmax:
                _xmax = _x + 1.0
                _resize = True
            if _y < _ymin:
                _ymin = _y - 1.0
                _resize = True
            if _y > _ymax:
                _ymax = _y + 1.0
                _resize = True
        if _resize:
            self.resize(_xmin, _ymin, _xmax, _ymax)
        for _node in self.getNodes(_x, _y):
            _node.addObject(obj)
        quadtree.Quadtree.addObject(self, obj)
        obj.connect('moved', self, PointQuadtree.movePoint)

    def old_addObject(self, obj):
        if not isinstance(obj, Point):
            raise TypeError, "Invalid Point object: " + `obj`
        if obj in self:
            return
        _x, _y = obj.getCoords()
        _node = self.getTreeRoot()
        _bounds = _node.getBoundary()
        _xmin = _ymin = _xmax = _ymax = None
        _resize = False
        if _bounds is None: # first node in tree
            _resize = True
            _xmin = _x - 1.0
            _ymin = _y - 1.0
            _xmax = _x + 1.0
            _ymax = _y + 1.0
        else:
            _xmin, _ymin, _xmax, _ymax = _bounds
            if _x < _xmin:
                _xmin = _x - 1.0
                _resize = True
            if _x > _xmax:
                _xmax = _x + 1.0
                _resize = True
            if _y < _ymin:
                _ymin = _y - 1.0
                _resize = True
            if _y > _ymax:
                _ymax = _y + 1.0
                _resize = True
        if _resize:
            self.resize(_xmin, _ymin, _xmax, _ymax)
            _node = self.getTreeRoot()
        while True:
            if ((_x > _xmax) or
                (_y > _ymax) or
                (_x < _xmin) or
                (_y < _ymin)):
                print "invalid node:"
                print "boundary: (%g, %g) -> (%g, %g)" % (_xmin, _ymin,
                                                          _xmax, _ymax)
                print "point: (%g, %g)" % (_x, _y)
                raise ValueError, "Failed to add point: " + `obj`
            if _node.hasSubnodes():
                _xmid = (_xmin + _xmax)/2.0
                _ymid = (_ymin + _ymax)/2.0
                if _x < _xmid:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SWNODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NWNODE)
                else:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SENODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NENODE)
                _xmin, _ymin, _xmax, _ymax = _node.getBoundary()
            else:
                _node.addObject(obj)
                obj.connect('moved', self, PointQuadtree.movePoint)
                break
        quadtree.Quadtree.addObject(self, obj)

    def delObject(self, obj):
        if obj not in self:
            return
        _x, _y = obj.getCoords()
        for _node in self.getNodes(_x, _y):
            _node.delObject(obj)
            _parent = _node.getParent()
            if _parent is not None:
                self.purgeSubnodes(_parent)
        quadtree.Quadtree.delObject(self, obj)
        obj.disconnect(self)

    def old_delObject(self, obj):
        if obj not in self:
            return
        _x, _y = obj.getCoords()
        _node = self.getTreeRoot()
        while True:
            _xmin, _ymin, _xmax, _ymax = _node.getBoundary()
            if ((_x > _xmax) or
                (_y > _ymax) or
                (_x < _xmin) or
                (_y < _ymin)):
                print "invalid node:"
                print "boundary: (%g, %g) -> (%g, %g)" % (_xmin, _ymin,
                                                          _xmax, _ymax)
                print "point: (%g, %g)" % (_x, _y)
                raise ValueError, "Failed to delete point: " + `obj`
            if _node.hasSubnodes():
                _xmid = (_xmin + _xmax)/2.0
                _ymid = (_ymin + _ymax)/2.0
                if _x < _xmid:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SWNODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NWNODE)
                else:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SENODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NENODE)
                _xmin, _ymin, _xmax, _ymax = _node.getBoundary()
            else:
                _node.delObject(obj)
                _parent = _node.getParent()
                if _parent is not None:
                    self.purgeSubnodes(_parent)
                break
        quadtree.Quadtree.delObject(self, obj)
        obj.disconnect(self)

    def find(self, *args):
        _alen = len(args)
        if _alen < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _x = args[0]
        if not isinstance(_x, float):
            _x = float(args[0])
        _y = args[1]
        if not isinstance(_y, float):
            _y = float(args[1])
        _t = tolerance.TOL
        if _alen > 2 :
            _t = tolerance.toltest(args[2])
        if not len(self):
            return None
        _nodes = [self.getTreeRoot()]
        _p = None
        while len(_nodes):
            _node = _nodes.pop()
            _xmin, _ymin, _xmax, _ymax = _node.getBoundary()
            if ((_x < (_xmin - _t)) or
                (_y < (_ymin - _t)) or
                (_x > (_xmax + _t)) or
                (_y > (_ymax + _t))):
                continue
            if _node.hasSubnodes():
                for _subnode in _node.getSubnodes():
                    _xmin, _ymin, _xmax, _ymax = _subnode.getBoundary()
                    if ((_xmin - _t) < _x < (_xmax + _t) and
                        (_ymin - _t) < _y < (_ymax + _t)):
                        _nodes.append(_subnode)
            else:
                for _pt in _node.getObjects():
                    _xp, _yp = _pt.getCoords()
                    _xsep = _xp - _x
                    _ysep = _yp - _y
                    if abs(_xsep) < _t and abs(_ysep) < _t:
                        if _p is None:
                            _p = _pt
                        else:
                            if math.hypot(_xsep, _ysep) < (_p - _pt):
                                _p = _pt
        return _p

    def movePoint(self, obj, *args):
        if obj not in self:
            raise ValueError, "Point not stored in Quadtree: " + `obj`
        _alen = len(args)
        if len(args) < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _x = args[0]
        if not isinstance(_x, float):
            _x = float(args[0])
        _y = args[1]
        if not isinstance(_y, float):
            _y = float(args[1])
        for _node in self.getNodes(_x, _y):
            _node.delObject(obj)
        quadtree.Quadtree.delObject(self, obj)
        obj.disconnect(self)
        self.addObject(obj)

    def old_movePoint(self, obj, *args):
        if obj not in self:
            raise ValueError, "Point not stored in Quadtree: " + `obj`
        _alen = len(args)
        if len(args) < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _x = args[0]
        if not isinstance(_x, float):
            _x = float(args[0])
        _y = args[1]
        if not isinstance(_y, float):
            _y = float(args[1])
        _node = self.getTreeRoot()
        _xmin, _ymin, _xmax, _ymax = _node.getBoundary()
        while True:
            if _node.hasSubnodes():
                _xmid = (_xmin + _xmax)/2.0
                _ymid = (_ymin + _ymax)/2.0
                if _x < _xmid:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SWNODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NWNODE)
                else:
                    if _y < _ymid:
                        _node = _node.getSubnode(quadtree.QTreeNode.SENODE)
                    else:
                        _node = _node.getSubnode(quadtree.QTreeNode.NENODE)
                _xmin, _ymin, _xmax, _ymax = _node.getBoundary()
            else:
                _node.delObject(obj)
                quadtree.Quadtree.delObject(self, obj)
                obj.disconnect(self)
                break
        self.addObject(obj)

    def getClosest(self, x, y, tol=tolerance.TOL):
        return self.find(x, y, tol)

    def getInRegion(self, xmin, ymin, xmax, ymax):
        _xmin = xmin
        if not isinstance(_xmin, float):
            _xmin = float(xmin)
        _ymin = ymin
        if not isinstance(_ymin, float):
            _ymin = float(ymin)
        _xmax = xmax
        if not isinstance(_xmax, float):
            _xmax = float(xmax)
        if _xmax < _xmin:
            raise ValueError, "Illegal values: xmax < xmin"
        _ymax = ymax
        if not isinstance(_ymax, float):
            _ymax = float(ymax)
        if _ymax < _ymin:
            raise ValueError, "Illegal values: ymax < ymin"
        _pts = []        
        if not len(self):
            return _pts
        _nodes = [self.getTreeRoot()]
        while len(_nodes):
            _node = _nodes.pop()
            if _node.hasSubnodes():
                for _subnode in _node.getSubnodes():
                    _sxmin, _symin, _sxmax, _symax = _subnode.getBoundary()
                    if ((_sxmin > _xmax) or
                        (_symin > _ymax) or
                        (_sxmax < _xmin) or
                        (_symax < _ymin)):
                        continue
                    _nodes.append(_subnode)
            else:
                for _pt in _node.getObjects():
                    if _pt.inRegion(_xmin, _ymin, _xmax, _ymax):
                        _pts.append(_pt)
        return _pts

#
# Point history class
#

class PointLog(logger.Logger):
    def __init__(self, p):
        if not isinstance(p, Point):
            raise TypeError, "Invalid point: " + `p`
        logger.Logger.__init__(self)
        self.__p = p
        p.connect('moved', self, PointLog.movePoint)

    def detatch(self):
        self.__p.disconnect(self)
        self.__p = None
        
    def movePoint(self, p, *args):
        _alen = len(args)
        if _alen < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _x = args[0]
        if not isinstance(_x, float):
            _x = float(args[0])
        _y = args[1]
        if not isinstance(_y, float):
            _y = float(args[1])
        self.saveUndoData('moved', _x, _y)

    def execute(self, undo, *args):
        if undo is not True and undo is not False:
            raise ValueError, "Invalid undo value: " + str(undo)
        _alen = len(args)
        if _alen == 0:
            raise ValueError, "No arguments to execute()"
        _p = self.__p
        _op = args[0]
        if _op == 'moved':        
            if _alen < 3:
                raise ValueError, "Invalid argument count: %d" % _alen
            _x = args[1]
            if not isinstance(_x, float):
                _x = float(args[1])
            _y = args[2]
            if not isinstance(_y, float):
                _y = float(args[2])
            _px = _p.x
            _py = _p.y
            if undo:
                self.saveRedoData('moved', _px, _py)
            else:
                self.saveUndoData('moved', _px, _py)
            self.ignore('moved')
            try:
                if undo:
                    _p.startUndo(True)
                    try:
                        _p.setCoords(_x, _y)
                    finally:
                        _p.endUndo()
                    if abs(_x - _px) > 1e-10 or abs(_y - _py) > 1e-10:
                        _p.sendMessage('moved', _px, _py)
                else:
                    _p.setCoords(_x, _y)
            finally:
                self.receive('moved')
        else:
            raise ValueError, "Unexpected operation: %s" % _op
        
