#ifndef VISUAL_DISPLAY_H
#define VISUAL_DISPLAY_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "cvisual.h"
#include <list>
#include <boost/shared_ptr.hpp>
#include <boost/enable_shared_from_this.hpp>
#include "color.h"
#include "vector.h"
#include "vcache.h"
#include "tmatrix.h"
#include "glcontext.h"
#include "light.h"


namespace visual {

enum stereotype { NONE, ACTIVE, PASSIVE, RED_BLUE, RED_CYAN, YELLOW_BLUE, 
	GREEN_MAGENTA, CROSSEYED };

/* Base type that wraps around a glContext, implemented in glDevice.
 * I think that this object doesn't need to be a virtual base, since it has only
 * ONE implementation.
 */

class Display;
class mouseObject;
class kbObject;
class cursorObject;
class DisplayObject;

class Device 
{
 public:
	Display* display;

	virtual ~Device() {};

	virtual bool show() = 0;
	virtual void hide() = 0;
	virtual void join() = 0;
	virtual bool closed() = 0;
	virtual void frame() = 0;
	virtual std::string info() = 0;
	virtual void onClose(bool quit) = 0;

	virtual tmatrix get_wct() = 0;

	virtual void setX(int) = 0;
	virtual void setY(int) = 0;
	virtual void setWidth(int) = 0;
	virtual void setHeight(int) = 0;
	virtual void setFullScreen(bool) = 0;
	virtual void setStereo( stereotype) = 0;
	virtual void setStereoDepth(float) = 0;
	virtual void setNewmouse(bool) = 0;
	virtual int getX() = 0;
	virtual int getY() = 0;
	virtual int getWidth() = 0;
	virtual int getHeight() = 0;
	virtual bool getFullScreen() const = 0;
	virtual stereotype getStereo() const = 0;
	virtual float getStereoDepth() const = 0;
	virtual glContext& get_context() = 0;
};

class Display : public Cache,
                public boost::enable_shared_from_this<Display> 
{
	boost::python::object self;
	friend class Device;
 public:
	lighting lights;
	static boost::shared_ptr<Display> selected;
	boost::shared_ptr<mouseObject> mouse;
	boost::shared_ptr<kbObject> kb;
	boost::shared_ptr<cursorObject> cursor;

	Display();
	Display( Device& dev);
	~Display();


	// The following functions are called by devices to change the display:
	void setForward( const vector& v) { write_lock L(mtx); forward=v; }
	void setRegion( const vector& x1, const vector& x2) { write_lock L(mtx); region_min=x1; region_max=x2; region_valid=true; }
	void addObject();

	static void shutdown();
	// Destroys all devices of all displays.
	static bool allclosed();
	// Returns true if all devices of all displays are closed.
	static void waitclose();
	// Blocks until allclosed().
	static void internal_shutdown();
	// Simmilar to shutdown(), except done from the render loop.

	// Returns a reference to the currently selected display.
	static boost::python::object get_selected();

	void show();
	// wakes up all devices
	void hide();
	// hides all devices

	// Python type interface
	void py_select( void);
	void py_select( boost::python::object);
	std::string py_info( void);
	vector py_project( vector v) const;
	void exit_on_close( bool code);
	boost::python::list py_objects();
	rgb fgcolor() { return foreground; }

	// Cached attributes
	std::string c_title;
	bool c_title_changed;
	tmatrix model;
	tmatrix imodel;
	tmatrix view;
	tmatrix iview;
	vector c_forward;
	vector c_up;
	vector c_center;
	vector c_extent;
	bool c_uniform;
	bool c_rotation_enabled;
	bool c_zoom_enabled;
	double tanfov;
	rgb bgcolor;

	// We need appends, iteration, and reference-counting semantics.
	std::list< boost::shared_ptr<DisplayObject> > objects;
	// This mutex is used to synchronize operations on the list itself.
	mutex list_mutex;

	// Python getters
	int get_x();
	int get_y();
	int get_width();
	int get_height();
	vector get_extent();
	inline vector& get_up() { return up; }
	inline vector& get_forward(){ return forward; }
	inline vector& get_scale() { /* Cache::read_lock L( mtx); */ return scale; }
	inline vector& get_center() { return center; }
	inline float get_ambient() const { return lights.ambient; }
	std::string get_title() const;
	inline double get_fov() const { return fov; }
	inline rgb get_foreground() { return foreground; }
	inline rgb get_background() { return background; }
	vector get_range();
	inline bool is_selected() { return selected == shared_from_this(); }
	inline double get_maxscale() { return autoscale_max; }
	inline double get_minscale() { return autoscale_min; }
	inline bool is_userspin() const { return rotation_enabled; }
	inline bool is_userzoom() const { return zoom_enabled; }
	inline bool is_autoscale() const { return autoscale_enabled; }
	inline bool is_autocenter() const { return autocenter_enabled; }
	inline bool is_uniform() const { return uniform; }
	inline bool is_visible() { return !device->closed(); }
	inline boost::python::list get_lights() { return lights.as_list(); }
	boost::shared_ptr<kbObject> get_keyboard_object();
	inline boost::shared_ptr<cursorObject> get_cursor_object() { return cursor; }
	boost::shared_ptr<mouseObject> get_mouse();
	std::string get_stereo() const;
	bool get_fullscreen() const;
	float get_stereodepth() const;
	
	// Python setters.  Some of these have locking and/or checking semantics.
	void set_x( int x);
	void set_y( int y);
	void set_width( int width);
	void set_widthd( double width);
	void set_height( int height);
	void set_heightd( double height);
	void set_title( std::string);
	void set_forward( const vector& v);
	void set_scale( const vector& v);
	void set_up( const vector& v);
	void set_fov( const double& _fov);
	void set_uniform( bool);
	void set_foreground( rgb fg);
	void set_background( rgb bg);
	void set_autoscale( bool);
	void set_autocenter( bool);
	void set_range( const vector& v);
	void set_ambient( const float& l);
	void set_userzoom( bool zoom);
	void set_userspin( bool spin);
	void set_maxscale( double scale);
	void set_minscale( double scale);
	void set_visible( bool vis);
	void set_lighting( boost::python::list);
	void set_center( const vector& v);
	void set_fullscreen( bool fs);
	void set_stereo( std::string mode);
	void set_stereodepth( float depth);
	inline void set_range_f( float f) { set_range( vector(f,f,f)); }
	
	boost::python::object getObject() { return self; }
	
 protected:
	std::string title;
	bool title_changed;

	vector center;
	vector forward;
	vector scale;
	vector up;
	rgb foreground;
	rgb background;
	double fov;
	bool uniform;
	bool autoscale_enabled;
	bool autocenter_enabled;
	bool rotation_enabled;
	bool zoom_enabled;
	double autoscale_max; 
	double autoscale_min;
	bool auto_show;

	vector region_min;
	vector region_max;
	bool region_valid;

	double get_tanfov();
	// calculates the field of view ratio actually used
	//   in projection

	void window_changed();
	// called to report that x,y,width, or height have
	//   changed.

	virtual void refreshCache();
	// Called by updateCache() when the cache needs to be refreshed.
      
	Device* device;

	static std::vector<Display*> all;
	int displayID;
};

void display_init_type();

} // !namespace visual

#endif // !VISUAL_DISPLAY_H
