"""
Generators for the small graph atlas.

See
"An Atlas of Graphs" by Ronald C. Read and Robin J. Wilson,
Oxford University Press, 1998.

"""
#    Copyright (C) 2004,2005 by 
#    Aric Hagberg <hagberg@lanl.gov>
#    Dan Schult <dschult@colgate.edu>
#    Pieter Swart <swart@lanl.gov>
#    Distributed under the terms of the GNU Lesser General Public License
#    http://www.gnu.org/copyleft/lesser.html
__author__ = """Pieter Swart (swart@lanl.gov)"""
__date__ = "$Date: 2005-03-30 16:56:28 -0700 (Wed, 30 Mar 2005) $"
__credits__ = """"""
__revision__ = "$Revision: 911 $"
 
from networkx.generators.small import make_small_graph

def graph_atlas_g():
    """
    Return the list [G1,G2,...,G1252] of graphs as named in the Graph Atlas.
    G1,...,G1252 are all graphs with up to 7 nodes.

    The graphs are listed:
      1. in increasing order of number of nodes;
      2. for a fixed number of nodes,
         in increasing order of the number of edges;
      3. for fixed numbers of nodes and edges,
         in increasing order of the degree sequence,
         for example 111223 < 112222;
      4. for fixed degree sequence, in increasing number of automorphisms.

    Note that indexing is set up so that for
    GAG=graph_atlas_g(), then
    G123=GAG[123] and G[0]=empty_graph(0)
    """

    descr_list=[
     ['edgelist', 'G0', 0, []],
     ['edgelist', 'G1', 1, []],
     ['edgelist', 'G2', 2, []],
     ['edgelist', 'G3', 2, [[1, 2]]],
     ['edgelist', 'G4', 3, []],
     ['edgelist', 'G5', 3, [[2, 3]]],
     ['edgelist', 'G6', 3, [[1, 2], [1, 3]]],
     ['edgelist', 'G7', 3, [[1, 2], [1, 3], [2, 3]]],
     ['edgelist', 'G8', 4, []],
     ['edgelist', 'G9', 4, [[4, 3]]],
     ['edgelist', 'G10', 4, [[4, 3], [4, 2]]],
     ['edgelist', 'G11', 4, [[1, 2], [4, 3]]],
     ['edgelist', 'G12', 4, [[4, 3], [2, 3], [4, 2]]],
     ['edgelist', 'G13', 4, [[4, 1], [4, 2], [4, 3]]],
     ['edgelist', 'G14', 4, [[1, 2], [2, 3], [1, 4]]],
     ['edgelist', 'G15', 4, [[4, 3], [2, 3], [4, 2], [4, 1]]],
     ['edgelist', 'G16', 4, [[1, 2], [2, 3], [3, 4], [1, 4]]],
     ['edgelist', 'G17', 4, [[1, 2], [1, 3], [1, 4], [2, 3], [3, 4]]],
     ['edgelist', 'G18', 4, [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3]]],
     ['edgelist', 'G19', 5, []],
     ['edgelist', 'G20', 5, [[5, 4]]],
     ['edgelist', 'G21', 5, [[2, 3], [1, 2]]],
     ['edgelist', 'G22', 5, [[1, 3], [5, 4]]],
     ['edgelist', 'G23', 5, [[2, 3], [1, 2], [3, 1]]],
     ['edgelist', 'G24', 5, [[5, 4], [4, 3], [4, 2]]],
     ['edgelist', 'G25', 5, [[4, 3], [5, 4], [1, 5]]],
     ['edgelist', 'G26', 5, [[2, 3], [1, 2], [5, 4]]],
     ['edgelist', 'G27', 5, [[5, 4], [2, 3], [4, 2], [4, 3]]],
     ['edgelist', 'G28', 5, [[1, 4], [2, 1], [3, 2], [4, 3]]],
     ['edgelist', 'G29', 5, [[5, 4], [5, 1], [5, 2], [5, 3]]],
     ['edgelist', 'G30', 5, [[5, 1], [4, 2], [5, 4], [4, 3]]],
     ['edgelist', 'G31', 5, [[3, 4], [2, 3], [1, 2], [5, 1]]],
     ['edgelist', 'G32', 5, [[2, 3], [1, 2], [3, 1], [5, 4]]],
     ['edgelist', 'G33', 5, [[1, 4], [3, 1], [4, 3], [2, 1], [3, 2]]],
     ['edgelist', 'G34', 5, [[5, 3], [5, 4], [3, 4], [5, 2], [5, 1]]],
     ['edgelist', 'G35', 5, [[1, 2], [2, 3], [3, 4], [1, 5], [1, 3]]],
     ['edgelist', 'G36', 5, [[5, 1], [2, 3], [5, 4], [4, 3], [4, 2]]],
     ['edgelist', 'G37', 5, [[2, 1], [5, 2], [3, 5], [4, 3], [2, 4]]],
     ['edgelist', 'G38', 5, [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5]]],
     ['edgelist', 'G39', 5, [[2, 1], [5, 2], [5, 1], [1, 4], [2, 4], [4, 5]]],
     ['edgelist', 'G40', 5, [[2, 1], [5, 2], [3, 5], [4, 3], [2, 4], [3, 2]]],
     ['edgelist', 'G41', 5, [[2, 1], [5, 2], [3, 5], [4, 3], [2, 4], [4, 5]]],
     ['edgelist', 'G42', 5, [[1, 2], [5, 4], [3, 4], [5, 3], [5, 1], [5, 2]]],
     ['edgelist', 'G43', 5, [[1, 5], [4, 1], [5, 4], [3, 4], [2, 3], [1, 2]]],
     ['edgelist', 'G44', 5, [[3, 2], [1, 3], [4, 1], [2, 4], [5, 2], [1, 5]]],
     ['edgelist',
      'G45',
      5,
      [[5, 1], [2, 3], [5, 4], [4, 3], [4, 2], [5, 2], [3, 5]]],
     ['edgelist',
      'G46',
      5,
      [[5, 2], [3, 5], [4, 3], [2, 4], [4, 5], [1, 4], [5, 1]]],
     ['edgelist',
      'G47',
      5,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2]]],
     ['edgelist',
      'G48',
      5,
      [[3, 2], [1, 3], [4, 1], [2, 4], [5, 2], [1, 5], [3, 5]]],
     ['edgelist',
      'G49',
      5,
      [[2, 1], [5, 2], [3, 5], [4, 3], [2, 4], [5, 1], [4, 5], [1, 4]]],
     ['edgelist',
      'G50',
      5,
      [[1, 2], [2, 3], [3, 4], [1, 4], [5, 1], [5, 2], [5, 3], [5, 4]]],
     ['edgelist',
      'G51',
      5,
      [[1, 2], [4, 5], [1, 4], [1, 5], [2, 3], [2, 4], [2, 5], [3, 4], [3, 5]]],
     ['edgelist',
      'G52',
      5,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5]]],
     ['edgelist', 'G53', 6, []],
     ['edgelist', 'G54', 6, [[6, 5]]],
     ['edgelist', 'G55', 6, [[1, 4], [6, 5]]],
     ['edgelist', 'G56', 6, [[2, 4], [2, 3]]],
     ['edgelist', 'G57', 6, [[2, 4], [3, 2], [4, 3]]],
     ['edgelist', 'G58', 6, [[1, 4], [6, 1], [5, 1]]],
     ['edgelist', 'G59', 6, [[5, 4], [6, 5], [1, 6]]],
     ['edgelist', 'G60', 6, [[5, 4], [6, 2], [6, 3]]],
     ['edgelist', 'G61', 6, [[2, 3], [4, 1], [6, 5]]],
     ['edgelist', 'G62', 6, [[1, 4], [5, 1], [6, 5], [1, 6]]],
     ['edgelist', 'G63', 6, [[4, 1], [6, 4], [5, 6], [1, 5]]],
     ['edgelist', 'G64', 6, [[6, 2], [6, 4], [6, 3], [1, 6]]],
     ['edgelist', 'G65', 6, [[5, 4], [4, 2], [5, 1], [4, 3]]],
     ['edgelist', 'G66', 6, [[1, 3], [2, 4], [3, 2], [6, 4]]],
     ['edgelist', 'G67', 6, [[2, 4], [3, 2], [4, 3], [1, 6]]],
     ['edgelist', 'G68', 6, [[2, 3], [1, 4], [6, 1], [5, 1]]],
     ['edgelist', 'G69', 6, [[5, 6], [2, 3], [1, 6], [4, 5]]],
     ['edgelist', 'G70', 6, [[1, 3], [5, 1], [4, 2], [6, 4]]],
     ['edgelist', 'G71', 6, [[4, 1], [6, 4], [5, 6], [1, 5], [6, 1]]],
     ['edgelist', 'G72', 6, [[6, 4], [4, 2], [4, 3], [5, 4], [5, 6]]],
     ['edgelist', 'G73', 6, [[6, 4], [6, 5], [3, 4], [4, 5], [1, 5]]],
     ['edgelist', 'G74', 6, [[5, 4], [2, 3], [5, 1], [4, 3], [4, 2]]],
     ['edgelist', 'G75', 6, [[2, 5], [4, 5], [5, 1], [3, 2], [4, 3]]],
     ['edgelist', 'G76', 6, [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5]]],
     ['edgelist', 'G77', 6, [[6, 4], [6, 5], [6, 1], [6, 2], [6, 3]]],
     ['edgelist', 'G78', 6, [[2, 5], [6, 2], [2, 1], [3, 2], [3, 4]]],
     ['edgelist', 'G79', 6, [[1, 2], [4, 5], [1, 3], [4, 1], [6, 4]]],
     ['edgelist', 'G80', 6, [[2, 1], [3, 2], [3, 5], [2, 4], [6, 4]]],
     ['edgelist', 'G81', 6, [[5, 4], [1, 6], [5, 1], [4, 3], [4, 2]]],
     ['edgelist', 'G82', 6, [[2, 3], [1, 2], [5, 6], [2, 4], [3, 4]]],
     ['edgelist', 'G83', 6, [[1, 2], [1, 6], [3, 4], [4, 5], [5, 6]]],
     ['edgelist', 'G84', 6, [[5, 4], [6, 2], [6, 3], [1, 4], [5, 1]]],
     ['edgelist', 'G85', 6, [[2, 3], [4, 1], [6, 4], [5, 6], [1, 5]]],
     ['edgelist', 'G86', 6, [[1, 4], [6, 1], [5, 6], [4, 5], [6, 4], [5, 1]]],
     ['edgelist', 'G87', 6, [[2, 5], [3, 5], [5, 1], [3, 4], [4, 2], [4, 5]]],
     ['edgelist', 'G88', 6, [[2, 5], [3, 5], [5, 1], [3, 2], [4, 2], [3, 4]]],
     ['edgelist', 'G89', 6, [[3, 1], [6, 5], [5, 4], [6, 4], [5, 1], [3, 5]]],
     ['edgelist', 'G90', 6, [[4, 3], [5, 4], [1, 5], [2, 1], [3, 2], [1, 4]]],
     ['edgelist', 'G91', 6, [[5, 2], [4, 2], [5, 3], [4, 3], [3, 1], [2, 1]]],
     ['edgelist', 'G92', 6, [[6, 3], [6, 4], [6, 5], [4, 5], [6, 2], [6, 1]]],
     ['edgelist', 'G93', 6, [[5, 4], [5, 3], [5, 1], [2, 5], [4, 1], [6, 4]]],
     ['edgelist', 'G94', 6, [[5, 4], [4, 6], [6, 5], [6, 2], [4, 3], [5, 1]]],
     ['edgelist', 'G95', 6, [[5, 3], [2, 3], [5, 4], [5, 2], [5, 1], [1, 6]]],
     ['edgelist', 'G96', 6, [[2, 3], [4, 2], [1, 4], [3, 1], [5, 1], [6, 1]]],
     ['edgelist', 'G97', 6, [[3, 1], [5, 3], [2, 5], [3, 2], [4, 2], [6, 4]]],
     ['edgelist', 'G98', 6, [[2, 3], [4, 2], [1, 4], [3, 1], [5, 1], [6, 4]]],
     ['edgelist', 'G99', 6, [[6, 4], [3, 6], [3, 1], [5, 3], [5, 4], [4, 2]]],
     ['edgelist', 'G100', 6, [[1, 3], [4, 5], [2, 1], [6, 4], [5, 6], [4, 1]]],
     ['edgelist', 'G101', 6, [[2, 3], [4, 1], [6, 4], [5, 6], [1, 5], [6, 1]]],
     ['edgelist', 'G102', 6, [[5, 4], [2, 3], [5, 1], [4, 3], [4, 2], [6, 1]]],
     ['edgelist', 'G103', 6, [[2, 5], [3, 5], [5, 1], [1, 6], [4, 2], [3, 4]]],
     ['edgelist', 'G104', 6, [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 6]]],
     ['edgelist', 'G105', 6, [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6]]],
     ['edgelist', 'G106', 6, [[2, 4], [3, 2], [4, 3], [1, 5], [6, 1], [5, 6]]],
     ['edgelist',
      'G107',
      6,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [1, 6]]],
     ['edgelist',
      'G108',
      6,
      [[2, 5], [3, 5], [3, 2], [4, 2], [3, 4], [3, 1], [1, 2]]],
     ['edgelist',
      'G109',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2]]],
     ['edgelist',
      'G110',
      6,
      [[1, 2], [4, 3], [1, 3], [4, 1], [4, 2], [6, 2], [6, 3]]],
     ['edgelist',
      'G111',
      6,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [4, 5]]],
     ['edgelist',
      'G112',
      6,
      [[2, 1], [5, 2], [3, 5], [4, 3], [6, 2], [3, 6], [2, 3]]],
     ['edgelist',
      'G113',
      6,
      [[1, 5], [3, 1], [2, 3], [4, 2], [6, 4], [4, 1], [3, 4]]],
     ['edgelist',
      'G114',
      6,
      [[2, 5], [3, 5], [3, 4], [3, 2], [4, 2], [5, 6], [1, 5]]],
     ['edgelist',
      'G115',
      6,
      [[2, 1], [5, 2], [3, 5], [4, 3], [6, 2], [3, 6], [5, 6]]],
     ['edgelist',
      'G116',
      6,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [6, 5]]],
     ['edgelist',
      'G117',
      6,
      [[1, 6], [5, 1], [6, 5], [1, 3], [4, 1], [4, 3], [1, 2]]],
     ['edgelist',
      'G118',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 6], [5, 2]]],
     ['edgelist',
      'G119',
      6,
      [[1, 2], [5, 1], [2, 5], [1, 3], [4, 1], [4, 3], [4, 6]]],
     ['edgelist',
      'G120',
      6,
      [[2, 5], [3, 5], [5, 1], [1, 6], [4, 2], [3, 4], [4, 5]]],
     ['edgelist',
      'G121',
      6,
      [[3, 1], [4, 3], [5, 4], [6, 5], [3, 6], [2, 3], [5, 2]]],
     ['edgelist',
      'G122',
      6,
      [[2, 6], [1, 2], [5, 1], [4, 5], [3, 4], [2, 3], [1, 4]]],
     ['edgelist',
      'G123',
      6,
      [[2, 5], [3, 5], [5, 1], [1, 6], [4, 2], [3, 4], [3, 2]]],
     ['edgelist',
      'G124',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [1, 3], [6, 2]]],
     ['edgelist',
      'G125',
      6,
      [[3, 1], [5, 2], [2, 3], [6, 5], [3, 6], [4, 2], [6, 4]]],
     ['edgelist',
      'G126',
      6,
      [[6, 1], [4, 6], [3, 4], [1, 3], [2, 4], [5, 2], [4, 5]]],
     ['edgelist',
      'G127',
      6,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [3, 4]]],
     ['edgelist',
      'G128',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [1, 4]]],
     ['edgelist',
      'G129',
      6,
      [[5, 4], [1, 5], [2, 1], [3, 2], [4, 3], [1, 6], [6, 4]]],
     ['edgelist',
      'G130',
      6,
      [[2, 3], [1, 2], [3, 1], [4, 1], [5, 4], [6, 5], [4, 6]]],
     ['edgelist',
      'G131',
      6,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2]]],
     ['edgelist',
      'G132',
      6,
      [[1, 2], [2, 3], [3, 4], [1, 4], [5, 1], [5, 2], [5, 3], [5, 4]]],
     ['edgelist',
      'G133',
      6,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [6, 1], [1, 5]]],
     ['edgelist',
      'G134',
      6,
      [[2, 3], [4, 2], [1, 4], [2, 1], [3, 1], [4, 3], [6, 4], [5, 1]]],
     ['edgelist',
      'G135',
      6,
      [[1, 2], [3, 5], [1, 3], [6, 3], [4, 2], [4, 3], [3, 2], [5, 2]]],
     ['edgelist',
      'G136',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [2, 6]]],
     ['edgelist',
      'G137',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 5]]],
     ['edgelist',
      'G138',
      6,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [3, 2], [6, 2]]],
     ['edgelist',
      'G139',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 1]]],
     ['edgelist',
      'G140',
      6,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [4, 1], [6, 2]]],
     ['edgelist',
      'G141',
      6,
      [[3, 1], [4, 3], [5, 4], [6, 5], [3, 6], [2, 3], [5, 2], [6, 4]]],
     ['edgelist',
      'G142',
      6,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [1, 6], [6, 5]]],
     ['edgelist',
      'G143',
      6,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [6, 2], [6, 4]]],
     ['edgelist',
      'G144',
      6,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [4, 5]]],
     ['edgelist',
      'G145',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 1], [6, 3], [1, 3]]],
     ['edgelist',
      'G146',
      6,
      [[2, 6], [5, 2], [1, 5], [6, 1], [3, 6], [5, 3], [4, 5], [6, 4]]],
     ['edgelist',
      'G147',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 3]]],
     ['edgelist',
      'G148',
      6,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [2, 5], [1, 2]]],
     ['edgelist',
      'G149',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1]]],
     ['edgelist',
      'G150',
      6,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [3, 2]]],
     ['edgelist',
      'G151',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [5, 6], [6, 4], [2, 6]]],
     ['edgelist',
      'G152',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 2]]],
     ['edgelist',
      'G153',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 6], [6, 3], [6, 1]]],
     ['edgelist',
      'G154',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [5, 2], [6, 3]]],
     ['edgelist',
      'G155',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [3, 5], [1, 4]]],
     ['edgelist',
      'G156',
      6,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2], [5, 3]]],
     ['edgelist',
      'G157',
      6,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2], [1, 5]]],
     ['edgelist',
      'G158',
      6,
      [[1, 2], [2, 3], [3, 4], [1, 4], [5, 1], [5, 2], [5, 3], [5, 4], [5, 6]]],
     ['edgelist',
      'G159',
      6,
      [[3, 1], [5, 2], [2, 3], [6, 5], [3, 6], [4, 2], [6, 4], [4, 3], [5, 4]]],
     ['edgelist',
      'G160',
      6,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2], [5, 6]]],
     ['edgelist',
      'G161',
      6,
      [[2, 6], [5, 2], [1, 5], [6, 1], [3, 6], [5, 3], [4, 5], [6, 4], [5, 6]]],
     ['edgelist',
      'G162',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [3, 6], [1, 6], [3, 1], [4, 1]]],
     ['edgelist',
      'G163',
      6,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [1, 5], [2, 1], [5, 2]]],
     ['edgelist',
      'G164',
      6,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [5, 2], [2, 1], [6, 2]]],
     ['edgelist',
      'G165',
      6,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [6, 5], [5, 1], [6, 1]]],
     ['edgelist',
      'G166',
      6,
      [[5, 4], [1, 5], [2, 1], [3, 2], [4, 3], [1, 6], [6, 4], [1, 4], [2, 6]]],
     ['edgelist',
      'G167',
      6,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [4, 3], [1, 4], [5, 1]]],
     ['edgelist',
      'G168',
      6,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [4, 3], [1, 4], [3, 5]]],
     ['edgelist',
      'G169',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [3, 6], [1, 6], [3, 1], [6, 2]]],
     ['edgelist',
      'G170',
      6,
      [[2, 6], [5, 2], [1, 5], [6, 1], [3, 6], [5, 3], [4, 5], [6, 4], [3, 1]]],
     ['edgelist',
      'G171',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 1], [6, 5], [6, 3], [6, 4]]],
     ['edgelist',
      'G172',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1], [6, 2]]],
     ['edgelist',
      'G173',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [1, 4], [5, 3], [6, 3]]],
     ['edgelist',
      'G174',
      6,
      [[3, 4], [1, 3], [4, 1], [5, 4], [2, 5], [6, 2], [5, 6], [2, 1], [6, 3]]],
     ['edgelist',
      'G175',
      6,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [1, 4], [6, 3], [5, 2]]],
     ['edgelist',
      'G176',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 3]]],
     ['edgelist',
      'G177',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [5, 6]]],
     ['edgelist',
      'G178',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 6]]],
     ['edgelist',
      'G179',
      6,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [5, 6],
       [2, 1]]],
     ['edgelist',
      'G180',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [6, 5],
       [4, 6],
       [2, 6]]],
     ['edgelist',
      'G181',
      6,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [5, 1],
       [3, 5]]],
     ['edgelist',
      'G182',
      6,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 3]]],
     ['edgelist',
      'G183',
      6,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [5, 6],
       [4, 5],
       [2, 4],
       [6, 2],
       [3, 4],
       [2, 3]]],
     ['edgelist',
      'G184',
      6,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [6, 3]]],
     ['edgelist',
      'G185',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [5, 2]]],
     ['edgelist',
      'G186',
      6,
      [[1, 2],
       [3, 5],
       [1, 3],
       [5, 6],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4]]],
     ['edgelist',
      'G187',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 5]]],
     ['edgelist',
      'G188',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [1, 3],
       [2, 4],
       [6, 2]]],
     ['edgelist',
      'G189',
      6,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [4, 3],
       [1, 4]]],
     ['edgelist',
      'G190',
      6,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [6, 4],
       [3, 6],
       [2, 1]]],
     ['edgelist',
      'G191',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 3],
       [2, 6]]],
     ['edgelist',
      'G192',
      6,
      [[1, 2],
       [3, 5],
       [1, 3],
       [3, 2],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4]]],
     ['edgelist',
      'G193',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1],
       [5, 6]]],
     ['edgelist',
      'G194',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [1, 3]]],
     ['edgelist',
      'G195',
      6,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [6, 4],
       [3, 6],
       [2, 1],
       [6, 2]]],
     ['edgelist',
      'G196',
      6,
      [[2, 4],
       [5, 2],
       [4, 5],
       [3, 4],
       [1, 3],
       [5, 1],
       [6, 5],
       [3, 6],
       [5, 3],
       [1, 6],
       [2, 6]]],
     ['edgelist',
      'G197',
      6,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2]]],
     ['edgelist',
      'G198',
      6,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3]]],
     ['edgelist',
      'G199',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [1, 4]]],
     ['edgelist',
      'G200',
      6,
      [[1, 2],
       [2, 3],
       [1, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [5, 4]]],
     ['edgelist',
      'G201',
      6,
      [[4, 3],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [3, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2],
       [1, 5]]],
     ['edgelist',
      'G202',
      6,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3],
       [5, 6]]],
     ['edgelist',
      'G203',
      6,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2],
       [3, 4]]],
     ['edgelist',
      'G204',
      6,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 3],
       [4, 2],
       [5, 1],
       [3, 5],
       [6, 2],
       [1, 6],
       [5, 6],
       [4, 5],
       [6, 4]]],
     ['edgelist',
      'G205',
      6,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2],
       [3, 4],
       [1, 5]]],
     ['edgelist',
      'G206',
      6,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 3],
       [4, 2],
       [5, 1],
       [3, 5],
       [6, 2],
       [1, 6],
       [5, 6],
       [4, 5],
       [6, 4],
       [4, 1]]],
     ['edgelist',
      'G207',
      6,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [2, 4],
       [3, 1],
       [5, 1],
       [6, 4]]],
     ['edgelist',
      'G208',
      6,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [5, 6]]],
     ['edgelist', 'G209', 7, []],
     ['edgelist', 'G210', 7, [[7, 6]]],
     ['edgelist', 'G211', 7, [[3, 4], [2, 3]]],
     ['edgelist', 'G212', 7, [[6, 5], [7, 1]]],
     ['edgelist', 'G213', 7, [[1, 5], [5, 3], [3, 1]]],
     ['edgelist', 'G214', 7, [[1, 2], [1, 7], [1, 6]]],
     ['edgelist', 'G215', 7, [[6, 5], [7, 1], [6, 7]]],
     ['edgelist', 'G216', 7, [[4, 3], [2, 3], [6, 7]]],
     ['edgelist', 'G217', 7, [[4, 2], [6, 7], [1, 5]]],
     ['edgelist', 'G218', 7, [[3, 6], [7, 3], [6, 7], [2, 3]]],
     ['edgelist', 'G219', 7, [[2, 3], [5, 2], [6, 5], [3, 6]]],
     ['edgelist', 'G220', 7, [[2, 1], [6, 2], [2, 3], [5, 2]]],
     ['edgelist', 'G221', 7, [[2, 1], [3, 2], [6, 3], [7, 3]]],
     ['edgelist', 'G222', 7, [[4, 5], [3, 4], [2, 3], [1, 2]]],
     ['edgelist', 'G223', 7, [[5, 3], [1, 5], [3, 1], [6, 7]]],
     ['edgelist', 'G224', 7, [[1, 2], [7, 1], [1, 6], [5, 3]]],
     ['edgelist', 'G225', 7, [[4, 2], [6, 5], [7, 6], [1, 7]]],
     ['edgelist', 'G226', 7, [[1, 5], [4, 1], [3, 6], [7, 3]]],
     ['edgelist', 'G227', 7, [[3, 4], [2, 3], [7, 1], [6, 5]]],
     ['edgelist', 'G228', 7, [[1, 5], [4, 1], [2, 4], [5, 2], [2, 1]]],
     ['edgelist', 'G229', 7, [[3, 6], [7, 3], [6, 7], [5, 3], [4, 3]]],
     ['edgelist', 'G230', 7, [[5, 3], [5, 1], [3, 1], [6, 5], [7, 1]]],
     ['edgelist', 'G231', 7, [[3, 6], [7, 3], [6, 7], [2, 3], [1, 2]]],
     ['edgelist', 'G232', 7, [[5, 2], [1, 5], [4, 1], [2, 4], [3, 2]]],
     ['edgelist', 'G233', 7, [[2, 3], [1, 2], [5, 1], [4, 5], [3, 4]]],
     ['edgelist', 'G234', 7, [[6, 2], [6, 1], [3, 6], [4, 6], [5, 6]]],
     ['edgelist', 'G235', 7, [[2, 6], [7, 2], [2, 1], [3, 2], [4, 3]]],
     ['edgelist', 'G236', 7, [[2, 6], [5, 2], [3, 4], [7, 3], [3, 2]]],
     ['edgelist', 'G237', 7, [[2, 6], [7, 2], [2, 3], [3, 4], [5, 4]]],
     ['edgelist', 'G238', 7, [[3, 2], [4, 3], [5, 4], [6, 5], [4, 7]]],
     ['edgelist', 'G239', 7, [[7, 6], [3, 7], [2, 3], [6, 3], [4, 5]]],
     ['edgelist', 'G240', 7, [[5, 4], [6, 5], [7, 6], [1, 7], [2, 1]]],
     ['edgelist', 'G241', 7, [[1, 5], [4, 1], [3, 6], [7, 3], [6, 7]]],
     ['edgelist', 'G242', 7, [[5, 2], [6, 3], [7, 6], [4, 7], [3, 4]]],
     ['edgelist', 'G243', 7, [[2, 5], [4, 2], [2, 1], [3, 2], [7, 6]]],
     ['edgelist', 'G244', 7, [[1, 5], [4, 1], [2, 1], [3, 2], [7, 6]]],
     ['edgelist', 'G245', 7, [[1, 5], [4, 1], [3, 2], [6, 3], [7, 3]]],
     ['edgelist', 'G246', 7, [[7, 6], [4, 5], [3, 4], [2, 3], [1, 2]]],
     ['edgelist', 'G247', 7, [[3, 4], [2, 3], [7, 1], [6, 7], [6, 5]]],
     ['edgelist', 'G248', 7, [[1, 2], [5, 7], [6, 5], [4, 3], [7, 6]]],
     ['edgelist', 'G249', 7, [[2, 6], [7, 2], [6, 7], [3, 6], [2, 3], [7, 3]]],
     ['edgelist', 'G250', 7, [[2, 5], [4, 2], [3, 4], [5, 3], [2, 1], [3, 2]]],
     ['edgelist', 'G251', 7, [[1, 5], [4, 1], [2, 4], [3, 2], [2, 5], [4, 5]]],
     ['edgelist', 'G252', 7, [[6, 3], [5, 6], [3, 5], [4, 3], [7, 4], [3, 7]]],
     ['edgelist', 'G253', 7, [[2, 3], [5, 2], [6, 5], [3, 6], [1, 2], [5, 1]]],
     ['edgelist', 'G254', 7, [[2, 3], [6, 2], [5, 6], [3, 5], [1, 3], [6, 1]]],
     ['edgelist', 'G255', 7, [[3, 6], [7, 3], [6, 7], [3, 5], [2, 3], [4, 3]]],
     ['edgelist', 'G256', 7, [[2, 5], [4, 2], [3, 4], [2, 3], [3, 6], [7, 3]]],
     ['edgelist', 'G257', 7, [[6, 5], [7, 6], [2, 7], [6, 2], [4, 7], [1, 2]]],
     ['edgelist', 'G258', 7, [[7, 6], [2, 7], [6, 2], [4, 2], [1, 4], [2, 5]]],
     ['edgelist', 'G259', 7, [[1, 5], [4, 1], [3, 4], [5, 3], [3, 6], [7, 3]]],
     ['edgelist', 'G260', 7, [[2, 5], [4, 2], [3, 4], [2, 3], [3, 6], [7, 6]]],
     ['edgelist', 'G261', 7, [[3, 4], [2, 3], [4, 7], [6, 5], [7, 6], [6, 3]]],
     ['edgelist', 'G262', 7, [[3, 6], [7, 3], [6, 7], [2, 5], [4, 2], [3, 2]]],
     ['edgelist', 'G263', 7, [[5, 6], [1, 5], [4, 1], [3, 4], [5, 3], [7, 4]]],
     ['edgelist', 'G264', 7, [[1, 5], [4, 1], [2, 4], [7, 6], [2, 5], [2, 1]]],
     ['edgelist', 'G265', 7, [[2, 5], [4, 2], [3, 4], [6, 3], [7, 6], [3, 7]]],
     ['edgelist', 'G266', 7, [[7, 4], [6, 7], [5, 6], [2, 5], [3, 2], [6, 3]]],
     ['edgelist', 'G267', 7, [[2, 1], [4, 2], [7, 4], [6, 7], [5, 6], [2, 5]]],
     ['edgelist', 'G268', 7, [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6]]],
     ['edgelist', 'G269', 7, [[1, 5], [4, 1], [5, 4], [3, 6], [7, 3], [6, 7]]],
     ['edgelist', 'G270', 7, [[7, 4], [1, 7], [7, 3], [6, 7], [7, 2], [5, 7]]],
     ['edgelist', 'G271', 7, [[3, 5], [6, 3], [3, 4], [7, 3], [2, 3], [2, 1]]],
     ['edgelist', 'G272', 7, [[2, 1], [3, 2], [6, 3], [2, 5], [4, 2], [7, 3]]],
     ['edgelist', 'G273', 7, [[2, 1], [3, 2], [4, 7], [2, 4], [5, 2], [6, 5]]],
     ['edgelist', 'G274', 7, [[2, 1], [3, 2], [6, 3], [7, 6], [2, 5], [4, 2]]],
     ['edgelist', 'G275', 7, [[2, 1], [3, 5], [6, 3], [7, 6], [3, 7], [4, 3]]],
     ['edgelist', 'G276', 7, [[5, 1], [2, 5], [4, 2], [3, 2], [6, 3], [7, 3]]],
     ['edgelist', 'G277', 7, [[7, 6], [2, 3], [1, 2], [3, 1], [4, 3], [1, 5]]],
     ['edgelist', 'G278', 7, [[1, 5], [4, 1], [2, 1], [3, 2], [6, 3], [7, 3]]],
     ['edgelist', 'G279', 7, [[2, 1], [4, 2], [7, 4], [3, 7], [5, 2], [6, 5]]],
     ['edgelist', 'G280', 7, [[3, 6], [7, 3], [5, 3], [2, 5], [4, 2], [1, 4]]],
     ['edgelist', 'G281', 7, [[1, 5], [4, 1], [3, 4], [5, 3], [2, 3], [7, 6]]],
     ['edgelist', 'G282', 7, [[1, 5], [4, 1], [3, 2], [6, 3], [7, 6], [3, 7]]],
     ['edgelist', 'G283', 7, [[4, 5], [2, 1], [3, 2], [6, 3], [7, 6], [3, 7]]],
     ['edgelist', 'G284', 7, [[5, 6], [1, 5], [4, 1], [7, 4], [2, 1], [3, 2]]],
     ['edgelist', 'G285', 7, [[3, 6], [7, 3], [6, 7], [2, 5], [4, 2], [2, 1]]],
     ['edgelist', 'G286', 7, [[5, 6], [4, 5], [3, 4], [2, 3], [1, 2], [7, 1]]],
     ['edgelist', 'G287', 7, [[7, 5], [6, 7], [5, 6], [3, 4], [2, 3], [1, 2]]],
     ['edgelist', 'G288', 7, [[1, 2], [5, 1], [3, 4], [6, 3], [7, 6], [4, 7]]],
     ['edgelist', 'G289', 7, [[2, 3], [1, 2], [5, 1], [4, 5], [3, 4], [7, 6]]],
     ['edgelist',
      'G290',
      7,
      [[2, 5], [4, 2], [3, 4], [5, 3], [2, 1], [3, 2], [4, 5]]],
     ['edgelist',
      'G291',
      7,
      [[2, 3], [6, 2], [5, 6], [3, 5], [1, 3], [6, 1], [6, 3]]],
     ['edgelist',
      'G292',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2]]],
     ['edgelist',
      'G293',
      7,
      [[2, 3], [6, 2], [5, 6], [3, 5], [1, 3], [6, 1], [2, 1]]],
     ['edgelist',
      'G294',
      7,
      [[1, 5], [4, 1], [3, 4], [5, 3], [3, 6], [7, 3], [3, 1]]],
     ['edgelist',
      'G295',
      7,
      [[2, 5], [4, 2], [3, 4], [5, 3], [2, 1], [3, 2], [3, 7]]],
     ['edgelist',
      'G296',
      7,
      [[2, 5], [4, 2], [3, 4], [5, 3], [2, 1], [4, 5], [7, 4]]],
     ['edgelist',
      'G297',
      7,
      [[1, 5], [4, 1], [3, 4], [5, 3], [3, 6], [7, 3], [4, 5]]],
     ['edgelist',
      'G298',
      7,
      [[1, 5], [4, 1], [2, 4], [4, 7], [2, 5], [2, 1], [6, 5]]],
     ['edgelist',
      'G299',
      7,
      [[1, 5], [4, 1], [2, 4], [7, 6], [2, 5], [2, 1], [4, 5]]],
     ['edgelist',
      'G300',
      7,
      [[6, 3], [5, 6], [3, 5], [4, 3], [7, 4], [3, 7], [3, 2]]],
     ['edgelist',
      'G301',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [1, 3], [3, 6]]],
     ['edgelist',
      'G302',
      7,
      [[6, 3], [5, 6], [3, 5], [4, 3], [7, 4], [3, 7], [4, 2]]],
     ['edgelist',
      'G303',
      7,
      [[2, 5], [4, 2], [3, 4], [5, 3], [3, 1], [3, 2], [7, 1]]],
     ['edgelist',
      'G304',
      7,
      [[2, 3], [6, 2], [5, 6], [3, 5], [1, 3], [6, 1], [4, 6]]],
     ['edgelist',
      'G305',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [1, 3], [4, 6]]],
     ['edgelist',
      'G306',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [1, 3], [2, 6]]],
     ['edgelist',
      'G307',
      7,
      [[4, 3], [5, 4], [4, 6], [3, 5], [6, 3], [7, 2], [7, 5]]],
     ['edgelist',
      'G308',
      7,
      [[2, 3], [6, 2], [5, 6], [3, 5], [1, 3], [6, 1], [1, 4]]],
     ['edgelist',
      'G309',
      7,
      [[4, 5], [2, 4], [3, 2], [7, 3], [6, 7], [2, 6], [5, 2]]],
     ['edgelist',
      'G310',
      7,
      [[1, 2], [5, 1], [2, 5], [3, 2], [4, 3], [6, 4], [5, 6]]],
     ['edgelist',
      'G311',
      7,
      [[7, 4], [6, 7], [2, 6], [3, 2], [4, 3], [5, 3], [6, 5]]],
     ['edgelist',
      'G312',
      7,
      [[2, 3], [5, 2], [6, 5], [7, 6], [4, 7], [3, 4], [6, 3]]],
     ['edgelist',
      'G313',
      7,
      [[5, 2], [4, 5], [2, 4], [3, 2], [7, 3], [6, 7], [3, 6]]],
     ['edgelist',
      'G314',
      7,
      [[4, 1], [7, 4], [1, 7], [2, 1], [1, 3], [6, 1], [1, 5]]],
     ['edgelist',
      'G315',
      7,
      [[2, 6], [7, 2], [2, 3], [4, 2], [5, 4], [2, 5], [5, 1]]],
     ['edgelist',
      'G316',
      7,
      [[6, 1], [7, 6], [1, 7], [6, 3], [2, 6], [7, 4], [5, 7]]],
     ['edgelist',
      'G317',
      7,
      [[5, 2], [1, 5], [2, 1], [3, 2], [1, 4], [7, 1], [5, 6]]],
     ['edgelist',
      'G318',
      7,
      [[6, 3], [7, 6], [3, 7], [3, 5], [4, 3], [2, 1], [3, 2]]],
     ['edgelist',
      'G319',
      7,
      [[5, 2], [1, 5], [4, 1], [2, 4], [3, 2], [2, 6], [7, 2]]],
     ['edgelist',
      'G320',
      7,
      [[2, 1], [5, 2], [1, 5], [6, 5], [3, 2], [4, 3], [7, 2]]],
     ['edgelist',
      'G321',
      7,
      [[1, 2], [5, 1], [2, 5], [3, 2], [4, 3], [6, 5], [7, 5]]],
     ['edgelist',
      'G322',
      7,
      [[3, 4], [6, 3], [7, 6], [4, 7], [2, 3], [5, 6], [1, 6]]],
     ['edgelist',
      'G323',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [2, 1], [3, 2], [7, 6]]],
     ['edgelist',
      'G324',
      7,
      [[3, 6], [7, 3], [6, 7], [5, 3], [2, 3], [1, 2], [4, 2]]],
     ['edgelist',
      'G325',
      7,
      [[3, 6], [7, 3], [5, 3], [2, 5], [4, 2], [3, 4], [1, 2]]],
     ['edgelist',
      'G326',
      7,
      [[7, 3], [6, 7], [3, 6], [2, 3], [1, 2], [5, 2], [4, 2]]],
     ['edgelist',
      'G327',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [6, 5], [3, 2], [7, 4]]],
     ['edgelist',
      'G328',
      7,
      [[3, 6], [7, 3], [6, 7], [5, 6], [4, 7], [2, 3], [1, 2]]],
     ['edgelist',
      'G329',
      7,
      [[3, 6], [7, 3], [2, 5], [2, 3], [1, 2], [5, 1], [1, 4]]],
     ['edgelist',
      'G330',
      7,
      [[7, 6], [2, 3], [5, 2], [1, 5], [4, 1], [2, 4], [4, 5]]],
     ['edgelist',
      'G331',
      7,
      [[5, 2], [1, 5], [2, 1], [4, 7], [3, 4], [1, 3], [6, 1]]],
     ['edgelist',
      'G332',
      7,
      [[5, 2], [1, 5], [4, 1], [2, 4], [3, 2], [6, 3], [7, 2]]],
     ['edgelist',
      'G333',
      7,
      [[5, 2], [1, 5], [2, 1], [3, 4], [1, 3], [6, 1], [7, 6]]],
     ['edgelist',
      'G334',
      7,
      [[1, 2], [6, 1], [7, 6], [4, 7], [3, 4], [1, 3], [5, 1]]],
     ['edgelist',
      'G335',
      7,
      [[2, 1], [5, 2], [3, 5], [4, 3], [5, 4], [1, 5], [7, 6]]],
     ['edgelist',
      'G336',
      7,
      [[4, 7], [3, 4], [2, 3], [1, 2], [5, 1], [2, 5], [6, 5]]],
     ['edgelist',
      'G337',
      7,
      [[2, 1], [6, 2], [7, 6], [3, 7], [2, 3], [4, 3], [5, 4]]],
     ['edgelist',
      'G338',
      7,
      [[3, 4], [2, 3], [1, 2], [5, 1], [6, 5], [7, 6], [5, 2]]],
     ['edgelist',
      'G339',
      7,
      [[6, 3], [7, 6], [3, 7], [2, 3], [5, 2], [1, 5], [4, 2]]],
     ['edgelist',
      'G340',
      7,
      [[3, 4], [2, 3], [1, 2], [5, 1], [6, 5], [7, 6], [6, 3]]],
     ['edgelist',
      'G341',
      7,
      [[2, 5], [1, 2], [3, 1], [4, 3], [6, 4], [1, 6], [7, 4]]],
     ['edgelist',
      'G342',
      7,
      [[3, 2], [4, 3], [7, 4], [6, 7], [1, 6], [3, 1], [6, 5]]],
     ['edgelist',
      'G343',
      7,
      [[6, 3], [7, 6], [3, 7], [2, 3], [1, 2], [5, 1], [4, 1]]],
     ['edgelist',
      'G344',
      7,
      [[5, 2], [1, 5], [4, 1], [2, 4], [3, 2], [6, 3], [7, 3]]],
     ['edgelist',
      'G345',
      7,
      [[2, 1], [3, 2], [6, 3], [5, 6], [1, 5], [5, 2], [7, 4]]],
     ['edgelist',
      'G346',
      7,
      [[3, 6], [7, 3], [1, 5], [4, 1], [2, 4], [5, 2], [2, 1]]],
     ['edgelist',
      'G347',
      7,
      [[7, 6], [1, 5], [4, 1], [2, 4], [5, 2], [3, 5], [4, 3]]],
     ['edgelist',
      'G348',
      7,
      [[3, 2], [6, 3], [5, 6], [1, 5], [4, 1], [7, 4], [3, 7]]],
     ['edgelist',
      'G349',
      7,
      [[5, 1], [4, 5], [2, 4], [3, 2], [6, 3], [7, 6], [3, 7]]],
     ['edgelist',
      'G350',
      7,
      [[7, 6], [3, 7], [2, 3], [5, 2], [1, 5], [4, 1], [2, 4]]],
     ['edgelist',
      'G351',
      7,
      [[5, 2], [1, 5], [3, 1], [4, 3], [7, 4], [6, 7], [1, 6]]],
     ['edgelist',
      'G352',
      7,
      [[1, 5], [4, 1], [5, 4], [3, 2], [6, 3], [7, 6], [3, 7]]],
     ['edgelist',
      'G353',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [6, 7], [1, 7]]],
     ['edgelist',
      'G354',
      7,
      [[2, 1], [5, 2], [1, 5], [6, 3], [7, 6], [4, 7], [3, 4]]],
     ['edgelist',
      'G355',
      7,
      [[1, 2], [5, 1], [6, 5], [3, 6], [2, 3], [6, 2], [5, 2], [3, 5]]],
     ['edgelist',
      'G356',
      7,
      [[5, 2], [6, 5], [3, 6], [2, 3], [1, 2], [6, 1], [1, 5], [3, 1]]],
     ['edgelist',
      'G357',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [2, 1], [4, 5], [6, 2], [7, 2]]],
     ['edgelist',
      'G358',
      7,
      [[5, 2], [6, 5], [3, 6], [2, 3], [6, 2], [7, 6], [3, 5], [4, 3]]],
     ['edgelist',
      'G359',
      7,
      [[2, 4], [1, 2], [5, 1], [3, 5], [2, 3], [5, 2], [6, 5], [2, 6]]],
     ['edgelist',
      'G360',
      7,
      [[3, 1], [4, 3], [7, 4], [6, 7], [1, 6], [4, 1], [1, 7], [5, 1]]],
     ['edgelist',
      'G361',
      7,
      [[2, 1], [3, 2], [6, 3], [5, 6], [1, 5], [3, 1], [6, 1], [7, 6]]],
     ['edgelist',
      'G362',
      7,
      [[2, 1], [3, 2], [4, 3], [2, 4], [5, 4], [3, 5], [6, 3], [4, 6]]],
     ['edgelist',
      'G363',
      7,
      [[3, 1], [4, 3], [7, 4], [6, 7], [1, 6], [4, 1], [7, 1], [5, 6]]],
     ['edgelist',
      'G364',
      7,
      [[2, 1], [3, 2], [5, 4], [2, 6], [5, 2], [3, 5], [6, 3], [4, 6]]],
     ['edgelist',
      'G365',
      7,
      [[4, 6], [3, 2], [5, 4], [2, 6], [5, 2], [3, 5], [6, 3], [5, 7]]],
     ['edgelist',
      'G366',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [2, 1], [4, 5], [3, 2], [6, 3]]],
     ['edgelist',
      'G367',
      7,
      [[4, 6], [3, 2], [5, 4], [2, 6], [5, 2], [3, 5], [6, 3], [1, 4]]],
     ['edgelist',
      'G368',
      7,
      [[5, 1], [3, 5], [1, 3], [4, 1], [3, 4], [6, 3], [7, 6], [3, 7]]],
     ['edgelist',
      'G369',
      7,
      [[4, 3], [7, 4], [6, 7], [3, 6], [1, 3], [6, 1], [5, 6], [3, 5]]],
     ['edgelist',
      'G370',
      7,
      [[1, 6], [5, 1], [3, 5], [6, 3], [2, 6], [5, 2], [4, 5], [6, 4]]],
     ['edgelist',
      'G371',
      7,
      [[3, 4], [2, 3], [5, 2], [6, 5], [2, 6], [6, 3], [7, 6], [4, 7]]],
     ['edgelist',
      'G372',
      7,
      [[6, 3], [5, 6], [1, 5], [4, 1], [7, 4], [3, 7], [5, 3], [4, 3]]],
     ['edgelist',
      'G373',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [3, 5], [4, 3], [6, 5], [3, 6]]],
     ['edgelist',
      'G374',
      7,
      [[6, 7], [3, 6], [7, 3], [4, 3], [5, 4], [1, 5], [4, 1], [3, 5]]],
     ['edgelist',
      'G375',
      7,
      [[2, 1], [6, 1], [4, 3], [2, 4], [6, 3], [7, 2], [7, 3], [7, 6]]],
     ['edgelist',
      'G376',
      7,
      [[6, 5], [7, 6], [4, 7], [1, 4], [5, 1], [3, 5], [4, 3], [1, 3]]],
     ['edgelist',
      'G377',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [5, 3], [2, 6]]],
     ['edgelist',
      'G378',
      7,
      [[6, 1], [7, 3], [1, 7], [2, 1], [3, 2], [6, 3], [5, 6], [5, 7]]],
     ['edgelist',
      'G379',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [2, 1], [3, 2], [2, 6], [7, 2]]],
     ['edgelist',
      'G380',
      7,
      [[1, 3], [5, 1], [2, 5], [1, 2], [4, 1], [2, 4], [6, 2], [7, 2]]],
     ['edgelist',
      'G381',
      7,
      [[5, 3], [1, 5], [4, 1], [2, 4], [5, 2], [2, 1], [2, 6], [7, 2]]],
     ['edgelist',
      'G382',
      7,
      [[1, 5], [4, 1], [5, 4], [2, 5], [4, 2], [2, 6], [3, 2], [7, 2]]],
     ['edgelist',
      'G383',
      7,
      [[3, 2], [1, 3], [4, 1], [6, 4], [3, 6], [4, 3], [5, 4], [7, 6]]],
     ['edgelist',
      'G384',
      7,
      [[5, 3], [1, 5], [4, 1], [2, 4], [5, 2], [4, 5], [2, 6], [7, 2]]],
     ['edgelist',
      'G385',
      7,
      [[3, 2], [1, 3], [4, 1], [6, 4], [3, 6], [7, 6], [5, 4], [6, 1]]],
     ['edgelist',
      'G386',
      7,
      [[2, 1], [3, 2], [4, 3], [2, 4], [5, 3], [4, 5], [5, 6], [7, 5]]],
     ['edgelist',
      'G387',
      7,
      [[7, 6], [2, 3], [5, 2], [1, 5], [4, 1], [2, 4], [1, 2], [4, 5]]],
     ['edgelist',
      'G388',
      7,
      [[1, 2], [7, 6], [3, 4], [7, 5], [7, 4], [7, 3], [7, 1], [7, 2]]],
     ['edgelist',
      'G389',
      7,
      [[7, 5], [2, 3], [3, 4], [7, 6], [5, 6], [7, 3], [7, 1], [7, 2]]],
     ['edgelist',
      'G390',
      7,
      [[1, 2], [2, 3], [3, 4], [7, 6], [7, 5], [7, 4], [7, 1], [7, 3]]],
     ['edgelist',
      'G391',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [2, 1], [7, 2], [6, 2], [3, 6]]],
     ['edgelist',
      'G392',
      7,
      [[4, 1], [3, 4], [5, 3], [1, 5], [2, 1], [3, 2], [6, 3], [7, 3]]],
     ['edgelist',
      'G393',
      7,
      [[3, 2], [4, 3], [7, 4], [6, 7], [1, 6], [3, 1], [6, 3], [5, 6]]],
     ['edgelist',
      'G394',
      7,
      [[2, 1], [3, 2], [4, 3], [5, 4], [6, 3], [2, 6], [7, 2], [3, 7]]],
     ['edgelist',
      'G395',
      7,
      [[3, 6], [5, 3], [2, 5], [4, 2], [1, 4], [2, 1], [3, 2], [7, 3]]],
     ['edgelist',
      'G396',
      7,
      [[5, 6], [1, 5], [4, 1], [3, 4], [5, 3], [2, 5], [4, 2], [7, 4]]],
     ['edgelist',
      'G397',
      7,
      [[1, 2], [5, 1], [2, 5], [3, 2], [5, 3], [6, 5], [2, 6], [7, 4]]],
     ['edgelist',
      'G398',
      7,
      [[1, 2], [3, 1], [4, 3], [5, 4], [2, 5], [3, 2], [2, 7], [6, 1]]],
     ['edgelist',
      'G399',
      7,
      [[5, 6], [1, 5], [2, 1], [5, 2], [4, 1], [2, 4], [7, 2], [3, 7]]],
     ['edgelist',
      'G400',
      7,
      [[3, 6], [5, 3], [1, 5], [2, 1], [5, 2], [4, 1], [2, 4], [7, 2]]],
     ['edgelist',
      'G401',
      7,
      [[2, 7], [3, 2], [1, 3], [2, 1], [5, 2], [4, 5], [3, 4], [5, 6]]],
     ['edgelist',
      'G402',
      7,
      [[1, 2], [3, 1], [4, 3], [5, 4], [2, 5], [3, 2], [2, 7], [6, 4]]],
     ['edgelist',
      'G403',
      7,
      [[1, 5], [4, 1], [5, 4], [2, 5], [4, 2], [6, 2], [7, 3], [2, 7]]],
     ['edgelist',
      'G404',
      7,
      [[3, 4], [2, 3], [1, 2], [6, 1], [5, 6], [1, 5], [3, 1], [7, 6]]],
     ['edgelist',
      'G405',
      7,
      [[5, 6], [1, 5], [4, 1], [2, 4], [5, 2], [3, 5], [4, 3], [7, 3]]],
     ['edgelist',
      'G406',
      7,
      [[3, 4], [2, 3], [1, 2], [5, 1], [6, 5], [5, 2], [3, 7], [6, 3]]],
     ['edgelist',
      'G407',
      7,
      [[1, 2], [2, 3], [3, 4], [7, 4], [5, 6], [7, 3], [7, 1], [7, 2]]],
     ['edgelist',
      'G408',
      7,
      [[5, 2], [1, 5], [4, 1], [2, 4], [1, 2], [3, 2], [6, 3], [7, 3]]],
     ['edgelist',
      'G409',
      7,
      [[1, 2], [2, 3], [3, 4], [7, 6], [5, 6], [7, 3], [7, 5], [7, 2]]],
     ['edgelist',
      'G410',
      7,
      [[1, 2], [5, 1], [1, 3], [6, 1], [7, 6], [4, 7], [3, 4], [6, 3]]],
     ['edgelist',
      'G411',
      7,
      [[1, 5], [4, 1], [3, 4], [5, 3], [2, 5], [4, 2], [3, 6], [7, 3]]],
     ['edgelist',
      'G412',
      7,
      [[5, 6], [4, 5], [2, 4], [3, 2], [7, 3], [5, 7], [4, 3], [1, 2]]],
     ['edgelist',
      'G413',
      7,
      [[2, 1], [3, 7], [4, 3], [5, 4], [6, 3], [2, 6], [7, 2], [7, 6]]],
     ['edgelist',
      'G414',
      7,
      [[3, 4], [2, 3], [1, 2], [5, 1], [6, 5], [7, 6], [6, 3], [5, 2]]],
     ['edgelist',
      'G415',
      7,
      [[5, 2], [1, 5], [4, 1], [2, 4], [4, 5], [3, 2], [3, 6], [7, 3]]],
     ['edgelist',
      'G416',
      7,
      [[1, 7], [5, 1], [2, 5], [4, 2], [1, 4], [3, 5], [4, 3], [6, 3]]],
     ['edgelist',
      'G417',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [3, 5], [4, 3], [2, 1], [7, 6]]],
     ['edgelist',
      'G418',
      7,
      [[1, 2], [5, 1], [4, 3], [7, 4], [6, 7], [3, 6], [7, 3], [4, 6]]],
     ['edgelist',
      'G419',
      7,
      [[6, 3], [7, 6], [3, 7], [5, 3], [1, 5], [4, 1], [3, 4], [2, 3]]],
     ['edgelist',
      'G420',
      7,
      [[3, 1], [2, 3], [1, 2], [6, 1], [5, 6], [1, 5], [7, 1], [4, 7]]],
     ['edgelist',
      'G421',
      7,
      [[1, 2], [3, 1], [4, 3], [3, 2], [2, 5], [6, 5], [6, 4], [2, 7]]],
     ['edgelist',
      'G422',
      7,
      [[2, 7], [3, 2], [1, 3], [2, 1], [5, 2], [4, 5], [3, 4], [6, 7]]],
     ['edgelist',
      'G423',
      7,
      [[7, 2], [1, 7], [2, 1], [6, 2], [1, 6], [3, 2], [4, 3], [5, 4]]],
     ['edgelist',
      'G424',
      7,
      [[7, 6], [3, 7], [2, 3], [5, 2], [4, 5], [1, 4], [5, 1], [3, 5]]],
     ['edgelist',
      'G425',
      7,
      [[2, 7], [1, 2], [6, 1], [2, 6], [4, 1], [5, 4], [3, 5], [1, 3]]],
     ['edgelist',
      'G426',
      7,
      [[3, 7], [5, 3], [1, 5], [2, 1], [5, 2], [4, 5], [6, 4], [3, 6]]],
     ['edgelist',
      'G427',
      7,
      [[2, 1], [3, 2], [7, 3], [6, 7], [2, 6], [5, 2], [4, 5], [3, 4]]],
     ['edgelist',
      'G428',
      7,
      [[7, 2], [5, 4], [2, 1], [6, 2], [4, 3], [3, 2], [5, 7], [6, 5]]],
     ['edgelist',
      'G429',
      7,
      [[5, 3], [1, 5], [2, 1], [5, 2], [4, 5], [7, 4], [6, 7], [4, 6]]],
     ['edgelist',
      'G430',
      7,
      [[5, 2], [3, 5], [1, 3], [7, 1], [4, 7], [1, 4], [6, 1], [5, 6]]],
     ['edgelist',
      'G431',
      7,
      [[6, 7], [5, 6], [1, 5], [4, 1], [3, 4], [5, 3], [2, 5], [4, 2]]],
     ['edgelist',
      'G432',
      7,
      [[7, 4], [6, 7], [5, 6], [1, 5], [2, 1], [3, 2], [6, 3], [5, 2]]],
     ['edgelist',
      'G433',
      7,
      [[1, 2], [3, 1], [4, 3], [3, 2], [2, 5], [6, 5], [6, 4], [5, 7]]],
     ['edgelist',
      'G434',
      7,
      [[5, 1], [4, 5], [3, 4], [7, 3], [6, 7], [2, 6], [5, 2], [3, 2]]],
     ['edgelist',
      'G435',
      7,
      [[7, 2], [1, 7], [5, 4], [6, 2], [1, 6], [3, 2], [4, 3], [6, 7]]],
     ['edgelist',
      'G436',
      7,
      [[7, 3], [6, 7], [4, 6], [7, 4], [5, 4], [1, 5], [2, 1], [5, 2]]],
     ['edgelist',
      'G437',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [1, 7], [6, 2]]],
     ['edgelist',
      'G438',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [1, 7], [5, 3]]],
     ['edgelist',
      'G439',
      7,
      [[1, 2], [3, 1], [4, 3], [5, 4], [2, 5], [3, 2], [6, 7], [1, 6]]],
     ['edgelist',
      'G440',
      7,
      [[5, 1], [3, 5], [4, 3], [7, 4], [6, 7], [5, 6], [2, 3], [6, 2]]],
     ['edgelist',
      'G441',
      7,
      [[6, 2], [3, 5], [4, 3], [1, 4], [6, 1], [5, 6], [2, 3], [1, 7]]],
     ['edgelist',
      'G442',
      7,
      [[6, 7], [3, 6], [5, 3], [1, 5], [4, 1], [3, 4], [2, 5], [4, 2]]],
     ['edgelist',
      'G443',
      7,
      [[1, 5], [2, 1], [5, 2], [4, 5], [6, 4], [7, 6], [3, 7], [5, 3]]],
     ['edgelist',
      'G444',
      7,
      [[1, 2], [7, 6], [3, 4], [4, 5], [7, 5], [1, 6], [7, 3], [7, 2]]],
     ['edgelist',
      'G445',
      7,
      [[2, 3], [1, 2], [5, 1], [6, 5], [3, 6], [4, 3], [7, 4], [6, 7]]],
     ['edgelist',
      'G446',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 7], [2, 7]]],
     ['edgelist',
      'G447',
      7,
      [[7, 3], [6, 7], [3, 6], [2, 3], [5, 2], [1, 5], [4, 1], [2, 4]]],
     ['edgelist',
      'G448',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [7, 6], [7, 2]]],
     ['edgelist',
      'G449',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [7, 1], [7, 4]]],
     ['edgelist',
      'G450',
      7,
      [[1, 5], [2, 1], [4, 3], [2, 5], [3, 6], [6, 4], [7, 5], [7, 4]]],
     ['edgelist',
      'G451',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [2, 1], [7, 3], [6, 7], [3, 6]]],
     ['edgelist',
      'G452',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [3, 5], [1, 4]]],
     ['edgelist',
      'G453',
      7,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2], [5, 3]]],
     ['edgelist',
      'G454',
      7,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2], [1, 5]]],
     ['edgelist',
      'G455',
      7,
      [[1, 2], [2, 3], [3, 4], [1, 4], [5, 1], [5, 2], [5, 3], [5, 4], [5, 6]]],
     ['edgelist',
      'G456',
      7,
      [[3, 1], [5, 2], [2, 3], [6, 5], [3, 6], [4, 2], [6, 4], [4, 3], [5, 4]]],
     ['edgelist',
      'G457',
      7,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2], [5, 6]]],
     ['edgelist',
      'G458',
      7,
      [[2, 6], [5, 2], [1, 5], [6, 1], [3, 6], [5, 3], [4, 5], [6, 4], [5, 6]]],
     ['edgelist',
      'G459',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [3, 6], [1, 6], [3, 1], [4, 1]]],
     ['edgelist',
      'G460',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [1, 5], [2, 1], [5, 2]]],
     ['edgelist',
      'G461',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [5, 2], [2, 1], [6, 2]]],
     ['edgelist',
      'G462',
      7,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [6, 5], [5, 1], [6, 1]]],
     ['edgelist',
      'G463',
      7,
      [[5, 4], [1, 5], [2, 1], [3, 2], [4, 3], [1, 6], [6, 4], [1, 4], [2, 6]]],
     ['edgelist',
      'G464',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [4, 3], [1, 4], [5, 1]]],
     ['edgelist',
      'G465',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [5, 6], [4, 5], [4, 3], [1, 4], [3, 5]]],
     ['edgelist',
      'G466',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [3, 6], [1, 6], [3, 1], [6, 2]]],
     ['edgelist',
      'G467',
      7,
      [[2, 6], [5, 2], [1, 5], [6, 1], [3, 6], [5, 3], [4, 5], [6, 4], [3, 1]]],
     ['edgelist',
      'G468',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 1], [6, 5], [6, 3], [6, 4]]],
     ['edgelist',
      'G469',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1], [6, 2]]],
     ['edgelist',
      'G467',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [1, 4], [5, 3], [6, 3]]],
     ['edgelist',
      'G471',
      7,
      [[3, 4], [1, 3], [4, 1], [5, 4], [2, 5], [6, 2], [5, 6], [2, 1], [6, 3]]],
     ['edgelist',
      'G472',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [1, 4], [6, 3], [5, 2]]],
     ['edgelist',
      'G473',
      7,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [6, 1], [1, 5], [1, 7]]],
     ['edgelist',
      'G474',
      7,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [6, 1], [1, 5], [3, 7]]],
     ['edgelist',
      'G475',
      7,
      [[2, 3], [4, 2], [1, 4], [2, 1], [3, 1], [4, 3], [6, 4], [5, 1], [2, 7]]],
     ['edgelist',
      'G476',
      7,
      [[1, 2], [3, 5], [1, 3], [4, 2], [4, 3], [3, 2], [5, 2], [6, 3], [3, 7]]],
     ['edgelist',
      'G477',
      7,
      [[1, 2], [3, 5], [1, 3], [6, 3], [4, 2], [4, 3], [3, 2], [5, 2], [2, 7]]],
     ['edgelist',
      'G478',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [2, 6], [2, 7]]],
     ['edgelist',
      'G479',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [2, 6], [5, 7]]],
     ['edgelist',
      'G480',
      7,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [3, 2], [6, 2], [2, 7]]],
     ['edgelist',
      'G481',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 5], [5, 7]]],
     ['edgelist',
      'G482',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 5], [4, 7]]],
     ['edgelist',
      'G483',
      7,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [3, 2], [6, 2], [1, 7]]],
     ['edgelist',
      'G484',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 1], [2, 7]]],
     ['edgelist',
      'G485',
      7,
      [[3, 1], [4, 3], [5, 4], [6, 5], [3, 6], [2, 3], [5, 2], [6, 4], [3, 7]]],
     ['edgelist',
      'G486',
      7,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [4, 1], [6, 2], [1, 7]]],
     ['edgelist',
      'G487',
      7,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [6, 1], [1, 5], [6, 7]]],
     ['edgelist',
      'G488',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 1], [5, 7]]],
     ['edgelist',
      'G489',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 5], [3, 7]]],
     ['edgelist',
      'G490',
      7,
      [[3, 1], [4, 3], [5, 4], [6, 5], [3, 6], [2, 3], [5, 2], [6, 4], [6, 7]]],
     ['edgelist',
      'G491',
      7,
      [[2, 3], [4, 2], [1, 4], [2, 1], [3, 1], [4, 3], [5, 1], [7, 6], [7, 4]]],
     ['edgelist',
      'G492',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 1], [1, 7]]],
     ['edgelist',
      'G493',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 7], [6, 5], [1, 4], [3, 5]]],
     ['edgelist',
      'G494',
      7,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [3, 2], [6, 2], [6, 7]]],
     ['edgelist',
      'G495',
      7,
      [[3, 1], [4, 3], [5, 4], [6, 5], [3, 6], [2, 3], [5, 2], [6, 4], [5, 7]]],
     ['edgelist',
      'G496',
      7,
      [[1, 2], [3, 6], [1, 3], [4, 1], [4, 2], [4, 3], [3, 2], [6, 2], [5, 7]]],
     ['edgelist',
      'G497',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 1], [3, 7]]],
     ['edgelist',
      'G498',
      7,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [4, 1], [6, 2], [6, 7]]],
     ['edgelist',
      'G499',
      7,
      [[1, 2], [3, 6], [1, 3], [6, 5], [4, 2], [4, 3], [4, 1], [6, 2], [3, 7]]],
     ['edgelist',
      'G500',
      7,
      [[1, 2], [3, 6], [1, 3], [5, 1], [4, 2], [4, 3], [6, 2], [6, 4], [1, 7]]],
     ['edgelist',
      'G501',
      7,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [1, 6], [6, 5], [6, 7]]],
     ['edgelist',
      'G502',
      7,
      [[1, 2], [2, 3], [3, 4], [1, 4], [5, 1], [5, 2], [5, 3], [5, 4], [6, 7]]],
     ['edgelist',
      'G503',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [4, 5], [5, 7]]],
     ['edgelist',
      'G504',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 1], [6, 3], [1, 3], [1, 7]]],
     ['edgelist',
      'G505',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [4, 5], [4, 7]]],
     ['edgelist',
      'G506',
      7,
      [[1, 2], [3, 5], [1, 3], [6, 3], [4, 2], [4, 3], [3, 2], [5, 2], [6, 7]]],
     ['edgelist',
      'G507',
      7,
      [[2, 6], [5, 2], [1, 5], [6, 1], [3, 6], [5, 3], [4, 5], [6, 4], [5, 7]]],
     ['edgelist',
      'G508',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 3], [3, 7]]],
     ['edgelist',
      'G509',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [7, 6], [7, 2]]],
     ['edgelist',
      'G510',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [4, 5], [3, 7]]],
     ['edgelist',
      'G511',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [4, 5], [1, 7]]],
     ['edgelist',
      'G512',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [7, 6], [4, 7], [2, 7], [1, 2], [2, 5]]],
     ['edgelist',
      'G513',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1], [1, 7]]],
     ['edgelist',
      'G514',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [3, 2], [5, 7]]],
     ['edgelist',
      'G515',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 5], [6, 7]]],
     ['edgelist',
      'G516',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 3], [5, 7]]],
     ['edgelist',
      'G517',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 3], [6, 7]]],
     ['edgelist',
      'G518',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [7, 6], [4, 7], [2, 7], [1, 2], [1, 5]]],
     ['edgelist',
      'G519',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 1], [6, 3], [1, 3], [2, 7]]],
     ['edgelist',
      'G520',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 1], [6, 3], [1, 3], [5, 7]]],
     ['edgelist',
      'G521',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [3, 2], [3, 7]]],
     ['edgelist',
      'G522',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1], [4, 7]]],
     ['edgelist',
      'G523',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1], [3, 7]]],
     ['edgelist',
      'G524',
      7,
      [[1, 2], [3, 6], [1, 3], [6, 2], [4, 2], [4, 3], [3, 2], [7, 1], [7, 5]]],
     ['edgelist',
      'G525',
      7,
      [[2, 6], [5, 2], [1, 5], [6, 1], [3, 6], [5, 3], [4, 5], [6, 4], [2, 7]]],
     ['edgelist',
      'G526',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 4], [5, 2], [6, 1], [6, 7]]],
     ['edgelist',
      'G527',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 3], [2, 7]]],
     ['edgelist',
      'G528',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 3], [1, 7]]],
     ['edgelist',
      'G529',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 3], [4, 7]]],
     ['edgelist',
      'G530',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [7, 6], [4, 7], [2, 7], [1, 2], [3, 5]]],
     ['edgelist',
      'G531',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [5, 6], [6, 4], [2, 6], [4, 7]]],
     ['edgelist',
      'G532',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1], [2, 7]]],
     ['edgelist',
      'G533',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 2], [5, 7]]],
     ['edgelist',
      'G534',
      7,
      [[1, 2], [3, 6], [1, 3], [6, 2], [4, 2], [4, 3], [4, 1], [7, 5], [7, 1]]],
     ['edgelist',
      'G535',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 6], [6, 3], [6, 1], [2, 7]]],
     ['edgelist',
      'G536',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [3, 2], [7, 1]]],
     ['edgelist',
      'G537',
      7,
      [[6, 4], [4, 3], [5, 4], [6, 5], [3, 6], [2, 3], [5, 2], [7, 1], [7, 3]]],
     ['edgelist',
      'G538',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 6], [6, 3], [6, 1], [1, 7]]],
     ['edgelist',
      'G539',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [5, 6], [6, 4], [2, 6], [6, 7]]],
     ['edgelist',
      'G540',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [4, 1], [6, 3], [6, 1], [5, 7]]],
     ['edgelist',
      'G541',
      7,
      [[2, 4], [3, 2], [1, 3], [6, 1], [7, 6], [4, 7], [2, 7], [1, 2], [6, 5]]],
     ['edgelist',
      'G542',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [5, 2], [6, 3], [6, 7]]],
     ['edgelist',
      'G543',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [5, 6], [6, 4], [2, 6], [2, 7]]],
     ['edgelist',
      'G544',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [3, 2], [4, 7]]],
     ['edgelist',
      'G545',
      7,
      [[1, 2], [2, 3], [1, 3], [4, 1], [4, 2], [4, 3], [1, 6], [6, 5], [5, 7]]],
     ['edgelist',
      'G546',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [5, 6], [6, 4], [2, 6], [1, 7]]],
     ['edgelist',
      'G547',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [2, 6], [6, 3], [6, 1], [5, 7]]],
     ['edgelist',
      'G548',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [3, 5], [6, 2], [1, 7]]],
     ['edgelist',
      'G549',
      7,
      [[1, 2], [3, 6], [1, 3], [6, 4], [4, 2], [4, 3], [6, 2], [7, 5], [7, 1]]],
     ['edgelist',
      'G550',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [5, 2], [6, 3], [1, 7]]],
     ['edgelist',
      'G551',
      7,
      [[7, 4], [2, 3], [7, 6], [4, 5], [7, 5], [1, 6], [7, 1], [7, 2], [7, 3]]],
     ['edgelist',
      'G552',
      7,
      [[1, 2], [3, 1], [4, 3], [5, 4], [2, 5], [3, 2], [7, 3], [6, 7], [3, 6]]],
     ['edgelist',
      'G553',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [4, 5], [7, 6], [7, 1]]],
     ['edgelist',
      'G554',
      7,
      [[2, 5], [3, 5], [3, 4], [1, 5], [4, 2], [5, 6], [1, 6], [7, 5], [7, 4]]],
     ['edgelist',
      'G555',
      7,
      [[5, 2], [6, 5], [7, 6], [4, 7], [3, 4], [2, 3], [6, 3], [1, 6], [3, 1]]],
     ['edgelist',
      'G556',
      7,
      [[5, 2], [4, 2], [3, 4], [5, 1], [6, 1], [6, 3], [6, 5], [7, 5], [6, 7]]],
     ['edgelist',
      'G557',
      7,
      [[2, 1], [3, 2], [7, 3], [4, 7], [6, 4], [5, 6], [4, 5], [3, 4], [1, 3]]],
     ['edgelist',
      'G558',
      7,
      [[1, 3], [6, 1], [2, 6], [3, 2], [5, 3], [6, 5], [7, 6], [4, 7], [3, 4]]],
     ['edgelist',
      'G559',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [6, 7], [1, 7], [2, 4], [5, 2]]],
     ['edgelist',
      'G560',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 5], [6, 2], [7, 2], [1, 7]]],
     ['edgelist',
      'G561',
      7,
      [[1, 5], [2, 1], [5, 2], [4, 5], [3, 4], [7, 3], [6, 7], [2, 6], [3, 2]]],
     ['edgelist',
      'G562',
      7,
      [[1, 2], [3, 1], [4, 3], [5, 4], [2, 5], [3, 2], [6, 4], [7, 6], [4, 7]]],
     ['edgelist',
      'G563',
      7,
      [[7, 6], [4, 7], [3, 4], [1, 5], [1, 6], [2, 1], [3, 1], [2, 3], [6, 5]]],
     ['edgelist',
      'G564',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 4], [6, 2], [7, 2], [1, 7]]],
     ['edgelist',
      'G565',
      7,
      [[6, 3], [7, 6], [4, 7], [3, 4], [1, 3], [5, 1], [6, 5], [2, 6], [1, 2]]],
     ['edgelist',
      'G566',
      7,
      [[3, 5], [2, 3], [5, 2], [6, 5], [1, 6], [2, 1], [7, 5], [4, 7], [3, 4]]],
     ['edgelist',
      'G567',
      7,
      [[7, 3], [6, 7], [3, 6], [2, 3], [1, 2], [5, 1], [2, 5], [4, 2], [1, 4]]],
     ['edgelist',
      'G568',
      7,
      [[1, 6], [7, 1], [2, 7], [5, 2], [3, 5], [4, 3], [2, 4], [6, 2], [7, 6]]],
     ['edgelist',
      'G569',
      7,
      [[7, 6], [4, 7], [3, 4], [6, 3], [1, 6], [2, 1], [5, 2], [1, 5], [3, 1]]],
     ['edgelist',
      'G570',
      7,
      [[1, 5], [4, 1], [2, 4], [5, 2], [3, 5], [7, 3], [6, 7], [3, 6], [4, 3]]],
     ['edgelist',
      'G571',
      7,
      [[2, 1], [5, 2], [6, 5], [1, 6], [7, 1], [4, 7], [3, 4], [1, 3], [4, 5]]],
     ['edgelist',
      'G572',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [7, 1], [7, 2], [7, 4]]],
     ['edgelist',
      'G573',
      7,
      [[1, 2], [2, 3], [3, 4], [1, 4], [5, 2], [6, 5], [6, 4], [7, 1], [7, 5]]],
     ['edgelist',
      'G574',
      7,
      [[1, 2], [5, 1], [2, 5], [3, 2], [6, 3], [5, 6], [7, 6], [4, 7], [3, 4]]],
     ['edgelist',
      'G575',
      7,
      [[2, 1], [7, 4], [1, 5], [6, 1], [4, 6], [6, 7], [2, 3], [2, 5], [7, 3]]],
     ['edgelist',
      'G576',
      7,
      [[7, 3], [6, 7], [3, 6], [2, 3], [1, 4], [5, 1], [2, 5], [4, 2], [4, 5]]],
     ['edgelist',
      'G577',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [7, 2], [6, 7], [7, 1]]],
     ['edgelist',
      'G578',
      7,
      [[1, 5], [2, 1], [3, 2], [4, 3], [1, 4], [3, 5], [6, 5], [7, 6], [4, 7]]],
     ['edgelist',
      'G579',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [5, 3], [7, 2], [6, 7]]],
     ['edgelist',
      'G580',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [1, 5], [6, 4], [6, 5], [7, 2], [7, 6]]],
     ['edgelist',
      'G581',
      7,
      [[1, 2], [2, 3], [3, 4], [4, 5], [5, 6], [1, 6], [7, 1], [7, 5], [7, 3]]],
     ['edgelist',
      'G582',
      7,
      [[1, 5], [4, 1], [5, 4], [7, 2], [6, 7], [2, 6], [3, 2], [6, 3], [7, 3]]],
     ['edgelist',
      'G583',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 3]]],
     ['edgelist',
      'G584',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 5]]],
     ['edgelist',
      'G585',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1]]],
     ['edgelist',
      'G586',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [5, 6],
       [2, 1]]],
     ['edgelist',
      'G587',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [6, 5],
       [4, 6],
       [2, 6]]],
     ['edgelist',
      'G588',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [5, 1],
       [3, 5]]],
     ['edgelist',
      'G589',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [5, 6],
       [4, 5],
       [2, 4],
       [6, 2],
       [3, 4],
       [2, 3]]],
     ['edgelist',
      'G590',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 3]]],
     ['edgelist',
      'G591',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [5, 2]]],
     ['edgelist',
      'G592',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [6, 3]]],
     ['edgelist',
      'G593',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [5, 6],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4]]],
     ['edgelist',
      'G594',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 5]]],
     ['edgelist',
      'G595',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [1, 3],
       [2, 4],
       [6, 2]]],
     ['edgelist',
      'G596',
      7,
      [[1, 2],
       [2, 3],
       [4, 5],
       [1, 3],
       [4, 1],
       [3, 5],
       [6, 3],
       [2, 6],
       [5, 2],
       [4, 6]]],
     ['edgelist',
      'G597',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [6, 4],
       [3, 6],
       [2, 1]]],
     ['edgelist',
      'G598',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [5, 3],
       [3, 7]]],
     ['edgelist',
      'G599',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [5, 3],
       [2, 7]]],
     ['edgelist',
      'G600',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [1, 5],
       [2, 7]]],
     ['edgelist',
      'G601',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [1, 5],
       [1, 7]]],
     ['edgelist',
      'G602',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [1, 5],
       [4, 7]]],
     ['edgelist',
      'G603',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G604',
      7,
      [[3, 1],
       [5, 2],
       [2, 3],
       [6, 5],
       [3, 6],
       [4, 2],
       [6, 4],
       [4, 3],
       [5, 4],
       [4, 7]]],
     ['edgelist',
      'G605',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [5, 6],
       [3, 7]]],
     ['edgelist',
      'G606',
      7,
      [[3, 1],
       [5, 2],
       [2, 3],
       [6, 5],
       [3, 6],
       [4, 2],
       [6, 4],
       [4, 3],
       [5, 4],
       [3, 7]]],
     ['edgelist',
      'G607',
      7,
      [[3, 4],
       [2, 3],
       [5, 2],
       [6, 5],
       [3, 6],
       [1, 3],
       [5, 1],
       [1, 2],
       [6, 1],
       [7, 6]]],
     ['edgelist',
      'G608',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [1, 5],
       [6, 7]]],
     ['edgelist',
      'G609',
      7,
      [[3, 1],
       [5, 2],
       [2, 3],
       [6, 5],
       [3, 6],
       [4, 2],
       [6, 4],
       [4, 3],
       [5, 4],
       [5, 7]]],
     ['edgelist',
      'G610',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [5, 6],
       [7, 6]]],
     ['edgelist',
      'G611',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 7]]],
     ['edgelist',
      'G612',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [5, 6],
       [7, 6]]],
     ['edgelist',
      'G613',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [4, 1],
       [1, 7]]],
     ['edgelist',
      'G614',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [4, 1],
       [3, 7]]],
     ['edgelist',
      'G615',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [5, 6],
       [2, 7]]],
     ['edgelist',
      'G616',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [4, 1],
       [4, 7]]],
     ['edgelist',
      'G617',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [1, 5],
       [2, 1],
       [5, 2],
       [1, 7]]],
     ['edgelist',
      'G618',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [2, 1],
       [6, 2],
       [2, 7]]],
     ['edgelist',
      'G619',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [6, 5],
       [5, 1],
       [6, 1],
       [1, 7]]],
     ['edgelist',
      'G620',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [4, 7]]],
     ['edgelist',
      'G621',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [4, 1],
       [6, 7]]],
     ['edgelist',
      'G622',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [5, 1],
       [4, 7]]],
     ['edgelist',
      'G623',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [2, 1],
       [6, 2],
       [1, 7]]],
     ['edgelist',
      'G624',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [3, 1],
       [6, 4],
       [3, 4],
       [7, 3]]],
     ['edgelist',
      'G625',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [7, 5],
       [7, 3]]],
     ['edgelist',
      'G626',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [3, 7]]],
     ['edgelist',
      'G627',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [2, 1],
       [6, 2],
       [6, 7]]],
     ['edgelist',
      'G628',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [4, 1],
       [5, 7]]],
     ['edgelist',
      'G629',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [4, 7]]],
     ['edgelist',
      'G630',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [6, 5],
       [5, 1],
       [6, 1],
       [3, 7]]],
     ['edgelist',
      'G631',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [3, 1],
       [6, 7]]],
     ['edgelist',
      'G632',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [5, 1],
       [3, 7]]],
     ['edgelist',
      'G633',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [3, 1],
       [6, 4],
       [3, 4],
       [1, 7]]],
     ['edgelist',
      'G634',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [2, 7]]],
     ['edgelist',
      'G635',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [5, 7]]],
     ['edgelist',
      'G636',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [7, 5],
       [7, 1]]],
     ['edgelist',
      'G637',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [6, 7]]],
     ['edgelist',
      'G638',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [1, 5],
       [2, 1],
       [5, 2],
       [6, 7]]],
     ['edgelist',
      'G639',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [3, 1],
       [1, 7]]],
     ['edgelist',
      'G640',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [2, 1],
       [6, 2],
       [3, 7]]],
     ['edgelist',
      'G641',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [1, 4],
       [5, 3],
       [6, 3],
       [3, 7]]],
     ['edgelist',
      'G642',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 5],
       [6, 3],
       [6, 4],
       [6, 7]]],
     ['edgelist',
      'G643',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [4, 1],
       [6, 3],
       [6, 1],
       [6, 2],
       [1, 7]]],
     ['edgelist',
      'G644',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [6, 5],
       [5, 1],
       [6, 1],
       [6, 7]]],
     ['edgelist',
      'G645',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [7, 6],
       [7, 5]]],
     ['edgelist',
      'G646',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [5, 1],
       [2, 7]]],
     ['edgelist',
      'G647',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [5, 7]]],
     ['edgelist',
      'G648',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 5],
       [6, 3],
       [6, 4],
       [5, 7]]],
     ['edgelist',
      'G649',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [3, 7]]],
     ['edgelist',
      'G650',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 5],
       [6, 3],
       [6, 4],
       [1, 7]]],
     ['edgelist',
      'G651',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [1, 4],
       [5, 3],
       [6, 3],
       [6, 7]]],
     ['edgelist',
      'G652',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [4, 1],
       [6, 3],
       [6, 1],
       [6, 2],
       [2, 7]]],
     ['edgelist',
      'G653',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [4, 7]]],
     ['edgelist',
      'G654',
      7,
      [[5, 4],
       [5, 2],
       [2, 3],
       [6, 5],
       [3, 6],
       [4, 2],
       [6, 4],
       [4, 3],
       [7, 1],
       [7, 3]]],
     ['edgelist',
      'G655',
      7,
      [[2, 1],
       [3, 2],
       [4, 3],
       [5, 4],
       [6, 5],
       [2, 6],
       [7, 2],
       [5, 7],
       [3, 7],
       [6, 3]]],
     ['edgelist',
      'G656',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [1, 4],
       [5, 3],
       [6, 3],
       [1, 7]]],
     ['edgelist',
      'G657',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [4, 1],
       [6, 3],
       [6, 1],
       [6, 2],
       [4, 7]]],
     ['edgelist',
      'G658',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [4, 1],
       [6, 3],
       [6, 1],
       [6, 2],
       [3, 7]]],
     ['edgelist',
      'G659',
      7,
      [[1, 2],
       [3, 6],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [3, 2],
       [6, 2],
       [7, 6],
       [7, 5]]],
     ['edgelist',
      'G660',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [2, 7]]],
     ['edgelist',
      'G661',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 7]]],
     ['edgelist',
      'G662',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [3, 1],
       [2, 7]]],
     ['edgelist',
      'G663',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 5],
       [6, 3],
       [6, 4],
       [2, 7]]],
     ['edgelist',
      'G664',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [1, 4],
       [5, 3],
       [6, 3],
       [2, 7]]],
     ['edgelist',
      'G665',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [4, 1],
       [6, 3],
       [6, 1],
       [6, 2],
       [5, 7]]],
     ['edgelist',
      'G666',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [2, 1],
       [6, 3],
       [5, 7]]],
     ['edgelist',
      'G667',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [1, 4],
       [6, 3],
       [5, 2],
       [1, 7]]],
     ['edgelist',
      'G668',
      7,
      [[5, 1],
       [2, 5],
       [4, 2],
       [3, 4],
       [2, 3],
       [7, 2],
       [1, 7],
       [6, 1],
       [2, 6],
       [1, 2]]],
     ['edgelist',
      'G669',
      7,
      [[4, 3],
       [7, 4],
       [6, 7],
       [1, 6],
       [3, 1],
       [6, 3],
       [2, 6],
       [3, 2],
       [5, 3],
       [6, 5]]],
     ['edgelist',
      'G670',
      7,
      [[3, 1],
       [2, 3],
       [4, 2],
       [1, 4],
       [7, 1],
       [2, 7],
       [6, 2],
       [1, 6],
       [5, 1],
       [2, 5]]],
     ['edgelist',
      'G671',
      7,
      [[7, 5],
       [2, 3],
       [7, 6],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [7, 4]]],
     ['edgelist',
      'G672',
      7,
      [[1, 2],
       [7, 6],
       [3, 4],
       [4, 5],
       [7, 5],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [7, 4]]],
     ['edgelist',
      'G673',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [3, 2],
       [1, 6],
       [6, 3],
       [7, 2],
       [3, 7]]],
     ['edgelist',
      'G674',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [3, 2],
       [6, 1],
       [6, 3],
       [7, 3],
       [1, 7]]],
     ['edgelist',
      'G675',
      7,
      [[7, 5],
       [2, 3],
       [7, 6],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 4],
       [7, 2],
       [7, 3],
       [1, 5]]],
     ['edgelist',
      'G676',
      7,
      [[2, 1],
       [3, 2],
       [1, 3],
       [4, 3],
       [5, 4],
       [3, 5],
       [6, 3],
       [5, 6],
       [7, 5],
       [2, 7]]],
     ['edgelist',
      'G677',
      7,
      [[1, 2],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 6],
       [3, 7]]],
     ['edgelist',
      'G678',
      7,
      [[1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [2, 3],
       [6, 2],
       [7, 6],
       [4, 7],
       [3, 4],
       [3, 7]]],
     ['edgelist',
      'G679',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [3, 2],
       [6, 2],
       [1, 6],
       [7, 1],
       [3, 7]]],
     ['edgelist',
      'G680',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 3],
       [7, 5],
       [1, 3],
       [5, 1]]],
     ['edgelist',
      'G681',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [6, 3],
       [7, 6],
       [3, 7],
       [5, 3],
       [2, 5],
       [4, 2],
       [5, 4]]],
     ['edgelist',
      'G682',
      7,
      [[2, 7],
       [3, 2],
       [1, 3],
       [2, 1],
       [5, 2],
       [4, 5],
       [3, 4],
       [6, 7],
       [5, 6],
       [4, 2]]],
     ['edgelist',
      'G683',
      7,
      [[7, 6],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [7, 5]]],
     ['edgelist',
      'G684',
      7,
      [[1, 2],
       [7, 6],
       [3, 4],
       [4, 5],
       [7, 5],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [6, 4]]],
     ['edgelist',
      'G685',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [6, 5],
       [1, 5],
       [6, 1],
       [6, 4],
       [6, 3],
       [7, 6],
       [7, 2]]],
     ['edgelist',
      'G686',
      7,
      [[1, 4],
       [3, 1],
       [2, 3],
       [4, 2],
       [5, 4],
       [3, 5],
       [1, 5],
       [7, 1],
       [6, 7],
       [1, 6]]],
     ['edgelist',
      'G687',
      7,
      [[1, 4],
       [3, 1],
       [2, 3],
       [4, 2],
       [5, 4],
       [1, 6],
       [1, 5],
       [7, 1],
       [6, 7],
       [2, 5]]],
     ['edgelist',
      'G688',
      7,
      [[1, 2],
       [7, 6],
       [3, 4],
       [4, 5],
       [7, 5],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [5, 3]]],
     ['edgelist',
      'G689',
      7,
      [[2, 3],
       [6, 2],
       [7, 6],
       [3, 7],
       [2, 7],
       [6, 3],
       [5, 2],
       [1, 5],
       [4, 1],
       [2, 4]]],
     ['edgelist',
      'G690',
      7,
      [[5, 3],
       [7, 3],
       [6, 4],
       [5, 2],
       [3, 1],
       [7, 4],
       [6, 3],
       [1, 2],
       [1, 5],
       [7, 1]]],
     ['edgelist',
      'G691',
      7,
      [[5, 3],
       [4, 7],
       [6, 4],
       [6, 2],
       [3, 1],
       [7, 1],
       [6, 3],
       [2, 5],
       [1, 5],
       [6, 5]]],
     ['edgelist',
      'G692',
      7,
      [[5, 1],
       [6, 5],
       [5, 2],
       [3, 2],
       [4, 3],
       [1, 4],
       [4, 5],
       [6, 4],
       [7, 2],
       [7, 6]]],
     ['edgelist',
      'G693',
      7,
      [[1, 5],
       [2, 1],
       [3, 2],
       [5, 3],
       [4, 5],
       [6, 4],
       [5, 6],
       [6, 3],
       [7, 4],
       [3, 7]]],
     ['edgelist',
      'G694',
      7,
      [[2, 7],
       [3, 2],
       [1, 3],
       [2, 1],
       [5, 2],
       [4, 5],
       [3, 4],
       [6, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G695',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 4],
       [7, 2],
       [7, 6],
       [6, 2]]],
     ['edgelist',
      'G696',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [3, 1],
       [4, 3],
       [7, 4],
       [6, 7],
       [1, 6],
       [6, 3],
       [7, 3]]],
     ['edgelist',
      'G697',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 4],
       [6, 2],
       [6, 5],
       [7, 2],
       [6, 7]]],
     ['edgelist',
      'G698',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [5, 2],
       [7, 2],
       [7, 6]]],
     ['edgelist',
      'G699',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [3, 2],
       [6, 4],
       [3, 6],
       [7, 2],
       [5, 7]]],
     ['edgelist',
      'G700',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 3],
       [6, 5],
       [7, 6],
       [7, 1],
       [1, 3]]],
     ['edgelist',
      'G701',
      7,
      [[3, 1],
       [6, 3],
       [2, 6],
       [1, 2],
       [4, 1],
       [6, 4],
       [7, 6],
       [5, 7],
       [1, 5],
       [5, 4]]],
     ['edgelist',
      'G702',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [5, 3],
       [2, 6],
       [7, 3],
       [7, 6]]],
     ['edgelist',
      'G703',
      7,
      [[6, 1],
       [7, 6],
       [3, 7],
       [4, 3],
       [1, 4],
       [5, 1],
       [3, 5],
       [5, 4],
       [2, 5],
       [4, 2]]],
     ['edgelist',
      'G704',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [7, 4],
       [6, 7],
       [4, 6],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G705',
      7,
      [[6, 3],
       [3, 2],
       [4, 3],
       [5, 4],
       [2, 5],
       [6, 1],
       [7, 2],
       [7, 1],
       [2, 6],
       [3, 7]]],
     ['edgelist',
      'G706',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 3],
       [7, 5],
       [5, 3],
       [6, 2]]],
     ['edgelist',
      'G707',
      7,
      [[5, 3],
       [3, 4],
       [5, 2],
       [1, 2],
       [4, 1],
       [7, 5],
       [1, 7],
       [6, 1],
       [5, 6],
       [2, 6]]],
     ['edgelist',
      'G708',
      7,
      [[3, 2],
       [6, 3],
       [4, 6],
       [1, 4],
       [5, 1],
       [7, 5],
       [4, 7],
       [2, 4],
       [5, 2],
       [6, 5]]],
     ['edgelist',
      'G709',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [7, 6],
       [7, 4]]],
     ['edgelist',
      'G710',
      7,
      [[1, 2],
       [5, 1],
       [2, 5],
       [3, 2],
       [6, 3],
       [5, 6],
       [7, 6],
       [4, 7],
       [3, 4],
       [6, 4]]],
     ['edgelist',
      'G711',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 6],
       [7, 2],
       [7, 3],
       [5, 3]]],
     ['edgelist',
      'G712',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 4],
       [6, 3],
       [7, 6],
       [7, 5]]],
     ['edgelist',
      'G713',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 4],
       [7, 3],
       [5, 1]]],
     ['edgelist',
      'G714',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 5],
       [7, 6],
       [7, 4]]],
     ['edgelist',
      'G715',
      7,
      [[1, 6],
       [7, 1],
       [2, 7],
       [1, 2],
       [2, 6],
       [3, 2],
       [4, 3],
       [5, 4],
       [7, 5],
       [5, 6]]],
     ['edgelist',
      'G716',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 6],
       [7, 5],
       [3, 1]]],
     ['edgelist',
      'G717',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 6],
       [7, 4]]],
     ['edgelist',
      'G718',
      7,
      [[3, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [6, 2],
       [6, 1],
       [7, 1],
       [2, 7],
       [7, 6]]],
     ['edgelist',
      'G719',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 4],
       [6, 2],
       [7, 2],
       [7, 5],
       [7, 6]]],
     ['edgelist',
      'G720',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [6, 3],
       [5, 2],
       [7, 1],
       [6, 7]]],
     ['edgelist',
      'G721',
      7,
      [[4, 2],
       [1, 4],
       [6, 1],
       [2, 6],
       [3, 2],
       [7, 3],
       [1, 7],
       [1, 5],
       [5, 3],
       [5, 7]]],
     ['edgelist',
      'G722',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 5],
       [7, 2],
       [7, 3],
       [7, 6]]],
     ['edgelist',
      'G723',
      7,
      [[1, 4],
       [3, 1],
       [2, 3],
       [4, 2],
       [5, 4],
       [3, 5],
       [6, 5],
       [6, 1],
       [7, 5],
       [7, 2]]],
     ['edgelist',
      'G724',
      7,
      [[1, 2],
       [7, 6],
       [3, 4],
       [4, 5],
       [7, 5],
       [1, 6],
       [7, 3],
       [7, 2],
       [5, 3],
       [6, 2]]],
     ['edgelist',
      'G725',
      7,
      [[6, 3],
       [7, 6],
       [3, 7],
       [5, 3],
       [1, 5],
       [4, 1],
       [3, 4],
       [2, 1],
       [2, 4],
       [5, 2]]],
     ['edgelist',
      'G726',
      7,
      [[4, 5],
       [2, 4],
       [5, 2],
       [1, 5],
       [4, 1],
       [2, 1],
       [3, 2],
       [6, 3],
       [7, 6],
       [3, 7]]],
     ['edgelist',
      'G727',
      7,
      [[6, 7],
       [3, 6],
       [7, 3],
       [4, 7],
       [1, 4],
       [5, 1],
       [6, 5],
       [2, 5],
       [4, 2],
       [3, 2]]],
     ['edgelist',
      'G728',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 6],
       [5, 3]]],
     ['edgelist',
      'G729',
      7,
      [[2, 1],
       [3, 2],
       [4, 3],
       [1, 4],
       [6, 1],
       [2, 6],
       [5, 6],
       [7, 5],
       [4, 7],
       [3, 7]]],
     ['edgelist',
      'G730',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [5, 2],
       [3, 6],
       [7, 1],
       [4, 7]]],
     ['edgelist',
      'G731',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 3],
       [2, 6]]],
     ['edgelist',
      'G732',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [3, 2],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4]]],
     ['edgelist',
      'G733',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1],
       [5, 6]]],
     ['edgelist',
      'G734',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [1, 3]]],
     ['edgelist',
      'G735',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [6, 2]]],
     ['edgelist',
      'G736',
      7,
      [[2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G737',
      7,
      [[4, 7],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [7, 6],
       [3, 7],
       [6, 2],
       [1, 4],
       [2, 7],
       [1, 2]]],
     ['edgelist',
      'G738',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3]]],
     ['edgelist',
      'G739',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [1, 4]]],
     ['edgelist',
      'G740',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 5],
       [7, 5]]],
     ['edgelist',
      'G741',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [7, 5]]],
     ['edgelist',
      'G742',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 3],
       [5, 7]]],
     ['edgelist',
      'G743',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [3, 6],
       [7, 3]]],
     ['edgelist',
      'G744',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 3],
       [1, 7]]],
     ['edgelist',
      'G745',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [7, 6]]],
     ['edgelist',
      'G746',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [5, 6],
       [2, 1],
       [5, 7]]],
     ['edgelist',
      'G747',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [6, 5],
       [4, 6],
       [2, 6],
       [2, 7]]],
     ['edgelist',
      'G748',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [6, 5],
       [4, 6],
       [2, 6],
       [7, 5]]],
     ['edgelist',
      'G749',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [5, 6],
       [2, 1],
       [2, 7]]],
     ['edgelist',
      'G750',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [5, 1],
       [3, 5],
       [3, 7]]],
     ['edgelist',
      'G751',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [6, 5],
       [4, 6],
       [2, 6],
       [6, 7]]],
     ['edgelist',
      'G752',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 3],
       [3, 7]]],
     ['edgelist',
      'G753',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [5, 6],
       [4, 5],
       [2, 4],
       [6, 2],
       [3, 4],
       [2, 3],
       [7, 2]]],
     ['edgelist',
      'G754',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 3],
       [4, 7]]],
     ['edgelist',
      'G755',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [5, 6],
       [4, 5],
       [2, 4],
       [6, 2],
       [3, 4],
       [2, 3],
       [7, 5]]],
     ['edgelist',
      'G756',
      7,
      [[1, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G757',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [6, 3],
       [1, 7]]],
     ['edgelist',
      'G758',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 3],
       [1, 7]]],
     ['edgelist',
      'G759',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [5, 2],
       [2, 7]]],
     ['edgelist',
      'G760',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [5, 6],
       [4, 5],
       [2, 4],
       [6, 2],
       [3, 4],
       [2, 3],
       [6, 7]]],
     ['edgelist',
      'G761',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [6, 5],
       [4, 6],
       [2, 6],
       [1, 7]]],
     ['edgelist',
      'G762',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [5, 6],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [3, 7]]],
     ['edgelist',
      'G763',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [5, 6],
       [4, 5],
       [2, 4],
       [6, 2],
       [3, 4],
       [2, 3],
       [4, 7]]],
     ['edgelist',
      'G764',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [5, 2],
       [3, 7]]],
     ['edgelist',
      'G765',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 3],
       [6, 7]]],
     ['edgelist',
      'G766',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [6, 3],
       [6, 7]]],
     ['edgelist',
      'G767',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [5, 6],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 7]]],
     ['edgelist',
      'G768',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 5],
       [6, 7]]],
     ['edgelist',
      'G769',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [6, 3],
       [2, 7]]],
     ['edgelist',
      'G770',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [5, 2],
       [5, 7]]],
     ['edgelist',
      'G771',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [5, 2],
       [6, 7]]],
     ['edgelist',
      'G772',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [5, 6],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [5, 7]]],
     ['edgelist',
      'G773',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [5, 1],
       [3, 5],
       [2, 7]]],
     ['edgelist',
      'G774',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [7, 6],
       [7, 3]]],
     ['edgelist',
      'G775',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 5],
       [6, 7]]],
     ['edgelist',
      'G776',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [1, 3],
       [2, 4],
       [6, 2],
       [2, 7]]],
     ['edgelist',
      'G777',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 5],
       [2, 7]]],
     ['edgelist',
      'G778',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [5, 6],
       [4, 5],
       [2, 4],
       [6, 2],
       [3, 4],
       [2, 3],
       [3, 7]]],
     ['edgelist',
      'G779',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [4, 3],
       [1, 4],
       [3, 5],
       [6, 3],
       [2, 7]]],
     ['edgelist',
      'G780',
      7,
      [[1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [6, 7]]],
     ['edgelist',
      'G781',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [6, 4],
       [3, 6],
       [2, 1],
       [2, 7]]],
     ['edgelist',
      'G782',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [1, 3],
       [2, 4],
       [6, 2],
       [6, 7]]],
     ['edgelist',
      'G783',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [1, 3],
       [2, 4],
       [6, 2],
       [7, 4]]],
     ['edgelist',
      'G784',
      7,
      [[5, 4],
       [1, 5],
       [2, 1],
       [3, 2],
       [4, 3],
       [1, 6],
       [6, 4],
       [1, 4],
       [2, 6],
       [6, 3],
       [5, 7]]],
     ['edgelist',
      'G785',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 6],
       [1, 6],
       [3, 1],
       [6, 2],
       [5, 2],
       [7, 4]]],
     ['edgelist',
      'G786',
      7,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [4, 3],
       [1, 4],
       [2, 7]]],
     ['edgelist',
      'G787',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [6, 4],
       [3, 6],
       [2, 1],
       [7, 3]]],
     ['edgelist',
      'G788',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [5, 6],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 7]]],
     ['edgelist',
      'G789',
      7,
      [[1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6]]],
     ['edgelist',
      'G790',
      7,
      [[7, 6],
       [1, 7],
       [6, 1],
       [2, 6],
       [7, 2],
       [3, 7],
       [6, 3],
       [4, 6],
       [7, 4],
       [5, 7],
       [6, 5]]],
     ['edgelist',
      'G791',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [3, 2],
       [6, 2],
       [3, 6],
       [7, 3],
       [2, 7],
       [4, 2]]],
     ['edgelist',
      'G792',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 2],
       [4, 6],
       [7, 2],
       [5, 7],
       [2, 5],
       [4, 2]]],
     ['edgelist',
      'G793',
      7,
      [[2, 5],
       [3, 4],
       [5, 3],
       [1, 7],
       [5, 6],
       [7, 6],
       [4, 2],
       [7, 5],
       [4, 1],
       [4, 7],
       [5, 4]]],
     ['edgelist',
      'G794',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [5, 3],
       [7, 5],
       [3, 7],
       [6, 3],
       [1, 3]]],
     ['edgelist',
      'G795',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [5, 1],
       [4, 1],
       [3, 1],
       [7, 1],
       [4, 7]]],
     ['edgelist',
      'G796',
      7,
      [[1, 2],
       [3, 1],
       [6, 3],
       [7, 6],
       [3, 7],
       [2, 3],
       [5, 2],
       [3, 5],
       [4, 3],
       [5, 4],
       [4, 2]]],
     ['edgelist',
      'G797',
      7,
      [[5, 6],
       [2, 5],
       [3, 2],
       [4, 3],
       [7, 4],
       [6, 7],
       [3, 6],
       [5, 3],
       [4, 6],
       [1, 6],
       [3, 1]]],
     ['edgelist',
      'G798',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [5, 3],
       [6, 3],
       [6, 1],
       [7, 3],
       [5, 7],
       [6, 5]]],
     ['edgelist',
      'G799',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [7, 2],
       [3, 7],
       [1, 3],
       [7, 1],
       [6, 3],
       [1, 6]]],
     ['edgelist',
      'G800',
      7,
      [[1, 6],
       [7, 1],
       [2, 7],
       [6, 2],
       [3, 6],
       [7, 3],
       [5, 4],
       [4, 3],
       [5, 6],
       [7, 5],
       [7, 6]]],
     ['edgelist',
      'G801',
      7,
      [[1, 6],
       [7, 1],
       [2, 7],
       [6, 2],
       [3, 6],
       [7, 3],
       [4, 7],
       [6, 4],
       [5, 6],
       [7, 5],
       [5, 4]]],
     ['edgelist',
      'G802',
      7,
      [[1, 6],
       [1, 7],
       [2, 3],
       [2, 7],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G803',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [1, 3],
       [3, 5],
       [6, 3],
       [5, 6],
       [7, 6],
       [7, 1]]],
     ['edgelist',
      'G804',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [6, 7],
       [1, 7],
       [5, 3],
       [1, 5],
       [3, 1],
       [7, 5]]],
     ['edgelist',
      'G805',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [6, 2],
       [6, 3],
       [7, 2],
       [3, 7],
       [5, 3],
       [6, 5]]],
     ['edgelist',
      'G806',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [6, 2],
       [3, 6],
       [5, 3],
       [7, 3],
       [5, 7]]],
     ['edgelist',
      'G807',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 5],
       [7, 3],
       [1, 3],
       [5, 1]]],
     ['edgelist',
      'G808',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [4, 2],
       [6, 4],
       [5, 6],
       [2, 5],
       [7, 6],
       [7, 2]]],
     ['edgelist',
      'G809',
      7,
      [[1, 5],
       [4, 1],
       [5, 4],
       [3, 5],
       [4, 3],
       [2, 4],
       [3, 2],
       [5, 2],
       [6, 3],
       [7, 6],
       [3, 7]]],
     ['edgelist',
      'G810',
      7,
      [[1, 6],
       [1, 7],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G811',
      7,
      [[1, 2],
       [5, 1],
       [6, 5],
       [7, 6],
       [4, 7],
       [3, 4],
       [2, 3],
       [5, 2],
       [3, 5],
       [6, 3],
       [2, 6]]],
     ['edgelist',
      'G812',
      7,
      [[1, 5],
       [4, 1],
       [5, 4],
       [3, 5],
       [7, 3],
       [2, 7],
       [6, 2],
       [3, 6],
       [4, 3],
       [2, 4],
       [5, 2]]],
     ['edgelist',
      'G813',
      7,
      [[1, 2],
       [7, 6],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [7, 4],
       [7, 5]]],
     ['edgelist',
      'G814',
      7,
      [[5, 2],
       [1, 5],
       [2, 1],
       [4, 2],
       [1, 4],
       [6, 2],
       [7, 6],
       [2, 7],
       [3, 2],
       [6, 3],
       [7, 3]]],
     ['edgelist',
      'G815',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 5],
       [7, 6],
       [5, 7]]],
     ['edgelist',
      'G816',
      7,
      [[2, 1],
       [3, 2],
       [4, 3],
       [5, 4],
       [1, 5],
       [3, 1],
       [6, 3],
       [7, 6],
       [4, 7],
       [7, 1],
       [1, 6]]],
     ['edgelist',
      'G817',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [1, 3],
       [5, 1],
       [7, 5],
       [1, 7],
       [4, 7]]],
     ['edgelist',
      'G818',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [3, 1],
       [6, 3],
       [7, 6],
       [5, 7],
       [1, 6],
       [7, 1]]],
     ['edgelist',
      'G819',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [3, 7],
       [4, 7],
       [1, 4],
       [5, 1]]],
     ['edgelist',
      'G820',
      7,
      [[5, 7],
       [6, 5],
       [7, 6],
       [4, 7],
       [6, 4],
       [3, 6],
       [4, 3],
       [6, 1],
       [7, 1],
       [2, 1],
       [3, 2]]],
     ['edgelist',
      'G821',
      7,
      [[3, 1],
       [5, 3],
       [6, 5],
       [4, 6],
       [2, 4],
       [1, 2],
       [3, 2],
       [4, 3],
       [7, 4],
       [6, 7],
       [5, 4]]],
     ['edgelist',
      'G822',
      7,
      [[5, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [1, 4],
       [2, 1],
       [4, 3],
       [4, 6],
       [3, 6],
       [7, 1],
       [5, 7]]],
     ['edgelist',
      'G823',
      7,
      [[1, 2],
       [1, 3],
       [3, 4],
       [6, 2],
       [2, 4],
       [6, 3],
       [7, 4],
       [7, 1],
       [6, 4],
       [5, 6],
       [4, 5]]],
     ['edgelist',
      'G824',
      7,
      [[5, 1],
       [2, 5],
       [7, 2],
       [1, 7],
       [4, 1],
       [2, 4],
       [6, 2],
       [1, 6],
       [7, 6],
       [3, 4],
       [1, 3]]],
     ['edgelist',
      'G825',
      7,
      [[1, 2],
       [6, 1],
       [5, 6],
       [2, 5],
       [3, 2],
       [4, 3],
       [5, 4],
       [5, 3],
       [7, 5],
       [3, 7],
       [4, 7]]],
     ['edgelist',
      'G826',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [6, 2],
       [4, 6],
       [7, 4],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G827',
      7,
      [[7, 4],
       [6, 7],
       [3, 6],
       [4, 3],
       [6, 4],
       [5, 6],
       [3, 5],
       [2, 3],
       [6, 2],
       [1, 2],
       [5, 1]]],
     ['edgelist',
      'G828',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 2],
       [7, 6],
       [5, 7]]],
     ['edgelist',
      'G829',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [6, 3],
       [7, 6],
       [3, 7],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2]]],
     ['edgelist',
      'G830',
      7,
      [[6, 1],
       [1, 2],
       [4, 1],
       [6, 4],
       [3, 6],
       [7, 3],
       [5, 7],
       [6, 5],
       [2, 6],
       [7, 2],
       [4, 7]]],
     ['edgelist',
      'G831',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [3, 2],
       [6, 2],
       [3, 6],
       [7, 5],
       [7, 3],
       [4, 7]]],
     ['edgelist',
      'G832',
      7,
      [[4, 3],
       [7, 4],
       [6, 7],
       [1, 6],
       [3, 1],
       [2, 3],
       [1, 2],
       [6, 2],
       [3, 6],
       [5, 3],
       [7, 5]]],
     ['edgelist',
      'G833',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [5, 2],
       [6, 4],
       [6, 2],
       [7, 5],
       [7, 6],
       [4, 7]]],
     ['edgelist',
      'G834',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 4],
       [7, 1],
       [7, 3],
       [7, 4],
       [6, 7]]],
     ['edgelist',
      'G835',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [6, 2],
       [7, 6],
       [5, 7],
       [4, 7],
       [2, 7]]],
     ['edgelist',
      'G836',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [6, 2],
       [7, 6],
       [2, 7],
       [3, 7],
       [5, 3]]],
     ['edgelist',
      'G837',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 4],
       [5, 3],
       [7, 2],
       [7, 5],
       [6, 3],
       [4, 6]]],
     ['edgelist',
      'G838',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [6, 2],
       [7, 2],
       [5, 7],
       [7, 6],
       [3, 7]]],
     ['edgelist',
      'G839',
      7,
      [[1, 4],
       [1, 7],
       [2, 3],
       [2, 6],
       [3, 5],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G840',
      7,
      [[6, 2],
       [7, 6],
       [5, 7],
       [4, 5],
       [3, 4],
       [1, 3],
       [2, 1],
       [6, 1],
       [7, 1],
       [3, 7],
       [5, 3]]],
     ['edgelist',
      'G841',
      7,
      [[2, 1],
       [3, 2],
       [4, 3],
       [5, 4],
       [1, 5],
       [6, 3],
       [4, 6],
       [7, 1],
       [7, 6],
       [7, 3],
       [4, 7]]],
     ['edgelist',
      'G842',
      7,
      [[1, 4],
       [5, 1],
       [3, 5],
       [4, 3],
       [2, 4],
       [5, 2],
       [6, 2],
       [7, 1],
       [7, 2],
       [6, 4],
       [5, 6]]],
     ['edgelist',
      'G843',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G844',
      7,
      [[1, 3],
       [2, 1],
       [3, 2],
       [1, 4],
       [4, 2],
       [6, 5],
       [6, 4],
       [7, 5],
       [7, 3],
       [7, 1],
       [2, 7]]],
     ['edgelist',
      'G845',
      7,
      [[5, 2],
       [6, 5],
       [3, 6],
       [2, 3],
       [1, 2],
       [6, 1],
       [7, 6],
       [4, 7],
       [3, 4],
       [1, 3],
       [5, 1]]],
     ['edgelist',
      'G846',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [7, 2],
       [7, 3],
       [6, 2],
       [4, 6],
       [6, 3],
       [5, 6]]],
     ['edgelist',
      'G847',
      7,
      [[1, 6],
       [1, 7],
       [2, 5],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6]]],
     ['edgelist',
      'G848',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 4],
       [7, 5],
       [7, 3],
       [3, 1],
       [5, 1]]],
     ['edgelist',
      'G849',
      7,
      [[1, 3],
       [2, 1],
       [3, 2],
       [1, 4],
       [4, 2],
       [6, 5],
       [6, 4],
       [7, 5],
       [7, 3],
       [5, 1],
       [2, 5]]],
     ['edgelist',
      'G850',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [7, 3],
       [1, 7],
       [2, 7]]],
     ['edgelist',
      'G851',
      7,
      [[1, 4],
       [5, 1],
       [2, 5],
       [4, 2],
       [5, 4],
       [1, 2],
       [3, 5],
       [7, 3],
       [6, 7],
       [3, 6],
       [2, 3]]],
     ['edgelist',
      'G852',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 5],
       [6, 3],
       [6, 4],
       [7, 2],
       [7, 6]]],
     ['edgelist',
      'G853',
      7,
      [[5, 2],
       [6, 5],
       [3, 6],
       [2, 3],
       [1, 2],
       [5, 1],
       [6, 1],
       [7, 6],
       [4, 7],
       [3, 4],
       [6, 4]]],
     ['edgelist',
      'G854',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 4],
       [5, 3],
       [6, 2],
       [6, 5],
       [7, 6],
       [5, 7]]],
     ['edgelist',
      'G855',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [6, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [7, 4],
       [7, 5]]],
     ['edgelist',
      'G856',
      7,
      [[1, 5],
       [4, 1],
       [2, 4],
       [5, 2],
       [4, 5],
       [6, 2],
       [7, 6],
       [2, 7],
       [3, 2],
       [6, 3],
       [7, 3]]],
     ['edgelist',
      'G857',
      7,
      [[5, 2],
       [1, 5],
       [4, 1],
       [3, 6],
       [6, 5],
       [7, 6],
       [3, 7],
       [2, 3],
       [4, 2],
       [7, 4],
       [3, 4]]],
     ['edgelist',
      'G858',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 3],
       [6, 5],
       [6, 4],
       [7, 1],
       [7, 6],
       [4, 7]]],
     ['edgelist',
      'G859',
      7,
      [[6, 3],
       [3, 5],
       [6, 4],
       [5, 2],
       [6, 5],
       [1, 2],
       [4, 1],
       [1, 3],
       [7, 3],
       [7, 4],
       [1, 7]]],
     ['edgelist',
      'G860',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 5],
       [6, 3],
       [6, 4],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G861',
      7,
      [[1, 4],
       [1, 5],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G862',
      7,
      [[5, 1],
       [4, 5],
       [6, 4],
       [1, 6],
       [2, 1],
       [3, 2],
       [4, 3],
       [5, 2],
       [6, 3],
       [7, 5],
       [6, 7]]],
     ['edgelist',
      'G863',
      7,
      [[3, 4],
       [5, 3],
       [1, 5],
       [6, 1],
       [2, 6],
       [5, 2],
       [4, 5],
       [6, 4],
       [2, 1],
       [7, 6],
       [7, 3]]],
     ['edgelist',
      'G864',
      7,
      [[5, 2],
       [1, 5],
       [4, 1],
       [5, 4],
       [6, 5],
       [7, 6],
       [3, 7],
       [2, 3],
       [4, 2],
       [7, 4],
       [3, 6]]],
     ['edgelist',
      'G865',
      7,
      [[1, 4],
       [5, 1],
       [3, 5],
       [4, 3],
       [2, 4],
       [1, 2],
       [7, 1],
       [6, 7],
       [3, 6],
       [2, 6],
       [5, 2]]],
     ['edgelist',
      'G866',
      7,
      [[1, 4],
       [1, 5],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G867',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 4],
       [5, 3],
       [6, 2],
       [6, 5],
       [7, 2],
       [6, 7]]],
     ['edgelist',
      'G868',
      7,
      [[5, 2],
       [6, 5],
       [7, 6],
       [4, 7],
       [3, 4],
       [2, 3],
       [1, 2],
       [6, 1],
       [5, 1],
       [6, 3],
       [7, 3]]],
     ['edgelist',
      'G869',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [5, 3],
       [4, 6],
       [5, 6],
       [4, 1],
       [7, 6],
       [7, 2]]],
     ['edgelist',
      'G870',
      7,
      [[1, 5],
       [2, 1],
       [5, 2],
       [4, 5],
       [3, 4],
       [2, 3],
       [7, 2],
       [6, 7],
       [4, 6],
       [6, 5],
       [3, 7]]],
     ['edgelist',
      'G871',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [6, 2],
       [5, 3],
       [7, 3],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G872',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 3],
       [2, 7],
       [6, 3],
       [5, 2],
       [1, 4]]],
     ['edgelist',
      'G873',
      7,
      [[1, 4],
       [1, 5],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G874',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [6, 5],
       [1, 5],
       [6, 4],
       [6, 2],
       [7, 4],
       [7, 5],
       [5, 3],
       [1, 4]]],
     ['edgelist',
      'G875',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 7],
       [4, 7],
       [5, 6]]],
     ['edgelist',
      'G876',
      7,
      [[5, 4],
       [3, 5],
       [4, 3],
       [1, 4],
       [3, 2],
       [6, 5],
       [6, 1],
       [7, 5],
       [7, 2],
       [2, 6],
       [1, 7]]],
     ['edgelist',
      'G877',
      7,
      [[7, 5],
       [4, 7],
       [2, 4],
       [5, 2],
       [1, 5],
       [3, 1],
       [4, 3],
       [1, 2],
       [6, 1],
       [7, 6],
       [6, 3]]],
     ['edgelist',
      'G878',
      7,
      [[7, 2],
       [3, 7],
       [2, 3],
       [1, 2],
       [4, 1],
       [5, 4],
       [6, 5],
       [4, 6],
       [3, 1],
       [5, 1],
       [6, 7]]],
     ['edgelist',
      'G879',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [5, 4],
       [1, 3]]],
     ['edgelist',
      'G880',
      7,
      [[4, 7],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [7, 6],
       [3, 7],
       [6, 2],
       [1, 4],
       [2, 7],
       [1, 2],
       [1, 7]]],
     ['edgelist',
      'G881',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [6, 2],
       [3, 5]]],
     ['edgelist',
      'G882',
      7,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2],
       [3, 4]]],
     ['edgelist',
      'G883',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 3],
       [4, 2],
       [5, 1],
       [3, 5],
       [6, 2],
       [1, 6],
       [5, 6],
       [4, 5],
       [6, 4]]],
     ['edgelist',
      'G884',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 3],
       [2, 6],
       [7, 2]]],
     ['edgelist',
      'G885',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 3],
       [5, 7],
       [6, 4]]],
     ['edgelist',
      'G886',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [3, 2],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4],
       [2, 7]]],
     ['edgelist',
      'G887',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [3, 2],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4],
       [4, 7]]],
     ['edgelist',
      'G888',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G889',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1],
       [5, 6],
       [7, 2]]],
     ['edgelist',
      'G890',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [3, 2],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4],
       [1, 7]]],
     ['edgelist',
      'G891',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1],
       [5, 6],
       [1, 7]]],
     ['edgelist',
      'G892',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1],
       [5, 6],
       [3, 7]]],
     ['edgelist',
      'G893',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [1, 3],
       [2, 7]]],
     ['edgelist',
      'G894',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [1, 3],
       [5, 7]]],
     ['edgelist',
      'G895',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [1, 3],
       [7, 2],
       [7, 6]]],
     ['edgelist',
      'G896',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 2],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [2, 7]]],
     ['edgelist',
      'G897',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [4, 6],
       [1, 4],
       [6, 7]]],
     ['edgelist',
      'G898',
      7,
      [[4, 7],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [7, 6],
       [3, 7],
       [6, 2],
       [1, 4],
       [2, 7],
       [1, 2],
       [2, 5]]],
     ['edgelist',
      'G899',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [4, 6],
       [1, 4],
       [4, 7]]],
     ['edgelist',
      'G900',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [3, 2],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4],
       [5, 7]]],
     ['edgelist',
      'G901',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 2],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [3, 7]]],
     ['edgelist',
      'G902',
      7,
      [[4, 7],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [7, 6],
       [3, 7],
       [6, 2],
       [1, 4],
       [2, 7],
       [1, 2],
       [1, 5]]],
     ['edgelist',
      'G903',
      7,
      [[2, 4],
       [5, 2],
       [4, 5],
       [3, 4],
       [1, 3],
       [5, 1],
       [6, 5],
       [3, 6],
       [5, 3],
       [1, 6],
       [2, 6],
       [4, 7]]],
     ['edgelist',
      'G904',
      7,
      [[2, 4],
       [5, 2],
       [4, 5],
       [3, 4],
       [1, 3],
       [5, 1],
       [6, 5],
       [3, 6],
       [5, 3],
       [1, 6],
       [2, 6],
       [1, 7]]],
     ['edgelist',
      'G905',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 1],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G906',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [1, 4],
       [6, 7]]],
     ['edgelist',
      'G907',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3],
       [5, 7]]],
     ['edgelist',
      'G908',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3],
       [1, 7]]],
     ['edgelist',
      'G909',
      7,
      [[4, 7],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [7, 6],
       [3, 7],
       [6, 2],
       [1, 4],
       [2, 7],
       [1, 2],
       [5, 6]]],
     ['edgelist',
      'G910',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3],
       [4, 7]]],
     ['edgelist',
      'G911',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [1, 4],
       [1, 7]]],
     ['edgelist',
      'G912',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [1, 3],
       [6, 7]]],
     ['edgelist',
      'G913',
      7,
      [[1, 4],
       [7, 1],
       [6, 7],
       [4, 6],
       [2, 4],
       [7, 2],
       [5, 7],
       [4, 5],
       [3, 4],
       [7, 3],
       [4, 7],
       [6, 5]]],
     ['edgelist',
      'G914',
      7,
      [[1, 2],
       [5, 1],
       [6, 5],
       [2, 6],
       [5, 2],
       [3, 5],
       [2, 3],
       [7, 2],
       [5, 7],
       [3, 7],
       [4, 3],
       [5, 4]]],
     ['edgelist',
      'G915',
      7,
      [[5, 2],
       [4, 3],
       [4, 1],
       [5, 3],
       [6, 2],
       [6, 1],
       [4, 6],
       [5, 4],
       [6, 5],
       [7, 6],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G916',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 6],
       [7, 4],
       [2, 7]]],
     ['edgelist',
      'G917',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [7, 4],
       [1, 7],
       [6, 1],
       [2, 6],
       [5, 2],
       [3, 5]]],
     ['edgelist',
      'G918',
      7,
      [[7, 3],
       [6, 7],
       [4, 6],
       [3, 4],
       [2, 3],
       [5, 2],
       [6, 5],
       [3, 6],
       [5, 3],
       [1, 5],
       [2, 1],
       [6, 2]]],
     ['edgelist',
      'G919',
      7,
      [[6, 5],
       [7, 6],
       [4, 7],
       [5, 4],
       [1, 5],
       [4, 1],
       [2, 4],
       [1, 2],
       [5, 2],
       [4, 6],
       [3, 4],
       [5, 3]]],
     ['edgelist',
      'G920',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [6, 1],
       [2, 6],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G921',
      7,
      [[2, 3],
       [1, 2],
       [3, 1],
       [4, 3],
       [1, 4],
       [2, 4],
       [5, 3],
       [1, 5],
       [6, 5],
       [3, 6],
       [7, 3],
       [2, 7]]],
     ['edgelist',
      'G922',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 3],
       [5, 6],
       [7, 5],
       [4, 7]]],
     ['edgelist',
      'G923',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [6, 1],
       [2, 6],
       [7, 2],
       [3, 7]]],
     ['edgelist',
      'G924',
      7,
      [[2, 3],
       [1, 2],
       [3, 1],
       [4, 3],
       [1, 4],
       [2, 4],
       [5, 3],
       [1, 5],
       [7, 5],
       [3, 7],
       [6, 3],
       [5, 6]]],
     ['edgelist',
      'G925',
      7,
      [[2, 1],
       [3, 2],
       [1, 3],
       [4, 1],
       [5, 4],
       [1, 5],
       [6, 1],
       [4, 6],
       [5, 6],
       [7, 5],
       [4, 7],
       [7, 1]]],
     ['edgelist',
      'G926',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [7, 6],
       [3, 7]]],
     ['edgelist',
      'G927',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [7, 5],
       [1, 7],
       [6, 1],
       [4, 6]]],
     ['edgelist',
      'G928',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [2, 6],
       [7, 2],
       [1, 7],
       [6, 1],
       [5, 6]]],
     ['edgelist',
      'G929',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [7, 2],
       [1, 7],
       [6, 1],
       [3, 6]]],
     ['edgelist',
      'G930',
      7,
      [[6, 5],
       [4, 6],
       [5, 4],
       [7, 5],
       [4, 7],
       [3, 4],
       [5, 3],
       [1, 5],
       [4, 1],
       [2, 1],
       [3, 2],
       [7, 3]]],
     ['edgelist',
      'G931',
      7,
      [[5, 2],
       [4, 3],
       [4, 1],
       [5, 3],
       [6, 2],
       [6, 1],
       [4, 6],
       [5, 4],
       [6, 5],
       [7, 6],
       [1, 7],
       [4, 7]]],
     ['edgelist',
      'G932',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [7, 2],
       [1, 7],
       [6, 1],
       [2, 6]]],
     ['edgelist',
      'G933',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [4, 2],
       [6, 2],
       [7, 6],
       [5, 7]]],
     ['edgelist',
      'G934',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [6, 5],
       [4, 6],
       [7, 4],
       [5, 7]]],
     ['edgelist',
      'G935',
      7,
      [[2, 1],
       [3, 2],
       [4, 3],
       [1, 4],
       [5, 4],
       [2, 5],
       [5, 1],
       [6, 5],
       [1, 6],
       [4, 6],
       [7, 1],
       [2, 7]]],
     ['edgelist',
      'G936',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [7, 3],
       [5, 7],
       [6, 1],
       [2, 6]]],
     ['edgelist',
      'G937',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [6, 5],
       [3, 6],
       [2, 6],
       [7, 2],
       [6, 7]]],
     ['edgelist',
      'G938',
      7,
      [[1, 3],
       [2, 1],
       [3, 2],
       [1, 4],
       [4, 2],
       [5, 3],
       [6, 4],
       [7, 2],
       [7, 5],
       [5, 1],
       [4, 5],
       [2, 6]]],
     ['edgelist',
      'G939',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 3],
       [5, 4],
       [5, 2],
       [1, 5],
       [6, 3],
       [6, 5],
       [7, 1],
       [4, 7]]],
     ['edgelist',
      'G940',
      7,
      [[6, 1],
       [3, 6],
       [7, 3],
       [4, 7],
       [3, 4],
       [2, 3],
       [1, 2],
       [5, 1],
       [2, 5],
       [6, 2],
       [7, 6],
       [1, 3]]],
     ['edgelist',
      'G941',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [7, 5],
       [3, 7],
       [6, 3],
       [4, 6]]],
     ['edgelist',
      'G942',
      7,
      [[1, 3],
       [2, 1],
       [6, 2],
       [4, 6],
       [7, 4],
       [3, 7],
       [5, 3],
       [4, 5],
       [6, 5],
       [3, 6],
       [2, 3],
       [5, 2]]],
     ['edgelist',
      'G943',
      7,
      [[1, 3],
       [2, 1],
       [3, 2],
       [1, 4],
       [4, 2],
       [5, 1],
       [2, 5],
       [5, 3],
       [4, 5],
       [6, 5],
       [7, 6],
       [4, 7]]],
     ['edgelist',
      'G944',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [4, 2],
       [4, 3],
       [7, 2],
       [3, 7],
       [5, 7],
       [4, 5],
       [6, 4],
       [7, 6]]],
     ['edgelist',
      'G945',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [6, 1],
       [7, 6],
       [1, 7],
       [4, 5]]],
     ['edgelist',
      'G946',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 2],
       [3, 6],
       [7, 1],
       [4, 7]]],
     ['edgelist',
      'G947',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 4],
       [6, 2],
       [6, 5],
       [7, 4],
       [5, 7],
       [2, 7],
       [7, 6]]],
     ['edgelist',
      'G948',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G949',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 3],
       [7, 6],
       [1, 7],
       [7, 3],
       [1, 6],
       [2, 6],
       [7, 2]]],
     ['edgelist',
      'G950',
      7,
      [[1, 2],
       [7, 6],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [7, 4],
       [7, 5],
       [6, 2]]],
     ['edgelist',
      'G951',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 5],
       [7, 2],
       [6, 7]]],
     ['edgelist',
      'G952',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 2],
       [5, 6],
       [7, 5],
       [6, 7]]],
     ['edgelist',
      'G953',
      7,
      [[3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 2],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [7, 4],
       [7, 2]]],
     ['edgelist',
      'G954',
      7,
      [[1, 5],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G955',
      7,
      [[1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G956',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [3, 2],
       [5, 7],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4],
       [7, 2],
       [3, 7]]],
     ['edgelist',
      'G957',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [6, 5],
       [1, 5],
       [6, 4],
       [6, 2],
       [7, 4],
       [7, 5],
       [5, 3],
       [1, 4],
       [5, 4]]],
     ['edgelist',
      'G958',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [7, 2],
       [7, 6]]],
     ['edgelist',
      'G959',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 2],
       [7, 6],
       [1, 7],
       [2, 7]]],
     ['edgelist',
      'G960',
      7,
      [[1, 4],
       [5, 1],
       [3, 5],
       [4, 3],
       [2, 4],
       [5, 2],
       [2, 1],
       [6, 2],
       [6, 3],
       [7, 2],
       [3, 7],
       [5, 7]]],
     ['edgelist',
      'G961',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 3],
       [6, 1],
       [6, 5],
       [5, 2],
       [2, 7],
       [6, 4],
       [2, 6],
       [7, 3]]],
     ['edgelist',
      'G962',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 3],
       [6, 1],
       [6, 5],
       [5, 2],
       [2, 6],
       [6, 4],
       [7, 2],
       [5, 7]]],
     ['edgelist',
      'G963',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 3],
       [6, 1],
       [6, 5],
       [5, 2],
       [2, 6],
       [6, 4],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G964',
      7,
      [[5, 4],
       [2, 3],
       [1, 2],
       [1, 4],
       [5, 1],
       [7, 5],
       [5, 3],
       [6, 5],
       [7, 3],
       [7, 4],
       [4, 3],
       [6, 2]]],
     ['edgelist',
      'G965',
      7,
      [[3, 4],
       [5, 3],
       [1, 5],
       [7, 1],
       [7, 6],
       [5, 6],
       [2, 4],
       [6, 2],
       [1, 6],
       [7, 2],
       [4, 7],
       [6, 4]]],
     ['edgelist',
      'G966',
      7,
      [[1, 4],
       [1, 6],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G967',
      7,
      [[1, 4],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G968',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [5, 4],
       [6, 4],
       [7, 2],
       [4, 7],
       [7, 3],
       [1, 7],
       [5, 7]]],
     ['edgelist',
      'G969',
      7,
      [[1, 2],
       [3, 5],
       [1, 3],
       [7, 2],
       [4, 2],
       [4, 3],
       [5, 2],
       [6, 2],
       [6, 3],
       [6, 4],
       [1, 4],
       [5, 7]]],
     ['edgelist',
      'G970',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [7, 4],
       [2, 7]]],
     ['edgelist',
      'G971',
      7,
      [[5, 4],
       [2, 3],
       [6, 1],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [6, 2],
       [7, 3],
       [7, 4],
       [4, 3],
       [7, 5]]],
     ['edgelist',
      'G972',
      7,
      [[3, 4],
       [5, 3],
       [6, 5],
       [1, 6],
       [7, 1],
       [2, 7],
       [4, 2],
       [7, 4],
       [6, 4],
       [2, 6],
       [5, 1],
       [4, 1]]],
     ['edgelist',
      'G973',
      7,
      [[1, 4],
       [1, 6],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G974',
      7,
      [[4, 3],
       [2, 3],
       [6, 1],
       [1, 4],
       [5, 1],
       [5, 2],
       [7, 5],
       [6, 2],
       [7, 3],
       [7, 4],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G975',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 7],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G976',
      7,
      [[1, 4],
       [1, 6],
       [2, 3],
       [2, 5],
       [2, 7],
       [3, 5],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G977',
      7,
      [[1, 4],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G978',
      7,
      [[1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G979',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 3],
       [6, 1],
       [6, 5],
       [5, 2],
       [4, 5],
       [6, 4],
       [3, 7],
       [7, 2]]],
     ['edgelist',
      'G980',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 3],
       [6, 1],
       [6, 5],
       [5, 2],
       [4, 5],
       [6, 4],
       [7, 2],
       [7, 6]]],
     ['edgelist',
      'G981',
      7,
      [[1, 3],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G982',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G983',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 2],
       [3, 5],
       [6, 3],
       [1, 6],
       [5, 4],
       [7, 6],
       [7, 5],
       [4, 6]]],
     ['edgelist',
      'G984',
      7,
      [[1, 3],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 6],
       [3, 4],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G985',
      7,
      [[1, 3],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G986',
      7,
      [[1, 3],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G987',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G988',
      7,
      [[1, 6],
       [1, 7],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G989',
      7,
      [[4, 1],
       [3, 4],
       [5, 3],
       [1, 5],
       [6, 2],
       [6, 3],
       [7, 2],
       [7, 1],
       [4, 7],
       [6, 4],
       [5, 6],
       [7, 5]]],
     ['edgelist',
      'G990',
      7,
      [[1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G991',
      7,
      [[1, 2],
       [1, 3],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G992',
      7,
      [[4, 1],
       [3, 4],
       [5, 3],
       [1, 5],
       [6, 2],
       [6, 3],
       [7, 2],
       [7, 1],
       [4, 7],
       [6, 4],
       [7, 5],
       [2, 4]]],
     ['edgelist',
      'G993',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G994',
      7,
      [[3, 4],
       [5, 3],
       [6, 3],
       [5, 2],
       [7, 1],
       [4, 1],
       [4, 2],
       [7, 4],
       [6, 7],
       [2, 6],
       [5, 1],
       [4, 5]]],
     ['edgelist',
      'G995',
      7,
      [[3, 4],
       [5, 3],
       [5, 2],
       [3, 6],
       [7, 1],
       [7, 5],
       [4, 2],
       [7, 4],
       [1, 4],
       [2, 6],
       [5, 1],
       [6, 4]]],
     ['edgelist',
      'G996',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 7],
       [3, 4],
       [3, 7],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G997',
      7,
      [[4, 1],
       [3, 4],
       [5, 3],
       [1, 5],
       [6, 2],
       [6, 3],
       [7, 2],
       [7, 1],
       [4, 7],
       [2, 4],
       [7, 5],
       [6, 5]]],
     ['edgelist',
      'G998',
      7,
      [[7, 4],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 3],
       [6, 1],
       [1, 7],
       [5, 2],
       [4, 5],
       [7, 6],
       [6, 2],
       [1, 5]]],
     ['edgelist',
      'G999',
      7,
      [[1, 4],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1000',
      7,
      [[1, 4],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 6],
       [4, 5],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1001',
      7,
      [[1, 4],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1002',
      7,
      [[1, 5],
       [4, 1],
       [2, 4],
       [5, 2],
       [2, 1],
       [5, 6],
       [3, 5],
       [7, 3],
       [6, 7],
       [3, 6],
       [4, 3],
       [7, 4]]],
     ['edgelist',
      'G1003',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [6, 3],
       [6, 5],
       [7, 5],
       [7, 4],
       [7, 3],
       [6, 4]]],
     ['edgelist',
      'G1004',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1005',
      7,
      [[4, 1],
       [5, 3],
       [4, 2],
       [5, 1],
       [6, 3],
       [6, 2],
       [5, 4],
       [6, 5],
       [4, 6],
       [7, 2],
       [1, 7],
       [3, 7]]],
     ['edgelist',
      'G1006',
      7,
      [[2, 1],
       [5, 2],
       [1, 5],
       [6, 1],
       [7, 6],
       [2, 7],
       [4, 5],
       [6, 4],
       [3, 4],
       [6, 3],
       [7, 4],
       [3, 7]]],
     ['edgelist',
      'G1007',
      7,
      [[1, 2],
       [3, 1],
       [3, 4],
       [4, 5],
       [1, 5],
       [1, 6],
       [7, 2],
       [5, 7],
       [7, 6],
       [3, 7],
       [4, 2],
       [6, 4]]],
     ['edgelist',
      'G1008',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [7, 1],
       [7, 2],
       [7, 3],
       [7, 4],
       [7, 5],
       [7, 6]]],
     ['edgelist',
      'G1009',
      7,
      [[4, 7],
       [2, 3],
       [1, 7],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 2],
       [3, 6],
       [5, 6],
       [7, 5]]],
     ['edgelist',
      'G1010',
      7,
      [[2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1011',
      7,
      [[2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1012',
      7,
      [[1, 7],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1013',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [5, 4],
       [1, 3],
       [7, 5]]],
     ['edgelist',
      'G1014',
      7,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2],
       [1, 5],
       [2, 7]]],
     ['edgelist',
      'G1015',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1016',
      7,
      [[1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1017',
      7,
      [[1, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1018',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1019',
      7,
      [[1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1020',
      7,
      [[1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1021',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3],
       [5, 6],
       [2, 7]]],
     ['edgelist',
      'G1022',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [5, 6],
       [1, 5],
       [2, 4],
       [5, 2],
       [3, 5],
       [1, 4],
       [6, 4],
       [5, 4],
       [1, 3],
       [6, 7]]],
     ['edgelist',
      'G1023',
      7,
      [[1, 6],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1024',
      7,
      [[1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1025',
      7,
      [[6, 7],
       [1, 6],
       [7, 1],
       [5, 7],
       [6, 5],
       [2, 6],
       [7, 2],
       [4, 7],
       [6, 4],
       [3, 6],
       [7, 3],
       [2, 1],
       [3, 2]]],
     ['edgelist',
      'G1026',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 7]]],
     ['edgelist',
      'G1027',
      7,
      [[4, 5],
       [1, 4],
       [5, 1],
       [2, 5],
       [4, 2],
       [3, 4],
       [5, 3],
       [2, 1],
       [3, 2],
       [7, 1],
       [4, 7],
       [6, 4],
       [5, 6]]],
     ['edgelist',
      'G1028',
      7,
      [[4, 5],
       [1, 4],
       [5, 1],
       [2, 5],
       [4, 2],
       [3, 4],
       [5, 3],
       [2, 1],
       [3, 2],
       [7, 1],
       [4, 7],
       [6, 4],
       [1, 6]]],
     ['edgelist',
      'G1029',
      7,
      [[4, 5],
       [1, 4],
       [5, 1],
       [2, 5],
       [4, 2],
       [3, 4],
       [5, 3],
       [2, 1],
       [3, 2],
       [7, 5],
       [1, 7],
       [6, 1],
       [4, 6]]],
     ['edgelist',
      'G1030',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1031',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 6],
       [5, 7]]],
     ['edgelist',
      'G1032',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [6, 2],
       [7, 6],
       [2, 7]]],
     ['edgelist',
      'G1033',
      7,
      [[1, 5],
       [1, 7],
       [2, 4],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1034',
      7,
      [[1, 6],
       [1, 7],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1035',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1036',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [6, 4],
       [7, 6],
       [5, 7]]],
     ['edgelist',
      'G1037',
      7,
      [[1, 6],
       [1, 7],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1038',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [7, 2],
       [7, 6],
       [6, 2]]],
     ['edgelist',
      'G1039',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 5],
       [1, 6],
       [7, 1],
       [4, 7],
       [7, 5]]],
     ['edgelist',
      'G1040',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [7, 2],
       [6, 2],
       [3, 7]]],
     ['edgelist',
      'G1041',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [6, 7]]],
     ['edgelist',
      'G1042',
      7,
      [[2, 1],
       [3, 2],
       [5, 3],
       [2, 5],
       [4, 2],
       [1, 4],
       [3, 4],
       [6, 3],
       [2, 6],
       [1, 6],
       [7, 1],
       [2, 7],
       [3, 7]]],
     ['edgelist',
      'G1043',
      7,
      [[3, 6],
       [7, 3],
       [6, 7],
       [5, 6],
       [4, 5],
       [1, 4],
       [5, 1],
       [2, 5],
       [4, 2],
       [7, 4],
       [3, 2],
       [5, 3],
       [4, 3]]],
     ['edgelist',
      'G1044',
      7,
      [[1, 4],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1045',
      7,
      [[3, 5],
       [4, 3],
       [2, 4],
       [5, 2],
       [1, 5],
       [4, 1],
       [7, 4],
       [2, 7],
       [6, 2],
       [5, 6],
       [7, 5],
       [4, 6],
       [2, 3]]],
     ['edgelist',
      'G1046',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 5],
       [3, 6],
       [4, 6],
       [4, 7]]],
     ['edgelist',
      'G1047',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 6],
       [6, 7]]],
     ['edgelist',
      'G1048',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 6],
       [4, 7],
       [5, 6]]],
     ['edgelist',
      'G1049',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 7],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6]]],
     ['edgelist',
      'G1050',
      7,
      [[1, 3],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1051',
      7,
      [[3, 6],
       [2, 3],
       [6, 2],
       [5, 6],
       [4, 5],
       [1, 4],
       [5, 1],
       [4, 3],
       [5, 3],
       [2, 4],
       [7, 4],
       [3, 7],
       [2, 7]]],
     ['edgelist',
      'G1052',
      7,
      [[1, 5],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1053',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [1, 5],
       [7, 2],
       [5, 7]]],
     ['edgelist',
      'G1054',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [2, 1],
       [6, 3],
       [6, 1],
       [7, 6],
       [2, 7],
       [5, 1]]],
     ['edgelist',
      'G1055',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [2, 4],
       [4, 3],
       [5, 1],
       [3, 6],
       [4, 5],
       [6, 4],
       [1, 6],
       [7, 5],
       [7, 2]]],
     ['edgelist',
      'G1056',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [2, 1],
       [6, 3],
       [7, 3],
       [6, 7],
       [1, 6],
       [2, 3]]],
     ['edgelist',
      'G1057',
      7,
      [[6, 5],
       [7, 3],
       [7, 5],
       [5, 4],
       [6, 1],
       [4, 2],
       [4, 3],
       [7, 4],
       [6, 7],
       [5, 1],
       [2, 5],
       [6, 2],
       [1, 4]]],
     ['edgelist',
      'G1058',
      7,
      [[1, 3],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1059',
      7,
      [[2, 6],
       [5, 2],
       [1, 5],
       [6, 1],
       [3, 6],
       [5, 3],
       [4, 5],
       [6, 4],
       [1, 2],
       [3, 1],
       [4, 3],
       [7, 6],
       [7, 5]]],
     ['edgelist',
      'G1060',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [1, 5],
       [1, 7],
       [5, 7]]],
     ['edgelist',
      'G1061',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [6, 1],
       [5, 6],
       [2, 6],
       [7, 2],
       [1, 7],
       [4, 7]]],
     ['edgelist',
      'G1062',
      7,
      [[1, 6],
       [1, 7],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1063',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [5, 2],
       [6, 4],
       [3, 6],
       [2, 1],
       [7, 4],
       [5, 7],
       [6, 2]]],
     ['edgelist',
      'G1064',
      7,
      [[6, 3],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [2, 1],
       [7, 3],
       [7, 4],
       [2, 3],
       [4, 2],
       [5, 1]]],
     ['edgelist',
      'G1065',
      7,
      [[2, 1],
       [3, 2],
       [1, 3],
       [1, 4],
       [4, 3],
       [7, 3],
       [2, 7],
       [6, 2],
       [7, 6],
       [5, 7],
       [6, 5],
       [1, 6],
       [5, 1]]],
     ['edgelist',
      'G1066',
      7,
      [[1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G1067',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1068',
      7,
      [[1, 2],
       [2, 3],
       [5, 2],
       [4, 2],
       [1, 5],
       [3, 4],
       [1, 4],
       [3, 1],
       [6, 1],
       [7, 6],
       [5, 7],
       [4, 6],
       [5, 3]]],
     ['edgelist',
      'G1069',
      7,
      [[1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1070',
      7,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [1, 5],
       [3, 4],
       [7, 6],
       [1, 7]]],
     ['edgelist',
      'G1071',
      7,
      [[6, 3],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [2, 1],
       [7, 3],
       [7, 4],
       [3, 4],
       [6, 1],
       [5, 1]]],
     ['edgelist',
      'G1072',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [6, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [7, 4],
       [7, 5],
       [5, 3],
       [1, 4]]],
     ['edgelist',
      'G1073',
      7,
      [[1, 2],
       [1, 7],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1074',
      7,
      [[1, 2],
       [1, 7],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1075',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [7, 3],
       [1, 7],
       [6, 1],
       [3, 6],
       [6, 4],
       [5, 6],
       [7, 5],
       [4, 7]]],
     ['edgelist',
      'G1076',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [7, 6],
       [1, 7],
       [1, 3],
       [3, 6],
       [6, 4],
       [5, 6],
       [7, 5],
       [4, 7]]],
     ['edgelist',
      'G1077',
      7,
      [[4, 5],
       [1, 4],
       [5, 1],
       [4, 7],
       [4, 2],
       [3, 4],
       [5, 3],
       [2, 1],
       [3, 2],
       [6, 3],
       [4, 6],
       [7, 3],
       [6, 7]]],
     ['edgelist',
      'G1078',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 5],
       [7, 6],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G1079',
      7,
      [[2, 1],
       [3, 2],
       [7, 1],
       [2, 5],
       [4, 2],
       [1, 4],
       [3, 4],
       [2, 7],
       [2, 6],
       [3, 7],
       [5, 4],
       [6, 5],
       [7, 6]]],
     ['edgelist',
      'G1080',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [5, 7]]],
     ['edgelist',
      'G1081',
      7,
      [[1, 7],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 3],
       [6, 1],
       [7, 4],
       [5, 2],
       [4, 5],
       [7, 6],
       [2, 6],
       [4, 6],
       [2, 4]]],
     ['edgelist',
      'G1082',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [7, 5],
       [5, 4],
       [6, 5],
       [6, 3],
       [7, 1],
       [4, 7],
       [4, 6]]],
     ['edgelist',
      'G1083',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 7],
       [1, 5],
       [7, 6],
       [1, 7],
       [7, 5],
       [3, 6],
       [6, 4],
       [5, 6],
       [2, 6],
       [7, 2]]],
     ['edgelist',
      'G1084',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1085',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1086',
      7,
      [[3, 4],
       [6, 3],
       [7, 6],
       [4, 7],
       [5, 4],
       [6, 5],
       [1, 6],
       [3, 1],
       [2, 3],
       [1, 2],
       [6, 2],
       [5, 3],
       [7, 5]]],
     ['edgelist',
      'G1087',
      7,
      [[3, 2],
       [1, 6],
       [7, 1],
       [5, 7],
       [6, 5],
       [2, 6],
       [7, 2],
       [4, 7],
       [6, 4],
       [3, 6],
       [7, 3],
       [2, 1],
       [4, 5]]],
     ['edgelist',
      'G1088',
      7,
      [[1, 2],
       [3, 1],
       [3, 4],
       [4, 5],
       [1, 5],
       [1, 6],
       [7, 2],
       [5, 7],
       [7, 6],
       [3, 7],
       [4, 2],
       [6, 4],
       [7, 1]]],
     ['edgelist',
      'G1089',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1090',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [5, 7],
       [6, 2],
       [5, 6],
       [4, 2],
       [6, 3],
       [7, 1],
       [7, 2],
       [3, 2],
       [5, 2]]],
     ['edgelist',
      'G1091',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 4],
       [6, 5],
       [6, 3],
       [6, 2],
       [7, 6],
       [2, 7],
       [3, 7]]],
     ['edgelist',
      'G1092',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G1093',
      7,
      [[4, 1],
       [3, 4],
       [5, 3],
       [1, 5],
       [6, 2],
       [6, 3],
       [7, 2],
       [7, 1],
       [4, 7],
       [2, 4],
       [7, 5],
       [6, 5],
       [6, 4]]],
     ['edgelist',
      'G1094',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1095',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1096',
      7,
      [[1, 3],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 7],
       [3, 4],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1097',
      7,
      [[4, 5],
       [6, 1],
       [4, 6],
       [1, 7],
       [7, 5],
       [3, 4],
       [5, 3],
       [2, 1],
       [3, 2],
       [2, 7],
       [6, 2],
       [3, 6],
       [7, 3]]],
     ['edgelist',
      'G1098',
      7,
      [[1, 3],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1099',
      7,
      [[4, 1],
       [3, 4],
       [5, 3],
       [1, 5],
       [6, 4],
       [6, 3],
       [6, 5],
       [2, 4],
       [2, 1],
       [5, 2],
       [7, 1],
       [4, 7],
       [2, 7]]],
     ['edgelist',
      'G1100',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [7, 1],
       [2, 7],
       [7, 4],
       [5, 7],
       [2, 3],
       [6, 1]]],
     ['edgelist',
      'G1101',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1102',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6]]],
     ['edgelist',
      'G1103',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [6, 4],
       [6, 5],
       [7, 5],
       [7, 3],
       [7, 6],
       [6, 3],
       [4, 7]]],
     ['edgelist',
      'G1104',
      7,
      [[1, 2],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1105',
      7,
      [[1, 2],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1106',
      7,
      [[1, 2],
       [3, 1],
       [3, 4],
       [4, 5],
       [1, 5],
       [1, 6],
       [7, 2],
       [5, 7],
       [7, 6],
       [3, 7],
       [4, 2],
       [6, 4],
       [3, 2]]],
     ['edgelist',
      'G1107',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [2, 4],
       [3, 1],
       [5, 1],
       [6, 4]]],
     ['edgelist',
      'G1108',
      7,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2],
       [3, 4],
       [1, 5],
       [2, 7]]],
     ['edgelist',
      'G1109',
      7,
      [[1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1110',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1111',
      7,
      [[1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1112',
      7,
      [[1, 4],
       [2, 3],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1113',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 7],
       [6, 4],
       [5, 6],
       [7, 5]]],
     ['edgelist',
      'G1114',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [7, 3],
       [2, 7],
       [6, 2],
       [1, 6]]],
     ['edgelist',
      'G1115',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [6, 3],
       [4, 6],
       [7, 5],
       [1, 7]]],
     ['edgelist',
      'G1116',
      7,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [3, 5],
       [6, 2],
       [1, 4],
       [2, 5],
       [1, 2],
       [1, 5],
       [7, 5],
       [1, 7]]],
     ['edgelist',
      'G1117',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [4, 5],
       [4, 6],
       [5, 7]]],
     ['edgelist',
      'G1118',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 7],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1119',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1120',
      7,
      [[4, 5],
       [2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [1, 2],
       [6, 2],
       [1, 5],
       [2, 5],
       [6, 4],
       [3, 6],
       [7, 5],
       [1, 7]]],
     ['edgelist',
      'G1121',
      7,
      [[2, 4],
       [3, 2],
       [1, 3],
       [6, 1],
       [5, 6],
       [4, 5],
       [6, 4],
       [3, 6],
       [2, 1],
       [5, 2],
       [7, 2],
       [6, 2],
       [3, 7],
       [1, 5]]],
     ['edgelist',
      'G1122',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 7],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1123',
      7,
      [[3, 4],
       [5, 3],
       [7, 4],
       [5, 1],
       [7, 1],
       [4, 5],
       [4, 2],
       [6, 5],
       [6, 1],
       [1, 4],
       [2, 6],
       [6, 4],
       [7, 5],
       [7, 2]]],
     ['edgelist',
      'G1124',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [6, 4],
       [7, 6],
       [5, 7],
       [6, 5]]],
     ['edgelist',
      'G1125',
      7,
      [[4, 2],
       [2, 5],
       [3, 4],
       [4, 5],
       [1, 5],
       [2, 6],
       [1, 2],
       [1, 3],
       [3, 6],
       [6, 4],
       [5, 6],
       [2, 3],
       [7, 3],
       [6, 7]]],
     ['edgelist',
      'G1126',
      7,
      [[1, 4],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1127',
      7,
      [[1, 4],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 6],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1128',
      7,
      [[1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1129',
      7,
      [[1, 2],
       [1, 7],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1130',
      7,
      [[1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1131',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [5, 7],
       [6, 3],
       [2, 6],
       [1, 6],
       [7, 4]]],
     ['edgelist',
      'G1132',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [6, 1],
       [6, 2],
       [6, 3],
       [6, 4],
       [6, 5],
       [5, 3],
       [4, 1],
       [7, 2],
       [6, 7]]],
     ['edgelist',
      'G1133',
      7,
      [[1, 5],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1134',
      7,
      [[1, 5],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1135',
      7,
      [[1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1136',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [2, 3],
       [6, 3],
       [5, 1],
       [4, 2],
       [6, 1],
       [7, 6],
       [7, 5],
       [1, 2]]],
     ['edgelist',
      'G1137',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [6, 3],
       [7, 1],
       [7, 2],
       [6, 1],
       [2, 3],
       [4, 2],
       [5, 1]]],
     ['edgelist',
      'G1138',
      7,
      [[6, 7],
       [1, 6],
       [7, 1],
       [5, 7],
       [6, 5],
       [2, 6],
       [7, 2],
       [4, 7],
       [6, 4],
       [3, 6],
       [7, 3],
       [2, 1],
       [3, 2],
       [4, 5]]],
     ['edgelist',
      'G1139',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 6],
       [2, 7],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1140',
      7,
      [[1, 2],
       [3, 1],
       [3, 4],
       [4, 5],
       [1, 5],
       [1, 6],
       [7, 2],
       [5, 7],
       [7, 6],
       [3, 7],
       [4, 2],
       [6, 4],
       [7, 1],
       [4, 7]]],
     ['edgelist',
      'G1141',
      7,
      [[4, 2],
       [5, 3],
       [5, 6],
       [5, 1],
       [2, 5],
       [1, 4],
       [6, 1],
       [6, 3],
       [7, 2],
       [4, 7],
       [7, 1],
       [6, 7],
       [7, 3],
       [5, 7]]],
     ['edgelist',
      'G1142',
      7,
      [[1, 5],
       [4, 1],
       [3, 4],
       [5, 3],
       [2, 5],
       [4, 2],
       [2, 1],
       [3, 2],
       [6, 5],
       [2, 6],
       [7, 2],
       [4, 7],
       [1, 6],
       [7, 1]]],
     ['edgelist',
      'G1143',
      7,
      [[4, 5],
       [5, 3],
       [2, 6],
       [5, 1],
       [2, 5],
       [6, 4],
       [4, 1],
       [6, 3],
       [7, 5],
       [1, 7],
       [4, 7],
       [3, 7],
       [6, 7],
       [2, 7]]],
     ['edgelist',
      'G1144',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 7],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1145',
      7,
      [[3, 4],
       [5, 3],
       [7, 4],
       [5, 1],
       [5, 6],
       [4, 5],
       [4, 2],
       [6, 3],
       [2, 7],
       [6, 7],
       [7, 1],
       [6, 4],
       [7, 5],
       [1, 2]]],
     ['edgelist',
      'G1146',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1147',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1148',
      7,
      [[3, 4],
       [5, 3],
       [7, 4],
       [5, 1],
       [2, 5],
       [7, 1],
       [4, 2],
       [6, 3],
       [5, 6],
       [6, 7],
       [2, 6],
       [6, 4],
       [7, 5],
       [4, 1]]],
     ['edgelist',
      'G1149',
      7,
      [[4, 2],
       [5, 3],
       [1, 4],
       [5, 1],
       [2, 5],
       [6, 4],
       [6, 1],
       [6, 3],
       [7, 5],
       [2, 7],
       [7, 4],
       [1, 7],
       [7, 6],
       [3, 7]]],
     ['edgelist',
      'G1150',
      7,
      [[1, 2],
       [5, 3],
       [4, 1],
       [5, 1],
       [5, 6],
       [6, 4],
       [2, 4],
       [6, 3],
       [7, 5],
       [3, 7],
       [7, 6],
       [4, 7],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G1151',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 7],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1152',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1153',
      7,
      [[3, 4],
       [5, 3],
       [7, 4],
       [5, 1],
       [2, 5],
       [7, 2],
       [4, 2],
       [6, 3],
       [6, 1],
       [6, 7],
       [5, 6],
       [6, 4],
       [7, 1],
       [4, 1]]],
     ['edgelist',
      'G1154',
      7,
      [[3, 4],
       [5, 3],
       [4, 1],
       [5, 1],
       [5, 6],
       [4, 5],
       [4, 2],
       [6, 3],
       [1, 2],
       [6, 7],
       [7, 1],
       [6, 4],
       [7, 5],
       [2, 7]]],
     ['edgelist',
      'G1155',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1156',
      7,
      [[1, 4],
       [1, 5],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1157',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1158',
      7,
      [[1, 2],
       [1, 6],
       [1, 7],
       [2, 5],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1159',
      7,
      [[1, 2],
       [1, 5],
       [1, 7],
       [2, 4],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1160',
      7,
      [[3, 4],
       [5, 3],
       [7, 4],
       [5, 1],
       [2, 5],
       [5, 6],
       [4, 2],
       [6, 3],
       [6, 1],
       [7, 2],
       [1, 7],
       [6, 4],
       [7, 5],
       [4, 1]]],
     ['edgelist',
      'G1161',
      7,
      [[1, 2],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1162',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1163',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [6, 7]]],
     ['edgelist',
      'G1164',
      7,
      [[3, 4],
       [5, 3],
       [7, 4],
       [5, 1],
       [5, 6],
       [4, 6],
       [4, 2],
       [6, 3],
       [4, 1],
       [2, 5],
       [7, 1],
       [2, 7],
       [7, 5],
       [1, 2]]],
     ['edgelist',
      'G1165',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G1166',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1167',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1168',
      7,
      [[1, 4],
       [1, 5],
       [1, 6],
       [2, 3],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1169',
      7,
      [[1, 4],
       [1, 5],
       [1, 6],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 7],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1170',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [6, 7],
       [1, 7],
       [1, 3],
       [6, 1],
       [4, 6],
       [2, 4],
       [7, 2],
       [5, 7],
       [3, 5]]],
     ['edgelist',
      'G1171',
      7,
      [[1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 7],
       [5, 6]]],
     ['edgelist',
      'G1172',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [5, 6]]],
     ['edgelist',
      'G1173',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [2, 4],
       [3, 1],
       [5, 1],
       [6, 4],
       [2, 7]]],
     ['edgelist',
      'G1174',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [3, 5],
       [6, 3],
       [2, 6],
       [2, 5],
       [2, 4],
       [3, 1],
       [5, 1],
       [6, 4],
       [1, 7]]],
     ['edgelist',
      'G1175',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [6, 7]]],
     ['edgelist',
      'G1176',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 6],
       [2, 7],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1177',
      7,
      [[4, 5],
       [5, 6],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [4, 7],
       [5, 7],
       [6, 7],
       [2, 6],
       [4, 6],
       [3, 4],
       [3, 5],
       [2, 7],
       [1, 2]]],
     ['edgelist',
      'G1178',
      7,
      [[4, 5],
       [5, 6],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [4, 6],
       [3, 4],
       [3, 5],
       [7, 2],
       [5, 7]]],
     ['edgelist',
      'G1179',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 2],
       [5, 3],
       [5, 4],
       [6, 2],
       [1, 6],
       [6, 3],
       [4, 6],
       [5, 6],
       [7, 2],
       [6, 7]]],
     ['edgelist',
      'G1180',
      7,
      [[5, 4],
       [5, 6],
       [6, 4],
       [1, 2],
       [1, 6],
       [1, 4],
       [3, 5],
       [2, 6],
       [2, 4],
       [7, 6],
       [4, 7],
       [7, 1],
       [2, 7],
       [7, 3],
       [5, 7]]],
     ['edgelist',
      'G1181',
      7,
      [[4, 5],
       [5, 6],
       [6, 7],
       [1, 5],
       [1, 6],
       [1, 7],
       [4, 7],
       [2, 4],
       [5, 7],
       [2, 6],
       [4, 6],
       [3, 4],
       [3, 5],
       [2, 7],
       [1, 2]]],
     ['edgelist',
      'G1182',
      7,
      [[1, 3],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1183',
      7,
      [[7, 2],
       [5, 6],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [4, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [5, 7]]],
     ['edgelist',
      'G1184',
      7,
      [[4, 5],
       [5, 6],
       [1, 4],
       [1, 5],
       [1, 6],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [4, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [6, 3]]],
     ['edgelist',
      'G1185',
      7,
      [[4, 5],
       [5, 6],
       [1, 4],
       [1, 5],
       [7, 1],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [4, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [6, 3]]],
     ['edgelist',
      'G1186',
      7,
      [[1, 2],
       [2, 3],
       [1, 3],
       [4, 1],
       [2, 4],
       [3, 4],
       [6, 2],
       [4, 6],
       [5, 4],
       [3, 5],
       [7, 3],
       [4, 7],
       [7, 2],
       [1, 6],
       [5, 1]]],
     ['edgelist',
      'G1187',
      7,
      [[1, 2],
       [3, 1],
       [4, 3],
       [5, 4],
       [2, 5],
       [4, 2],
       [5, 3],
       [1, 5],
       [4, 1],
       [7, 4],
       [5, 7],
       [6, 5],
       [4, 6],
       [7, 3],
       [6, 2]]],
     ['edgelist',
      'G1188',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [2, 3],
       [2, 4],
       [2, 5],
       [3, 4],
       [3, 5],
       [4, 5],
       [6, 4],
       [5, 6],
       [7, 5],
       [6, 7],
       [7, 4]]],
     ['edgelist',
      'G1189',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [1, 4],
       [5, 1],
       [5, 4],
       [5, 3],
       [7, 2],
       [3, 7],
       [6, 3],
       [5, 6],
       [7, 5],
       [1, 7],
       [7, 6],
       [4, 7]]],
     ['edgelist',
      'G1190',
      7,
      [[1, 2],
       [6, 4],
       [2, 4],
       [1, 5],
       [4, 1],
       [5, 4],
       [3, 5],
       [6, 3],
       [5, 6],
       [7, 5],
       [3, 7],
       [7, 6],
       [4, 7],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G1191',
      7,
      [[6, 3],
       [5, 6],
       [4, 2],
       [1, 5],
       [1, 6],
       [1, 4],
       [3, 5],
       [2, 6],
       [2, 5],
       [7, 4],
       [2, 7],
       [7, 1],
       [6, 7],
       [7, 3],
       [5, 7]]],
     ['edgelist',
      'G1192',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [5, 6],
       [1, 6],
       [1, 4],
       [3, 1],
       [4, 2],
       [7, 5],
       [6, 7],
       [7, 4],
       [1, 7],
       [7, 3],
       [2, 7]]],
     ['edgelist',
      'G1193',
      7,
      [[6, 3],
       [4, 1],
       [6, 4],
       [1, 5],
       [1, 6],
       [5, 4],
       [3, 5],
       [2, 6],
       [2, 5],
       [7, 5],
       [3, 7],
       [7, 1],
       [6, 7],
       [7, 4],
       [2, 7]]],
     ['edgelist',
      'G1194',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1195',
      7,
      [[7, 2],
       [5, 6],
       [1, 4],
       [1, 5],
       [1, 6],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [1, 7],
       [4, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [6, 3]]],
     ['edgelist',
      'G1196',
      7,
      [[4, 5],
       [1, 2],
       [1, 4],
       [1, 5],
       [1, 6],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [2, 7],
       [3, 4],
       [3, 5],
       [6, 7],
       [6, 3]]],
     ['edgelist',
      'G1197',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 7]]],
     ['edgelist',
      'G1198',
      7,
      [[6, 3],
       [5, 6],
       [6, 4],
       [1, 5],
       [1, 2],
       [2, 4],
       [3, 5],
       [4, 1],
       [2, 5],
       [7, 5],
       [2, 7],
       [7, 1],
       [4, 7],
       [7, 3],
       [6, 7]]],
     ['edgelist',
      'G1199',
      7,
      [[6, 1],
       [5, 4],
       [6, 4],
       [6, 3],
       [1, 2],
       [2, 4],
       [3, 5],
       [4, 1],
       [2, 5],
       [7, 3],
       [6, 7],
       [7, 5],
       [4, 7],
       [7, 1],
       [2, 7]]],
     ['edgelist',
      'G1200',
      7,
      [[4, 5],
       [5, 6],
       [1, 4],
       [5, 7],
       [1, 2],
       [2, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [1, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [6, 3]]],
     ['edgelist',
      'G1201',
      7,
      [[1, 3],
       [1, 4],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1202',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1203',
      7,
      [[4, 5],
       [6, 1],
       [1, 4],
       [1, 5],
       [5, 7],
       [2, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [2, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [6, 3]]],
     ['edgelist',
      'G1204',
      7,
      [[7, 5],
       [6, 3],
       [1, 4],
       [1, 5],
       [3, 5],
       [2, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [4, 6],
       [3, 4],
       [1, 2],
       [6, 7],
       [5, 6]]],
     ['edgelist',
      'G1205',
      7,
      [[1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1206',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1207',
      7,
      [[3, 4],
       [1, 3],
       [4, 1],
       [5, 4],
       [2, 5],
       [6, 2],
       [5, 6],
       [2, 1],
       [6, 3],
       [7, 3],
       [6, 7],
       [7, 2],
       [1, 7],
       [7, 5],
       [4, 7]]],
     ['edgelist',
      'G1208',
      7,
      [[4, 1],
       [4, 6],
       [4, 5],
       [3, 1],
       [3, 6],
       [3, 5],
       [2, 5],
       [2, 6],
       [2, 1],
       [7, 1],
       [2, 7],
       [7, 6],
       [4, 7],
       [7, 5],
       [3, 7]]],
     ['edgelist',
      'G1209',
      7,
      [[1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1210',
      7,
      [[4, 5],
       [7, 3],
       [1, 4],
       [1, 5],
       [6, 1],
       [2, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [5, 6],
       [3, 4],
       [3, 5],
       [6, 2],
       [6, 3]]],
     ['edgelist',
      'G1211',
      7,
      [[4, 5],
       [7, 3],
       [1, 4],
       [1, 5],
       [6, 1],
       [6, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [1, 2],
       [5, 7],
       [3, 4],
       [3, 5],
       [6, 2],
       [6, 3]]],
     ['edgelist',
      'G1212',
      7,
      [[1, 2],
       [2, 3],
       [3, 4],
       [4, 5],
       [1, 5],
       [7, 3],
       [2, 7],
       [7, 1],
       [5, 7],
       [6, 5],
       [1, 6],
       [4, 6],
       [7, 4],
       [6, 3],
       [2, 6]]],
     ['edgelist',
      'G1213',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [5, 6],
       [3, 7]]],
     ['edgelist',
      'G1214',
      7,
      [[4, 1],
       [5, 2],
       [5, 4],
       [2, 4],
       [5, 1],
       [3, 6],
       [7, 3],
       [6, 7],
       [2, 6],
       [5, 6],
       [4, 6],
       [1, 6],
       [1, 7],
       [4, 7],
       [5, 7],
       [7, 2]]],
     ['edgelist',
      'G1215',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1216',
      7,
      [[1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1217',
      7,
      [[4, 5],
       [6, 2],
       [1, 4],
       [1, 5],
       [6, 1],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [4, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [5, 6],
       [3, 6]]],
     ['edgelist',
      'G1218',
      7,
      [[3, 5],
       [4, 2],
       [4, 1],
       [5, 4],
       [5, 1],
       [6, 3],
       [5, 6],
       [6, 1],
       [4, 6],
       [6, 2],
       [7, 6],
       [2, 7],
       [4, 7],
       [7, 1],
       [5, 7],
       [7, 3]]],
     ['edgelist',
      'G1219',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1220',
      7,
      [[3, 5],
       [5, 2],
       [4, 1],
       [4, 2],
       [5, 1],
       [6, 3],
       [5, 6],
       [6, 1],
       [4, 6],
       [7, 6],
       [2, 6],
       [7, 2],
       [4, 7],
       [5, 7],
       [7, 3],
       [7, 1]]],
     ['edgelist',
      'G1221',
      7,
      [[1, 2],
       [1, 4],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1222',
      7,
      [[3, 6],
       [1, 2],
       [5, 6],
       [2, 4],
       [6, 1],
       [5, 4],
       [6, 4],
       [3, 5],
       [2, 5],
       [4, 1],
       [7, 4],
       [3, 7],
       [7, 5],
       [6, 7],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G1223',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 4],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1224',
      7,
      [[3, 6],
       [6, 2],
       [4, 2],
       [1, 5],
       [6, 1],
       [5, 4],
       [6, 4],
       [3, 5],
       [2, 5],
       [4, 1],
       [7, 3],
       [5, 7],
       [6, 7],
       [7, 2],
       [1, 7],
       [4, 7]]],
     ['edgelist',
      'G1225',
      7,
      [[2, 7],
       [1, 2],
       [1, 4],
       [1, 5],
       [6, 1],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [4, 6],
       [3, 4],
       [3, 5],
       [6, 7],
       [5, 6],
       [3, 6]]],
     ['edgelist',
      'G1226',
      7,
      [[4, 5],
       [6, 2],
       [1, 4],
       [1, 5],
       [6, 1],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [2, 7],
       [3, 4],
       [3, 5],
       [6, 7],
       [1, 2],
       [3, 6]]],
     ['edgelist',
      'G1227',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6]]],
     ['edgelist',
      'G1228',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [6, 7]]],
     ['edgelist',
      'G1229',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [6, 7]]],
     ['edgelist',
      'G1230',
      7,
      [[3, 6],
       [6, 2],
       [4, 6],
       [1, 5],
       [1, 2],
       [5, 4],
       [4, 3],
       [3, 5],
       [2, 5],
       [1, 6],
       [7, 5],
       [3, 7],
       [7, 4],
       [6, 7],
       [7, 2],
       [1, 7]]],
     ['edgelist',
      'G1231',
      7,
      [[6, 7],
       [6, 2],
       [1, 4],
       [1, 5],
       [1, 2],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [4, 6],
       [3, 4],
       [3, 5],
       [7, 3],
       [5, 6],
       [3, 6]]],
     ['edgelist',
      'G1232',
      7,
      [[4, 5],
       [6, 2],
       [1, 4],
       [1, 5],
       [1, 2],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [6, 1],
       [3, 4],
       [3, 5],
       [7, 3],
       [7, 6],
       [3, 6]]],
     ['edgelist',
      'G1233',
      7,
      [[6, 1],
       [6, 2],
       [1, 4],
       [1, 5],
       [7, 2],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [7, 1],
       [4, 6],
       [3, 4],
       [3, 5],
       [7, 3],
       [5, 6],
       [3, 6]]],
     ['edgelist',
      'G1234',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [5, 6],
       [7, 3],
       [2, 7]]],
     ['edgelist',
      'G1235',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1236',
      7,
      [[5, 1],
       [5, 4],
       [1, 2],
       [4, 1],
       [3, 5],
       [4, 2],
       [6, 4],
       [5, 6],
       [6, 3],
       [7, 6],
       [6, 1],
       [2, 6],
       [7, 2],
       [1, 7],
       [7, 5],
       [3, 7],
       [4, 7]]],
     ['edgelist',
      'G1237',
      7,
      [[1, 2],
       [6, 2],
       [6, 4],
       [1, 5],
       [6, 1],
       [5, 4],
       [4, 2],
       [3, 6],
       [2, 5],
       [4, 1],
       [3, 5],
       [7, 3],
       [6, 7],
       [7, 4],
       [2, 7],
       [7, 1],
       [5, 7]]],
     ['edgelist',
      'G1238',
      7,
      [[4, 5],
       [6, 2],
       [1, 4],
       [1, 5],
       [5, 6],
       [5, 7],
       [4, 7],
       [2, 4],
       [2, 5],
       [1, 2],
       [4, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [6, 1],
       [6, 7],
       [7, 3]]],
     ['edgelist',
      'G1239',
      7,
      [[4, 3],
       [5, 2],
       [1, 2],
       [4, 1],
       [3, 5],
       [5, 4],
       [6, 2],
       [5, 6],
       [6, 3],
       [1, 6],
       [6, 4],
       [7, 6],
       [3, 7],
       [7, 4],
       [1, 7],
       [2, 7],
       [5, 7]]],
     ['edgelist',
      'G1240',
      7,
      [[4, 3],
       [5, 2],
       [5, 1],
       [4, 1],
       [3, 5],
       [4, 2],
       [6, 3],
       [5, 6],
       [6, 1],
       [4, 6],
       [6, 2],
       [7, 6],
       [3, 7],
       [7, 1],
       [4, 7],
       [7, 5],
       [2, 7]]],
     ['edgelist',
      'G1241',
      7,
      [[4, 3],
       [6, 2],
       [6, 1],
       [1, 5],
       [1, 2],
       [5, 4],
       [6, 4],
       [3, 6],
       [2, 5],
       [4, 1],
       [3, 5],
       [7, 5],
       [6, 7],
       [7, 3],
       [4, 7],
       [7, 1],
       [2, 7]]],
     ['edgelist',
      'G1242',
      7,
      [[4, 3],
       [6, 2],
       [6, 1],
       [1, 5],
       [5, 6],
       [1, 2],
       [4, 2],
       [3, 6],
       [2, 5],
       [4, 1],
       [3, 5],
       [7, 1],
       [4, 7],
       [7, 2],
       [6, 7],
       [7, 3],
       [5, 7]]],
     ['edgelist',
      'G1243',
      7,
      [[1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7]]],
     ['edgelist',
      'G1244',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [4, 5],
       [4, 6],
       [5, 6],
       [7, 2],
       [1, 7],
       [6, 7]]],
     ['edgelist',
      'G1245',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7]]],
     ['edgelist',
      'G1246',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 5],
       [2, 6],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1247',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1248',
      7,
      [[5, 1],
       [5, 6],
       [4, 1],
       [4, 6],
       [3, 1],
       [3, 6],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 1],
       [3, 4],
       [3, 5],
       [7, 1],
       [6, 7],
       [7, 2],
       [3, 7],
       [7, 5],
       [4, 7]]],
     ['edgelist',
      'G1249',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1250',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1251',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]],
     ['edgelist',
      'G1252',
      7,
      [[1, 2],
       [1, 3],
       [1, 4],
       [1, 5],
       [1, 6],
       [1, 7],
       [2, 3],
       [2, 4],
       [2, 5],
       [2, 6],
       [2, 7],
       [3, 4],
       [3, 5],
       [3, 6],
       [3, 7],
       [4, 5],
       [4, 6],
       [4, 7],
       [5, 6],
       [5, 7],
       [6, 7]]]]

    GAG=[]

    for i in range(1253):
        g=make_small_graph(descr_list[i])
        GAG.append(g)

    return GAG


def _test_suite():
    """test hook"""
    import doctest
    suite = doctest.DocFileSuite('tests/generators/atlas.txt',
                                 package='networkx')
    return suite

if __name__ == "__main__":
    import os
    import sys
    import unittest
    if sys.version_info[:2] < (2, 4):
        print "Python version 2.4 or later required (%d.%d detected)." \
              %  sys.version_info[:2]
        sys.exit(-1)
    # directory of networkx package (relative to this)
    nxbase=sys.path[0]+os.sep+os.pardir
    sys.path.insert(0,nxbase) # prepend to search path
    unittest.TextTestRunner().run(_test_suite())
    


