/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//
//  BBOBEXServerSession.h
//  BTUtil
//
//  A Bluetooth OBEX server session. 
// 
//  Server sessions are created using incoming client channels (i.e. the client
//  connections that are spawned from server connections). A BBOBEXServerSession
//  talks to a single client session, and a BBOBEXServer instance uses 
//  BBOBEXServerSession objects to talk to multiple client sessions.
//
//  See the included BTUtil "SimpleOBEXServer" example.
//

#import <Cocoa/Cocoa.h>
#import <Foundation/Foundation.h>

#import <IOBluetooth/Bluetooth.h>
#import <IOBluetooth/OBEX.h>

@class BBOBEXServer, BBOBEXFileReader, IOBluetoothOBEXSession;

@interface BBOBEXServerSession : NSObject {
	BBOBEXServer *mOBEXServer;
	IOBluetoothOBEXSession *mOBEXSession;

	BOOL mRunning;
	
	OBEXOpCode mCurrentOp;	// only used for PUT, GET
	OBEXMaxPacketLength	mMaxPacketLength;
	
	NSFileHandle *mPutFileHandle;

	BBOBEXFileReader *mCurrentGetBodyHandler;
	NSFileHandle *mGetFileHandle;
	CFMutableDataRef mGetHeadersDataRef;
	int mNextGetDelegateReply;
}

/*
 * Sets whether debug messages should be displayed (default is NO).
 */
+ (void)setDebug:(BOOL)doDebug;

/*
 * Creates a server session.
 */
- (id)initWithServer:(BBOBEXServer *)server;

/*
 * Starts the OBEX session.
 * Returns kOBEXSuccess if the session was successfully started,
 * kOBEXSessionAlreadyConnectedError if the session is already running, or one 
 * of the other OBEX error codes if there was an error.
 */
- (OBEXError)runWithIncomingRFCOMMChannel:(IOBluetoothRFCOMMChannel *)channel;

/*
 * Closes the OBEX session. (This is automatically called when the client
 * disconnects.)
 * Returns kOBEXSuccess if the transport connection was successfully closed,
 * kOBEXSessionNotConnectedError if the session is not running, or one 
 * of the other OBEX error codes if there was an error.
 */
- (OBEXError)close;

/*
 * Returns NO if the session is closed or has not been successfully started,
 * otherwise returns YES.
 */
- (BOOL)isRunning;

/* 
 * Returns whether the session has an opened OBEX connection.
 */
- (BOOL)hasOpenOBEXConnection;

/* 
 * Returns whether the session has an opened transport connection.
 */
- (BOOL)hasOpenTransportConnection;

/* 
 * Sets the maximum packet length for this OBEX session. This should be set 
 * before the OBEX connection for this session is established.
 * Note this value can change after the session is connected if the client
 * negotiates a lower max packet length.
 * (The max packet length defaults to 0x2000 bytes.)
 */
- (void)setMaxPacketLength:(OBEXMaxPacketLength)length;

/* 
 * Returns the associated RFCOMM channel for this OBEX session.
 */
- (IOBluetoothRFCOMMChannel *)getRFCOMMChannel;

/*
 * Returns the BBOBEXServer used to create this OBEX session.
 */
- (BBOBEXServer *)getOBEXServer;

@end


/*
 * Informal protocol to describe the BBOBEXServerSessionnDelegate delegate 
 * methods that can be implemented.
 *
 * The request methods (e.g. obexServerSessionConnectRequest:) allow the 
 * delegate to either allow or deny a client request. 
 *	- To allow a request, the method should return kOBEXSuccess. 
 *	- To deny a request, the method should return one of the other response 
 *	  codes listed in the OBEXOpCodeResponseValues enum in <IOBluetooth/OBEX.h>.
 *    Use the codes that have 'WithFinalBit' on the end. For example, you could
 *	  return kOBEXResponseCodeNotFoundWithFinalBit if a client requests a non-
 *	  existent file.
 */
@protocol BBOBEXServerSessionDelegate

/*
 * Called when a CONNECT request is received.
 * This method should return kOBEXSuccess if the request should be allowed, or
 * an appropriate response code if the request should be denied (see above).
 * If this method is not implemented by the delegate, connections will  
 * automatically be accepted.
 *
 * Arguments:
 *	- session: the server session which received the request
 */
- (int)obexServerSessionConnectRequest:(BBOBEXServerSession *)session;

/*
 * Called when a connection is succesfully established.
 * If an error occured during the connection process, this method will not be
 * called; obexServerError:error:summary: will be called instead.
 *
 * Arguments:
 *	- session: the server which received the request
 */
- (void)obexServerSessionConnectComplete:(BBOBEXServerSession *)session;

/*
 * Called when a session is succesfully disconnected.
 * If an error occured during the disconnection process, this method will not be
 * called; obexServerError:error:summary: will be called instead.
 *
 * Arguments:
 *	- session: the server which received the request
 */
- (void)obexServerSessionDisconnectComplete:(BBOBEXServerSession *)session;

/*
 * Called when a PUT request is received.
 * This method should return kOBEXSuccess if the request should be allowed, or
 * an appropriate response code if the request should be denied (see above).
 * If this method is not implemented by the delegate, the server will refuse
 * the request and send a "Not Implemented" response to the client.
 *
 * Arguments:
 *	- session: the server which received the request
 *	- name: the name of the data that will be sent, as specified by the client.
 *	  This is nil if the client did not specify a name.
 *	- type: the type of the data that will be sent, as specified by the client.
 *	  This is nil if the client did not specify a type.
 *	- totalLength: the total length of the data that will be sent. This is -1
 *	  if the client did not specify the total length.
 *	- destFile: set this value to point to a file handle to which the received 
 *	  PUT body data will be written. This cannot be nil if kOBEXSuccess is 
 *	  returned.
 */
- (int)obexServerSessionPutRequest:(BBOBEXServerSession *)session
						  withName:(NSString *)name
							  type:(NSString *)type
							length:(unsigned)totalLength
					   writeToFile:(NSFileHandle **)destFile;

/*
 * Called each time data is received for a PUT request.
 * This method should return kOBEXSuccess if the PUT should continue, or
 * an appropriate response code if it should be stopped (see above).
 *
 * Arguments:
 *	- server: the server which received the request
 *  - destFile: the file to which the received PUT body data is written
 *	- bytecount: the amount of body data received in this packet, in bytes
 */
- (int)obexServerSessionPutProgress:(BBOBEXServerSession *)session
							forFile:(NSFileHandle *)destFile
						   byAmount:(unsigned)bytecount;

/*
 * Called when a PUT operation is successfully completed.
 * This will not be called if the PUT is aborted by the client, or if the PUT
 * was refused or stopped, or if some error occurs. If an error occurs, 
 * obexServerError:error:summary: will be called.
 *
 * Arguments:
 *	- session: the server which received the request
 *  - destFile: the file to which the received PUT body data was written 
 */
- (void)obexServerSessionPutComplete:(BBOBEXServerSession *)session
							 forFile:(NSFileHandle *)destFile;

/*
 * Called when a GET request is received.
 * This method should return kOBEXSuccess if the request should be allowed, or
 * an appropriate response code if the request should be denied (see above).
 * If this method is not implemented by the delegate, the server will refuse
 * the request and send a "Not Implemented" response to the client.
 *
 * Arguments:
 *	- session: the server which received the request
 *	- name: the name of the requested data, as specified by the client
 *	  This is nil if the client did not specify a name.
 *	- type: the type of the requested data, as specified by the client.
 *	  This is nil if the client did not specify a type.
 *	- sourceFile: set this value to point to a file handle to which the data 
 *	  will be read and sent for the GET operation. This cannot be nil if 
 *    kOBEXSuccess is returned.
 */
- (int)obexServerSessionGetRequest:(BBOBEXServerSession *)session
						  withName:(NSString *)name
							  type:(NSString *)type
					  readFromFile:(NSFileHandle **)sourceFile;

/*
 * Called each time data is sent for a GET request.
 * This method should return kOBEXSuccess if the GET should continue, or
 * an appropriate response code if it should be stopped (see above).
 *
 * Arguments:
 *	- server: the server which received the request
 * 	- sourceFile: the file from which data is being read for the GET operation
 *	- bytecount: the amount of body data sent in this packet, in bytes
 */
- (int)obexServerSessionGetProgress:(BBOBEXServerSession *)session
							 forFile:(NSFileHandle *)sourceFile
						   byAmount:(unsigned)bytecount;

/*
 * Called when a GET operation is successfully completed.
 * This will not be called if the GET is aborted by the client, or if the GET
 * was refused or stopped, or if some error occurs. If an error occurs, 
 * obexServerError:error:summary: will be called.
 *
 * Arguments:
 *	- session: the server which received the request
 * 	- sourceFile: the file from which data was read for the GET operation
 */
- (void)obexServerSessionGetComplete:(BBOBEXServerSession *)session
							 forFile:(NSFileHandle *)sourceFile;

/*
 * Called when a client sends a SETPATH request.
 * This method should return kOBEXSuccess if the request should be allowed, or
 * an appropriate response code if the request should be denied (see above).
 * If this method is not implemented by the delegate, the server will send a 
 * "Not Implemented" response to the client.
 *
 * Arguments:
 *	- session: the server which received the request
 *	- pathName: the name of the requested path, as specified by the client
 *	  This is nil if the client did not specify a name.
 *	- flags: the flags for the request, as specified by the client.
 */
- (int)obexServerSessionSetPathRequest:(BBOBEXServerSession *)session
							   forPath:(NSString *)pathName
							 withFlags:(OBEXFlags)flags;

/*
 * Called when a SETPATH process is successfully completed.
 * If an error occured during the SETPATH process, this method will not be
 * called; obexServerError:error:summary: will be called instead.
 *
 * Arguments:
 *	- session: the server which received the request
 */
- (void)obexServerSessionSetPathComplete:(BBOBEXServerSession *)session;

/*
 * Called when a client sends an ABORT request and the server session is in the
 * middle of a multi-packet operation - i.e. PUT or GET. (Single-packet 
 * operations cannot be aborted.)
 * This method should return kOBEXSuccess if the request should be allowed, or
 * an appropriate response code if the request should be denied (see above).
 * If this method is not implemented by the delegate, the server will send a 
 * "Not Implemented" response to the client.
 *
 * Arguments:
 *	- server: the server which received the request
 *	- op: the operation currently in progress. This is either kOBEXOpCodePut or
 *	  kOBEXOpCodeGet. 
 */
- (int)obexServerSessionAbortRequest:(BBOBEXServerSession *)session
						forOperation:(OBEXOpCode)op;

/*
 * Called when an ABORT process is successfully completed.
 * If an error occured during the ABORT process, this method will not be
 * called; obexServerError:error:summary: will be called instead.
 *
 * Arguments:
 *	- session: the server which received the request
 */
- (void)obexServerSessionAbortComplete:(BBOBEXServerSession *)session;

/*
 * Called when an error occurs during the server session.
 *
 * Arguments:
 *	- session: the server session on which the error occured
 *	- error: the corresponding error code
 *	- summary: an appropriate string description of the error
 */
- (void)obexServerSessionError:(BBOBEXServerSession *)session
						 error:(OBEXError)error 
					   summary:(NSString *)summary;

@end