# simpletask.py - simple, extensible task abstractions for synchronous threads
#
# Copyright 2005 Lars Wirzenius
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# You may contact the author at <liw@iki.fi>.


"""A simple background computation task abstraction.

This module defines the classes BackgroundTask and TaskQueue, which implement
a simple abstraction for background computation tasks. The computation is to
be implemented so that it can be done in suitably small pieces, and BackgroundTask
and TaskQueue then call the computation function whenever there is time to do
the background computation.

BackgroundTask is meant for the actual computation. You should derive a subclass
from it and implement the computation in the subclass.

TaskQueue is for when you have many background computations. For example, if
you are computing checksums for a list of files, each file might correspond to
one BackgroundTask, and each of those is added to the TaskQueue. The TaskQueue
makes sure the computations are done in order and also provides some tools
for measuring the speed and estimating the remaining time.

"""


class BackgroundTask:

    """A background task.
    
    To use: derive a subclass and define the methods setup, work, and
    cleanup. Then call the run method when you want to start the task.
    Call the do_work method whenever you want the task to do things for
    a while, until it returns False. Call the stop method if you want to
    stop the task before it finishes normally.
    
    For progress reporting purposes, the method estimate is used."""

    def estimate(self):
        """Return (total, done) giving total amount of work, and amount done.
        
        If self.total and self.done exist, they are returned, otherwise
        (0,0).

        """

        if "total" in dir(self) and "done" in dir(self):
            return self.total, self.done
        return (0, 0)

    def description(self):
        """Return string describing to people what is currently going on."""
        return ""

    def setup(self):
        """Set up the task so it can start running."""
        pass
        
    def work(self):
        """Do some work. Return False if done, True if more work to do."""
        return False
        
    def cleanup(self):
        """Clean up the task after all work has been done."""
        pass

    def is_running(self):
        """Is the task running?"""
        if "running" not in dir(self):
            self.running = False
        return self.running

    def start(self):
        """Start running the task. Call setup()."""
        self.setup()
        self.running = True
    
    def do_work(self):
        """Do some work by calling work()."""
        if not self.is_running():
            return False
        elif self.work():
            return True
        else:
            self.running = False
            return False

    def stop(self):
        """Stop task processing. cleanup() is not called."""
        self.running = False

    def run_completely(self):
        """Run the task until it has finished."""
        self.start()
        while self.do_work():
            pass
        self.cleanup()


class TaskQueue(BackgroundTask):

    """A queue of tasks.
    
    A task queue is a queue of other tasks. If you need, for example, to
    do simple tasks A, B, and C, you can create a TaskQueue and add the
    simple tasks to it:
    
        q = TaskQueue()
        q.add(A)
        q.add(B)
        q.add(C)
        q.run_completely()
        
    The task queue behaves as a single BackgroundTask. It will execute
    the tasks in order and start the next one when the previous
    finishes.
    
    Subclasses may define start_hook, work_hook, and cleanup_hook
    functions to add things to be done when the queued tasks are
    started, they work, or they finish."""

    def init_tasks(self):
        if "tasks" not in dir(self):
            self.tasks = []
            self.current_task = 0

    def add(self, task):
        """Add a task to the queue. The queue may already be running."""
        self.init_tasks()
        self.tasks.append(task)
        
    def get_current_task(self):
        """Get the currently executing task, or None."""
        self.init_tasks()
        if self.current_task < len(self.tasks):
            return self.tasks[self.current_task]
        else:
            return None

    def work(self):
        task = self.get_current_task()
        if task:
            if not task.is_running():
                task.start()
                self.setup_hook(task)

            more_to_do = task.work()
            self.work_hook(task)

            if not more_to_do:
                task.cleanup()
                self.cleanup_hook(task)
                self.current_task += 1

        return self.current_task < len(self.tasks)

    def stop(self):
        task = self.get_current_task()
        if task:
            task.stop()
        BackgroundTask.stop(self)
        self.tasks = []

    def estimate(self):
        self.init_tasks()
        total = 0
        done = 0
        for task in self.tasks:
            (task_total, task_done) = task.estimate()
            total += task_total
            done += task_done
        return (total, done)

    def description(self):
        task = self.get_current_task()
        if task:
            return task.description()
        else:
            return ""

    # The following hooks are called after each sub-task has been set up,
    # after its work method has been called, and after it has finished.
    # Subclasses may override these to provide additional processing.

    def setup_hook(self, task):
        pass
        
    def work_hook(self, task):
        pass
        
    def cleanup_hook(self, task):
        pass


import unittest


ITERATIONS = 10


class SimpleTask(BackgroundTask):

    def __init__(self):
        self.did_setup = 0
        self.did_work = 0
        self.did_cleanup = 0
        
    def setup(self):
        self.did_setup += 1
        assert self.did_setup < 2
        
    def work(self):
        self.did_work += 1
        if self.did_work >= ITERATIONS:
            return False
        else:
            return True
        
    def cleanup(self):
        self.did_cleanup += 1


class BackgroundTaskTests(unittest.TestCase):

    def testSimple(self):
        t = SimpleTask()
        
        self.failUnlessEqual(t.did_setup, 0)
        self.failUnlessEqual(t.did_work, 0)
        self.failUnlessEqual(t.did_cleanup, 0)
        
        t.run_completely()
        
        self.failUnlessEqual(t.did_setup, 1)
        self.failUnlessEqual(t.did_work, ITERATIONS)
        self.failUnlessEqual(t.did_cleanup, 1)


class TaskQueueTests(unittest.TestCase):
    
    def testQueue(self):
        tasks = map(lambda i: SimpleTask(), range(10))
        
        for task in tasks:
            self.failUnlessEqual(task.did_setup, 0)
            self.failUnlessEqual(task.did_work, 0)
            self.failUnlessEqual(task.did_cleanup, 0)
        
        q = TaskQueue()
        for task in tasks:
            q.add(task)
            
        for task in tasks:
            self.failUnlessEqual(task.did_setup, 0)
            self.failUnlessEqual(task.did_work, 0)
            self.failUnlessEqual(task.did_cleanup, 0)
            
        q.run_completely()
            
        for task in tasks:
            self.failUnlessEqual(task.did_setup, 1)
            self.failUnlessEqual(task.did_work, ITERATIONS)
            self.failUnlessEqual(task.did_cleanup, 1)

    def testQueueInQueue(self):
        st = SimpleTask()

        q1 = TaskQueue()
        q1.add(st)
        q2 = TaskQueue()
        q2.add(q1)

        self.failUnlessEqual(st.did_setup, 0)
        self.failUnlessEqual(st.did_work, 0)
        self.failUnlessEqual(st.did_cleanup, 0)

        q2.run_completely()

        self.failUnlessEqual(st.did_setup, 1)
        self.failUnlessEqual(st.did_work, ITERATIONS)
        self.failUnlessEqual(st.did_cleanup, 1)

if __name__ == "__main__":
    unittest.main()
